#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "pvm3.h"

int sort_function(const void *a, const void *b)
{
  int *aa = (int *)a;
  int *bb = (int *)b;

  return ( *aa > *bb ) ? 1 : -1;
}

void compare_exchange(int ME, int NPROC, int *taskid,
                      int dest, int *data, int Nlocal)
{
  static int k = 0;

  if (dest != ME && dest >=0 && dest < NPROC) {
    int *buf1 = (int *)malloc(Nlocal*sizeof(*buf1));
    int *buf2 = (int *)malloc(Nlocal*sizeof(*buf2));
      
    /* Send local contribution to destination */
    pvm_initsend(PvmDataDefault);
    pvm_pkint(data,Nlocal,1);
    pvm_send(taskid[dest],1000 + k);

    /* Receive contribution from destination */
    pvm_recv(taskid[dest],1000 + k);
    pvm_upkint(buf2,Nlocal,1);

    /* Merge with current data set */
    memcpy(buf1,data,Nlocal*sizeof(*data));
    
    if (dest > ME) {
      int *p1 = buf1, *p2 = buf2;
      int rembuf1 = Nlocal, rembuf2 = Nlocal;
      int i = 0;
      while (rembuf1 > 0 && rembuf2 > 0) {
        if (*p1 <= *p2) {
          data[i++] = *p1++; /* Take from buf1[] */
          rembuf1--;
        }
        else {
          data[i++] = *p2++; /* Take from buf2[] */
          rembuf2--;
        }
      } /*  while (rembuf1 > 0 && rembuf2 > 0) */
    } /* if (dest > ME) */
    else { /* dest < ME */
      int *p1 = buf1 + Nlocal - 1, *p2 = buf2 + Nlocal - 1;
      int rembuf1 = Nlocal, rembuf2 = Nlocal;
      int i = Nlocal - 1;
      while (rembuf1 > 0 && rembuf2 > 0) {
        if (*p1 > *p2) {
          data[i--] = *p1--; /* Take from buf1[] */
          rembuf1--;
        }
        else {
          data[i--] = *p2--; /* Take from buf2[] */
          rembuf2--;
        }
      } /* while (rembuf1 > 0 && rembuf2 > 0) */
    } /*  dest < ME */
    free(buf1);
    free(buf2);
  } /* if (dest != ME && dest >=0 && dest < NPROC) */

  k++;
}

int *setup_spmd(int argc, char *argv[], int NPROC, int *ME)
{
  int mytid = pvm_mytid();
  int parent = pvm_parent();
  int *taskid = (int *)malloc(NPROC*sizeof(*taskid));
  int i, msglabel = 0;

  if (parent < 0) {
    char *a_out = strrchr(argv[0],'/'); 
    taskid[0] = mytid;
    pvm_spawn((a_out) ? a_out+1 : argv[0], argv+1,
              PvmTaskDefault,"*",
              NPROC-1,
              &taskid[1]);
    /* Send taskid tables to the other processes */
    pvm_initsend(PvmDataDefault);
    pvm_pkint(taskid,NPROC,1);
    pvm_mcast(&taskid[1],NPROC-1,msglabel);
  }
  else {
    /* Receive taskids from process #0 */
    pvm_recv(parent,msglabel);
    pvm_upkint(taskid,NPROC,1);
  }

  /* Find process index in taskid-table */
  for (i=0; i<NPROC; i++) {
    if (taskid[i] == mytid) {
      *ME = i;
      return taskid;
    }
  }

  *ME = -1; 
  return NULL;
}

void verify_data(char *note, int ME, int NPROC, int *taskid,
            int *data, int Nlocal)
{
  static int msglabel = 1000;
  int i, N = Nlocal * NPROC;

  if (ME > 0) {
    pvm_initsend(PvmDataDefault);
    pvm_pkint(data,Nlocal,1);
    pvm_send(taskid[0],msglabel);
  }
  else {
    int *full_data = (int *)malloc(N*sizeof(*full_data));
    memcpy(full_data,data,N*sizeof(*data));
    for (i=1; i<NPROC; i++) {
      pvm_recv(taskid[i],msglabel);
      pvm_upkint(&full_data[Nlocal*i],Nlocal,1);
    }
    printf("Data %s:\n",note);
    for (i=0; i<N; i++)
      printf("%5d%s",full_data[i],(i+1)%10==0?"\n":"");
    printf("\n");
    free(full_data);
  }
  msglabel++;
}

int main(int argc, char *argv[])
{
  int N = atoi(argv[1]);     /* GLOBAL number of elements to sort */
  int NPROC = atoi(argv[2]); /* Number of SPMD-processes */
  int ME;                    /* Process id [0..NPROC-1] */
  int *data;                 /* Local data array to be sorted */
  int *taskid;               /* PVM taskids */
  int i, phase, Nlocal;

  Nlocal = (N+NPROC-1)/NPROC;
  N = Nlocal*NPROC;

  taskid = setup_spmd(argc, argv, NPROC, &ME);

  if (ME == 0) {
    printf("N, Nlocal, NPROC = %d, %d, %d\n",
           N, Nlocal, NPROC);
  }

  data = (int *)malloc(Nlocal*sizeof(*data));
  for (i=0; i<Nlocal; i++) data[i] = N - (i + ME*Nlocal);

  /* Check and display ALL unsorted data */
  verify_data("before sort",ME,NPROC,taskid,data,Nlocal);

  /* Sorting of local data */
  qsort(data,Nlocal,sizeof(*data),sort_function);

  /* Data after local qsort() */
  verify_data("after local qsort()'s",ME,NPROC,taskid,data,Nlocal);

  /* Odd-Even loop */

  for (phase=1; phase<=NPROC; phase++) {
    if ( phase%2 != 0 ) { /* Odd phase */
      if (ME%2 == 0) 
        compare_exchange(ME,NPROC,taskid,ME+1,data,Nlocal);
      else 
        compare_exchange(ME,NPROC,taskid,ME-1,data,Nlocal);
    }
    else { /* Even phase */
      if (ME%2 == 0) 
        compare_exchange(ME,NPROC,taskid,ME-1,data,Nlocal);
      else 
        compare_exchange(ME,NPROC,taskid,ME+1,data,Nlocal);
    }
  }

  /* Check result */
  verify_data("after sort",ME,NPROC,taskid,data,Nlocal);

  pvm_exit(); /* Finish with PVM */

  return 0;
}

