/*
 * iocmds.c
 *
 * Extended Tcl file I/O commands.
 *---------------------------------------------------------------------------
 * Copyright 1991 Karl Lehenbauer and Mark Diekhans.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies.  Karl Lehenbauer and
 * Mark Diekhans make no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or
 * implied warranty.
 */

#include "tclExtdInt.h"
#include <math.h>

#ifdef TCL_USE_BZERO_MACRO
#    define bzero(to,length)    memset(to,'\0',length)
#endif

/*
 * Macro to enable line buffering mode on a file.
 */
#ifdef TCL_HAVE_SETLINEBUF
#   define SET_LINE_BUF(fp)  setlinebuf (fp)
#else
#   define SET_LINE_BUF(fp)  setvbuf (fp, NULL, _IOLBF, BUFSIZ)
#endif


/*
 * Control block used to pass data used by the binary search routines.
 */
typedef struct binSearchCB_t {
    Tcl_Interp   *interp;         /* Pointer to the interpreter.             */
    char         *cmdName;        /* Cmd name to include in error msg.       */
    char         *fileHandle;     /* Handle of file.                         */
    char         *key;            /* The key to search for.                  */

    FILE         *fileCBPtr;      /* Open file structure.                    */
    dynamicBuf_t  dynBuf;         /* Dynamic buffer to hold a line of file.  */
    long          lastRecOffset;  /* Offset of last record read.             */
    int           cmpResult;      /* -1, 0 or 1 result of string compare.    */
    char         *tclProc;        /* Name of Tcl comparsion proc, or NULL.   */
    } binSearchCB_t;

/*
 * Prototypes of internal functions.
 */
int
StandardKeyCompare _ANSI_ARGS_((char *key,
                                char *line));

int
TclProcKeyCompare _ANSI_ARGS_((binSearchCB_t *searchCBPtr));

int
ReadAndCompare _ANSI_ARGS_((long           fileOffset,
                            binSearchCB_t *searchCBPtr));

int
BinSearch _ANSI_ARGS_((binSearchCB_t *searchCBPtr));

FILE *
DoNormalDup _ANSI_ARGS_((Interp     *iPtr,
                         char       *tclCommand,
                         OpenFile   *oldFilePtr));

FILE *
DoSpecialDup _ANSI_ARGS_((Interp     *iPtr,
                          char       *tclCommand,
                          OpenFile   *oldFilePtr,
                          char       *newHandleName));

int
GetFcntlFlags _ANSI_ARGS_((Tcl_Interp *interp,
                           char       *cmdName,
                           OpenFile   *filePtr));

int
SetFcntlFlag _ANSI_ARGS_((Tcl_Interp *interp,
                          char       *cmdName,
                          char       *flagName,
                          char       *valueStr,
                          OpenFile   *filePtr));

int
ParseSelectFileList _ANSI_ARGS_((Tcl_Interp *interp,
                                 char       *handleList,
                                 fd_set     *fileDescSetPtr,
                                 int       **fileDescListPtr,
                                 int        *maxFileIdPtr));

static char *
ReturnSelectedFileList _ANSI_ARGS_((fd_set     *fileDescSetPtr,
                                    int         fileDescCnt,
                                    int        *fileDescList));

/*
 *----------------------------------------------------------------------
 *
 * StandardKeyCompare --
 *    Standard comparison routine for BinSearch, compares the key to the
 *    first white-space seperated field in the line.
 *
 * Parameters:
 *   o key (I) - The key to search for.
 *   o line (I) - The line to compare the key to.
 *
 * Results:
 *   o < 0 if key < line-key
 *   o = 0 if key == line-key
 *   o > 0 if key > line-key.
 *----------------------------------------------------------------------
 */
static int
StandardKeyCompare (key, line)
    char *key;
    char *line;
{
    int  cmpResult, fieldLen;
    char saveChar;

    fieldLen = strcspn (line, " \t\r\n\v\f");

    saveChar = line [fieldLen];
    line [fieldLen] = 0;
    cmpResult = strcmp (key, line);
    line [fieldLen] = saveChar;

    return cmpResult;
}

/*
 *----------------------------------------------------------------------
 *
 * TclProcKeyCompare --
 *    Comparison routine for BinSearch that runs a Tcl procedure to, 
 *    compare the key to a line from the file.
 *
 * Parameters:
 *   o searchCBPtr (I/O) - The search control block, the line should be in
 *     dynBuf, the comparsion result is returned in cmpResult.
 *
 * Results:
 *   TCL_OK or TCL_ERROR.
 *----------------------------------------------------------------------
 */
static int
TclProcKeyCompare (searchCBPtr)
    binSearchCB_t *searchCBPtr;
{
    char *cmdArgv [3];
    char *command;
    int   result;

    cmdArgv [0] = searchCBPtr->tclProc;
    cmdArgv [1] = searchCBPtr->key;
    cmdArgv [2] = searchCBPtr->dynBuf.ptr;
    command = Tcl_Merge (3, cmdArgv);

    result = Tcl_Eval (searchCBPtr->interp, command, 0, (char **) NULL);

    ckfree (command);
    if (result == TCL_ERROR)
        return TCL_ERROR;

    if (!Tcl_StrToInt (searchCBPtr->interp->result, 0, 
                       &searchCBPtr->cmpResult)) {
        char *oldResult = ckalloc (strlen (searchCBPtr->interp->result + 1));
        
        strcpy (oldResult, searchCBPtr->interp->result);
        Tcl_ResetResult (searchCBPtr->interp);
        Tcl_AppendResult (searchCBPtr->interp, "invalid integer \"", oldResult,
                          "\" returned from compare proc \"",
                          searchCBPtr->tclProc, "\"", (char *) NULL);
        ckfree (oldResult);
        return TCL_ERROR;
    }
    Tcl_ResetResult (searchCBPtr->interp);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * ReadAndCompare --
 *    Search for the next line in the file starting at the specified
 *    offset.  Read the line into the dynamic buffer and compare it to
 *    the key using the specified comparison method.  The start of the
 *    last line read is saved in the control block, and if the start of
 *    the same line is found in the search, then it will not be recompared.
 *    This is needed since the search algorithm has to hit the same line
 *    a couple of times before failing, due to the fact that the records are
 *    not fixed length.
 *
 * Parameters:
 *   o fileOffset (I) - The offset of the next byte of the search, not
 *     necessarly the start of a record.
 *   o searchCBPtr (I/O) - The search control block, the comparsion result
 *     is returned in cmpResult.  If the EOF is hit, a less-than result is
 *     returned.
 *
 * Results:
 *   TCL_OK or TCL_ERROR.
 *----------------------------------------------------------------------
 */
static int
ReadAndCompare (fileOffset, searchCBPtr)
    long           fileOffset;
    binSearchCB_t *searchCBPtr;
{
    int  recChar, status;

    if (fseek (searchCBPtr->fileCBPtr, fileOffset, SEEK_SET) != 0)
        goto unixError;

    /*
     * Go to beginning of next line.
     */
    
    if (fileOffset != 0) {
        while (((recChar = getc (searchCBPtr->fileCBPtr)) != EOF) &&
                (recChar != '\n'))
            fileOffset++;
        if ((recChar == EOF) && ferror (searchCBPtr->fileCBPtr))
            goto unixError;
    }
    /*
     * If this is the same line as before, then just leave the comparison
     * result unchanged.
     */
    if (fileOffset == searchCBPtr->lastRecOffset)
        return TCL_OK;

    searchCBPtr->lastRecOffset = fileOffset;

    status = Tcl_DynamicFgets (&searchCBPtr->dynBuf, searchCBPtr->fileCBPtr);
    if (status < 0)
        goto unixError;

    /* 
     * Only compare if EOF was not hit, otherwise, treat as if we went
     * above the key we are looking for.
     */
    if (status == 0) {
        searchCBPtr->cmpResult = -1;
        return TCL_OK;
    }

    if (searchCBPtr->tclProc == NULL) {
        searchCBPtr->cmpResult = StandardKeyCompare (searchCBPtr->key, 
                                                     searchCBPtr->dynBuf.ptr);
    } else {
        if (TclProcKeyCompare (searchCBPtr) != TCL_OK)
            return TCL_ERROR;
    }

    return TCL_OK;

unixError:
   Tcl_AppendResult (searchCBPtr->interp, searchCBPtr->cmdName, 
                     ": ", searchCBPtr->fileHandle, ": ",
                     Tcl_UnixError (searchCBPtr->interp), (char *) NULL);
   return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * BinSearch --
 *      Binary search a sorted ASCII file.
 *
 * Parameters:
 *   o searchCBPtr (I/O) - The search control block, if the line is found,
 *     it is returned in dynBuf.
 * Results:
 *     TCL_OK - If the key was found.
 *     TCL_BREAK - If it was not found.
 *     TCL_ERROR - If there was an error.
 *
 * based on getpath.c from smail 2.5 (9/15/87)
 *
 *----------------------------------------------------------------------
 */
static int
BinSearch (searchCBPtr)
    binSearchCB_t *searchCBPtr;
{
    OpenFile   *filePtr;
    long        middle, high, low;
    struct stat statBuf;

    if (TclGetOpenFile (searchCBPtr->interp, searchCBPtr->fileHandle, 
                        &filePtr) != TCL_OK)
        goto unixError;

    searchCBPtr->fileCBPtr = filePtr->f;
    searchCBPtr->lastRecOffset = -1;

    if (fstat (fileno (searchCBPtr->fileCBPtr), &statBuf) < 0)
        goto unixError;

    low = 0;
    high = statBuf.st_size;

    /*
     * "Binary search routines are never written right the first time around."
     * - Robert G. Sheldon.
     */

    while (TRUE) {
        middle = (high + low + 1) / 2;

        if (ReadAndCompare (middle, searchCBPtr) != TCL_OK)
            return TCL_ERROR;

        if (searchCBPtr->cmpResult == 0)
            return TCL_OK;     /* Found   */
        
        if (low >= middle)  
            return TCL_BREAK;  /* Failure */

        /*
         * Close window.
         */
        if (searchCBPtr->cmpResult > 0) {
            low = middle;
        } else {
            high = middle - 1;
        }
    }

unixError:
   Tcl_AppendResult (searchCBPtr->interp, searchCBPtr->cmdName, 
                     ": ", searchCBPtr->fileHandle, ": ",
                     Tcl_UnixError (searchCBPtr->interp), (char *) NULL);
   return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_BsearchCmd --
 *     Implements the TCL bsearch command:
 *        bsearch filehandle key [retvar]
 *
 * Results:
 *      Standard TCL results.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_BsearchCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    int           status;
    binSearchCB_t searchCB;

    if ((argc < 3) || (argc > 5)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " handle key [retvar] [compare_proc]"
                          , (char *) NULL);
        return TCL_ERROR;
    }

    searchCB.interp = interp;
    searchCB.cmdName = argv [0];
    searchCB.fileHandle = argv [1];
    searchCB.key = argv [2];
    searchCB.tclProc = (argc == 5) ? argv [4] : NULL;
    Tcl_DynBufInit (&searchCB.dynBuf);

    status = BinSearch (&searchCB);
    if (status == TCL_ERROR) {
        Tcl_DynBufFree (&searchCB.dynBuf);
        return TCL_ERROR;
    }

    if (status == TCL_BREAK) {
        Tcl_DynBufFree (&searchCB.dynBuf);
        if ((argc >= 4) && (argv [3][0] != '\0'))
            interp->result = "0";
        return TCL_OK;
    }

    if ((argc == 3) || (argv [3][0] == '\0')) {
        Tcl_DynBufReturn (interp, &searchCB.dynBuf);
    } else {
        char *varPtr;

        varPtr = Tcl_SetVar (interp, argv[3], searchCB.dynBuf.ptr,
                             TCL_LEAVE_ERR_MSG);
        Tcl_DynBufFree (&searchCB.dynBuf);
        if (varPtr == NULL)
            return TCL_ERROR;
        interp->result = "1";
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * DoNormalDup --
 *   Process a normal dup command (i.e. the new file is not specified).
 *
 * Parameters:
 *   o iPtr (I) - If an error occures, the error message is in result,
 *     otherwise the file handle is in result.
 *   o tclCommand (I) - The command name (argv [0]), for error reporting.
 *   o oldFilePtr (I) - Tcl file control block for the file to dup.
 * Returns:
 *   A pointer to the FILE structure for the new file, or NULL if an
 *   error occured. 
 *----------------------------------------------------------------------
 */
static FILE *
DoNormalDup (iPtr, tclCommand, oldFilePtr)
    Interp     *iPtr;
    char       *tclCommand;
    OpenFile   *oldFilePtr;
{
    int       newFileId;
    FILE     *newFileCbPtr;
    char     *mode;

    newFileId = dup (fileno (oldFilePtr->f));
    if (newFileId < 0)
        goto unixError;

    TclMakeFileTable (iPtr, newFileId);
    if (iPtr->filePtrArray [newFileId] != NULL) {
        panic ("Tcl_OpenCmd found file already open");
    }
    /*
     * Set up a stdio FILE control block for the new file.
     */
    if (oldFilePtr->readable && oldFilePtr->writable) {
        mode = "r+";
    } else if (oldFilePtr->writable) {
        mode = "w";
    } else {
        mode = "r";
    }
    if ((newFileCbPtr = fdopen (newFileId, mode)) == NULL)
        goto unixError;

    sprintf (iPtr->result, "file%d", newFileId);
    return newFileCbPtr;

unixError:
    Tcl_AppendResult ((Tcl_Interp *) iPtr, tclCommand, ": ", 
                      Tcl_UnixError ((Tcl_Interp *) iPtr), (char *) NULL);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * DoSpecialDup --
 *   Process a special dup command.  This is the case were the file is
 *   dup-ed to stdin, stdout or stderr.  The new file may or be open or
 *   closed
 * Parameters:
 *   o iPtr (I) - If an error occures, the error message is in result,
 *     otherwise nothing is returned.
 *   o tclCommand (I) - The command name (argv [0]), for error reporting.
 *   o oldFilePtr (I) - Tcl file control block for the file to dup.
 *   o newFileHandle (I) - The handle name for the new file.
 * Returns:
 *   A pointer to the FILE structure for the new file, or NULL if an
 *   error occured. 
 *----------------------------------------------------------------------
 */
static FILE *
DoSpecialDup (iPtr, tclCommand, oldFilePtr, newHandleName)
    Interp     *iPtr;
    char       *tclCommand;
    OpenFile   *oldFilePtr;
    char       *newHandleName;
{
    int       newFileId;
    FILE     *newFileCbPtr;

    /*
     * Duplicate the old file to the specified file id.
     */
    newFileId = Tcl_ConvertFileHandle ((Tcl_Interp *) iPtr, newHandleName);
    if (newFileId < 0)
        return NULL;
    if (newFileId > 2) {
        Tcl_AppendResult (iPtr, "target handle must be one of stdin, ",
                          "stdout, stderr, file0, file1, or file2: got \"",
                          newHandleName, "\"", (char *) NULL);
        return NULL;
    }
    switch (newFileId) {
        case 0: 
            newFileCbPtr = stdin;
            break;
        case 1: 
            newFileCbPtr = stdout;
            break;
        case 2: 
            newFileCbPtr = stderr;
            break;
    }

    /*
     * If the specified id is not open, set up a stdio file descriptor.
     */
    TclMakeFileTable (iPtr, newFileId);
    if (iPtr->filePtrArray [newFileId] == NULL) {
        char *mode;

        /*
         * Set up a stdio FILE control block for the new file.
         */
        if (oldFilePtr->readable && oldFilePtr->writable) {
            mode = "r+";
        } else if (oldFilePtr->writable) {
            mode = "w";
        } else {
            mode = "r";
        }
        if (freopen ("/dev/null", mode, newFileCbPtr) == NULL)
            goto unixError;
    }
    
    /*
     * This functionallity may be obtained with dup2 on most systems.  Being
     * open is optional.
     */
    close (newFileId);
    if (fcntl (fileno (oldFilePtr->f), F_DUPFD, newFileId) < 0)
        goto unixError;

    return newFileCbPtr;

unixError:
    Tcl_AppendResult ((Tcl_Interp *) iPtr, tclCommand, ": ", 
                      Tcl_UnixError ((Tcl_Interp *) iPtr), (char *) NULL);
    return NULL;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_DupCmd --
 *     Implements the dup TCL command:
 *         dup filehandle [stdhandle]
 *
 * Results:
 *      Returns TCL_OK and interp->result containing a filehandle
 *      if the requested file or pipe was successfully duplicated.
 *
 *      Return TCL_ERROR and interp->result containing an
 *      explanation of what went wrong if an error occured.
 *
 * Side effects:
 *      Locates and creates an entry in the handles table
 *
 *----------------------------------------------------------------------
 */
int
Tcl_DupCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Interp   *iPtr = (Interp *) interp;
    OpenFile *oldFilePtr;
    FILE     *newFileCbPtr;
    OpenFile *newFilePtr;
    long      seekOffset = -1;

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # arg: ", argv[0], 
                          " filehandle [stdhandle]", (char *) NULL);
        return TCL_ERROR;
    }

    if (TclGetOpenFile(interp, argv[1], &oldFilePtr) != TCL_OK)
	return TCL_ERROR;
    if (oldFilePtr->numPids > 0) { /*??????*/
        Tcl_AppendResult (interp, "can not `dup' a pipeline", (char *) NULL);
        return TCL_ERROR;
    }

    /*
     * If writable, flush out the buffer.  If readable, remember were we are
     * so the we can set it up for the next stdio read to come from the same
     * place.  The location is only recorded if the file is a reqular file,
     * since you cann't seek on other types of files.
     */
    if (oldFilePtr->writable) {
        if (fflush (oldFilePtr->f) != 0)
            goto unixError;
    }
    if (oldFilePtr->readable) {
        struct stat statBuf;
        
        if (fstat (fileno (oldFilePtr->f), &statBuf) < 0)
            goto unixError;
        if ((statBuf.st_mode & S_IFMT) == S_IFREG) {
            seekOffset = ftell (oldFilePtr->f);
            if (seekOffset < 0)
                goto unixError;
        }
    }
    /*
     * Process the dup depending on if dup-ing to a new file or a target
     * file handle.
     */
    if (argc == 2)
        newFileCbPtr = DoNormalDup (iPtr, argv [0], oldFilePtr);
    else
        newFileCbPtr = DoSpecialDup (iPtr, argv [0], oldFilePtr, argv [2]);

    if (newFileCbPtr == NULL)
        return TCL_ERROR;

    /*
     * Set up a Tcl OpenFile structure for the new file handle.
     */
    newFilePtr = iPtr->filePtrArray [fileno (newFileCbPtr)];
    if (newFilePtr == NULL) {
        newFilePtr = (OpenFile*) ckalloc (sizeof (OpenFile));
        iPtr->filePtrArray [fileno (newFileCbPtr)] = newFilePtr;
    }
    newFilePtr->f = newFileCbPtr;
    newFilePtr->f2 = NULL;
    newFilePtr->readable = oldFilePtr->readable;
    newFilePtr->writable = oldFilePtr->writable;
    newFilePtr->numPids = 0;
    newFilePtr->pidPtr = NULL;
    newFilePtr->errorId = -1;

    if (seekOffset >= 0) {
        if (fseek (newFilePtr->f, seekOffset, SEEK_SET) != 0)
            goto unixError;
    }
    return TCL_OK;

unixError:
    Tcl_AppendResult (interp, argv [0], ": ", Tcl_UnixError (interp), 
                      (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_PipeCmd --
 *     Implements the pipe TCL command:
 *         pipe [handle_var_r handle_var_w]
 *
 * Results:
 *      Standard TCL result.
 *
 * Side effects:
 *      Locates and creates entries in the handles table
 *
 *----------------------------------------------------------------------
 */
int
Tcl_PipeCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Interp    *iPtr = (Interp *) interp;
    FILE      *file0CbPtr, *file1CbPtr;
    OpenFile  *file0Ptr,   *file1Ptr;
    int        fileIds [2];
    char       fHandle [12];

    if (!((argc == 1) || (argc == 3))) {
        Tcl_AppendResult (interp, "wrong # args: ", argv[0], 
                          " [handle_var_r handle_var_w]", (char*) NULL);
    }

    if (pipe (fileIds) < 0)
        goto unixError;

    if (((file0CbPtr = fdopen (fileIds[0], "r")) == NULL) ||
            ((file1CbPtr = fdopen (fileIds[1], "w")) == NULL)) {
        close (fileIds [0]);
        close (fileIds [1]);
        goto unixError;
    }

    TclMakeFileTable (iPtr,
                      (fileIds [0] > fileIds [1]) ? fileIds [0] : fileIds [1]);
    file0Ptr = (OpenFile*) ckalloc (sizeof (OpenFile));
    file0Ptr->f = file0CbPtr;
    file0Ptr->f2 = NULL;
    file0Ptr->readable = TRUE;
    file0Ptr->writable = FALSE;
    file0Ptr->numPids = 0;
    file0Ptr->pidPtr = NULL;
    file0Ptr->errorId = -1;
    iPtr->filePtrArray[fileIds [0]] = file0Ptr;

    file1Ptr = (OpenFile*) ckalloc (sizeof (OpenFile));
    file1Ptr->f = file1CbPtr;
    file1Ptr->f2 = NULL;
    file1Ptr->readable = FALSE;
    file1Ptr->writable = TRUE;
    file1Ptr->numPids = 0;
    file1Ptr->pidPtr = NULL;
    file1Ptr->errorId = -1;
    iPtr->filePtrArray[fileIds [1]] = file1Ptr;


    if (argc == 1)      
        sprintf (interp->result, "file%d file%d", fileIds [0], fileIds [1]);
    else {
        sprintf (fHandle, "file%d", fileIds [0]);
        if (Tcl_SetVar (interp, argv[1], fHandle, TCL_LEAVE_ERR_MSG) == NULL)
            return TCL_ERROR;

        sprintf (fHandle, "file%d", fileIds [1]);
        if (Tcl_SetVar (interp, argv[2], fHandle, TCL_LEAVE_ERR_MSG) == NULL)
            return TCL_ERROR;
    }
        
    return TCL_OK;

unixError:
    Tcl_AppendResult (interp, argv [0], ": ", Tcl_UnixError (interp), 
                      (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_CopyfileCmd --
 *     Implements the copyfile TCL command:
 *         copyfile handle1 handle2 [lines]
 *
 * Results:
 *      Nothing if it worked, else an error.
 *
 *----------------------------------------------------------------------
 */
int
Tcl_CopyfileCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    OpenFile  *fromFilePtr, *toFilePtr;
    char       transferBuffer [2048];
    int        bytesRead;

    if (argc != 3) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " fromfilehandle tofilehandle", (char *) NULL);
        return TCL_ERROR;
    }

    if (TclGetOpenFile (interp, argv[1], &fromFilePtr) != TCL_OK)
	return TCL_ERROR;
    if (TclGetOpenFile (interp, argv[2], &toFilePtr) != TCL_OK)
	return TCL_ERROR;

    if (!fromFilePtr->readable) {
        interp->result = "Source file is not open for read access";
	return TCL_ERROR;
    }
    if (!toFilePtr->writable) {
        interp->result = "Target file is not open for write access";
	return TCL_ERROR;
    }

    while (TRUE) {
        bytesRead = fread (transferBuffer, sizeof (char), 
                           sizeof (transferBuffer), fromFilePtr->f);
        if (bytesRead <= 0) {
            if (feof (fromFilePtr->f))
                break;
            else
                goto unixError;
        }
        if (fwrite (transferBuffer, sizeof (char), bytesRead, toFilePtr->f) != 
                    bytesRead)
            goto unixError;
    }

    return TCL_OK;

unixError:
    Tcl_AppendResult (interp, argv [0], ": ", Tcl_UnixError (interp), 
                      (char *) NULL);
    return TCL_ERROR;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_FstatCmd --
 *     Implements the fstat TCL command:
 *         fstat handle [arrayvar]
 *----------------------------------------------------------------------
 */
int
Tcl_FstatCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    OpenFile    *filePtr;
    struct stat  statBuf;

    if ((argc < 2) || (argc > 3)) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " handle [arrayVar]", (char *) NULL);
        return TCL_ERROR;
    }

    if (TclGetOpenFile (interp, argv[1], &filePtr) != TCL_OK)
	return TCL_ERROR;
    
    if (fstat (fileno (filePtr->f), &statBuf)) {
        Tcl_AppendResult (interp, argv [0], ": ", Tcl_UnixError (interp), 
                          (char *) NULL);
        return TCL_ERROR;
    }
    /*
     * Either return the arguments in an array or a list of keyword & value
     * elements.
     */
    if (argc == 2) {
        char statList [160];

        sprintf (statList, 
                 "{atime %d} {ctime %d} {dev %d} {gid %d} {ino %d} ",
                  statBuf.st_atime, statBuf.st_ctime, statBuf.st_dev,
                  statBuf.st_gid,   statBuf.st_ino);
        Tcl_AppendResult (interp, statList, (char *) NULL);

        sprintf (statList, 
                 "{mode %d} {mtime %d} {nlink %d} {size %d} {uid %d}",
                 statBuf.st_mode, statBuf.st_mtime, statBuf.st_nlink, 
                 statBuf.st_size, statBuf.st_uid);
        Tcl_AppendResult (interp, statList, (char *) NULL);
        
    } else {
        char numBuf [30];

	sprintf (numBuf, "%d", statBuf.st_dev);
	if  (Tcl_SetVar2 (interp, argv[2], "dev", numBuf, 
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_ino);
	if  (Tcl_SetVar2 (interp, argv[2], "ino", numBuf,
                             TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_mode);
	if  (Tcl_SetVar2 (interp, argv[2], "mode", numBuf, 
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_nlink);
	if  (Tcl_SetVar2 (interp, argv[2], "nlink", numBuf,
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_uid);
	if  (Tcl_SetVar2 (interp, argv[2], "uid", numBuf,
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_gid);
	if  (Tcl_SetVar2 (interp, argv[2], "gid", numBuf,
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_size);
	if  (Tcl_SetVar2 (interp, argv[2], "size", numBuf,
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_atime);
	if  (Tcl_SetVar2 (interp, argv[2], "atime", numBuf,
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_mtime);
	if  (Tcl_SetVar2 (interp, argv[2], "mtime", numBuf,
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

	sprintf (numBuf, "%d", statBuf.st_ctime);
	if  (Tcl_SetVar2 (interp, argv[2], "ctime", numBuf,
                          TCL_LEAVE_ERR_MSG) == NULL)
	    return TCL_ERROR;

    }
    return TCL_OK;

}

/*
 *----------------------------------------------------------------------
 *
 * GetFcntlFlags --
 *    Return the fcntl values as a symbolic list in the result.
 * Result:
 *   Returns TCL_OK if all is well, TCL_ERROR if fcntl returns an error.
 *----------------------------------------------------------------------
 */
static int
GetFcntlFlags (interp, cmdName, filePtr)
    Tcl_Interp *interp;
    char       *cmdName;
    OpenFile   *filePtr;
{
    int   flags;
    int   listArgc = 0;
    char *listArgv [9];

    flags = fcntl (fileno (filePtr->f), F_GETFL, 0);
    if (flags == -1)
        goto unixError;

    if (flags & O_RDONLY)
        listArgv [listArgc++] = "RDONLY";
    if (flags & O_WRONLY)
        listArgv [listArgc++] = "WRONLY";
    if (flags & O_RDWR)
        listArgv [listArgc++] = "RDWR";
    if (flags & O_NDELAY)
        listArgv [listArgc++] = "NDELAY";
    if (flags & O_APPEND)
        listArgv [listArgc++] = "APPEND";

    flags = fcntl (fileno (filePtr->f), F_GETFD, 0);
    if (flags == -1)
        goto unixError;
    if (flags & 1) 
        listArgv [listArgc++] = "CLEXEC";

    /*
     * Poke the stdio FILE structure to see if its buffered.
     */

#ifdef _IONBF
#ifndef TCL_386BSD
    if (filePtr->f->_flag & _IONBF)
        listArgv [listArgc++] = "NOBUF";
    if (filePtr->f->_flag & _IOLBF)
        listArgv [listArgc++] = "LINEBUF";
#endif
#else
    if (filePtr->f->_flags & _SNBF)
        listArgv [listArgc++] = "NOBUF";
    if (filePtr->f->_flags & _SLBF)
        listArgv [listArgc++] = "LINEBUF";
#endif

    Tcl_SetResult (interp, Tcl_Merge (listArgc, listArgv), TCL_DYNAMIC);
    return TCL_OK;

unixError:
    Tcl_AppendResult (interp, cmdName, ": ", Tcl_UnixError (interp), 
                      (char *) NULL);
    return TCL_ERROR;
   
}

/*
 *----------------------------------------------------------------------
 *
 * SetFcntlFlag --
 *    Set the specified fcntl flag to the given boolean value.
 * Result:
 *   Returns TCL_OK if all is well, TCL_ERROR if there is an error.
 *----------------------------------------------------------------------
 */
static int
SetFcntlFlag (interp, cmdName, flagName, valueStr, filePtr)
    Tcl_Interp *interp;
    char       *cmdName;
    char       *flagName;
    char       *valueStr;
    OpenFile   *filePtr;
{
#define   MAX_FLAG_NAME_LEN  12
#define   CLEXEC_FLAG   1
#define   NOBUF_FLAG    2
#define   LINEBUF_FLAG  4

    int   setFlag = 0, otherFlag = 0, setValue;
    char  flagNameUp [MAX_FLAG_NAME_LEN + 1];
 
    
    if (Tcl_GetBoolean (interp, valueStr, &setValue) != TCL_OK)
        return TCL_ERROR;

    if (strlen (flagName) > MAX_FLAG_NAME_LEN)
        goto invalidFlagName;
    Tcl_UpShift (flagNameUp, flagName);

    if (STREQU (flagNameUp, "NDELAY"))
        setFlag = O_NDELAY;
    else if (STREQU (flagNameUp, "APPEND"))
        setFlag = O_APPEND;
    else if (STREQU (flagNameUp, "CLEXEC"))
        otherFlag = CLEXEC_FLAG;
    else if (STREQU (flagNameUp, "NOBUF"))
        otherFlag = NOBUF_FLAG;
    else if (STREQU (flagNameUp, "LINEBUF"))
        otherFlag = LINEBUF_FLAG;
    else {
        Tcl_AppendResult (interp, "unknown attribute name \"", flagName,
                          "\", expected one of: APPEND, CLEXEC, NDELAY, ",
                          "NOBUF, LINEBUF", (char *) NULL);
        return TCL_ERROR;
    }

    if (otherFlag == CLEXEC_FLAG) {
        if (fcntl (fileno (filePtr->f), F_SETFD, setValue) == -1)
            goto unixError;
    } else if (otherFlag != 0) {
        if (setValue != 1) {
            Tcl_AppendResult (interp, flagNameUp, " flag may not be cleared",
                              (char *) NULL);
            return TCL_ERROR;
        }
        if (otherFlag == NOBUF_FLAG)
            setbuf (filePtr->f, NULL);
        else
            if (SET_LINE_BUF (filePtr->f) != 0) goto unixError;
    } else {
        int flags;

        flags = fcntl (fileno (filePtr->f), F_GETFL, 0);
        if (flags == -1)
            goto unixError;
        flags = flags & ~setFlag;
        if (setValue)
            flags = flags | setFlag;
        if (fcntl (fileno (filePtr->f), F_SETFL, flags) == -1)
            goto unixError;
    }
    return TCL_OK;

invalidFlagName:
    Tcl_AppendResult (interp, cmdName, ": invalid flag name \"", flagName,
                      "\", expected one of: NDELAY, APPEND, CLEXEC",
                      (char *) NULL);
    return TCL_ERROR;
unixError:
    Tcl_AppendResult (interp, cmdName, ": ", Tcl_UnixError (interp), 
                      (char *) NULL);
    return TCL_ERROR;
   
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_FcntlCmd --
 *     Implements the fcntl TCL command:
 *         fcntl handle [attribute value]
 *----------------------------------------------------------------------
 */
int
Tcl_FcntlCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    OpenFile    *filePtr;

    if (!((argc == 2) || (argc == 4))) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " handle [attribute value]", (char *) NULL);
        return TCL_ERROR;
    }

    if (TclGetOpenFile (interp, argv[1], &filePtr) != TCL_OK)
	return TCL_ERROR;
    if (argc == 2) {    
        if (GetFcntlFlags (interp, argv [0], filePtr) != TCL_OK)
            return TCL_ERROR;
    } else {
        if (SetFcntlFlag (interp, argv [0], argv [2], argv [3], 
                          filePtr) != TCL_OK)
            return TCL_ERROR;
    }
    return TCL_OK;
}
#ifndef TCL_NO_SELECT

/*
 *----------------------------------------------------------------------
 *
 * ParseSelectFileList --
 *
 *   Parse a list of file handles for select.
 *
 * Parameters:
 *   o interp (O) - Error messages are returned in the result.
 *   o handleList (I) - The list of file handles to parse, may be empty.
 *   o fileDescSetPtr (O) - The select fd_set for the parsed handles is
 *     filled in.  Should be cleared before this procedure is called.
 *   o fileDescListPtr (O) - A pointer to a dynamically allocated list of
 *     the integer file ids that are in the set.  If the list is empty,
 *     NULL is returned.
 *   o maxFileIdPtr (I/O) - If a file id greater than the current value is
 *     encountered, it will be set to that file id.
 * Returns:
 *   The number of files in the list, or -1 if an error occured.
 *----------------------------------------------------------------------
 */
static int
ParseSelectFileList (interp, handleList, fileDescSetPtr, fileDescListPtr,
                     maxFileIdPtr)
    Tcl_Interp *interp;
    char       *handleList;
    fd_set     *fileDescSetPtr;
    int       **fileDescListPtr;
    int        *maxFileIdPtr;
{
    int    handleCnt, idx;
    char **handleArgv;
    int   *fileDescList;

    if (Tcl_SplitList (interp, handleList, &handleCnt, &handleArgv) != TCL_OK)
        return -1;

    /*
     * Handle case of an empty list.
     */
    if (handleCnt == 0) {
        *fileDescListPtr = NULL;
        ckfree ((char *) handleArgv);
        return 0;
    }

    fileDescList = (int *) ckalloc (sizeof (int) * handleCnt);

    for (idx = 0; idx < handleCnt; idx++) {
        OpenFile *filePtr;
        int       fileId;

        if (TclGetOpenFile (interp, handleArgv [idx], &filePtr) != TCL_OK) {
            ckfree ((char *) handleArgv);
            ckfree ((char *) fileDescList);
            return -1;
        }
        fileId = fileno (filePtr->f);
        fileDescList [idx] = fileId;

        FD_SET (fileId, fileDescSetPtr);
        if (fileId > *maxFileIdPtr)
            *maxFileIdPtr = fileId;
    }

    *fileDescListPtr = fileDescList;
    ckfree ((char *) handleArgv);
    return handleCnt;
}

/*
 *----------------------------------------------------------------------
 *
 * ReturnSelectedFileList --
 *
 *   Take the resulting file descriptor sets from a select, and the
 *   list of file descritpors and build up a list of Tcl file handles.
 *
 * Parameters:
 *   o fileDescSetPtr (I) - The select fd_set.
 *   o fileDescCnt (I) - Number of descriptors in the list.
 *   o fileDescListPtr (I) - A pointer to a list of the integer file
 *     ids that are in the set.  If the list is empty,
 *     NULL is returned.
 * Returns:
 *   A dynamicly allocated list of file handles.  If the handles are empty,
 *   it still returns a NULL list to make clean up easy.
 *----------------------------------------------------------------------
 */
static char *
ReturnSelectedFileList (fileDescSetPtr, fileDescCnt, fileDescList) 
    fd_set     *fileDescSetPtr;
    int         fileDescCnt;
    int        *fileDescList;
{
    int    idx, handleCnt;
    char  *fileHandleList;
    char **fileHandleArgv;

    /*
     * Special case the empty list.
     */
    if (fileDescCnt == 0) {
        fileHandleList = ckalloc (1);
        fileHandleList [0] = '\0';
        return fileHandleList;
    }

    handleCnt = 0;
    fileHandleArgv = (char **) ckalloc (sizeof (char *) * fileDescCnt);

    for (idx = 0; idx < fileDescCnt; idx++) {
        if (FD_ISSET (fileDescList [idx], fileDescSetPtr)) {
            fileHandleArgv [handleCnt] = ckalloc (8);  /* fileNNN */
            sprintf (fileHandleArgv [handleCnt], "file%d", fileDescList [idx]);
            handleCnt++;
        }
    }

    fileHandleList = Tcl_Merge (handleCnt, fileHandleArgv);
    for (idx = 0; idx < handleCnt; idx++)
        ckfree ((char *) fileHandleArgv [idx]);
    ckfree ((char *) fileHandleArgv);

    return fileHandleList;
}

/*
 *----------------------------------------------------------------------
 *
 * Tcl_SelectCmd --
 *     Implements the select TCL command:
 *          select readhandles [writehandles] [excepthandles] [timeout]
 *
 * Results:
 *     A list in the form:
 *        {readhandles writehandles excepthandles}
 *     or {} it the timeout expired.
 *----------------------------------------------------------------------
 */
int
Tcl_SelectCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{

    fd_set readFdSet,            writeFdSet,            exceptFdSet;
    int    readDescCnt = 0,      writeDescCnt = 0,      exceptDescCnt = 0;
    int   *readDescList = NULL, *writeDescList = NULL, *exceptDescList = NULL;
    char  *retListArgv [3];

    int             numSelected, maxFileId = 0;
    int             result = TCL_ERROR;
    struct timeval  timeoutRec;
    struct timeval *timeoutRecPtr;


    if (argc < 2) {
        Tcl_AppendResult (interp, "wrong # args: ", argv [0], 
                          " readhandles [writehandles] [excepthandles]",
                          " [timeout]", (char *) NULL);
        return TCL_ERROR;
    }
    
    /*
     * Parse the file handles and set everything up for the select call.
     */
    FD_ZERO (&readFdSet);
    FD_ZERO (&writeFdSet);
    FD_ZERO (&exceptFdSet);
    readDescCnt = ParseSelectFileList (interp, argv [1], &readFdSet, 
                                       &readDescList, &maxFileId);
    if (readDescCnt < 0)
        goto exitPoint;
    if (argc > 2) {
        writeDescCnt = ParseSelectFileList (interp, argv [2], &writeFdSet, 
                                            &writeDescList, &maxFileId);
        if (writeDescCnt < 0)
            goto exitPoint;
    }
    if (argc > 3) {
        exceptDescCnt = ParseSelectFileList (interp, argv [3], &exceptFdSet, 
                                             &exceptDescList, &maxFileId);
        if (exceptDescCnt < 0)
            goto exitPoint;
    }
    
    /*
     * Get the time out.  Zero is different that not specified.
     */
    timeoutRecPtr = NULL;
    if ((argc > 4) && (argv [4][0] != '\0')) {
        double  timeout, seconds, microseconds;

        if (Tcl_GetDouble (interp, argv [4], &timeout) != TCL_OK)
            goto exitPoint;
        if (timeout < 0) {
            Tcl_AppendResult (interp, "timeout must be greater than or equal",
                              " to zero", (char *) NULL);
            goto exitPoint;
        }
        seconds = floor (timeout);
        microseconds = (timeout - seconds) * 1000000.0;
        timeoutRec.tv_sec = seconds;
        timeoutRec.tv_usec = microseconds;
        timeoutRecPtr = &timeoutRec;
    }

    /*
     * All set, do the select.
     */
    numSelected = select (maxFileId + 1, &readFdSet, &writeFdSet, &exceptFdSet,
                          timeoutRecPtr);
    if (numSelected < 0) {
        Tcl_AppendResult (interp, argv [0], ": system call error:", 
                          Tcl_UnixError (interp), (char *) NULL);
        goto exitPoint;
    }

    /*
     * Return the result, either a 3 element list, or leave the result
     * empty if the timeout occured.
     */
    if (numSelected > 0) {
        retListArgv [0] = ReturnSelectedFileList (&readFdSet, readDescCnt,
                                                  readDescList);
        retListArgv [1] = ReturnSelectedFileList (&writeFdSet, writeDescCnt,
                                                  writeDescList);
        retListArgv [2] = ReturnSelectedFileList (&exceptFdSet, exceptDescCnt,
                                                  exceptDescList);
        Tcl_SetResult (interp, Tcl_Merge (3, retListArgv), TCL_DYNAMIC); 
        ckfree ((char *) retListArgv [0]);
        ckfree ((char *) retListArgv [1]);
        ckfree ((char *) retListArgv [2]);
    }

    result = TCL_OK;

exitPoint:
    if (readDescList != NULL)
        ckfree ((char *) readDescList);
    if (writeDescList != NULL)
        ckfree ((char *) writeDescList);
    if (exceptDescList != NULL)
        ckfree ((char *) exceptDescList);
    return result;

unixError:
    return TCL_ERROR;
}
#else
/*
 *----------------------------------------------------------------------
 *
 * Tcl_SelectCmd --
 *     Dummy select command that returns an error for systems that don't
 *     have select.
 *----------------------------------------------------------------------
 */
int
Tcl_SelectCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    Tcl_AppendResult (interp, 
                      "select is not available on this version of Unix",
                      (char *) NULL);
    return TCL_ERROR;
}
#endif


/*
 *----------------------------------------------------------------------
 *
 * TtyName --
 *      Return the tty name corresponding to a file handle, or an 
 *      empty string if it's not a tty.
 *
 * Parameters:
 *   o searchCBPtr (I/O) - The search control block, if the line is found,
 *     it is returned in dynBuf.
 * Results:
 *     TCL_OK - If the key was found.
 *     TCL_BREAK - If it was not found.
 *     TCL_ERROR - If there was an error.
 *
 * based on getpath.c from smail 2.5 (9/15/87)
 *
 *----------------------------------------------------------------------
 */
int
Tcl_TtyNameCmd (clientData, interp, argc, argv)
    ClientData  clientData;
    Tcl_Interp *interp;
    int         argc;
    char      **argv;
{
    OpenFile *filePtr;
    char     *ttyName;

    if (argc != 2) {
        Tcl_AppendResult (interp, "wrong # arg: ", argv[0], 
                          " filehandle ", (char *) NULL);
        return TCL_ERROR;
    }

    if (TclGetOpenFile(interp, argv[1], &filePtr) != TCL_OK)
	return TCL_ERROR;

    ttyName = ttyname (fileno (filePtr->f));
    if (ttyName != NULL)
        Tcl_SetResult (interp, ttyName, TCL_VOLATILE);
    return TCL_OK;
}
