// SPDX-License-Identifier: GPL-2.0 OR BSD-3-Clause
/* Copyright(c) 2019-2020  Realtek Corporation
 */

#include "coex.h"
#include "debug.h"
#include "fw.h"
#include "mac.h"
#include "ps.h"
#include "reg.h"

#define RTW89_COEX_VERSION 0x06030013
#define FCXDEF_STEP 50 /* MUST <= FCXMAX_STEP and match with wl fw*/

enum btc_fbtc_tdma_template {
	CXTD_OFF = 0x0,
	CXTD_OFF_B2,
	CXTD_OFF_EXT,
	CXTD_FIX,
	CXTD_PFIX,
	CXTD_AUTO,
	CXTD_PAUTO,
	CXTD_AUTO2,
	CXTD_PAUTO2,
	CXTD_MAX,
};

enum btc_fbtc_tdma_type {
	CXTDMA_OFF = 0x0,
	CXTDMA_FIX = 0x1,
	CXTDMA_AUTO = 0x2,
	CXTDMA_AUTO2 = 0x3,
	CXTDMA_MAX
};

enum btc_fbtc_tdma_rx_flow_ctrl {
	CXFLC_OFF = 0x0,
	CXFLC_NULLP = 0x1,
	CXFLC_QOSNULL = 0x2,
	CXFLC_CTS = 0x3,
	CXFLC_MAX
};

enum btc_fbtc_tdma_wlan_tx_pause {
	CXTPS_OFF = 0x0,  /* no wl tx pause*/
	CXTPS_ON = 0x1,
	CXTPS_MAX
};

enum btc_mlme_state {
	MLME_NO_LINK,
	MLME_LINKING,
	MLME_LINKED,
};

#define FCXONESLOT_VER 1
struct btc_fbtc_1slot {
	u8 fver;
	u8 sid; /* slot id */
	struct rtw89_btc_fbtc_slot slot;
} __packed;

static const struct rtw89_btc_fbtc_tdma t_def[] = {
	[CXTD_OFF]	= { CXTDMA_OFF,    CXFLC_OFF, CXTPS_OFF, 0, 0, 0, 0, 0},
	[CXTD_OFF_B2]	= { CXTDMA_OFF,    CXFLC_OFF, CXTPS_OFF, 0, 0, 1, 0, 0},
	[CXTD_OFF_EXT]	= { CXTDMA_OFF,    CXFLC_OFF, CXTPS_OFF, 0, 0, 3, 0, 0},
	[CXTD_FIX]	= { CXTDMA_FIX,    CXFLC_OFF, CXTPS_OFF, 0, 0, 0, 0, 0},
	[CXTD_PFIX]	= { CXTDMA_FIX,  CXFLC_NULLP,  CXTPS_ON, 0, 5, 0, 0, 0},
	[CXTD_AUTO]	= { CXTDMA_AUTO,   CXFLC_OFF, CXTPS_OFF, 0, 0, 0, 0, 0},
	[CXTD_PAUTO]	= { CXTDMA_AUTO, CXFLC_NULLP,  CXTPS_ON, 0, 5, 0, 0, 0},
	[CXTD_AUTO2]	= {CXTDMA_AUTO2,   CXFLC_OFF, CXTPS_OFF, 0, 0, 0, 0, 0},
	[CXTD_PAUTO2]	= {CXTDMA_AUTO2, CXFLC_NULLP,  CXTPS_ON, 0, 5, 0, 0, 0}
};

#define __DEF_FBTC_SLOT(__dur, __cxtbl, __cxtype) \
	{ .dur = cpu_to_le16(__dur), .cxtbl = cpu_to_le32(__cxtbl), \
	  .cxtype = cpu_to_le16(__cxtype),}

static const struct rtw89_btc_fbtc_slot s_def[] = {
	[CXST_OFF]	= __DEF_FBTC_SLOT(100, 0x55555555, SLOT_MIX),
	[CXST_B2W]	= __DEF_FBTC_SLOT(5,   0xea5a5a5a, SLOT_ISO),
	[CXST_W1]	= __DEF_FBTC_SLOT(70,  0xea5a5a5a, SLOT_ISO),
	[CXST_W2]	= __DEF_FBTC_SLOT(70,  0xea5a5aaa, SLOT_ISO),
	[CXST_W2B]	= __DEF_FBTC_SLOT(15,  0xea5a5a5a, SLOT_ISO),
	[CXST_B1]	= __DEF_FBTC_SLOT(100, 0xe5555555, SLOT_MIX),
	[CXST_B2]	= __DEF_FBTC_SLOT(7,   0xea5a5a5a, SLOT_MIX),
	[CXST_B3]	= __DEF_FBTC_SLOT(5,   0xe5555555, SLOT_MIX),
	[CXST_B4]	= __DEF_FBTC_SLOT(50,  0xe5555555, SLOT_MIX),
	[CXST_LK]	= __DEF_FBTC_SLOT(20,  0xea5a5a5a, SLOT_ISO),
	[CXST_BLK]	= __DEF_FBTC_SLOT(250, 0x55555555, SLOT_MIX),
	[CXST_E2G]	= __DEF_FBTC_SLOT(20,  0xea5a5a5a, SLOT_MIX),
	[CXST_E5G]	= __DEF_FBTC_SLOT(20,  0xffffffff, SLOT_MIX),
	[CXST_EBT]	= __DEF_FBTC_SLOT(20,  0xe5555555, SLOT_MIX),
	[CXST_ENULL]	= __DEF_FBTC_SLOT(7,   0xaaaaaaaa, SLOT_ISO),
	[CXST_WLK]	= __DEF_FBTC_SLOT(250, 0xea5a5a5a, SLOT_MIX),
	[CXST_W1FDD]	= __DEF_FBTC_SLOT(35,  0xfafafafa, SLOT_ISO),
	[CXST_B1FDD]	= __DEF_FBTC_SLOT(100, 0xffffffff, SLOT_MIX),
};

static const u32 cxtbl[] = {
	0xffffffff, /* 0 */
	0xaaaaaaaa, /* 1 */
	0xe5555555, /* 2 */
	0xee555555, /* 3 */
	0xd5555555, /* 4 */
	0x5a5a5a5a, /* 5 */
	0xfa5a5a5a, /* 6 */
	0xda5a5a5a, /* 7 */
	0xea5a5a5a, /* 8 */
	0x6a5a5aaa, /* 9 */
	0x6a5a6a5a, /* 10 */
	0x6a5a6aaa, /* 11 */
	0x6afa5afa, /* 12 */
	0xaaaa5aaa, /* 13 */
	0xaaffffaa, /* 14 */
	0xaa5555aa, /* 15 */
	0xfafafafa, /* 16 */
	0xffffddff, /* 17 */
	0xdaffdaff, /* 18 */
	0xfafadafa  /* 19 */
};

struct rtw89_btc_btf_tlv {
	u8 type;
	u8 len;
	u8 val[1];
} __packed;

enum btc_btf_set_report_en {
	RPT_EN_TDMA = BIT(0),
	RPT_EN_CYCLE = BIT(1),
	RPT_EN_MREG = BIT(2),
	RPT_EN_BT_VER_INFO = BIT(3),
	RPT_EN_BT_SCAN_INFO = BIT(4),
	RPT_EN_BT_AFH_MAP = BIT(5),
	RPT_EN_BT_DEVICE_INFO = BIT(6),
	RPT_EN_WL_ALL = GENMASK(2, 0),
	RPT_EN_BT_ALL = GENMASK(6, 3),
	RPT_EN_ALL = GENMASK(6, 0),
};

#define BTF_SET_REPORT_VER 1
struct rtw89_btc_btf_set_report {
	u8 fver;
	__le32 enable;
	__le32 para;
} __packed;

#define BTF_SET_SLOT_TABLE_VER 1
struct rtw89_btc_btf_set_slot_table {
	u8 fver;
	u8 tbl_num;
	u8 buf[];
} __packed;

#define BTF_SET_MON_REG_VER 1
struct rtw89_btc_btf_set_mon_reg {
	u8 fver;
	u8 reg_num;
	u8 buf[];
} __packed;

enum btc_btf_set_cx_policy {
	CXPOLICY_TDMA = 0x0,
	CXPOLICY_SLOT = 0x1,
	CXPOLICY_TYPE = 0x2,
	CXPOLICY_MAX,
};

enum btc_b2w_scoreboard {
	BTC_BSCB_ACT = BIT(0),
	BTC_BSCB_ON = BIT(1),
	BTC_BSCB_WHQL = BIT(2),
	BTC_BSCB_BT_S1 = BIT(3),
	BTC_BSCB_A2DP_ACT = BIT(4),
	BTC_BSCB_RFK_RUN = BIT(5),
	BTC_BSCB_RFK_REQ = BIT(6),
	BTC_BSCB_LPS = BIT(7),
	BTC_BSCB_WLRFK = BIT(11),
	BTC_BSCB_BT_HILNA = BIT(13),
	BTC_BSCB_BT_CONNECT = BIT(16),
	BTC_BSCB_PATCH_CODE = BIT(30),
	BTC_BSCB_ALL = GENMASK(30, 0),
};

enum btc_phymap {
	BTC_PHY_0 = BIT(0),
	BTC_PHY_1 = BIT(1),
	BTC_PHY_ALL = BIT(0) | BIT(1),
};

enum btc_cx_state_map {
	BTC_WIDLE = 0,
	BTC_WBUSY_BNOSCAN,
	BTC_WBUSY_BSCAN,
	BTC_WSCAN_BNOSCAN,
	BTC_WSCAN_BSCAN,
	BTC_WLINKING
};

enum btc_ant_phase {
	BTC_ANT_WPOWERON = 0,
	BTC_ANT_WINIT,
	BTC_ANT_WONLY,
	BTC_ANT_WOFF,
	BTC_ANT_W2G,
	BTC_ANT_W5G,
	BTC_ANT_W25G,
	BTC_ANT_FREERUN,
	BTC_ANT_WRFK,
	BTC_ANT_BRFK,
	BTC_ANT_MAX
};

enum btc_plt {
	BTC_PLT_NONE = 0,
	BTC_PLT_LTE_RX = BIT(0),
	BTC_PLT_GNT_BT_TX = BIT(1),
	BTC_PLT_GNT_BT_RX = BIT(2),
	BTC_PLT_GNT_WL = BIT(3),
	BTC_PLT_BT = BIT(1) | BIT(2),
	BTC_PLT_ALL = 0xf
};

enum btc_cx_poicy_main_type {
	BTC_CXP_OFF = 0,
	BTC_CXP_OFFB,
	BTC_CXP_OFFE,
	BTC_CXP_FIX,
	BTC_CXP_PFIX,
	BTC_CXP_AUTO,
	BTC_CXP_PAUTO,
	BTC_CXP_AUTO2,
	BTC_CXP_PAUTO2,
	BTC_CXP_MANUAL,
	BTC_CXP_USERDEF0,
	BTC_CXP_MAIN_MAX
};

enum btc_cx_poicy_type {
	/* TDMA off + pri: BT > WL */
	BTC_CXP_OFF_BT = (BTC_CXP_OFF << 8) | 0,

	/* TDMA off + pri: WL > BT */
	BTC_CXP_OFF_WL = (BTC_CXP_OFF << 8) | 1,

	/* TDMA off + pri: BT = WL */
	BTC_CXP_OFF_EQ0 = (BTC_CXP_OFF << 8) | 2,

	/* TDMA off + pri: BT = WL > BT_Lo */
	BTC_CXP_OFF_EQ1 = (BTC_CXP_OFF << 8) | 3,

	/* TDMA off + pri: WL = BT, BT_Rx > WL_Lo_Tx */
	BTC_CXP_OFF_EQ2 = (BTC_CXP_OFF << 8) | 4,

	/* TDMA off + pri: WL_Rx = BT, BT_HI > WL_Tx > BT_Lo */
	BTC_CXP_OFF_EQ3 = (BTC_CXP_OFF << 8) | 5,

	/* TDMA off + pri: BT_Hi > WL > BT_Lo */
	BTC_CXP_OFF_BWB0 = (BTC_CXP_OFF << 8) | 6,

	/* TDMA off + pri: WL_Hi-Tx > BT_Hi_Rx, BT_Hi > WL > BT_Lo */
	BTC_CXP_OFF_BWB1 = (BTC_CXP_OFF << 8) | 7,

	/* TDMA off + pri: WL_Hi-Tx > BT, BT_Hi > other-WL > BT_Lo */
	BTC_CXP_OFF_BWB2 = (BTC_CXP_OFF << 8) | 8,

	/* TDMA off + pri: WL_Hi-Tx = BT */
	BTC_CXP_OFF_BWB3 = (BTC_CXP_OFF << 8) | 9,

	/* TDMA off+Bcn-Protect + pri: WL_Hi-Tx > BT_Hi_Rx, BT_Hi > WL > BT_Lo*/
	BTC_CXP_OFFB_BWB0 = (BTC_CXP_OFFB << 8) | 0,

	/* TDMA off + Ext-Ctrl + pri: default */
	BTC_CXP_OFFE_DEF = (BTC_CXP_OFFE << 8) | 0,

	/* TDMA off + Ext-Ctrl + pri: E2G-slot block all BT */
	BTC_CXP_OFFE_DEF2 = (BTC_CXP_OFFE << 8) | 1,

	/* TDMA off + Ext-Ctrl + pri: default */
	BTC_CXP_OFFE_2GBWISOB = (BTC_CXP_OFFE << 8) | 2,

	/* TDMA off + Ext-Ctrl + pri: E2G-slot block all BT */
	BTC_CXP_OFFE_2GISOB = (BTC_CXP_OFFE << 8) | 3,

	/* TDMA off + Ext-Ctrl + pri: E2G-slot WL > BT */
	BTC_CXP_OFFE_2GBWMIXB = (BTC_CXP_OFFE << 8) | 4,

	/* TDMA off + Ext-Ctrl + pri: E2G/EBT-slot WL > BT */
	BTC_CXP_OFFE_WL = (BTC_CXP_OFFE << 8) | 5,

	/* TDMA off + Ext-Ctrl + pri: default */
	BTC_CXP_OFFE_2GBWMIXB2 = (BTC_CXP_OFFE << 8) | 6,

	/* TDMA Fix slot-0: W1:B1 = 30:30 */
	BTC_CXP_FIX_TD3030 = (BTC_CXP_FIX << 8) | 0,

	/* TDMA Fix slot-1: W1:B1 = 50:50 */
	BTC_CXP_FIX_TD5050 = (BTC_CXP_FIX << 8) | 1,

	/* TDMA Fix slot-2: W1:B1 = 20:30 */
	BTC_CXP_FIX_TD2030 = (BTC_CXP_FIX << 8) | 2,

	/* TDMA Fix slot-3: W1:B1 = 40:10 */
	BTC_CXP_FIX_TD4010 = (BTC_CXP_FIX << 8) | 3,

	/* TDMA Fix slot-4: W1:B1 = 70:10 */
	BTC_CXP_FIX_TD7010 = (BTC_CXP_FIX << 8) | 4,

	/* TDMA Fix slot-5: W1:B1 = 20:60 */
	BTC_CXP_FIX_TD2060 = (BTC_CXP_FIX << 8) | 5,

	/* TDMA Fix slot-6: W1:B1 = 30:60 */
	BTC_CXP_FIX_TD3060 = (BTC_CXP_FIX << 8) | 6,

	/* TDMA Fix slot-7: W1:B1 = 20:80 */
	BTC_CXP_FIX_TD2080 = (BTC_CXP_FIX << 8) | 7,

	/* TDMA Fix slot-8: W1:B1 = user-define */
	BTC_CXP_FIX_TDW1B1 = (BTC_CXP_FIX << 8) | 8,

	/* TDMA Fix slot-9: W1:B1 = 40:20 */
	BTC_CXP_FIX_TD4020 = (BTC_CXP_FIX << 8) | 9,

	/* TDMA Fix slot-9: W1:B1 = 40:10 */
	BTC_CXP_FIX_TD4010ISO = (BTC_CXP_FIX << 8) | 10,

	/* PS-TDMA Fix slot-0: W1:B1 = 30:30 */
	BTC_CXP_PFIX_TD3030 = (BTC_CXP_PFIX << 8) | 0,

	/* PS-TDMA Fix slot-1: W1:B1 = 50:50 */
	BTC_CXP_PFIX_TD5050 = (BTC_CXP_PFIX << 8) | 1,

	/* PS-TDMA Fix slot-2: W1:B1 = 20:30 */
	BTC_CXP_PFIX_TD2030 = (BTC_CXP_PFIX << 8) | 2,

	/* PS-TDMA Fix slot-3: W1:B1 = 20:60 */
	BTC_CXP_PFIX_TD2060 = (BTC_CXP_PFIX << 8) | 3,

	/* PS-TDMA Fix slot-4: W1:B1 = 30:70 */
	BTC_CXP_PFIX_TD3070 = (BTC_CXP_PFIX << 8) | 4,

	/* PS-TDMA Fix slot-5: W1:B1 = 20:80 */
	BTC_CXP_PFIX_TD2080 = (BTC_CXP_PFIX << 8) | 5,

	/* PS-TDMA Fix slot-6: W1:B1 = user-define */
	BTC_CXP_PFIX_TDW1B1 = (BTC_CXP_PFIX << 8) | 6,

	/* TDMA Auto slot-0: W1:B1 = 50:200 */
	BTC_CXP_AUTO_TD50B1 = (BTC_CXP_AUTO << 8) | 0,

	/* TDMA Auto slot-1: W1:B1 = 60:200 */
	BTC_CXP_AUTO_TD60B1 = (BTC_CXP_AUTO << 8) | 1,

	/* TDMA Auto slot-2: W1:B1 = 20:200 */
	BTC_CXP_AUTO_TD20B1 = (BTC_CXP_AUTO << 8) | 2,

	/* TDMA Auto slot-3: W1:B1 = user-define */
	BTC_CXP_AUTO_TDW1B1 = (BTC_CXP_AUTO << 8) | 3,

	/* PS-TDMA Auto slot-0: W1:B1 = 50:200 */
	BTC_CXP_PAUTO_TD50B1 = (BTC_CXP_PAUTO << 8) | 0,

	/* PS-TDMA Auto slot-1: W1:B1 = 60:200 */
	BTC_CXP_PAUTO_TD60B1 = (BTC_CXP_PAUTO << 8) | 1,

	/* PS-TDMA Auto slot-2: W1:B1 = 20:200 */
	BTC_CXP_PAUTO_TD20B1 = (BTC_CXP_PAUTO << 8) | 2,

	/* PS-TDMA Auto slot-3: W1:B1 = user-define */
	BTC_CXP_PAUTO_TDW1B1 = (BTC_CXP_PAUTO << 8) | 3,

	/* TDMA Auto slot2-0: W1:B4 = 30:50 */
	BTC_CXP_AUTO2_TD3050 = (BTC_CXP_AUTO2 << 8) | 0,

	/* TDMA Auto slot2-1: W1:B4 = 30:70 */
	BTC_CXP_AUTO2_TD3070 = (BTC_CXP_AUTO2 << 8) | 1,

	/* TDMA Auto slot2-2: W1:B4 = 50:50 */
	BTC_CXP_AUTO2_TD5050 = (BTC_CXP_AUTO2 << 8) | 2,

	/* TDMA Auto slot2-3: W1:B4 = 60:60 */
	BTC_CXP_AUTO2_TD6060 = (BTC_CXP_AUTO2 << 8) | 3,

	/* TDMA Auto slot2-4: W1:B4 = 20:80 */
	BTC_CXP_AUTO2_TD2080 = (BTC_CXP_AUTO2 << 8) | 4,

	/* TDMA Auto slot2-5: W1:B4 = user-define */
	BTC_CXP_AUTO2_TDW1B4 = (BTC_CXP_AUTO2 << 8) | 5,

	/* PS-TDMA Auto slot2-0: W1:B4 = 30:50 */
	BTC_CXP_PAUTO2_TD3050 = (BTC_CXP_PAUTO2 << 8) | 0,

	/* PS-TDMA Auto slot2-1: W1:B4 = 30:70 */
	BTC_CXP_PAUTO2_TD3070 = (BTC_CXP_PAUTO2 << 8) | 1,

	/* PS-TDMA Auto slot2-2: W1:B4 = 50:50 */
	BTC_CXP_PAUTO2_TD5050 = (BTC_CXP_PAUTO2 << 8) | 2,

	/* PS-TDMA Auto slot2-3: W1:B4 = 60:60 */
	BTC_CXP_PAUTO2_TD6060 = (BTC_CXP_PAUTO2 << 8) | 3,

	/* PS-TDMA Auto slot2-4: W1:B4 = 20:80 */
	BTC_CXP_PAUTO2_TD2080 = (BTC_CXP_PAUTO2 << 8) | 4,

	/* PS-TDMA Auto slot2-5: W1:B4 = user-define */
	BTC_CXP_PAUTO2_TDW1B4 = (BTC_CXP_PAUTO2 << 8) | 5,

	BTC_CXP_MAX = 0xffff
};

enum btc_wl_rfk_result {
	BTC_WRFK_REJECT = 0,
	BTC_WRFK_ALLOW = 1,
};

enum btc_coex_info_map_en {
	BTC_COEX_INFO_CX = BIT(0),
	BTC_COEX_INFO_WL = BIT(1),
	BTC_COEX_INFO_BT = BIT(2),
	BTC_COEX_INFO_DM = BIT(3),
	BTC_COEX_INFO_MREG = BIT(4),
	BTC_COEX_INFO_SUMMARY = BIT(5),
	BTC_COEX_INFO_ALL = GENMASK(7, 0),
};

#define BTC_CXP_MASK GENMASK(15, 8)

enum btc_w2b_scoreboard {
	BTC_WSCB_ACTIVE = BIT(0),
	BTC_WSCB_ON = BIT(1),
	BTC_WSCB_SCAN = BIT(2),
	BTC_WSCB_UNDERTEST = BIT(3),
	BTC_WSCB_RXGAIN = BIT(4),
	BTC_WSCB_WLBUSY = BIT(7),
	BTC_WSCB_EXTFEM = BIT(8),
	BTC_WSCB_TDMA = BIT(9),
	BTC_WSCB_FIX2M = BIT(10),
	BTC_WSCB_WLRFK = BIT(11),
	BTC_WSCB_RXSCAN_PRI = BIT(12),
	BTC_WSCB_BT_HILNA = BIT(13),
	BTC_WSCB_BTLOG = BIT(14),
	BTC_WSCB_ALL = GENMASK(23, 0),
};

enum btc_wl_link_mode {
	BTC_WLINK_NOLINK = 0x0,
	BTC_WLINK_2G_STA,
	BTC_WLINK_2G_AP,
	BTC_WLINK_2G_GO,
	BTC_WLINK_2G_GC,
	BTC_WLINK_2G_SCC,
	BTC_WLINK_2G_MCC,
	BTC_WLINK_25G_MCC,
	BTC_WLINK_25G_DBCC,
	BTC_WLINK_5G,
	BTC_WLINK_2G_NAN,
	BTC_WLINK_OTHER,
	BTC_WLINK_MAX
};

enum btc_wl_mrole_type {
	BTC_WLMROLE_NONE = 0x0,
	BTC_WLMROLE_STA_GC,
	BTC_WLMROLE_STA_GC_NOA,
	BTC_WLMROLE_STA_GO,
	BTC_WLMROLE_STA_GO_NOA,
	BTC_WLMROLE_STA_STA,
	BTC_WLMROLE_MAX
};

enum btc_bt_hid_type {
	BTC_HID_218 = BIT(0),
	BTC_HID_418 = BIT(1),
	BTC_HID_BLE = BIT(2),
	BTC_HID_RCU = BIT(3),
	BTC_HID_RCU_VOICE = BIT(4),
	BTC_HID_OTHER_LEGACY = BIT(5)
};

enum btc_reset_module {
	BTC_RESET_CX = BIT(0),
	BTC_RESET_DM = BIT(1),
	BTC_RESET_CTRL = BIT(2),
	BTC_RESET_CXDM = BIT(0) | BIT(1),
	BTC_RESET_BTINFO = BIT(3),
	BTC_RESET_MDINFO = BIT(4),
	BTC_RESET_ALL =  GENMASK(7, 0),
};

enum btc_gnt_state {
	BTC_GNT_HW	= 0,
	BTC_GNT_SW_LO,
	BTC_GNT_SW_HI,
	BTC_GNT_MAX
};

enum btc_ctr_path {
	BTC_CTRL_BY_BT = 0,
	BTC_CTRL_BY_WL
};

enum btc_wl_max_tx_time {
	BTC_MAX_TX_TIME_L1 = 500,
	BTC_MAX_TX_TIME_L2 = 1000,
	BTC_MAX_TX_TIME_L3 = 2000,
	BTC_MAX_TX_TIME_DEF = 5280
};

enum btc_wl_max_tx_retry {
	BTC_MAX_TX_RETRY_L1 = 7,
	BTC_MAX_TX_RETRY_L2 = 15,
	BTC_MAX_TX_RETRY_DEF = 31,
};

enum btc_reason_and_action {
	BTC_RSN_NONE,
	BTC_RSN_NTFY_INIT,
	BTC_RSN_NTFY_SWBAND,
	BTC_RSN_NTFY_WL_STA,
	BTC_RSN_NTFY_RADIO_STATE,
	BTC_RSN_UPDATE_BT_SCBD,
	BTC_RSN_NTFY_WL_RFK,
	BTC_RSN_UPDATE_BT_INFO,
	BTC_RSN_NTFY_SCAN_START,
	BTC_RSN_NTFY_SCAN_FINISH,
	BTC_RSN_NTFY_SPECIFIC_PACKET,
	BTC_RSN_NTFY_POWEROFF,
	BTC_RSN_NTFY_ROLE_INFO,
	BTC_RSN_CMD_SET_COEX,
	BTC_RSN_ACT1_WORK,
	BTC_RSN_BT_DEVINFO_WORK,
	BTC_RSN_RFK_CHK_WORK,
	BTC_RSN_NUM,
	BTC_ACT_NONE = 100,
	BTC_ACT_WL_ONLY,
	BTC_ACT_WL_5G,
	BTC_ACT_WL_OTHER,
	BTC_ACT_WL_IDLE,
	BTC_ACT_WL_NC,
	BTC_ACT_WL_RFK,
	BTC_ACT_WL_INIT,
	BTC_ACT_WL_OFF,
	BTC_ACT_FREERUN,
	BTC_ACT_BT_WHQL,
	BTC_ACT_BT_RFK,
	BTC_ACT_BT_OFF,
	BTC_ACT_BT_IDLE,
	BTC_ACT_BT_HFP,
	BTC_ACT_BT_HID,
	BTC_ACT_BT_A2DP,
	BTC_ACT_BT_A2DPSINK,
	BTC_ACT_BT_PAN,
	BTC_ACT_BT_A2DP_HID,
	BTC_ACT_BT_A2DP_PAN,
	BTC_ACT_BT_PAN_HID,
	BTC_ACT_BT_A2DP_PAN_HID,
	BTC_ACT_WL_25G_MCC,
	BTC_ACT_WL_2G_MCC,
	BTC_ACT_WL_2G_SCC,
	BTC_ACT_WL_2G_AP,
	BTC_ACT_WL_2G_GO,
	BTC_ACT_WL_2G_GC,
	BTC_ACT_WL_2G_NAN,
	BTC_ACT_LAST,
	BTC_ACT_NUM = BTC_ACT_LAST - BTC_ACT_NONE,
	BTC_ACT_EXT_BIT = BIT(14),
	BTC_POLICY_EXT_BIT = BIT(15),
};

#define BTC_FREERUN_ANTISO_MIN 30
#define BTC_TDMA_BTHID_MAX 2
#define BTC_BLINK_NOCONNECT 0
#define BTC_B1_MAX 250 /* unit ms */

static void _run_coex(struct rtw89_dev *rtwdev,
		      enum btc_reason_and_action reason);
static void _write_scbd(struct rtw89_dev *rtwdev, u32 val, bool state);
static void _update_bt_scbd(struct rtw89_dev *rtwdev, bool only_update);

static void _send_fw_cmd(struct rtw89_dev *rtwdev, u8 h2c_class, u8 h2c_func,
			 void *param, u16 len)
{
	struct rtw89_btc *btc = &rtwdev->btc;
	struct rtw89_btc_btf_fwinfo *pfwinfo = &btc->fwinfo;
	struct rtw89_btc_cx *cx = &btc->cx;
	struct rtw89_btc_wl_info *wl = &cx->wl;
	int ret;

	if (!wl->status.map.init_ok) {
		rtw89_debug(rtwdev, RTW89_DBG_BTC,
			    "[BTC], %s(): return by btc not init!!\n", __func__);
		pfwinfo->cnt_h2c_fail++;
		return;
	} else if ((wl->status.map.rf_off_pre == BTC_LPS_RF_OFF &&
		    wl->status.map.rf_off == BTC_LPS_RF_OFF) ||
		   (wl->status.map.lps_pre == BTC_LPS_RF_OFF &&
		    wl->status.map.lps == BTC_LPS_RF_OFF)) {
		rtw89_debug(rtwdev, RTW89_DBG_BTC,
			    "[BTC], %s(): return by wl off!!\n", __func__);
		pfwinfo->cnt_h2c_fail++;
		return;
	}

	pfwinfo->cnt_h2c++;

	ret = rtw89_fw_h2c_raw_with_hdr(rtwdev, h2c_class, h2c_func, param, len,
					false, true);
	if (ret != 0)
		pfwinfo->cnt_h2c_fail++;
}

static void _reset_btc_var(struct rtw89_dev *rtwdev, u8 type)
{
	struct rtw89_btc *btc = &rtwdev->btc;
	struct rtw89_btc_cx *cx = &btc->cx;
	struct rtw89_btc_wl_info *wl = &btc->cx.wl;
	struct rtw89_btc_bt_info *bt = &btc->cx.bt;
	struct rtw89_btc_bt_link_info *bt_linfo = &bt->link_info;
	struct rtw89_btc_wl_link_info *wl_linfo = wl->link_info;
	u8 i;

	rtw89_debug(rtwdev, RTW89_DBG_BTC, "[BTC], %s\n", __func__);

	if (type & BTC_RESET_CX)
		memset(cx, 0, sizeof(*cx));
	else if (type & BTC_RESET_BTINFO) /* only for BT enable */
		memset(bt, 0, sizeof(*bt));

	if (type & BTC_RESET_CTRL) {
		memset(&btc->ctrl, 0, sizeof(btc->ctrl));
		btc->ctrl.trace_step = FCXDEF_STEP;
	}

	/* Init Coex variables that are not zero */
	if (type & BTC_RESET_DM) {
		memset(&btc->dm, 0, sizeof(btc->dm));
		memset(bt_linfo->rssi_state, 0, sizeof(bt_linfo->rssi_state));

		for (i = 0; i < RTW89_PORT_NUM; i++)
			memset(wl_linfo[i].rssi_state, 0,
			       sizeof(wl_linfo[i].rssi_state));

		/* set the slot_now table to original */
		btc->dm.tdma_now = t_def[CXTD_OFF];
		btc->dm.tdma = t_def[CXTD_OFF];
		memcpy(&btc->dm.slot_now, s_def, sizeof(btc->dm.slot_now));
		memcpy(&btc->dm.slot, s_def, sizeof(btc->dm.slot));

		btc->policy_len = 0;
		btc->bt_req_len = 0;

		btc->dm.coex_info_map = BTC_COEX_INFO_ALL;
		btc->dm.wl_tx_limit.tx_time = BTC_MAX_TX_TIME_DEF;
		btc->dm.wl_tx_limit.tx_retry = BTC_MAX_TX_RETRY_DEF;
	}

	if (type & BTC_RESET_MDINFO)
		memset(&btc->mdinfo, 0, sizeof(btc->mdinfo));
}

#define BTC_RPT_HDR_SIZE 3
#define BTC_CHK_WLSLOT_DRIFT_MAX 15
#define BTC_CHK_HANG_MAX 3

static void _chk_btc_err(struct rtw89_dev *rtwdev, u8 type, u32 cnt)
{
	struct rtw89_btc *btc = &rtwdev->btc;
	struct rtw89_btc_cx *cx = &btc->cx;
	struct rtw89_btc_dm *dm = &btc->dm;
	struct rtw89_btc_bt_info *bt = &cx->bt;

	rtw89_debug(rtwdev, RTW89_DBG_BTC,
		    "[BTC], %s(): type:%d cnt:%d\n",
		    __func__, type, cnt);

	switch (type) {
	case BTC_DCNT_RPT_FREEZE:
		if (dm->cnt_dm[BTC_DCNT_RPT] == cnt && btc->fwinfo.rpt_en_map)
			dm->cnt_dm[BTC_DCNT_RPT_FREEZE]++;
		else
			dm->cnt_dm[BTC_DCNT_RPT_FREEZE] = 0;

		if (dm->cnt_dm[BTC_DCNT_RPT_FREEZE] >= BTC_CHK_HANG_MAX)
			dm->error.map.wl_fw_hang = true;
		else
			dm->error.map.wl_fw_hang = false;

		dm->cnt_dm[BTC_DCNT_RPT] = cnt;
		break;
	case BTC_DCNT_CYCLE_FREEZE:
		if (dm->cnt_dm[BTC_DCNT_CYCLE] == cnt &&
		    (dm->tdma_now.type != CXTDMA_OFF ||
		     dm->tdma_now.ext_ctrl == CXECTL_EXT))
			dm->cnt_dm[BTC_DCNT_CYCLE_FREEZE]++;
		else
			dm->cnt_dm[BTC_DCNT_CYCLE_FREEZE] = 0;

		if (dm->cnt_dm[BTC_DCNT_CYCLE_FREEZE] >= BTC_CHK_HANG_MAX)
			dm->error.map.cycle_hang = true;
		else
			dm->error.map.cycle_hang = false;

		dm->cnt_dm[BTC_DCNT_CYCLE] = cnt;
		break;
	case BTC_DCNT_W1_FREEZE:
		if (dm->cnt_dm[BTC_DCNT_W1] == cnt &&
		    dm->tdma_now.type != CXTDMA_OFF)
			dm->cnt_dm[BTC_DCNT_W1_FREEZE]++;
		else
			dm->cnt_dm[BTC_DCNT_W1_FREEZE] = 0;

		if (dm->cnt_dm[BTC_DCNT_W1_FREEZE] >= BTC_CHK_HANG_MAX)
			dm->error.map.w1_hang = true;
		else
			dm->error.map.w1_hang = false;

		dm->cnt_dm[BTC_DCNT_W1] = cnt;
		break;
	case BTC_DCNT_B1_FREEZE:
		if (dm->cnt_dm[BTC_DCNT_B1] == cnt &&
		    dm->tdma_now.type != CXTDMA_OFF)
			dm->cnt_dm[BTC_DCNT_B1_FREEZE]++;
		else
			dm->cnt_dm[BTC_DCNT_B1_FREEZE] = 0;

		if (dm->cnt_dm[BTC_DCNT_B1_FREEZE] >= BTC_CHK_HANG_MAX)
			dm->error.map.b1_hang = true;
		else
			dm->error.map.b1_hang = false;

		dm->cnt_dm[BTC_DCNT_B1] = cnt;
		break;
	case BTC_DCNT_TDMA_NONSYNC:
		if (cnt != 0) /* if tdma not sync between drv/fw  */
			dm->cnt_dm[BTC_DCNT_TDMA_NONSYNC]++;
		else
			dm->cnt_dm[BTC_DCNT_TDMA_NONSYNC] = 0;

		if (dm->cnt_dm[BTC_DCNT_TDMA_NONSYNC] >= BTC_CHK_HANG_MAX)
			dm->error.map.tdma_no_sync = true;
		else
			dm->error.map.tdma_no_sync = false;
		break;
	case BTC_DCNT_SLOT_NONSYNC:
		if (cnt != 0) /* if slot not sync between drv/fw  */
			dm->cnt_dm[BTC_DCNT_SLOT_NONSYNC]++;
		else
			dm->cnt_dm[BTC_DCNT_SLOT_NONSYNC] = 0;

		if (dm->cnt_dm[BTC_DCNT_SLOT_NONSYNC] >= BTC_CHK_HANG_MAX)
			dm->error.map.tdma_no_sync = true;
		else
			dm->error.map.tdma_no_sync = false;
		break;
	case BTC_DCNT_BTCNT_FREEZE:
		cnt = cx->cnt_bt[BTC_BCNT_HIPRI_RX] +
		      cx->cnt_bt[BTC_BCNT_HIPRI_TX] +
		      cx->cnt_bt[BTC_BCNT_LOPRI_RX] +
		      cx->cnt_bt[BTC_BCNT_LOPRI_TX];

		if (cnt == 0)
			dm->cnt_dm[BTC_DCNT_BTCNT_FREEZE]++;
		else
			dm->cnt_dm[BTC_DCNT_BTCNT_FREEZE] = 0;

		if ((dm->cnt_dm[BTC_DCNT_BTCNT_FREEZE] >= BTC_CHK_HANG_MAX &&
		     bt->enable.now) || (!dm->cnt_dm[BTC_DCNT_BTCNT_FREEZE] &&
		     !bt->enable.now))
			_update_bt_scbd(rtwdev, false);
		break;
	case BTC_DCNT_WL_SLOT_DRIFT:
		if (cnt >= BTC_CHK_WLSLOT_DRIFT_MAX)
			dm->cnt_dm[BTC_DCNT_WL_SLOT_DRIFT]++;
		else
			dm->cnt_dm[BTC_DCNT_WL_SLOT_DRIFT] = 0;

		if (dm->cnt_dm[BTC_DCNT_WL_SLOT_DRIFT] >= BTC_CHK_HANG_MAX)
			dm->error.map.wl_slot_drift = true;
		else
			dm->error.map.wl_slot_drift = false;
		break;
	}
}

static void _update_bt_report(struct rtw89_dev *rtwdev, u8 rpt_type, u8 *pfinfo)
{
	struct rtw89_btc *btc = &rtwdev->btc;
	struct rtw89_btc_bt_info *bt = &btc->cx.bt;
	struct rtw89_btc_bt_link_info *bt_linfo = &bt->link_info;
	struct rtw89_btc_bt_a2dp_desc *a2dp = &bt_linfo->a2dp_desc;
	struct rtw89_btc_fbtc_btver *pver = NULL;
	struct rtw89_btc_fbtc_btscan *pscan = NULL;
	struct rtw89_btc_fbtc_btafh *pafh = NULL;
	struct rtw89_btc_fbtc_btdevinfo *pdev = NULL;

	pver = (struct rtw89_btc_fbtc_btver *)pfinfo;
	pscan = (struct rtw89_btc_fbtc_btscan *)pfinfo;
	pafh = (struct rtw89_btc_fbtc_btafh *)pfinfo;
	pdev = (struct rtw89_btc_fbtc_btdevinfo *)pfinfo;

	rtw89_debug(rtwdev, RTW89_DBG_BTC,
		    "[BTC], %s(): rpt_type:%d\n",
		    __func__, rpt_type);

	switch (rpt_type) {
	case BTC_RPT_TYPE_BT_VER:
		bt->ver_info.fw = le32_to_cpu(pver->fw_ver);
		bt->ver_info.fw_coex = le32_get_bits(pver->coex_ver, GENMASK(7, 0));
		bt->feature = le32_to_cpu(pver->feature);
		break;
	case BTC_RPT_TYPE_BT_SCAN:
		memcpy(bt->scan_info, pscan->scan, BTC_SCAN_MAX1);
		break;
	case BTC_RPT_TYPE_BT_AFH:
		memcpy(&bt_linfo->afh_map[0], pafh->afh_l, 4);
		memcpy(&bt_linfo->afh_map[4], pafh->afh_m, 4);
		memcpy(&bt_linfo->afh_map[8], pafh->afh_h, 2);
		break;
	case BTC_RPT_TYPE_BT_DEVICE:
		a2dp->device_name = le32_to_cpu(pdev->dev_name);
		a2dp->vendor_id = le16_to_cpu(pdev->vendor_id);
		a2dp->flush_time = le32_to_cpu(pdev->flush_time);
		break;
	default:
		break;
	}
}

struct rtw89_btc_fbtc_cysta_cpu {
	u8 fver;
	u8 rsvd;
	u16 cycles;
	u16 cycles_a2dp[CXT_FLCTRL_MAX];
	u16 a2dpept;
	u16 a2dpeptto;
	u16 tavg_cycle[CXT_MAX];
	u16 tmax_cycle[CXT_MAX];
	u16 tmaxdiff_cycle[CXT_MAX];
	u16 tavg_a2dp[CXT_FLCTRL_MAX];
	u16 tmax_a2dp[CXT_FLCTRL_MAX];
	u16 tavg_a2dpept;
	u16 tmax_a2dpept;
	u16 tavg_lk;
	u16 tmax_lk;
	u32 slot_cnt[CXST_MAX];
	u32 bcn_cnt[CXBCN_MAX];
	u32 leakrx_cnt;
	u32 collision_cnt;
	u32 skip_cnt;
	u32 exception;
	u32 except_cnt;
	u16 tslot_cycle[BTC_CYCLE_SLOT_MAX];
};

static void rtw89_btc_fbtc_cysta_to_cpu(const struct rtw89_btc_fbtc_cysta *src,
					struct rtw89_btc_fbtc_cysta_cpu *dst)
{
	static_assert(sizeof(*src) == sizeof(*dst));

#define __CPY_U8(_x)	({dst->_x = src->_x; })
#define __CPY_LE16(_x)	({dst->_x = le16_to_cpu(src->_x); })
#define __CPY_LE16S(_x)	({int _i; for (_i = 0; _i < ARRAY_SIZE(dst->_x); _i++) \
				   dst->_x[_i] = le16_to_cpu(src->_x[_i]); })
#define __CPY_LE32(_x)	({dst->_x = le32_to_cpu(src->_x); })
#define __CPY_LE32S(_x)	({int _i; for (_i = 0; _i < ARRAY_SIZE(dst->_x); _i++) \
				   dst->_x[_i] = le32_to_cpu(src->_x[_i]); })

	__CPY_U8(fver);
	__CPY_U8(rsvd);
	__CPY_LE16(cycles);
	__CPY_LE16S(cycles_a2dp);
	__CPY_LE16(a2dpept);
	__CPY_LE16(a2dpeptto);
	__CPY_LE16S(tavg_cycle);
	__CPY_LE16S(tmax_cycle);
	__CPY_LE16S(tmaxdiff_cycle);
	__CPY_LE16S(tavg_a2dp);
	__CPY_LE16S(tmax_a2dp);
	__CPY_LE16(tavg_a2dpept);
	__CPY_LE16(tmax_a2dpept);
	__CPY_LE16(tavg_lk);
	__CPY_LE16(tmax_lk);
	__CPY_LE32S(slot_cnt);
	__CPY_LE32S(bcn_cnt);
	__CPY_LE32(leakrx_cnt);
	__CPY_LE32(collision_cnt);
	__CPY_LE32(skip_cnt);
	__CPY_LE32(exception);
	__CPY_LE32(except_cnt);
	__CPY_LE16S(tslot_cycle);

#undef __CPY_U8
#undef __CPY_LE16
#undef __CPY_LE16S
#undef __CPY_LE32
#undef __CPY_LE32S
}

#define BTC_LEAK_AP_TH 10
#define BTC_CYSTA_CHK_PERIOD 100

struct rtw89_btc_prpt {
	u8 type;
	__le16 len;
	u8 content[];
} __packed;

static u32 _chk_btc_report(struct rtw89_dev *rtwdev,
			   struct rtw89_btc_btf_fwinfo *pfwinfo,
			   u8 *prptbuf, u32 index)
{
	const struct rtw89_chip_info *chip = rtwdev->chip;
	struct rtw89_btc *btc = &rtwdev->btc;
	struct rtw89_btc_dm *dm = &btc->dm;
	struct rtw89_btc_rpt_cmn_info *pcinfo = NULL;
	struct rtw89_btc_wl_info *wl = &btc->cx.wl;
	struct rtw89_btc_bt_info *bt = &btc->cx.bt;
	struct rtw89_btc_fbtc_rpt_ctrl *prpt;
	struct rtw89_btc_fbtc_rpt_ctrl_v1 *prpt_v1;
	struct rtw89_btc_fbtc_cysta *pcysta_le32 = NULL;
	struct rtw89_btc_fbtc_cysta_v1 *pcysta_v1 = NULL;
	struct rtw89_btc_fbtc_cysta_cpu pcysta[1];
	struct rtw89_btc_prpt *btc_prpt = NULL;
	struct rtw89_btc_fbtc_slot *rtp_slot = NULL;
	void *rpt_content = NULL, *pfinfo = NULL;
	u8 rpt_type = 0;
	u16 wl_slot_set = 0, wl_slot_real = 0;
	u32 trace_step = btc->ctrl.trace_step, rpt_len = 0, diff_t;
	u32 cnt_leak_slot = 0, bt_slot_real = 0, cnt_rx_imr = 0;
	u8 i;

	rtw89_debug(rtwdev, RTW89_DBG_BTC,
		    "[BTC], %s(): index:%d\n",
		    __func__, index);

	if (!p