/************************************************************************/
/* File Name            : unipowerd.c                                   */
/* Program Name         : unipowerd                   Version: 1.0.0    */
/* Author               : Tom Webster <webster@kaiwan.com>              */
/* Created              : 1994/04/20                                    */
/* Last Modified By     : Tom Webster                 Date: 1995/04/09  */
/*                                                                      */
/* Compiler (created)   : GCC 2.5.8                                     */
/* Compiler (env)       : Linux 1.0.9                                   */
/* ANSI C Compatable    : No                                            */
/* POSIX Compatable     : Yes?                                          */
/*                                                                      */
/* Purpose              : Monitor the serial port connected to a UPS.   */
/*                      : Functions performed depend on the values in   */                                                
/*                      : the unipowerd.h file.  As a minimum, the      */
/*                      : power status will be monitored.  Low battery  */
/*                      : sensing and physical cable conection checks   */
/*                      : are also supported.                           */
/*                      :                                               */
/*                      : If one of the monitored lines of the serial   */
/*                      : connection changes state, unipowerd will      */
/*                      : notify init.                                  */
/*                      :                                               */
/*                      : If sent the "-k" option (and compiled to      */
/*                      : support shutting the inverter down), will     */
/*                      : shut the inverter down (killing power to      */
/*                      : the system while in powerfail mode).          */
/*                                                                      */
/* Usage                : unipowerd [-k] <serial device>                */
/*                      : i.e. unipowerd /dev/cua4                      */
/*                      : or:  unipowerd  -k /dev/cua4                  */
/*                                                                      */
/* History/Credits      : unipowerd is a hack to make the powerd        */
/*                      : deamon easier to configure and to add         */
/*                      : additional features.                          */
/*                      :                                               */
/*                      : powerd was written by Miquel van Smoorenburg  */
/*                      : <miquels@drinkel.nl.mugnet.org> for his       */
/*                      : employer Cistron Electronics and released     */
/*                      : into the public domain.                       */
/*                                                                      */
/* Copyright            : GNU Copyleft                                  */
/************************************************************************/
 
#include <sys/types.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <errno.h>
#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <syslog.h>
#include "unipowerd.h"

#ifndef INVERTERTIME
#define INVERTERTIME 2
#endif

int unitest(int, int, int);
void powerfail(int);

/* Main program. */
int main(int argc, char **argv) {
	int fd, fd2;
  	int flags;
  	int pstatus, poldstat = 1;
  	int bstatus, boldstat = 1;
  	int count = 0;
  	int tries = 0;
	int dtr_bit = TIOCM_DTR;
        int rts_bit = TIOCM_RTS;
	int cable_bit = CABLEPOWER;
        int killbit = INVERTERBIT;
	int ikill = 0;
	char *program_name;


	program_name = argv[0];
	/* Parse the command line */
	while ((argc > 1) && (argv[1][0] == '-')) {
		switch (argv[1][1]) {
			case 'k':
				ikill = 1;
				break;
			default:
				fprintf(stderr, "Usage: %s [-k] <device>\n", program_name);
				exit(1);
		} 					/* switch (argv[1][1]) */
		argv++;
		argc--;
	}						/* while ((argc > 1) && (argv[1][0] == '-')) */
	
	/* Done with options, make sure that one port is specified */
	if (argc != 2) {
		fprintf(stderr, "Usage: %s [-k] <device>\n", program_name);
                exit(1);
	}						/* if (argc != 2) */
	
	/* Kill the inverter and close out if inverter kill was selected */
	if (ikill) {
		if (INVERTERKILL) {
 		       if ((fd = open(argv[1], O_RDWR | O_NDELAY)) < 0) {
		                fprintf(stderr, "%s: %s: %s\n",program_name, argv[1], sys_errlist[errno]);
		                exit(1);
		        } /* if ((fd = open(argv[1], O_RDWR | O_NDELAY)) < 0) */
	
		        /* Force high to send the UPS the inverter kill signal. */
		        ioctl(fd, TIOCMBIS, &killbit);
		        sleep(INVERTERTIME);
		        close(fd);
		        exit(0);
		} else {
			fprintf(stderr, "Error: %s was not compiled to support killing the inverter.\n", program_name);
			exit(1);
		}					/* if (INVERTERKILL) */
	}						/* if (ikill) */

	/****************************************/
	/* If no kill signal, monitor the line. */
	/****************************************/

  	/* Start syslog. */
 	openlog(program_name, LOG_CONS|LOG_PERROR, LOG_DAEMON);

  	/* Open monitor device. */
  	if ((fd = open(argv[1], O_RDWR | O_NDELAY)) < 0) {
		syslog(LOG_ERR, "%s: %s", argv[1], sys_errlist[errno]);
		closelog();
		exit(1);
  	} 						/* if ((fd = open(argv[1], O_RDWR | O_NDELAY)) < 0) */

  	/* Line is opened, so DTR is high. Clear it and RTS, can cause problems  */
  	/* with some UPSs if it remains set.                             */
  	ioctl(fd, TIOCMBIC, &dtr_bit);
        ioctl(fd, TIOCMBIC, &rts_bit);

  	/* Set line to provide power high to enable monitoring of line. */
  	ioctl(fd, TIOCMBIS, &cable_bit);
 
  	/* Daemonize. */
  	switch(fork()) {
		case 0: /* Child */
			closelog();
			setsid();
			break;
		case -1: /* Error */
			syslog(LOG_ERR, "can't fork.");
			closelog();
			exit(1);
		default: /* Parent */
			closelog();
			exit(0);
  	} 						/* switch(fork()) */

  	/* Restart syslog. */
  	openlog(program_name, LOG_CONS, LOG_DAEMON);

        /* Create an info file for powerfail scripts. */
        unlink(UPSSTAT);
        if ((fd2 = open(UPSSTAT, O_CREAT|O_WRONLY, 0644)) >= 0) {
                write(fd2, "OK\n", 3);
                close(fd2);
        }
 
  	/* Now sample the line. */
  	while(1) {
		/* Get the status. */
		ioctl(fd, TIOCMGET, &flags);

		/* Calculate present status. */
 		pstatus = unitest(POWERBIT, POWEROK, flags);

#if BATTCHECK
 		bstatus = unitest(BATTBIT, BATTOK, flags);
#else
		bstatus = 1;
#endif

#if CABLECHECK
		/* Check the connection. */
		tries = 0;
		while((unitest(CABLEBIT, CABLEOK, flags) == 0) {
			/* Keep on trying, and warn every two minutes. */
			if ((tries % 60) == 0)
			    syslog(LOG_ALERT, "UPS connection error");
			sleep(2);
			tries++;
  			ioctl(fd, TIOCMGET, &flags);
		}					/*  */
		if (tries > 0)
			syslog(LOG_ALERT, "UPS connection OK");
#else
		/* Do pseudo-cable check, bad power on startup == possible bad cable */
		if (tries < 1) {
			tries++;

			/* Do startup failure check */
			if (!pstatus) {
				/* Power is out: assume bad cable, but semi-scram to be safe */
				syslog(LOG_ALERT, "No power on startup; UPS connection error?");
				
				/* Set status registers to prevent further processing until */
				/* the status of the cable is changed.                      */
				poldstat = pstatus;
				boldstat = bstatus;
				
				powerfail(3);
			} 			/* if (!pstatus) */
		} 				/* if (tries < 1) */
#endif

		/* If anything has changed, process the change */
		if (pstatus != poldstat || bstatus != boldstat) {
			count++;
			if (count < 4) {
				/* Wait a little to ride out short brown-outs */
				sleep(1);
				continue;
			} 				/* if (count < 4) */

			if (pstatus != poldstat) {
				if (pstatus) {
					/* Power is OK */
					syslog(LOG_ALERT, "Line power restored");
					powerfail(0);
				}else{
					/* Power has FAILED */
					if (bstatus) {
						/* Battery OK, normal shutdown */
						syslog(LOG_ALERT, "Line power has failed");
						powerfail(1);
					}else{
						/* Low Battery, SCRAM! */
						syslog(LOG_ALERT, "UPS battery power is low!");
						powerfail(2);
					} 		/* if (bstatus) */
				} 			/* if (pstatus) */
			} 				/* if (pstatus != poldstat) */
	
			if (bstatus != boldstat) {
				if (!bstatus && !pstatus) {
					/* Power is out and Battery is now low, SCRAM! */
					syslog(LOG_ALERT, "UPS battery power is low!");
					powerfail(2);
				} 			/* if (!bstatus && !pstatus) */
			} 				/* if (bstatus != boldstat) */
		} 					/* if (pstatus != poldstat || bstatus != boldstat) */

	
		/* Reset count, remember status and sleep 2 seconds. */
		count = 0;
		poldstat = pstatus;
		boldstat = bstatus;
		sleep(2);
  	} 						/* while(1) */
  	/* Never happens */
  	return(0);
} 							/* main */
/************************************************************************/
/* End of Function main                                                 */
/************************************************************************/
 

/************************************************************************/
/* Function             : unitest                                       */
/* Author               : Tom Webster <webster@kaiwan.com>              */
/* Created              : 1994/11/04                                    */
/* Last Modified By     :                             Date:             */
/*                                                                      */
/* Takes                : Line of serial connection to test (int).      */
/*                      :                                               */
/*                      : True/False value of a positive result,        */
/*                      : 0 == True, 0 != False.  (int)                 */
/*                      :                                               */
/*                      : The current status of the serial connection   */
/*                      : as an int.                                    */
/*                                                                      */
/* Returns              : int containing result of test.                */
/*                      :                                               */
/*                      : 1 == Tested line is in normal mode.           */
/*                      : 0 == Tested line is in failure mode.          */
/*                                                                      */
/* Purpose              : Tests the condition of a serial control line  */
/*                      : to see if it is in normal or failure mode.    */
/*                                                                      */
/************************************************************************/                                                
int unitest(int testline, int okvalue, int testflags) 
{
        int unireturn;
        if (okvalue) {
                if ((testflags & testline)) {
                        /* OK Value Returned */
                        unireturn = 1;
                }else{
                        /* Fail Value Returned */
                        unireturn = 0;
                }
        }else{
                if (!(testflags & testline)) {
                        /* OK Value Returned */
                        unireturn = 1;
                }else{
                        /* Fail Value Returned */
                        unireturn = 0;
                }
        }
        return(unireturn);
} /* EOF unitest */
/************************************************************************/
/* End of Function unitest                                              */
/************************************************************************/
 

/************************************************************************/
/* Function             : powerfail                                     */
/* Author               : Miquel van Smoorenburg                        */
/*                      :  <miquels@drinkel.nl.mugnet.org>              */
/* Created              : 1993/10/28                                    */
/* Last Modified By     : Tom Webster                 Date: 1995/01/21  */
/*                      :  <webster@kaiwan.com>                         */
/*                                                                      */
/* Takes                : Failure mode as an int.                       */
/* Returns              : None.                                         */
/*                                                                      */
/* Purpose              : Communicates power status to init via the     */
/*                      : SIGPWR signal and status files.               */
/*                                                                      */
/************************************************************************/                                                
void powerfail(int ok)
{
	int fd;

	/* Create an info file for init. */
	unlink(PWRSTAT);
	if ((fd = open(PWRSTAT, O_CREAT|O_WRONLY, 0644)) >= 0) {
		if (ok) {
			/* Problem */
			write(fd, "FAIL\n", 5);
		} else {
			/* No problem */
			write(fd, "OK\n", 3);
 		} 					/* if (ok) */
		close(fd);
  	}

        /* Create an info file for powerfail scripts. */
        unlink(UPSSTAT);
        if ((fd = open(UPSSTAT, O_CREAT|O_WRONLY, 0644)) >= 0) {
                switch(ok) {
                        case 0: /* Power OK */
                                write(fd, "OK\n", 3);
                                break;
                        case 1: /* Line Fail */
                                write(fd, "FAIL\n", 5);
                                break;
                        case 2: /* Line Fail and Low Batt */
                                write(fd, "SCRAM\n", 6);
				break;
                        case 3: /* Bad cable? */
                                write(fd, "CABLE\n", 6);
				break;
                        default: /* Should never get here */
                                write(fd, "SCRAM\n", 6);
                }                                       /* Switch(ok) */
                close(fd);
        }
  	kill(1, SIGPWR);
} /*EOF powerfail */
/************************************************************************/
/* End of Function powerfail                                            */
/************************************************************************/
 
