use strict;
use warnings;
use Test::More;

BEGIN { use_ok( 'Barcode::DataMatrix' ) }

my $expected = render_expected();
for my $encoding (qw(ASCII C40 TEXT BASE256 NONE AUTO)) {
    my $code = new_ok('Barcode::DataMatrix' => [encoding_mode => $encoding]);

    my $output = $code->barcode('MONKEY');
    is_deeply($output, $expected->{$encoding}->{short}, 'barcode() ' . $encoding . ' output matches for short example');

    $output = $code->barcode('MONKEY'x30);
    is_deeply($output, $expected->{$encoding}->{long}, 'barcode() ' . $encoding . ' output matches for long example');
}

done_testing;

sub render_expected {
    return {"TEXT" => {"short" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,0,0,1,1,0,1,1,0,0,1,1,1,1,1],[1,0,0,1,0,0,1,1,0,1,0,1,1,1,1,0],[1,1,1,0,0,1,1,1,0,1,1,0,0,0,1,1],[1,1,0,0,1,1,1,0,1,0,0,0,1,0,0,0],[1,1,0,0,1,1,0,0,1,1,1,0,0,1,1,1],[1,0,1,1,0,0,0,0,0,0,1,0,1,1,0,0],[1,1,1,1,1,0,0,1,0,0,1,1,0,1,1,1],[1,1,1,1,1,1,0,0,1,0,1,1,0,0,0,0],[1,1,0,1,1,0,0,1,1,0,0,0,1,1,1,1],[1,1,0,1,0,1,0,1,1,1,0,1,1,0,0,0],[1,0,1,0,1,0,0,0,1,1,1,1,1,1,1,1],[1,0,1,1,0,1,1,0,1,1,0,0,0,1,1,0],[1,0,0,0,0,1,0,1,0,1,0,1,1,1,0,1],[1,0,0,1,0,0,0,1,0,1,0,0,1,0,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]],"long" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,0,0,1,1,0,1,1,0,0,1,1,0,1,1,1,1,1,0,0,0,1,0,0,0,1,1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,1,1,0,1,1,1,1,0,0,0,1,0,1,1,1,1,0,0,1,1],[1,0,0,1,0,0,1,1,0,0,1,1,0,0,0,0,1,1,1,0,1,1,1,0,1,1,0,0,0,1,0,0,1,0,1,1,1,0,0,1,1,0,1,1,1,1,0,0,1,1,0,1,0,1,1,0,1,1,0,0,1,1,0,0],[1,1,1,0,0,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,1,0,0,1,0,0,0,1,1,0,1,1,0,1,0,0,1,1,0,1,0,1,0,0,0,1,1,1,0,0,1,1,1,0,1,1,0,1,1,1,1,1,1],[1,1,0,0,1,1,1,0,1,1,1,0,1,1,1,0,1,1,1,1,0,1,0,1,1,0,0,1,0,0,1,0,1,1,0,0,1,1,0,0,0,1,0,0,1,1,0,0,1,0,1,1,0,0,0,1,0,0,0,1,1,1,0,0],[1,1,0,0,1,1,0,1,1,0,1,0,1,0,0,1,1,0,1,1,0,1,1,1,0,1,1,0,1,1,1,1,1,0,1,1,1,1,0,0,1,0,0,1,1,1,0,1,1,1,1,0,0,0,1,1,0,1,1,1,1,1,1,1],[1,0,1,1,0,0,0,1,0,0,0,1,0,0,0,0,1,1,0,0,1,1,0,1,0,1,0,0,0,1,1,0,1,0,1,1,1,0,1,1,0,1,1,1,0,0,1,0,1,1,1,0,0,1,1,0,1,1,1,1,1,1,0,0],[1,1,1,0,0,0,1,1,0,0,1,0,0,1,1,1,1,0,0,1,1,0,0,0,1,0,0,1,1,1,0,1,1,1,1,0,0,0,1,0,0,0,1,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,0,1,1,0,1,1],[1,1,1,0,0,1,1,0,1,1,0,0,1,1,0,0,1,0,1,1,1,0,1,1,1,0,1,1,1,0,0,0,1,1,0,1,0,1,1,0,0,1,0,0,1,1,0,0,1,1,0,1,0,0,0,1,0,1,0,1,1,0,0,0],[1,1,0,1,1,0,0,0,1,0,1,1,1,1,0,1,1,0,1,0,0,1,1,1,1,0,1,0,0,0,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,0,1,1,0,1,0,0,0,1,0,0,0,1,0,0,0,0,1],[1,1,0,1,0,1,0,1,1,0,1,1,0,0,1,0,1,1,0,1,1,1,0,0,0,1,0,0,0,1,0,0,1,0,1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,1,1,0,1,1,0,0,1,1,0,0,0,0,1,0],[1,0,1,0,0,1,1,1,0,1,1,0,1,1,1,1,1,0,0,0,1,1,0,0,1,1,0,1,1,0,0,1,1,1,1,0,1,1,1,1,0,1,1,0,0,1,1,1,1,1,0,0,1,0,0,1,1,1,1,1,0,0,0,1],[1,1,0,0,1,1,0,0,0,1,0,0,0,1,1,0,1,0,0,1,1,0,1,1,0,1,0,1,0,0,1,0,1,1,1,0,1,1,1,0,1,1,1,1,1,1,0,0,1,1,0,1,1,1,1,1,1,1,1,0,1,0,1,0],[1,1,0,1,1,0,0,0,1,1,0,1,1,0,0,1,1,1,1,0,0,0,1,0,0,0,1,0,0,0,1,1,1,1,0,1,1,0,1,0,1,0,0,0,1,1,0,1,1,1,1,0,0,0,1,1,0,0,1,0,0,1,1,1],[1,0,1,1,1,0,0,1,1,0,1,1,1,0,1,0,1,1,0,1,0,1,1,0,1,1,0,0,1,1,0,0,1,0,0,1,0,0,0,1,0,0,0,1,0,0,1,0,1,1,0,0,0,1,0,1,0,1,1,1,1,0,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,1,1,0,1,1,0,1,0,1,0,1,1,1,1,1,1,0,1,1,0,0,1,1,1,1,1,1,0,0,1,1,0,1,1,0,0,1,0,0,1,1,0,0,1,1,1,1,0,0,1,1,0,1,0,0,0,0,0,1,0,0,1],[1,1,0,1,0,1,0,0,0,1,0,0,1,1,0,0,1,0,1,1,0,1,1,1,0,0,0,1,1,0,0,0,1,1,1,0,1,1,0,0,1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,0,0,1,1,0,0,1,0],[1,0,0,0,1,0,0,0,1,1,0,1,1,0,0,1,1,1,1,0,0,0,1,1,0,1,1,1,0,1,1,1,1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,1,1,1,1,1,1,1,1,1,0,1,1,0,0,0,0,1],[1,0,1,1,1,0,1,1,0,0,0,1,0,0,1,0,1,1,1,0,0,1,1,0,1,1,1,1,0,1,0,0,1,1,0,1,1,0,1,1,0,0,1,1,0,0,1,0,1,1,1,0,0,0,0,1,0,1,0,1,0,1,0,0],[1,1,1,1,0,0,1,0,0,0,1,1,0,0,1,1,1,0,0,1,1,0,1,0,1,0,0,0,1,0,0,1,1,1,1,1,0,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,0,1,0,1,1,0,1,0,0,0,0,1],[1,1,0,1,0,1,1,0,0,1,0,0,1,1,0,0,1,0,1,1,0,0,0,1,0,0,1,1,0,0,1,0,1,1,0,0,0,1,0,0,0,1,1,0,0,1,0,0,1,0,0,0,0,1,0,0,1,0,1,0,1,1,1,0],[1,1,0,1,1,1,0,1,1,0,1,1,1,0,1,1,1,1,1,1,0,0,1,0,0,1,1,1,0,1,1,1,1,0,0,0,1,1,0,1,1,0,0,0,1,1,1,1,1,0,0,0,0,0,1,0,1,1,1,0,0,0,1,1],[1,0,1,1,0,1,0,1,0,0,0,1,1,0,1,0,1,1,1,0,1,1,0,1,1,1,0,0,1,1,1,0,1,0,0,1,1,0,1,1,1,0,0,1,1,0,1,0,1,1,1,0,0,0,0,0,0,0,0,1,0,0,0,0],[1,1,1,0,0,0,1,0,0,1,1,1,0,1,1,1,1,0,0,0,1,0,0,0,1,1,0,0,1,1,0,1,1,1,1,0,1,0,1,1,0,1,1,0,1,1,1,1,1,0,1,0,1,1,1,0,0,1,1,1,0,1,0,1],[1,1,1,0,1,1,1,0,1,1,1,0,0,1,0,0,1,1,0,1,1,0,0,1,0,0,1,1,0,1,0,0,1,1,0,0,0,1,0,1,0,1,0,1,1,0,1,0,1,1,0,0,0,1,1,1,1,1,1,1,0,0,1,0],[1,0,0,1,1,1,1,0,1,0,0,0,1,1,0,1,1,1,1,0,0,1,1,0,1,1,1,0,0,0,1,1,1,0,0,0,1,0,0,0,1,0,0,1,0,1,1,1,1,1,0,1,1,1,1,1,1,0,1,0,0,1,0,1],[1,1,1,1,0,0,0,1,0,0,0,1,0,0,1,0,1,1,0,0,1,1,0,0,0,1,1,0,1,1,1,0,1,0,1,1,0,0,1,1,1,1,1,0,0,1,0,0,1,1,1,0,0,0,0,1,1,0,1,0,1,1,0,0],[1,0,1,1,0,0,1,1,0,1,1,0,0,1,1,1,1,0,1,1,1,1,0,1,1,0,0,1,1,1,0,1,1,0,1,0,0,1,1,1,1,0,1,0,1,0,1,1,1,1,0,1,0,0,0,1,0,0,1,0,0,0,0,1],[1,1,1,0,1,1,0,1,0,1,0,0,1,1,1,0,1,0,1,1,1,0,1,1,1,1,1,1,0,1,0,0,1,1,1,1,1,1,1,0,0,1,0,1,0,1,0,0,1,0,0,0,1,1,0,0,0,0,0,0,0,0,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,0,0,1,0,0,0,1,0,0,0,1,1,0,1,1,1,1,0,1,0,1,0,0,0,1,1,0,1,1,1,1,0,0,0,1,1,0,0,0,1,1,0,0,0,0,1,1,1,0,1,0,1,0,0,1,1,0,0,1,1,1,1],[1,1,0,1,1,0,1,1,0,0,1,1,0,0,0,0,1,1,0,0,0,1,0,0,0,1,0,0,1,1,0,0,1,0,0,1,0,0,0,1,0,0,1,0,1,0,0,0,1,1,1,1,0,1,0,0,0,1,1,1,0,1,0,0],[1,1,1,0,0,1,1,1,1,1,1,0,0,0,1,1,1,1,0,0,1,0,0,1,1,0,0,1,1,0,0,1,1,1,1,0,0,1,1,0,0,0,0,1,1,1,1,1,1,0,0,0,1,1,1,0,1,0,1,0,0,0,0,1],[1,1,0,1,1,1,0,0,0,1,1,0,0,1,1,0,1,0,1,1,0,0,1,1,0,0,1,1,1,0,1,0,1,1,1,0,0,0,1,1,1,1,1,0,1,1,0,0,1,1,0,0,1,1,0,1,0,1,1,0,0,1,1,0],[1,0,0,0,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,0,1,1,1,1,0,0,1,0,1,1,1,1,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,0,1,0,0,1,0,1,0,0,1,1,1,1],[1,0,0,1,1,0,1,1,1,1,0,1,0,1,0,0,1,1,1,0,1,1,0,0,1,1,0,0,1,1,0,0,1,1,1,1,1,1,0,0,0,0,1,0,1,0,1,0,1,0,1,0,1,1,1,1,0,0,1,1,0,0,1,0],[1,1,1,0,1,0,1,0,0,0,1,0,0,1,1,1,1,1,0,1,1,0,1,1,1,1,0,1,1,0,0,1,1,0,0,0,0,1,0,0,0,0,1,0,0,0,0,1,1,1,0,0,1,1,0,0,1,0,1,0,1,0,0,1],[1,1,0,0,0,1,0,0,1,1,0,0,1,1,0,0,1,0,0,1,0,0,0,1,1,0,0,1,1,1,1,0,1,1,0,0,0,1,0,0,1,1,0,1,1,1,1,0,1,0,0,0,1,0,1,1,1,0,0,1,0,0,1,0],[1,1,0,0,1,0,0,1,1,1,0,1,1,0,0,1,1,0,1,1,0,1,1,0,0,0,1,1,0,1,1,1,1,0,1,1,1,0,1,1,1,0,0,0,1,0,0,1,1,0,0,1,0,1,0,0,0,1,1,0,1,1,1,1],[1,0,1,1,0,1,1,1,0,0,1,1,1,0,0,0,1,1,1,0,1,1,1,0,0,1,0,1,1,0,0,0,1,1,1,0,0,0,1,1,1,0,1,1,0,0,1,0,1,0,0,0,1,1,0,1,1,1,0,0,0,0,1,0],[1,0,1,0,0,0,1,1,0,0,1,1,0,1,1,1,1,0,1,0,1,1,0,1,1,0,1,1,0,1,1,1,1,1,1,0,0,1,1,0,1,0,1,1,1,0,1,1,1,1,1,0,0,0,0,1,1,0,1,0,0,1,0,1],[1,1,1,0,0,1,0,0,1,1,0,1,0,1,0,0,1,0,0,1,0,1,0,1,0,0,0,1,1,0,1,0,1,1,0,1,1,0,1,1,1,1,0,1,1,1,0,0,1,0,1,1,0,0,0,1,1,1,1,1,1,1,0,0],[1,0,0,1,1,0,1,1,1,0,0,0,1,0,0,1,1,0,1,0,0,0,1,0,0,0,1,0,1,1,1,1,1,1,1,0,0,0,1,0,0,0,0,1,1,1,1,1,1,0,0,1,1,0,1,1,0,0,1,1,0,0,0,1],[1,0,1,1,0,0,0,1,1,0,1,1,1,0,1,0,1,1,0,0,1,1,1,0,0,1,1,0,0,1,1,0,1,1,1,0,0,0,1,1,1,1,0,1,0,1,1,0,1,1,1,1,0,1,1,1,1,1,0,1,0,0,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,1,1,1,0,1,1,0,0,1,1,1,0,0,1,1,1,0,0,1,1,1,1,0,0,1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,0,1,0,1,0,0,0,1,1,0,1,0,0,1,1,0,1,1,1,1,0,0,1,1],[1,1,1,0,1,1,1,1,1,1,0,1,0,1,1,0,1,1,1,1,0,0,0,0,1,1,1,0,0,1,1,0,1,0,1,1,0,1,0,1,0,1,1,0,1,0,0,0,1,0,1,1,1,0,1,0,0,1,1,1,0,0,1,0],[1,0,1,0,1,0,0,0,1,1,0,1,1,0,0,1,1,0,1,1,1,1,1,0,0,0,1,0,0,1,0,1,1,0,0,0,1,1,1,0,1,0,0,0,1,0,0,1,1,1,1,1,0,1,1,1,1,1,0,1,1,1,1,1],[1,0,0,1,0,0,0,1,0,0,1,1,0,0,0,0,1,1,1,1,0,0,0,0,1,1,0,0,1,0,0,0,1,1,0,1,0,0,0,1,0,1,0,1,1,1,1,0,1,1,0,0,1,0,0,0,1,0,1,1,0,0,0,0],[1,0,1,0,0,1,1,0,0,1,1,0,0,1,1,1,1,0,0,0,1,1,0,1,0,0,0,0,1,1,0,1,1,1,0,1,0,1,1,0,0,0,0,1,0,1,0,1,1,0,1,1,1,1,0,1,0,0,0,0,0,1,0,1],[1,1,0,0,1,1,0,0,1,1,1,0,1,1,0,0,1,1,1,1,0,0,0,0,1,0,0,1,0,1,1,0,1,1,1,0,0,0,1,1,0,1,0,1,1,0,1,0,1,0,0,0,1,0,0,0,1,1,1,0,1,1,1,0],[1,0,1,1,1,1,0,0,1,0,1,1,1,1,1,1,1,0,0,1,1,1,0,1,1,1,1,0,0,0,0,1,1,1,1,1,1,1,0,1,1,0,0,0,0,0,1,1,1,0,1,0,1,0,0,1,0,0,1,1,0,1,1,1],[1,0,1,1,0,0,1,1,0,0,1,1,1,1,1,0,1,0,1,1,0,1,1,1,0,1,0,1,0,0,1,0,1,0,0,0,0,1,0,0,1,0,1,1,1,1,0,0,1,1,0,1,1,1,1,1,0,1,1,0,0,0,1,0],[1,1,1,0,1,1,1,1,0,1,1,0,1,1,1,1,1,0,1,1,0,1,1,1,0,0,1,1,0,0,1,1,1,1,0,1,0,1,0,1,0,1,0,1,1,1,1,1,1,1,0,0,0,1,0,1,1,0,0,0,1,1,0,1],[1,1,0,0,0,1,1,0,0,1,1,0,0,1,1,0,1,0,1,0,1,1,0,1,1,0,0,1,1,0,1,0,1,0,1,1,1,1,1,0,0,0,1,1,0,0,0,0,1,0,1,0,0,0,1,0,1,0,0,1,0,0,0,0],[1,1,0,1,1,0,0,0,1,1,0,1,0,1,1,1,1,1,1,1,1,1,0,0,1,1,0,1,1,0,1,1,1,0,1,0,1,0,1,0,1,1,1,1,0,1,1,1,1,0,0,1,0,1,0,1,1,1,1,1,1,0,1,1],[1,0,1,1,1,0,0,1,0,0,0,1,1,0,0,0,1,0,1,1,1,0,0,0,1,1,0,1,1,1,0,0,1,0,1,1,1,0,0,1,1,0,1,0,0,1,0,0,1,0,1,0,0,1,0,1,1,1,0,0,1,0,1,0],[1,0,1,1,0,1,1,0,1,0,1,0,0,0,0,1,1,0,0,1,1,1,0,0,1,0,1,0,1,0,1,1,1,0,0,1,1,0,0,0,0,1,1,1,0,1,0,1,1,0,0,1,0,1,1,1,0,0,0,1,1,0,1,1],[1,1,0,0,0,1,0,1,0,1,0,0,0,1,0,0,1,0,0,1,1,1,0,1,0,0,1,1,1,0,1,0,1,1,0,0,0,0,0,1,1,1,0,0,1,0,1,0,1,0,0,1,1,0,0,1,1,0,1,1,0,0,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]]},"NONE" => {"short" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,1,1,1,0,0,0,1,1,1,1,1,1],[1,0,0,1,0,1,0,1,1,0,1,0,0,0],[1,1,1,1,0,0,0,0,0,0,1,1,0,1],[1,1,0,1,1,0,1,0,1,0,0,1,0,0],[1,1,0,1,1,1,0,0,1,1,1,0,1,1],[1,0,0,0,1,0,0,0,0,0,0,1,0,0],[1,0,0,1,1,0,0,1,1,0,1,0,1,1],[1,0,1,1,0,1,1,1,0,0,1,0,0,0],[1,0,1,1,1,0,0,0,0,1,0,1,0,1],[1,1,1,0,1,0,1,0,0,0,0,1,1,0],[1,1,1,1,0,0,0,0,1,1,0,0,0,1],[1,0,0,1,0,1,0,0,0,1,0,1,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1]],"long" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,1,1,1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,1,1,0,0,0,1,1,1,0,1,0,0,1,0,0,1,0,1,1,0,1,0,1,1,0,1,0,0,1,0,0,1,1],[1,0,0,1,0,1,0,1,1,0,0,1,0,1,1,1,0,0,0,1,0,1,0,1,1,0,1,0,0,0,0,1,1,0,1,0,0,1,0,1,0,0,1,0,0,0,0,1,1,0,1,0],[1,1,1,1,0,0,0,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,0,0,1,1,1,1,0,1,0,0,1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1],[1,1,0,1,1,0,1,0,1,1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,0,0,1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,0],[1,1,0,1,1,0,1,0,0,1,0,0,0,0,1,1,0,1,0,1,1,0,1,0,1,1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,1,0,1,1,0,0,1,0,1,1,0,1],[1,1,0,0,1,0,0,0,0,1,1,0,1,0,1,1,0,1,0,0,1,0,0,1,0,0,1,1,1,1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,0],[1,0,1,0,1,1,0,1,0,1,1,0,1,0,0,1,0,0,1,0,1,1,1,0,0,1,1,0,1,0,1,1,1,1,0,0,1,0,1,1,1,0,0,0,1,1,0,0,0,0,0,1],[1,0,1,0,0,1,0,1,0,0,1,0,0,0,0,1,1,0,1,0,0,1,0,1,0,0,1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,1,0,1,1,1,1,0,1,1,1,0],[1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,0,1,1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,0,1,0,0,0,1,0,1,1,1],[1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,0,1,1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,0,0,0,1,1,1,0,0],[1,1,1,0,0,0,1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,1,1,0,1,1,1,0,0,1,0,0,0,0,1,1,0,1,0,1,1,0,1,0,0,1,1,1,0,0,1,1],[1,1,0,1,1,1,1,0,0,1,0,1,1,1,0,0,1,1,0,1,0,0,1,0,0,0,1,0,0,1,1,0,1,0,1,1,0,1,0,0,1,0,1,1,0,0,1,0,1,0,0,0],[1,1,0,0,1,0,1,1,1,0,0,0,1,0,1,0,0,1,0,0,1,0,1,1,0,1,1,1,0,1,1,0,1,0,0,1,0,0,1,0,1,0,0,1,0,0,1,0,1,1,1,1],[1,1,1,1,0,0,0,1,0,1,0,1,1,0,0,0,0,1,1,0,1,0,0,1,0,0,1,1,0,0,1,0,0,0,0,1,1,0,1,0,0,0,1,0,1,0,1,0,0,1,0,0],[1,0,1,0,1,1,1,1,0,0,0,0,1,1,0,1,0,0,1,0,1,0,1,1,0,1,1,0,1,0,1,1,0,1,0,0,1,0,1,1,0,1,1,0,0,0,0,0,0,0,0,1],[1,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,1,0,0,1,0,1,1,1,1,0,0,1,0,0,0,1,1,1,1,0,0,1,0,0],[1,0,1,1,0,1,0,1,1,0,1,0,1,1,0,0,1,0,1,1,1,0,0,0,1,1,1,0,1,0,1,1,0,0,1,0,1,0,1,1,1,1,1,1,1,1,1,1,1,0,0,1],[1,0,1,1,0,1,0,0,1,0,0,1,0,1,1,1,0,0,0,1,0,1,1,1,1,0,1,0,0,1,0,1,1,1,0,0,0,1,1,1,0,1,0,1,1,1,1,0,0,1,0,0],[1,0,0,1,0,0,1,0,1,1,1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,1,1,1,1,0,0,0,1,0,1,1,1,0,0,1,0,1,0,0,0,0,1,1,1,0,1,1],[1,0,0,1,1,0,1,0,0,1,0,1,0,1,1,0,0,1,0,1,1,1,0,0,0,0,1,1,0,1,0,1,1,0,0,1,0,1,0,0,0,0,0,0,0,1,1,0,1,0,0,0],[1,1,0,0,1,0,1,1,1,1,0,0,0,0,1,1,0,1,0,0,1,0,1,1,1,1,1,1,0,0,0,0,1,1,1,1,1,0,0,0,1,0,0,1,0,1,0,1,0,1,0,1],[1,1,1,1,1,0,0,1,0,1,1,0,1,0,1,1,0,1,0,1,1,0,0,1,0,0,1,1,1,0,1,0,0,1,1,1,1,1,1,0,0,1,1,0,1,0,1,1,0,1,0,0],[1,0,1,0,1,1,0,1,0,1,1,0,1,0,0,1,0,0,0,0,1,1,0,1,0,1,1,1,1,0,1,1,0,1,1,1,1,0,1,0,1,1,1,0,1,0,1,1,1,1,0,1],[1,1,0,0,1,1,0,1,0,0,1,0,0,0,0,1,1,0,1,0,1,1,0,1,0,0,1,0,1,0,1,1,1,0,1,0,1,1,0,1,1,0,0,0,0,1,1,1,1,1,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,1,0,0,1,0,0,1,0,1,1,0,1,0,1,1,0,1,0,0,1,0,0,1,1,1,0,1,0,0,1,0,1,0,0,1,1,0,0,1,0,0,0,0,0,0,1,0,1,0,1],[1,0,0,0,0,1,1,0,1,0,0,1,0,1,0,0,1,0,0,0,0,1,1,0,1,0,1,0,0,0,0,0,1,0,1,1,1,1,0,0,1,1,0,1,1,1,0,1,0,0,1,0],[1,1,0,1,0,0,1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,1,1,0,1,1,1,1,0,1,0,1,1,1,1,1,0,0,0,1,1,0,1,1],[1,1,0,1,1,1,1,0,0,1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,0,1,1,1,1,1,0,0,1,0,1,1,0,1,0,0,0,0,0,1,1,1,0,1,0,0,0],[1,1,0,0,1,0,1,1,1,0,0,0,1,0,1,0,1,1,0,0,1,0,1,0,1,1,1,1,0,0,0,0,1,0,0,1,0,1,0,0,1,1,1,1,1,0,0,1,0,0,1,1],[1,1,1,1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,0,1,1,0,1,0,1,1,0,0,0,1,1,1,0,1,0,1,0,1,0,1,1,1,0,0,0],[1,0,1,0,1,1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,1,1,1,0,0,1,1,1,1,1,1,1,0,1,1,0,1,1,0,1,0,0,1,1,0,0,1,0,1,0,0,1],[1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,1,0,1,1,0,0,1,1,1,0,0,1,0,1,1,1,0,1,1,1,1,1,1,0,0,1,1,0,1,1,0,0,0,1,1,0,0],[1,0,1,1,0,1,0,0,1,0,1,1,1,1,0,0,0,0,1,1,1,1,1,0,1,1,1,0,1,1,1,0,0,1,1,1,0,1,0,0,0,1,1,1,1,1,1,1,0,0,0,1],[1,0,1,1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,0,1,1,0,1,1,1,0,1,0,0,1,0,1,0,0,1,1,0,0,1,0,1,1,0,1,0,1,1,0,1,0,0,0],[1,0,0,1,0,0,0,0,1,1,0,1,0,1,1,0,1,1,0,1,0,0,0,1,0,1,1,0,1,0,0,1,0,0,1,1,0,0,1,1,0,1,0,1,1,0,0,1,0,0,1,1],[1,0,0,1,1,0,1,0,1,1,0,1,0,0,1,0,1,1,0,0,1,1,1,0,0,0,1,1,0,1,1,1,0,1,1,0,1,1,1,0,0,1,0,1,0,0,0,0,1,0,0,0],[1,1,0,1,1,0,1,0,0,1,0,0,1,0,1,0,0,1,0,1,1,0,1,0,1,1,1,1,1,1,1,0,1,0,1,0,1,0,0,0,1,1,1,0,1,0,1,0,1,1,0,1],[1,1,0,0,1,0,0,0,0,1,1,0,1,0,0,0,1,0,1,1,0,0,0,0,1,0,1,0,1,1,1,1,0,0,1,0,1,1,1,1,0,1,1,1,1,0,1,0,0,0,0,0],[1,0,1,0,1,1,0,1,0,0,1,0,1,1,0,1,1,1,0,1,0,0,0,1,0,1,1,1,1,1,1,0,1,0,0,1,0,0,0,1,1,0,1,0,0,1,1,0,1,0,0,1],[1,0,1,0,0,1,0,1,1,1,1,0,0,1,0,1,1,0,0,1,1,1,0,0,1,0,1,0,1,0,1,1,0,0,1,0,1,0,0,1,0,1,1,1,1,1,1,1,1,0,1,0],[1,0,1,0,1,1,0,0,1,0,1,0,1,1,0,0,1,1,0,1,0,1,1,0,0,1,1,1,1,1,0,1,0,1,1,1,1,0,1,0,0,0,0,0,1,0,1,0,1,1,1,1],[1,0,0,1,0,1,1,1,0,0,0,1,1,1,0,1,1,0,1,1,0,0,1,0,0,0,1,0,0,0,1,0,0,1,0,1,0,1,0,1,0,0,1,0,0,0,0,0,0,1,1,0],[1,1,1,0,0,0,1,0,1,1,1,0,1,0,0,1,0,1,0,1,0,1,1,0,0,1,1,0,1,1,0,1,1,1,1,0,1,1,0,1,1,1,0,0,1,0,1,1,0,0,1,1],[1,1,0,1,0,1,1,0,0,1,0,1,0,0,1,1,0,1,0,1,1,0,0,0,1,0,1,0,0,0,1,1,1,0,1,0,0,0,0,0,1,1,1,0,0,1,0,1,0,0,0,0],[1,1,0,0,0,0,1,1,1,1,1,1,1,1,0,1,0,1,0,1,0,0,1,1,0,1,1,0,1,1,0,0,0,1,1,0,0,1,1,0,1,1,1,1,0,0,0,0,0,1,1,1],[1,1,1,0,1,0,0,1,0,0,0,0,0,1,0,0,1,0,1,1,0,0,0,0,0,0,1,0,0,1,1,1,0,1,0,1,1,1,1,0,1,1,1,1,1,1,1,0,0,1,0,0],[1,1,1,0,1,1,0,1,0,1,0,0,1,1,1,0,1,1,1,0,1,1,1,0,0,1,1,1,1,0,1,1,1,1,1,0,1,0,1,1,1,0,1,0,0,0,1,0,0,0,0,1],[1,0,1,0,0,1,0,1,0,0,0,0,0,1,0,1,0,1,0,0,0,1,0,1,1,0,1,0,1,0,1,1,0,1,0,0,0,0,0,0,0,1,0,1,0,0,1,1,0,1,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]]},"ASCII" => {"short" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,1,1,1,1,0,0,1,0,1,1,1,1],[1,0,1,0,0,1,1,0,0,0,0,0,0,0],[1,0,0,0,0,0,0,1,1,0,0,1,0,1],[1,1,0,1,1,1,0,0,1,1,1,0,0,0],[1,0,0,1,1,1,0,0,0,0,1,1,1,1],[1,0,0,1,0,1,1,0,0,1,1,1,1,0],[1,0,0,1,0,0,1,0,0,0,1,0,1,1],[1,0,1,1,1,1,0,0,0,0,0,0,0,0],[1,1,0,1,1,0,0,0,1,1,0,1,1,1],[1,0,1,1,0,0,1,1,0,0,0,0,1,0],[1,0,0,1,0,1,1,0,1,0,1,0,0,1],[1,0,0,1,1,0,1,0,0,1,0,1,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1]],"long" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,1,1,1,1,0,0,1,0,1,0,0,0,0,0,1,0,1,1,1,1,0,0,1,1,1,0,1,0,1,0,0,0,1,0,1,1,1,0,0,1,1,0,1,0,1,0,0,0,1,1],[1,0,1,0,0,1,1,0,0,0,0,1,0,1,1,1,1,0,1,0,0,1,1,0,0,0,1,0,0,0,0,1,1,1,0,0,0,1,0,1,0,1,0,0,0,0,0,1,1,1,0,0],[1,0,0,0,0,0,0,0,1,1,1,0,0,0,1,0,1,0,0,0,0,0,0,0,1,1,1,1,1,0,0,1,0,0,1,1,0,0,0,0,1,0,1,1,1,0,0,1,0,0,1,1],[1,1,0,1,1,1,0,0,1,1,0,1,1,0,0,0,0,1,0,1,1,1,0,0,0,0,1,1,0,1,0,0,0,0,0,1,0,1,1,1,0,0,0,1,0,1,0,0,0,0,0,0],[1,0,0,1,1,0,1,0,1,0,0,0,0,0,1,1,1,0,0,1,1,0,1,1,0,1,1,0,0,0,1,0,1,1,1,1,0,1,0,0,1,1,0,0,0,0,1,0,1,1,1,1],[1,1,0,1,0,0,0,0,0,1,1,1,0,0,1,1,0,1,0,1,0,0,0,1,0,0,1,1,1,1,1,0,1,0,0,1,0,0,0,0,0,1,0,1,1,1,1,0,0,1,0,0],[1,0,1,0,1,1,1,0,0,1,1,0,1,0,1,0,0,0,1,0,1,1,1,1,0,1,1,0,1,0,1,0,0,0,0,0,1,0,1,1,1,1,0,0,1,1,0,0,0,0,0,1],[1,1,0,0,0,1,0,1,0,1,0,0,0,0,0,1,1,1,0,0,1,0,0,1,1,0,1,0,0,0,0,1,0,1,1,1,1,0,1,0,0,1,1,0,0,1,1,0,1,1,1,0],[1,0,1,1,0,0,0,0,1,0,1,1,1,0,0,1,0,0,1,0,0,0,0,0,0,1,1,0,1,1,1,0,0,0,1,0,1,0,0,0,0,0,0,1,0,0,0,1,0,1,1,1],[1,0,0,1,0,1,1,1,0,0,0,1,0,1,0,0,0,0,0,1,0,1,1,1,0,0,1,0,1,1,0,1,1,0,0,0,0,1,0,1,1,1,0,0,0,0,0,1,1,1,0,0],[1,1,1,1,0,1,0,0,1,1,0,0,0,0,1,0,1,1,1,0,0,1,1,0,1,1,1,0,1,0,0,0,0,0,1,1,1,0,0,1,1,0,1,0,0,1,1,1,0,0,1,1],[1,0,0,1,0,0,0,0,0,1,0,1,1,1,1,0,1,1,0,1,0,1,0,0,0,0,1,0,0,1,1,1,0,0,1,1,0,1,0,1,0,0,1,1,0,0,1,0,1,0,1,0],[1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,0,1,0,0,0,1,0,1,1,1,1,1,0,0,1,1,0,1,0,1,0,0,0,1,0,1,0,1,0,0,0,1,0,0,0,0,1],[1,1,1,1,1,0,1,0,0,1,1,0,0,0,0,0,0,1,1,1,0,0,0,1,0,0,1,1,0,1,0,0,0,0,0,1,1,1,0,0,0,0,1,0,1,0,1,0,1,0,0,0],[1,0,1,0,1,0,0,0,0,0,0,0,1,1,1,0,0,1,0,0,1,1,0,0,0,1,1,0,1,0,1,1,1,0,0,1,0,0,1,1,1,0,1,0,0,0,1,0,1,0,0,1],[1,0,0,0,0,1,0,1,1,1,0,0,0,1,0,1,0,0,0,0,0,1,0,1,1,0,1,1,0,0,0,1,0,1,0,0,0,0,0,1,0,0,0,1,1,1,1,1,1,0,1,0],[1,0,1,1,1,0,0,1,1,0,1,1,0,0,0,0,1,0,1,1,1,1,0,1,0,1,1,0,1,1,0,0,0,0,1,0,1,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1],[1,0,1,1,0,1,0,1,0,0,0,1,0,1,1,1,1,0,1,0,0,1,0,0,0,0,1,0,0,1,0,1,1,1,1,0,0,1,1,1,0,1,0,1,0,1,0,0,1,0,1,0],[1,0,1,0,0,0,1,0,1,1,1,1,0,0,1,0,1,0,0,0,0,0,1,0,1,1,1,1,1,1,0,0,1,0,1,1,1,1,0,1,0,1,0,0,0,0,0,0,0,0,1,1],[1,0,0,1,1,1,0,0,1,0,0,1,1,0,0,0,0,1,0,1,1,1,1,0,1,0,1,0,0,1,1,0,0,0,1,0,0,1,0,0,0,1,0,0,0,1,0,0,1,0,0,0],[1,0,0,1,0,0,1,0,0,0,0,0,0,0,1,1,1,0,0,0,1,0,1,0,0,1,1,0,0,0,0,0,1,0,0,0,1,0,0,1,0,0,0,1,1,1,0,1,0,1,1,1],[1,1,0,0,0,0,0,1,0,1,1,1,0,0,1,1,0,1,1,0,0,0,0,1,0,0,1,1,1,1,0,0,0,1,1,1,1,1,1,0,0,0,0,0,1,0,1,1,0,1,0,0],[1,0,1,0,1,1,1,0,0,1,1,0,1,0,1,0,0,0,0,0,1,1,1,0,0,1,1,1,1,0,1,1,1,0,1,1,1,0,0,0,0,0,1,0,1,0,0,1,1,0,0,1],[1,1,1,0,1,1,0,1,0,1,0,0,0,0,0,1,1,1,0,0,1,1,0,1,0,0,1,1,0,0,1,1,1,0,1,0,1,1,1,1,0,0,0,0,1,1,0,1,1,1,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,1,0,1,0,0,0,1,0,1,1,1,0,0,1,1,0,1,0,1,0,0,0,1,1,1,0,1,0,1,0,0,1,0,0,1,0,0,0,0,0,0,1,0,0,0,1,0,0,1,1],[1,0,0,0,0,1,1,1,0,0,0,1,0,1,0,1,0,0,0,0,0,1,1,1,0,0,1,0,0,0,0,0,1,0,1,0,0,1,0,1,0,0,1,1,0,0,1,0,1,0,0,0],[1,1,1,0,0,1,0,0,1,1,0,0,0,0,1,0,1,1,1,0,0,1,0,0,1,1,1,1,1,0,1,0,1,1,0,0,1,0,1,0,1,0,0,1,0,1,1,1,0,0,1,1],[1,1,0,1,0,0,0,0,0,1,0,1,1,1,0,0,0,1,0,1,0,0,0,0,0,0,1,1,1,1,1,0,0,1,0,0,1,1,1,1,0,0,1,1,0,1,0,0,0,0,0,0],[1,0,0,0,1,0,1,1,1,1,0,1,0,0,1,1,0,0,0,0,1,0,1,1,0,1,1,0,0,0,0,0,0,0,1,0,0,1,0,0,0,1,1,1,0,1,1,0,1,1,0,1],[1,1,1,1,1,0,1,0,0,1,0,0,0,0,0,1,0,1,1,1,1,0,0,1,0,0,1,1,0,1,0,0,0,0,0,0,0,1,1,0,0,1,0,0,0,0,0,0,1,0,0,0],[1,0,1,0,1,0,0,0,0,0,1,0,1,1,1,1,0,0,1,0,1,1,1,1,0,1,1,1,1,0,1,0,0,1,1,0,0,0,1,0,0,0,1,0,0,0,1,0,1,0,0,1],[1,0,0,0,0,1,0,1,1,1,1,0,1,0,0,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,1,0,1,0,1,1,0,0,1,1,1,0,1,1,0,1,1,0,0,0,1,0],[1,0,1,1,1,0,0,0,1,0,1,0,0,0,0,0,0,0,1,0,0,0,1,0,1,1,1,1,0,0,1,1,1,1,1,1,0,1,0,1,0,1,0,0,0,1,0,1,1,0,0,1],[1,0,1,1,0,1,1,0,0,0,0,1,0,1,1,1,0,0,0,1,1,0,1,1,1,0,1,1,1,0,1,1,0,1,0,1,1,0,0,1,0,0,0,1,0,0,1,1,0,0,1,0],[1,0,1,0,0,0,0,0,1,1,1,0,0,1,1,0,1,1,1,0,0,0,0,0,1,1,1,0,1,0,1,0,0,1,0,1,0,1,1,0,1,1,0,0,0,1,1,1,1,0,1,1],[1,0,0,1,1,1,0,0,1,1,0,1,0,1,0,0,1,1,0,0,1,1,1,1,0,0,1,0,0,0,0,0,1,0,0,1,1,1,0,1,0,0,0,0,1,1,1,1,0,0,0,0],[1,0,0,1,1,0,1,0,1,0,0,0,1,0,1,0,1,0,0,1,1,1,0,1,1,1,1,0,1,1,0,1,0,1,0,1,0,1,0,1,1,1,0,0,1,1,1,0,1,1,1,1],[1,1,0,1,0,0,0,0,0,1,1,1,0,0,0,0,1,0,1,1,0,1,0,0,1,0,1,0,0,1,1,0,0,0,1,1,1,1,1,0,0,1,1,0,0,0,0,0,0,1,0,0],[1,0,1,0,1,1,1,0,0,1,0,0,1,1,1,0,1,1,0,0,0,1,1,0,1,1,1,1,1,1,1,1,1,0,0,0,0,0,0,1,0,0,0,0,0,1,0,1,0,0,0,1],[1,1,0,0,0,1,0,1,0,0,0,0,0,1,0,1,1,0,0,0,0,0,1,1,0,0,1,0,0,1,0,0,0,1,0,0,1,0,0,0,0,0,1,1,1,1,1,1,1,1,1,0],[1,0,1,1,0,0,0,0,1,0,1,1,0,0,0,0,1,1,0,1,0,1,1,1,0,1,1,1,1,0,1,1,0,1,1,1,0,0,0,0,1,1,0,0,0,0,1,0,0,1,1,1],[1,0,0,1,0,1,1,1,1,0,0,1,1,1,0,1,1,0,1,1,0,0,0,0,0,0,1,1,0,1,1,0,0,1,1,1,1,0,0,1,0,1,0,1,0,1,1,1,0,1,0,0],[1,1,1,1,0,0,1,0,1,1,1,1,1,0,0,1,0,0,0,0,1,0,0,0,1,1,1,1,1,0,1,1,1,1,0,1,0,1,1,1,0,1,0,1,1,1,1,1,1,0,1,1],[1,0,0,1,1,0,0,0,1,0,0,1,0,0,1,0,0,0,1,1,1,0,0,1,0,0,1,0,1,0,0,0,0,0,0,1,1,0,0,1,1,1,1,1,1,1,0,0,1,0,1,0],[1,0,0,0,0,0,1,0,0,0,1,1,1,1,0,0,1,1,1,1,1,0,1,1,0,1,1,1,0,0,0,1,0,1,0,0,0,1,1,0,1,0,0,1,1,1,1,1,1,0,0,1],[1,1,1,1,0,0,0,1,0,0,0,0,0,1,0,0,0,1,1,1,1,1,0,1,1,0,1,1,0,1,0,0,0,0,1,1,1,1,0,0,1,0,1,0,1,0,1,0,1,0,0,0],[1,1,1,0,1,1,1,0,0,1,0,0,1,1,1,0,0,1,0,0,1,1,0,1,0,1,1,0,0,0,1,1,1,0,0,0,1,0,1,0,0,0,0,1,0,0,1,1,1,0,0,1],[1,1,0,0,0,1,0,1,0,0,0,0,1,0,0,1,0,1,1,0,0,1,0,1,0,0,1,0,0,0,1,1,0,1,0,1,1,0,0,0,0,1,0,0,1,0,1,1,0,1,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]]},"AUTO" => {"short" => [[1,0,1,0,1,0,1,0,1,0,1,0],[1,1,0,1,0,0,0,1,1,1,1,1],[1,1,0,0,1,1,0,0,0,1,0,0],[1,1,1,0,1,1,0,0,0,1,1,1],[1,0,1,0,1,1,1,0,0,1,0,0],[1,0,0,0,0,1,1,0,1,1,0,1],[1,1,0,0,1,0,0,1,1,1,0,0],[1,1,0,1,0,1,1,0,0,1,1,1],[1,0,1,1,0,0,0,1,1,1,1,0],[1,1,1,1,0,1,1,0,1,1,0,1],[1,1,1,1,1,0,0,1,1,0,1,0],[1,1,1,1,1,1,1,1,1,1,1,1]],"long" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1],[1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,0,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,0,0],[1,1,1,0,1,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,1,1],[1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,1,0,0],[1,0,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,0,0,1,1,1],[1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,0,0,1,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,0],[1,1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,1,1,1,1],[1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,0,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,1,0,1,1,1,1,0],[1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,1,1,0,1,1,0,1,0,1,0,0,0,1,1,0,0,0,0,0,0,1,0,1],[1,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,0,1,0,1,1,1,1,0,0,1,1,0,0,0,0,1,0,0,1,1,0,1,0],[1,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,1,1,1,0,0,1,1,1,1,0,1,1,0,0,1,0,1,0,1,0,1,1,0,1],[1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,1,0,0,1,1,1,0,1,1,0,1,1,0,1,0,1,1,1,1,1,1,0,0,1,0,0,0,1,0],[1,0,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,1,1,0,1,1,1,1,0,0,1,1,0,1,0,0,0,0,0,0,0,0,1],[1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,0,1,1,1,1,1,1,0,1,0,0,1,0,1,0,0,1,0,0,1,0],[1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,0,1,1,1,1,1,1,0,0,0,1,1,1,0,1,0,1,1,0,1,1,1,0,1,1,0,1,1],[1,1,0,1,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,0,1,1,1,0,1,0,0,0,0,1,0,0,0,1,0,1,0,0,0,1,1,0],[1,1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,1,1,0,0,1,1,1,0,1,0,1,1,1,0,1,1,1,0,1,0,1,0,1],[1,0,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,0,1,1,0,0,0,1,1,0,1,1,0,0,1,1,1,1,0,1,1,1,0,0],[1,0,0,0,1,1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,1,1,0,0,0,1,1,0,0,0,0,0,1,0,1,1,1,1,1,1,1,1],[1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,0,1,1,1,1,0,0,0,1,1,1,1,1,1,1,1,0,0,1,1,1,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,0,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,1,0,0,0,0,0,1,1,1,0,0,1,1,1,1,0,0,1,1,0,1],[1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,0,1,0,0,1,1,1,1,1,0,0,1,0,0,0,0,1,0,0,0,1,1,0],[1,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,1,0,0,1,1,1,0,0,1,1,0,0,0,1,0,1,1,0,1,0,1,1],[1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,0,0,0,1,0,1,0,1,0,0,0,0,0,1,0,0,0,0,0,0,1,1,1,1,0],[1,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,1,0,0,1,1,1,0,0,1,1,1,1,0,0,0,0,0,1,1,1,0,0,0,0,0,1],[1,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,1,0,0,1,0,1,0,0,0,1,1,0,1,0,1,0,0,0,0,0,1,1,1,1,0],[1,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,1,1,1,0,1,1,1,0,0,0,0,0,1,1,1,0,1,0,0,0,0,0,0,0,1,1,0,1],[1,0,1,1,1,1,0,1,1,0,1,0,1,1,1,1,1,0,1,0,0,0,1,1,1,0,1,0,1,1,0,1,1,1,0,0,1,1,1,0,0,1,0,0],[1,0,1,1,1,1,0,1,1,1,1,0,0,1,0,1,1,1,1,1,1,1,1,1,0,0,1,0,0,1,1,1,0,1,0,0,1,0,1,0,0,1,1,1],[1,0,1,0,0,0,1,1,1,1,1,1,0,0,1,1,0,1,0,0,1,0,1,1,1,0,0,0,1,0,0,1,1,1,1,0,1,1,1,0,0,1,1,0],[1,1,1,1,1,0,0,0,1,1,1,1,1,0,0,0,0,1,0,1,1,1,1,1,1,0,0,1,0,0,0,1,0,1,1,0,1,0,0,1,1,0,0,1],[1,0,0,1,1,1,1,0,1,0,0,1,1,1,0,1,0,0,0,0,1,0,1,1,0,1,1,1,1,0,1,0,1,1,0,0,0,0,1,1,0,1,0,0],[1,1,0,1,0,0,0,1,1,1,0,1,0,0,1,0,1,0,0,0,1,1,1,0,0,0,0,0,1,0,0,1,1,0,0,0,1,0,1,0,1,0,1,1],[1,1,0,0,1,1,0,0,0,1,0,0,0,1,1,0,0,1,1,0,1,0,1,1,0,1,1,0,0,0,1,1,1,1,0,0,0,0,0,1,0,0,0,0],[1,1,1,0,1,1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,1,0,0,0,0,0,1,0,0,1,0,0,0,1,1,0,0,1,0,1,1,1],[1,0,1,0,1,1,1,1,1,0,0,1,1,0,1,0,0,1,1,0,0,0,1,0,1,0,0,1,1,1,1,0,0,0,0,0,1,0,0,1,0,1,1,0],[1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,0,0,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,0,0,1,1,1,1,1,1,0,1,1,1],[1,1,1,1,0,0,1,1,1,1,1,0,1,1,1,0,0,1,1,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,0,0,0,1,1,1,0,1,0],[1,1,1,1,1,0,0,0,0,0,0,1,0,0,1,1,0,1,1,1,1,1,1,1,0,1,0,0,1,1,0,0,0,0,1,0,0,1,1,0,1,1,1,1],[1,0,0,1,1,1,1,0,1,1,0,1,0,0,1,1,1,1,1,1,1,0,1,0,1,0,1,1,0,0,1,1,1,1,0,1,1,1,1,1,1,0,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]]},"C40" => {"short" => [[1,0,1,0,1,0,1,0,1,0,1,0],[1,1,0,1,0,0,0,1,1,1,1,1],[1,1,0,0,1,1,0,0,0,1,0,0],[1,1,1,0,1,1,0,0,0,1,1,1],[1,0,1,0,1,1,1,0,0,1,0,0],[1,0,0,0,0,1,1,0,1,1,0,1],[1,1,0,0,1,0,0,1,1,1,0,0],[1,1,0,1,0,1,1,0,0,1,1,1],[1,0,1,1,0,0,0,1,1,1,1,0],[1,1,1,1,0,1,1,0,1,1,0,1],[1,1,1,1,1,0,0,1,1,0,1,0],[1,1,1,1,1,1,1,1,1,1,1,1]],"long" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1],[1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,0,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,0,0],[1,1,1,0,1,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,1,1],[1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,1,0,0],[1,0,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,0,0,1,1,1],[1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,0,0,1,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,0],[1,1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,1,1,1,1],[1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,0,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,1,0,1,1,1,1,0],[1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,1,1,0,1,1,0,1,0,1,0,0,0,1,1,0,0,0,0,0,0,1,0,1],[1,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,0,1,0,1,1,1,1,0,0,1,1,0,0,0,0,1,0,0,1,1,0,1,0],[1,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,1,1,1,0,0,1,1,1,1,0,1,1,0,0,1,0,1,0,1,0,1,1,0,1],[1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,1,0,0,1,1,1,0,1,1,0,1,1,0,1,0,1,1,1,1,1,1,0,0,1,0,0,0,1,0],[1,0,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,1,1,1,1,1,1,0,1,1,1,1,0,0,1,1,0,1,0,0,0,0,0,0,0,0,1],[1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,0,1,1,1,1,1,1,0,1,0,0,1,0,1,0,0,1,0,0,1,0],[1,0,0,1,1,1,1,0,1,1,0,1,0,1,1,1,0,1,1,1,1,1,1,0,0,0,1,1,1,0,1,0,1,1,0,1,1,1,0,1,1,0,1,1],[1,1,0,1,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,0,1,1,1,0,1,0,0,0,0,1,0,0,0,1,0,1,0,0,0,1,1,0],[1,1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,0,1,0,1,1,0,0,1,1,1,0,1,0,1,1,1,0,1,1,1,0,1,0,1,0,1],[1,0,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,0,1,1,0,0,0,1,1,0,1,1,0,0,1,1,1,1,0,1,1,1,0,0],[1,0,0,0,1,1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,1,1,0,0,0,1,1,0,0,0,0,0,1,0,1,1,1,1,1,1,1,1],[1,1,1,0,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,0,1,0,1,1,1,1,0,0,0,1,1,1,1,1,1,1,1,0,0,1,1,1,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,0,1,1,1,1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,1,1,0,0,0,0,0,1,1,1,0,0,1,1,1,1,0,0,1,1,0,1],[1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,0,1,0,0,1,1,1,1,1,0,0,1,0,0,0,0,1,0,0,0,1,1,0],[1,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,1,0,0,1,1,1,0,0,1,1,0,0,0,1,0,1,1,0,1,0,1,1],[1,1,1,0,1,1,1,1,0,0,0,1,1,1,1,0,1,0,0,0,0,0,1,0,1,0,1,0,0,0,0,0,1,0,0,0,0,0,0,1,1,1,1,0],[1,1,1,0,0,0,1,1,0,1,0,1,0,0,0,1,1,1,1,0,0,1,1,1,0,0,1,1,1,1,0,0,0,0,0,1,1,1,0,0,0,0,0,1],[1,1,1,1,1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,1,0,0,1,0,1,0,0,0,1,1,0,1,0,1,0,0,0,0,0,1,1,1,1,0],[1,1,0,0,1,0,0,1,1,1,1,0,1,1,0,1,1,1,1,0,1,1,1,0,0,0,0,0,1,1,1,0,1,0,0,0,0,0,0,0,1,1,0,1],[1,0,1,1,1,1,0,1,1,0,1,0,1,1,1,1,1,0,1,0,0,0,1,1,1,0,1,0,1,1,0,1,1,1,0,0,1,1,1,0,0,1,0,0],[1,0,1,1,1,1,0,1,1,1,1,0,0,1,0,1,1,1,1,1,1,1,1,1,0,0,1,0,0,1,1,1,0,1,0,0,1,0,1,0,0,1,1,1],[1,0,1,0,0,0,1,1,1,1,1,1,0,0,1,1,0,1,0,0,1,0,1,1,1,0,0,0,1,0,0,1,1,1,1,0,1,1,1,0,0,1,1,0],[1,1,1,1,1,0,0,0,1,1,1,1,1,0,0,0,0,1,0,1,1,1,1,1,1,0,0,1,0,0,0,1,0,1,1,0,1,0,0,1,1,0,0,1],[1,0,0,1,1,1,1,0,1,0,0,1,1,1,0,1,0,0,0,0,1,0,1,1,0,1,1,1,1,0,1,0,1,1,0,0,0,0,1,1,0,1,0,0],[1,1,0,1,0,0,0,1,1,1,0,1,0,0,1,0,1,0,0,0,1,1,1,0,0,0,0,0,1,0,0,1,1,0,0,0,1,0,1,0,1,0,1,1],[1,1,0,0,1,1,0,0,0,1,0,0,0,1,1,0,0,1,1,0,1,0,1,1,0,1,1,0,0,0,1,1,1,1,0,0,0,0,0,1,0,0,0,0],[1,1,1,0,1,1,0,1,1,1,1,0,0,1,1,0,0,0,1,1,0,1,1,0,0,0,0,0,1,0,0,1,0,0,0,1,1,0,0,1,0,1,1,1],[1,0,1,0,1,1,1,1,1,0,0,1,1,0,1,0,0,1,1,0,0,0,1,0,1,0,0,1,1,1,1,0,0,0,0,0,1,0,0,1,0,1,1,0],[1,1,1,0,0,1,0,1,1,1,0,0,0,1,0,0,0,1,0,1,0,1,1,1,0,1,1,1,1,0,0,0,0,0,1,1,1,1,1,1,0,1,1,1],[1,1,1,1,0,0,1,1,1,1,1,0,1,1,1,0,0,1,1,1,1,0,1,0,1,0,0,0,1,1,1,1,1,1,0,0,0,0,1,1,1,0,1,0],[1,1,1,1,1,0,0,0,0,0,0,1,0,0,1,1,0,1,1,1,1,1,1,1,0,1,0,0,1,1,0,0,0,0,1,0,0,1,1,0,1,1,1,1],[1,0,0,1,1,1,1,0,1,1,0,1,0,0,1,1,1,1,1,1,1,0,1,0,1,0,1,1,0,0,1,1,1,1,0,1,1,1,1,1,1,0,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]]},"BASE256" => {"short" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,0,0,1,1,0,1,0,0,1,0,0,0,1],[1,1,1,0,0,0,1,1,0,1,0,0,1,0],[1,0,1,0,1,1,1,0,1,1,0,0,1,1],[1,0,1,1,0,1,0,1,1,1,0,1,0,0],[1,1,0,0,1,0,1,1,1,1,1,1,1,1],[1,1,1,0,1,1,0,0,0,0,0,0,0,0],[1,1,1,1,1,0,0,1,1,0,0,1,1,1],[1,0,1,1,1,1,1,1,0,0,1,1,0,0],[1,1,1,1,1,0,1,0,1,0,0,1,1,1],[1,0,1,0,1,1,1,1,0,1,1,0,0,0],[1,1,1,1,1,1,0,0,0,1,0,1,0,1],[1,0,0,0,0,1,1,0,0,1,1,0,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1]],"long" => [[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,1,1,1,1,0,1,0,0,1,1,0,0,0,0,1,0,0,0,1,1,0,1,0,1,1,1,1,1,0,1,1,1,0,1,0,0,0,0,1,1,0,0,0,1,1,1,0,1,1,1,1],[1,1,0,0,0,0,1,1,0,0,1,1,1,0,0,0,1,0,0,1,1,1,1,0,1,0,1,0,0,1,1,0,0,1,0,1,1,1,1,1,1,0,0,0,1,1,0,0,1,1,0,0],[1,0,0,0,1,1,1,0,1,1,1,0,1,0,1,0,1,1,0,0,0,0,0,0,0,1,1,0,0,0,0,1,0,1,0,1,0,0,1,1,0,1,1,1,0,0,1,0,0,1,1,1],[1,0,1,1,0,1,0,0,0,1,0,0,0,0,1,1,0,0,0,1,0,1,0,1,1,0,1,0,0,0,1,0,0,1,0,0,1,0,0,1,1,0,0,0,1,0,1,1,1,1,0,0],[1,1,0,0,1,1,0,1,1,1,1,1,1,1,1,0,1,1,0,1,0,1,0,0,1,1,1,1,0,1,1,0,0,1,0,1,0,1,1,0,0,1,1,1,1,1,0,1,0,1,1,1],[1,1,1,0,1,1,1,1,1,1,1,0,1,0,1,0,1,1,1,0,1,0,1,1,1,0,1,0,1,0,1,0,1,1,0,0,1,0,1,1,0,1,0,1,0,0,1,0,0,0,0,0],[1,1,1,0,0,1,0,0,0,1,0,0,1,0,1,1,1,0,0,0,0,1,0,0,1,1,1,1,0,0,1,0,1,1,1,0,0,1,0,1,1,1,0,0,1,1,0,0,0,1,0,1],[1,0,1,1,0,0,0,1,0,0,0,0,0,0,1,0,1,0,0,0,1,0,0,1,1,0,1,0,1,1,1,0,1,0,1,1,0,1,0,0,1,1,0,0,0,1,1,0,1,1,1,0],[1,1,0,1,1,1,1,0,0,1,0,0,0,0,0,0,1,0,1,0,1,0,0,0,1,1,1,1,1,0,1,0,1,1,1,1,1,1,1,0,0,1,1,0,0,0,1,0,0,1,1,1],[1,0,1,0,0,0,0,0,0,1,0,1,0,0,1,1,0,1,0,1,1,1,0,0,1,0,1,1,0,0,0,1,1,1,0,1,1,1,0,1,1,0,1,0,0,1,0,1,1,1,0,0],[1,0,0,0,0,0,1,0,0,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0,1,1,1,0,1,0,0,1,0,0,1,1,0,0,1,1,0,0,1,0,0,0,1,1,0,0,0,1],[1,0,1,0,0,0,1,0,1,0,1,0,1,1,1,0,1,1,0,1,1,0,0,0,1,0,1,0,1,1,0,1,1,1,0,1,1,1,0,0,0,1,1,1,0,0,1,0,1,0,0,0],[1,0,0,1,0,1,0,1,0,0,1,0,1,0,1,0,0,1,1,0,1,1,0,0,0,1,1,0,1,0,0,1,1,1,0,1,1,1,1,1,1,0,0,1,0,0,1,1,0,1,1,1],[1,0,0,1,1,0,0,1,0,1,1,1,1,0,0,0,0,0,1,0,0,1,1,1,0,0,1,1,1,0,1,0,1,1,1,0,1,1,1,1,0,0,1,0,1,0,1,0,1,0,0,0],[1,0,1,1,1,0,1,1,0,0,0,1,0,1,1,0,0,1,0,0,0,0,1,1,1,1,1,1,0,0,1,0,1,0,1,0,1,0,1,1,1,0,1,0,0,0,0,1,0,1,1,1],[1,1,0,1,0,0,1,1,0,1,1,1,1,0,1,0,1,0,1,1,0,0,0,0,1,0,1,0,1,0,0,0,0,0,0,0,1,0,1,0,0,0,0,1,1,0,0,1,1,1,0,0],[1,0,1,0,0,1,0,1,0,0,0,0,1,0,0,1,1,1,0,0,1,1,0,1,1,1,1,1,0,1,0,1,1,1,0,1,0,0,1,0,1,1,1,0,1,0,1,0,0,0,0,1],[1,0,1,0,0,1,1,1,0,0,1,1,0,0,1,1,1,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,0,1,0,1,1,1,1,1,0,1,0,1,0,0,0,1,1,0,0,0],[1,0,1,0,1,0,0,1,0,0,1,0,1,1,0,1,1,0,1,0,1,0,1,0,0,1,1,1,1,0,0,0,1,0,0,0,1,0,0,1,0,1,1,0,0,1,0,0,1,1,0,1],[1,1,0,0,1,1,0,0,1,0,1,1,1,1,0,1,1,0,0,0,0,0,0,1,0,0,1,0,0,1,0,1,0,1,1,0,0,1,0,0,0,1,0,1,1,1,1,0,0,1,1,0],[1,1,0,0,1,1,1,0,0,1,0,0,1,0,1,0,0,0,1,1,1,0,1,1,0,1,1,0,0,1,1,1,0,0,0,0,1,0,0,1,0,1,0,0,0,1,0,0,1,1,1,1],[1,0,1,1,1,1,0,0,0,1,0,1,1,1,0,0,1,1,1,1,1,1,1,1,1,0,1,1,1,1,1,0,0,0,1,1,1,1,1,0,1,1,1,0,1,0,1,1,1,1,1,0],[1,1,1,0,1,1,1,1,0,1,1,1,1,0,0,1,0,1,0,1,1,0,1,1,1,1,1,1,0,1,1,1,1,1,1,1,1,0,1,0,0,1,0,0,1,1,0,0,1,1,0,1],[1,0,1,0,1,1,1,1,1,1,0,0,0,1,0,1,1,0,0,1,0,1,1,0,0,0,1,0,1,1,1,0,1,0,1,0,1,0,1,1,0,1,1,1,0,0,1,1,1,0,0,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1],[1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0,1,0],[1,1,1,0,0,0,1,0,1,0,0,1,1,0,1,0,0,0,1,0,1,1,0,0,0,1,1,0,1,1,1,1,0,1,0,1,0,0,1,1,1,0,1,0,1,0,1,0,1,1,0,1],[1,0,0,0,1,0,1,0,1,1,1,1,1,1,1,1,1,0,1,1,0,0,0,0,1,0,1,1,0,0,0,0,1,0,1,0,1,0,0,1,1,1,0,1,0,0,0,0,1,1,0,0],[1,1,0,1,0,1,0,1,0,0,0,1,1,1,0,1,1,0,0,1,1,0,1,1,1,1,1,1,0,1,1,0,1,1,0,0,0,0,0,1,0,0,0,0,0,0,1,0,1,0,0,1],[1,0,0,0,1,1,1,1,0,0,1,0,1,1,0,0,0,0,1,0,0,1,0,0,1,0,1,0,1,1,1,0,0,0,0,1,1,0,0,0,0,1,1,1,0,0,1,1,1,1,1,0],[1,0,1,0,0,0,0,0,1,0,1,0,0,0,0,1,0,0,1,1,0,0,0,0,0,1,1,0,0,0,0,0,1,1,1,1,0,0,0,0,0,0,0,0,1,1,1,0,0,0,0,1],[1,0,0,0,0,0,1,1,0,0,0,0,0,1,0,1,0,1,1,0,0,1,1,1,0,0,1,1,0,1,0,0,1,0,0,1,1,0,0,1,1,1,0,1,1,1,1,1,1,0,1,0],[1,1,1,0,1,0,0,0,1,0,1,1,0,1,0,0,0,0,1,1,0,0,0,0,0,1,1,1,1,1,1,0,1,0,1,0,0,0,1,1,0,1,0,0,0,0,1,1,0,1,1,1],[1,0,0,1,0,1,1,1,0,0,1,1,0,0,1,1,0,1,1,1,0,1,1,1,0,0,1,0,1,0,1,1,1,1,1,1,0,1,1,1,1,1,0,1,0,1,1,1,0,0,0,0],[1,1,1,1,1,1,0,0,0,1,1,0,1,1,1,0,0,0,0,1,1,1,1,0,1,1,1,0,0,0,0,0,1,0,1,1,1,0,1,1,0,0,1,0,0,1,1,0,0,0,1,1],[1,0,1,1,0,1,0,0,1,1,1,1,0,1,0,0,0,0,0,0,1,0,1,1,1,0,1,1,0,1,1,1,0,1,1,0,1,0,1,1,1,0,0,1,1,0,0,0,1,1,0,0],[1,1,1,1,0,1,0,0,1,0,0,1,1,1,0,0,1,1,0,1,0,0,0,0,1,1,1,0,0,0,1,1,1,0,0,0,1,0,0,1,1,0,1,1,0,0,1,1,1,0,1,1],[1,1,1,1,1,1,0,1,0,1,0,0,0,1,1,1,1,0,0,0,1,1,1,1,0,0,1,0,1,1,1,0,1,1,1,0,1,0,1,1,1,1,1,0,0,1,1,1,0,0,0,0],[1,0,1,0,1,1,1,0,1,0,0,0,0,1,1,1,1,0,0,1,1,0,0,1,1,1,1,0,1,1,0,0,1,1,0,1,0,1,0,0,1,0,0,1,0,1,1,0,0,1,0,1],[1,1,0,0,0,0,1,0,1,0,0,1,0,1,0,0,1,0,1,1,0,1,1,1,1,0,1,0,0,0,0,1,0,0,1,0,1,0,1,1,1,1,1,0,1,1,1,1,1,0,0,0],[1,1,1,0,1,1,1,1,1,0,1,0,1,0,1,1,1,1,0,1,0,1,1,0,1,1,1,0,0,1,1,1,0,0,0,1,1,1,0,0,1,0,0,1,0,0,0,1,1,0,1,1],[1,0,0,0,0,0,0,0,1,0,0,0,0,1,0,1,1,0,0,1,0,1,0,1,1,0,1,1,1,0,0,1,1,0,0,1,0,1,1,1,0,0,0,1,1,1,1,1,0,1,0,0],[1,1,0,0,1,0,1,1,0,1,0,1,1,0,0,0,1,1,1,1,1,1,0,0,1,1,1,1,1,1,0,0,0,1,0,0,1,1,0,0,0,0,0,1,0,1,1,1,1,1,0,1],[1,1,0,1,1,1,1,0,1,1,1,0,1,1,0,1,1,1,1,0,1,1,0,0,0,0,1,0,1,0,1,0,0,0,1,0,0,1,1,0,0,0,0,1,1,0,1,0,1,1,1,0],[1,0,1,1,0,0,1,0,0,1,1,1,1,0,0,1,1,0,0,1,1,1,1,0,0,1,1,0,1,0,0,1,1,1,0,1,1,0,0,1,0,0,0,0,1,1,1,0,0,0,0,1],[1,0,0,1,1,0,1,1,0,1,0,1,0,0,1,1,0,1,0,1,1,0,0,0,1,0,1,0,0,1,1,1,0,1,0,0,1,1,1,1,0,0,1,1,1,0,0,0,0,1,0,0],[1,1,1,0,0,1,0,1,0,1,1,1,1,1,0,0,1,1,0,0,1,0,0,1,1,1,1,0,1,1,1,1,0,1,0,1,1,1,1,1,0,1,1,0,1,0,1,1,0,0,0,1],[1,1,0,1,0,0,0,0,0,0,0,0,0,1,0,0,0,0,1,0,0,0,1,0,1,0,1,0,0,1,0,0,0,0,1,0,0,1,0,1,1,0,0,1,0,0,0,1,0,0,1,0],[1,1,0,0,0,1,0,0,0,1,0,1,0,1,1,0,0,0,0,1,0,0,0,0,0,1,1,0,0,0,1,1,0,0,1,1,1,1,1,1,1,1,1,0,0,1,1,0,1,0,1,1],[1,0,0,0,0,1,1,0,0,0,0,0,0,1,0,0,0,0,1,0,1,0,0,0,0,0,1,1,1,0,1,1,1,1,0,1,1,0,1,0,0,1,1,0,1,1,0,1,0,1,1,0],[1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1,1]]}};
}
