#!perl

use strict;

use Test::More tests => 8000;
use Test::Number::Delta within => 0.0001;

use App::SourcePlot::Source;
use Astro::Telescope;

my $jcmt = new Astro::Telescope('JCMT');

my @col = qw/LST EL AZ PA ELEX ELEY AZX AZY/;
my $n = 0;

foreach (<DATA>) {
  chomp;
  my ($date, $time, $ra, $dec, $sys, @ref) = split;
  $n ++;

  my $s = new App::SourcePlot::Source('test', $ra, $dec, $sys);
  my @out = $s->calcPoint($date, $time, $jcmt);

  for (my $i = 0; $i < 8; $i ++) {
    delta_ok($out[$i], $ref[$i], $col[$i] . '_' . $n);
  }
}

# The following test data was created from App::SourcePlot::Source before
# conversion to using Astro::Coords.  However it was necessary first to fix
# a bug in the original Source module's handling of negative declinations.
# This data should help check that the calculations have not inadvertently
# been changed.  The times, dates and coordinates are all randomly selected.

__DATA__
2055/03/04 22:13:09 08:13:39 +86:59:06 RB 2.2842652269247 23.1963968649055 0.138659962188263 -177.784357159175 -1.159818764769 -29.9775719569296 29.9775719569296 -1.159818764769
2019/05/12 04:52:41 23:09:11 -12:42:52 RJ 5.19725469367338 31.264806780372 119.36583460725 -57.1543681387552 -25.2040471900258 16.2713246308637 -16.2713246308637 -25.2040471900258
2054/09/19 14:52:16 12:05:52 +73:05:05 RJ 3.78280947618005 33.5764150420331 348.352531103765 140.085529900348 19.2493007803918 -23.0100938604346 23.0100938604346 19.2493007803918
2050/02/19 21:59:09 21:07:19 -25:19:16 RB 2.00312362285619 -67.6981942266085 252.486906794875 98.4468819019656 29.6745741093172 -4.40677335821101 4.40677335821101 29.6745741093172
2057/08/18 09:01:48 04:24:48 -12:52:24 RB 1.70361019619126 46.0143907607019 224.90479563144 42.8892411743806 20.4174990690424 21.9801212864182 -21.9801212864182 20.4174990690424
2071/08/19 05:05:19 06:01:01 -84:16:35 RB 0.679491271379108 -15.7848752480222 175.694077549984 -44.7951490307896 -21.1372239407653 21.2889117635904 -21.2889117635904 -21.1372239407653
1989/08/07 03:35:42 03:52:35 -37:41:35 RB 0.0785816868058791 13.3814542152103 138.762327897675 -51.4797386522718 -23.4716390939325 18.6837404778641 -18.6837404778641 -23.4716390939325
2004/09/09 12:11:48 09:41:09 -01:07:52 RJ 2.91058081749989 60.3162770087633 227.514111089557 43.9385368630972 20.8165893172886 21.6025371008904 -21.6025371008904 20.8165893172886
2022/01/18 08:16:09 08:01:59 +20:01:35 RJ 4.13107980788788 -15.6691373884092 298.559171664722 61.5351564130701 26.3732918853947 14.2985829762174 -14.2985829762174 26.3732918853947
2075/07/07 22:35:19 22:42:06 -25:39:16 RJ 4.53426650546484 -1.30509138030916 116.451493837083 -68.6352510471891 -27.9384038676185 10.9291165850603 -10.9291165850603 -27.9384038676185
1982/05/18 23:05:11 08:06:01 -18:42:11 RB 3.79659364555757 -11.3759679617724 253.924537699855 72.7279303324792 28.6471694774393 8.9072824660986 -8.9072824660986 28.6471694774393
2047/10/19 01:39:19 11:13:16 -51:07:52 RB 0.824393780320757 -35.2357373947847 140.012685286977 -102.937053825399 -29.2384983599713 -6.71641374946154 6.71641374946154 -29.2384983599713
2034/01/07 07:52:09 00:16:52 +58:01:09 RJ 3.83852837574004 -6.68622482893858 17.8256918362255 -33.1406468728744 -16.4008835829586 25.1199326770244 -25.1199326770244 -16.4008835829586
2062/05/07 08:07:09 02:13:24 -29:24:09 RB 5.97223743113089 19.7993753152146 132.491607239295 -52.39457212742 -23.766955144705 18.3066065438022 -18.3066065438022 -23.766955144705
2013/06/25 08:05:07 11:19:09 +29:16:52 RJ 0.520867710110055 -27.6900951237231 39.1962109418883 -42.9324848588729 -20.4340826316591 21.9647050288077 -21.9647050288077 -20.4340826316591
2061/05/23 09:39:07 00:09:42 +03:05:16 RB 0.370822131604679 66.599694738498 228.697972692424 45.0892023981297 21.246204031051 21.1801514222858 -21.1801514222858 21.246204031051
2047/10/01 07:52:07 04:41:06 -51:39:19 RJ 2.14584854083744 5.24815937692063 209.632957467479 48.4336075518079 22.445621880256 19.9046240458988 -19.9046240458988 22.445621880256
2022/12/19 22:13:19 17:39:06 +16:48:24 RJ 0.990481567377875 -44.0440314559213 320.887496262365 38.3090210632999 18.5970775499753 23.5403633489418 -23.5403633489418 18.5970775499753
2026/05/09 08:59:19 14:01:13 -09:41:11 RB 6.23007843684939 -55.4653854044252 281.328614620518 69.5218517159263 28.1041705425689 10.4955037094083 -10.4955037094083 28.1041705425689
2025/10/04 03:09:39 09:11:59 +09:01:35 RB 0.967174413190927 8.92478506789861 83.8921223149934 -71.1428992642963 -28.3898286422703 9.6962688526325 -9.6962688526325 -28.3898286422703
1996/03/09 03:35:52 08:16:06 -09:09:41 RJ 3.77009681396724 -4.97889225473213 262.047242303274 70.6856990692714 28.3115527903629 9.92249860662595 -9.92249860662595 28.3115527903629
2042/03/09 00:52:24 08:01:09 +03:06:11 RB 3.05240626269176 35.2434344728552 259.035181923519 67.6242634777371 27.7412197639577 11.4203645304257 -11.4203645304257 27.7412197639577
2080/08/19 04:11:52 22:05:06 -01:09:05 RJ 0.459739188235092 33.2379458014427 255.346103030218 65.5362995881234 27.3067144810331 12.4234996780834 -12.4234996780834 27.3067144810331
2080/12/12 21:06:41 02:41:42 +25:01:41 RJ 0.594973941720127 81.1192064629774 49.9561093965669 -127.148983438917 -23.9120382438331 -18.116689184988 18.116689184988 -23.9120382438331
2010/07/14 14:35:42 08:11:19 -03:35:24 RB 2.5519471759381 57.5764426277999 225.732455320905 42.4635762268951 20.2536411912854 22.1312001141978 -22.1312001141978 20.2536411912854
2074/09/18 20:35:42 19:48:13 +37:39:07 RB 5.27088682649373 71.5432969945736 350.443080025144 168.571252748401 5.94447453134093 -29.4051563938408 29.4051563938408 5.94447453134093
2080/12/23 09:48:19 08:24:19 -18:59:52 RJ 4.09935381866834 -22.6148789340743 256.718257227912 75.902827995463 29.0965211523655 7.30701422127733 -7.30701422127733 29.0965211523655
2078/09/09 12:39:52 10:01:59 -86:16:35 RJ 3.03471680908582 -16.807135871727 181.539962065772 25.6250763271282 12.9744122239037 27.0492999436994 -27.0492999436994 12.9744122239037
2040/12/25 12:42:35 02:13:59 +51:05:35 RB 4.89078488448007 1.14214224196738 34.4059714783288 -58.6705738187454 -25.6257570264518 15.5987363854017 -15.5987363854017 -25.6257570264518
2058/03/12 18:59:59 15:52:35 +01:48:59 RB 1.581411270311 -52.7894479323298 58.6129622957347 -53.4522963342045 -24.1008402273979 17.8647558151081 -17.8647558151081 -24.1008402273979
2045/07/25 12:05:48 08:39:13 -46:13:42 RB 2.09430085082812 22.9180057996622 172.108647318874 -10.8291222823455 -5.63641701122768 29.4657564517788 -29.4657564517788 -5.63641701122768
2078/06/07 20:59:05 07:09:41 -18:24:19 RJ 3.6018530382402 -13.4955785198638 254.867445916248 73.3007699814146 28.7347906954129 8.62042943772589 -8.62042943772589 28.7347906954129
2050/09/18 05:05:06 12:07:13 -51:48:59 RB 1.19615400221602 -30.5006668868214 139.889800498114 -96.866773028181 -29.7848053425725 -3.58683296308367 3.58683296308367 -29.7848053425725
2054/10/12 04:42:01 09:35:01 -03:39:06 RB 1.50846148836438 27.6408606310804 105.965972273082 -65.0673988653475 -27.2041289789901 12.6465555189732 -12.6465555189732 -27.2041289789901
2050/11/18 19:35:24 22:41:19 -86:06:41 RJ 6.05331343912151 -15.6824575508119 180.342618865504 4.45489521505233 2.33022813749147 29.9093636981338 -29.9093636981338 2.33022813749147
2008/08/04 13:35:42 22:13:05 -09:01:48 RB 2.65899800870037 -78.7907900040421 351.306650455083 8.27119740696153 4.31576244261129 29.6879469572914 -29.6879469572914 4.31576244261129
2011/07/01 08:05:48 07:59:42 +84:41:13 RJ 0.618226054221094 20.1793653352755 5.675289161967 -92.8502908586684 -29.9628862542925 -1.49179332091585 1.49179332091585 -29.9628862542925
2046/10/09 21:52:19 06:01:13 -20:16:19 RB 5.96372805532195 -24.4755871422487 103.923230688987 -76.7590390086862 -29.2024621532133 6.87140481925954 -6.87140481925954 -29.2024621532133
2045/07/18 09:07:06 22:09:09 -61:35:06 RJ 1.19201966289385 -19.7827927169362 210.486549688869 84.6472832559997 29.8691786499663 2.79860086050084 -2.79860086050084 29.8691786499663
2027/01/01 00:59:39 20:01:48 +73:11:05 RJ 1.92530402661136 3.32209772262214 357.119565920226 9.44756645342678 4.92434848248502 29.5930868958115 -29.5930868958115 4.92434848248502
2043/06/19 08:01:19 16:13:59 +86:35:35 RJ 0.39639305943218 17.0319651363664 357.738789560152 37.1972714783245 18.1368354623527 23.896761274523 -23.896761274523 18.1368354623527
2052/04/09 14:59:48 02:09:39 +58:13:09 RJ 1.02010772374306 47.2708666501503 340.902270253914 143.952905644635 17.6535007932972 -24.2560077040937 24.2560077040937 17.6535007932972
1996/01/07 01:39:35 04:06:35 -25:39:24 RJ 2.19475916314764 12.880080639111 236.302533581192 60.2728232230046 26.0518922603234 14.8761187699111 -14.8761187699111 26.0518922603234
1998/03/23 01:35:48 10:09:41 +01:52:24 RB 3.47722520228052 41.3535454570805 253.921814734673 64.7311735231304 27.1294479889446 12.8059771909507 -12.8059771909507 27.1294479889446
2001/08/23 16:24:11 14:05:05 +18:52:35 RB 3.71764162825072 88.4042608140389 222.154958724634 41.7827078833391 19.9892235594693 22.3703138442347 -22.3703138442347 19.9892235594693
1981/06/14 21:07:11 12:16:11 +18:16:05 RB 3.74896653929915 61.2743947601581 271.678350967467 81.6021998607277 29.6783382319304 4.38135136586188 -4.38135136586188 29.6783382319304
2012/12/25 00:24:01 22:59:19 -01:06:42 RJ 1.65571461190327 -19.0151750654828 275.958958857385 69.3604728544253 28.0744975470388 10.5746199686473 -10.5746199686473 28.0744975470388
2060/01/25 06:05:11 14:16:39 +20:52:16 RJ 3.67500387617487 85.8956606033309 78.4575943818046 -100.057769923983 -29.538965018159 -5.23923139935427 5.23923139935427 -29.538965018159
1990/11/19 22:11:39 03:06:39 -84:05:11 RJ 0.462914924763504 -14.3015899274692 177.884711495067 -19.8140808038481 -10.1690741242046 28.2239248060299 -28.2239248060299 -10.1690741242046
1994/11/19 05:09:48 13:24:11 -41:48:35 RJ 2.27575751688389 0.419458153880303 135.308205588405 -62.5284585457195 -26.6172021638969 13.8392394648782 -13.8392394648782 -26.6172021638969
2054/03/01 17:19:39 00:07:24 -86:41:39 RB 0.952705263858899 -17.3985752416224 183.200089449926 50.8610152394794 23.2685132340559 18.936110790671 -18.936110790671 23.2685132340559
2027/09/09 07:48:16 21:35:13 +09:19:07 RJ 1.74784133264899 -37.719469921884 299.956710411364 55.7172915244906 24.7880497701894 16.8983013522251 -16.8983013522251 24.7880497701894
2007/12/23 09:52:39 15:06:35 -09:52:52 RB 4.09563312906196 59.2383753138897 194.027759949938 13.3924911981859 6.94861248003754 29.184187235595 -29.184187235595 6.94861248003754
2008/01/12 15:59:19 16:09:24 -18:16:07 RB 6.04395968874145 -18.0312826112836 256.357829474941 74.4856644120835 28.9069067744261 8.02438413428948 -8.02438413428948 28.9069067744261
1998/09/09 16:24:52 05:41:07 +16:16:09 RJ 4.00841020928461 -39.7456151831956 313.394093229619 45.4057406134776 21.362891788842 21.062451291771 -21.062451291771 21.362891788842
2058/06/18 05:24:35 00:59:48 +01:11:16 RB 5.98287514458166 52.5757487257471 114.652841446298 -58.8045408504388 -25.6621593768148 15.538776532242 -15.538776532242 -25.6621593768148
2047/07/09 10:19:42 19:35:59 +25:06:35 RB 1.34653560035035 -31.7407158592622 319.275195931356 42.7720184747918 20.3724867535243 22.0218478670042 -22.0218478670042 20.3724867535243
2063/01/07 22:35:11 10:52:16 +58:19:07 RJ 1.41845622625786 20.5034946430811 34.1673762458297 -85.1323415434831 -29.8918005733241 2.54563518293935 -2.54563518293935 -29.8918005733241
2032/04/23 05:11:59 09:52:07 +25:24:19 RJ 4.96961521335993 -28.029398785453 314.87867574772 47.4798166540218 22.1111791115191 20.2754964994281 -20.2754964994281 22.1111791115191
2015/08/25 06:06:06 12:42:39 -73:09:07 RJ 1.04112650501017 -30.2389850577943 165.46839277723 -125.058399521766 -24.557009793174 -17.232332111991 17.232332111991 -24.557009793174
2033/05/04 22:11:39 06:59:59 +86:01:42 RJ 3.33278871825112 20.2606842023479 355.74383253797 95.5167674360715 29.8610432032459 -2.88411144269479 2.88411144269479 29.8610432032459
2050/11/01 08:16:35 01:16:16 -16:48:59 RB 2.79085986909125 -51.509596775424 268.531050576228 78.4503808160574 29.3925504279212 6.00649476334657 -6.00649476334657 29.3925504279212
1986/04/07 01:01:16 16:11:35 -18:24:05 RJ 3.58254856760177 37.0191104069611 133.73903702361 -45.7397088478384 -21.4852979578196 20.9375732037816 -20.9375732037816 -21.4852979578196
2053/01/18 05:35:24 00:19:59 +32:35:35 RB 3.4362862294454 -36.0653599728156 10.9111812458729 -12.2821324441059 -6.38177059578407 29.313358798725 -29.313358798725 -6.38177059578407
2009/07/23 09:05:16 03:24:09 -73:01:52 RJ 1.26519581602912 -3.90931091167941 186.161083430523 20.1903412910389 10.3541995931119 28.15653655523 -28.15653655523 10.3541995931119
1995/04/19 05:48:11 01:07:52 +29:09:11 RJ 5.04131046452012 11.125734816817 62.8442641924042 -73.377775759971 -28.7463506343283 8.5818019790865 -8.5818019790865 -28.7463506343283
2023/11/19 14:24:01 14:42:06 +18:42:05 RB 4.70012000606795 44.847002411026 276.572511232199 80.0176352321847 29.5458346209914 5.20035157936724 -5.20035157936724 29.5458346209914
2051/05/23 22:35:41 21:01:01 -09:06:41 RB 3.75862501664615 -13.4722868793156 94.5340625244169 -71.5730165596751 -28.4618175464222 9.48287624901704 -9.48287624901704 -28.4618175464222
2020/03/18 02:16:59 09:35:52 +51:19:24 RJ 3.58290999553795 33.3802335153574 319.029893568107 99.8986381082648 29.5534023763304 -5.1571705403941 5.1571705403941 29.5534023763304
2003/03/01 19:19:11 02:07:07 -16:19:59 RB 1.46450281282633 27.9510608835071 238.351841193713 56.4576468201988 25.0043280334232 16.5765973467708 -16.5765973467708 25.0043280334232
2054/01/14 14:07:16 14:41:39 -58:52:48 RB 5.60283676945056 -21.3283157056943 212.801761976661 86.8233135812045 29.9539018568363 1.66245708245837 -1.66245708245837 29.9539018568363
1983/06/18 00:06:09 07:35:16 +84:06:24 RJ 4.5752597114789 14.7579142015106 356.925252916258 29.6546171781016 14.8431146316478 26.0707105394495 -26.0707105394495 14.8431146316478
2062/04/12 12:24:41 15:13:07 -84:01:09 RB 0.385760361862065 -24.5083076436062 183.295275097243 146.286120500765 16.6513783858713 -24.954590716951 24.954590716951 16.6513783858713
2058/03/19 05:35:59 11:13:05 -25:16:07 RJ 4.46730001646921 -5.83131546559969 244.871253215771 70.7896115605709 28.3295018127076 9.87113605639159 -9.87113605639159 28.3295018127076
2058/07/18 05:13:48 13:16:24 +38:59:59 RB 0.168599806281663 -30.8871754906924 350.276039020678 11.7037791372522 6.0855564862011 29.3762829890586 -29.3762829890586 6.0855564862011
2001/08/14 19:01:39 03:01:09 -25:59:52 RJ 4.25177647408873 -72.0669609891603 113.51220516174 -106.32468050349 -28.7905291215172 -8.43240374407379 8.43240374407379 -28.7905291215172
2061/12/25 14:05:05 23:35:42 +46:41:39 RJ 5.25031895369933 38.7771909793766 44.9036566854214 -102.969704978426 -29.2346661290066 -6.73307480468735 6.73307480468735 -29.2346661290066
2048/04/23 06:06:07 12:52:42 -01:19:59 RJ 5.2086414732021 -14.2971775777425 273.512631120865 69.9416378159704 28.1803124229822 10.2893144447583 -10.2893144447583 28.1803124229822
2012/04/12 10:05:13 17:11:42 -12:19:24 RB 6.0606949410514 -2.74476758114272 257.811406481602 70.3016644674379 28.244410115381 10.1120372445005 -10.1120372445005 28.244410115381
2053/05/23 00:59:06 04:42:59 -37:16:35 RJ 4.37773039427466 -72.6362199841195 181.022297890558 178.792850004737 0.632015499302884 -29.9933418679653 29.9933418679653 0.632015499302884
2035/10/07 16:01:48 04:06:16 +51:13:52 RB 4.3898366699095 -18.3557308980805 5.45283116903896 -8.24705188256147 -4.30325099779317 29.6897630649015 -29.6897630649015 -4.30325099779317
2052/09/04 03:05:13 02:05:41 -01:01:05 RJ 0.439621204430269 68.3240605994108 160.995038442069 -17.8412885536511 -9.19144050481943 28.55726565073 -28.55726565073 -9.19144050481943
1983/07/09 19:01:01 19:07:05 +86:13:01 RB 3.61868433546641 20.6051272270348 3.89001698654857 -101.418673260876 -29.4062011224318 -5.93930429823988 5.93930429823988 -29.4062011224318
2051/10/09 21:24:35 13:13:59 +73:24:19 RB 5.83872322548604 7.29066365984655 347.957348022686 41.7881611789331 19.9913526298343 22.3684112093286 -22.3684112093286 19.9913526298343
2028/10/23 01:05:13 11:59:42 +12:13:41 RJ 0.754355028872463 -37.0641689591222 56.5805627480293 -53.4124164890132 -24.0883999021701 17.8815265051151 -17.8815265051151 -24.0883999021701
2025/06/09 11:59:06 20:01:07 -51:41:52 RB 1.27092604198062 -40.5821721109479 218.158401442515 111.047397447686 27.9985095073851 -10.7742036998039 10.7742036998039 27.9985095073851
2054/07/04 18:48:52 11:24:07 +46:59:07 RB 3.49339181118155 55.0699160812409 325.954645656991 130.189092106648 22.9175668847348 -19.3593679670517 19.3593679670517 22.9175668847348
1995/06/19 10:42:19 07:07:52 -41:48:41 RB 1.09440874126642 15.7848978332084 147.084363765075 -43.3699297876065 -20.6011827664852 21.8080551315763 -21.8080551315763 -20.6011827664852
2027/09/04 02:19:09 02:09:42 -12:52:07 RJ 0.221852555938908 51.9258821539339 147.150265398376 -31.5446488039781 -15.6948852522261 25.5669821629275 -25.5669821629275 -15.6948852522261
2016/05/14 00:19:13 14:11:19 -20:07:09 RB 4.0477254422655 45.9982103648809 204.859646746998 24.9626714490104 12.6608312106152 27.1974879916567 -27.1974879916567 12.6608312106152
2069/01/07 22:07:13 05:19:05 -20:24:39 RB 1.30554704890721 49.4082227422276 170.957832550112 -9.07014947341273 -4.72930839364086 29.6248821452143 -29.6248821452143 -4.72930839364086
1995/11/25 01:01:59 04:24:48 -16:19:59 RB 1.29052306571815 53.2547074366994 191.628199926906 11.3901615189955 5.92467033876875 29.4091530203934 -29.4091530203934 5.92467033876875
2011/12/14 10:48:52 00:48:24 +29:41:13 RJ 4.18734463278027 -22.450196159931 43.9632524783113 -48.785927053317 -22.5675929816004 19.7662274351182 -19.7662274351182 -22.5675929816004
2079/07/09 08:59:35 12:05:07 -20:05:13 RB 1.00019745172103 -39.2604538463786 101.132051000365 -80.9136050945461 -29.6235400239212 4.73770793223272 -4.73770793223272 -29.6235400239212
2067/05/07 13:07:42 22:52:59 -32:07:41 RB 1.00048813265335 3.70128174043303 234.560807371179 64.0124512133061 26.9666786866679 13.1452744592882 -13.1452744592882 26.9666786866679
2051/12/04 17:48:35 17:39:42 +73:09:52 RB 5.85701844760067 24.3548544357207 342.42962351315 102.106738348241 29.3327573255966 -6.29200664952419 6.29200664952419 29.3327573255966
2011/04/14 12:13:52 22:09:16 +20:59:41 RJ 0.361761567452831 44.919745401625 280.352340306764 82.5658306125461 29.7478252928496 3.88160924696121 -3.88160924696121 29.7478252928496
2028/08/14 09:42:13 00:24:48 +01:06:59 RB 1.81218115878966 -5.6880156378165 273.720418305851 69.9024317477685 28.1732651056627 10.308595116991 -10.308595116991 28.1732651056627
2052/05/19 02:13:16 06:09:52 -58:19:16 RJ 4.63761017741021 -51.1511427128008 185.804351302367 169.556870250735 5.43778450469895 -29.5030591580001 29.5030591580001 5.43778450469895
1985/12/23 05:19:07 11:19:48 -37:01:42 RB 2.90441400485238 32.8333982607532 176.236043875654 -4.44794423842995 -2.32659959326668 29.9096461753163 -29.9096461753163 -2.32659959326668
2061/07/25 14:42:52 03:52:01 -29:01:19 RB 2.78359718210745 -18.1236576250482 245.227902536344 76.842929510205 29.212491721406 6.82864023263692 -6.82864023263692 29.212491721406
2052/12/18 07:35:52 11:52:24 +86:35:42 RJ 3.43006917558291 23.3492458857424 358.775690561624 161.879552852633 9.3304686129419 -28.5121439997575 28.5121439997575 9.3304686129419
2003/03/07 10:41:41 15:59:19 -25:11:52 RJ 5.58669982230635 0.00877615810788733 243.078643770576 67.9834063758992 27.8122597381244 11.2462530764298 -11.2462530764298 27.8122597381244
1998/09/25 02:35:01 02:52:41 -37:05:39 RB 0.652833273165665 32.9756567223598 174.040900300511 -7.01471329893985 -3.66372662338835 29.775444702457 -29.775444702457 -3.66372662338835
1995/11/01 21:09:39 12:41:59 -49:24:42 RB 6.1615237623676 -56.8550753208273 201.590105661449 147.672996982638 16.0425196439949 -25.350297108161 25.350297108161 16.0425196439949
2007/07/23 20:59:13 13:05:11 -73:35:09 RJ 4.38000846672856 -9.87773736934285 193.471567028173 51.0515822979637 23.3313663477006 18.8586145925247 -18.8586145925247 23.3313663477006
1982/12/18 23:52:19 04:05:07 +32:39:39 RJ 1.40102966143622 68.5611635990398 310.50751303169 121.877774943181 25.4752981561169 -15.8432693550589 15.8432693550589 25.4752981561169
2010/03/14 21:11:39 15:06:16 +12:09:07 RJ 2.18558948241544 -6.41862246552085 74.6376259228728 -68.090011489518 -27.8331364783921 11.1944858647103 -11.1944858647103 -27.8331364783921
2068/04/19 18:52:16 06:39:41 +18:11:59 RB 2.21143907202246 66.2594277148421 269.976180165009 81.7212549574135 29.6873781796033 4.31967323083779 -4.31967323083779 29.6873781796033
1984/01/09 20:59:52 15:01:16 -86:39:41 RB 1.02081622749143 -22.9830627047891 179.369524005658 -169.361011075366 -5.53860542510278 -29.4842983627731 29.4842983627731 -5.53860542510278
2013/04/18 09:41:06 18:19:39 +29:19:39 RJ 6.05421368503982 24.9488374675323 293.992069199855 80.3512913109794 29.5756171852838 5.02820724608115 -5.02820724608115 29.5756171852838
2001/08/18 04:09:24 00:35:59 +73:42:01 RB 0.416754792198102 35.2967961294011 355.291063656116 163.750177319076 8.39478128597162 -28.8015216118993 28.8015216118993 8.39478128597162
2040/10/14 03:39:06 15:41:39 +38:09:06 RJ 1.27429029733162 -29.9265253538096 15.6192890432441 -18.7564263449645 -9.6463701538883 28.4068221181844 -28.4068221181844 -9.6463701538883
1992/12/14 00:35:24 00:16:41 -46:11:16 RJ 1.51361351566635 -9.29123784470172 224.042445788436 70.9769125077809 28.3616193076694 9.77847381991878 -9.77847381991878 28.3616193076694
2028/10/09 13:13:39 15:52:13 -61:11:41 RB 3.70060846968903 5.76956286132176 167.079249374523 -26.090178680126 -13.1935568569008 26.9430892338597 -26.9430892338597 -13.1935568569008
2055/08/12 23:39:39 12:13:48 +09:41:07 RB 5.43236971564551 -29.8938759705558 293.653691378114 60.7740968436345 26.1810428693452 14.647627598813 -14.647627598813 26.1810428693452
2042/03/18 06:11:41 11:16:59 +41:48:35 RB 4.6041824454575 10.5635616584932 310.278685309433 72.8217951885945 28.6617233908903 8.86033928605954 -8.86033928605954 28.6617233908903
2058/08/01 17:48:19 03:24:35 +51:05:19 RJ 3.71065375446387 -16.8373097699465 347.337279541325 19.2527202881037 9.89206405003799 28.3222009884463 -28.3222009884463 9.89206405003799
2075/09/01 14:09:11 12:01:59 +41:52:42 RJ 3.28315240221077 67.6327742947747 346.792273620604 163.332174483197 8.60467826127623 -28.7395113392681 28.7395113392681 8.60467826127623
2062/12/07 18:52:11 08:07:41 -38:41:39 RJ 6.19266308140968 -41.338694932659 124.853352028556 -97.4162961227877 -29.7490347030211 -3.87232930397786 3.87232930397786 -29.7490347030211
2031/08/19 08:24:59 04:16:07 +03:52:11 RJ 1.54777218935554 61.5416013314232 239.236041401066 54.1240518601227 24.3086315666365 17.5809678731724 -17.5809678731724 24.3086315666365
2032/05/14 05:11:52 11:52:19 +25:09:39 RJ 5.33036337053466 -21.6456841372875 308.75691228081 54.0286226994648 24.2793158285299 17.6214307846582 -17.6214307846582 24.2793158285299
2048/02/23 18:24:52 16:13:24 -49:59:16 RB 1.12552828097785 -59.5942371326958 180.903732819839 178.671104461064 0.695745694559116 -29.9919312137198 29.9919312137198 0.695745694559116
2078/08/23 07:48:11 00:35:41 +20:06:01 RB 1.46607937038259 21.8335259820473 285.208260076961 76.1982684966065 29.1338121264383 7.15688416710763 -7.15688416710763 29.1338121264383
2053/07/04 09:07:42 22:06:39 +38:39:19 RB 0.954822407814097 18.455509160007 305.973695895328 79.0909387516035 29.4578638570065 5.67752208116108 -5.67752208116108 29.4578638570065
2080/02/18 20:06:39 02:42:35 +01:11:06 RB 1.4890525695749 44.2670954812034 252.144254123214 63.616329540731 26.8751534032732 13.3313963841203 -13.3313963841203 26.8751534032732
1990/02/04 17:41:52 10:07:41 +25:09:16 RJ 0.611320400240169 -13.8244646548718 56.3023266762709 -59.8834355886128 -25.9501918769572 15.0528250354909 -15.0528250354909 -25.9501918769572
1988/11/12 01:42:01 17:24:48 -51:41:59 RB 1.25399574490432 -57.1196573930196 191.555977154874 162.283736747992 9.12910377174406 -28.577254317459 28.577254317459 9.12910377174406
2073/10/12 18:13:13 00:48:13 -38:41:16 RB 5.06452752734844 -7.06378184527374 127.953269613305 -70.3143850246348 -28.2466544493404 10.1057662955147 -10.1057662955147 -28.2466544493404
2008/01/12 02:35:09 20:42:11 -37:07:09 RJ 2.52551392673313 -68.8330872642146 147.620531472059 -140.832589136067 -18.9476526461756 -23.2591156151706 23.2591156151706 -18.9476526461756
2051/01/09 09:42:35 00:41:07 -32:06:19 RB 4.35413166002251 -37.0625582989565 115.142575435872 -88.0017608460235 -29.9817569729412 1.04606348540217 -1.04606348540217 -29.9817569729412
1983/12/07 10:01:05 19:07:09 +58:19:39 RB 3.85394824290147 29.2458874144102 33.3192207261873 -99.6831062930646 -29.5725931641281 -5.04596210340138 5.04596210340138 -29.5725931641281
2048/11/12 17:11:13 16:16:59 +16:42:41 RB 5.32761308599864 33.2830286114724 277.125914739446 76.7708740560318 29.2038808913215 6.86537259626207 -6.86537259626207 29.2038808913215
2008/05/04 14:42:13 11:05:11 -37:01:07 RJ 1.36735214110585 -10.3072137766285 125.851465146123 -72.8672773052728 -28.668747809458 8.83758445717476 -8.83758445717476 -28.668747809458
2022/12/01 14:05:42 12:39:48 -01:52:05 RB 4.83055977739661 3.09390076018989 266.4704435232 70.0005084899183 28.190869683517 10.2603541111874 -10.2603541111874 28.190869683517
2029/11/04 04:11:09 21:06:48 +73:11:42 RJ 1.77013233098766 5.99651528230586 350.405470206406 33.1189678590326 16.3913777759769 25.126136483853 -25.126136483853 16.3913777759769
2043/05/18 18:09:19 13:19:48 -84:16:19 RB 2.50606198006819 -17.1081623953329 175.273657941454 -58.0707463321364 -25.4610531219559 15.8661518308928 -15.8661518308928 -25.4610531219559
1995/05/25 17:16:11 18:05:07 +46:52:11 RJ 2.38760846159788 -11.7102820998834 29.8919646138211 -43.2957490837969 -20.5729306731365 21.834709146639 -21.834709146639 -20.5729306731365
1995/12/14 09:13:59 18:48:39 +38:48:11 RJ 3.77001033585575 30.6187280945455 55.8971212056001 -88.2950890261319 -29.9867194102463 0.892557567418424 -0.892557567418424 -29.9867194102463
1998/09/12 10:35:24 02:06:11 +58:06:11 RJ 2.53100753123524 5.14769382076225 330.874943723084 60.0323726917942 25.9892331169734 14.9853182145596 -14.9853182145596 25.9892331169734
1981/08/01 03:05:11 13:24:42 +29:35:19 RJ 6.12385080900694 -32.6285792950987 328.864966482284 34.0507651694691 16.7978168164878 24.8562537442737 -24.8562537442737 16.7978168164878
2053/10/23 09:48:06 03:06:05 +29:59:13 RJ 3.04107646161513 -18.5226612196001 313.198090301784 52.505762212353 23.8024367747676 18.2604491615949 -18.2604491615949 23.8024367747676
1992/03/04 13:35:05 16:48:39 -73:59:05 RB 0.0221469670722261 -23.8887486825842 196.630098379958 101.53940260146 29.3936209967562 -6.00125359396282 6.00125359396282 29.3936209967562
2017/08/12 01:35:06 02:06:24 +61:48:41 RJ 5.9237994562453 34.6699966850297 27.0401627413835 -114.814290581825 -27.2301849456671 -12.5903545551651 12.5903545551651 -27.2301849456671
2021/04/04 18:39:48 03:24:42 -32:48:16 RJ 1.88809394758292 14.4737572003905 226.630963326936 54.3885436923892 24.3895304741513 17.4685661475246 -17.4685661475246 24.3895304741513
2020/09/01 18:35:19 08:59:48 -37:59:11 RJ 4.4530653894378 -35.4092409832583 236.781540584582 91.6175018355079 29.9880461799185 -0.846809489245073 0.846809489245073 29.9880461799185
1989/05/25 03:19:07 20:35:52 +41:11:48 RJ 5.0161916682694 61.9270262474887 35.8422976947171 -132.978178356194 -21.9484018244 -20.451593027309 20.451593027309 -21.9484018244
2071/08/12 09:48:41 00:07:16 +16:01:05 RJ 1.79887936594602 -3.77754614169482 288.958426719832 68.0529421686876 27.8258880261314 11.2124910504847 -11.2124910504847 27.8258880261314
1992/10/23 23:52:05 23:41:52 +16:52:35 RJ 0.44674386433625 61.2039258996275 269.056710757397 79.3477047847527 29.4830112897978 5.54545266733546 -5.54545266733546 29.4830112897978
2005/07/12 23:59:13 11:09:35 +61:19:09 RJ 4.98660806975742 4.79930746023247 334.872041244146 56.2756969097029 24.9515609856095 16.6559179987599 -16.6559179987599 24.9515609856095
2065/10/12 18:11:42 17:24:48 +03:05:39 RJ 5.0569373515956 58.1707665111133 241.720831147509 56.0613843906906 24.8890858022195 16.7491315574795 -16.7491315574795 24.8890858022195
2038/04/12 00:09:35 02:19:59 -51:35:39 RB 3.44938543215614 -55.5413931766889 200.151631240005 148.855906927469 15.5157639810438 -25.676079686832 25.676079686832 15.5157639810438
1999/03/07 16:09:05 11:42:11 +86:48:59 RB 0.735459003175905 17.3853770095454 2.58665917389247 -44.7587188603045 -21.1236836164539 21.3023470648649 -21.3023470648649 -21.1236836164539
2058/10/19 08:52:59 06:06:05 +37:59:01 RB 2.72743659887822 33.136601418977 303.057651485042 89.7492663490672 29.9997127430466 0.131283413605091 -0.131283413605091 29.9997127430466
2002/03/09 04:01:19 20:42:07 -84:13:24 RJ 3.87355348665285 -19.5820639114551 173.859248492816 -86.5814160719666 -29.9466161804791 1.78890450809109 -1.78890450809109 -29.9466161804791
2041/04/25 17:09:09 17:19:05 +01:24:09 RB 1.85543925700451 -57.3473580823084 52.5775549598478 -48.3582988055079 -22.4194401483666 19.9341090453978 -19.9341090453978 -22.4194401483666
2030/03/14 04:19:13 05:05:41 +51:41:39 RJ 4.04179830343549 -15.0600223375201 344.00701902586 24.7420662409557 12.5560194822433 27.2460341107018 -27.2460341107018 12.5560194822433
1988/07/12 15:07:19 00:01:59 +29:05:52 RB 2.66145983759928 -33.7017395881364 329.989069712096 32.6572064344426 16.1883496508071 25.2574213961603 -25.2574213961603 16.1883496508071
1989/09/07 16:59:41 20:06:24 +01:05:01 RB 4.12951059475248 23.3926233703897 97.5652475808392 -68.8695389858873 -27.9828603746149 10.8147827187977 -10.8147827187977 -27.9828603746149
2002/05/25 22:24:13 23:09:52 +86:05:59 RJ 3.74046024409525 17.1366188800764 2.97388044801744 -45.934726598059 -21.5564386212109 20.8643225140422 -20.8643225140422 -21.5564386212109
2015/07/18 10:48:19 14:59:48 -01:06:41 RB 1.62219811946283 -40.3697982686038 74.1441263293964 -64.8514636388697 -27.1562737982786 12.748991857905 -12.748991857905 -27.1562737982786
2009/06/12 11:52:52 21:52:48 -84:41:48 RB 1.2931682209478 -21.0161717302654 185.832607300567 101.286471789794 29.4198268923277 -5.87143812242553 5.87143812242553 29.4198268923277
2007/08/04 01:16:19 01:24:35 +73:41:07 RJ 5.69412242482639 28.4627433080755 15.1490806157696 -118.708334618622 -26.3122889392895 -14.410532633298 14.410532633298 -26.3122889392895
2073/03/12 03:13:52 19:52:13 +29:41:35 RJ 3.73143433285293 13.9007986032322 62.8450326219621 -74.8886188961944 -28.9626259742097 7.82088848392768 -7.82088848392768 -28.9626259742097
2016/10/07 11:01:06 20:35:24 -84:01:24 RB 3.0845502046916 -24.1991835862992 175.237617108039 -133.829385205644 -21.6421546408966 -20.7753975292776 20.7753975292776 -21.6421546408966
2024/01/19 16:19:48 01:07:06 -49:59:35 RJ 6.25608667282134 18.3862909038904 167.495330017629 -18.4229683389068 -9.48088171435938 28.4624820055863 -28.4624820055863 -9.48088171435938
2080/03/09 00:48:16 16:05:09 -73:05:05 RB 3.0480511587336 -13.6015256928204 163.927475061145 -65.80617743051 -27.3649292201473 12.2947406957742 -12.2947406957742 -27.3649292201473
2004/08/19 01:42:59 23:24:59 +38:16:39 RJ 6.08126567042325 71.3540397168892 7.06794584056864 -171.517664553748 -4.42513457837953 -29.6718416004674 29.6718416004674 -4.42513457837953
2078/12/04 10:48:42 10:05:09 +58:48:42 RJ 4.02777223541974 22.9657366462507 326.179199228471 89.2027174761307 29.997095552868 0.417442681224413 -0.417442681224413 29.997095552868
2018/05/25 06:05:11 17:07:24 -16:06:24 RJ 5.74227631309426 10.7591484443604 248.341803659008 65.5265109103273 27.3045915939899 12.4281647030209 -12.4281647030209 27.3045915939899
1996/07/09 03:42:19 06:59:06 -58:52:13 RB 5.89705037659525 -35.6670911969233 149.553906036157 -112.492703901975 -27.7178476885748 -11.4769734474281 11.4769734474281 -27.7178476885748
2034/01/09 22:13:41 06:42:24 +58:41:06 RB 1.35918870450389 47.3354829811829 18.6443573628989 -144.75573528716 -17.3119032484961 -24.5009796929574 24.5009796929574 -17.3119032484961
1981/12/23 01:06:42 04:35:01 +73:42:01 RJ 1.79945940761984 32.8522576085981 348.959703226146 140.171462919417 19.2147682795878 -23.0389383427654 23.0389383427654 19.2147682795878
2057/02/01 08:52:41 06:19:07 +61:42:52 RJ 4.5407579011506 -7.499168021312 352.453152374677 15.0995633173588 7.81491450711818 28.9642384889787 -28.9642384889787 7.81491450711818
2037/05/25 04:11:19 17:11:06 -84:16:16 RJ 5.25092781371638 -15.4568439785361 183.833841142292 39.3592823363178 19.0254361265172 23.1955336260198 -23.1955336260198 19.0254361265172
1982/12/01 11:48:59 10:19:42 +32:05:11 RJ 4.22698444185454 12.4230111036313 300.010594484489 74.2367737766069 28.8717765355877 8.14987850701408 -8.14987850701408 28.8717765355877
2077/05/19 02:59:59 20:19:07 +49:13:06 RB 4.84098154427268 52.6167891230471 30.4628657707822 -132.591128536135 -22.086056497054 -20.3028596116147 20.3028596116147 -22.086056497054
2043/07/23 23:11:19 14:19:05 -51:13:24 RJ 4.96279238230294 -3.06307922662469 215.61450243208 61.4742974446528 26.3580892075065 14.3265883352991 -14.3265883352991 26.3580892075065
2055/07/14 17:13:39 21:52:11 -20:39:19 RB 3.24463559745544 -55.8681381194021 96.941811544123 -84.1208018384282 -29.8422021442827 3.07294177943066 -3.07294177943066 -29.8422021442827
2055/07/12 13:16:35 01:24:11 -12:06:59 RJ 2.17300006295058 -15.8487509909555 262.867090874637 72.5000110994743 28.6115102700495 9.02116844243873 -9.02116844243873 28.6115102700495
2025/07/01 05:19:35 02:07:19 -86:24:59 RJ 6.18458636786321 -16.8303730012999 177.68685358327 -35.9135984669268 -17.5969378169685 24.2970734753333 -24.2970734753333 -17.5969378169685
1989/07/18 10:59:41 12:16:11 +73:11:24 RJ 1.67707207920142 19.5696581805288 17.7949284555057 -86.1209003867986 -29.9312706933847 2.02954050940786 -2.02954050940786 -29.9312706933847
2065/04/23 07:01:07 14:42:52 +16:48:13 RB 5.44725359119898 5.44235301403091 285.407100763491 70.9099150157414 28.350165688737 9.81163112948911 -9.81163112948911 28.350165688737
2015/03/01 23:09:16 10:52:16 +29:11:41 RJ 2.47287587066494 68.3100221640022 60.464608228022 -110.473086068001 -28.1050977351656 -10.4930205992742 10.4930205992742 -28.1050977351656
1993/12/07 07:11:05 14:48:13 +16:19:05 RJ 3.12023878397805 48.8887279580144 87.6023021526921 -78.3775491036033 -29.3848915083599 6.04385233455604 -6.04385233455604 -29.3848915083599
2076/03/12 06:41:41 02:11:05 -51:07:48 RJ 4.6453563330994 -38.5445302596795 139.973295398519 -106.885028562174 -28.7066853639639 -8.7135650232486 8.7135650232486 -28.7066853639639
2072/10/09 00:16:59 00:01:05 -37:41:39 RJ 0.308297183709047 30.8182221733128 195.219804923873 18.0846275476222 9.31264185505057 28.5179715561882 -28.5179715561882 9.31264185505057
2018/05/14 14:07:42 06:07:06 -01:01:39 RJ 1.38099926757909 65.6197615909771 147.336171528284 -30.5181670005323 -15.2343461287697 25.844045697778 -25.844045697778 -15.2343461287697
1986/05/14 06:05:42 15:48:06 +38:19:35 RB 5.55102963843418 19.2313914446824 304.797644062045 79.4872817535625 29.4964329467806 5.47361337838852 -5.47361337838852 29.4964329467806
2014/03/04 06:52:48 14:24:42 +37:24:52 RJ 4.53955342787162 48.2243049120217 304.361707460413 102.351602992654 29.3055994215897 -6.41730804475852 6.41730804475852 29.3055994215897
2022/02/01 15:16:09 13:42:16 -58:07:48 RJ 6.20953434598275 -45.7696372496948 202.276015357569 137.357342093969 20.3227146512044 -22.0677880451513 22.0677880451513 20.3227146512044
2030/04/23 19:11:01 09:35:42 +41:41:07 RJ 2.34858366392259 66.7221454780274 18.8913348885218 -155.984001927361 -12.2097511951292 -27.4029556025083 27.4029556025083 -12.2097511951292
2050/03/14 09:24:52 16:01:42 +03:59:09 RJ 5.38177388102017 22.6459422261582 265.807266659622 70.1098977384414 28.2104073872693 10.2065133637447 -10.2065133637447 28.2104073872693
1984/08/04 14:42:39 19:01:09 -12:07:48 RB 2.94857905995918 -29.0893089392088 93.0954344844048 -73.8715831352574 -28.8192449052561 8.33373404248489 -8.33373404248489 -28.8192449052561
2038/05/01 12:59:42 12:52:06 -51:35:39 RJ 0.862518799208777 -47.5699104164948 147.546748500967 -125.277401777672 -24.4909632412741 -17.3260705156871 17.3260705156871 -24.4909632412741
2062/08/25 16:35:39 12:09:09 +38:06:16 RJ 3.80620052110935 54.7717079196201 308.166415651209 110.684432428659 28.0662011194547 -10.5966199668724 10.5966199668724 28.0662011194547
2016/12/09 18:48:42 01:48:11 +86:35:59 RB 6.21419842861965 22.3240868106822 1.91189414995478 -144.152955583423 -17.5687027794758 -24.317497458547 24.317497458547 -17.5687027794758
2046/07/01 03:39:11 11:07:06 +29:05:39 RB 5.74388101539427 -38.2499289468591 338.674008989323 22.9274574642198 11.686960764525 27.6299646776548 -27.6299646776548 11.686960764525
2010/10/19 01:13:48 16:05:05 +38:01:59 RJ 0.712080953191997 -29.0258148187469 341.545479364225 22.2061207844987 11.3381908025734 27.7749064683293 -27.7749064683293 11.3381908025734
1985/05/09 19:13:07 15:42:24 -58:05:19 RB 2.6311373442096 -14.470596265449 147.139525985683 -75.6176640351182 -29.0597935526513 7.45173796354158 -7.45173796354158 -29.0597935526513
2069/11/19 21:01:19 21:24:13 +46:41:13 RB 0.170087067985166 42.8263582913481 316.854655178954 108.710027147325 28.4146246142469 -9.6233626156054 9.6233626156054 28.4146246142469
2020/06/23 23:42:59 06:05:11 +20:59:16 RB 4.59499186757864 -48.2490195821901 347.237155089072 12.8603118263099 6.67724567553129 29.2474680987705 -29.2474680987705 6.67724567553129
2051/09/04 13:11:09 16:41:05 -38:16:05 RB 3.07773193966981 -1.62763517293915 130.619703454316 -65.7556949310333 -27.3540858778681 12.3188467717648 -12.3188467717648 -27.3540858778681
1982/01/19 10:07:11 07:39:39 -84:05:07 RB 4.62869943252187 -24.9226520269347 183.079842046092 150.217857235141 14.9011044768963 -26.0376090562982 26.0376090562982 14.9011044768963
1999/11/18 02:24:41 01:39:13 -09:52:09 RB 1.53238068531401 21.9258927562842 250.331908932711 63.9607281371212 26.954800961729 13.1696129447138 -13.1696129447138 26.954800961729
1982/01/25 11:52:48 14:11:39 +73:05:19 RB 5.19401945680432 20.4991978964652 341.826136089233 89.0959663550002 29.9962657260797 0.473331269214617 -0.473331269214617 29.9962657260797
2002/08/04 20:24:09 19:07:52 +86:42:16 RB 4.43654527926539 22.6416395606065 1.6678864261285 -150.815397769218 -14.6287515270008 -26.1915946204364 26.1915946204364 -14.6287515270008
1989/09/12 07:41:19 09:09:42 +38:19:13 RB 1.77252107226948 53.5338494971997 51.8865483888424 -109.726857012869 -28.2393729072984 -10.1260958815596 10.1260958815596 -28.2393729072984
2036/02/07 05:42:35 20:11:35 -01:01:48 RJ 3.79654208005283 3.61117016334153 92.2845031742825 -70.0716034548824 -28.203579448011 10.2253658281613 -10.2253658281613 -28.203579448011
2005/03/23 19:19:48 21:06:24 +37:39:01 RJ 1.85456504399534 -25.7972902905978 333.594398950557 31.9052110980241 15.8554664161859 25.4677086665684 -25.4677086665684 15.8554664161859
2017/09/14 01:42:39 19:11:39 -58:41:05 RJ 0.241334078508009 -14.567861688308 212.400302417156 75.7068644218814 29.071359498136 7.406487489366 -7.406487489366 29.071359498136
2077/05/09 18:09:19 10:16:41 +16:01:09 RJ 2.3643456673628 70.7460046180049 99.4037478626106 -74.5257893789232 -28.9125192642342 8.00413829186417 -8.00413829186417 -28.9125192642342
2057/07/18 03:13:05 22:41:06 -86:19:24 RJ 5.92778069617809 -15.8451580721542 179.787739500207 -2.87611338022207 -1.50529708290011 29.9622108779077 -29.9622108779077 -1.50529708290011
2073/11/09 15:06:19 20:13:59 +29:11:52 RB 4.72846436490104 57.8816931734078 65.6068721360854 -99.8503342786022 -29.5577396819286 -5.13225339352458 5.13225339352458 -29.5577396819286
2041/01/14 08:48:39 17:19:16 -86:05:01 RB 4.21132408807299 -16.3703956145488 178.171464581531 -26.4436366639429 -13.3595168202612 26.8611859442051 -26.8611859442051 -13.3595168202612
2023/10/07 06:13:06 23:42:52 +20:01:39 RB 1.81250676854523 -8.17372579577948 295.264020347425 65.2197989424796 27.2376710647015 12.5741510636354 -12.5741510636354 27.2376710647015
2078/11/23 22:01:59 18:05:01 +84:01:06 RJ 0.501152205688548 16.6919529788329 354.629310642555 57.6356176110316 25.3398256849155 16.0590545879294 -16.0590545879294 25.3398256849155
2007/01/01 15:42:06 16:07:16 +25:01:19 RJ 5.78354117764786 8.68269232546054 293.539867000817 72.1036043118458 28.5484121073841 9.21890265416482 -9.21890265416482 28.5484121073841
2051/08/04 19:09:52 04:06:39 -46:24:13 RJ 4.11393243765436 -63.0497756707357 189.704658887349 166.738978465253 6.88162883789083 -29.200054529701 29.200054529701 6.88162883789083
2065/01/01 21:19:07 10:35:39 -86:07:13 RB 0.991338507212358 -20.3457904787299 176.540579027929 -98.5467251297367 -29.6668498451827 -4.45847734808424 4.45847734808424 -29.6668498451827
2033/03/19 11:06:42 13:48:19 +37:13:09 RJ 5.91531730890909 -17.0337917116517 321.289419823021 47.4910354183907 22.1151487185747 20.2711666451471 -20.2711666451471 22.1151487185747
2036/08/04 23:16:41 06:48:24 -12:19:39 RJ 5.20461841431626 -73.2035434409553 66.036392980502 -61.6532249778527 -26.4027007467776 14.2442056035467 -14.2442056035467 -26.4027007467776
1984/06/18 05:06:52 08:24:01 -51:16:59 RJ 5.90402069061927 -49.6227867815405 148.814832273123 -128.925363118998 -23.3389527575449 -18.8492250286606 18.8492250286606 -23.3389527575449
2012/07/18 17:52:35 14:35:06 -03:35:42 RB 3.49104044678601 59.4857267704716 138.826788602122 -38.3701888989179 -18.6221981715175 23.5204960674883 -23.5204960674883 -18.6221981715175
1993/01/25 13:16:07 01:13:35 +01:42:16 RB 5.56447786043669 28.6630227665305 98.9866085283851 -68.392228482277 -27.8917963634033 11.0475198855871 -11.0475198855871 -27.8917963634033
2056/03/18 13:07:42 10:19:24 +01:07:01 RB 0.151631485066974 -52.2043211081592 61.165737837002 -55.5026555008339 -24.7245731729584 16.9910412104446 -16.9910412104446 -24.7245731729584
2076/01/18 00:52:11 13:06:35 -84:42:59 RB 2.18725213350255 -18.6064071959443 175.272967453043 -73.9863573749558 -28.8358811108902 8.27598698395571 -8.27598698395571 -28.8358811108902
1984/12/19 13:11:13 23:52:41 -37:19:13 RJ 4.90531505800174 -2.11427803432695 129.264212820561 -66.4856541746388 -27.5088061828894 11.9693601496581 -11.9693601496581 -27.5088061828894
2052/11/01 02:42:07 03:11:19 +03:16:07 RB 1.33630106314465 58.7812620535849 242.335594762163 56.6047114468271 25.0467937984503 16.5123626540211 -16.5123626540211 25.0467937984503
1988/08/25 17:39:09 11:35:16 -37:41:16 RJ 4.08269823308896 9.36280845593668 224.181180237632 55.8728546245918 24.8338387289335 16.8309374066129 -16.8309374066129 24.8338387289335
2066/07/01 04:24:16 22:11:05 +37:09:52 RB 5.94383579225901 71.2073321991417 343.84655732943 160.671746895297 9.92939249747723 -28.3091357097147 28.3091357097147 9.92939249747723
2058/02/09 19:11:42 06:24:09 +29:11:39 RB 1.09939358916612 57.1567260443421 66.7043269242327 -98.469197718144 -29.6728554887965 -4.4183308093668 4.4183308093668 -29.6728554887965
2027/10/12 06:35:13 09:59:59 +03:42:41 RB 1.99591485425605 50.6450162856158 109.998951885486 -62.3153609698507 -26.5655465355954 13.9381396629935 -13.9381396629935 -26.5655465355954
2060/10/25 20:19:07 08:48:19 +03:24:42 RJ 5.84157569851957 -58.5499853819025 45.3593708172537 -42.0982312765241 -20.1121114142812 22.2598961017238 -22.2598961017238 -20.1121114142812
2002/05/18 03:09:07 00:19:35 +09:39:59 RB 4.8493529548494 5.48220391729773 81.3746319670705 -70.7897962775782 -28.329533636313 9.8710447242939 -9.8710447242939 -28.329533636313
2038/07/09 22:52:41 16:39:59 +38:05:41 RB 4.64398049800048 67.5370653349868 326.970518281733 139.449049250739 19.5037196638654 -22.7948441379482 22.7948441379482 19.5037196638654
1982/12/14 22:05:01 16:05:06 -46:59:39 RB 0.862750132175257 -60.9229973399555 197.547234016801 155.386852216877 12.494682768665 -27.2742168083783 27.2742168083783 12.494682768665
2061/05/01 11:52:16 23:01:48 +58:48:13 RJ 0.574923794709107 38.4013290585813 331.463060115564 118.852805394335 26.2758693080359 -14.4768329446401 14.4768329446401 26.2758693080359
1989/12/25 20:09:05 22:24:48 -58:19:09 RB 0.550110571976992 -0.0328567879438035 205.52145557557 50.1227669170438 23.0225992851013 19.2343422595538 -19.2343422595538 23.0225992851013
2013/01/12 10:52:19 10:06:39 -32:24:07 RJ 4.71435136775915 -33.9363424570789 243.562474549956 86.7711264882653 29.9523752050643 1.68973949916845 -1.68973949916845 29.9523752050643
2038/12/25 09:05:11 19:05:05 -20:09:24 RJ 3.93071413148456 17.6609802526241 119.872280771226 -60.3008875702649 -26.0591756848748 14.8633563714536 -14.8633563714536 -26.0591756848748
2036/08/04 17:09:05 01:48:19 -01:06:11 RB 3.59626929893792 -70.7399145788677 353.522697464768 6.09242098118567 3.18397621634062 29.8305597576341 -29.8305597576341 3.18397621634062
2025/03/23 10:07:11 23:24:07 +46:59:13 RB 5.7226257257338 56.2217259054976 29.7253927530949 -136.439382127694 -20.6736486881575 -21.7393709641895 21.7393709641895 -20.6736486881575
2029/02/12 19:09:59 05:06:39 -41:24:41 RB 1.14404324012345 27.8963085439994 170.029326865341 -12.5250392593102 -6.50598755140031 29.2860397797487 -29.2860397797487 -6.50598755140031
2065/03/14 17:35:42 09:42:16 +37:07:11 RB 1.25243440597893 22.9843132732663 57.6367101686375 -81.725694225815 -29.6877127781283 4.31737304426649 -4.31737304426649 -29.6877127781283
2001/05/23 09:41:09 17:48:11 +46:19:07 RJ 0.371596869538897 -1.22784907403301 320.951983682132 59.0999158885385 25.7419245549868 15.4062753514718 -15.4062753514718 25.7419245549868
1997/12/07 17:52:48 00:05:41 -03:35:11 RJ 5.92837131449441 58.3127326170011 135.312468970043 -41.5196461354011 -19.8863045548759 22.4618541343034 -22.4618541343034 -19.8863045548759
2078/04/23 09:39:35 20:06:35 -12:06:42 RJ 6.13791386994785 32.3244617555623 240.849210863814 57.0960279088399 25.187466191023 16.2969796856986 -16.2969796856986 25.187466191023
2023/04/01 06:48:48 04:39:07 +41:52:16 RB 5.00055199461873 -20.0241668184735 27.3053857778567 -35.5074284363139 -17.4242550436213 24.4212066895729 -24.4212066895729 -17.4242550436213
2050/05/04 18:35:11 19:06:16 -32:11:13 RJ 2.38370532895034 -61.1546673158061 121.093823339086 -108.013238621147 -28.5295527061748 -9.27710204673808 9.27710204673808 -28.5295527061748
2068/04/12 04:01:11 09:07:19 +09:41:11 RJ 4.47547803486589 -22.9480793664547 289.952399996303 63.6790843654953 26.8897388695238 13.3019526209058 -13.3019526209058 26.8897388695238
2051/08/18 20:13:07 06:24:52 -16:48:13 RJ 4.63150532431309 -78.7544687023166 283.54755742825 72.8477015182255 28.6657266699943 8.84737896109229 -8.84737896109229 28.6657266699943
2073/11/01 09:01:05 10:01:48 +73:06:52 RB 2.99284715767585 36.2844988015304 353.237538061844 158.379811609294 11.0535642055859 -27.8894015416787 27.8894015416787 11.0535642055859
2029/04/09 16:52:52 23:13:42 -84:16:19 RB 1.50746850263116 -20.3663059366346 186.540602267149 93.9260503271001 29.9295975484099 -2.05406684171106 2.05406684171106 29.9295975484099
2059/11/19 10:09:13 10:16:06 -41:59:06 RB 3.59011375655453 12.2185648664947 215.58235241565 47.9722625143724 22.2846241586404 20.0847087633389 -20.0847087633389 22.2846241586404
2030/10/04 09:07:52 12:13:35 -09:16:11 RJ 2.53089632002868 41.8978300835756 123.892582814954 -52.3370235841951 -23.7485557920208 18.3304691100166 -18.3304691100166 -23.7485557920208
2006/09/04 17:35:48 18:06:19 +25:07:01 RB 4.23387031464377 62.2184873266459 73.4874062788768 -94.974301151817 -29.8870106942055 -2.60126733813462 2.60126733813462 -29.8870106942055
2029/12/12 05:05:06 09:24:48 +25:41:41 RB 2.65989001874005 79.2201019268175 302.633451774077 118.766314480803 26.2976928824898 -14.4371516947853 14.4371516947853 26.2976928824898
2077/12/07 17:06:16 17:39:13 +41:07:48 RJ 5.73548743578261 32.855474603548 306.792463057047 91.5818142395672 29.9885678116544 -0.828130790277932 0.828130790277932 29.9885678116544
2076/12/23 20:05:06 21:52:19 +49:52:41 RB 0.514148908444711 33.9920388701789 318.24311201241 100.063177803653 29.5384703801267 -5.24201942025935 5.24201942025935 29.5384703801267
2028/02/04 16:24:01 13:59:05 +84:59:07 RJ 0.267223076097796 14.8126951436872 358.712980602101 13.5921929635398 7.05029018942489 29.1597909499519 -29.1597909499519 7.05029018942489
2047/08/04 12:16:42 05:07:52 +01:59:41 RB 2.30571732401513 34.5278198214466 258.43885162154 67.2627189874802 27.6686038266323 11.5951870310429 -11.5951870310429 27.6686038266323
1984/04/04 23:24:24 09:48:13 +73:35:13 RJ 3.13262349569395 33.2194220163392 349.510296977907 142.484104693293 18.2694450593074 -23.795532715721 23.795532715721 18.2694450593074
2036/06/12 00:01:59 11:19:35 +58:42:07 RJ 4.47386347992317 18.912384700525 326.578238372161 82.7072681000905 29.7573165913312 3.80816350532483 -3.80816350532483 29.7573165913312
2070/03/25 19:05:19 14:13:39 +58:48:39 RJ 1.8301046647298 7.36074915563567 29.8719289125873 -63.6769789230209 -26.889250045801 13.3029407265609 -13.3029407265609 -26.889250045801
2011/12/19 20:24:16 09:06:35 +41:01:07 RB 0.507702653729844 -0.311459924768343 45.8930803810299 -63.1020952224804 -26.7544222164465 13.5720629185143 -13.5720629185143 -26.7544222164465
2016/11/14 04:07:01 10:05:35 -61:06:11 RB 1.92652435886597 2.2311085481665 161.489247947091 -38.6413482359929 -18.7333028550218 23.4321011465475 -23.4321011465475 -18.7333028550218
2073/01/07 05:59:01 12:48:07 -32:07:24 RJ 3.35303813860076 37.6536969905623 179.035376264577 -1.07617945159767 -0.563453111113466 29.9947082098089 -29.9947082098089 -0.563453111113466
2076/01/09 12:41:07 20:07:59 -32:48:05 RJ 5.13420138548651 36.8960713174479 170.492171873289 -10.6264356640517 -5.53214540837554 29.4855111398902 -29.4855111398902 -5.53214540837554
2048/03/09 09:01:42 08:01:35 +73:52:05 RB 5.20274818712706 3.46693263085382 358.53701852246 4.87470307638105 2.54931041470866 29.8914873569292 -29.8914873569292 2.54931041470866
2053/12/19 12:42:01 14:39:13 +49:01:11 RB 4.78256848801988 38.7605938674874 317.119673292224 104.634181398678 29.0267585779064 -7.5793988191636 7.5793988191636 29.0267585779064
2003/03/18 04:11:13 20:48:52 +49:52:06 RB 4.06753301677699 21.6708923502402 42.7956150952752 -84.6161717341616 -29.8676546105716 2.81481936606952 -2.81481936606952 -29.8676546105716
2029/08/14 22:41:19 22:01:48 +51:05:11 RB 5.21679320225416 49.5092820533237 30.8463236030268 -129.24637795635 -23.2329792001937 -18.9796911851422 18.9796911851422 -23.2329792001937
2028/03/19 12:35:16 12:39:52 +73:06:07 RJ 0.0233014937387122 2.95836320381444 357.387591130983 8.40609892589536 4.385649964543 29.6777033206497 -29.6777033206497 4.385649964543
2051/11/04 23:39:09 02:24:39 +38:06:06 RJ 0.59157147166884 71.2974164957952 7.54675109088185 -170.937505897544 -4.7253501286114 -29.6255137704316 29.6255137704316 -4.7253501286114
2062/07/18 13:59:01 05:01:48 -37:19:19 RB 2.46718076470433 6.4742650956889 226.59245025829 59.0497097462878 25.7284147327727 15.4288261166702 -15.4288261166702 25.7284147327727
2003/01/12 00:06:41 18:16:39 +25:06:48 RB 1.87945229208395 -43.302383620167 16.2378396651692 -16.8919292434848 -8.71702238201766 28.7056356973923 -28.7056356973923 -8.71702238201766
2075/08/25 19:48:39 11:19:09 -01:41:11 RJ 4.64799378789082 -5.94527907870292 269.909330896083 70.2838359336181 28.2412622196442 10.1208254723267 -10.1208254723267 28.2412622196442
2009/08/01 02:41:35 22:48:42 +29:59:24 RB 6.02448773146256 79.3133560286078 349.284228669528 168.310366541777 6.0783036292513 -29.3777845487135 29.3777845487135 6.0783036292513
2057/08/18 00:52:41 08:11:42 +09:11:52 RB 5.84677844129734 -48.3828657082527 49.268871049095 -46.1787687801837 -21.645111049568 20.7723173394754 -20.7723173394754 -21.645111049568
2047/05/01 05:01:19 20:13:48 +41:35:05 RJ 5.04970457471059 64.8684072865813 26.1756313017799 -146.218218424164 -16.6809407404708 -24.9348394021879 24.9348394021879 -16.6809407404708
2033/06/07 19:59:42 10:07:07 +32:16:35 RB 3.34036591666611 53.6204267944762 297.18301150684 99.8212390712976 29.5603420658581 -5.11724310048483 5.11724310048483 29.5603420658581
1991/01/14 16:52:48 16:19:07 -32:52:05 RJ 0.0312295746687141 -32.9994430972382 243.039411566859 86.4435940189353 29.9422264285124 1.86093430934449 -1.86093430934449 29.9422264285124
2074/09/19 22:35:06 00:35:09 -25:52:07 RB 5.81049623236137 32.1321853053975 139.511803738603 -42.4520791862303 -20.249199910081 22.135263788841 -22.135263788841 -20.249199910081
2048/07/19 11:24:13 11:19:16 +03:35:59 RJ 1.81387985074426 23.2019007427393 95.007028650868 -69.8415980809282 -28.1623040997366 10.3385022026383 -10.3385022026383 -28.1623040997366
2012/09/23 01:35:24 16:09:19 -18:48:07 RJ 0.368179486085669 -51.0001163175188 264.253014961779 81.481294174818 29.6690266229379 4.44396886210984 -4.44396886210984 29.6690266229379
1980/02/14 14:48:42 18:01:24 +32:07:13 RJ 0.0156618054106517 9.78981550547112 300.750879926598 72.6628820187826 28.6370385152887 8.9398000578242 -8.9398000578242 28.6370385152887
2058/07/09 09:06:39 08:05:11 -38:19:24 RB 1.03255590814783 6.92752375863977 135.433989379794 -57.6909248312343 -25.3553155712698 16.0345867511864 -16.0345867511864 -25.3553155712698
2055/08/01 17:19:39 11:52:19 +41:19:39 RB 3.58053741887427 59.7091524022063 319.326422045203 125.974025471059 24.278501352493 -17.6225529386919 17.6225529386919 24.278501352493
2032/12/12 12:48:06 19:35:41 +32:24:01 RJ 4.6903026204461 63.948872882135 55.7808577287496 -112.760055451631 -27.6639926822767 -11.6061840789702 11.6061840789702 -27.6639926822767
2053/10/23 23:01:24 17:39:07 -58:59:59 RJ 0.228792173992358 -25.6923125560527 213.075443546569 94.0300850740403 29.925818542822 -2.10840805872406 2.10840805872406 29.925818542822
2065/01/14 12:05:19 01:16:16 +38:05:09 RB 5.07513888526914 12.2903563200772 53.0172027288238 -74.3253287659152 -28.8843383037523 8.10524526182829 -8.10524526182829 -28.8843383037523
2034/04/23 22:05:48 00:41:11 +12:06:11 RJ 3.11377157342194 -55.6707770306582 338.281074521944 20.8724570755803 10.6886661683691 28.0312756673891 -28.0312756673891 10.6886661683691
2021/07/01 00:35:42 22:39:13 -03:06:48 RJ 4.9419010222184 29.6604223368731 105.625413680305 -65.1244804626173 -27.216714756697 12.6194468124634 -12.6194468124634 -27.216714756697
1996/08/19 23:59:01 13:24:09 -01:01:05 RB 5.64256792089868 -30.0731061584974 280.473085058083 67.7130480967546 27.7588832639467 11.3773634880221 -11.3773634880221 27.7588832639467
2010/07/07 03:59:24 03:09:41 -16:24:11 RB 5.93072558998488 13.8478334935116 113.191714886772 -64.205242372814 -27.0107576948458 13.0544616415357 -13.0544616415357 -27.0107576948458
2024/09/23 20:07:05 08:35:48 +18:35:41 RJ 5.23363091635236 -50.5649859854639 345.835550984671 14.0498363110658 7.28297324061198 29.1025480117554 -29.1025480117554 7.28297324061198
1980/05/23 18:41:05 21:09:52 +37:05:09 RJ 2.73546117072211 -30.2842072885883 17.9431330540961 -21.278506217787 -10.8870507845801 27.9548229329748 -27.9548229329748 -10.8870507845801
2044/03/18 12:07:16 14:05:52 +12:13:16 RJ 6.16887274208762 -40.4642171869898 306.737255088068 50.4226077635558 23.1229409246475 19.1135973326651 -19.1135973326651 23.1229409246475
2019/01/19 14:05:42 02:48:39 +49:39:52 RB 5.67294887389564 22.4664779039446 42.9929589803099 -85.5056071240346 -29.9077502155267 2.35084602763802 -2.35084602763802 -29.9077502155267
2037/07/23 07:24:07 08:16:01 -84:41:06 RJ 0.826266228449092 -18.417738168798 174.700994792501 -73.4659953114617 -28.7595301402147 8.53753044586554 -8.53753044586554 -28.7595301402147
2042/06/25 17:42:11 22:13:42 -51:06:42 RJ 3.04520207106331 -55.0225998899063 157.477794941247 -145.157017234443 -17.1398827679391 -24.6216250215396 24.6216250215396 -17.1398827679391
2030/04/18 22:11:13 21:52:39 +18:24:39 RJ 3.05099144486152 -43.767309895305 35.54775842909 -35.2337385396831 -17.3074017250919 24.5041597596467 -24.5041597596467 -17.3074017250919
2005/08/01 12:16:09 07:19:39 +16:59:41 RB 2.25456164231967 72.2362494725866 263.518228897714 77.6537143078347 29.3061948498175 6.41458832853155 -6.41458832853155 29.3061948498175
1992/04/07 17:11:39 03:13:07 +58:42:52 RJ 1.55457277939795 41.2237841417445 333.075641594382 124.947272616491 24.5903862149517 -17.1846706631089 17.1846706631089 24.5903862149517
2037/03/23 20:41:05 02:01:35 -37:07:42 RB 2.21446862647825 -16.0124564942693 236.101722172951 76.9335366710279 29.2232539553349 6.78243527517974 -6.78243527517974 29.2232539553349
1986/04/14 21:01:05 12:24:59 -86:39:24 RB 2.66930265021735 -17.216279152128 178.15049066191 -33.6440313820238 -16.6209443871327 24.9748715247915 -24.9748715247915 -16.6209443871327
2041/03/23 00:39:39 20:59:16 +20:42:09 RB 3.24160440067942 -26.3454211158414 52.7910266059097 -53.4040619503986 -24.0857922655032 17.885038745919 -17.885038745919 -24.0857922655032
2047/07/12 19:59:01 21:24:01 +46:06:11 RB 3.93280973238864 9.90265926764876 43.9333747627048 -71.5664658219133 -28.4607331647402 9.48613028191453 -9.48613028191453 -28.4607331647402
2033/02/23 17:07:35 00:05:09 -73:06:01 RJ 0.798233312655682 -7.21409783183153 191.882238830611 41.2563591561844 19.7828776918681 22.5529986970378 -22.5529986970378 19.7828776918681
1989/06/25 02:42:48 09:13:52 +03:07:59 RJ 5.3905891140292 -65.1892668973051 336.952201905896 21.6458279240474 11.0660434459618 27.884452342696 -27.884452342696 11.0660434459618
2000/08/19 08:59:05 19:16:24 +58:41:19 RJ 1.70556991410398 -10.9213232358886 354.002096120306 10.9049478278309 5.67540720599473 29.4582713859137 -29.4582713859137 5.67540720599473
2062/02/14 21:35:16 10:39:09 +41:09:07 RB 1.8141567963905 37.2175804619924 53.5438686294825 -95.1395134267301 -29.8793857064011 -2.68743554492246 2.68743554492246 -29.8793857064011
2001/04/09 07:35:19 19:09:05 -12:48:19 RB 5.34730646048232 52.7600676257974 210.572096521388 29.374944500918 14.7156817217292 26.1428520147433 -26.1428520147433 14.7156817217292
2053/02/25 20:42:52 04:16:06 +12:35:09 RJ 1.77721396159411 53.7969886848949 264.46298611724 73.7150641848779 28.796371547585 8.41243042725697 -8.41243042725697 28.796371547585
2000/08/14 17:39:24 10:19:06 -51:48:01 RB 3.89608364496696 -2.89019359491235 214.80112407251 60.8275065974542 26.1946856600682 14.6232158969981 -14.6232158969981 26.1946856600682
2012/04/09 19:39:52 10:16:09 -01:52:09 RB 2.24014898692739 55.9404779947694 127.273969946119 -48.5161421660018 -22.4742712086909 19.8722704701352 -19.8722704701352 -22.4742712086909
2021/11/25 23:01:13 05:16:19 -16:09:16 RJ 0.79133669791664 40.9285595149427 134.702809798269 -44.1108098520091 -20.8814481391726 21.53984966547 -21.53984966547 -20.8814481391726
2028/02/23 19:06:09 07:01:59 -37:09:39 RB 1.30345498492628 25.6433207708357 152.535468496622 -33.0451528648747 -16.3589938114804 25.1472328791051 -25.1472328791051 -16.3589938114804
2047/12/09 01:41:52 05:19:13 -41:01:19 RB 1.71289765751401 27.1012832231722 194.819163715863 18.5703547360028 9.55406652527214 28.4379994519775 -28.4379994519775 9.55406652527214
2002/05/12 15:48:05 09:13:59 -86:19:05 RB 1.78363218165551 -16.9334747916436 177.986286899325 -33.1683138651712 -16.4130115862107 25.112010088221 -25.112010088221 -16.4130115862107
1981/06/07 21:48:05 11:41:13 +49:52:16 RB 3.80749659906769 45.0475459265837 321.852976188972 116.045372520991 26.9533985537124 -13.1724829248223 13.1724829248223 26.9533985537124
2034/09/25 23:35:07 02:24:19 -41:01:05 RB 6.1709804739372 17.3510838837087 146.920624230307 -42.5805784513667 -20.2987924798061 22.0897945635935 -22.0897945635935 -20.2987924798061
1999/08/18 01:11:59 18:11:06 +25:24:06 RB 5.91484140767538 29.7606518557804 288.633954700043 80.749862413827 29.6098794168349 4.82234807127148 -4.82234807127148 29.6098794168349
2048/02/19 21:35:19 00:52:05 +51:13:16 RJ 1.88998665646494 12.6279209619698 320.497234222902 73.9446940139031 28.8298554976091 8.29695317492974 -8.29695317492974 28.8298554976091
2074/02/04 21:13:41 01:39:09 -25:39:35 RJ 1.54925730991597 13.8691902247819 236.166178750555 59.8012998908869 25.9285864180166 15.0900101512042 -15.0900101512042 25.9285864180166
2043/11/07 05:19:06 20:35:41 -46:01:52 RJ 2.11224982647672 -62.9850833521282 193.334818984829 161.839548420962 9.35037377396181 -28.5056224329027 28.5056224329027 9.35037377396181
2004/01/25 10:19:06 10:06:05 +18:48:52 RJ 4.7784286219992 -21.4308407922332 300.623364719879 58.7750773982275 25.6541654130592 15.5519708384306 -15.5519708384306 25.6541654130592
2038/07/23 13:13:42 12:16:01 +18:07:41 RB 2.35161297672105 42.4105685962635 83.8557937971297 -78.9612295502827 -29.4449353085046 5.7441957381327 -5.7441957381327 -29.4449353085046
2054/02/23 02:52:35 05:59:24 -12:16:05 RJ 3.33902549465807 -14.1731047650348 261.833983394498 72.3622326426928 28.5897344747626 9.0899440406732 -9.0899440406732 28.5897344747626
2010/12/25 19:19:35 14:41:05 +29:13:19 RJ 0.33208419478425 -36.8483607613612 336.580176556883 25.3551611382567 12.8468419434215 27.110120842238 -27.110120842238 12.8468419434215
2072/12/19 07:39:11 13:07:48 +51:59:48 RJ 3.46443741636223 58.2107355104661 359.087353607755 178.61750396499 0.723802991743569 -29.9912672161271 29.9912672161271 0.723802991743569
2066/03/04 08:07:48 00:48:41 -61:48:16 RB 4.87472300889486 -19.2353946633146 149.380014960551 -83.6329222154589 -29.8149541580684 3.32693681218587 -3.32693681218587 -29.8149541580684
2018/02/19 22:13:35 17:41:07 +03:35:24 RB 2.06185011361437 -50.813194993604 56.8601520674789 -52.1151665898793 -23.677399882098 18.4222890766383 -18.4222890766383 -23.677399882098
1997/07/12 18:39:19 10:42:09 +29:39:01 RB 3.58589754263821 48.8835904359517 292.37209660785 93.0768034511778 29.9567544789426 -1.61023634549414 1.61023634549414 29.9567544789426
2002/12/18 21:05:07 18:41:52 -38:41:09 RB 0.672176999522347 -33.1640915930355 236.136883447655 90.2112529083576 29.9997960841915 -0.110611513540999 0.110611513540999 29.9997960841915
2000/12/18 04:05:16 09:07:11 -73:42:48 RB 2.50157352838519 -3.83525826447657 181.810944895471 6.15982254267766 3.21906605424486 29.826793554427 -29.826793554427 3.21906605424486
2009/09/14 17:07:24 02:16:07 -03:07:16 RJ 4.28640474789641 -55.0556799694695 65.3908862612959 -58.9289878606446 -25.6958491971841 15.4830014543554 -15.4830014543554 -25.6958491971841
2070/01/07 11:48:19 13:05:05 -51:11:35 RJ 4.87669035114333 -10.6747482586723 218.800017795343 71.4822331733783 28.4467565020036 9.52796119407063 -9.52796119407063 28.4467565020036
2034/08/19 16:01:09 03:39:16 -16:48:41 RB 3.54825863678606 -59.0107443957539 270.708887357355 78.8984501820792 29.4386236807398 5.77645529574915 -5.77645529574915 29.4386236807398
2059/05/14 04:41:39 15:24:39 -29:06:48 RB 5.18877234928467 10.7257865225247 233.06880683142 59.7592197755018 25.9174967709191 15.1090489816335 -15.1090489816335 25.9174967709191
2080/01/19 10:24:06 14:06:42 +20:11:52 RJ 4.7073378204744 36.5649710188461 280.445589309098 79.5422579542597 29.5016713857696 5.44530857216235 -5.44530857216235 29.5016713857696
2074/05/25 22:39:16 10:01:41 -37:39:39 RJ 3.81599962750507 4.33415331245903 226.841742433239 60.5939649765122 26.1348629666773 14.7298655022033 -14.7298655022033 26.1348629666773
2037/09/01 05:09:35 00:52:52 -37:11:48 RJ 0.925754278924155 22.1338759248956 213.165473432747 40.1152934167688 19.3298333999396 22.9424833165588 -22.9424833165588 19.3298333999396
2065/08/25 13:42:07 17:59:13 +38:41:52 RJ 3.0517107253994 8.13817623398246 51.694011158541 -71.0720662008237 -28.3778197423217 9.73135893245616 -9.73135893245616 -28.3778197423217
2023/03/25 14:42:13 21:01:01 +25:52:39 RJ 0.668271802237557 14.7523035621865 292.72548396035 74.8221886138701 28.9535387589877 7.85446327458465 -7.85446327458465 28.9535387589877
1999/11/01 08:13:41 06:05:42 +37:35:48 RJ 2.76690376330281 29.7891796213721 302.743141491477 86.996774659559 29.9587976110277 1.57176515468533 -1.57176515468533 29.9587976110277
2035/06/14 08:59:19 21:39:52 +03:59:01 RB 0.563047309956406 23.8375145509615 265.951554689974 70.2441872141132 28.2342518400209 10.1403660207251 -10.1403660207251 28.2342518400209
2029/04/01 10:16:07 19:13:16 -58:06:16 RJ 5.9171429669357 1.81754421193434 203.958570706039 46.2101189645446 21.6564736718148 20.7604708063664 -20.7604708063664 21.6564736718148
2010/10/18 07:42:39 11:35:41 -01:35:05 RJ 2.39620211304084 48.0369137009728 116.517771735251 -57.3648866984362 -25.2636617121012 16.1786092386371 -16.1786092386371 -25.2636617121012
2046/12/23 03:42:06 21:41:13 +01:06:52 RB 2.48395018762571 -68.2174769360269 348.565779818408 10.7524126162543 5.59696214880681 29.4732762804684 -29.4732762804684 5.59696214880681
2030/09/25 02:19:24 05:09:52 -18:35:09 RJ 0.588920785501529 32.2080113482197 128.803797652465 -50.6489770512678 -23.1982759683566 19.0220922113203 -19.0220922113203 -23.1982759683566
2020/10/12 08:35:07 06:01:19 -37:13:35 RJ 2.53233570678031 13.2366159323039 221.811031956189 51.9628827080007 23.6283525332749 18.4851550321679 -18.4851550321679 23.6283525332749
2070/09/07 02:24:39 18:24:52 +46:19:19 RB 0.307573083483095 7.28346451219713 316.891872747001 68.804169096368 27.9705033648997 10.8467018726493 -10.8467018726493 27.9705033648997
2050/10/25 12:07:07 12:09:41 -73:42:05 RB 3.67908716310603 -5.6802946605326 187.105710927596 25.4023253912069 12.8691538672449 27.0995365041761 -27.0995365041761 12.8691538672449
1994/02/19 03:41:35 11:39:41 +18:06:07 RJ 3.47663231890275 66.9376493951355 269.869787991115 81.8450216321933 29.6966399905018 4.25553442877955 -4.25553442877955 29.6966399905018
2062/08/12 04:41:11 17:01:01 +16:39:19 RJ 0.456581101299986 -29.0324469511434 303.14626499276 55.2642593044064 24.6536630586596 17.0937678055507 -17.0937678055507 24.6536630586596
2008/01/14 10:06:05 21:59:41 -25:52:11 RJ 4.53287369632811 7.89736024372157 121.174249891922 -63.4146628466301 -26.8280638740943 13.4259073722231 -13.4259073722231 -26.8280638740943
2031/04/12 22:11:05 01:05:06 -18:19:06 RJ 2.94301898496302 -63.4679899139008 268.89024582418 81.8213837074997 29.6948818016446 4.26778570061319 -4.26778570061319 29.6948818016446
2065/12/19 16:05:41 07:11:59 -49:06:52 RB 5.67537613369004 -48.6599944216516 144.15741253667 -122.324129830066 -25.3511015537549 -16.0412483931647 16.0412483931647 -25.3511015537549
2032/07/12 00:42:42 19:16:06 +73:41:01 RB 5.16765622490063 35.8309422833289 357.439824368011 171.318982766122 4.52799936876634 -29.6563184113681 29.6563184113681 4.52799936876634
2066/10/18 17:35:11 13:06:11 +46:11:35 RB 4.99622215327891 15.0879615026706 313.552432216736 76.8969802628899 29.21892061246 6.80107919691885 -6.80107919691885 29.21892061246
2032/12/18 00:06:59 16:52:19 +12:19:06 RB 1.4634310599812 -56.563563886161 17.5858332774531 -16.9051061308785 -8.72362387554659 28.7036301968583 -28.7036301968583 -8.72362387554659
2017/11/25 17:19:48 00:48:39 -86:11:06 RB 5.58020924410589 -17.2295131382802 176.544080941116 -51.1029442723802 -23.3482625037388 18.8376919514177 -18.8376919514177 -23.3482625037388
2070/10/01 16:05:24 20:48:01 +61:09:06 RB 4.3114369769981 28.8777380141013 29.6408507371134 -101.886306578812 -29.3567471695165 -6.1791096142635 6.1791096142635 -29.3567471695165
1987/06/25 13:16:05 07:59:06 +12:07:35 RJ 1.86927045811585 75.7655137126035 120.670253630015 -55.8667166606532 -24.832035527158 16.83359770156 -16.83359770156 -24.832035527158
2053/01/19 14:07:05 19:06:07 +18:41:13 RJ 5.69224182832222 53.2263890746737 275.113612268743 81.7506740333259 29.6895922443858 4.30442938867724 -4.30442938867724 29.6895922443858
1984/04/18 06:24:42 15:42:07 +49:09:41 RB 5.19518476642504 33.148935429716 316.34427657371 97.7392025613717 29.7267387769545 -4.03992595064694 4.03992595064694 29.7267387769545
1984/08/19 09:05:48 12:07:59 -03:59:39 RB 1.7328106851646 5.00103074180843 96.2801332227197 -69.6521125675262 -28.127959216437 10.4315823496944 -10.4315823496944 -28.127959216437
2037/07/19 09:19:16 15:07:19 -20:01:09 RJ 1.26126685381935 -66.6193177799481 95.129631712628 -86.4963425742499 -29.9439269822515 1.8333676340525 -1.8333676340525 -29.9439269822515
2025/03/01 21:52:39 13:05:41 -12:09:01 RB 2.14758744726893 10.0055422632751 107.349383632995 -66.9167907846324 -27.5980930191176 11.7620262586059 -11.7620262586059 -27.5980930191176
2019/01/09 20:05:41 03:24:52 +46:16:24 RJ 0.792759126215765 63.0104655572337 9.34210130165405 -167.220493874322 -6.63599059766589 -29.2568560988307 29.2568560988307 -6.63599059766589
2076/01/09 13:42:13 06:07:24 +03:07:19 RB 5.40153034813047 -47.8926590930845 61.0367057757412 -55.5159511756819 -24.7285153344531 16.9853033341685 -16.9853033341685 -24.7285153344531
2022/11/01 09:01:01 20:52:13 -37:42:41 RJ 2.98140132117327 -53.0228043072562 126.884874412585 -108.180204338067 -28.5023971912468 -9.36020055086468 9.36020055086468 -28.5023971912468
2032/09/12 21:19:35 11:52:19 +16:07:06 RJ 5.36273525625391 -28.2473576858938 301.669054191334 56.3740441583352 24.9801138190742 16.6130645452938 -16.6130645452938 24.9801138190742
2035/04/19 19:24:11 11:06:11 +37:01:01 RB 2.33365169807046 55.9335014977368 53.334166392117 -110.050244535237 -28.1817699271167 -10.2853217633217 10.2853217633217 -28.1817699271167
2040/08/12 18:06:35 20:13:13 +20:13:11 RB 3.98599327717069 19.4485622325953 74.4834540029754 -75.4086349542829 -29.0324144537224 7.55770540490232 -7.55770540490232 -29.0324144537224
2025/08/04 00:05:41 00:42:09 +73:59:07 RB 5.39608871485371 26.2943827926033 15.4529737154279 -111.283453670292 -27.9538827391534 -10.8894646243818 10.8894646243818 -27.9538827391534
2078/06/09 16:39:41 09:39:48 +12:11:19 RJ 2.50131494381326 81.6016823990973 161.687044118085 -17.5780826413803 -9.06015733627462 28.5991879087842 -28.5991879087842 -9.06015733627462
2007/05/25 22:59:05 05:59:59 +61:24:41 RB 3.88948129233219 -0.0436678414559783 339.013134881991 44.7586547763095 21.1236597902616 21.3023706911998 -21.3023706911998 21.1236597902616
1996/06/01 00:05:05 06:11:07 -37:48:05 RJ 4.29288575098925 -60.6263259182 226.530077361668 120.219226815524 25.9231786280916 -15.0992983219769 15.0992983219769 25.9231786280916
2070/06/07 22:48:42 15:39:41 -20:06:39 RJ 4.08046646617965 49.7889280464102 176.86684875978 -3.1433699496484 -1.64503914517533 29.9548634817594 -29.9548634817594 -1.64503914517533
1985/09/04 00:19:24 03:06:24 -38:42:01 RJ 5.98394741631184 6.53711711801018 135.316967521052 -58.0189571169487 -25.4467014299773 15.8891593966953 -15.8891593966953 -25.4467014299773
2056/10/07 16:59:52 17:05:11 -12:24:01 RJ 4.65958071289534 56.270448159237 197.58063324696 16.9188110919609 8.73048943939008 28.701542713741 -28.701542713741 8.73048943939008
2029/09/12 11:35:16 16:41:39 +32:05:19 RJ 2.80152232830769 10.2156561817893 59.4688227842214 -72.9197924584483 -28.6768359632132 8.81130405439312 -8.81130405439312 -28.6768359632132
2079/12/09 05:35:39 12:52:39 -73:42:06 RB 2.73996592677827 -7.37989182916679 170.291914233834 -36.1435781381806 -17.6943218828944 24.2262455428926 -24.2262455428926 -17.6943218828944
2037/01/04 03:05:01 01:48:06 +41:09:01 RB 2.53526788937928 -5.36886185647026 317.941243950578 57.4032268941029 25.2744821760646 16.1617001188548 -16.1617001188548 25.2744821760646
2001/05/01 04:41:11 17:35:01 +49:59:52 RJ 4.96388708407662 55.6594539822094 336.311051104603 143.989605551391 17.6379603701991 -24.2673103985441 24.2673103985441 17.6379603701991
2040/12/12 01:05:41 11:24:39 +51:41:35 RJ 1.61801990141201 22.1399373243221 41.3209152199314 -85.5956933410502 -29.9114094838666 2.30381915272388 -2.30381915272388 -29.9114094838666
2076/06/12 22:41:39 00:06:07 +46:11:16 RJ 4.14493677103459 -7.15584341125124 34.5383830459033 -50.9290250039345 -23.2909738924533 18.9084778642031 -18.9084778642031 -23.2909738924533
2004/01/23 04:07:52 03:59:52 -49:07:01 RB 3.11977765916014 -33.3275197035338 223.723125550877 97.8970387622936 29.7154969652906 -4.12180059025212 4.12180059025212 29.7154969652906
2072/07/14 03:24:19 19:05:48 -32:39:13 RJ 5.91448690696922 18.3085371035346 223.829399214981 50.5997165694475 23.1819130081595 19.0420300725035 -19.0420300725035 23.1819130081595
2062/04/12 17:13:24 20:11:41 +37:01:13 RJ 1.64897367867245 -26.7564444260865 334.361260035113 30.7343306156901 15.3317410342871 25.7863862698432 -25.7863862698432 15.3317410342871
1996/09/12 20:09:59 10:41:09 -37:01:41 RB 5.05334815852519 -42.3479184474429 237.148278324734 96.7216807877801 29.7937929028763 -3.51139636904166 3.51139636904166 29.7937929028763
2056/03/19 19:11:59 11:13:07 +46:19:05 RB 1.76267145282095 28.7288265778191 47.8912088492681 -90.4792258988763 -29.9989506411642 -0.250919168242953 0.250919168242953 -29.9989506411642
1983/12/04 04:24:11 15:16:41 -29:52:41 RB 2.32830961982066 -14.9863207162891 116.980204687426 -75.4744288599655 -29.0410739765745 7.52436191893552 -7.52436191893552 -29.0410739765745
2048/09/25 23:35:48 19:13:35 -49:59:59 RB 6.18453815447865 0.36653060727549 215.466145158219 57.7794895842524 25.3800706772808 15.9953747194692 -15.9953747194692 25.3800706772808
1987/04/01 23:41:06 14:11:59 -51:09:13 RJ 3.14164031271081 13.4624054481029 159.531868625139 -31.5878507665155 -15.714158714948 25.5551406938296 -25.5551406938296 -15.714158714948
2018/06/12 07:09:48 01:41:59 -46:06:01 RJ 0.0514576373280842 21.0210208234987 163.291921652127 -22.916556353287 -11.6817036682707 27.6321877419561 -27.6321877419561 -11.6817036682707
2046/03/14 05:41:11 07:11:01 +46:42:52 RJ 4.40255464485063 -15.8779020862693 334.958736891948 35.4458727840115 17.3980080986656 24.4399123197847 -24.4399123197847 17.3980080986656
2042/11/18 20:35:07 01:09:59 -86:42:24 RB 0.0302581498230508 -16.1581467561016 179.005919038487 -14.3157590037735 -7.41796583293621 29.0684327561943 -29.0684327561943 -7.41796583293621
2029/09/14 10:52:01 09:19:01 -16:19:42 RJ 2.64669803312009 51.9925875746553 198.157302123842 17.7990145965384 9.17036789062015 28.5640395033805 -28.5640395033805 9.17036789062015
2037/03/19 08:01:48 12:07:09 -37:59:52 RJ 5.10677423398528 -27.8057366697423 236.41682000415 85.824416807729 29.9203679217502 2.18439543743847 -2.18439543743847 29.9203679217502
2021/01/07 11:19:13 12:35:48 -49:35:52 RJ 4.746967704823 -10.4758800700433 220.721470031102 71.6281180541119 28.4709240890551 9.45550006690603 -9.45550006690603 28.4709240890551
2032/03/09 22:59:52 09:35:41 -32:16:35 RJ 2.58459285706018 37.623365401194 184.061028382461 4.52703597826076 2.36788497614309 29.9064060150958 -29.9064060150958 2.36788497614309
2037/08/23 14:59:11 01:16:01 +16:06:05 RJ 3.35059174379584 -53.128542290875 347.896659345686 11.8593511715487 6.16529773783071 29.359650948264 -29.359650948264 6.16529773783071
2004/11/19 15:07:39 20:48:09 -03:48:24 RJ 4.90136741845318 51.2527888220589 124.155680839954 -51.2777389105486 -23.4056227003244 18.766374876626 -18.766374876626 -23.4056227003244
2074/08/12 02:05:05 08:35:52 +49:11:16 RJ 6.05832732620903 -13.8317719331857 23.9363426907799 -35.5128617028008 -17.4265707889706 24.4195542657313 -24.4195542657313 -17.4265707889706
1992/03/23 01:41:35 05:24:42 -38:13:06 RB 3.51042667362119 -35.0897500217998 236.640181066018 91.4508603149136 29.9903822387712 -0.759587501473067 0.759587501473067 29.9903822387712
2048/04/25 20:24:42 21:19:07 +37:01:09 RB 2.71639391126111 -31.0536667047586 13.7536154399989 -16.3604098051622 -8.45035570323972 28.7852651279908 -28.7852651279908 -8.45035570323972
2026/12/14 19:42:05 13:41:59 -46:13:41 RB 0.24341318822801 -61.2579845992026 198.34830549705 154.46290809959 12.9328595524365 -27.0691917832238 27.0691917832238 12.9328595524365
1991/05/04 12:42:01 11:42:13 -20:19:05 RB 0.826285672428472 -42.0909967291215 100.210412877547 -81.4092453383841 -29.6634149230819 4.48127382571983 -4.48127382571983 -29.6634149230819
1991/05/23 11:19:59 05:41:52 -51:42:05 RB 0.794221387534111 10.3377380654389 156.000572831911 -38.108647120353 -18.5146389874212 23.6052566850153 -23.6052566850153 -18.5146389874212
1983/10/23 16:09:06 19:39:19 -12:59:19 RJ 4.68998927041271 48.432399849185 140.138712519109 -38.2344469094276 -18.5664225044818 23.5645486989475 -23.5645486989475 -18.5664225044818
2075/07/07 16:52:41 19:01:48 +51:01:19 RJ 3.03515330423597 2.44769775035423 35.607010546867 -60.7978921951297 -26.187123876964 14.636753159514 -14.636753159514 -26.187123876964
2041/11/04 21:35:52 22:24:09 -37:01:06 RJ 0.0593578266038635 27.9936495060416 204.09236831687 28.6620384240373 14.3892670485858 26.3239243617756 -26.3239243617756 14.3892670485858
2058/06/07 10:24:13 16:42:11 -73:59:19 RJ 0.821434129316436 -34.000815660856 188.127576350124 150.970766434096 14.5576742195638 -26.231166983515 26.231166983515 14.5576742195638
2022/01/18 16:16:07 09:06:06 -51:01:24 RJ 6.2310634168725 -45.5913610596734 144.524261652097 -119.586224751777 -26.0884097872593 -14.8119841605381 14.8119841605381 -26.0884097872593
2079/09/14 08:35:19 21:41:19 -86:16:07 RJ 2.04661252558763 -23.2602153283754 182.407753861282 146.464210248793 16.5737329062432 -25.0062267755955 25.0062267755955 16.5737329062432
2007/08/09 06:07:52 08:19:39 -01:35:19 RJ 0.772562699721452 8.15077103429481 94.6917635953854 -69.7123276618434 -28.1389067695007 10.4020154690017 -10.4020154690017 -28.1389067695007
1996/01/19 19:59:01 22:13:24 +86:05:09 RB 0.928325846813594 20.3529343886739 356.121192094161 97.6098955542589 29.7357805040017 -3.97282743368424 3.97282743368424 29.7357805040017
2004/04/04 13:39:41 07:01:52 -38:13:09 RB 0.577038963965287 0.416795297415382 131.414999125921 -64.0305055169332 -26.9708195163626 13.1367764164499 -13.1367764164499 -26.9708195163626
2041/08/12 04:24:41 07:35:24 -86:01:39 RB 0.38581180873811 -19.5017391365315 175.984343305734 -84.4726341829829 -29.8605091991285 2.88963498884608 -2.88963498884608 -29.8605091991285
1980/10/01 02:01:48 09:59:05 -84:52:05 RB 0.616887919628601 -21.7683890389515 175.08618149221 -112.067434665467 -27.8022694997255 -11.2709276754238 11.2709276754238 -27.8022694997255
2023/03/12 04:05:35 12:01:35 +32:48:39 RB 3.94237949237906 48.3984226278238 296.867253167036 96.3077343130953 29.8183840034749 -3.29605452341308 3.29605452341308 29.8183840034749
1992/07/07 11:07:01 00:06:48 +12:09:06 RB 1.5246636684902 8.69491068632484 280.113424370049 71.4797091751089 28.4463367478922 9.52921432362178 -9.52921432362178 28.4463367478922
2026/08/23 17:19:39 12:09:13 -37:05:01 RB 3.95949819706751 19.525112669044 215.453027508687 43.4641448548914 20.6370152600277 21.7741498377637 -21.7741498377637 20.6370152600277
2023/08/19 23:59:42 11:59:06 +37:24:42 RB 5.63623586319824 -22.9336244433363 327.912733083412 38.7396644303172 18.7734833656969 23.3999214169173 -23.3999214169173 18.7734833656969
1999/09/14 11:42:13 07:01:52 -32:13:11 RB 2.85356682562089 14.2022212063501 227.39001363913 54.9886955250389 24.5711658461483 17.2121413241082 -17.2121413241082 24.5711658461483
1986/11/07 22:35:16 08:39:09 +29:48:09 RJ 0.359184789720059 -5.56341062497696 55.4795095042906 -63.3375171311933 -26.8099622918207 13.4620177503653 -13.4620177503653 -26.8099622918207
2074/01/25 16:07:09 13:05:52 -61:05:13 RB 0.0360652080900037 -46.5561307011407 191.191739102492 157.316279599985 11.5693171075648 -27.6794310213308 27.6794310213308 11.5693171075648
2000/07/19 12:16:19 13:13:11 +51:24:35 RB 2.03523436743948 20.1221165618822 41.445581682186 -83.0576609526687 -29.7800488522784 3.6261122922373 -3.6261122922373 -29.7800488522784
2060/04/04 12:52:39 11:48:52 -86:09:01 RJ 0.37878660947682 -23.0289006001215 178.453946410575 -155.537993617604 -12.4226923116269 -27.30708178716 27.30708178716 -12.4226923116269
2019/07/23 03:41:42 08:52:01 -01:11:59 RB 6.12529144490404 -49.3764381411341 67.7591874591731 -60.5796935429746 -26.1311931894621 14.7363748084463 -14.7363748084463 -26.1311931894621
2047/11/23 15:09:52 16:07:19 -12:59:11 RB 4.97286905658241 37.2837031922345 234.522438753166 51.9062423005807 23.6100672763816 18.5085040779836 -18.5085040779836 23.6100672763816
2069/08/25 05:35:41 04:35:05 +86:52:11 RB 0.923950823843663 22.400930110158 1.58174205648342 -149.926677769727 -15.0332356620743 -25.9615451298365 25.9615451298365 -15.0332356620743
1998/12/18 05:06:09 22:42:42 -58:01:05 RJ 2.75911267355592 -51.7482219156356 182.192089961855 176.10331512481 2.03872694086914 -29.9306463755893 29.9306463755893 2.03872694086914
2066/10/14 18:41:24 16:39:06 +03:01:01 RJ 5.21712889323433 39.9447659668617 256.590025869356 66.3858709920176 27.4879192964002 12.017249799958 -12.017249799958 27.4879192964002
2068/08/25 07:24:35 04:48:59 +46:06:09 RB 1.40460285203301 63.072676394675 350.749183300586 167.35845553604 6.56552427638138 -29.2727499729056 29.2727499729056 6.56552427638138
2048/02/09 08:39:52 06:59:42 +41:41:09 RB 4.60834289543938 -25.158181820135 341.451473229798 23.5656017726672 11.9939643234687 27.4980875663629 -27.4980875663629 11.9939643234687
2008/08/18 14:42:13 10:41:19 -38:24:48 RB 3.19086517101427 27.9815115254356 199.182828660736 23.3408451531453 11.8860043879539 27.5449251167895 -27.5449251167895 11.8860043879539
2029/02/19 21:07:06 17:13:41 -37:59:24 RB 1.77687688217984 -63.588777675782 138.405169966244 -127.503332190847 -23.7995380435213 -18.26422702758 18.26422702758 -23.7995380435213
2014/01/01 05:39:19 17:35:35 -16:59:16 RB 3.15147412794239 -0.528663539755944 107.929078736357 -69.4002827831358 -28.0818381672269 10.5551108544475 -10.5551108544475 -28.0818381672269
2043/08/07 02:19:48 02:13:42 +16:52:13 RB 6.02829849015015 43.3411285063431 84.5750493477662 -78.7912394755328 -29.427763363535 5.8315301096527 -5.8315301096527 -29.427763363535
2045/10/09 04:41:06 20:07:07 -49:19:35 RB 1.45602546383334 -46.6964354185034 218.00465055955 117.912932159294 26.5097997516123 -14.0438782795002 14.0438782795002 26.5097997516123
1994/05/07 00:39:42 19:16:39 -49:24:39 RB 4.00544519683778 2.55692733046311 145.393581067168 -55.0569379299833 -24.5916490138975 17.1828635208825 -17.1828635208825 -24.5916490138975
2007/08/04 22:35:06 00:16:05 -46:48:01 RJ 5.00595871306074 -6.04730233201161 137.768131677753 -67.3505550427934 -27.6863470572734 11.5527566677486 -11.5527566677486 -27.6863470572734
2056/02/04 07:05:05 13:16:01 +01:13:41 RJ 4.10855893249841 50.313200176968 245.96638061204 59.2384516061864 25.7791001727435 15.343988864818 -15.343988864818 25.7791001727435
2039/03/19 12:13:35 12:48:06 +25:09:42 RB 6.20005079545788 -42.2188398946734 337.866246363702 22.9585020262746 11.7019297871286 27.6236282782895 -27.6236282782895 11.7019297871286
2008/01/19 00:35:19 07:42:48 +58:41:59 RJ 2.1216309545039 50.9234074696375 355.323112590851 171.514084992322 4.42698832198257 -29.6715650816911 29.6715650816911 4.42698832198257
2056/03/25 11:09:24 02:19:41 -41:41:42 RJ 6.0376412578689 13.4822287002257 144.090969826115 -47.3968830804382 -22.0818079090005 20.3074803821155 -20.3074803821155 -22.0818079090005
2055/07/18 03:01:42 00:09:52 +32:06:13 RB 5.86912036290236 62.2555065469752 56.7815729843567 -110.755022317641 -28.0531245109711 -10.6311902048618 10.6311902048618 -28.0531245109711
1998/09/19 07:07:24 00:06:09 -16:48:09 RB 1.7413701389809 -12.4706947826745 256.701366575305 72.7438807611639 28.6496480434119 8.89930710722059 -8.89930710722059 28.6496480434119
2053/11/01 05:48:41 23:01:41 -51:41:39 RB 2.1483883120993 -43.6204984236841 216.889519068609 115.84812221142 26.9985872159295 -13.07961346309 13.07961346309 26.9985872159295
1985/10/23 12:48:11 05:48:09 -18:09:41 RB 3.81982416677717 -44.197019901498 263.611823232807 79.6848157542384 29.5151285429888 5.37188859628058 -5.37188859628058 29.5151285429888
1982/05/01 12:39:41 13:42:59 -32:39:05 RB 0.767416641612925 -69.5107721584292 132.921749104264 -124.946929047805 -24.5904892607507 -17.1845232088907 17.1845232088907 -24.5904892607507
2061/09/09 15:52:35 15:11:13 -29:13:19 RJ 3.87995670925987 40.3362715200781 172.651312754607 -7.94296499332524 -4.14561813452457 29.7121835327312 -29.7121835327312 -4.14561813452457
2013/08/25 22:19:48 03:41:06 +12:09:52 RJ 5.30970622172015 -15.1562378182031 70.699739124578 -65.2874979563074 -27.2525092962328 12.5419590598416 -12.5419590598416 -27.2525092962328
2071/03/04 01:52:52 21:52:16 -58:11:42 RJ 3.23059815415829 -43.8742807498134 154.419231295034 -130.231342916712 -22.9032847521531 -19.3762624765921 19.3762624765921 -22.9032847521531
2013/09/04 09:19:06 19:41:06 +32:35:35 RB 2.06596034595374 -37.371636646855 2.62219931010763 -2.93339485768986 -1.53525105741471 29.9606909831984 -29.9606909831984 -1.53525105741471
2063/01/14 04:41:39 11:39:01 +20:48:24 RJ 3.12506706569798 86.6769454250932 281.543050102474 100.346266475194 29.5122100190613 -5.38789938573631 5.38789938573631 29.5122100190613
2029/01/25 01:07:41 07:09:48 -16:48:01 RJ 2.38222150822298 43.7403974148513 219.513941093599 38.7129438507134 18.7625684882286 23.4086741129123 -23.4086741129123 18.7625684882286
2010/09/18 09:39:11 05:13:05 -25:09:16 RB 2.38998871631642 17.8944157754194 233.848628831827 57.0037836961231 25.1611959889467 16.3375094921415 -16.3375094921415 25.1611959889467
2026/04/23 17:35:06 19:07:13 -49:52:07 RJ 1.92833977159537 -59.891359193797 175.91832800613 -174.042241490341 -3.1138566176524 -29.8379606703391 29.8379606703391 -3.1138566176524
2066/12/19 15:05:59 12:01:13 -58:39:59 RJ 5.41001260586796 -36.4446982290696 209.899170413505 114.293897787015 27.3434129781452 -12.3425186532815 12.3425186532815 27.3434129781452
1988/12/23 04:05:13 23:41:16 -51:16:05 RB 2.58585905013888 -51.9420522598514 208.273520808321 134.851564426865 21.2680890090709 -21.1581754861386 21.1581754861386 21.2680890090709
2077/10/07 21:13:24 20:05:11 -84:11:19 RB 5.76738904977555 -14.0469599213645 182.391155108087 21.2678460636493 10.8818494675073 27.9568480370465 -27.9568480370465 10.8818494675073
1982/07/09 14:16:39 15:42:13 +32:41:19 RJ 2.37867129818833 3.34667181975817 56.2945183936263 -68.5138166363537 -27.915177634761 10.9883054935572 -10.9883054935572 -27.915177634761
2048/01/09 05:48:24 17:48:59 +16:01:05 RB 3.32483935710719 16.4910910952736 78.5348896786208 -73.5582357096644 -28.7732374239453 8.49121947338992 -8.49121947338992 -28.7732374239453
2032/06/23 01:06:13 03:39:35 +16:13:16 RJ 4.94369273214911 -30.663132769764 55.8733568382575 -54.2382664149082 -24.3436294991403 17.5324756611434 -17.5324756611434 -24.3436294991403
2017/03/23 20:59:16 15:35:06 +09:41:07 RJ 2.2913649613644 -8.46236031640046 76.5041860180646 -68.1002955678298 -27.8351453397962 11.1894898861575 -11.1894898861575 -27.8351453397962
1990/11/18 22:39:13 17:48:41 -25:05:41 RB 0.566323254123572 -38.9174551241491 253.223923785733 84.0878157605326 29.8404280580786 3.0901218925208 -3.0901218925208 29.8404280580786
2031/12/12 03:06:07 13:09:13 -49:05:01 RJ 2.13095213992902 -6.02949519385603 140.505221565886 -66.4336360251923 -27.4979280089789 11.9943301277317 -11.9943301277317 -27.4979280089789
2013/12/14 03:24:59 16:11:09 +86:59:19 RJ 2.25407527826788 18.6377512714668 2.97450932034984 -66.637259032335 -27.5403808207621 11.8965299246208 -11.8965299246208 -27.5403808207621
1986/10/25 07:16:39 19:48:11 -01:41:42 RJ 2.39953712756708 -62.9187627825392 50.6139627902186 -46.6707427129576 -21.8226737776919 20.5856967137971 -20.5856967137971 -21.8226737776919
2068/06/14 21:42:48 10:11:07 +38:11:16 RB 3.92092712196102 27.4048631981406 302.931553373488 85.246318628051 29.8968053952795 2.48616716187562 -2.48616716187562 29.8968053952795
2041/04/12 02:05:48 01:19:59 +09:16:48 RJ 3.96258912806246 -50.7479604649639 44.0909358556638 -41.5794593928282 -19.9097425038352 22.4410818239892 -22.4410818239892 -19.9097425038352
2009/04/09 16:41:01 09:52:39 -84:59:09 RJ 1.45291775288347 -17.6432531681361 175.288169146229 -63.2236232535318 -26.7831492297649 13.5152845821368 -13.5152845821368 -26.7831492297649
2067/06/25 07:35:24 05:01:05 -16:42:24 RJ 0.389527285937884 25.8901998890773 120.849274632146 -57.4401962086758 -25.284905029045 16.1453887432968 -16.1453887432968 -25.284905029045
2030/09/07 18:11:16 04:24:24 +58:48:11 RB 4.44394832955716 -11.0028623738399 3.55998120011106 -6.50770782554415 -3.40010624755573 29.8066985341438 -29.8066985341438 -3.40010624755573
2015/03/07 20:52:07 13:39:11 +25:35:13 RB 1.97602148081365 6.30854503044053 65.3896667153058 -71.0292780704061 -28.370544511736 9.75254859550094 -9.75254859550094 -28.370544511736
2028/12/18 11:09:52 02:05:19 -73:41:24 RJ 4.36322473130808 -32.2404002657738 167.95642170738 -136.10152464043 -20.8014796250564 -21.6170868853406 21.6170868853406 -20.8014796250564
2039/11/07 03:19:42 21:42:41 +51:42:42 RJ 1.58922776468806 -3.80710813300511 329.578277291942 50.532244447602 23.1594728414162 19.0693161310967 -19.0693161310967 23.1594728414162
1993/12/19 16:52:16 18:59:07 -84:01:19 RB 5.86950891026104 -15.7775848431109 184.720954479631 47.31385498909 22.0523568888707 20.3394580961705 -20.3394580961705 22.0523568888707
1998/12/07 22:35:41 16:48:01 -09:16:52 RB 0.878687902817483 -66.0399154352668 292.669500446383 61.6160692696039 26.3934579787641 14.2613244799778 -14.2613244799778 26.3934579787641
1993/03/12 04:09:41 02:07:19 -41:35:19 RJ 3.96500793778389 -64.5773272963456 152.418046932609 -144.359576770154 -17.4808944847441 -24.3806958065852 24.3806958065852 -17.4808944847441
1990/04/01 21:16:24 16:59:11 +18:13:48 RB 2.51331381257487 -12.6001131059829 65.1547140135927 -63.9562474301411 -26.9537709748846 13.1717208531561 -13.1717208531561 -26.9537709748846
1989/10/18 20:52:06 13:39:52 +16:52:06 RB 5.85170674428506 -28.6859106160578 303.009920917759 55.4354159375128 24.7046162854574 17.0200450701022 -17.0200450701022 24.7046162854574
2020/05/19 12:06:42 21:39:48 -38:13:11 RB 0.94645986841322 -10.6653225010941 233.379696708833 73.1001408430748 28.7044289595508 8.72099524745326 -8.72099524745326 28.7044289595508
1996/07/14 10:09:11 06:24:24 +29:07:48 RB 1.39252628920035 71.9528707175368 55.7848201275259 -117.08912427785 -26.7089777644123 -13.6612776408405 13.6612776408405 -26.7089777644123
2020/02/04 12:41:13 04:42:13 -09:35:48 RJ 5.57438277603449 -23.2587596641724 92.1364290376808 -72.4282341799407 -28.6001866112507 9.05700423990388 -9.05700423990388 -28.6001866112507
1986/12/12 03:52:42 11:24:19 +41:09:35 RB 2.33294034881155 51.5187352198687 48.2249295081501 -111.728645116396 -27.8684279982407 -11.1063369707061 11.1063369707061 -27.8684279982407
2074/01/19 04:07:05 03:59:24 +86:39:24 RB 3.06554646407013 19.1151004364957 356.839464241302 76.1157491796077 29.1234743279386 7.19883623024607 -7.19883623024607 29.1234743279386
1989/03/25 16:09:11 04:01:13 +58:42:07 RJ 1.05288314557993 51.1452813267122 359.79294356418 179.625306091386 0.196187873375618 -29.9993584984469 29.9993584984469 0.196187873375618
2012/09/01 02:07:05 18:48:48 -01:09:09 RB 0.128345821118512 4.96028703354357 267.061565394539 69.9967955990827 28.1902047311164 10.2621809191682 -10.2621809191682 28.1902047311164
2034/07/23 03:42:16 00:39:41 -37:05:39 RB 6.13415697319277 30.6268824206081 161.880537900091 -21.3805672501106 -10.9368294412623 27.9353854774323 -27.9353854774323 -10.9368294412623
2005/05/18 01:09:09 22:07:19 +03:41:52 RB 4.32921751070461 6.529888369353 88.1226226755307 -70.4755570081244 -28.2749699923217 10.0262690934019 -10.0262690934019 -28.2749699923217
2016/12/07 07:09:39 14:05:13 -16:35:13 RB 3.12126126696097 40.7720870497764 135.962510259398 -43.114285999557 -20.5036742833077 21.8997566398356 -21.8997566398356 -20.5036742833077
2068/12/23 11:06:39 02:48:59 -46:24:39 RJ 4.44052950345244 -53.1300760524546 142.806898927384 -124.854775816483 -24.6180966396393 -17.1449502140245 17.1449502140245 -24.6180966396393
1981/10/18 12:13:41 01:09:59 -61:09:35 RB 3.58230444454637 -48.4969012843285 174.563416761829 -169.408592389059 -5.51411826520893 -29.4888877334716 29.4888877334716 -5.51411826520893
2048/11/12 09:48:05 17:11:24 -18:07:59 RB 3.38878536953884 15.6406679145779 116.516364538084 -62.415272335547 -26.5898112144624 13.8917939654765 -13.8917939654765 -26.5898112144624
1980/07/23 00:42:19 19:48:52 +58:07:52 RJ 5.34812648317249 51.1392191170026 352.180648716633 165.989982466322 7.26274612148732 -29.1076024223023 29.1076024223023 7.26274612148732
1983/04/18 01:05:09 18:16:06 +84:39:52 RB 3.78402572404079 22.7611758767259 4.77389469950932 -122.56281428158 -25.2840566675135 -16.1467172649416 16.1467172649416 -25.2840566675135
2025/10/23 05:09:07 10:11:52 -25:41:41 RB 1.81672328046331 23.3886586303978 131.669342488788 -51.4726171708733 -23.4693166489105 18.6866577009687 -18.6866577009687 -23.4693166489105
2076/12/19 09:39:09 12:01:35 -84:05:42 RB 3.98982258651879 -16.1932994214848 183.925155279019 45.2128536147773 21.2918638451314 21.1342502587715 -21.1342502587715 21.2918638451314
1986/05/14 11:13:52 14:05:06 -01:59:16 RB 0.6161566280442 -72.0006579241874 11.5697926898914 -10.8834462626407 -5.66435190959374 29.4603991392561 -29.4603991392561 -5.66435190959374
2043/01/07 15:59:06 14:39:35 -32:16:06 RB 5.96594678458358 -35.823418068163 243.430150962639 88.0477249384503 29.9825865032724 1.02201104386902 -1.02201104386902 29.9825865032724
2013/07/14 21:11:41 04:16:07 +41:52:11 RB 4.28916199217733 -28.1561691380914 0.522965282828283 -0.662217100019195 -0.346728343051083 29.9979962573523 -29.9979962573523 -0.346728343051083
2039/11/19 10:05:05 23:11:42 +03:24:16 RB 3.56932372187861 -47.8722371569554 59.5901687975626 -54.4099666626254 -24.3960602898915 17.4594456479001 -17.4594456479001 -24.3960602898915
2017/02/19 15:52:19 09:35:24 +38:41:13 RJ 0.397893245689542 -9.81893405753069 42.6474682037154 -54.6470628920175 -24.4681003094532 17.3583428715571 -17.3583428715571 -24.4681003094532
2068/10/09 09:41:19 00:19:52 -09:01:13 RJ 2.7769624678488 -61.8258311985408 289.569749885643 63.7105261998359 26.8970344468547 13.2871945106824 -13.2871945106824 26.8970344468547
1981/07/12 11:13:41 00:07:24 +58:59:39 RJ 1.63392589872604 15.8609504025923 327.529137251588 77.7901781022547 29.3213896106097 6.34477039007957 -6.34477039007957 29.3213896106097
1984/02/01 22:19:06 22:19:24 -20:42:59 RB 1.76315200470783 -39.2888675384623 259.21383568167 80.721359210289 29.607476756594 4.83707764127774 -4.83707764127774 29.607476756594
2073/03/19 22:39:11 17:01:48 -12:07:05 RJ 2.66724253086198 -16.7827388064217 97.2508863005158 -72.7157621944228 -28.6452771555646 8.91336618124103 -8.91336618124103 -28.6452771555646
1989/02/07 15:01:24 21:35:24 -18:35:39 RB 6.2481754012931 39.5120675479504 222.890814392286 42.4425854546565 20.2455318881015 22.1386187140902 -22.1386187140902 20.2455318881015
2047/02/12 13:24:52 03:39:16 +61:48:35 RJ 5.91106228609392 23.4308467193486 29.9621602096766 -91.5771624730248 -29.9886349476325 -0.825696056449135 0.825696056449135 -29.9886349476325
1984/09/01 11:07:01 04:05:59 -20:24:35 RB 2.48680213676812 1.50326939245413 247.751958153475 68.1954051271304 27.8536812541938 11.1432688466973 -11.1432688466973 27.8536812541938
2012/03/01 22:09:59 06:41:05 +20:41:48 RB 2.2260494054329 65.2981756006092 276.464252938706 87.2137226308479 29.9645342538824 1.45831647728027 -1.45831647728027 29.9645342538824
1993/05/19 20:11:39 06:24:42 -84:24:13 RJ 3.06046726010689 -18.6573403569379 185.797637007114 77.0340859732242 29.2351115561361 6.73114049030892 -6.73114049030892 29.2351115561361
2054/11/12 07:52:06 15:42:48 +03:09:24 RB 2.87340875697778 17.5340467557877 93.3659256401921 -70.0978533749389 -28.2082612140254 10.2124433551091 -10.2124433551091 -28.2082612140254
2017/07/23 12:01:13 18:24:05 +37:35:09 RB 2.03598995552695 -29.5260263728949 18.1819856520623 -21.7564014264226 -11.1198362202939 27.8630443855986 -27.8630443855986 -11.1198362202939
2061/09/07 11:05:35 18:13:19 -20:07:35 RJ 2.58984778098115 -39.3507983272276 100.19321193226 -80.3961401986243 -29.5795440074807 5.00505507557258 -5.00505507557258 -29.5795440074807
2046/06/25 16:42:59 15:41:52 +32:42:48 RB 2.7868147477752 21.3297670411908 61.9064162232479 -79.4377875928244 -29.491693637418 5.4990914155595 -5.4990914155595 -29.491693637418
1981/11/04 06:09:41 02:06:39 -03:39:07 RB 2.28215224544656 -9.35877747704079 269.633540031633 70.4733774366434 28.2745885655826 10.0273446857594 -10.0273446857594 28.2745885655826
2049/08/18 11:07:42 16:59:35 -16:06:48 RJ 2.25352723908441 -39.1901162356068 95.0686175252921 -77.3503814852911 -29.2718244766557 6.5696492910857 -6.5696492910857 -29.2718244766557
2060/11/23 21:48:19 11:16:52 +12:11:59 RB 0.447548475428327 -43.3833520527682 50.6089652672018 -47.9174580943325 -22.2654025902734 20.1060152067248 -20.1060152067248 -22.2654025902734
2065/03/04 02:35:05 04:16:07 +41:48:52 RJ 3.42315450932618 -13.4015298305062 324.756529963079 46.9025237361412 21.9057711725238 20.4972483357407 -20.4972483357407 21.9057711725238
2066/11/19 07:35:13 05:19:19 -03:06:16 RB 2.92169653818359 2.64732487437958 265.854581796686 69.9792748049785 28.1870652839156 10.2708008782314 -10.2708008782314 28.1870652839156
2041/12/25 09:01:52 08:35:05 +37:06:24 RJ 3.92094355957299 7.79965483543014 306.563153812348 71.0172147114721 28.3684905293431 9.75852166501509 -9.75852166501509 28.3684905293431
2064/07/19 15:39:06 00:48:39 +38:11:42 RB 2.93119052993196 -26.5740781233668 337.715046485563 27.2485929036077 13.7355624624775 26.670851576944 -26.670851576944 13.7355624624775
2064/03/23 01:48:41 01:01:01 -51:59:05 RB 3.55113811786774 -57.9769457967139 171.785725673962 -167.56583069817 -6.45953231663161 -29.2963213091745 29.2963213091745 -6.45953231663161
2020/04/07 23:13:05 00:11:07 +86:09:01 RJ 3.13955107820515 16.0877466630298 359.785593987532 3.09316262594425 1.61878957609575 29.9562935008376 -29.9562935008376 1.61878957609575
2079/11/12 02:24:06 02:42:39 -73:39:19 RB 1.43740255456584 -6.89278075958351 191.202588456167 38.9733959488246 18.868784171055 23.323142667877 -23.323142667877 18.868784171055
2049/02/23 14:39:01 07:41:39 -84:19:42 RJ 0.150406825542055 -21.159495566106 174.245228602824 -102.906470505356 -29.242079279327 -6.70080587851589 6.70080587851589 -29.242079279327
2064/02/19 10:01:48 02:11:42 -25:42:41 RJ 5.1409690712605 -16.2540218958006 111.721511840802 -75.3716258075458 -29.0275266377858 7.57645677692636 -7.57645677692636 -29.0275266377858
2038/02/14 01:52:09 23:07:52 -41:52:41 RB 2.91759270306005 -68.4023517683629 182.063590917201 177.411510743031 1.35486880988435 -29.969389892155 29.969389892155 1.35486880988435
2055/04/18 13:01:06 23:11:59 +32:39:52 RJ 0.643015179959245 45.29748014588 297.254376209742 94.6255569382891 29.902290190441 -2.41930596796957 2.41930596796957 29.902290190441
2027/07/18 03:01:48 19:06:35 +41:48:59 RB 5.8659099142787 43.4283231629882 309.575117093141 102.874760164119 29.2457833607002 -6.68462082836356 6.68462082836356 29.2457833607002
2042/07/25 00:07:09 04:59:16 -46:41:19 RB 5.22842095086069 -45.3013581885031 137.127135845959 -111.454617445351 -27.921227176423 -10.9729245400932 10.9729245400932 -27.921227176423
1980/01/12 18:59:05 07:24:35 +20:05:06 RB 0.543462820641667 15.230574040585 73.7972032526509 -74.0468114850502 -28.8445972444213 8.24555697373571 -8.24555697373571 -28.8445972444213
2017/05/09 07:05:42 15:41:06 -16:41:52 RJ 5.73599344495407 -8.44323060724818 255.153902321681 71.7392112441971 28.4892042251005 9.40027885865712 -9.40027885865712 28.4892042251005
2043/12/01 05:16:11 12:07:16 +32:13:35 RJ 2.51235842409245 53.5859573669588 62.6317564145957 -99.9609446040679 -29.5477767200345 -5.18930543550771 5.18930543550771 -29.5477767200345
2019/06/25 04:48:39 09:11:48 -51:52:24 RJ 5.9365344710422 -53.5568911007572 157.089406920229 -143.539362861165 -17.8281116246741 -24.1279596298187 24.1279596298187 -17.8281116246741
2078/10/19 11:07:59 02:24:59 -61:07:13 RB 3.32080932502833 -45.0678104583665 198.344031532064 142.982015389974 18.0619703318211 -23.9533969977624 23.9533969977624 18.0619703318211
2058/12/09 17:06:06 16:09:11 +25:42:09 RJ 5.76230245680803 10.7739035480572 293.460556047099 73.0434597718513 28.6957874846046 8.74938744361067 -8.74938744361067 28.6957874846046
2071/07/18 05:16:13 06:19:41 -46:13:35 RB 0.176694614316612 -11.288220660825 135.363872370102 -73.0444838082626 -28.6959438561348 8.74887456794077 -8.74887456794077 -28.6959438561348
1999/02/19 03:16:35 18:13:05 +18:52:42 RJ 3.3635077689365 14.8328233720058 74.9119867078389 -73.7282552903787 -28.7983075632996 8.40580046691564 -8.40580046691564 -28.7983075632996
2013/12/23 22:19:39 05:41:07 -25:13:01 RJ 1.09018968012493 39.7317944302264 152.674800844839 -28.5101527795076 -14.3194343597845 26.3619763981349 -26.3619763981349 -14.3194343597845
2064/04/09 19:52:09 15:59:11 +25:48:59 RB 2.30085940141079 -7.63640005731338 59.3366873018846 -63.7043089923077 -26.8955924851761 13.2901130496071 -13.2901130496071 -26.8955924851761
2046/12/04 18:39:35 07:41:39 +46:07:52 RB 6.08382601140798 -9.49350859531822 33.4768024668775 -48.204423092176 -22.3658235634877 19.9942475809153 -19.9942475809153 -22.3658235634877
2021/11/18 00:11:06 07:42:05 -12:59:39 RJ 0.959469997641776 21.7502422294912 113.709301002735 -62.1475623000864 -26.5246128855009 14.0158806813671 -14.0158806813671 -26.5246128855009
1998/07/12 05:06:19 10:19:05 -01:48:59 RB 0.024613438810634 -58.9804820400399 58.3131219378639 -53.228410638999 -24.0308491156302 17.9587942463245 -17.9587942463245 -24.0308491156302
2029/06/18 07:52:39 21:05:59 +61:19:05 RB 0.348073150778408 30.0959418718107 330.716902979919 104.389993564207 29.0587973659443 -7.45562174771426 7.45562174771426 29.0587973659443
2023/04/09 20:52:16 12:07:16 -84:59:35 RJ 2.54537992422398 -15.8713888406177 176.99666406862 -35.4361408686959 -17.3938566317096 24.4428670878752 -24.4428670878752 -17.3938566317096
2026/06/07 22:05:52 14:35:39 +29:35:39 RJ 3.88714647581225 79.8319749455649 342.422866150009 160.952861647559 9.79037822137541 -28.3575121313986 28.3575121313986 9.79037822137541
2025/05/12 19:06:07 10:48:24 -46:52:24 RB 2.65755562914244 22.215712872168 172.206396295223 -10.8379676697114 -5.64096590091189 29.4648859442006 -29.4648859442006 -5.64096590091189
2000/03/23 16:07:07 09:39:59 +46:52:16 RJ 1.01511484854633 16.4374753542309 45.3719009108073 -78.3331017298823 -29.3801941303509 6.06664593188826 -6.06664593188826 -29.3801941303509
2065/08/04 22:39:41 16:59:05 -29:11:05 RJ 5.04245187687601 31.4391833327617 213.93140803147 37.0161442931981 18.0612009492634 23.9539771284504 -23.9539771284504 18.0612009492634
2080/07/12 17:09:06 22:11:09 +12:16:24 RB 3.20663387782852 -46.3441929658327 43.7666407598643 -41.8855891547104 -20.0293598269994 22.3343848117781 -22.3343848117781 -20.0293598269994
2007/11/07 02:09:05 04:07:59 -32:06:59 RJ 1.2760654149952 37.0210684737831 191.716253011777 13.0315033385181 6.76460295028383 29.2273869328924 -29.2273869328924 6.76460295028383
1981/01/23 03:07:35 14:52:59 +61:07:39 RB 2.86582890834361 32.0038968705094 29.4201832598804 -107.613282682838 -28.5936163443405 -9.07772572578959 9.07772572578959 -28.5936163443405
2068/06/12 22:39:24 07:06:48 +38:41:16 RJ 4.13416082897612 -14.5630896182163 321.118706180809 49.053617768694 22.6596958212279 19.6605743885937 -19.6605743885937 22.6596958212279
1994/09/09 10:24:41 19:01:13 +49:52:09 RJ 2.43206965177191 -14.0634258112206 21.8543900118891 -32.9094858746641 -16.2994034927028 25.1858977561267 -25.1858977561267 -16.2994034927028
2045/04/12 06:11:52 06:19:19 +49:13:19 RJ 5.03982797815057 -20.001268647679 9.03972066817848 -13.0748704622678 -6.78672319322793 29.22225843939 -29.22225843939 -6.78672319322793
2072/12/14 04:06:41 07:19:24 -25:35:13 RB 2.44867947605934 36.2702270318215 212.975895667845 34.6705110104813 17.0656892096232 24.6731078666744 -24.6731078666744 17.0656892096232
2058/12/04 14:24:19 03:24:16 -09:59:24 RJ 4.96844318259668 -38.0934407172239 86.9608758793732 -72.4192644678975 -28.5987683776425 9.06148151694618 -9.06148151694618 -28.5987683776425
1988/09/14 06:16:42 18:11:48 -51:42:05 RB 1.4408471257909 -56.9500399452434 192.717045669021 160.482486851852 10.0228492023311 -28.2761824486144 28.2761824486144 10.0228492023311
1981/05/12 04:13:48 00:52:13 +73:24:52 RB 5.03063346519974 20.2962069442668 17.4827985568828 -88.6192785902421 -29.9912896201303 0.722874070268193 -0.722874070268193 -29.9912896201303
2041/06/25 21:24:05 02:07:41 -84:09:39 RJ 4.02021922308358 -25.5184181521647 177.817680528578 -160.097937746396 -10.2124018167489 -28.2082762524275 28.2082762524275 -10.2124018167489
2000/03/14 22:48:13 10:13:48 -01:09:42 RJ 2.61521375036824 68.7160133358443 170.004165223645 -9.39995152335203 -4.89975382576299 29.5971689937893 -29.5971689937893 -4.89975382576299
2029/06/04 07:16:52 07:16:59 -46:05:48 RJ 6.23385545952572 -29.4514468025612 132.598863655713 -91.4718279634244 -29.9901022561145 -0.770562565789484 0.770562565789484 -29.9901022561145
2071/04/01 20:52:41 08:09:11 +58:59:59 RJ 2.41608808521858 49.611956869137 348.313607703296 158.427482970804 11.0303557805794 -27.8985886982449 27.8985886982449 11.0303557805794
2055/10/19 23:41:35 18:52:41 +16:11:16 RJ 0.327419527652928 0.961928834321707 286.959896342167 69.6104190170117 28.1203608143969 10.4520480130991 -10.4520480130991 28.1203608143969
2018/09/12 17:42:35 00:24:07 +09:06:19 RJ 4.40271145499522 -18.8844238359932 72.3559665900676 -65.2573393419813 -27.2459038463735 12.5563021464988 -12.5563021464988 -27.2459038463735
2027/05/25 03:35:06 06:19:16 +32:52:42 RB 5.08264455888342 -35.3878200894253 15.60519198631 -17.5298060500203 -9.03605686278484 28.6068117128162 -28.6068117128162 -9.03605686278484
2012/10/01 00:35:01 10:48:19 +61:41:39 RJ 0.24160534555662 -4.73649600769688 14.4569635422326 -29.6150201014085 -14.8250936369777 26.0809623797674 -26.0809623797674 -14.8250936369777
2052/03/25 03:52:24 20:24:06 +12:05:06 RJ 4.12519974696422 22.463621861368 84.5406486829081 -73.399604681202 -28.7496180990806 8.57084938363858 -8.57084938363858 -28.7496180990806
2051/03/19 15:11:35 16:48:01 +49:52:39 RJ 0.697396955314686 -14.7119537339241 339.052175240457 31.3947364149385 15.627936497554 25.6079597162371 -25.6079597162371 15.627936497554
2048/03/12 06:42:13 05:11:11 -38:13:13 RJ 4.64407892733449 -70.4236202725915 161.243598127536 -157.371431049954 -11.5426682404981 -27.6905545247797 27.6905545247797 -11.5426682404981
1985/11/19 19:07:09 04:41:11 -37:59:05 RJ 5.94238246030124 -11.8295358404083 126.389045137881 -73.9760033211155 -28.8343850671579 8.28119784806901 -8.28119784806901 -28.8343850671579
2060/01/01 05:42:13 14:42:11 -16:59:11 RB 3.16164694328638 35.2004396579333 130.112109338787 -48.9518206194316 -22.6247292227435 19.7008027145478 -19.7008027145478 -22.6247292227435
2029/08/18 01:39:41 16:13:42 +32:41:48 RJ 6.04880453980861 0.374910519612896 304.804332154451 66.5174368510875 27.5154415021257 11.9540988344207 -11.9540988344207 27.5154415021257
2005/04/23 10:16:19 10:01:41 +61:05:24 RJ 0.0099611873458052 -5.57349116325818 14.0865905798485 -28.2498376887406 -14.1995151521286 26.4267623715897 -26.4267623715897 -14.1995151521286
1985/09/25 15:16:19 09:42:05 +41:42:01 RJ 3.98627232638319 18.0656908955926 308.843718964479 79.226431587328 29.4712076585596 5.6078444295518 -5.6078444295518 29.4712076585596
2016/01/01 13:52:48 00:16:59 +73:24:35 RB 5.30221343403652 26.8315828793754 15.9522555565271 -112.229682627347 -27.7702414960505 -11.3496117665334 11.3496117665334 -27.7702414960505
2042/07/09 08:59:59 18:11:09 -51:24:11 RJ 1.00127845982351 -47.5858599403682 213.177827401246 124.406731110234 24.7514136072204 -16.9519180166819 16.9519180166819 24.7514136072204
2035/07/25 17:09:13 06:48:59 -38:59:19 RJ 3.41180933440375 -14.542316114411 233.26908862515 76.0720475848744 29.1179750402202 7.22104767724964 -7.22104767724964 29.1179750402202
2032/05/01 03:06:39 04:11:11 -16:24:06 RB 4.55887091539877 -72.9992087177273 80.5160051820748 -75.0714785397046 -28.9874388176632 7.72841450701453 -7.72841450701453 -28.9874388176632
2070/10/01 07:09:41 01:05:19 -84:35:11 RJ 1.96753239159809 -20.3817453692339 186.139687983988 94.4754984106114 29.9085238292762 -2.3409832023358 2.3409832023358 29.9085238292762
1993/09/04 18:13:13 20:13:06 -84:16:01 RB 4.40019610657091 -16.286717353314 175.07092994035 -52.2201861019196 -23.7111269709635 18.3788589897972 -18.3788589897972 -23.7111269709635
2056/06/01 12:09:09 13:01:39 -12:13:35 RJ 1.18566470000046 -39.9202748315451 89.944789907014 -74.513086586985 -28.9107439918448 8.01054816089453 -8.01054816089453 -28.9107439918448
2038/04/25 16:11:59 06:19:07 -01:19:59 RB 1.60059524676206 68.402387395612 168.569502662927 -10.7480324248801 -5.594708936989 29.4737040751646 -29.4737040751646 -5.594708936989
2030/10/19 03:41:16 23:59:41 -46:24:42 RJ 1.35997212507067 -6.03581133774499 222.786751341353 67.5041466472524 27.7172167771458 11.4784970326567 -11.4784970326567 27.7172167771458
2073/08/18 01:09:52 03:39:01 +29:05:11 RJ 5.924140015807 20.9865968264514 65.2174318116994 -78.3959739715946 -29.3868335384388 6.03440258717375 -6.03440258717375 -29.3868335384388
2007/02/07 20:19:06 13:16:39 +61:35:11 RB 1.34881435883824 3.151730403482 23.9668266332461 -52.6785865579429 -23.8574083856732 18.1885696281816 -18.1885696281816 -23.8574083856732
1982/03/09 22:42:06 20:07:24 -41:24:42 RB 2.49141227863174 -62.4133325244696 145.64427827601 -135.022018031855 -21.2050499079449 -21.2213538305539 21.2213538305539 -21.2050499079449
1980/11/01 16:07:07 06:52:16 +09:39:39 RB 4.84865749905528 -60.0294491851166 348.732724860802 10.7445563873596 5.59292080718326 29.4740434424016 -29.4740434424016 5.59292080718326
2029/06/12 11:39:52 06:41:41 +29:59:09 RJ 1.23898942387664 61.1520947890675 63.53175686113 -103.594045247458 -29.1595630097387 -7.05123287667349 7.05123287667349 -29.1595630097387
2008/11/04 11:11:07 10:05:07 +12:16:05 RB 3.60905700205217 37.0105821755683 270.262095423052 74.0873915443369 28.8504299716071 8.22512555851856 -8.22512555851856 28.8504299716071
1998/03/09 23:24:52 09:24:13 -49:16:35 RB 2.68072437676746 19.8566289788231 188.330873713425 12.1125988765168 6.29500693277397 29.3321135910171 -29.3321135910171 6.29500693277397
2046/11/01 21:48:41 03:24:06 -49:42:11 RB 0.0603079412096625 8.62763008902076 150.525510195955 -45.2952911326329 -21.3222498843108 21.1035935297998 -21.1035935297998 -21.3222498843108
2054/04/18 02:19:41 18:24:42 +29:09:41 RB 4.12401707541872 51.8232429410952 67.5881099656795 -94.7556387924764 -29.8967204418272 -2.48718853793403 2.48718853793403 -29.8967204418272
1987/07/25 18:19:42 15:41:48 +03:01:06 RB 3.71376599465497 61.7750644869441 123.908342018935 -51.4215558208489 -23.4526539934663 18.7075658668558 -18.7075658668558 -23.4526539934663
2046/10/09 09:01:19 13:59:59 -03:59:07 RB 2.59039545915423 23.6965017549923 104.374451716844 -66.0709950387239 -27.4214622728721 12.1681307856817 -12.1681307856817 -27.4214622728721
2020/04/19 21:59:11 23:35:07 -37:48:48 RB 3.02265145180885 -72.347997251619 183.970214709845 175.295353621335 2.46057990830221 -29.8989221630958 29.8989221630958 2.46057990830221
1981/09/14 21:01:07 19:52:59 -18:48:39 RJ 5.30507904747564 50.8773370475422 188.987891493754 8.93448635064822 4.65915034930865 29.6359969972757 -29.6359969972757 4.65915034930865
2003/01/14 01:52:09 05:42:19 -09:11:05 RJ 2.37530340555291 32.4708268774584 244.50169694402 59.3324558597391 25.8042400977764 15.3016728816266 -15.3016728816266 25.8042400977764
2014/01/25 14:01:48 13:01:42 -16:13:59 RB 5.76284184457638 -46.2221914977151 266.437257074782 78.4189481938824 29.3892508254805 6.02261869264475 -6.02261869264475 29.3892508254805
1998/02/07 09:11:52 10:06:24 -86:06:35 RJ 4.71572607789196 -21.6411088708543 183.689615719874 117.123497110778 26.7007772970901 -13.6772984076241 13.6772984076241 26.7007772970901
2034/07/04 13:06:59 07:16:42 -09:09:11 RB 1.9949064417668 60.5824659775312 188.29834274048 7.90837769563662 4.1276812574733 29.7146806719625 -29.7146806719625 4.1276812574733
1984/01/07 23:41:41 00:39:42 +51:48:13 RJ 1.69440266779716 17.0234912397186 319.66603280972 79.367420961978 29.4849178021355 5.53530687508047 -5.53530687508047 29.4849178021355
2001/10/18 07:07:06 22:35:35 -09:06:52 RJ 2.24359841074656 -58.7706256216733 285.66765137988 66.5419940974563 27.5205625588392 11.9423044863635 -11.9423044863635 27.5205625588392
2052/01/14 09:05:42 01:19:01 +32:48:16 RB 4.27458340328774 -22.4346883545762 38.6596148722907 -44.7033885649814 -21.1031021822929 21.3227361819187 -21.3227361819187 -21.1031021822929
2080/10/19 00:16:39 17:48:09 +09:07:48 RJ 0.479968359018127 -23.8123743521162 290.063223493904 63.501556933373 26.8483945823571 13.3852048303362 -13.3852048303362 26.8483945823571
2040/07/01 02:48:39 20:06:11 -18:07:09 RB 5.53045208910142 49.8892056085917 200.982075028894 20.7258493741023 10.616905188655 28.0585338928304 -28.0585338928304 10.616905188655
2001/03/12 07:01:01 19:41:52 -49:24:16 RJ 4.71556238931805 17.2190599824353 163.074922600744 -24.8876712385255 -12.6252188452832 27.2140377215272 -27.2140377215272 -12.6252188452832
2068/07/07 23:39:35 08:07:16 +25:09:39 RJ 4.82755248119812 -38.4552060259231 329.204311503162 32.0894418888037 15.9372740435538 25.4165948950812 -25.4165948950812 15.9372740435538
1991/01/01 11:59:52 00:35:41 +61:13:01 RJ 4.80911417166759 15.7454628010206 30.0092258138683 -77.381942604771 -29.2754388969111 6.5535240438425 -6.5535240438425 -29.2754388969111
2059/09/18 18:19:59 13:35:59 +37:16:07 RJ 4.67078182799411 33.063047779484 301.879134372085 89.0571815600334 29.9959384450761 0.493636302560925 -0.493636302560925 29.9959384450761
2037/12/23 11:11:11 04:06:24 +73:11:16 RB 4.45176291020503 3.51157287108928 3.09399020497917 -10.2448168306376 -5.33563575321242 29.5217037297823 -29.5217037297823 -5.33563575321242
1987/07/07 06:41:13 12:41:48 +84:59:07 RB 0.348057008237553 14.6763312172937 0.880528507074846 -9.15251199746959 -4.7718891785339 29.6180531714661 -29.6180531714661 -4.7718891785339
2063/10/18 18:09:13 04:59:09 +86:13:41 RB 5.14038450087478 16.5942136538041 1.83082945552922 -28.0288712845682 -14.0974925865401 26.4813274359962 -26.4813274359962 -14.0974925865401
2059/06/18 03:52:01 12:59:41 +41:39:13 RJ 5.57371673191514 -9.69306447261951 320.701322733798 52.5214263962464 23.8074281382863 18.2539410933731 -18.2539410933731 23.8074281382863
1994/09/14 09:16:48 17:16:11 +20:52:39 RJ 2.22107644129346 -27.6553402481535 51.9183734691279 -52.4275252643381 -23.7774800884257 18.2929341726395 -18.2929341726395 -23.7774800884257
2059/11/09 08:35:05 22:39:16 +51:35:59 RJ 3.0062268157818 -17.5220168174229 7.60483887823429 -11.6446226624615 -6.05522296258608 29.3825505168181 -29.3825505168181 -6.05522296258608
1980/11/25 04:09:11 20:48:19 +12:07:11 RB 2.12037257132191 -56.1742476849122 340.512577530044 18.7312833757293 9.63390352800409 28.4110524763359 -28.4110524763359 9.63390352800409
2008/02/01 04:42:06 12:13:35 -29:48:42 RJ 3.42501233685024 38.8288506131679 194.203708393775 15.4357475462702 7.98472730465318 28.9178859854991 -28.9178859854991 7.98472730465318
2048/06/07 10:42:11 07:05:42 -46:42:19 RB 0.90744937652381 6.90653795433182 145.594727818854 -51.0301570571587 -23.3243127075241 18.8673378281523 -18.8673378281523 -23.3243127075241
2070/09/01 15:35:06 01:19:11 +49:13:01 RB 3.66279070870593 -19.952038513851 5.63690086831183 -8.23822612028832 -4.29867757690423 29.6904255794662 -29.6904255794662 -4.29867757690423
2028/02/23 20:24:35 06:01:13 -09:24:24 RJ 1.64662193942513 60.5417194129269 187.436608726746 7.08955175126018 3.70261549993683 29.7706338269716 -29.7706338269716 3.70261549993683
2003/01/25 22:19:11 08:39:59 +37:48:19 RJ 1.6499520161552 54.3828503335119 52.0180724157259 -110.222754396031 -28.1506745198246 -10.3701265218366 10.3701265218366 -28.1506745198246
2073/02/23 13:52:48 13:06:11 +61:05:59 RB 6.23449730090784 -8.0218820617598 349.926390991422 19.4852129281123 10.0069070486901 28.2818282881231 -28.2818282881231 10.0069070486901
2028/04/19 11:41:01 14:59:07 +16:41:41 RB 0.319225254975969 -44.8761206242872 321.44669705869 37.671055378179 18.3338175808625 23.745970877428 -23.745970877428 18.3338175808625
2061/12/01 23:06:09 23:06:13 -51:41:42 RB 0.921574502384205 -0.61972224785843 214.58197956392 58.2270632537096 25.5042450596612 15.79662887887 -15.79662887887 25.5042450596612
2015/04/23 14:59:13 03:39:35 +84:35:07 RJ 1.2405099360877 24.9926751189623 358.441996603196 164.128448737995 8.20444971477532 -28.856316550761 28.856316550761 8.20444971477532
2025/02/07 08:42:01 18:05:19 +25:19:07 RJ 4.59307849609123 80.4869642076662 53.1851956403214 -123.575766699234 -24.9946567095044 -16.591176449367 16.591176449367 -24.9946567095044
2001/02/04 08:05:01 06:13:41 +03:59:42 RB 4.37628371110103 -56.939915010124 313.540486310174 43.1239586546048 20.5073711009761 21.8962949040894 -21.8962949040894 20.5073711009761
1999/09/12 02:42:39 22:07:35 -03:01:19 RB 0.458412732882952 32.6521755310854 253.014480185386 64.2589036223361 27.0229722049473 13.029158576464 -13.029158576464 27.0229722049473
2016/04/19 12:39:07 07:06:01 +58:52:13 RJ 0.571733582544791 25.0616204343297 33.3155585301081 -92.8604561547562 -29.9626211119038 -1.49710924935215 1.49710924935215 -29.9626211119038
2079/11/12 23:35:19 10:19:07 -51:07:52 RJ 0.716133467070056 -30.5640726179982 138.905813517104 -96.3229760116019 -29.8175061391818 -3.30398662828025 3.30398662828025 -29.8175061391818
1990/01/19 00:09:06 15:19:05 -86:13:16 RJ 2.01312050306363 -21.31364000552 176.245700979146 -112.315462698975 -27.7532183688037 -11.3911751006425 11.3911751006425 -27.7532183688037
2049/03/25 13:59:39 23:59:41 +25:19:05 RJ 0.494246011103898 63.7967468261613 287.902583309691 96.9881723035496 29.7771386476531 -3.64993341835763 3.64993341835763 29.7771386476531
2040/08/07 14:48:39 09:16:07 -12:05:07 RJ 3.03396953151683 43.4127216167078 229.338051314539 46.9077223968491 21.9076308744755 20.4952606586919 -20.4952606586919 21.9076308744755
2011/11/01 03:42:41 01:05:07 +18:13:07 RJ 1.58295428502182 20.4174477167542 282.808453755967 75.0435733579604 28.9836713536913 7.74253155377585 -7.74253155377585 28.9836713536913
2019/03/09 10:19:01 15:01:11 -18:19:05 RB 5.52408007528063 -6.41406986819725 252.517975965321 71.2025309823698 28.3999048484716 9.66671632964169 -9.66671632964169 28.3999048484716
2003/01/07 22:09:19 06:05:48 -51:07:48 RJ 1.29713116998844 17.461517414434 168.811505553641 -16.9106999031535 -8.7264261620296 28.7027783749003 -28.7027783749003 -8.7264261620296
2051/06/07 00:16:07 05:01:13 +51:13:05 RJ 4.43888830469681 -18.8702606911389 358.695138692708 1.96291092845469 1.02757672063429 29.9823962698649 -29.9823962698649 1.02757672063429
2070/10/19 20:01:41 18:35:35 -41:13:05 RB 5.65488463823385 17.1993908685876 212.532168719801 42.1725405242257 20.1409642640761 22.2337931652071 -22.2337931652071 20.1409642640761
2073/11/09 13:24:35 02:09:19 +49:35:13 RB 4.28335364259736 -14.8284583583108 19.9844259678986 -30.1249844786328 -15.0566384585487 25.9479794652407 -25.9479794652407 -15.0566384585487
2057/11/04 14:48:52 13:07:11 -38:13:41 RJ 4.56392855726495 6.4650369624388 224.987747893119 58.2302053070422 25.505111295421 15.795230223336 -15.795230223336 25.505111295421
2035/11/07 21:35:59 09:01:11 -37:24:06 RB 0.102075846301301 -43.5224490036088 123.707351638549 -98.2610164444664 -29.6887133621111 -4.31048708412479 4.31048708412479 -29.6887133621111
2043/05/07 06:48:35 09:39:35 +61:41:19 RJ 5.62162213324097 -8.62946791939155 358.256503987842 3.4384878185671 1.79930750773268 29.9459929288146 -29.9459929288146 1.79930750773268
2057/03/23 06:42:06 23:42:19 -25:16:05 RB 4.82955175348536 0.184553412706462 116.419976987852 -67.9958506088407 -27.8147016877344 11.2402121876033 -11.2402121876033 -27.8147016877344
2003/10/18 01:11:24 03:59:39 -84:42:05 RJ 0.679001179727819 -14.8503512564739 178.038996302758 -20.3443843027804 -10.4298625545006 28.1285969627748 -28.1285969627748 -10.4298625545006
2023/10/14 01:07:01 10:19:19 +18:42:52 RJ 0.593726642503443 -20.6586922353035 60.1184881735421 -59.3860413451605 -25.8185395952427 15.2775329542725 -15.2775329542725 -25.8185395952427
2070/01/04 22:59:59 00:48:39 +29:01:19 RB 1.48061916328001 25.6831670630587 294.24348232369 80.8477466385718 29.618074711713 4.77175548109795 -4.77175548109795 29.618074711713
2017/03/18 13:48:41 01:35:41 -29:41:48 RB 0.321433063256273 40.4464819361973 172.815122831879 -7.75866997596895 -4.05002598560989 29.7253644135086 -29.7253644135086 -4.05002598560989
2028/03/19 14:16:41 08:39:01 +25:19:01 RB 0.467026747321177 -3.72959763795157 61.6879911724656 -66.0766317008872 -27.4226592203597 12.1654330495886 -12.1654330495886 -27.4226592203597
2034/01/01 04:01:16 13:19:01 +61:24:39 RB 2.72513229228985 38.5160950966502 25.7162131055128 -122.741285347916 -25.233638553392 -16.2253963081574 16.2253963081574 -25.233638553392
2080/01/18 21:52:35 07:19:48 -18:05:35 RB 1.41924956916531 41.7524454435007 140.378970931927 -39.201559493289 -18.9615118528962 23.2478185697599 -23.2478185697599 -18.9615118528962
2030/05/19 14:39:24 18:39:24 -20:41:24 RB 1.60745920157263 -81.5617257993749 263.124644310594 93.77555056305 29.9348897416268 -1.97544328104964 1.97544328104964 29.9348897416268
2031/02/18 00:39:05 14:07:42 -09:41:19 RJ 2.66161940911838 24.0578446230973 111.089193999626 -62.9782715866547 -26.7250287670654 13.6298509676198 -13.6298509676198 -26.7250287670654
2021/01/09 20:59:48 20:59:48 +46:41:59 RJ 1.03839737126833 5.00254665475036 318.25093099949 66.174161420844 27.4433276606671 12.1187361927414 -12.1187361927414 27.4433276606671
2003/07/12 01:59:13 11:59:59 +84:41:41 RB 5.48552807063029 15.9132107725557 355.791575092784 45.0656654985795 21.2375015088036 21.1888774989041 -21.1888774989041 21.2375015088036
2028/10/07 12:16:41 02:13:24 +86:59:52 RB 3.41695783551008 17.3987457649327 358.871761211113 23.6239417820171 12.0219573481177 27.4858607563969 -27.4858607563969 12.0219573481177
2074/12/09 09:24:19 01:48:39 +61:24:42 RJ 3.74402760189483 -8.252076978949 2.87689836251641 -5.73117770774535 -2.99583594408545 29.8500413231896 -29.8500413231896 -2.99583594408545
2031/11/18 15:06:05 05:13:48 +61:09:06 RB 4.86813263197291 -7.57615480635355 8.90478935782798 -17.613166396366 -9.07766769268732 28.5936347682686 -28.5936347682686 -9.07766769268732
1993/05/09 21:59:11 02:42:16 +09:59:01 RJ 3.35892828872239 -49.4044131506549 314.670524375205 42.7849118649426 20.3774418605456 22.0172628457782 -22.0172628457782 20.3774418605456
2048/09/18 10:11:11 11:05:05 -58:52:09 RJ 2.54370737834634 9.17148073587527 169.246636394686 -20.0062550574191 -10.2636818638905 28.1896582915942 -28.1896582915942 -10.2636818638905
2018/01/01 10:52:13 01:52:48 -25:19:07 RB 4.52093615196537 -43.6957525963469 106.063450205109 -85.864851939407 -29.9219020557617 2.16327930822754 -2.16327930822754 -29.9219020557617
2054/02/01 06:16:16 11:11:39 -84:06:19 RJ 3.85173534348204 -16.364563662996 184.639549213724 51.1958981651463 23.3787931211836 18.7997880891487 -18.7997880891487 23.3787931211836
2024/09/04 12:24:42 10:06:24 +20:35:16 RB 2.88372821470283 78.1563406947238 274.095106036756 89.7649420069286 29.9997475383404 0.123075732122253 -0.123075732122253 29.9997475383404
2036/06/01 18:59:24 12:07:11 +01:42:19 RJ 2.97795075951051 68.4331115439242 146.732095813899 -31.0795730074807 -15.4868406354443 25.6935355125049 -25.6935355125049 -15.4868406354443
2067/05/23 11:16:01 07:07:11 -46:52:06 RB 0.787087692721379 2.71102022640501 142.76843010056 -56.6806608707461 -25.0686600383035 16.4791469404204 -16.4791469404204 -25.0686600383035
2058/10/18 08:42:48 15:16:16 +73:11:01 RB 2.66567910797716 22.9688085869176 18.1868462797229 -97.0938547801181 -29.7703556691056 -3.7048513242709 3.7048513242709 -29.7703556691056
1981/07/09 20:24:16 03:59:24 +86:07:59 RB 3.99126800537912 16.1414993450269 359.06016553717 13.5061535671682 7.00649385139196 29.1703452826738 -29.1703452826738 7.00649385139196
2027/10/18 05:13:24 01:06:11 -20:24:24 RJ 1.74115967767019 -0.447096404631853 248.579743690532 68.985923488943 28.0047706188005 10.757919064037 -10.757919064037 28.0047706188005
2021/01/07 03:59:06 18:13:42 -32:13:05 RB 2.82133867643735 -29.3879250965582 116.546132847288 -83.9524598386852 -29.8330446716839 3.1606084251794 -3.1606084251794 -29.8330446716839
2000/07/18 00:01:07 20:19:06 -25:11:07 RB 5.08451880891498 43.0735207826074 162.270839788327 -18.4299656790605 -9.48435767050157 28.4613239252498 -28.4613239252498 -9.48435767050157
1989/05/25 00:09:42 16:01:48 -18:39:52 RJ 4.18744264635488 51.5367408788445 179.41854697851 -0.577275456600902 -0.302255608395074 29.9984773204773 -29.9984773204773 -0.302255608395074
2011/08/07 10:07:07 14:42:52 -25:07:39 RJ 1.78552469044909 -33.4799610047531 107.685264550011 -82.0483400511112 -29.7115540698219 4.15012707709642 -4.15012707709642 -29.7115540698219
2053/12/18 02:06:52 21:59:16 +01:42:52 RB 1.98641358742538 -47.2588825816251 296.790135675021 57.1823286559339 25.2119846451987 16.2590230410768 -16.2590230410768 25.2119846451987
2020/11/07 20:09:11 22:39:59 -09:01:01 RB 6.01633530658871 61.2873370588753 187.815915877565 7.43511963080282 3.88210261039534 29.7477609127538 -29.7477609127538 3.88210261039534
2006/10/23 12:19:05 15:11:11 -18:41:06 RJ 3.69107902675765 48.2427958142734 156.320187897747 -23.5100512136671 -11.9672981903288 27.5097032703691 -27.5097032703691 -11.9672981903288
2070/01/25 00:42:07 22:09:48 -46:07:39 RJ 2.27153887378999 -57.7004928323097 211.186550311879 135.692471383162 20.9552796245901 -21.4680286904792 21.4680286904792 20.9552796245901
2071/06/09 12:05:41 12:42:09 -38:09:11 RB 1.29730907982651 -33.7286960834935 124.093223335978 -90.7921592670906 -29.9971327597108 -0.414760408291703 0.414760408291703 -29.9971327597108
1989/06/25 03:13:35 02:11:01 -29:59:07 RB 5.525274487493 1.17230151087052 122.391676336446 -66.2608221829659 -27.4616260405087 12.0772138844707 -12.0772138844707 -27.4616260405087
2029/05/18 23:06:05 09:52:07 +46:42:42 RJ 3.81129970013065 27.9594644829446 313.05418735253 90.1316451665805 29.9999208126809 -0.0689291873869403 0.0689291873869403 29.9999208126809
2048/08/12 21:35:52 08:19:07 -29:06:41 RB 4.90288176345959 -65.5009410491461 242.066709222771 107.395166909297 28.6279665063799 -8.96880893483569 8.96880893483569 28.6279665063799
1989/06/18 03:11:07 18:09:09 +16:41:39 RJ 5.39406027407741 54.9044958069039 271.030516112411 79.1001789983418 29.4587791034009 5.67277125724565 -5.67277125724565 29.4587791034009
2051/06/25 16:48:35 21:59:39 +25:48:24 RJ 2.80766582032671 -43.0323668314773 12.7662190294042 -13.3793340418728 -6.94191056568495 29.1857821155787 -29.1857821155787 -6.94191056568495
1998/12/09 01:52:19 14:19:59 -32:35:09 RJ 1.75621456257165 -30.6262473067631 116.853163022758 -84.8712648317096 -29.879890727604 2.68181470397268 -2.68181470397268 -29.879890727604
2026/12/19 18:07:13 09:19:01 -32:13:24 RB 6.19754447660874 -56.6989636921659 119.533038066165 -103.822515628306 -29.1312140398205 -7.16745202747578 7.16745202747578 -29.1312140398205
2017/11/14 04:16:48 01:59:05 -46:11:01 RB 1.96514156833553 -8.85000701883539 224.28284527735 70.5879369914417 28.2945811229543 9.97079129640962 -9.97079129640962 28.2945811229543
2041/04/14 18:48:52 02:19:05 -84:09:05 RB 2.10249489365271 -19.3977953160774 186.640182331809 85.0015372227117 29.8859110826014 2.61387045640071 -2.61387045640071 29.8859110826014
2004/01/12 04:06:52 07:35:52 -29:24:13 RB 2.92616839256815 18.8933233894036 227.386152833986 52.7169915903078 23.8695947191544 18.1725740593708 -18.1725740593708 23.8695947191544
2021/08/25 14:19:19 08:11:16 -29:48:24 RB 3.20841726262355 13.5312353038391 230.672666989325 57.1875774951552 25.2134740208025 16.2567133148838 -16.2567133148838 25.2134740208025
2006/06/09 23:07:01 21:24:01 +49:19:52 RB 4.18637697612807 20.2954919492258 43.1637950515594 -82.9095551802539 -29.770576110741 3.70307953392557 -3.70307953392557 -29.770576110741
1999/04/04 14:41:24 10:52:59 -18:39:41 RB 0.833492000282388 -30.09604056029 100.928672190182 -77.5482153257071 -29.2943339656374 6.46853905528154 -6.46853905528154 -29.2943339656374
2050/07/18 16:06:39 03:35:59 +51:06:11 RB 3.02404951840009 -0.283406421640885 326.385529149407 56.6321744939122 25.054705636399 16.5003553135499 -16.5003553135499 25.054705636399
2049/06/25 17:09:52 06:07:09 -58:39:16 RJ 2.90915080485394 -9.24951583512726 210.549359682845 66.8463661210236 27.5836149718156 11.7959393473614 -11.7959393473614 27.5836149718156
2038/07/18 03:13:11 04:42:41 +86:42:35 RJ 5.92136353800952 19.4871521537548 3.41229178587187 -83.4641185203813 -29.8050230201751 3.41476247590251 -3.41476247590251 -29.8050230201751
2020/10/12 04:35:42 21:52:59 -29:59:06 RJ 1.48482329054753 -32.2271189555929 246.4886219672 84.2585003043038 29.8495011309417 3.00121346023748 -3.00121346023748 29.8495011309417
2075/05/09 18:52:59 12:48:24 -25:42:19 RB 2.54652761129912 24.4895695554758 133.102791840879 -50.0625549367192 -23.002373257455 19.2585260216019 -19.2585260216019 -23.002373257455
2054/11/12 02:16:48 17:11:19 +18:59:59 RJ 1.40638097480849 -51.1780982111742 3.25740154243073 -3.23979905976404 -1.695450990565 29.9520524495166 -29.9520524495166 -1.695450990565
2069/11/04 02:42:48 02:42:05 +86:41:59 RJ 1.38892475473838 22.3400927489802 358.214847592482 146.181902603632 16.696741853889 -24.924261502893 24.924261502893 16.696741853889
2055/01/09 12:39:24 02:11:35 +32:35:39 RB 5.12819218867278 2.27702404023665 55.505472405621 -67.7198737060998 -27.7602384449605 11.3740565093961 -11.3740565093961 -27.7602384449605
2007/06/19 11:39:24 23:24:06 -86:39:41 RJ 1.34576001887056 -19.540675441371 183.583127309745 84.6172459516001 29.8677073793324 2.8142593879342 -2.8142593879342 29.8677073793324
2028/06/07 04:06:35 21:06:07 +03:19:52 RB 5.4570820709539 73.1485357593194 163.061342817343 -15.9406382345712 -8.23923854824653 28.8464026898518 -28.8464026898518 -8.23923854824653
2025/04/18 20:24:39 03:24:35 +49:06:42 RJ 2.58829325823822 10.6545569117314 318.676859207232 71.9323783345872 28.5207343725307 9.30417706471378 -9.30417706471378 28.5207343725307
2004/09/14 10:01:07 13:39:07 +03:01:35 RJ 2.42481863060239 23.6644108750893 95.5738704562594 -69.6501650112308 -28.1276046174582 10.4325384487163 -10.4325384487163 -28.1276046174582
1999/01/07 12:48:16 18:01:59 +73:06:07 RJ 5.12505746751938 35.1607667319936 351.961643704377 153.090118279332 13.5776552885729 -26.7515845673614 26.7515845673614 13.5776552885729
2075/03/07 16:42:42 01:11:41 +49:52:52 RB 0.892734863720715 50.5637694367 328.616647831808 129.570664203703 23.125185204686 -19.1108819589513 19.1108819589513 23.125185204686
2072/09/19 23:11:09 11:48:11 +20:24:07 RB 5.97659585935131 -47.3937635719169 337.193133817492 22.7916397884637 11.6214321092649 27.657590562624 -27.657590562624 11.6214321092649
2075/10/19 09:35:13 14:09:48 -09:35:59 RJ 2.91020099952206 35.2245090511567 118.643433984172 -56.953144297709 -25.1467466767474 16.3597411830228 -16.3597411830228 -25.1467466767474
2035/09/04 15:35:11 06:52:39 +32:24:24 RB 3.70569782072084 -3.53162992616251 306.240768845875 63.8378999878006 26.926506613687 13.22736714479 -13.22736714479 26.926506613687
2037/01/01 06:05:35 00:35:35 +58:09:35 RJ 3.27368572190927 -11.7960233361802 359.00700331568 1.78127240305769 0.932521813896462 29.9855032151639 -29.9855032151639 0.932521813896462
1982/06/01 15:09:24 22:19:42 +73:48:39 RB 1.95575403950542 7.81502892977866 349.035227000989 40.3821951456981 19.4364965762382 22.8521902854383 -22.8521902854383 19.4364965762382
2055/05/07 11:13:06 13:52:07 +58:07:09 RJ 0.497336269749857 -12.3263778184258 359.992126075727 0.0139203106722124 0.00728865755221343 29.9999991145912 -29.9999991145912 0.00728865755221343
2055/04/09 17:48:59 23:11:24 -25:42:01 RJ 1.74775804161732 -27.2416396838453 250.897373435178 79.7607743468404 29.5222242643306 5.33275486841183 -5.33275486841183 29.5222242643306
2049/03/14 22:24:24 07:39:09 +41:19:19 RB 2.51343579230864 58.4072509330519 318.356839664346 123.970109346593 24.8798755574827 -16.7628097956213 16.7628097956213 24.8798755574827
2000/11/14 22:39:07 18:07:11 -73:41:11 RB 0.506882237016762 -26.0989691125415 196.387547440735 109.215410119326 28.3286365496795 -9.87361895336067 9.87361895336067 28.3286365496795
2074/12/14 13:42:39 04:52:48 -49:48:07 RJ 4.96031869801388 -51.5309879941553 148.128399051624 -129.846630264 -23.0328694465704 -19.2220426869062 19.2220426869062 -23.0328694465704
1991/12/01 14:05:05 20:06:39 -32:16:59 RJ 4.81951821034815 32.4810187044111 154.550354221222 -28.5745072662625 -14.3490350318449 26.3458762172543 -26.3458762172543 -14.3490350318449
2056/08/04 02:16:05 12:52:11 +49:13:05 RJ 5.97500689059791 -15.8077254691802 339.195372366658 30.5621224771068 15.254168363201 25.8323507940551 -25.8323507940551 15.254168363201
1988/05/09 01:09:42 16:52:41 +73:19:24 RJ 4.17885504646812 35.9312861680092 4.85170400295751 -163.889869518182 -8.32453573743881 -28.8219032118995 28.8219032118995 -8.32453573743881
2057/11/07 02:16:09 17:24:09 -51:01:13 RJ 1.32219778981105 -58.3494201280147 187.673383528314 168.469003316955 5.99694116324439 -29.3945011300513 29.3945011300513 5.99694116324439
2034/04/23 20:39:39 23:39:05 +38:06:42 RJ 2.73684209405866 -29.3695916323699 343.499625649801 19.9050179917003 10.2138569937066 28.2077493840276 -28.2077493840276 10.2138569937066
2004/05/23 23:52:01 10:13:16 +25:59:01 RB 4.09910112038698 16.4315834666844 291.997452158907 75.4996289659518 29.0443805686664 7.51158820639648 -7.51158820639648 29.0443805686664
2046/07/25 21:09:35 21:09:24 +16:35:42 RB 4.46934953431257 31.0480681722713 81.6251634746122 -76.7087265125941 -29.1964169859426 6.89704539516449 -6.89704539516449 -29.1964169859426
1984/11/07 22:11:07 01:16:09 +18:01:11 RB 0.261812737692432 85.444152154742 110.067480454335 -68.4641400950435 -27.9056400718106 11.0125043556204 -11.0125043556204 -27.9056400718106
2070/12/23 03:42:52 11:24:19 +20:39:59 RJ 2.49048229794591 62.4916177112793 83.96062152988 -85.7914279079567 -29.9191052641813 2.20162217258957 -2.20162217258957 -29.9191052641813
2029/05/25 20:48:11 08:24:41 +12:07:35 RJ 3.32836849514457 28.1783633371499 273.336562189666 73.7877227638414 28.8070164576312 8.37590608888156 -8.37590608888156 28.8070164576312
2007/12/18 07:06:09 01:11:41 +58:13:59 RJ 3.28113425897793 -11.4515534778952 354.627078016126 9.64541762655983 5.02650841800848 29.5759059560935 -29.5759059560935 5.02650841800848
2052/02/25 04:42:19 18:11:11 -25:06:52 RJ 3.8447246849496 21.4177852326457 128.732654189474 -54.1299203270496 -24.3104321534089 17.5784779919793 -17.5784779919793 -24.3104321534089
2063/09/14 08:05:41 00:01:52 +12:06:11 RB 1.91487121868636 -11.8548891912819 288.371143170374 66.2527604733182 27.4599264639048 12.0810777084224 -12.0810777084224 27.4599264639048
1985/03/12 14:59:13 11:42:06 +25:35:07 RB 0.522497323741535 -34.0278337676218 37.5016360121074 -39.3413502719767 -19.0181756060039 23.2014869484522 -23.2014869484522 -19.0181756060039
2036/07/18 19:13:52 02:59:24 -58:09:07 RB 3.84978079903017 -51.8482234523492 184.163483337403 172.634042077082 3.84619128294524 -29.7524253232404 29.7524253232404 3.84619128294524
2005/01/07 17:16:48 01:07:39 -58:35:59 RJ 0.0261950702961715 10.5620554279871 171.870890162424 -14.7833587822017 -7.65494817216965 29.0069262156713 -29.0069262156713 -7.65494817216965
2072/01/18 07:59:35 11:07:13 +61:42:07 RB 4.05676991051309 29.8568823427743 329.930466438194 103.274505990405 29.1984341424289 -6.88850082603176 6.88850082603176 29.1984341424289
1989/12/25 21:48:52 09:11:48 +16:48:13 RJ 0.986689591875903 13.5732134048177 76.712523006828 -73.0582251752704 -28.6980412917703 8.74199210797216 -8.74199210797216 -28.6980412917703
2060/05/07 13:48:16 11:05:16 -86:39:48 RB 1.18981768502107 -20.1183356013708 177.10364729862 -95.6992459804856 -29.8517063080559 -2.97919964043587 2.97919964043587 -29.8517063080559
1987/06/25 13:52:11 17:59:59 +01:06:07 RB 2.02721770336323 -57.2671774964384 53.2340408036996 -48.9169914934965 -22.6127492617749 19.7145522603 -19.7145522603 -22.6127492617749
2063/06/12 21:41:16 22:07:09 +38:41:05 RJ 3.86619945433813 -2.75630396409822 46.6079201972791 -61.5888661291199 -26.3866839502746 14.2738540733862 -14.2738540733862 -26.3866839502746
2013/01/07 08:06:59 06:13:09 +38:06:48 RB 3.90495651580438 -15.0903338357311 320.927319909378 48.8841613763197 22.6014492372328 19.7275059847105 -19.7275059847105 22.6014492372328
2051/05/09 18:48:19 22:52:06 +37:24:48 RB 2.52299547317928 -29.3454931132612 342.295092989132 21.2735398830188 10.8846276505659 27.9557665054732 -27.9557665054732 10.8846276505659
1983/09/14 19:06:13 16:42:16 +86:35:05 RB 4.79402125549457 22.9233369393739 358.311724229546 152.799560346292 13.7131425569122 -26.682385973031 26.682385973031 13.7131425569122
2017/05/07 03:59:24 03:16:09 -25:42:24 RJ 4.88647589630574 -43.098164689584 107.025788889707 -86.2211667860099 -29.9347765132919 1.97715833945304 -1.97715833945304 -29.9347765132919
2052/04/25 02:05:07 12:16:48 +84:41:06 RB 4.18908571095237 23.1208898242744 354.76752920875 123.129452782306 25.1231364994975 -16.3959754948468 16.3959754948468 25.1231364994975
2077/06/01 05:19:05 16:41:35 +51:09:24 RB 5.67321997455603 25.3571447963865 317.930980819711 89.3304824152255 29.9979518345832 0.350550609793393 -0.350550609793393 29.9979518345832
2057/11/14 17:05:42 09:09:24 -49:39:48 RJ 5.33463825119137 -58.4032496780065 195.035644533515 157.738107250617 11.3652216433976 -27.7638566664728 27.7638566664728 11.3652216433976
2015/06/07 22:59:35 22:07:52 -86:01:48 RB 4.11636856702163 -20.4524235489396 175.466078314859 -97.6279929478538 -29.7345241669836 -3.98221955234404 3.98221955234404 -29.7345241669836
2039/06/01 17:39:16 01:35:11 -41:16:52 RJ 2.61481960594048 -39.4699668489322 232.553956776572 97.7627746861462 29.725074190501 -4.05215552135021 4.05215552135021 29.725074190501
2015/05/14 00:59:09 16:41:09 +86:41:59 RJ 4.20943326841114 23.1188101741386 0.502882542843511 -171.825547224434 -4.26562784902616 -29.6951918507628 29.6951918507628 -4.26562784902616
2010/12/01 16:24:01 14:42:39 +16:35:06 RJ 5.4342444685215 5.02286331918852 285.786575995486 70.791133560874 28.3297640187464 9.87038351038815 -9.87038351038815 28.3297640187464
2024/05/12 11:16:11 03:35:35 -20:41:09 RJ 0.605614076775066 45.308637980205 153.677706190551 -26.4653780942425 -13.3697085915392 26.8561146143168 -26.8561146143168 -13.3697085915392
2047/01/19 20:39:35 11:41:09 -51:07:42 RJ 1.11701015291819 -28.926624821712 138.57277869239 -94.07757565457 -29.9240606733477 -2.13321185488055 2.13321185488055 -29.9240606733477
2008/05/18 19:07:06 15:07:05 -58:09:35 RJ 2.76712899121872 -6.06622504156747 150.477028126135 -61.5949241308102 -26.3881930069976 14.2710640747438 -14.2710640747438 -26.3881930069976
1991/06/12 22:13:09 08:41:24 -46:06:01 RJ 3.99606137475452 -20.0451024701963 226.887180221527 81.8626371080804 29.697946942731 4.24640405363602 -4.24640405363602 29.697946942731
2039/02/19 09:07:24 11:19:39 +09:19:19 RB 4.90377451997538 -15.325369096309 285.545989597151 66.5207712846478 27.5161371463585 11.9524975023127 -11.9524975023127 27.5161371463585
1998/08/14 22:35:13 05:16:42 -29:24:01 RB 5.1815237174569 -54.9900973045045 113.171995439192 -97.1597297037701 -29.7660763988043 -3.73907686742914 3.73907686742914 -29.7660763988043
1980/06/12 15:06:59 12:59:19 +73:06:42 RJ 2.14277330799961 24.1454054716178 17.496928453509 -101.504450103207 -29.397276508993 -5.98332130624853 5.98332130624853 -29.397276508993
2025/08/14 19:24:11 13:07:06 +09:24:52 RJ 4.35368098046583 38.4453370074957 266.139874024319 72.0000988213911 28.5317114782042 9.27046062094614 -9.27046062094614 28.5317114782042
1995/08/12 23:11:59 03:16:11 -49:41:11 RB 5.3033554999587 -24.9354876800655 136.386757362859 -88.2040607939155 -29.985263522057 0.940197592421336 -0.940197592421336 -29.985263522057
2071/01/09 22:41:35 16:48:16 +86:52:13 RJ 1.48193753222914 16.7194609506191 1.09173316266238 -18.3034569191703 -9.42149215868084 28.4821959389355 -28.4821959389355 -9.42149215868084
2053/11/01 23:19:13 03:59:39 -32:19:06 RJ 0.461569660563312 28.6627456065766 147.35713600747 -36.832081187403 -17.9841555594225 24.011875162396 -24.011875162396 -17.9841555594225
2063/08/23 17:16:19 06:11:59 -41:06:06 RJ 3.94558321159934 -44.6207040265299 231.221407090776 103.235322604689 29.2031382158411 -6.86853101809108 6.86853101809108 29.2031382158411
2067/08/18 23:16:11 04:09:11 -46:59:05 RJ 5.4346618506972 -28.8438001453283 133.328835525902 -91.2033423598581 -29.993383802814 -0.630022267143913 0.630022267143913 -29.993383802814
2031/04/01 21:41:39 02:06:06 +73:16:19 RJ 2.62501199553625 11.4357127176675 345.131060860071 57.7853367066044 25.3817028976893 15.9927845609646 -15.9927845609646 25.3817028976893
2062/10/07 10:07:39 00:11:07 +25:19:13 RJ 2.84831250244883 -40.375155835742 335.643568260952 25.4971605333703 12.9139909776149 27.0781985558508 -27.0781985558508 12.9139909776149
2048/07/09 19:19:06 10:09:05 +38:48:39 RB 3.71959506380882 35.8835255081319 303.558027002295 92.195551103388 29.9779768239232 -1.14930654932709 1.14930654932709 29.9779768239232
2006/05/07 08:24:52 14:59:16 -29:16:48 RJ 6.04222232424896 -36.2916806215548 247.613091831498 85.9716181636514 29.9258812064684 2.10751844982269 -2.10751844982269 29.9258812064684
2054/03/04 06:41:41 20:11:05 -32:59:39 RJ 4.49622292898329 21.4223100541564 139.504266335905 -46.6364626946703 -21.8103534669811 20.5987495165446 -20.5987495165446 -21.8103534669811
2026/06/19 08:42:05 09:19:09 +20:13:48 RB 0.576815399764045 -8.92798295109349 64.978136481114 -65.0414043236453 -27.1983885593307 12.6588964675311 -12.6588964675311 -27.1983885593307
2059/06/25 11:06:42 17:01:19 -09:41:35 RJ 1.31280929502963 -79.9128187117695 354.762555712227 4.99907826520879 2.61419149939512 29.8858830019207 -29.8858830019207 2.61419149939512
1980/01/18 13:48:48 20:01:11 +25:01:13 RJ 5.57229407037709 71.5463086081515 289.702457388266 102.324756217879 29.3086031278794 -6.40357577392963 6.40357577392963 29.3086031278794
2006/07/01 23:52:05 20:09:39 -58:52:05 RJ 4.76202386618313 7.61080482836471 165.007272786922 -28.0638621534127 -14.1136622587193 26.4727130768042 -26.4727130768042 -14.1136622587193
2073/09/14 08:48:42 08:05:07 +29:52:39 RB 2.11294398936235 80.1234807319735 10.8479583982572 -168.261621556076 -6.10329488502286 -29.3726027370142 29.3726027370142 -6.10329488502286
1993/10/01 10:41:39 18:41:06 -86:11:01 RJ 2.88893996279629 -21.342830051463 176.272040572531 -112.870964847684 -27.6414743802415 -11.6597124358387 11.6597124358387 -27.6414743802415
2079/03/12 04:09:01 20:41:09 -03:06:09 RB 3.96493644755829 3.9999382544137 94.253707268448 -69.9086945859303 -28.1743917404687 10.3055155161986 -10.3055155161986 -28.1743917404687
2073/09/23 20:35:41 14:39:11 +61:35:42 RJ 5.36100798314842 18.7656422752954 329.558025721444 82.8050847393318 29.763774604829 3.75735562236758 -3.75735562236758 29.763774604829
2002/10/01 11:48:13 10:39:07 -18:42:35 RB 3.17695442709014 45.7863544734643 209.967813419405 29.7974223014112 14.9080476102134 26.0336343304505 -26.0336343304505 14.9080476102134
2076/10/23 10:16:24 09:06:35 -20:39:09 RB 3.17223688435697 30.8024897474953 228.565799347987 49.1405082269594 22.689485497311 19.626187807807 -19.626187807807 22.689485497311
2060/10/25 04:59:48 17:59:07 -32:41:05 RJ 1.81931754157072 -72.3870815561385 139.614008343644 -133.595263315216 -21.726866109728 -20.6867902065534 20.6867902065534 -21.726866109728
2066/07/23 11:05:52 21:59:48 -58:41:59 RJ 1.79622535487331 -38.1381641661185 209.803618264535 116.918376383124 26.7495712525084 -13.581621324679 13.581621324679 26.7495712525084
2078/06/18 17:16:42 11:19:01 -32:05:35 RB 2.81809997491013 36.5404634862812 169.684234522317 -11.561217305218 -6.0124444391192 29.3913339586094 -29.3913339586094 -6.0124444391192
2040/05/01 22:11:01 15:13:06 +18:19:39 RB 3.28355535506273 51.0969262970544 85.6286792694506 -80.4883980499625 -29.5875648137901 4.95741952932871 -4.95741952932871 -29.5875648137901
2073/01/25 22:24:52 13:59:59 +01:13:59 RB 1.69286406493303 -22.7897183181619 80.5359026467627 -68.1251362492061 -27.8399939457203 11.1774208609256 -11.1774208609256 -27.8399939457203
2072/04/23 06:19:41 13:39:24 +09:01:07 RJ 5.27113125716717 -2.88793926686166 280.273299034952 69.4433404129699 28.0897623753505 10.5340044473288 -10.5340044473288 28.0897623753505
2072/12/25 06:35:48 14:48:42 -51:13:05 RJ 3.29033665188004 12.3924262326674 158.602909453288 -33.4669785922545 -16.5436889504652 25.0261134799284 -25.0261134799284 -16.5436889504652
2037/11/14 22:11:48 15:06:07 -29:52:01 RJ 0.388039842365422 -65.3349927350596 240.730575451041 108.618208973402 28.4300098438278 -9.5778150055142 9.5778150055142 28.4300098438278
2036/07/25 17:01:05 11:01:05 -41:19:42 RJ 3.38923975735983 23.2109714961832 202.873381732979 29.2374715670158 14.6529134322399 26.1780848792895 -26.1780848792895 14.6529134322399
1985/05/23 04:13:59 04:59:41 +61:42:35 RB 5.2212113444504 -1.39949202126346 18.9926503443674 -40.3217422710294 -19.4123743770347 22.8726850423348 -22.8726850423348 -19.4123743770347
2064/12/09 10:59:39 03:41:42 -09:01:11 RJ 4.16852188852741 -78.7038804669882 13.332731296784 -12.6816691590283 -6.58602260690416 29.268144905705 -29.268144905705 -6.58602260690416
2042/05/12 15:05:52 10:39:07 +58:09:11 RJ 1.60434406103707 28.044436617809 34.0386404214447 -97.2745113100196 -29.7585261179878 -3.79870020994504 3.79870020994504 -29.7585261179878
2053/03/01 23:48:16 11:16:41 +84:09:05 RB 2.65720072558823 25.867227489359 2.33771765211453 -159.917495455154 -10.3011876912861 -28.1759743779855 28.1759743779855 -10.3011876912861
2062/01/04 08:11:07 07:16:42 -20:11:06 RJ 3.87364958150047 -26.7071508933088 256.617964595818 77.3696741502401 29.2740349529411 6.55979249473525 -6.55979249473525 29.2740349529411
2021/09/12 01:35:01 12:48:01 -73:59:42 RB 0.174045865168045 -35.4099013417693 181.066717437898 176.269530272015 1.95188963130705 -29.9364347721501 29.9364347721501 1.95188963130705
2009/11/14 02:39:13 16:41:13 +61:48:42 RB 1.53723202001509 -7.26931908861752 8.30886544406705 -16.6638716538186 -8.60269498668181 28.740105061849 -28.740105061849 -8.60269498668181
2062/11/04 21:59:09 10:24:05 -18:01:06 RJ 0.159809142237461 -59.4096654074019 92.7121014401807 -81.8691219655769 -29.698427369503 4.24304275000329 -4.24304275000329 -29.698427369503
2016/10/09 14:07:01 05:09:05 +61:11:07 RB 3.93239141343884 -4.58779729730943 344.564940273491 31.3765177476492 15.6197929979597 25.6129277260701 -25.6129277260701 15.6197929979597
1992/08/18 16:01:13 19:06:35 +86:11:24 RJ 3.53438647273651 20.131442144292 4.04922268261597 -93.9422234664251 -29.9290165452831 -2.06251512289504 2.06251512289504 -29.9290165452831
2021/01/12 04:06:24 05:35:11 -20:52:24 RJ 2.9392946640746 -1.99271877685101 248.515876973084 69.5209377689187 28.1040031211785 10.4959520085026 -10.4959520085026 28.1040031211785
1997/06/19 04:13:07 09:06:42 +20:19:19 RJ 5.68355803502358 -48.8875736108807 12.9051684241607 -12.9478708011561 -6.72193361969895 29.2372298347905 -29.2372298347905 -6.72193361969895
2010/07/19 00:16:09 23:48:07 +46:35:01 RB 5.16038251718374 33.2351124823684 46.2091732141664 -96.1493091641506 -29.8273837286094 -3.21359299014148 3.21359299014148 -29.8273837286094
2012/10/18 02:24:24 04:13:19 +49:48:05 RJ 1.01263099465272 59.6654980976709 7.11012578292748 -169.599875623694 -5.4156384109057 -29.507132368333 29.507132368333 -5.4156384109057
2004/09/04 05:13:19 23:41:35 -01:41:48 RB 0.993588601407564 26.7544795140607 257.839679852048 66.9127404659262 27.5972614760908 11.7639771769702 -11.7639771769702 27.5972614760908
2032/11/04 07:52:06 08:19:41 +49:59:16 RJ 2.74151447915792 50.8875944835847 327.68365033742 128.702389653365 23.4121298839355 -18.7582561635599 18.7582561635599 23.4121298839355
2054/04/25 04:35:09 14:42:05 -18:39:59 RB 4.8371377775355 23.3860805778017 237.663520179508 57.2644080201538 25.2352507142888 16.2228888113981 -16.2228888113981 25.2352507142888
2080/01/14 11:24:35 14:35:19 -84:19:52 RB 4.88595287967781 -16.7779513072274 184.357353027244 52.9780147362177 23.9521357698739 18.0636428236812 -18.0636428236812 23.9521357698739
2079/01/25 17:52:35 00:19:05 +32:07:59 RJ 0.493748076936714 66.2155816573163 306.947056508195 116.854186366292 26.7647703211775 -13.5516445369047 13.5516445369047 26.7647703211775
2058/05/23 10:35:01 10:01:24 +03:24:05 RB 0.610642590770285 -23.9586221117053 77.3781430428553 -66.8047060094964 -27.5750307814666 11.815992442498 -11.815992442498 -27.5750307814666
2036/07/23 18:06:01 09:16:19 +38:24:52 RJ 3.63893507372724 28.4547656091857 303.597481427269 86.3406379889274 29.9388341364255 1.91473511264728 -1.91473511264728 29.9388341364255
2025/12/09 07:24:06 18:16:52 +84:35:19 RJ 3.21585191004726 19.8528315822358 5.74417268652306 -89.3423767917523 -29.998023961101 0.344323146489539 -0.344323146489539 -29.998023961101
2071/01/04 16:52:39 07:59:42 -32:35:11 RB 6.15243251911286 -42.6083677917556 116.973142696012 -92.6823806119456 -29.9671294120231 -1.40397820604966 1.40397820604966 -29.9671294120231
2036/09/12 04:05:24 21:35:52 -38:48:07 RJ 0.838344639961408 -7.38547507417021 231.499252304129 70.478744574482 28.2755277452876 10.02469604155 -10.02469604155 28.2755277452876
2030/07/07 21:16:48 16:09:41 -29:59:06 RB 4.18913350714641 39.8626157349999 175.861079703992 -4.50538161982692 -2.35658198463406 29.9072987972785 -29.9072987972785 -2.35658198463406
2025/04/14 17:35:13 04:01:42 +29:39:05 RJ 1.77816460860645 51.6352347413505 293.188299785548 95.2831254468851 29.8725557898845 -2.76231978240029 2.76231978240029 29.8725557898845
2002/12/07 19:11:52 11:48:48 +38:06:11 RJ 6.27063165906599 -32.0596961769121 1.90136285131851 -2.27272247788513 -1.18968266892775 29.9764016377425 -29.9764016377425 -1.18968266892775
1986/01/23 04:09:41 19:11:24 +46:24:07 RB 3.13391913552756 2.27818439570339 40.8152680211683 -63.204570985706 -26.7786535817542 13.5241898961972 -13.5241898961972 -26.7786535817542
1989/09/07 15:52:07 20:41:05 +32:41:06 RJ 3.83388820686516 10.1334391530603 58.7914458491042 -72.8654932864694 -28.6684726196754 8.83847711174949 -8.83847711174949 -28.6684726196754
1998/02/14 03:06:41 04:48:39 -03:16:24 RJ 3.23836907078645 -23.109484170096 275.030004468227 69.8279714349034 28.1598444989714 10.3451997464404 -10.3451997464404 28.1598444989714
2009/06/19 16:19:05 05:39:06 +51:06:24 RJ 2.57835675034825 32.288495976579 318.674608501305 98.3077532488 29.6851873730585 -4.3347030609265 4.3347030609265 29.6851873730585
2008/03/12 22:01:19 15:09:19 -41:42:59 RB 2.37679001564899 -14.6230100673893 129.800255532333 -76.3064958835669 -29.1472789551964 7.10183986780628 -7.10183986780628 -29.1472789551964
2064/07/04 15:35:19 07:01:09 +32:42:05 RB 2.65659354707765 47.9483577289275 296.979863203628 96.1121725445782 29.8294603736166 -3.19425951025255 3.19425951025255 29.8294603736166
1993/12/14 09:35:16 14:07:19 +18:13:41 RJ 3.87150157797793 80.3600995199086 262.309592825094 79.018845468161 29.4506967113334 5.71458338088204 -5.71458338088204 29.4506967113334
2061/09/01 00:09:07 06:24:19 +86:59:48 RB 5.8975982101037 17.7511588587158 2.49177524677407 -48.3152085372506 -22.4044420227912 19.9509643287584 -19.9509643287584 -22.4044420227912
1992/02/19 02:24:35 12:41:07 +37:05:19 RJ 3.13087739130592 70.2948176242351 26.3217646894217 -148.454197874685 -15.6953998933414 -25.5666662314056 25.5666662314056 -15.6953998933414
2072/02/14 20:35:41 03:05:01 -61:16:01 RJ 1.54620018468703 2.44513889210284 198.927374309106 39.0043745653848 18.8813917425627 23.312937306652 -23.312937306652 18.8813917425627
2058/04/07 03:13:13 11:41:39 +41:11:11 RJ 4.16950627722314 33.2323342284697 306.533197943465 91.7621477528956 29.9858128094454 -0.922512957579945 0.922512957579945 29.9858128094454
1988/11/23 11:07:52 03:07:13 +84:13:48 RJ 3.91897420045624 14.0166725746603 359.820467814641 1.66902517089998 0.873775949054015 29.9872725600521 -29.9872725600521 0.873775949054015
2003/11/14 08:42:59 10:35:11 -25:06:24 RJ 3.11927553274439 41.0724381312407 204.110884368571 25.1155782891652 12.7333687311331 27.1636028677531 -27.1636028677531 12.7333687311331
1980/11/19 20:11:01 23:52:11 +32:01:11 RJ 6.22543396123249 77.8661352557698 4.49861816312774 -175.013063601359 -2.60785815744111 -29.8864363186826 29.8864363186826 -2.60785815744111
1991/08/01 14:24:09 03:52:06 -51:35:41 RJ 2.80420595927861 -23.219684307489 221.236202516565 86.9785331495578 29.9582956828865 1.58130318938076 -1.58130318938076 29.9582956828865
2022/12/01 07:13:06 07:35:52 -37:52:48 RJ 3.02532366124159 9.89859321594178 223.435702210255 55.0831958776633 24.5995211251628 17.1715916680041 -17.1715916680041 24.5995211251628
2002/03/12 02:39:39 14:07:39 -01:01:07 RJ 3.56784756387209 67.882428249797 159.669229909161 -19.0810175582698 -9.80714441039731 28.3517180875095 -28.3517180875095 -9.80714441039731
2003/08/04 03:19:05 14:24:09 +51:06:01 RB 6.23063198267059 -11.2451474460264 335.829296319424 37.6118420069622 18.3092670865615 23.7649056121196 -23.7649056121196 18.3092670865615
2046/11/14 21:35:52 18:07:11 +01:11:07 RJ 0.227867207641725 -9.61372730638973 274.792372710851 69.6623559271796 28.1298237288771 10.4265534565504 -10.4265534565504 28.1298237288771
2020/11/25 07:52:06 11:59:09 -84:41:06 RJ 3.10104816409775 -14.6211447107522 179.780361331973 -2.2778089638329 -1.19234384708407 29.9762959044362 -29.9762959044362 -1.19234384708407
2035/08/14 22:13:19 01:09:11 -49:39:41 RB 5.08638408819277 -12.8477204443617 138.002712910147 -74.4492708688097 -28.9018039575977 8.04274381020493 -8.04274381020493 -28.9018039575977
1995/11/01 11:41:19 01:48:41 -84:05:01 RB 3.67491241617219 -25.9506362422446 179.540554558894 -175.957218385883 -2.11503946139545 -29.925350592378 29.925350592378 -2.11503946139545
2025/03/09 03:41:11 12:39:41 -32:11:48 RB 3.79293532562598 31.8136764136038 206.134981159925 29.4678566732227 14.7580562130452 26.1189543590971 -26.1189543590971 14.7580562130452
2031/12/23 03:24:35 22:07:52 +16:41:41 RJ 2.40098190073939 -50.5507211060072 337.39394816796 22.2005018236383 11.3354668805435 27.7760182639647 -27.7760182639647 11.3354668805435
2009/07/18 10:16:39 23:05:59 -03:35:48 RJ 1.49150030568616 -9.50246017375876 269.641712732113 70.480539422328 28.2758417650871 10.0238102773225 -10.0238102773225 28.2758417650871
1990/03/23 08:59:59 14:06:59 +51:05:24 RB 5.41965385320214 10.1657816985666 320.658332013877 70.9975612989665 28.3651415240242 9.76825195836383 -9.76825195836383 28.3651415240242
2070/03/18 23:48:01 08:24:39 +41:39:13 RJ 2.94657569469166 48.8386604221255 311.00366793917 108.785729396298 28.4018849145947 -9.66089712698146 9.66089712698146 28.4018849145947
2029/06/04 04:11:42 03:01:41 -32:41:52 RB 5.42370142857647 -14.9209688392307 119.550490265736 -75.7254808519742 -29.0737644646399 7.39704129058653 -7.39704129058653 -29.0737644646399
2035/09/14 18:35:42 00:59:35 -16:48:35 RB 4.66753313509532 -22.4772488958529 100.058883240253 -74.8653344131229 -28.9594452447923 7.83265798524862 -7.83265798524862 -28.9594452447923
2036/01/18 14:48:35 02:19:41 -18:24:42 RJ 5.84138420050154 19.3661956222256 118.659396982576 -60.3687625872714 -26.0767651593685 14.8324751414976 -14.8324751414976 -26.0767651593685
2019/10/23 22:13:59 10:16:35 +46:59:42 RJ 0.0081060888454707 -19.2736762952406 18.5183247748402 -25.9285588288727 -13.1175034409894 26.9801983587154 -26.9801983587154 -13.1175034409894
2022/03/23 03:42:42 08:52:41 -03:24:01 RB 4.03563672343585 -7.96419062399193 268.944431362852 70.4804107555276 28.275819254953 10.0238737752044 -10.0238737752044 28.275819254953
2069/03/23 23:59:39 13:24:01 +84:39:52 RJ 3.08767340827583 25.0261796845283 2.51451162554089 -155.438403276343 -12.4701381274879 -27.285447679695 27.285447679695 -12.4701381274879
2014/11/07 07:05:39 14:52:41 -38:09:19 RB 2.5788560373946 -2.1682036206866 130.337880132912 -66.2291386072589 -27.4549433515833 12.0923978416752 -12.0923978416752 -27.4549433515833
2046/06/19 20:52:06 20:09:19 -29:07:35 RJ 3.77355053558464 -6.83826918547545 118.396700925154 -71.0940201789775 -28.3815464160774 9.72048471178493 -9.72048471178493 -28.3815464160774
2042/03/14 01:05:09 11:39:07 +38:35:24 RB 3.1942040448855 70.7572134157807 343.040428456899 159.59809745022 10.4580951086523 -28.118112431285 28.118112431285 10.4580951086523
2029/08/19 00:35:07 11:35:09 +46:41:16 RJ 5.78351120191513 -20.5211058883069 343.450431282249 22.9211444567777 11.6839163476692 27.6312522115931 -27.6312522115931 11.6839163476692
2030/07/14 13:42:48 08:13:41 +03:07:13 RJ 2.32318346898945 70.9108694246558 209.532405811929 27.6685205505852 13.9306656776861 26.5694665692885 -26.5694665692885 13.9306656776861
2018/01/25 09:39:16 02:05:42 -38:41:01 RJ 4.61462828604983 -41.0098677901626 124.43497550011 -96.8477070656278 -29.7859972617328 -3.57692145902653 3.57692145902653 -29.7859972617328
2068/07/14 06:35:06 23:35:06 -73:13:52 RJ 0.465583487826706 -5.07279279289462 189.000146304787 29.929544380202 14.9680405775603 25.9991877040132 -25.9991877040132 14.9680405775603
2058/12/12 19:48:41 04:42:01 -38:19:16 RB 0.242071240469906 10.7995847555971 137.46268120719 -53.9468700906008 -24.2541479446234 17.6560558302335 -17.6560558302335 -24.2541479446234
2043/07/09 18:05:13 03:09:41 +32:19:41 RJ 3.38268135921106 -28.2648193362104 327.362969149028 36.9779407591897 18.0452249636216 23.9660146042743 -23.9660146042743 18.0452249636216
2006/04/18 20:09:01 20:11:19 +38:19:16 RJ 2.51302896902094 -28.5632748856944 18.7559627479775 -22.6820570083543 -11.5685135228247 27.6797668861611 -27.6797668861611 -11.5685135228247
2052/06/09 20:01:13 22:19:35 +25:59:13 RB 3.38825284410058 -30.7793029467863 39.9274757362319 -42.4306957081843 -20.2409373518191 22.1428194934551 -22.1428194934551 -20.2409373518191
2036/01/19 14:07:35 19:16:24 +29:19:41 RJ 5.67920067563234 56.0978082562192 293.51541961316 98.0753802394479 29.7025233298697 -4.21427429559872 4.21427429559872 29.7025233298697
1986/01/01 15:48:09 17:39:09 -29:39:59 RB 5.81143321983446 8.23734162615587 234.260673498167 61.5146218986474 26.3681656526911 14.3080341106751 -14.3080341106751 26.3681656526911
2026/08/25 02:39:59 00:01:13 -38:24:16 RB 0.145125282608224 31.8277764232824 186.514107546196 7.78102898313363 4.06162565238719 29.7237816749462 -29.7237816749462 4.06162565238719
1991/05/09 20:39:52 19:11:52 +61:13:39 RJ 3.00302124527257 5.70335101440129 25.8153303538573 -58.273830795332 -25.5171305221628 15.7758058404288 -15.7758058404288 -25.5171305221628
1997/03/14 12:41:24 08:41:52 +49:07:11 RJ 6.23877333766495 -9.38832894095596 29.0342506286004 -44.24646282637 -20.9323871213384 21.4903506114355 -21.4903506114355 -20.9323871213384
2063/07/14 17:11:06 13:19:13 +46:59:07 RJ 3.23464364383285 60.4410959780013 21.338458783164 -150.083954643791 -14.9619146907923 -26.0027134888929 26.0027134888929 -14.9619146907923
2078/04/09 19:52:52 09:19:05 -12:42:19 RJ 2.29717008450847 55.9398466571342 164.047426361576 -15.3907838465338 -7.96203111491161 28.9241432116006 -28.9241432116006 -7.96203111491161
2053/06/07 00:11:16 01:01:59 +16:06:19 RJ 4.4264889934202 -22.9755056576461 61.4065585707403 -59.4280984457624 -25.8297468806872 15.2585771315555 -15.2585771315555 -25.8297468806872
2035/06/23 05:24:07 03:09:59 -09:19:48 RB 6.05950139394164 23.1060841450215 109.56382428909 -63.8337439368101 -26.9255470723131 13.2293202719056 -13.2293202719056 -26.9255470723131
2061/11/12 13:05:41 07:42:48 +25:09:13 RJ 4.25069715688981 -21.6332271568767 308.777777034722 54.0203169810217 24.2767611332326 17.6249501809216 -17.6249501809216 24.2767611332326
2042/06/19 13:35:24 16:48:52 +51:07:09 RB 1.86224177388621 -13.2026280998229 21.1475466685059 -32.607514347183 -16.166438042661 25.2714518976018 -25.2714518976018 -16.166438042661
2076/07/07 20:35:06 08:42:09 +16:42:06 RJ 4.02132165886311 -2.41743159045975 288.423972750136 68.5124304244811 27.914911775953 10.9889808690687 -10.9889808690687 27.914911775953
2033/05/18 01:19:19 13:48:06 -49:11:07 RJ 4.37751610423722 10.8972735765227 207.03455179007 41.0276934291197 19.6927120445253 22.6317717453011 -22.6317717453011 19.6927120445253
2059/11/25 15:42:05 06:42:11 +58:42:19 RB 5.14971509841438 -10.9585502502749 6.39904787778243 -11.6029134306729 -6.03383193799465 29.3869507119067 -29.3869507119067 -6.03383193799465
2077/10/23 19:05:48 20:42:01 +61:19:59 RJ 5.48434757069032 48.1388209040697 357.625326213398 175.296276212488 2.46009846796579 -29.8989617801005 29.8989617801005 2.46009846796579
2053/09/18 07:35:07 04:13:35 +09:13:13 RJ 1.85714875761659 47.893761481249 261.879853542504 70.7098586669949 28.3157342390022 9.91055974737048 -9.91055974737048 28.3157342390022
2061/11/18 15:48:48 16:35:05 -03:11:48 RB 5.0675957811597 44.2588472283647 243.915523114382 57.8256182366057 25.3929402764611 15.9749361224413 -15.9749361224413 25.3929402764611
2069/04/14 18:11:19 14:13:35 -18:59:59 RB 1.94207909341907 -18.9541915820574 104.614297462906 -75.016932647399 -28.9800681904566 7.75600719935737 -7.75600719935737 -28.9800681904566
1991/04/23 05:41:39 13:01:09 -12:41:42 RJ 5.08102361029924 -9.74449950002365 259.959983338664 71.694195717205 28.4818099238097 9.42265904423886 -9.42265904423886 28.4818099238097
2055/06/18 00:05:59 10:19:06 +09:52:52 RJ 4.5842263093791 -12.5341297265735 285.17594747884 67.0492083465612 27.6252027731471 11.6982123310575 -11.6982123310575 27.6252027731471
1989/09/23 18:52:09 13:42:07 -86:59:41 RJ 4.89682730638028 -19.0440206803257 183.130612267457 74.7045061992824 28.9373450569808 7.91391565871354 -7.91391565871354 28.9373450569808
2003/07/23 12:11:16 21:35:16 +37:52:52 RJ 2.06945777655212 -27.5790785873632 337.649974607855 26.9567044635956 13.59951244228 26.7404798261413 -26.7404798261413 13.59951244228
2059/07/07 05:01:06 19:13:24 +12:13:52 RB 6.20283190446701 26.7483845039411 274.272812821032 73.8717335261219 28.8192667796732 8.33365839724823 -8.33365839724823 28.8192667796732
1994/03/04 03:09:16 16:52:59 +29:52:42 RJ 3.55887269531373 44.5099562013505 67.1499189090113 -90.9426624871293 -29.9959397885889 -0.493554656899544 0.493554656899544 -29.9959397885889
2018/07/12 23:16:24 03:42:52 -46:39:39 RB 4.79668248892331 -48.8089411864941 139.276284799758 -117.034058091731 -26.7220950931837 -13.6356017040264 13.6356017040264 -26.7220950931837
1986/06/01 12:52:01 09:16:19 -03:42:35 RB 1.35524208216095 24.8029816934536 104.209304643038 -66.0697783425438 -27.4212038721205 12.168713087406 -12.168713087406 -27.4212038721205
2078/12/04 21:52:05 08:09:07 +58:24:16 RJ 0.64706806847224 18.4004576120246 33.7148984450617 -81.8514738391029 -29.6971190275281 4.25219019622012 -4.25219019622012 -29.6971190275281
1986/06/12 16:05:19 12:13:19 +01:42:13 RJ 2.39021544192681 41.3977075625845 105.888646204311 -64.8558435885138 -27.1572483097783 12.7469158717332 -12.7469158717332 -27.1572483097783
2070/11/23 16:19:09 20:16:41 -41:05:09 RB 5.28333886250414 29.3925064385568 176.9726090254 -3.75743741395506 -1.96597974337753 29.9355127507218 -29.9355127507218 -1.96597974337753
2023/05/23 15:59:13 16:07:42 +09:52:39 RJ 2.02013329259683 -29.5447937890015 65.6323986038823 -60.4201232765723 -26.0900507060438 14.8090936304713 -14.8090936304713 -26.0900507060438
2026/04/12 00:42:11 02:19:24 +18:05:35 RJ 3.59051079862627 -50.8411097293392 345.641917262942 14.2167788856739 7.36773822973137 29.0812041253136 -29.0812041253136 7.36773822973137
2051/02/09 07:42:01 21:48:42 +09:13:39 RB 4.35990854299212 13.8998296462985 84.5267246627144 -71.8145688740983 -28.5015431915654 9.36280063332174 -9.36280063332174 -28.5015431915654
2022/09/19 20:11:41 21:07:07 -01:35:16 RJ 5.17603211194086 60.6883843327998 134.324221057568 -42.3173420164948 -20.1970905219603 22.1828207053959 -22.1828207053959 -20.1970905219603
2065/05/09 20:16:09 12:06:48 -09:19:11 RB 2.91779875719547 56.2968594535652 150.685165990915 -27.8819908297363 -14.0295601962381 26.5173799742759 -26.5173799742759 -14.0295601962381
1988/02/01 05:16:13 01:24:11 -09:41:01 RJ 3.57156291249504 -79.2969569480567 20.2459086445623 -19.2877715927095 -9.90938860662237 28.3161441132606 -28.3161441132606 -9.90938860662237
2019/05/07 09:35:09 05:05:16 -20:16:01 RJ 0.0639216330865136 8.21282468983464 115.066867417563 -65.2650301881575 -27.2475890407328 12.5526447996961 -12.5526447996961 -27.2475890407328
2056/10/19 14:48:05 22:59:01 -73:41:48 RJ 4.28942559451836 -21.8158543228353 162.354051389633 -93.6773463705572 -29.9382316524847 -1.92413240764896 1.92413240764896 -29.9382316524847
2054/11/18 06:35:24 02:11:19 +51:39:09 RB 2.64104302264772 0.46108687006649 326.756724735709 57.1673152081347 25.2077233612893 16.265628882423 -16.265628882423 25.2077233612893
1985/07/18 01:24:11 19:35:19 -20:07:24 RB 5.44165886719124 46.6320963224952 204.192354054114 24.2283594251742 12.3112335436762 27.357513202648 -27.357513202648 12.3112335436762
1983/08/12 13:11:13 07:52:52 +58:35:52 RJ 2.67311201255578 43.5612075318441 335.482090239323 131.394336330236 22.5052930930574 -19.837131415494 19.837131415494 22.5052930930574
2017/12/07 21:01:42 14:41:19 -25:24:59 RB 0.474331925577895 -75.7169903950458 243.101308401389 111.400071835229 27.9316607518025 -10.9463385497717 10.9463385497717 27.9316607518025
2035/07/12 10:13:39 06:59:59 -12:39:06 RB 1.36995654924388 47.6333657247633 138.088444486155 -40.1181210095124 -19.3309656063811 22.9415293458154 -22.9415293458154 -19.3309656063811
2036/09/04 22:24:05 18:16:07 +41:19:09 RB 5.50776121639862 49.3014479066991 311.156984733732 109.315986049781 28.3112609487445 -9.923331269896 9.923331269896 28.3112609487445
2009/05/04 03:06:16 12:05:06 +73:13:59 RB 4.60142220089286 21.3736714685308 341.801908543432 91.9328911080129 29.9829305478702 -1.01186746246307 1.01186746246307 29.9829305478702
2029/08/23 21:13:01 15:35:07 +86:42:13 RB 4.98528321999514 21.7562073377723 356.790190934403 122.214962200842 25.3816194642807 -15.9929169750376 15.9929169750376 25.3816194642807
2021/09/04 01:24:42 19:09:01 +32:13:48 RB 6.27447365240346 25.6417479617825 297.254862421026 81.8824617694353 29.699414444625 4.23612814329296 -4.23612814329296 29.699414444625
1989/06/04 09:09:13 18:19:59 +58:59:59 RB 0.436817558489676 7.24036904245703 330.789340773883 63.0994806437981 26.7538028545543 13.5732837891073 -13.5732837891073 26.7538028545543
2061/06/18 11:13:13 21:48:41 -32:11:13 RB 1.22981259576006 -19.8627878784689 242.418986115629 78.4142108701166 29.3887527634832 6.02504863107855 -6.02504863107855 29.3887527634832
2025/01/18 13:48:39 03:41:24 -16:52:16 RB 5.59062413042139 -10.9303024734219 103.906609012237 -72.3763855768267 -28.5919789577654 9.08288166160384 -9.08288166160384 -28.5919789577654
1993/05/23 03:59:39 23:19:59 +32:42:35 RB 5.15970295762042 39.6713573840428 63.122153883272 -90.7158014948439 -29.997658869906 -0.374783036902413 0.374783036902413 -29.997658869906
2024/04/18 13:11:16 07:35:09 +58:41:52 RB 0.696266750514971 24.3631904370635 33.6905171870691 -91.5348327415846 -29.9892367820679 -0.803540433995558 0.803540433995558 -29.9892367820679
2061/01/01 21:05:13 20:35:05 +41:41:16 RB 0.929892842615651 3.667589552379 313.686390432667 66.4251063696428 27.4961421014712 11.9984236271147 -11.9984236271147 27.4961421014712
2019/04/07 20:59:05 08:24:42 -18:09:41 RB 2.54023311185814 47.6559042824187 206.680668946317 26.433289273531 13.35466557884 26.863598181877 -26.863598181877 13.35466557884
2077/01/14 17:42:48 09:09:16 -84:05:07 RB 0.27001510391412 -22.3337061929209 174.860951022649 -116.694975455076 -26.8023236356751 -13.4772195844889 13.4772195844889 -26.8023236356751
2029/05/25 19:11:13 07:24:52 -61:07:07 RJ 2.90411317521503 -2.15292151133392 203.299406096581 50.5293222080562 23.1585002246541 19.0704973019766 -19.0704973019766 23.1585002246541
1989/01/12 00:59:42 11:11:35 -20:05:48 RJ 2.1182344052621 29.5591157219619 128.545022519029 -51.55237012363 -23.4953048276136 18.6539714553635 -18.6539714553635 -23.4953048276136
2011/03/12 00:07:24 23:42:01 +46:09:42 RB 2.89877239139521 -23.0397087270271 352.409263732889 10.4010857998356 5.41613354000444 29.5070414897332 -29.5070414897332 5.41613354000444
2075/02/14 06:01:19 11:59:01 +61:35:05 RJ 4.00844235692896 36.5150820503578 333.092467026955 118.027313325074 26.4817107584972 -14.0967725137101 14.0967725137101 26.4817107584972
2053/09/12 01:48:59 07:41:05 -32:16:35 RJ 0.239507892002253 -20.3464071946354 118.284750939585 -78.8713380656529 -29.435886991179 5.79038487861918 -5.79038487861918 -29.435886991179
1980/11/19 13:42:35 14:07:41 -20:35:16 RJ 4.52593349895299 28.3385761264258 231.886451898715 52.2018174650072 23.7052336154583 18.3864596710881 -18.3864596710881 23.7052336154583
2048/08/01 00:05:52 10:39:35 +86:41:06 RJ 5.35273876253449 16.8948578937655 357.848579631455 34.558311702789 17.0173404715434 24.7064793784054 -24.7064793784054 17.0173404715434
2004/06/09 19:13:35 07:01:09 -25:39:35 RB 3.17333121461948 3.32419142840357 241.0946560331 66.1021008103604 27.4280642869355 12.1532419325764 -12.1532419325764 27.4280642869355
2049/07/23 20:59:01 01:05:24 -61:35:01 RB 4.3934253774339 -34.1879629312059 151.589537534076 -112.39459118269 -27.7374600951844 -11.4294928788662 11.4294928788662 -27.7374600951844
2006/07/14 15:42:13 07:24:39 -32:09:39 RJ 2.84236484540081 18.2796511663157 224.345500304564 50.9721347613682 23.3051941734253 18.8909482170415 -18.8909482170415 23.3051941734253
1999/05/07 16:05:09 17:06:24 +46:13:05 RB 1.76760954693983 -20.4430591801829 17.7243724452308 -24.4213334995838 -12.40330454883 27.3158934737448 -27.3158934737448 -12.40330454883
2073/02/04 02:35:05 13:16:05 -16:52:42 RJ 2.94245421149069 41.7675688827744 138.142746552455 -41.0973690787631 -19.7202192839376 22.6078073106043 -22.6078073106043 -19.7202192839376
2027/04/23 04:41:42 09:48:39 -32:19:52 RJ 4.82354454005814 -42.8837618190936 243.647094757536 92.4249659895525 29.9731346521468 -1.26933018725447 1.26933018725447 29.9731346521468
1980/04/25 23:59:39 05:35:42 -49:48:41 RJ 3.64759908911454 -37.5105950437179 221.645862067634 104.274893388902 29.0737161485026 -7.3972311925679 7.3972311925679 29.0737161485026
2036/06/09 00:39:59 06:09:19 -58:07:07 RB 4.58851420141452 -51.0220070447914 188.169025411508 165.329155786229 7.59797110690223 -29.0219026781271 29.0219026781271 7.59797110690223
2063/02/23 17:07:05 03:06:01 +73:16:07 RJ 0.791468390401827 36.2862671069666 0.994150466383012 -176.702791923239 -1.72546138315902 -29.9503386126973 29.9503386126973 -1.72546138315902
2003/07/18 01:48:59 16:07:19 +29:01:24 RJ 5.543972788854 21.5047445979723 294.340834364114 78.5651516469867 29.4045232338759 5.94760568535371 -5.94760568535371 29.4045232338759
2020/04/07 23:59:59 14:16:52 +86:09:13 RB 3.34475120091382 23.7246487391635 1.59587484039877 -158.900713389914 -10.7995557565509 -27.9887405122337 27.9887405122337 -10.7995557565509
2045/04/18 06:11:41 07:07:05 +38:09:06 RB 5.14223995451681 -31.8854313166996 5.78893294155079 -6.91551553022533 -3.61217010670179 29.7817431847139 -29.7817431847139 -3.61217010670179
2055/06/19 09:07:39 22:05:19 +29:13:59 RJ 0.688181123437584 28.7310849389911 293.537963730682 82.2897497134344 29.7287764068119 4.02490414218569 -4.02490414218569 29.7287764068119
2012/10/12 02:39:59 05:39:52 +38:52:16 RJ 0.977597287764392 58.457982719708 46.5236686398729 -118.732866362981 -26.306116518206 -14.4217971740893 14.4217971740893 -26.306116518206
2069/07/14 10:48:01 14:24:48 -58:19:19 RB 1.56797830175704 -36.4090571099836 149.834762768608 -113.936625346898 -27.4198436589799 -12.1717777549994 12.1717777549994 -27.4198436589799
2074/08/23 13:42:16 15:59:59 +37:35:01 RJ 3.01473707220068 29.1191760122107 57.453412949454 -86.3946223526739 -29.9406249198486 1.88652575093468 -1.88652575093468 -29.9406249198486
2026/07/18 02:13:42 06:16:11 -46:59:09 RJ 5.65960554267527 -41.3009205234168 135.890217197292 -106.266139286592 -28.7991297908949 -8.40298299933954 8.40298299933954 -28.7991297908949
2069/06/18 22:24:52 18:07:48 +51:09:13 RB 4.16961038258429 48.9776102146796 32.0594051591413 -127.193312925265 -23.8980142875415 -18.1351843969799 18.1351843969799 -23.8980142875415
2021/05/01 01:05:39 16:11:16 -86:13:52 RB 4.02356159357734 -16.4030108820649 178.783482987424 -18.4942774818361 -9.51629817918804 28.45066025534 -28.45066025534 -9.51629817918804
2061/03/19 23:07:11 22:11:13 +41:16:06 RJ 2.78814959697122 -28.3322279397619 5.35713339978203 -6.74171524654258 -3.52181406837035 29.7925632611199 -29.7925632611199 -3.52181406837035
1981/05/23 02:42:24 19:05:48 +12:52:16 RJ 4.81996418841621 78.0839347643219 124.334109398318 -52.8217685188344 -23.9027870183481 18.1288933130373 -18.1288933130373 -23.9027870183481
2050/06/19 06:59:24 17:42:19 +25:05:48 RJ 0.130798535083005 -1.40662319972475 297.357245060536 67.2934281295681 27.6748145899452 11.580355668638 -11.580355668638 27.6748145899452
1995/01/01 08:48:16 18:16:05 -46:07:24 RJ 3.97133524239516 11.8504071169356 149.154754940065 -44.0992846813309 -20.8771149291337 21.5440495783811 -21.5440495783811 -20.8771149291337
1988/03/18 23:11:01 22:59:11 -61:19:59 RJ 2.7822421493438 -48.266349527775 183.735241260199 172.645094262162 3.84045205726532 -29.7531666885366 29.7531666885366 3.84045205726532
2058/01/19 21:05:39 12:09:35 -61:07:42 RJ 1.2366940318157 -27.9419286536773 149.96255302603 -99.8814994231623 -29.5549437004314 -5.14833010444484 5.14833010444484 -29.5549437004314
1994/07/01 03:09:39 07:06:13 -61:39:35 RB 5.60767506534911 -41.704925500966 158.874789101507 -134.280026038928 -21.4780849718996 -20.944972330845 20.944972330845 -21.4780849718996
2078/03/04 02:09:35 13:41:13 -38:06:19 RJ 3.3089074161096 29.5718779510565 164.86262378332 -18.2932839467378 -9.41643494323262 28.4838682934721 -28.4838682934721 -9.41643494323262
2065/10/07 03:07:09 11:06:48 -49:19:01 RB 1.01327553755166 -27.7870016635208 136.863360569348 -91.7828358358292 -29.9854777582317 -0.933340029459605 0.933340029459605 -29.9854777582317
2032/08/19 06:24:13 14:09:24 -20:06:09 RB 1.03240397400902 -65.9365515783544 96.0422723572731 -87.0717694465935 -29.9608292399883 1.53255057086585 -1.53255057086585 -29.9608292399883
2054/05/09 09:42:05 06:07:06 +09:52:11 RJ 0.137705988855431 8.32723102834067 82.4659026369408 -71.1899473087662 -28.3977810961167 9.67295346918592 -9.67295346918592 -28.3977810961167
2080/11/14 00:52:01 19:09:39 -41:39:19 RB 1.08198007430503 -44.3516771930589 230.860041534863 103.310692620602 29.194077708555 -6.90694047656784 6.90694047656784 29.194077708555
2053/09/04 02:41:06 10:59:16 +32:41:24 RB 0.329910517453458 -29.353837278438 31.6350972419599 -35.6397485268001 -17.4806073815872 24.3809016562308 -24.3809016562308 -17.4806073815872
2001/05/14 18:07:01 09:09:59 +25:42:24 RB 2.4300760215738 84.252428554104 351.010014650789 170.626813491294 4.88592734028368 -29.5994546237843 29.5994546237843 4.88592734028368
2046/10/01 07:01:11 19:52:09 -46:24:39 RB 1.92715627587126 -62.5701155347849 194.157454991843 160.600442589167 9.96461537460074 -28.2967567123208 28.2967567123208 9.96461537460074
1996/01/09 03:52:39 14:06:59 +37:05:59 RB 2.81136732863577 42.4950207582915 57.6610445490399 -96.4533848355757 -29.8099088285637 -3.37184454456609 3.37184454456609 -29.8099088285637
2062/12/23 13:09:35 10:01:48 +61:19:42 RJ 4.96894415296946 -0.855859578751078 339.286389072271 43.3709844282566 20.6015841828142 21.8076759229043 -21.8076759229043 20.6015841828142
1985/03/01 07:05:11 02:39:16 -49:41:16 RB 4.54242982367992 -46.4385969604701 142.620058934752 -118.337614566536 -26.4049777744982 -14.2399841547755 14.2399841547755 -26.4049777744982
2016/10/12 22:39:41 11:35:06 -37:41:59 RJ 6.22705381754439 -71.8689829359375 172.854948209352 -171.485222651176 -4.44193459939349 -29.6693312532438 29.6693312532438 -4.44193459939349
1996/06/07 15:05:35 11:01:13 +01:39:09 RB 2.05284951179683 39.3559481353143 105.190411532615 -65.2506191680043 -27.2444309408102 12.5594977093603 -12.5594977093603 -27.2444309408102
2010/09/04 07:39:35 02:05:11 +32:16:48 RB 1.62586919194211 34.5682927921293 296.52092624672 87.2180883114464 29.9646452840529 1.45603331036342 -1.45603331036342 29.9646452840529
1991/01/04 08:59:42 03:11:11 +58:35:06 RJ 4.07244535180856 -11.4780608544345 3.03191356807041 -5.47354530589 -2.86158439438404 29.8632103892702 -29.8632103892702 -2.86158439438404
2074/04/18 04:59:07 09:19:52 -03:39:35 RB 4.82424265119312 -43.3825746861388 283.510229465582 66.5163460685515 27.5152139180341 11.9546226642593 -11.9546226642593 27.5152139180341
2025/02/09 18:07:13 09:19:19 -49:01:39 RB 0.817202458746105 -17.2450138772442 137.099312464685 -79.4238682892185 -29.4903568306533 5.5062558967724 -5.5062558967724 -29.4903568306533
2020/09/04 17:48:52 00:59:13 +25:01:07 RJ 4.30144071691607 -22.8109305763712 50.1101420954919 -52.8704061391766 -23.9181677791085 18.1085960275891 -18.1085960275891 -23.9181677791085
2028/02/25 13:01:24 22:01:19 -84:13:39 RB 6.02516553156589 -13.7947708067906 181.309733082761 11.5655306341219 6.01465705434284 29.3908812477381 -29.3908812477381 6.01465705434284
2011/07/23 19:59:42 23:06:52 -73:59:07 RJ 4.12019360318504 -24.7500843304974 163.437640233059 -104.539410635685 -29.0392556591694 -7.53137641878265 7.53137641878265 -29.0392556591694
2016/11/19 08:11:59 09:48:35 +61:06:52 RJ 3.08433566796882 43.9594495309303 340.792438332148 140.282696575558 19.1700044513425 -23.0761983293503 23.0761983293503 19.1700044513425
2023/12/04 15:24:42 10:09:13 -29:09:24 RJ 5.22367046699131 -58.4132330178748 246.04735043967 99.7493550643114 29.5667389570861 -5.08015230515083 5.08015230515083 29.5667389570861
2029/12/12 15:42:24 18:52:13 -38:41:16 RB 5.44824916384789 26.0654123098247 203.893229674111 29.1721076135267 14.6230395202882 26.1947841218073 -26.1947841218073 14.6230395202882
2024/03/09 13:16:35 23:05:48 -38:07:11 RJ 0.0314208385664441 30.486063823152 193.711679219124 16.4361377288421 8.48839384230999 28.7740711401399 -28.7740711401399 8.48839384230999
2005/04/01 01:48:16 20:09:16 +73:11:42 RB 3.69183228734421 18.7821289708813 17.6138553041877 -83.5140862588236 -29.8079897051947 3.38876817368931 -3.38876817368931 -29.8079897051947
2076/09/09 18:11:59 12:35:35 +61:24:42 RB 4.49612657924059 28.2440475045172 329.188472651052 99.8440522590355 29.5583022144077 -5.12901259520204 5.12901259520204 29.5583022144077
2066/01/12 09:19:24 18:24:01 -41:52:01 RJ 4.31065048882273 22.3200474219995 156.100774823014 -30.7603145805973 -15.3434337314625 25.7794305858027 -25.7794305858027 -15.3434337314625
1981/12/12 19:13:19 06:16:24 -01:05:13 RJ 0.0812835690585687 0.371886519908359 91.2828709433876 -70.1655940849603 -28.2203156517637 10.1790856423759 -10.1790856423759 -28.2203156517637
2039/10/14 09:13:09 02:19:19 -16:01:05 RJ 2.72280095869645 -33.6840449662104 263.780885103298 76.4343725226246 29.1630567192198 7.03676934335431 -7.03676934335431 29.1630567192198
2073/10/25 21:48:42 01:06:24 +84:01:39 RJ 6.23095921646737 24.9883707105689 2.2871708435095 -157.292414624262 -11.5808452067481 -27.6746097406511 27.6746097406511 -11.5808452067481
2041/03/12 10:48:19 16:42:35 -20:05:05 RJ 5.71545820299856 5.36215807404969 246.308197278257 66.5906385310227 27.5306917216 11.9189350753422 -11.9189350753422 27.5306917216
2053/09/01 11:06:35 00:16:13 +16:06:05 RJ 2.48992879919024 -35.0269490802721 308.255238102366 50.3593888790029 23.10183733673 19.1390990296635 -19.1390990296635 23.10183733673
2020/02/18 15:59:05 11:11:19 -84:13:19 RJ 0.397757040777056 -24.4387372437999 176.440283643285 -143.771835792735 -17.7300679249992 -24.2000969290396 24.2000969290396 -17.7300679249992
2076/09/12 21:48:06 15:13:41 -84:19:48 RJ 5.49330265703375 -18.9822205788476 185.65063947009 80.1049450336313 29.5537248357296 5.15532233075508 -5.15532233075508 29.5537248357296
2073/04/25 15:11:19 03:13:06 +61:39:59 RB 1.3442001882701 43.9941566309262 343.318546311998 144.740620060294 17.318366261117 -24.4964117789892 24.4964117789892 17.318366261117
2026/05/14 03:41:06 11:42:13 +01:06:42 RJ 4.92380720531847 -14.891345847876 276.566321572274 69.1808118913389 28.0412010057062 10.662600346894 -10.662600346894 28.0412010057062
2035/01/23 07:05:59 19:48:11 +01:01:52 RJ 3.90758997027547 15.7977495142121 94.6096644748403 -69.6989073649898 -28.1364695505379 10.408606113772 -10.408606113772 -28.1364695505379
2046/06/18 12:06:01 06:01:13 +84:41:13 RJ 1.45459048486491 25.0542225875304 1.04817373226868 -169.303982136447 -5.56794965939371 -29.4787709477593 29.4787709477593 -5.56794965939371
2047/08/09 03:42:06 13:06:42 -58:59:16 RB 0.140216376725693 -49.6436776371868 187.948449970826 165.14340374957 7.69201957548935 -28.9971176990109 28.9971176990109 7.69201957548935
2030/05/25 20:35:06 13:39:11 +86:42:42 RJ 3.26695947780607 23.1116621317379 1.0845199163983 -162.738425069387 -8.9020350801842 -28.648800523428 28.648800523428 -8.9020350801842
1981/12/07 16:05:16 22:07:39 +32:06:07 RJ 5.45568286882355 68.9786776453616 50.7294524903974 -120.802931602168 -25.7680108969399 -15.3626044151109 15.3626044151109 -25.7680108969399
2069/07/23 19:48:24 15:48:42 -73:24:39 RB 4.08712684091539 -3.6773043884775 178.185361460407 -6.11396397217696 -3.19519216584354 29.8293604863284 -29.8293604863284 -3.19519216584354
2009/04/07 23:06:24 10:07:24 -51:41:39 RB 3.10466902906038 14.8226952042694 195.895155032999 24.7339995244433 12.5521833678828 27.2478016122227 -27.2478016122227 12.5521833678828
2044/01/09 19:01:01 13:52:11 -16:59:16 RJ 0.508960128892546 -87.3356376070152 10.679138623521 -10.5150368592919 -5.47480701869431 29.4962114195714 -29.4962114195714 -5.47480701869431
2012/04/07 07:05:42 18:13:35 -61:09:35 RB 5.18924609704035 7.04745816797115 190.725370436447 21.2606239773842 10.8783254431815 27.958219463196 -27.958219463196 10.8783254431815
2008/12/19 23:01:07 19:52:42 -49:59:19 RB 1.20644622107205 -40.3902544734562 220.550566751556 108.522509625001 28.445967704237 -9.53031591131708 9.53031591131708 28.445967704237
1998/08/18 10:59:39 19:59:42 -37:11:41 RB 2.20704408165071 -72.0562518310895 165.021645229912 -162.261730289392 -9.14007919738144 -28.5737458563905 28.5737458563905 -9.14007919738144
2035/08/07 06:48:05 14:05:01 +51:05:39 RB 0.917820082252408 -17.1929395065083 13.4586169764743 -20.2217221499457 -10.3696193549869 28.1508613444186 -28.1508613444186 -10.3696193549869
2063/05/09 03:11:52 15:06:01 +49:07:42 RB 4.71055858151125 45.4695868566184 320.553635287346 115.106018328858 27.1657270876467 -12.7288362311524 12.7288362311524 27.1657270876467
2076/03/01 16:07:09 21:48:35 -38:41:41 RB 0.647012126117251 3.88540274108194 227.006980994068 60.975505542544 26.2323710033618 14.5555045100465 -14.5555045100465 26.2323710033618
