#
# Command Line Registrar Client
# -----------------------------
#
# This provides a very light-weight command line Registrar Client (see rc.tcl
# for information on Registrar Clients in general).  The idea with this client
# is to allow joining "standard" conferences very easily, such as a the
# PostIt conference for example (see postit.tcl).  As such, this Registrar
# Client could be started up in a .xinitrc file.
#
# This registrar client is started up indirectly via a one line shell
# script (see cmdrc).  A typical invocation would be:
#
#      cmdrc PostIt
#
# This will join a conference named "PostIt" if it already exists, or create
# the conference if it doesn't yet exist.  The conference will be created
# with the type "PostIt" (which indirectly specifies the groupware conference
# application to be used; see the documentation for the proc createConference
# in rc.tcl for details). 
#
# This Registrar Client remains running until the conference that it spawns
# terminates.
#

initGkErrorHandling

#
# Maintain a flag specifying if we've initiated a join to the conference.
# The global target_name is the name of the conference we'd like to join,
#   e.g. PostIt
#

set already_joined 0
set target_name $argv
wm withdraw .

PollConferences


#
# This is called by updatelist when a new conference is found.  If the
# conference is the one we're looking for, join to it.  Update our conference
# list.
#

proc foundNewConf conf {    global conflist userlist already_joined target_name
    PollUsers [keylget conf confnum]
    lappend conflist $conf
    set userlist([keylget conf confnum]) {}
    if {([keylget conf confname] == $target_name) && ($already_joined ==0)} {
	set already_joined 1
	callJoinConference [keylget conf confnum]
	PollUsers [keylget conf confnum]
    }
}


#
# This is called bu updatelist when a new user is found.  If the new user
# is us, create the conference.  Update our user list.
#

proc foundNewUser user {    global host myport userlist
    set confnum [keylget user confnum]
    lappend userlist($confnum) $user
    if {([keylget user host] == $host) && ([keylget user port] == $myport)} {
	createConference [findconfinfo $confnum] [keylget user usernum]
    } else {
	joinTo $user $user
    }
}


#
# This is called by updatelist when a deleted user is found.  Update our
# user list.  Pass the delete on to our conference.
#

proc foundDeletedUser {conf user} {  global userlist
    set idx 0; foreach i $userlist($conf) {
	if {[keylget i usernum] == $user} {
	    toConf $conf "removeUser [list [lindex $userlist($conf) $idx]]"
	    set userlist($conf) [lreplace $userlist($conf) $idx $idx]
	} else { incr idx }
    }
}


#
# This is called by updatelist when a deleted conference is found.  Update
# our conference list.  Pass the delete on to our conference.
#

proc foundDeletedConf conf { global conflist userlist
    set idx 0; foreach i $conflist {
	if {[keylget i confnum] == $conf} {
	    set conflist [lreplace $conflist $idx $idx]
	    set userlist($conf) ""
	    toConf $conf "deleteconf"
	} else { incr idx }
    }
}


#
# This overrides the conflist routine in rc.tcl.  Basically after going
# through the entire list of conferences, we check to see if we've found
# the conference we're looking for.  If not, it must not be there so we
# create the conference ourself.
#

proc conflist confs {  global already_joined host myport target_name registrar
    updatelist $confs conflist conflist confnum foundNewConf foundDeletedConf
    if {$already_joined == 0} {
	keylset conf confname $target_name confType $target_name \
	    originator $host$myport
	callNewConference $conf
	after 1000 PollConferences
    }
}


#
# This is called by errors.tcl if a socket connection dies.  If it is the
# conference that we spawned dying, first pass this on to the central
# Registrar and then kill ourself.
#

proc socket_closed filedesc { global conferences
    foreach i $conferences {
	if {[keylget i filedesc] == $filedesc} {
	    userLeft [keylget i confnum] [keylget i usernum]
	    after 1000 destroy .
	}
    }
}



