/*
 * Copyright 1992 by the National Optical Astronomy Observatories(*)
 *
 * Permission to use, copy, and distribute
 * is hereby granted without fee, providing that the above copyright
 * notice appear in all copies and that both the copyright notice and this
 * permission notice appear in supporting documentation.
 *
 * This software is provided "as is" without any express or implied warranty.
 *
 * (*) Operated by the Association of Universities for Research in
 *     Astronomy, Inc. (AURA) under cooperative agreement with the
 *     National Science Foundation.
 */
/* Program: PortMgrUser.c
 *      C-level interface code to the TCL ipc Port Manager.  Based on
 *      the functions in George Drapeau's MAEstro PortManager.
 *
 * Created: K. Gillies 26 June 1992
 * SCCS INFO
 *      @(#)PortMgrUser.c	1.1 9/11/92
 *
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <string.h>
#include "tcl.h"
#include "tclIPC.h"
#include "tclipcP.h"
#include "tclipc.h"

/* For Future use */
#define LaunchNewApp ":LaunchNewApp:" /* Used by SenderGetPortFromName */

/*
 *--------------------------------------------------------------
 *
 * pmConnectWithPortMgr --
 *
 *      Register a Port with a port manager
 *
 * Results:
 *      Tcl_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
int pmConnectWithPortMgr(sender, receivingPort)
Sender sender;                 /* PortManager Sender */
Port* receivingPort;           /* The Port that should be registered */
{
  /* Connect to the process on the portmanager machine */
  return ipcRemoteSend(sender, "connect %s %s %d", 
                           receivingPort->appName, ipcThisHostName,
			   receivingPort->portNumber);
}            

/*
 *--------------------------------------------------------------
 *
 * pmIsPortUnique --
 *
 *      Check to see if a desired receiver Port is unique.
 *
 * Results:
 *      Tcl_OK or TCL_ERROR
 *--------------------------------------------------------------
 */
int pmIsPortUnique(sender, receivingPort)
Sender sender;                 /* PortManager Sender */
Port* receivingPort;           /* The Port that should be registered */
{
  /* Connect to the process on the portmanager machine */
  return ipcRemoteSend(sender, "isunique %s %s", 
			   receivingPort->appName, receivingPort->hostName);
}            

/*
 *--------------------------------------------------------------
 *
 * pmGetOpenApps
 *
 *      Get the list of registered apps on a port manager.
 *
 * Results:
 *      Tcl_OK or TCL_ERROR and openAppsReturn is set to the open apps list.
 *--------------------------------------------------------------
 */
int pmGetOpenApps(sender, openAppsReturn)
Sender sender;
PortMgrResult* openAppsReturn;
{
  int result;

  /* Connect the process on the portmanager machine */
  result = ipcRemoteSend(sender, "openapps");
  if (result != TCL_OK) {
    Tcl_AppendResult(sender->interp, 
		     "The \"openapps\" message failed.\n", (char *)NULL);
    return TCL_ERROR;
  }
  /* Check to see if any process/port information was returned */
  *openAppsReturn = (strlen(sender->interp->result) == 0) ?
                               NULL : 
			       strdup(sender->interp->result);
  return TCL_OK;
} 

/*
 *--------------------------------------------------------------
 *
 * pmDisconnectFromPortMgr
 *
 *      Remove an application registration from a port manager.
 *
 * Results:
 *      Tcl_OK or TCL_ERROR 
 *--------------------------------------------------------------
 */
int pmDisconnectFromPortMgr(sender, appPort)
Sender sender;
Port* appPort;
{
  int result;
  
  /* Use the local hostname.  You can only disconnect programs on
   * your own machine here */
  /* Disconnect the entry on the PortManager */
  result = ipcRemoteSend(sender, "disconnect %s %s %d", 
			  appPort->appName, ipcThisHostName, 
			  appPort->portNumber);
  if (result != TCL_OK) {
    Tcl_AppendResult(sender->interp, 
		     "The DisconnectFromPortMgr message failed.\n", 
		     (char *)NULL);
    return TCL_ERROR;
  }
  return TCL_OK;
}


/******************************************************************
 *      pmGetPortFromName
 *
 *      This function is responsible for making sure that the
 *      application specified by the "appNameAndHost" argument is
 *      open and listening for messages.
 *
 *      The function sends a message to the Port Manager to see if
 *      the requested application is already alive.  If not, this
 *      function will attempt to launch the application.
 *      In addition, this function allows the caller to explicitly
 *      request that a new version be launched, even if one is
 *      already running.  To do so, the caller must set the
 *      appNameAndHost->hostName field to the defined constant
 *      "LaunchNewApp".
 *
 *      If the test fails, the function will try to launch a new copy of
 *      the requested app, using the SenderLaunchApplication() function.
 *      The function will sleep for a time, giving the new application
 *      some time to start up.
 *      After waiting, the function sends another GetPortFromName
 *      message to the Port Manager, to see if the new application is
 *      alive and listening for messages.  If so, the function will
 *      return successfully.  If not, the function goes into a loop,
 *      sleeping for about one second, then retrying the GetPortFromName
 *      request.
 */

/*
 *--------------------------------------------------------------
 *
 * pmGetPortFromName
 *
 *      Given an application label, host, and progra, return its
 *      Port.  Returns a PortMgrResult in matchingPortsReturn.
 *
 * Results:
 *      Tcl_OK or TCL_ERROR 
 *--------------------------------------------------------------
 */
int pmGetPortFromName(sender, appNameAndHost, matchingPortsReturn)
Sender sender;
Port* appNameAndHost;
PortMgrResult* matchingPortsReturn;
{
  int result;

  /* If ANYHOST, user doesn't care about which machine */
  if (strcmp(appNameAndHost->hostName, ANYHOST) == 0) {
    result = ipcRemoteSend(sender,
			   "portfromname %s", appNameAndHost->appName);
  } else {
    /* App on specific machine */
    result = ipcRemoteSend(sender, 
			    "portfromname %s %s", 
			    appNameAndHost->appName, 
			    appNameAndHost->hostName);
  }
  if (result != TCL_OK) {
    Tcl_AppendResult(sender->interp, 
		     "GetPortFromName message failed.\n", (char *)NULL); 
    return TCL_ERROR;
  }

  /* Check to see if any process/port information was returned */
  /* A 0 length string indicates that no matching apps were found */
  *matchingPortsReturn = (strlen(sender->interp->result) == 0) ?
                           NULL : 
			   strdup(sender->interp->result);

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * pmGetPortFromResult
 *
 *      Get Port data from a Port Manager Result.  whichone is an
 *      index indicating which list member.  appPortReturn is set if
 *      successful.
 *
 * Results:
 *      Tcl_OK or TCL_ERROR and appOrtReturn is set if OK.
 *--------------------------------------------------------------
 */
int pmGetPortFromResult(interp, whichone, pmresult, appPortReturn)
Tcl_Interp *interp;
int whichone;
PortMgrResult pmresult;
Port *appPortReturn;
{

  int argc;
  char **argv = NULL;
  char hostName[MAXHOSTNAMELEN];
  int portNumber;

  if (pmresult == NULL) {
    /* The PortManager found no applications */
    return TCL_ERROR;
  }
  /* Break the list into pieces and fetch "whichone" */
  if (Tcl_SplitList(interp, pmresult, &argc, &argv) != TCL_OK) {
    Tcl_AppendResult(interp, "PortFromResult failed.\n", (char *)NULL);
    return TCL_ERROR;
  }
  if (whichone > argc) {
    Tcl_AppendResult(interp, "There are only ", argv[0],
		     " applications registered.", (char *)NULL);
    return TCL_ERROR;
  }
  /* Get the host name and its port */
  sscanf(argv[whichone], "%s%d", hostName, &portNumber);
  /* The caller must free this port name */
  appPortReturn->hostName = strdup(hostName);
  appPortReturn->portNumber = portNumber;

  /* Free the split list */
  ckfree((char *)argv); 

  return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * pmDestroyResult
 *
 *      User interface to destroy a pmResult.
 *
 * Results:
 *      Tcl_OK or TCL_ERROR 
 *--------------------------------------------------------------
 */
int pmDestroyResult(pmresult)
PortMgrResult pmresult;
{
  ckfree((char *)pmresult);
  return TCL_OK;
}
