
#---------------------------------------------------------------------------
#
#	Simple text editing widget with the possibility of attaching
#	font and style attributes to portions of text.
#
#	Juergen Wagner, Jan 1993
#
#---------------------------------------------------------------------------

defwidget Textedit {} {{fonts {
			{fixed Typewriter}
			{small Small}
			{smallbold Small Bold}
			{text Normal}
			{textbold Normal Bold}
			{large Large}
			{largebold Large Bold}
			{huge Very Large}
			{hugebold Very Large Bold}}}
}


defmethod Textedit new {name args} {

  args	layout {title {Text Editor}} actions readonly embedded \
	initmethod init {width 60} {height 15} {textfont text} \
	values raw fit

  if { $embedded == "true" } {
    Frame new $name -relief flat
  } {
    Toplevel new $name -title $title -resizable true \
	-buttons {Save} \
	-actions [concat {{Dismiss Dismiss}} $actions] \
	-handler $name
  }
  defsuper $name $self
  $name slot _raw $raw

  Scrollbar new $name.scroll -dir vertical -action "$name.entry yview" \
	-layout {right filly}

  Text new $name.entry -textfont $textfont \
	-width $width -height $height -wrap word -state normal \
	-yscroll "$name.scroll! set" \
	-filter [list $name _filter] \
	-layout {left fill expand}

  set fonts [$self slot fonts]
  foreach f $fonts {
    $name.entry! tag configure [lindex $f 0] -font [Font slot [lindex $f 0]]
  }
  $name.entry! tag configure format -font [Font slot textitalics]

  $name slot _initmethod $initmethod
  $name slot _init $init
  $name slot _values $values

  if { $initmethod != {} } {
    $name init-$initmethod $init $values
  }

  if { $readonly == "true" } {
    bind $name.entry <1> nil
    bind $name.entry <B1-Motion> nil
    bind $name.entry <B1-ButtonRelease> nil

    $name.entry! configure -state disabled
  } {
    set menu [Popup new $name.popup -parent $name.entry]

    $menu addItem -text "Add format control" -action [list $name _format]
    foreach f $fonts {
      $menu addItem -text [lrange $f 1 end] -textfont [lindex $f 0] \
	-action [list $name _font [lindex $f 0]]
    }

    focus $name.entry
  }

  if { $fit == "true" } {
    $name layout $layout

    for {set i 8} {$i < 50} {incr i 4} {
      $name.entry! configure -height $i
      update
      set region [$name.scroll! get]
      if { [lindex $region 0] <= [lindex $region 1] } {
	break
      }
    }
  }

  $name layout $layout
}

defmethod Textedit _filter {key} {

  if { $key == "\r" } {
    return "\n"
  }
  if { $key >= " " || $key == "\t" || $key == "\n" } {
    return $key
  }
  return
}

#---------------------------------------------------------------------------

defmethod Textedit load {text tags values} {

  $self.entry set $text

  foreach spec $tags {
    set tag [lindex $spec 0]
    set from {}
    foreach range [lrange $spec 1 end] {
      if { $from == {} } {
	set from $range
      } {
	$self.entry! tag add $tag $from $range
	set from {}
      }
    }
  }

  foreach value $values {
    if { [catch {set pos [$self.entry! index format.first]}] } {
      break
    }
    $self.entry! insert $pos $value
    $self.entry! tag add textbold $pos format.first
    $self.entry! delete format.first
  }

  if { [$self slot _raw] != "true" } {
    while { ! [catch {$self.entry! delete format.first}] } {
    }
  }

  return $self
}

defmethod Textedit init-Formatted {file values} {
  global _textedit

  set _textedit(text) {}
  set _textedit(tags) {}

  catch {source $file}

  set text $_textedit(text)
  set tags $_textedit(tags)

  unset _textedit(text)
  unset _textedit(tags)

  $self load $text $tags $values
}

defmethod Textedit init-Text {file values} {

  $self load [app(getfile) $file] {{bold 0.0 end}} $values
}

defmethod Textedit init-String {text values} {

  $self load $text {} {}
}

defmethod Textedit init-List {list values} {

  $self load [lindex $list 0] [lrange $list 1 end] $values
}

defmethod Textedit init-Form {expression} {

  set list [uplevel #0 [list eval [list $form]]]
  $self load [lindex $list 0] [lrange $list 1 end] $values
}

#---------------------------------------------------------------------------

defmethod Textedit Save {action} {

  if { $action != {} } {
    set text [$self.entry! get 0.0 end]
    set tags {}
    foreach f [Textedit slot fonts] {
      set tag [lindex $f 0]
      lappend tags [concat $tag [$self.entry! tag ranges $tag]]
    }
    lappend tags [concat format [$self.entry! tag ranges format]]

    eval [concat $action [list $text $tags]]
  }
}

defmethod Textedit save-Formatted {file text {tags {}}} {

  set fp [open $file w]
  puts $fp [list set _textedit(text) $text]
  puts $fp [list set _textedit(tags) $tags]
  close $fp
  return
}

defmethod Textedit save-Text {file text {tags {}}} {

  set fp [open $file w]
  puts $fp $text
  close $fp
  return
}

#---------------------------------------------------------------------------

defmethod Textedit _format {} {

  set start [$self.entry! index insert]
  $self.entry! insert insert "@"
  set end [$self.entry! index insert]

  $self.entry! tag add format $start $end
  return
}

defmethod Textedit _font {newfont} {

  if { [catch {set startsel [$self.entry index sel.first]}] } {
    return
  }
  if { [catch {set endsel [$self.entry index sel.last]}] } {
    return
  }

  foreach f [Textedit slot fonts] {
    $self.entry tag remove [lindex $f 0] $startsel $endsel
  }
  $self.entry tag add $newfont $startsel $endsel

  return
}

Window addDemo Textedit

defmethod Textedit demo {} {

  Textedit new * -initmethod Text -init /etc/hosts -layout -0-0
}

