#include "protos.h"

/*
 * This software is Copyright (C) 1988 by Steven Dorner and the
 * University of Illinois Board of Trustees.  No warranties of any
 * kind are expressed or implied.  No support will be provided.
 * This software may not be redistributed for commercial purposes.
 * You may direct questions to nameserv@uiuc.edu
 */

#include <sys/param.h>

/*
 * this file contains functions that deal with fields
 */
FDESC	**FieldDescriptors;
static int LineNumber;

#define FREEFD(FD)    {free((FD)->fdName);free(FD);}
#define IS_A_COMMENT(l) (*(l) == '#')
#define IS_BLANK(l)   (!(*(StrSkip(l," \t\n\f"))))

extern int IndicateAlways;

/*
 * read the field config file.	Read into global FieldDescriptors
 */
#define GUESSCOUNT 35		/* guess how many FD's there are */
int 
GetFieldConfig()
{
	int	fieldCount;
	int	i;
	FILE	*fp;
	FDESC	*fd;
	char	fconfig[MAXPATHLEN];

	(void) sprintf(fconfig, "%s.cnf", Database);

	if ((fp = fopen(fconfig, "r")) == NULL)
	{
		IssueMessage(LOG_INFO, "Couln't open field config file %s.\n", fconfig);
		return (0);
	}
	if (FieldDescriptors)
	{
		for (; *FieldDescriptors; FieldDescriptors++)
			FREEFD(*FieldDescriptors);
		FieldDescriptors = NULL;
	}
	fieldCount = GUESSCOUNT;
	FieldDescriptors = (FDESC **) malloc(GUESSCOUNT * sizeof (FDESC *));
	LineNumber = 0;

	for (i = 0; fd = GetFD(fp); i++)
	{
		if (fd == NULL)
		{
			IssueMessage(LOG_INFO, "Error in %s, line %d.\n", fconfig, LineNumber);
			i--;	/* back up loop counter! */
		} else
		{
			FieldDescriptors[i] = fd;
			/* DumpFD(stdout,FieldDescriptors[i]); */
			if (i == fieldCount - 1)
			{
				fieldCount += GUESSCOUNT;
				FieldDescriptors = (FDESC **)
				    realloc(FieldDescriptors, fieldCount * sizeof (FDESC *));
			}
		}
	}

	FieldDescriptors[i] = NULL;

	FieldDescriptors =
	    (FDESC **) realloc(FieldDescriptors, (i + 1) * sizeof (FDESC *));

	return (i);
}

/*
 * read a single field descriptor from a file.	storage is malloc'ed
 */
FDESC *
GetFD(fp)
	FILE *fp;
{
	static char Line[MAX_LINE + 2];
	static FDESC blankFD;
	FDESC	*fd;
	FDESC	tempFD;
	int	number;
	char	*token;
	char	help[MAX_LINE];

	*Line = '\0';
	do
	{
		if (fgets(Line, MAX_LINE + 1, fp) == NULL)
			return (NULL);
	}
	while (!*Line || IS_A_COMMENT(Line) || IS_BLANK(Line));

	/* got a non-comment, non-blank line.  now try to parse it. */

	tempFD = blankFD;

	/* strip the newline */
	/* Line[strlen(Line) - 2] = '\0'; */
	if ((token = index(Line, '\n')) != NULL)
		*token = '\0';

	/* Id number */
	if (!(token = strtok(Line, ":")))
		return (NULL);
	tempFD.fdId = atoi(token);

	/* name */
	if (!(token = strtok(NULL, ":")))
		return (NULL);
	tempFD.fdName = make_str(token);

	/* max length */
	if (!(token = strtok(NULL, ":")))
	{
		free(tempFD.fdName);
		return (NULL);
	}
	tempFD.fdMax = atoi(token);

	/* help */
	*help = '\0';
	for (token = strtok(NULL, ":"); token; token = strtok(NULL, ":"))
	{
		number = strlen(token);
		if (number)
		{
			if (token[number - 1] == '\\')	/* ends in backslash? */
				token[number - 1] = ':';	/* turn it into colon */
			else
				number = 0;
		}
		strcate(help, token);
		if (!number)
			break;	/* stop with help */
	}
	tempFD.fdHelp = make_str(help);

	/* merge flags */
	if (token = strtok(NULL, ":"))
		tempFD.fdMerge = make_str(token);
	else
		tempFD.fdMerge = "";

	/* flags */
	for (token = strtok(NULL, ":"); token; token = strtok(NULL, ":"))
	{
		if (islower(*token))
			*token = toupper(*token);
		switch (*token)
		{
		    case 'A':
			tempFD.fdAlways = 1;
			break;
		    case 'C':
			tempFD.fdChange = 1;
			break;
		    case 'D':
			tempFD.fdDefault = 1;
			break;
		    case 'E':
			tempFD.fdEncrypt = 1;
			break;
		    case 'F':
			tempFD.fdForcePub = 1;
			break;
		    case 'I':
			tempFD.fdIndexed = 1;
			break;
		    case 'L':
			tempFD.fdLookup = 1;
			break;
		    case 'N':
			tempFD.fdNoPeople = 1;
			break;
		    case 'P':
			tempFD.fdPublic = 1;
			break;
		    case 'S':
			tempFD.fdSacred = 1;
			break;
		    case 'T':
			tempFD.fdTurn = 1;
			break;
		    case 'W':
			tempFD.fdAny = 1;
			break;	/* Wildany */
		    default:
			IssueMessage(LOG_INFO, "%s: unknown field flag\n", token);
			break;
		}
	}

	fd = (FDESC *) malloc(sizeof (FDESC));
	*fd = tempFD;	/* intentional structure assignment */
	return (fd);
}

/*
 * dump a field descriptor
 */
void 
DumpFD(fp, fd)
	FILE *fp;
	FDESC *fd;
{
	fprintf(fp, "%d\t%s\t%s\t%s\t%s\t%s\t%d\t%s\n%s",
		fd->fdId,
		fd->fdIndexed ? "idx" : "!idx",
		fd->fdLookup ? "lup" : "!lup",
		fd->fdPublic ? "pub" : "!pub",
		fd->fdDefault ? "dft" : "!dft",
		fd->fdAlways ? "alw" : "!alw",
		fd->fdAny ? "any" : "!any",
		fd->fdChange ? "chg" : "!chg",
		fd->fdNoPeople ? "npl" : "!npl",
		fd->fdMax,
		fd->fdName,
		fd->fdHelp);
}

/*
 * advance a string pointer past all occurrences of a set of chars.
 */
char *
StrSkip(str, skipChars)
	char *str, *skipChars;
{
	register char *skipThis;/* current char in skip set */
	register char chr;	/* current char in str */

	for (chr = *str;; chr = *(++str))
	{
		for (skipThis = skipChars; *skipThis; skipThis++)
			if (chr == *skipThis)
				goto outerFor;	/* chr is in skipChars */
		return (str);	/* chr is NOT in skipChars */
	      outerFor:;
	}
}

/*
 * find a numbered field in a QDIR structure
 * returns index of field, or -1 if not found
 */
int 
FindField(indir, num)
	QDIR indir;
	int num;
{
	char	ascii[20];
	register char **dir;
	int	len;

	(void) sprintf(ascii, "%d:", num);
	len = strlen(ascii);

	for (dir = indir; *dir; dir++)
		if (!strncmp(ascii, *dir, len))
			return (dir - indir);

	return (-1);
}

/*
 * find a field descriptor by Id
 */
FDESC *
FindFDI(indx)
	int indx;
{
	FDESC **fd;

	for (fd = FieldDescriptors; *fd; fd++)
		if ((*fd)->fdId == indx)
			return (*fd);

	return (NULL);
}

/*
 * find a field descriptor by name
 */
FDESC *
FindFD(name)
	char *name;
{
	FDESC **fd;

	for (fd = FieldDescriptors; *fd; fd++)
		if (!stricmp((*fd)->fdName, name))
		{		/* always field/property. Probably a better way to do this.  mae */
			if (!stricmp("Always", name))
			{
				IndicateAlways = 1;
			}
			return (*fd);
		}
	return (NULL);
}
