/*
**  Lexical analyzer for the gag language.  We have our own FSA
**  for comments because it's much smaller and quicker that way.
*/
%{
#include "gate.h"
#include "gag.h"

/* State of our automaton. */
typedef enum {
    S_STAR, S_NORMAL, S_END
} STATE;

/* Key-value pair. */
typedef struct {
    char	*name;
    int		value;
} PAIR;

char		yyfilename[SM_SIZE];

/* List of GAG's keywords. */
STATIC PAIR	Keywords[] = {
    {	"command",		COMMAND		},
    {	"default",		DEFAULT		},
    {	"directory",		DIRECTORY	},
    {	"distributions",	DISTRIBUTIONS	},
    {	"do",			DO		},
    {	"dotify",		DOTIFY		},
    {	"false",		FALSEt		},
    {	"flags",		FLAGS		},
    {	"gateway",		GATEWAY		},
    {	"inews",		INEWSt		},
    {	"mail2news",		MAIL2NEWS	},
    {	"mailcontact",		MAILCONTACT	},
    {	"mailhost",		MAILHOST	},
    {	"mailinglist",		MAILINGLIST	},
    {	"mailpost",		MAILPOST	},
    {	"moderator",		MODERATOR	},
    {	"news2mail",		NEWS2MAIL	},
    {	"no",			NO		},
    {	"owner",		OWNER		},
    {	"organization",		ORGANIZATION	},
    {	"owner",		OWNER		},
    {	"request_address",	REQUESTADDR	},
    {	"site",			SITE		},
    {	"true",			TRUEt		},
    {	"user",			USER		},
    {	NULL,			0		}
};

%}

%%

[-+0-9A-Za-z_.]+	{
		    /* A simple ID or keyword. */
		    register PAIR	*p;

		    /* Keyword? */
		    for (p = Keywords; p->name; p++)
			if (EQ(p->name, yytext))
			    return p->value;
		    yylval.String = COPY(yytext);
		    return ID;
		}

^#[ \t]+[0-9]+[ \t]+"[^\n]+"[^\n]*$	{
		    /* C pre-processor control line. */
		    register char	*p;
		    char		*namep;

		    /* Find the line number. */
		    for (p = yytext; *p && !isdigit(*p); p++)
			;
		    /* Parse the number, find the start of the filename. */
		    for (yylineno = atoi(p); *p && *p != '"'; p++)
			;
		    /* March down to the end of the filename. */
		    for (namep = p; *++p && *p != '"'; p++)
			;
		    *p = '\0';
		    (void)strncpy(yyfilename, namep, sizeof yyfilename - 1);
		    yyfilename[sizeof yyfilename - 1] = '\0';
		}

\"[^"]*		{
		    /* Quoted string. */
		    int		c;

		    /* See the Lex paper in Volume 2A or PS1:16
		     * for details on this code. */
		    if (yytext[yyleng - 1] == '\\')
			yymore();
		    else {
			if ((c = input()) == '"') {
			    yylval.String = COPY(&yytext[1]);
			    return ID;
			}
			unput(c);
			yyerror("Bad string");
		    }
		}

"/*"	        {
		    /* Comment. */
		    register STATE	S;

		    for (S = S_NORMAL; S != S_END; )
			switch (input()) {
			case '/':
			    if (S == S_STAR) {
				S = S_END;
				break;
			    }
			    /* FALLTHROUGH */
			default:
			    S = S_NORMAL;
			    break;
			case '\0':
			    S = S_END;
			    break;
			case '*':
			    S = S_STAR;
			    break;
			}
		}

[ \t\n]		{
		    /* Tasty whitespace. */
#ifdef	lint
		    /* I am compulsive about lint natterings. */
		    yytext[0] = yyinput();
		    yyoutput(yytext[0]);
		    yyunput(yytext[0]);
		    REJECT;
#endif	/* lint */
		}

.		{
		    /* Random special character. */
		    return *yytext;
		}

%%


/*
**  Called by lex at end-of-stream.  Return one if no more input.
*/
int
yywrap()
{
    return 1;
}
