use warnings;
use strict;

package TextFileParser 0.1821902;

# ABSTRACT: an extensible Perl class to parse any text file by specifying grammar in derived classes.

use Exporter 'import';
our (@EXPORT_OK) = ();
our (@EXPORT)    = (@EXPORT_OK);


use Exception::Class (
    'TextFileParser::Exception',
    'TextFileParser::Exception::ParsingError' => {
        isa         => 'TextFileParser::Exception',
        description => 'For all parsing errors',
        alias       => 'throw_text_parsing_error'
    },
    'TextFileParser::Exception::FileNotFound' => {
        isa         => 'TextFileParser::Exception',
        description => 'File not found',
        alias       => 'throw_file_not_found'
    },
    'TextFileParser::Exception::FileCantOpen' => {
        isa         => 'TextFileParser::Exception',
        description => 'Error opening file',
        alias       => 'throw_cant_open'
    }
);

use English;
use Try::Tiny;


sub new {
    my $pkg = shift;
    bless {}, $pkg;
}


sub read {
    my ( $self, $fname ) = @_;
    return                    if not $self->__is_file_known_or_opened($fname);
    $self->filename($fname)   if not exists $self->{__filehandle};
    delete $self->{__records} if exists $self->{__records};
    $self->__read_file_handle;
    $self->__close_file;
}

sub __is_file_known_or_opened {
    my ( $self, $fname ) = @_;
    return 0 if not defined $fname and not exists $self->{__filehandle};
    return 0 if defined $fname and not $fname;
    return 1;
}


sub filename {
    my ( $self, $fname ) = @_;
    $self->__check_and_open_file($fname) if defined $fname;
    return ( exists $self->{__filename} and defined $self->{__filename} )
        ? $self->{__filename}
        : undef;
}

sub __check_and_open_file {
    my ( $self, $fname ) = @_;
    throw_file_not_found error =>
        "No such file $fname or it has no read permissions"
        if not -f $fname or not -r $fname;
    $self->__open_file($fname);
    $self->{__filename} = $fname;
}

sub __open_file {
    my ( $self, $fname ) = @_;
    $self->__close_file if exists $self->{__filehandle};
    open my $fh, "<$fname"
        or throw_cant_open error => "Error while opening file $fname";
    $self->{__filehandle} = $fh;
}

sub __read_file_handle {
    my ($self) = @_;
    my $fh = $self->{__filehandle};
    while (<$fh>) {
        $self->lines_parsed( $self->lines_parsed + 1 );
        $self->__try_to_parse($_);
    }
}


sub lines_parsed {
    my $self = shift;
    return $self->{__current_line} = shift if @_;
    return ( exists $self->{__current_line} ) ? $self->{__current_line} : 0;
}

sub __try_to_parse {
    my ( $self, $line ) = @_;
    try { $self->save_record($line); }
    catch {
        $self->__close_file;
        $_->rethrow;
    };
}


sub save_record {
    my $self = shift;
    return if not @_;
    $self->{__records} = [] if not exists $self->{__records};
    push @{ $self->{__records} }, shift;
}

sub __close_file {
    my $self = shift;
    close $self->{__filehandle};
    delete $self->{__filehandle};
}


sub get_records {
    my $self = shift;
    return () if not exists $self->{__records};
    return @{ $self->{__records} };
}


sub record_list_pointer {
    my $self = shift;
    return ( exists $self->{__records} ) ? $self->{__records} : undef;
}


sub last_record {
    my $self = shift;
    return undef if not exists $self->{__records};
    my (@record) = @{ $self->{__records} };
    return $record[$#record];
}

1;

__END__

=pod

=encoding UTF-8

=head1 NAME

TextFileParser - an extensible Perl class to parse any text file by specifying grammar in derived classes.

=head1 VERSION

version 0.1821902

=head1 SYNOPSIS

    use strict;
    use warnings;
    use TextFileParser;

    my $parser = new TextFileParser;
    $parser->read(shift @ARGV);
    print $parser->get_records, "\n";

The above code reads a text file and prints the content to C<STDOUT>.

Here's another parser which is derived from C<TextFileParser> as the base class. See how simple it is to make your own parser.

    use strict;
    use warnings;
    package WordCounter;
    use parent 'TextFileParser';
    use Exception::Class (
        'SomeException' => {
            description => 'Some description',
            alias => 'throw_some_error'
        }
    );

    sub save_record {
        my ($self, $line) = @_;
        my (@words) = split /\s+/, $line;
        throw_some_error 
            error => $self->filename . ': expected at least one character: ' . $self->lines_parsed
                if not @words;
        $self->SUPER::save_record(\@words);
    }

    package main;
    use Data::Dumper qw(Dumper);
    use Try::Tiny;

    try {
        my $a_parser = new WordCounter;
        $a_parser->read(shift @ARGV);
        print Dumper($a_parser->get_records);
    } catch {
        print STDERR $_, "\n";
    };

=head1 METHODS

=head2 new

Takes no arguments. Returns a blessed reference of the object.

    my $pars = new TextFileParser;

=head2 read

Takes zero or one string argument with the name of the file. Throws an exception if filename provided is either non-existent or cannot be read for any reason.

    $pars->read($filename);

    # The above is equivalent to the following
    $pars->filename($anotherfile);
    $pars->read();

Returns once all records have been read or if an exception is thrown for any parsing errors. This function will handle all C<open> and C<close> operations on all files even if any exception is thrown.

    use Try::Tiny;

    try {
        $pars->read('myfile.txt');
    } catch {
        print STDERR $_, "\n";
    }

You're better-off not overriding this subroutine. Override C<save_record> instead. If you want to intervene in the file C<open> step you can't do it for now. A new version will explain how you can do that.

=head2 filename

Takes zero or one string argument with the name of a file. Returns the name of the file that was last opened if any. Returns undef if no file has been opened. This is most useful in generating error messages.

=head2 lines_parsed

Takes no arguments. Returns the number of lines last parsed.

    print $pars->lines_parsed, " lines were parsed\n";

This is also very useful for error message generation. See example under L<Synopsis|/SYNOPSIS>.

=head2 save_record

Takes exactly one string argument. This method can be overridden in derived classes to extract the relevant information from each line and store records. In general once the relevant data has been collected, you would want to call C<SUPER::save_record>. By default, this method saves the input string as the record.

See L<Synopsis|/SYNOPSIS> for how a derived class could write their own method to handle data.

    package MyParser;
    use parent 'TextFileParser';

    sub save_record {
        my ($self, $line) = @_;
        my $data = __extract_some_info($line);
        $self->SUPER::save_record($data);
    }

Here's an example of a parser that reads multi-line records: if a line starts with a C<'+'> character then it is to be treated as a continuation of the previous line.

    use strict;
    use warnings;
    package MultilineWordCounter;
    use parent 'TextFileParser';

    sub save_record {
        my ($self, $line) = @_;
        my (@words) = split /\s+/, $line;
        my $method = ($#words >= 0 and $words[0] eq '+') ? '__append_last_record' : 'SUPER::save_record';
        $self->$method(\@words);
    }

    sub __append_last_record {
        my ($self, $plus, $words) = @_;
        my $last_rec = $self->last_record;
        push @{last_rec}, @{$words};
    }

=head2 get_records

Takes no arguments. Returns an array containing all the records that were read by the parser.

=head2 record_list_pointer

Takes no arguments and returns the reference to the array containing all the records. This may be useful if you want to re-order the records in some way.

=head2 last_record

Takes no arguments and returns the last saved record.

=head1 AUTHOR

Balaji Ramasubramanian <balajiram@cpan.org>

=head1 COPYRIGHT AND LICENSE

This software is copyright (c) 2018 by Balaji Ramasubramanian.

This is free software; you can redistribute it and/or modify it under
the same terms as the Perl 5 programming language system itself.

=cut
