# NAME

Automate::Animate::FFmpeg - Create animation from a sequence of images using ffmpeg

# VERSION

Version 0.01

# SYNOPSIS

This module creates an animation from a sequence of input
images using [https://ffmpeg.org/|FFmpeg](https://ffmpeg.org/|FFmpeg).
An excellent, open source program.

    use Automate::Animate::FFmpeg;
    my $aaFFobj = Automate::Animate::FFmpeg->new({
      # specify input images in any of these 3 ways:
      'input-images' => [
        'im1.png',
        'im2.png',
        ...
      ],
      'input-pattern' => ['*.png', './'],
      'input-images-from-file' => 'file-containing-a-list-of-pathnames-to-images.txt',
      # optionally specify the duration of each frame
      'frame-duration' => 5.3, # seconds
      'output-filename' => 'out.mp4',
    });

    # options can be set after construction as well:

    # optionally add some extra params to FFmpeg as an arrayref
    $aaFF->ffmpeg_extra_params(['-x', 'abc', '-y', 1, 2, 3]);
    # you can also add images here, order is important
    $aaFF->input_images(['img1.png', 'img2.png']) or die;
    # or add images via a search pattern and optional search dir
    $aaFF->input_pattern(['*.png', './']);
    # or add images via multiple search patterns
    $aaFF->input_patterns([
        ['*.png', './'],
        ['*.jpg', '/images'],
        ['*.tiff'], # this defaults to current dir
    ]) or die;

    # and make the animation:
    die "make_animation() has failed"
      unless $aaFF->make_animation()
    ;

# METHODS

## `new`

    my $ret = Automate::Animate::FFmpeg->new({ ... });

All arguments are supplied via a hashref with the following keys:

- `input-images` : an array of pathnames to input images. Image types can be what ffmpeg understands: png, jpeg, tiff, and lots more.
- `input-pattern` : an arrayref of 1 or 2 items. The first item is the pattern
which complies to what [File::Find::Rule](https://metacpan.org/pod/File%3A%3AFind%3A%3ARule) understands (See \[https://metacpan.org/pod/File::Find::Rule#Matching-Rules\]).
For example `*.png`, regular expressions can be passed by enclosing them in `regex(/.../modifiers)`
and should include the `//`. Modifiers can be after the last `/`. For example `regex(/\.(mp3|ogg)$/i)`.
The optional second parameter is the search path. If not specified, the current working dir will be used.
- `input-patterns` : same as above but it expects an array of `input-pattern`.
- `input-images-from-file` : specify the file which contains pathnames to image files, each on its own line.
- `ffmpeg-extra-params` : pass extra parameters to the `ffmpeg` executable as an arrayref of arguments, each argument must be a separate item as in : `['-i', 'file']`.
- `frame-duration` : set the duration of each frame (i.e. each input image) in the animation in (fractional) seconds.
- `ffmpeg-executable` : set the path to the `ffmpeg` executable.
- `qw/verbosity` : set the verbosity, 0 being mute.

Return value:

- `undef` on failure or the blessed object on success.

This is the constructor. It instantiates the object which does the animations. Its
input parameters can be set also via their own setter methods.
If input images are specified during construction then the list
of filenames is constructed and kept in memory. Just the filenames.

## `make_animation()`

    $aaFF->make_animation() or die "failed";

It initiates the making of the animation by shelling out to `ffmpeg`
with all the input images specified via one or more calls to any of:

- input\_images($m)
- input\_pattern($m)
- input\_patterns($m)
- input\_file\_with\_images($m)

On success, the resultant animation will be
written to the output file (specified using ["output\_filename($m) before the call"](#output_filename-m-before-the-call)

It returns 0 on failure, 1 on success.

## `input_images($m)`

    my $ret = $aaFF->input_images($m);

Sets or gets the list (as an ARRAYref) of all input images currently in the list
of images to make up the animation. The optional input parameter, `$m`,
is an ARRAYref of input images (their fullpath that is) to make up
the animation.

## `input_pattern($m)`

    $aaFF->input_pattern($m) or die "failed";

Initiates a search via [File::Find::Rule](https://metacpan.org/pod/File%3A%3AFind%3A%3ARule) for the
input image files to make up the animation using
the pattern `$m-`\[0\]> with starting search dir being `$m-`\[1\]>,
which is optional -- default being `Cwd::cwd` (current working dir).

The pattern (`$m-`\[0\]>) can be a shell wildcard, e.g. `*.png`,
or a regex specified as `regex(/REGEX-HERE/modifiers)`, for example
`regex(/\.(mp3|ogg)$/i)` Both shell wildcards and regular expressions
must comply with what [File::Find::Rule](https://metacpan.org/pod/File%3A%3AFind%3A%3ARule) expects, see \[https://metacpan.org/pod/File::Find::Rule#Matching-Rules\].

The results of the search will be added to the list of input images
in the order of appearance.

Multiple calls to `input_pattern()` will load
input images in the order they are found.

`input_pattern()` can be combined with `input_patterns()`
and `input_images()`. The input images list will increase
in the order they are called.

It returns 1 on success or 0 on failure.

## `input_patterns($m)`

    $aaFF->input_patterns($m) or die "failed";

Argument `$m` is an array of arrays each composed of one or two items.
The first argument, which is mandatory, is the search pattern.
The optional second argument is the directory to start the search.
For each item of `@$m` it calls [input\_pattern($m)](https://metacpan.org/pod/input_pattern%28%24m%29).

`input_patterns()` can be combined with `input_pattern()`
and `input_images()`. The input images list will increase
in the order they are called.

It returns 1 on success or 0 on failure.

## `output_filename($m)`

    my $ret = $aaFF->output_filename($m);

Sets or gets the output filename of the animation.

## `input_file_with_images($m)`

    $aaFF->input_file_with_images($m) or die "failed";

Reads file `$m` which must contain filenames, one filename
per line, and adds the up to the list of input images to make up the
animation.

It returns 0 on failure, 1 on success.

## `num_input_images()`

    my $N = $aaFF->num_input_images();

It returns the number of input images currently
in the list to make up the animation.

## `ffmpeg_executable()`

    my $ret = $aaFF->ffmpeg_executable();

It returns the path to `ffmpeg` executable as it was set during construction.
You can not change the path to the executable mid-stream. Set it via
the constructor or rely on the installation to detect it
and hardcode it to the module file.

## `verbosity($m)`

    my $ret = $aaFF->verbosity($m);

It sets or gets the verbosity level. Zero being mute.

## `frame_duration($m)`

    my $ret = $aaFF->frame_duration($m);

It sets or gets the frame duration in (fractional) seconds.
Frame duration is the time that each frame(=image) appears
in the produced animation.

# AUTHOR

Andreas Hadjiprocopis, `<bliako at cpan.org>`

# BUGS

Please report any bugs or feature requests to `bug-automate-animate-ffmpeg at rt.cpan.org`, or through
the web interface at [https://rt.cpan.org/NoAuth/ReportBug.html?Queue=Automate-Animate-FFmpeg](https://rt.cpan.org/NoAuth/ReportBug.html?Queue=Automate-Animate-FFmpeg).  I will be notified, and then you'll
automatically be notified of progress on your bug as I make changes.

# SUPPORT

You can find documentation for this module with the perldoc command.

    perldoc Automate::Animate::FFmpeg

You can also look for information at:

- RT: CPAN's request tracker (report bugs here)

    [https://rt.cpan.org/NoAuth/Bugs.html?Dist=Automate-Animate](https://rt.cpan.org/NoAuth/Bugs.html?Dist=Automate-Animate)

- AnnoCPAN: Annotated CPAN documentation

    [http://annocpan.org/dist/Automate-Animate](http://annocpan.org/dist/Automate-Animate)

- CPAN Ratings

    [https://cpanratings.perl.org/d/Automate-Animate](https://cpanratings.perl.org/d/Automate-Animate)

- Search CPAN

    [https://metacpan.org/release/Automate-Animate](https://metacpan.org/release/Automate-Animate)

# ACKNOWLEDGEMENTS

# LICENSE AND COPYRIGHT

Copyright 2019 Andreas Hadjiprocopis.

This program is free software; you can redistribute it and/or modify it
under the terms of the the Artistic License (2.0). You may obtain a
copy of the full license at:

[http://www.perlfoundation.org/artistic\_license\_2\_0](http://www.perlfoundation.org/artistic_license_2_0)

Any use, modification, and distribution of the Standard or Modified
Versions is governed by this Artistic License. By using, modifying or
distributing the Package, you accept this license. Do not use, modify,
or distribute the Package, if you do not accept this license.

If your Modified Version has been derived from a Modified Version made
by someone other than you, you are nevertheless required to ensure that
your Modified Version complies with the requirements of this license.

This license does not grant you the right to use any trademark, service
mark, tradename, or logo of the Copyright Holder.

This license includes the non-exclusive, worldwide, free-of-charge
patent license to make, have made, use, offer to sell, sell, import and
otherwise transfer the Package with respect to any patent claims
licensable by the Copyright Holder that are necessarily infringed by the
Package. If you institute patent litigation (including a cross-claim or
counterclaim) against any party alleging that the Package constitutes
direct or contributory patent infringement, then this Artistic License
to you shall terminate on the date that such litigation is filed.

Disclaimer of Warranty: THE PACKAGE IS PROVIDED BY THE COPYRIGHT HOLDER
AND CONTRIBUTORS "AS IS' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES.
THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
PURPOSE, OR NON-INFRINGEMENT ARE DISCLAIMED TO THE EXTENT PERMITTED BY
YOUR LOCAL LAW. UNLESS REQUIRED BY LAW, NO COPYRIGHT HOLDER OR
CONTRIBUTOR WILL BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, OR
CONSEQUENTIAL DAMAGES ARISING IN ANY WAY OUT OF THE USE OF THE PACKAGE,
EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
