package Object::Relation::Exceptions;

# $Id: Exceptions.pm 3076 2006-07-28 17:20:08Z theory $

use strict;
use version;

our $VERSION = version->new('0.1.0');

=head1 Name

Object::Relation::Exceptions - Object::Relation exception object definitions

=head1 Synopsis

  use Object::Relation::Exceptions ':all';
  throw_fatal 'Whoops!';

  # The error is fully localizable.
  throw_fatal ['Unknown value "[_1]"', 'foo'];

=head1 Description

This class defines Object::Relation exception objects. It subclasses Exception::Class,
which provides a robust exception implementation. It extends Exception::Class
by requiring localizable error messages. All error messages must be represented
in the appropriate Object::Relation::Language lexicons.

There currently four major classes of exceptions:

=over

=item Object::Relation::Exception::Fatal

This class and its subclasses represent fatal, non-recoverable errors.
Exceptions of this sort are unexpected, and should be reported to an
administrator or to the Object::Relation developers.

=item Object::Relation::Exception::Error

This class and its subclasses represent non-fatal errors triggered by invalid
data. These can be used to let users know that the data they've entered is
invalid.

=item Object::Relation::Exception::ExternalLib

This class will never has its error message localized. This is so that it can
be thrown by libraries not under direct Object::Relation control and therefore are not
localizable. This class is also used for the global C<$SIG{__DIE__}> handler,
so that exceptions always include a nicely formatted stack trace.

=item Object::Relation::Exception::DBI

This class is used solely for handling exceptions thrown by L<DBI|DBI>. You
should never need to access it directly.

=back

=cut

##############################################################################

=head1 Exception Classes

The exception classes generated by this module are as follows:

=over 4

=item Object::Relation::Exception

Object::Relation exception base class. It generally should not be thrown, only its
subclasses.

=cut

use Exception::Class(
    'Object::Relation::Exception' => {
        description => 'Object::Relation Exception',
    }
);

##############################################################################
# Fatal exceptions.
##############################################################################

=item Object::Relation::Exception::Fatal

Base class for fatal exceptions. Alias: C<throw_fatal>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal' => {
        description => 'Object::Relation fatal exception',
        isa         => 'Object::Relation::Exception',
        alias       => 'throw_fatal',
    },
);

=item Object::Relation::Exception::Fatal::Invalid

Invalid data exception. Thrown when an invalid value is assigned to a Object::Relation
class attribute. Alias: C<throw_invalid>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Invalid' => {
        description => 'Invalid data',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_invalid',
    },
);

=item Object::Relation::Exception::Fatal::ReadOnly

ReadOnly exception. Thrown when an someone attempts to assign a value to a
read-only attribute. Alias: C<throw_read_only>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::ReadOnly' => {
        description => 'Assignment to read-only value',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_read_only',
    },
);

=item Object::Relation::Exception::Fatal::Language

Localization exception. Thrown if an error is encountered while attempting to
localize a string. Alias: C<throw_lang>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Language' => {
        description => 'Localization exception',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_lang',
    },
);

=item Object::Relation::Exception::Fatal::Stat

File status exception. Thrown if an error is encountered while attempting to
stat a file. Alias: C<throw_stat>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Stat' => {
        description => 'File status exception',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_stat',
    },
);

=item Object::Relation::Exception::Fatal::IO

IO exception. Thrown if an error is encountered while attempting to read or
write to a file. Alias: C<throw_io>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::IO' => {
        description => 'File IO exception',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_io',
    },
);

=item Object::Relation::Exception::Fatal::Unimplemented

Unimplemented exception. Thrown if a feature has not yet been implemented or
must be overridden in a subclass. Alias: C<throw_unimplemented>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Unimplemented' => {
        description => 'Unimplemented exception',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_unimplemented',
    },
);

=item Object::Relation::Exception::Fatal::RequiredArguments

Missing arguments exception. Thrown if required arguments to a method are not
present.  Alias: C<throw_required>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::RequiredArguments' => {
        description => 'Missing arguments to method call',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_required',
    },
);

=item Object::Relation::Exception::Fatal::NotFound

Object Not Found exception. Thrown if something searched for is not found.
Usually this is for objects in data stores, but also might be used for
anything that is being looked for (such as a file). Alias: C<throw_not_found>.


Alias: C<throw_not_found>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::NotFound' => {
        description => 'Object not found exception',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_not_found',
    },
);

=item Object::Relation::Exception::Fatal::Unsupported

Unsupported feature exception. Thrown if code attempts to use an unsupported
feature. Alias: C<throw_unsupported>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Unsupported' => {
        description => 'Unsupported feature requested.',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_unsupported',
    },
);

=item Object::Relation::Exception::Fatal::XML

XML parse exception. Thrown if a problem is found parsing XML.
Alias: C<throw_xml>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::XML' => {
        description => 'Error parsing XML',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_xml',
    },
);

=item Object::Relation::Exception::Fatal::InvalidClass

Invalid class exception. Thrown if an incorrect class is encountered.
Alias: C<throw_invalid_class>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::InvalidClass' => {
        description => 'Invalid class',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_invalid_class',
    },
);

=item Object::Relation::Exception::Fatal::InvalidAttribute

Invalid attribute exception. Thrown if an incorrect attribute is encountered.
Alias: C<throw_invalid_attr>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::InvalidAttribute' => {
        description => 'Invalid attribute',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_invalid_attr',
    },
);

=item Object::Relation::Exception::Fatal::UnknownClass

Unknown class exception. Thrown if an unknown class is encountered.
Alias: C<throw_unknown_class>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::UnknownClass' => {
        description => 'Invalid class',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_unknown_class',
    },
);

=item Object::Relation::Exception::Fatal::Attribute

Object::Relation attribute exception. Thrown if a Object::Relation attribute is used
incorrectly, such as "unknown attributes" or trying to use a non-unique
attribute where a unique attribute would be used. Alias: C<throw_attribute>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Attribute' => {
        description => 'Improper use of attribute',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_attribute',
    },
);

=item Object::Relation::Exception::Fatal::Search

Store search exception. Thrown if data store cannot figure out how to respond
to a search request. Alias: C<throw_search>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Search' => {
        description => 'Bad search request',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_search',
    },
);

=item Object::Relation::Exception::Fatal::Setup

Setup exception. Thrown if there is an error when setting up a data store.
Alias: C<throw_setup>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Setup' => {
        description => 'Setup error',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'throw_setup',
    },
);

=item Object::Relation::Exception::Fatal::Panic

Panic exception. This internal exception should (theoretically) never be
thrown.  It only occurs if the internal state of something has reached a point
that cannot occur.  For example, an "ANY" search in a store class can only
occur if the values searched on are contained in an array reference.  If an ANY
search occurs and the values are not an array reference, a panic is thrown.
Alias: C<panic>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Fatal::Panic' => {
        description => 'Invalid class',
        isa         => 'Object::Relation::Exception::Fatal',
        alias       => 'panic',
    },
);

##############################################################################
# Non-fatal errors.
##############################################################################

=item Object::Relation::Exception::Error

Base class for error exceptions. These are non-fatal errors, generally
triggered by problems with data entered by users. Alias: C<throw_error>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Error' => {
        description => 'Object::Relation error',
        isa         => 'Object::Relation::Exception',
        alias       => 'throw_error',
    },
);

=item Object::Relation::Exception::Error::Auth

Authentication failure. Thrown when authentication fails. Alias:
C<throw_auth>.

=cut

use Exception::Class(
    'Object::Relation::Exception::Error::Auth' => {
        description => 'Object::Relation authentication error',
        isa         => 'Object::Relation::Exception::Error',
        alias       => 'throw_auth',
    },
);

##############################################################################
# Unlocalized exceptions.
##############################################################################

=item Object::Relation::Exception::ExternalLib

Class for exceptions thrown external to Object::Relation code. Exceptions of this class
behave just like any other Object::Relation exception objects, except that error
messages are not localized. Alias: C<throw_exlib>.

=cut

use Exception::Class(
    'Object::Relation::Exception::ExternalLib' => {
        description => 'External library exception',
        isa         => 'Object::Relation::Exception',
        alias       => 'throw_exlib',
    },
);

##############################################################################

use Exporter::Tidy all => [qw(
    panic isa_obj_rel_exception isa_exception throw_exlib throw_fatal
    throw_invalid throw_read_only throw_lang throw_stat throw_io throw_error
    throw_auth throw_required throw_xml throw_unknown_class throw_setup
    throw_invalid_class throw_invalid_attr throw_not_found throw_unsupported
    throw_unimplemented throw_search throw_attribute sig_handlers
)];

##############################################################################

=back

=cut

=head1 Interface

=head2 Functions

In addition to the functions that can be imported to throw the above
exceptions, there are a few other functions that may be imported into a client
class.

=head3 isa_obj_rel_exception

  if (isa_obj_rel_exception($@))
      print 'A Object::Relation exception was thrown';
      print "...and it was fatal!' if isa_obj_rel_exception($@, 'Fatal');
  }

This function returns true if the argument passed to it is a Object::Relation exception.
The optional second argument can be used to test for a specific Object::Relation
exception. If no such exception exists, an exception will be thrown.

=cut

sub isa_obj_rel_exception {
    my ($err, $name) = @_;
    return unless $err;

    my $class = "Object::Relation::Exception";
    if ($name) {
        $class .= "::$name";
        throw_fatal qq{No such exception class "$class"}
          unless isa_exception($class);
    }
    return UNIVERSAL::isa($err, $class);
}

##############################################################################

=head3 isa_exception

  if (isa_exception($@)) {
      print "What we have here...is an exception object";
  }

This function returns true if the argument passed to it is an Exception::Class
object.

=cut

sub isa_exception {
    my $err = shift;
    return $err && UNIVERSAL::isa($err, 'Exception::Class::Base');
}

##############################################################################

=head3 sig_handlers

  sig_handlers(0);

This function accepts a boolean value. If true, it turns on stack traces via
the DIE signal handler. If false, it disables them. If called without
arguments, it merely returns a boolean value indicating whether or not the
signal handlers are enabled.

=cut

my $SIG_DIE = sub {
    my $err = shift;
    $err->rethrow if UNIVERSAL::can($err, 'rethrow');
    Object::Relation::Exception::ExternalLib->throw($err);
};

my $HANDLERS = 1;
sub sig_handlers {
    _set_handlers(shift) if @_;
    return $HANDLERS;
}

sub _set_handlers {
    $HANDLERS = shift;
    if ($HANDLERS) {
        $SIG{__DIE__} = $SIG_DIE;
    }
    else {
        $SIG{__DIE__} = undef;
    }
}

# Always use exception objects for exceptions.
sig_handlers(1);
##############################################################################
# From here on in, we're modifying the behavior of Exception::Class::Base.

package Object::Relation::Exception;
use aliased 'Object::Relation::Language';

=head2 Constructors

=head3 new

  my $err = Object::Relation::Exception->new("Whoops!");
  my $err = Object::Relation::Exception->new(error => 'Whoops!');
  my $err = Object::Relation::Exception->new(['Unknown value "[_1]"', 'foo']);
  my $err = Object::Relation::Exception->new(
      error => ['Unknown value "[_1]"', 'foo']
  );

Creates and returns a new exception object. Use this method with C<die> to
throw exceptions yourself, or if you don't want to import any C<throw_>
functions into your namespace. Otherwise, a C<throw_> function is generally the
preferred way to throw an exception. Besides, it requires less typing!

The base class supports only a single parameter, C<error>, for the exception
error message. If only a single argument is passed to C<new()>, it is assumed
to be the C<error> parameter. Other exception classes may support other
parameters; consult their <descriptions|"Exception Classes"> for details. All
C<throw_> functions for the exception subclasses support the parameters that
correspond to their respective classes.

=cut

sub new {
    my $class = shift;
    my %p =  @_ == 1 ? ( error => $_[0] ) : @_;

    # Localize the error message.
    $p{error} = Language->get_handle->maketext(
        ref $p{error} ? @{$p{error}} : $p{error}
    ) unless $class->isa('Object::Relation::Exception::ExternalLib');

    $class->SUPER::new(%p);
}

##############################################################################

=head2 Instance Methods

=head3 as_string

  my $string $err->as_string;

Returns a stringified representation of the exception, which includes the full
error message and the stack trace. The method overrides stringification
(double-quoted string context), so its return value is output whenever an
exception object is printed.

We override the base class version of this method in order to provide a nicer,
more legible stack trace, rather than the default Carp-style trace.

=cut

sub as_string {
    my $self = shift;
    return sprintf("%s\n%s\n", $self->full_message, $self->trace_as_text);
}

##############################################################################

=head3 trace_as_text

  my $trace = $err->trace_as_text;

Returns a stringified representation of the stack trace for the exception
object. Used by C<as_string()> to pretty-print the stack trace.

=cut

sub trace_as_text {
    my $self = shift;
    return join "\n", map {
        sprintf("[%s:%d]", $_->filename, $_->line);
    } $self->_filtered_frames;
}

##############################################################################

=begin private

=head2 Private Instance Methods

=head3 _filtered_frames

  my @trace_frames = $err->_filtered_frames;

Returns a list of stack trace frames, filtering out those that derive from
Exception::Class::Base, Object::Relation::Exception::__ANON__ (the C<throw_>
fucnctions), and C<(eval)>. Used by C<trace_as_text()>.

=cut

sub _filtered_frames {
    my $self = shift;
    my %ignore_subs = map { $_ => 1 } qw{
         (eval)
         Exception::Class::Base::throw
         Object::Relation::Exception::__ANON__
    };

    my $faultregex = qr{/Object/Relation/Util/Exception\.pm|/dev/null};
    my @frames;
    my $trace = $self->trace;
    while (my $frame = $trace->next_frame) {
        push @frames, $frame
          unless $frame->filename =~ $faultregex
          || $ignore_subs{$frame->subroutine};
    }

    return @frames
        ? @frames
        : grep { $_->filename !~ $faultregex } $trace->frames;
}

# Make Exception::Class::DBI inherit from this class, too.
package Object::Relation::Exception::DBI;
use base qw(Object::Relation::Exception::ExternalLib Exception::Class::DBI);

sub full_message {
    my $self = shift;
    return $self->SUPER::full_message unless $self->can('statement');
    return $self->SUPER::full_message
        . ' [for Statement "' . $self->statement . '"]';
}

# Make sure that fields from all parent classes are recognized.
sub Fields {
    return (
        map  { $_->Fields }
        grep { $_->isa('Exception::Class::Base') }
        @Object::Relation::Exception::DBI::ISA
    );
}

package Object::Relation::Exception::DBI::H;
use base qw(Object::Relation::Exception::DBI Exception::Class::DBI::H);

package Object::Relation::Exception::DBI::DRH;
use base qw(Object::Relation::Exception::DBI Exception::Class::DBI::DRH);

package Object::Relation::Exception::DBI::DBH;
use base qw(Object::Relation::Exception::DBI Exception::Class::DBI::DBH);

package Object::Relation::Exception::DBI::STH;
use base qw(Object::Relation::Exception::DBI Exception::Class::DBI::STH);

package Object::Relation::Exception::DBI::Unknown;
use base qw(Object::Relation::Exception::DBI Exception::Class::DBI::Unknown);

1;
__END__

##############################################################################

=end private

=head1 Copyright and License

Copyright (c) 2004-2006 Kineticode, Inc. <info@obj_relode.com>

This module is free software; you can redistribute it and/or modify it under the
same terms as Perl itself.

=cut
