#
# Copyright (c) 1993-1994 Regents of the University of California.
# All rights reserved.
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions
# are met:
# 1. Redistributions of source code must retain the above copyright
#    notice, this list of conditions and the following disclaimer.
# 2. Redistributions in binary form must reproduce the above copyright
#    notice, this list of conditions and the following disclaimer in the
#    documentation and/or other materials provided with the distribution.
# 3. All advertising materials mentioning features or use of this software
#    must display the following acknowledgement:
#	This product includes software developed by the Computer Systems
#	Engineering Group at Lawrence Berkeley Laboratory.
# 4. Neither the name of the University nor of the Laboratory may be used
#    to endorse or promote products derived from this software without
#    specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
# ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
# IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
# ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
# FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
# DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
# OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
# HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
# LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
# OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
# SUCH DAMAGE.
#
# @(#) $Header: vat.tcl,v 1.25 94/06/22 00:30:11 van Exp $ (LBL)
#
# NOTE: This file is provided for reference purposes.  The code
#       shown here is compiled into vat.  DON'T use this as your
#	vat startup file (~/.vat.tcl).
#

set tcl_precision 17

#
# Turn entry into mini-emacs editor.
#
bind Entry <Control-a> {
	%W icursor 0
	tk_entrySeeCaret %W
}

bind Entry <Control-d> { 
	if { [catch { %W delete sel.first sel.last; tk_entrySeeCaret %W }] } {
		set x [%W index insert]
		%W delete $x $x
		tk_entrySeeCaret %W
	}
}
bind Entry <Control-e> {
	%W icursor end
	tk_entrySeeCaret %W
}
bind Entry <Control-f> { 
	set x [expr [%W index insert] + 1]
	if { $x <= [%W index end] } { 
		%W icursor $x
		tk_entrySeeCaret %W
	}
}
bind Entry <Control-b> { 
	set x [expr [%W index insert] - 1]
	if { $x >= 0 } {
		%W icursor $x 
		tk_entrySeeCaret %W
	}
}
bind Entry <Control-k> { %W delete insert end }
bind Entry <Any-KeyPress> {
	if { %s > 2 } {
		# ignore ctrl-meta combos
		return
	}
	if {"%A" != ""} {
		%W insert insert %A
		tk_entrySeeCaret %W
	}
}
bind Entry <1> {
	%W icursor @%x
	%W select from @%x
	if { [lindex [%W config -state] 4] == "normal" } {
		focus %W
		global entryTab
		set entryTab(%W:focus) 1
	}
}
#
# have middle button paste-in current x selection
#
bind Entry <2> {
	if { [lindex [%W config -state] 4] == "normal" } { 
		if { [catch {selection get} text] == 0 } {
			%W icursor @%x
			%W insert insert $text
			tk_entrySeeCaret %W
			focus %W
			global entryTab
			set entryTab(%W:focus) 1
		}
	}
}
bind Entry <Enter> {
	catch { 
		global entryTab
		if { $entryTab(%W:focus) } {
			focus %W
		}
	}
}
bind Entry <Return> {
	focus .
	%W select clear
	catch {
		global entryTab
		set entryTab(%W:focus) 0
		set v [%W get]
		if { [$entryTab(%W:action) %W $v] } {
			%W delete 0 end
			%W insert 0 $entryTab(%W:value)
		} else {
			set entryTab(%W:value) $v
		}
	}
}
bind Entry <Escape> {
	focus .
	%W select clear
	%W delete 0 end
	catch {
		global entryTab
		set entryTab(%W:focus) 0
		%W insert 0 $entryTab(%W:value)
	}
}
bind Entry <Control-g> [bind Entry <Escape>]

set helv10 "-*-helvetica-medium-r-normal--*-100-75-75-*-*-*-*"
set helv10b "-*-helvetica-bold-r-normal--*-100-75-75-*-*-*-*"
set helv10o "-*-helvetica-bold-o-normal--*-100-75-75-*-*-*-*"
set helv12 "-*-helvetica-medium-r-normal--*-120-75-75-*-*-*-*"
set helv12b "-*-helvetica-bold-r-normal--*-120-75-75-*-*-*-*"
set helv14 "-*-helvetica-medium-r-normal--*-140-75-75-*-*-*-*"
set helv14b "-*-helvetica-bold-r-normal--*-140-75-75-*-*-*-*"
set times14 "-*-times-medium-r-normal--*-140-75-75-*-*-*-*"

# globals
option add *background gray80 startupFile
option add *foreground black startupFile
option add *activeBackground gray95 startupFile
option add *selectBackground gray95 startupFile

option add *Font $helv14b startupFile
option add *Checkbutton.anchor w startupFile
option add *Radiobutton.anchor w startupFile
option add *Radiobutton.relief flat startupFile
option add *Radiobutton.font $helv12b 100
option add *Scale.sliderForeground gray66 startupFile
option add *Scale.activeForeground gray80 startupFile
option add *Scale.background gray70 startupFile

# vat widgets
option add *Vatname.font $helv12b startupFile
option add *Vatname.foreground black startupFile
option add *Vatname.disabled gray50 startupFile
option add *VatVU.foreground black startupFile
option add *VatVU.peak gray50 startupFile
option add *VatVU.hot firebrick1 startupFile
option add *VatVU.hotLevel 90 startupFile

if { [tk colormodel .] == "monochrome" } {
	# make mono look better
	option add *selectBackground black startupFile
	option add *selectForeground white startupFile
	option add *activeForeground black startupFile
	option add *Vatname.background white startupFile
	option add *VatVU.background white startupFile
	option add *VatVU.hot gray50 startupFile
}

option add Vat.iconPrefix "" startupFile
option add Vat.audioFont $helv10 startupFile
option add Vat.helpFont $times14 startupFile
option add Vat.ctrlTitleFont $helv12b startupFile
option add Vat.ctrlFont $helv10b startupFile
option add Vat.entryFont $helv10 startupFile
option add Vat.noAudioFont $helv10o startupFile

# colors
option add Vat.titleReleased "gray95" startupFile
option add Vat.titleHave "#aaaaaa" startupFile
option add Vat.titlePinned "black" startupFile

option add Vat.mikeGain "32" startupFile
option add Vat.lineinGain "180" startupFile
option add Vat.linein2Gain "180" startupFile
option add Vat.linein3Gain "180" startupFile
option add Vat.speakerGain "180" startupFile
option add Vat.jackGain "180" startupFile
option add Vat.lineoutGain "180" startupFile
option add Vat.lineout2Gain "180" startupFile

option add Vat.speakerMute "false" startupFile
option add Vat.mikeMute "true" startupFile

option add Vat.speakerMode "NetMutesMike" startupFile
option add Vat.jackMode "FullDuplex" startupFile
option add Vat.lineoutMode "NetMutesMike" startupFile
option add Vat.lineout2Mode "NetMutesMike" startupFile
option add Vat.maxPlayout "6" startupFile
option add Vat.vtMode "false" startupFile
option add Vat.lectureMode "false" startupFile
option add Vat.useNames "false" startupFile
option add Vat.defaultHost "224.2.0.1" startupFile
option add Vat.defaultVTHost "224.1.0.200" startupFile
option add Vat.defaultPort "3456" startupFile
option add Vat.defaultTTL "16" startupFile

option add Vat.filterLength "256" startupFile
option add Vat.filterMaxTaps "35" startupFile

option add Vat.meterDisable "false" startupFile
option add Vat.meterStyle "discrete" startupFile
option add Vat.inputPort "Mike" startupFile
option add Vat.outputPort "Speaker" startupFile
option add Vat.audioFormat "pcm2" startupFile
option add Vat.mikeAGC "false" startupFile
option add Vat.mikeAGCLevel "0" startupFile
option add Vat.speakerAGC "false" startupFile
option add Vat.speakerAGCLevel "0" startupFile

option add Vat.defaultPriority "100" startupFile
option add Vat.idleHoldTime "10" startupFile
option add Vat.idleDropTime "20" startupFile
option add Vat.autoRaise "true" startupFile
option add Vat.externalEchoCancel "false" startupFile
option add Vat.echoSuppressTime "120" startupFile
option add Vat.keepSites "false" startupFile
option add Vat.muteNewSites "false" startupFile
option add Vat.compressionSlope "0.0625" startupFile
option add Vat.key "" startupFile
option add Vat.afDevice "-1" startupFile
option add Vat.afBlocks "2" startupFile
option add Vat.afSoftOuputGain "0" startupFile
option add Vat.afSoftInputGain "0" startupFile
option add Vat.siteDropTime "30" startupFile
option add Vat.audioFileName "/dev/audio" startupFile
option add Vat.ipcChannel "0" startupFile
option add Vat.statTimeConst 0.1 startupFile

set title "LBL Visual Audio Tool v[version]"

proc mk.meter { w c } {
	frame $w.$c -borderwidth 1 -relief flat -background black
	meter $w.$c.meter
	pack $w.$c.meter -expand 1 -fill y
}

proc mk.tests { w } {
	label $w.label -text "Audio Tests" -font [ctitlefont]
	frame $w.frame -borderwidth 2 -relief sunken
	frame $w.frame.p1
	frame $w.frame.p2

	set f [ctrlfont]
	set p $w.frame.p1
	radiobutton $p.none -text none -relief flat -command "audio test 0" \
		-anchor w -variable audioTest -font $f
	$p.none select
	radiobutton $p.loop -text "loopback" -relief flat \
		-command "audio test 1" \
		-anchor w -variable audioTest -font $f
	pack $p.none $p.loop -fill x

	set p $w.frame.p2
	radiobutton $p.t6 -text "-6dBm tone" -relief flat \
		-command "audio test 2" -anchor w -variable audioTest -font $f
	radiobutton $p.t0 -text "0dBm tone" -relief flat \
		-command "audio test 3" -anchor w -variable audioTest -font $f
	radiobutton $p.tmax -text "max tone" -relief flat \
		-command "audio test 4" -anchor w -variable audioTest -font $f
	pack $p.t6 $p.t0 $p.tmax -expand 1 -fill x

	pack $w.frame.p1 -side left -anchor center
	pack $w.frame.p2 -side left -expand 1 -fill both

	pack $w.label -fill x
	pack $w.frame -fill both -expand 1
}

proc pri_accept { w pri } {
	global audioPri
	if { $audioPri == 0 } {
		aa priority $pri
	}
	return 0
}

proc mk.pri { w } {
	label $w.label -text "Priority" -font [ctitlefont]
	frame $w.frame -borderwidth 2 -relief sunken
	set f [ctrlfont]
	set p $w.frame.inset
	frame $p -borderwidth 0
	pack $p -anchor c
	radiobutton $p.high -text "high (200)" -relief flat -value 200 \
		-command "aa priority 200" -variable audioPri -font $f
	radiobutton $p.med -text "med (100)" -relief flat -value 100 \
		-command "aa priority 100" -variable audioPri -font $f
	radiobutton $p.low -text "low (10)" -relief flat -value 10 \
		-command "aa priority 10" -variable audioPri -font $f
	frame $p.f
	radiobutton $p.f.rb -text "" -relief flat -value 0 \
		-command "aa priority \[$p.f.entry get\]" \
		-variable audioPri -font $f
	entry $p.f.entry -relief raised -borderwidth 1 -font $f -width 8 \
		-exportselection 0

	global entryTab
	set entryTab($p.f.entry:action) pri_accept

	pack $p.f.rb $p.f.entry -side left

	set pri [option get . defaultPriority Vat]
	aa priority $pri
	if { $pri == 10 } {
		$p.low select
	} elseif { $pri == 100 } {
		$p.med select
	} elseif { $pri == 200 } {
		$p.high select
	} else {
		$p.f.rb select
	}
	$p.f.entry insert 0 $pri
	set entryTab($p.f.entry:value) $pri

	pack $p.high $p.med $p.low $p.f -expand 1 -fill x
	pack $w.label $w.frame -expand 1 -fill x
}

proc mk.oradio { w } {
	set f [ctrlfont]
	set labels [audio output names]
	set i 0
	set n [llength $labels]
	while { $i < $n } {
		set p $w.p$i
		frame $p
		set label [lindex $labels $i]
		set port [string tolower $label]
		global omode$i
		if { $label == "Speaker" } { set label "Spkr" }
		label $p.label -text $label -font $f
		radiobutton $p.mmn -text "" -relief flat -value MikeMutesNet \
			-command "audio mode $port mikemutesnet" \
			-variable omode$i -font $f
		radiobutton $p.nmm -text "" -relief flat -value NetMutesMike \
			-command "audio mode $port netmutesmike" \
			-variable omode$i -font $f
		radiobutton $p.fd -text "" -relief flat -value FullDuplex \
			-command "audio mode $port fullduplex" \
			-variable omode$i -font $f
		pack $p.label $p.mmn $p.nmm $p.fd
		if { [yesno externalEchoCancel] } {
			radiobutton $p.ec -text "" -relief flat \
			-value EchoCancel \
			-command "audio mode $port echocancel" \
			-variable omode$i -font $f
			pack $p.ec
		}
		pack $p -side left
		set omode$i [option get . $port\Mode Vat]
		eval "audio mode $port \$omode$i"

		incr i
	}
	frame $w.label
	label $w.label.blank -text "" -font $f
	label $w.label.mmn -text "Mike mutes net" -font $f
	label $w.label.nmm -text "Net mutes mike" -font $f
	label $w.label.fd -text "Full duplex" -font $f
	pack $w.label.blank $w.label.mmn $w.label.nmm $w.label.fd -anchor w
	if { [yesno externalEchoCancel] } {
		label $w.label.ec -text "Ext. Echo Cancel" -font $f
		pack $w.label.ec -anchor w
	}
	pack $w.label -side left
}

proc yesno { attr } {
	set v [option get . $attr Vat]
	if { [string match \[0-9\]* $v] } {
		return $v
	}
	if { $v == "true" || $v == "True" || $v == "t" } {
		return 1
	} elseif { $v == "false" || $v == "False" || $v == "f" } {
		return 0
	}
	return 0
}

#
# Initialize some global tcl variables.  Called
# after command line options have been parsed.
#
proc init_globals { } {
	global autoRaise keepSites muteNewSites mikeAGC \
		speakerAGC meterDisable disableSS audioFormat lectureMode \
		gaintab porttab iconPrefix

	set autoRaise [yesno autoRaise]
	set keepSites [yesno keepSites]
	set muteNewSites [yesno muteNewSites]
	set mikeAGC [yesno mikeAGC]
	set speakerAGC [yesno speakerAGC]
	set meterDisable [yesno meterDisable]
	set disableSS 0
	set audioFormat [option get . audioFormat Vat]
	set lectureMode [yesno lectureMode]
	set iconPrefix [option get . iconPrefix Vat]
}

proc mk.obuttons { w } {
	set f [ctrlfont]
	frame $w.p0 -borderwidth 0
	frame $w.p1 -borderwidth 0
	pack $w.p0 $w.p1 -side left -fill x -anchor n

	set p $w.p0
	checkbutton $p.ar -text "Autoraise" -relief flat -font $f \
		-command "aa autoraise \$autoRaise" -variable autoRaise
	global autoRaise
	aa autoraise $autoRaise
	global meterDisable
	checkbutton $p.dm -text "Disable Meters" -relief flat -font $f \
		-command "meter_disable \$meterDisable" \
		-variable meterDisable
	meter_disable $meterDisable

	checkbutton $p.nss -text "No Silence Suppr" -relief flat -font $f \
		-command "ss_disable \$disableSS" -variable disableSS

	pack $p.ar $p.dm $p.nss -expand 1 -fill x

	set p $w.p1
	global sitebox keepSites muteNewSites
	checkbutton $p.kas -text "Keep All Sites" -relief flat -font $f \
		-command "$sitebox keepsites \$keepSites" -variable keepSites
	$sitebox keepsites $keepSites

	checkbutton $p.mns -text "Mute New Sites" -relief flat -font $f \
		-command "$sitebox mutesites \$muteNewSites" \
		-variable muteNewSites
	$sitebox mutesites $muteNewSites

	pack $p.kas $p.mns -expand 1 -fill x
}

proc setAGC { w which level } {
	$w.label configure -text "$level dB"
	agc $which $level
}

proc enableAGC { w which state } {
	if { $state } {
		agc $which [$w.scale get]
		agc $which on
		$w.scale configure -state normal
	} else {
		agc $which off
		$w.scale configure -state disabled
	}
}

proc oneagc { w which label } {
	set f [ctrlfont]
	checkbutton $w.button -text $label -relief flat -font $f \
		-command "enableAGC $w $which \$$which\AGC" \
		-variable $which\AGC
	scale $w.scale -orient horizontal \
			-showvalue 0 \
			-from -10 -to 10 \
			-command "setAGC $w $which" \
			-relief groove -borderwidth 2 -width 10 \
			-state disabled
	label  $w.label -text "0 dB" -width 5 -font $f
	pack $w.button $w.scale $w.label -side left
	pack $w.scale -expand 1 -fill x -pady 3

	global $which\AGCbutton
	set $which\AGCbutton $w.button
}

proc mk.agc { w } {
	label $w.label -text "Automatic Gain Control" -font [ctitlefont]
	frame $w.frame -borderwidth 2 -relief sunken

	frame $w.frame.spkr -borderwidth 0
	frame $w.frame.mike -borderwidth 0

	oneagc $w.frame.spkr output Spkr
	$w.frame.spkr.scale set [option get . speakerAGCLevel Vat]
	oneagc $w.frame.mike input Mike
	$w.frame.mike.scale set [option get . mikeAGCLevel Vat]

	pack $w.frame.spkr $w.frame.mike -fill x
	pack $w.label $w.frame -expand 1 -fill x
	pack $w.frame -padx 6
}

proc mk.omode { w } {
	label $w.label -text "Output Mode" -font [ctitlefont]
	frame $w.frame -borderwidth 2 -relief sunken
	frame $w.frame.radios -borderwidth 0
	frame $w.frame.buttons -borderwidth 0
	mk.oradio $w.frame.radios
	mk.obuttons $w.frame.buttons
	pack $w.frame.radios $w.frame.buttons -anchor c -pady 4
	pack $w.label $w.frame -expand 1 -fill x
}

proc mk.me { w } {
	set f [ctrlfont]
	frame $w.mode -borderwidth 2 -relief sunken
	frame $w.mode.inset -borderwidth 0
	set p $w.mode.inset
	radiobutton $p.conf -text "Conference" \
		-command "connection mode conference" \
		-variable lectureMode -font $f -value 0
	radiobutton $p.lec -text "Lecture" \
		-command "connection mode lecture" \
		-variable lectureMode -font $f -value 1

	pack $p.conf $p.lec -fill x
	pack $p -anchor c
	pack $p -side left -expand 1 -fill x

	frame $w.fmt -borderwidth 2 -relief sunken
	frame $w.fmt.p1
	set p $w.fmt.p1
	radiobutton $p.pcm -text PCM -font $f -value pcm \
		-command { net format $audioFormat } -variable audioFormat
	radiobutton $p.pcm2 -text PCM2 -font $f -value pcm2 \
		-command { net format $audioFormat } -variable audioFormat
	radiobutton $p.pcm4 -text PCM4 -font $f -value pcm4 \
		-command { net format $audioFormat } -variable audioFormat
	pack $p.pcm $p.pcm2 $p.pcm4 -expand 1 -fill x

	frame $w.fmt.p2
	set p $w.fmt.p2
	radiobutton $p.dvi -text DVI -font $f -value dvi \
		-command { net format $audioFormat } -variable audioFormat 
	radiobutton $p.dvi2 -text DVI2 -font $f -value dvi2 \
		-command { net format $audioFormat } -variable audioFormat
	radiobutton $p.dvi4 -text DVI4 -font $f -value dvi4 \
		-command { net format $audioFormat } -variable audioFormat
	pack $p.dvi $p.dvi2 $p.dvi4 -expand 1 -fill x

	frame $w.fmt.p3
	set p $w.fmt.p3
	radiobutton $p.gsm -text GSM -font $f -value gsm \
		-command { net format $audioFormat } -variable audioFormat
	radiobutton $p.lpc4 -text LPC4 -font $f -value lpc4 \
		-command { net format $audioFormat } -variable audioFormat
	pack $p.gsm $p.lpc4 -expand 1 -fill x
	pack $w.fmt.p1 $w.fmt.p2 $w.fmt.p3 -side left
	pack $w.mode -side left -expand 1 -fill both
	pack $w.fmt -side left
	if {[net ttl] > 160} {
		global audioFormat
		$w.fmt.p1.pcm configure -state disabled
		if {$audioFormat  == "pcm"} {
			set audioFormat pcm2
		}
		if {[net ttl] > 192} {
			$w.fmt.p1.pcm2 configure -state disabled
			$w.fmt.p1.pcm4 configure -state disabled
			if {[regexp -nocase pcm $audioFormat]} {
				set audioFormat dvi2
			}
			if {[net ttl] > 200} {
				$w.fmt.p2.dvi configure -state disabled
				$w.fmt.p2.dvi2 configure -state disabled
				$w.fmt.p2.dvi4 configure -state disabled
				if {[regexp -nocase dvi $audioFormat]} {
					set audioFormat gsm
				}
			}
		}
	}
}

proc mk.info { w } {
	label $w.label -font [ctrlfont] -text \
	 "Dest: [net addr]  Port: [net port]  ID: [net confid]  TTL: [net ttl]"
	pack $w.label -expand 1 -fill x
}

proc mk.entry { w action text } {
	entry $w.entry -relief raised -borderwidth 1 -exportselection 1 \
		-font [option get . entryFont Vat]
	global entryTab
	set entryTab($w.entry:action) $action
	set entryTab($w.entry:value) $text
	$w.entry insert 0 $text
}

proc mk.ent { w label text action } {
	frame $w -borderwidth 0
	label $w.label -text $label -font [ctrlfont] -width 6 -anchor e
	mk.entry $w $action $text
	pack $w.label -side left
	pack $w.entry -side left -expand 1 -fill x

	$action $w.entry $text
}

proc setcrypt { w key } {
	global sessionKey doEncryption entryTab
	if { ![net crypt $key] || $key == "" } {
		# illegal encryption string, clear it out
		# and disable encryption
		set sessionKey ""
	        set entryTab($w.entry:value) ""
		$w.button configure -state disabled
		set doEncryption 0
		return -1
	} else {
		set sessionKey $key
		$w.button configure -state normal
		set doEncryption 1
		return 0
	}
}

proc toggleKey { w } {
        global sessionKey doEncryption
	if { $doEncryption } {
		setcrypt $w $sessionKey
	} else {
		net crypt ""
	}
}

proc updateKey { w key } {
	return [setcrypt [winfo parent $w] [string trim $key]]
}

proc updateName { w name } {
	global sessionName
	set name [string trim $name]
	if { $name != ""} {
		set sessionName $name
		global sitebox
		$sitebox name $name
		return 0
	}
	return -1
}

proc updateTitle { w name } {
	global confName
	if { $name != ""} {
		set confName $name
		global iconPrefix
	        wm iconname . $iconPrefix$confName
	        wm title . $confName
		return 0
	}
	return -1
}

proc mk.entries { w } {
	global sessionKey sessionName confName
	set sessionKey [option get . sessionKey Vat]
	set sessionName [option get . sessionName Vat]
	mk.ent $w.name "Name: " $sessionName updateName
	mk.ent $w.title "Title: " $confName updateTitle

	frame $w.key
	checkbutton $w.key.button -text Key: -relief flat -font [ctrlfont] \
		-command "toggleKey $w.key" -variable doEncryption \
		-disabledforeground gray40
	mk.entry $w.key updateKey ""

	global doEncryption
	if { $sessionKey == "" } {
		set doEncryption 0
		$w.key.button configure -state disabled
	} else {
		set doEncryption 1
	}
	pack $w.key.button -side left
	pack $w.key.entry -side left -fill x -expand 1
	pack $w.name $w.title $w.key -expand 1 -fill x
}

proc mk.net { w } {
	label $w.label -text "Network" -font [ctitlefont]
	frame $w.frame -borderwidth 0
	frame $w.frame.me -borderwidth 0
	frame $w.frame.ie -borderwidth 2 -relief sunken
	frame $w.frame.ie.info -borderwidth 0
	frame $w.frame.ie.entries -borderwidth 0

	mk.me $w.frame.me
	mk.info $w.frame.ie.info
	mk.entries $w.frame.ie.entries

	pack $w.label $w.frame -expand 1 -fill x
	pack $w.frame -padx 6

	pack $w.frame.ie.info $w.frame.ie.entries -expand 1 -fill x
	pack $w.frame.me $w.frame.ie -expand 1 -fill x
}

proc mk.menu { } {
	set w .menu
	toplevel $w
	bind $w <Enter> "focus $w"
	wm withdraw $w

	frame $w.tp
	frame $w.tp.tests
	frame $w.tp.pri
	frame $w.omode
	frame $w.agc
	frame $w.net

	mk.tests $w.tp.tests
	mk.pri $w.tp.pri
	mk.omode $w.omode
	mk.agc $w.agc
	mk.net $w.net
	button $w.ok -text " Dismiss " -borderwidth 2 -relief raised \
		-command "toggle_window $w" -font [ctitlefont]
	frame $w.pad -borderwidth 0 -height 6

	pack $w.tp.tests -side left -expand 1 -fill both -padx 2
	pack $w.tp.pri -side left -expand 1 -fill x -padx 2
	pack $w.tp $w.omode $w.agc $w.net -expand 1 -fill x
	pack $w.ok -pady 6 -anchor c
	pack $w.tp -padx 4
	pack $w.omode -padx 6
}

proc toggle_window { w } {
	if { ![winfo exists $w] } { mk$w }
	global mapped$w
	if { ! [info exists mapped$w] } {
		set mapped$w 1
		wm transient $w .
		update idletasks
		set x [winfo rootx .]
		set y [winfo rooty .]
		incr y [winfo height .]
		incr y -[winfo reqheight $w]
		incr y -20
 		# adjust for virtual desktops
		incr x [winfo vrootx .]
		incr y [winfo vrooty .]
		if { $y < 0 } { set y 0 }
		if { $x < 0 } {
			set x 0
		} else {
			set right [expr [winfo screenwidth .] - \
					[winfo reqwidth $w]]
			if { $x > $right } {
				set x $right
			}
		}
		wm geometry $w +$x+$y
		wm deiconify $w
	} elseif { [winfo ismapped $w] } {
		wm withdraw $w
	} else {
		wm deiconify $w
	}
}

proc mute_button_mute { w which } {
	global mute_state
	set mute_state($w) 1
	audio $which mute
	$w configure -foreground gray95 \
		-activeforeground gray \
		-disabledforeground gray95 \
		-background black -activebackground black \
		-command "mute_button_unmute $w $which"
}

proc mute_button_unmute { w which } {
	global mute_state
	set mute_state($w) 0
	audio $which unmute
	set fg [option get . foreground Button]
	$w configure \
		-foreground $fg -activeforeground $fg \
		-disabledforeground $fg \
		-background [option get $w background Button] \
		-activebackground [option get $w activeBackground Button] \
		-command "mute_button_mute $w $which"
}

proc lookup_bitmap { name } {
	switch -glob $name {
		Mike { return mike }
		Speaker { return speaker }
		Jack { return headphone }
		Lineout* { return lineout }
		default { return linein }
	}
}

proc setPort { which button scale port } {
	audio $which set $port
	$button configure -bitmap [lookup_bitmap $port]
	global gaintab porttab
	set porttab($which) $port
	$scale set $gaintab($port)
	# XXX the audio class remembers the gain levels but
	# that doesn't work the first time we select a port
	audio $which gain $gaintab($port)
}

proc changePort { which button scale } {
	set ports [audio $which names]
	set n [audio $which current]
	incr n
	if { $n >= [llength $ports] } {
		set n 0
	}
	setPort $which $button $scale [lindex $ports $n]
}

proc setgain { which level } {
	audio $which gain $level
	global gaintab porttab
	set gaintab($porttab($which)) $level
}

proc mk.pane { w which bitmap } {
	set f [option get . audioFont Vat]
	frame $w.mute -borderwidth 1 -relief raised
	button $w.mute.b -text "mute" -font $f -borderwidth 1 \
		-anchor c \
		-command "mute_button_mute $w.mute.b $which" \
		-disabledforeground black
	global mute_state
	set mute_state($w.mute.b) 0

	global $which\Mutebutton
	set $which\Mutebutton $w.mute.b

	pack $w.mute.b -expand 1 -fill x

	frame $w.select -borderwidth 2 -relief raised
	button $w.select.b -bitmap $bitmap -relief flat -borderwidth 1 \
		-command "changePort $which $w.select.b $w.frame.scale" \
		-height 24
	pack $w.select.b -expand 1 -fill x

	frame $w.frame -borderwidth 2 -relief raised
	meter $w.frame.meter
	scale $w.frame.scale -orient vertical \
			-showvalue 0 \
			-from 256 -to 0 \
			-command "setgain $which" \
			-relief groove -borderwidth 2 -length 200
	global $which\Scale $which\PortButton
	set $which\Scale $w.frame.scale
	set $which\PortButton $w.select.b

	pack $w.frame.meter $w.frame.scale -side left -expand 1 -fill y

	pack $w.mute $w.select -fill x
	pack $w.frame -expand 1 -fill y
}

proc mk.ab { top } {
	set w $top.audio
	frame $w
	pack $w -expand 1 -fill y

	frame $w.spkr -borderwidth 0
	mk.pane $w.spkr output speaker
	frame $w.mike -borderwidth 0
	mk.pane $w.mike input mike
	pack $w.spkr $w.mike -side left -expand 1 -fill y

	global spkr_meter mike_meter
	set spkr_meter $w.spkr.frame.meter
	set mike_meter $w.mike.frame.meter

	set f [ctrlfont]
	button $top.button -text "Keep Audio" -font $f -command "aa hold"
	pack $top.button -fill x -side top

	global holdButton
	set holdButton $top.button
}

proc ptt_press { button } {
	global mute_state
	if { $mute_state($button) } {
		$button invoke
	}
}

#
# Build the user-interface.
#
proc mktk { } {
	init_globals

	global confName
	set confName [option get . conferenceName Vat]

	#
	# emulate implicit keyboard focus
	#
	bind . <Enter> { focus %W }

	#wm focusmodel . active
	bind . q { adios }
	bind . <Control-c> { adios }
	bind . <Control-d> { adios }

	bind . p { audio psetup }
	bind . P { audio psetup }

	. configure -background [option get . background Vat]

	frame .m
	frame .m.left
	frame .m.right
	frame .m.left.sites -relief raised -borderwidth 2

	global sitebox
	set sitebox .m.left.sites.sb
	sitebox $sitebox
	$sitebox name [option get . sessionName Vat]

	global title_bar title
	set title_bar .m.left.label
	label $title_bar -text $title -font [ctrlfont]\
		-background gray -borderwidth 2 -relief raised

	pack .m.left.sites -expand 1 -fill both
	pack .m.left.sites.sb -expand 1 -fill both
	pack .m.left.label -fill x

	set a .m.right
	frame $a.ab
	mk.ab $a.ab

	#
	# Setup sitebox bindings
	#	
	bind . c "$sitebox purge"
	bind . C "$sitebox purge"
	bind . o "$sitebox sort"
	bind . O "$sitebox sort"
	bind . l "$sitebox list"
	bind . L "$sitebox list"

	bind $sitebox <Any-ButtonRelease-1> sb_release

	bind . a { connection dump 2 }
	bind . A { connection dump 2 }
	bind . n { connection dump 1 }
	bind . N { connection dump 1 }
	bind . s { connection dump 0 }
	bind . S { connection dump 0 }

	global inputMutebutton
	bind all <ButtonPress-3> "ptt_press $inputMutebutton"
	bind all <ButtonRelease-3> "$inputMutebutton invoke"

	frame $a.bb -borderwidth 0
	button $a.bb.quit -text Quit -borderwidth 2 -relief raised \
		-background gray -font [ctrlfont] -command adios
	button $a.bb.menu -text Menu -borderwidth 2 -relief raised \
		-background gray -font [ctrlfont] \
		-command "toggle_window .menu"
	button $a.bb.help -text Help -borderwidth 2 -relief raised \
		-background gray -font [ctrlfont] \
		-command "toggle_window .help"

	pack $a.bb.menu $a.bb.help $a.bb.quit \
		-side left -fill both -expand 1

	pack $a.ab -expand 1 -fill both
	pack $a.bb -fill x

	pack .m.left -side left -expand 1 -fill both
	pack .m.right -side left -fill y

	pack .m -expand 1 -fill both

	set v [option get . geometry Vat]
	if { $v != "" } { 
		if { [ catch "wm geometry . $v" ] } {
			puts "vat: bad geometry $v"
			adios
		}
	}
	update idletasks
	global minwidth minheight iconPrefix
	set minwidth [winfo reqwidth .]
	set minheight [winfo reqheight .]
	wm minsize . $minwidth $minheight
	wm iconname . $iconPrefix$confName
	wm title . $confName
}

#
# Do any initialization that needs to wait for Controller
# object to be built.  Happens after mktk called, just
# before Tk_MainLoop is called.
#
proc init_tk { } {
	bind . <Unmap> { canmark 1 }
	bind . <Map> { canmark 0 ; mark_icon "" }

	mk.menu

	global outputMutebutton inputMutebutton \
		inputAGCbutton outputAGCbutton 
	if { [yesno mikeMute] } {
		$inputMutebutton invoke
	}
	if { [yesno speakerMute] } {
		$outputMutebutton invoke
	}
	if { [yesno mikeAGC] } {
		$inputAGCbutton invoke
	}
	if { [yesno speakerAGC] } {
		$outputAGCbutton invoke
	}

	global gaintab porttab inputScale outputScale 
	global inputPortButton outputPortButton
	set names [audio input names]
	foreach port $names {
		set attr [string tolower $port]
		set gaintab($port) [option get . $attr\Gain Vat]
	}
	if { [llength $names] <= 1 } {
		$inputPortButton configure -state disabled \
			-disabledforeground [option get . foreground Button]
	}
	set names [audio output names]
	foreach port $names {
		set attr [string tolower $port]
		set gaintab($port) [option get . $attr\Gain Vat]
	}
	if { [llength $names] <= 1 } {
		$outputPortButton configure -state disabled \
			-disabledforeground [option get . foreground Button]
	}
	setPort input $inputPortButton $inputScale [option get . inputPort Vat]
	setPort output $outputPortButton $outputScale \
		[option get . outputPort Vat]
}

proc ctrlfont { } {
	return [option get . ctrlFont Vat]
}

proc ctitlefont { } {
	return [option get . ctrlTitleFont Vat]
}

proc reverse { button } {
	set fg gray95
	set bg [option get . foreground Button]
	$button configure -foreground $fg -background $bg \
		-activeforeground gray -activebackground $bg
}

proc normal { button } {
	$button configure \
		-foreground [option get . foreground Button] \
		-background [option get . background Button] \
		-activeforeground [option get . foreground Button] \
		-activebackground [option get .  activeBackground Button]
}

proc aa_display { state } {
	global holdButton title_bar
	if { ![winfo exists $holdButton] } { return }
        set fg [option get . foreground Button]
	if { $state == 0 } {
		$title_bar configure -font [option get . noAudioFont Vat]
		normal $holdButton
	} elseif { $state & 1 } {
		# pinned flag
		$title_bar configure -font [ctrlfont]
		reverse $holdButton
	} else {
		$title_bar configure -font [ctrlfont]
		normal $holdButton
	}
}

proc mark_icon { mark } {
	global confName iconPrefix
	append mark $iconPrefix$confName
	wm iconname . $mark
}

set summary_window ""
proc sb_release { } {
	global summary_window
	if { [winfo exists $summary_window] } { stat_destroy $summary_window }
}

proc fmttime { t } {
	if { $t > 1000 } {
		return [format "%3.1fs " [expr $t / 1000.]]
	} elseif { $t >= 10. } {
		return [format "%3.fms" $t]
	} else {
		return [format "%3.1fms" $t]
	}
}

set statNames "Packets Missing Duplicate Misordered Dropped Bad-Header \
			Bad-ID Bad-Encoding Playout"

proc setstats { w s d f } {
	global statWindow statNames
	set p $statWindow($w)
	set n [llength $statNames]
	set k [option get . statTimeConst Vat]

	#
	# Total packets
	#
	set value [lindex $s 0]
	set old [lindex $d 0]
	set delta [expr $value - $old]
	$p.delta.packets configure -text $delta
	$p.total.packets configure -text $value
	set t [lindex $f 0]
	set t [expr $delta * $k + (1 - $k) * $t]
	set f [lreplace $f 0 0 $t]
	$p.lpf.packets configure -text [expr int($t)]
	set pkts $t

	#
	# Missing packets
	#
	set value [lindex $s 1]
	set old [lindex $d 1]
	set delta [expr $value - $old]
	$p.delta.missing configure -text $delta
	$p.total.missing configure -text $value
	set t [lindex $f 1]
	set t [expr $delta * $k + (1 - $k) * $t]
	set f [lreplace $f 1 1 $t]
	set pkts [expr $pkts + $t]
	if { $pkts == 0 } { set pkts 1.0 }
	set t [expr $t / $pkts * 100]
	$p.lpf.missing configure -text [expr int($t)]

	set i 2
	while { $i < $n } {
		set id [string tolower [lindex $statNames $i]]
		set value [lindex $s $i]
		if { $id == "playout" } {
			set value [fmttime $value]
			$p.total.$id configure -text $value
		} else {
			set old [lindex $d $i]
			set delta [expr $value - $old]
			$p.delta.$id configure -text $delta
			$p.total.$id configure -text $value
			set t [lindex $f $i]
			set t [expr $delta * $k + (1 - $k) * $t]
			set f [lreplace $f $i $i $t]
			set t [expr $t / $pkts * 100]
			$p.lpf.$id configure -text [expr int($t)]
		}
		incr i
	}
	if { ! [winfo exists $w.sc] } { return $f }

	global statIndex maxStat
	set i $statIndex($w)
	# special case playout since we don't want deltas
	if { $i < 8 } {
		set v [expr [lindex $s $i] - [lindex $d $i]]
	} else {
		set v [expr [lindex $s $i]]
	}
	if {! [info exists maxStat($w)] } {
		set max [expr 2 * $v]
		if { $max <= 0 } { set max 1}
		set maxStat($w) $max
		$w.sc configure -max $max
	} elseif { $v > $maxStat($w) } {
		set maxStat($w) $v
		$w.sc configure -max $v
	}
	$w.sc set $v

	return $f
}

proc stat_mon { w i } {
	$w.sc reset
	global maxStat
	if { [info exists maxStat($w)] } { unset maxStat($w) }
}

proc mk.stats { w } {
	set f [ctrlfont]
	set i 0
	global statNames
	set n [llength $statNames]
	set p $w.f
	frame $w.f
	frame $p.name
	frame $p.delta
	frame $p.total
	frame $p.lpf
	set top [winfo toplevel $w]
	while { $i < $n } {
		set name [lindex $statNames $i]
		set id [string tolower $name]
		radiobutton $p.name.$id -text $name -font $f -anchor w \
			-command "stat_mon $top $i" \
			-variable statIndex($top) -value $i
		label $p.delta.$id -font $f -anchor e -width 4
		label $p.lpf.$id -font $f -anchor e -width 7
		label $p.total.$id -font $f -anchor e -width 7
		pack $p.name.$id -anchor w -fill x
		pack $p.total.$id $p.delta.$id $p.lpf.$id \
			-expand 1 -fill x -anchor e
		incr i
	}

	pack $p.name $p.delta $p.lpf $p.total -side left

	global statWindow statIndex maxStat
	set statWindow($top) $p
	set statIndex($top) 0
	if { [info exists maxStat($top)] } { unset maxStat($top) }

	pack $w.f -anchor c
}

proc stat_update { w addr mixer } {
	$w.title.name configure -text [connection name $addr]
	set fmt [connection format $mixer]
	if { $fmt == "" } { set fmt "?" }
	$w.title.encoding configure -text "Encoding: $fmt"
	set t [connection lastactive $addr]
	if { $t == "" } { set t "never" }
	$w.title.time configure -text "Last active: $t"

	set stats [connection stats $addr]
	if { $stats != "" } {
		global oldStats lpfStats
		if { ![info exists oldStats($w)] } {
			set oldStats($w) $stats
			set lpfStats($w) "0 0 0 0 0 0 0 0 0"
		}
		set fs [setstats $w $stats $oldStats($w) $lpfStats($w)]
		set oldStats($w) $stats
		set lpfStats($w) $fs
	} else {
		set zstats "0 0 0 0 0 0 0 0 0"
		setstats $w $zstats $zstats $zstats
	}
}

proc stat_timer { w addr mixer id } {
	global stat_timer_id
	if { $id != $stat_timer_id($w) || ![winfo exists $w] } {
		return
	}
	stat_update $w $addr $mixer
	after 1000 "stat_timer $w $addr $mixer $id"
}

proc stat_destroy { w } {
	global oldStats
	if { [info exists oldStats($w)] } { unset oldStats($w) }
	destroy $w
}

proc site_summary { name addr mixer x y keep } {
	global summary_window
	set summary_window ""
	set w [split $addr .]
	set w .[lindex $w 0]_[lindex $w 1]_[lindex $w 2]_[lindex $w 3]

	if { [winfo exists $w] } { return }

	toplevel $w
	bind $w <Enter> "focus $w"
	global iconPrefix
	wm iconname $w "$iconPrefix$name"
	wm title $w "$name"
	wm transient $w .
	wm withdraw $w
	set f [ctitlefont]
	frame $w.title -borderwidth 2 -relief raised
	label $w.title.name -borderwidth 0 -font $f -anchor w
	label $w.title.addr -borderwidth 0 -font $f -anchor w \
		-text [expr {$mixer == $addr ? $addr : "$addr via $mixer"}]
	label $w.title.encoding -borderwidth 0 -font $f -anchor w
	label $w.title.time -borderwidth 0 -font $f -anchor w
	frame $w.frame -borderwidth 2 -relief raised

	mk.stats $w.frame
	pack $w.title.name $w.title.addr $w.title.encoding \
		$w.title.time -fill x

	pack $w.title -fill x
	pack $w.frame -expand 1 -fill x -anchor center

	if { $keep } {
		stripchart $w.sc -max 200 -stripwidth 1 -width 1 \
			-relief flat -striprelief flat \
			 -borderwidth 0
		button $w.ok -borderwidth 2 -relief raised -font $f \
			-command "stat_destroy $w" -text "Dismiss" 
		pack $w.sc -expand 1 -fill x
		pack $w.ok -fill x
	}
	stat_update $w $addr $mixer

	incr x [winfo rootx .]
	incr y [winfo rooty .]
	# adjust for virtual desktops
	incr x [winfo vrootx .]
	incr y [winfo vrooty .]

	#
	# Need to do an update so that $w gets laid out allowing us to
	# look up its size.  This is tricky because of a possible race:
	# if the user releases the mouse, summary_window might get
	# destroyed during the update idletasks.  So we check
	# that the window still exists before proceeeding.
	#

	if { !$keep } { set summary_window $w }
	update idletasks
	if { ![winfo exists $w] } { return }

	#
	# Check if window goes off the bottom or right.  Don't need
	# to check top or left since upper-left corner is at mouse.
	#
	set right [expr [winfo screenwidth .] - [winfo reqwidth $w] - 5]
	if { $x > $right } {
		set x $right
	}
	set bot [expr [winfo screenheight .] - [winfo reqheight $w] - 5]
	if { $y > $bot } {
		set y $bot
	}
	wm geometry $w +$x+$y
	wm deiconify $w
	global stat_timer_id
	if { ! [info exists stat_timer_id($w)] } {
		set stat_timer_id($w) 0
	}
	incr stat_timer_id($w)
	after 1000 "stat_timer $w $addr $mixer $stat_timer_id($w)"

}

set helpno 0
proc helpitem { w text } {
	global helpno
	set f [option get . helpFont Vat]
	set h $w.h$helpno
	incr helpno
	frame $h
	canvas $h.bullet -width 12 -height 12 
	$h.bullet create oval 6 3 12 9 -fill black
	message $h.msg -justify left -anchor w -font $f -width 450 -text $text
	pack $h.bullet -side left -anchor ne -pady 5
	pack $h.msg -side left -expand 1 -fill x -anchor nw
	pack $h -expand 1 -fill both
}

proc mk.help { } {
	set w .help
	if { [winfo exists $w] } { return }
	toplevel $w
	bind $w <Enter> "focus $w"
	wm withdraw $w
	wm iconname $w "Vat Help"
	wm title $w "Vat Help"

	frame $w.frame -borderwidth 2 -relief raised
	set p $w.frame
	helpitem $p "Read the man page."
	helpitem $p "Before transmitting audio, adjust the mike \
level so that the output meter peaks around 80% of full scale.  Below this, \
you are hard to hear, and above this, your signal is distorted."
	helpitem $p "To talk, temporarily unmute the mike by depressing \
the right mouse button anywhere in the vat window.  The mike is \
live only while the button is depressed.  For hands-free operation, \
you can leave the mike active by de-selecting the ``mute button'' above \
the mike icon."
	helpitem $p "Mute individual sites by clicking on checkbox next \
to name."
	helpitem $p "Select input/output line by clicking on mike \
or speaker icon."
	helpitem $p "Prevent other vats from taking the audio device \
by clicking on the ``Keep Audio'' button.  Different vats will \
cooperate so that only one instance ever has ``Keep Audio'' selected.  \
The vat title is italicized when the audio is not being held."
	helpitem $p "Get network statistics for a site by \
clicking (and holding) left mouse button on site name.  Use shift-left \
for a persistent window.  The stripchart plots the selected statistic."	
helpitem $p "If the user interface looks peculiar, you might \
have X resources that conflict with tk.  A common problem is \
defining ``*background'' and/or ``*foreground''."

	button $w.frame.ok -text " Dismiss " -borderwidth 2 -relief raised \
		-command "wm withdraw $w" -font [ctitlefont] 
	pack $w.frame.ok -pady 6 -padx 6 -anchor e

	pack $w.frame -expand 1 -fill both
}
