/*
 * Gated Release 4.x, 5.x, 6.x, 7.x
 *  
 *  $Id: trace.c,v 1.12 2000/04/08 13:46:54 mrr Exp $
 */

/*
 * GateD Releases Unicast, Multicast, IPv6, RSd
 * 
 * Copyright (c) 1996,1997,1998,1999 
 * The Regents of the University of Michigan.
 * All Rights Reserved.
 * 
 * License to use, copy, modify, and distribute this software and its
 * documentation can be obtained from Merit Network, Inc. at the 
 * University of Michigan.
 * 
 * Merit GateD Consortium
 * Merit Network, Inc.
 * 4251 Plymouth Road, Suite C
 * Ann Arbor, MI 48105
 * 
 * THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND, EITHER
 * EXPRESS OR IMPLIED, INCLUDING WITHOUT LIMITATION WARRANTIES OF 
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE. THE REGENTS OF THE
 * UNIVERSITY OF MICHIGAN AND MERIT DO NOT WARRANT THAT THE FUNCTIONS 
 * CONTAINED IN THE SOFTWARE WILL MEET LICENSEE'S REQUIREMENTS OR THAT 
 * OPERATION WILL BE UNINTERRUPTED OR ERROR FREE. The Regents of the
 * University of Michigan and Merit shall not be liable for any special, 
 * indirect, incidental or consequential damages with respect to any claim 
 * by Licensee or any third party arising from use of the software. 
 * GateD was originated and developed through release 3.0 by Cornell 
 * University and its collaborators.
 * 
 * Please send questions or comments to gated-people@gated.org.
 *
 * Please submit bugs, bug fixes, and enhancements using the send-pr(1) 
 * utility or via the web at 
 * www.gated.org/gated-web/support/html/report_prob.html.
 * 
 * ------------------------------------------------------------------------
 *
 *      Copyright (c) 1990,1991,1992,1993,1994,1995 by Cornell University.
 *          All rights reserved.
 *
 *      THIS SOFTWARE IS PROVIDED "AS IS" AND WITHOUT ANY
 *      EXPRESS OR IMPLIED WARRANTIES, INCLUDING, WITHOUT
 *      LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTABILITY
 *      AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 *      GateD is based on Kirton's EGP, UC Berkeley's routing
 *      daemon   (routed), and DCN's HELLO routing Protocol.
 *      Development of GateD has been supported in part by the
 *      National Science Foundation.
 *
 * ------------------------------------------------------------------------
 *
 *      Portions of this software may fall under the following
 *      copyrights:
 *
 *      Copyright (c) 1988 Regents of the University of California.
 *      All rights reserved.
 *
 *      Redistribution and use in source and binary forms are
 *      permitted provided that the above copyright notice and
 *      this paragraph are duplicated in all such forms and that
 *      any documentation, advertising materials, and other
 *      materials related to such distribution and use
 *      acknowledge that the software was developed by the
 *      University of California, Berkeley.  The name of the
 *      University may not be used to endorse or promote
 *      products derived from this software without specific
 *      prior written permission.  THIS SOFTWARE IS PROVIDED
 *      ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES,
 *      INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *      MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 * __END_OF_COPYRIGHT__
 */


#define	INCLUDE_TIME
#define	INCLUDE_FILE
#define	INCLUDE_FCNTL /* to get O_RDWR */
#define	INCLUDE_STAT

#include "include.h"
#include "krt/krt.h"
#include "parse.h"

trace *trace_global; /* Global options */

static trace_file trace_files = { &trace_files, &trace_files };	/* file list */
int trace_nosyslog = TRACE_LOG_NORMAL;
int trace_require_lock;

char trace_buffer[BUFSIZ];
char *trace_ptr = trace_buffer;

static block_t trace_trace_index;
static block_t trace_file_index;
static char trace_errbuf[BUFSIZ];

static int bits_index;
static char bits_strings[BITS_N_STRINGS][LINE_MAX];

static task *trace_dump_task;	/* Pointer to the dump task */

static char *syslog_name = (char *)0;
static flag_t syslog_options = (flag_t)0;
static flag_t syslog_mask = LOG_UPTO(LOG_DEBUG);


static const bits trace_types[] = {
	{ TR_ALL,		"all" },
	{ TR_NORMAL|TR_ROUTE,	"general" },
	{ TR_STATE,		"state" },
	{ TR_NORMAL,		"normal" },
	{ TR_POLICY,		"policy" },
	{ TR_TASK,		"task" },
	{ TR_TIMER,		"timer" },
	{ TR_ROUTE,		"route" },
	{ TR_DETAIL,		"detail" },
	{ TR_DETAIL_RECV,	"recv-detail" },
	{ TR_DETAIL_SEND,	"send-detail" },
	{ TR_PACKET,		"packet" },
	{ TR_PACKET_RECV,	"recv-packet" },
	{ TR_PACKET_SEND,	"send-packet" },
	{ 0,			NULL }
};

static const bits trace_global_types[] = {
	{ TR_PARSE,		"parse" },
	{ TR_ADV,		"adv" },
	{ TR_KRT_SYMBOLS,	"symbols" },
	{ TR_KRT_IFLIST,	"iflist" },
	{ 0,			NULL }
};

static const bits trace_control_types[] = {
	{ TRC_NOSTAMP,		"nostamp" },
	{ TRC_LOGONLY,		"logonly" },
	{ TRC_NL_BEFORE,	"nl_before" },
	{ TRC_NL_AFTER,		"nl_after" },
	{ 0,			NULL }
};

#ifdef	SYSLOG_DEBUG

static const bits syslog_pri_bits[] = {
	{ LOG_MASK(LOG_EMERG),	"EMERG" },
	{ LOG_MASK(LOG_ALERT),	"ALERT" },
	{ LOG_MASK(LOG_CRIT),	"CRIT" },
	{ LOG_MASK(LOG_ERR),	"ERR" },
	{ LOG_MASK(LOG_WARNING),"WARNING" },
	{ LOG_MASK(LOG_NOTICE),	"NOTICE" },
	{ LOG_MASK(LOG_INFO),	"INFO" },
	{ LOG_MASK(LOG_DEBUG),	"DEBUG" },
	{ 0,			NULL }
};

static const bits syslog_facility_bits[] = {
	{ LOG_KERN,	"KERN" },
	{ LOG_USER,	"USER" },
	{ LOG_MAIL,	"MAIL" },
	{ LOG_DAEMON,	"DAEMON" },
	{ LOG_AUTH,	"AUTH" },
	{ LOG_SYSLOG,	"SYSLOG" },
	{ LOG_LPR,	"LPR" },
	{ LOG_NEWS,	"NEWS" },
	{ LOG_UUCP,	"UUCP" },
	{ LOG_CRON,	"CRON" },
	{ LOG_LOCAL0,	"LOCAL0" },
	{ LOG_LOCAL1,	"LOCAL1" },
	{ LOG_LOCAL2,	"LOCAL2" },
	{ LOG_LOCAL3,	"LOCAL3" },
	{ LOG_LOCAL4,	"LOCAL4" },
	{ LOG_LOCAL5,	"LOCAL5" },
	{ LOG_LOCAL6,	"LOCAL6" },
	{ LOG_LOCAL7,	"LOCAL7" },
	{ 0,		NULL }
};

static const bits syslog_bits[] = {
	{ LOG_PID,	"PID" },
	{ LOG_CONS,	"CONS" },
	{ LOG_ODELAY,	"ODELAY" },
	{ LOG_NDELAY,	"NDELAY" },
	{ LOG_NOWAIT,	"NOWAIT" },
	{ 0, NULL }
};

/*
 * display trace options enabled
 */
#define trace_display(trp, level)                               \
	trace_tf((trp), (level), TRC_NL_BEFORE|TRC_NL_AFTER,    \
	    ("Tracing flags enabled:%s%s %s",                   \
	    (trp)->tr_control ? " " : "",                       \
	    trace_bits(trace_control_types, (trp)->tr_control), \
	    trace_string((trp)->tr_flags, (trp)->tr_names)));

/*
 * set the syslog name
 */
void
openlog(const char *name, flag_t opts, flag_t facility)
{

	if (syslog_name) {
		task_mem_free((task *)0, syslog_name);
	}

	syslog_name = task_mem_strdup((task *)0, name);
	syslog_options = opts;

	(void)fprintf(stderr,
	    "openlog: name %s options %s facility LOG_%s\n",
	    syslog_name,
	    trace_bits(syslog_bits, opts),
	    trace_bits(syslog_facility_bits, facility));
}

/*
 * set the syslog mask
 */
void
setlogmask(flag_t mask)
{

	(void)fprintf(stderr,
	    "setlogmask: old %s new %s\n",
	    trace_bits(syslog_pri_bits, syslog_mask),
	    trace_bits(syslog_pri_bits, mask));
		   
	syslog_mask = mask;
}

/*
 * output text to the syslog
 */
void
syslog(int pri, const char *msg, ...)
{
	va_list ap;
	flag_t pri_mask;
	char buf[BUFSIZ];

	va_start(ap, msg);
	pri_mask = LOG_MASK(pri);
    
	(void)vsprintf(buf, msg, ap);

	(void)fprintf(stderr,
	    "SYSLOG\nSYSLOG LOG_%s %s%s",
	    trace_bits(syslog_pri_bits, pri_mask),
	    BIT_TEST(syslog_mask, pri_mask) ? "" : "SKIP ", syslog_name);

	if (BIT_TEST(syslog_options, LOG_PID))
		(void)fprintf(stderr, "[%d]", task_pid);

	(void)fprintf(stderr, ": %s\nSYSLOG\n", buf);

	va_end(ap);
}
#endif	/* SYSLOG_DEBUG */

/*
 * trace the fact that we syslogged
 */
void
trace_syslog(int pri)
{

	if (trace_nosyslog == TRACE_LOG_TRACE) {
		if (pri < LOG_NOTICE) {
			(void)fwrite((char *)trace_buffer, sizeof(char),
			    (size_t)(trace_ptr - trace_buffer), stderr);

			(void)fprintf(stderr, "\n");
		}
	}
}

/*
 * convert tracing flags to printable
 */
const char *
trace_string(flag_t tr_flags, const bits *types)
{

	switch (tr_flags) {
	case TR_ALL:
		return (" all");
	case 0:
		return (" none");
	default:
		return (trace_bits2(types, trace_types, tr_flags));
	}
}

/*
 * close trace file
 */
static void
trace_close(trace_file *tfp)
{

	if ((tfp->trf_fd != -1) && (tfp->trf_fd != fileno(stderr)))
		(void)close(tfp->trf_fd);

	tfp->trf_fd = -1;
}


void
trace_file_free(trace_file *tfp)
{

	GASSERT(!tfp || tfp->trf_refcount);

	if (tfp && !--tfp->trf_refcount) {
#ifdef DEBUG
		syslog(LOG_CRIT, "trace_file_free: freeing %s",
		    tfp->trf_file ? tfp->trf_file : "stderr");
#endif /* DEBUG */

		if (tfp->trf_fd != -1)
			trace_close(tfp);

		if (tfp->trf_file)
			task_mem_free((task *)0, tfp->trf_file);

		REMQUE(tfp);
		task_block_free(trace_file_index, (void_t)tfp);
	}
}

/*
 * create a new trace
 */
trace *
trace_create(void)
{
	trace *trp;

	trp = (trace *)task_block_alloc(trace_trace_index);

	return (trace_alloc(trp));
}

/*
 * refcount an existing trace
 */
trace *
trace_alloc(trace *trp)
{

	if (trp)
		trp->tr_refcount++;

	return (trp);
}

/*
 * de-refcount an existing trace, free'ing if necessary
 */
trace *
trace_free(trace *trp)
{

	GASSERT(trp->tr_refcount);

	if (!--trp->tr_refcount) {

		if (trp->tr_file)
			trace_file_free(trp->tr_file);

		task_block_free(trace_trace_index, (void_t)trp);
    	}

	return NULL;
}

/*
 * turn off tracing.
 */
void
trace_off(trace_file *tfp)
{

	if (tfp->trf_fd != -1) {

		trace_log_tf(trace_global,
		    TRC_NL_BEFORE|TRC_NL_AFTER, LOG_INFO,
		    ("Tracing to \"%s\" suspended",
		    tfp->trf_file ? tfp->trf_file : "(stderr)"));

		if (tfp->trf_fd != fileno(stderr))
			(void)close(tfp->trf_fd);

		tfp->trf_fd = -1;
    	}
}

/*
 * close all trace files
 */
void
trace_close_all(void)
{
	trace_file *tfp;

	TRACE_FILES(tfp) {

		trace_close(tfp);

	} TRACE_FILES_END(tfp);
}

/*
 * locate a given trace file descriptor
 */
trace_file *
trace_file_locate(char *file, off_t limit_size, u_int limit_files, flag_t flags)
{
	int check_limits;
	register trace_file *tfp;

	check_limits = 0;

#ifndef	FLAT_FS
	/*
 	 * for systems without a filesystem hierarchy
 	 */
	if (file && *file != '/' && TRACE_REALFILE(file)) {

		/* Make trace file name absolute */
		char *fn = task_mem_malloc(NULL,
		    (size_t)(strlen(file) + strlen(task_path_start) + 2));

		(void)strcpy(fn, task_path_start);
		(void)strcat(fn, "/");
		(void)strcat(fn, file);

		/*
		 * created an absolute path for file name,
		 * free the memory holding trace file name
		 */
		task_mem_free(NULL, (void_t)file);
		file = fn;
    	}
#endif	/* FLAT_FS */

 	/* Locate an already created trace_file blocks */
	if (file) {

		/* Search the list of existing blocks */
		TRACE_FILES(tfp) {

			if (tfp->trf_file && !strcmp(file, tfp->trf_file)) {
				/* Found it! */

				if (tfp->trf_file && tfp->trf_refcount
				    && TRACE_REALFILE(file)) {

					check_limits++;

					/*
					 * we found the trace file, free up
					 * memory pointed to by file
					 */
					task_mem_free(NULL, (void_t)file);
				}
				break;
			}
		} TRACE_FILES_END(tfp) ;

	} else {

		/* Inherit the global file if it exists */

		if (trace_global && trace_global->tr_file)
			/* Inherit global file */
			tfp = trace_global->tr_file;
		else
			/* No global tracing to inherit */
			return NULL;
	}

	if (tfp == &trace_files) {

		/* Not found, create and insert on list */

		INSQUE(tfp = (trace_file *) task_block_alloc(trace_file_index),
		    trace_files.trf_back);

		tfp->trf_file = file;
		tfp->trf_limit_size = limit_size;
		tfp->trf_limit_files = limit_files;
		tfp->trf_flags = flags;
		tfp->trf_fd = -1;
    	}

	/* If tracing to a real file */
	if (check_limits) {
	
		/* Verify that limits match */
		if (tfp->trf_limit_size != limit_size) {
			/*
			 * save the limit sizes as int, since I can't figure out
			 * how to make gated print a quad_t type.
			 */
			u_int l1 = tfp->trf_limit_size;
			u_int l2 = limit_size;

			trace_log_tf(trace_global, 0, LOG_WARNING,
			    ("trace_file_locate: replacing %s file size "
			    "limit %u with %u",
			    tfp->trf_file ? tfp->trf_file : "stderr", l1, l2));

			tfp->trf_limit_size = limit_size;
		}

		if (tfp->trf_limit_files != limit_files) {

			trace_log_tf(trace_global, 0, LOG_WARNING, 
			    ("trace_file_locate: replacing %s file "
			    "count limit %u with %u",
			    tfp->trf_file ? tfp->trf_file : "stderr",
			    tfp->trf_limit_files, limit_files));

			tfp->trf_limit_files = limit_files;
		}

		if ((flags ^ tfp->trf_flags) & TRF_REPLACE) {

			trace_log_tf(trace_global, 0, LOG_WARNING,
			    ("trace_file_locate: changing %s to %s",
			    tfp->trf_file ? tfp->trf_file : "stderr",
			    BIT_TEST(flags, TRF_REPLACE) ?
			    "replace" : "append"));

			tfp->trf_flags = (tfp->trf_flags & ~TRF_REPLACE) |
			    (flags & TRF_REPLACE);
		}
    	}
	
	return (trace_file_alloc(tfp));
}

/*
 * turn on tracing
 */
void
trace_on(trace_file *tfp)
{
    char tbuf[BUFSIZ];

	if (tfp->trf_fd == -1) {

		if (!tfp->trf_file)

			/* No file specified, use stderr */
			tfp->trf_fd = fileno(stderr);

		else if (!TRACE_REALFILE(tfp->trf_file))

			/* the user stated "stderr" as the tracefile, oblige */
			tfp->trf_fd = fileno(stderr);

		else {

			int append, fd;
			struct stat stbuf;

			append = BIT_TEST(tfp->trf_flags, TRF_REPLACE) == 0;

			if (BIT_TEST(task_state, TASKS_RECONFIG))
				/* don't replace if reconfiguring to same */
				append = TRUE;
	
			if (stat(tfp->trf_file, &stbuf) < 0) {

				if (errno == ENOENT)
					tfp->trf_size = 0;
				else {
					syslog(LOG_ERR,
					    "trace_on: stat(%s): %m",
					    tfp->trf_file);
					return;
				}

			} else {

				/* 
				 * Indicate current file size.
				 * Will be reset later if not appending
				 */
				tfp->trf_size = stbuf.st_size;

				switch (stbuf.st_mode & S_IFMT) {
				default:
					/* 
					 * This may be a pipe, where we won't
					 * be able to seek
					 */
					append = FALSE;

					if (tfp->trf_limit_size) {

					/* can't rename a pipe! */

						tfp->trf_limit_size = 0;
						trace_log_tf(trace_global,
						    0, LOG_WARNING, 
						    ("trace_on: \"%s\" is "
						    "not a regular file, "
						    "ignoring size limit",
						    tfp->trf_file));
			    		}
				/* Fall through */
				case S_IFREG:
				    break;
	
				case S_IFDIR:
				case S_IFBLK:
				case S_IFLNK:
					trace_log_tf(trace_global, 0, LOG_ERR, 
					    ("trace_on: \"%s\" is "
					    "not a regular file",
					    tfp->trf_file));
					return;
				}
			}

			/* First try to open the file */
			fd = open(tfp->trf_file, O_NONBLOCK | O_RDWR | O_CREAT |
			    (append ? O_APPEND : O_TRUNC), 0644);

			if (fd < 0) {
				trace_log_tf(trace_global, 0, LOG_ERR, 
				    ("Could not open \"%s\": %m",
				    tfp->trf_file));
				return;
		    	}

			/*
			 * Then try to lock the file, to insure
			 * only one gated writes to it at a time 
			 */
			if (flock(fd, LOCK_EX|LOCK_NB) < 0) {

				int error = errno;

				switch (error) {
				case EWOULDBLOCK:
#if defined(EAGAIN) && EAGAIN != EWOULDBLOCK
				case EAGAIN:	/* System V style */
#endif	/* EAGAIN */
					trace_log_tf(trace_global, 0, LOG_ERR, 
					    ("trace_on: trace file \"%s\" "
					    "appears to be in use",
				    	tfp->trf_file));

					(void)close(fd);
					return;

				default:
					trace_log_tf(trace_global, 0, LOG_ERR,
					    ("trace_on: Could not obtain lock "
					    "on \"%s\": %m", tfp->trf_file));

					if (trace_require_lock == 0)
						break;
					else {
						(void)close(fd);
						return;
					}
				}
 			}

			if (!append)
				/* Indicate the file is empty */
				tfp->trf_size = 0;

			/* Now close the old one */
			if (tfp->trf_fd != -1)
				trace_close(tfp);

			/* set the new one */
			tfp->trf_fd = fd;
		}

		if (tfp->trf_file) {
			syslog(LOG_INFO,
			    "trace_on: tracing to \"%s\" started",
			    tfp->trf_file);

			sprintf(tbuf,
			    "%s%strace_on: Tracing to \"%s\" started\n",
			    time_string,
			    *time_string ? " " : "", tfp->trf_file);

			trace_write(tfp, tbuf, strlen(tbuf));
		}
	}

	/* Reset replace indication */
	BIT_RESET(tfp->trf_flags, TRF_REPLACE);
}

/*
 * Toggle tracing
 */
void
trace_toggle(void)
{
	int turn_off;
	trace_file *tfp;

	turn_off = 0;

	/* First figure out if tracing is on or off */
	TRACE_FILES(tfp) {

		if (tfp->trf_fd != -1) {
		    turn_off++;
		    break;
		}

	} TRACE_FILES_END(tfp) ;
    
	if (turn_off) {
		/* Tracing is on, turn it off */

		/* Then any others */
		TRACE_FILES(tfp) {

			if (tfp->trf_fd != fileno(stderr))
				trace_off(tfp);

		} TRACE_FILES_END(tfp) ;

	} else {
		/* Tracing is off, try to turn it on */

		/* Then any others */
		TRACE_FILES(tfp) {

			if (tfp->trf_fd != fileno(stderr))
				trace_on(tfp);
	
		} TRACE_FILES_END(tfp) ;
    	}
}

trace *
trace_set_global(const bits *types, flag_t inherit)
{
	trace *trp;

	trp = NULL;
    
	if (trace_global && trace_global->tr_flags) {
		trp = trace_create();
		trp->tr_names = types;
		trp->tr_flags = trace_global->tr_flags & (inherit|TR_INHERIT);
		trp->tr_control = trace_global->tr_control & TRC_INHERIT;
		trp->tr_file = trace_file_alloc(trace_global->tr_file);
    	}

	return (trp);
}

/*
 *  Rotate the trace files
 */
static void
trace_rotate(trace_file *tfp)
{
	char file1[MAXPATHLEN];
	char file2[MAXPATHLEN];
	char *cp, tbuf[BUFSIZ];
	int i;

	i = tfp->trf_limit_files - 2;
	cp = file1;

	sprintf(tbuf, "trace_rotate: Rotating trace files\n");
	trace_write(tfp, tbuf, strlen(tbuf));

	syslog(LOG_INFO, "trace_rotate: rotating %s", tfp->trf_file);

	/* Close this one */
	trace_close(tfp);

	while (i >= 0) {

		(void)sprintf(file2, "%s.%d", tfp->trf_file, i--);

		if (i < 0)
	    		cp = tfp->trf_file;
		else
			(void)sprintf(file1, "%s.%d", tfp->trf_file, i);

		if (rename(cp, file2) < 0) {

			if (errno != ENOENT)
				syslog(LOG_ERR,
				    "trace_rotate: rename(%s, %s): %m",
				    file1, file2);

		}
    	}

	/* And open new file */
	trace_on(tfp);
}

/*
 * parse trace flags specified on the command line
 */
flag_t
trace_args(char *flags)
{
	int i;
	char *cp;
	flag_t tr_flags;

	i = 1;
	tr_flags = 0;
	cp = flags;

	while ((cp = (char *)index(cp, ','))) {
		i++;
		*cp++ = (char)0;
	}

	for (cp = flags; i--; cp += strlen(cp) + 1) {

		if (!strcasecmp(cp, "none"))
			tr_flags = (flag_t) 0;
		else {

			static const bits *types[] = {
				trace_types,
				trace_global_types,
				(bits *)0
		    	};
			const bits *p = (bits *)0, **pp;

			for (pp = types; *pp; pp++) {

				for (p = *pp; p->t_bits; p++) {
					if (!strcasecmp(cp, p->t_name)) {
						BIT_SET(tr_flags, p->t_bits);
						break;
					}
				}

				if (p->t_bits)
					break;
			}
		
			if (!p->t_bits)
				trace_log_tf(trace_global, 0, LOG_ERR, 
				    ("%s: unknown trace flag: %s",
				    task_progname, cp));
		}
	}
	return (tr_flags);
}

/*
 * return pointer to static string with current trace flags
 */
char *
trace_bits(const bits *bp, flag_t mask)
{
	char *dp, *s;
	const bits *p;
	flag_t seen;

	s = BITS_STRING;
	dp = s;
	*dp = (char)0;
	seen = 0;

	for (p = bp; p->t_bits; p++) {

		if (BIT_MATCH(mask, p->t_bits) && !BIT_MATCH(seen, p->t_bits)) {

			const char *sp = p->t_name;

			BIT_SET(seen, p->t_bits);
	    
			if (dp > s)
				*dp++ = ' ';

			while (*sp)
				*dp++ = *sp++;

		}
	}

	*dp = (char)0;

	BITS_NEW_INDEX;

	return (s);
}


/*
 * return pointer to static string with current trace flags
 */
char *
trace_bits2(const bits *bp1, const bits *bp2, flag_t mask)
{
	char *dp, *s;
	const bits *p, *bp;
	flag_t seen;

	s = BITS_STRING;
	dp = s;
	*dp = (char)0;
	bp = bp1;
	seen = 0;

	do {
		for (p = bp; p && p->t_bits; p++) {
			if (BIT_MATCH(mask, p->t_bits) &&
			    !BIT_MATCH(seen, p->t_bits)) {

				const char *sp = p->t_name;

				BIT_SET(seen, p->t_bits);
	    
				if (dp > s)
					*dp++ = ' ';
			
				while (*sp)
					*dp++ = *sp++;
		    	}
		}

    } while (bp != bp2 && (bp = bp2)) ;

    *dp = (char)0;

    BITS_NEW_INDEX;
	
    return (s);
}


const char *
trace_value(const bits *bp, int value)
{
	const bits *p;

	for (p = bp; p->t_name; p++) {
		if (p->t_bits == (u_int) value)
			return p->t_name;
	}

	return NULL;
}

/*
 * dump tracing state for a task
 */
void
trace_task_dump(FILE *fd, trace *trp)
{

	(void)fprintf(fd, "\t\tTrace options:%s%s %s\n",
	    trp->tr_control ? " " : "",
	    trace_bits(trace_control_types, trp->tr_control),
	    trace_string(trp->tr_flags, trp->tr_names));

	if (trp->tr_file) {

		(void)fprintf(fd, "\t\tTrace file: %s",
		    trp->tr_file->trf_file
		    ? trp->tr_file->trf_file
		    : "(stderr)");

		if (trp->tr_file->trf_limit_size)
			(void)fprintf(fd, "\tsize %u\tfiles %u\n",
			    trp->tr_file->trf_limit_size,
			    trp->tr_file->trf_limit_files);
		else
			(void)fprintf(fd, "\n");
	}
}

/*
 * dump everything
 */
static void
trace_do_dump(task *tp)
{
	int tries;
	FILE *fp;
	char path_dump[MAXPATHLEN];

	tries = 10;
	fp = NULL;

	(void)sprintf(path_dump, _PATH_DUMP, task_progname);

	while (fp == NULL && tries--) {

		int fd;
		int can_seek = TRUE;
		struct stat stbuf;

		if (stat(path_dump, &stbuf) < 0) {

			if (errno != ENOENT) {
				trace_log_tf(trace_global, 0, LOG_ERR, 
				    ("trace_do_dump: stat(%s): %m", path_dump));
				return;
		    	}

		} else {

			switch (stbuf.st_mode & S_IFMT) {
			default:
				/* Might be a FIFO (pipe) -- can't seek */
				can_seek = FALSE;
				/* Fall through */

			case S_IFREG:
			break;

			case S_IFDIR:
			case S_IFBLK:
			case S_IFLNK:
				trace_log_tf(trace_global, 0, LOG_ERR,
				    ("trace_do_dump: \"%s\" is not "
				    "a regular file", path_dump));
				return;
			}
		}

		/* First try to open the file */
		fd = open(path_dump,
		    O_RDWR | O_CREAT | (can_seek ? O_APPEND : 0), 0644);

		if (fd < 0) {
			trace_log_tf(trace_global, 0, LOG_ERR, 
			    ("Could not open \"%s\": %m", path_dump));
			return;
		}

		/*
		 * then try to lock the file to insure only one gated
		 * writes to it at a time 
		 */
		if (flock(fd, LOCK_EX|LOCK_NB) < 0) {

			int error = errno;

			switch (error) {
			case EWOULDBLOCK:
#if defined(EAGAIN) && EAGAIN != EWOULDBLOCK
			case EAGAIN:	/* System V style */
#endif /* EAGAIN */

				trace_log_tf(trace_global, 0, LOG_ERR,
				    ("trace_do_dump: dump file \"%s\" in use",
				    path_dump));
				break;
				trace_log_tf(trace_global, 0, LOG_ERR, 
				    ("trace_do_dump: dump file \"%s\" "
				    "in use, waiting...", path_dump));

				(void)close(fd);

				/* XXX sleep?! */
				sleep(15);

				continue;
			default:
				trace_log_tf(trace_global, 0, LOG_ERR, 
				    ("trace_do_dump: Could not obtain "
				    "lock on \"%s\": %m", path_dump));
				break;
		    	}

			(void)close(fd);
			return;
		}

		/* And finally open the stream */
		fp = fdopen(fd, can_seek ? "a" : "w");

		if (!fp) {

			trace_log_tf(trace_global, 0, LOG_ERR, 
			    ("trace_do_dump: can not open \"%s\" "
			    "for writing: %m", path_dump));
			return;
		}
	}
    
#ifndef vax11c
	setvbuf(fp, NULL, _IOLBF, 0);
#endif	/* vax11c */

	trace_log_tf(trace_global, 0, LOG_INFO,
	    ("trace_do_dump: processing dump to %s", path_dump));

	(void)fprintf(fp, "\f\n\t%s[%d] version %s memory dump on %s at %s\n",
	    task_progname, task_mpid, gated_version, task_hostname, time_full);

	if (krt_version_kernel)
		(void)fprintf(fp, "\t\t%s\n\n", krt_version_kernel);

	(void)fprintf(fp, "\t\tStarted at %s\n", task_time_start.gt_ctime);

	/* Dump all the trace files */
	{
		trace_file *tfp;

		(void)fprintf(fp, "Tracing:\n\tGlobal:\t%s\n\tFiles:\n",
		    trace_string(trace_global ? trace_global->tr_flags :
		    (flag_t) 0, (bits *) 0));

		TRACE_FILES(tfp) {

			(void)fprintf(fp,
			    "\t\t%s:\n\t\t\t%s  refcount %u "
			    "size %u  limit %u  files %u\n",
			    tfp->trf_file ? tfp->trf_file : "(stderr)",
			    tfp->trf_fd != -1 ? "opened" : "closed",
			    tfp->trf_refcount,
			    tfp->trf_size,
			    tfp->trf_limit_size,
			    tfp->trf_limit_files);

		} TRACE_FILES_END(tfp) ;

		(void)fprintf(fp, "\n");
    	}
    
	/* Task_dump dumps all protocols */
	task_dump(fp);

	(void)fprintf(fp, "\nDone\n");
    
	(void)fflush(fp);
	(void)fclose(fp);

	task_timer_peek();
    
	trace_log_tf(trace_global, 0, LOG_INFO, 
	    ("trace_do_dump: dump completed to %s", path_dump));
}

static void
trace_dump_done(task *tp)
{
    trace_dump_task = (task *) 0;
    task_delete(tp);
}

void
trace_dump(int now)
{

	if (trace_dump_task) {

		trace_log_tf(trace_global, 0, LOG_ERR, 
		    ("trace_dump: %s already active",
		    task_name(trace_dump_task)));

	} else if (now)
		trace_do_dump((task *) 0);
	else {

		trace_dump_task = task_alloc("TraceDump", TASKPRI_PROTO,
		    trace_set_global((bits *)0, (flag_t)0));

		task_set_child(trace_dump_task, trace_dump_done);
		task_set_process(trace_dump_task, trace_do_dump);

		if (!task_fork(trace_dump_task))
			task_quit(EINVAL);
    	}
}


void
trace_cleanup(void)
{

	if (trace_global
	    && trace_global->tr_file
	    && trace_global->tr_file->trf_fd != fileno(stderr))
		/* Release tracing */
		trace_freeup(trace_global);

#ifdef	DEBUG
	{
		trace_file *tfp;
	
		TRACE_FILES(tfp) {

			syslog(LOG_CRIT,
			    "trace_cleanup: file %s still open, refcount %d",
			    tfp->trf_file ? tfp->trf_file : "stderr",
			    tfp->trf_refcount);

		} TRACE_FILES_END(tfp) ;
	}
#endif	/* DEBUG */
}


/*
 * restart tracing
 */
void
trace_reinit(void)
{
	trace_file *tfp;

	if (trace_global && (!trace_global->tr_flags
	    || !trace_global->tr_refcount))
		/* No flags set, get rid of trace block */
		trace_global = trace_free(trace_global);
    
	TRACE_FILES(tfp) {

		if (tfp->trf_fd == -1)
			trace_on(tfp);

	} TRACE_FILES_END(tfp) ;
	
}

/*
 * initialize the trace structures
 */
void
trace_init(void)
{
	trace_file *trf;
    
	/* Set allocation index */
	trace_trace_index = task_block_init(sizeof (trace), "trace");
	trace_file_index = task_block_init(sizeof (trace_file), "trace_file");

	/* Start tracing to stderr */
	trace_global = trace_create();
	trace_global->tr_names = trace_global_types;

	trf = (trace_file *)task_block_alloc(trace_file_index);
	trf->trf_fd = -1;

	INSQUE(trf, trace_files.trf_back);
	trace_global->tr_file = trace_file_alloc(trf);
    
	trace_on(trace_global->tr_file);
}

/*
 * called from task.c
 */
void
trace_init2(void)
{

	if (!trace_global->tr_file->trf_file) {

		/* No trace file specified */
		if (trace_global->tr_flags)
			/*
			 * Tracing flags were specified, we trace
			 * to the console
			 */
			task_newstate(TASKS_NODETACH, (flag_t) 0);
		else
			/*
			 * No trace flags specifed, close tracing
			 * so we can daemonize
			 */
			trace_freeup(trace_global);

    	}
}

/*
 * do the actual output of trace data
 */
void
trace_trace(trace *trp, flag_t cf)
{
	char *dp, *sp;
	size_t tbsize, tsize, size;
	int fd;
	char time_buffer[LINE_MAX];

	dp = time_buffer;
	fd = trp->tr_file->trf_fd;

	if (!BIT_TEST(cf, TRC_NOSTAMP)) {

		sp = time_string;

		while (*sp)
		    *dp++ = *sp++;

		*dp++ = ' ';
	}

	if (task_mpid != task_pid) {

		*dp++ = '[';
		sp = task_pid_str;

		while (*sp)
		    *dp++ = *sp++;

		*dp++ = ']';
		*dp++ = ' ';
    	}

	*dp = (char)0;

	size = (tbsize = dp - time_buffer) +
	    (tsize = trace_ptr - trace_buffer) + 1;

	if (fd != fileno(stderr) &&
	    trp->tr_file->trf_limit_size &&
	    (trp->tr_file->trf_size +
	    (off_t)size) > trp->tr_file->trf_limit_size) {

		/* Time to rotate files */
	
		/* Inform our audience */
		if (tbsize)
			trace_write(trp->tr_file, time_buffer, tbsize);

		/* Rotate them around */
		trace_rotate(trp->tr_file);
		fd = trp->tr_file->trf_fd;
    	}

	trp->tr_file->trf_size += size;

	if (BIT_TEST(cf, TRC_NL_BEFORE)) {

		if (tbsize)
			trace_write(trp->tr_file, time_buffer, tbsize);

		trace_write(trp->tr_file, "\n", 1);
    	}

	if (tbsize)
		trace_write(trp->tr_file, time_buffer, tbsize);

	trace_write(trp->tr_file, trace_buffer, tsize);

	if (BIT_TEST(cf, TRC_NL_AFTER)) {

		trace_write(trp->tr_file, "\n", 1);

		if (tbsize)
			trace_write(trp->tr_file, time_buffer, tbsize);

    	}

	trace_write(trp->tr_file, "\n", 1);
}

void
trace_write(trace_file *tfp, const char *str, size_t len)
{
	int nw;

	if (tfp->trf_fd == -1)
		return;

	nw = write(tfp->trf_fd, str, len);

	if (nw < 0) {
		switch (errno) {
#if defined(EAGAIN) && EAGAIN != EWOULDBLOCK
		case EAGAIN:	/* System V Style */
#endif /* EAGAIN */
		case EWOULDBLOCK:
		default:
			if (!tfp->trf_blocked) {
				/* 
				 * note in the global trace that we are
				 * losing trace data
				 */
				snprintf(trace_errbuf, BUFSIZ,
				    "Cannot write to trace file %s! (%s)\n",
				    tfp->trf_file, strerror(errno));

				trace_write(trace_global->tr_file,
				    trace_errbuf, strlen(trace_errbuf));

				tfp->trf_blocked++;
			}
			break;
		}
	} else if (nw != len) {
		/* 
		 * incomplete write.  warn and get out.
		 * we could do more here.  the caller is attempting
		 * to write too much anyway.
		 */
		snprintf(trace_errbuf, BUFSIZ,
		    "Incomplete write to trace file %s! (%s)\n",
		    tfp->trf_file, strerror(errno));

		trace_write(trace_global->tr_file,
		    trace_errbuf, strlen(trace_errbuf));

	} else {
		/*
		 * if we were blocked before, reset it now
		 */
		if (tfp->trf_blocked) {

			snprintf(trace_errbuf, BUFSIZ,
			    "Tracing resumed to trace file %s"
			    " (some trace data may have been lost)\n",
			    tfp->trf_file);

			trace_write(trace_global->tr_file,
			    trace_errbuf, strlen(trace_errbuf));
		
			tfp->trf_blocked = 0;
		}
	}
}

/*
 * display trace options enabled
 */
void
trace_display(trace *trp, flag_t level)
{

	trace_tf(trp, level, TRC_NL_BEFORE|TRC_NL_AFTER,
	    ("Tracing flags enabled:%s%s %s",
	    trp->tr_control ? " " : "",
	    trace_bits(trace_control_types, trp->tr_control),
	    trace_string(trp->tr_flags, trp->tr_names)));
}


flag_t
trace_parse_packet(u_int detail, u_int inout, u_int indx)
{
	static flag_t flags[2][3][6] = {

	/* Packets */
	{
		/* Recv and Send */	     
		{ TR_PACKET, TR_PACKET_1, TR_PACKET_2,
		    TR_PACKET_3, TR_PACKET_4, TR_PACKET_5 },

		/* Recv */
		{ TR_PACKET_RECV, TR_PACKET_RECV_1, TR_PACKET_RECV_2,
		    TR_PACKET_RECV_3, TR_PACKET_RECV_4, TR_PACKET_RECV_5 },
		/* Send */
		{ TR_PACKET_SEND, TR_PACKET_SEND_1, TR_PACKET_SEND_2,
		    TR_PACKET_SEND_3, TR_PACKET_SEND_4, TR_PACKET_SEND_5 }

	},

	/* Detail */
	{
		/* Recv and Send */
		{ TR_DETAIL, TR_DETAIL_1, TR_DETAIL_2,
		    TR_DETAIL_3, TR_DETAIL_4, TR_DETAIL_5 },		
		/* Recv */
		{ TR_DETAIL_RECV, TR_DETAIL_RECV_1, TR_DETAIL_RECV_2,
		    TR_DETAIL_RECV_3, TR_DETAIL_RECV_4, TR_DETAIL_RECV_5 },
		/* Send */
		{ TR_DETAIL_SEND, TR_DETAIL_SEND_1, TR_DETAIL_SEND_2,
		    TR_DETAIL_SEND_3, TR_DETAIL_SEND_4, TR_DETAIL_SEND_5 }
	}

	};

	return (flags[detail][inout][indx]);
}


/*
 * prefill the trace buffer
 */
void
tracef(const char *fmt,...)
{
	va_list ap;

	va_start(ap, fmt);

	if (fmt && *fmt)
		trace_ptr += vsprintf(trace_ptr, fmt, ap);

	va_end(ap);
}
