/*
 *  Ball And Paddle
 *
 *  Copyright (C) 2007 by Eric Hutchins
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 3 of the License, or
 *  (at your option) any later version.
 *  
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *  
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **/

#include "game.h"

Game
  Game::m_instance;

void
Game::init ()
{
  paddle = NULL;

  score = 0;

  m_paused = false;

  m_gameover = false;
  m_transitioning = false;

  extraBallCount = 5;

  level = 1;

  double direction =
    PI / 4 + PI / 2 * ((double) rand () / ((double) (RAND_MAX) + (double) 1));
  balls.push_back (new Ball (251, 250, direction, imgBall));

  powerBalls = false;

  m_started = false;
}

void
Game::cleanup ()
{
  SDL_FreeSurface (imgBlocks);
  SDL_FreeSurface (imgBall);
  SDL_FreeSurface (imgLaser);
  SDL_FreeSurface (imgPowerUps);
}

void
Game::pause ()
{
  m_paused = true;
}

void
Game::resume ()
{
  m_paused = false;
}

void
Game::drawBox (SDL_Surface * dest, int x, int y, int width, int height,
	       Uint32 inside, Uint32 outside)
{
  SDL_Rect d = { x + 1, y + 1, width - 2, height - 2 };
  SDL_FillRect (dest, &d, inside);
  d.x = x;
  d.y = y;
  d.w = 1;
  d.h = height;
  SDL_FillRect (dest, &d, outside);
  d.x = x + width - 1;
  SDL_FillRect (dest, &d, outside);
  d.x = x;
  d.w = width;
  d.h = 1;
  SDL_FillRect (dest, &d, outside);
  d.y = y + height - 1;
  SDL_FillRect (dest, &d, outside);
}

void
Game::loadHighScores ()
{
  ifstream file;
#ifndef SINGLEDIR
  string location =
    string (getenv ("HOME")) + "/.ballandpaddle/" + levelsetFilename +
    "_scores";
#endif
#ifdef SINGLEDIR
  string location = levelsetFilename + "_scores";
#endif
  file.open (location.c_str (), ifstream::in);
  if (!file.fail ())
    {
      for (int i = 0; i < 5; i++)
	{
	  int buffer;
	  file >> buffer;
	  scores.push_back (buffer);
	}
      char buffer[50];
      // get garbage data on end of line (should be just a newline character)
      file.getline (buffer, 50);
      for (int i = 0; i < 5; i++)
	{
	  char strName[20];
	  file.getline (strName, 20);
	  names.push_back (string (strName));
	}
      file.close ();
    }
  else
    {
      for (int i = 0; i < 5; i++)
	scores.push_back (0);
      for (int i = 0; i < 5; i++)
	names.push_back ("0");
    }
}

void
Game::saveHighScores ()
{
  if (scores.size () < 5)
    {
      int size = scores.size ();
      for (int i = 0; i < 5 - size; i++)
	scores.push_back (0);
    }
  ofstream file;
#ifndef SINGLEDIR
  string location =
    string (getenv ("HOME")) + "/.ballandpaddle/" + levelsetFilename +
    "_scores";
#endif
#ifdef SINGLEDIR
  string location = levelsetFilename + "_scores";
#endif
  file.open (location.c_str ());
  for (int i = 0; i < 5; i++)
    {
      file << scores[i] << endl;
    }
  for (int i = 0; i < 5; i++)
    {
      file << names[i] << endl;
    }
  file.close ();
}

void
Game::endGame (bool won, GameStateManager * gameStateManager)
{
  loadHighScores ();
  scores.push_back (score);
  names.push_back ("");
  int scoreIndex = names.size () - 1;
  for (int i = 0; i < scores.size (); i++)
    {
      int highestIndex = i;
      for (int j = i; j < scores.size (); j++)
	if (scores[j] >= scores[highestIndex])
	  highestIndex = j;
      int temp = scores[i];
      string strTemp = names[i];
      scores[i] = scores[highestIndex];
      scores[highestIndex] = temp;
      names[i] = names[highestIndex];
      names[highestIndex] = strTemp;
      if (i == scoreIndex)
	scoreIndex = highestIndex;
      else if (highestIndex == scoreIndex)
	scoreIndex = i;
    }
  saveHighScores ();
  if (gameStateManager->isFullscreen ())
    SDL_ShowCursor (true);
  // if it's in the top 5, setup name input
  if (scoreIndex < 5)
    {
      gameStateManager->setScoreIndex (scoreIndex);
      gameStateManager->setEnteringHighScore (true);
    }
  gameStateManager->changeState (HighScoreMenu::instance ());
  HighScoreMenu::instance ()->loadHighScores (levelsetFilename);
}

void
Game::loadImages (GameStateManager * gameStateManager)
{
  levelsetFilename = gameStateManager->getLevelsetFilename ();

  string location = LEVELPATH + "/" + levelsetFilename + "/ball.png";
  SDL_Surface *temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgBall = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  SDL_SetColorKey (imgBall, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0x00ff00ff);

  location = LEVELPATH + "/" + levelsetFilename + "/paddle.png";
  temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgPaddle = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  SDL_SetColorKey (imgPaddle, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0x00ff00ff);

  location = LEVELPATH + "/" + levelsetFilename + "/laser.png";
  temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgLaser = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  SDL_SetColorKey (imgLaser, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0x00ff00ff);

  location = LEVELPATH + "/" + levelsetFilename + "/blocks.png";
  temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgBlocks = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);

  location = LEVELPATH + "/" + levelsetFilename + "/powerups.png";
  temp = IMG_Load (location.c_str ());
  if (temp != NULL)
    imgPowerUps = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);
  SDL_SetColorKey (imgPowerUps, SDL_SRCCOLORKEY | SDL_RLEACCEL, 0x00ff00ff);
}

bool
Game::loadLevel (int level)
{
  blocks = vector < vector < Block * > >();

  for (int i = 0; i < 16; i++)
    blocks.push_back (vector < Block * >());

  stringstream ss (stringstream::in | stringstream::out);
  ss << LEVELPATH << "/" << levelsetFilename;
  ss << "/level";
  ss << level;
  ss << ".lvl";

  ifstream file (ss.str ().c_str (), ifstream::in);

  if (!file.is_open ())
    {
      return false;
    }

  file >> bonusTime;

  for (int i = 0; i < 25; i++)
    {
      for (int j = 0; j < 16; j++)
	{
	  int type;
	  int strength;
	  file >> type;
	  file >> strength;
	  blocks[j].
	    push_back (new Block (j * 32, i * 16, type, strength, imgBlocks));
	}
    }

  file.close ();

  stringstream ss2 (stringstream::in | stringstream::out);
  ss2 << LEVELPATH << "/" << levelsetFilename;
  ss2 << "/level";
  ss2 << level;
  ss2 << ".png";

  SDL_Surface *temp = IMG_Load (ss2.str ().c_str ());
  if (temp != NULL)
    imgBackground = SDL_DisplayFormat (temp);
  SDL_FreeSurface (temp);

  balls = vector < Ball * >();

  double direction =
    PI / 4 + PI / 2 * ((double) rand () / ((double) (RAND_MAX) + (double) 1));
  balls.push_back (new Ball (251, 250, direction, imgBall));

  powerBalls = false;
  reverse = false;

  lasers = vector < Laser * >();

  if (paddle != NULL)
    delete paddle;
  paddle = new Paddle (imgPaddle);

  powerUps = vector < PowerUp * >();

  timeTilReady = 2700;

  tickingBonusTime = false;

  return true;
}

void
Game::draw (GameStateManager * gameStateManager)
{
  SDL_FillRect (gameStateManager->screen, NULL,
		SDL_MapRGB (gameStateManager->screen->format, 192, 64, 64));
  SDL_BlitSurface (gameStateManager->imgSkin, NULL, gameStateManager->screen,
		   NULL);


  SDL_Surface *buffer =
    SDL_CreateRGBSurface (SDL_HWSURFACE, 512, 400, 32, 0, 0, 0, 0);

  SDL_BlitSurface (imgBackground, NULL, buffer, NULL);

  for (int i = 0; i < lasers.size (); i++)
    lasers[i]->draw (buffer);

  paddle->draw (buffer);

  for (int i = 0; i < balls.size (); i++)
    balls[i]->draw (buffer, powerBalls);

  for (int i = 0; i < powerUps.size (); i++)
    powerUps[i]->draw (imgPowerUps, buffer);

  for (int i = 0; i < 16; i++)
    for (int j = 0; j < 25; j++)
      blocks[i][j]->draw (buffer);

  if (timeTilReady > 0)
    {
      drawBox (buffer, 120, 200, 250, 40,
	       SDL_MapRGB (buffer->format, 64, 64, 64),
	       SDL_MapRGB (buffer->format, 0, 0, 0));
      gameStateManager->drawString (FONT_WHITE, "Get Ready", 150, 205,
				    buffer);
      if (timeTilReady < 1800)
	gameStateManager->drawString (FONT_WHITE, ".", 294, 200, buffer);
      if (timeTilReady < 1200)
	gameStateManager->drawString (FONT_WHITE, ".", 310, 200, buffer);
      if (timeTilReady < 600)
	gameStateManager->drawString (FONT_WHITE, ".", 326, 200, buffer);
    }

  if (m_paused)
    {
      drawBox (buffer, 180, 200, 130, 40,
	       SDL_MapRGB (buffer->format, 64, 64, 64),
	       SDL_MapRGB (buffer->format, 0, 0, 0));
      gameStateManager->drawString (FONT_WHITE, "Paused", 190, 205, buffer);
    }

  SDL_Rect d = { 12, 12, 0, 0 };
  SDL_BlitSurface (buffer, NULL, gameStateManager->screen, &d);
  SDL_FreeSurface (buffer);

  // draw row of extra balls
  SDL_Rect ballRect = { 15, 427, 0, 0 };
  SDL_Rect s = { 0, 0, 10, 10 };
  for (int i = 0; i < extraBallCount; i++)
    {
      ballRect.x = 15 + i * 15;
      SDL_BlitSurface (imgBall, &s, gameStateManager->screen, &ballRect);
    }

  // draw the row of bonus time
  SDL_Rect bonusRect = { 12, 452, 512 * bonusTime / 100000, 16 };
  SDL_FillRect (gameStateManager->screen, &bonusRect,
		SDL_MapRGB (gameStateManager->screen->format, 192, 192, 0));

  // draw the level number
  stringstream ssLevel (stringstream::in | stringstream::out);
  ssLevel << level;
  string strLevel;
  int levelBuffer = 5 - ssLevel.str ().length ();
  if (levelBuffer < 0)
    levelBuffer = 0;
  for (int i = 0; i < levelBuffer; i++)
    strLevel += ' ';
  strLevel += ssLevel.str ();
  gameStateManager->drawString (FONT_WHITE, "Level: ", 534, 10);
  gameStateManager->drawString (FONT_WHITE, strLevel.c_str (), 546, 39);

  // draw the score
  stringstream ssScore (stringstream::in | stringstream::out);
  ssScore << score;
  string strScore;
  int scoreBuffer = 5 - ssScore.str ().length ();
  if (scoreBuffer < 0)
    scoreBuffer = 0;
  for (int i = 0; i < scoreBuffer; i++)
    strScore += ' ';
  strScore += ssScore.str ();
  gameStateManager->drawString (FONT_WHITE, "Score: ", 534, 72);
  gameStateManager->drawString (FONT_WHITE, strScore.c_str (), 546, 101);
}

void
Game::update (GameStateManager * gameStateManager)
{
  if (m_gameover)
    {
      score = gameStateManager->getScore ();
      endGame (true, gameStateManager);
      return;
    }
  if (!m_started)
    {
      if (gameStateManager->isFullscreen ())
	SDL_WarpMouse (268, 0);
      levelsetName = gameStateManager->getLevelsetName ();
      levelsetFilename = gameStateManager->getLevelsetFilename ();
      if (level == 1)
	level = gameStateManager->getStartLevel ();

      loadImages (gameStateManager);

      // load the level
      if (!loadLevel (level))
	{
	  m_gameover = true;
	  m_transitioning = true;
	  gameStateManager->setScore (score);
	  GameOver *gameover = GameOver::instance ();
	  gameover->setLevelsetFilename (gameStateManager->
					 getLevelsetFilename ());
	  gameover->init ();
	  gameover->setBallCount (extraBallCount);
	  gameStateManager->pushState (gameover);
	  //endGame(true, gameStateManager);
	  return;
	}
      m_started = true;
    }

  int time = gameStateManager->getTime ();

  // if ticking away bonus after level is over
  if (tickingBonusTime)
    {
      int prevScore = score;
      bonusTime -= time * 8;
      bonusScore += time;
      if (bonusTime < 0)
	{
	  // stop the ticking
	  gameStateManager->stopSound ();
	  // subtract back the overlapped time lost to the score
	  bonusScore += bonusTime / 8;
	  // move to next level
	  level++;
	  // reset mouse and level
	  m_started = false;
	  // reset the ticking bonus meter
	  tickingBonusTime = false;
	}
      score = oldScore + bonusScore / 4;
      if (score != 0 &&
	  (score / EXTRA_BALL_SCORE) > (prevScore / EXTRA_BALL_SCORE))
	extraBallCount++;
      return;
    }

  // do nothing if paused
  if (m_paused)
    return;

  // update the paddle
  paddle->update (time);

  // if waiting for ready, tick away
  if (timeTilReady > 0)
    {
      timeTilReady -= time;
      return;
    }

  // seed the random number generator for anything random during this frame
  srand (SDL_GetTicks ());

  // update the bonus time meter
  if (bonusTime > 0)
    bonusTime -= time;

  // update all balls
  for (int i = 0; i < balls.size (); i++)
    {
      int prevScore = score;
      balls[i]->update (time, blocks, paddle, powerUps, powerBalls, score,
			gameStateManager);
      if (score != prevScore
	  && (score / EXTRA_BALL_SCORE) > (prevScore / EXTRA_BALL_SCORE))
	extraBallCount++;
      if (balls[i]->getY () >= 400)
	{
	  delete balls[i];
	  balls.erase (balls.begin () + i);
	  i--;
	}
    }
  // update all lasers
  for (int i = 0; i < lasers.size (); i++)
    {
      int prevScore = score;
      lasers[i]->update (time, blocks, powerUps, score);
      if (score != prevScore &&
	  (score / EXTRA_BALL_SCORE) > (prevScore / EXTRA_BALL_SCORE))
	extraBallCount++;
      if (lasers[i]->getY () < -lasers[i]->getHeight ())
	{
	  delete lasers[i];
	  lasers.erase (lasers.begin () + i);
	  i--;
	}
    }
  // update all powerups
  for (int i = 0; i < powerUps.size (); i++)
    {
      powerUps[i]->update (time);
      if (powerUps[i]->getY () >= 400)
	{
	  delete powerUps[i];
	  powerUps.erase (powerUps.begin () + i);
	  i--;
	  continue;
	}
      if ((int) powerUps[i]->getX () <
	  (int) (paddle->getX () + paddle->getWidth ()) &&
	  (int) (powerUps[i]->getX () + powerUps[i]->getWidth ()) >=
	  (int) paddle->getX () &&
	  (int) powerUps[i]->getY () <
	  (int) (paddle->getY () + paddle->getHeight ()) &&
	  (int) (powerUps[i]->getY () + powerUps[i]->getHeight ()) >=
	  (int) paddle->getY ())
	{
	  int type = powerUps[i]->getType ();
	  handlePowerUp (type);
	  delete powerUps[i];
	  powerUps.erase (powerUps.begin () + i);
	  i--;
	  continue;
	}
    }
  // if there are no balls on the field, lose a life or game over
  if (balls.size () == 0)
    {
      extraBallCount--;
      if (extraBallCount >= 0)
	{
	  timeTilReady = 2700;

	  balls = vector < Ball * >();

	  double direction =
	    PI / 4 +
	    PI / 2 * ((double) rand () / ((double) (RAND_MAX) + (double) 1));
	  balls.push_back (new Ball (251, 250, direction, imgBall));

	  powerBalls = false;

	  powerUps = vector < PowerUp * >();
	}
      else
	{
	  m_gameover = true;
	  m_transitioning = true;
	  gameStateManager->setScore (score);
	  GameOver *gameover = GameOver::instance ();
	  gameover->init ();
	  gameover->setBallCount (extraBallCount);
	  gameStateManager->pushState (gameover);
	  //endGame(false, gameStateManager);
	  return;
	}
    }

  bool won = true;
  for (int i = 0; i < 16; i++)
    for (int j = 0; j < 25; j++)
      if (blocks[i][j]->isAlive () && blocks[i][j]->getType () < 8)
	won = false;
  if (won)
    {
      oldScore = score;
      bonusScore = 0;
      tickingBonusTime = true;
      // make sure all channels aren't being used by other sounds
      gameStateManager->stopSound ();
      // start ticking
      gameStateManager->loopSound ("bonustick");
    }
}

void
Game::handleEvents (GameStateManager * gameStateManager)
{
  if (!m_started)
    return;

  SDL_Event event;

  if (m_paused || tickingBonusTime)
    {
      while (SDL_PollEvent (&event))
	{
	  switch (event.type)
	    {
	    case SDL_QUIT:
	      gameStateManager->quit ();
	      break;
	    case SDL_KEYDOWN:
	      switch (event.key.keysym.sym)
		{
		case SDLK_ESCAPE:
		  endGame (false, gameStateManager);
		  break;

		case SDLK_p:
		  // p to pause
		  m_paused = !m_paused;
		  // if unpausing make sure the mouse is still in the same place (to avoid cheating)
		  if (!m_paused)
		    SDL_WarpMouse ((int)
				   (paddle->getX () +
				    paddle->getWidth () / 2 + 12), 0);
		  break;
		}
	      break;
	    }
	}
    }

  // if paused, don't do anything else
  if (m_paused || tickingBonusTime)
    return;

  int paddledirection = paddle->getDirection ();

  switch (paddledirection)
    {
    case PADDLE_LEFT:
      paddledirection = -1;
      break;
    case PADDLE_NONE:
      paddledirection = 0;
      break;
    case PADDLE_RIGHT:
      paddledirection = 1;
      break;
    }

  int paddleX;
  while (SDL_PollEvent (&event))
    {
      switch (event.type)
	{
	case SDL_QUIT:
	  gameStateManager->quit ();
	  break;
	case SDL_KEYDOWN:
	  switch (event.key.keysym.sym)
	    {
	    case SDLK_ESCAPE:
	      endGame (false, gameStateManager);
	      break;

	    case SDLK_p:
	      // p to pause
	      m_paused = !m_paused;
	      // if unpausing make sure the mouse is still in the same place (to avoid cheating)
	      if (!m_paused)
		SDL_WarpMouse ((int)
			       (paddle->getX () + paddle->getWidth () / 2 +
				12), 0);
	      break;
	    case SDLK_LEFT:
	      if (!reverse)
		paddledirection--;
	      else
		paddledirection++;
	      break;
	    case SDLK_RIGHT:
	      if (!reverse)
		paddledirection++;
	      else
		paddledirection--;
	      break;
	    case SDLK_SPACE:
	      if (timeTilReady > 0)
		break;
	      if (paddle->isLaser ())
		lasers.
		  push_back (new
			     Laser ((int) paddle->getX () +
				    paddle->getWidth () / 2 - 4,
				    (int) paddle->getY (), 8, 30, imgLaser));
	      else if (paddle->isGlue ())
		{
		  for (int i = 0; i < balls.size (); i++)
		    {
		      if (balls[i]->isAttached ())
			balls[i]->release ();
		    }
		}
	      break;
	    }
	  break;
	case SDL_KEYUP:
	  switch (event.key.keysym.sym)
	    {
	    case SDLK_LEFT:
	      if (!reverse)
		paddledirection++;
	      else
		paddledirection--;
	      break;
	    case SDLK_RIGHT:
	      if (!reverse)
		paddledirection--;
	      else
		paddledirection++;
	      break;
	    }
	  break;
	case SDL_MOUSEMOTION:
	  if (!reverse)
	    paddleX = event.motion.x - 12 - paddle->getWidth () / 2;
	  else
	    paddleX = 512 - (event.motion.x - 12) - paddle->getWidth () / 2;
	  if (paddleX < 0)
	    paddleX = 0;
	  if (paddleX > 512 - paddle->getWidth ())
	    paddleX = 512 - paddle->getWidth ();
	  paddle->setX (paddleX);

	  if (gameStateManager->isFullscreen ())
	    {
	      if (event.motion.x < 12 + paddle->getWidth () / 2)
		{
		  paddleX = 0;
		  SDL_WarpMouse (paddleX + 12 + paddle->getWidth () / 2,
				 event.motion.y);
		}
	      if (event.motion.x > 524 - paddle->getWidth () / 2)
		{
		  paddleX = 512 - paddle->getWidth ();
		  SDL_WarpMouse (paddleX + 12 + paddle->getWidth () / 2,
				 event.motion.y);
		}
	    }
	  break;
	case SDL_MOUSEBUTTONDOWN:
	  if (timeTilReady > 0)
	    break;
	  if (event.button.button == SDL_BUTTON_LEFT)
	    {
	      if (paddle->isLaser ())
		{
		  lasers.
		    push_back (new
			       Laser ((int) paddle->getX () +
				      paddle->getWidth () / 2 - 4,
				      (int) paddle->getY (), 8, 30,
				      imgLaser));
		  gameStateManager->playSound ("laser");
		}
	      else if (paddle->isGlue ())
		{
		  for (int i = 0; i < balls.size (); i++)
		    {
		      if (balls[i]->isAttached ())
			balls[i]->release ();
		    }
		}
	    }
	}
    }

  switch (paddledirection)
    {
    case -1:
      paddle->setDirection (PADDLE_LEFT);
      break;
    case 0:
      paddle->setDirection (PADDLE_NONE);
      break;
    case 1:
      paddle->setDirection (PADDLE_RIGHT);
      break;
    }

  // update position of balls to be aligned new paddle position
  if (paddle->isGlue ())
    {
      for (int i = 0; i < balls.size (); i++)
	{
	  if (balls[i]->isAttached ())
	    balls[i]->update (0, blocks, paddle, powerUps, false, score,
			      gameStateManager);
	}
    }
}

void
Game::handlePowerUp (int type)
{
  int n;
  double speed;
  int mouseX;
  switch (type)
    {
    case 0:
      paddle->stretch ();
      break;
    case 1:
      paddle->shrink ();
      break;
    case 2:
      n = balls.size ();
      if (balls.size () >= 28)
	break;
      for (int i = 0; i < n; i++)
	{
	  double direction = balls[i]->getDirection ();
	  double x = balls[i]->getX ();
	  double y = balls[i]->getY ();

	  balls.push_back (new Ball ((int) x,
				     (int) y, direction - PI / 12, imgBall));
	  if (balls[i]->isAttached ())
	    balls.back ()->setAttached (paddle);
	  balls.back ()->setSpeed (balls[i]->getSpeed ());

	  balls.push_back (new Ball ((int) x,
				     (int) y, direction + PI / 12, imgBall));
	  if (balls[i]->isAttached ())
	    balls.back ()->setAttached (paddle);
	  balls.back ()->setSpeed (balls[i]->getSpeed ());
	}
      break;
    case 3:
      paddle->setLaser ();
      for (int i = 0; i < balls.size (); i++)
	if (balls[i]->isAttached ())
	  balls[i]->release ();
      break;
    case 4:
      paddle->setGlue ();
      break;
    case 5:
      powerBalls = true;
      break;
    case 6:
      reverse = !reverse;
      mouseX =
	524 - paddle->getWidth () / 2 - (mouseX - 12 -
					 paddle->getWidth () / 2);
      SDL_WarpMouse (mouseX, 0);
      break;
    case 7:
      speed = balls[0]->getSpeed ();
      if (speed < 0.25)
	for (int i = 0; i < balls.size (); i++)
	  balls[i]->setSpeed (0.25);
      else if (speed < 0.30)
	for (int i = 0; i < balls.size (); i++)
	  balls[i]->setSpeed (0.30);
      break;
    case 8:
      speed = balls[0]->getSpeed ();
      if (speed > 0.25)
	for (int i = 0; i < balls.size (); i++)
	  balls[i]->setSpeed (0.20);
      if (speed > 0.20)
	for (int i = 0; i < balls.size (); i++)
	  balls[i]->setSpeed (0.20);
      break;
    }
}
