/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import { NotificationType, TalerCorebankApiClient, WalletNotification } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  BankService,
  ExchangeService,
  GlobalTestState,
  WalletClient,
  WalletService,
  generateRandomPayto,
  setupDb,
} from "../harness/harness.js";
import { withdrawViaBankV3 } from "../harness/helpers.js";

export async function runWalletObservabilityTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    allowRegistrations: true,
    currency: "TESTKUDOS",
    database: db.connStr,
    httpPort: 8082,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  let receiverName = "Exchange";
  let exchangeBankUsername = "exchange";
  let exchangeBankPassword = "mypw";
  let exchangePaytoUri = generateRandomPayto(exchangeBankUsername);

  await exchange.addBankAccount("1", {
    accountName: exchangeBankUsername,
    accountPassword: exchangeBankPassword,
    wireGatewayApiBaseUrl: new URL("accounts/exchange/taler-wire-gateway/", bank.baseUrl).href,
    accountPaytoUri: exchangePaytoUri,
  });

  bank.setSuggestedExchange(exchange, exchangePaytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "adminpw",
    },
  });

  await bankClient.registerAccountExtended({
    name: receiverName,
    password: exchangeBankPassword,
    username: exchangeBankUsername,
    is_taler_exchange: true,
    payto_uri: exchangePaytoUri,
  });

  const coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS"));
  exchange.addCoinConfigList(coinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  const walletService = new WalletService(t, {
    name: "wallet",
    useInMemoryDb: true,
  });
  await walletService.start();
  await walletService.pingUntilAvailable();

  const allNotifications: WalletNotification[] = [];

  const walletClient = new WalletClient({
    name: "wallet",
    unixPath: walletService.socketPath,
    onNotification(n) {
      console.log("got notification", n);
      allNotifications.push(n);
    },
  });
  await walletClient.connect();
  await walletClient.client.call(WalletApiOperation.InitWallet, {
    config: {
      testing: {
        skipDefaults: true,
        emitObservabilityEvents: true,
      },
    },
  });

  const wres = await withdrawViaBankV3(t, {
    amount: "TESTKUDOS:10",
    bankClient,
    exchange,
    walletClient,
  });
  await wres.withdrawalFinishedCond;

  const requestObsEvents = allNotifications.filter(
    (x) => x.type === NotificationType.RequestObservabilityEvent,
  );
  const taskObsEvents = allNotifications.filter(
    (x) => x.type === NotificationType.TaskObservabilityEvent,
  );

  console.log(`req events: ${requestObsEvents.length}`);
  console.log(`task events: ${taskObsEvents.length}`);

  t.assertTrue(requestObsEvents.length > 30);
  t.assertTrue(taskObsEvents.length > 30);
}

runWalletObservabilityTest.suites = ["wallet"];
