//
// Part of the ht://Dig package   <http://www.htdig.org/>
// Copyright (c) 1999, 2000 The ht://Dig Group
// For copyright details, see the file COPYING in your distribution
// or the GNU General Public License version 2 or later
// <http://www.gnu.org/copyleft/gpl.html>
//
// $Id: WordSearch.h,v 1.1 2000/10/24 14:23:29 loic Exp $
//
// ************************* WordSearch implementation ********************
//
// NAME
//
// Solve a query from a WordTree syntax tree
//
// SYNOPSIS
//
// #include <WordSearch.h>
//
// WordTree* expr = get_query();
// WordSearch search;
// search.limit_count = NUMBER_OF_RESULTS;
// WordMatch* search.Search(expr);
// ...
//  
// DESCRIPTION
//
// The WordSearch class is a wrapper to query an inverted index
// using a WordTree syntax tree. 
// 
// END
//

#ifndef _WordSearch_h
#define _WordSearch_h

#include <WordList.h>
#include <WordMatch.h>
#include <WordTree.h>

class WordSearch {
public:
  WordSearch(WordList* words);

  ~WordSearch() {
    delete expr;
  }

  //-
  // Perform a search from the <b>expr</b> specifications.
  // Restore the context from <i>context_in</i> on <b>expr</b>.
  // Then skip (using WalkNext) <i>limit_bottom</i> entries.
  // Then collect in a WordMatch array of size <i>limit_count</i>
  // each match returned by WalkNext. When finished store
  // the context (ContextSave) in <i>context_out</i>.
  // It is the responsibility of the caller to free the WordMatch
  // array. If no match are found a null pointer is returned.
  //
  WordMatch **Search();

  //-
  // Search backend, only run the WalkNext loop but does not
  // allocate/deallocate data.
  //
  int SearchLoop(WordTree *expr);
  //-
  // Search backend, only run the WalkNext loop but does not
  // allocate/deallocate data. If limit_bottom is above all matches
  // return the last valid limit_count range and reset limit_bottom 
  // accordingly.
  //
  int SearchLoopBounded(WordTree *expr);

  //
  // Return a context description string to resume the
  // search at a given point.
  //
  const String& Context() const { return context_out; }

  inline int Verbose(int verbosity) { return verbose = verbosity; }

  //
  // Internal
  //
  WordList* words;
  int verbose;

  //
  // Input/Output
  //
  //
  // Input: Index of the first match to return, relative to context_in
  //        position.
  // Output: If bounded set to 1 contains the index of the effective
  //         match range returned.
  //
  unsigned int limit_bottom;

  //
  // Input
  //
  //
  // Maximum number of matches returned
  //
  unsigned int limit_count;
  //
  // Initial position in the index file. If the position contains a match
  // it won't be returned. To be accurate we should say that context_in
  // is used to move the cursor immediately after the specified position.
  //
  String context_in;
  //
  // Query tree
  //
  WordTree* expr;
  //
  // If limit_bottom is past the last matches, return the last valid
  // range instead of nothing.
  //
  int bounded;
  
  //
  // Output
  //
  //
  // Array of at most limit_count matches. The number of valid elements in
  // the array is matches_length;
  //
  WordMatch** matches;
  //
  // Effective allocated size of the array
  //
  unsigned int matches_size;
  //
  // Number of valid elements in the matches array.
  //
  unsigned int matches_length;
  //
  // Estimated number of matches.
  //
  unsigned int matches_total;
  //
  // String representation of the position in the index after retrieving
  // the last match. May be stored in the context_in data member before
  // searching to re-start at this position.
  //
  String context_out;
};

#endif /* _WordSearch_h */
