/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  Configuration,
  Logger,
  TransactionMajorState,
  TransactionMinorState,
} from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import {
  configureCommonKyc,
  createKycTestkudosEnvironment,
  postAmlDecisionNoRules,
  withdrawViaBankV3,
} from "../harness/environments.js";
import { GlobalTestState } from "../harness/harness.js";

const logger = new Logger("test-kyc-deposit-deposit.ts");

function adjustExchangeConfig(config: Configuration) {
  configureCommonKyc(config);

  config.setString("KYC-RULE-R1", "operation_type", "deposit");
  config.setString("KYC-RULE-R1", "enabled", "yes");
  config.setString("KYC-RULE-R1", "exposed", "yes");
  config.setString("KYC-RULE-R1", "is_and_combinator", "yes");
  config.setString("KYC-RULE-R1", "threshold", "TESTKUDOS:5");
  config.setString("KYC-RULE-R1", "timeframe", "1d");
  config.setString("KYC-RULE-R1", "next_measures", "M1");

  config.setString("KYC-MEASURE-M1", "check_name", "C1");
  config.setString("KYC-MEASURE-M1", "context", "{}");
  config.setString("KYC-MEASURE-M1", "program", "NONE");

  config.setString("KYC-CHECK-C1", "type", "INFO");
  config.setString("KYC-CHECK-C1", "description", "my check!");
  config.setString("KYC-CHECK-C1", "fallback", "FREEZE");
}

export async function runKycDepositDepositTest(t: GlobalTestState) {
  // Set up test environment

  const { walletClient, bankClient, exchange, amlKeypair } =
    await createKycTestkudosEnvironment(t, { adjustExchangeConfig });

  // Withdraw digital cash into the wallet.

  const wres = await withdrawViaBankV3(t, {
    bankClient,
    amount: "TESTKUDOS:50",
    exchange: exchange,
    walletClient: walletClient,
  });

  await wres.withdrawalFinishedCond;

  const depositResp = await walletClient.call(
    WalletApiOperation.CreateDepositGroup,
    {
      amount: "TESTKUDOS:10",
      depositPaytoUri: wres.accountPaytoUri,
    },
  );

  console.log("waiting for kyc-required");

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: {
      major: TransactionMajorState.Pending,
      minor: TransactionMinorState.KycRequired,
    },
  });

  const txDetails = await walletClient.call(
    WalletApiOperation.GetTransactionById,
    {
      transactionId: depositResp.transactionId,
    },
  );

  const kycPaytoHash = txDetails.kycPaytoHash;

  t.assertTrue(!!kycPaytoHash);

  await postAmlDecisionNoRules(t, {
    amlPriv: amlKeypair.priv,
    amlPub: amlKeypair.pub,
    exchangeBaseUrl: exchange.baseUrl,
    paytoHash: kycPaytoHash,
  });

  logger.info(`made decision to have no rules on ${kycPaytoHash}`);

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: [
      {
        major: TransactionMajorState.Done,
      },
      {
        major: TransactionMajorState.Finalizing,
        minor: TransactionMinorState.Track,
      },
    ],
  });

  await exchange.runAggregatorOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60,
  });

  await exchange.runTransferOnceWithTimetravel({
    timetravelMicroseconds: 1000 * 1000 * 60 * 60,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionState, {
    transactionId: depositResp.transactionId,
    txState: {
      major: TransactionMajorState.Done,
    },
  });
}

runKycDepositDepositTest.suites = ["wallet"];
