/* macrofuncs.c: -*- C -*-  Functions which allow the definition of
   Meta-HTML functions and macros. */

/*  Copyright (c) 1997 Brian J. Fox
    Author: Brian J. Fox (bfox@ai.mit.edu) Sat Jun 21 12:02:48 1997.

   This file is part of <Meta-HTML>(tm), a system for the rapid deployment
   of Internet and Intranet applications via the use of the Meta-HTML
   language.

   Copyright (c) 1995, 1996, Brian J. Fox (bfox@ai.mit.edu).
   Copyright (c) 1996, Universal Access Inc. (http://www.ua.com).

   Meta-HTML is free software; you can redistribute it and/or modify
   it under the terms of the UAI Free Software License as published
   by Universal Access Inc.; either version 1, or (at your option) any
   later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   UAI Free Software License for more details.

   You should have received a copy of the UAI Free Software License
   along with this program; if you have not, you may obtain one by
   writing to:

   Universal Access Inc.
   129 El Paseo Court
   Santa Barbara, CA
   93101  */

#include "language.h"

#if defined (__cplusplus)
extern "C"
{
#endif

static void pf_define_container (PFunArgs);
static void pf_define_tag (PFunArgs);
static void pf_define_function (PFunArgs);
static void pf_defweakmacro (PFunArgs);
static void pf_undef (PFunArgs);
static void pf_function_def (PFunArgs);

/************************************************************/
/*							    */
/*		   Macro Manipulation Functions		    */
/*							    */
/************************************************************/

static PFunDesc func_table[] =
{
  { "DEFINE-CONTAINER",	1, 0, pf_define_container },
  { "DEFINE-TAG",	1, 0, pf_define_tag },
  { "DEFINE-FUNCTION",	1, 0, pf_define_function },
  { "DEFMACRO",		1, 0, pf_define_container },
  { "DEFSUBST",		1, 0, pf_define_tag },
  { "DEFUN",		1, 0, pf_define_function },
  { "DEFWEAKMACRO",	1, 0, pf_defweakmacro },
  { "UNDEF",		0, 0, pf_undef },
  { "FUNCTION-DEF",	0, 0, pf_function_def },

  { (char *)NULL,	0, 0, (PFunHandler *)NULL }
};

PACKAGE_INITIALIZER (initialize_macro_functions)
DOC_SECTION (MACRO-COMMANDS)

DEFMACROX (pf_defsubst, name &optional named-parameters &key
	  package=packname whitespace=delete,
"A synonym for <funref macro-commands define-tag>.")

DEFMACRO (pf_define_tag, name &optional named-parameters &key
	  package=packname whitespace=delete,
"Define <var name> as a simple tag.  Within <var body>, the values of
<code>%0...%9</code> are defined to be the positional arguments that
were found in the opening tag of the invocation, and
<code>%body</code> is all of that material in a single string.

If any <var named-parameter>s are supplied, the values that were
passed in the opening tag are evaluated and bound to the named
parameters.

A keyword argument of <var package-name> wraps the entire body of the
macro in an <funref packages in-package> statement.

The keyword argument <var whitespace> can be set to the string
<code>delete</code> to remove whitespace from the starts and ends of
lines in the subst definition before it is stored.  This effectively
concatenates all of the lines of the subst definition into a single
long line.

Also see <funref macro-commands define-function> and <funref
macro-commands define-container>.")
{
  char *temp = get_positional_arg (vars, 0);
  char *subst_name = mhtml_evaluate_string (temp);
  char *subst_body = body ? body->buffer : "";

  if (!empty_string_p (subst_name))
    mhtml_add_user_function (user_SUBST, subst_name, subst_body, vars);

  if (subst_name && subst_name != temp)
    free (subst_name);
}

DEFMACROX (pf_defmacro, name &optional named-parameters
	  &key package=packname whitespace=delete,
"A synonym for <funref macro-commands define-container>.")

DEFMACRO (pf_define_container, name &optional named-parameters
	  &key package=packname whitespace=delete,
 "Define <var name> as a complex tag. At invocation time, various
substitutions are made within <var body>.  Specifically, if the text
string is:

<ul>
<li><b>%0</b>,<b>%1</b>, and so on, upto <b>%9</b> are replaced
with the exact text of the positional arguments that were found in
the opening tag of the invocation

<li><b>%attributes</b> is replaced by all of the arguments which
appeared in the opening tag.

<li><b>%body</b> is replaced with the exact text of the material
that appeared between the opening and closing tags

<li><b>%qbody</b> is similar to <b>%body</b>, but the string is
first surrounded by double quotes, and double quote characters which
appear in the string are escaped.

<li><b>%xbody</b> is replaced with the evaluation of the material
that appeared between the opening and closing tags
</ul>

If any <var named-parameter>s are supplied, the values that were
passed in the opening tag are evaluated and bound to the named
parameters.

A keyword argument of <var package-name> wraps the entire body of the
macro in an <funref packages in-package> statement.

The keyword argument <var whitespace> can be set to the string
<code>delete</code> to remove whitespace from the starts and ends of
lines in the macro definition before it is stored.  This effectively
concatenates all of the lines of the macro definition into a single
long line.")
{
  char *temp = get_positional_arg (vars, 0);
  char *subst_name = mhtml_evaluate_string (temp);
  char *subst_body = body ? body->buffer : "";

  if (!empty_string_p (subst_name))
    mhtml_add_user_function (user_MACRO, subst_name, subst_body, vars);

  if (subst_name && subst_name != temp)
    free (subst_name);
}

DEFMACRO (pf_defweakmacro, name  &optional named-parameters
	  &key package=packname whitespace=delete,
"<code>defweakmacro</code> is exactly like <funref macro-commands
define-container>, with one exception: at invocation time, the closing
tag does not have to be present -- in that case, the invocation is
treated as if the definition were a <funref macro-commands defsubst>.

This facility exists primarily to allow the redefinition of standard
HTML constructs which allow the closing tag to be missing, and yet,
still inexplicably operate correctly.

For example, the <example code><p></example> tag is often used without
its closing counterpart of <example code></p></example>.  If you
wished to redefine <example code><p></example> to do something special
when a closing tag was found, you might write the following
definition:

<example>
<defweakmacro p>
  <verbatim><P></verbatim>
  <when %qbody> Look ma! %body See? </when>
  <verbatim></P></verbatim>
</defweakmacro>
</example>

then, a simple <example code><P></example> would produce
<example code><P></P></example code>, while a complex invocation, such as:
<example>
<P> this is a list </P>
</example>
produces
<example>
  <P> Look ma!  this is a list See? </P>
</example>")
{
  char *temp = get_positional_arg (vars, 0);
  char *subst_name = mhtml_evaluate_string (temp);
  char *subst_body = body ? body->buffer : "";

  if (!empty_string_p (subst_name))
    {
      UserFunction *uf;
      mhtml_add_user_function (user_MACRO, subst_name, subst_body, vars);
      uf = mhtml_find_user_function (subst_name);
      uf->flags |= user_WEAK_MACRO;
    }

  if (subst_name && subst_name != temp)
    free (subst_name);
}

DEFMACROX (pf_defun, name  &optional named-parameters
	  &key package=packname whitespace=delete,
"A synonym for <funref macro-commands define-function>.")

DEFMACRO (pf_define_function, name  &optional named-parameters
	  &key package=packname whitespace=delete,
"Define <var name> as a simple tag.

The only differences between <funref macro-commands define-function>
and <funref macro-commands define-tag> are:
<ol>
<li> The <i>whitespace=delete</i> option is assumed.
<li> The <var named-parameter>s are evaluated in the context of the
caller, not of the definition of the defun.
<li> By default, a local package is wrapped around the invocation of
the defined function.  This can be changed by the use of the <var
package=packname> keyword.
</ol>

<example>
<define-function factorial num>
   <if <lt num 2> 1
      <mul num <factorial <sub num 1>>>>
</define-function>
.blank
<factorial 5> --> 120
</example>")
{
  char *temp = get_positional_arg (vars, 0);
  char *subst_name = mhtml_evaluate_string (temp);
  char *subst_body = body ? body->buffer : "";

  if (!empty_string_p (subst_name))
    mhtml_add_user_function (user_DEFUN, subst_name, subst_body, vars);

  if (subst_name && subst_name != temp)
    free (subst_name);
}

DEFUN (pf_undef, &optional name...,
"Remove the definition of a user-defined <funref macros defun>,
<funref macros defmacro> or <funref macros defsubst>.  For every <var
name> that has been defined in this way, the definition is removed.")
{
  register int i;
  char *name;

  if (!mhtml_user_keywords)
    return;

  for (i = 0; (name = get_positional_arg (vars, i)) != (char *)NULL; i++)
    {
      char *varname = name;

      varname = mhtml_evaluate_string (name);

      if (varname)
	{
	  UserFunction *uf = mhtml_find_user_function (varname);
	  Symbol *sym;

	  if (uf)
	    {
	      free (uf->body);
	      free (uf->name);
	      free (uf);
	    }

	  sym = symbol_remove_in_package (mhtml_user_keywords, varname);

	  if (sym)
	    {
	      sym->values= (char **)NULL;
	      sym->values_index = 0;
	      symbol_free (sym);
	    }

	  free (varname);
	}
    }
}

DEFUN (pf_function_def, funname,
"Return a human readable rendition of the function named by <var funname>.")
{
  if (mhtml_user_keywords == (Package *)NULL)
    return;
  else
    {
      register int i;
      char *name;
    
      for (i = 0; (name = get_positional_arg (vars, i)) != (char *)NULL; i++)
	{
	  char *varname = mhtml_evaluate_string (name);

	  if (varname != (char *)NULL)
	    {
	      UserFunction *uf = mhtml_find_user_function (varname);

	      if (uf)
		{
		  BPRINTF_BUFFER *insertion = bprintf_create_buffer ();

		  if (uf->type == user_MACRO)
		    {
		      if (uf->flags & user_WEAK_MACRO)
			bprintf (insertion, "<defweakmacro %s", uf->name);
		      else
			bprintf (insertion, "<define-container %s", uf->name);
		    }
		  else if (uf->type == user_DEFUN)
		    bprintf (insertion, "<define-function %s", uf->name);
		  else if (uf->type == user_SUBST)
		    bprintf (insertion, "<define-tag %s", uf->name);

		  /* Any named parameters? */
		  if (uf->named_parameters != (char **)NULL)
		    {
		      register int j;

		      for (j = 0; uf->named_parameters[j] != (char *)NULL; j++)
			bprintf (insertion, " %s", uf->named_parameters[j]);
		    }

		  /* Any special package name? */
		  if (uf->packname != (char *)NULL)
		    {
		      if ((uf->type != user_DEFUN) ||
			  (strcmp (uf->packname, "local") != 0))
			bprintf (insertion, " package=%s", uf->packname);
		    }

		  /* Any CR's in body? */
		  if ((uf->type != user_DEFUN) &&
		      (uf->flags & user_WHITESPACE_DELETED))
		    bprintf (insertion, " whitespace=delete>\n");
		  else
		    bprintf (insertion, ">\n");

		  bprintf (insertion, "%s", uf->body);

		  bprintf (insertion, "\n");

		  if (uf->type == user_MACRO)
		    {
		      if (uf->flags & user_WEAK_MACRO)
			bprintf (insertion, "</defweakmacro>");
		      else
			bprintf (insertion, "</define-container>");
		    }
		  else if (uf->type == user_DEFUN)
		    bprintf (insertion, "</define-function>");
		  else if (uf->type == user_SUBST)
		    bprintf (insertion, "</define-tag>");

		  bprintf_insert (page, start, "%s\n", insertion->buffer);
		  start += insertion->bindex + 1;
		  bprintf_free_buffer (insertion);
		}

	      free (varname);
	    }
	}

      *newstart = start;
    }
}

#if defined (__cplusplus)
}
#endif
