;; Calculator for GNU Emacs version 1.07
;; Copyright (C) 1990 Dave Gillespie

;; This file is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.


;;;; [calc.el]

;;; This is part I of the Emacs Calculator.  It defines simple arithmetic
;;; commands only.  Assuming the autoload commands shown below have been
;;; done, the Calculator will autoload the remaining commands from calc-ext.elc
;;; whenever one is first needed.  If you wish, you can concatenate calc-ext
;;; onto the end of calc to make one big file (.el or .elc).  On the other
;;; hand, you may wish to use calc-install to split Calc into many small,
;;; fast-loading files.

;;; Suggested usage:
;;;
;;;   (autoload 'calc ".../calc.elc" "Calculator Mode" t nil)
;;;   (autoload 'quick-calc ".../calc.elc" "Quick Calculator" t nil)
;;;   (autoload 'full-calc ".../calc.elc" "Full-screen Calculator" t nil)
;;;   (autoload 'calc-keypad ".../calc.elc" "X windows Calculator" t nil)
;;;   (autoload 'calc-eval ".../calc.elc" "Call Calculator" nil nil)
;;;   (autoload 'calc-grab-region ".../calc-ext.elc" nil t nil)
;;;   (autoload 'defmath ".../calc-ext.elc" nil t t)
;;;   (autoload 'calc-extensions ".../calc-ext.elc" nil nil nil)
;;;   (global-set-key "\e#" 'calc)
;;;   `M-x calc' or `M-#' to start.
;;;
;;; where ".../calc.elc" represents the full path for "calc.elc",
;;; and ".../calc-ext.elc" is the path of the companion file containing
;;; all of the more advanced Calc commands.


;;; Author's address:
;;;   Dave Gillespie, 256-80 Caltech, Pasadena CA 91125.
;;;   daveg@csvax.cs.caltech.edu, ...!cit-vax!daveg.
;;;
;;; Note that csvax's name has changed slightly.  The old name,
;;; csvax.caltech.edu, will also still work for a while.
;;;
;;; This file and the manual, calc.texinfo, are available from anonymous FTP
;;; on csvax.cs.caltech.edu [131.215.131.131]; look in ~ftp/pub/calc*.
;;;
;;; Bug reports and suggestions are always welcome!


;;; All functions, macros, and Lisp variables defined here begin with one
;;; of the prefixes "math", "Math", or "calc", with the exceptions of
;;; "full-calc", "another-calc", "quick-calc", "report-calc-bug",
;;; and "defmath".  User-accessible variables begin with "var-".


;;; Version 1.07:
;;;  * Added `m F' (calc-settings-file-name) command.
;;;  * Added calc-autoload-directory variable.
;;;  * Extended Z ` to accept a prefix argument.
;;;  * Added keystrokes (v h, v k) for head, tail, cons.
;;;  * Extended `v e' to accept a vector as the filler.
;;;  * Changed `V M', `V R' to accept mapping-mode keys in uppercase, too.
;;;  * Changed V M ' etc. to accept $, $$, ... as argument indicators.
;;;  * Changed `t y' to accept a prefix argument.
;;;  * Put in a cleaner and safer random number generator for `k r' et al.
;;;  * Fixed a bug which completely broke `a r' command!
;;;  * Fixed "0 * matrix" to generate a zero matrix instead of 0.
;;;  * Fixed a bug in `a R' which sometimes caused it to crash.
;;;  * Fixed a fatal typo in the TeX version of the manual.
;;;  * Fixed a bug that prevented C-k, C-w, M-w from working in Trail buffer.
;;;  * Fixed another bug in `Z P' command.
;;;  * Fixed a bug in `u s' which incorrectly simplified subtractions.
;;;  * Fixed an argument-name aliasing bug evaluating lambda( ) formulas.
;;;  * Fixed overfull hboxes in the manual.
;;;  * Fixed various other bugs in the manual.
;;; 
;;; Version 1.06:
;;;  * Added "calc-keypad" mode for X window system users (try it!).
;;;  * Improved "calc-eval" for calling/operating Calc from user-written Lisp.
;;;  * Moved vector accumulate command to `V U' (old `H V R' still supported).
;;;  * Added right-to-left reductions: `I V R' and `I V U'.
;;;  * Added set operations on vectors: intersect, union, diff, xor.
;;;  * Added `I v s' to remove a subvector from a vector.
;;;  * Introduced `H |' to append two vectors with no magical special cases.
;;;  * Introduced rhead, rtail, and rcons for isolating last vector element.
;;;  * Changed `g p' to keep temp files around until data actually change.
;;;  * Improved `a S' to solve many higher-order polynomial equations.
;;;  * Added `a P' to produce a vector of all solutions to an equation.
;;;  * Enhanced `a v' and `j v' to allow top-level-only evaluation.
;;;  * Changed `j DEL' to delete a side of an eqn or ineq, leaving other side.
;;;  * Fixed binding for keys `j 1' through `j 9'.
;;;  * Introduced "let" marker in rewrite rules.
;;;  * Enhanced the "sign" function to provide a two-argument version.
;;;  * Changed "max-specpdl-size exceeded" error message to be user-friendly.
;;;  * Put "<Aborted>" in the trail in above case and when user presses C-g.
;;;  * Changed TeX mode to generate \ldots instead of \dots, recognize both.
;;;  * Changed "sin(0)" etc. (for integer 0) to generate "0" instead of "0.".
;;;  * Enhanced Programming Tutorial exercise 2.
;;;  * Fixed an error in the answer to Types Tutorial exercise 3.
;;;  * Fixed several bugs relating to head, tail, and cons functions.
;;;  * Fixed some other minor typos in the manual.
;;;  * Fixed several bugs in `Z P' (calc-user-define-permanent).
;;;  * Fixed several bugs that broke the `g P' command.
;;; 
;;; Version 1.05:
;;;  * Created a calc-install command to ease installation.
;;;  * Added lots of exercises to the Tutorial section of the manual.
;;;  * Added ability to select and operate on sub-formulas.
;;;  * Substantially improved the algebraic rewrite-rule system.
;;;  * Added a set of graphing commands that use GNUPLOT.
;;;  * Added a command (`a R') for finding numerical roots to equations.
;;;  * Added several new math functions, such as erf and Bessel functions.
;;;  * Added key bindings for miscellaneous commands using the "f" prefix key.
;;;  * Added lots of new vector operations, many of them in the spirit of APL.
;;;  * Added more control over vector display, including an abbreviated mode.
;;;  * Improved keyboard macro editing; added read-kbd-macro to macedit.el.
;;;  * Introduced the `m S' (calc-shift-prefix) command.
;;;  * Enhanced the calc-edit command in several ways.
;;;  * Made it possible to hit ` (calc-edit) during numeric/algebraic entry.
;;;  * Enhanced the calc-solve-for command to handle inequalities.
;;;  * Enhanced calc-simplify to handle equations and inequalities.
;;;  * Taught log10 and log to look for exact integer or rational results.
;;;  * Added ability to take Nth roots directly.
;;;  * Added "increment" and "decrement" commands for integers and floats.
;;;  * Added "full-help" command, changed "h" key to invoke it.
;;;  * Added special help for Inverse and Hyperbolic prefixes.
;;;  * Added an optional prefix argument to `o' (calc-realign).
;;;  * Changed `t s' and `t r' to use RET as the search exit key.
;;;  * Made handling of operator keys for V M, V R, etc. more regular.
;;;  * Improved TeX mode; added support for \matrix format.
;;;  * Added a variant of `m a' mode that only affects ( and [ keys.
;;;  * Fixed "Mismatch" message for algebraic entry of semi-open intervals.
;;;  * Trimmed fat from calc.el to speed loading, moved more to calc-ext.el.
;;;  * Fixed a bug in which minibuffer entry rounded to out-of-date precision.
;;;  * Fixed a bug which crashed Calc 1.04 under Epoch.
;;;  * Fixed a bug which messed up Calc Trail's mode line, among other things.
;;;  * Fixed a bug which caused trail ">" to show only when in Trail buffer.
;;;  * Fixed a bug in which "calc" called "calc-grab-region" with too few args.
;;;  * Fixed bugs in both implementation and documentation of calc-perm.
;;;  * Fixed a bug in which calc-simplify-extended always used radians.
;;;  * Fixed a bug where calc-comma failed to override "polar" mode.
;;;  * Fixed a bug doing mixed arithmetic on rectangular+polar complex numbers.
;;;  * Fixed several bugs in transcendental functions with complex arguments.
;;;  * Fixed a bug in which `a s' simplified "x / .5" to ".5 x".
;;;  * Fixed numerous other bugs in various parts of Calc.
;;;  * Completed the "Hooks" section of the "Internals" chapter of the manual.
;;; 
;;; Version 1.04:
;;;  * Included a copy of revision history (from README) in calc.el.
;;;  * Added the "calc-split" feature to split calc-ext.el into smaller bits.
;;;  * Changed calc-unpack to unpack floats and fractions, too.
;;;  * Added "mant", "xpon", and "scf" functions for decomposing floats.
;;;  * Fixed a bug in the "y" command with positive prefix arguments.
;;;  * Rearranged binary shift/rotate command keys to be a bit more convenient.
;;;  * Fixed a bug in which simplifying "(0/0) * 2" crashed with a Lisp error.
;;;  * Made `H F' [ffloor] and friends faster for very large arguments.
;;;  * Made calc-define-del more robust.
;;;  * Handled pasting of data into the Calculator using the mouse under X.
;;;  * Made overlay-arrow variables buffer-local to avoid interference.
;;;  * Fixed a problem in which Calc Trail buffer got stuck after a C-x C-w.
;;; 
;;; Version 1.03:
;;;  * Changed math-choose to compute n-choose-m faster when m is large.
;;;  * Fixed some problems with TeX mode.
;;;  * Fixed a bug that prevented `b s' from working without a prefix argument.
;;;  * Added "calc-eval" function.
;;;  * Improved calc-grab-region.
;;; 
;;; Version 1.02:
;;;  * Fixed a bug in Tutorial: telephone pole height/distance were switched!
;;;  * Fixed a few other things in the manual.
;;;  * Added "full-calc" command.
;;;  * Added "calc-insert-variables" (`Z I') command.
;;;  * Quick Calc now works even if you are already in the minibuffer.
;;;  * Fixed a bug in math-mul-bignum-digit which affected math-and, etc.
;;;  * Definition of "Hectares" was wrong in units table.
;;;  * Fixed a bug in calc-execute-kbd-macro concerning undo and refresh.
;;;  * Bound "calc-undo" to `C-x u' as well as `C-_' and `U'.
;;; 
;;; Version 1.01:
;;;  * Added a tutorial section to the manual.
;;;  * Next and Prev for node Strings in the manual were reversed; fixed.
;;;  * Changed "'bignum" in calc-isqrt-bignum-iter to "'bigpos".
;;;  * Fixed a bug that prevented "$" from working during algebraic entry.
;;;  * Fixed a bug caused by an X (last-X) command following a K (macro) cmd.
;;;  * Fixed a bug in which K command incorrectly formatted stack in Big mode.
;;;  * Added space between unary operators and non-flat compositions.
;;;    (Otherwise, "-(a/b)" in Big mode blended the minus sign into the rule!)
;;;  * Fixed formatting of (-1)^n in Big mode.
;;;  * Fixed some problems relating to "not" operator in Pascal language mode.
;;;  * Fixed several bugs relating to V M ' and V M $ sequences.
;;;  * Fixed matrix-vector multiplication to produce a vector.
;;;  * Introduced Z ` ... Z ' commands; renamed old Z ' to Z #.
;;;  * Fixed various other bugs.
;;;  * Added calc-settings-file variable suggested by C. Witty.
;;; 
;;; Version 1.00:
;;;  * First official release of Calc.


(provide 'calc)

;;;; [calc-macs.el]

(provide 'calc-macs)

(defun calc-need-macros ())

(defmacro calc-record-compilation-date-macro ()
  (` (setq calc-version (, (concat "Emacs Calc Mode"
				   " v1.07 by Dave Gillespie"
				   ", installed "
				   (current-time-string)
				   " by "
				   (user-full-name)))))
)

;;;; [calc.el]

(defun calc-record-compilation-date ()
  (calc-record-compilation-date-macro)
)
(calc-record-compilation-date)


(defvar calc-info-filename "calc-info"
  "*File name in which to look for the Calculator's Info documentation.")

(defvar calc-settings-file "~/.emacs"
  "*File in which to record permanent settings; default is \"~/.emacs\".")

(defvar calc-autoload-directory nil
  "Name of directory from which .elc files made by calc-split or calc-install
should be loaded.  Should include a trailing \"/\".
If nil, use original installation directory.")

(defvar calc-gnuplot-name "gnuplot"
  "*Name of GNUPLOT program, for calc-graph features.")

(defvar calc-gnuplot-plot-command nil
  "*Name of command for displaying GNUPLOT output; %s = file name to print.")

(defvar calc-gnuplot-print-command "lp %s"
  "*Name of command for printing GNUPLOT output; %s = file name to print.")


;; Address of the author of Calc, for use by report-calc-bug.
(defvar calc-bug-address "daveg@csvax.cs.caltech.edu")


;; If T, scan keymaps to find all DEL-like keys.
;; If NIL, only DEL itself is mapped to calc-pop.
(defvar calc-scan-for-dels t)



(defvar calc-extensions-loaded nil)



;;; IDEAS:
;;;
;;;   Make math-simplify faster!
;;;   Provide analogues of Mathematica's polynomial-munching ops:
;;;      Together, Apart, Cancel, Factor, GCD, quotient/remainder.
;;;   Automatically generate derivatives for functions defined with Z F.
;;;
;;;   Fix rewrite mechanism to do less gratuitous rearrangement of terms.
;;;   Implement a pattern-based "refers" predicate.
;;;
;;;   Make it possible to Undo a selection command.
;;;   Figure out how to allow selecting rows of matrices.
;;;   If cursor was in selection before, move it after j n, j p, j L, etc.
;;;   Consider reimplementing calc-delete-selection using rewrites.
;;;
;;;   When the stack is truncated, put the "." on the *second*-to-bottom line.
;;;   When formatting formulas, suppress extra spaces inside vectors!
;;;   Implement line-breaking in non-flat compositions.
;;;   Implement structured line-breaking using level information.
;;;   Implement matrix formatting with multi-line components.
;;;
;;;   Support lambda notation in Z F and Z G commands.
;;;   Have "Z R" define a user command based on a set of rewrite rules.
;;;   Support "incf" and "decf" in defmath definitions.
;;;   Have defmath generate calls to calc-binary-op or calc-unary-op.
;;;   Make some way to define algebraic functions using keyboard macros.
;;;
;;;   Allow calc-word-size=0 => Common Lisp-style signed bitwise arithmetic.
;;;   Consider digamma function (and thus Euler's gamma constant).
;;;   May as well make continued-fractions stuff available to the user.
;;;   Provide date arithmetic a la HP 48.
;;;   Consider providing infinities and indeterminate values as a data type.
;;;
;;;   Implement some more built-in statistical functions:
;;;      mean, median, variance, std dev of a list.
;;;   How about fitting a list or Nx2 matrix to a line or curve.
;;;
;;;   How about matrix eigenvalues, SVD, pseudo-inverse, etc.?
;;;   Should cache matrix inverses as well as decompositions.
;;;   If dividing by a non-square matrix, use least-squares automatically.
;;;
;;;   Handle TeX-mode parsing of \matrix{ ... } where ... contains braces.
;;;   Support AmS-TeX's \{d,t,}frac, \{d,t,}binom notations.
;;;
;;;   If GNUPLOT isn't there, do low-res plots in an Emacs text buffer.
;;;   Support polar plotting with GNUPLOT.
;;;   Make a calc-graph-histogram function.
;;;
;;;   Replace hokey formulas for complex functions with formulas designed
;;;      to minimize roundoff while maintaining the proper branch cuts.
;;;   Test accuracy of advanced math functions over whole complex plane.
;;;   Extend Bessel functions to provide arbitrary precision.
;;;   Extend advanced math functions to handle error forms and intervals.
;;;   Provide a better implementation for math-sin-cos-raw.
;;;   Provide a better implementation for math-hypot.
;;;   Provide a better implementation for math-make-frac.
;;;   Provide a better implementation for math-prime-factors.
;;;   Provide a better implementation for math-integral.
;;;   Provide a better initial guess for math-nth-root.
;;;   Clean up some of the definitions in the units table.
;;;
;;;   Provide more examples in the tutorial section of the manual.
;;;   Cover in the tutorial:  language modes, simplification modes,
;;;       bitwise stuff, rewrite rules, selections.
;;;   Provide more Lisp programming examples in the manual.
;;;   Finish the Internals section of the manual (and bring it up to date).
;;;   Make a TeX or PostScript reference card for Calc.
;;;
;;;   Tim suggests adding spreadsheet-like features.
;;;   Is constructive real arithmetic worth implementing?
;;;   Implement language modes for Eqn, Gnuplot, Lisp, Ada, ...?
;;;


;;; For atan series, if x > tan(pi/12) (about 0.268) reduce using the identity
;;;   atan(x) = atan((x * sqrt(3) - 1) / (sqrt(3) + x)) + pi/6.


;;; A better integration algorithm:
;;;   Use breadth-first instead of depth-first search, as follows:
;;;	The integral cache allows unfinished integrals in symbolic notation
;;;	on the righthand side.  An entry with no unfinished integrals on the
;;;	RHS is "complete"; references to it elsewhere are replaced by the
;;;	integrated value.  More than one cache entry for the same integral
;;;	may exist, though if one becomes complete, the others may be deleted.
;;;	The integrator works by using every applicable rule (such as
;;;	substitution, parts, linearity, etc.) to generate possible righthand
;;;	sides, all of which are entered into the cache.  Now, as long as the
;;;	target integral is not complete (and the time limit has not run out)
;;;	choose an incomplete integral from the cache and, for every integral
;;;	appearing in its RHS's, add those integrals to the cache using the
;;;	same substitition, parts, etc. rules.  The cache should be organized
;;;	as a priority queue, choosing the "simplest" incomplete integral at
;;;	each step, or choosing randomly among equally simple integrals.
;;;	Simplicity equals small size, and few steps removed from the original
;;;	target integral.  Note that when the integrator finishes, incomplete
;;;	integrals can be left in the cache, so the algorithm can start where
;;;	it left off if another similar integral is later requested.
;;;   Breadth-first search would avoid the nagging problem of, e.g., whether
;;;   to use parts or substitution first, and which decomposition is best.
;;;   All are tried, and any path that diverges will quickly be put on the
;;;   back burner by the priority queue.
;;;   Note: Probably a good idea to call math-simplify-extended before
;;;   measuring a formula's simplicity.






;; Calculator stack.
;; Entries are 3-lists:  Formula, Height (in lines), Selection (or nil).
(defvar calc-stack '((top-of-stack 1 nil)))

;; Index into calc-stack of "top" of stack.
;; This is 1 unless calc-truncate-stack has been used.
;;(defvar calc-stack-top 1)

;; If non-NIL, load the calc-ext module automatically when calc is loaded.
;;(defvar calc-always-load-extensions nil)

;; If non-NIL, display line numbers in Calculator stack.
;;(defvar calc-line-numbering t)

;; If non-NIL, break long values across multiple lines in Calculator stack.
;;(defvar calc-line-breaking t)

;; If NIL, stack display is left-justified.
;; If 'right, stack display is right-justified.
;; If 'center, stack display is centered."
;;(defvar calc-display-just nil)

;; Radix for entry and display of numbers in calc-mode, 2-36.
;;(defvar calc-number-radix 10)

;; If non-NIL, leading zeros are provided to pad integers to calc-word-size.
;;(defvar calc-leading-zeros nil)

;; If non-NIL, group digits in large displayed integers by inserting spaces.
;; If an integer, group that many digits at a time.
;; If 't', use 4 for binary and hex, 3 otherwise.
;;(defvar calc-group-digits nil)

;; The character (in the form of a string) to be used for grouping digits.
;; This is used only when calc-group-digits mode is on.
;;(defvar calc-group-char ",")

;; The character (in the form of a string) to be used as a decimal point.
;;(defvar calc-point-char ".")

;; Format of displayed fractions; a string of one or two of ":" or "/".
;;(defvar calc-frac-format ":")

;; If non-NIL, prefer fractional over floating-point results.
;;(defvar calc-prefer-frac nil)

;; Format of display hours-minutes-seconds angles, a format string.
;; String must contain three %s marks for hours, minutes, seconds respectively.
;;(defvar calc-hms-format "%s@ %s' %s\"")

;; Format to use for display of floating-point numbers in calc-mode.
;; Must be a list of one of the following forms:
;;  (float 0)      Floating point format, display full precision.
;;  (float N)      N > 0: Floating point format, at most N significant figures.
;;  (float -N)     -N < 0: Floating point format, calc-internal-prec - N figs.
;;  (fix N)        N >= 0: Fixed point format, N places after decimal point.
;;  (sci 0)        Scientific notation, full precision.
;;  (sci N)        N > 0: Scientific notation, N significant figures.
;;  (sci -N)       -N < 0: Scientific notation, calc-internal-prec - N figs.
;;  (eng 0)        Engineering notation, full precision.
;;  (eng N)        N > 0: Engineering notation, N significant figures.
;;  (eng -N)       -N < 0: Engineering notation, calc-internal-prec - N figs.
;;(defvar calc-float-format '(float 0))

;; Format to use when full precision must be displayed.
;;(defvar calc-full-float-format '(float 0))

;; Format to use for display of complex numbers in calc-mode.  Must be one of:
;;   nil            Use (x, y) form.
;;   i              Use x + yi form.
;;   j              Use x + yj form.
;;(defvar calc-complex-format nil)

;; Preferred form, either 'cplx or 'polar, for complex numbers.
;;(defvar calc-complex-mode 'cplx)

;; If non-NIL, display vectors of byte-sized integers as strings.
;;(defvar calc-display-strings nil)

;; If NIL, vector elements are left-justified.
;; If 'right, vector elements are right-justified.
;; If 'center, vector elements are centered."
;;(defvar calc-matrix-just 'center)

;; If non-NIL, display vectors one element per line.
;;(defvar calc-break-vectors nil)

;; If non-NIL, display long vectors in full.  If NIL, use abbreviated form.
;;(defvar calc-full-vectors t)

;; If non-NIL, display long vectors in full in the trail.
;;(defvar calc-full-trail-vectors t)

;; If non-NIL, separate elements of displayed vectors with this string.
;;(defvar calc-vector-commas ",")

;; If non-NIL, surround displayed vectors with these characters.
;;(defvar calc-vector-brackets "[]")

;; Language or format for entry and display of stack values.  Must be one of:
;;   nil            Use standard Calc notation.
;;   flat           Use standard Calc notation, one-line format.
;;   big 	    Display formulas in 2-d notation (enter w/std notation).
;;   unform	    Use unformatted display: add(a, mul(b,c)).
;;   c              Use C language notation.
;;   pascal         Use Pascal language notation.
;;   fortran        Use Fortran language notation.
;;   tex            Use TeX notation.
;;   math           Use Mathematica(tm) notation.
;;(defvar calc-language nil)

;; Numeric prefix argument for the command that set calc-language.
;;(defvar calc-language-option nil)

;; Open-parenthesis string for function call notation.
;;(defvar calc-function-open "(")

;; Close-parenthesis string for function call notation.
;;(defvar calc-function-close ")")

;; Function through which to pass strings after formatting.
;;(defvar calc-language-output-filter nil)

;; Function through which to pass strings before parsing.
;;(defvar calc-language-input-filter nil)

;; Formatting function used for non-decimal integers.
;;(defvar calc-radix-formatter nil)

;; Minimum number of bits per word, if any, for binary operations in calc-mode.
;;(defvar calc-word-size 32)

;; Most recently used value of M in a modulo form.
;;(defvar calc-previous-modulo nil)

;; Type of simplification applied to results.
;; If 'none, results are not simplified when pushed on the stack.
;; If 'num, functions are simplified only when args are constant.
;; If NIL, only fast simplifications are applied.
;; If 'binary, math-clip is applied if appropriate.
;; If 'alg, math-simplify is applied.
;; If 'ext, math-simplify-extended is applied.
;; If 'units, math-simplify-units is applied.
;;(defvar calc-simplify-mode nil)

;; Last direction specified in a Map or Reduce command.
;;(defvar calc-mapping-dir nil)

;; If non-NIL, display shows unformatted Lisp exprs.  (For debugging)
;;(defvar calc-display-raw nil)

;; Number of digits of internal precision for calc-mode calculations.
;;(defvar calc-internal-prec 12)

;; If non-NIL, next operation is Inverse.
;;(defvar calc-inverse-flag nil)

;; If non-NIL, next operation is Hyperbolic.
;;(defvar calc-hyperbolic-flag nil)

;; If deg, angles are in degrees; if rad, angles are in radians.
;; If hms, angles are in degrees-minutes-seconds.
;;(defvar calc-angle-mode 'deg)

;; If non-NIL, numeric entry accepts whole algebraic expressions.
;; If NIL, algebraic expressions must be preceded by "'".
;;(defvar calc-algebraic-mode nil)

;; Like calc-algebraic-mode except only affects ( and [ keys.
;;(defvar calc-incomplete-algebraic-mode nil)

;; If non-NIL, inexact numeric computations like sqrt(2) are postponed.
;; If NIL, computations on numbers always yield numbers where possible.
;;(defvar calc-symbolic-mode nil)

;; An integer which governs how long calc-integral will look for an integral.
;; The integrator often uses substitution or integration by parts to transform
;; an integral into another one; this controls how many levels of nested
;; sub-integrations are allowed before a given path is abandoned.
;;(defvar calc-integral-limit 3)

;; If non-NIL, shifted letter keys are prefix keys rather than normal meanings.
;;(defvar calc-shift-prefix nil)

;; Initial height of Calculator window.
;;(defvar calc-window-height 7)

;; If non-NIL, M-x calc creates a window to display Calculator trail.
;;(defvar calc-display-trail t)

;; If non-NIL, selected sub-formulas are shown by obscuring rest of formula.
;; If NIL, selected sub-formulas are highlighted by obscuring the sub-formulas.
;;(defvar calc-show-selections t)

;; If non-NIL, commands operate only on selected portions of formulas.
;; If NIL, selections displayed but ignored.
;;(defvar calc-use-selections t)

;; If non-NIL, selection hides deep structure of associative formulas.
;;(defvar calc-assoc-selections t)

;; If non-NIL, display "Working..." for potentially slow Calculator commands.
;;(defvar calc-display-working-message 'lots)

;; If non-NIL, automatically execute a "why" command to explain odd results.
;;(defvar calc-auto-why nil)


;; Floating-point numbers with this positive exponent or higher above the
;; current precision are displayed in scientific notation in calc-mode.
(defvar calc-display-sci-high 0)

;; Floating-point numbers with this negative exponent or lower are displayed
;; scientific notation in calc-mode.
(defvar calc-display-sci-low -3)


;; List of used-defined strings to append to Calculator mode line.
(defvar calc-other-modes nil)

;; T if calc-settings-file has been loaded yet.
(defvar calc-loaded-settings-file nil)


  
(defconst calc-mode-var-list '((calc-always-load-extensions nil)
			       (calc-line-numbering t)
			       (calc-line-breaking t)
			       (calc-display-just nil)
			       (calc-number-radix 10)
			       (calc-leading-zeros nil)
			       (calc-group-digits nil)
			       (calc-group-char ",")
			       (calc-point-char ".")
			       (calc-frac-format ":")
			       (calc-prefer-frac nil)
			       (calc-hms-format "%s@ %s' %s\"")
			       (calc-float-format (float 0))
			       (calc-full-float-format (float 0))
			       (calc-complex-format nil)
			       (calc-matrix-just center)
			       (calc-full-vectors t)
			       (calc-full-trail-vectors t)
			       (calc-break-vectors nil)
			       (calc-vector-commas ",")
			       (calc-vector-brackets "[]")
			       (calc-complex-mode cplx)
			       (calc-display-strings nil)
			       (calc-simplify-mode nil)
			       (calc-mapping-dir nil)
			       (calc-word-size 32)
			       (calc-previous-modulo nil)
			       (calc-display-raw nil)
			       (calc-internal-prec 12)
			       (calc-angle-mode deg)
			       (calc-algebraic-mode nil)
			       (calc-incomplete-algebraic-mode nil)
			       (calc-symbolic-mode nil)
			       (calc-integral-limit 3)
			       (calc-shift-prefix nil)
			       (calc-window-height 7)
			       (calc-language nil)
			       (calc-language-option nil)
			       (calc-show-selections t)
			       (calc-use-selections t)
			       (calc-assoc-selections t)
			       (calc-display-trail t)
			       (calc-display-working-message lots)
			       (calc-auto-why nil)
			       (calc-gnuplot-default-device "default")
			       (calc-gnuplot-default-output "/dev/null")
			       (calc-gnuplot-print-device "postscript")
			       (calc-gnuplot-print-output "auto")
			       (calc-gnuplot-geometry nil)
			       (calc-graph-default-resolution 15)))

(defconst calc-local-var-list '(calc-stack
				calc-stack-top
				calc-undo-list
				calc-redo-list
				calc-always-load-extensions
				calc-display-raw
				calc-line-numbering
				calc-line-breaking
				calc-display-just
				calc-auto-why
				calc-algebraic-mode
				calc-incomplete-algebraic-mode
				calc-symbolic-mode
				calc-integral-limit
				calc-inverse-flag
				calc-hyperbolic-flag
				calc-angle-mode
				calc-number-radix
				calc-leading-zeros
				calc-group-digits
				calc-group-char
				calc-point-char
				calc-frac-format
				calc-prefer-frac
				calc-hms-format
				calc-float-format
				calc-full-float-format
				calc-complex-format
				calc-matrix-just
				calc-full-vectors
				calc-full-trail-vectors
				calc-break-vectors
				calc-vector-commas
				calc-vector-brackets
				calc-complex-mode
				calc-display-strings
				calc-simplify-mode
				calc-mapping-dir
				calc-show-selections
				calc-use-selections
				calc-assoc-selections
				calc-word-size
				calc-internal-prec))


(defun calc-init-base ()

  ;; Set up the standard keystroke (M-#) to run the Calculator, if that key
  ;; has not yet been bound to anything.  For best results, the user should
  ;; do this before Calc is even loaded, so that M-# can auto-load Calc.
  (or (global-key-binding "\e#")
      (global-set-key "\e#" 'calc))
  
  (put 'calc-mode 'mode-class 'special)
  (put 'calc-trail-mode 'mode-class 'special)
  
  ;; Define "inexact-result" as an e-lisp error symbol.
  (put 'inexact-result 'error-conditions '(error inexact-result calc-error))
  (put 'inexact-result 'error-message "Calc internal error (inexact-result)")
  
  (setq calc-trail-pointer nil		; "Current" entry in trail buffer.
        calc-trail-overlay nil		; Value of overlay-arrow-string.
        calc-undo-list nil		; List of previous operations for undo.
        calc-redo-list nil		; List of recent undo operations.
        calc-main-buffer nil		; Pointer to Calculator buffer.
	calc-trail-buffer nil		; Pointer to Calc Trail buffer.
        calc-why nil			; Explanations of most recent errors.
        calc-next-why nil
	calc-inverse-flag nil
	calc-hyperbolic-flag nil
	calc-function-open "("
	calc-function-close ")"
	calc-language-output-filter nil
	calc-language-input-filter nil
	calc-radix-formatter nil
        calc-last-kill nil		; Last number killed in calc-mode.
        calc-previous-alg-entry nil	; Previous algebraic entry.
        calc-dollar-values nil		; Values to be used for '$'.
        calc-dollar-used nil		; Highest order of '$' that occurred.
        calc-quick-prev-results nil	; Previous results from Quick Calc.
	calc-said-hello nil		; Has calc-summary been called yet?
	calc-executing-macro nil	; Kbd macro executing from "K" key.
	calc-any-selections nil 	; Nil means no selections present.
	calc-help-phase 0		; Count of consecutive "?" keystrokes.
	calc-full-help-flag nil		; Executing calc-full-help?
	calc-refresh-count 0		; Count of calc-refresh calls.
	calc-prepared-composition nil
	calc-selection-cache-default-entry nil
	math-radix-explicit-format t
	math-expr-function-mapping nil
	math-expr-variable-mapping nil
	math-read-expr-quotes nil
        var-i '(special-const (math-imaginary 1))
        var-pi '(special-const (math-pi))
        var-e '(special-const (math-e)))

  (mapcar (function (lambda (v) (or (boundp (car v)) (set (car v) (nth 1 v)))))
	  calc-mode-var-list)
  (mapcar (function (lambda (v) (or (boundp v) (set v nil))))
	  calc-local-var-list)

  (if (boundp 'calc-mode-map)
      nil
    (setq calc-mode-map (make-keymap))
    (suppress-keymap calc-mode-map t)
    (define-key calc-mode-map "+" 'calc-plus)
    (define-key calc-mode-map "-" 'calc-minus)
    (define-key calc-mode-map "*" 'calc-times)
    (define-key calc-mode-map "/" 'calc-divide)
    (define-key calc-mode-map "%" 'calc-mod)
    (define-key calc-mode-map "&" 'calc-inv)
    (define-key calc-mode-map "^" 'calc-power)
    (define-key calc-mode-map "e" 'calcDigit-start)
    (define-key calc-mode-map "i" 'calc-info)
    (define-key calc-mode-map "n" 'calc-change-sign)
    (define-key calc-mode-map "q" 'calc-quit)
    (define-key calc-mode-map "?" 'calc-help)
    (define-key calc-mode-map " " 'calc-enter)
    (define-key calc-mode-map "'" 'calc-algebraic-entry)
    (define-key calc-mode-map "$" 'calc-algebraic-entry)
    (define-key calc-mode-map "\"" 'calc-algebraic-entry)
    (define-key calc-mode-map "\t" 'calc-roll-down)
    (define-key calc-mode-map "\M-\t" 'calc-roll-up)
    (define-key calc-mode-map "\C-m" 'calc-enter)
    (define-key calc-mode-map "\C-j" 'calc-over)

    (mapcar (function
	     (lambda (x)
	       (define-key calc-mode-map (char-to-string x) 'undefined)))
	    "jOWY{}")
    (mapcar (function
	     (lambda (x)
	       (define-key calc-mode-map (char-to-string x)
		 'calc-missing-key)))
	    (concat "ABCDEFGHIJKLMNPQRSTUVXZabcdfghklmoprstuvwxyz"
		    ":\\|!()[]<>,;=~`\C-k\M-k\C-w\M-w\C-y\C-_"))
    (mapcar (function
	     (lambda (x)
	       (define-key calc-mode-map (char-to-string x) 'calcDigit-start)))
	    "_0123456789.#@")

    (setq calc-digit-map (make-keymap))
    (let ((i 0))
      (while (< i 128)
	(aset calc-digit-map i
	      (if (eq (aref calc-mode-map i) 'undefined)
		  'undefined 'calcDigit-nondigit))
	(setq i (1+ i))))
    (mapcar (function
	     (lambda (x)
	       (define-key calc-digit-map (char-to-string x)
		 'calcDigit-key)))
	    "_0123456789.e+-:n#@oh'\"mspM")
    (mapcar (function
	     (lambda (x)
	       (define-key calc-digit-map (char-to-string x)
		 'calcDigit-letter)))
	    "abcdfgijklqrtuvwxyzABCDEFGHIJKLNOPQRSTUVWXYZ")
    (define-key calc-digit-map "'" 'calcDigit-algebraic)
    (define-key calc-digit-map "`" 'calcDigit-edit)
    (define-key calc-digit-map "\C-g" 'abort-recursive-edit)

    (mapcar (function
	     (lambda (x)
	       (condition-case err
		   (progn
		     (define-key calc-digit-map x 'calcDigit-backspace)
		     (define-key calc-mode-map x 'calc-pop))
		 (error nil))))
	    (if calc-scan-for-dels
		(append (where-is-internal 'delete-backward-char global-map)
			(where-is-internal 'backward-delete-char global-map)
			'("\C-d"))
	      '("\177" "\C-d"))))

;;;; (Autoloads here)

)

(calc-init-base)


(defun calc-mode ()
  "Calculator major mode.

This is an RPN calculator featuring arbitrary-precision integer, rational,
floating-point, complex, matrix, and symbolic arithmetic.

RPN calculation:  2 RET 3 +    produces 5.
Algebraic style:  ' 2+3 RET    produces 5.

Operators are +, -, *, /, ^ (power), % (modulo), n (change-sign).

Press ? repeatedly for more complete help.

Notations:  3.14e6     3.14 * 10^6
            _23        negative number -23
            17:3       the fraction 17/3
            5:2:3      the fraction 5 and 2/3
            16#12C     the integer 12C base 16 = 300 base 10
            8#177:100  the fraction 177:100 base 8 = 127:64 base 10
            (2, 4)     complex number 2 + 4i
            (2; 4)     polar complex number (r; theta)
            [1, 2, 3]  vector  ([[1, 2], [3, 4]] is a matrix)
            [1 .. 4)   semi-open interval, 1 <= x < 4
            2 +/- 3    (p key) number with mean 2, standard deviation 3
            2 mod 3    (M key) number 2 computed modulo 3

\\{calc-mode-map}
"
  (interactive)
  (mapcar (function
	   (lambda (v) (set-default v (symbol-value v)))) calc-local-var-list)
  (kill-all-local-variables)
  (use-local-map calc-mode-map)
  (mapcar (function (lambda (v) (make-local-variable v))) calc-local-var-list)
  (make-local-variable 'overlay-arrow-position)
  (make-local-variable 'overlay-arrow-string)
  (setq truncate-lines t)
  (setq buffer-read-only t)
  (setq major-mode 'calc-mode)
  (setq mode-name "Calculator")
  (setq calc-stack-top (length (or (memq (assq 'top-of-stack calc-stack)
					 calc-stack)
				   (setq calc-stack (list (list 'top-of-stack
								1 nil))))))
  (setq calc-stack-top (- (length calc-stack) calc-stack-top -1))
  (or calc-loaded-settings-file
      (string-match "\\.emacs" calc-settings-file)
      (progn
	(setq calc-loaded-settings-file t)
	(load calc-settings-file t)))   ; t = missing-ok
  (if (and (eq window-system 'x) (fboundp 'x-paste-text))
      (progn
	(or (boundp 'calc-old-x-paste-text)
	    (setq calc-old-x-paste-text (symbol-function 'x-paste-text)))
	(substitute-key-definition 'x-paste-text 'calc-x-paste-text
				   mouse-map)))
  (run-hooks 'calc-mode-hook)
  (calc-refresh t)
  (calc-set-mode-line)
  (if (and (boundp 'calc-defs)
	   calc-defs)
      (progn
	(message "Evaluating calc-defs...")
	(eval (cons 'progn calc-defs))
	(setq calc-defs nil)
	(calc-refresh t)
	(calc-set-mode-line)))
)

(defun calc-trail-mode (&optional buf)
  (interactive)
  (fundamental-mode)
  (use-local-map calc-mode-map)
  (setq major-mode 'calc-trail-mode)
  (setq mode-name "Calc Trail")
  (setq truncate-lines nil)
  (setq buffer-read-only t)
  (make-local-variable 'overlay-arrow-position)
  (make-local-variable 'overlay-arrow-string)
  (if buf
      (progn
	(make-local-variable 'calc-main-buffer)
	(setq calc-main-buffer buf)))
  (if (= (buffer-size) 0)
      (let ((buffer-read-only nil))
	(insert calc-version
		"\n")))
  (run-hooks 'calc-trail-mode-hook)
)

(defun calc-create-buffer ()
  (set-buffer (get-buffer-create "*Calculator*"))
  (if (or (not (eq major-mode 'calc-mode))
	  (and (boundp 'calc-defs) calc-defs))
      (calc-mode))
  (setq max-lisp-eval-depth (max max-lisp-eval-depth 1000))
  (if calc-always-load-extensions
      (calc-extensions))
  (if calc-language
      (progn
	(calc-extensions)
	(calc-set-language calc-language calc-language-option t)))
)

(defun calc (&optional arg full-display)
  "The Emacs Calculator.  Full documentation is listed under \"calc-mode\"."
  (interactive "P")
  (or (fboundp 'calc-extensions)
      (autoload 'calc-extensions "calc-ext"))
  (if arg
      (or (eq arg 0)
	  (progn
	    (calc-extensions)
	    (if (= (prefix-numeric-value arg) -1)
		(calc-grab-region (region-beginning) (region-end) nil)
	      (if (= (prefix-numeric-value arg) -2)
		  (calc-keypad)))))
    (if (eq major-mode 'calc-mode)
	(calc-quit)
      (if (get-buffer-window "*Calc Keypad*")
	  (calc-keypad))
      (let ((oldbuf (current-buffer)))
	(calc-create-buffer)
	(if full-display
	    (switch-to-buffer (current-buffer) t)
	  (if (get-buffer-window (current-buffer))
	      (select-window (get-buffer-window (current-buffer)))
	    (if (and (boundp 'calc-window-hook) calc-window-hook)
		(run-hooks 'calc-window-hook)
	      (let ((w (get-largest-window)))
		(if (and pop-up-windows
			 (> (window-height w)
			    (+ window-min-height calc-window-height 2)))
		    (progn
		      (setq w (split-window w
					    (- (window-height w)
					       calc-window-height 2)
					    nil))
		      (set-window-buffer w (current-buffer))
		      (select-window w))
		  (pop-to-buffer (current-buffer)))))))
	(save-excursion
	  (set-buffer (calc-trail-buffer))
	  (and calc-display-trail
	       (= (window-width) (screen-width))
	       (calc-trail-display 1 t)))
	(calc-summary full-display)
	(run-hooks 'calc-start-hook)
	(and calc-said-hello
	     (interactive-p)
	     (progn
	       (sit-for 2)
	       (message "")))
	(setq calc-said-hello t))))
)

(defun full-calc ()
  "Invoke the Calculator and give it a full-sized window."
  (interactive)
  (calc nil t)
)

;;;; [calc-misc.el]

(defun another-calc ()
  "Create another, independent Calculator buffer."
  (interactive)
  (if (eq major-mode 'calc-mode)
      (mapcar (function
	       (lambda (v)
		 (set-default v (symbol-value v)))) calc-local-var-list))
  (set-buffer (generate-new-buffer "*Calculator*"))
  (pop-to-buffer (current-buffer))
  (calc-mode)
)

;;;; [calc.el]

(defun calc-quit ()
  "Close the Calculator window(s).
This does not destroy the Calculator buffers or forget the stack contents,
it only closes the windows."
  (interactive)
  (calc-select-buffer)
  (run-hooks 'calc-end-hook)
  (setq calc-undo-list nil calc-redo-list nil)
  (mapcar (function
	   (lambda (v) (set-default v (symbol-value v)))) calc-local-var-list)
  (let ((buf (current-buffer))
	(win (get-buffer-window (current-buffer))))
    (delete-windows-on (calc-trail-buffer))
    (if (and win
	     (< (window-height win) (1- (screen-height)))
	     (= (window-width win) (screen-width)))  ; avoid calc-keypad mode
	(setq calc-window-height (- (window-height win) 2)))
    (delete-windows-on buf)
    (delete-windows-on (get-buffer "*Calc Keypad*"))
    (bury-buffer buf)
    (bury-buffer calc-trail-buffer))
)

(defun quick-calc ()
  "Do a quick calculation in the minibuffer without invoking full Calculator."
  (interactive)
  (calc-do-quick-calc)
)

(defun calc-eval (str &optional separator &rest args)
  "Do a quick calculation and return the result as a string.
Return value will either be the formatted result in string form,
or a list containing a character position and an error message in string form."
  "Do a quick calculation in the minibuffer without invoking full Calculator."
  (interactive)
  (calc-do-calc-eval str separator args)
)

(defun calc-keypad ()
  "Invoke the Calculator in \"visual keypad\" mode.
This is most useful in the X window system.
In this mode, click on the Calc \"buttons\" using the left mouse button.
Or, position the cursor manually and do M-x calc-keypad-press."
  (interactive)
  (calc-extensions)
  (calc-do-keypad)
)

;;;; [calc-aent.el]

(defun calc-do-quick-calc ()
  (if (eq major-mode 'calc)
      (calc-algebraic-entry)
    (save-excursion
      (calc-create-buffer)
      (let* ((calc-command-flags nil)
	     (calc-language (if (memq calc-language '(nil big))
				'flat calc-language))
	     (calc-dollar-values calc-quick-prev-results)
	     (calc-dollar-used 0)
	     (enable-recursive-minibuffers t)
	     (alg-exp (calc-do-alg-entry "" "Quick calc: "))
	     buf)
	(setq calc-quick-prev-results alg-exp)
	(while alg-exp
	  (setq buf (concat buf
			    (and buf " ")
			    (math-format-value (car alg-exp) 1000))
		alg-exp (cdr alg-exp)))
	(calc-handle-whys)
	(message buf))))
)

(defun calc-do-calc-eval (str separator args)
  (catch 'calc-error
    (save-excursion
      (calc-create-buffer)
      (cond
       ((and (consp str) (not (symbolp (car str))))
	(let ((calc-language nil)
	      (math-expr-opers math-standard-opers)
	      (calc-internal-prec 12)
	      (calc-word-size 32)
	      (calc-symbolic-mode nil)
	      (calc-angle-mode 'deg)
	      (calc-number-radix 10)
	      (calc-leading-zeros nil)
	      (calc-group-digits nil)
	      (calc-point-char ".")
	      (calc-frac-format ":")
	      (calc-prefer-frac nil)
	      (calc-hms-format "%s@ %s' %s\"")
	      (calc-float-format '(float 0))
	      (calc-complex-format nil)
	      (calc-matrix-just nil)
	      (calc-full-vectors t)
	      (calc-break-vectors nil)
	      (calc-vector-commas ",")
	      (calc-vector-brackets "[]")
	      (calc-complex-mode 'cplx)
	      (calc-display-strings nil)
	      (calc-simplify-mode nil)
	      (calc-display-working-message 'lots)
	      (strp (cdr str)))
	  (while strp
	    (set (car strp) (nth 1 strp))
	    (setq strp (cdr (cdr strp))))
	  (calc-do-calc-eval (car str) separator args)))
       ((eq separator 'eval)
	(eval str))
       ((eq separator 'macro)
	(calc-extensions)
	(let* ((calc-buffer (current-buffer))
	       (calc-window (get-buffer-window calc-buffer))
	       (save-window (selected-window)))
	  (if calc-window
	      (unwind-protect
		  (progn
		    (select-window calc-window)
		    (calc-execute-kbd-macro str nil (car args)))
		(and (window-point save-window)
		     (select-window save-window)))
	    (save-window-excursion
	      (select-window (get-largest-window))
	      (switch-to-buffer calc-buffer)
	      (calc-execute-kbd-macro str nil (car args)))))
	nil)
       ((eq separator 'pop)
	(or (not (integerp str))
	    (= str 0)
	    (calc-pop (min str (calc-stack-size))))
	(calc-stack-size))
       ((eq separator 'top)
	(and (integerp str)
	     (> str 0)
	     (<= str (calc-stack-size))
	     (math-format-value (calc-top-n str (car args)) 1000)))
       ((eq separator 'rawtop)
	(and (integerp str)
	     (> str 0)
	     (<= str (calc-stack-size))
	     (calc-top-n str (car args))))
       (t
	(let* ((calc-command-flags nil)
	       (calc-next-why nil)
	       (calc-language (if (memq calc-language '(nil big))
				  'flat calc-language))
	       (calc-dollar-values (mapcar
				    (function
				     (lambda (x)
				       (if (stringp x)
					   (progn
					     (setq x (math-read-exprs x))
					     (if (eq (car-safe x)
						     'error)
						 (throw 'calc-error
							(calc-eval-error
							 (cdr x)))
					       (car x)))
					 x)))
				    args))
	       (calc-dollar-used 0)
	       (res (if (stringp str)
			(math-read-exprs str)
		      (list str)))
	       buf)
	  (if (eq (car res) 'error)
	      (calc-eval-error (cdr res))
	    (setq res (mapcar 'calc-normalize res))
	    (and (memq 'clear-message calc-command-flags)
		 (message ""))
	    (cond ((eq separator 'raw)
		   (if (= (length res) 1)
		       (car res)
		     (calc-eval-error '(0 "Single value expected"))))
		  ((eq separator 'list)
		   res)
		  ((memq separator '(num rawnum))
		   (if (= (length res) 1)
		       (if (math-constp (car res))
			   (if (eq separator 'num)
			       (math-format-value (car res) 1000)
			     (car res))
			 (calc-eval-error
			  (list 0
				(if calc-next-why
				    (calc-explain-why (car calc-next-why))
				  "Number expected"))))
		     (calc-eval-error '(0 "Single value expected"))))
		  ((eq separator 'push)
		   (calc-push-list res)
		   nil)
		  (t (while res
		       (setq buf (concat buf
					 (and buf (or separator ", "))
					 (math-format-value (car res) 1000))
			     res (cdr res)))
		     buf))))))))
)

(defun calc-eval-error (msg)
  (if (and (boundp 'calc-eval-error)
	   calc-eval-error)
      (if (eq calc-eval-error 'string)
	  (nth 1 msg)
	(error "%s" (nth 1 msg)))
    msg)
)

;;;; [calc.el]

(defun calc-summary (&optional full)
  (interactive)
  (if full
      (message "Welcome to GNU Emacs Calc!  Press `?' or `i' for help, `C-x C-c' to quit.")
      (message "Welcome to the GNU Emacs Calculator!  Press `?' or `i' for help, `q' to quit."))
)

;;;; [calc-misc.el]

(defun calc-info ()
  "Run the Emacs Info system on the Calculator documentation."
  (interactive)
  (require 'info)
  (select-window (get-largest-window))
  (Info-find-node calc-info-filename "Top")
)

(defun calc-help ()
  (interactive)
  (let ((msgs
	 '("Letter keys: Help, Info (manual), Why; Xtended cmd; Quit"
	   "Letter keys: Negate; Precision; Store, Recall, Let; Yank"
	   "Letter keys: SHIFT + Undo, reDo, last-X; Inverse, Hyperbolic"
	   "Letter keys: SHIFT + sQrt; Sin, Cos, Tan; Exp, Ln, logB"
	   "Letter keys: SHIFT + Floor, Round; Abs, conJ, arG; Pi"
	   "Letter keys: SHIFT + Num-eval; More-recn; Kbd-macro"
	   "Other keys: +, -, *, /, ^, \\ (int div), : (frac div)"
	   "Other keys: & (1/x), | (concat), % (modulo), ! (factorial)"
	   "Other keys: ' (alg-entry), = (evaluate), ` (edit)"
	   "Other keys: SPC/RET (enter/dup), LFD (over), DEL (drop)"
	   "Other keys: TAB (swap/roll-dn), M-TAB (roll-up); < > (hscroll)"
	   "Other keys: [ , ; ] (vector), ( , ) (complex), ( ; ) (polar)"
	   "Prefix keys: Algebra, Binary, Convert, Display, Functions, Graph"
	   "Prefix keys: J (select), Komb/stat, Modes, Trail, Units, Vectors"
	   "Prefix keys: Z (user), SHIFT + Z (define-user)"
	   "Prefix keys: prefix + ? gives further help for that prefix"
	   "  Copyright (C) 1990 Dave Gillespie, daveg@csvax.cs.caltech.edu")))
    (if calc-full-help-flag
	msgs
      (if (or calc-inverse-flag calc-hyperbolic-flag)
	  (if calc-inverse-flag
	      (if calc-hyperbolic-flag
		  (calc-inv-hyp-prefix-help)
		(calc-inverse-prefix-help))
	    (calc-hyperbolic-prefix-help))
	(setq calc-help-phase
	      (if (eq this-command last-command)
		  (% (1+ calc-help-phase) (1+ (length msgs)))
		0))
	(let ((msg (nth calc-help-phase msgs)))
	  (message "%s" (if msg
			    (concat msg ":"
				    (make-string (- (apply 'max
							   (mapcar 'length
								   msgs))
						    (length msg)) 32)
				    "  [?=MORE]")
			  ""))))))
)




;;;; Stack and buffer management.

;;;; [calc-macs.el]

(defmacro calc-wrapper (&rest body)
  (list 'calc-do (list 'function (append (list 'lambda ()) body)))
)

;; We use "point" here to generate slightly smaller byte-code than "t".
(defmacro calc-slow-wrapper (&rest body)
  (list 'calc-do (list 'function (append (list 'lambda ()) body)) (point))
)

;;;; [calc.el]

(defun calc-do (do-body &optional do-slow)
  (let ((calc-command-flags nil))
    (setq calc-aborted-prefix "")
    (unwind-protect
	(condition-case err
	    (save-excursion
	      (calc-select-buffer)
	      (and do-slow calc-display-working-message
		   (progn
		     (message "Working...")
		     (calc-set-command-flag 'clear-message)))
	      (funcall do-body)
	      (setq calc-aborted-prefix nil)
	      (and (memq 'renum-stack calc-command-flags)
		   (calc-renumber-stack))
	      (and (memq 'clear-message calc-command-flags)
		   (message "")))
	  (error
	   (if (and (eq (car err) 'error)
		    (stringp (nth 1 err))
		    (string-match "max-specpdl-size\\|max-lisp-eval-depth"
				  (nth 1 err)))
	       (error "Computation got stuck or ran too long.  Type `M' to increase the limit.")
	     (setq calc-aborted-prefix nil)
	     (signal (car err) (cdr err)))))
      (setq calc-old-aborted-prefix calc-aborted-prefix)
      (and calc-aborted-prefix
	   (calc-record "<Aborted>" calc-aborted-prefix))
      (or (memq 'no-align calc-command-flags)
	  (eq major-mode 'calc-trail-mode)
	  (calc-align-stack-window))
      (and (memq 'position-point calc-command-flags)
	   (progn
	     (goto-line calc-final-point-line)
	     (move-to-column calc-final-point-column)))
      (or (memq 'keep-flags calc-command-flags)
	  (save-excursion
	    (calc-select-buffer)
	    (setq calc-inverse-flag nil
		  calc-hyperbolic-flag nil)))
      (and (memq 'do-edit calc-command-flags)
	   (switch-to-buffer (get-buffer-create "*Calc Edit*")))
      (calc-set-mode-line)))
)
(setq calc-aborted-prefix nil)

(defun calc-set-command-flag (f)
  (if (not (memq f calc-command-flags))
      (setq calc-command-flags (cons f calc-command-flags)))
)

(defun calc-select-buffer ()
  (if (not (eq major-mode 'calc-mode))
      (if calc-main-buffer
	  (set-buffer calc-main-buffer)
	(let ((buf (get-buffer "*Calculator*")))
	  (if buf
	      (set-buffer buf)
	    (error "Calculator buffer not available")))))
)

(defun calc-cursor-stack-index (&optional index)
  (goto-char (point-max))
  (forward-line (- (calc-substack-height (or index 1))))
)

(defun calc-stack-size ()
  (- (length calc-stack) calc-stack-top)
)

(defun calc-substack-height (n)
  (let ((sum 0)
	(stack calc-stack))
    (setq n (+ n calc-stack-top))
    (while (and (> n 0) stack)
      (setq sum (+ sum (nth 1 (car stack)))
	    n (1- n)
	    stack (cdr stack)))
    sum)
)

(defun calc-set-mode-line ()
  (save-excursion
    (calc-select-buffer)
    (let* ((fmt (car calc-float-format))
	   (figs (nth 1 calc-float-format))
	   (new-mode-string
	    (format "Calc%s: %d %s %-14s"
		    (if (and (> (length (buffer-name)) 12)
			     (equal (substring (buffer-name) 0 12)
				    "*Calculator*"))
			(substring (buffer-name) 12)
		      "")
		    calc-internal-prec
		    (capitalize (symbol-name calc-angle-mode))
		    (concat
		     (cond ((= calc-number-radix 10) "")
			   ((= calc-number-radix 2) "Bin ")
			   ((= calc-number-radix 8) "Oct ")
			   ((= calc-number-radix 16) "Hex ")
			   (t (format "Radix%d " calc-number-radix)))
		     (if calc-algebraic-mode "Alg "
		       (if calc-incomplete-algebraic-mode "IncAlg " ""))
		     (if calc-symbolic-mode "Symb " "")
		     (cond ((eq calc-simplify-mode 'none) "NoSimp ")
			   ((eq calc-simplify-mode 'num) "NumSimp ")
			   ((eq calc-simplify-mode 'binary)
			    (format "BinSimp%d " calc-word-size))
			   ((eq calc-simplify-mode 'alg) "AlgSimp ")
			   ((eq calc-simplify-mode 'units) "UnitSimp ")
			   (t ""))
		     (cond ((null calc-language) "")
			   ((eq calc-language 'tex) "TeX ")
			   (t (concat
			       (capitalize (symbol-name calc-language))
			       " ")))
		     (if (eq calc-complex-mode 'polar) "Polar " "")
		     (if calc-prefer-frac "Frac " "")
		     (cond ((eq fmt 'float)
			    (if (zerop figs) "" (format "Norm%d " figs)))
			   ((eq fmt 'fix) (format "Fix%d " figs))
			   ((eq fmt 'sci)
			    (if (zerop figs) "Sci " (format "Sci%d " figs)))
			   ((eq fmt 'eng)
			    (if (zerop figs) "Eng " (format "Eng%d " figs))))
		     (if calc-assoc-selections "" "Break ")
		     (if (and (fboundp 'calc-gnuplot-alive)
			      (calc-gnuplot-alive)) "Graph " "")
		     (if calc-inverse-flag "Inv " "")
		     (if calc-hyperbolic-flag "Hyp " "")
		     (if (/= calc-stack-top 1) "Narrow " "")
		     (apply 'concat calc-other-modes)))))
      (if (equal new-mode-string mode-line-buffer-identification)
	  nil
	(setq mode-line-buffer-identification new-mode-string)
	(set-buffer-modified-p (buffer-modified-p)))))
)

(defun calc-align-stack-window ()
  (if (eq major-mode 'calc-mode)
      (progn
	(let ((win (get-buffer-window (current-buffer))))
	  (if win
	      (progn
		(calc-cursor-stack-index 0)
		(vertical-motion (- 2 (window-height win)))
		(set-window-start win (point)))))
	(calc-cursor-stack-index 0)
	(if (looking-at " *\\.$")
	    (goto-char (1- (match-end 0)))))
    (save-excursion
      (calc-select-buffer)
      (calc-align-stack-window)))
)

(defun calc-check-stack (n)
  (if (> n (calc-stack-size))
      (error "Too few elements on stack"))
  (if (< n 0)
      (error "Invalid argument"))
)

(defun calc-push-list (vals &optional m sels)
  (while vals
    (if calc-executing-macro
	(calc-push-list-in-macro vals m sels)
      (save-excursion
	(calc-select-buffer)
	(let* ((val (car vals))
	       (entry (list val 1 (car sels)))
	       (mm (+ (or m 1) calc-stack-top)))
	  (calc-cursor-stack-index (1- (or m 1)))
	  (if (> mm 1)
	      (setcdr (nthcdr (- mm 2) calc-stack)
		      (cons entry (nthcdr (1- mm) calc-stack)))
	    (setq calc-stack (cons entry calc-stack)))
	  (let ((buffer-read-only nil))
	    (insert (math-format-stack-value entry) "\n"))
	  (calc-record-undo (list 'push mm))
	  (calc-set-command-flag 'renum-stack))))
    (setq vals (cdr vals)
	  sels (cdr sels)))
)

(defun calc-pop-push-list (n vals &optional m sels)
  (if (and calc-any-selections (null sels))
      (calc-replace-selections n vals m)
    (calc-pop-stack n m sels)
    (calc-push-list vals m sels))
)

(defun calc-pop-push-record-list (n prefix vals &optional m sels)
  (or (and (consp vals)
	   (or (integerp (car vals))
	       (consp (car vals))))
      (and vals (setq vals (list vals)
		      sels (and sels (list sels)))))
  (calc-check-stack (+ n (or m 1) -1))
  (if prefix
      (if (cdr vals)
	  (calc-record-list vals prefix)
	(calc-record (car vals) prefix)))
  (calc-pop-push-list n vals m sels)
)

(defun calc-enter-result (n prefix vals &optional m)
  (setq calc-aborted-prefix prefix)
  (if (and (consp vals)
	   (or (integerp (car vals))
	       (consp (car vals))))
      (setq vals (mapcar 'calc-normalize vals))
    (setq vals (calc-normalize vals)))
  (or (and (consp vals)
	   (or (integerp (car vals))
	       (consp (car vals))))
      (setq vals (list vals)))
  (if (equal vals '((nil)))
      (setq vals nil))
  (calc-pop-push-record-list n prefix vals m)
  (calc-handle-whys)
)

(defun calc-normalize (val)
  (if (memq calc-simplify-mode '(nil none num))
      (math-normalize val)
    (calc-extensions)
    (calc-normalize-fancy val))
)

(defun calc-handle-whys ()
  (setq calc-why calc-next-why
	calc-next-why nil)
  (if (and calc-why calc-auto-why)
      (progn
	(calc-extensions)
	(calc-explain-why (car calc-why))
	(calc-clear-command-flag 'clear-message)))
)

;;;; [calc-misc.el]

(defun calc-record-why (&rest stuff)
  (setq calc-next-why (cons stuff calc-next-why))
  nil
)

;;;; [calc.el]

(defun calc-pop-stack (&optional n m sel-ok)  ; pop N objs at level M of stack.
  (or n (setq n 1))
  (or m (setq m 1))
  (let ((mm (+ m calc-stack-top)))
    (if (and calc-any-selections (not sel-ok)
	     (calc-top-selected n m))
	(calc-sel-error))
    (if calc-executing-macro
	(calc-pop-stack-in-macro n mm)
      (calc-record-undo (list 'pop mm (calc-top-list n m 'full)))
      (save-excursion
	(calc-select-buffer)
	(let ((buffer-read-only nil))
	  (if (> mm 1)
	      (progn
		(calc-cursor-stack-index (1- m))
		(let ((bot (point)))
		  (calc-cursor-stack-index (+ n m -1))
		  (delete-region (point) bot))
		(setcdr (nthcdr (- mm 2) calc-stack)
			(nthcdr (+ n mm -1) calc-stack)))
	    (calc-cursor-stack-index n)
	    (setq calc-stack (nthcdr n calc-stack))
	    (delete-region (point) (point-max)))
	  (calc-set-command-flag 'renum-stack)))))
)

(defun calc-get-stack-element (x)
  (cond ((eq sel-mode 'entry)
	 x)
	((eq sel-mode 'sel)
	 (nth 2 x))
	((or (null (nth 2 x))
	     (eq sel-mode 'full)
	     (not calc-use-selections))
	 (car x))
	(sel-mode
	 (calc-sel-error))
	(t (nth 2 x)))
)

;; Get the Nth element of the stack (N=1 is the top element).
(defun calc-top (&optional n sel-mode)
  (or n (setq n 1))
  (calc-check-stack n)
  (calc-get-stack-element (nth (+ n calc-stack-top -1) calc-stack))
)

(defun calc-top-n (&optional n sel-mode)    ; in case precision has changed
  (math-check-complete (calc-normalize (calc-top n sel-mode)))
)

(defun calc-top-list (&optional n m sel-mode)
  (or n (setq n 1))
  (or m (setq m 1))
  (calc-check-stack (+ n m -1))
  (and (> n 0)
       (let ((top (copy-sequence (nthcdr (+ m calc-stack-top -1)
					 calc-stack))))
	 (setcdr (nthcdr (1- n) top) nil)
	 (nreverse (mapcar 'calc-get-stack-element top))))
)

(defun calc-top-list-n (&optional n m sel-mode)
  (mapcar 'math-check-complete
	  (mapcar 'calc-normalize (calc-top-list n m sel-mode)))
)

;;;; [calc-misc.el]

(defun calc-roll-down-stack (n &optional m)
  (if (< n 0)
      (calc-roll-up-stack (- n) m)
    (if (or (= n 0) (> n (calc-stack-size))) (setq n (calc-stack-size)))
    (or m (setq m 1))
    (and (> n 1)
	 (< m n)
	 (if (and calc-any-selections
		  (not calc-use-selections))
	     (calc-roll-down-with-selections n m)
	   (calc-pop-push-list n
			       (append (calc-top-list m 1)
				       (calc-top-list (- n m) (1+ m)))))))
)

(defun calc-roll-up-stack (n &optional m)
  (if (< n 0)
      (calc-roll-down-stack (- n) m)
    (if (or (= n 0) (> n (calc-stack-size))) (setq n (calc-stack-size)))
    (or m (setq m 1))
    (and (> n 1)
	 (< m n)
	 (if (and calc-any-selections
		  (not calc-use-selections))
	     (calc-roll-up-with-selections n m)
	   (calc-pop-push-list n
			       (append (calc-top-list (- n m) 1)
				       (calc-top-list m (- n m -1)))))))
)

;;;; [calc.el]

(defun calc-renumber-stack ()
  (if calc-line-numbering
      (save-excursion
	(calc-cursor-stack-index 0)
	(let ((lnum 1)
	      (buffer-read-only nil)
	      (stack (nthcdr calc-stack-top calc-stack)))
	  (if (re-search-forward "^[0-9]+[:*]" nil t)
	      (progn
		(beginning-of-line)
		(while (re-search-forward "^[0-9]+[:*]" nil t)
		  (let ((buffer-read-only nil))
		    (beginning-of-line)
		    (delete-char 4)
		    (insert "    ")))
		(calc-cursor-stack-index 0)))
	  (while (re-search-backward "^[0-9]+[:*]" nil t)
	    (delete-char 4)
	    (if (> lnum 999)
		(insert (format "%03d%s" (% lnum 1000)
				(if (and (nth 2 (car stack))
					 calc-use-selections) "*" ":")))
	      (let ((prefix (int-to-string lnum)))
		(insert prefix (if (and (nth 2 (car stack))
					calc-use-selections) "*" ":")
			(make-string (- 3 (length prefix)) 32))))
	    (beginning-of-line)
	    (setq lnum (1+ lnum)
		  stack (cdr stack))))))
)

(defun calc-refresh (&optional align)
  "Refresh the contents of the Calculator buffer from memory."
  (interactive)
  (and (eq major-mode 'calc-mode)
       (not calc-executing-macro)
       (let ((buffer-read-only nil)
	     (save-point (point))
	     (save-mark (mark))
	     (save-aligned (looking-at "\\.$"))
	     (thing calc-stack))
	 (setq calc-any-selections nil)
	 (erase-buffer)
	 (insert "--- Emacs Calculator Mode ---\n")
	 (while thing
	   (goto-char (point-min))
	   (forward-line 1)
	   (insert (math-format-stack-value (car thing)) "\n")
	   (setq thing (cdr thing)))
	 (calc-renumber-stack)
	 (if (or align save-aligned)
	     (calc-align-stack-window)
	   (goto-char save-point))
	 (set-mark save-mark)))
  (setq calc-refresh-count (1+ calc-refresh-count))
)

(defun calc-x-paste-text (arg)
  "Move point to mouse position and insert window system cut buffer contents.
If mouse is pressed in Calc window, push cut buffer contents onto the stack."
  (x-mouse-select arg)
  (if (memq major-mode '(calc-mode calc-trail-mode))
      (progn
	(calc-wrapper
	 (calc-extensions)
	 (let ((val (math-read-exprs (calc-clean-newlines
				      (x-get-cut-buffer)))))
	   (if (eq (car-safe val) 'error)
	       (error "%s in yanked data" (nth 2 val))
	     (calc-enter-result 0 "Xynk" val)))))
    (funcall calc-old-x-paste-text arg))
)



;;;; The Calc Trail buffer.

(defun calc-check-trail-aligned ()
  (save-excursion
    (let ((win (get-buffer-window (current-buffer))))
      (and win
	   (pos-visible-in-window-p (1- (point-max)) win))))
)

;;;; [calc-macs.el]

(defmacro math-showing-full-precision (body)
  (list 'let
	'((calc-float-format calc-full-float-format))
	body)
)

;;;; [calc.el]

(defun calc-trail-buffer ()
  (and (or (null calc-trail-buffer)
	   (null (buffer-name calc-trail-buffer)))
       (save-excursion
	 (setq calc-trail-buffer (get-buffer-create "*Calc Trail*"))
	 (let ((buf (or (and (not (eq major-mode 'calc-mode))
			     (get-buffer "*Calculator*"))
			(current-buffer))))
	   (set-buffer calc-trail-buffer)
	   (or (eq major-mode 'calc-trail-mode)
	       (calc-trail-mode buf)))))
  (or (and calc-trail-pointer
	   (eq (marker-buffer calc-trail-pointer) calc-trail-buffer))
      (save-excursion
	(set-buffer calc-trail-buffer)
	(goto-line 2)
	(setq calc-trail-pointer (point-marker))))
  calc-trail-buffer
)

(defun calc-record (val &optional prefix)
  (setq calc-aborted-prefix nil)
  (or calc-executing-macro
      (let* ((mainbuf (current-buffer))
	     (buf (calc-trail-buffer))
	     (calc-display-raw nil)
	     (calc-can-abbrev-vectors t)
	     (fval (if val
		       (if (stringp val)
			   val
			 (math-showing-full-precision
			  (math-format-flat-expr val 0)))
		     "")))
	(save-excursion
	  (set-buffer buf)
	  (let ((aligned (calc-check-trail-aligned))
		(buffer-read-only nil))
	    (goto-char (point-max))
	    (cond ((null prefix) (insert "     "))
		  ((and (> (length prefix) 4)
			(string-match " " prefix 4))
		   (insert (substring prefix 0 4) " "))
		  (t (insert (format "%4s " prefix))))
	    (insert fval "\n")
	    (let ((win (get-buffer-window buf)))
	      (if (and aligned win (not (memq 'hold-trail calc-command-flags)))
		  (calc-trail-here))
	      (goto-char (1- (point-max))))))))
  val
)

;;;; [calc-misc.el]

(defun calc-record-list (vals &optional prefix)
  (while vals
    (or (eq (car vals) 'top-of-stack)
	(progn
	  (calc-record (car vals) prefix)
	  (setq prefix "...")))
    (setq vals (cdr vals)))
)

;;;; [calc.el]

(defun calc-trail-display (flag &optional no-refresh)
  "Turn the Trail display on or off.
With prefix argument 1, turn it on; with argument 0, turn it off."
  (interactive "P")
  (let ((win (get-buffer-window (calc-trail-buffer))))
    (if (setq calc-display-trail
	      (not (if flag (memq flag '(nil 0)) win)))
	(if (null win)
	    (progn
	      (if (and (boundp 'calc-trail-window-hook) calc-trail-window-hook)
		  (run-hooks 'calc-trail-window-hook)
		(let ((w (split-window nil (/ (* (window-width) 2) 3) t)))
		  (set-window-buffer w calc-trail-buffer)))
	      (calc-wrapper
	       (setq overlay-arrow-string calc-trail-overlay
		     overlay-arrow-position calc-trail-pointer)
	       (or no-refresh
		   (calc-refresh)))))
      (if win
	  (progn
	    (delete-window win)
	    (calc-wrapper
	     (or no-refresh
		 (calc-refresh)))))))
  calc-trail-buffer
)

(defun calc-trail-here ()
  "Move the trail pointer to the current cursor line."
  (interactive)
  (if (eq major-mode 'calc-trail-mode)
      (progn
	(beginning-of-line)
	(if (bobp)
	    (forward-line 1)
	  (if (eobp)
	      (forward-line -1)))
	(if (or (bobp) (eobp))
	    (setq overlay-arrow-position nil)   ; trail is empty
	  (set-marker calc-trail-pointer (point) (current-buffer))
	  (setq calc-trail-overlay (concat (buffer-substring (point)
							     (+ (point) 4))
					   ">")
		overlay-arrow-string calc-trail-overlay
		overlay-arrow-position calc-trail-pointer)
	  (forward-char 4)
	  (let ((win (get-buffer-window (current-buffer))))
	    (if win
		(save-excursion
		  (forward-line (/ (window-height) 2))
		  (forward-line (- 1 (window-height)))
		  (set-window-start win (point))
		  (set-window-point win (+ calc-trail-pointer 4))
		  (set-buffer calc-main-buffer)
		  (setq overlay-arrow-string calc-trail-overlay
			overlay-arrow-position calc-trail-pointer))))))
    (error "Not in Calc Trail buffer"))
)




;;;; The Undo list.

(defun calc-record-undo (rec)
  (or calc-executing-macro
      (if (memq 'undo calc-command-flags)
	  (setq calc-undo-list (cons (cons rec (car calc-undo-list))
				     (cdr calc-undo-list)))
	(setq calc-undo-list (cons (list rec) calc-undo-list)
	      calc-redo-list nil)
	(calc-set-command-flag 'undo)))
)



;;; Arithmetic commands.

(defun calc-binary-op (name func arg &optional ident unary func2)
  (setq calc-aborted-prefix name)
  (if (null arg)
      (calc-enter-result 2 name (cons (or func2 func) (calc-top-list-n 2)))
    (calc-extensions)
    (calc-binary-op-fancy name func arg ident unary))
)

(defun calc-unary-op (name func arg &optional func2)
  (setq calc-aborted-prefix name)
  (if (null arg)
      (calc-enter-result 1 name (list (or func2 func) (calc-top-n 1)))
    (calc-extensions)
    (calc-unary-op-fancy name func arg))
)


(defun calc-plus (arg)
  "Add the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "+" 'calcFunc-add arg 0 nil '+))
)

(defun calc-minus (arg)
  "Subtract the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "-" 'calcFunc-sub arg 0 'neg '-))
)

(defun calc-times (arg)
  "Multiply the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "*" 'calcFunc-mul arg 1 nil '*))
)

(defun calc-divide (arg)
  "Divide the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "/" 'calcFunc-div arg 0 'calcFunc-inv '/))
)

;;;; [calc-misc.el]

(defun calc-power (arg)
  "Compute y^x for the top two elements of the Calculator stack.
With Inverse flag, compute y^(1/x), i.e., the x'th root of y."
  (interactive "P")
  (calc-slow-wrapper
   (if (and calc-extensions-loaded
	    (calc-is-inverse))
       (calc-binary-op "root" 'calcFunc-nroot arg nil nil)
     (calc-binary-op "^" 'calcFunc-pow arg nil nil '^)))
)

(defun calc-mod (arg)
  "Compute the modulo of the top two elements of the Calculator stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-binary-op "%" 'calcFunc-mod arg nil nil '%))
)

(defun calc-inv (arg)
  "Invert the number or square matrix on the top of the stack."
  (interactive "P")
  (calc-slow-wrapper
   (calc-unary-op "inv" 'calcFunc-inv arg))
)

;;;; [calc.el]

(defun calc-change-sign (arg)
  "Change the sign of the top element of the Calculator stack."
  (interactive "P")
  (calc-wrapper
   (calc-unary-op "chs" 'neg arg))
)



;;; Stack management commands.

(defun calc-enter (n)
  "Duplicate the top element (or top N elements) of the Calculator stack.
With a negative argument -N, duplicate the Nth element of the stack."
  (interactive "p")
  (calc-wrapper
   (cond ((< n 0)
	  (calc-push-list (calc-top-list 1 (- n))))
	 ((= n 0)
	  (calc-push-list (calc-top-list (calc-stack-size))))
	 (t
	  (calc-push-list (calc-top-list n)))))
)

;;;; [calc-misc.el]

(defun calc-over (n)
  "Duplicate the second (or Nth) element of the Calculator stack.
With a negative argument -N, duplicate the top N elements of the stack."
  (interactive "P")
  (if n
      (calc-enter (- (prefix-numeric-value n)))
    (calc-enter -2))
)

;;;; [calc.el]

(defun calc-pop (n)
  "Pop (and discard) the top element (or top N elements) of the stack.
With a negative argument -N, remove the Nth element from the stack.
With a zero prefix argument, clear the whole stack."
  (interactive "P")
  (calc-wrapper
   (let* ((nn (prefix-numeric-value n))
	  (top (and (null n) (calc-top 1))))
     (cond ((and (null n)
		 (eq (car-safe top) 'incomplete)
		 (> (length top) (if (eq (nth 1 top) 'intv) 3 2)))
	    (calc-pop-push-list 1 (let ((tt (copy-sequence top)))
				    (setcdr (nthcdr (- (length tt) 2) tt) nil)
				    (list tt))))
	   ((< nn 0)
	    (if (and calc-any-selections
		     (calc-top-selected 1 (- nn)))
		(calc-delete-selection (- nn))
	      (calc-pop-stack 1 (- nn) t)))
	   ((= nn 0)
	    (calc-pop-stack (calc-stack-size) 1 t))
	   (t
	    (if (and calc-any-selections
		     (= nn 1)
		     (calc-top-selected 1 1))
		(calc-delete-selection 1)
	      (calc-pop-stack nn))))))
)

;;;; [calc-misc.el]

(defun calc-roll-down (n)
  "Exchange the top two elements of the Calculator stack.
With a positive numeric prefix, roll down the top N elements.
With a negative numeric prefix, roll down the whole stack N times.
With a zero prefix, reverse the order of elements in the stack."
  (interactive "P")
  (calc-wrapper
   (let ((nn (prefix-numeric-value n)))
     (cond ((null n)
	    (calc-roll-down-stack 2))
	   ((> nn 0)
	    (calc-roll-down-stack nn))
	   ((= nn 0)
	    (calc-pop-push-list (calc-stack-size)
				(reverse
				 (calc-top-list (calc-stack-size)))))
	   (t
	    (calc-roll-down-stack (calc-stack-size) (- nn))))))
)

(defun calc-roll-up (n)
  "Roll up (rotate) the top three elements of the Calculator stack.
With a numeric prefix, roll up the top N elements.
With a negative numeric prefix, roll up the whole stack N times.
With a zero prefix, reverse the order of elements in the stack."
  (interactive "P")
  (calc-wrapper
   (let ((nn (prefix-numeric-value n)))
     (cond ((null n)
	    (calc-roll-up-stack 3))
	   ((> nn 0)
	    (calc-roll-up-stack nn))
	   ((= nn 0)
	    (calc-pop-push-list (calc-stack-size)
				(reverse
				 (calc-top-list (calc-stack-size)))))
	   (t
	    (calc-roll-up-stack (calc-stack-size) (- nn))))))
)




;;;; [calc-misc.el]

;;; Other commands.

(defun calc-num-prefix-name (n)
  (cond ((eq n '-) "- ")
	((equal n '(4)) "C-u ")
	((consp n) (format "%d " (car n)))
	((integerp n) (format "%d " n))
	(t ""))
)

(defun calc-missing-key (n)
  "This is a placeholder for a command which needs to be loaded from calc-ext.
When this key is used, calc-ext (the Calculator extensions module) will be
loaded and the keystroke automatically re-typed."
  (interactive "P")
  (calc-extensions)
  (if (keymapp (key-binding (char-to-string last-command-char)))
      (message "%s%c-" (calc-num-prefix-name n) last-command-char))
  (setq unread-command-char last-command-char
	prefix-arg n)
)



;;;; [calc-aent.el]

;;;; Reading an expression in algebraic form.

(defun calc-algebraic-entry ()
  "Read an algebraic expression (e.g., 1+2*3) and push the result on the stack."
  (interactive)
  (calc-wrapper
   (calc-alg-entry (and (memq last-command-char '(?$ ?\"))
			(char-to-string last-command-char))))
)

(defun calc-alg-entry (&optional initial prompt)
  (let* ((sel-mode nil)
	 (calc-dollar-values (mapcar 'calc-get-stack-element
				     (nthcdr calc-stack-top calc-stack)))
	 (calc-dollar-used 0)
	 (calc-plain-entry t)
	 (alg-exp (calc-do-alg-entry initial prompt t)))
    (if (stringp alg-exp)
	(progn
	  (calc-extensions)
	  (calc-alg-edit alg-exp))
      (let ((nvals (mapcar 'calc-normalize alg-exp)))
	(while alg-exp
	  (calc-record (if calc-extensions-loaded (car alg-exp) (car nvals))
		       "alg'")
	  (calc-pop-push-record-list calc-dollar-used
				     (and (not (equal (car alg-exp)
						      (car nvals)))
					  calc-extensions-loaded
					  "")
				     (list (car nvals)))
	  (setq alg-exp (cdr alg-exp)
		nvals (cdr nvals)
		calc-dollar-used 0)))
      (calc-handle-whys)))
)

(defun calc-do-alg-entry (&optional initial prompt no-normalize)
  (let* ((calc-buffer (current-buffer))
	 (blink-paren-hook 'calcAlg-blink-matching-open)
	 (alg-exp 'error))
    (if (boundp 'calc-alg-ent-map)
	()
      (setq calc-alg-ent-map (copy-keymap minibuffer-local-map))
      (define-key calc-alg-ent-map "'" 'calcAlg-previous)
      (define-key calc-alg-ent-map "`" 'calcAlg-edit)
      (define-key calc-alg-ent-map "\ep" 'calcAlg-plus-minus)
      (define-key calc-alg-ent-map "\em" 'calcAlg-mod)
      (define-key calc-alg-ent-map "\C-m" 'calcAlg-enter)
      (define-key calc-alg-ent-map "\C-j" 'calcAlg-enter))
    (setq calc-aborted-prefix nil)
    (let ((buf (read-from-minibuffer (or prompt "Algebraic: ")
				     (or initial "")
				     calc-alg-ent-map nil)))
      (if (eq alg-exp 'error)
	  (if (eq (car-safe (setq alg-exp (math-read-exprs buf))) 'error)
	      (setq alg-exp nil)))
      (setq calc-aborted-prefix "alg'")
      (or no-normalize
	  (and alg-exp (setq alg-exp (mapcar 'calc-normalize alg-exp))))
      alg-exp))
)

(defun calcAlg-plus-minus ()
  (interactive)
  (if (calc-minibuffer-contains ".* \\'")
      (insert "+/- ")
    (insert " +/- "))
)

(defun calcAlg-mod ()
  (interactive)
  (if (not (calc-minibuffer-contains ".* \\'"))
      (insert " "))
  (if (calc-minibuffer-contains ".* mod +\\'")
      (if calc-previous-modulo
	  (insert (math-format-flat-expr calc-previous-modulo 0))
	(beep))
    (insert "mod "))
)

(defun calcAlg-previous ()
  (interactive)
  (if (calc-minibuffer-contains "\\`\\'")
      (if calc-previous-alg-entry
	  (insert calc-previous-alg-entry)
	(beep))
    (insert "'"))
)

(defun calcAlg-edit ()
  (interactive)
  (if (or (not calc-plain-entry)
	  (calc-minibuffer-contains
	   "\\`\\([^\"]*\"[^\"]*\"\\)*[^\"]*\"[^\"]*\\'"))
      (insert "`")
    (setq alg-exp (buffer-string))
    (and (> (length alg-exp) 0) (setq calc-previous-alg-entry alg-exp))
    (exit-minibuffer))
)
(setq calc-plain-entry nil)

(defun calcAlg-enter ()
  (interactive)
  (let* ((str (buffer-string))
	 (exp (and (> (length str) 0)
		   (save-excursion
		     (set-buffer calc-buffer)
		     (math-read-exprs str)))))
    (if (eq (car-safe exp) 'error)
	(progn
	  (goto-char (point-min))
	  (forward-char (nth 1 exp))
	  (beep)
	  (calc-temp-minibuffer-message
	   (concat " [" (or (nth 2 exp) "Error") "]"))
	  (setq unread-command-char -1))
      (setq alg-exp (if (calc-minibuffer-contains "\\` *\\[ *\\'")
			'((incomplete vec))
		      exp))
      (and (> (length str) 0) (setq calc-previous-alg-entry str))
      (exit-minibuffer)))
)

(defun calcAlg-blink-matching-open ()
  (let ((oldpos (point))
	(blinkpos nil))
    (save-excursion
      (condition-case ()
	  (setq blinkpos (scan-sexps oldpos -1))
	(error nil)))
    (if (and blinkpos
	     (> oldpos (1+ (point-min)))
	     (or (and (= (char-after (1- oldpos)) ?\))
		      (= (char-after blinkpos) ?\[))
		 (and (= (char-after (1- oldpos)) ?\])
		      (= (char-after blinkpos) ?\()))
	     (save-excursion
	       (goto-char blinkpos)
	       (looking-at ".+\\(\\.\\.\\|\\\\dots\\|\\\\ldots\\)")))
	(let ((saved (aref (syntax-table) (char-after blinkpos))))
	  (unwind-protect
	      (progn
		(aset (syntax-table) (char-after blinkpos)
		      (+ (logand saved 255)
			 (lsh (char-after (1- oldpos)) 8)))
		(blink-matching-open))
	    (aset (syntax-table) (char-after blinkpos) saved)))
      (blink-matching-open)))
)

;;;; [calc.el]



;;;; Reading a number using the minibuffer.

(defun calcDigit-start ()
  "Begin digit entry in the Calculator."
  (interactive)
  (calc-wrapper
   (if calc-algebraic-mode
       (calc-alg-digit-entry)
     (setq unread-command-char last-command-char
	   calc-aborted-prefix nil)
     (let* ((calc-digit-value nil)
	    (calc-prev-char nil)
	    (calc-prev-prev-char nil)
	    (calc-buffer (current-buffer))
	    (buf (read-from-minibuffer "Calc: " "" calc-digit-map)))
       (or calc-digit-value (setq calc-digit-value (math-read-number buf)))
       (if (stringp calc-digit-value)
	   (calc-alg-entry calc-digit-value)
	 (if calc-digit-value
	     (calc-push-list (list (calc-record (calc-normalize
						 calc-digit-value))))))
       (if (eq calc-prev-char 'dots)
	   (progn
	     (calc-extensions)
	     (calc-dots))))))
)

(defun calcDigit-nondigit ()
  (interactive)
  (let ((str (buffer-string)))
    (setq calc-digit-value (save-excursion
			     (set-buffer calc-buffer)
			     (math-read-number str))))
  (if (and (null calc-digit-value) (> (buffer-size) 0))
      (progn
	(beep)
	(calc-temp-minibuffer-message " [Bad format]"))
    (or (memq last-command-char '(32 13))
	(setq prefix-arg current-prefix-arg
	      unread-command-char last-command-char))
    (exit-minibuffer))
)

;;;; [calc-aent.el]

(defun calc-alg-digit-entry ()
  (calc-alg-entry 
   (cond ((eq last-command-char ?e) "1e")
	 ((eq last-command-char ?#) (format "%d#" calc-number-radix))
	 ((eq last-command-char ?_) "-")
	 ((eq last-command-char ?@) "0@ ")
	 (t (char-to-string last-command-char))))
)

(defun calcDigit-algebraic ()
  (interactive)
  (if (calc-minibuffer-contains ".*[@oh] *[^'m ]+[^'m]*\\'")
      (calcDigit-key)
    (setq calc-digit-value (buffer-string))
    (exit-minibuffer))
)

(defun calcDigit-edit ()
  (interactive)
  (setq unread-command-char last-command-char)
  (setq calc-digit-value (buffer-string))
  (exit-minibuffer)
)

;;;; [calc.el]

(defun calc-minibuffer-contains (rex)
  (save-excursion
    (goto-char (point-min))
    (looking-at rex))
)

(defun calcDigit-key ()
  (interactive)
  (goto-char (point-max))
  (if (or (and (memq last-command-char '(?+ ?-))
	       (> (buffer-size) 0)
	       (/= (preceding-char) ?e))
	  (and (memq last-command-char '(?m ?s))
	       (not (calc-minibuffer-contains "[-+]?[0-9]+\\.?0*[@oh].*"))
	       (not (calc-minibuffer-contains "[-+]?\\(1[1-9]\\|[2-9][0-9]\\)#.*"))))
      (calcDigit-nondigit)
    (if (calc-minibuffer-contains "\\([-+]?\\|.* \\)\\'")
	(cond ((memq last-command-char '(?. ?@)) (insert "0"))
	      ((and (memq last-command-char '(?o ?h ?m))
		    (not (calc-minibuffer-contains ".*#.*"))) (insert "0"))
	      ((memq last-command-char '(?: ?e)) (insert "1"))
	      ((eq last-command-char ?#)
	       (insert (int-to-string calc-number-radix)))))
    (if (and (calc-minibuffer-contains "\\([-+]?[0-9]+#\\|[^:]*:\\)\\'")
	     (eq last-command-char ?:))
	(insert "1"))
    (if (or (and (memq last-command-char '(?e ?h ?o ?m ?s ?p))
		 (calc-minibuffer-contains ".*#.*"))
	    (and (eq last-command-char ?n)
		 (calc-minibuffer-contains "[-+]?\\(2[4-9]\\|[3-9][0-9]\\)#.*")))
	(setq last-command-char (upcase last-command-char)))
    (cond
     ((memq last-command-char '(?_ ?n))
      (goto-char (point-min))
      (if (and (search-forward " +/- " nil t)
	       (not (search-forward "e" nil t)))
	  (beep)
	(and (not (calc-minibuffer-contains ".*#.*"))
	     (search-forward "e" nil t))
	(if (looking-at "+")
	    (delete-char 1))
	(if (looking-at "-")
	    (delete-char 1)
	  (insert "-")))
      (goto-char (point-max)))
     ((eq last-command-char ?p)
      (if (or (calc-minibuffer-contains ".*\\+/-.*")
	      (calc-minibuffer-contains ".*mod.*")
	      (calc-minibuffer-contains ".*#.*")
	      (calc-minibuffer-contains ".*[-+e:]\\'"))
	  (beep)
	(if (not (calc-minibuffer-contains ".* \\'"))
	    (insert " "))
	(insert "+/- ")))
     ((and (eq last-command-char ?M)
	   (not (calc-minibuffer-contains
		 "[-+]?\\(2[3-9]\\|[3-9][0-9]\\)#.*")))
      (if (or (calc-minibuffer-contains ".*\\+/-.*")
	      (calc-minibuffer-contains ".*mod *[^ ]+")
	      (calc-minibuffer-contains ".*[-+e:]\\'"))
	  (beep)
	(if (calc-minibuffer-contains ".*mod \\'")
	    (if calc-previous-modulo
		(insert (math-format-flat-expr calc-previous-modulo 0))
	      (beep))
	  (if (not (calc-minibuffer-contains ".* \\'"))
	      (insert " "))
	  (insert "mod "))))
     (t
      (insert (char-to-string last-command-char))
      (if (or (and (calc-minibuffer-contains "[-+]?\\(.*\\+/- *\\|.*mod *\\)?\\([0-9][0-9]?\\)#[0-9a-zA-Z]*\\(:[0-9a-zA-Z]*\\)?\\(:[0-9a-zA-Z]*\\)?\\'")
		   (let ((radix (string-to-int
				 (buffer-substring
				  (match-beginning 2) (match-end 2)))))
		     (and (>= radix 2)
			  (<= radix 36)
			  (or (memq last-command-char '(?# ?:))
			      (let ((dig (math-read-radix-digit
					  (upcase last-command-char))))
				(and dig
				     (< dig radix)))))))
	      (save-excursion
		(goto-char (point-min))
         	(looking-at
		 "[-+]?\\(.*\\+/- *\\|.*mod *\\)?\\([0-9]+\\.?0*[@oh] *\\)?\\([0-9]+\\.?0*['m] *\\)?[0-9]*\\(\\.?[0-9]*\\(e[-+]?[0-9]*\\)?\\|[0-9]:\\([0-9]+:\\)?[0-9]*\\)?[\"s]?\\'")))
	  (if (and (memq last-command-char '(?@ ?o ?h ?\' ?m))
		   (string-match " " calc-hms-format))
	      (insert " "))
	(if (and (eq this-command last-command)
		 (eq last-command-char ?.))
	    (progn
	      (calc-extensions)
	      (calc-digit-dots))
	  (delete-backward-char 1)
	  (beep)
	  (calc-temp-minibuffer-message " [Bad format]"))))))
  (setq calc-prev-prev-char calc-prev-char
	calc-prev-char last-command-char)
)

;;;; [calc-misc.el]

(defun calcDigit-letter ()
  (interactive)
  (if (calc-minibuffer-contains "[-+]?\\(1[1-9]\\|[2-9][0-9]\\)#.*")
      (progn
	(setq last-command-char (upcase last-command-char))
	(calcDigit-key))
    (calcDigit-nondigit))
)

;;;; [calc.el]

(defun calcDigit-backspace ()
  (interactive)
  (goto-char (point-max))
  (cond ((calc-minibuffer-contains ".* \\+/- \\'")
	 (backward-delete-char 5))
	((calc-minibuffer-contains ".* mod \\'")
	 (backward-delete-char 5))
	((calc-minibuffer-contains ".* \\'")
	 (backward-delete-char 2))
	((eq last-command 'calcDigit-start)
	 (erase-buffer))
	(t (backward-delete-char 1)))
  (if (= (buffer-size) 0)
      (progn
	(setq last-command-char 13)
	(calcDigit-nondigit)))
)

;;;; [calc-misc.el]

(defun calc-temp-minibuffer-message (m)
  "A Lisp version of temp_minibuffer_message from minibuf.c."
  (let ((savemax (point-max)))
    (save-excursion
      (goto-char (point-max))
      (insert m))
    (let ((inhibit-quit t))
      (sit-for 2)
      (delete-region savemax (point-max))
      (if quit-flag
	  (setq quit-flag nil
		unread-command-char 7))))
)

;;;; [calc.el]






;;;; Arithmetic routines.
;;;
;;; An object as manipulated by one of these routines may take any of the
;;; following forms:
;;;
;;; integer                 An integer.  For normalized numbers, this format
;;;			    is used only for -999999 ... 999999.
;;;
;;; (bigpos N0 N1 N2 ...)   A big positive integer, N0 + N1*1000 + N2*10^6 ...
;;; (bigneg N0 N1 N2 ...)   A big negative integer, - N0 - N1*1000 ...
;;;			    Each digit N is in the range 0 ... 999.
;;;			    Normalized, always at least three N present,
;;;			    and the most significant N is nonzero.
;;;
;;; (frac NUM DEN)          A fraction.  NUM and DEN are small or big integers.
;;;                         Normalized, DEN > 1.
;;;
;;; (float NUM EXP)         A floating-point number, NUM * 10^EXP;
;;;                         NUM is a small or big integer, EXP is a small int.
;;;			    Normalized, NUM is not a multiple of 10, and
;;;			    abs(NUM) < 10^calc-internal-prec.
;;;			    Normalized zero is stored as (float 0 0).
;;;
;;; (cplx REAL IMAG)        A complex number; REAL and IMAG are any of above.
;;;			    Normalized, IMAG is nonzero.
;;;
;;; (polar R THETA)         Polar complex number.  Normalized, R > 0 and THETA
;;;                         is neither zero nor 180 degrees (pi radians).
;;;
;;; (vec A B C ...)         Vector of objects A, B, C, ...  A matrix is a
;;;                         vector of vectors.
;;;
;;; (hms H M S)             Angle in hours-minutes-seconds form.  All three
;;;                         components have the same sign; H and M must be
;;;                         numerically integers; M and S are expected to
;;;                         lie in the range [0,60).
;;;
;;; (sdev X SIGMA)          Error form, X +/- SIGMA.  When normalized,
;;;                         SIGMA > 0.  X and SIGMA are any real numbers,
;;;                         or symbolic expressions which are assumed real.
;;;
;;; (intv MASK LO HI)       Interval form.  MASK is 0=(), 1=(], 2=[), or 3=[].
;;;                         LO and HI are any real numbers, or symbolic
;;;			    expressions which are assumed real, and LO < HI.
;;;			    For [LO..HI], if LO = HI normalization produces LO,
;;;			    and if LO > HI normalization produces [LO..LO).
;;;			    For other intervals, if LO > HI normalization
;;;			    sets HI equal to LO.
;;;
;;; (mod N M)	    	    Number modulo M.  When normalized, 0 <= N < M.
;;;			    N and M are real numbers.
;;;
;;; (var V S)		    Symbolic variable.  V is a Lisp symbol which
;;;			    represents the variable's visible name.  S is
;;;			    the symbol which actually stores the variable's
;;;			    value:  (var pi var-pi).
;;;
;;; In general, combining rational numbers in a calculation always produces
;;; a rational result, but if either argument is a float, result is a float.

;;; In the following comments, [x y z] means result is x, args must be y, z,
;;; respectively, where the code letters are:
;;;
;;;    O  Normalized object (vector or number)
;;;    V  Normalized vector
;;;    N  Normalized number of any type
;;;    N  Normalized complex number
;;;    R  Normalized real number (float or rational)
;;;    F  Normalized floating-point number
;;;    T  Normalized rational number
;;;    I  Normalized integer
;;;    B  Normalized big integer
;;;    S  Normalized small integer
;;;    D  Digit (small integer, 0..999)
;;;    L  Normalized bignum digit list (without "bigpos" or "bigneg" symbol)
;;;       or normalized vector element list (without "vec")
;;;    P  Predicate (truth value)
;;;    X  Any Lisp object
;;;    Z  "nil"
;;;
;;; Lower-case letters signify possibly un-normalized values.
;;; "L.D" means a cons of an L and a D.
;;; [N N; n n] means result will be normalized if argument is.
;;; Also, [Public] marks routines intended to be called from outside.
;;; [This notation has been neglected in many recent routines.]

;;; Reduce an object to canonical (normalized) form.  [O o; Z Z] [Public]
(defun math-normalize (a)
  (cond
   ((not (consp a))
    (if (integerp a)
	(if (or (>= a 1000000) (<= a -1000000))
	    (math-bignum a)
	  a)
      a))
   ((eq (car a) 'bigpos)
    (if (eq (nth (1- (length a)) a) 0)
	(let* ((last (setq a (copy-sequence a))) (digs a))
	  (while (setq digs (cdr digs))
	    (or (eq (car digs) 0) (setq last digs)))
	  (setcdr last nil)))
    (if (cdr (cdr (cdr a)))
	a
      (cond
       ((cdr (cdr a)) (+ (nth 1 a) (* (nth 2 a) 1000)))
       ((cdr a) (nth 1 a))
       (t 0))))
   ((eq (car a) 'bigneg)
    (if (eq (nth (1- (length a)) a) 0)
	(let* ((last (setq a (copy-sequence a))) (digs a))
	  (while (setq digs (cdr digs))
	    (or (eq (car digs) 0) (setq last digs)))
	  (setcdr last nil)))
    (if (cdr (cdr (cdr a)))
	a
      (cond
       ((cdr (cdr a)) (- (+ (nth 1 a) (* (nth 2 a) 1000))))
       ((cdr a) (- (nth 1 a)))
       (t 0))))
   ((eq (car a) 'float)
    (math-make-float (math-normalize (nth 1 a)) (nth 2 a)))
   ((or (memq (car a) '(frac cplx polar hms mod sdev intv vec var quote
			     special-const calcFunc-if calcFunc-lambda
			     calcFunc-quote))
	(integerp (car a))
	(and (consp (car a)) (not (eq (car (car a)) 'lambda))))
    (calc-extensions)
    (math-normalize-fancy a))
   (t
    (or (and calc-simplify-mode
	     (calc-extensions)
	     (math-normalize-nonstandard))
	(let ((args (mapcar 'math-normalize (cdr a))))
	  (or (condition-case err
		  (let ((func (assq (car a) '( ( + . math-add )
					       ( - . math-sub )
					       ( * . math-mul )
					       ( / . math-div )
					       ( % . math-mod )
					       ( ^ . math-pow )
					       ( neg . math-neg )
					       ( | . math-concat ) ))))
		    (if func
			(apply (cdr func) args)
		      (and (or (consp (car a))
			       (fboundp (car a))
			       (and (not calc-extensions-loaded)
				    (calc-extensions)
				    (fboundp (car a))))
			   (apply (car a) args))))
		(wrong-number-of-arguments
		 (calc-record-why "Wrong number of arguments") nil)
		(wrong-type-argument
		 (or calc-next-why (calc-record-why "Wrong type of argument"))
		 nil)
		(args-out-of-range
		 (calc-record-why "Argument out of range") nil)
		(inexact-result
		 (calc-record-why "No exact representation for result") nil))
	      (if (consp (car a))
		  (math-dimension-error)
		(cons (car a) args)))))))
)

;;;; [calc-macs.el]

(defmacro math-with-extra-prec (delta &rest body)
  (` (math-normalize
      (let ((calc-internal-prec (+ calc-internal-prec (, delta))))
	(,@ body))))
)
(put 'math-with-extra-prec 'lisp-indent-hook 1)


;;;; [calc-misc.el]

;;; Concatenate two vectors, or a vector and an object.  [V O O] [Public]
(defun math-concat (v1 v2)
  (if (stringp v1)
      (concat v1 v2)
    (calc-extensions)
    (if (and (math-objvecp v1) (math-objvecp v2))
	(append (if (and (math-vectorp v1)
			 (or (math-matrixp v1)
			     (not (math-matrixp v2))))
		    v1
		  (list 'vec v1))
		(if (and (math-vectorp v2)
			 (or (math-matrixp v2)
			     (not (math-matrixp v1))))
		    (cdr v2)
		  (list v2)))
      (list '| v1 v2)))
)
(defun calcFunc-vconcat (a b)
  (math-normalize (list '| a b))
)


;;; True if A is zero.  Works for un-normalized values.  [P n] [Public]
(defun math-zerop (a)
  (if (consp a)
      (cond ((memq (car a) '(bigpos bigneg))
	     (while (eq (car (setq a (cdr a))) 0))
	     (null a))
	    ((memq (car a) '(frac float polar mod))
	     (math-zerop (nth 1 a)))
	    ((eq (car a) 'cplx)
	     (and (math-zerop (nth 1 a)) (math-zerop (nth 2 a))))
	    ((eq (car a) 'hms)
	     (and (math-zerop (nth 1 a))
		  (math-zerop (nth 2 a))
		  (math-zerop (nth 3 a)))))
    (eq a 0))
)

;;;; [calc-macs.el]

;;; Faster in-line version zerop, normalized values only.
(defmacro Math-zerop (a)   ; [P N]
  (` (if (consp (, a))
	 (and (not (memq (car (, a)) '(bigpos bigneg)))
	      (if (eq (car (, a)) 'float)
		  (eq (nth 1 (, a)) 0)
		(math-zerop (, a))))
       (eq (, a) 0)))
)

(defmacro Math-natnum-lessp (a b)
  (` (if (consp (, a))
	 (and (consp (, b))
	      (= (math-compare-bignum (cdr (, a)) (cdr (, b))) -1))
       (or (consp (, b))
	   (< (, a) (, b)))))
)

(defmacro Math-integer-negp (a)
  (` (if (consp (, a))
	 (eq (car (, a)) 'bigneg)
       (< (, a) 0)))
)

(defmacro Math-integer-posp (a)
  (` (if (consp (, a))
	 (eq (car (, a)) 'bigpos)
       (> (, a) 0)))
)

;;;; [calc-misc.el]

;;; True if A is real and negative.  [P n] [Public]

(defun math-negp (a)
  (if (consp a)
      (cond ((eq (car a) 'bigpos) nil)
	    ((eq (car a) 'bigneg) (cdr a))
	    ((memq (car a) '(float frac))
	     (Math-integer-negp (nth 1 a)))
	    ((eq (car a) 'hms)
	     (if (math-zerop (nth 1 a))
		 (if (math-zerop (nth 2 a))
		     (math-negp (nth 3 a))
		   (math-negp (nth 2 a)))
	       (math-negp (nth 1 a))))
	    ((eq (car a) 'intv)
	     (or (math-negp (nth 3 a))
		 (and (math-zerop (nth 3 a))
		      (memq (nth 1 a) '(0 2))))))
    (< a 0))
)

;;;; [calc-macs.el]

(defmacro Math-negp (a)
  (` (if (consp (, a))
	 (or (eq (car (, a)) 'bigneg)
	     (and (not (eq (car (, a)) 'bigpos))
		  (if (memq (car (, a)) '(frac float))
		      (Math-integer-negp (nth 1 (, a)))
		    (math-negp (, a)))))
       (< (, a) 0)))
)

;;;; [calc-misc.el]

;;; True if A is a negative number or an expression the starts with '-'.
(defun math-looks-negp (a)   ; [P x] [Public]
  (or (Math-negp a)
      (eq (car-safe a) 'neg)
      (and (memq (car-safe a) '(* /))
	   (or (math-looks-negp (nth 1 a))
	       (math-looks-negp (nth 2 a)))))
)

;;;; [calc-macs.el]

(defmacro Math-looks-negp (a)   ; [P x] [Public]
  (` (or (Math-negp (, a))
	 (and (consp (, a)) (or (eq (car (, a)) 'neg)
				(and (memq (car (, a)) '(* /))
				     (or (math-looks-negp (nth 1 (, a)))
					 (math-looks-negp (nth 2 (, a)))))))))
)

;;;; [calc-misc.el]

;;; True if A is real and positive.  [P n] [Public]
(defun math-posp (a)
  (if (consp a)
      (cond ((eq (car a) 'bigpos) (cdr a))
	    ((eq (car a) 'bigneg) nil)
	    ((memq (car a) '(float frac))
	     (Math-integer-posp (nth 1 a)))
	    ((eq (car a) 'hms)
	     (if (math-zerop (nth 1 a))
		 (if (math-zerop (nth 2 a))
		     (math-posp (nth 3 a))
		   (math-posp (nth 2 a)))
	       (math-posp (nth 1 a))))
	    ((eq (car a) 'mod)
	     (not (math-zerop (nth 1 a))))
	    ((eq (car a) 'intv)
	     (or (math-posp (nth 2 a))
		 (and (math-zerop (nth 2 a))
		      (memq (nth 1 a) '(0 1))))))
    (> a 0))
)

;;;; [calc-macs.el]

(defmacro Math-posp (a)
  (` (if (consp (, a))
	 (or (eq (car (, a)) 'bigpos)
	     (and (not (eq (car (, a)) 'bigneg))
		  (if (memq (car (, a)) '(frac float))
		      (Math-integer-posp (nth 1 (, a)))
		    (math-posp (, a)))))
       (> (, a) 0)))
)


(defmacro Math-integerp (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg))))
)

;;;; [calc-misc.el]

(fset 'math-fixnump (symbol-function 'integerp))
(fset 'math-fixnatnump (symbol-function 'natnump))

;;;; [calc-macs.el]

(defmacro Math-natnump (a)
  (` (if (consp (, a))
	 (eq (car (, a)) 'bigpos)
       (>= (, a) 0)))
)

(defmacro Math-ratp (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg frac))))
)

(defmacro Math-realp (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg frac float))))
)

(defmacro Math-anglep (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg frac float hms))))
)

;;;; [calc.el]

;;; True if A is a floating-point real or complex number.  [P x] [Public]
(defun math-floatp (a)
  (or (eq (car-safe a) 'float)
      (and (memq (car-safe a) '(cplx polar mod sdev intv))
	   (or (math-floatp (nth 1 a))
	       (math-floatp (nth 2 a))
	       (and (eq (car a) 'intv) (math-floatp (nth 3 a))))))
)

;;;; [calc-macs.el]

(defmacro Math-numberp (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg frac float cplx polar))))
)

(defmacro Math-scalarp (a)
  (` (or (not (consp (, a)))
	 (memq (car (, a)) '(bigpos bigneg frac float cplx polar hms))))
)

(defmacro Math-vectorp (a)
  (` (and (consp (, a)) (eq (car (, a)) 'vec)))
)

(defmacro Math-messy-integerp (a)
  (` (and (consp (, a))
	  (eq (car (, a)) 'float)
	  (>= (nth 2 (, a)) 0)))
)

(defmacro Math-objvecp (a)    ;  [Public]
  (` (or (not (consp (, a)))
	 (memq (car (, a))
	       '(bigpos bigneg frac float cplx polar hms sdev intv mod vec))))
)


;;;; [calc-misc.el]

;;; True if A is an even integer.  [P R R] [Public]
(defun math-evenp (a)
  (if (consp a)
      (and (memq (car a) '(bigpos bigneg))
	   (= (% (nth 1 a) 2) 0))
    (= (% a 2) 0))
)

;;; Compute A / 2, for small or big integer A.  [I i]
;;; If A is negative, type of truncation is undefined.
(defun math-div2 (a)
  (if (consp a)
      (if (cdr a)
	  (math-normalize (cons (car a) (math-div2-bignum (cdr a))))
	0)
    (/ a 2))
)

(defun math-div2-bignum (a)   ; [l l]
  (if (cdr a)
      (cons (+ (/ (car a) 2) (* (% (nth 1 a) 2) 500))
	    (math-div2-bignum (cdr a)))
    (list (/ (car a) 2)))
)

;;;; [calc.el]


;;; Verify that A is a complete object and return A.  [x x] [Public]
(defun math-check-complete (a)
  (cond ((integerp a) a)
	((eq (car-safe a) 'incomplete)
	 (calc-incomplete-error a))
	((consp a) a)
	(t (error "Invalid data object encountered")))
)

;;;; [calc-misc.el]

;;; Reject an argument to a calculator function.  [Public]
(defun math-reject-arg (&optional a p)
  (calc-record-why p a)
  (signal 'wrong-type-argument (and a (if p (list p a) (list a))))
)


;;; Coerce A to be an integer (by truncation toward zero).  [I N] [Public]
(defun math-trunc (a)
  (cond ((Math-integerp a) a)
	((Math-looks-negp a)
	 (math-neg (math-trunc (math-neg a))))
	((eq (car a) 'float) (math-scale-int (nth 1 a) (nth 2 a)))
	(t (calc-extensions)
	   (math-trunc-fancy a)))
)
(fset 'calcFunc-trunc (symbol-function 'math-trunc))

;;; Coerce A to be an integer (by truncation toward minus infinity).  [I N]
(defun math-floor (a)    ;  [Public]
  (cond ((Math-integerp a) a)
	((Math-messy-integerp a) (math-trunc a))
	((Math-realp a)
	 (if (Math-negp a)
	     (math-add (math-trunc a) -1)
	   (math-trunc a)))
	(t (calc-extensions)
	   (math-floor-fancy a)))
)
(fset 'calcFunc-floor (symbol-function 'math-floor))

;;;; [calc.el]


;;; Coerce integer A to be a bignum.  [B S]
(defun math-bignum (a)
  (if (>= a 0)
      (cons 'bigpos (math-bignum-big a))
    (cons 'bigneg (math-bignum-big (- a))))
)

(defun math-bignum-big (a)   ; [L s]
  (if (= a 0)
      nil
    (cons (% a 1000) (math-bignum-big (/ a 1000))))
)


;;; Build a normalized floating-point number.  [F I S]
(defun math-make-float (mant exp)
  (if (eq mant 0)
      '(float 0 0)
    (let* ((ldiff (- calc-internal-prec (math-numdigs mant))))
      (if (< ldiff 0)
	  (setq mant (math-scale-rounding mant ldiff)
		exp (- exp ldiff))))
    (if (consp mant)
	(let ((digs (cdr mant)))
	  (if (= (% (car digs) 10) 0)
	      (progn
		(while (= (car digs) 0)
		  (setq digs (cdr digs)
			exp (+ exp 3)))
		(while (= (% (car digs) 10) 0)
		  (setq digs (math-div10-bignum digs)
			exp (1+ exp)))
		(setq mant (math-normalize (cons (car mant) digs))))))
      (while (= (% mant 10) 0)
	(setq mant (/ mant 10)
	      exp (1+ exp))))
    (list 'float mant exp))
)

(defun math-div10-bignum (a)   ; [l l]
  (if (cdr a)
      (cons (+ (/ (car a) 10) (* (% (nth 1 a) 10) 100))
	    (math-div10-bignum (cdr a)))
    (list (/ (car a) 10)))
)

;;; Coerce A to be a float.  [F N; V V] [Public]
(defun math-float (a)
  (cond ((Math-integerp a) (math-make-float a 0))
	((eq (car a) 'frac) (math-div (math-float (nth 1 a)) (nth 2 a)))
	((eq (car a) 'float) a)
	((memq (car a) '(cplx polar vec hms sdev mod))
	 (cons (car a) (mapcar 'math-float (cdr a))))
	((eq (car a) 'intv)
	 (cons (car a) (cons (nth 1 a) (mapcar 'math-float (nthcdr 2 a)))))
	((and (memq (car a) '(* /))
	      (math-numberp (nth 1 a)))
	 (list (car a) (math-float (nth 1 a))
	       (list 'calcFunc-float (nth 2 a))))
	((and (eq (car a) '/)
	      (eq (car (nth 1 a)) '*)
	      (math-numberp (nth 1 (nth 1 a))))
	 (list '* (math-float (nth 1 (nth 1 a)))
	       (list 'calcFunc-float (list '/ (nth 2 (nth 1 a)) (nth 2 a)))))
	(t (math-reject-arg a 'objectp)))
)
(fset 'calcFunc-float (symbol-function 'math-float))

;;;; [calc-macs.el]

;;; Compute the negative of A.  [O O; o o] [Public]
(defmacro Math-integer-neg (a)
  (` (if (consp (, a))
	 (if (eq (car (, a)) 'bigpos)
	     (cons 'bigneg (cdr (, a)))
	   (cons 'bigpos (cdr (, a))))
       (- (, a))))
)

;;;; [calc.el]

(defun math-neg (a)
  (cond ((not (consp a)) (- a))
	((eq (car a) 'bigpos) (cons 'bigneg (cdr a)))
	((eq (car a) 'bigneg) (cons 'bigpos (cdr a)))
	((memq (car a) '(frac float))
	 (list (car a) (Math-integer-neg (nth 1 a)) (nth 2 a)))
	((memq (car a) '(cplx vec hms))
	 (cons (car a) (mapcar 'math-neg (cdr a))))
	(t (math-neg-fancy a)))
)

;;;; [calc-misc.el]

(defun calcFunc-neg (a)
  (math-normalize (list 'neg a))
)

;;;; [calc.el]


;;; Compute the number of decimal digits in integer A.  [S I]
(defun math-numdigs (a)
  (if (consp a)
      (if (cdr a)
	  (let* ((len (1- (length a)))
		 (top (nth len a)))
	    (+ (* len 3) (cond ((>= top 100) 0) ((>= top 10) -1) (t -2))))
	0)
    (cond ((>= a 100) (+ (math-numdigs (/ a 1000)) 3))
	  ((>= a 10) 2)
	  ((>= a 1) 1)
	  ((= a 0) 0)
	  ((> a -10) 1)
	  ((> a -100) 2)
	  (t (math-numdigs (- a)))))
)

;;; Multiply (with truncation toward 0) the integer A by 10^N.  [I i S]
(defun math-scale-int (a n)
  (cond ((= n 0) a)
	((> n 0) (math-scale-left a n))
	(t (math-normalize (math-scale-right a (- n)))))
)

(defun math-scale-left (a n)   ; [I I S]
  (if (= n 0)
      a
    (if (consp a)
	(cons (car a) (math-scale-left-bignum (cdr a) n))
      (if (>= n 3)
	  (if (or (>= a 1000) (<= a -1000))
	      (math-scale-left (math-bignum a) n)
	    (math-scale-left (* a 1000) (- n 3)))
	(if (= n 2)
	    (if (or (>= a 10000) (<= a -10000))
		(math-scale-left (math-bignum a) 2)
	      (* a 100))
	  (if (or (>= a 100000) (<= a -100000))
	      (math-scale-left (math-bignum a) 1)
	    (* a 10))))))
)

(defun math-scale-left-bignum (a n)
  (if (>= n 3)
      (while (>= (setq a (cons 0 a)
		       n (- n 3)) 3)))
  (if (> n 0)
      (math-mul-bignum-digit a (if (= n 2) 100 10) 0)
    a)
)

(defun math-scale-right (a n)   ; [i i S]
  (if (= n 0)
      a
    (if (consp a)
	(cons (car a) (math-scale-right-bignum (cdr a) n))
      (if (<= a 0)
	  (if (= a 0)
	      0
	    (- (math-scale-right (- a) n)))
	(if (>= n 3)
	    (while (and (> (setq a (/ a 1000)) 0)
			(>= (setq n (- n 3)) 3))))
	(if (= n 2)
	    (/ a 100)
	  (if (= n 1)
	      (/ a 10)
	    a)))))
)

(defun math-scale-right-bignum (a n)   ; [L L S; l l S]
  (if (>= n 3)
      (setq a (nthcdr (/ n 3) a)
	    n (% n 3)))
  (if (> n 0)
      (cdr (math-mul-bignum-digit a (if (= n 2) 10 100) 0))
    a)
)

;;; Multiply (with rounding) the integer A by 10^N.   [I i S]
(defun math-scale-rounding (a n)
  (cond ((>= n 0)
	 (math-scale-left a n))
	((consp a)
	 (math-normalize
	  (cons (car a)
		(let ((val (if (< n -3)
			       (math-scale-right-bignum (cdr a) (- -3 n))
			     (if (= n -2)
				 (math-mul-bignum-digit (cdr a) 10 0)
			       (if (= n -1)
				   (math-mul-bignum-digit (cdr a) 100 0)
				 (cdr a))))))  ; n = -3
		  (if (and val (>= (car val) 500))
		      (if (cdr val)
			  (if (eq (car (cdr val)) 999)
			      (math-add-bignum (cdr val) '(1))
			    (cons (1+ (car (cdr val))) (cdr (cdr val))))
			'(1))
		    (cdr val))))))
	(t
	 (if (< a 0)
	     (- (math-scale-rounding (- a) n))
	   (if (= n -1)
	       (/ (+ a 5) 10)
	     (/ (+ (math-scale-right a (- -1 n)) 5) 10)))))
)


;;; Compute the sum of A and B.  [O O O] [Public]
(defun math-add (a b)
  (or
   (and (not (or (consp a) (consp b)))
	(progn
	  (setq a (+ a b))
	  (if (or (<= a -1000000) (>= a 1000000))
	      (math-bignum a)
	    a)))
   (and (Math-zerop a) (not (eq (car-safe a) 'mod))
	(if (and (math-floatp a) (Math-ratp b)) (math-float b) b))
   (and (Math-zerop b) (not (eq (car-safe b) 'mod))
	(if (and (math-floatp b) (Math-ratp a)) (math-float a) a))
   (and (Math-objvecp a) (Math-objvecp b)
	(or
	 (and (Math-integerp a) (Math-integerp b)
	      (progn
		(or (consp a) (setq a (math-bignum a)))
		(or (consp b) (setq b (math-bignum b)))
		(if (eq (car a) 'bigneg)
		    (if (eq (car b) 'bigneg)
			(cons 'bigneg (math-add-bignum (cdr a) (cdr b)))
		      (math-normalize
		       (let ((diff (math-sub-bignum (cdr b) (cdr a))))
			 (if (eq diff 'neg)
			     (cons 'bigneg (math-sub-bignum (cdr a) (cdr b)))
			   (cons 'bigpos diff)))))
		  (if (eq (car b) 'bigneg)
		      (math-normalize
		       (let ((diff (math-sub-bignum (cdr a) (cdr b))))
			 (if (eq diff 'neg)
			     (cons 'bigneg (math-sub-bignum (cdr b) (cdr a)))
			   (cons 'bigpos diff))))
		    (cons 'bigpos (math-add-bignum (cdr a) (cdr b)))))))
	 (and (Math-ratp a) (Math-ratp b)
	      (calc-extensions)
	      (calc-add-fractions a b))
	 (and (Math-realp a) (Math-realp b)
	      (progn
		(or (and (consp a) (eq (car a) 'float))
		    (setq a (math-float a)))
		(or (and (consp b) (eq (car b) 'float))
		    (setq b (math-float b)))
		(math-add-float a b)))
	 (and (calc-extensions)
	      (math-add-objects-fancy a b))))
   (and (calc-extensions)
	(math-add-symb-fancy a b)))
)

;;;; [calc-misc.el]

(defun calcFunc-add (&rest rest)
  (if rest
      (let ((a (car rest)))
	(while (setq rest (cdr rest))
	  (setq a (list '+ a (car rest))))
	(math-normalize a))
    0)
)

;;;; [calc.el]

(defun math-add-bignum (a b)   ; [L L L; l l l]
  (if a
      (if b
	  (let* ((a (copy-sequence a)) (aa a) (carry nil) sum)
	    (while (and aa b)
	      (if carry
		  (if (< (setq sum (+ (car aa) (car b))) 999)
		      (progn
			(setcar aa (1+ sum))
			(setq carry nil))
		    (setcar aa (+ sum -999)))
		(if (< (setq sum (+ (car aa) (car b))) 1000)
		    (setcar aa sum)
		  (setcar aa (+ sum -1000))
		  (setq carry t)))
	      (setq aa (cdr aa)
		    b (cdr b)))
	    (if carry
		(if b
		    (nconc a (math-add-bignum b '(1)))
		  (while (eq (car aa) 999)
		    (setcar aa 0)
		    (setq aa (cdr aa)))
		  (if aa
		      (progn
			(setcar aa (1+ (car aa)))
			a)
		    (nconc a '(1))))
	      (if b
		  (nconc a b)
		a)))
	a)
    b)
)

(defun math-sub-bignum (a b)   ; [l l l]
  (if b
      (if a
	  (let* ((a (copy-sequence a)) (aa a) (borrow nil) sum)
	    (while (and aa b)
	      (if borrow
		  (if (>= (setq diff (- (car aa) (car b))) 1)
		      (progn
			(setcar aa (1- diff))
			(setq borrow nil))
		    (setcar aa (+ diff 999)))
		(if (>= (setq diff (- (car aa) (car b))) 0)
		    (setcar aa diff)
		  (setcar aa (+ diff 1000))
		  (setq borrow t)))
	      (setq aa (cdr aa)
		    b (cdr b)))
	    (if borrow
		(progn
		  (while (eq (car aa) 0)
		    (setcar aa 999)
		    (setq aa (cdr aa)))
		  (if aa
		      (progn
			(setcar aa (1- (car aa)))
			a)
		    'neg))
	      (while (eq (car b) 0)
		(setq b (cdr b)))
	      (if b
		  'neg
		a)))
	(while (eq (car b) 0)
	  (setq b (cdr b)))
	(and b
	     'neg))
    a)
)

(defun math-add-float (a b)   ; [F F F]
  (let ((ediff (- (nth 2 a) (nth 2 b))))
    (if (>= ediff 0)
	(if (>= ediff (+ calc-internal-prec calc-internal-prec))
	    a
	  (math-make-float (math-add (nth 1 b)
				     (math-scale-int (nth 1 a) ediff))
			   (nth 2 b)))
      (if (>= (setq ediff (- ediff))
	      (+ calc-internal-prec calc-internal-prec))
	  b
	(math-make-float (math-add (nth 1 a)
				   (math-scale-int (nth 1 b) ediff))
			 (nth 2 a)))))
)

;;; Compute the difference of A and B.  [O O O] [Public]
(defun math-sub (a b)
  (if (or (consp a) (consp b))
      (math-add a (math-neg b))
    (setq a (- a b))
    (if (or (<= a -1000000) (>= a 1000000))
	(math-bignum a)
      a))
)

;;;; [calc-misc.el]

(defun calcFunc-sub (&rest rest)
  (if rest
      (let ((a (car rest)))
	(while (setq rest (cdr rest))
	  (setq a (list '- a (car rest))))
	(math-normalize a))
    0)
)

;;;; [calc.el]

(defun math-sub-float (a b)   ; [F F F]
  (let ((ediff (- (nth 2 a) (nth 2 b))))
    (if (>= ediff 0)
	(if (>= ediff (+ calc-internal-prec calc-internal-prec))
	    a
	  (math-make-float (math-add (Math-integer-neg (nth 1 b))
				     (math-scale-int (nth 1 a) ediff))
			   (nth 2 b)))
      (if (>= (setq ediff (- ediff))
	      (+ calc-internal-prec calc-internal-prec))
	  b
	(math-make-float (math-add (nth 1 a)
				   (Math-integer-neg
				    (math-scale-int (nth 1 b) ediff)))
			 (nth 2 a)))))
)


;;; Compute the product of A and B.  [O O O] [Public]
(defun math-mul (a b)
  (or
   (and (not (consp a)) (not (consp b))
	(< a 1000) (> a -1000) (< b 1000) (> b -1000)
	(* a b))
   (and (Math-zerop a) (not (eq (car-safe b) 'mod))
	(if (math-matrixp b)
	    (math-diag-matrix 0 (1- (length b)))
	  (if (and (math-floatp b) (Math-ratp a)) (math-float a) a)))
   (and (Math-zerop b) (not (eq (car-safe a) 'mod))
	(if (math-matrixp a)
	    (math-diag-matrix 0 (1- (length a)))
	  (if (and (math-floatp a) (Math-ratp b)) (math-float b) b)))
   (and (Math-objvecp a) (Math-objvecp b)
	(or
	 (and (Math-integerp a) (Math-integerp b)
	      (progn
		(or (consp a) (setq a (math-bignum a)))
		(or (consp b) (setq b (math-bignum b)))
		(math-normalize
		 (cons (if (eq (car a) (car b)) 'bigpos 'bigneg)
		       (if (cdr (cdr a))
			   (if (cdr (cdr b))
			       (math-mul-bignum (cdr a) (cdr b))
			     (math-mul-bignum-digit (cdr a) (nth 1 b) 0))
			 (math-mul-bignum-digit (cdr b) (nth 1 a) 0))))))
	 (and (Math-ratp a) (Math-ratp b)
	      (calc-extensions)
	      (calc-mul-fractions a b))
	 (and (Math-realp a) (Math-realp b)
	      (progn
		(or (and (consp a) (eq (car a) 'float))
		    (setq a (math-float a)))
		(or (and (consp b) (eq (car b) 'float))
		    (setq b (math-float b)))
		(math-make-float (math-mul (nth 1 a) (nth 1 b))
				 (+ (nth 2 a) (nth 2 b)))))
	 (and (calc-extensions)
	      (math-mul-objects-fancy a b))))
   (and (calc-extensions)
	(math-mul-symb-fancy a b)))
)

;;;; [calc-misc.el]

(defun calcFunc-mul (&rest rest)
  (if rest
      (let ((a (car rest)))
	(while (setq rest (cdr rest))
	  (setq a (list '* a (car rest))))
	(math-normalize a))
    1)
)

;;;; [calc.el]

;;; Multiply digit lists A and B.  [L L L; l l l]
(defun math-mul-bignum (a b)
  (and a b
       (let* ((sum (if (<= (car b) 1)
		       (if (= (car b) 0)
			   (list 0)
			 (copy-sequence a))
		     (math-mul-bignum-digit a (car b) 0)))
	      (sump sum) c d aa prod)
	 (while (setq b (cdr b))
	   (setq ss (setq sump (or (cdr sump) (setcdr sump (list 0))))
		 d (car b)
		 c 0
		 aa a)
	   (while (progn
		    (setcar ss (% (setq prod (+ (+ (car ss) (* (car aa) d))
						c)) 1000))
		    (setq aa (cdr aa)))
	     (setq c (/ prod 1000)
		   ss (or (cdr ss) (setcdr ss (list 0)))))
	   (if (>= prod 1000)
	       (if (cdr ss)
		   (setcar (cdr ss) (+ (/ prod 1000) (car (cdr ss))))
		 (setcdr ss (list (/ prod 1000))))))
	 sum))
)

;;; Multiply digit list A by digit D.  [L L D D; l l D D]
(defun math-mul-bignum-digit (a d c)
  (if a
      (if (<= d 1)
	  (and (= d 1) a)
	(let* ((a (copy-sequence a)) (aa a) prod)
	  (while (progn
		   (setcar aa (% (setq prod (+ (* (car aa) d) c)) 1000))
		   (cdr aa))
	    (setq aa (cdr aa)
		  c (/ prod 1000)))
	  (if (>= prod 1000)
	      (setcdr aa (list (/ prod 1000))))
	  a))
    (and (> c 0)
	 (list c)))
)


;;;; [calc-misc.el]

;;; Compute the square of A.  [O O] [Public]
(defun math-sqr (a)
  (if (eq (car-safe a) 'calcFunc-sqrt)
      (nth 1 a)
    (math-mul a a))
)

;;;; [calc.el]


;;; Compute the integer (quotient . remainder) of A and B, which may be
;;; small or big integers.  Type and consistency of truncation is undefined
;;; if A or B is negative.  B must be nonzero.  [I.I I I] [Public]
(defun math-idivmod (a b)
  (if (eq b 0)
      (math-reject-arg a "Division by zero"))
  (if (or (consp a) (consp b))
      (if (and (natnump b) (< b 1000))
	  (let ((res (math-div-bignum-digit (cdr a) b)))
	    (cons
	     (math-normalize (cons (car a) (car res)))
	     (cdr res)))
	(or (consp a) (setq a (math-bignum a)))
	(or (consp b) (setq b (math-bignum b)))
	(let ((res (math-div-bignum (cdr a) (cdr b))))
	  (cons
	   (math-normalize (cons (if (eq (car a) (car b)) 'bigpos 'bigneg)
				 (car res)))
	   (math-normalize (cons (car a) (cdr res))))))
    (cons (/ a b) (% a b)))
)

(defun math-quotient (a b)   ; [I I I] [Public]
  (if (and (not (consp a)) (not (consp b)))
      (if (= b 0)
	  (math-reject-arg a "Division by zero")
	(/ a b))
    (if (and (natnump b) (< b 1000))
	(if (= b 0)
	    (math-reject-arg a "Division by zero")
	  (math-normalize (cons (car a)
				(car (math-div-bignum-digit (cdr a) b)))))
      (or (consp a) (setq a (math-bignum a)))
      (or (consp b) (setq b (math-bignum b)))
      (let* ((alen (1- (length a)))
	     (blen (1- (length b)))
	     (d (/ 1000 (1+ (nth (1- blen) (cdr b)))))
	     (res (math-div-bignum-big (math-mul-bignum-digit (cdr a) d 0)
				       (math-mul-bignum-digit (cdr b) d 0)
				       alen blen)))
	(math-normalize (cons (if (eq (car a) (car b)) 'bigpos 'bigneg)
			      (car res))))))
)

;;;; [calc-misc.el]

(defun math-imod (a b)   ; [I I I] [Public]
  (if (and (not (consp a)) (not (consp b)))
      (if (= b 0)
	  (math-reject-arg a "Division by zero")
	(% a b))
    (cdr (math-idivmod a b)))
)

;;;; [calc.el]

;;; Divide a bignum digit list by another.  [l.l l L]
;;; The following division algorithm is borrowed from Knuth vol. II, sec. 4.3.1
(defun math-div-bignum (a b)
  (if (cdr b)
      (let* ((alen (length a))
	     (blen (length b))
	     (d (/ 1000 (1+ (nth (1- blen) b))))
	     (res (math-div-bignum-big (math-mul-bignum-digit a d 0)
				       (math-mul-bignum-digit b d 0)
				       alen blen)))
	(if (= d 1)
	    res
	  (cons (car res)
		(car (math-div-bignum-digit (cdr res) d)))))
    (let ((res (math-div-bignum-digit a (car b))))
      (cons (car res) (list (cdr res)))))
)

;;; Divide a bignum digit list by a digit.  [l.D l D]
(defun math-div-bignum-digit (a b)
  (if a
      (let* ((res (math-div-bignum-digit (cdr a) b))
	     (num (+ (* (cdr res) 1000) (car a))))
	(cons
	 (cons (/ num b) (car res))
	 (% num b)))
    '(nil . 0))
)

(defun math-div-bignum-big (a b alen blen)   ; [l.l l L]
  (if (< alen blen)
      (cons nil a)
    (let* ((res (math-div-bignum-big (cdr a) b (1- alen) blen))
	   (num (cons (car a) (cdr res)))
	   (res2 (math-div-bignum-part num b blen)))
      (cons
       (cons (car res2) (car res))
       (cdr res2))))
)

(defun math-div-bignum-part (a b blen)   ; a < b*1000  [D.l l L]
  (let* ((num (+ (* (or (nth blen a) 0) 1000) (or (nth (1- blen) a) 0)))
	 (den (nth (1- blen) b))
	 (guess (min (/ num den) 999)))
    (math-div-bignum-try a b (math-mul-bignum-digit b guess 0) guess))
)

(defun math-div-bignum-try (a b c guess)   ; [D.l l l D]
  (let ((rem (math-sub-bignum a c)))
    (if (eq rem 'neg)
	(math-div-bignum-try a b (math-sub-bignum c b) (1- guess))
      (cons guess rem)))
)


;;; Compute the quotient of A and B.  [O O N] [Public]
(defun math-div (a b)
  (or
   (and (Math-zerop b)
	(math-reject-arg a "Division by zero"))
   (and (Math-zerop a) (not (eq (car-safe b) 'mod))
	(if (and (math-floatp b) (Math-ratp a)) (math-float a) a))
   (and (Math-objvecp a) (Math-objvecp b)
	(or
	 (and (Math-integerp a) (Math-integerp b)
	      (let ((q (math-idivmod a b)))
		(if (eq (cdr q) 0)
		    (car q)
		  (if calc-prefer-frac
		      (progn
			(calc-extensions)
			(math-make-frac a b))
		    (math-div-float (math-make-float a 0)
				    (math-make-float b 0))))))
	 (and (Math-ratp a) (Math-ratp b)
	      (calc-extensions)
	      (calc-div-fractions a b))
	 (and (Math-realp a) (Math-realp b)
	      (progn
		(or (and (consp a) (eq (car a) 'float))
		    (setq a (math-float a)))
		(or (and (consp b) (eq (car b) 'float))
		    (setq b (math-float b)))
		(math-div-float a b)))
	 (and (calc-extensions)
	      (math-div-objects-fancy a b))))
   (and (calc-extensions)
	(math-div-symb-fancy a b)))
)

;;;; [calc-misc.el]

(defun calcFunc-div (a &rest rest)
  (while rest
    (setq a (list '/ a (car rest))
	  rest (cdr rest)))
  (math-normalize a)
)

;;;; [calc.el]

(defun math-div-float (a b)   ; [F F F]
  (let ((ldiff (max (- (1+ calc-internal-prec)
		       (- (math-numdigs (nth 1 a)) (math-numdigs (nth 1 b))))
		    0)))
    (math-make-float (math-quotient (math-scale-int (nth 1 a) ldiff) (nth 1 b))
		     (- (- (nth 2 a) (nth 2 b)) ldiff)))
)

;;;; [calc-misc.el]

(defun math-inv (m)
  (if (Math-vectorp m)
      (progn
	(calc-extensions)
	(if (math-square-matrixp m)
	    (or (math-with-extra-prec 2 (math-matrix-inv-raw m))
		(math-reject-arg m "Singular matrix"))
	  (math-reject-arg m 'square-matrixp)))
    (math-div 1 m))
)
(fset 'calcFunc-inv (symbol-function 'math-inv))


;;;; [calc-macs.el]

(defmacro math-working (msg arg)    ; [Public]
  (` (if (eq calc-display-working-message 'lots)
	 (progn
	   (calc-set-command-flag 'clear-message)
	   (message "Working... %s = %s"
		    (, msg)
		    (math-showing-full-precision
		     (math-format-number (, arg)))))))
)


;;;; [calc-misc.el]

;;; Compute A modulo B, defined in terms of truncation toward minus infinity.
(defun math-mod (a b)   ; [R R R] [Public]
  (cond ((and (Math-zerop a) (not (eq (car-safe a) 'mod))) a)
	((Math-zerop b)
	 (math-reject-arg a "Division by zero"))
	((and (Math-natnump a) (Math-natnump b))
	 (math-imod a b))
	((and (Math-anglep a) (Math-anglep b))
	 (math-sub a (math-mul (math-floor (math-div a b)) b)))
	(t (calc-extensions)
	   (math-mod-fancy a b)))
)
(defun calcFunc-mod (a b)
  (math-normalize (list '% a b))
)


;;; Compute the greatest common divisor of A and B.   [I I I] [Public]
(defun math-gcd (a b)
  (cond
   ((not (or (consp a) (consp b)))
    (if (< a 0) (setq a (- a)))
    (if (< b 0) (setq b (- b)))
    (let (c)
      (if (< a b)
	  (setq c b b a a c))
      (while (> b 0)
	(setq c b
	      b (% a b)
	      a c))
      a))
   ((Math-looks-negp a) (math-gcd (math-neg a) b))
   ((Math-looks-negp b) (math-gcd a (math-neg b)))
   ((eq a 0) b)
   ((eq b 0) a)
   ((not (Math-integerp a))
    (if (Math-messy-integerp a)
	(math-gcd (math-trunc a) b)
      (calc-record-why 'integerp a)
      (list 'calcFunc-gcd a b)))
   ((not (Math-integerp b))
    (if (Math-messy-integerp b)
	(math-gcd a (math-trunc b))
      (calc-record-why 'integerp b)
      (list 'calcFunc-gcd a b)))
   (t
    (let (c)
      (if (Math-natnum-lessp a b)
	  (setq c b b a a c))
      (while (and (consp a) (not (eq b 0)))
	(setq c b
	      b (math-imod a b)
	      a c))
      (while (> b 0)
	(setq c b
	      b (% a b)
	      a c))
      a)))
)
(fset 'calcFunc-gcd (symbol-function 'math-gcd))



;;; General exponentiation.

(defun math-pow (a b)   ; [O O N] [Public]
  (cond ((Math-zerop a)
	 (if (Math-zerop b)
	     (math-reject-arg (list '^ a b) "Indeterminate form")
	   (if (math-floatp b) (math-float a) a)))
	((or (eq a 1) (eq b 1)) a)
	((or (equal a '(float 1 0)) (equal b '(float 1 0))) a)
	((Math-zerop b)
	 (if (eq (car-safe a) 'mod)
	     (math-make-mod 1 (nth 2 a))
	   (if (math-matrixp a)
	       (math-diag-matrix 1 (1- (length a)))
	     (if (or (math-floatp a) (math-floatp b))
		 '(float 1 0) 1))))
	((and (Math-integerp b) (or (Math-numberp a) (Math-vectorp a)))
	 (math-with-extra-prec 2
	   (math-ipow a b)))
	(t
	 (calc-extensions)
	 (math-pow-fancy a b)))
)
(defun calcFunc-pow (a b)
  (math-normalize (list '^ a b))
)

(defun math-ipow (a n)   ; [O O I] [Public]
  (cond ((Math-integer-negp n)
	 (math-ipow (math-div 1 a) (Math-integer-neg n)))
	((not (consp n))
	 (if (and (Math-ratp a) (> n 20))
	     (math-iipow-show a n)
	   (math-iipow a n)))
	((math-evenp n)
	 (math-ipow (math-sqr a) (math-div2 n)))
	(t
	 (math-mul a (math-ipow (math-sqr a)
				(math-div2 (math-add n -1))))))
)

(defun math-iipow (a n)   ; [O O S]
  (cond ((= n 0) 1)
	((= n 1) a)
	((= (% n 2) 0) (math-iipow (math-sqr a) (/ n 2)))
	(t (math-mul a (math-iipow (math-sqr a) (/ n 2)))))
)

(defun math-iipow-show (a n)   ; [O O S]
  (math-working "pow" a)
  (let ((val (cond
	      ((= n 0) 1)
	      ((= n 1) a)
	      ((= (% n 2) 0) (math-iipow-show (math-sqr a) (/ n 2)))
	      (t (math-mul a (math-iipow-show (math-sqr a) (/ n 2)))))))
    (math-working "pow" val)
    val)
)

;;;; [calc.el]




;;; Format the number A as a string.  [X N; X Z] [Public]
;;; Target line-width is W.
(defun math-format-stack-value (entry &optional w)
  (or w (setq w (calc-window-width)))
  (setq calc-selection-cache-entry calc-selection-cache-default-entry)
  (let* ((a (car entry))
	 (math-comp-selected (nth 2 entry))
	 (c (cond ((null a) "<nil>")
		  ((eq calc-display-raw t) (format "%s" a))
		  ((stringp a) a)
		  ((eq a 'top-of-stack) ".")
		  (calc-prepared-composition
		   calc-prepared-composition)
		  ((and (Math-scalarp a)
			(memq calc-language '(nil flat unform))
			(null math-comp-selected))
		   (math-format-number a))
		  (t (calc-extensions)
		     (math-compose-expr a 0))))
	 s ww)
    (and math-comp-selected (setq calc-any-selections t))
    (if (and calc-display-just
	     (< (setq ww (if (stringp c)
			     (length c)
			   (math-comp-width c))) w))
	(setq c (math-comp-concat
		 (make-string (if (eq calc-display-just 'center)
				  (/ (- w ww) 2)
				(- w ww)) 32)
		 c))
      (if calc-line-numbering
	  (setq c (math-comp-concat
		   (if (eq calc-language 'big)
		       (if math-comp-selected '(tag t "1:  ") "1:  ")
		     "    ") c))))
    (setq s (if (stringp c)
		(if calc-display-raw
		    (prin1-to-string c)
		  c)
	      (math-composition-to-string c w)))
    (if calc-language-output-filter
	(setq s (funcall calc-language-output-filter s)))
    (if (eq calc-language 'big)
	(setq s (concat s "\n"))
      (if calc-line-numbering
	  (progn
	    (aset s 0 ?1)
	    (aset s 1 ?:))))
    (setcar (cdr entry) (calc-count-lines s))
    s)
)

(defun calc-count-lines (s)
  (let ((pos 0)
	(num 1))
    (while (setq newpos (string-match "\n" s pos))
      (setq pos (1+ newpos)
	    num (1+ num)))
    num)
)

(defun math-format-value (a &optional w)
  (if (and (Math-scalarp a)
	   (memq calc-language '(nil flat unform)))
      (math-format-number a)
    (calc-extensions)
    (math-composition-to-string (math-compose-expr a 0) w))
)

(defun calc-window-width ()
  (1- (window-width (get-buffer-window (current-buffer))))
)

(defun math-comp-concat (c1 c2)
  (if (and (stringp c1) (stringp c2))
      (concat c1 c2)
    (list 'horiz c1 c2))
)



;;; Format an expression as a one-line string suitable for re-reading.

(defun math-format-flat-expr (a prec)
  (cond
   ((or (not (or (consp a) (integerp a)))
	(eq calc-display-raw t))
    (let ((print-escape-newlines t))
      (concat "'" (prin1-to-string a))))
   ((Math-scalarp a)
    (let ((calc-group-digits nil)
	  (calc-point-char ".")
	  (calc-frac-format (if (> (length calc-frac-format) 1) "::" ":"))
	  (calc-complex-format nil)
	  (calc-hms-format "%s@ %s' %s\"")
	  (calc-language nil))
      (math-format-number a)))
   (t
    (calc-extensions)
    (math-format-flat-expr-fancy a prec)))
)



;;; Format a number as a string.
(defun math-format-number (a)   ; [X N]   [Public]
  (cond
   ((eq calc-display-raw t) (format "%s" a))
   ((integerp a)
    (if (not (or calc-group-digits calc-leading-zeros))
	(if (= calc-number-radix 10)
	    (int-to-string a)
	  (if (< a 0)
	      (concat "-" (math-format-number (- a)))
	    (calc-extensions)
	    (if math-radix-explicit-format
		(if calc-radix-formatter
		    (funcall calc-radix-formatter
			     calc-number-radix
			     (if (= calc-number-radix 2)
				 (math-format-binary a)
			       (math-format-radix a)))
		  (format "%d#%s" calc-number-radix
			  (if (= calc-number-radix 2)
			      (math-format-binary a)
			    (math-format-radix a))))
	      (math-format-radix a))))
      (math-format-number (math-bignum a))))
   ((stringp a) a)
   ((eq (car a) 'bigpos) (math-format-bignum (cdr a)))
   ((eq (car a) 'bigneg) (concat "-" (math-format-bignum (cdr a))))
   ((eq (car a) 'float)
    (if (Math-integer-negp (nth 1 a))
	(concat "-" (math-format-number (math-neg a)))
      (let ((mant (nth 1 a))
	    (exp (nth 2 a))
	    (fmt (car calc-float-format))
	    (figs (nth 1 calc-float-format))
	    (point calc-point-char)
	    str)
	(if (and (eq fmt 'fix)
		 (or (and (< figs 0) (setq figs (- figs)))
		     (> (+ exp (math-numdigs mant)) (- figs))))
	    (progn
	      (setq mant (math-scale-rounding mant (+ exp figs))
		    str (if (integerp mant)
			    (int-to-string mant)
			  (math-format-bignum-decimal (cdr mant))))
	      (if (<= (length str) figs)
		  (setq str (concat (make-string (1+ (- figs (length str))) ?0)
				    str)))
	      (if (> figs 0)
		  (setq str (concat (substring str 0 (- figs)) point
				    (substring str (- figs))))
		(setq str (concat str point)))
	      (if calc-group-digits
		  (setq str (math-group-float str))))
	  (if (< figs 0)
	      (setq figs (+ calc-internal-prec figs)))
	  (if (> figs 0)
	      (let ((adj (- figs (math-numdigs mant))))
		(if (< adj 0)
		    (setq mant (math-scale-rounding mant adj)
			  exp (- exp adj)))))
	  (setq str (if (integerp mant)
			(int-to-string mant)
		      (math-format-bignum-decimal (cdr mant))))
	  (let* ((len (length str))
		 (dpos (+ exp len)))
	    (if (and (eq fmt 'float)
		     (<= dpos (+ calc-internal-prec calc-display-sci-high))
		     (>= dpos (+ calc-display-sci-low 2)))
		(progn
		  (cond
		   ((= dpos 0)
		    (setq str (concat "0" point str)))
		   ((and (<= exp 0) (> dpos 0))
		    (setq str (concat (substring str 0 dpos) point
				      (substring str dpos))))
		   ((> exp 0)
		    (setq str (concat str (make-string exp ?0) point)))
		   (t   ; (< dpos 0)
		    (setq str (concat "0" point
				      (make-string (- dpos) ?0) str))))
		  (if calc-group-digits
		      (setq str (math-group-float str))))
	      (let* ((eadj (+ exp len))
		     (scale (if (eq fmt 'eng)
				(1+ (% (+ eadj 300002) 3))
			      1)))
		(if (> scale (length str))
		    (setq str (concat str (make-string (- scale (length str))
						       ?0))))
		(if (< scale (length str))
		    (setq str (concat (substring str 0 scale) point
				      (substring str scale))))
		(if calc-group-digits
		    (setq str (math-group-float str)))
		(setq str (concat str
				  (if (eq calc-language 'math)
				      "*10.^" "e")
				  (int-to-string (- eadj scale))))))))
	str)))
   (t
    (calc-extensions)
    (math-format-number-fancy a)))
)

(defun math-format-bignum (a)   ; [X L]
  (if (and (= calc-number-radix 10)
	   (not calc-leading-zeros)
	   (not calc-group-digits))
      (math-format-bignum-decimal a)
    (calc-extensions)
    (math-format-bignum-fancy a))
)

(defun math-format-bignum-decimal (a)   ; [X L]
  (if a
      (let ((s ""))
	(while (cdr (cdr a))
	  (setq s (concat (format "%06d" (+ (* (nth 1 a) 1000) (car a))) s)
		a (cdr (cdr a))))
	(concat (int-to-string (+ (* (or (nth 1 a) 0) 1000) (car a))) s))
    "0")
)



;;; Parse a simple number in string form.   [N X] [Public]
(defun math-read-number (s)
  (math-normalize
   (cond

    ;; Integers (most common case)
    ((string-match "\\` *\\([0-9]+\\) *\\'" s)
     (let ((digs (math-match-substring s 1)))
       (if (and (eq calc-language 'c)
		(> (length digs) 1)
		(eq (aref digs 0) ?0))
	   (math-read-number (concat "8#" digs))
	 (if (<= (length digs) 6)
	     (string-to-int digs)
	   (cons 'bigpos (math-read-bignum digs))))))

    ;; Clean up the string if necessary
    ((string-match "\\`\\(.*\\)[ \t\n]+\\([^\001]\\)*\\'" s)
     (math-read-number (concat (math-match-substring s 1)
			       (math-match-substring s 2))))

    ;; Plus and minus signs
    ((string-match "^[-_+]\\(.*\\)$" s)
     (let ((val (math-read-number (math-match-substring s 1))))
       (and val (if (eq (aref s 0) ?+) val (math-neg val)))))

    ;; Forms that require extensions module
    ((string-match "[^-+0-9eE.]" s)
     (calc-extensions)
     (math-read-number-fancy s))

    ;; Decimal point
    ((string-match "^\\([0-9]*\\)\\.\\([0-9]*\\)$" s)
     (let ((int (math-match-substring s 1))
	   (frac (math-match-substring s 2)))
       (let ((ilen (length int))
	     (flen (length frac)))
	 (let ((int (if (> ilen 0) (math-read-number int) 0))
	       (frac (if (> flen 0) (math-read-number frac) 0)))
	   (and int frac (or (> ilen 0) (> flen 0))
		(list 'float
		      (math-add (math-scale-int int flen) frac)
		      (- flen)))))))

    ;; "e" notation
    ((string-match "^\\(.*\\)[eE]\\([-+]?[0-9]+\\)$" s)
     (let ((mant (math-match-substring s 1))
	   (exp (math-match-substring s 2)))
       (let ((mant (if (> (length mant) 0) (math-read-number mant) 1))
	     (exp (string-to-int exp)))
	 (and mant (Math-realp mant)
	      (let ((mant (math-float mant)))
		(list 'float (nth 1 mant) (+ (nth 2 mant) exp)))))))

    ;; Syntax error!
    (t nil)))
)

(defun math-match-substring (s n)
  (if (match-beginning n)
      (substring s (match-beginning n) (match-end n))
    "")
)

(defun math-read-bignum (s)   ; [l X]
  (if (> (length s) 3)
      (cons (string-to-int (substring s -3))
	    (math-read-bignum (substring s 0 -3)))
    (list (string-to-int s)))
)

;;;; [calc-misc.el]

(defun math-read-radix-digit (dig)   ; [D S; Z S]
  (if (> dig ?9)
      (if (< dig ?A)
	  nil
	(- dig 55))
    (if (>= dig ?0)
	(- dig ?0)
      nil))
)




;;;; [calc-aent.el]

;;; Algebraic expression parsing.   [Public]

(defun math-read-exprs (exp-str)
  (let ((exp-pos 0)
	(exp-old-pos 0)
	(exp-keep-spaces nil)
	exp-token exp-data)
    (if calc-language-input-filter
	(setq exp-str (funcall calc-language-input-filter exp-str)))
    (while (setq exp-token (string-match "\\.\\." exp-str))
      (setq exp-str (concat (substring exp-str 0 exp-token) "\\dots"
			    (substring exp-str (+ exp-token 2)))))
    (math-read-token)
    (let ((val (catch 'syntax (math-read-expr-list))))
      (if (stringp val)
	  (list 'error exp-old-pos val)
	(if (equal exp-token 'end)
	    val
	  (list 'error exp-old-pos "Syntax error")))))
)

(defun math-read-expr-list ()
  (let* ((exp-keep-spaces nil)
	 (val (list (math-read-expr-level 0)))
	 (last val))
    (while (equal exp-data ",")
      (math-read-token)
      (let ((rest (list (math-read-expr-level 0))))
	(setcdr last rest)
	(setq last rest)))
    val)
)

(defun math-read-token ()
  (if (>= exp-pos (length exp-str))
      (setq exp-old-pos exp-pos
	    exp-token 'end
	    exp-data "\000")
    (let ((ch (elt exp-str exp-pos)))
      (setq exp-old-pos exp-pos)
      (cond ((memq ch '(32 10 9))
	     (setq exp-pos (1+ exp-pos))
	     (if exp-keep-spaces
		 (setq exp-token 'space
		       exp-data " ")
	       (math-read-token)))
	    ((or (and (>= ch ?a) (<= ch ?z))
		 (and (>= ch ?A) (<= ch ?Z)))
	     (string-match (if (eq calc-language 'tex)
			       "[a-zA-Z0-9']*"
			     "[a-zA-Z0-9'_]*")
			   exp-str exp-pos)
	     (setq exp-token 'symbol
		   exp-pos (match-end 0)
		   exp-data (math-restore-dashes
			     (math-match-substring exp-str 0))))
	    ((or (and (>= ch ?0) (<= ch ?9))
		 (eq ch '?\.)
		 (and (eq ch '?_)
		      (or (not (eq calc-language 'tex))
			  (eq exp-pos 0)
			  (memq (elt exp-str (1- exp-pos)) '(32 ?\()))))
	     (or (and (eq calc-language 'c)
		      (string-match "0[xX][0-9a-fA-F]+" exp-str exp-pos))
		 (string-match "_?\\([0-9]+.?0*@ *\\)?\\([0-9]+.?0*' *\\)?\\([0-9]+\\(#\\|\\^\\^\\)[0-9a-zA-Z:]+\\|[0-9]+:[0-9:]+\\|[0-9.]+\\([eE][-+_]?[0-9]+\\)?\"?\\)?" exp-str exp-pos))
	     (setq exp-token 'number
		   exp-data (math-match-substring exp-str 0)
		   exp-pos (match-end 0)))
	    ((eq ch ?\$)
	     (string-match "\\$+" exp-str exp-pos)
	     (setq exp-token 'dollar
		   exp-data (- (match-end 0) (match-beginning 0))
		   exp-pos (match-end 0)))
	    ((eq (string-match "~=\\|<=\\|>=\\|<>\\|/=\\|\\+/-\\|\\\\dots\\|\\\\ldots\\|\\*\\*\\|<<\\|>>\\|==\\|!=\\|&&\\|||\\|!!"
			       exp-str exp-pos)
		 exp-pos)
	     (setq exp-token 'punc
		   exp-data (math-match-substring exp-str 0)
		   exp-pos (match-end 0)))
	    ((and (eq ch ?\")
		  (string-match "\\(\"\\([^\"\\]\\|\\\\.\\)*\\)\\(\"\\|\\'\\)" exp-str exp-pos))
	     (setq exp-token 'string
		   exp-data (math-match-substring exp-str 1)
		   exp-pos (match-end 0)))
	    ((and (= ch ?\\) (eq calc-language 'tex))
	     (or (string-match "\\\\hbox *{\\([a-zA-Z0-9]+\\)}" exp-str exp-pos)
		 (string-match "\\(\\\\\\([a-zA-Z]+\\|[^a-zA-Z]\\)\\)" exp-str exp-pos))
	     (setq exp-token 'symbol
		   exp-pos (match-end 0)
		   exp-data (math-restore-dashes
			     (math-match-substring exp-str 1)))
	     (let ((code (assoc exp-data math-tex-ignore-words)))
	       (cond ((null code))
		     ((null (cdr code))
		      (math-read-token))
		     ((eq (nth 1 code) 'punc)
		      (setq exp-token 'punc
			    exp-data (nth 2 code)))
		     ((and (eq (nth 1 code) 'mat)
			   (string-match " *{" exp-str exp-pos))
		      (setq exp-pos (match-end 0)
			    exp-token 'punc
			    exp-data "[")
		      (let ((right (string-match "}" exp-str exp-pos)))
			(and right
			     (setq exp-str (copy-sequence exp-str))
			     (aset exp-str right ?\])))))))
	    (t
	     (if (and (eq ch ?\{) (eq calc-language 'tex))
		 (setq ch ?\())
	     (if (and (eq ch ?\}) (eq calc-language 'tex))
		 (setq ch ?\)))
	     (if (and (eq ch ?\&) (eq calc-language 'tex))
		 (setq ch ?\,))
	     (setq exp-token 'punc
		   exp-data (char-to-string ch)
		   exp-pos (1+ exp-pos))))))
)

;;;; [calc.el]

(defconst math-tex-ignore-words
  '( ("\\hbox") ("\\mbox") ("\\text") ("\\left") ("\\right")
     ("\\,") ("\\>") ("\\:") ("\\;") ("\\!") ("\\ ")
     ("\\quad") ("\\qquad") ("\\hfil") ("\\hfill")
     ("\\displaystyle") ("\\textstyle") ("\\dsize") ("\\tsize")
     ("\\scriptstyle") ("\\scriptscriptstyle") ("\\ssize") ("\\sssize")
     ("\\rm") ("\\bf") ("\\it") ("\\sl")
     ("\\roman") ("\\bold") ("\\italic") ("\\slanted")
     ("\\cal") ("\\mit") ("\\Cal") ("\\Bbb") ("\\frak") ("\\goth")
     ("\\matrix" mat) ("\\bmatrix" mat) ("\\pmatrix" mat)
     ("\\cr" punc ";") ("\\\\" punc ";") ("\\*" punc "*")
     ("\\{" punc "[") ("\\}" punc "]")
))

(defconst math-standard-opers
  '( ( "u+"    ident	     -1 1000 )
     ( "u-"    neg	     -1 1000 )
     ( "u!"    calcFunc-lnot -1 1000 )
     ( "mod"   mod	     400 400 )
     ( "+/-"   sdev	     300 300 )
     ( "!"     calcFunc-fact 210  -1 )
     ( "^"     ^             201 200 )
     ( "*"     *             196 195 )
     ( "2x"    *             196 195 )
     ( "/"     /             190 191 )
     ( "%"     %             190 191 )
     ( "\\"    calcFunc-idiv 190 191 )
     ( "+"     +	     180 181 )
     ( "-"     -	     180 181 )
     ( "|"     |	     170 171 )
     ( "<"     calcFunc-lt   160 161 )
     ( ">"     calcFunc-gt   160 161 )
     ( "<="    calcFunc-leq  160 161 )
     ( ">="    calcFunc-geq  160 161 )
     ( "="     calcFunc-eq   160 161 )
     ( "=="    calcFunc-eq   160 161 )
     ( "!="    calcFunc-neq  160 161 )
     ( "&&"    calcFunc-land 110 111 )
     ( "||"    calcFunc-lor  100 101 )
     ( "?"     calcFunc-if    91  90 )
))
(setq math-expr-opers math-standard-opers)

;;;; [calc-aent.el]

(defun math-read-expr-level (exp-prec)
  (let* ((x (math-read-factor)) op)
    (while (and (or (and (setq op (assoc exp-data math-expr-opers))
			 (/= (nth 2 op) -1))
		    (and (or (eq (nth 2 op) -1)
			     (memq exp-token '(symbol number dollar))
			     (equal exp-data "(")
			     (and (equal exp-data "[")
				  (not (eq calc-language 'math))
				  (not (and exp-keep-spaces
					    (eq (car-safe x) 'vec)))))
			 (setq op (assoc "2x" math-expr-opers))))
		(>= (nth 2 op) exp-prec))
      (if (not (equal (car op) "2x"))
	  (math-read-token))
      (and (memq (nth 1 op) '(sdev mod))
	   (calc-extensions))
      (setq x (cond ((eq (nth 3 op) -1)
		     (if (eq (nth 1 op) 'ident)
			 x
		       (list (nth 1 op) x)))
		    ((equal (car op) "?")
		     (let ((y (math-read-expr-level 0)))
		       (or (equal exp-data ":")
			   (throw 'syntax "Expected ':'"))
		       (math-read-token)
		       (list (nth 1 op)
			     x
			     y
			     (math-read-expr-level (nth 3 op)))))
		    (t (list (nth 1 op)
			     x
			     (math-read-expr-level (nth 3 op)))))))
    x)
)

(defun math-remove-dashes (x)
  (if (string-match "\\`\\(.*\\)-\\(.*\\)\\'" x)
      (math-remove-dashes
       (concat (math-match-substring x 1) "_" (math-match-substring x 2)))
    x)
)

(defun math-restore-dashes (x)
  (if (string-match "\\`\\(.*\\)_\\(.*\\)\\'" x)
      (math-restore-dashes
       (concat (math-match-substring x 1) "-" (math-match-substring x 2)))
    x)
)

(defun math-read-factor ()
  (let (op)
    (cond ((eq exp-token 'number)
	   (let ((num (math-read-number exp-data)))
	     (if (not num)
		 (progn
		   (setq exp-old-pos exp-pos)
		   (throw 'syntax "Bad format")))
	     (math-read-token)
	     (if (and math-read-expr-quotes
		      (consp num))
		 (list 'quote num)
	       num)))
	  ((or (equal exp-data "-")
	       (equal exp-data "+")
	       (equal exp-data "!")
	       (equal exp-data "|"))
	   (setq exp-data (concat "u" exp-data))
	   (math-read-factor))
	  ((and (setq op (assoc exp-data math-expr-opers))
		(eq (nth 2 op) -1))
	   (math-read-token)
	   (let ((val (math-read-expr-level (nth 3 op))))
	     (cond ((eq (nth 1 op) 'ident)
		    val)
		   ((and (Math-numberp val)
			 (equal (car op) "u-"))
		    (math-neg val))
		   (t (list (nth 1 op) val)))))
	  ((eq exp-token 'symbol)
	   (let ((sym (intern exp-data)))
	     (math-read-token)
	     (if (equal exp-data calc-function-open)
		 (progn
		   (math-read-token)
		   (let ((args (if (equal exp-data calc-function-close)
				   nil
				 (math-read-expr-list))))
		     (if (not (or (equal exp-data calc-function-close)
				  (eq exp-token 'end)))
			 (throw 'syntax "Expected `)'"))
		     (math-read-token)
		     (let ((f (assq sym math-expr-function-mapping)))
		       (if f
			   (setq sym (cdr f))
			 (or (string-match "-" (symbol-name sym))
			     (setq sym (intern (concat "calcFunc-"
						       (symbol-name sym)))))))
		     (cons sym args)))
	       (if math-read-expr-quotes
		   sym
		 (let ((val (list 'var
				  (intern (math-remove-dashes
					   (symbol-name sym)))
				  (if (string-match "-" (symbol-name sym))
				      sym
				    (intern (concat "var-"
						    (symbol-name sym)))))))
		   (let ((v (assq (nth 1 val) math-expr-variable-mapping)))
		     (and v (setq val (list 'var
					    (intern
					     (substring (symbol-name (cdr v)) 4))
					    (cdr v)))))
		   (while (and (memq calc-language '(c pascal))
			       (equal exp-data "["))
		     (math-read-token)
		     (setq val (append (list 'calcFunc-subscr val)
				       (math-read-expr-list)))
		     (if (equal exp-data "]")
			 (math-read-token)
		       (throw 'syntax "Expected ']'")))
		   val)))))
	  ((eq exp-token 'dollar)
	   (if (>= (length calc-dollar-values) exp-data)
	       (let ((num exp-data))
		 (math-read-token)
		 (setq calc-dollar-used (max calc-dollar-used num))
		 (math-check-complete (nth (1- num) calc-dollar-values)))
	     (throw 'syntax (if calc-dollar-values
				"Too many $'s"
			      "$'s not allowed in this context"))))
	  ((equal exp-data "(")
	   (let* ((exp (let ((exp-keep-spaces nil))
			 (math-read-token)
			 (math-read-expr-level 0))))
	     (let ((exp-keep-spaces nil))
	       (cond
		((equal exp-data ",")
		 (progn
		   (math-read-token)
		   (let ((exp2 (math-read-expr-level 0)))
		     (setq exp
			   (if (and exp2 (Math-realp exp) (Math-realp exp2))
			       (math-normalize (list 'cplx exp exp2))
			     (list '+ exp (list '* exp2 '(var i var-i))))))))
		((equal exp-data ";")
		 (progn
		   (math-read-token)
		   (let ((exp2 (math-read-expr-level 0)))
		     (setq exp (if (and exp2 (Math-realp exp)
					(Math-anglep exp2))
				   (math-normalize (list 'polar exp exp2))
				 (list '* exp
				       (list 'calcFunc-exp
					     (list '* exp2
						   '(var i var-i)))))))))
		((or (equal exp-data "\\dots")
		     (equal exp-data "\\ldots"))
		 (progn
		   (math-read-token)
		   (let ((exp2 (math-read-expr-level 0)))
		     (setq exp
			   (list 'intv
				 (if (equal exp-data ")") 0 1)
				 exp
				 exp2)))))))
	     (if (not (or (equal exp-data ")")
			  (and (equal exp-data "]") (eq (car-safe exp) 'intv))
			  (eq exp-token 'end)))
		 (throw 'syntax "Expected `)'"))
	     (math-read-token)
	     exp))
	  ((eq exp-token 'string)
	   (calc-extensions)
	   (math-read-string))
	  ((equal exp-data "[")
	   (calc-extensions)
	   (math-read-brackets t "]"))
	  ((equal exp-data "{")
	   (calc-extensions)
	   (math-read-brackets nil "}"))
	  (t (throw 'syntax "Expected a number"))))
)




;;;; [calc-misc.el]

;;; Bug reporting

(defun report-calc-bug (topic)
  "Report a bug in Calc, the GNU Emacs calculator.
Prompts for bug subject.  Leaves you in a mail buffer."
  (interactive "sBug Subject: ")
  (mail nil calc-bug-address topic)
  (goto-char (point-max))
  (insert "\nIn Calc 1.07, Emacs " (emacs-version) "\n\n")
  (message (substitute-command-keys "Type \\[mail-send] to send bug report."))
)

;;;; [calc.el]


;;; User-programmability.

(defmacro defmath (func args &rest body)   ;  [Public]
  (calc-extensions)
  (math-do-defmath func args body)
)



(if calc-always-load-extensions
    (calc-extensions)
)



;;;; [end]



(defun calc-install ()
  "Split \"calc.el\" and \"calc-ext.el\"; format \"calc.texinfo\".
Usage, from the Unix shell:
       cd <Calc home directory>
       emacs -batch -l calc -f calc-install"
  (interactive)
  (setq max-lisp-eval-depth (max 400 max-lisp-eval-depth))
  (find-file "calc-ext.el")
  (end-of-buffer)
  (backward-paragraph 2)
  (eval-last-sexp nil)
  (find-file "calc.el")
  (search-forward "calc-info-filename \"")
  (insert (file-name-directory buffer-file-name))
  (search-forward "calc-autoload-directory nil")
  (delete-backward-char 3)
  (insert "\"" default-directory "\"")
  (calc-split (file-name-directory (buffer-file-name)) nil t)
  (find-file "calc-ext.el")
  (calc-split (file-name-directory (buffer-file-name)) nil t)
  (byte-compile-file "macedit.el")
  (find-file "calc.texinfo")
  (texinfo-format-buffer)
  (save-buffer)
)



;;; End.

