;;; Mailing, forwarding, and replying commands for VM
;;; Copyright (C) 1989, 1990, 1991 Kyle E. Jones
;;;
;;; This program is free software; you can redistribute it and/or modify
;;; it under the terms of the GNU General Public License as published by
;;; the Free Software Foundation; either version 1, or (at your option)
;;; any later version.
;;;
;;; This program is distributed in the hope that it will be useful,
;;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;;; GNU General Public License for more details.
;;;
;;; You should have received a copy of the GNU General Public License
;;; along with this program; if not, write to the Free Software
;;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

(require 'vm)

(defun vm-do-reply (to-all include-text count)
  (save-restriction
    (widen)
    (let ((mail-buffer (current-buffer))
	  (mlist (vm-select-marked-or-prefixed-messages count))
	  (dir default-directory)
	  (message-pointer vm-message-pointer)
	  to cc subject mp in-reply-to tmp tmp2)
      (setq mp mlist)
      (while mp 
	(cond
	 ((eq mlist mp)
	  (cond ((setq to (vm-get-header-contents (car mp) "Reply-To")))
		((setq to (vm-get-header-contents (car mp) "From")))
		((setq to (vm-grok-From_-author (car mp))))
		(t (error "No From: or Reply-To: header in message")))
	  (setq subject (vm-get-header-contents (car mp) "Subject")
		in-reply-to (and vm-in-reply-to-format
				 (vm-sprintf 'vm-in-reply-to-format (car mp))))
	  (and subject vm-reply-subject-prefix
	       (let ((case-fold-search t))
		 (not
		  (equal
		   (string-match (regexp-quote vm-reply-subject-prefix)
				 subject)
		   0)))
	       (setq subject (concat vm-reply-subject-prefix subject))))
	 (t (cond ((setq tmp (vm-get-header-contents (car mp) "Reply-To"))
		   (setq to (concat to ",\n\t" tmp)))
		  ((setq tmp (vm-get-header-contents (car mp) "From"))
		   (setq to (concat to ",\n\t" tmp)))
		  ((setq tmp (vm-grok-From_-author (car mp)))
		   (setq to (concat to ",\n\t" tmp)))
		  (t (error "No From: or Reply-To: header in message")))))
	(if to-all
	    (progn
	      (setq tmp (vm-get-header-contents (car mp) "To"))
	      (setq tmp2 (vm-get-header-contents (car mp) "Cc"))
	      (if tmp
		  (if cc
		      (setq cc (concat cc ",\n\t" tmp))
		    (setq cc tmp)))
	      (if tmp2
		  (if cc
		      (setq cc (concat cc ",\n\t" tmp2))
		    (setq cc tmp2)))))
	(setq mp (cdr mp)))
      (if vm-strip-reply-headers
	  (let ((mail-use-rfc822 t))
	    (require 'mail-utils)
	    (and to (setq to (mail-strip-quoted-names to)))
	    (and cc (setq cc (mail-strip-quoted-names cc)))))
      (if vm-reply-ignored-addresses
	  (progn
	    (require 'rfc822)
	    (and to (setq to (mapconcat 'identity
					(vm-strip-ignored-addresses
					 (rfc822-addresses to))
					", ")))
	    (and cc (setq cc (mapconcat 'identity
					(vm-strip-ignored-addresses
					 (rfc822-addresses cc))
					", ")))))
      (if (vm-mail-internal
	   (format "reply to %s%s" (vm-su-full-name (car mlist))
		   (if (cdr mlist) ", ..." ""))
	   to subject in-reply-to cc)
	  (progn
	    (use-local-map (copy-keymap (current-local-map)))
	    (local-set-key "\C-c\C-y" 'vm-yank-message)
	    (local-set-key "\C-cy" 'vm-yank-message-other-folder)
	    (local-set-key "\C-c\C-s" 'vm-mail-send)
	    (local-set-key "\C-c\C-c" 'vm-mail-send-and-exit)
	    (local-set-key "\C-c\C-v" vm-mode-map)
	    (make-local-variable 'vm-reply-list)
	    (setq vm-mail-buffer mail-buffer
		  vm-system-state 'replying
		  vm-local-message-pointer message-pointer
		  vm-reply-list mlist
		  default-directory dir)
	    (if include-text
		(save-excursion
		  (while mlist
		    (vm-yank-message (car mlist))
		    (goto-char (point-max))
		    (setq mlist (cdr mlist))))))))))

(defun vm-strip-ignored-addresses (addresses)
  (setq addresses (copy-sequence addresses))
  (let (re-list list)
    (setq re-list vm-reply-ignored-addresses)
    (while re-list
      (setq addr-list addresses)
      (while addr-list
	(if (string-match (car re-list) (car addr-list))
	    (setq addresses (delq (car addr-list) addresses)))
	(setq addr-list (cdr addr-list)))
      (setq re-list (cdr re-list))))
  addresses )

(defun vm-mail-yank-default (message)
  (save-excursion
    (delete-region (point) (progn (search-forward "\n\n") (point)))
    (if vm-included-text-attribution-format
	(insert (vm-sprintf 'vm-included-text-attribution-format message)))
    (while (and (re-search-forward "^" nil t) (< (point) (mark)))
      (replace-match vm-included-text-prefix t t))))

(defun vm-yank-message-other-folder (folder &optional prefix-argument)
  "Like vm-yank-message except the message is yanked from a folder other
than the one that spawned the current *mail* buffer.  The name of the
folder is read from the minibuffer.

Don't call this function from a program."
  (interactive
   (list
    (let ((dir (if vm-folder-directory
		    (expand-file-name vm-folder-directory)
		  default-directory)))
      (read-file-name "Yank from folder: " dir nil t))
    current-prefix-arg ))
  (let ((b (current-buffer)) newbuf sumbuf)
    (set-buffer (or (get-file-buffer folder) (find-file-noselect folder)))
    (setq newbuf (current-buffer))
    (if (not (eq major-mode 'vm-mode))
	(vm-mode)
      (vm-set-folder-variables))
    (if (null vm-message-pointer)
	(error "No messages in folder %s" folder))
    (save-excursion
      (save-window-excursion
	(save-window-excursion
	  (vm-summarize))
	(switch-to-buffer vm-summary-buffer)
	(setq sumbuf (current-buffer))
	(delete-other-windows)
	(set-buffer b)
	(unwind-protect
	    (let ((prefix-arg prefix-argument)
		  (vm-mail-buffer newbuf))
	      (command-execute 'vm-yank-message))
	  (bury-buffer newbuf)
	  (bury-buffer sumbuf))))))

(defun vm-yank-message (message &optional prefix)
  "Yank message number N into the current buffer at point.
When called interactively N is always read from the minibuffer.  When
called non-interactively the first argument is expected to be a message
struct.

This command is meant to be used in VM created *mail* buffers; the
yanked message comes from the mail buffer containing the message you
are replying to, forwarding, or invoked VM's mail command from.

All message headers are yanked along with the text.  Point is left
before the inserted text, the mark after.  Any hook functions bound to
mail-yank-hooks are run, aftert inserting the text and setting point
and mark.

Prefix arg means to ignore mail-yank-hooks, don't set the mark, prepend the
value of vm-included-text-prefix to every yanked line, and don't yank any
headers other than those specified in vm-visible-headers/vm-invisible-headers."
  (interactive
   (list
   ;; What we really want for the first argument is a message struct,
   ;; but if called interactively, we let the user type in a message
   ;; number instead.
    (let (mp default (result 0) prompt)
      (save-excursion
	(vm-select-folder-buffer)
	(setq default (and vm-local-message-pointer
			   (vm-number-of (car vm-local-message-pointer)))
	      prompt (if default
			 (format "Yank message number: (default %s) "
				 default)
		       "Yank message number: "))
	(while (zerop result)
	  (setq result (read-string prompt))
	  (and (string= result "") default (setq result default))
	  (setq result (string-to-int result)))
	(if (null (setq mp (nthcdr (1- result) vm-message-list)))
	    (error "No such message.")))
      (car mp))
    current-prefix-arg ))
  (if (not (bufferp vm-mail-buffer))
      (error "This is not a VM *mail* buffer."))
  (if (null (buffer-name vm-mail-buffer))
      (error "The mail buffer containing message %d has been killed."
	     (vm-number-of message)))
  (let ((b (current-buffer)) (start (point)) mp end)
    (save-restriction
      (widen)
      (save-excursion
	(set-buffer (marker-buffer (vm-start-of message)))
	(save-restriction
	  (widen)
	  (append-to-buffer b (if prefix
				  (vm-vheaders-of message)
				(vm-start-of message))
			    (vm-text-end-of message))
	  (setq end (vm-marker (+ start (- (vm-text-end-of message)
					   (if prefix
					       (vm-vheaders-of message)
					     (vm-start-of message)))) b))))
      (if prefix
	  (save-excursion
	    (while (and (< (point) end) (re-search-forward "^" end t))
	      (replace-match vm-included-text-prefix t t)
	      (forward-line)))
	;; Delete UNIX From or MMDF ^A^A^A^A line
	(delete-region (point) (progn (forward-line) (point)))
	(push-mark end)
	(if mail-yank-hooks
	    (run-hooks 'mail-yank-hooks)
	  (vm-mail-yank-default message))))))

(defun vm-mail-send-and-exit (arg)
  "Just like mail-send-and-exit except that VM flags the appropriate message(s)
as having been replied to, if appropriate."
  (interactive "P")
  (let ((reply-buf (current-buffer)))
    (mail-send-and-exit arg)
    (save-excursion
      (set-buffer reply-buf)
      (cond ((eq vm-system-state 'replying)
	     (vm-mark-replied))
	    ((eq vm-system-state 'forwarding)
	     (vm-mark-forwarded)))
      ;; keep this buffer if the user demands it
      (if (memq (current-buffer) vm-kept-mail-buffers)
	  (setq vm-kept-mail-buffers
		(delq (current-buffer) vm-kept-mail-buffers)))
      (setq vm-kept-mail-buffers (cons (current-buffer) vm-kept-mail-buffers))
      (if (not (eq vm-keep-sent-messages t))
	  (let ((extras (nthcdr (or vm-keep-sent-messages 0) vm-kept-mail-buffers)))
	    (mapcar 'kill-buffer extras)
	    (and vm-kept-mail-buffers extras
		 (setcdr (memq (car extras) vm-kept-mail-buffers) nil)))))))

(defun vm-mail-send ()
  "Just like mail-send except that VM flags the appropriate message(s)
as having been replied to, if appropriate."
  (interactive)
  (mail-send)
  (cond ((eq vm-system-state 'replying)
	 (vm-mark-replied))
	((eq vm-system-state 'forwarding)
	 (vm-mark-forwarded))))

(defun vm-mark-replied ()
  (save-excursion
    (let ((mp vm-reply-list))
      (while mp
	(if (null (marker-buffer (vm-start-of (car mp))))
	    ()
	  (set-buffer (marker-buffer (vm-start-of (car mp))))
	  (cond ((and (memq (car mp) vm-local-message-list)
		      (null (vm-replied-flag (car mp))))
		 (vm-set-replied-flag (car mp) t))))
	(setq mp (cdr mp)))
      (vm-update-summary-and-mode-line))))

(defun vm-mark-forwarded ()
  (save-excursion
    (let ((mp vm-forward-list))
      (while mp
	(if (null (marker-buffer (vm-start-of (car mp))))
	    ()
	  (set-buffer (marker-buffer (vm-start-of (car mp))))
	  (cond ((and (memq (car mp) vm-local-message-list)
		      (null (vm-forwarded-flag (car mp))))
		 (vm-set-forwarded-flag (car mp) t))))
	(setq mp (cdr mp)))
      (vm-update-summary-and-mode-line))))

(defun vm-reply (count)
  "Reply to the sender of the current message.
Numeric prefix argument N mans to reply to the current message plus the
next N-1 messages.  A negative N means reply to the current message and
the previous N-1 messages. 

If invoked on marked messages (via vm-next-command-uses-marks),
all marked messages will be replied to.

You will be placed into a standard Emacs *mail* buffer to compose and
send your message.  See the documentation for the function `mail' for
more info.

Note that the normal binding of C-c C-y in the *mail* buffer is
automatically changed to vm-yank-message during a reply.  This allows
you to yank any message from the current folder into a reply.

Normal VM commands may be accessed in the reply buffer by prefixing them
with C-c C-v."
  (interactive "p")
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (vm-do-reply nil nil count))

(defun vm-reply-include-text (count)
  "Reply to the sender (only) of the current message and include text
from the message.  See the documentation for function vm-reply for details."
  (interactive "p")
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (vm-do-reply nil t count))

(defun vm-followup (count)
  "Reply to all recipients of the current message.
See the documentation for the function vm-reply for details."
  (interactive "p")
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (vm-do-reply t nil count))

(defun vm-followup-include-text (count)
  "Reply to all recipients of the current message and include text from
the message.  See the documentation for the function vm-reply for details."
  (interactive "p")
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (vm-do-reply t t count))

(defun vm-forward-message ()
  "Forward the current message to one or more third parties.
You will be placed in a *mail* buffer as is usual with replies, but you
must fill in the To: and Subject: headers manually."
  (interactive)
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (if (eq last-command 'vm-next-command-uses-marks)
      (error
       (substitute-command-keys
	"Multiple messages should forwarded as a digest.  Use \\[vm-send-digest]")))
  (let ((b (current-buffer))
	(dir default-directory)
	(mp vm-message-pointer)
	start)
    (save-restriction
      (widen)
      (cond ((vm-mail-internal
	      (format "forward of %s:%s" (buffer-name)
		      (vm-number-of (car vm-message-pointer)))
	      nil
	      (and vm-forwarding-subject-format
		   (vm-sprintf 'vm-forwarding-subject-format
			       (car mp))))
	     (use-local-map (copy-keymap (current-local-map)))
	     (local-set-key "\C-c\C-y" 'vm-yank-message)
	     (local-set-key "\C-cy" 'vm-yank-message-other-folder)
	     (local-set-key "\C-c\C-s" 'vm-mail-send)
	     (local-set-key "\C-c\C-c" 'vm-mail-send-and-exit)
	     (local-set-key "\C-c\C-v" vm-mode-map)
	     (make-local-variable 'vm-forward-list)
	     (setq vm-mail-buffer b
		   vm-system-state 'forwarding
		   vm-forward-list (list (car mp))
		   vm-local-message-pointer mp
		   default-directory dir)
	     (goto-char (point-max))
	     (insert "------- Start of forwarded message -------\n")
	     (setq start (point))
	     (insert-buffer-substring b
				      (save-excursion
					(set-buffer b)
					(goto-char (vm-start-of (car mp)))
					(forward-line 1)
					(point))
				      (vm-text-end-of (car mp)))
	     (if vm-rfc934-forwarding
		 (vm-rfc934-char-stuff-region start (point)))
	     (insert "------- End of forwarded message -------\n")
	     (goto-char (point-min))
	     (end-of-line))))))

(defun vm-mail ()
  "Send a mail message from within VM, or from without."
  (interactive)
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (let ((mail-buffer (if (memq major-mode '(vm-mode vm-virtual-mode))
			 (current-buffer))))
    (cond ((vm-mail-internal)
	   (if (null mail-buffer)
	       ()
	     (use-local-map (copy-keymap (current-local-map)))
	     (local-set-key "\C-c\C-y" 'vm-yank-message)
	     (local-set-key "\C-c\C-v" vm-mode-map)
	     (setq vm-mail-buffer mail-buffer))
	   (local-set-key "\C-c\C-s" 'vm-mail-send)
	   (local-set-key "\C-c\C-c" 'vm-mail-send-and-exit)
	   (local-set-key "\C-cy" 'vm-yank-message-other-folder)))))

(defun vm-mail-internal (&optional buffer-name to subject in-reply-to cc)
  (switch-to-buffer (generate-new-buffer (or buffer-name "*VM-mail*")))
  (auto-save-mode auto-save-default)
  (mail-mode)
  (mail-setup to subject in-reply-to cc nil)
  (vm-scrub-auto-save-file-name)
  t )

(defun vm-scrub-auto-save-file-name ()
  ;; remove whitespace from auto-save filename.
  ;; while we're at it, go ahead and fix up any shell
  ;; meta-characers, too.
  (if buffer-auto-save-file-name
      (let ((str buffer-auto-save-file-name) (i 0))
	(while (setq i (string-match "[][!?*$^%&|<>()`'\" \n\r\t\f]" str i))
	  (aset str i ?_)))))

(defun vm-resend-bounced-message ()
  "Extract the original text from a bounced message and resend it.
You will be placed in a *mail* buffer with the extracted message and
you can change the recipient address before resending the message."
  (interactive)
  (vm-follow-summary-cursor)
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (let ((b (current-buffer)) start
	(dir default-directory)
	(lim (vm-text-end-of (car vm-message-pointer))))
      (save-restriction
	(widen)
	(save-excursion
	  (goto-char (vm-text-of (car vm-message-pointer)))
	  (let (case-fold-search)
	    ;; What a wonderful world it would be if mailers used the
	    ;; message encapsulation standard instead the following
	    ;; ad hockeries.
	    (or
	     ;; sendmail
	     (search-forward "----- Unsent message follows" lim t)
	     ;; smail 2.x
	     (search-forward "======= text of message follows" lim t)
	     ;; smail 3.x (?)
	     (search-forward "- Message text follows:" lim t)
	     ;; MMDF
	     (search-forward "Your message follows:\n" lim t)
 	     (search-forward "    Your message begins as follows:\n" lim t)
	     ;; zmailer (?)
	     (search-forward "---  Original Message  ---" lim t)
	     ;; Grapevine
	     (search-forward "The text of your message was\n---" lim t)
	     (error "This does not appear to be a bounced message."))
	    (forward-line 1)
	    (setq start (point))))
	(cond ((vm-mail-internal
		(format "retry of %s:%s" (buffer-name)
			(vm-number-of (car vm-message-pointer))))
	       (use-local-map (copy-keymap (current-local-map)))
	       (local-set-key "\C-c\C-y" 'vm-yank-message)
	       (local-set-key "\C-cy" 'vm-yank-message-other-folder)
	       (local-set-key "\C-c\C-s" 'vm-mail-send)
	       (local-set-key "\C-c\C-c" 'vm-mail-send-and-exit)
	       (local-set-key "\C-c\C-v" vm-mode-map)
	       (goto-char (point-min))
	       (insert-buffer-substring b start lim)
	       (delete-region (point) (point-max))
	       (goto-char (point-min))
	       ;; some mailers leave grot at the top of the message.
	       ;; trim it.
	       (while (not (looking-at vm-generic-header-regexp))
		 (delete-region (point) (progn (forward-line 1) (point))))
	       ;; delete all but pertinent headers
	       (while (looking-at vm-generic-header-regexp)
		 (let ((match-end-0 (match-end 0)))
		   (if (or
			(looking-at "From:\\|To:\\|Cc:\\|Subject:\\|In-Reply-To\\|Resent-")
			(looking-at "Newsgroups\\|References"))
		       (goto-char match-end-0)
		     (delete-region (point) match-end-0))))
	       (insert mail-header-separator)
	       (if (= (following-char) ?\n)
		   (forward-char 1)
		 (insert "\n"))
	       (setq vm-mail-buffer b
		     default-directory dir))))))

(defun vm-send-digest (&optional prefix)
  "Send a digest of all messages in the current folder to recipients.
You will be placed in a *mail* buffer as is usual with replies, but you
must fill in the To: and Subject: headers manually.

Prefix arg means to insert a list of preamble lines at the beginning of
the digest.  One line is generated for each message being digestified.
The variable vm-digest-preamble-format determines the format of the
preamble lines.

If invoked on marked messages (via vm-next-command-uses-marks),
only marked messages will be put into the digest."
  (interactive "P")
  (vm-select-folder-buffer)
  (vm-check-for-killed-summary)
  (vm-error-if-folder-empty)
  (let ((b (current-buffer))
	(dir default-directory)
	(mp vm-message-pointer)
	;; prefix arg doesn't have "normal" meaning here, so only call
	;; vm-select-marked-or-prefixed-messages if we're using marks.
	(mlist (if (eq last-command 'vm-next-command-uses-marks)
		   (vm-select-marked-or-prefixed-messages 0)
		 vm-message-list))
	start)
    (save-restriction
      (widen)
      (cond
       ((vm-mail-internal (format "digest from %s" (current-buffer)))
	(use-local-map (copy-keymap (current-local-map)))
	(local-set-key "\C-c\C-y" 'vm-yank-message)
	(local-set-key "\C-cy" 'vm-yank-message-other-folder)
	(local-set-key "\C-c\C-s" 'vm-mail-send)
	(local-set-key "\C-c\C-c" 'vm-mail-send-and-exit)
	(local-set-key "\C-c\C-v" vm-mode-map)
	(make-local-variable 'vm-forward-list)
	(setq vm-mail-buffer b
	      vm-local-message-pointer mp
	      vm-system-state 'forwarding
	      vm-forward-list mlist
	      default-directory dir)
	(goto-char (point-max))
	(setq start (point)
	      mp mlist)
	(message "Building digest...")
	(while mp
	  (insert-buffer-substring (marker-buffer (vm-start-of (car mp)))
				   (vm-start-of (car mp))
				   (vm-end-of (car mp)))
	  (setq mp (cdr mp)))
	(vm-digestify-region start (point))
	(goto-char start)
	(setq mp mlist)
	(if prefix
	    (progn
	      (message "Building digest preamble...")
	      (while mp
		(insert (vm-sprintf 'vm-digest-preamble-format (car mp)) "\n")
		(if vm-digest-center-preamble
		    (progn
		      (forward-char -1)
		      (center-line)
		      (forward-char 1)))
		(setq mp (cdr mp)))))
	(goto-char (point-min))
	(end-of-line)
	(message "Building digest... done"))))))
