;;!emacs
;;
;; FILE:         hmouse-drv.el
;; SUMMARY:      Smart Key/Mouse driver functions.
;; USAGE:        GNU Emacs Lisp Library
;;
;; AUTHOR:       Bob Weiner
;; ORIG-DATE:    04-Feb-90
;;
;; This file is part of Hyperbole.
;;
;; Copyright (C) 1989, 1990, 1991, Brown University, Providence, RI
;; Available for use and distribution under the same terms as GNU Emacs.
;;
;; DESCRIPTION:  
;; DESCRIP-END.

;;; ************************************************************************
;;; Public variables
;;; ************************************************************************

(defvar smart-key-point-prev nil
  "Value of point prior to last smart-key press.
Used to determine a region when the smart-key is clicked within the region.")

;;; ************************************************************************
;;; smart-key driver functions
;;; ************************************************************************

(defun smart-key-mouse (&optional arg-list)
  "Set point to the current mouse cursor position and execute 'smart-key'.
Optional ARG-LIST will be passed to 'smart-key'."
  (interactive)
  (if *smart-key-meta-depressed*
      (progn (message *smart-key-help-msg*)
	     (setq *smart-key-meta-depressed* nil))
    (setq *smart-key-depressed* nil)
    (smart-key-mouse-func 'smart-key arg-list)))

(defun smart-key-mouse-meta (&optional arg-list)
  "Set point to the current mouse cursor position and execute 'smart-key-meta'.
Optional ARG-LIST will be passed to 'smart-key-meta'."
  (interactive)
  (if *smart-key-depressed*
      (progn (message *smart-key-help-msg*)
	     (setq *smart-key-depressed* nil))
    (setq *smart-key-meta-depressed* nil)
    (smart-key-mouse-func 'smart-key-meta arg-list)))

(defun smart-key-mouse-func (func set-point-arg-list)
  "Execute FUNC.
SET-POINT-ARG-LIST is passed to the call of the command bound to
'mouse-set-point-command'.  Return nil if 'mouse-set-point-command' variables
is not bound to a valid function."
  (if (null (smart-key-set-point set-point-arg-list))
      nil
    (and (eq major-mode 'br-mode)
	 (setq *smart-key-mouse-prev-window* 
	       (if (br-in-view-window-p)
		   (save-window-excursion
		     (br-next-class-window)
		     (selected-window))
		 (selected-window))))
    (setq *smart-key-mouse-prefix-arg* current-prefix-arg)
    (funcall func)
    (setq *smart-key-mouse-prev-window* nil
	  *smart-key-mouse-prefix-arg* nil)
    t))

(defun smart-key ()
  "Use one key to perform functions that vary by buffer.
Default function is given by 'smart-key-other-mode-cmd' variable.
Use \\[smart-menu] to pop up a menu whose items are selected with the smart
key.  Each item typically displays a subsystem buffer in which the smart key is
useful.

Be sure to clear any binding of the 'up' transition of any mouse key to which
you use bind this command.

Return t unless 'smart-key-other-mode-cmd' variable is not bound to a valid
function."
  (interactive)
  (or (smart-key-execute nil)
      (if (fboundp smart-key-other-mode-cmd)
	 (progn (funcall smart-key-other-mode-cmd)
		t))))

(defun smart-key-meta ()
  "Use one meta-key to perform functions that vary by buffer.
Default function is given by 'smart-key-meta-other-mode-cmd' variable.
Use \\[smart-menu] to pop up a menu whose items are selected with the smart
key.  Each item typically displays a subsystem buffer in which the smart key is
useful.  This command exits from the menu.

Be sure to clear any binding of the 'up' transition of any mouse key to which
you use bind this command.

Return non-nil unless 'smart-key-meta-other-mode-cmd' variable is not bound to
a valid function."
  (interactive)
  (or (smart-key-execute t)
      (if (fboundp smart-key-meta-other-mode-cmd)
	  (progn (funcall smart-key-meta-other-mode-cmd)
		 t))))

(defun smart-key-execute (meta)
  "Evaluate form associated with non-nil predicate from 'smart-key-alist'.
Non-nil META means evaluate second form, otherwise evaluate first form.
Returns non-nil iff a non-nil predicate is found."
    (let ((pred-forms smart-key-alist)
	  (pred-form) (pred-t))
      (while (and (null pred-t) (setq pred-form (car pred-forms)))
	(if (setq pred-t (eval (car pred-form)))
	    (eval (if meta (cdr (cdr pred-form)) (car (cdr pred-form))))
	  (setq pred-forms (cdr pred-forms))))
      pred-t))

(defun smart-key-help (meta)
  "Display documentation associated with smart key command in current context.
Non-nil META means use smart-key-meta command, otherwise evaluate
smart-key command.  Returns non-nil iff associated documentation is found."
  (interactive "P")
  (let ((pred-forms smart-key-alist)
	(pred-form) (pred-t) (call) (cmd-sym) (doc))
    (while (and (null pred-t) (setq pred-form (car pred-forms)))
      (or (setq pred-t (eval (car pred-form)))
	  (setq pred-forms (cdr pred-forms))))
    (if pred-t
	(setq call (if meta (cdr (cdr pred-form))
		     (car (cdr pred-form)))
	      cmd-sym (car call))
      (setq cmd-sym
	    (if meta smart-key-meta-other-mode-cmd smart-key-other-mode-cmd)
	    call cmd-sym))
    (setq *smart-key-help-msg*
	  (if (and cmd-sym (symbolp cmd-sym))
	      (progn
		(if (and (fboundp 'br-in-browser) (br-in-browser))
		    (br-to-view-window))
		(setq doc (documentation cmd-sym))
		(let ((condition (car pred-form))
		      (temp-buffer-show-hook
		       '(lambda (buf)
			  (set-buffer buf)
			  (setq buffer-read-only t)
			  (display-buffer buf 'other-win))))
		  (with-output-to-temp-buffer (hypb:help-buf-name "Smart")
		    (princ (format "A click of the %sSmart Key"
				   (if meta "secondary " "")))
		    (terpri)
		    (princ "WHEN  ")
		    (princ
		     (or condition
			 "there is no matching smart-key-alist entry"))
		    (terpri)
		    (princ "CALLS ") (princ call)
		    (if doc (progn (princ " WHICH:") (terpri) (terpri)
				   (princ doc)))))
		"")
	    (message "No %sSmart Key command for current context."
		     (if meta "secondary " ""))))
    doc))

(defun smart-key-help-hide ()
  "Restores display to configuration prior to help buffer display.
Point must be in the help buffer."
  (let ((buf (current-buffer)))
    (if *smart-key-screen-config*
	(set-window-configuration *smart-key-screen-config*)
      (switch-to-buffer (other-buffer)))
    (bury-buffer buf)
    (setq *smart-key-screen-config* nil)))

(defun smart-key-help-show (buffer)
  "Saves prior screen configuration if BUFFER displays help.  Displays BUFFER."
  (if (bufferp buffer) (setq buffer (buffer-name buffer)))
  (and (stringp buffer)
       (string-match "Help\\*$" buffer)
       (not (memq t (mapcar '(lambda (wind)
			       (string-match "Help\\*$" buffer))
			    (hypb:window-list 'no-mini))))
       (setq *smart-key-screen-config* (current-window-configuration)))
  (let ((wind (display-buffer (get-buffer-create buffer))))
    (setq minibuffer-scroll-window wind)))


;;; ************************************************************************
;;; Global smart-key key bindings
;;; ************************************************************************

(defun smart-key-summarize ()
  "Displays smart key operation summary in help buffer."
  (let* ((doc-file (concat hyperb:dir "hmouse-doc"))
	 (buf-name (hypb:help-buf-name "Smart"))
	 (wind (get-buffer-window buf-name))
	 owind)
    (if (file-readable-p doc-file)
	(progn
	  (if (and (fboundp 'br-in-browser) (br-in-browser))
	      (br-to-view-window))
	  (setq owind (selected-window))
	  (unwind-protect
	      (progn
		(if wind
		    (select-window wind)
		  (smart-key-help-show buf-name)
		  (select-window (get-buffer-window buf-name)))
		(setq buffer-read-only nil) (erase-buffer)
		(insert-file-contents doc-file)
		(goto-char (point-min))
		(set-buffer-modified-p nil))
	    (select-window owind))))))

(and (boundp 'smart-key-init) smart-key-init
     (global-set-key
      "\M-\C-m"
      '(lambda (arg)
	 (interactive "P")
	 (funcall (if arg 'smart-key 'smart-key-meta)))))

;; ************************************************************************
;; Private variables
;; ************************************************************************

(defvar *smart-key-mouse-prev-window* nil
  "Window point was in prior to current invocation of 'smart-key-mouse(-meta)'.")

(defvar *smart-key-mouse-prefix-arg* nil
  "Prefix argument to pass to 'smart-br-cmd-select'.")

(defvar *smart-key-depressed* nil "t while Smart key is depressed.")
(defvar *smart-key-meta-depressed* nil "t while Smart key is depressed.")
(defvar *smart-key-help-msg* "" "Holds last Smart key help message.")
(defvar *smart-key-screen-config* nil
  "Screen configuration prior to display of a help buffer.")

;;; ************************************************************************
;;; smart-key support functions
;;; ************************************************************************

;; Most 'smart' functions use the end-of-line position to scroll a buffer up or
;; down a screen.  These next two functions are used to keep point at the end
;; of line when using a keyboard key and meta-key to execute 'smart' functions.
;; Each subsequent push of such a key repeats the scroll action.
;; They may also be used to test whether the scroll action would be
;; successful, no action is taken if it would fail and nil is returned.

(defun scroll-up-eol ()
  (if (pos-visible-in-window-p (point-max))
      nil
    (scroll-up)
    (end-of-line)
    t))

(defun scroll-down-eol ()
  (if (pos-visible-in-window-p (point-min))
      nil
    (scroll-down)
    (end-of-line)
    t))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defun smart-key-set-point (set-point-arg-list)
  "Set point to cursor position."
  (if (fboundp mouse-set-point-command)
      (progn
	(and (boundp 'drag-button) drag-button
	     (progn (delete-button drag-button)
		    (setq drag-button nil)))
	(setq smart-key-point-prev (point))
	(if set-point-arg-list
	    (funcall mouse-set-point-command set-point-arg-list)
	  (funcall mouse-set-point-command))
	t)))
    
(provide 'hmouse-drv)
