;;!emacs
;; $Id:
;;
;; FILE:         hmouse-tag.el
;; SUMMARY:      Smart Key support of programming language tags location.
;; USAGE:        GNU Emacs Lisp Library
;;
;; AUTHOR:       Bob Weiner
;; ORG:          Brown U.
;;
;; ORIG-DATE:    24-Aug-91
;; LAST-MOD:      3-Nov-92 at 18:45:42 by Bob Weiner
;;
;; This file is part of Hyperbole.
;;
;; Copyright (C) 1991, 1992, Brown University, Providence, RI
;; Available for use and distribution under the same terms as GNU Emacs.
;;
;;
;; DESCRIPTION:
;;
;;   Only supports C, Lisp, and if the OO-Browser is available, C++.
;;   See the GNU Emacs manual for information on how to create a TAGS file
;;     from the 'etags' program.
;;   Does not support the 'ctags' tags file format.
;;
;;   YOU MUST APPROPRIATELY SET THE PUBLIC VARIABLES BELOW BEFORE USE.
;;
;; DESCRIP-END.

;;; ************************************************************************
;;; Other required Elisp libraries
;;; ************************************************************************

(if (featurep 'tags)
    nil
  ;; wtags is Bob Weiner's unreleased changes to "tags.el" to
  ;; do exact tag matching.  That package is not required to use this code.
  (or (load "wtags" t)
      (and (load "tags" t) (provide 'tags))
      ;; For Emacs V19
      (load "etags")))

;;; ************************************************************************
;;; Public variables
;;; ************************************************************************

(defvar smart-emacs-tags-file nil
  "*Full path name of etags file for GNU Emacs source.")

(defvar smart-c-cpp-include-dirs '("/usr/include/")
  "*Ordered list of include directories by default searched by C/C++ preprocessor.
Each directory must end with a directory separator.  See also
'smart-c-include-dirs'.")

(defvar smart-c-include-dirs nil
  "*Ordered list of directories to search for C/C++ include files.
Each directory must end with a directory separator.  Directories normally
searched by the C/C++ pre-processor should be set instead in
'smart-c-cpp-include-dirs'.")

(defvar smart-c-use-lib-man nil
  "When non-nil makes 'smart-c' and 'smart-c++' display man pages for recognized lib symbols.
When nil, 'smart-c' and 'smart-c++' look up only symbols defined in an etags
TAGS file.

Create the file ~/.CLIBS-LIST and populate it with the full pathnames (one per
line) of all of the C/C++ libraries whose symbols you want to match against.
Your MANPATH environment variable must include paths for the man pages of
these libraries also.

Your smart-clib-sym executable script must output a 1 if a symbol is from a
C/C++ library listed in ~/.CLIBS-LIST or 0 if not!  Otherwise, don't set this
variable to t.")


;;; ************************************************************************
;;; Public functions
;;; ************************************************************************

(defun smart-c (&optional identifier next)
  "Jumps to the definition of optional C IDENTIFIER or the one at point.
Optional second arg NEXT means jump to next matching C tag.

It assumes that its caller has already checked that the key was pressed in an
appropriate buffer and has moved the cursor to the selected buffer.

If:
 (1) on a '#include' statement, the include file is displayed;
 (2) on a C identifier, the identifier definition is displayed,
     assuming the identifier is found within an 'etags' generated tag file
     in the current directory or any of its ancestor directories.
 (3) if 'smart-c-use-lib-man' is non-nil, the C identifier is
     recognized as a library symbol, and a man page is found for the
     identifier, then the man page is displayed."

  (interactive)
  (or
    (or identifier (smart-c-include-file))
   (let ((tags-file-name (smart-tags-file buffer-file-name))
	 (tag (or identifier (smart-c-at-tag-p))))
     (message "Looking for '%s' in '%s'..." tag tags-file-name)
     (condition-case junk
	 (progn
	   (funcall (if (and (fboundp 'br-in-browser) (br-in-browser))
			'find-tag 'find-tag-other-window)
		    tag next)
	   (message "Found definition for '%s'." tag))
       (error
	(if (not smart-c-use-lib-man)
	    (progn
	      (message "'%s' not found in '%s'." tag tags-file-name)
	      (beep))
	  (message "Checking if '%s' is a C library function..." tag)
	      (let ((buf (get-buffer-create "*junk*"))
		    (found))
		(save-excursion
		  (set-buffer buf)
		  (setq buffer-read-only nil)
		  (erase-buffer)
		  (call-process (expand-file-name "smart-clib-sym"
						  hyperb:dir)
				nil buf nil tag)
		  (setq found (string= (buffer-substring 1 2) "1"))
		  (set-buffer-modified-p nil)
		  (kill-buffer buf))
		(if found
		    (progn
		      (message "C lib man page for '%s'." tag)
		      (manual-entry tag))
		  (message "'%s' not found in '%s' or C libs."
			   tag tags-file-name)
		  (beep)))))))))

(defun smart-c-at-tag-p ()
  "Return C tag name that point is within, else nil."
  (let* ((identifier-chars "_a-zA-Z0-9")
	 (identifier (concat "[_a-zA-Z][" identifier-chars "]*")))
    (save-excursion
      (skip-chars-backward identifier-chars)
      (if (looking-at identifier)
	  (progn (setq identifier
		       (buffer-substring (point) (match-end 0)))
		 (and (re-search-backward "/\\*\\|\\*/" nil t)
		      (looking-at "/\\*")
		      (setq identifier nil))
		 identifier)))))

(defun smart-c-include-file ()
  "If point is on an include file line, tries to display file.
Returns non-nil iff on an include file line, even if file is not found.
Look for include file in 'smart-c-cpp-include-dirs' and in directory list
'smart-c-include-dirs'."
  (let ((opoint (point)))
    (beginning-of-line)
    (if (looking-at smart-c-include-regexp)
	(let ((incl-type (string-to-char
			  (buffer-substring (match-beginning 2)
					    (1+ (match-beginning 2)))))
	      (file (buffer-substring (match-beginning 3) (match-end 3)))
	      (path)
	      (dir-list smart-c-include-dirs)
	      (found))
	  (goto-char opoint)
	  (setq dir-list (if (= incl-type ?<)
			     (append dir-list smart-c-cpp-include-dirs)
			   (cons (file-name-directory buffer-file-name)
				 dir-list)))
	  (while dir-list
	    (setq path (expand-file-name file (car dir-list))
		  dir-list (if (setq found (file-exists-p path))
			       nil
			     (cdr dir-list))))
	  ;;
	  ;; If found, display file
	  ;;
	  (if found
	      (if (and (file-readable-p path)
		       (progn
			 (find-file-other-window path)
			 (cond ((featurep 'c-mode) t)
			       ((load "c-mode" nil 'nomessage)
				(provide 'c-mode))
			       (t
				(beep)
				(message
				 "(smart-c-include-file):  c-mode undefined.")
				nil
				))))
		  nil
		(beep)
		(message "(smart-c-include-file):  '%s' unreadable." path))
	    (beep)
	    (message "(smart-c-include-file):  '%s' not found." file))
	  path)
      (goto-char opoint)
      nil)))


(defun smart-emacs-lisp-mode-p ()
  (or (eq major-mode 'emacs-lisp-mode)
      (eq major-mode 'lisp-interaction-mode)
      (eq major-mode 'debugger-mode)))

(defun smart-lisp (&optional next)
  "Jumps to the definition of any selected Lisp construct.
If on an Emacs Lisp require, load, or autoload clause and 'find-library'
from load-library package by Hallvard Furuseth (hallvard@ifi.uio.no) has
been loaded, jumps to library source, if possible.

Otherwise, the identifier must be found within an 'etags' generated tag file
in the current directory or any of its ancestor directories in order for its
definition to be located.

Optional NEXT means jump to next matching Lisp tag.  When matching to an Emacs
Lisp tag using 'wtags' (Bob Weiner's personal modifications to 'tags'), there
is no next tag, so display documentation for current tag instead.

This command assumes that its caller has already checked that the key was
pressed in an appropriate buffer and has moved the cursor to the selected
buffer."

  (interactive)
  ;; Handle 'require', 'load', and 'autoload' clauses in Emacs Lisp.
  (or (and (fboundp 'find-library)
	   (smart-emacs-lisp-mode-p)
	   (let ((req)
		 (opoint (point)))
	     (setq req (and (search-backward "\(" nil t)
			    (looking-at (concat
					 "(\\(require\\|load\\|autoload\\)"
					 "[ \t]+.*['\"]"
					 "\\([^][() \t\n\^M`'\"]+\\)"))))
	     (goto-char opoint)
	     (if req (progn
		       (setq req (buffer-substring (match-beginning 2)
						   (match-end 2)))
		       (pop-to-buffer nil t)
		       (find-library req)
		       t))))
      (let ((tags-file-name (smart-tags-file default-directory))
	    (tag (smart-lisp-at-tag-p)))
	;; This part only works properly for Emacs Lisp, so is conditionalized.
	(if (and next (smart-emacs-lisp-mode-p) (featurep 'wtags))
	    (progn (setq tag (intern tag))
		   (cond ((fboundp tag) (describe-function tag))
			 ((boundp tag) (describe-variable tag))
			 (t (error "(smart-lisp): Unbound symbol: %s" tag))))
	  (condition-case ()
	      (funcall (if (and (fboundp 'br-in-browser) (br-in-browser))
			   'find-tag 'find-tag-other-window)
		       tag next)
	    (error (if (equal tags-file-name smart-emacs-tags-file)
		       nil
		     (setq tags-file-name smart-emacs-tags-file)
		     (funcall (if (and (fboundp 'br-in-browser)
				       (br-in-browser))
				  'find-tag 'find-tag-other-window)
			      tag next))))))))

(defun smart-lisp-at-tag-p ()
  "Return Lisp tag name that point is within, else nil."
  (let* ((identifier-chars "-_*:+%$#!<>a-zA-Z0-9")
	 (identifier (concat "[-<*a-zA-Z][" identifier-chars "]*"))
	 (opoint (point)))
    (save-excursion
      (beginning-of-line)
      (if (and (looking-at "\\(;*[ \t]*\\)?(def")
	       (> opoint (match-end 0)))
	  nil
	(goto-char opoint)
	(skip-chars-backward identifier-chars)
	(if (looking-at identifier)
	    (progn (setq identifier
			 (buffer-substring (point) (match-end 0)))
		   (let ((pnt (point)))
		     (beginning-of-line)
		     (if (search-forward ";" pnt t)
			 nil
		       identifier))))))))

(defun smart-lisp-mode-p ()
  (or (eq major-mode 'lisp-mode)
      (eq major-mode 'emacs-lisp-mode)
      (eq major-mode 'debugger-mode)
      (eq major-mode 'scheme-mode)
      (eq major-mode 'lisp-interaction-mode)))

(defun smart-c++ (&optional identifier next)
  "Jumps to the definition of optional C++ IDENTIFIER or the one at point.
Optional second arg NEXT means jump to next matching C++ tag.

It assumes that its caller has already checked that the key was pressed in an
appropriate buffer and has moved the cursor to the selected buffer.

If:
 (1) on a '#include' statement, the include file is displayed;
 (2) on a C++ identifier, the identifier definition is displayed,
     assuming the identifier is found within an 'etags' generated tag file
     in the current directory or any of its ancestor directories.
 (3) if 'smart-c-use-lib-man' is non-nil, the C++ identifier is
     recognized as a library symbol, and a man page is found for the
     identifier, then the man page is displayed."

  (interactive)
  (or
    (or identifier (smart-c-include-file))
    (let ((tags-file-name (smart-tags-file buffer-file-name))
	  (tag (or identifier (smart-c++-at-tag-p))))
      (message "Looking for '%s' in '%s'..." tag tags-file-name)
      (condition-case junk
	  (progn
	    (funcall (if (and (fboundp 'br-in-browser) (br-in-browser))
			 'find-tag 'find-tag-other-window)
		     tag next)
	    (message "Found definition for '%s'." tag))
	(error
	  (if (not smart-c-use-lib-man)
	      (progn
		(message "'%s' not found in '%s'." tag tags-file-name)
		(beep))
	    (message "Checking if '%s' is a C++ library function..." tag)
	    (let ((buf (get-buffer-create "*junk*"))
		  (found))
	      (save-excursion
		(set-buffer buf)
		(setq buffer-read-only nil)
		(erase-buffer)
		(call-process (expand-file-name "smart-clib-sym"
						hyperb:dir)
			      nil buf nil tag)
		(setq found (string= (buffer-substring 1 2) "1"))
		(set-buffer-modified-p nil)
		(kill-buffer buf))
	      (if found
		  (progn
		    (message "C++ lib man page for '%s'." tag)
		    (manual-entry tag))
		(message "'%s' not found in '%s' or C++ libs."
			 tag tags-file-name)
		(beep)))))))))

;;; The following should not be called if the OO-Browser is unavailable.
(defun smart-c++-oobr (&optional junk)
  "Jumps to the definition of selected C++ construct via OO-Browser support.
Optional JUNK is ignored.  Does nothing if the OO-Browser is not available.

It assumes that its caller has already checked that the key was pressed in an
appropriate buffer and has moved the cursor to the selected buffer.

If key is pressed:
 (1) on a '#include' statement, the include file is displayed;
 (2) within a method declaration, its definition is displayed;
 (3) on a class name, the class definition is shown.

 (2) and (3) require that an OO-Browser Environment has been loaded with
     the {M-x br-env-load RTN} command."

  (interactive)
  (c++-to-definition 'other-win))

(defun smart-c++-at-tag-p ()
  "Return C++ tag name that point is within, else nil."
  (let* ((identifier-chars "_:~<>a-zA-Z0-9")
	 (identifier (concat "\\([_~:<a-zA-Z][" identifier-chars "]*"
			     "[ \t]*[^]) \t:;.,?~{}][^[( \t:;.,~^!|?{}]?[=*]?\\)[ \t\n]*(")))
    (save-excursion
      (skip-chars-backward identifier-chars)
      (if (looking-at identifier)
	  (progn (setq identifier
		       (buffer-substring (point) (match-end 1)))
		 (and (re-search-backward "/\\*\\|\\*/" nil t)
		      (looking-at "/\\*")
		      (setq identifier nil))
		 identifier)))))



;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defun smart-tags-file (curr-filename)
  "Return appropriate tags file name for CURR-FILENAME or 'tags-file-name'."
  (let ((path curr-filename)
	(tags-file))
    (while (and (not (equal
		      (setq path (directory-file-name
				  (file-name-directory path)))
		      "/"))
		(not (file-exists-p
		      (setq tags-file (expand-file-name "TAGS" path)))))
      (setq tags-file nil))
    (if (and (not tags-file)
	     (eq major-mode 'emacs-lisp-mode)
	     (let ((path (file-name-directory curr-filename)))
	       (delq nil (mapcar
			  '(lambda (p)
			     (and p (equal (file-name-as-directory p)
					   path)))
			  load-path))))
	(setq tags-file smart-emacs-tags-file))
    (or tags-file tags-file-name
	(call-interactively 'visit-tags-table))))

;;; ************************************************************************
;;; Private functions
;;; ************************************************************************

(defconst smart-c-include-regexp
  "[ \t/*]*#[ \t]*\\(include\\|import\\)[ \t]+\\([\"<]\\)\\([^\">]+\\)[\">]"
  "Regexp to match to C or Objective-C include file lines.
Include keyword matched is grouping 1.  Type of include, user-specified via
double quote, or system-related starting with '<' is given by grouping 2.
File name is grouping 3.")

(provide 'hmouse-tag)
