/* Gcov.c: prepend line execution counts and branch probabilities to a
   source file.
   Contributed by James E. Wilson of Cygnus Support.

Gcov is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

Gcov is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Gcov; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

/* ??? No need to print an execution count on every line, could just print
   it on the first line of each block, and only print it on a subsequent
   line in the same block if the count changes.  */

/* ??? Print a list of the ten blocks with the highest execution counts,
   and list the line numbers corresponding to those blocks.  Also, perhaps
   list the line numbers with the highest execution counts, only printing
   the first if there are several which are all listed in the same block.  */

/* ??? Does not correctly handle the case where two .bb files refer to the
   same included source file.  For example, if one has a short file containing
   only inline functions, which is then included in two other files, then
   there will be two .bb files which refer to the include file, but there
   is no way to get the total execution counts for the included file, can
   only get execution counts for one or the other of the including files.  */

#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <strings.h>
#include <malloc.h>

/* The .bb file format consists of a list of line numbers for each basic
   block in the file.  Each list is terminated by a zero.  These lists
   correspond to the basic blocks in the reconstructed program flow graph.
   A minus one indicates that a source file name (padded to a four byte
   boundary) follows.  The padded file name is followed by another -1 to
   make it easy to scan past file names.

   The .bbg file contains enough info to enable gcov to reconstruct the
   program flow graph.  The first word is the number of basic blocks,
   the second word is the number of arcs, followed by the list of arcs
   (source bb, dest bb pairs), then a -1, then the number of instrumented
   arcs followed by the instrumented arcs, followed by another -1.  This
   is repeated for each function.

   The .da file contains the execution count for each instrumented branch.

   The .bb and .bbg files are created by giving GCC the -ftest-coverage option,
   and the .da files are created when an executable compiled with
   -fprofile-arcs is run.  */

/* The functions in this file for creating and solution program flow graphs
   are very similar to functions in the gcc source file profile.c.  */

#define TRUE  1
#define FALSE 0

char version_string[] = "GNU gcov version 1.3\n";

/* This is the size of the buffer used to read in source file lines.  */

#define STRING_SIZE 200

/* One copy of this structure is created for each source file mentioned in the
   .bb file.  */

struct sourcefile
{
  char *name;
  int maxlineno;
  struct sourcefile *next;
};

/* This points to the head of the sourcefile structure list.  */

struct sourcefile *sources;

/* One of these is dynamically created whenever we identify an arc in the
   function.  */

struct adj_list {
  int source;
  int target;
  int arc_count;
  unsigned int count_valid : 1;
  unsigned int on_tree : 1;
  unsigned int fake : 1;
  unsigned int fall_through : 1;
#if 0
  /* Not needed for gcov, but defined in profile.c.  */
  rtx branch_insn;
#endif
  struct adj_list *pred_next;
  struct adj_list *succ_next;
};

/* Count the number of basic blocks, and create an array of these structures,
   one for each bb in the function.  */

struct bb_info {
  struct adj_list *succ;
  struct adj_list *pred;
  int succ_count;
  int pred_count;
  int exec_count;
  unsigned int count_valid : 1;
  unsigned int on_tree : 1;
#if 0
  /* Not needed for gcov, but defined in profile.c.  */
  rtx first_insn;
#endif
};

/* When outputting branch probabilities, one of these structures is created
   for each branch/call.  */

struct arcdata
{
  int prob;
  int call_insn;
  struct arcdata *next;
};

/* Used to save the list of bb_graphs, one per function.  */

struct bb_info_list {
  /* Indexed by block number, holds the basic block graph for one function.  */
  struct bb_info *bb_graph;
  int max_block_num;
  struct bb_info_list *next;
};

/* Holds a list of function basic block graphs.  */

static struct bb_info_list *bb_graph_list = 0;

/* Name and file pointer of the input file for the basic block graph.  */

static char *bbg_file_name;
static FILE *bbg_file;

/* Name and file pointer of the input file for the arc count data.  */

static char *d_file_name;
static FILE *d_file;

/* Name and file pointer of the input file for the basic block line counts.  */

static char *bb_file_name;
static FILE *bb_file;

/* Holds the entire contents of the bb_file read into memory.  */

static long *bb_data;

/* Size of bb_data array in longs.  */

static long bb_data_size;

/* Name and file pointer of the output file.  */

static char *gcov_file_name;
static FILE *gcov_file;

/* Name and file pointer for the input source file.  */

static char *source_file_name = 0;
static FILE *source_file;

/* Output branch probabilities if true.  */

static int output_branch_probs = 0;

/* Output a gcov file if this is true.  This is on by default, and can
   be turned off by the -n option.  */

static int output_gcov_file = 1;

/* Object directory file prefix.  This is the directory where .bb and .bbg
   files are looked for, if non-zero.  */

static char *object_directory = 0;

/* Source directory file prefix.  This is the directory where .da files
   are looked for, if non-zero.  */

static char *source_directory = 0;

/* Forward declarations.  */
static void process_args ();
static void open_files ();
static void read_files ();
static void scan_for_source_files ();
static void output_data ();

int
main (argc, argv)
     int argc;
     char **argv;
{
  process_args (argc, argv);

  open_files ();

  read_files ();

  scan_for_source_files ();

  output_data ();

  return 0;
}

/* Print a usage message and exit.  */

static void
print_usage ()
{
  fprintf (stderr, "gcov [-v] [-b] [-o OBJDIR] [-s SRCDIR] [-n] file\n");
  exit (1);
}


/* Parse the command line.  */

static void
process_args (argc, argv)
     int argc;
     char **argv;
{
  int i;

  for (i = 1; i < argc; i++)
    {
      if (argv[i][0] == '-')
	{
	  if (argv[i][1] == 'v')
	    fputs (version_string, stderr);
	  else if (argv[i][1] == 'b')
	    output_branch_probs = 1;
	  else if (argv[i][1] == 'n')
	    output_gcov_file = 0;
	  else if (argv[i][1] == 'o' && argv[i][2] == '\0')
	    object_directory = argv[++i];
	  else if (argv[i][1] == 's' && argv[i][2] == '\0')
	    source_directory = argv[++i];
	  else
	    print_usage ();
	}
      else if (! source_file_name)
	source_file_name = argv[i];
      else
	print_usage ();
    }

  if (! source_file_name)
    print_usage ();
}


/* Find and open the .bb, .da, and .bbg files.  */

static void
open_files ()
{
  int count, objdir_count, srcdir_count;
  char *cptr;

  /* Determine the names of the .bb, .bbg, and .da files.  Strip off the
     extension, if any, and append the new extensions.  */
  count = strlen (source_file_name);
  if (object_directory)
    objdir_count = strlen (object_directory);
  else
    objdir_count = 0;
  if (source_directory)
    srcdir_count = strlen (source_directory);
  else
    srcdir_count = 0;

  d_file_name = malloc (count + srcdir_count + 3);
  bb_file_name = malloc (count + objdir_count + 4);
  bbg_file_name = malloc (count + objdir_count + 5);

  if (source_directory)
    {
      strcpy (d_file_name, source_directory);

      if (source_directory[strlen (source_directory)] != '/')
	strcat (d_file_name, "/");

      cptr = rindex (source_file_name, '/');
      if (cptr)
	strcat (d_file_name, cptr);
      else
	strcat (d_file_name, source_file_name);
    }
  else
    strcpy (d_file_name, source_file_name);

  if (object_directory)
    {
      strcpy (bb_file_name, object_directory);
      strcpy (bbg_file_name, object_directory);

      if (object_directory[strlen (object_directory)] != '/')
	{
	  strcat (bb_file_name, "/");
	  strcat (bbg_file_name, "/");
	}

      cptr = rindex (source_file_name, '/');
      if (cptr)
	{
	  strcat (bb_file_name, cptr);
	  strcat (bbg_file_name, cptr);
	}
      else
	{
	  strcat (bb_file_name, source_file_name);
	  strcat (bbg_file_name, source_file_name);
	}
    }
  else
    {
      strcpy (bb_file_name, source_file_name);
      strcpy (bbg_file_name, source_file_name);
    }

  cptr = rindex (bb_file_name, '.');
  if (cptr)
    strcpy (cptr, ".bb");
  else
    strcat (bb_file_name, ".bb");

  cptr = rindex (d_file_name, '.');
  if (cptr)
    strcpy (cptr, ".da");
  else
    strcat (d_file_name, ".da");

  cptr = rindex (bbg_file_name, '.');
  if (cptr)
    strcpy (cptr, ".bbg");
  else
    strcat (bbg_file_name, ".bbg");

  bb_file = fopen (bb_file_name, "r");
  if (bb_file == NULL)
    {
      fprintf (stderr, "Could not open basic block file %s.\n", bb_file_name);
      exit (1);
    }

  /* If none of the functions in the file were executed, then there won't
     be a .da file.  Just assume that all counts are zero in this case.  */
  d_file = fopen (d_file_name, "r");
  if (d_file == NULL)
    {
      fprintf (stderr, "Could not open data file %s.\n", d_file_name);
      fprintf (stderr, "Assuming that all execution counts are zero.\n");
    }
    
  bbg_file = fopen (bbg_file_name, "r");
  if (bbg_file == NULL)
    {
      fprintf (stderr, "Could not open program flow graph file %s.\n",
	       bbg_file_name);
      exit (1);
    }

  /* Check for empty .bbg file.  This indicates that there is no executable
     code in this source file.  */
  /* Set the EOF condition if at the end of file.  */
  ungetc (getc (bbg_file), bbg_file);
  if (feof (bbg_file))
    {
      fprintf (stderr, "No executable code associated with file %s.\n",
	       source_file_name);
      exit (2);
    }
}

/* Initialize a new arc.  */

static void
init_arc (arcptr, source, target, bb_graph)
     struct adj_list *arcptr;
     int source, target;
     struct bb_info *bb_graph;
{
  arcptr->target = target;
  arcptr->source = source;

  arcptr->arc_count = 0;
  arcptr->count_valid = 0;
  arcptr->on_tree = 0;
  arcptr->fake = 0;
  arcptr->fall_through = 0;

  arcptr->succ_next = bb_graph[source].succ;
  bb_graph[source].succ = arcptr;
  bb_graph[source].succ_count++;

  arcptr->pred_next = bb_graph[target].pred;
  bb_graph[target].pred = arcptr;
  bb_graph[target].pred_count++;
}


/* Reverse the arcs on a arc list.  */

static struct adj_list *
reverse_arcs (arcptr)
     struct adj_list *arcptr;
{
  struct adj_list *prev = 0;
  struct adj_list *next;

  for ( ; arcptr; arcptr = next)
    {
      next = arcptr->succ_next;
      arcptr->succ_next = prev;
      prev = arcptr;
    }

  return prev;
}


/* Construct the program flow graph from the .bbg file, and read in the data
   in the .da file.  */

static void
create_program_flow_graph (bptr)
     struct bb_info_list *bptr;
{
  int max_block_num, src, dest;
  int number_arcs, i, flag_bits;
  struct adj_list *arcptr;
  struct bb_info *bb_graph;

  /* Read the number of blocks.  */
  fread (&max_block_num, sizeof (max_block_num), 1, bbg_file);

  /* Create an array of size bb number of bb_info structs.  Bzero it.  */
  bb_graph = (struct bb_info *) malloc (max_block_num
					* sizeof (struct bb_info));
  bzero (bb_graph, sizeof (struct bb_info) * max_block_num);

  bptr->bb_graph = bb_graph;
  bptr->max_block_num = max_block_num;

  /* Read and create each arc from the .bbg file.  */
  fread (&number_arcs, sizeof (number_arcs), 1, bbg_file);
  for (i = 0; i < number_arcs; i++)
    {
      fread (&src, sizeof (src), 1, bbg_file);
      fread (&dest, sizeof (dest), 1, bbg_file);

      arcptr = (struct adj_list *) malloc (sizeof (struct adj_list));
      init_arc (arcptr, src, dest, bb_graph);

      fread (&flag_bits, sizeof (flag_bits), 1, bbg_file);
      arcptr->on_tree = flag_bits & 0x1;
      arcptr->fake = !! (flag_bits & 0x2);
      arcptr->fall_through = !! (flag_bits & 0x4);
    }

  /* Read and ignore the -1 separating the arc list from the arc list of the
     next function.  */
  fread (&src, sizeof (src), 1, bbg_file);
  if (src != -1)
    abort ();

  /* Must reverse the order of all succ arcs, to ensure that they match
     the order of the data in the .da file.  */

  for (i = 0; i < max_block_num; i++)
    if (bb_graph[i].succ)
      bb_graph[i].succ = reverse_arcs (bb_graph[i].succ);

  /* For each arc not on the spanning tree, set its execution count from
     the .da file.  */

  /* The first count in the .da file is the number of times that the function
     was entered.  This is the exec_count for block zero.  */

  /* This duplicates code in branch_prob in profile.c.  */

  for (i = 0; i < max_block_num; i++)
    for (arcptr = bb_graph[i].succ; arcptr; arcptr = arcptr->succ_next)
      if (! arcptr->on_tree)
	{
	  if (d_file)
	    fread (&arcptr->arc_count, sizeof (arcptr->arc_count), 1, d_file);
	  else
	    arcptr->arc_count = 0;
	  arcptr->count_valid = 1;
	  bb_graph[i].succ_count--;
	  bb_graph[arcptr->target].pred_count--;
	}
}
  

static void
solve_program_flow_graph (bptr)
     struct bb_info_list *bptr;
{
  int passes, changes, total;
  int i;
  struct adj_list *arcptr;
  struct bb_info *bb_graph;
  int max_block_num;

  max_block_num = bptr->max_block_num;
  bb_graph = bptr->bb_graph;

  /* For every block in the file,
     - if every exit/entrance arc has a known count, then set the block count
     - if the block count is known, and every exit/entrance arc but one has
       a known execution count, then set the count of the remaining arc

     As arc counts are set, decrement the succ/pred count, but don't delete
     the arc, that way we can easily tell when all arcs are known, or only
     one arc is unknown.  */

  /* The order that the basic blocks are iterated through is important.
     Since the code that finds spanning trees starts with block 0, low numbered
     arcs are put on the spanning tree in preference to high numbered arcs.
     Hence, most instrumented arcs are at the end.  Graph solving works much
     faster if we propagate numbers from the end to the start.

     This takes an average of slightly more than 3 passes.  */

  changes = 1;
  passes = 0;
  while (changes)
    {
      passes++;
      changes = 0;

      for (i = max_block_num - 1; i >= 0; i--)
	{
	  if (! bb_graph[i].count_valid)
	    {
	      if (bb_graph[i].succ_count == 0)
		{
		  total = 0;
		  for (arcptr = bb_graph[i].succ; arcptr;
		       arcptr = arcptr->succ_next)
		    total += arcptr->arc_count;
		  bb_graph[i].exec_count = total;
		  bb_graph[i].count_valid = 1;
		  changes = 1;
		}
	      else if (bb_graph[i].pred_count == 0)
		{
		  total = 0;
		  for (arcptr = bb_graph[i].pred; arcptr;
		       arcptr = arcptr->pred_next)
		    total += arcptr->arc_count;
		  bb_graph[i].exec_count = total;
		  bb_graph[i].count_valid = 1;
		  changes = 1;
		}
	    }
	  if (bb_graph[i].count_valid)
	    {
	      if (bb_graph[i].succ_count == 1)
		{
		  total = 0;
		  /* One of the counts will be invalid, but it is zero,
		     so adding it in also doesn't hurt.  */
		  for (arcptr = bb_graph[i].succ; arcptr;
		       arcptr = arcptr->succ_next)
		    total += arcptr->arc_count;
		  /* Calculate count for remaining arc by conservation.  */
		  total = bb_graph[i].exec_count - total;
		  /* Search for the invalid arc, and set its count.  */
		  for (arcptr = bb_graph[i].succ; arcptr;
		       arcptr = arcptr->succ_next)
		    if (! arcptr->count_valid)
		      break;
		  if (! arcptr)
		    abort ();
		  arcptr->count_valid = 1;
		  arcptr->arc_count = total;
		  bb_graph[i].succ_count--;

		  bb_graph[arcptr->target].pred_count--;
		  changes = 1;
		}
	      if (bb_graph[i].pred_count == 1)
		{
		  total = 0;
		  /* One of the counts will be invalid, but it is zero,
		     so adding it in also doesn't hurt.  */
		  for (arcptr = bb_graph[i].pred; arcptr;
		       arcptr = arcptr->pred_next)
		    total += arcptr->arc_count;
		  /* Calculate count for remaining arc by conservation.  */
		  total = bb_graph[i].exec_count - total;
		  /* Search for the invalid arc, and set its count.  */
		  for (arcptr = bb_graph[i].pred; arcptr;
		       arcptr = arcptr->pred_next)
		    if (! arcptr->count_valid)
		      break;
		  if (! arcptr)
		    abort ();
		  arcptr->count_valid = 1;
		  arcptr->arc_count = total;
		  bb_graph[i].pred_count--;

		  bb_graph[arcptr->source].succ_count--;
		  changes = 1;
		}
	    }
	}
    }
	      
  /* If the graph has been correctly solved, every block will have a
     succ and pred count of zero.  */
  for (i = 0; i < max_block_num; i++)
    if (bb_graph[i].succ_count || bb_graph[i].pred_count)
      abort ();
}


static void
read_files ()
{
  struct stat buf;
  struct bb_info_list *list_end = 0;
  struct bb_info_list *b_ptr;
  int total;

  /* Read and ignore the first word of the .da file, which is the count of
     how many numbers follow.  */
  if (d_file)
    fread (&total, sizeof (total), 1, d_file);

  while (! feof (bbg_file))
    {
      b_ptr = (struct bb_info_list *) malloc (sizeof (struct bb_info_list));

      b_ptr->next = 0;
      if (list_end)
	list_end->next = b_ptr;
      else
	bb_graph_list = b_ptr;
      list_end = b_ptr;

      /* Read in the data in the .bbg file and reconstruct the program flow
	 graph for one function.  */
      create_program_flow_graph (b_ptr);

      /* Calculate all of the basic block execution counts and branch
	 taken probabilities.  */
      solve_program_flow_graph (b_ptr);

      /* Set the EOF condition if at the end of file.  */
      ungetc (getc (bbg_file), bbg_file);
    }

  /* Read in all of the data from the .bb file.   This info will be accessed
     sequentially twice.  */
  stat (bb_file_name, &buf);
  bb_data_size = buf.st_size >> 2;

  bb_data = (long *) malloc (buf.st_size);
  fread (bb_data, sizeof (char), buf.st_size, bb_file);
  
  fclose (bb_file);
  if (d_file)
    fclose (d_file);
  fclose (bbg_file);
}


/* Scan the data in the .bb file to find all source files referenced,
   and the largest line number mentioned in each one.  */

static void
scan_for_source_files ()
{
  struct sourcefile *s_ptr;
  long *ptr;
  int count;
  int line_num;

  /* Search the bb_data to find:
     1) The number of sources files contained herein, and
     2) The largest line number for each source file.  */

  ptr = bb_data;
  sources = 0;
  for (count = 0; count < bb_data_size; count++)
    {
      line_num = *ptr++;
      if (line_num == -1)
	{
	  /* A source file name follows.  Check to see if we already have
	   a sourcefile structure for this file.  */
	  s_ptr = sources;
	  while (s_ptr && strcmp (s_ptr->name, ptr))
	    s_ptr = s_ptr->next;

	  if (s_ptr == 0)
	    {
	      /* No sourcefile structure for this file name exists, create
		 a new one, and append it to the front of the sources list.  */
	      s_ptr = (struct sourcefile *) malloc (sizeof(struct sourcefile));
	      s_ptr->name = malloc (strlen ((char *) ptr) + 1);
	      strcpy (s_ptr->name, ptr);
	      s_ptr->maxlineno = 0;
	      s_ptr->next = sources;
	      sources = s_ptr;
	    }

	  /* Scan past the file name.  */
	  count++;
	  while (*ptr++ != -1) count++;
	}
      /* There will be a zero before the first file name, in which case s_ptr
	 will still be uninitialized.  So, only try to set the maxlineno
	 field if line_num is non-zero or not -2.  */
      else if (line_num != 0 && line_num != -2)
	{
	  if (s_ptr->maxlineno <= line_num)
	    s_ptr->maxlineno = line_num + 1;
	}
    }
}

/* Calculate the branch taken probabilities for all arcs branches at the
   end of this block.  */

static void
calculate_branch_probs (current_graph, block_num, branch_probs, last_line_num)
     struct bb_info_list *current_graph;
     int block_num;
     struct arcdata **branch_probs;
     int last_line_num;
{
  int total;
  struct adj_list *arcptr;
  struct arcdata *end_ptr, *a_ptr;

  total = current_graph->bb_graph[block_num].exec_count;
  for (arcptr = current_graph->bb_graph[block_num].succ; arcptr;
       arcptr = arcptr->succ_next)
    {
      /* Ignore fall through arcs as they aren't really branches.  */

      if (arcptr->fall_through)
	continue;
		      
      a_ptr = (struct arcdata *) malloc (sizeof (struct arcdata));
      if (total == 0)
	a_ptr->prob = -1;
      else
	a_ptr->prob = ((arcptr->arc_count * 100) + (total >> 1)) / total;
      a_ptr->call_insn = arcptr->fake;

      /* Append the new branch to the end of the list.  */
      a_ptr->next = 0;
      if (! branch_probs[last_line_num])
	branch_probs[last_line_num] = a_ptr;
      else
	{
	  end_ptr = branch_probs[last_line_num];
	  while (end_ptr->next != 0)
	    end_ptr = end_ptr->next;
	  end_ptr->next = a_ptr;
	}
    }
}


/* Calculate line execution counts, and output the data to a .tcov file.  */

static void
output_data ()
{
  /* When scanning data, this is true only if the data applies to the
     current source file.  */
  int this_file;
  /* The number of the function being processed, used only for better error
     messages.  */
  int function_number;
  /* An array indexed by line number which indicates how many times that line
     was executed.  */
  long *line_counts;
  /* An array indexed by line number which indicates whether the line was
     present in the bb file (i.e. whether it had code associate with it).
     Lines never executed are those which both exist, and have zero execution
     counts.  */
  char *line_exists;
  /* An array indexed by line number, which contains a list of branch
     probabilities, one for each branch on that line.  */
  struct arcdata **branch_probs;
  /* Pointer into the bb_data, incremented while scanning the data.  */
  long *ptr;
  struct sourcefile *s_ptr;
  struct bb_info_list *current_graph;
  int count;
  char *cptr;
  long block_num;
  long line_num;
  long last_line_num;
  int i;
  struct arcdata *a_ptr;
  /* Buffer used for reading in lines from the source file.  */
  char string[STRING_SIZE];
  /* For calculating coverage.  */
  int total_source_lines;
  int total_source_lines_executed;

  /* Now, for each source file, allocate an array big enough to hold a count
     for each line.  Scan through the bb_data, and when the file name matches
     the current file name, then for each following line number, increment
     the line number execution count indicated by the execution count of
     the appropriate basic block.  */

  for (s_ptr = sources; s_ptr; s_ptr = s_ptr->next)
    {
      line_counts = (long *) malloc (sizeof (long) * s_ptr->maxlineno);
      bzero (line_counts, sizeof (long) * s_ptr->maxlineno);
      line_exists = malloc (s_ptr->maxlineno);
      bzero (line_exists, s_ptr->maxlineno);
      if (output_branch_probs)
	{
	  branch_probs = (struct arcdata **) malloc (sizeof (struct arcdata **)
						     * s_ptr->maxlineno);
	  bzero (branch_probs, sizeof (struct arcdata **) * s_ptr->maxlineno);
	}
      
      ptr = bb_data;

      /* There will be a zero at the beginning of the bb info, before the
	 first list of line numbers, so must initialize block_num to 0.  */
      block_num = 0;
      this_file = FALSE;
      function_number = 0;
      current_graph = 0;

      for (count = 0; count < bb_data_size; count++)
	{
	  line_num = *ptr++;

	  if (line_num == -1)
	    {
	      /* Marks the beginning of a file name.  Check to see whether
		 this is the filename we are currently collecting data for.  */

	      if (strcmp (s_ptr->name, ptr))
		this_file = FALSE;
	      else
		this_file = TRUE;
	      
	      /* Scan past the file name.  */
	      count++;
	      while (*ptr++ != -1) count++;
	    }
	  else if (line_num == -2)
	    {
	      /* Marks the start of new a function.  Advance to the next
		 program flow graph.  */

	      if (! current_graph)
		current_graph = bb_graph_list;
	      else
		{
		  if (block_num == current_graph->max_block_num)
		    /* Last block falls through to exit.  */
		    ;
		  else if (block_num == current_graph->max_block_num - 1)
		    {
		      if (output_branch_probs && this_file)
			calculate_branch_probs (current_graph, block_num,
						branch_probs, last_line_num);
		    }
		  else
		    {
		      fprintf (stderr,
			       "didn't use all bb entries of graph, function #%d\n",
			       function_number);
		      fprintf (stderr, "block_num = %d, max_block_num = %d\n",
			       block_num, current_graph->max_block_num);
		    }

		  current_graph = current_graph->next;
		  block_num = 0;
		}

	      function_number++;
	    }
	  else if (line_num == 0)
	    {
	      /* Marks the end of a block.  */

	      if (block_num >= current_graph->max_block_num)
		{
		  fprintf (stderr, "ERROR: too many basic blocks in .bb file %d\n", function_number);
		  abort ();
		}
		  
	      if (output_branch_probs && this_file)
		calculate_branch_probs (current_graph, block_num,
					branch_probs, last_line_num);

	      block_num++;
	    }
	  else if (this_file)
	    {
	      /* Accumulate execution data for this line number.  */

	      line_counts[line_num]
		+= current_graph->bb_graph[block_num].exec_count;
	      line_exists[line_num] = 1;
	      last_line_num = line_num;
	    }
	}

      /* Calculate summary test coverage statistics.  */

      total_source_lines = 0;
      total_source_lines_executed = 0;

      for (count = 1; count < s_ptr->maxlineno; count++)
	{
	  if (line_exists[count])
	    {
	      total_source_lines++;
	      if (line_counts[count])
		total_source_lines_executed++;
	    }
	}

      if (total_source_lines)
	fprintf (stdout,
		 "%.2lf%% of %d source lines executed in file %s\n",
		 (((double) total_source_lines_executed / total_source_lines)
		  * 100), total_source_lines, s_ptr->name);
      else
	fprintf (stdout, "No executable source lines in file %s\n",
		 s_ptr->name);

      if (output_gcov_file)
	{
	  /* Now the statistics are ready.  Read in the source file one line
	     at a time, and output that line to the gcov file preceeded by
	     its execution count if non zero.  */
      
	  source_file = fopen (s_ptr->name, "r");
	  if (source_file == NULL)
	    {
	      fprintf (stderr, "Could not open source file %s.\n",
		       s_ptr->name);
	      free (line_counts);
	      free (line_exists);
	      continue;
	    }

	  count = strlen (s_ptr->name);
	  gcov_file_name = malloc (count + 6);
	  strcpy (gcov_file_name, s_ptr->name);

	  /* Don't strip off the ending for compatibility with tcov, since
	     this results in confusion if there is more than one file with
	     the same basename, e.g. tmp.c and tmp.h.  */
	  strcat (gcov_file_name, ".gcov");

	  gcov_file = fopen (gcov_file_name, "w");

	  if (gcov_file == NULL)
	    {
	      fprintf (stderr, "Could not open output file %s.\n",
		       gcov_file_name);
	      fclose (source_file);
	      free (line_counts);
	      free (line_exists);
	      continue;
	    }

	  fprintf (stderr, "Creating %s.\n", gcov_file_name);

	  for (count = 1; count < s_ptr->maxlineno; count++)
	    {
	      char *retval;
	      int len;

	      retval = fgets (string, STRING_SIZE, source_file);

	      /* For lines which don't exist in the .bb file, print nothing
		 before the source line.  For lines which exist but were never
		 executed, print ###### before the source line.  Otherwise,
		 print the execution count before the source line.  */
	      /* There are 16 spaces of identation added before the source line
		 so that tabs won't be messed up.  */
	      if (line_exists[count])
		{
		  if (line_counts[count])
		    fprintf (gcov_file, "%12d    %s", line_counts[count],
			     string);
		  else
		    fprintf (gcov_file, "      ######    %s", string);
		}
	      else
		fprintf (gcov_file, "\t\t%s", string);

	      /* In case the source file line is larger than our buffer, keep
		 reading and outputing lines until we get a newline.  */
	      len = strlen (string);
	      while ((len == 0 || string[strlen (string) - 1] != '\n') &&
		     retval != NULL)
		{
		  retval = fgets (string, STRING_SIZE, source_file);
		  fputs (string, gcov_file);
		}

	      if (output_branch_probs)
		{
		  for (i = 0, a_ptr = branch_probs[count]; a_ptr;
		       a_ptr = a_ptr->next, i++)
		    {
		      if (a_ptr->call_insn)
			{
			  if (a_ptr->prob == -1)
			    fprintf (gcov_file, "call %d never executed\n", i);
			  else
			    fprintf (gcov_file,
				     "call %d doesn't return = %d%%\n",
				     i, a_ptr->prob);
			}
		      else
			{
			  if (a_ptr->prob == -1)
			    fprintf (gcov_file, "branch %d never executed\n",
				     i);
			  else
			    fprintf (gcov_file, "branch %d taken = %d%%\n", i,
				     a_ptr->prob);
			}
		    }
		}

	      /* Gracefully handle errors while reading the source file.  */
	      if (retval == NULL)
		{
		  fprintf (stderr,
			   "Unexpected EOF while reading source file %s.\n",
			   s_ptr->name);
		  break;
		}
	    }

	  fclose (source_file);
	  fclose (gcov_file_name);
	}

      free (line_counts);
      free (line_exists);
    }
}
