/*
FUNCTION
       <<ldexp>>, <<ldexpf>>---load exponent

INDEX
	ldexp
INDEX
	ldexpf

ANSI_SYNOPSIS
       #include <math.h>
       double ldexp(double <[val]>, int <[exp]>);
       float ldexpf(float <[val]>, int <[exp]>);

TRAD_SYNOPSIS
       #include <math.h>

       double ldexp(<[val]>, <[exp]>)
              double <[val]>;
              int <[exp]>;

       float ldexpf(<[val]>, <[exp]>)
              float <[val]>;
              int <[exp]>;


DESCRIPTION
<<ldexp>> calculates the value 
@ifinfo
<[val]> times 2 to the power <[exp]>.
@end ifinfo
@tex
$val\times 2^{exp}$.
@end tex
<<ldexpf>> is identical, save that it takes and returns <<float>>
rather than <<double>> values.

RETURNS
<<ldexp>> returns the calculated value.

Underflow and overflow both set <<errno>> to <<ERANGE>>.
On underflow, <<ldexp>> and <<ldexpf>> return 0.0.
On overflow, <<ldexp>> returns plus or minus <<DBL_MAX>>; <<ldexpf>>
returns plus or minus <<FLT_MAX>>.

PORTABILITY
<<ldexp>> is ANSI, <<ldexpf>> is an extension.
              
*/   

#include "mathimpl.h"

double
_DEFUN(ldexp,(val,exp),double val _AND int exp)
{
  __ieee_double_shape_type un;

  
  int oe;
  
  if (exp == 0 || val == 0.0)	/* nothing to do for zero */
   return (val);

  un.value = val;
  oe = un.number.exponent - __IEEE_DBL_EXPBIAS;
  
  if (exp > 0) 
  {
    if (exp + oe > DBL_MAX_EXP) 
    {

      return __matherror("ldexp", val, (double)exp, OVERFLOW, 
			 val <	0 ? -DBL_MAX : DBL_MAX);
    }
    un.number.exponent += exp;
    return un.value;
  }

  if (exp + oe < -DBL_MAX_EXP) 
  {
    return __matherror("ldexp", val, (double)exp, UNDERFLOW, 0.0);

  }

  un.number.exponent -= abs(exp);
  return un.value;
}


_FLOAT_RET
_DEFUN(ldexpf,(rval,exp),_FLOAT_ARG rval _AND int exp)
{
  __ieee_float_shape_type un;
  float val = rval;
  int oe;
  
  if (exp == 0 || val == 0.0)	/* nothing to do for zero */
   return (val);
  un.value = val;
  oe = un.number.exponent - __IEEE_FLT_EXPBIAS;

  if (exp > 0) {
      if (exp + oe > FLT_MAX_EXP) {
      return __matherror("ldexpf", val, (double)exp, OVERFLOW, 
			 val <	0 ? -FLT_MAX : FLT_MAX);

	}
      un.number.exponent += exp;
      return un.value;
    }
  /* exp is negative */
  if (exp + oe < -FLT_MAX_EXP) {
    return __matherror("ldexpf", val, (double)exp, UNDERFLOW, 0.0);
    }
  un.number.exponent -= abs(exp);
  return un.value;
}

