char rcsid[] = "$Id: comm960.c,v 1.17 89/12/10 10:43:08 chrisb Exp $";
/******************************************************************************
 * This utility provides communication between a host Unix system and an
 * i80960 board with a NINDY monitor.  The functions provided are:
 *
 *	o terminal emulation (the terminal at the host looks as if it is
 *	  connected directly to NINDY's serial port).
 *
 *	o object file download:
 *	  	- Intel lnk960 (COFF) and GNU gld960 (b.out) files are supported
 *		- (a copy of) the object file is stripped of all symbols and
 *			relocation information to speed up download.
 *		- 'b.out' format files are converted to coff format (NINDY only
 *			understands COFF).
 *		- NINDY is reset and told to start up its download.
 *		- the 'sx' utility is invoked to perform the xmodem transfer.
 *
 * INVOCATION:
 *	comm960 [/dev/ttyNN [objfile] ]
 *
 *		If tty is omitted it will be prompted for.
 *		If objfile is specified, it will be downloaded before
 *			beginning NINDY terminal emulation.
 *		During terminal emulation, download can be initiated by
 *			pressing the Esc key.
 *
 *	comm960 -d NINDY-file-descriptor# objfile
 *
 *		Silent download and exit.  The file descriptor must already be
 *		open and properly initialized for communication with NINDY;
 *		hence this mode can only be used by a parent process exec'ing
 *		this utility.  It is intended to support the gdb960 symbolic
 *		debugger.
 *
 *		[ I would have preferred to take the tty name (and open it here)
 *		instead of the file descriptor number.  But the USG/386 system
 *		I tried it on hung when 2 processes opened the same tty. ]
 *
 * IMPORTANT:
 *	Looks for 'sx' utility in $G960BIN, $G960BASE/bin, and
 *		DEFAULT_BASE/bin/HOST/bin
 *	where
 *		DEFAULT_BASE is defined in env.h, and
 *		HOST must be defined on the compiler invocation line.
 *
 * COMPILING UNDER USG (SystemV) UNIX:
 *	Make sure the symbol 'USG' is defined.
 *
 *****************************************************************************/

#include <stdio.h>
#include <sys/ioctl.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <ctype.h>
#include <setjmp.h>
#include "wait.h"
#include "demux.h"
#include "ttycntl.h"
#include "coff.h"
#include "env.h"

char *getenv();
char *malloc();
/* The following are defined differently in coff.h & b.out.h -- and we
 * don't need either version.  #undef them just to let this compile.
 */
#undef		N_ABS
#undef		n_name

#include "b.out.h"

#ifdef USG
#	include "sysv.h"

#	define SIGCHLD	SIGCLD

#	include <unistd.h>
#	define L_SET	SEEK_SET
#	define L_INCR	SEEK_CUR

#else	/* BSD */
#	include "string.h"

#endif

char *concat();
char *get_reply();
char *getenv();
char *mycopy();
char *xmalloc();

/******************************************************************************
 * GDB-style commands sent to NINDY.
 *	See the NINDY or gdb960 source for a full description of commands.
 ******************************************************************************/
#define GDB_EXIT	"\020E"	/* Get out of GDB mode, give prompt	*/
#define GDB_RESET	"\020X"	/* Reset NINDY silently			*/
#define GDB_DOWNLOAD	"\020D"	/* Enter GDB mode, start up download	*/

/******************************************************************************
 * Invocation
 ******************************************************************************/
char *	progname;		/* Name by which we were invoked	*/
int	download_only = 0;	/* 1=> download and exit ('-d' specified)
				 * 0=> terminal emulation
				 */

/******************************************************************************
 * Talking to NINDY
 ******************************************************************************/
char *	ttyname = NULL;	/* "/dev/ttyNN" via which we talk to NINDY	*/
int	nindy_fd = -1;	/* File descriptor for NINDY I/O		*/

/******************************************************************************
 * Error recovery
 ******************************************************************************/
jmp_buf	ioerr;		/* For recovery from I/O errors during file processing*/

/******************************************************************************
 * Terminal control
 *	Note that we save/restore the terminal status flags because the USG
 *	version of DEMUX_READ uses non-blocking reads (kludge).
 ******************************************************************************/
TTY_STRUCT orig_term;	/* Terminal attributes when we were invoked.	*/
int orig_term_flags;	/* Terminal status flags (see fcntl(2) when invoked */
char is_nindyterm = 0;	/* 1 => stdout is configured for output from NINDY.
			 * 0 => stdout has its original configuration.
			 */

/* All code #ifdef'd with FLASH_KLUDGE supports download to flash memory
 * in versions of NINDY up to and including X1.04.  It is necessary so
 * people with older versions of NINDY can load Flash memory with updates
 * of NINDY.  It should be be removed once the field has a (newer version of)
 * NINDY smart enough to decide for itself whether or not it is downloading
 * into flash memory.
 *
 * To summarize usage of comm960 with different versions of NINDY:
 *
 * prior to X1.04:
 * ----- -- ------
 * - NINDY does not contain "GDB interface" commands
 * - not possible to download programs to RAM with comm960, or to run gdb960.
 * - Load new NINDY image to flash with
 *	(1) comm960 -f /dev/ttyXX
 *	(2) at NINDY prmompt ("=>") erase flash with "ef" command
 *	(3) Press <ESC> key; at prompt enter name of file containing new
 *		NINDY image.
 * X1.04:
 * ------
 * - NINDY contains GDB interface, but NINDY "download" command only supports
 *	RAM.
 * - can download programs to RAM with normal comm960 command, or load and run
 *	from gdb960 (see man pages).
 * - Load new NINDY image (or any other program) to flash as for older versions
 *
 * after X1.04
 * ----- -----
 * - We'll see.  NINDY *should* be fixed such that the comm960 "-f" switch is
 *	obsolete, comm960 can download to RAM and flash, gdb960 can run.
 *	
 */
#define FLASH_KLUDGE 1

#ifdef FLASH_KLUDGE
int toflash = 0;	/* only TRUE when "-f" switch specified */
#endif

void sig_handler();
void ctrl_c();
void ctrl_z();
void cont();

main( argc, argv )
    int argc;
    char *argv[];
{
	int i;
	char *filename = NULL;	/* Name of file to download */
	char *p;


	if ( (argc > 1 ) && !strcmp(argv[1],"-v960") ){
		extern char comm960_ver[];
		puts( comm960_ver );
		exit(0);
	}

	progname = argv[0];	/* For error reporting */

	/* Save terminal attibutes (restored by 'quit()').
	 */
	orig_term_flags = fcntl( 0, F_GETFL, 0 );
	ioctl( 0, TIOCGETP, &orig_term );

	/* Set up signal handlers
	 */
	for ( i = 1; i < NSIG; i++ ){
		signal( i, sig_handler );
	}
	signal( SIGCHLD, SIG_DFL );	/* We'll 'wait' for this one, thanks */
#ifdef SIGTSTP
	signal( SIGTSTP, ctrl_z );
	signal( SIGCONT, cont );
#endif


	/* Parse arguments
	 */
	if ( argc > 4 ){
		usage();
		quit(1);
	}

	for ( i = 1; i < argc; i++ ){
		if ( !strcmp(argv[i],"-d") ){
			download_only = 1;
#ifdef FLASH_KLUDGE
		} else if ( !strcmp(argv[i],"-f") ){
			toflash = 1;
#endif
		} else if ( argv[i][0] == '-' ){/* -h or unknown switch */
			usage();
			quit(1);
		} else if ( ttyname == NULL ){
			ttyname = argv[i];	/* 1st non-switch is tty name */
		} else {
			filename = argv[i];	/* 2nd non-switch is obj file */
		}
	}

	if ( download_only ){
		if ( (ttyname==NULL) || (filename==NULL) ){
			err("Must specify file descriptor & object file name");
			quit(1);
		}

		/* ttyname in this case must be the number of an open file
		 * descriptor.  Convert it to binary and confirm it's valid.
		 */
		nindy_fd = 0;
		for ( p = ttyname; isdigit(*p); p++ ){
			nindy_fd = (10 * nindy_fd) +  (*p - '0');
		}
		if ( (*p != '\0') || (fcntl(nindy_fd,F_GETFD,0) == -1) ){
			err( "Invalid file descriptor" );
			quit(1);
		}
		TTY_FLUSH( nindy_fd );
		if ( !download(filename) ){
			quit(1);
		}
	} else {	/* Interactive */
		signal( SIGINT, ctrl_c );
		get_tty();
		TTY_FLUSH( nindy_fd );
		if ( filename ){
			download( filename );
		}
		emul_term();
	}

	close( nindy_fd );
	quit(0);
}

		/*************************************
		 *                                   *
		 *  OPEN A CONNECTION TO THE i80960  *
		 *                                   *
		 *************************************/

/******************************************************************************
 * connect:
 *	Open the specified tty.  Put descriptor in a global.
 *	Return 1 on success, 0 on failure.
 *
 ******************************************************************************/
int
connect( name )
    char *name;		/* "/dev/ttyXX" to be opened */
{
	TTY_STRUCT tty;

	nindy_fd = open(name, O_RDWR);
	if ( nindy_fd < 0 ){
		return 0;
	}

	ioctl( nindy_fd, TIOCGETP, &tty );
	TTY_REMOTE( tty );
	ioctl( nindy_fd, TIOCSETP, &tty );
	ttyname = mycopy( name );
	return 1;
}

/******************************************************************************
 * get_tty::
 *	If the global 'ttyname' is set, try to connect to that tty for comm
 *	with NINDY.  Otherwise prompt for a ttyname and try to open it.
 *	Either way, continue to prompt until a tty gets opened.
 *
 *	We'll accept a full path ('/dev/ttyNN'), or 'ttyNN', or 'NN'.
 *
 ******************************************************************************/
get_tty()
{
	char buf[50];	/* User input buffer				*/
	char *tty;	/* Pointer to tty name to try: a full path	*/

	buf[0] = '\0';
	if ( ttyname != NULL ){
		tty = ttyname;
	} else {
		tty = buf;
	}

	while ( 1 ){
		while ( *tty == '\0' ){
			printf("Enter tty to which the i80960 is attached: ");
			fflush( stdout );
			tty = get_reply( buf, sizeof(buf)-1 );
		}

		if ( *tty == '/' ){
			if ( connect(tty) ){
				break;
			}
		} else {
			if ( connect( concat("/dev/",tty) )
			||   connect( concat("/dev/tty",tty) ) ) {
				break;
			}
		}
		printf( "Can't open tty\n" );
		tty = "";
	}
	printf( "Connected to %s\n", ttyname );
}

		/************************
		 *                      *
		 *  TERMINAL EMULATION  *
		 *                      *
		 ************************/


#define ESC	'\033'
#define	BSIZE	1000


/******************************************************************************
 * host_term::
 *	Restore attributes terminal had when we were invoked.
 *
 ******************************************************************************/
host_term()
{
	if ( is_nindyterm ){
		ioctl( 0, TIOCSETP, &orig_term );
		fcntl( 0, F_SETFL, orig_term_flags );
		is_nindyterm = 0;
	}
}


/******************************************************************************
 * nindy_term::
 *	Set up terminal attributes for acting as terminal emulator for NINDY.
 *
 ******************************************************************************/
nindy_term()
{
	TTY_STRUCT tty;

	if ( !is_nindyterm ){
		tty = orig_term;
		TTY_NINDYTERM( tty );
		ioctl( 0, TIOCSETP, &tty );
		is_nindyterm = 1;
	}
}


#ifdef SIGTSTP
/******************************************************************************
 * ctrl_z:
 *	^Z interrupt handler: (temporarily) restore normal terminal attributes.
 *
 ******************************************************************************/
void
ctrl_z()
{
	host_term();
	signal( SIGTSTP, SIG_DFL );	/* Restore default ^Z handler */
	kill( 0, SIGTSTP );
}


/******************************************************************************
 * cont:
 *	Handle return from ^Z interrupt: back to NINDY terminal attributes.
 *
 ******************************************************************************/
void
cont()
{
	nindy_term();
	signal( SIGTSTP, ctrl_z );
}
#endif	/* SIGTSTP */


/******************************************************************************
 * ctrl_c:
 *	^C interrupt handler for interactive use: the normal way to exit from
 *	interactive use.
 *
 ******************************************************************************/
void
ctrl_c()
{
	quit( 0 );
}


/******************************************************************************
 * sig_handler:
 *	Handle most signals: clean up and exit.
 *
 ******************************************************************************/
void
sig_handler( signum )
    int signum;
{
	char msg[100];

	sprintf( msg, "Unexpected signal #%d, quitting", signum );
	err( msg );
	quit( 1 );
}


/******************************************************************************
 * emul_term:
 *	Perform terminal emulation for NINDY.  Set up terminal with a approriate
 *	attributes.  Pass output from NINDY through to display.  Pass input
 *	from keyboard on to NINDY, except for ESC which initiates a download.
 *
 ******************************************************************************/
emul_term()
{
	DEMUX_DECL;	/* OS-dependent data structure for MUX'd file
			 *	descriptors (stdin and NINDY)
			 */
	int n;		/* # of characters of input in system buffer	*/
	char buf[BSIZE];/* Input from NINDY or from keyboard		*/
	int i;		/* # of characters preceding ESC in buf		*/
	char *fn;	/* -> name of file to download			*/


	/* Give user instructions
	 */
	puts( "\n\tPress your interrupt key to exit." );
	puts( "\tPress <Esc> to download a program." );

	nindy_term();		/* Set terminal attributes for NINDY	*/

#ifdef FLASH_KLUDGE
	if ( toflash ){
		/* Can't use GDB interface commands */
		write( nindy_fd, "\r", 1 );
	} else
#endif
	putpkt( GDB_EXIT );	/* Get out of GDB mode, just in case;
				 * also gets us a NINDY prompt.
				 */

	while ( 1 ){		/* Until ^C from user	*/

		/* Wait for input from either keyboard or NINDY
		 */
		DEMUX_WAIT( nindy_fd );

		/* Send input from NINDY to stdout
		 */
		n = DEMUX_READ( nindy_fd, buf, sizeof(buf) );
		if ( n > 0 ){
			write( 1, buf, n );
		}

		/* Send input from stdin to NINDY
		 */
		n = DEMUX_READ( 0, buf, sizeof(buf) );
		if ( n > 0 ){
			/* Write out any characters in buffer preceding ESC */
			i = non_esc( buf, n );
			if ( i > 0 ){
				write( nindy_fd, buf, i );
			}

			/* Perform download if there was an ESC in the buffer */
			if ( i != n ){
				host_term();
				fn = "";
				printf("\nFile to download?: ");
				fflush( stdout );
				fn = get_reply( buf, sizeof(buf)-1 );
				if ( *fn != '\0' ){
					download( fn );
				}
				nindy_term();
				/* Get NINDY out of GDB mode, get prompt */
#ifdef FLASH_KLUDGE
				if ( toflash ){
					/* Can't use GDB interface commands */
					write( nindy_fd, "\r", 1 );
				} else
#endif
				putpkt( GDB_EXIT );
			}
		}
	}
}


/******************************************************************************
 * non_esc:
 *	Return number of characters in buffer that precede the first ESC in the
 *	buffer (or total number if there is no ESC).
 *
 ******************************************************************************/
int
non_esc( buf, n )
    char *buf;		/* Character buffer; NOT '\0'-terminated */
    int n;		/* Number of characters in buffer */
{
	int i;

	for ( i = 0; (i < n) && (buf[i] != ESC); i++ ){
		;
	}
	return i;
}

		/***********************
		 *                     *
		 *  DOWNLOAD TO NINDY  *
		 *                     *
		 ***********************/

/******************************************************************************
 * tohex:
 *	Convert a hex digit from binary to ASCII.  Return the result.
 *
 ******************************************************************************/
int
tohex( n )
    int n;	/* Number in range 0-15 */
{
	return  (n < 10) ? '0'+n : 'a'+n-10;
}


/******************************************************************************
 * putpkt:
 *	Checksum and send a gdb command to NINDY, and wait for positive
 *	acknowledgement.
 *
 ******************************************************************************/
putpkt( cmd )
    char *cmd;	/* Command to be sent */
{
	char csum;	/* Message checksum			*/
	char buf[500];	/* Command with checksum appended	*/
	char errmsg[80];/* Format error message here		*/
	char *p;	/* Pointer into buf			*/
	char ack;	/* Response received from NINDY		*/

	/* Copy the command into buffer, appending checksum.
	 */
	for ( csum = 0, p= buf; *cmd; ){
		csum += *cmd;
		*p++ = *cmd++;
	}
	*p++ = '#';
	*p++ = tohex( (csum >> 4) & 0xf );
	*p++ = tohex( csum & 0xf );

	/* Send checksummed message over and over until we get a positive ack
	 */
	write( nindy_fd, buf, p-buf );
	do {
		read( nindy_fd, &ack, 1 );
		if ( ack == '-' ){
			err( "Remote NAK, resending" );
			write( nindy_fd, buf, p-buf );
		} else if ( ack != '+' ){
			sprintf( errmsg,"Bad ACK, ignored: <%c>", ack );
			err( errmsg );
		}
	} while ( ack != '+' );
}



/******************************************************************************
 * concat_path:
 *	Concatenates a component of a path onto a base, making sure that
 *	there is one (and only one) "/" between them.  Assumes the buffer
 *	containing the base has been allocated large enough to hold the
 *	concatenated value.
 *****************************************************************************/
concat_path( base, next )
    char *base;	/* Base of path */
    char *next;	/* Next component of path */
{
	if ( next == NULL ){
		return;
	}
	if ( next[0] == '/' ){
		next++;
	}
	if ( base[strlen(base)-1] != '/' ){
		strcat( base, "/" );
	}
	strcat( base, next );
}


/******************************************************************************
 * exists:
 *	Creates a full pathname by concatenating up to three name components
 *	onto a specified base name; optionally looks up the base name as a
 *	runtime environment variable;  and checks to see if the file or
 *	directory specified by the pathname actually exists.
 *
 *	Returns:  the full pathname if it exists, NULL otherwise.
 *		(returned pathname is in malloc'd memory and must be freed
 *		by caller).
 *****************************************************************************/
char *
exists( base, c1, c2, c3, env )
    char *base;		/* Base directory of path */
    char *c1, *c2, *c3;	/* Components (subdirectories and/or file name) to be
			 *	appended onto the base directory name.  One or
			 *	more may be omitted by passing NULL pointers.
			 */
    int env;		/* If 1, '*base' is the name of an environment variable
			 *	to be examined for the base directory name;
			 *	otherwise, '*base' is the actual base directory
			 *	name.
			 */
{
	struct stat buf;	/* For call to 'stat' -- never examined */
	char *path;		/* Pointer to full pathname (malloc'd memory) */
	int len;		/* Length of full pathname (incl. terminator) */

	if ( env ){
		base = getenv( base );
		if ( base == NULL ){
			return NULL;
		}
	}

	len = strlen(base) + 4;
			/* +4 for terminator and "/" before each component */
	if ( c1 != NULL ){
		len += strlen(c1);
	}
	if ( c2 != NULL ){
		len += strlen(c2);
	}
	if ( c3 != NULL ){
		len += strlen(c3);
	}
	path = malloc( len );

	strcpy( path, base );
	concat_path( path, c1 );
	concat_path( path, c2 );
	concat_path( path, c3 );

	if ( stat(path,&buf) != 0 ){
		free( path );
		path = NULL;
	}
	return path;
}


/******************************************************************************
 * download:
 *	Strip/convert specified object file (see comments in file header).
 *	Start up NINDY downloader.  Invoke 'sx' to download from host end.
 *	Return 1 on success, 0 on failure.
 *
 ******************************************************************************/

#define XMODEM	"sx"	/* Name of xmodem transfer utility	*/

int
download( fn )
    char *fn;
{
	extern char *mktemp();
	static char template[] = "/tmp/commXXXXXX";

	char *p;	/* Pointer to full pathname of sx utility	*/
	char *tempfile;	/* Stripped copy of object file			*/
	int success;	/* Return value					*/
	int infd = -1;	/* Descriptor of object file to be downloaded	*/
	int outfd = -1;	/* Descriptor on which to write stripped object file */
	char ack;	/* Character received from NINDY after reset	*/
	int pid;	/* Process ID of xmodem transfer utility	*/
	WAITTYPE w;	/* xmodem transfer completion status		*/
	char msg[200];	/* Build error message here			*/


	/* Make sure the xmodem utility is findable.  This must be done before
	 * we start up the NINDY end of the download (NINDY will hang if we
	 * don't complete the download).
	 */
	if ( ((p = exists("G960BIN",XMODEM,NULL,NULL,1)) == NULL)
	  && ((p = exists("G960BASE","bin",XMODEM, NULL,1)) == NULL)
	  && ((p = exists(DEFAULT_BASE,HOST,"bin",XMODEM,0)) == NULL)
	){
		sprintf(msg, "Can't find '%s' download utility\n",XMODEM);
		strcat(msg, "Check G960BIN & G960BASE environment variables");
		err( msg );
		return 0;
	}

	if ( setjmp(ioerr) != 0 ){
		/* Come here on I/O error
		 */
		success = 0;
	} else {
		infd = xopen( fn, O_RDONLY );
		tempfile = mktemp( template );
		outfd = xopen( tempfile, O_WRONLY|O_CREAT|O_TRUNC );

		printf( "Downloading %s\n", fn );

		if ( (success = strip(infd,outfd)) != 0 ){

			success = 0;	/* assume failure */

#ifdef FLASH_KLUDGE
			if ( !toflash ){
			/* Reset NINDY,  wait until "reset-complete" ack,
			 * and start up the NINDY end of the download.
			 */
				putpkt( GDB_RESET );
				do {
					read( nindy_fd, &ack, 1 );
				} while ( ack != '+' );
				putpkt( GDB_DOWNLOAD );
			} else {
				/* Can't use GDB interface commands */
			 	int i;

				/* End of "df" prompt prior X1.04 */
				static char a[] = "receiving file\n\r";
				int alen = strlen(a);

				/* End of "df" prompt in X1.04 */
				static char b[] = "Receiving file";
				int blen = strlen(b);

				write( nindy_fd, "rs\r", 3 );  /* RESET NINDY */
				do {
					/* Wait for end of power-up prompt */
					read( nindy_fd, msg, 1 );
				} while (msg[0] != '>');

				write( nindy_fd, "df\r", 3 ); /* LOAD FLASH */
	 
				i=0;
				while (1){
					read( nindy_fd, &msg[i], 1 );
					msg[++i] = '\0';

					if ( (i >= alen)
					&& !strcmp(&msg[i-alen],a) ){
						/* End of prompt, preX1.04 */
						break;
					}

					if ( (i >= blen )
					&&   !strcmp(&msg[i-blen],b) ){
						/* End of prompt, X1.04 */
						break;
					}

					if  ( msg[i-1] == '\r' ){
						/* Start reading another
						 * prompt line
						 */
						i =0;
					}
				}
			}
#else
			/* Reset NINDY,  wait until "reset-complete" ack,
			 * and start up the NINDY end of the download.
			 */
			putpkt( GDB_RESET );
			do {
				read( nindy_fd, &ack, 1 );
			} while ( ack != '+' );
			putpkt( GDB_DOWNLOAD );
#endif

			/* Invoke x-modem transfer, a separate process.  DON'T
			 * use system() to do this -- on sys V, the redirection
			 * of stdin/stdout causes the nindy tty to lose all
			 * the transmission parameters we've set up.
			 */
			pid = fork();
			if ( pid == -1 ){
				perr( "Can't fork process:" );

			} else if ( pid == 0 ){		/* CHILD */
				dup2( nindy_fd, 0 );	/* Redirect stdin */
				dup2( nindy_fd, 1 );	/* Redirect stout */
				execlp( XMODEM, XMODEM, tempfile, (char*) 0 );
				/* Don't get here unless execlp fails */
				perr( concat("Can't exec ",XMODEM) );
				exit( 1 );

			} else {			/* PARENT */
				if ( wait(&w) == -1 ){
					perr( "Wait failed" );
				} else if (WIFEXITED(w) && (WRETCODE(w) == 0)){
					success = 1;
				}
			}
		}
	}
	if ( infd != -1 ){
		close( infd );
	}
	if ( outfd != -1 ){
		close( outfd );
		unlink( tempfile );
	}
	return success;
}

	/************************************
	 *                                  *
	 *  PREPARE A FILE FOR DOWNLOADING  *
	 *                                  *
	 ************************************/


/* PUT_LITTLE_END
 *	takes a number in host byte order and places it in a
 *	different location in little-endian (i80960) order.
 *
 * MAKE_LITTLE_END
 *	converts a number from host byte order to little-endian (i80960)
 *	order in place.
 */
#define PUT_LITTLE_END(n,w)	put_little_end( (char*)&w, (long) n, sizeof(w) )
#define MAKE_LITTLE_END(n)	put_little_end( (char*)&n, (long) n, sizeof(n) )


/******************************************************************************
 * put_little_end:
 *	Convert a number from host byte order to little-endian (i80960) byte
 *	order.
 *
 ******************************************************************************/
put_little_end( buf, value, n )
    char *buf;		/* Put output here		*/
    long value;		/* The integer to be converted	*/
    int	n;		/* Number of bytes to output (# of significant low-order
			 *	bytes in 'value').
			 */
{
	while ( n-- ){
		*buf++ = value;
		value >>= 8;
	}
}

/******************************************************************************
 * make_scn:
 *	Write a COFF section header to a stripped (no symbol info or relocation
 *	information) file.  Fields in the header are written in i80960
 *	(little-endian) byte order.
 *
 ******************************************************************************/
put_shdr( fd, name, paddr, size, flags, fileaddr , align )
    int fd;		/* Descriptor of file to which header is written */
    char *name;		/* Section name					*/
    long paddr;		/* Section runtime physical address		*/
    long size;		/* Size of section				*/
    long flags;		/* Section flags				*/
    long fileaddr;	/* Address of section binary in output file	*/
    long align;		/* Section alignment				*/
    /* NOTE THAT ALL OTHER FIELDS OF THE HEADER ARE ZEROED, BECAUSE
     * FILE IS ASSUMED TO BE STRIPPED!
     */
{
	struct scnhdr s;

	bzero( &s, sizeof(s) );
	strcpy( s.s_name, name );
	PUT_LITTLE_END( paddr, s.s_paddr );
	s.s_vaddr = s.s_paddr;
	PUT_LITTLE_END( size, s.s_size );
	PUT_LITTLE_END( flags, s.s_flags );
	PUT_LITTLE_END( fileaddr, s.s_scnptr );
	PUT_LITTLE_END( align, s.s_align );
	xwrite( fd, &s, sizeof(s) );
}

/******************************************************************************
 * strip_bout:
 *	Strip symbol and relocation information from a 'b.out' format object
 *	file and convert it into a COFF file.  The COFF file will have 3
 *	sections:  one each for data, text, and bss.
 *
 ******************************************************************************/
strip_bout( infd, outfd )
    int infd;	/* Descriptor on which b.out file should be read.  File type
		 *	assumed already to have been validated.
		 */
    int outfd;	/* Descriptor on which to write out coff file.	*/
{
	struct exec hdr;/* Header from input file			*/
	char *buf;	/* Dynamic buffer for copying between files	*/
	int buflen;	/* Current size of buffer 'buf' (in bytes)	*/
	struct filehdr f; /* COFF file header				*/
	struct aouthdr a; /* COFF 'optional' (aout) header		*/

#	define HDRSIZE	(sizeof(f)+sizeof(a)+(3*sizeof(struct scnhdr)))
			/* Total size of COFF file headers	*/


	/* Read b.out file header.
	 */
	xseek( infd, 0 );
	xread( infd, &hdr, sizeof(hdr) );

	/* Generate COFF file header
	 */
	bzero( &f, sizeof(f) );
	PUT_LITTLE_END( I960RWMAGIC, f.f_magic );
	PUT_LITTLE_END( 3, f.f_nscns );
	PUT_LITTLE_END( sizeof(a), f.f_opthdr );
	PUT_LITTLE_END( F_RELFLG|F_EXEC|F_LNNO|F_LSYMS|F_AR32WR|F_I960BA,
								f.f_flags );
	xwrite( outfd, &f, sizeof(f) );

	/* Generate COFF aout header
	 */
	bzero( &a, sizeof(a) );
	PUT_LITTLE_END( hdr.a_entry, a.entry );
	PUT_LITTLE_END( BMAGIC, a.magic );
	PUT_LITTLE_END( hdr.a_text, a.tsize );
	PUT_LITTLE_END( hdr.a_data, a.dsize );
	PUT_LITTLE_END( hdr.a_bss, a.bsize );
	PUT_LITTLE_END( hdr.a_entry, a.entry );
	PUT_LITTLE_END( hdr.a_tload, a.text_start );
	PUT_LITTLE_END( hdr.a_dload, a.data_start );
	xwrite( outfd, &a, sizeof(a) );

	/* Generate section headers
	 */
	put_shdr( outfd, ".text", hdr.a_tload, hdr.a_text, STYP_TEXT,
							HDRSIZE, 0 );
	put_shdr( outfd, ".data", hdr.a_dload, hdr.a_data, STYP_DATA,
							HDRSIZE+hdr.a_text, 0 );
	put_shdr( outfd, ".bss",  hdr.a_dload+hdr.a_data, hdr.a_bss, STYP_BSS,
							0, 0);

	/* Write text and data
	 */
	buflen = hdr.a_text + hdr.a_data;
	buf = xmalloc( buflen );
	xseek( infd, N_TXTOFF(hdr) );
	xread( infd, buf, buflen );
	xwrite( outfd, buf, buflen );
	free( buf );
}


/******************************************************************************
 * strip_coff:
 *	Strip symbol and relocation information from a COFF file.
 *
 ******************************************************************************/
strip_coff( infd, outfd )
    int infd;	/* Descriptor on which COFF file should be read.  File type
		 *	assumed already to have been validated.
		 */
    int outfd;	/* Descriptor on which to write out stripped file.	*/
{
	char *buf;	/* Dynamic buffer for copying from infd to outf */
	struct filehdr f;	/* Header of output file		*/
	struct aouthdr a;	/* "Optional" a.out header of output file */
	struct scnhdr *shdrs;	/* Array of section headers from infd */
	int shdr_size;		/* Size of shdrs array, in bytes	*/
	struct scnhdr *sP;	/* Pointer to an entry in shdrs array	*/
	int nsecs;		/* Number of sections (& sect. hdrs) in files */
	int pos;		/* Address in 'outfd' to which next section
				 *	will be written.
				 */
	int sec_size;		/* Size of next section to be copied	*/
	int i;			/* Loop counter				*/


	xseek( infd, 0 );

	/* Read file header, strip it, put it into i80960 byte order, write
	 * it out.
	 */
	xread( infd, &f, sizeof(f) );
	nsecs = f.f_nscns;
	MAKE_LITTLE_END( f.f_magic );
	MAKE_LITTLE_END( f.f_nscns );
	MAKE_LITTLE_END( f.f_timdat );
	f.f_nsyms = f.f_symptr = 0;
	MAKE_LITTLE_END( f.f_opthdr );
	f.f_flags |= F_RELFLG | F_LNNO | F_LSYMS;
	MAKE_LITTLE_END( f.f_flags );
	xwrite( outfd, &f, sizeof(f) );

	/* If there's an "optional" aout header:  read it, put into i80960
	 * byte order, write it out.
	 */
	if ( f.f_opthdr ){
		xread( infd, &a, sizeof(a) );
		MAKE_LITTLE_END( a.magic );
		MAKE_LITTLE_END( a.vstamp );
		MAKE_LITTLE_END( a.tsize );
		MAKE_LITTLE_END( a.dsize );
		MAKE_LITTLE_END( a.bsize );
		MAKE_LITTLE_END( a.entry );
		MAKE_LITTLE_END( a.text_start );
		MAKE_LITTLE_END( a.data_start );
		a.tagentries = 0;	/* ASSUMES THERE NEVER ARE ANY */
		xwrite( outfd, &a, sizeof(a) );
	}

	/* Read in ALL the section headers.
	 */
	shdr_size = nsecs * sizeof(struct scnhdr);
	shdrs = (struct scnhdr *) xmalloc( shdr_size ); 
	xread( infd, shdrs, shdr_size );

	/* Write out section headers, one at a time.  For robustness, we will
	 * NOT assume that section binary immediately follows the headers, nor
	 * will we assume that the section binaries come in the same order as
	 * the corresponding section headers -- although we'll make both
	 * assumptions true in the output file.  Therefore, we update the
	 * file pointer of each section via the variable 'pos', which is
	 * initialized to point just after the headers.
	 */
	pos = lseek(outfd,0,L_INCR) + shdr_size;
	for ( sP= shdrs, i = 0; i < nsecs; i++, sP++ ){
		if ( sP->s_scnptr == 0 ){
			/* This section has no binary in the file.
			 */
			put_shdr( outfd, sP->s_name, sP->s_paddr, sP->s_size,
						sP->s_flags, 0, sP->s_align );
		} else {
			/* This section DOES have binary: update 'pos'.
			 */
			put_shdr( outfd, sP->s_name, sP->s_paddr, sP->s_size,
						sP->s_flags, pos, sP->s_align );
			pos += sP->s_size;
		}
	}

	/* Copy binary for each section.
	 */
	for ( i = 0; i < nsecs; i++ ){
		if ( shdrs[i].s_scnptr != 0 ){
			sec_size = shdrs[i].s_size;
			buf = xmalloc( sec_size );
			xseek( infd, shdrs[i].s_scnptr );
			xread( infd, buf, sec_size );
			xwrite( outfd, buf, sec_size );
			free( buf );
		}
	}
	free( shdrs );
}

/******************************************************************************
 * strip:
 *	Distinguish COFF from b.out files; call the appropriate routine to
 *	strip the symbols and relocation info from an object file.
 *	Return 1 on success, 0 on failure.
 ******************************************************************************/
strip( infd, outfd )
    int infd;	/* Descriptor of input object file	*/
    int outfd;	/* Strip 'infd' and write results to this descriptor */
{
	struct filehdr *coff_hdr;
	struct exec *bout_hdr;
	char buf[10];		/* The first 10 bytes of the input file */

	xseek( infd, 0 );
	xread( infd, buf, sizeof(buf) );

	coff_hdr = (struct filehdr *) buf;
	bout_hdr = (struct exec *) buf;

	if ( (coff_hdr->f_magic == I960ROMAGIC)
	||   (coff_hdr->f_magic == I960RWMAGIC) ){
		strip_coff( infd, outfd );

	} else if ( bout_hdr->a_magic == BMAGIC ){
		strip_bout( infd, outfd );

	} else {
		err( "Unknown input file format" );
		return 0;
	}

	return 1;
}

		/*************************
		 *                       *
		 *  LOW-LEVEL UTILITIES  *
		 *                       *
		 *************************/

/******************************************************************************
 * my_copy:
 *	Allocate space for a copy of a text string and copy it.
 *	Return a pointer to the new copy.
 *
 ******************************************************************************/
char *
mycopy( textP )
    char *textP;	/* Test to be copied	*/
{
	char *copyP;	/* Address of copy	*/

	copyP = xmalloc( strlen(textP) + 1 );
	strcpy( copyP, textP );
	return copyP;
}


/******************************************************************************
 * xmalloc:
 *	Perform memory allocation with error checking.  On error, issue message
 *	and die.
 *
 ******************************************************************************/
char *
xmalloc( len )
    int len;	/* Allocate this many bytes	*/
{
	char *p;	/* Pointer to allocated memory */
	extern char *malloc();

	p = malloc( len );
	if ( p == NULL ){
		perr( "malloc failed" );
		quit( 1 );	/* Something is dreadfully wrong */
	}
	return p;
}


/******************************************************************************
 * get_reply:
 *	Get a user keyboard response.  Strip off leading and trailing
 *	whitespace.  Return a pointer to the result.
 *
 ******************************************************************************/
char *
get_reply( bufP, len )
    char *bufP;		/* Buffer to use for user input	*/
    int len;		/* Length of buffer		*/
{
	char *p;	/* Pointer into the buffer	*/

	fgets( bufP, len, stdin );

	/* Strip leading whitespace */
	for ( ; isspace(*bufP); bufP++ ){
		;
	}
	if ( *bufP != '\0' ){
		/* Strip trailing whitespace */
		for ( p= bufP + strlen(bufP) - 1; isspace(*p); p-- ){
			;
		}
		*(p+1) = '\0';
	}

	return bufP;
}


/******************************************************************************
 * usage:
 *	Give instructions to user.
 *
 ******************************************************************************/
usage()
{
	putchar( '\n' );
#ifdef FLASH_KLUDGE
	printf( "For interactive usage:\t%s [-f] [/dev/ttyNN [objfile] ]\n\n",
								progname );
	puts( "\t-f for download to flash memory" );
#else
	printf( "For interactive usage:\t%s [/dev/ttyNN [objfile] ]\n\n",
								progname );
#endif
	puts( "\tIf tty is not specified, you will be prompted." );
	puts( "\tIf 'objfile' is specified, it will be downloaded." );
	puts( "" );
	printf( "For silent download:\t%s -d file-descriptor# objfile\n\n",
								progname );
	puts( "\tAll args MUST be specified." );
}


/******************************************************************************
 * concat:
 *	Concatenate 2 strings.  Return a pointer to the result.
 *
 * WARNING:  concatenated string is clobbered by next call to this routine.
 *
 ******************************************************************************/
char *
concat( s1, s2 )
    char *s1;
    char *s2;
{
	static char *p = NULL;

	if ( p ){
		free(p);
	}
	p = xmalloc( strlen(s1) + strlen(s2) + 1 );
	strcpy( p, s1 );
	strcat( p, s2 );
	return p;
}


/******************************************************************************
 * err:
 *	Print an error message, including the name of this program.
 *
 ******************************************************************************/
err( msg )
    char *msg;	/* Text of message, trailing '\n' unnecessary	*/
{
	char was_nindyterm;

	was_nindyterm = is_nindyterm;
	if ( was_nindyterm ){
		host_term();
	}

	fprintf( stderr, "%s: %s\n", progname, msg );

	if ( was_nindyterm ){
		nindy_term();
	}
}


/******************************************************************************
 * perr:
 *	Print an error message, including the name of this program.
 *	Append the most recent system error.
 *
 ******************************************************************************/
perr( msg )
    char *msg;	/* Text of message, trailing '\n' unnecessary	*/
{
	char was_nindyterm;

	was_nindyterm = is_nindyterm;
	if ( was_nindyterm ){
		host_term();
	}

	fprintf( stderr, "%s: ", progname );
	perror( msg );

	if ( was_nindyterm ){
		nindy_term();
	}
}


/******************************************************************************
 * quit:
 *	Clean up terminal and NINDY tty, exit.
 ******************************************************************************/
quit( status )
    int status;
{
	is_nindyterm = 1;	/* Just to make sure cleanup gets done */
	host_term();
	exit( status );
}

		/**********************************
		 *                                *
		 *  FILE I/O WITH ERROR-CHECKING  *
		 *                                *
		 **********************************/

/* NOTE!:	All of these routines assume that the jumpbuf 'ioerr'
 *		has been set up to restore control correctly after an error
 *		is detected
 *
 *		As of this writing, there routines are only used while
 *		stripping/converting a file for download.  The 'longjmp's
 *		result in abortion of the download.
 */


/******************************************************************************
 * xopen:
 *	Open file for I/O, with error checking.
 *	Return file descriptor.
 ******************************************************************************/
int
xopen( fn, mode )
    char *fn;	/* Name of file			*/
    int mode;	/* Mode in which to open it	*/
{
	int fd;	/* File descriptor	*/

	fd = open( fn, mode, 0777 );
	if ( fd < 0 ){
		perr( concat("Can't open file ",fn) );
		longjmp( ioerr, 1 );
	}
	return fd;
}


/******************************************************************************
 * xread:
 *	Read from file, with error checking.
 ******************************************************************************/
xread( fd, bufp, n )
    int fd;	/* File descriptor		*/
    char *bufp;	/* Pointer to input buffer	*/
    int n;	/* Size of input buffer		*/
{
	if ( read( fd, bufp, n ) != n ){
		perr( "File read failed" );
		longjmp( ioerr, 1 );
	}
}


/******************************************************************************
 * xseek:
 *	Seek to location relative to beginning of file, with error checking.
 *	Return position.
 ******************************************************************************/
int
xseek( fd, offset )
    int fd;	/* File descriptor					*/
    int offset;	/* Offset into file at which to position I/O pointer	*/
{
	int pos;

	pos = lseek( fd, offset, L_SET );

 	if ( pos == -1 ){
		perr( "Seek failed" );
		longjmp( ioerr, 1 );
	}
	return pos;
}


/******************************************************************************
 * xwrite:
 *	Write to file, with error checking.
 ******************************************************************************/
xwrite( fd, bufp, n )
    int fd;	/* File descriptor		*/
    char *bufp;	/* Pointer to output buffer	*/
    int n;	/* Size of output buffer	*/
{
	if ( write( fd, bufp, n ) != n ){
		perr( "File write failed" );
		longjmp( ioerr, 1 );
	}
}
