/* Memory-access and commands for inferior process, for GDB.
   Copyright (C)  1989 Free Software Foundation, Inc.

   MODIFIED BY CHRIS BENENATI, FOR INTEL CORPORATION, 12/88
	Except for the data cache routines, this file bears little
	resemblence to the one that the Free Software Foundation
	distributes.  A new (although similar) protocol has been specified,
	and portions of the code are entirely dependent on having an
	i80960 with a NINDY ROM monitor at the other end of the line.

GDB is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY.  No author or distributor accepts responsibility to anyone
for the consequences of using it or for whether it serves any
particular purpose or works at all, unless he says so in writing.
Refer to the GDB General Public License for full details.

Everyone is granted permission to copy, modify and redistribute GDB,
but only under the conditions described in the GDB General Public
License.  A copy of this license is supposed to have been given to you
along with GDB so you can know your rights and responsibilities.  It
should be in a file named COPYING.  Among other things, the copyright
notice and this notice must be preserved on all copies.

In other words, go ahead and share GDB, but don't try to stop
anyone else from sharing it farther.  Help stamp out software hoarding!
*/

/*****************************************************************************
 *
 * REMOTE COMMUNICATION PROTOCOL BETWEEN GDB960 AND THE NINDY ROM MONITOR.
 *
 *
 * MODES OF OPERATION
 * ----- -- ---------
 *	
 * As far as NINDY is concerned, GDB is always in one of two modes: command
 * mode or passthrough mode.
 *
 * In command mode (the default) pre-defined packets containing requests
 * are sent by GDB to NINDY.  NINDY never talks except in reponse to a request.
 *
 * Once the the user program is started, GDB enters passthrough mode, to give
 * the user program access to the terminal.  GDB remains in this mode until
 * NINDY indicates that the program has stopped.
 *
 *
 * PASSTHROUGH MODE
 * ----------- ----
 *
 * GDB writes all input received from the keyboard directly to NINDY, and writes
 * all characters received from NINDY directly to the monitor.
 *
 * Keyboard input is neither buffered nor echoed to the monitor.
 *
 * GDB remains in passthrough mode until NINDY sends a single ^P character,
 * to indicate that the user process has stopped.
 *
 * Note:
 *	GDB assumes NINDY performs a 'flushreg' when the user program stops.
 *
 *
 * COMMAND MODE
 * ------- ----
 *
 * All info (except for message ack and nak) is transferred between gdb
 * and the remote processor in messages of the following format:
 *
 *		<info>#<checksum>
 *
 * where 
 *	#	is a literal character
 *
 *	<info>	ASCII information;  all numeric information is in the
 *		form of hex digits ('0'-'9' and lowercase 'a'-'f').
 *
 *	<checksum>
 *		is a pair of ASCII hex digits representing an 8-bit
 *		checksum formed by adding together each of the
 *		characters in <info>.
 *
 * The receiver of a message always sends a single character to the sender
 * to indicate that the checksum was good ('+') or bad ('-');  the sender
 * re-transmits the entire message over until a '+' is received.
 *
 * In response to a command NINDY always sends back either data or
 * a result code of the form "Xnn", where "nn" are hex digits and "X00"
 * means no errors.  (Exceptions: the "s" and "c" commands don't respond.)
 *
 * SEE THE HEADER OF THE FILE "gdb.c" IN THE NINDY MONITOR SOURCE CODE FOR A
 * FULL DESCRIPTION OF LEGAL COMMANDS.
 *
 * SEE THE FILE "fault.c" IN THE NINDY MONITOR SOURCE CODE FOR A LIST
 * OF STOP CODES.
 *
 ******************************************************************************/

#include <stdio.h>
#include <signal.h>
#include <sys/types.h>
#include <sys/stat.h>

#include "defs.h"
#include "param.h"
#include "frame.h"
#include "inferior.h"

#include "wait.h"
#include <sys/ioctl.h>
#include <sys/file.h>
#include "ttycntl.h"

#ifndef I80960

	remote_error() { error( "Remote operation not implemented" );  }
	remote_wait() { remote_error(); }
	remote_resume() { remote_error(); }
	remote_fetch_word() { remote_error(); }
	remote_store_word() { remote_error(); }

#else

#include "demux.h"
#include "env.h"

extern char *malloc();
extern char *mktemp();
extern char *getenv();


#define DLE	'\020'	/* Character NINDY sends to indicate user program has
			 * halted.
			 */
#define TRUE	1
#define FALSE	0

#define	PBUFSIZ	800

static int remote_desc;		/* Descriptor for I/O to remote machine.  */
static int have_regs = 0;	/* 1 iff regs read since i960 last halted */
static int regs_changed = 0;	/* 1 iff regs were modified since last read */

static void remote_send ();
static void putpkt ();
static void getpkt ();
static void dcache_flush ();
static void remote_fetch_registers();
static void remote_store_registers();

/* Image of i960 registers
 */
static char registers[REGISTER_BYTES];

/* Copy LEN bytes of consecutive data from registers starting
 * with the REGBYTE'th byte of register data into memory at MYADDR.
 */
read_register_bytes( regbyte, myaddr, len )
    int regbyte;
    char *myaddr;
    int len;
{
	remote_fetch_registers();
	bcopy( &registers[regbyte], myaddr, len );
}

/* Copy LEN bytes of consecutive data from memory at MYADDR
 * into registers starting with the REGBYTE'th byte of register data.
 */
write_register_bytes( regbyte, myaddr, len )
    int regbyte;
    char *myaddr;
    int len;
{
	remote_fetch_registers();
	bcopy( myaddr, &registers[regbyte], len );
	regs_changed = 1;
}


/* Return the contents of register REGNO, regarding it as an integer.
 *
 * This loses when REGISTER_RAW_SIZE (regno) != sizeof (long)
 */
CORE_ADDR
read_register( regno )
    int regno;
{
	unsigned int regval;

	if ( (regno!=IP_REGNUM) && (regno!=FP_REGNUM) && (regno!=SP_REGNUM) ){
		/* We always get the ip, fp, and sp when the processor halts,
		 * without having to read all the registers back.  DON'T fetch
		 * the other registers when retrieving them because we may be
		 * single-stepping through all the instructions that make up a
		 * C statement.
		 */
		remote_fetch_registers();
	}
	regval = *(int *) &registers[ REGISTER_BYTE(regno) ];
	byteswap( &regval, 4 );		/* i960 byte-order to host byte-order */
	return regval;
}

/* Store VALUE in the register number REGNO, regarded as an integer.
 *
 * This loses when REGISTER_RAW_SIZE (regno) != sizeof (int)
 */
void
write_register( regno, val )
    int regno, val;
{
	remote_fetch_registers();
	byteswap( &val, 4 );		/* host byte-order to i960 byte-order */
	*(int *) &registers[ REGISTER_BYTE(regno) ] = val;
	regs_changed = 1;
}

/* Restore TTY to normal operation */

static TTY_STRUCT orig_tty;	/* TTY attributes before entering passthrough */

static
restore_tty()
{
	ioctl( 0, TIOCSETP, &orig_tty );
}


/* Recover from ^Z or ^C while remote process is running */

#ifdef USG
static void (*old_ctrlc)();    /* Signal handlers before entering passthrough */
#else /* BSD */
static int (*old_ctrlc)();    /* Signal handlers before entering passthrough */
#endif

#ifdef SIGTSTP
static int (*old_ctrlz)();
#endif

static
#ifdef USG
void
#endif
cleanup()
{
	restore_tty();
	signal(SIGINT, old_ctrlc);
#ifdef SIGTSTP
	signal(SIGTSTP, old_ctrlz);
#endif
	error("\n\nYou may need to reset the 80960 and/or reload your program.\n");
}

/* Open a connection to a remote debugger.
 */

void
remote_open (name, from_tty)
    char *name;		/* "/dev/ttyXX" to be opened */
    int from_tty;
{
	TTY_STRUCT tty;

	remote_debugging = have_regs = regs_changed = 0;
	dcache_init();

	printf( "Attaching remote machine %s\n", name );

	if ( remote_desc ){
		close( remote_desc );
	}
	remote_desc = open( name, O_RDWR );
	if ( remote_desc < 0 ){
		perror_with_name (name);
	}

	ioctl( remote_desc, TIOCGETP, &tty );
	TTY_REMOTE( tty );
	ioctl( remote_desc, TIOCSETP, &tty );
	TTY_FLUSH( remote_desc );

	/* Get current processor registers.  Allow user to interrupt if we
	 * hang because there's nothing at the other end of the remote tty.
	 */
	immediate_quit++;
	remote_fetch_registers();
	immediate_quit--;

	remote_debugging = 1;
}

/******************************************************************************
 * concat_path:
 *	Concatenates a component of a path onto a base, making sure that
 *	there is one (and only one) "/" between them.  Assumes the buffer
 *	containing the base has been allocated large enough to hold the
 *	concatenated value.
 *****************************************************************************/
static
concat_path( base, next )
    char *base;	/* Base of path */
    char *next;	/* Next component of path */
{
	if ( next == NULL ){
		return;
	}
	if ( next[0] == '/' ){
		next++;
	}
	if ( base[strlen(base)-1] != '/' ){
		strcat( base, "/" );
	}
	strcat( base, next );
}


/******************************************************************************
 * exists:
 *	Creates a full pathname by concatenating up to three name components
 *	onto a specified base name; optionally looks up the base name as a
 *	runtime environment variable;  and checks to see if the file or
 *	directory specified by the pathname actually exists.
 *
 *	Returns:  the full pathname if it exists, NULL otherwise.
 *		(returned pathname is in malloc'd memory and must be freed
 *		by caller).
 *****************************************************************************/
static
char *
exists( base, c1, c2, c3, env )
    char *base;		/* Base directory of path */
    char *c1, *c2, *c3;	/* Components (subdirectories and/or file name) to be
			 *	appended onto the base directory name.  One or
			 *	more may be omitted by passing NULL pointers.
			 */
    int env;		/* If 1, '*base' is the name of an environment variable
			 *	to be examined for the base directory name;
			 *	otherwise, '*base' is the actual base directory
			 *	name.
			 */
{
	struct stat buf;	/* For call to 'stat' -- never examined */
	char *path;		/* Pointer to full pathname (malloc'd memory) */
	int len;		/* Length of full pathname (incl. terminator) */

	if ( env ){
		base = getenv( base );
		if ( base == NULL ){
			return NULL;
		}
	}

	len = strlen(base) + 4;
			/* +4 for terminator and "/" before each component */
	if ( c1 != NULL ){
		len += strlen(c1);
	}
	if ( c2 != NULL ){
		len += strlen(c2);
	}
	if ( c3 != NULL ){
		len += strlen(c3);
	}
	path = malloc( len );

	strcpy( path, base );
	concat_path( path, c1 );
	concat_path( path, c2 );
	concat_path( path, c3 );

	if ( stat(path,&buf) != 0 ){
		free( path );
		path = NULL;
	}
	return path;
}



/******************************************************************************
 * remote_load:
 *	Download an object file to the remote system by invoking the "comm960"
 *	utility.  We look for "comm960" in $G960BIN, $G960BASE/bin, and
 *	DEFAULT_BASE/bin/HOST/bin where
 *		DEFAULT_BASE is defined in env.h, and
 *		HOST must be defined on the compiler invocation line.
 ******************************************************************************/

#define COMM	"comm960"

remote_load( fn )
    char *fn;	/* Full path of file to download -- existence assumed already
		 * to have been validated.
		 */
{
	char buf[200];
	int pid;	/* Process ID of comm utility		*/
	WAITTYPE w;	/* comm utility completion status	*/
	char *p;	/* Pointer to full pathname of comm utility	*/



	if ( !remote_debugging ){
		error( "Use 'attach' command to open remote tty" );
	}

	have_regs = regs_changed = 0;
	flush_breakpoints();
	inferior_pid = 0;
	dcache_flush();

	if ( ((p = exists("G960BIN",COMM,NULL,NULL,1)) == NULL)
	  && ((p = exists("G960BASE","bin",COMM, NULL,1)) == NULL)
	  && ((p = exists(DEFAULT_BASE,HOST,"bin",COMM,0)) == NULL)
	){
		sprintf(buf, "Can't find '%s' utility: ",COMM);
		strcat(buf, "check G960BIN & G960BASE environment variables");
		error( buf );
		return 0;
	}

	/* Invoke comm utility, a separate process.  On system V (at least
	 * the 386 system I tried) 2 separate processes can't independently
	 * open the same tty:  they both get hung up.  So we just let the
	 * the child process inherit the file descriptor for NINDY.  The
	 * child passes the descriptor number along as one of the arguments
	 * to the comm utility when it is exec'd.
	 */
	pid = fork();
	if ( pid == -1 ){
		perror_with_name("Can't fork download process"); /* No return */

	} else if ( pid == 0 ){					/* CHILD */
		sprintf( buf, "%d", remote_desc );
		execlp( p, p, "-d", buf, fn, (char*) 0 );
		/* Don't get here unless execlp fails */
		sprintf( buf, "Can't exec %s", p );
		perror( buf );
		exit(1);

	} else {						/* PARENT */
		if ( wait(&w) == -1 ){
			perror_with_name("Download wait failed");/* No return */
		}
		
		if ( !WIFEXITED(w) || (WRETCODE(w) != 0) ){
			error( "Download failed" );		/* No return */
		}
	}
	printf( "\n" );
}

/* Convert hex digit A to a number.  */
static
int
fromhex( a )
    int a;
{
	if (a >= '0' && a <= '9'){
		a -= '0';
	} else if (a >= 'a' && a <= 'f'){
		a -= 'a' - 10;
	} else {
		error ("Reply contains invalid hex digit");
	}
	return (a & 0xff);
}


/* Convert two ASCII characters (hex digits) to an integer */
static
int
hextoint( p )
     char *p;	/* Pointer to 2 ASCII characters (hex digits) */
{
	return ( (fromhex(*p) << 4) | fromhex(*(p+1)) );
}

/* Convert a string of ASCII hex digits to a string of binary bytes
 */
static
void
hextobin( n, hexp, binp )
    int n;		/* Number of bytes to convert (twice this many digits)*/
    char *hexp;		/* Get hex from here		*/
    char *binp;		/* Put binary here		*/
{
	while ( n-- ){
		*binp++ = hextoint( hexp );
		hexp += 2;
	}
}


/* Convert a string of binary bytes to a string of ASCII hex digits 
 */
static
void
bintohex( n, binp, hexp )
    int n;		/* Number of bytes to convert	*/
    char *binp;		/* Get binary from here		*/
    char *hexp;		/* Place hex here		*/
{
	static char tohex[] = "0123456789abcdef";

	while ( n-- ){
		*hexp++ = tohex[ (*binp >> 4) & 0xf ];
		*hexp++ = tohex[ *binp & 0xf ];
		binp++;
	}
}


/* Return the number of characters in the buffer before the first DLE character.
 */

static
int
non_dle( buf, n )
    char *buf;		/* Character buffer; NOT '\0'-terminated */
    int n;		/* Number of characters in buffer */
{
	int i;

	for ( i = 0; i < n; i++ ){
		if ( buf[i] == DLE ){
			break;
		}
	}
	return i;
}

/* Tell the remote machine to resume.  */

int
remote_resume (step, signal)
     int step, signal;
{
	dcache_flush();
	remote_store_registers();
	have_regs = 0;
	putpkt( step ? "\020s": "\020c" );
}

/* Wait until the remote machine stops. While waiting, operate in passthrough
 * mode; i.e., pass everything NINDY sends to stdout, and everything from
 * stdin to NINDY.
 *
 * Return to caller, storing status in 'status' just as `wait' would.
 */


/* SOME SPECIFIC VALUES THAT NINDY MIGHT RETURN FOR 'stop_code'
 */
#define SINGLE_STEP	0x10	/* Stopped due to single-step trace */
#define GDB_BPT		0xff	/* Stopped due to fmark instruction, other
				 *	than user exit.
				 */

int
remote_wait( status )
    WAITTYPE *status;
{
	DEMUX_DECL;	/* OS-dependent data needed by DEMUX... macros */
	char buf[PBUFSIZ];
	int i, n;
	int stop_exit;
	int stop_code;
	TTY_STRUCT tty;
	char *p;


	WSETEXIT( (*status), 0 );

	/* OPERATE IN PASSTHROUGH MODE UNTIL NINDY SENDS A DLE CHARACTER */

	/* Save current tty attributes, set up signals to restore them.
	 */
	ioctl( 0, TIOCGETP, &orig_tty );
	old_ctrlc = signal( SIGINT, cleanup );
#ifdef SIGTSTP
	old_ctrlz = signal( SIGTSTP, cleanup );
#endif

	/* Pass input from keyboard to NINDY as it arrives.
	 * NINDY will interpret <CR> and perform echo.
	 */
	tty = orig_tty;
	TTY_NINDYTERM( tty );
	ioctl( 0, TIOCSETP, &tty );

	while ( 1 ){
		/* Go to sleep until there's something for us on either
		 * the remote port or stdin.
		 */

		DEMUX_WAIT( remote_desc );

		/* Pass input through to correct place */

		n = DEMUX_READ( 0, buf, sizeof(buf) );
		if ( n ){				/* Input on stdin */
			write( remote_desc, buf, n );
		}

		n = DEMUX_READ( remote_desc, buf, sizeof(buf) );
		if ( n ){				/* Input on remote */
			/* Write out any characters in buffer preceding DLE */
			i = non_dle( buf, n );
			if ( i > 0 ){
				write( 1, buf, i );
			}

			/* Exit loop if there was a DLE in the buffer */
			if ( i != n ){
				break;
			}
		}
	}

	signal( SIGINT, old_ctrlc );
#ifdef SIGTSTP
	signal( SIGTSTP, old_ctrlz );
#endif
	restore_tty();

	/* Find out why remote process stopped */
	strcpy( buf, "\020?" );
	remote_send( buf, FALSE );
	stop_exit = hextoint( buf );
	stop_code = hextoint( buf+2 );
	hextobin( 4, buf+4, &registers[ REGISTER_BYTE(IP_REGNUM) ] );
	hextobin( 4, buf+12, &registers[ REGISTER_BYTE(FP_REGNUM) ] );
	hextobin( 4, buf+20, &registers[ REGISTER_BYTE(SP_REGNUM) ] );

	if ( stop_exit ){			/* User program exited */
		WSETEXIT( (*status), stop_code );
	} else {				/* Fault or trace */
		switch (stop_code){
		case GDB_BPT:
		case SINGLE_STEP:
			/* Make it look like a VAX trace trap */
			stop_code = SIGTRAP;
			break;
		default:
			/* The 80960 is not running Unix, and its
			 * faults/traces do not map nicely into Unix signals.
			 * Make sure they do not get confused with Unix signals
			 * by numbering them with values higher than the highest
			 * legal Unix signal.  code in i80960-fault() will
			 * decode the value at the right time.  I hope.
			 */
			stop_code += NSIG;
			break;
		}
		WSETSTOP( (*status), stop_code );
	}
}


/* Print out text describing a "signal number" with which the i80960 halted.
 *
 * SEE THE FILE "fault.c" IN THE NINDY MONITOR SOURCE CODE FOR A LIST
 * OF STOP CODES.
 */
i80960_fault( signal )
    int signal;		/* Signal number, as returned by remote_wait() */
{
	static char unknown[] = "Unknown fault or trace";
	static char *sigmsgs[] = {
		/* FAULTS */
		"parallel fault",	/* 0x00 */
		unknown,		/* 0x01 */
		"operation fault",	/* 0x02 */
		"arithmetic fault",	/* 0x03 */
		"floating point fault",	/* 0x04 */
		"constraint fault",	/* 0x05 */
		"virtual memory fault",	/* 0x06 */
		"protection fault",	/* 0x07 */
		"machine fault",	/* 0x08 */
		"structural fault",	/* 0x09 */
		"type fault",		/* 0x0a */
		"reserved (0xb) fault",	/* 0x0b */
		"process fault",	/* 0x0c */
		"descriptor fault",	/* 0x0d */
		"event fault",		/* 0x0e */
		"reserved (0xf) fault",	/* 0x0f */

		/* TRACES */
		"single-step trace",	/* 0x10 */
		"branch trace",		/* 0x11 */
		"call trace",		/* 0x12 */
		"return trace",		/* 0x13 */
		"pre-return trace",	/* 0x14 */
		"supervisor call trace",/* 0x15 */
		"breakpoint trace",	/* 0x16 */
	};
#	define NUMMSGS ( sizeof(sigmsgs) / sizeof(sigmsgs[0]) )


	if ( signal == SIGTRAP ){
		printf( "\nUnexpected breakpoint.\n" );
		printf( "You should probably re-load your program.\n" );

	} else {

		/* remote_wait() biases the 80960 "signal number" by adding
		 * NSIG to it, so it won't get confused with any of the Unix
		 * signals elsewhere in GDB.  We need to "unbias" it before
		 * using it.
		 */
		signal -= NSIG;

		printf( "\nProgram stopped for reason #%d: %s.\n", signal,
				signal < NUMMSGS ? sigmsgs[signal] : unknown );
		printf( "Proceed at your own risk.\n\n" );
	}
	fflush (stdout);
}


/* Read the remote registers into the block REGS.  */

static
void
remote_fetch_registers()
{
	char buf[PBUFSIZ];	   /* Registers in ASCII hex */

	if ( !have_regs ){

		strcpy( buf, "\020r" );
		remote_send( buf, FALSE );
		if ( strlen(buf) < 2*REGISTER_BYTES ){
			error ("Remote reply is too short: %s", buf);
		}

		hextobin( REGISTER_BYTES, buf, registers );
		have_regs = 1;
	}
}

/* Store the remote registers from the contents of the block REGS.  */

static
void
remote_store_registers()
{
	char buf[PBUFSIZ];	   /* Registers in ASCII hex */

	if ( regs_changed ){

		strcpy( buf, "\020R" );
		bintohex( REGISTER_BYTES, registers, buf+2 );
		buf[ (2*REGISTER_BYTES)+2 ] = '\0';

		remote_send( buf, TRUE );
		regs_changed = 0;
	}
}

/* Read a word from remote address ADDR and return it.
 * This goes through the data cache.
 */
int
remote_fetch_word (addr)
     CORE_ADDR addr;
{
	return dcache_fetch (addr);
}

/* Write a word WORD into remote address ADDR.
   This goes through the data cache.  */

void
remote_store_word (addr, word)
     CORE_ADDR addr;
     int word;
{
	dcache_poke (addr, word);
}

/* Write memory data directly to the remote machine.
 * This does not inform the data cache; the data cache uses this.
 */
static
void
remote_write_bytes( memaddr, myaddr, len )
     CORE_ADDR memaddr;	/* Address in the remote memory space	*/
     char *myaddr;	/* Address of the buffer in our space	*/
     int len;		/* Number of bytes to transmit		*/
{
	char buf[PBUFSIZ];
	char *p;

	if (len > PBUFSIZ / 2 - 20){
		abort ();
	}

	sprintf( buf, "\020M%x,", memaddr );
	p = buf + strlen (buf);
	bintohex( len, myaddr, p );
	*(p+(2*len)) = '\0';

	remote_send( buf, TRUE );
}

/* Read memory data directly from the remote machine.
 * This does not use the data cache; the data cache uses this.
 */
static
void
remote_read_bytes (memaddr, myaddr, len)
     CORE_ADDR memaddr;	/* Address in the remote memory space	*/
     char *myaddr;	/* Address of the buffer in our space	*/
     int len;		/* Number of bytes to read		*/
{
	char buf[PBUFSIZ];
	int i;
	char *p;

	if (len > PBUFSIZ / 2 - 1){
		abort ();
	}

	sprintf( buf, "\020m%x,%x", memaddr, len );
	remote_send( buf, FALSE );
	if (strlen(buf) < 2*len ){
		error ("Remote reply is too short: %s", buf);
	}

	/* Reply describes registers byte by byte,
	 * each byte encoded as two hex characters.
	 */

	p = buf;
	for (i = 0; i < len; i++){
		myaddr[i] = fromhex (p[0]) * 16 + fromhex (p[1]);
		p += 2;
	}
}

/*
 * A communication packet whose contents are <data>
 * is encapsulated for transmission in the form:
 * 
 * 	<data> # CSUM1 CSUM2
 * 
 * 	CSUM1 and CSUM2 are ascii hex representation of an 8-bit 
 * 	checksum of <data>, the most significant nibble is sent first.
 * 	the hex digits 0-9,a-f are used.
 * 
 * Receiver responds with:
 * 
 * 	+	- if CSUM is correct and ready for next packet
 * 	-	- if CSUM is incorrect
 * 
 */

/* Send the command in 'buf' to the remote machine, and return the reply in
 * 'buf' (CLOBBERS 'buf'!).
 *
 * Check NINDY's response.  Call error (no return) if the
 * reponse indicates a problem with the command sent.
 */
static
void
remote_send( buf, ack_required )
    char *buf;		/* Buffer containing message sent by NINDY */
    int ack_required;	/* TRUE means the message MUST be either "X00" (no
			 *	error) or an error code "Xnn".
			 * FALSE means the message is OK as long as it doesn't
			 *	begin with "Xnn".
			 */
{
	int errnum;
	static char *errmsg[] = {
		NULL,						/* X00" */
		"Buffer overflow",				/* X01" */
		"Unknown command",				/* X02" */
		"Wrong amount of data to load register(s)",	/* X03" */
		"Missing command argument(s)",			/* X04" */
		"Odd number of digits sent to load memory",	/* X05" */
		"Unknown register name",			/* X06" */
	};
#	define NUMERRS	( sizeof(errmsg) / sizeof(errmsg[0]) )

	putpkt (buf);
	getpkt (buf);
	if ( buf[0] != 'X' ){
		if ( !ack_required ){
			return;
		}
		error( "NINDY failed to acknowledge command: <%s>", buf );
	}

	if ( !strcmp(buf,"X00") ){	/* If command completed successfully */
		return;
	}

	errnum = hextoint( &buf[1] );
	if ( errnum > NUMERRS ){
		error( "Unknown error response from NINDY: <%s>", buf );
	} else{
		error( "Error response %s from NINDY:\n\t%s",
							buf,  errmsg[errnum] );
	}
}

/* Send a packet to the remote machine, with error checking
 */
static
void
putpkt (buf)
    char *buf;	/* Packet to be sent */
{
	int i;
	char csum = 0;
	char buf2[500];
	char ack;
	int cnt = strlen (buf);
	char *p;

	/* Copy the packet into buffer BUF2, encapsulating it
	 *   and giving it a checksum.
	 */

	p = buf2;

	for (i = 0; i < cnt; i++){
		csum += buf[i];
		*p++ = buf[i];
	}
	*p++ = '#';
	bintohex( 1, &csum, p );
	p += 2;

	/* Send it over and over until we get a positive ack */

	write( remote_desc, buf2, p - buf2 );
	for ( ack = '?'; ack != '+'; ){
		read( remote_desc, &ack, 1 );
		if ( ack == '-' ){
			printf( "Remote NAK, resending\n" );
			write( remote_desc, buf2, p - buf2 );

		} else if ( ack != '+' ){
			printf( "Bad ACK, ignored: <%c>\n", ack );
		}
	}
}

static
int
readchar ()
{
  char buf[1];
  while (read (remote_desc, buf, 1) != 1) ;
  return buf[0] & 0x7f;
}

/* Read a packet from the remote machine, with error checking,
 * and store it in BUF.
 */
static
void
getpkt (buf)
     char *buf;
{
	char *bp;
	int csum;		/* Calculated checksum */
	char recvsum[2];	/* Checksum received, as 2 ASCII digits */
	int recv;		/* Checksum received, in binary	*/
	int c;

	while (1){
		csum = 0;
		bp = buf;
		while ( (c = readchar()) != '#' ){
			*bp++ = c;
			csum += c;
		}
		*bp = 0;

		recvsum[0] = readchar();
		recvsum[1] = readchar();
		recv = hextoint(recvsum);
		csum &= 0xff;
		if ( csum == recv ){
			break;
		}
		printf("Bad checksum received (0x%02x; calc=0x%02x), requesting resend\n",
						recv, csum );
		write (remote_desc, "-", 1);
	}

	write (remote_desc, "+", 1);
}

/* The data cache records all the data read from the remote machine
   since the last time it stopped.

   Each cache block holds 16 bytes of data
   starting at a multiple-of-16 address.  */

#define DCACHE_SIZE 64		/* Number of cache blocks */

struct dcache_block {
	struct dcache_block *next, *last;
	unsigned int addr;	/* Address for which data is recorded.  */
	int data[4];
};

struct dcache_block dcache_free, dcache_valid;

/* Free all the data cache blocks, thus discarding all cached data.  */ 
static
void
dcache_flush ()
{
  register struct dcache_block *db;

  while ((db = dcache_valid.next) != &dcache_valid)
    {
      remque (db);
      insque (db, &dcache_free);
    }
}

/*
 * If addr is present in the dcache, return the address of the block
 * containing it.
 */
static
struct dcache_block *
dcache_hit (addr)
{
  register struct dcache_block *db;

  if (addr & 3)
    abort ();

  /* Search all cache blocks for one that is at this address.  */
  db = dcache_valid.next;
  while (db != &dcache_valid)
    {
      if ((addr & 0xfffffff0) == db->addr)
	return db;
      db = db->next;
    }
  return NULL;
}

/*  Return the int data at address ADDR in dcache block DC.  */
static
int
dcache_value (db, addr)
     struct dcache_block *db;
     unsigned int addr;
{
  if (addr & 3)
    abort ();
  return (db->data[(addr>>2)&3]);
}

/* Get a free cache block, put it on the valid list,
   and return its address.  The caller should store into the block
   the address and data that it describes.  */
static
struct dcache_block *
dcache_alloc ()
{
  register struct dcache_block *db;

  if ((db = dcache_free.next) == &dcache_free)
    /* If we can't get one from the free list, take last valid */
    db = dcache_valid.last;

  remque (db);
  insque (db, &dcache_valid);
  return (db);
}

/* Return the contents of the word at address ADDR in the remote machine,
   using the data cache.  */
static
int
dcache_fetch (addr)
     CORE_ADDR addr;
{
  register struct dcache_block *db;

  db = dcache_hit (addr);
  if (db == 0)
    {
      db = dcache_alloc ();
      remote_read_bytes (addr & ~0xf, db->data, 16);
      db->addr = addr & ~0xf;
    }
  return (dcache_value (db, addr));
}

/* Write the word at ADDR both in the data cache and in the remote machine.  */
static
dcache_poke (addr, data)
     CORE_ADDR addr;
     int data;
{
  register struct dcache_block *db;

  /* First make sure the word is IN the cache.  DB is its cache block.  */
  db = dcache_hit (addr);
  if (db == 0)
    {
      db = dcache_alloc ();
      remote_read_bytes (addr & ~0xf, db->data, 16);
      db->addr = addr & ~0xf;
    }

  /* Modify the word in the cache.  */
  db->data[(addr>>2)&3] = data;

  /* Send the changed word.  */
  remote_write_bytes (addr, &data, 4);
}

/* Initialize the data cache.  */
static
dcache_init ()
{
  register i;
  register struct dcache_block *db;

  db = (struct dcache_block *) xmalloc (sizeof (struct dcache_block) * 
					DCACHE_SIZE);
  dcache_free.next = dcache_free.last = &dcache_free;
  dcache_valid.next = dcache_valid.last = &dcache_valid;
  for (i=0;i<DCACHE_SIZE;i++,db++)
    insque (db, &dcache_free);
}

#endif	/* I80960 */
