package Tutorial::Elastic::Search::With::Perl::First::Steps::Cheat::Sheet::Geo::Point::Distance;
use Moose;
use ElasticSearch;
use Data::Printer;
use Try::Tiny;
use File::Slurp;
use utf8;

=pod

=head1 Elastic search example of geo point usage

Quickly create an index and start inserting some stuff, and see for yourself if it works or not.

=head2 More tutorials

Tutorial::Elastic::Search::With::Perl::First::Steps::Cheat::Sheet
Tutorial::Elastic::Search::With::Perl::First::Steps::Cheat::Sheet::Geo::Point::Distance

=head2 Dependencys

Make sure you have these installed before executing this module.

    Moose;
    ElasticSearch;
    Data::Printer;
    Try::Tiny;
    File::Slurp;

=head2 How to use

This module has code also, so you can read the methos and see how they are executed.

    $ perl lib/Tutorial/Elastic/Search/With/Perl/First/Steps/Cheat/Sheet/Geo/Point/Distance.pm

    This will:
    - delete index: 'world'
    - create index: 'world'
    - alter mapping so we can insert some coordinates properly
    - insert some objects of type 'city' in our index called 'world'.

    I included some cities on the source code of this tutorial. (from maxmind.com)
    If you want more cities, you can replace with a list from: http://www.maxmind.com/en/worldcities
    I chose brasilian cities.. if you want a place you know, replace the content and execute some querys

    Then after inserting the cities, you usually want to search them... and there is an example of:

    - Search using latitude and longitude values. Define a point and give it some distance. That will be the radio distance of the area you are searching.
    - Show results sorted by distance. and you get to know the distance as well. This is cool!

=head2 Delete an index

    use Try::Tiny;
    ...

    sub delete_index {
        try {
            my $result = $es->delete_index( index => 'world',  );
        } catch {
            warn "$_";
        }
    }

=head2 Create an index

    sub create_index {
        my $result = $es->create_index(
            index      => 'world',
            "settings" => {
            }
        );
    }

=head2 Modify mapping, so you can use latitude longitue points

    sub alter_mapping_for_geo_point_distance {
        my $result = $es->put_mapping(
            index               => 'world',
            type                => 'cidade',
            mapping             => {
                "properties" => {
                    "location" => {
                        "type" => "geo_point",
                    }
                }
            },
            ignore_conflicts    => 0 | 1
        );
        warn p $result; sleep 1;
    }

=head2 Insert some data into elastic search

    sub insert_cities {
        my $cidades = &cidades();
    #   my @lines = read_file("/home/youruser/Downloads/worldcitiespop.txt"); #get at maxmind.com
        my @lines = split /\n/, $cidades;
        my @rows;
        my $id = 0;
          foreach my $line ( @lines ) {
              $line =~ s/\n//g;
              my @row = split ",", $line; #quick hack... not safe at all
              my $obj = {
                  country     => $row[0], #Country,
                  city        => $row[1], #City,
                  accentcity  => $row[2], #AccentCity,
                  region      => $row[3], #Region,
                  population  => $row[4], #Population,
                  location => {
                      lat    => $row[5], #Latitude,
                      lon    => $row[6], #Longitude
                  },
              };
              crud_create( $id++, $obj );
          }
    }


=head2 Search by proximity using latitude longitude

    sub search_by_proximity {
        my $item = $es->search(
            {
                index => 'world',
                type  => 'cidade',
                "query" => {
                    "match_all" => {}
                },
                "filter" => {
                    "geo_distance" => {
                        "distance" => "5km",
                        "location" => {
    #                       São Paulo
    #                       "lat" => -23.473293,
    #                       "lon" => -46.665803

    #                       Lapa
                            lat => -23.53333,
                            lon => -46.7,
                        }
                    }
                },
                size => 50,
                "sort" => [
                    {
                        "_geo_distance" => {
                            "location" => {
                                lat => -23.53333,
                                lon => -46.7,
                            },
                            "order" => "asc",
                            "unit" => "km"
                        }
                    }
                ],
            }
        );
    }

=head2 Distance between 2 coordinate points

    use GIS::Distance;
    my $gis = GIS::Distance->new(); # defaults to Haversine formula!u can change
    #$gis->formula( 'Polar' );# This is optional. Allows to change the formula.
    my $distance = $gis->distance( $lat1,$lon1 => $lat2,$lon2 );
    print $distance->meters();

=head2 What is Haversine

From wikipedia: http://en.wikipedia.org/wiki/Haversine_formula

The haversine formula is an equation important in navigation, giving great-circle distances between two points on a sphere from their longitudes and latitudes. It is a special case of a more general formula in spherical trigonometry, the law of haversines, relating the sides and angles of spherical "triangles".

=head2 AUTHOR

    If you liked this article, i am accepting donations at:
    Hernan Lopes  C<< <hernanlopes____gmail.com> >>

=cut

my $es = ElasticSearch->new(
    servers      => 'localhost:9200',
    transport    => 'http',
    max_requests => 10_000,
    trace_calls  => 1, # or 'log_file'
    no_refresh   => 0 | 1,
);

sub delete_index {
    try {
        my $result = $es->delete_index( index => 'world',  );
    } catch {
        warn "$_";
    }
}

sub create_index {
    my $result = $es->create_index(
        index      => 'world',
        "settings" => {

        }
    );
}

delete_index();sleep 1;
create_index();sleep 1;
alter_mapping_for_geo_point_distance();
insert_cities();
sleep 1; search_by_proximity();

sub alter_mapping_for_geo_point_distance {
    my $result = $es->put_mapping(
        index               => 'world',
        type                => 'cidade',
        mapping             => {
            "properties" => {
                "location" => {
                    "type" => "geo_point",
                }
            }
        },
        ignore_conflicts    => 0 | 1
    );
    warn p $result; sleep 1;
}

sub crud_create {
    my ( $id, $object ) = @_;
    my $result = $es->create(
        index => 'world',
        type  => 'cidade',
        data  => $object,
        id    => $id,
    );
}

sub insert_cities {
    my $cidades = &cidades();
    my @lines = split /\n/, $cidades;
    my @rows;
    my $id = 0;
    foreach my $line ( @lines ) {
        $line =~ s/\n//g;
        my @row = split ",", $line;
        my $obj = {
            country     => $row[0], #Country,
            city        => $row[1], #City,
            accentcity  => $row[2], #AccentCity,
            region      => $row[3], #Region,
            population  => $row[4], #Population,
            location => {
                lat     => $row[5], #Latitude,
                lon     => $row[6], #Longitude
            },
        };
        crud_create( $id++, $obj );
    }
}

sub search_by_proximity {
    my $item = $es->search(
        {
            index => 'world',
            type  => 'cidade',
            "query" => {
                "match_all" => {}
            },
            "filter" => {
                "geo_distance" => {
                    "distance" => "5km",
                    "location" => {
#                       São Paulo
#                       "lat" => -23.473293,
#                       "lon" => -46.665803

#                       Lapa
                        lat => -23.53333,
                        lon => -46.7,
                    }
                }
            },
#           from => 30,
            size => 50,
            "sort" => [
                {
                    "_geo_distance" => {
                        "location" => {
                            lat => -23.53333,
                            lon => -46.7,
                        },
                        "order" => "asc",
                        "unit" => "km"
                    }
                }
            ],
        }
    );

}

sub cidades {
my $world_city_maxmind = <<'CITY';
br,10 de novembro,10 de Novembro,26,,-27.033333,-50.933333
br,13 de maio,13 de Maio,26,,-28.583333,-49.166667
br,14 de julho,14 de Julho,21,,-22.764965,-43.289678
br,14 de maio,14 de Maio,11,,-22.533333,-54.35
br,15 de novembro,15 de Novembro,23,,-28.75,-53.1
br,21 de abril,21 de Abril,23,,-28.633333,-54.116667
br,23 de julho,23 de Julho,23,,-27.966667,-54.616667
br,25 de julio,25 de Julio,08,,-19.766667,-40.6
br,28 de setembro,28 de Setembro,23,,-29.1,-51.866667
br,2 de julho,2 de Julho,05,,-12.066667,-38.966667
br,2 de novembro,2 de Novembro,24,,-8.85,-62.283333
br,3 buritys,3 Buritys,24,,-12.166667,-60.083333
br,3 de maio,3 de Maio,23,,-27.783333,-54.233333
br,3 do maio,3 do Maio,23,,-27.783333,-54.233333
br,3 ilhas,3 Ilhas,15,,-22.032222,-43.489444
br,3 vendas,3 Vendas,23,,-29.85,-53.016667
br,4a legua,4a Légua,23,,-29.2444,-51.1402
br,5a legua,5a Légua,23,,-29.2238,-51.1732
br,7 voltas,7 Voltas,11,,-21.783333,-55.183333
br,a. alvarenga,A. Alvarenga,15,,-20.7,-45.366667
br,aba,Aba,17,,-6.716667,-37.983333
br,abacate,Abacate,03,,.316667,-50.883333
br,abacate,Abacate,04,,-3.946389,-62.903889
br,abacate,Abacate,16,,-1.116667,-49.65
br,abacaxi,Abacaxí,04,,-3.916667,-58.75
br,abacaxis,Abacaxis,04,,-3.916667,-58.75
br,abacaxy,Abacaxy,04,,-3.916667,-58.75
br,aba da serra,Aba da Serra,06,,-5.916667,-39.516667
br,abade,Abade,16,,-.7,-47.883333
br,abadia,Abadia,05,,-11.566667,-37.5
br,abadia,Abadia,15,,-17.716667,-43.1
br,abadia,Abadia,21,,-21.733333,-41.233333
br,abadia de pitangui,Abadia de Pitanguí,15,,-19.333333,-45.216667
br,abadia do pitangui,Abadia do Pitanguí,15,,-19.333333,-45.216667
br,abadia dos dourados,Abadia dos Dourados,15,,-18.466667,-47.4
br,abadiania,Abadiânia,29,7062,-16.1,-48.8
br,abaete,Abaeté,15,20371,-19.15,-45.45
br,abaete,Abaeté,16,,-1.7,-48.9
br,abaetetuba,Abaetetuba,16,78052,-1.7,-48.9
br,abaetezinho,Abaetèzinho,16,,-1.033333,-47.7
br,abaiara,Abaiara,06,,-7.35,-39.05
br,abaiba,Abaíba,15,,-21.6,-42.5
br,abaira,Abaíra,05,,-13.283333,-41.7
br,abaitinga,Abaitinga,27,,-24.033333,-48.05
br,abapa,Abapã,18,,-24.95,-49.816667
br,abapan,Abapan,18,,-24.95,-49.816667
br,a. barauna,A. Baraúna,17,,-7.116667,-36.95
br,a. barbosa,A. Barbosa,11,,-21.633333,-55.533333
br,abare,Abaré,05,,-8.716667,-39.116667
br,abarracamento,Abarracamento,21,,-22.2,-43.5
br,abate,Abaté,23,,-28.1,-52.916667
br,abatia,Abatiá,18,,-23.316667,-50.3
br,abaulado,Abaulado,27,,-23.071781,-47.873628
br,abbadia,Abbadia,05,,-11.566667,-37.5
br,abbadia,Abbadia,15,,-19.333333,-45.216667
br,abbadia dos dourados,Abbadia dos Dourados,15,,-18.466667,-47.4
br,abdon batista,Abdon Batista,26,,-27.616667,-50.983333
br,abel,Abel,15,,-22.9,-46.133333
br,abelardo luz,Abelardo Luz,26,,-26.6,-52.35
br,abelar do luz,Abelar do Luz,26,,-27.35,-51.683333
br,abelha,Abelha,11,,-21.333333,-57.7
br,abelha,Abelha,26,,-26.690898,-49.984099
br,abelhas,Abelhas,11,,-21.333333,-57.7
br,abelhas,Abelhas,22,,-5.948056,-36.061944
br,abertao,Abertão,15,,-22.3,-45.783333
br,abertinha,Abertinha,27,,-24.8867,-48.5365
br,abico,Ábico,04,,-3.15,-64.266667
br,abismo,Abismo,01,,-10.716667,-70.45
br,abobora,Abóbora,06,,-5.85,-40.416667
br,abobora,Abóbora,20,,-10.704167,-45.055833
br,abobora,Abóbora,27,,-24.637699,-48.172946
br,aboboras,Abóboras,05,,-9.816667,-40.05
br,aboboras,Abóboras,20,,-10.704167,-45.055833
br,aboboras,Abóboras,30,,-7.866667,-38.233333
br,abobral,Abobral,11,,-21.2,-56.833333
br,abolicao,Abolição,14,,-15.733333,-55.666667
br,abolicao,Abolição,21,,-22.882414,-43.301796
br,abolicao,Abolição,23,,-30.866667,-52.25
br,aborrecido,Aborrecido,06,,-2.893889,-40.918611
br,abraao,Abraão,21,,-23.133333,-44.166667
br,abraao,Abraão,24,,-9,-63.266667
br,abrahao,Abrahão,21,,-23.133333,-44.166667
br,abrahao,Abrahão,24,,-9,-63.266667
br,abranches,Abranches,18,,-25.266667,-49.816667
br,abrantes,Abrantes,05,,-12.816667,-38.266667
br,abre campo,Abre Campo,15,,-20.3,-42.483333
br,abreu,Abreu,13,,-3.350278,-42.419722
br,abreu,Abreu,18,,-23.327092,-49.807253
br,abreu,Abreu,20,,-7.8,-45.5
br,abreu,Abreu,23,,-29.933333,-57.183333
br,abreu e lima,Abreu e Lima,30,81960,-7.9,-34.883333
br,abreulandia,Abreulândia,31,,-9.565556,-49.056389
br,abreus,Abreus,05,,-10.183333,-40.966667
br,abreus,Abreus,15,,-21,-43.3
br,abrigo do bom pastor,Abrigo do Bom Pastor,27,,-23.629246,-47.825019
br,abucari,Abucari,04,,-5.016667,-62.783333
br,abufari,Abufari,04,,-5.396389,-63.090556
br,abufary,Abufary,04,,-5.396389,-63.090556
br,abuna,Abunã,24,,-9.7,-65.383333
br,abunai,Abunaí,04,,-1.833333,-66.55
br,abunan,Abunan,24,,-9.7,-65.383333
br,abundancia,Abundância,13,,-4.116667,-44.6
br,abura,Aburá,17,,-7.35,-35.433333
br,abutari,Abutari,04,,-5.396389,-63.090556
br,acaa,Acaã,05,,-12.366667,-43.3
br,acaam,Acaam,05,,-12.366667,-43.3
br,acacias,Acácias,21,,-22.44,-44.3228
br,academia,Academia,04,,-6.95,-71.251111
br,acaiaca,Acaiaca,15,,-20.35,-43.15
br,acai,Açaí,01,,-8.266667,-73.1
br,acai,Açaí,18,,-23.366667,-50.816667
br,acai,Açaí,27,,-21.383333,-48.983333
br,acaituba,Açaituba,04,,-7.016667,-64.583333
br,acaituba,Açaituba,25,,.65,-61.55
br,acaju,Acaju,05,,-13.033333,-39.533333
br,acajutiba,Acajutiba,05,14349,-11.666667,-38.016667
br,acampamento,Acampamento,01,,-7.45,-73.666667
br,acampamento,Acampamento,13,,-3.648333,-42.9625
br,acampamento,Acampamento,16,,-1,-57.016667
br,acampamento grande,Acampamento Grande,03,,.3,-53.066667
br,acampamonte,Acampamonte,20,,-6.983333,-41.166667
br,acapucai,Açapucaí,23,,-29.833333,-51.15
br,acara,Acará,04,,-4.369444,-59.660278
br,acara,Acará,13,,-3.416667,-45.316667
br,acara,Acará,16,12803,-1.95,-48.183333
br,acarahu,Acarahú,06,,-2.883333,-40.116667
br,acarahy,Acarahy,04,,-3.416667,-68.25
br,acarai,Acaraí,04,,-3.416667,-68.25
br,acarai,Acaraí,05,,-13.95,-39.116667
br,acarai,Acaraí,06,,-3.016667,-39.633333
br,acarai,Açaraí,15,,-19.583333,-41.483333
br,acara-mirim,Acará-Mirim,13,,-3.533333,-45.366667
br,acara-mirim,Acará-Mirim,16,,-2.4,-48.283333
br,acarape,Acarape,06,,-3.666667,-40.966667
br,acarau,Acaraú,06,28389,-2.883333,-40.116667
br,acare,Açaré,06,,-6.866667,-39.866667
br,acari,Acari,05,,-9.133333,-40.083333
br,acari,Acari,16,,-1.616667,-47.55
br,acari,Acari,21,,-22.824605,-43.341365
br,acari,Acari,22,9159,-6.516667,-36.633333
br,acari,Acari,25,,3.833333,-61.4
br,acary,Acary,22,,-6.516667,-36.633333
br,a. cassara,A. Cassara,24,,-13.216667,-62.1
br,acaua,Acauã,13,,-2.683333,-45.15
br,acaua,Acauã,20,,-8.2,-41.083333
br,acaua,Acauã,30,,-7.883333,-39.133333
br,acau,Acaú,17,,-7.116667,-35.35
br,acaua de baixo,Acauã de Baixo,30,,-7.9,-39.15
br,accioly,Accioly,08,,-19.666667,-40.466667
br,acegua,Aceguá,23,,-31.866667,-54.15
br,aceme,Aceme,30,,-8.133333,-38.95
br,acesita,Acesita,15,,-19.533333,-42.633333
br,aciline,Aciline,03,,-.383333,-52.1
br,acima,Acimã,04,,-7.716667,-65.883333
br,acioli,Acióli,08,,-19.666667,-40.466667
br,acopiara,Acopiara,06,24562,-6.1,-39.45
br,acoriana,Açoriana,23,,-32.1168,-52.5985
br,acorizal,Acorizal,14,,-15.2,-56.366667
br,acotea,Açotéa,23,,-31.766667,-53.8
br,acoteia,Açotéia,23,,-31.766667,-53.8
br,acougue,Açougue,05,,-11.983333,-40.416667
br,acougue,Açougue,06,,-3.45,-40.066667
br,acre,Acre,20,,-10.883333,-45.083333
br,acua,Açuã,20,,-8.2,-41.083333
br,acu,Açu,04,,-4.132222,-62.34
br,acu,Açu,05,,-12.5,-38.966667
br,acu,Açu,13,,-2.166667,-44.633333
br,acu,Açu,22,36126,-5.577222,-36.9075
br,acuarana,Açuarana,02,,-9.633333,-36.95
br,acucar,Açúcar,27,,-24.853518,-48.230402
br,acucena,Açucena,15,14506,-19.066667,-42.533333
br,acu da torre,Acu da Torre,05,,-12.55,-38
br,acude,Açude,02,,-9.416667,-36.633333
br,acude,Açude,05,,-13.833333,-41.4
br,acude,Açude,06,,-3.166667,-40.433333
br,acude,Açude,13,,-4.116667,-44.933333
br,acude,Açude,15,,-20.783333,-46.45
br,acude,Açude,17,,-6.833333,-38.016667
br,acude,Açude,20,,-4.6,-41.466667
br,acude,Açude,21,,-22.960392,-43.289884
br,acude,Açude,30,,-8.033333,-37.666667
br,acude angelim,Açude Angelim,30,,-8.883333,-36.283333
br,acude de baixo,Açude de Baixo,30,,-8.016667,-38.6
br,acude de leite,Açude de Leite,17,,-6.65,-37.75
br,acude de macaubas,Açude de Macaúbas,05,,-13,-42.533333
br,acude do mado,Açude do Mado,06,,-7.416667,-38.766667
br,acude do meio,Açude do Meio,30,,-7.883333,-37.366667
br,acude dos alves,Açude dos Alves,17,,-7.516667,-38.416667
br,acude dos carlos,Acude dos Carlos,30,,-7.75,-37.883333
br,acude do severino,Açude do Severino,17,,-7.366667,-37.833333
br,acude novo,Açude Nôvo,06,,-6.083333,-38.483333
br,acude novo,Açude Nôvo,22,,-5.918056,-37.086111
br,acudes,Açudes,22,,-6.25,-37.8
br,acude tapuio,Açude Tapuio,22,,-6.133333,-38.083333
br,acude velho,Açude Velho,06,,-3.75,-40.216667
br,acude velho,Açude Velho,17,,-7.116667,-36.366667
br,acude velho,Açude Velho,20,,-3.718056,-42.243889
br,acude velho,Açude Velho,22,,-5.871667,-36.078056
br,acudina,Açudina,05,,-13.2,-44.166667
br,acudinho,Açudinho,06,,-3.383333,-39.933333
br,acudinho,Açudinho,17,,-7.716667,-38.583333
br,acudinho,Açudinho,20,,-6.616667,-40.65
br,acudinho,Açudinho,30,,-8.1,-36.416667
br,acungui,Açungui,18,,-24.983333,-49.5
br,acupe,Acupe,05,,-12.65,-38.75
br,acuria,Acuriá,01,,-9.066667,-72.65
br,acurizal,Acurizal,14,,-15.2,-56.366667
br,acurizal,Acurizal,24,,-13.516667,-61.733333
br,acurui,Acuruí,15,,-20.2,-43.733333
br,acuru tendi,Acuru Tendi,11,,-23.716667,-54.783333
br,acutinga,Acutinga,05,,-12.616667,-38.816667
br,acutuba,Açutuba,04,,-4.073889,-62.445833
br,adalberto,Adalberto,21,,-22.316667,-41.9
br,adamantina,Adamantina,27,31079,-21.683333,-51.066667
br,adao candido lima,Adão Cândido Lima,15,,-20.85,-46.266667
br,adelia,Adélia,04,,-6.876389,-71.123333
br,adelino,Adelino,15,,-21.066667,-45.816667
br,adelino franca,Adelino França,15,,-16.45,-41.883333
br,adeque,Adequê,22,,-6.033333,-36.95
br,adiquinhao,Adiquinhão,22,,-5.948056,-37.372222
br,adolfo,Adolfo,26,,-26.733333,-51.033333
br,adolfo,Adolfo,27,,-21.233333,-49.65
br,adolfo,Adolfo,31,,-11.63,-49.2225
br,adolfo conder,Adolfo Cónder,26,,-26.733333,-51.033333
br,adolfo correia,Adolfo Correia,15,,-20.4,-45.083333
br,adolfo konder,Adolfo Konder,26,,-26.733333,-51.033333
br,adolpho conder,Adolpho Conder,26,,-26.733333,-51.033333
br,adolpho condor,Adolpho Condor,26,,-27.2,-49.8
br,adonis,Adônis,15,,-20.266667,-41.933333
br,adrianopolis,Adrianópolis,18,,-24.659641,-48.99079
br,adrianopolis,Adrianópolis,21,,-22.657778,-43.487222
br,adustina,Adustina,05,,-10.533333,-38.116667
br,aelegrete,Aelegrete,11,,-21.383333,-54.75
br,afeto,Afeto,20,,-8.233333,-40.783333
br,affligidos,Affligidos,05,,-12.483333,-38.85
br,affonso,Affonso,14,,-14.5,-57
br,affonso arinos,Affonso Arinos,15,,-15.916667,-46.066667
br,affonso camargo,Affonso Camargo,18,,-23.504,-49.9239
br,affonso claudio,Affonso Claudio,08,,-20.083333,-41.133333
br,affonso penha,Affonso Penha,08,,-19.683333,-41.016667
br,affonso penna,Affonso Penna,05,,-12.8,-39.2
br,affonso penna,Affonso Penna,06,,-6.1,-39.45
br,affua,Affua,16,,-.166667,-50.383333
br,afligidos,Afligidos,05,,-12.483333,-38.85
br,afogados,Afogados,30,,-7.75,-37.65
br,afogados da ingazeira,Afogados da Ingàzeira,30,25060,-7.75,-37.65
br,afonso,Afonso,11,,-21.95,-53.516667
br,afonso,Afonso,14,,-14.5,-57
br,afonso,Afonso,15,,-16.416667,-40.866667
br,afonso,Afonso,21,,-22.4,-42.6
br,afonso arinos,Afonso Arinos,21,,-22.009722,-43.338889
br,afonso bezerra,Afonso Bezerra,22,6947,-5.498056,-36.509167
br,afonso claudio,Afonso Cláudio,08,13543,-20.083333,-41.133333
br,afonso de paula,Afonso de Paula,15,,-20.016667,-45.283333
br,afonso martins,Afonso Martins,15,,-16.483333,-40.916667
br,afonso nonato,Afonso Nonato,15,,-16.25,-40.55
br,afonso pena,Afonso Pena,05,,-12.8,-39.2
br,afonso-pena,Afonso-Pena,06,,-6.1,-39.45
br,afonso pena,Afonso Pena,08,,-19.683333,-41.016667
br,afonsopolis,Afonsópolis,05,,-18.133333,-39.783333
br,afonso souto,Afonso Souto,23,,-29.966667,-55.2
br,afranio,Afrânio,30,,-8.516667,-41
br,afranio peixoto,Afrânio Peixoto,05,,-12.283333,-41.416667
br,afua,Afuá,16,7031,-.166667,-50.383333
br,agachi,Agachi,11,,-20.066667,-56.3
br,agarrado,Agarrado,04,,-5.373611,-59.973056
br,agenor alves,Agenor Alves,15,,-19.383333,-50.066667
br,agiota,Agiota,30,,-8.033333,-38.416667
br,agisse,Agissê,27,,-22.516667,-50.916667
br,agostinho,Agostinho,01,,-9.95,-68.55
br,agostinho,Agostinho,11,,-18.266667,-55.283333
br,agostinho,Agostinho,27,,-21.4,-47.55
br,agostinho,Agostinho,30,,-7.833333,-37.266667
br,agostinho porto,Agostinho Porto,21,,-22.787918,-43.377155
br,agostinho rodrigues,Agostinho Rodrigues,11,,-19.45,-56.566667
br,agreate,Agreate,06,,-5.583333,-40.6
br,agreste,Agreste,02,,-9.583333,-37.283333
br,agreste,Agreste,05,,-13.866667,-43.233333
br,agreste,Agreste,06,,-5.9,-39.833333
br,agreste,Agreste,30,,-8.35,-36.866667
br,agrestina,Agrestina,30,14453,-8.45,-35.95
br,agrinco,Agrinco,18,,-24.916667,-53.016667
br,agrioes,Agriões,21,,-22.418257,-42.974767
br,agropastoril papal do ceu,Agropastoril Papal do Céu,18,,-23.550578,-50.579507
br,agro-pecuaria palmeirense,Agro-Pecuária Palmeirense,18,,-25.52433,-50.35314
br,agrovila,Agrovila,18,,-25.4094,-50.5715
br,agua amarela,Água Amarela,03,,.383333,-52.183333
br,agua amarela,Água Amarela,31,,-5.766667,-48.05
br,agua amarela de baixo,Água Amarela de Baixo,18,,-25.8851,-50.180098
br,agua amarela de cima,Água Amarela de Cima,18,,-25.897993,-50.039159
br,agua amarela do meio,Água Amarela do Meio,18,,-25.892423,-50.111217
br,agua amarla do meio,Água Amarla do Meio,18,,-25.88441,-50.121859
br,agua azul,Água Azul,11,,-21.933333,-56.183333
br,agua azul,Água Azul,18,,-25.863209,-50.145768
br,agua azul,Agua Azul,20,,-6.183333,-41.583333
br,agua azul de baixo,Água Azul de Baixo,18,,-25.813198,-50.145404
br,agua azul de cima,Água Azul de Cima,18,,-25.829931,-50.121164
br,agua azul de meio,Água Azul de Meio,18,,-25.820023,-50.133088
br,agua azulzinha,Água Azulzinha,18,,-25.8479,-50.152161
br,agua bela,Água Bela,05,,-14.333333,-40.7
br,agua bela,Água Bela,15,,-16.166667,-40.816667
br,agua bela,Água Bela,17,,-6.766667,-37.933333
br,agua bela,Água Bela,20,,-5.816667,-42.833333
br,agua boa,Água Boa,05,,-12.416667,-44.616667
br,agua boa,Água Boa,06,,-3.05,-40.433333
br,agua boa,Água Boa,08,,-19.766667,-40.283333
br,agua boa,Água Boa,15,,-17.983333,-42.4
br,agua boa,Água Boa,18,,-23.5,-52.166667
br,agua boa,Água Boa,25,,2.666667,-60.766667
br,agua bonita,Água Bonita,27,,-22.733333,-47.75
br,agua bonita,Água Bonita,31,,-13.033333,-49.816667
br,agua branca,Água Branca,02,5161,-9.283333,-37.916667
br,agua branca,Água Branca,05,,-12.566667,-38.45
br,agua branca,Agua Branca,06,,-6.433333,-38.783333
br,agua branca,Água Branca,13,,-3.0825,-42.037778
br,agua branca,Água Branca,14,,-16.25,-55.466667
br,agua branca,Água Branca,15,,-14.75,-43.416667
br,agua branca,Água Branca,17,,-7.516667,-37.666667
br,agua branca,Água Branca,18,,-23.712166,-50.037343
br,agua-branca,Água-Branca,18,,-24.679441,-48.839312
br,agua branca,Água Branca,20,13912,-5.883333,-42.633333
br,agua branca,Água Branca,22,,-6.033333,-37.833333
br,agua branca,Água Branca,27,,-23.174735,-47.930774
br,agua-branca,Água-Branca,27,,-23.232136,-47.626292
br,agua branca,Água Branca,27,,-23.429777,-47.924835
br,agua branca,Agua Branca,30,,-7.833333,-37.583333
br,agua branca de baixo,Água Branca de Baixo,27,,-23.410557,-47.853598
br,agua branca de cima,Água Branca de Cima,18,,-25.69739,-50.42745
br,agua branca de cima,Água Branca de Cima,20,,-5.166667,-41.266667
br,agua branca de cima,Água Branca de Cima,27,,-23.440545,-47.862891
br,aguacerita,Aguacerita,14,,-15.816667,-58.75
br,agua clara,Água Clara,11,,-20.45,-52.866667
br,agua clara,Água Clara,18,,-23.326366,-52.59526
br,agua clara,Água Clara,26,,-27.116667,-48.916667
br,agua clara,Agua Clará,27,,-21.533333,-50.566667
br,agua clara de baixo,Água Clara de Baixo,18,,-25.956825,-49.422122
br,agua comprida,Água Comprida,05,,-12.316667,-37.9
br,agua comprida,Água Comprida,15,,-20.066667,-48.133333
br,agua comprida,Água Comprida,18,,-25.607999,-50.182499
br,aguacu,Aguaçu,11,,-19.45,-57.7
br,aguacu,Aguaçu,14,,-15.15,-56.016667
br,aguacuzinho,Aguaçuzinho,11,,-19.2,-56.466667
br,aguada,Aguada,05,,-12.716667,-42.233333
br,agua da fortuna,Água da Fortuna,18,,-25.0923,-53.9021
br,agua da mula,Água da Mula,18,,-23.342991,-49.794705
br,aguada nova,Aguada Nova,05,,-11.45,-41.8
br,aguada nova,Aguada Nova,15,,-16.3,-41.183333
br,agua da pedra,Água da Pedra,27,,-23.099043,-47.761872
br,agua da peroba,Água da Peroba,18,,-23.575174,-49.99037
br,agua da rega,Água da Rega,05,,-12.116667,-41.733333
br,agua da romana,Água da Romana,18,,-25.0576,-53.8151
br,agua da rosa,Água da Rosa,27,,-22.716667,-48.616667
br,agua das palmeiras,Água das Palmeiras,18,,-23.6513,-49.845
br,agua de rega,Água de Rega,05,,-12.116667,-41.733333
br,agua de rego,Água de Rêgo,05,,-12.116667,-41.733333
br,aguado,Aguado,05,,-12.716667,-42.233333
br,agua do barreirinho,Água do Barreirinho,18,,-23.116297,-49.990839
br,agua do barreiro,Água do Barreiro,18,,-23.8335,-50.8471
br,agua do boi,Água do Boi,18,,-23.0113,-50.1186
br,agua do cacador,Água do Caçador,18,,-23.928526,-50.512746
br,agua doce,Água Doce,03,,2.6,-50.833333
br,agua doce,Água Doce,05,,-14.8,-39.833333
br,agua doce,Água Dôce,06,,-5.316667,-39.266667
br,agua doce,Água Doce,08,,-18.55,-40.983333
br,agua doce,Água Doce,17,,-7.666667,-36.683333
br,agua doce,Água Doce,26,,-27,-51.55
br,agua doce,Água Doce,29,,-15.183333,-56.816667
br,agua doce do mantena,Água Doce do Mantena,08,,-18.55,-40.983333
br,agua do coriango,Água do Coriango,18,,-23.548612,-50.619678
br,agua do noratinho,Água do Noratinho,18,,-23.806027,-50.836333
br,agua do peixe,Água do Peixe,18,,-23.1413,-50.0282
br,agua dos coqueiros,Água dos Coqueiros,18,,-23.0271,-50.1315
br,agua emendada,Água Emendada,14,,-15.9,-54.3
br,agua ferro,Água Ferro,21,,-22.955,-43.908889
br,agua fetida,Água Fétida,05,,-12.533333,-43.566667
br,agua fina,Água Fina,14,,-14.266667,-55.966667
br,agua fria,Água Fria,05,,-11.85,-38.766667
br,agua fria,Água Fria,06,,-6.366667,-39.05
br,agua fria,Água Fria,08,,-18.433333,-40.4
br,agua fria,Água Fria,11,,-21.383333,-55.3
br,agua fria,Água Fria,13,,-3.169722,-43.479722
br,agua fria,Água Fria,14,,-15.183333,-55.75
br,agua fria,Água Fria,15,,-18.566667,-47.766667
br,agua fria,Água Fria,17,,-6.883333,-37.5
br,agua fria,Água Fria,27,,-23.35,-46.883333
br,agua fria,Água Fria,30,,-7.916667,-40.5
br,agua grande,Água Grande,05,,-12.316667,-38.433333
br,agua grande,Água Grande,18,,-23.7469,-49.7795
br,aguai,Aguaí,27,27503,-22.066667,-46.966667
br,agua limpa,Água Limpa,05,,-17.383333,-40.166667
br,agua limpa,Água Limpa,08,,-18.833333,-40.666667
br,agua limpa,Água Limpa,11,,-19.35,-54.7
br,agua limpa,Água Limpa,13,,-3.408889,-43.439444
br,agua limpa,Água Limpa,14,,-14.383333,-57.75
br,agua limpa,Água Limpa,15,,-19.15,-48.333333
br,agua limpa,Água Limpa,20,,-6.216667,-41.366667
br,agua limpa,Água Limpa,27,,-22.166667,-48.85
br,agua-limpa,Água-Limpa,29,,-17.1,-50.116667
br,agua limpa,Água Limpa,29,,-18.1,-48.766667
br,aguamare,Aguamare,22,,-5.106389,-36.320278
br,agua negra,Água Negra,26,,-26.9177,-49.0125
br,aguanil,Aguanil,15,,-20.95,-45.383333
br,agua nova,Agua Nova,22,,-5.331111,-36.535
br,aguapaba,Aguapaba,17,,-7.6,-35.583333
br,agua parada,Água Parada,27,,-24.647747,-48.099655
br,agua paranhos,Água Paranhos,18,,-23.397625,-52.709511
br,aguape,Aguapé,06,,-2.944444,-40.953056
br,aguape,Aguapé,15,,-21.9,-45.266667
br,aguape,Aguapé,30,,-8.616667,-38.383333
br,aguapei,Aguapeí,14,,-16.2,-59.716667
br,aguapei do alto,Aguapeí do Alto,27,,-21.6,-51.2
br,agua preta,Água Preta,04,,-8.833333,-66.166667
br,agua preta,Agua Preta,05,,-14.583333,-39.266667
br,agua preta,Água Preta,30,16444,-8.7,-35.516667
br,agua quente,Água Quente,05,,-13.183333,-44.3
br,agua quente,Água Quente,14,,-15.833333,-55.433333
br,agua quente,Água Quente,15,,-17.016667,-41.033333
br,agua quente,Água Quente,27,,-24.8814,-48.5078
br,agua quente,Água Quente,29,,-14.55,-48.933333
br,agua rasa,Água Rasa,15,,-20.433333,-45.166667
br,agua rica,Água Rica,13,,-3.693889,-43.018056
br,agua rosa,Agua Rosa,15,,-20.433333,-45.166667
br,agua santa,Água Santa,15,,-20.866667,-42.166667
br,agua santa,Água Santa,21,,-22.909708,-43.308343
br,agua santa,Água Santa,23,,-28.166667,-52.033333
br,agua santa,Água Santa,27,,-22.662,-45.5611
br,aguas belas,Águas Belas,05,,-12.5,-41.2
br,aguas belas,Águas Belas,06,,-4.05,-38.183333
br,aguas belas,Águas Belas,15,,-16.1,-41.05
br,aguas belas,Águas Belas,17,,-6.933333,-38.383333
br,aguas belas,Aguas Belas,20,,-7.033333,-42.283333
br,aguas belas,Águas Belas,30,22563,-9.116667,-37.116667
br,aguas bellas,Aguas Bellas,06,,-4.9,-40.766667
br,aguas bellas,Aguas Bellas,15,,-17.083333,-40.95
br,aguas bellas,Aguas Bellas,30,,-9.116667,-37.116667
br,aguas boas,Águas Boas,13,,-1.966667,-45.816667
br,aguas brancas,Águas Brancas,05,,-12.566667,-38.45
br,aguas brancas,Águas Brancas,23,,-28.583333,-50.4
br,aguas brancas,Águas Brancas,26,,-27.95,-49.583333
br,aguas claras,Águas Claras,08,,-18.85,-40.833333
br,aguas claras,Águas Claras,18,,-23.333313,-52.620215
br,aguas claras,Águas Claras,26,,-27.116667,-48.916667
br,aguas claras do sul,Águas Claras do Sul,27,,-21.416667,-51.183333
br,aguas da prata,Águas da Prata,27,,-21.933333,-46.716667
br,aguas de chapeco,Águas de Chapecó,26,,-27.083333,-52.983333
br,aguas de contendas,Águas de Contendas,15,,-21.9,-45.016667
br,aguas de lindoia,Águas de Lindóia,27,18806,-22.483333,-46.65
br,aguas de sao pedro,Águas de São Pedro,27,,-22.6,-47.883333
br,aguas do pardo,Aguas do Pardo,23,,-27.133333,-53.4
br,aguas do paulista,Águas do Paulista,05,,-12.5,-43.1
br,aguas do vere,Águas do Verê,18,,-25.766667,-52.95
br,aguas formosas,Águas Formosas,15,11512,-17.083333,-40.95
br,aguas lindas,Aguas Lindas,21,,-22.711389,-43.722778
br,aguas m. de ibira,Águas M. de Ibira,27,,-21.116667,-49.2
br,aguas pretas,Aguas Pretas,17,,-7.066667,-36.116667
br,aguas santas,Águas Santas,15,,-21.1,-44.216667
br,agua suja,Água Suja,05,,-14.033333,-44.45
br,agua suja,Água Suja,11,,-21.95,-54.466667
br,agua suja,Agua Suja,15,,-18.883333,-47.633333
br,agua suja,Água Suja,18,,-23.0442,-50.1685
br,agua sumida,Água Sumida,27,,-22.3,-52.216667
br,aguas vermelhas,Aguas Vermelhas,05,,-13.866667,-39.85
br,aguas vermelhas,Águas Vermelhas,15,16410,-15.75,-41.466667
br,agua verde,Água Verde,06,,-4.166667,-38.7
br,agua verde,Água Verde,18,,-23.367,-49.882
br,agua verde,Água Verde,20,,-7.366667,-41.716667
br,agua vermelha,Água Vermelha,05,,-13.866667,-39.85
br,agua vermelha,Água Vermelha,06,,-7.4,-38.933333
br,agua vermelha,Água Vermelha,13,,-6.483333,-43.133333
br,agua vermelha,Agua Vermelha,15,,-15.75,-41.466667
br,agua vermelha,Água Vermelha,27,,-21.9,-47.883333
br,agua viva,Água Viva,15,,-21.683333,-42.55
br,aguazinha,Águazinha,17,,-7.9,-36.7
br,agudo,Agudo,23,,-29.633333,-53.25
br,agudos,Agudos,18,,-25.9892,-49.3342
br,agudos,Agudos,27,30808,-22.466667,-49
br,agudos do sul,Agudos do Sul,18,,-25.9892,-49.3342
br,aguia,Águia,04,,-6.5,-69.433333
br,aguia branca,Águia Branca,08,2554,-18.983333,-40.733333
br,aguiar,Aguiar,17,,-7.083333,-38.183333
br,aguiar,Aguiar,23,,-2.7,-54.2
br,aguiar velho,Aguiar Velho,17,,-7.1,-38.1
br,aguilhadas,Aguilhadas,06,,-6.533333,-40.366667
br,aguinhas,Aguinhas,26,,-27,-52.95
br,agulha,Agulha,27,,-21.35,-48.7
br,agulhas negras,Agulhas Negras,21,,-22.4548,-44.4377
br,aguti,Aguti,26,,-27.3,-49.15
br,aiapua,Aiapuá,04,,-4.463333,-62.055278
br,aideia do prata,Aideia do Prata,15,,-16.816667,-40.616667
br,aimin,Aimin,04,,-4.803889,-62.451667
br,aimores,Aimorés,05,,-17.833333,-40.166667
br,aimores,Aimorés,15,19133,-19.5,-41.066667
br,aiparuca,Aiparuça,04,,-2.981111,-60.608056
br,aiquara,Aiquara,05,,-14.116667,-39.866667
br,airao,Airão,04,,-1.95,-61.366667
br,airi,Airi,04,,-6.9,-58.283333
br,airi,Airi,30,,-8.533333,-38.183333
br,airituba,Airituba,08,,-20.95,-41.666667
br,airoes,Airões,15,,-20.8,-42.966667
br,airy,Airy,04,,-6.9,-58.283333
br,aito calcado,Aito Calçado,08,,-20.95,-41.616667
br,aiua,Aiuá,06,,-3.45,-40.466667
br,aiuaba,Aiuaba,06,,-6.633333,-40.116667
br,aiure,Aiuré,26,,-28.1,-49.3
br,aiuruoca,Aiuruoca,15,,-21.966667,-44.6
br,ajacio,Ajácio,15,,-16.1,-40.35
br,ajapi,Ajapi,27,,-22.283333,-47.55
br,ajara,Ajará,04,,-4.428333,-62.268889
br,ajaru,Ajaru,04,,-2.211389,-61.114444
br,ajerico,Ajericó,16,,-.833333,-51.816667
br,ajuda,Ajuda,05,,-16.5,-39.083333
br,ajudante,Ajudante,27,,-23.166667,-47.316667
br,ajura,Ajurá,04,,-3.904444,-62.732222
br,ajuricaba,Ajuricaba,04,,-7.583333,-66.616667
br,ajuricaba,Ajuricaba,23,,-28.25,-53.766667
br,ajurical,Ajurical,04,,-7.583333,-66.616667
br,ajuritiba,Ajuritiba,27,,-23.133333,-46.466667
br,ajusta conta,Ajusta Conta,14,,-15.75,-57.633333
br,ajusta-contas,Ajusta-Contas,14,,-15.75,-57.633333
br,akotipa,Akotipa,16,,1.916667,-55.55
br,alagacico,Alagaciço,05,,-9.533333,-38.366667
br,alagadico,Alagadiço,05,,-10.666667,-41.283333
br,alagadico,Alagadiço,06,,-5.25,-40.966667
br,alagadico,Alagadiço,15,,-15.1,-43.666667
br,alagadico,Alagadiço,20,,-3.75,-42.533333
br,alagadico,Alagadiço,25,,3.383333,-60.616667
br,alagadico grande,Alagadico Grande,22,,-5.816667,-35.416667
br,alagamar,Alagamar,06,,-4.2,-38.4
br,alagoa,Alagoa,15,,-22.166667,-44.633333
br,alagoa,Alagoa,27,,-23.502437,-47.013252
br,alagoa da serra negra,Alagôa da Serra Negra,05,,-10.033333,-37.9
br,alagoa de baixo,Alagoa de Baixo,30,,-8.083333,-37.266667
br,alagoa de roca,Alagoa de Roça,17,,-7.083333,-35.866667
br,alagoado,Alagoado,20,,-6.033333,-42.283333
br,alagoa do monteiro,Alagôa do Monteiro,17,,-7.883333,-37.116667
br,alagoa grande,Alagoa Grande,17,16406,-7.05,-35.633333
br,alagoa grande,Alagoa Grande,20,,-8.416667,-44.166667
br,alagoa nova,Alagoa Nova,17,9182,-7.066667,-35.766667
br,alagoas,Alagôas,02,,-9.716667,-35.9
br,alagoas,Alagoas,05,,-12.433333,-40.3
br,alagoas,Alagoas,13,,-3.771389,-43.408056
br,alagoas,Alagoas,15,,-18.633333,-46.666667
br,alagoas,Alagoas,29,,-13.8,-47.816667
br,alagoinha,Alagoinha,02,,-10.166667,-36.383333
br,alagoinha,Alagoinha,05,,-11.683333,-41.333333
br,alagoinha,Alagoinha,06,,-6.783333,-38.716667
br,alagoinha,Alagoinha,13,,-3.077778,-42.026111
br,alagoinha,Alagoinha,17,7474,-6.933333,-35.55
br,alagoinha,Alagoinha,20,,-6.933333,-41.716667
br,alagoinha,Alagoinha,22,,-5.066667,-37.4
br,alagoinha,Alagoinha,30,,-8.483333,-36.783333
br,alagoinhas,Alagoinhas,02,,-10.166667,-36.383333
br,alagoinhas,Alagoinhas,05,122692,-12.116667,-38.433333
br,alagoinhas,Alagoinhas,20,,-4.033333,-41.85
br,alagoinhas de santana,Alagoinhas de Santana,05,,-13.983333,-41.533333
br,alamao,Alamão,05,,-12.783333,-38.85
br,alamari,Alamari,27,,-23.5521,-47.8989
br,alambari,Alambari,27,,-20.583333,-47.233333
br,alambary,Alambary,27,,-23.5521,-47.8989
br,alambique,Alambique,08,,-20.083333,-41.133333
br,alambique,Alambique,18,,-23.4221,-52.545324
br,alambique,Alambique,23,,-29.769,-50.7919
br,alamoique,Alamoique,16,,1.983056,-55.166667
br,alasca,Alasca,24,,-13.297222,-60.370833
br,alaska,Alaska,24,,-13.297222,-60.370833
br,alavanca,Alavanca,11,,-21.55,-54.3
br,alazao,Alazão,05,,-12.7,-44.666667
br,alazao,Alazão,30,,-7.766667,-37.483333
br,alazao dos cardosos,Alazão dos Cardosos,13,,-3.992778,-42.906111
br,albano,Albano,04,,-2.45,-57.55
br,albano,Albano,14,,-15.75,-55.583333
br,albertina,Albertina,15,,-22.2,-46.616667
br,albertina,Albertina,26,,-27.25,-49.65
br,alberto isaacson,Alberto Isaacson,15,,-19.5,-45.066667
br,alberto moreira,Alberto Moreira,27,,-20.433333,-48.583333
br,albertos,Albertos,15,,-20.583333,-45.483333
br,alberto torres,Alberto Tôrres,21,,-22.236111,-43.096389
br,albino,Albino,13,,-3.609722,-42.987778
br,albino,Albino,15,,-18.45,-48.583333
br,albino alves garcia,Albino Alves Garcia,15,,-20.45,-45.2
br,albinopolis,Albinópolis,27,,-21.733333,-47.533333
br,albuquerque,Albuquerque,04,,-6.887222,-58.340833
br,albuquerque,Albuquerque,06,,-4.583333,-37.766667
br,albuquerque,Albuquerque,11,,-19.383333,-57.433333
br,albuquerque,Albuquerque,21,,-22.385802,-42.908931
br,albuquerque,Albuquerque,27,,-20.983333,-48.833333
br,albuquerque,Albuquerque,30,,-8.016667,-37.4
br,albuquerque de fora,Albuquerque de Fora,22,,-6.383333,-38.2
br,albuquerque lins,Albuquerque Lins,27,,-21.666667,-49.75
br,albuquerque ne,Albuquerque Né,30,,-8.016667,-37.4
br,alcacus,Alcaçus,22,,-6.016667,-35.133333
br,alcantara,Alcântara,13,5655,-2.4,-44.4
br,alcantara,Alcântara,21,,-22.824444,-43.035278
br,alcantaras,Alcântaras,06,,-3.583333,-40.5
br,alcantilado,Alcantilado,14,,-16.383333,-53.516667
br,alcantil,Alcantil,17,,-7.733333,-36.05
br,alceu garcia,Alceu Garcia,15,,-18.55,-49.183333
br,alcibiades vargas,Alcibiades Vargas,11,,-23.383333,-54.65
br,alcides batista,Alcides Batista,15,,-17.516667,-40.65
br,alcidio,Alcídio,27,,-21.033333,-47.55
br,alcobaca,Alcobaça,05,,-17.5,-39.216667
br,alcobaca,Alcobaça,16,,-3.7,-49.7
br,alcobaco,Alcobaço,27,,-23.47624,-46.614322
br,aldeamento,Aldeamento,08,,-19.266667,-40.733333
br,aldeamento,Aldeamento,15,,-19.383333,-41.316667
br,aldeia 20 de septembro,Aldeia 20 de Septembro,14,,-12.883333,-59.3
br,aldeia,Aldeia,05,,-12.366667,-40
br,aldeia,Aldeia,11,,-18.2,-55.166667
br,aldeia,Aldeia,13,,-3.013889,-42.004167
br,aldeia,Aldeia,14,,-15.116667,-56.383333
br,aldeia,Aldeia,15,,-16.033333,-41.3
br,aldeia,Aldeia,17,,-7.35,-36.3
br,aldeia,Aldeia,20,,-7.383333,-40.85
br,aldeia,Aldeia,27,,-23.5,-46.85
br,aldeia,Aldeia,28,,-10.783333,-36.966667
br,aldeia,Aldeia,30,,-7.833333,-35.116667
br,aldeia,Aldeia,31,,-7.916667,-48.233333
br,aldeia amanajas tauai,Aldeia Amanajás Tauaí,13,,-4.533333,-47.366667
br,aldeia amanajaz tauai,Aldeia Amanajaz Tauaí,13,,-4.533333,-47.366667
br,aldeia amanajaz taudi,Aldeia Amanajaz Taudí,13,,-4.533333,-47.366667
br,aldeia bacaeri,Aldeia Bacaeri,14,,-13.1,-54.1
br,aldeia bacaery,Aldeia Bacaery,14,,-14.2,-55.083333
br,aldeia bakairi,Aldeia Bakairi,14,,-14.2,-55.083333
br,aldeia cabixi,Aldeia Cabixí,14,,-12.983333,-60.2
br,aldeia cabixy,Aldeia Cabixy,14,,-12.983333,-60.2
br,aldeia caiabi,Aldeia Caiabi,14,,-11.616667,-55.683333
br,aldeia campista,Aldeia Campista,21,,-22.917417,-43.237362
br,aldeia capitao carutu,Aldeia Capitão Carutú,14,,-14.316667,-54.7
br,aldeia caraja,Aldeia Carajá,14,,-10.5,-50.55
br,aldeia caraja,Aldeia Carajá,31,,-7.066667,-49.183333
br,aldeia caraja do capitao maloe,Aldeia Caraja do Capitão Maloé,31,,-11.566667,-50.666667
br,aldeia curipi,Aldeia Curipi,03,,3.716667,-51.683333
br,aldeia de bororo,Aldeia de Borôro,14,,-16.566667,-57.866667
br,aldeia de pedra,Aldeia de Pedra,13,,-5.833333,-45.933333
br,aldeia do cabo,Aldeia do Cabo,21,,-22.966667,-42.016667
br,aldeia do capitao chimang,Aldeia do Capitão Chimang,13,,-2.866667,-46.583333
br,aldeia do capitao mariquinhas,Aldeia do Capitão Mariquinhas,13,,-3.216667,-46.233333
br,aldeia do capitao piarru,Aldeia do Capitão Piarru,13,,-3.066667,-46.3
br,aldeia do inacio,Aldeia do Inácio,14,,-16.866667,-57.483333
br,aldeia do meio,Aldeia do Meio,31,,-7.066667,-49.183333
br,aldeia do meio caraja,Aldeia do Meio Carajá,31,,-7.066667,-49.183333
br,aldeia do prata,Aldeia do Prata,15,,-16.816667,-40.616667
br,aldeia do ranchao,Aldeia do Ranchão,14,,-12.883333,-59.016667
br,aldeia dos bororos,Aldeia dos Bororos,14,,-16.566667,-57.866667
br,aldeia dos caiapos,Aldeia dos Caiapós,15,,-19.85,-50.25
br,aldeia dos indios,Aldeia dos Índios,16,,-8.1,-54.616667
br,aldeia dos machacalis,Aldeia dos Machacalis,15,,-16.883333,-40.633333
br,aldeia fichada,Aldeia Fichada,16,,-7.35,-49.466667
br,aldeia indigena catuquina,Aldeia Indígena Catuquina,01,,-8.294444,-71.665833
br,aldeia indigena espirro,Aldeia Indígena Espirro,14,,-12.725,-59.966667
br,aldeia indigena ipixuna,Aldeia Indígena Ipixuna,16,,-4.773333,-52.505833
br,aldeia indigena man-azde,Aldeia Indígena Man-Âzde,14,,-13.051389,-59.870833
br,aldeia indigena serra azul,Aldeia Indígena Serra Azul,14,,-13.670833,-59.5875
br,aldeia jose coroado,Aldeia José Coroado,14,,-14.3,-54.7
br,aldeia manau,Aldeia Manau,16,,-8.75,-49.65
br,aldeia manuel antonio,Aldeia Manuel Antônio,16,,-3.15,-46.733333
br,aldeiamento,Aldeiamento,08,,-19.266667,-40.733333
br,aldeiamento,Aldeiamento,15,,-19.383333,-41.316667
br,aldeia nambikwara,Aldeia Nambikwara,14,,-12.75,-59.8
br,aldeia nhambiquara,Aldeia Nhambiquara,14,,-12.85,-59.3
br,aldeia palikur,Aldeia Palikur,03,,3.45,-51.5
br,aldeia paracanti,Aldeia Paracanti,16,,-8.455556,-49.583333
br,aldeia paracaty,Aldeia Paracaty,16,,-8.455556,-49.583333
br,aldeia parakana,Aldeia Parakanã,16,,-4.423889,-49.9775
br,aldeia pequena,Aldeia Pequena,16,,-8.416667,-49.833333
br,aldeia pirizal,Aldeia Pirizal,14,,-16.833333,-56.133333
br,aldeia queimada,Aldeia Queimada,14,,-14.45,-58.116667
br,aldeia ranchao,Aldeia Ranchão,14,,-12.883333,-59.016667
br,aldeias das canoas,Aldeias das Canoas,16,,.8,-55.85
br,aldeia toliri,Aldeia Tolirí,14,,-14.633333,-58.4
br,aldeia toloiri,Aldeia Toloiri,14,,-14.633333,-58.4
br,aldeia velha,Aldeia Velha,11,,-20.866667,-57.833333
br,aldeia velha,Aldeia Velha,13,,-5.15,-45.583333
br,aldeia velha,Aldeia Velha,14,,-14.683333,-59.183333
br,aldeia velha,Aldeia Velha,15,,-16.85,-40.583333
br,aldeia velha,Aldeia Velha,16,,-8.338056,-50.464444
br,aldeia velha,Aldeia Velha,17,,-7.083333,-35.866667
br,aldeia velha,Aldeia Velha,21,,-22.466667,-42.3
br,aldeia velha,Aldeia Velha,30,,-8.2,-36.466667
br,aldeia velha,Aldeia Velha,31,,-6.033333,-48.3
br,aldeia vila batista,Aldeia Vila Batista,04,,-1.933611,-61.203889
br,aldeia vinte de setembro,Aldeia Vinte de Setembro,14,,-12.883333,-59.3
br,aldeia waikiso,Aldeia Waikiso,14,,-14.1875,-59.683333
br,aldeia xada,Aldeia Xadá,04,,-4.540278,-59.168056
br,aldeia zamaicara,Aldeia Zamaicara,14,,-14.633333,-58.55
br,aldeia zamaicare,Aldeia Zamaicare,14,,-14.633333,-58.55
br,aldeia zamicare,Aldeia Zamicaré,14,,-14.633333,-58.55
br,aldeinha,Aldeinha,13,,-5.95,-45.033333
br,aldeola,Aldeola,01,,-8.216667,-72.966667
br,alecrim,Alecrim,02,,-9.566667,-37.316667
br,alecrim,Alecrim,05,,-12.583333,-38.933333
br,alecrim,Alecrim,06,,-3.466667,-40.133333
br,alecrim,Alecrim,15,,-16.465556,-44.8125
br,alecrim,Alecrim,18,,-23.6149,-49.7811
br,alecrim,Alecrim,20,,-6.95,-40.5
br,alecrim,Alecrim,22,,-5.512222,-37.413889
br,alecrim,Alecrim,23,,-27.65,-54.766667
br,alecrim,Alecrim,27,,-24.283333,-47.233333
br,alecrim,Alecrim,30,,-7.9,-39.583333
br,alegre,Alegre,04,,-5.816667,-61.416667
br,alegre,Alegre,05,,-10.316667,-41.45
br,alegre,Alegre,06,,-3.616667,-40.283333
br,alegre,Alegre,08,19758,-20.766667,-41.533333
br,alegre,Alegre,11,,-18.616667,-55.116667
br,alegre,Alegre,13,,-1.833333,-44.75
br,alegre,Alegre,14,,-15.247778,-59.833333
br,alegre,Alegre,15,,-18.3,-47.016667
br,alegre,Alegre,16,,-2.2,-48.5
br,alegre,Alegre,20,,-5.766667,-41.783333
br,alegre,Alegre,26,,-26.997859,-49.990577
br,alegre,Alegre,27,,-24.116667,-48.9
br,alegre,Alegre,30,,-7.916667,-38.216667
br,alegre de cima,Alegre de Cima,15,,-22.556896,-45.668764
br,alegres,Alegres,05,,-10.516667,-38.233333
br,alegres,Alegres,06,,-5.95,-40.1
br,alegrete,Alegrete,04,,-5.166667,-60.4
br,alegrete,Alegrete,11,,-20.766667,-55.4
br,alegrete,Alegrete,14,,-15.166667,-56.416667
br,alegrete,Alegrete,20,,-7.233333,-40.85
br,alegrete,Alegrete,23,77540,-29.766667,-55.766667
br,alegrete,Alegrete,29,,-17.3,-49.916667
br,alegre velho,Alegre Velho,20,,-6.833333,-42.35
br,alegria,Alegria,03,,.316667,-51.2
br,alegria,Alegria,04,,-2.633333,-67.45
br,alegria,Alegria,05,,-13.416667,-39.95
br,alegria,Alegria,08,,-19.416667,-40.616667
br,alegria,Alegria,11,,-19.683333,-55.65
br,alegria,Alegria,13,,-3.698889,-43.354444
br,alegria,Alegria,14,,-8.85,-58.466667
br,alegria,Alegria,15,,-15.5,-42.066667
br,alegria,Alegria,16,,-.6,-47.666667
br,alegria,Alegria,18,,-26.219167,-53.5325
br,alegria,Alegria,20,,-3.835278,-42.253889
br,alegria,Alegria,22,,-5.616667,-36.683333
br,alegria,Alegria,23,,-27.8,-54.066667
br,alegria,Alegria,27,,-22.516667,-50.916667
br,alegria nova,Alegria Nova,11,,-21.183333,-57.15
br,alegrias,Alegrias,23,,-31.833333,-53.416667
br,alegria velha,Alegria Velha,11,,-21.166667,-57.116667
br,aleixo,Aleixo,14,,-15.166667,-56.216667
br,aleixo,Aleixo,22,,-5.721389,-37.343056
br,aleixo,Aleixo,26,,-28.25,-50.45
br,aleluia,Aleluia,21,,-21.8,-41.633333
br,aleluia,Aleluia,27,,-23.260704,-47.92205
br,aleluia de baixo,Aleluia de Baixo,27,,-23.230149,-47.9321
br,alemao,Alemão,22,,-5.233611,-36.796111
br,alemao do francisco,Alemão do Francisco,05,,-12.783333,-38.85
br,alemoa,Alemôa,18,,-23.55,-49.7
br,alem parahyba,Alem Parahyba,15,,-21.866667,-42.683333
br,alem paraiba,Além Paraíba,15,33907,-21.866667,-42.683333
br,alemquer,Alemquer,16,,-1.933333,-54.766667
br,alem tejo,Além Tejo,20,,-6.033333,-41.966667
br,alencar,Alencar,06,,-6.4,-39.15
br,alenquer,Alenquer,16,26291,-1.933333,-54.766667
br,alexandra,Alexandra,18,,-25.566667,-48.616667
br,alexandre,Alexandre,14,,-8.7,-58.016667
br,alexandre bode,Alexandre Bode,08,,-20.133333,-40.95
br,alexandre de gusmao,Alexandre de Gusmão,23,,-28.816667,-51.8
br,alexandre mendes,Alexandre Mendes,20,,-3.779444,-42.443056
br,alexandria,Alexandria,22,9769,-6.416667,-38.016667
br,alexandria,Alexandria,30,,-8.183333,-39.7
br,alexandrina,Alexandrina,04,,.866667,-66.9
br,alexandrina,Alexandrina,30,,-8.183333,-39.7
br,alexandrita,Alexandrita,15,,-19.7,-50.45
br,alexrandina,Alexrandina,30,,-8.183333,-39.7
br,alfa barra,Alfa Barra,21,,-23.008452,-43.369043
br,alfenas,Alfenas,15,71407,-21.433333,-45.95
br,alferes,Alferes,06,,-5.95,-40.1
br,alferez,Alferez,06,,-5.95,-40.1
br,alfeu,Alfeu,23,,-28.95,-52.633333
br,alfonso bezerra,Alfonso Bezerra,22,,-5.498056,-36.509167
br,alfredo,Alfredo,15,,-17.066667,-40.816667
br,alfredo brener,Alfredo Brener,23,,-28.566667,-53.133333
br,alfredo brenner,Alfredo Brenner,23,,-28.566667,-53.133333
br,alfredo castilho,Alfredo Castilho,27,,-20.866667,-51.483333
br,alfredo chaves,Alfredo Chaves,08,6422,-20.633333,-40.75
br,alfredo chaves,Alfredo Chaves,23,,-28.95,-51.55
br,alfredo chavez,Alfredo Chavez,08,,-20.633333,-40.75
br,alfredo dantas,Alfredo Dantas,31,,-7.116667,-48.35
br,alfredo dutra,Alfredo Dutra,06,,-4.383333,-38.883333
br,alfredo guedes,Alfredo Guedes,27,,-22.583333,-48.716667
br,alfredo manso,Alfredo Manso,15,,-21.45,-46.133333
br,alfredo marcondes,Alfredo Marcondes,27,,-21.95,-51.416667
br,alfredo silveira,Alfredo Silveira,23,,-30.35,-52.083333
br,alfredo teixeira,Alfredo Teixeira,08,,-20.233333,-40.45
br,alfredo wagner,Alfredo Wagner,26,,-27.7,-49.333333
br,algeria,Algeria,15,,-15.5,-42.066667
br,algodao,Algodão,05,,-12.416667,-40.583333
br,algodao,Algodão,06,,-3.433333,-40.416667
br,algodao,Algodão,17,,-6.9,-36
br,algodao,Algodão,30,,-7.883333,-36.166667
br,algodoal,Algodoal,16,,-.6,-47.583333
br,algodoes,Algodões,05,,-10.2,-41.95
br,algodoes,Algodões,06,,-3.516667,-41.4
br,algodoes,Algodões,20,,-7.033333,-42.683333
br,algodoes,Algodões,30,,-7.666667,-39.6
br,algoinhas,Algoinhas,30,,-8.483333,-36.783333
br,alhandra,Alhandra,17,10788,-7.433333,-34.9
br,alho,Alho,05,,-16.933333,-39.716667
br,alianca,Aliança,04,,-2.917222,-69.685556
br,alianca,Aliança,05,,-12.466667,-38.683333
br,alianca,Alianca,06,,-4.733333,-38.516667
br,alianca,Aliança,11,,-19.05,-55.266667
br,alianca,Aliança,14,,-9.1,-60.583333
br,alianca,Aliança,15,,-19.6,-43.433333
br,alianca,Aliança,21,,-22.276667,-43.53
br,alianca,Aliança,24,,-8.616667,-63.533333
br,alianca,Aliança,26,,-27.2,-48.916667
br,alianca,Aliança,27,,-20.5,-47.316667
br,alianca,Aliança,30,17329,-7.583333,-35.216667
br,alianca do tocantins,Aliança do Tocantins,31,,-11.3375,-48.850278
br,alianca velha,Aliança Velha,18,,-25.966172,-50.249492
br,aliatar,Aliatar,26,,-28.7,-49.3
br,alice,Alice,03,,3.65,-51.983333
br,alipio,Alípio,27,,-21,-47.216667
br,alipio paca,Alípio Paca,15,,-17.15,-40.65
br,alivio,Alívio,06,,-5.083333,-38.716667
br,alivio,Alívio,13,,-4.9,-44.583333
br,alivio,Alívio,17,,-6.583333,-38.15
br,alivio,Alívio,20,,-3.869444,-42.303333
br,alivio,Alívio,22,,-5.636944,-36.141111
br,aljezur,Aljezur,21,,-22.666667,-43.6
br,aljubarrota,Aljubarrota,04,,-4.297222,-69.5375
br,allemoa,Allemõa,18,,-23.55,-49.7
br,allianca,Alliança,04,,-6.583333,-64.35
br,allianca,Alliança,14,,-9.1,-60.583333
br,allianca,Alliança,15,,-19.6,-43.433333
br,allianca,Allianca,24,,-8.616667,-63.533333
br,allianca,Alliança,26,,-27.2,-48.916667
br,allianca,Alliança,30,,-7.583333,-35.216667
br,almadina,Almadina,05,,-14.7,-39.633333
br,almas,Almas,05,,-10.3,-41.4
br,almas,Almas,06,,-2.895,-41.275833
br,almas,Almas,13,,-3.780278,-42.829167
br,almas,Almas,15,,-19.15,-44.633333
br,almas,Almas,17,,-6.65,-37.683333
br,almas,Almas,20,,-3.579167,-42.0275
br,almas,Almas,30,,-7.85,-37.883333
br,almas,Almas,31,,-11.626111,-47.101667
br,almecega,Almécega,20,,-7.016667,-42.516667
br,almecegas,Almécegas,13,,-5.866667,-43.1
br,almecegas,Almécegas,20,,-5.85,-43.083333
br,almecega velha,Almécega Velha,06,,-2.9,-40
br,almeida,Almeida,05,,-13.45,-39.433333
br,almeida,Almeida,13,,-5.916667,-44.15
br,almeida,Almeida,15,,-19.433333,-43.833333
br,almeida,Almeida,18,,-25.331473,-48.4179
br,almeida,Almeida,20,,-5.566667,-42.366667
br,almeida,Almeida,21,,-21.2,-41.716667
br,almeida,Almeida,30,,-7.7,-37.833333
br,almeidas,Almeidas,15,,-22.366667,-46.533333
br,almeirim,Almeirim,16,20464,-1.533333,-52.566667
br,almenara,Almenara,15,29333,-16.183333,-40.7
br,almendra,Almendra,25,,-1.3,-61.7
br,almeriana,Almeriana,03,,2.3,-50.75
br,almerim,Almerim,16,,-1.533333,-52.566667
br,almesca,Almesca,13,,-5.866667,-43.1
br,almescar,Almescar,13,,-5.866667,-43.1
br,almesquinha,Almesquinha,13,,-6.016667,-43.066667
br,almesquinho,Almesquinho,13,,-6.016667,-43.066667
br,almfala,Almfala,06,,-2.933333,-39.816667
br,almino affonso,Almino Affonso,22,,-6.15,-37.766667
br,almino afonso,Almino Afonso,22,,-6.15,-37.766667
br,almirante tamandare,Almirante Tamandaré,18,89500,-25.3,-49.316667
br,almirante tamandare,Almirante Tamandaré,23,,-28.1,-52.916667
br,almoco,Almôço,16,,-1.25,-46.916667
br,almofada,Almofada,06,,-3.033333,-39.883333
br,almofala,Almofala,06,,-3.033333,-39.883333
br,aloandia,Aloândia,29,,-17.716667,-49.483333
br,alpercata,Alpercata,13,,-3.655833,-43.483056
br,alpercata,Alpercata,15,,-19.016667,-41.983333
br,alpes da cantareira,Alpes da Cantareira,27,,-23.411082,-46.647701
br,alpes de caieiras,Alpes de Caieiras,27,,-23.378383,-46.677328
br,alpestre,Alpestre,23,,-27.25,-53.033333
br,alpinino,Alpinino,30,,-8.533333,-36.4
br,alpinopolis,Alpinópolis,15,13286,-20.866667,-46.383333
br,alta,Alta,06,,-3.333333,-40.083333
br,alta floresta,Alta Floresta,14,40468,-9.9,-55.9
br,altair,Altair,27,,-20.516667,-49.066667
br,altamira,Altamira,04,,1.666667,-67.2
br,altamira,Altamira,05,,-11.066667,-45.416667
br,altamira,Altamira,07,,-15.616667,-47.666667
br,altamira,Altamira,13,,-5.733333,-45.6
br,altamira,Altamira,15,,-19.55,-43.566667
br,altamira,Altamira,16,70890,-3.2,-52.2
br,altamira,Altamira,20,,-3.767222,-42.578889
br,altamira do maranhao,Altamira do Maranhão,13,,-4.166667,-45.483333
br,alta mir,Alta Mir,07,,-15.616667,-47.666667
br,altaneira,Altaneira,06,,-6.983333,-39.733333
br,altaneira,Altaneira,18,,-23.166667,-52.033333
br,altaneiras,Altaneiras,18,,-23.166667,-52.033333
br,alteado,Alteado,15,,-14.633333,-44.866667
br,alter do chao,Alter do Chão,16,,-2.516667,-54.95
br,alterosa,Alterosa,15,9277,-21.25,-46.133333
br,altidao,Altidão,20,,-4.75,-41.8
br,altinho,Altinho,20,,-3.905278,-42.022222
br,altinho,Altinho,30,11913,-8.483333,-36.066667
br,altino,Altino,15,,-16.416667,-40.333333
br,altinopolis,Altinópolis,27,14616,-21.025278,-47.374444
br,alto alegre,Alto Alegre,01,,-10.616667,-68.183333
br,alto alegre,Alto Alegre,05,,-11.666667,-39.85
br,alto alegre,Alto Alegre,13,,-3.116667,-45.666667
br,alto alegre,Alto Alegre,18,,-22.866667,-51.866667
br,alto alegre,Alto Alegre,20,,-4.016667,-41.433333
br,alto alegre,Alto Alegre,22,,-5.8225,-37.410556
br,alto alegre,Alto Alegre,23,,-28.783333,-53
br,alto alegre,Alto Alegre,24,,-12.133333,-61.858333
br,alto alegre,Alto Alegre,27,,-21.566667,-50.166667
br,alto alegre,Alto Alegre,30,,-7.683333,-39.6
br,alto alegre dos parecis,Alto Alegre dos Parecis,24,,-12.133333,-61.858333
br,alto allegre,Alto Allegre,23,,-30.7232,-52.3438
br,alto,Alto,02,,-9.266667,-38.033333
br,alto,Alto,05,,-12.683333,-39.35
br,alto,Alto,13,,-3.014444,-42.079722
br,alto,Alto,21,,-22.44165,-42.98381
br,alto,Alto,30,,-8.3,-38.566667
br,alto amparo,Alto Amparo,18,,-24.783333,-50.533333
br,alto araguaia,Alto Araguaia,14,8780,-17.316667,-53.2
br,alto atafona,Alto Atafona,26,,-27.12944,-49.506696
br,alto bela alianca,Alto Bela Aliança,26,,-27.2261,-49.5749
br,alto bela vista,Alto Bela Vista,26,,-27.433333,-51.866667
br,alto belo,Alto Belo,20,,-5.983333,-42.316667
br,alto bergamo,Alto Bergamo,08,,-19.766667,-40.45
br,alto bergano,Alto Bergano,08,,-19.766667,-40.45
br,alto bonito,Alto Bonito,02,,-9.383333,-38.066667
br,alto bonito,Alto Bonito,05,,-11.733333,-40.4
br,alto bonito,Alto Bonito,06,,-3.833333,-41.283333
br,alto bonito,Alto Bonito,13,,-3.133333,-42.2
br,alto bonito,Alto Bonito,16,,-4.116667,-51.916667
br,alto bonito,Alto Bonito,20,,-3.555278,-42.243056
br,alto bonito,Alto Bonito,30,,-8.516667,-35.8
br,alto calcado,Alto Calçado,08,,-20.95,-41.616667
br,alto caminho da paz,Alto Caminho da Paz,26,,-26.978721,-49.835948
br,alto caminho do anum,Alto Caminho do Anum,26,,-26.947504,-49.778617
br,alto cananeia,Alto Cananéia,26,,-26.8391,-49.0063
br,alto capim,Alto Capim,15,,-19.85,-41.25
br,alto capivari,Alto Capivari,26,,-26.796349,-49.538043
br,alto cascavel,Alto Cascavel,18,,-25.4,-51.533333
br,alto coite,Alto Coité,14,,-15.783333,-54.333333
br,alto cutia,Alto Cutia,26,,-27.233112,-49.520823
br,alto da betania,Alto da Betânia,20,,-6.033333,-41.983333
br,alto da boa vista,Alto da Boa Vista,21,,-22.96519,-43.276055
br,alto da conceicao,Alto da Conceição,15,,-21.833333,-42.866667
br,alto da conceicao,Alto da Conceição,30,,-8.816667,-36.033333
br,alto da cruz,Alto da Cruz,20,,-5.3,-42.8
br,alto da doca,Alto da Doca,13,,-5.716667,-46.4
br,alto da gloria,Alto da Glória,18,,-25.60307,-50.69018
br,alto da grota funda,Alto da Grota Funda,02,,-9.733333,-37.433333
br,alto da jurema,Alto da Jurema,05,,-11.166667,-41.883333
br,alto da manela,Alto da Manela,20,,-6.333333,-41.95
br,alto da nova,Alto Da Nova,30,,-7.9,-38.683333
br,alto da piedade,Alto da Piedade,27,,-23.747202,-47.426103
br,alto da serra,Alto da Serra,18,,-25.8522,-48.9256
br,alto da serra,Alto da Serra,21,,-22.883333,-44.233333
br,alto da serra,Alto da Serra,27,,-23.783333,-46.316667
br,alto das pedras,Alto das Pedras,13,,-4.05,-44.933333
br,alto das piabas,Alto das Piabas,05,,-14.25,-39.95
br,alto da uniao,Alto da União,23,,-28.45,-53.866667
br,alto da varzea,Alto da Várzea,06,,-6.516667,-38.666667
br,alto de pedra,Alto de Pedra,13,,-4.05,-44.933333
br,alto de pedro,Alto de Pedro,30,,-7.983333,-38.033333
br,alto de santa helena,Alto de Santa Helena,15,,-18.6,-41.616667
br,alto de santa maria,Alto de Santa Maria,08,,-20.15,-40.9
br,alto de santo antonio,Alto de Santo Antônio,30,,-8.75,-36.183333
br,alto de sao antonio,Alto de São Antônio,08,,-19.833333,-40.65
br,alto de sao joao,Alto de São João,15,,-18.566667,-48.316667
br,alto do amparo,Alto do Amparo,18,,-24.783333,-50.533333
br,alto do bagre,Alto do Bagre,06,,-5.016667,-37.933333
br,alto do bonfim,Alto do Bonfim,05,,-9.666667,-43.366667
br,alto do catingueiro,Alto do Catingueiro,30,,-8.916667,-37.316667
br,alto do cobre,Alto do Cobre,18,,-25.116667,-52.133333
br,alto do coqueiro,Alto do Coqueiro,13,,-5.85,-43.183333
br,alto do doca,Alto do Doca,13,,-5.716667,-46.4
br,alto do feto,Alto do Feto,05,,-13.483333,-40.183333
br,alto do gloria,Alto do Glória,18,,-25.615365,-50.692288
br,alto do maranhao,Alto do Maranhão,15,,-20.566667,-43.866667
br,alto do matias,Alto do Matias,30,,-8.366667,-36.433333
br,alto do meio,Alto do Meio,20,,-6.416667,-42
br,alto do melo,Alto do Melo,05,,-12.4,-40.083333
br,alto do peba,Alto do Peba,02,,-9.25,-37.816667
br,alto do rodrigues,Alto do Rodrigues,22,,-5.289444,-36.761111
br,alto do sabia,Alto do Sabía,27,,-23.512018,-47.004669
br,alto dos coelhos,Alto dos Coelhos,17,,-7.366667,-38.166667
br,alto dos coelhos,Alto dos Coelhos,20,,-9.366667,-37.916667
br,alto dos gomes,Alto dos Gomes,17,,-6.566667,-38.45
br,alto dos pareiros,Alto dos Pareiros,20,,-9.366667,-37.916667
br,alto dos pereiras,Alto dos Pereiras,20,,-9.366667,-37.916667
br,alto do velame,Alto do Velame,06,,-4.95,-37.966667
br,alto do vista,Alto do Vista,20,,-6.25,-42.383333
br,alto faxinal,Alto Faxinal,26,,-26.875065,-49.929655
br,alto formoso,Alto Formoso,13,,-3.399444,-42.618611
br,alto formoso,Alto Formoso,20,,-6.9,-42.216667
br,alto garcas,Alto Garças,14,,-16.933333,-53.533333
br,alto garcia,Alto Garcia,26,,-26.9882,-49.085
br,alto grande,Alto Grande,06,,-4.05,-38.9
br,alto grande,Alto Grande,20,,-7.9,-41.3
br,alto grande,Alto Grande,22,,-5.866667,-37.254167
br,alto guandu,Alto Guandu,15,,-20.033333,-41.35
br,alto itauna,Alto Itaúna,08,,-18.1,-40.516667
br,alto itaunas,Alto Itaúnas,08,,-18.1,-40.516667
br,alto itoupava,Alto Itoupava,26,,-27.142099,-49.656141
br,alto lajeado,Alto Lajeado,27,,-20.2,-47.333333
br,alto leblon,Alto Leblon,21,,-22.984687,-43.233344
br,alto lindo,Alto Lindo,13,,-3.405278,-42.605556
br,alto longa,Alto Longá,20,4435,-5.25,-42.2
br,alto madeira,Alto Madeira,24,,-8.8,-63.933333
br,alto maranhao,Alto Maranhão,15,,-20.566667,-43.866667
br,alto mimaso,Alto Mimaso,20,,-5.6,-42.133333
br,alto mutum preto,Alto Mutum Prêto,08,,-19.266667,-40.9
br,altona,Altona,26,,-27.116667,-49.5
br,altonia,Altônia,18,11242,-23.933333,-53.9
br,alto nova esperanca,Alto Nova Esperança,26,,-26.992078,-49.806155
br,alto novo,Alto Novo,05,,-12.266667,-39.333333
br,alto novo,Alto Novo,20,,-5.616667,-42.216667
br,alto padre andre,Alto Padre André,27,,-24.620278,-48.0475
br,alto palmital,Alto Palmital,26,,-26.9007,-49.9145
br,alto paraguai,Alto Paraguai,14,6149,-14.5,-56.516667
br,alto paraguai do diamantino,Alto Paraguai do Diamantino,14,,-14.416667,-56.45
br,alto paraiso de goias,Alto Paraíso de Goiás,29,,-14.116667,-47.516667
br,alto parana,Alto Paraná,18,10888,-23.083333,-52.366667
br,alto parnaiba,Alto Parnaíba,13,,-9.1,-45.95
br,alto patrao mor,Alto Patrão Mor,08,,-19.45,-40.45
br,alto pau d'alho,Alto Pau d'Alho,18,,-23.516667,-50.883333
br,alto perimbo,Alto Perimbó,26,,-27.533333,-49.7
br,alto pimenta,Alto Pimenta,27,,-21.283333,-50.8
br,alto piquiri,Alto Piquiri,18,10039,-24.016667,-53.466667
br,alto pora,Alto Porã,27,,-20.066667,-47.5
br,alto quatipuru,Alto Quatipurú,16,,-1.083333,-46.9
br,alto queiroga,Alto Queiroga,15,,-19.216667,-41.916667
br,alto rafael,Alto Rafael,26,,-26.961875,-49.53026
br,alto ribeirao cambara,Alto Ribeirão Cambará,26,,-26.9134,-49.8399
br,alto ribeirao da laje,Alto Ribeirão da Laje,26,,-26.893382,-49.581326
br,alto rio bravo,Alto Rio Bravo,26,,-28.016667,-49.066667
br,alto rio doce,Alto Rio Doce,15,,-21.033333,-43.416667
br,alto rio forcacao,Alto Rio Forcação,26,,-26.747923,-49.632976
br,alto rio krauel,Alto Rio Krauel,26,,-26.973,-49.9346
br,alto rio novo,Alto Rio Novo,08,,-19.066667,-41.016667
br,alto rio preto,Alto Rio Preto,26,,-26.569104,-49.605047
br,alto rio scharlach,Alto Rio Scharlach,26,,-26.921018,-49.563472
br,altos,Altos,06,,-7.25,-40.15
br,altos,Altos,20,28769,-5.05,-42.466667
br,altos,Altos,22,,-5.3,-35.716667
br,alto santa helena,Alto Santa Helena,15,,-18.6,-41.616667
br,alto santa maria,Alto Santa Maria,08,,-19.9,-40.75
br,alto santa maria do rio doce,Alto Santa Maria do Rio Doce,08,,-19.9,-40.75
br,alto santo,Alto Santo,01,,-9.954722,-67.863889
br,alto santo,Alto Santo,06,,-5.516667,-38.25
br,alto santo antonio,Alto Santo Antônio,30,,-8.75,-36.183333
br,alto sao francisco,Alto São Francisco,08,,-18.85,-41.033333
br,alto sao joao,Alto São João,15,,-18.566667,-48.316667
br,alto seco,Alto Sêco,05,,-13.083333,-39.516667
br,alto sereno,Alto Sereno,20,,-6.983333,-42.4
br,alto sucuriu,Alto Sucuriú,11,,-19.316667,-52.783333
br,altos verdes,Altos Verdes,28,,-10.416667,-37.65
br,alto turi,Alto Turi,13,,-2.9,-45.633333
br,alto uru,Alto Uru,26,,-27.0151,-49.6723
br,alto uruguai,Alto Uruguai,23,,-27.3,-54.133333
br,alto uruguay,Alto Uruguay,23,,-27.3,-54.133333
br,alto verde,Alto Verde,20,,-5.633333,-41.283333
br,alto vere,Alto Verê,18,,-25.916667,-52.983333
br,alto vigante,Alto Vigante,26,,-26.825337,-49.608486
br,alto xarqueada,Alto Xarqueada,23,,-29.1729,-51.2185
br,alunas,Alunas,05,,-13.2,-39.3
br,alvacao,Alvação,13,,-6.7,-43.883333
br,alvacao,Alvação,15,,-16.432778,-44.283611
br,alvalhado,Alvalhado,20,,-7.266667,-41.6
br,alvaraes,Alvarães,04,5527,-3.2,-64.833333
br,alvareda,Alvareda,15,,-20.716667,-42.15
br,alvarenga,Alvarenga,15,,-19.416667,-41.733333
br,alvares florence,Álvares Florence,27,,-20.316667,-49.9
br,alvares machado,Álvares Machado,27,21642,-22.133333,-51.5
br,alvaro,Álvaro,04,,-7.283333,-72.35
br,alvaro de carvalho,Álvaro de Carvalho,27,,-22.1,-49.716667
br,alvaro melo,Álvaro Melo,15,,-21.166667,-47.066667
br,alvejado,Alvejado,22,,-5.938056,-36.790833
br,alvellos,Alvellos,04,,-4.1,-63.3
br,alvelos,Alvelos,04,,-4.1,-63.3
br,alves,Alves,15,,-18.166667,-42.75
br,alves,Alves,23,,-29.366667,-54.733333
br,alves,Alves,27,,-23.671212,-47.664083
br,alves lima,Alves Lima,27,,-21.916667,-50.233333
br,alvim fabricio,Alvim Fabrício,15,,-20.85,-45.633333
br,alvimlandia,Alvimlândia,27,,-22.45,-49.766667
br,alvinandia,Alvinândia,27,,-22.45,-49.766667
br,alvinlandia,Alvinlândia,27,,-22.45,-49.766667
br,alvino,Alvino,05,,-12.183333,-37.933333
br,alvinopolis,Alvinópolis,15,12068,-20.1,-43.05
br,alvino vieira,Alvino Vieira,15,,-17.066667,-41.65
br,alvorada,Alvorada,04,,-5.431389,-60.459167
br,alvorada,Alvorada,05,,-16.516667,-39.35
br,alvorada,Alvorada,06,,-5.933333,-40.55
br,alvorada,Alvorada,11,,-17.216667,-56.033333
br,alvorada,Alvorada,15,,-19.333333,-48.333333
br,alvorada,Alvorada,20,,-6.066667,-41.883333
br,alvorada,Alvorada,22,,-5.95,-35.55
br,alvorada,Alvorada,23,,-28.666667,-52.15
br,alvorada,Alvorada,31,10232,-12.47,-49.082222
br,alvorada do sul,Alvorada do Sul,18,,-22.783333,-51.183333
br,amacia,Amaciã,04,,-5.605833,-64.215278
br,amadeu amaral,Amadeu Amaral,27,,-22.366667,-50.05
br,amado bahia,Amado Bahia,05,,-12.55,-38.3
br,amado baia,Amado Baía,05,,-12.55,-38.3
br,amados,Amados,18,,-25.855418,-50.043221
br,amambai,Amambaí,11,20261,-23.083333,-55.216667
br,amambera,Amamberá,23,,-28.366667,-53.25
br,amana,Amaná,08,,-20.933333,-41.15
br,amanaiara,Amanaiara,06,,-4.066667,-40.533333
br,amanaju,Amanaju,06,,-5.5,-39.366667
br,aman,Aman,21,,-22.3922,-44.4823
br,amanari,Amanari,06,,-4.016667,-38.866667
br,amancio bento,Amâncio Bento,15,,-20.133333,-45.133333
br,amanda,Amanda,15,,-19.366667,-43.883333
br,amandaba,Amandaba,27,,-21.083333,-51.15
br,amanhece,Amanhece,15,,-18.55,-48.233333
br,amaniu,Amaniú,05,,-10.333333,-41.833333
br,amaniutuba,Amaniutuba,06,,-6.666667,-38.883333
br,amapa,Amapá,01,,-10.3,-69.3
br,amapa,Amapá,03,6256,2.05,-50.8
br,amapa,Amapá,04,,-5.45,-60.583333
br,amapa,Amapá,13,,-3.140556,-43.293333
br,amapa,Amapá,20,,-5.666667,-41.716667
br,amapari,Amapari,03,,.616667,-51.65
br,amapora,Amaporã,18,,-23.033333,-52.833333
br,amaragi,Amaragi,30,,-8.4,-35.45
br,amaragi d'agua,Amaragi d'Agua,30,,-8.4,-35.433333
br,amaragy,Amaragy,30,,-8.4,-35.45
br,amaraji,Amaraji,30,16660,-8.4,-35.45
br,amaraji d'agua,Amaraji d'Agua,30,,-8.4,-35.433333
br,amaral ferrador,Amaral Ferrador,23,,-30.866667,-52.25
br,amaralina,Amaralina,05,,-13.016667,-38.466667
br,amarante,Amarante,15,,-20.316667,-43.716667
br,amarante,Amarante,20,8828,-6.233333,-42.833333
br,amarante do grajau,Amarante do Grajau,13,,-5.6,-46.75
br,amarante do maranhao,Amarante do Maranhão,13,11510,-5.6,-46.75
br,amarantina,Amarantina,15,,-20.316667,-43.716667
br,amarela,Amarela,06,,-2.936111,-41.042222
br,amarelao,Amarelão,22,,-5.5,-35.9
br,amarelo,Amarelo,20,,-6.866667,-41.016667
br,amarelo ferrado,Amarelo Ferrado,20,,-6.85,-40.933333
br,amarelos,Amarelos,08,,-20.533333,-40.483333
br,amargosa,Amargosa,02,,-9.616667,-37.183333
br,amargosa,Amargosa,05,22531,-13.033333,-39.6
br,amargosa,Amargosa,20,,-6.1,-41.183333
br,amargoso,Amargoso,22,,-5.229722,-36.562778
br,amarilis,Amarílis,27,,-22.3,-50.283333
br,amaro,Amaro,06,,-6.816667,-39.683333
br,amaro,Amaro,11,,-21.716667,-56.583333
br,amaro,Amaro,18,,-25.889924,-50.027985
br,amaro,Amaro,25,,4.5,-60.116667
br,amaro,Amaro,30,,-8.783333,-37.05
br,amaro leite,Amaro Leite,13,,-8.966667,-46.866667
br,amaro leite,Amaro Leite,29,,-13.966667,-49.15
br,amaropolis,Amarópolis,23,,-29.933333,-51.9
br,amaro ribeiro,Amaro Ribeiro,15,,-20.7,-43.816667
br,amaro tempo,Amaro Tempo,15,,-17.083333,-40.816667
br,amarracao,Amarração,05,,-13.116667,-40.233333
br,amarracao,Amarracão,06,,-6.666667,-38.666667
br,amarracao,Amarração,20,,-2.883611,-41.666111
br,amatari,Amatari,04,,-3.266667,-58.883333
br,amataura,Amataurá,04,,-3.483333,-68.1
br,amatura,Amaturá,04,,-3.483333,-68.1
br,amazonas,Amazonas,05,,-12.466667,-40.883333
br,amazonas,Amazonas,17,,-6.1,-37.266667
br,amazonas,Amazonas,21,,-22.05,-41.833333
br,amazonas,Amazonas,22,,-5.291111,-36.009722
br,amazonia,Amazônia,05,,-12.166667,-39.333333
br,ambe,Ambé,03,,.333333,-50.916667
br,ambe,Ambé,16,,-3.15,-52.216667
br,ambicao,Ambição,11,,-21.366667,-56.216667
br,ambo,Ambó,17,,-7.683333,-36.216667
br,ambo,Ambó,30,,-7.383333,-37.266667
br,ambrosio aires,Ambrósio Aires,04,,-3.583333,-59.133333
br,ambrosio,Ambrósio,04,,-2.866667,-68.35
br,ambrosio,Ambrósio,05,,-10.983333,-39.216667
br,ambrosio,Ambrósio,20,,-7.85,-44.55
br,ambrosio,Ambrósio,25,,2.8,-60.25
br,ambrosio,Ambrósio,26,,-28.033333,-48.7
br,ambrosio,Ambrósio,30,,-8.816667,-38.566667
br,ambrosio vicente,Ambrósio Vicente,15,,-17.3,-41.683333
br,ambrozio,Ambrózio,30,,-7.583333,-39.866667
br,ameixa,Ameixa,17,,-6.516667,-37.7
br,ameixa,Ameixa,22,,-5.283333,-35.866667
br,ameixa,Ameixa,30,,-8.1,-35.783333
br,ameixas,Ameixas,30,,-8.1,-35.783333
br,amelia,Amelia,01,,-10.466667,-68.05
br,ameliopolis,Ameliópolis,27,,-21.833333,-51.3
br,amendoim,Amendoim,15,,-17.933333,-45.666667
br,america,América,04,,-5.166667,-60.433333
br,america,América,05,,-13.683333,-40.116667
br,america,América,08,,-21.166667,-41.433333
br,america,América,18,,-25.516667,-48.883333
br,america de baixo,América de Baixo,18,,-25.48983,-48.854266
br,america de cima,América de Cima,18,,-25.47675,-48.863254
br,america dourada,América Dourada,05,14018,-11.466667,-41.433333
br,americana,Americana,27,196026,-22.740278,-47.334444
br,americaninhas,Americaninhas,15,,-17.133333,-41.2
br,americano,Americano,16,,-1.316667,-48.066667
br,americano,Americano,27,,-23.325906,-47.776798
br,americano brasil,Americano Brasil,29,,-16.333333,-50.016667
br,americano do brasil,Americano do Brasil,29,,-16.333333,-50.016667
br,americas park,Américas Park,21,,-23.000375,-43.390957
br,americo alves,Américo Alves,05,,-9.683333,-41.583333
br,americo brasiliense,Américo Brasiliense,27,34332,-21.716667,-48.116667
br,americo de campos,Américo de Campos,27,,-20.3,-49.716667
br,americo v rech,Américo V Rech,23,,-29.0491,-51.1167
br,ametista,Ametista,08,,-18.883333,-41.1
br,ametista,Ametista,15,,-18.683333,-41.2
br,ametistas,Ametistas,23,,-28.316667,-52.05
br,amianto,Amianto,05,,-12.25,-40.666667
br,amianto,Amianto,15,,-20.116,-43.8795
br,amizade,Amizade,16,,-1.166667,-49.633333
br,amola faca,Amola Faca,26,,-27.783333,-50.483333
br,amolar,Amolar,02,,-9.183333,-36.133333
br,amolar,Amolar,11,,-18.016667,-57.5
br,amolar,Amolar,20,,-7.383333,-42.916667
br,amolar,Amolar,30,,-7.75,-39.283333
br,amontada,Amontada,06,13520,-3.383333,-39.833333
br,amontoado,Amontoado,21,,-21.333333,-41.05
br,amora preta,Amora Preta,18,,-23.899618,-50.294824
br,amoreira,Amoreira,05,,-13.95,-41.966667
br,amoreira,Amoreira,18,,-23.466667,-50.75
br,amoreira,Amoreira,27,,-20.433333,-48.583333
br,amorim,Amorim,08,,-20.533333,-41.416667
br,amorinha,Amorinha,18,,-23.889158,-50.243729
br,amorinopolis,Amorinópolis,29,,-16.6,-51.133333
br,amoro,Amoro,05,,-12.916667,-39.266667
br,amoroso,Amoroso,22,,-5.033333,-37.383333
br,amparinho,Amparinho,17,,-7.65,-37.016667
br,amparo,Amparo,05,,-11.05,-38.433333
br,amparo,Amparo,11,,-19.45,-55.566667
br,amparo,Amparo,15,,-19.7,-43.466667
br,amparo,Amparo,17,,-7.566667,-37.05
br,amparo,Amparo,18,,-24.783333,-50.533333
br,amparo,Amparo,21,,-21.75,-41.133333
br,amparo,Amparo,27,41498,-22.7,-46.783333
br,amparo,Ampáro,28,,-10.133333,-36.916667
br,amparo da serra,Amparo da Serra,15,,-20.516667,-42.816667
br,amparo de sao francisco,Amparo de São Francisco,28,,-10.133333,-36.916667
br,amparo do sao francisco,Amparo do São Francisco,28,,-10.133333,-36.916667
br,amparo do sitio,Amparo do Sítio,15,,-16.35,-41.983333
br,amparro da serra,Amparro da Serra,15,,-20.516667,-42.816667
br,ampere,Ampère,18,14275,-25.916667,-53.466667
br,amuamuaipe,Amuamuaipe,16,,1.916667,-54.966667
br,amuamuatpe,Amuamuatpe,16,,1.916667,-54.966667
br,amuri camaizore,Amuri Camaizorê,14,,-14.75,-58.05
br,amuri camalzore,Amuri Camalzorê,14,,-14.75,-58.05
br,anacetaba,Anacetaba,06,,-3.6,-38.966667
br,anacleto,Anacleto,15,,-20.033333,-46.366667
br,anacory,Anacory,04,,-5.664167,-63.131667
br,anadia,Anadia,02,9574,-9.7,-36.3
br,ana dias,Ana Dias,27,,-24.3,-47.066667
br,anafama,Anafamá,04,,-7.083333,-64.6
br,ana farias da luz,Ana Farias da Luz,23,,-30.583333,-55.816667
br,anage,Anagé,05,3762,-14.6,-41.133333
br,ana gomes,Ana Gomes,15,,-20.133333,-45
br,anaja,Anajá,13,,-3.673611,-43.2975
br,anaja,Anajá,20,,-3.599167,-42.444167
br,anaja dos pereiras,Anajá dos Pereiras,20,,-3.544444,-42.46
br,anajas,Anajás,16,,-.983333,-49.95
br,anajateua,Anajateua,16,,-3.1,-46.7
br,anajatuba,Anajatuba,04,,-7.016667,-57.833333
br,anajatuba,Anajatuba,13,5627,-3.266667,-44.616667
br,anajatuba,Anajatuba,16,,-4.116667,-56.6
br,anajazal,Anajazal,13,,-3.554722,-43.175
br,anajaz,Anajaz,16,,-.983333,-49.95
br,anajazinho,Anajazinho,13,,-3.371667,-42.503056
br,anaje,Anaje,05,,-14.6,-41.133333
br,analandia,Analândia,27,,-22.133333,-47.65
br,anama,Anamã,04,,-3.583333,-61.366667
br,anaman,Anamãn,04,,-3.583333,-61.366667
br,ananas,Ananás,13,,-3.659167,-43.297222
br,ananas,Ananás,20,,-9.905833,-45.038611
br,ananas,Ananás,31,8874,-6.360278,-47.91
br,ananatuba,Ananatuba,16,,-.583333,-49.183333
br,ananindeua,Ananindeua,16,433956,-1.366667,-48.383333
br,anapolandia,Anapolândia,31,,-7.1,-48.133333
br,anapolis,Anápolis,27,,-22.133333,-47.65
br,anapolis,Anápolis,28,,-10.733333,-37.816667
br,anapolis,Anápolis,29,319592,-16.333333,-48.966667
br,anapurus,Anapurus,13,,-3.205278,-42.408889
br,ana rech,Ana Rech,23,,-29.1119,-51.087
br,anastacio,Anastácio,11,18817,-20.516667,-55.8
br,anastacio,Anastácio,15,,-17.366667,-41.3
br,anastacio,Anastácio,20,,-6.933333,-40.883333
br,anati,Anati,04,,1.016667,-67.45
br,anaua,Anauá,06,,-7.25,-38.633333
br,anaurilandia,Anaurilândia,11,,-22.05,-52.75
br,ancari,Ancari,04,,-5.664167,-63.131667
br,anchieta,Anchieta,04,,.566667,-69.8
br,anchieta,Anchieta,05,,-13.15,-41.8
br,anchieta,Anchieta,08,,-20.8,-40.65
br,anchieta,Anchieta,20,,-6.866667,-40.75
br,anchieta,Anchieta,21,,-22.82327,-43.396679
br,anchieta,Anchieta,26,,-26.55,-53.283333
br,ancora,Ancora,17,,-7.55,-37.783333
br,ancori,Ancori,04,,-5.664167,-63.131667
br,ancuri,Ancuri,06,,-3.866667,-38.516667
br,andaia,Andaiá,05,,-12.866667,-39.35
br,andaia,Andaiá,15,,-20.116667,-41.566667
br,andarahy,Andarahy,04,,-5.65,-63.183333
br,andarahy,Andarahy,05,,-12.8,-41.333333
br,andarai,Andaraí,04,,-5.589444,-63.289167
br,andarai,Andaraí,05,,-11.733333,-40.616667
br,andarai,Andaraí,13,,-5.083333,-45.883333
br,andarai,Andaraí,21,,-22.931086,-43.253221
br,andarico,Andarico,11,,-22.1,-56.266667
br,anderai,Anderai,04,,-5.65,-63.183333
br,andes,Andes,08,,-20.983333,-41.283333
br,andira,Andira,04,,-2.85,-57
br,andira,Andirá,18,20758,-23.0488,-50.2276
br,andiroba,Andiroba,04,,-3.551111,-60.471944
br,andiroba,Andiroba,13,,-3.5075,-43.246944
br,andirobal,Andirobal,13,,-3.8,-45.166667
br,andorinha,Andorinha,05,,-10.366667,-39.816667
br,andorinhas,Andorinhas,18,,-23.833333,-51.266667
br,andradas,Andradas,15,26773,-22.066667,-46.566667
br,andradas,Andradas,21,,-22.276375,-42.969252
br,andrade,Andrade,04,,-4.508889,-63.556944
br,andrade,Andrade,15,,-21.016667,-42.966667
br,andrade,Andrade,21,,-22.243056,-43.424167
br,andrade araujo,Andrade Araujo,21,,-22.754546,-43.416973
br,andrade costa,Andrade Costa,21,,-22.254167,-43.364444
br,andrade pinto,Andrade Pinto,21,,-22.243056,-43.424167
br,andradina,Andradina,27,52407,-20.9,-51.383333
br,andre,André,20,,-7.083333,-40.75
br,andre da rocha,André da Rocha,23,,-28.633333,-51.566667
br,andre fernandes,André Fernandes,15,,-15.95,-41.5
br,andrelandia,Andrelândia,15,10079,-21.733333,-44.3
br,andre machado,André Machado,23,,-29.9,-56.516667
br,andrequice,Andrequicé,15,,-18.283333,-44.983333
br,andre resende,André Resende,15,,-21.566667,-45.183333
br,andre rodrigues,André Rodrigues,17,,-7.5,-36.5
br,andresa,Andresa,17,,-7.466667,-38.05
br,andres,Andrés,27,,-23.279659,-47.874478
br,andre siqueira,André Siqueira,31,,-11.733333,-49.083333
br,andreza,Andreza,06,,-4.333333,-38.15
br,andreza,Andreza,13,,-3.111944,-42.864444
br,andreza,Andreza,17,,-7.466667,-38.05
br,anel,Anel,02,,-9.35,-36.3
br,anelio,Anélio,16,,-1.983333,-46.55
br,anemutu,Anemutu,25,,4.083333,-59.883333
br,angai,Angaí,18,,-25.5122,-50.45927
br,angatuba,Angatuba,27,14476,-23.483333,-48.416667
br,angaturama,Angaturama,15,,-21.483333,-42.4
br,angeim,Angeim,20,,-5.366667,-41.2
br,angelica,Angelica,17,,-7.25,-37.5
br,angelim,Angelim,03,,1.35,-50.933333
br,angelim,Angelim,05,,-12.2,-40.25
br,angelim,Angelim,06,,-3.283333,-39.433333
br,angelim,Angelim,08,,-18.5,-39.866667
br,angelim,Angelim,13,,-3.048056,-42.1475
br,angelim,Angelim,17,,-7.216667,-38.5
br,angelim,Angelim,18,,-24.785933,-48.879113
br,angelim,Angelim,20,,-10.833333,-44.833333
br,angelim,Angelim,21,,-22.666667,-42.066667
br,angelim,Angelim,22,,-5.400278,-36.277222
br,angelim,Angelim,30,,-8.883333,-36.283333
br,angelina,Angelina,26,,-27.583333,-48.983333
br,angelo,Ângelo,05,,-17.366667,-40.333333
br,angelo,Ângelo,15,,-16.133333,-41.633333
br,angelo rios,Ángelo Rios,15,,-20.233333,-45.1
br,angelo saocal,Ángelo Saocal,23,,-29.766667,-54.083333
br,angical,Angical,05,,-11.233333,-41.75
br,angical,Angical,13,,-3.3575,-43.150556
br,angical,Angical,15,,-15.766667,-45.683333
br,angical,Angical,20,,-3.533333,-41.666667
br,angical,Angical,30,,-7.766667,-39.733333
br,angical,Angical,31,,-11.9,-48.55
br,angical do piaui,Angical do Piauí,20,,-6.083333,-42.733333
br,angicalinho,Angicalinho,13,,-3.345,-43.1375
br,angicao,Angicão,05,,-11.1,-41.383333
br,angicao,Angicão,15,,-15.45,-41.85
br,angico,Angico,05,,-10.033333,-43.1
br,angico,Angico,06,,-3.5,-39.966667
br,angico,Angico,13,,-3.753056,-43.449444
br,angico,Angico,17,,-6.85,-38.1
br,angico,Angico,22,,-5.166667,-35.683333
br,angico,Angico,27,,-21.783333,-47.733333
br,angico,Angico,28,,-10.266667,-37.55
br,angico,Angico,30,,-8.333333,-37.616667
br,angico,Angico,31,,-6.401389,-47.693611
br,angico branco,Angico Branco,13,,-3.060278,-42.180278
br,angico branco,Angico Branco,20,,-3.458333,-42.488889
br,angico branco,Angico Branco,31,,-6.533333,-47.45
br,angico branco i,Angico Branco I,20,,-3.424167,-42.198611
br,angico branco ii,Angico Branco II,20,,-3.519167,-42.0575
br,angico da escora,Angico da Escora,17,,-7.266667,-38.45
br,angico das araras,Angico das Araras,20,,-3.891111,-42.298889
br,angico de minas,Angico de Minas,15,,-16.083333,-44.416944
br,angico grosso,Angico Grosso,06,,-5.2,-37.966667
br,angicos,Angicos,05,,-10.183333,-38.866667
br,angicos,Angicos,06,,-4.466667,-41.066667
br,angicos,Angicos,13,,-8.466667,-45.966667
br,angicos,Angicos,15,,-16.083333,-44.416944
br,angicos,Angicos,17,,-7.116667,-37.716667
br,angicos,Angicos,20,,-6.966667,-41.716667
br,angicos,Angicos,22,7729,-5.666389,-36.600833
br,angicos,Angicos,30,,-7.816667,-37.883333
br,angico torto,Angico Tôrto,17,,-7.133333,-38.1
br,angico torto,Angico Torto,22,,-6.416667,-36.083333
br,angico torto,Angico Torto,30,,-7.883333,-39.016667
br,angico velho,Angico Velho,06,,-7.05,-40.416667
br,angola,Angola,15,,-21.533333,-46.433333
br,angola,Angola,21,,-21.25,-42.083333
br,angoritaba,Angoritaba,15,,-21.033333,-43.583333
br,angostura,Angostura,15,,-21.75,-42.683333
br,angra,Angra,21,,-23,-44.3
br,angra dos reis,Angra dos Reis,21,153637,-23,-44.3
br,anguera,Anguera,05,,-12.15,-39.216667
br,anguereta,Angueretá,15,,-19.15,-44.633333
br,angulo,Ângulo,18,,-23.183333,-51.916667
br,angustura,Angustura,15,,-21.75,-42.683333
br,angustura,Angustura,24,,-8.9,-62.266667
br,anhaia,Anháia,18,,-25.5245,-48.843
br,anhandui,Anhanduí,11,,-20.983333,-54.483333
br,anhanduizinho,Anhanduìzinho,11,,-20.75,-54.383333
br,anhanga,Anhangá,16,,-1.183333,-47.816667
br,anhanguara,Anhanguara,27,,-24.573974,-48.409735
br,anhanguera,Anhanguera,29,,-18.316667,-48.233333
br,anhema,Anhema,27,,-24.8515,-48.4807
br,anhembi,Anhembi,27,,-22.8,-48.116667
br,anhemby,Anhemby,27,,-22.8,-48.116667
br,anhoma,Anhoma,11,,-20.3,-55.916667
br,anhuma,Anhuma,11,,-20.3,-55.916667
br,anhumas,Anhumas,02,,-9.15,-36.116667
br,anhumas,Anhumas,11,,-20.3,-55.916667
br,anhumas,Anhumas,14,,-16.933333,-54.666667
br,anhumas,Anhumas,27,,-22.283333,-51.416667
br,anibal,Aníbal,15,,-20.066667,-41.433333
br,aniceto,Aniceto,06,,-5.166667,-40.016667
br,aniceto,Aniceto,15,,-15.033333,-45.633333
br,anicuns,Anicuns,29,15685,-16.466667,-49.966667
br,anil,Anil,06,,-4.05,-40.45
br,anil,Anil,13,,-2.533333,-44.233333
br,anil,Anil,21,,-22.95402,-43.334707
br,anil,Anil,30,,-8.466667,-38.533333
br,aninga,Aninga,13,,-3.226667,-42.205556
br,aninga,Aninga,20,,-3.424722,-42.185278
br,aningal,Aningal,25,,2.833333,-61.016667
br,aningas,Aningas,06,,-2.933333,-40.166667
br,aningas,Aningas,22,,-5.533333,-35.3
br,aninha,Aninha,15,,-14.466667,-44.733333
br,aninim,Aninim,04,,-4.035278,-59.379722
br,anisio de morais,Anisio de Morais,27,,-23.224167,-47.701389
br,anita garibaldi,Anita Garibaldi,26,,-27.616667,-51.083333
br,anitapolis,Anitápolis,26,,-27.9,-49.133333
br,anjelina,Anjelina,26,,-27.583333,-48.983333
br,anjico,Anjico,30,,-8.65,-38.7
br,anjico velho,Anjico Velho,22,,-5.233333,-35.75
br,anjinho,Anjinho,06,,-7.1,-39.883333
br,anjinho,Anjinho,20,,-3.956389,-42.083611
br,anna dias,Anna Dias,27,,-24.3,-47.066667
br,annapolis,Annapolis,27,,-22.133333,-47.65
br,annapolis,Annapolis,28,,-10.733333,-37.816667
br,annapolis,Annapolis,29,,-16.333333,-48.966667
br,anna reck,Anna Reck,23,,-29.1119,-51.087
br,annel,Annel,02,,-9.35,-36.3
br,annitapolis,Annitapolis,26,,-27.9,-49.133333
br,annory,Annory,04,,-3.783333,-61.633333
br,ano nuevo,Ano Nuevo,11,,-21.8,-54.15
br,anori,Anori,04,8064,-3.783333,-61.633333
br,anory,Anory,04,,-3.783333,-61.633333
br,anselmi,Anselmi,23,,-32.85,-52.733333
br,anselmos,Anselmos,27,,-23.057519,-47.513737
br,anta,Anta,05,,-15.75,-41.1
br,anta,Anta,15,,-19.5,-41.966667
br,anta,Anta,18,,-23.881522,-50.004666
br,anta gorda,Anta Gorda,05,,-12.716667,-39.9
br,anta gorda,Anta Gorda,13,,-3.4975,-43.299444
br,anta gorda,Anta Gorda,23,,-28.983333,-52.016667
br,anta gorda,Anta Gorda,26,,-26.95,-51.233333
br,anta gorda,Anta Gorda,27,,-24.683173,-48.745738
br,antari,Antari,05,,-12.15,-42.033333
br,antas,Antas,05,,-10.383333,-38.333333
br,antas,Antas,06,,-7.25,-38.666667
br,antas,Antas,18,,-24.133333,-50.516667
br,antas,Antas,21,,-22.335772,-42.788471
br,antas,Antas,26,,-26.6,-52.3
br,antas,Antas,27,,-20.866667,-47.25
br,antenor navarro,Antenor Navarro,17,,-6.733333,-38.45
br,antero,Antero,15,,-16.65,-40.533333
br,antiga aldeia pacarandi,Antiga Aldeia Pacarandi,16,,-8.455556,-49.583333
br,antiga,Antiga,11,,-22.216667,-55.883333
br,antiga colonia dourados,Antiga Coloniá Dourados,11,,-22.216667,-55.883333
br,antiga colonia sao lourenco,Antiga Colônia São Lourenço,14,,-16.666667,-55.25
br,antiga senzala,Antiga Senzala,15,,-20.1178,-43.9945
br,antigo aldeia do ignacio,Antigo Aldeia do Ignacio,14,,-16.866667,-57.483333
br,antigo aldeia lania,Antigo Aldeia Lania,11,,-20.55,-55.983333
br,antigo arraial de lavrinhas,Antigo Arraial de Lavrinhas,14,,-15.3,-59.233333
br,antigo arraial de santa barbara,Antigo Arraial de Santa Barbara,14,,-15.65,-59.15
br,antigo porto cunha bueno,Antigo Pôrto Cunha Bueno,27,,-21.7,-47.816667
br,antigo santa gertrudes,Antigo Santa Gertrudes,11,,-21.516667,-55.433333
br,antimari,Antimari,04,,-9.05,-67.4
br,antinha,Antinha,18,,-24.95,-49
br,antioa colonia sao lourenco,Antióa Colonia São Lourenço,14,,-16.666667,-55.25
br,antoniao,Antonião,13,,-4.816667,-45.316667
br,antonina,Antonina,18,16152,-25.4314,-48.7184
br,antonina,Antonina,27,,-20.566667,-50.616667
br,antonina,Antonina,31,,-6.1,-48.333333
br,antonina do norte,Antonina do Norte,06,,-6.716667,-39.966667
br,antonio alves,Antônio Alves,15,,-19.05,-48.583333
br,antonio amaral,Antônio Amaral,15,,-16.25,-41.6
br,antonio,Antônio,05,,-12.666667,-41.1
br,antonio,Antônio,25,,2.616667,-60.366667
br,antonio arangueiro,Antônio Arangueiro,15,,-19.033333,-48.433333
br,antonio barbosa,Antônio Barbosa,15,,-20.083333,-46.3
br,antonio bastos,Antônio Bastos,23,,-29.983333,-56.883333
br,antonio bernardo,Antônio Bernardo,05,,-16.933333,-39.766667
br,antonio bezerra,Antônio Bezerra,06,,-3.733333,-38.583333
br,antonio borges,Antônio Borges,05,,-12.333333,-41.066667
br,antonio borges,Antônio Borges,15,,-20.633333,-46.8
br,antonio borges,Antônio Borges,31,,-6.2,-48.416667
br,antonio bueno,Antônio Bueno,15,,-20.983333,-46.666667
br,antonio caetano,Antônio Caetano,15,,-18.466667,-48.033333
br,antonio candido da silva,Antônio Cândido da Silva,14,,-15.966667,-54.616667
br,antonio carlos,Antônio Carlos,15,,-18.4,-48.366667
br,antonio carlos,Antônio Carlos,26,,-27.466667,-48.85
br,antonio carmo,Antônio Carmo,15,,-17.266667,-41.916667
br,antonio cipriano,Antônio Cipriano,05,,-16.433333,-39.166667
br,antonio correia,Antônio Correia,15,,-16.883333,-41.516667
br,antonio da barra,Antonio da Barra,27,,-21.55,-46.533333
br,antonio da luiz,Antonio da Luiz,16,,-8.7,-49.616667
br,antonio danieli,Antonio Danieli,23,,-29.0981,-51.2171
br,antonio de alencar,Antônio de Alencar,31,,-6.8,-48.033333
br,antonio dias,Antônio Dias,15,,-16.516667,-41.883333
br,antonio diogo,Antônio Diogo,06,,-4.3,-38.75
br,antonio do azeite,Antônio do Azeite,15,,-21.55,-44.183333
br,antonio domingues,Antônio Domingues,15,,-20.7,-46.416667
br,antonio dos santos,Antônio dos Santos,15,,-19.783333,-43.566667
br,antonio duarte,Antônio Duarte,15,,-17.266667,-41.7
br,antonio eduardo,Antônio Eduardo,15,,-20.883333,-45.866667
br,antonio elias,Antônio Elias,15,,-16.583333,-40.733333
br,antonio fernandes,Antônio Fernandes,15,,-16.933333,-40.966667
br,antonio ferraz,Antônio Ferraz,15,,-16.266667,-40.333333
br,antonio ferreira,Antônio Ferreira,15,,-17.916667,-41.916667
br,antonio ferreira,Antônio Ferreira,17,,-7.016667,-36.55
br,antonio filinto,Antônio Filinto,04,,-1.75,-59.616667
br,antonio fonseca,Antônio Fonseca,15,,-20.283333,-45.283333
br,antonio fortunato,Antônio Fortunato,15,,-20.583333,-46.2
br,antonio franco,Antônio Franco,15,,-19.466667,-49.983333
br,antonio frotao,Antônio Frotão,15,,-20.65,-46.05
br,antonio garcia,Antônio Garcia,15,,-20.966667,-46.733333
br,antonio goncalves,Antônio Gonçalves,15,,-16.366667,-41.683333
br,antonio gusmao,Antônio Gusmão,15,,-16.416667,-41.85
br,antonio inacio,Antônio Inácio,26,,-27.9,-50.75
br,antonio joao,Antônio João,11,,-23.25,-55.516667
br,antonio laurindo,Antônio Laurindo,15,,-20.466667,-46.566667
br,antonio lemos,Antônio Lemos,16,,-1.366667,-50.833333
br,antonio lima,Antônio Lima,15,,-17.366667,-40.666667
br,antonio lopes,Antônio Lopes,15,,-21.3,-46.4
br,antonio manuel,Antônio Manuel,29,,-15.65,-48.6
br,antonio mariano,Antônio Mariano,21,,-21.183333,-41.633333
br,antonio martins,Antônio Martins,22,,-6.2,-37.883333
br,antonio matos,Antônio Matos,16,,-1.116667,-47.95
br,antonio melo,Antônio Melo,15,,-16.85,-41.75
br,antonio nonato,Antônio Nonato,05,,-12.4,-40.7
br,antonio olimpio,Antônio Olímpio,15,,-14.4,-44.383333
br,antonio olinto,Antônio Olinto,15,,-16.783333,-43.833333
br,antonio olinto,Antônio Olinto,18,,-25.98521,-50.196377
br,antonio olinto,Antônio Olinto,30,,-8.316667,-36.3
br,antonio olyntho,Antonio Olyntho,30,,-8.316667,-36.3
br,antonio ourives,Antônio Ourives,15,,-21.65,-45.566667
br,antonio pardinho,Antônio Pardinho,15,,-16.2,-39.933333
br,antonio pereira,Antônio Pereira,15,,-20.3,-43.483333
br,antonio pinheiro,Antônio Pinheiro,05,,-16.516667,-39.933333
br,antonio pinto,Antônio Pinto,14,,-15.916667,-54.666667
br,antonio pinto,Antônio Pinto,15,,-17.75,-40.8
br,antonio porto jatai,Antônio Pôrto Jataí,27,,-21.633333,-47.833333
br,antonio prado,Antônio Prado,15,,-21.016667,-42.1
br,antonio prado,Antônio Prado,23,,-28.85,-51.283333
br,antonio primo da silva,Antônio Primo da Silva,15,,-14.783333,-43.233333
br,antonio raimundo,Antônio Raimundo,05,,-12.333333,-41.05
br,antonio rocha,Antônio Rocha,15,,-17.416667,-41.5
br,antonio rocha,Antônio Rocha,21,,-22.65,-44.15
br,antonio rodrigues,Antônio Rodrigues,15,,-17.166667,-40.733333
br,antonio rodrigues de sousa,Antônio Rodrigues de Sousa,15,,-17.583333,-40.616667
br,antonio rondon,Antônio Rondon,11,,-19.483333,-56.416667
br,antonio roque,Antônio Roque,15,,-21.1,-46.616667
br,antonio rosa,Antônio Rosa,15,,-20.25,-45.516667
br,antonio sampaio,Antônio Sampaio,05,,-12.416667,-41.066667
br,antonio severino,Antônio Severino,15,,-20.233333,-45.35
br,antonio spineli,Antônio Spineli,15,,-21.166667,-46.666667
br,antonio taborda,Antônio Taborda,18,,-25.083333,-53.866667
br,antonio varga,Antônio Varga,08,,-20.333333,-41.25
br,antonio velho,Antônio Velho,02,,-9.783333,-37.183333
br,antonio vicente,Antônio Vicente,25,,3.2,-59.916667
br,antonio vieira,Antônio Vieira,15,,-16.133333,-39.883333
br,antonopolis,Antonópolis,06,,-4.8,-37.733333
br,antuerpia,Antuérpia,24,,-10.533333,-63.183333
br,antunes,Antunes,02,,-8.966667,-35.2
br,antunes,Antunes,15,,-19.9,-44.7
br,antunes de baixo,Antunes de Baixo,15,,-22.505208,-45.532931
br,anuan,Anuan,25,,2.95,-61
br,anuao,Anuão,25,,2.95,-61
br,anutiba,Anutiba,08,,-20.6,-41.466667
br,a. oliveira,A. Oliveira,11,,-19.683333,-55.183333
br,apa da brisa,Apa da Brisa,21,,-22.990106,-43.653849
br,apajatuba,Apajatuba,13,,-2.616667,-43.8
br,aparados da serra,Aparados da Serra,23,,-28.7,-50.4
br,aparaju,Aparaju,05,,-17.633333,-39.316667
br,apare,Apare,05,,-8.716667,-39.116667
br,aparecida,Aparecida,04,,-3.129722,-60.024444
br,aparecida,Aparecida,08,,-20.7,-41.666667
br,aparecida,Aparecida,14,,-14.666667,-56.75
br,aparecida,Aparecida,17,,-6.783333,-38.066667
br,aparecida,Aparecida,20,,-6.2,-41.883333
br,aparecida,Aparecida,21,,-22.033333,-42.8
br,aparecida,Aparecida,22,,-6.4,-38.383333
br,aparecida,Aparecida,27,34237,-22.8418,-45.2287
br,aparecida,Aparecida,29,,-15.883333,-48.483333
br,aparecida de goias,Aparecida de Goiás,29,,-14.95,-49.783333
br,aparecida de minas,Aparecida de Minas,15,,-20.133333,-49.233333
br,aparecida de monte alto,Aparecida de Monte Alto,27,,-21.25,-48.633333
br,aparecida de sao manuel,Aparecida de São Manuel,27,,-22.716667,-48.616667
br,aparecida d'oeste,Aparecida d'Oeste,18,,-23.9,-52.816667
br,aparecida d'oeste,Aparecida d'Oeste,27,,-20.466667,-50.883333
br,aparecida do rio negro,Aparecida do Rio Negro,31,,-9.941111,-47.853889
br,aparecida do taboado,Aparecida do Taboado,11,17332,-20.083333,-51.083333
br,aparecida do tabuado,Aparecida do Tabuado,11,,-20.083333,-51.083333
br,aparicio,Aparicio,04,,-2.95,-69.216667
br,apeca,Apecã,30,,-8.833333,-36.066667
br,apehu,Apehú,16,,-1.3,-47.983333
br,aperibe,Aperibé,21,,-21.616667,-42.1
br,apertado,Apertado,06,,-7.2,-40.433333
br,apertado do morro,Apertado do Morro,05,,-13.833333,-40.95
br,apeu,Apeú,16,,-1.3,-47.983333
br,apiaba,Apiaba,18,,-25.333333,-50.783333
br,apiaca,Apiacá,08,,-21.133333,-41.566667
br,apiahy,Apiahy,27,,-24.511926,-48.844671
br,apiai,Apiaí,27,18260,-24.511926,-48.844671
br,apicum,Apicum,05,,-12.716667,-38.75
br,apinage,Apinagé,31,,-11.523889,-48.301667
br,apinaje,Apinajé,31,,-11.523889,-48.301667
br,apipucos,Apipucos,30,,-8.033333,-34.933333
br,apiques,Apiques,06,,-3.1,-39.516667
br,apiuna,Apiúna,26,,-27.033333,-49.416667
br,apodi,Apodi,22,16757,-5.65,-37.8
br,apodi,Apodi,30,,-8.066667,-35.4
br,apody,Apody,22,,-5.65,-37.8
br,apolinario,Apolinário,16,,-3.55,-47.05
br,apolinario,Apolinário,27,,-23.371696,-46.603889
br,apolinario,Apolinário,30,,-8.083333,-36.433333
br,apolo xi,Apolo XI,21,,-22.716431,-43.33265
br,apora,Aporá,05,,-11.65,-38.083333
br,apore,Aporé,29,,-18.966667,-52.016667
br,aporema,Aporema,03,,1.233333,-50.816667
br,aposento,Aposento,20,,-3.976944,-42.115556
br,apoti,Apoti,30,,-8.066667,-35.4
br,apparecida,Apparecida,20,,-7.633333,-43.95
br,apparecida,Apparecida,27,,-22.8418,-45.2287
br,apparecida,Apparecida,29,,-16.816667,-49.25
br,apparecida dos corregos,Apparecida dos Corregos,15,,-18.883333,-43.516667
br,apparecido de monte alto,Apparecido de Monte Alto,27,,-21.25,-48.633333
br,aprasive,Aprasive,20,,-6.383333,-41.966667
br,aprazivel,Aprazível,20,,-4.833333,-42.716667
br,aprigio,Aprígio,15,,-16.166667,-39.95
br,aproaga,Aproaga,16,,-1.95,-47.783333
br,apua,Apuá,30,,-7.883333,-35.333333
br,apuarema,Apuarema,05,,-13.9,-39.75
br,apucarana,Apucarana,18,107086,-23.55,-51.483333
br,apucaraninha,Apucaraninha,18,,-23.7757,-50.9554
br,apucarena,Apucarena,18,,-24.783333,-51.166667
br,apuhy,Apuhy,16,,-4.533333,-56.2
br,apui,Apuí,04,,-.1,-65.083333
br,apui,Apuí,16,,-1.483333,-47.466667
br,apuiares,Apuiarés,06,,-3.933333,-39.4
br,apuires,Apuirés,06,,-3.933333,-39.4
br,apurima,Apurimá,04,,-3.75,-66.233333
br,aqua preta,Aqua Preta,04,,-8.833333,-66.166667
br,aquidaba,Aquidabã,02,,-9.033333,-35.25
br,aquidaba,Aquidabã,04,,-6.5,-69.683333
br,aquidaba,Aquidabã,11,,-22.383333,-55.716667
br,aquidaba,Aquidabã,13,,-2.183333,-44.833333
br,aquidaba,Aquidabã,18,,-23.6,-51.85
br,aquidaba,Aquidabã,26,,-27.033333,-49.416667
br,aquidaba,Aquidabã,28,10596,-10.283333,-37.033333
br,aquidaban,Aquidaban,04,,-6.5,-69.683333
br,aquidaban,Aquidaban,26,,-27.033333,-49.416667
br,aquidaban,Aquidaban,28,,-10.283333,-37.033333
br,aquidauana,Aquidauana,11,35304,-20.466667,-55.8
br,aquidavana,Aquidavana,11,,-20.466667,-55.8
br,aqui-ferto,Aqui-Ferto,16,,-4.716667,-56.55
br,aquiles stengel,Aquiles Stengel,26,,-26.333333,-51.083333
br,aquiperto,Aquiperto,16,,-4.716667,-56.55
br,aquiraz,Aquiraz,06,65117,-3.9,-38.366667
br,ara,Ará,04,,-4.485833,-59.836389
br,araba,Arabá,27,,-19.85,-50.433333
br,arabela,Arabela,27,,-21.4,-51.716667
br,arabia,Arábia,22,,-5.543611,-36.312778
br,araboladi,Araboladí,16,,-6.616667,-57.566667
br,araborladit,Araborladit,16,,-6.616667,-57.566667
br,arabuta,Arabutã,26,,-27.166667,-52.116667
br,araca,Araçá,02,,-9.483333,-37.933333
br,araca,Aracá,04,,-4.963333,-60.256944
br,araca,Araça,05,,-13.033333,-38.983333
br,araca,Aracá,06,,-3.983333,-38.35
br,araca,Araçá,13,,-3.072778,-42.196667
br,araca,Araça,15,,-16.099444,-44.613056
br,araca,Araçá,17,,-6.25,-37.516667
br,araca,Araçá,20,,-3.489444,-42.491944
br,araca,Araçá,21,,-22.65,-42.1
br,araca,Araçá,23,,-28.666667,-51.75
br,araca,Araça,25,,4.216667,-60.9
br,araca,Araçá,30,,-8.766667,-36.666667
br,aracacu,Aracaçu,27,,-23.716667,-48.45
br,araca de baixo,Araçá de Baixo,05,,-13.333333,-43.483333
br,aracagi,Araçagi,15,,-15.85,-41.1
br,aracagi,Araçagi,17,,-6.85,-35.366667
br,aracai,Araçaí,15,,-19.2,-44.233333
br,aracaiba,Araçaíba,27,,-24.416667,-48.85
br,aracaji,Araçaji,17,,-6.85,-35.366667
br,aracaju,Aracaju,28,490184,-10.916667,-37.066667
br,aracarariguama,Araçarariguama,27,,-23.433333,-47.066667
br,aracari,Araçari,04,,-2.443056,-61.075278
br,aracariguama,Araçariguama,27,,-23.433333,-47.066667
br,aracas,Araçás,05,,-12.2,-38.2
br,aracassu,Aracassú,27,,-23.716667,-48.45
br,aracatiacu,Aracatiaçu,06,,-3.883333,-40.016667
br,aracatiara,Aracatiara,06,,-3.166667,-39.733333
br,aracati,Aracati,01,,-9.316667,-72.216667
br,aracati,Aracati,06,44294,-4.566667,-37.766667
br,aracati,Aracati,15,,-21.4,-42.6
br,aracati,Aracati,22,,-5.3,-35.65
br,aracati,Aracati,25,,-3.8,-61.333333
br,aracati,Aracati,29,,-16.183333,-49.083333
br,aracatiba,Araçatiba,08,,-20.466667,-40.5
br,aracatiba,Araçatiba,21,,-23.044606,-43.554555
br,aracatu,Aracatu,05,,-14.416667,-41.45
br,aracatuba,Araçatuba,04,,-2.516667,-66.416667
br,aracatuba,Araçatuba,15,,-16.416667,-40.966667
br,aracatuba,Araçatuba,26,,-28.116667,-48.7
br,aracatuba,Araçatuba,27,170028,-21.2,-50.416667
br,aracaty,Aracaty,01,,-9.316667,-72.216667
br,aracaty,Aracaty,06,,-4.566667,-37.766667
br,aracazinho,Araçazinho,04,,-5.191944,-60.387778
br,aracazinho,Aracazinho,06,,-3.966667,-38.316667
br,arace,Aracê,08,,-20.4,-41
br,araci,Araci,05,15089,-11.333333,-38.95
br,araci,Araci,15,,-21.516667,-42.933333
br,aracitaba,Aracitaba,15,,-21.333333,-43.383333
br,aracoiaba,Araçoiaba,01,,-8.540833,-73.102778
br,aracoiaba,Aracoiaba,06,13425,-4.383333,-38.816667
br,aracoiaba,Araçoiaba,30,13095,-7.8,-35.05
br,aracoiaba da serra,Araçoiaba da Serra,27,15395,-23.5049,-47.6185
br,aracoyaba,Aracoyaba,06,,-4.383333,-38.816667
br,aracruz,Aracruz,08,58272,-19.816667,-40.266667
br,aracuai,Araçuaí,15,22005,-16.866667,-42.066667
br,aracu,Araçu,29,,-16.366667,-49.683333
br,aracui,Aracuí,08,,-20.65,-41.2
br,aracupeva,Araçupeva,27,,-25.213889,-48.108611
br,aracy,Aracy,05,,-11.333333,-38.95
br,arado,Arado,27,,-23.622352,-47.485413
br,aragarcas,Aragarças,29,16781,-15.916667,-52.25
br,aragoiania,Aragoiânia,29,,-16.95,-49.5
br,aragominas,Aragominas,31,,-7.131389,-48.377222
br,araguacema,Araguacema,31,,-8.791944,-49.430556
br,araguaci,Araguaci,31,,-6.8,-48.616667
br,araguacu,Araguaçu,27,,-22.416667,-50.566667
br,araguacu,Araguaçu,31,7603,-12.894167,-49.773056
br,araguaia,Araguaia,08,,-20.5,-40.766667
br,araguaia,Araguaia,15,,-17.383333,-40.666667
br,araguaiana,Araguaiana,14,,-15.716667,-51.85
br,araguaina,Araguaína,31,114959,-7.16,-48.0575
br,araguainha,Araguainha,14,,-16.816667,-53.083333
br,araguana,Araguanã,31,,-6.589722,-48.473889
br,araguar,Araguar,15,,-18.633333,-48.183333
br,araguari,Araguari,03,,1.233333,-50.816667
br,araguari,Araguari,15,96567,-18.633333,-48.183333
br,araguary,Araguary,15,,-18.633333,-48.183333
br,araguatins,Araguatins,31,17106,-5.600278,-47.933056
br,araguaya,Araguaya,08,,-20.5,-40.766667
br,araguiana,Araguiana,14,,-15.716667,-51.85
br,araia,Araia,31,,-11.583333,-47.916667
br,arai,Araí,29,,-13.616667,-47.716667
br,araim,Araim,13,,-6.1,-43.766667
br,araioses,Araioses,13,8667,-2.89,-41.903611
br,araioses,Araioses,20,,-3.988889,-42.216944
br,araiporanga,Araiporanga,18,,-23.7222,-50.7389
br,arajara,Arajara,06,,-7.35,-39.4
br,aramanduba,Aramanduba,16,,-1.483333,-52.483333
br,aramari,Aramari,05,,-12.066667,-38.5
br,aramary,Aramary,05,,-12.066667,-38.5
br,arambare,Arambaré,23,,-30.916667,-51.483333
br,arame,Arame,20,,-3.268889,-42.118611
br,aramina,Aramina,27,,-20.1,-47.8
br,aramirim,Aramirim,15,,-19.016667,-42.466667
br,arana,Aranã,15,,-18.083333,-41.55
br,aranau,Aranaú,06,,-2.833333,-40.216667
br,arandu,Arandu,27,,-23.116667,-49.033333
br,aranha,Aranha,14,,-15.866667,-56.716667
br,aranha,Aranha,15,,-20.2,-44.116667
br,aranhas,Aranhas,13,,-8.5,-45.85
br,arantes,Arantes,15,,-21.466667,-44.2
br,arantina,Arantina,15,,-21.933333,-44.25
br,arantina,Arantina,29,,-17.616667,-50.266667
br,arapa,Arapá,06,,-3.616667,-40.883333
br,arapapa,Arapapá,04,,-4.238333,-62.258889
br,arapari,Arapari,04,,-3.447222,-61.809167
br,arapari,Arapari,06,,-3.55,-39.616667
br,arapari,Arapari,16,,-1.75,-54.4
br,arapari,Arapari,25,,3.1,-60.4
br,arapei,Arapeí,27,,-22.683333,-44.45
br,arapema,Arapema,13,,-3.716667,-44.233333
br,arapira,Arapirã,03,,3.283333,-51.333333
br,arapira,Arapirá,04,,.55,-69.166667
br,arapiraca,Arapiraca,02,166566,-9.75,-36.65
br,arapiraca,Arapiraca,17,,-7,-37.716667
br,arapiranga,Arapiranga,05,,-13.466667,-41.766667
br,arapixi,Arapixi,04,,-9.016667,-68.016667
br,arapixy,Arapixy,04,,-9.016667,-68.016667
br,arapoema,Arapoema,31,,-7.618889,-48.945556
br,araponga,Araponga,05,,-17.266667,-39.483333
br,araponga,Araponga,06,,-7.116667,-39.766667
br,araponga,Araponga,15,,-20.666667,-42.533333
br,araponga,Araponga,18,,-23.383333,-51.45
br,arapongas,Arapongas,18,97513,-23.383333,-51.45
br,arapora,Araporã,15,,-18.433333,-49.183333
br,araporanga,Araporanga,06,,-7.116667,-39.766667
br,arapoti,Arapoti,18,,-24.166667,-49.666667
br,arapua,Arapuá,02,,-9.716667,-37.5
br,arapua,Arapuá,06,,-4.683333,-41.116667
br,arapua,Arapuá,15,,-19.033333,-46.15
br,arapua,Arapuá,17,,-6.566667,-38.433333
br,arapua,Arapuá,22,,-5.458056,-36.855833
br,arapua,Arapuá,30,,-7.933333,-36.266667
br,arapuca,Arapuca,06,,-7.2,-40.333333
br,araqua,Araquã,16,,1.5,-54.633333
br,araquara,Araquara,27,,-24.566667,-47.516667
br,araquari,Araquari,26,35691,-26.3692,-48.7221
br,araquem,Araquém,06,,-3.633333,-40.816667
br,arara,Arara,04,,-2.566667,-65.8
br,arara,Arara,05,,-13.45,-40.25
br,arara,Arara,06,,-5.633333,-38.25
br,arara,Arara,13,,-3.083056,-42.278889
br,arara,Arara,17,7641,-6.833333,-35.75
br,arara,Arara,30,,-8.383333,-36.166667
br,arara,Arara,31,,-11.416667,-47.066667
br,arara de baixo,Arara de Baixo,06,,-2.935,-41.1725
br,ararai,Araraí,08,,-20.583333,-41.55
br,ararangua,Araranguá,26,43445,-28.933333,-49.483333
br,ararapari,Ararapari,23,,-27.6,-52.816667
br,ararapira,Ararapira,18,,-25.2288,-48.0301
br,araraquara,Araraquara,27,185191,-21.783333,-48.166667
br,araras,Araras,01,,-9.016667,-68.1
br,araras,Araras,04,,-2.566667,-65.8
br,araras,Araras,05,,-14.516667,-41.166667
br,araras,Araras,06,,-2.937778,-41.153889
br,araras,Araras,13,,-3.25,-44.466667
br,araras,Araras,14,,-15.166667,-56.65
br,araras,Araras,15,,-19.15,-42.766667
br,araras,Araras,16,,-3.783333,-51.783333
br,araras,Araras,17,,-7.333333,-38.533333
br,araras,Araras,24,,-10.45,-64.15
br,araras,Araras,27,107465,-22.366667,-47.383333
br,ararenda,Ararendá,06,,-4.75,-40.833333
br,arari,Arari,13,16777,-3.466667,-44.783333
br,arari,Arari,15,,-21.083333,-47.05
br,arari,Arari,16,,-2.416667,-47.8
br,ararica,Ararica,23,,-29.6,-50.916667
br,araripe,Araripe,06,11860,-7.2,-40.133333
br,araripe,Araripe,30,,-7.6,-39.7
br,araripe de baixo,Araripe de Baixo,05,,-12.833333,-38.8
br,araripina,Araripina,30,38570,-7.55,-40.566667
br,arariuna,Arariúna,16,,-1.016667,-48.966667
br,ararius,Arariús,06,,-3.883333,-40.6
br,araruama,Araruama,21,109638,-22.883333,-42.333333
br,araruna,Araruna,16,,-2.783333,-46.65
br,araruna,Araruna,17,7667,-6.533333,-35.733333
br,araruna,Araruna,18,,-23.9,-52.5
br,ararunaquara,Ararunaquara,16,,-3.5,-52.216667
br,araruva,Araruva,18,,-23.75,-51.316667
br,arary,Arary,04,,-7.8,-70.766667
br,arary,Arary,13,,-3.466667,-44.783333
br,arary,Arary,15,,-21.083333,-47.05
br,arary,Arary,31,,-11.416667,-47.066667
br,arasselva,Arasselva,11,,-22,-54.816667
br,arassoiaba,Arassoiaba,30,,-7.8,-35.05
br,arassuahy,Arassuahy,15,,-16.866667,-42.066667
br,arassuai,Arassuaí,15,,-16.866667,-42.066667
br,arassui,Arássuí,15,,-16.866667,-42.066667
br,arata,Arata,05,,-12.816667,-38.45
br,arataca,Arataca,05,,-15.25,-39.416667
br,aratama,Aratama,06,,-7.05,-39.85
br,arati,Arati,04,,-7.8,-70.766667
br,aratiba,Aratiba,23,,-27.4,-52.316667
br,araticu,Araticu,16,,-1.966667,-49.85
br,araticum,Araticum,05,,-12,-39.3
br,araticum,Araticum,06,,-3.8,-40.883333
br,araticum,Araticum,28,,-9.816667,-37.333333
br,aratinga,Aratinga,18,,-25.0084,-53.8422
br,aratinga,Aratinga,23,,-29.183333,-51.733333
br,aratingauba,Aratingaúba,26,,-28.3,-48.866667
br,aratu,Aratu,05,,-12.816667,-38.45
br,aratuba,Aratuba,05,,-13.1,-38.75
br,aratuba,Aratuba,06,,-4.383333,-39.05
br,aratuhype,Aratuhype,05,,-13.083333,-39
br,aratuipe,Aratuípe,05,,-13.083333,-39
br,araty,Araty,04,,-7.8,-70.766667
br,araua,Arauá,04,,-5.300833,-60.423611
br,araua,Arauá,28,4747,-11.266667,-37.616667
br,araucaria,Araucária,18,111303,-25.583333,-49.416667
br,araucumirim,Araucumirim,16,,-.616667,-47.65
br,araujo,Araujo,06,,-6.066667,-38.5
br,araujo,Araújo,25,,4.216667,-60.883333
br,araujos,Araújos,15,,-19.933333,-45.066667
br,arauna,Araúna,15,,-20.683333,-45.9
br,araxa,Araxá,15,82597,-19.583333,-46.916667
br,araxanas,Araxanas,27,,-21.933333,-51.533333
br,araxans,Araxans,27,,-21.933333,-51.533333
br,araxas,Araxãs,27,,-21.933333,-51.533333
br,arayoses,Arayoses,13,,-2.89,-41.903611
br,arayozes,Arayozes,13,,-2.89,-41.903611
br,arcadas,Arcadas,27,,-22.7,-46.866667
br,arcadia,Arcádia,21,,-22.512222,-43.523611
br,arcado,Arcado,27,,-24.6846,-48.3306
br,arcangelo,Arcângelo,15,,-21.3,-44.316667
br,arceburgo,Arceburgo,15,,-21.366667,-46.933333
br,arcelino,Arcelino,16,,-5.35,-55.8
br,arco iris,Arco Íris,27,,-21.766667,-50.466667
br,arcos,Arcos,15,33356,-20.283333,-45.533333
br,arco verde,Arco Verde,05,,-13.05,-39.4
br,arcoverde,Arcoverde,15,,-21.133333,-43.4
br,arco verde,Arco Verde,23,,-29.35,-51.566667
br,arcoverde,Arcoverde,26,,-27.283333,-51.433333
br,arcoverde,Arcoverde,30,57164,-8.416667,-36.916667
br,arcozelo,Arcozelo,21,,-22.403056,-43.423056
br,ardeira,Ardeira,20,,-3.659444,-42.622222
br,ardeiras,Ardeiras,20,,-3.649722,-42.046111
br,areadinho,Areadinho,27,,-24.633388,-48.169083
br,areado,Areado,05,,-10.333333,-43.266667
br,areado,Areado,15,10689,-21.35,-46.15
br,areado,Areado,27,,-24.558837,-48.04375
br,areal,Areal,05,,-12.116667,-44.566667
br,areal,Areal,13,,-3.233333,-44.616667
br,areal,Areal,21,11041,-22.228889,-43.106944
br,areal,Areal,23,,-31.75,-52.283333
br,arealva,Arealva,27,,-22.033333,-48.883333
br,areao,Areão,05,,-12.983333,-44.083333
br,areao,Areão,16,,-4.466667,-49.566667
br,areao,Areão,18,,-24.766667,-51.2
br,areao,Areão,31,,-11.583333,-47.916667
br,are,Aré,21,,-21.266667,-41.8
br,are,Aré,23,,-27.45,-52.85
br,area residencial da mineracao plumbum,Área Residencial da Mineração Plumbum,18,,-24.679336,-48.919977
br,areas,Áreas,15,,-22.5647,-45.7108
br,areas,Áreas,27,,-22.5905,-45.7256
br,aredeco,Aredecô,17,,-7.383333,-36.8
br,aredo,Aredo,15,,-16.266667,-41.466667
br,areia,Areia,05,,-11.583333,-45.05
br,areia,Areia,11,,-21.55,-57.183333
br,areia,Areia,13,,-3.076111,-42.211667
br,areia,Areia,15,,-16.1,-40.766667
br,areia,Areia,17,13825,-6.966667,-35.7
br,areia branca,Areia Branca,05,,-12.833333,-38.35
br,areia branca,Areia Branca,06,,-4.983333,-38.65
br,areia branca,Areia Branca,15,,-21.316667,-42.7
br,areia branca,Areia Branca,18,,-23.9032,-50.4609
br,areia branca,Areia Branca,20,,-7.05,-41
br,areia branca,Areia Branca,21,,-22.7525,-43.403333
br,areia branca,Areia Branca,22,18372,-4.953889,-37.132222
br,areia branca,Areia Branca,27,,-22.8,-46.85
br,areia branca,Areia Branca,28,7414,-10.766667,-37.316667
br,areia branca dos assis,Areia Branca dos Assis,18,,-25.8698,-49.365
br,areia comprida,Areia Comprida,22,,-5.88,-36.114167
br,areia da teixeira,Areia da Teixeira,05,,-12.3,-38.633333
br,areia de tabuleiro,Areia de Tabuleiro,22,,-6.183333,-38.016667
br,areia do rosario,Areia do Rosário,18,,-24.966667,-49.45
br,areia do teixeira,Areia do Teixeira,05,,-12.3,-38.633333
br,areia dourada,Areia Dourada,27,,-22.1,-51.933333
br,areia fina,Areia Fina,05,,-13.133333,-39.35
br,areial,Areial,02,,-9.166667,-37.35
br,areial,Areial,05,,-12.116667,-44.566667
br,areial,Areial,06,,-3.966667,-40.333333
br,areial,Areial,13,,-3.560278,-43.350278
br,areial,Areial,17,,-7.033333,-35.883333
br,areiao,Areião,05,,-12.983333,-44.083333
br,areiao,Areião,16,,-2.516667,-49.45
br,areiao,Areião,18,,-24.766667,-51.2
br,areiao,Areião,29,,-18.233333,-48.4
br,areiao,Areião,31,,-11.583333,-47.916667
br,areia preta,Areia Preta,27,,-24.672218,-48.130287
br,areia-preta,Areia-Preta,27,,-24.676315,-48.138465
br,areia-pretinha,Areia-Pretinha,27,,-24.684058,-48.142021
br,areias alvas,Areias Alvas,22,,-4.910556,-37.215
br,areias,Areias,05,,-12.8,-38.233333
br,areias,Areias,06,,-4.738889,-37.312222
br,areias,Areias,11,,-20.983333,-55.883333
br,areias,Areias,13,,-3.366667,-45.633333
br,areias,Areias,14,,-16.233333,-58.183333
br,areias,Areias,15,,-16.616667,-41.133333
br,areias,Areias,16,,-1.2,-51.25
br,areias,Areias,17,,-6.45,-38.433333
br,areias,Areias,20,,-5.166667,-42.8
br,areias,Areias,27,,-21.516667,-46.583333
br,areias,Areias,30,,-7.666667,-35.1
br,areias barauna,Areias Baraúna,17,,-7.116667,-36.933333
br,areias brancas,Areias Brancas,23,,-29.5,-49.866667
br,areias grandes,Areias Grandes,26,,-26.3796,-48.6685
br,areias pequenas,Areias Pequenas,26,,-26.3667,-48.6852
br,areia vermelha,Areia Vermelha,27,,-23.140884,-47.708828
br,areinhas,Areinhas,13,,-2.466667,-43.45
br,areiopolis,Areiópolis,27,,-22.666667,-48.7
br,areja,Areja,06,,-5.733333,-39.266667
br,arejas,Arejas,16,,-1.2,-51.25
br,arembepe,Arembepe,05,,-12.75,-38.166667
br,arenal,Arenal,01,,-9.233333,-72.683333
br,arenal,Arenal,04,,-7.566667,-72.616667
br,arenapolis,Arenápolis,14,9060,-14.433333,-56.816667
br,arencas,Arencas,06,,-3.266667,-39.733333
br,arengas,Arengas,06,,-3.316667,-40
br,arenosa,Arenosa,22,,-5.373611,-36.856389
br,areopolis,Areopolis,27,,-22.666667,-48.7
br,arequembaua,Arequembaua,16,,-2.716667,-49.733333
br,arere,Arerê,06,,-4.216667,-38.583333
br,arere,Arere,16,,-.266667,-53.866667
br,ares,Arês,22,7231,-6.183333,-35.15
br,arez,Arez,22,,-6.183333,-35.15
br,argenita,Argenita,15,,-19.7,-46.666667
br,argenta,Argenta,23,,-28.5,-52.2
br,argentina,Argentina,15,,-16.033333,-41.116667
br,argentina,Argentina,27,,-21.85,-47.8
br,argerita,Argerita,15,,-21.616667,-42.833333
br,argirita,Argirita,15,,-21.616667,-42.833333
br,argoim,Argoim,05,,-12.533333,-39.533333
br,argola,Argola,08,,-20.333333,-40.35
br,argolas,Argolas,08,,-20.333333,-40.35
br,argolo,Argôlo,05,,-17.866667,-40.016667
br,arguelho,Arguelho,11,,-22.7,-55.233333
br,argyrita,Argyrita,15,,-21.616667,-42.833333
br,ari acu,Ari Açu,23,,-28,-53.083333
br,ari,Ari,05,,-12.366667,-39.4
br,ariau,Ariaú,04,,-3.15,-57.233056
br,aribao,Aribão,22,,-5.3,-35.783333
br,aribice,Aribicé,05,,-10.566667,-38.783333
br,aricanduva,Aricanduva,15,,-17.833333,-42.583333
br,aricanduva,Aricanduva,18,,-23.466667,-51.416667
br,aricaria,Aricaria,16,,-2.683333,-52.05
br,aricobe,Aricobé,05,,-12.083333,-44.433333
br,aricura,Aricura,16,,-2.333333,-49.516667
br,arikemes,Arikemes,24,,-9.933333,-63.066667
br,arima,Arimã,04,,-5.791111,-63.633056
br,arimateia,Arimatéia,15,,-22.166667,-44.4
br,arinos,Arinos,14,,-14.033333,-56.466667
br,arinos,Arinos,15,10078,-15.916667,-46.066667
br,aripibu,Aripibu,30,,-8.433333,-35.35
br,aripuana,Aripuanã,14,27043,-9.166667,-60.633333
br,ariquemes,Ariquemes,04,,-5.039722,-59.891667
br,ariquemes,Ariquemes,24,58102,-9.933333,-63.066667
br,ariramba,Ariramba,04,,-5.394167,-59.725
br,ariranha,Ariranha,04,,-5.394167,-59.725
br,ariranha,Ariranha,27,,-21.183333,-48.783333
br,ariri,Ariri,27,,-25.2169,-48.0386
br,ariry,Ariry,27,,-25.2169,-48.0386
br,arisco,Arisco,06,,-4.516667,-39.666667
br,arisco,Arisco,22,,-5.524444,-36.039722
br,aristeu,Aristeu,15,,-15.933333,-46.2
br,aristeu braganca,Aristeu Bragança,23,,-30.7,-55.183333
br,aristides,Aristides,15,,-20.966667,-46.566667
br,aristides lobo,Aristides Lôbo,21,,-22.433333,-43.783333
br,arisuel,Arisuel,08,,-19.233333,-40.5
br,arisvel,Arisvel,08,,-19.233333,-40.5
br,aritagua,Aritaguá,05,,-14.716667,-39.1
br,arius,Ariús,17,,-7.033333,-35.883333
br,arizona,Arizona,05,,-12.8,-40.933333
br,arizona,Arizona,15,,-16.216667,-40.066667
br,arizona,Arizona,20,,-6.45,-41.583333
br,arizona,Arizona,22,,-5.283333,-35.7
br,arizona,Arizona,29,,-17.05,-48.3
br,arlindo gomes,Arlindo Gomes,15,,-18.8,-48.483333
br,armacao,Armaçao,05,,-17.733333,-39.233333
br,armacao,Armação,26,,-27.35,-48.566667
br,armacao dos buzios,Armação dos Búzios,21,23463,-22.75,-41.883333
br,armacao do tairu,Armação do Tairu,05,,-13.066667,-38.733333
br,armador,Armador,06,,-5.45,-38.183333
br,armazem,Armazém,13,,-3.082778,-43.113611
br,armazem,Armazém,23,,-28.916667,-52.583333
br,armazem,Armazém,26,,-28.266667,-49.016667
br,arneiroz,Arneiroz,06,,-6.333333,-40.133333
br,arnoni,Arnoni,23,,-33.616667,-53.45
br,arnoresco,Arnoresco,23,,-33.233333,-53.066667
br,arnoreso,Arnoreso,23,,-33.233333,-53.066667
br,aroaba,Aroaba,08,,-20.133333,-40.383333
br,aroases,Aroases,20,,-6.116667,-41.8
br,aroazes,Aroazes,20,,-6.116667,-41.8
br,aroeira,Aroeira,05,,-10.65,-44.083333
br,aroeira,Aroeira,06,,-2.966667,-40.383333
br,aroeira,Aroeira,11,,-21.683333,-54.416667
br,aroeira,Aroeira,13,,-3.824167,-43.323611
br,aroeira,Aroeira,15,,-19.216667,-48.283333
br,aroeira,Aroeira,17,,-7.7,-36.066667
br,aroeira,Aroeira,18,,-24.95,-53.4
br,aroeira,Aroeira,20,,-3.709167,-42.274722
br,aroeira,Aroeira,22,,-5.420278,-37.493056
br,aroeira,Aroeira,23,,-28.9,-53.1
br,aroeira de dentro,Aroeira de Dentro,20,,-6.816667,-41.083333
br,aroeira grande,Aroeira Grande,20,,-7.733333,-41.4
br,aroeira ii,Aroeira II,20,,-5.066667,-41.566667
br,aroeiral,Aroeiral,20,,-3.864722,-42.025556
br,aroeiras altas,Aroeiras Altas,22,,-5.698611,-37.210278
br,aroeiras,Aroeiras,06,,-4.566667,-37.983333
br,aroeiras,Aroeiras,17,7019,-7.516667,-35.683333
br,aroeiras,Aroeiras,20,,-3.816944,-42.03
br,aroeira velha,Aroeira Velha,06,,-3.016667,-40.016667
br,aroeirinha,Aroeirinha,20,,-7.366667,-41.666667
br,arouca,Arouca,27,,-20.566667,-50.616667
br,arraia,Arraia,26,,-26.9,-48.933333
br,arraial,Arraial,05,,-14.016667,-39.25
br,arraial,Arraial,06,,-3.516667,-40.25
br,arraial,Arraial,13,,-3.433333,-44.316667
br,arraial,Arraial,20,,-3.819722,-42.446111
br,arraial,Arraial,23,,-32.016667,-52.266667
br,arraial,Arraial,26,,-26.9,-48.933333
br,arraial,Arraial,27,,-22.833333,-46.65
br,arraial do cabo,Arraial do Cabo,21,26164,-22.966667,-42.016667
br,arraial sao bento,Arraial São Bento,27,,-22.916667,-47.716667
br,arraias,Arraias,31,6881,-12.977222,-46.891667
br,arranca-dente,Arranca-Dente,15,,-16.166667,-40.983333
br,arrastao,Arrastão,21,,-22.783333,-42.183333
br,arrayas,Arrayas,31,,-12.977222,-46.891667
br,arrebol,Arrebol,21,,-21.983333,-41.883333
br,arrecife,Arrecife,05,,-11.033333,-41.883333
br,arrecife,Arrecife,20,,-5.216667,-41.45
br,arrecife,Arrecife,30,,-8.05,-34.9
br,arrecife grande,Arrecife Grande,05,,-11.033333,-41.7
br,arrecifes,Arrecifes,05,,-12.283333,-41.05
br,arreias,Arreias,29,,-16.05,-50.05
br,arreios,Arreios,29,,-16.05,-50.05
br,arrelia,Arrelia,21,,-22.936667,-43.253611
br,arrependido,Arrependido,16,,-4.483333,-49.55
br,arrepol,Arrepol,21,,-21.983333,-41.883333
br,arrieiro,Arrieiro,17,,-8.05,-36.816667
br,arrinha,Arrinha,05,,-12.333333,-40.683333
br,arroial de lavrinhas,Arroial de Lavrinhas,14,,-15.3,-59.233333
br,arroial velho,Arroial Velho,14,,-14.683333,-59.183333
br,arroio baixo laje,Arroio Baixo Laje,26,,-26.937666,-49.626324
br,arroio bonito,Arroio Bonito,23,,-29.366667,-53.05
br,arroio corrente,Arroio Corrente,26,,-28.7,-49.083333
br,arroio da cruz,Arroio da Cruz,18,,-25.896929,-50.228786
br,arroio da seca,Arroio da Sêca,23,,-29.366667,-51.783333
br,arroio do meio,Arroio do Meio,23,16331,-29.4,-51.95
br,arroio do pessegueiro,Arroio do Pessegueiro,18,,-25.515737,-50.626623
br,arroio do silva,Arroio do Silva,26,,-28.983333,-49.433333
br,arroio do so,Arroio do Só,23,,-29.783333,-53.566667
br,arroio dos ratos,Arroio dos Ratos,23,13405,-30.083333,-51.716667
br,arroio do tigre,Arroio do Tigre,23,,-29.333333,-53.1
br,arroio grande,Arroio Grande,18,,-25.2897,-50.6679
br,arroio grande,Arroio Grande,23,16117,-32.233333,-53.083333
br,arroio guacu,Arroio Guaçu,18,,-24.383333,-54.25
br,arroio miranda,Arroio Miranda,23,,-28.2,-52.366667
br,arroio moreira,Arroio Moreira,23,,-32.0371,-52.7105
br,arroio peixoto,Arroio Peixoto,05,,-12.283333,-41.416667
br,arroio teixeira,Arroio Teixeira,23,,-29.633333,-49.966667
br,arroio trinta,Arroio Trinta,04,,-26.866667,-51.316667
br,arrojado,Arrojado,05,,-12.383333,-38.45
br,arrojado,Arrojado,06,,-6.666667,-39.033333
br,arroyal de lavrinhas,Arroyal de Lavrinhas,14,,-15.3,-59.233333
br,arroyal velho,Arroyal Velho,14,,-14.683333,-59.183333
br,arrozal,Arrozal,11,,-18.633333,-56.216667
br,arrozal,Arrozal,14,,-14.816667,-59.933333
br,arrozal,Arrozal,15,,-20.5,-42.216667
br,arrozal,Arrozal,16,,-1.266667,-48.816667
br,arrozal,Arrozal,21,,-22.616667,-44.05
br,arrozal,Arrozal,27,,-23.098611,-46.668333
br,arrozal de sao sebastiao,Arrozal de São Sebastiao,21,,-22.816667,-43.766667
br,arroz,Arroz,05,,-11,-45.283333
br,arroz,Arroz,06,,-4.316667,-38.6
br,arroz,Arroz,17,,-7.45,-37.566667
br,arroz,Arroz,20,,-8.333333,-41.416667
br,arroz,Arroz,30,,-7.766667,-38.75
br,arroz bravo,Arroz Bravo,06,,-3.95,-38.383333
br,arroz-doce,Arroz-Doce,18,,-23.592648,-49.864692
br,arrozeira,Arrozeira,26,,-26.733333,-49.266667
br,arroz sem sal,Arroz Sem Sal,14,,-14.366667,-56.716667
br,arruda,Arruda,14,,-15.033333,-56.116667
br,arruda,Arruda,17,,-7.133333,-36.05
br,arruda camara,Arruda Câmara,17,,-6.816667,-37.916667
br,arrudas,Arrudas,27,,-23.5077,-47.8342
br,arrudeiro,Arrudeiro,30,,-7.75,-37.483333
br,arruido,Arruido,17,,-7.033333,-38.583333
br,artemis,Artemis,27,,-22.666667,-47.75
br,arthur nogueira,Arthur Nogueira,27,,-22.583333,-47.15
br,articura,Articura,16,,-2.333333,-49.516667
br,artur,Artur,05,,-12.416667,-40.633333
br,artur grande,Artur Grande,30,,-8.633333,-36.7
br,artur nogueira,Artur Nogueira,27,40404,-22.583333,-47.15
br,arua,Aruá,06,,-4.333333,-38.583333
br,aruaja,Aruajá,04,,-5.033333,-66.816667
br,aruana,Aruanã,04,,-7.616667,-60.316667
br,aruana,Aruanã,29,,-14.9,-51.083333
br,aru,Aru,04,,.65,-67.233333
br,aru,Aru,16,,-2.216667,-49.783333
br,aruaru,Aruaru,06,,-4.583333,-38.35
br,aruata,Aruatã,18,,-25.925441,-49.178695
br,aruau,Aruaú,04,,-.383333,-66.35
br,arueira,Arueira,05,,-11.566667,-41.783333
br,arueira,Arueira,13,,-6.366667,-46.483333
br,aruja,Arujá,27,72078,-23.4,-46.333333
br,aruma,Arumã,04,,-4.735556,-62.150556
br,arumanduba,Arumanduba,04,,-2.6,-66.366667
br,arumanduba,Arumanduba,16,,-1.483333,-52.483333
br,arumandupa,Arumandupa,04,,-2.6,-66.366667
br,arumim,Arumim,25,,3.433333,-60.916667
br,arunta,Arunta,13,,-2.566667,-44.15
br,aruti,Aruti,04,,-.316667,-66.083333
br,arvaja,Arvajá,04,,-5.033333,-66.816667
br,arvoredo,Arvoredo,06,,-5.066667,-39.166667
br,arvoredo,Arvoredo,11,,-21.35,-56.416667
br,arvoredo,Arvoredo,17,,-7.1,-37.3
br,arvoredo,Arvoredo,26,,-28.4,-50.116667
br,arvore grande,Árvore Grande,11,,-19.216667,-51.816667
br,arvores,Árvores,23,,-29.266667,-53.55
br,arvoresinha,Arvoresinha,23,,-28.883333,-52.166667
br,arvore verde,Árvore Verde,05,,-13.566667,-40.883333
br,arvorezinha,Àrvorezinha,23,,-28.883333,-52.166667
br,ascencao,Ascenção,15,,-19.8,-44.583333
br,ascuria,Ascuria,26,,-26.966667,-49.366667
br,ascurra,Ascurra,26,,-26.966667,-49.366667
br,asdiubal,Asdiubal,15,,-16.833333,-41.7
br,asoenangka,Asoenangka,16,,1.766667,-56.366667
br,assacaio,Assacaio,04,,-3.816667,-69.4
br,assahytuba,Assahytuba,04,,-7.016667,-64.583333
br,assai,Assaí,18,12545,-23.366667,-50.816667
br,assai,Assaí,27,,-21.383333,-48.983333
br,assaituba,Assaituba,04,,-7.016667,-64.583333
br,assaizal,Assaizal,31,,-6.433333,-48.383333
br,assa-peixe,Assa-Peixe,15,,-15.85,-43.95
br,assarai,Assaraí,15,,-19.583333,-41.483333
br,assare,Assaré,06,11369,-6.866667,-39.866667
br,assembleia,Assembléia,02,,-9.4,-36.233333
br,assentadinho,Assentadinho,27,,-24.8307,-48.2865
br,assentado,Assentado,27,,-24.8144,-48.2717
br,assis,Assis,06,,-4.916667,-40.866667
br,assis,Assis,27,87473,-22.666667,-50.416667
br,assis brasil,Assis Brasil,01,,-10.916667,-69.533333
br,assis brasil,Assis Brasil,23,,-28.266667,-53.566667
br,assituas,Assituas,13,,-2.733333,-46.65
br,assiz,Assiz,27,,-22.666667,-50.416667
br,assu,Assu,22,,-5.577222,-36.9075
br,assu de torre,Assú de Torre,05,,-12.55,-38
br,assumpcao,Assumpção,04,,-5.016667,-62.783333
br,assumpcao,Assumpção,06,,-3.566667,-39.633333
br,assumpcao,Assumpção,24,,-9.016667,-61.616667
br,assuncao,Assunção,04,,-5.016667,-62.783333
br,assuncao,Assuncão,05,,-12.816667,-42.766667
br,assuncao,Assunção,06,,-3.566667,-39.633333
br,assuncao,Assuncão,17,,-7.066667,-36.716667
br,assuncao,Assunção,20,,-5.866667,-41.066667
br,assuncao,Assunção,22,,-5.516667,-35.8
br,assuncao,Assunção,24,,-8.133333,-63.016667
br,assungui,Assungui,18,,-25.530951,-50.553859
br,assunguy,Assunguy,18,,-24.983333,-49.5
br,assunguy de cima,Assunguy de Cima,18,,-24.983333,-49.5
br,assurua,Assuruá,05,,-11.3,-42.666667
br,astolfo dutra,Astolfo Dutra,15,10999,-21.316667,-42.866667
br,astorga,Astorga,18,20129,-23.216667,-51.666667
br,astrogildo,Astrogildo,21,,-22.977002,-43.585487
br,atafona,Atafona,11,,-23.533333,-55.283333
br,atafona,Atafona,21,,-21.616667,-41.016667
br,atafona,Atafona,26,,-27.116667,-49.5
br,ataiba pinto,Ataíba Pinto,15,,-20.2,-46.916667
br,atalaia,Atalaia,02,18908,-9.516667,-36.033333
br,atalaia,Atalaia,04,,-7.416667,-71.233333
br,atalaia,Atalaia,08,,-18.633333,-40.016667
br,atalaia,Atalaia,15,,-21.05,-45.983333
br,atalaia,Atalaia,16,,-.983333,-49.316667
br,atalaia,Atalaia,18,,-23.151633,-52.054438
br,atalaia,Atalaia,20,,-2.878611,-41.643889
br,atalaia,Atalaia,28,,-11.416667,-37.316667
br,atalaia do norte,Atalaia do Norte,04,,-4.365278,-70.193056
br,atalaia nova,Atalaia Nova,28,,-10.933333,-37.033333
br,atalaia velha,Atalaia Velha,28,,-10.966667,-37.066667
br,ataleia,Ataléia,15,,-18.05,-41.1
br,atalho,Atalho,05,,-10.116667,-40.733333
br,atalho,Atalho,21,,-22.55,-42.15
br,atalho,Atalho,30,,-8.983333,-38.333333
br,atamasio,Atamásio,11,,-21.516667,-54.15
br,atanasio,Atanasio,11,,-21.516667,-54.15
br,atanasio,Atanásio,13,,-3.588333,-43.175556
br,ataque,Ataque,05,,-12.85,-40.2
br,ataque,Ataque,15,,-22.033333,-45.566667
br,a. tarraca,A. Tarraca,11,,-22.316667,-55.033333
br,atauba,Ataúba,25,,-1.033333,-61.85
br,atenas,Atenas,01,,-8.866667,-71.483333
br,aterradinho,Aterradinho,11,,-21.133333,-56.65
br,aterradinho,Aterradinho,14,,-16.95,-57.35
br,aterradinho,Aterradinho,15,,-20.366667,-47.116667
br,aterradinho,Aterradinho,27,,-23.45,-48.566667
br,aterrado alto,Aterrado Alto,18,,-25.840009,-50.223137
br,aterrado,Aterrado,13,,-4.25,-45.633333
br,aterrado,Aterrado,14,,-17.016667,-56.6
br,athenas,Athenas,01,,-8.866667,-71.483333
br,ati-acu,Ati-Açu,23,,-28.05,-53.066667
br,ati acu,Ati Açu,23,,-28,-53.083333
br,ati assu,Ati Assu,23,,-28,-53.083333
br,atibaia,Atibaia,27,110970,-23.124167,-46.562222
br,ati-mirim,Ati-Mirim,23,,-28.05,-52.683333
br,atininga,Atininga,04,,-5.879444,-61.241667
br,atirado,Atirado,15,,-22.133333,-45.583333
br,atlantica,Atlântica,27,,-21.583333,-51.216667
br,atlantida,Atlântida,27,,-21.583333,-51.216667
br,atoleiro,Atoleiro,05,,-11.95,-44.716667
br,atoleiro,Atoleiro,13,,-3.412778,-42.667778
br,atoleiro,Atoleiro,20,,-3.6725,-42.599167
br,atoleiro,Atoleiro,22,,-6.333333,-37.883333
br,atoleiro de dentro,Atoleiro de Dentro,13,,-3.926944,-43.184722
br,atomia,Atomia,18,,-23.266667,-50.583333
br,atras dos morros,Atrás dos Morros,20,,-6,-41.466667
br,atravessado,Atravessado,04,,-4.085,-62.305
br,atuba,Atuba,18,,-25.433333,-49.233333
br,aturiai,Aturiaí,16,,-1.066667,-46.5
br,auari,Auari,04,,-8.333333,-67.333333
br,augosto severo,Augosto Severo,22,,-5.863333,-37.310556
br,augusta vitoria,Augusta Vitória,18,,-24.25,-50.85
br,augustinho,Augustinho,05,,-9.883333,-38.016667
br,augustinopolis,Augustinópolis,31,,-5.41,-47.706389
br,augusto correa,Augusto Correa,16,16280,-1.033333,-46.666667
br,augusto de lima,Augusto de Lima,15,,-18.1,-44.316667
br,augusto franco,Augusto Franco,15,,-21.716667,-43.633333
br,augusto montenegro,Augusto Montenegro,04,,-2.716667,-57.7
br,augusto nicolau,Augusto Nicolau,15,,-21.633333,-46.416667
br,augusto queiroga,Augusto Queiroga,15,,-17.483333,-41.666667
br,augusto seio,Augusto Seio,15,,-19.35,-41.083333
br,augusto severo,Augusto Severo,22,,-5.863333,-37.310556
br,augusto severo,Augusto Severo,23,,-28.816667,-51.8
br,augusto soares,Augusto Soares,15,,-21.283333,-45.7
br,auler,Auler,23,,-28.766667,-52.416667
br,aunas,Aunas,05,,-13.2,-39.3
br,aurea,Áurea,23,,-27.7,-52.066667
br,aureliopolis,Aureliópolis,29,,-17.15,-48.8
br,aurerliopolis,Aurerliopolis,29,,-17.15,-48.8
br,auriflama,Auriflama,27,11971,-20.683333,-50.566667
br,aurilandia,Aurilândia,29,,-16.733333,-50.466667
br,aurizona,Aurizona,13,,-1.283333,-45.766667
br,aurora,Aurora,01,,-9.683333,-72.083333
br,aurora,Aurora,05,,-12.4,-38.816667
br,aurora,Aurora,06,10757,-6.95,-38.966667
br,aurora,Aurora,11,,-19.2,-55.483333
br,aurora,Aurora,18,,-26.060833,-53.688056
br,ausentes,Ausentes,23,,-28.733333,-50.066667
br,austin,Austin,21,,-22.720978,-43.527517
br,autazes,Autazes,04,9614,-3.583333,-59.133333
br,auxiliadora,Auxiliadora,23,,-30.025,-51.191667
br,avahy,Avahy,27,,-22.133333,-49.366667
br,avai,Avaí,01,,-8.183333,-73.4
br,avai,Avaí,26,,-27.35,-51.583333
br,avai,Avaí,27,,-22.133333,-49.366667
br,avanhandava,Avanhandava,27,,-21.466667,-49.95
br,avantiguara,Avantiguara,15,,-18.766667,-49.083333
br,avare,Avaré,27,81287,-23.083333,-48.916667
br,avarzeado,Avarzeado,06,,-5.683333,-40.583333
br,avatinguara,Avatinguara,15,,-18.766667,-49.083333
br,aveiro,Aveiro,16,,-3.616667,-55.316667
br,avelar,Avelar,15,,-21.666667,-45.3
br,avelar,Avelar,21,,-22.321944,-43.406389
br,avelinopolis,Avelinópolis,29,,-16.5,-49.716667
br,avelino silveira,Avelino Silveira,15,,-15.1,-42.816667
br,avenca,Avenca,18,,-24.983333,-50.633333
br,avencal,Avencal,18,,-25.5538,-50.42894
br,avencal,Avencal,26,,-26.216567,-49.604935
br,avencal de baixo,Avencal de Baixo,26,,-26.216567,-49.604935
br,avencas,Avencas,27,,-22.283333,-50.116667
br,avenida,Avenida,05,,-13.006667,-38.525
br,avenquinha de santo antonio,Avenquinha de Santo Antônio,26,,-26.161593,-49.326335
br,aventureio,Aventureio,15,,-21.75,-42.816667
br,aviacao,Aviação,23,,-29.733333,-53.85
br,aviario,Aviário,05,,-12.3,-38.933333
br,avobral,Avobral,11,,-21.2,-56.833333
br,axinal dos rodrigues,Axinal dos Rodrigues,18,,-25.721388,-50.221168
br,axinim,Axinim,04,,-4.035278,-59.379722
br,axioma,Axioma,04,,-6.75,-64.516667
br,axixa,Axixá,13,,-2.85,-44.066667
br,axixa de goias,Axixá de Goiás,31,,-5.551111,-47.595556
br,axixa do tocantins,Axixá do Tocantins,31,9007,-5.551111,-47.595556
br,axui,Axuí,13,,-2.516667,-43.6
br,ayapua,Ayapua,04,,-4.463333,-62.055278
br,aymores,Aymores,05,,-17.833333,-40.166667
br,aymores,Aymorés,15,,-19.5,-41.066667
br,ayrao,Ayrão,04,,-2.626944,-60.95
br,ayri,Ayri,04,,-6.9,-58.283333
br,ayuruoca,Ayuruoca,15,,-21.966667,-44.6
br,azambuja,Azambuja,26,,-28.5,-49.216667
br,azauri,Azauri,16,,.616667,-54.15
br,azeredo,Azeredo,08,,-20.283333,-40.45
br,azevedo,Azevedo,06,,-5.216667,-40.066667
br,azevedo,Azevedo,08,,-20.283333,-40.45
br,azevedo,Azevedo,23,,-29.7,-51.333333
br,azevedo sodre,Azevedo Sodré,23,,-30.066667,-54.6
br,azevedro sodre,Azevedro Sodre,23,,-30.066667,-54.6
br,azevem,Azevém,17,,-6.966667,-38.616667
br,azevem,Azevém,30,,-8.233333,-36.666667
br,azeveria,Azeveria,30,,-8.233333,-36.666667
br,azurita,Azurita,15,,-20.016667,-44.466667
br,babacu,Babaçu,13,,-4.65,-43.6
br,babaculandia,Babaçulândia,31,,-7.209167,-47.621667
br,babia,Babia,06,,-3.55,-40.233333
br,babi,Babi,21,,-22.7,-43.383333
br,babilonia,Babilônia,08,,-20.933333,-41.416667
br,babilonia,Babilônia,13,,-8.283333,-45.95
br,babilonia,Babilônia,15,,-20.5,-46.65
br,babilonia,Babilônia,21,,-22.55,-44.466667
br,baby,Baby,21,,-22.7,-43.383333
br,babylonia,Babylonia,21,,-22.55,-44.466667
br,bacaba,Bacaba,04,,-3.416667,-66.05
br,bacaba,Bacaba,13,,-2.983333,-43.85
br,bacaba,Bacaba,31,,-10.915278,-48.529444
br,bacaba i,Bacaba I,13,,-3.605556,-43.495556
br,bacaba ii,Bacaba II,13,,-3.395,-43.232778
br,bacabal,Bacabal,04,,-5.283333,-60.566667
br,bacabal,Bacabal,13,72374,-4.233333,-44.783333
br,bacabal,Bacabal,16,,-5.533333,-56.916667
br,bacabal,Bacabal,25,,1.45,-60.7
br,bacabal,Bacabal,31,,-11.933056,-49.190556
br,bacabal do carrinho,Bacabal do Carrinho,13,,-3.721667,-43.355278
br,bacabal dos crentes,Bacabal dos Crentes,13,,-3.7275,-43.336667
br,bacabal do tiberio,Bacabal do Tibério,13,,-3.4825,-43.449444
br,bacabal ii,Bacabal II,13,,-3.37,-43.420278
br,bacabal iii,Bacabal III,13,,-3.506389,-43.338333
br,bacabalzinho,Bacabalzinho,16,,-5.466667,-56.883333
br,bacada,Bacada,04,,-6.316667,-68.166667
br,bacaeri,Bacaeri,14,,-14.2,-55.083333
br,bacaetava,Bacaetava,27,,-23.433333,-47.6
br,bacaina,Bacaína,13,,-6.75,-44.066667
br,bacaiuval,Bacaiuval,14,,-14.65,-57.216667
br,bacalhau,Bacalhau,05,,-12.316667,-38.433333
br,bacao,Bação,15,,-20.333333,-43.833333
br,bacara,Bacará,25,,.710833,-61.648611
br,bacatuba,Bacatuba,13,,-5.666667,-43.7
br,bacatu,Bacatu,13,,-3.816667,-44.866667
br,bacauba,Bacaúba,20,,-8.944167,-45.431111
br,bacaubas,Bacaúbas,20,,-8.944167,-45.431111
br,bacaxa,Bacaxá,21,,-22.9,-42.466667
br,bacayuval,Bacayuval,14,,-14.65,-57.216667
br,bacelar,Bacelar,21,,-21.916667,-42.616667
br,bacia de pedra,Bacia de Pedra,21,,-22.4175,-43.756944
br,bacia de pedras,Bacia de Pedras,21,,-22.4175,-43.756944
br,bacre,Bacre,16,,-1.9,-50.2
br,bacue,Bacué,13,,-3.373611,-42.440278
br,bacumixa,Bacumixá,06,,-3.316667,-39.466667
br,bacupari,Bacupari,06,,-7.3,-38.8
br,bacurau,Bacurau,06,,-3.25,-39.433333
br,bacuri,Bacuri,04,,-4.476389,-62.193333
br,bacuri,Bacuri,13,,-1.733333,-45.133333
br,bacuri,Bacuri,20,,-3.7325,-42.643056
br,bacuri,Bacuri,27,,-21.05,-50.983333
br,bacuri,Bacuri,31,,-5.766667,-47.8
br,bacuri grande,Bacuri Grande,04,,-4.491944,-62.205
br,bacuri grande,Bacuri Grande,16,,-5.433333,-48.6
br,bacuri i,Bacuri I,13,,-3.435556,-43.4325
br,bacuri ii,Bacuri II,13,,-3.111389,-42.354167
br,bacuriteua,Bacuriteua,16,,-.983333,-46.733333
br,bacuriti,Bacuriti,27,,-21.616667,-49.433333
br,bacurituba,Bacurituba,13,,-2.716667,-44.716667
br,bacurizeiro,Bacurizeiro,13,,-3.143056,-42.219722
br,bacurizinho,Bacurizinho,16,,-5.416667,-48.633333
br,bacururi,Bacururi,04,,-3.75,-66.183333
br,bacururu,Bacururu,04,,-3.75,-66.183333
br,bacury,Bacury,04,,-4.85,-62.533333
br,bacury grande,Bacury Grande,16,,-5.433333,-48.6
br,bacury sinho,Bacury Sinho,16,,-5.416667,-48.633333
br,badajos,Badajós,04,,-3.416667,-62.666667
br,badajos,Badajós,16,,-2.516667,-47.783333
br,badajos,Badajós,28,,-10.566667,-36.816667
br,badajoz,Badajóz,16,,-2.516667,-47.783333
br,badalo,Badalo,17,,-7.416667,-36.433333
br,badaro,Badaró,21,,-21.316667,-41.666667
br,badia,Badia,28,,-9.666667,-37.75
br,bady bassitt,Bady Bassitt,27,14216,-20.933333,-49.466667
br,bady brassitt,Bady Brassitt,27,,-20.933333,-49.466667
br,baependi,Baependi,15,12853,-21.95,-44.883333
br,baependy,Baependy,15,,-21.95,-44.883333
br,baeta,Baeta,05,,-11.2,-43.083333
br,baetas,Baetas,04,,-6.984722,-62.812778
br,baete,Baetê,05,,-13.716667,-40.066667
br,bagaco,Bagaço,13,,-3.458333,-43.4525
br,bagagem,Bagagem,15,,-18.983333,-47.466667
br,bage,Bagé,23,98943,-31.333333,-54.1
br,bago,Bago,16,,-4.699167,-52.765
br,bagre,Bagre,14,,-15.95,-55.716667
br,bagre,Bagre,15,,-19.3,-42.5
br,bagre,Bagre,16,,-1.9,-50.2
br,bagre,Bagre,18,,-25.102805,-48.317008
br,baguacu,Baguaçu,11,,-21.383333,-57
br,baguacu,Baguaçu,27,,-20.166667,-47.583333
br,baguari,Baguari,15,,-18.366667,-42.8
br,baguari,Baguari,27,,-22.866667,-47.983333
br,baguassu,Baguassú,27,,-20.716667,-49.066667
br,bahia,Bahia,05,,-12.983333,-38.516667
br,bahia,Bahia,26,,-26.816667,-52.033333
br,bahia de pedra,Bahia de Pedra,14,,-16.466667,-58.083333
br,bahia do genipapo,Bahia do Genipapo,11,,-19.666667,-56.15
br,bahianos,Bahianos,04,,-6.3,-62.216667
br,bahus,Bahus,11,,-18.316667,-53.166667
br,bahusinho,Bahúsinho,11,,-18.316667,-53.166667
br,baia,Baía,05,,-12.983333,-38.516667
br,baia,Baía,11,,-19.933333,-56.283333
br,baia,Baía,14,,-16.733333,-56.916667
br,baia,Baía,16,,-.783333,-51.133333
br,baia,Baía,26,,-26.816667,-52.033333
br,baia bela,Baía Bela,14,,-16.166667,-58.75
br,baia bonita,Baía Bonita,05,,-13.75,-40.216667
br,baia branca,Baía Branca,27,,-23.328547,-46.648192
br,baiacus,Baiacus,22,,-5.75,-37.85
br,baia das garcas,Baía das Garças,11,,-21.066667,-56.833333
br,baia das gracas,Baía das Graças,11,,-21.066667,-56.833333
br,baia das pedras,Baía das Pedras,11,,-19.283333,-55.716667
br,baia das pedras,Baía das Pedras,14,,-16.466667,-58.083333
br,baia da traicao,Baía da Traição,17,,-6.7,-34.933333
br,baia de cima,Baía de Cima,17,,-7.433333,-34.833333
br,baia de las pedras,Baía de las Pedras,14,,-16.466667,-58.083333
br,baia de pedras,Baia de Pedras,14,,-16.466667,-58.083333
br,baia de sao felix,Baía de São Felix,14,,-16.466667,-55.933333
br,baia de sao luis,Baía de São Luís,14,,-16.7,-57.216667
br,baia do cervo,Baía do Cervo,11,,-19.5,-55.833333
br,baia do genipapo,Baía do Genipapo,11,,-19.666667,-56.15
br,baia do itajai,Baía do Itajaí,26,,-26.651238,-49.853402
br,baia dos cavalos,Baía dos Cavalos,14,,-16.333333,-56.2
br,baia dos nunes,Baía dos Nunes,14,,-15.716667,-55.9
br,baia dos passaros,Baía dos Pássaros,11,,-17.833333,-55.483333
br,baia dos passaros,Baía dos Pássaros,14,,-16.6,-55.866667
br,baia formosa,Baía Formosa,21,,-22.833333,-41.966667
br,baia formosa,Baía Formosa,22,,-6.366667,-35
br,baia funda,Baía Funda,20,,-8.883333,-45.966667
br,baia mineira,Baía Mineira,11,,-18.466667,-56.45
br,baiana,Baiana,15,,-17.85,-40.783333
br,baianopolis,Baianópolis,11,,-19.766667,-54.783333
br,baianos,Baianos,04,,-6.3,-62.216667
br,baianos,Baianos,18,,-24.416667,-52.983333
br,baiao,Baião,06,,-7.733333,-38.916667
br,baiao,Baião,13,,-3.166667,-43.65
br,baiao,Baião,16,12316,-2.683333,-49.683333
br,baiao,Baião,22,,-6,-38.433333
br,baiao,Baião,30,,-7.833333,-38.083333
br,baiao,Baião,31,,-11.516667,-47.7
br,baiao grande,Baião Grande,30,,-8.8,-37.35
br,baiaozinho,Baiãozinho,30,,-8.783333,-37.333333
br,baia piraputanga,Baía Piraputanga,14,,-16.3,-56.133333
br,baia santa clara,Baía Santa Clara,11,,-19.433333,-55.383333
br,baia verde,Baía Verde,11,,-17.716667,-55.55
br,baiazinha,Baìazinha,11,,-21.716667,-57.283333
br,baiepau,Baiepau,25,,2.116667,-59.75
br,baile,Baile,11,,-22.233333,-53.4
br,bailique,Bailique,03,,.966667,-50.066667
br,bailique,Bailique,16,,-2.766667,-49.783333
br,bainho,Bainho,14,,-16.683333,-55.433333
br,baioes,Baiões,15,,-20.616667,-45.333333
br,baireiro,Baireiro,23,,-27.916667,-53.1
br,bairrinho,Bairrinho,27,,-22.6,-47.1
br,bairro 29 de marco,Bairro 29 de Março,21,,-22.890486,-43.617473
br,bairro aboboral,Bairro Aboboral,27,,-24.501115,-48.06458
br,bairro adriana,Bairro Adriana,21,,-22.885084,-43.546989
br,bairro afonso vizeu,Bairro Afonso Vizeu,21,,-22.923753,-43.57351
br,bairro aguacai,Bairro Aguaçai,27,,-23.657582,-47.077553
br,bairro agua do barreiro,Bairro Água do Barreiro,18,,-23.904024,-50.597442
br,bairro agua do oleo,Bairro Água do Óleo,18,,-23.002824,-49.992588
br,bairro agua esprajada,Bairro Agua Esprajada,27,,-23.660448,-47.047817
br,bairro agua feia,Bairro Água Feia,18,,-23.161906,-49.941401
br,bairro agua preta,Bairro Água Preta,27,,-23.12204,-46.696542
br,bairro agua quente,Bairro Água Quente,27,,-24.8814,-48.5078
br,bairro agulhas negras,Bairro Agulhas Negras,21,,-22.933696,-43.562129
br,bairro alegre,Bairro Alegre,27,,-23.284309,-46.70061
br,bairro alessandra,Bairro Alessandra,21,,-22.886949,-43.574175
br,bairro alto,Bairro Alto,18,,-24.811987,-48.930066
br,bairro alto,Bairro Alto,27,,-23.483333,-45.35
br,bairro anapolis,Bairro Anapolis,21,,-22.914538,-43.612669
br,bairro andre lopes,Bairro André Lopes,27,,-24.6015,-48.4059
br,bairro andreotti,Bairro Andreotti,18,,-23.173442,-52.003231
br,bairro araujo,Bairro Araujo,21,,-22.824553,-43.321463
br,bairro areado,Bairro Areado,27,,-24.565427,-48.788241
br,bairro areia branca,Bairro Areia Branca,18,,-23.936,-50.4617
br,bairro areia branca,Bairro Areia Branca,27,,-24.9524,-48.5636
br,bairro arixiguana,Bairro Arixiguana,18,,-23.756404,-50.617754
br,bairro aroeira,Bairro Aroeira,27,,-24.8657,-48.5494
br,bairro aurora,Bairro Aurora,21,,-22.911084,-43.582225
br,bairro badenfurt,Bairro Badenfurt,26,,-26.8824,-49.1405
br,bairro bairro nova mairinque,Bairro Bairro Nova Mairinque,27,,-23.537143,-47.19141
br,bairro bananal pequeno,Bairro Bananal Pequeno,27,,-24.501354,-48.077049
br,bairro barata,Bairro Barata,21,,-22.891761,-43.44159
br,bairro barra das provas,Bairro Barra das Provas,18,,-24.674084,-48.94434
br,bairro barra das provas,Bairro Barra das Provas,27,,-24.671723,-48.938669
br,bairro barra do betari,Bairro Barra do Betari,27,,-24.612374,-48.613839
br,bairro barra do taquaruvira,Bairro Barra do Taquaruvira,27,,-24.611662,-48.622733
br,bairro barra grande,Bairro Barra Grande,18,,-23.98695,-50.598238
br,bairro barra grande,Bairro Barra Grande,27,,-24.508149,-48.923928
br,bairro barra mansa,Bairro Barra Mansa,27,,-23.9589,-47.1318
br,bairro barra ribeirao do ferro,Bairro Barra Ribeirão do Ferro,26,,-27.0659,-49.6501
br,bairro barreiro,Bairro Barreiro,18,,-25.328333,-50.333611
br,bairro batistini,Bairro Batistini,27,,-23.754,-46.5799
br,bairro beija-flor,Bairro Beija-flor,27,,-23.4341,-46.153
br,bairro bela vista,Bairro Bela Vista,18,,-23.054823,-49.977433
br,bairro bela vista,Bairro Bela Vista,26,,-26.8982,-49.0091
br,bairro bela vista,Bairro Bela Vista,27,,-24.7656,-48.487
br,bairro belchior,Bairro Belchior,27,,-23.989623,-47.120335
br,bairro benfica,Bairro Benfica,18,,-25.52638,-50.46454
br,bairro biquinha,Bairro Biquinha,21,,-22.331866,-42.94597
br,bairro boava,Bairro Boava,27,,-23.7719,-47.1384
br,bairro boa vista,Bairro Boa Vista,27,,-23.003611,-46.511111
br,bairro bombas,Bairro Bombas,27,,-24.606013,-48.653359
br,bairro borore,Bairro Bororé,27,,-23.7712,-46.6428
br,bairro briquituba,Bairro Briquituba,27,,-23.519922,-47.238803
br,bairro buriti,Bairro Buriti,21,,-22.716239,-43.431153
br,bairro cachoeira,Bairro Cachoeira,18,,-23.227782,-49.962968
br,bairro cachoeira,Bairro Cachoeira,27,,-23.3707,-46.221
br,bairro caetetuba,Bairro Caetetuba,27,,-23.136667,-46.643611
br,bairro cafundo,Bairro Cafundo,27,,-23.3564,-46.235
br,bairro caiacanga,Bairro Caiacanga,27,,-24.560803,-48.019199
br,bairro caiacoara,Bairro Caiaçoara,27,,-23.183056,-46.680833
br,bairro caicara,Bairro Caiçara,18,,-23.112365,-52.090032
br,bairro camargo,Bairro Camargo,27,,-24.509418,-48.581276
br,bairro camarista meier,Bairro Camarista Méier,21,,-22.913748,-43.299144
br,bairro cambui,Bairro Cambuí,18,,-23.864235,-50.417751
br,bairro caminho do sitio,Bairro Caminho do Sítio,27,,-23.8288,-46.7449
br,bairro campininha,Bairro Campininha,27,,-23.240833,-46.654167
br,bairro campista,Bairro Campista,27,,-22.673911,-45.594034
br,bairro campo dos aleixos,Bairro Campo dos Aleixos,27,,-23.070674,-46.685954
br,bairro cangume,Bairro Cangume,27,,-24.60504,-48.799002
br,bairro capelinha,Bairro Capelinha,27,,-23.733407,-47.024174
br,bairro capinzal,Bairro Capinzal,18,,-23.910858,-50.552356
br,bairro capoeirinha,Bairro Capoeirinha,27,,-24.501628,-48.84988
br,bairro capuava,Bairro Capuava,27,,-23.305983,-46.622562
br,bairro capuavinha,Bairro Capuavinha,27,,-23.325211,-46.56057
br,bairro caracas,Bairro Caraças,27,,-24.665456,-48.893277
br,bairro caracol,Bairro Caracol,27,,-24.569933,-48.555267
br,bairro caratuva,Bairro Caratuva,18,,-23.650488,-50.738772
br,bairro carmo messias,Bairro Carmo Messias,27,,-23.748419,-47.047244
br,bairro caroba,Bairro Caroba,21,,-22.896039,-43.544601
br,bairro carolina,Bairro Carolina,21,,-22.875711,-43.538073
br,bairro castelhanos,Bairro Castelhanos,27,,-24.554,-48.4302
br,bairro cateto,Bairro Cateto,18,,-23.204755,-49.797512
br,bairro cebandilha,Bairro Cebandilha,27,,-23.598324,-47.225261
br,bairro chapero,Bairro Chaperó,21,,-22.810306,-43.758947
br,bairro congonhas,Bairro Congonhas,18,,-23.574597,-50.617735
br,bairro convento,Bairro Convento,27,,-23.454208,-46.070375
br,bairro corda,Bairro Corda,27,,-24.591759,-48.389891
br,bairro coreiropolis,Bairro Coreirópolis,27,,-24.523967,-48.838267
br,bairro corrego da lavra,Bairro Córrego da Lavra,27,,-24.630792,-48.768138
br,bairro corrego da onca,Bairro Córrego da Onça,27,,-24.792761,-48.527787
br,bairro corrego das mocas,Bairro Córrego das Moças,18,,-24.725963,-48.820886
br,bairro corrego do chiqueiro,Bairro Córrego do Chiqueiro,27,,-24.502988,-48.976585
br,bairro cortesias,Bairro Cortesias,27,,-24.745674,-48.14466
br,bairro cristovao,Bairro Cristóvão,27,,-24.560246,-48.06816
br,bairro cruz alta,Bairro Cruz Alta,27,,-24.6499,-48.3138
br,bairro cruzeiro,Bairro Cruzeiro,23,,-29.1708,-51.1404
br,bairro curucutu,Bairro Curucutu,27,,-23.8668,-46.5945
br,bairro da caceia,Bairro da Caceia,27,,-23.325825,-46.639017
br,bairro da cachoeira,Bairro da Cachoeira,27,,-23.73059,-47.043646
br,bairro da cachoeirinha,Bairro da Cachoeirinha,27,,-23.911665,-47.067539
br,bairro da capelinha,Bairro da Capelinha,27,,-23.4887,-46.2311
br,bairro da capivara,Bairro da Capivara,27,,-24.530848,-48.958069
br,bairro da cascata,Bairro da Cascata,27,,-23.261507,-46.71176
br,bairro da enseada,Bairro da Enseada,27,,-23.716667,-45.433333
br,bairro da estrela,Bairro da Estrela,18,,-25.952069,-50.014187
br,bairro da fazenda,Bairro da Fazenda,27,,-24.651899,-48.81359
br,bairro da forquilha,Bairro da Forquilha,27,,-24.660452,-48.73796
br,bairro da imbuia,Bairro da Imbuia,18,,-24.866667,-51
br,bairro da lagoa,Bairro da Lagoa,18,,-25.4931,-50.6615
br,bairro da lagoa,Bairro da Lagoa,27,,-23.583853,-47.035675
br,bairro da luz,Bairro da Luz,21,,-22.900972,-43.311835
br,bairro da onca,Bairro da Onça,27,,-24.698708,-48.352026
br,bairro da palmerinha,Bairro da Palmerinha,27,,-23.872981,-47.008876
br,bairro da pedreira,Bairro da Pedreira,27,,-23.064444,-46.628611
br,bairro da ponte,Bairro da Ponte,27,,-23.190731,-47.533909
br,bairro da prata,Bairro da Prata,21,,-22.383605,-42.953589
br,bairro das andorinhas,Bairro das Andorinhas,18,,-24.677381,-48.580755
br,bairro das andorinhas,Bairro das Andorinhas,27,,-24.670499,-48.584397
br,bairro das chagas,Bairro das Chagas,27,,-24.504068,-48.959916
br,bairro da serra,Bairro da Serra,27,,-24.551673,-48.68187
br,bairro da serra da caneca fina,Bairro da Serra da Caneca Fina,21,,-22.485902,-42.959023
br,bairro da serrinha,Bairro da Serrinha,27,,-23.51418,-47.060286
br,bairro das garcas,Bairro das Garças,27,,-23.616079,-47.02439
br,bairro das ilhas,Bairro das Ilhas,27,,-24.7789,-48.5143
br,bairro das lagoas,Bairro das Lagoas,27,,-23.947929,-47.056116
br,bairro das laranjeiras,Bairro das Laranjeiras,27,,-23.380171,-46.708493
br,bairro das palmeiras,Bairro das Palmeiras,27,,-23.381396,-46.625779
br,bairro das pederneiras,Bairro das Pederneiras,27,,-21.733333,-51.833333
br,bairro das pedras,Bairro das Pedras,27,,-23.909982,-47.029584
br,bairro das ritas,Bairro das Ritas,27,,-23.90769,-47.097686
br,bairro das rosas,Bairro das Rosas,27,,-23.971,-47.1078
br,bairro das senhorinhas,Bairro das Senhorinhas,27,,-23.9697,-47.0471
br,bairro da usina,Bairro da Usina,27,,-23.087222,-46.643889
br,bairro da velha central,Bairro da Velha Central,26,,-26.9312,-49.1275
br,bairro de furnas,Bairro de Furnas,27,,-24.535039,-48.726149
br,bairro de itapuca,Bairro de Itapuca,21,,-22.474,-44.4783
br,bairro de lavras,Bairro de Lavras,27,,-24.7746,-48.2151
br,bairro de sao lourenco,Bairro de São Lourenco,27,,-23.816667,-46.05
br,bairro descalvado,Bairro Descalvado,27,,-24.64509,-48.646805
br,bairro diana,Bairro Diana,21,,-22.91526,-43.579973
br,bairro do albuquerque,Bairro do Albuquerque,21,,-22.379099,-42.920661
br,bairro do areado,Bairro do Areado,27,,-24.697398,-48.373357
br,bairro do azeite,Bairro do Azeite,27,,-24.814134,-48.177178
br,bairro do barreiro,Bairro do Barreiro,27,,-23.466667,-49.283333
br,bairro do barro,Bairro do Barro,27,,-23.716667,-45.4
br,bairro do bau,Bairro do Baú,27,,-24.564639,-48.567185
br,bairro do betari,Bairro do Betari,27,,-24.579576,-48.630717
br,bairro do bonito,Bairro do Bonito,27,,-22.833333,-45.033333
br,bairro do breu,Bairro do Breu,27,,-23.075833,-46.742222
br,bairro do capim azedo,Bairro do Capim Azedo,27,,-23.675824,-47.204927
br,bairro do capim grosso,Bairro do Capim Grosso,20,,-7.916667,-41.883333
br,bairro do capuava,Bairro do Capuava,18,,-24.695217,-48.81938
br,bairro do carmo,Bairro do Carmo,27,,-23.630976,-47.099544
br,bairro do corrego dos boavas,Bairro do Córrego dos Boavas,27,,-24.608912,-48.815561
br,bairro do cupim,Bairro do Cupim,27,,-23.716568,-47.22927
br,bairro do curral,Bairro do Curral,27,,-23.653796,-47.176678
br,bairro do descampado,Bairro do Descampado,18,,-24.716462,-48.943762
br,bairro do engano,Bairro do Engano,27,,-23.997723,-47.152318
br,bairro do feital,Bairro do Feital,27,,-23.71865,-47.19626
br,bairro do feixe,Bairro do Feixe,18,,-24.677873,-48.591815
br,bairro do felisberto,Bairro do Felisberto,18,,-24.033333,-50.65
br,bairro do fria,Bairro do Fria,27,,-24.726817,-48.473872
br,bairro do frigorifico,Bairro do Frigorífico,27,,-24.68962,-48.111615
br,bairro do gamba,Bairro do Gambá,18,,-23.6242,-50.6321
br,bairro do goiana,Bairro do Goianã,27,,-23.556161,-47.151466
br,bairro do gritem,Bairro do Gritem,18,,-25.910306,-50.160861
br,bairro do imbui,Bairro do Imbuí,21,,-22.391481,-42.997087
br,bairro do jacaranda,Bairro do Jacarandá,21,,-22.4306,-42.914187
br,bairro do jacu,Bairro do Jacu,27,,-23.316667,-48.533333
br,bairro do juquia,Bairro do Juquiá,27,,-23.922885,-47.117136
br,bairro do lajeado,Bairro do Lajeado,27,,-24.597722,-48.871156
br,bairro do laranjal,Bairro do Laranjal,18,,-24.717075,-48.97021
br,bairro do machadinho,Bairro do Machadinho,27,,-23.062222,-46.694722
br,bairro do olinto,Bairro do Olinto,27,,-23.738523,-47.099322
br,bairro do paiol,Bairro do Paiol,27,,-23.4178,-46.0019
br,bairro do paiol pequeno,Bairro do Paiol Pequeno,27,,-23.632525,-47.193981
br,bairro do paiolzinho,Bairro do Paiolzinho,27,,-23.73028,-47.202607
br,bairro do palmital,Bairro do Palmital,27,,-24.536206,-48.831648
br,bairro do pavao,Bairro do Pavão,27,,-23.629523,-47.15224
br,bairro do pedroso,Bairro do Pedroso,27,,-22.8,-45.066667
br,bairro do piai,Bairro do Piaí,27,,-23.7538,-47.2077
br,bairro do quebra-frascos,Bairro do Quebra-Frascos,21,,-22.415559,-42.996086
br,bairro do ribeirao,Bairro do Ribeirão,27,,-23.509081,-47.152617
br,bairro do rosario,Bairro do Rosário,27,,-23.258282,-46.541276
br,bairro dos alexandres,Bairro dos Alexandres,27,,-23.972938,-47.161631
br,bairro dos alves,Bairro dos Alves,18,,-23.884958,-50.577372
br,bairro dos alves,Bairro dos Alves,27,,-23.810972,-47.201294
br,bairro do sao jose,Bairro do São José,20,,-3.5175,-42.181667
br,bairro dos baianos,Bairro dos Baianos,18,,-23.5983,-50.5083
br,bairro dos barnabes,Bairro dos Barnabes,27,,-23.981137,-47.155438
br,bairro dos camargo,Bairro dos Camargo,27,,-23.92,-47.1044
br,bairro dos canudos,Bairro dos Canudos,27,,-24.8965,-48.5395
br,bairro dos carmos,Bairro dos Carmos,27,,-23.87496,-47.080122
br,bairro dos carreiros,Bairro dos Carreiros,18,,-23.861141,-50.569381
br,bairro dos coelhos,Bairro dos Coelhos,27,,-24.533159,-48.937854
br,bairro dos domingos,Bairro dos Domingos,27,,-23.730769,-47.120324
br,bairro do setubal,Bairro do Setubal,27,,-23.574557,-47.225422
br,bairro dos farias,Bairro dos Farias,27,,-23.877441,-47.125403
br,bairro dos ferreiras,Bairro dos Ferreiras,27,,-23.942653,-47.171522
br,bairro dos filipes,Bairro dos Filipes,27,,-23.78161,-47.140576
br,bairro dos garcias,Bairro dos Garcias,27,,-23.859163,-47.084951
br,bairro dos godoy,Bairro dos Godoy,18,,-23.580978,-50.522177
br,bairro dos grilos,Bairro dos Grilos,27,,-23.7737,-47.0319
br,bairro dos henrique,Bairro dos Henrique,27,,-24.630252,-48.806854
br,bairro dos justinos,Bairro dos Justinos,27,,-23.922898,-47.077277
br,bairro dos justos,Bairro dos Justos,27,,-23.946602,-47.024722
br,bairro dos mendes,Bairro dos Mendes,27,,-23.762692,-47.025153
br,bairro dos meninos,Bairro dos Meninos,27,,-24.556565,-48.142218
br,bairro dos mineiros,Bairro dos Mineiros,18,,-23.592538,-50.521858
br,bairro dos nogueiras,Bairro dos Nogueiras,27,,-23.4316,-46.039
br,bairro do soroca mirim,Bairro do Soroca Mirim,27,,-23.674431,-47.111738
br,bairro dos padeiros,Bairro dos Padeiros,27,,-23.88893,-47.046364
br,bairro dos paes,Bairro dos Paes,27,,-23.689519,-47.242808
br,bairro dos patos,Bairro dos Patos,27,,-22.133333,-49.2
br,bairro dos pederneiras,Bairro dos Pederneiras,27,,-21.733333,-51.833333
br,bairro dos pedrosos,Bairro dos Pedrosos,27,,-23.732722,-47.078196
br,bairro dos pereiras,Bairro dos Pereiras,27,,-23.117778,-46.67
br,bairro dos pessegueiros,Bairro dos Pessegueiros,21,,-22.316032,-42.942636
br,bairro dos pintos,Bairro dos Pintos,27,,-23.216389,-46.542778
br,bairro dos pires,Bairro dos Pires,27,,-23.665542,-47.006251
br,bairro dos policenos,Bairro dos Policenos,18,,-25.499167,-50.384722
br,bairro dos pratas,Bairro dos Pratas,27,,-23.953653,-47.047182
br,bairro dos remdios,Bairro dos Remdios,27,,-23.3217,-46.0904
br,bairro dos remedios,Bairro dos Remédios,27,,-23.267562,-46.535873
br,bairro dos ribeiro,Bairro dos Ribeiro,27,,-23.7746,-47.062207
br,bairro dos rodrigues,Bairro dos Rodrigues,27,,-23.725725,-47.091546
br,bairro dos rosas,Bairro dos Rosas,27,,-23.857348,-47.004594
br,bairro dos soares,Bairro dos Soares,27,,-23.8425,-47.0075
br,bairro dos tatetos,Bairro dos Tatetos,27,,-23.8288,-46.5727
br,bairro dos telles,Bairro dos Telles,18,,-23.833333,-50.45
br,bairro dos valos,Bairro dos Valos,27,,-23.300813,-46.666147
br,bairro dos veigas,Bairro dos Veigas,27,,-23.401336,-46.508078
br,bairro dos vitalino,Bairro dos Vitalino,27,,-23.9512,-47.1619
br,bairro do tanque,Bairro do Tanque,27,,-23.030556,-46.537222
br,bairro do taquari,Bairro do Taquari,27,,-25.0562,-48.1016
br,bairro do teles,Bairro do Teles,18,,-23.833333,-50.45
br,bairro do tigre,Bairro do Tigre,18,,-23.600948,-50.649561
br,bairro do tude,Bairro do Tude,27,,-24.543418,-48.556429
br,bairro do turvo,Bairro do Turvo,27,,-24.65657,-48.133115
br,bairro do vieirinha,Bairro do Vieirinha,27,,-23.747117,-47.232302
br,bairro do vital,Bairro do Vital,27,,-22.916667,-47.366667
br,bairro do xisto,Bairro do Xisto,18,,-23.848671,-50.54375
br,bairro duvida,Bairro Dúvida,27,,-24.9468,-48.5397
br,bairro elaine,Bairro Elaine,21,,-22.923769,-43.52259
br,bairro elisa maria,Bairro Elisa Maria,21,,-22.872494,-43.573322
br,bairro ema da flora,Bairro Ema da Flora,18,,-23.073585,-49.981238
br,bairro engenheiro neiva,Bairro Engenheiro Neiva,27,,-22.784039,-45.163282
br,bairro engenho,Bairro Engenho,27,,-24.52859,-48.049326
br,bairro engenho velho,Bairro Engenho Velho,27,,-24.510945,-48.86885
br,bairro engo goulart,Bairro Engo Goulart,27,,-23.49587,-46.509549
br,bairro esplanada,Bairro Esplanada,21,,-22.872965,-43.550881
br,bairro esplanada,Bairro Esplanada,23,,-29.2006,-51.2062
br,bairro estancia lago azul,Bairro Estância Lago Azul,27,,-23.298599,-46.713833
br,bairro estreitinho,Bairro Estreitinho,18,,-24.899,-48.5799
br,bairro estreitinho,Bairro Estreitinho,27,,-24.899765,-48.568937
br,bairro faxina,Bairro Faxina,18,,-25.447778,-50.312222
br,bairro feital,Bairro Feital,27,,-24.635577,-48.542046
br,bairro figueira,Bairro Figueira,27,,-23.2909,-46.1271
br,bairro florianos,Bairro Florianos,18,,-25.464167,-50.360833
br,bairro fortaleza,Bairro Fortaleza,26,,-26.8536,-49.0558
br,bairro fregueisa da escada,Bairro Fregueisa da Escada,27,,-23.3844,-46.0581
br,bairro geremuniz,Bairro Geremuniz,27,,-23.3554,-46.2082
br,bairro godinho,Bairro Godinho,27,,-23.9499,-47.1022
br,bairro goiabal,Bairro Goiabal,27,,-23.3505,-46.0815
br,bairro grotao,Bairro Grotão,21,,-22.386016,-42.922211
br,bairro guarau,Bairro Guaraú,27,,-24.8067,-48.09
br,bairro guarauna de docas,Bairro Guaraúna de Docas,18,,-25.4275,-50.273056
br,bairro guarauninha,Bairro Guarauninha,18,,-25.449167,-50.295
br,bairro guarda-mao,Bairro Guarda-Mão,27,,-24.597842,-48.851696
br,bairro guaxinduva,Bairro Guaxinduva,27,,-23.109722,-46.511944
br,bairro gurutuba,Bairro Gurutuba,27,,-24.63017,-48.826301
br,bairro gurutuba dos boavas,Bairro Gurutuba dos Boavas,27,,-24.614329,-48.824211
br,bairro gurutuba dos martins,Bairro Gurutuba dos Martins,27,,-24.612166,-48.833488
br,bairro hilario,Bairro Hilário,27,,-24.61987,-48.286907
br,bairro iguatemi,Bairro Iguatemi,27,,-24.527224,-48.909816
br,bairro ilha do cedro,Bairro Ilha do Cedro,27,,-24.736309,-48.520751
br,bairro indaiatuba,Bairro Indaiatuba,27,,-24.8752,-48.5434
br,bairro iolanda danon,Bairro Iolanda Danon,21,,-22.77303,-43.511296
br,bairro ipatinga,Bairro Ipatinga,21,,-22.879665,-43.541867
br,bairro iracema,Bairro Iracema,21,,-22.880965,-43.592762
br,bairro irmaos prata,Bairro Irmãos Prata,21,,-22.772226,-43.417413
br,bairro itapeti,Bairro Itapeti,27,,-23.4077,-46.182
br,bairro itapetinga,Bairro Itapetinga,27,,-23.158889,-46.5475
br,bairro itapoa,Bairro Itapoã,21,,-22.769086,-43.505454
br,bairro itoupava do norte,Bairro Itoupava do Norte,26,,-26.8895,-49.0669
br,bairro ivaporunduva,Bairro Ivaporunduva,27,,-24.5629,-48.3928
br,bairro jabour,Bairro Jabour,21,,-22.879052,-43.494169
br,bairro jardim aladim,Bairro Jardim Aladim,27,,-23.808063,-46.736069
br,bairro jardim alamo,Bairro Jardim Alamo,27,,-23.8105,-46.742
br,bairro jardim almeida prado,Bairro Jardim Almeida Prado,27,,-23.776796,-46.691306
br,bairro jardim alvorada,Bairro Jardim Alvorada,27,,-23.7651,-46.6854
br,bairro jardim betania,Bairro Jardim Betânia,27,,-23.598496,-47.020454
br,bairro jardim cambara,Bairro Jardim Cambará,27,,-23.419852,-46.555211
br,bairro jardim casa grande,Bairro Jardim Casa Grande,27,,-23.303392,-46.747034
br,bairro jardim city,Bairro Jardim City,27,,-23.425544,-46.540098
br,bairro jardim cruzeiro,Bairro Jardim Cruzeiro,27,,-23.544938,-47.191916
br,bairro jardim das montanhas,Bairro Jardim das Montanhas,27,,-23.401504,-46.620072
br,bairro jardim das orquideas,Bairro Jardim das Orquideas,27,,-23.7512,-46.5999
br,bairro jardim dos reis,Bairro Jardim dos Reis,27,,-23.305583,-46.722847
br,bairro jardim erpim,Bairro Jardim Erpim,27,,-23.791839,-46.712823
br,bairro jardim heliana,Bairro Jardim Heliana,27,,-23.294685,-46.748615
br,bairro jardim itaim,Bairro Jardim Itaim,27,,-23.810309,-46.713685
br,bairro jardim leonor,Bairro Jardim Leonor,18,,-23.835387,-50.415155
br,bairro jardim lucelia,Bairro Jardim Lucélia,27,,-23.758653,-46.678582
br,bairro jardim margarida,Bairro Jardim Margarida,27,,-23.615382,-47.040649
br,bairro jardim marilda,Bairro Jardim Marilda,27,,-23.787872,-46.681785
br,bairro jardim monte alegre,Bairro Jardim Monte Alegre,27,,-23.773822,-46.688939
br,bairro jardim nossa senhora aparecida,Bairro Jardim Nossa Senhora Aparecida,27,,-23.259514,-46.737097
br,bairro jardim novo parelheiros,Bairro Jardim Novo Parelheiros,27,,-23.832072,-46.723512
br,bairro jardim parate,Bairro Jardim Parate,27,,-23.318,-46.1057
br,bairro jardim paulo afonso,Bairro Jardim Paulo Afonso,27,,-23.7853,-46.718346
br,bairro jardim potao vermelho,Bairro Jardim Potão Vermelho,27,,-23.603025,-47.00548
br,bairro jardim primavera,Bairro Jardim Primavera,18,,-23.829295,-50.413947
br,bairro jardim sabia,Bairro Jardim Sabiá,27,,-23.782749,-46.692143
br,bairro jardim santa barbara,Bairro Jardim Santa Bárbara,18,,-23.8545,-50.4079
br,bairro jardim santa teresinha,Bairro Jardim Santa Teresinha,27,,-23.8513,-46.6958
br,bairro jardim sao goncalo,Bairro Jardim São Gonçalo,27,,-23.33071,-46.606224
br,bairro jardim silveira,Bairro Jardim Silveira,27,,-23.849888,-46.713126
br,bairro jardim suico,Bairro Jardim Suiço,27,,-23.330041,-46.59592
br,bairro jardim varginha,Bairro Jardim Varginha,27,,-23.783915,-46.68132
br,bairro jardim virginia,Bairro Jardim Virginia,27,,-23.37094,-46.710954
br,bairro jesus,Bairro Jesus,27,,-24.671723,-48.938669
br,bairro juqueri-mirim,Bairro Juqueri-Mirim,27,,-23.363235,-46.618251
br,bairro juquia,Bairro Juquiá,27,,-23.974652,-47.177667
br,bairro lagoinha,Bairro Lagoinha,21,,-22.796513,-43.599385
br,bairro lagoinhas,Bairro Lagoinhas,27,,-23.2894,-46.0055
br,bairro lajeadinho,Bairro Lajeadinho,27,,-23.700414,-47.168061
br,bairro lajeado liso,Bairro Lajeado Liso,18,,-23.945099,-50.564365
br,bairro lambari,Bairro Lambari,27,,-23.3671,-46.1801
br,bairro laranjal,Bairro Laranjal,27,,-23.038333,-46.606944
br,bairro lencol,Bairro Lençol,27,,-24.590045,-48.034708
br,bairro limeira,Bairro Limeira,18,,-25.319722,-50.405278
br,bairro limoeiro,Bairro Limoeiro,27,,-24.503043,-48.880641
br,bairro macedo,Bairro Macedo,21,,-22.878141,-43.576786
br,bairro machado,Bairro Machado,27,,-23.345815,-46.657192
br,bairro mambuca,Bairro Mambuca,18,,-23.931727,-50.65377
br,bairro manuel gomes,Bairro Manuel Gomes,27,,-24.746369,-48.170965
br,bairro maraba,Bairro Marabá,18,,-23.601,-50.6135
br,bairro maracana,Bairro Maracanã,27,,-23.158056,-46.718889
br,bairro maracatu,Bairro Maracatu,27,,-23.3574,-46.1202
br,bairro marciana,Bairro Marciana,27,,-22.8278,-45.2183
br,bairro maria jose,Bairro Maria José,21,,-22.922514,-43.553091
br,bairro maria luiza,Bairro Maria Luiza,21,,-22.897782,-43.590049
br,bairro mario lombardi,Bairro Mário Lombardi,21,,-22.883722,-43.602665
br,bairro marli,Bairro Marli,18,,-23.859609,-50.395315
br,bairro mato-dentro,Bairro Mato-Dentro,27,,-23.287524,-46.644398
br,bairro mineiros,Bairro Mineiros,27,,-24.546469,-48.90095
br,bairro monte belo,Bairro Monte Belo,27,,-21.316667,-47.7
br,bairro morada do campo,Bairro Morada do Campo,21,,-22.919911,-43.547725
br,bairro moricaba,Bairro Moriçaba,21,,-22.89842,-43.530601
br,bairro morro grande,Bairro Morro Grande,27,,-23.3092,-46.1768
br,bairro murundu,Bairro Murundu,27,,-23.8034,-47.2457
br,bairro nhunguara,Bairro Nhunguara,27,,-24.585952,-48.425262
br,bairro nossa senhora aparecida,Bairro Nossa Senhora Aparecida,21,,-22.910922,-43.621759
br,bairro nossa senhora de fatima,Bairro Nossa Senhora de Fátima,27,,-23.008603,-47.503935
br,bairro nova cidade,Bairro Nova Cidade,21,,-22.902599,-43.590674
br,bairro nova guara,Bairro Nova Guará,27,,-22.8086,-45.1878
br,bairro nova jersei,Bairro Nova Jersei,21,,-22.92592,-43.631227
br,bairro ocorrencia,Bairro Ocorrência,18,,-24.671129,-48.609922
br,bairro ocorrencia,Bairro Ocorrência,27,,-24.663243,-48.598206
br,bairro oito e meio,Bairro Oito e Meio,27,,-23.9572,-47.2365
br,bairro ouro grande,Bairro Ouro Grande,18,,-23.187611,-49.877633
br,bairro padre andre,Bairro Padre André,27,,-24.617166,-48.017789
br,bairro paiol grande,Bairro Paiol Grande,27,,-23.519504,-47.021335
br,bairro palmital,Bairro Palmital,27,,-24.660801,-48.84264
br,bairro paraiso,Bairro Paraíso,21,,-22.4532,-44.4315
br,bairro parque pauh,Bairro Parque Pauh,27,,-23.304991,-46.734816
br,bairro parque petropolis,Bairro Parque Petropolis,27,,-23.383376,-46.630789
br,bairro parque santa ines,Bairro Parque Santa Inês,27,,-23.405055,-46.671138
br,bairro parque sant'ana,Bairro Parque Sant'Ana,27,,-23.250586,-46.749568
br,bairro passa-vinte,Bairro Passa-Vinte,27,,-24.518231,-48.736822
br,bairro passo manso,Bairro Passo Manso,26,,-26.901,-49.1439
br,bairro pau arcado,Bairro Pau Arcado,27,,-23.2175,-46.721111
br,bairro paulo bueno,Bairro Paulo Bueno,27,,-23.9757,-47.0914
br,bairro pedra amarela,Bairro Pedra Amarela,18,,-23.620225,-50.523456
br,bairro pedra branca,Bairro Pedra Branca,27,,-23.9287,-47.0069
br,bairro pedregulho,Bairro Pedregulho,27,,-22.8001,-45.1996
br,bairro peixoto,Bairro Peixoto,21,,-22.964887,-43.190867
br,bairro piedade,Bairro Píedade,27,,-22.82122,-45.180502
br,bairro pimentas,Bairro Pimentas,18,,-24.84,-48.5671
br,bairro pimentas,Bairro Pimentas,27,,-24.8575,-48.5639
br,bairro pindauva do meio,Bairro Pindaúva do Meio,27,,-24.884,-48.0793
br,bairro pinhal,Bairro Pinhal,27,,-23.272,-46.1305
br,bairro pinheirinho,Bairro Pinheirinho,27,,-24.9872,-48.5555
br,bairro piraquara,Bairro Piraquara,21,,-22.895329,-43.436881
br,bairro pirucaia,Bairro Pirucaia,27,,-23.31898,-46.530479
br,bairro poco do peixe,Bairro Poço do Peixe,21,,-22.358179,-42.959313
br,bairro ponte alta,Bairro Ponte Alta,27,,-23.011042,-46.659852
br,bairro ponte lavrada,Bairro Ponte Lavrada,27,,-23.590751,-47.086537
br,bairro portao,Bairro Portão,27,,-23.224722,-46.628889
br,bairro portinho,Bairro Portinho,18,,-25.4234,-48.7248
br,bairro porto alegre,Bairro Porto Alegre,27,,-23.0061,-47.5156
br,bairro posse,Bairro Posse,21,,-22.378339,-42.992188
br,bairro praia grande,Bairro Praia Grande,27,,-24.667583,-48.653002
br,bairro preto,Bairro Preto,18,,-23.247792,-49.795019
br,bairro primeiro ribeirao,Bairro Primeiro Ribeirão,27,,-24.739912,-48.477459
br,bairro rancho alegre,Bairro Rancho Alegre,18,,-23.47002,-49.99377
br,bairro reginaldo,Bairro Reginaldo,27,,-24.8167,-48.4941
br,bairro reta grande,Bairro Reta Grande,18,,-23.928482,-50.568503
br,bairro ribeirao,Bairro Ribeirão,27,,-24.523373,-48.586352
br,bairro rio abaixo,Bairro Rio Abaixo,27,,-23.044167,-46.641111
br,bairro rio acima,Bairro Rio Acima,27,,-23.2175,-46.667778
br,bairro rio bonito,Bairro Rio Bonito,27,,-23.9455,-47.2159
br,bairro rio da prata,Bairro Rio da Prata,21,,-22.887021,-43.474311
br,bairro rio das pedras,Bairro Rio das Pedras,27,,-24.610084,-48.591589
br,bairro rio de una,Bairro Rio de Una,27,,-23.676908,-47.239699
br,bairro rio do meio,Bairro Rio do Meio,27,,-24.626622,-48.780832
br,bairro rio pardinho,Bairro Rio Pardinho,27,,-25.048181,-48.560783
br,bairro rio sao pedrinho,Bairro Rio São Pedrinho,27,,-25.02801,-48.552362
br,bairro rio vermelho,Bairro Rio Vermelho,27,,-24.9818,-48.5048
br,bairro rodeio bonito,Bairro Rodeio Bonito,18,,-23.895184,-50.6537
br,bairro rolas,Bairro Rolas,21,,-22.927834,-43.669136
br,bairro rosario,Bairro Rosário,27,,-23.564102,-47.072152
br,bairro salvador,Bairro Salvador,21,,-22.927161,-43.557533
br,bairro santa ana,Bairro Santa Ana,27,,-23.29,-46.0588
br,bairro santa clara do guandu,Bairro Santa Clara do Guandu,21,,-22.834552,-43.616959
br,bairro santa cruz,Bairro Santa Cruz,27,,-24.563788,-48.986959
br,bairro santa fe,Bairro Santa Fe,27,,-23.362257,-46.559105
br,bairro santa ines,Bairro Santa Inês,21,,-22.873793,-43.564115
br,bairro santa ines,Bairro Santa Inês,27,,-23.366465,-46.667085
br,bairro santa margarida,Bairro Santa Margarida,21,,-22.891235,-43.604982
br,bairro santa rosa,Bairro Santa Rosa,18,,-25.328889,-50.434722
br,bairro santa teresa,Bairro Santa Teresa,18,,-23.207475,-49.759507
br,bairro santiago,Bairro Santiago,27,,-24.949595,-48.501192
br,bairro santo angelo,Bairro Santo Angelo,27,,-23.2694,-46.1552
br,bairro santo antonio,Bairro Santo Antônio,27,,-23.307298,-46.563076
br,bairro sao caetano,Bairro São Caetano,21,,-22.4584,-44.4784
br,bairro sao joao,Bairro São João,27,,-22.647778,-47.469722
br,bairro sao joao velho,Bairro São João Velho,27,,-23.539117,-47.026037
br,bairro sao jose,Bairro São José,21,,-22.4468,-44.3066
br,bairro sao roque,Bairro São Roque,18,,-23.938805,-50.501158
br,bairro sao vicente,Bairro São Vicente,27,,-23.358673,-46.637886
br,bairro sapatu,Bairro Sapatu,27,,-24.600459,-48.362303
br,bairro sarassara,Bairro Sarassará,27,,-23.696503,-47.090789
br,bairro serrinha,Bairro Serrinha,27,,-24.575537,-48.885117
br,bairro silex,Bairro Silex,21,,-22.816178,-43.632293
br,bairro silvestre,Bairro Silvestre,21,,-22.86972,-43.551249
br,bairro soares,Bairro Soares,27,,-24.552286,-48.604202
br,bairro so bento,Bairro So Bento,27,,-23.4366,-46.2425
br,bairro sombra,Bairro Sombra,06,,-7.033333,-38.766667
br,bairro sorocabucu,Bairro Sorocabuçu,27,,-23.724435,-47.173041
br,bairro soroca mirim,Bairro Soroca Mirim,27,,-23.593556,-47.145372
br,bairro surubi,Bairro Surubi,21,,-22.4664,-44.4323
br,bairro tabuao,Bairro Tabuão,18,,-23.945,-50.4469
br,bairro tabuo,Bairro Tabuo,27,,-23.3986,-46.2396
br,bairro taipas de pedra,Bairro Taipas de Pedra,27,,-23.576764,-47.058255
br,bairro tamandare,Bairro Tamandaré,27,,-22.8236,-45.1894
br,bairro taquarivai,Bairro Taquarivaí,27,,-23.933333,-48.683333
br,bairro taquarucu,Bairro Taquaruçu,27,,-24.509623,-48.904362
br,bairro tatapui,Bairro Tatapuí,18,,-23.8781,-50.4185
br,bairro tatupeba,Bairro Tatupeba,18,,-24.703577,-48.755501
br,bairro tiburcio,Bairro Tibúrcio,27,,-23.7535,-47.1132
br,bairro toca do tatu,Bairro Toca do Tatu,27,,-23.7795,-46.6653
br,bairro tocas,Bairro Tocas,18,,-25.433889,-50.261389
br,bairro tres canais,Bairro Três Canais,18,,-24.8193,-48.5593
br,bairro tres corregos,Bairro Três Córregos,21,,-22.34358,-42.944115
br,bairro tronquini,Bairro Tronquini,18,,-23.388774,-52.691472
br,bairro turvo,Bairro Turvo,18,,-25.404444,-50.303611
br,bairro varadouro,Bairro Varadouro,27,,-23.277,-46.19
br,bairro varanguera,Bairro Varanguera,27,,-23.511385,-47.142949
br,bairro vargem do salto,Bairro Vargem do Salto,27,,-23.748065,-47.265368
br,bairro vargem grande,Bairro Vargem Grande,27,,-23.313729,-46.684388
br,bairro verava,Bairro Verava,27,,-23.7793,-47.102
br,bairro vicentino,Bairro Vicentino,21,,-22.4771,-44.4478
br,bairro vila davi,Bairro Vila Davi,27,,-23.29434,-46.553059
br,bairro vila ida,Bairro Vila Ida,27,,-23.308121,-46.731114
br,bairro vila marcelo,Bairro Vila Marcelo,27,,-23.789984,-46.724393
br,bairro vila nova silva,Bairro Vila Nova Silva,27,,-23.493246,-46.506187
br,bairro vila operaria,Bairro Vila Operária,27,,-23.311267,-46.566257
br,bairro vila renascenca,Bairro Vila Renascença,27,,-23.367294,-46.569123
br,bairro vila rosckel,Bairro Vila Rosckel,27,,-23.840793,-46.732542
br,bairro vila santista,Bairro Vila Santista,27,,-23.324088,-46.745462
br,bairro vinte e nove de marco,Bairro Vinte e Nove de Março,21,,-22.890486,-43.617473
br,bairro vista alegre,Bairro Vista Alegre,21,,-22.864745,-43.780628
br,bairro vitor dumas,Bairro Vitor Dumas,21,,-22.933531,-43.697766
br,bairro volta grande,Bairro Volta Grande,27,,-23.546234,-47.052449
br,bairro vorstaht,Bairro Vorstaht,26,,-26.9025,-49.039
br,bairro votorantim,Bairro Votorantim,27,,-23.336535,-46.569467
br,bairrro vila aurora,Bairrro Vila Aurora,27,,-23.431662,-46.747962
br,baitaca,Baitaca,18,,-26.0405,-49.8174
br,baitaca,Baitaca,23,,-27.583333,-53.066667
br,baitapora,Baitaporã,11,,-22.333333,-53.283333
br,baixa alegre,Baixa Alegre,05,,-13.133333,-39.466667
br,baixa,Baixa,05,,-12.45,-40.633333
br,baixa,Baixa,06,,-4.75,-37.566667
br,baixa,Baixa,13,,-4.183333,-43.75
br,baixa,Baixa,15,,-19.966667,-47.966667
br,baixa,Baixa,20,,-6.966667,-40.866667
br,baixa,Baixa,22,,-5.503056,-37.498056
br,baixa bonita,Baixa Bonita,15,,-15.783333,-40.9
br,baixa branca,Baixa Branca,06,,-5.216667,-37.816667
br,baixa calada,Baixa Calada,04,,-5.845833,-61.276944
br,baixa da alegria,Baixa da Alegria,22,,-5.302778,-37.278611
br,baixa da areia,Baixa da Areia,28,,-10.15,-37.05
br,baixa da aroeira,Baixa da Aroeira,22,,-5.335278,-37.343056
br,baixa da carnauba,Baixa da Carnaúba,22,,-6.283333,-37.95
br,baixa da conceicao,Baixa da Conceição,13,,-3.760556,-43.295
br,baixada do burgo,Baixada do Burgo,28,,-9.883333,-37.55
br,baixa d'agua,Baixa d'Agua,13,,-3.071111,-43.032778
br,baixa da imbira,Baixa da Imbira,05,,-11.966667,-44.633333
br,baixa da lagoa,Baixa da Lagoa,05,,-12.633333,-39.816667
br,baixa da lagoa,Baixa da Lagoa,20,,-6.483333,-41.35
br,baixa da manipeba,Baixa da Manipeba,22,,-5.483333,-35.666667
br,baixadao,Baixadão,11,,-19.183333,-54.316667
br,baixa da onca,Baixa da Onça,06,,-7.6,-38.916667
br,baixa da onca,Baixa da Onça,13,,-3.378611,-43.116667
br,baixa-d'areia,Baixa-D'Areia,28,,-10.15,-37.05
br,baixa da ribaca,Baixa da Ribacã,22,,-5.7,-35.783333
br,baixa das cotias,Baixa das Cotias,13,,-3.228611,-43.014722
br,baixa da serra,Baixa da Serra,13,,-6.583333,-43.866667
br,baixa da umburana,Baixa da Umburana,30,,-8.55,-37.766667
br,baixa da vaca,Baixa da Vaca,13,,-3.379167,-43.183333
br,baixa de sao francisco,Baixa de São Francisco,22,,-5.413056,-36.962222
br,baixa do angico,Baixa do Angico,20,,-7.9,-41.6
br,baixa do aroeira,Baixa do Aroeira,05,,-13.533333,-40.966667
br,baixa do bezerro,Baixa do Bezerro,22,,-5.567222,-36.143333
br,baixa do cafe,Baixa do Café,20,,-7.983333,-41.016667
br,baixa do cajueiro,Baixa do Cajueiro,20,,-5.383333,-41.4
br,baixa do caldeirao,Baixa do Caldeirão,20,,-6.3,-41.683333
br,baixa do capim,Baixa do Capim,13,,-3.879722,-43.128611
br,baixa do cedro,Baixa do Cedro,05,,-13.016667,-43.666667
br,baixa do cerral,Baixa do Cerral,20,,-6.233333,-42.133333
br,baixa do cocal i,Baixa do Cocal I,13,,-3.280833,-43.261389
br,baixa do cocal ii,Baixa do Cocal II,13,,-3.263889,-43.225833
br,baixa do correio,Baixa do Correio,22,,-5.722222,-37.478611
br,baixa do coxo,Baixa do Coxo,20,,-3.659722,-42.181667
br,baixa do curral,Baixa do Curral,20,,-6.233333,-41.933333
br,baixa do curralinho,Baixa do Curralinho,20,,-3.476667,-42.175556
br,baixa do feijao,Baixa do Feijão,22,,-5.346667,-36.055
br,baixa do jacare,Baixa do Jacaré,20,,-3.475556,-42.045833
br,baixa do jacu,Baixa do Jacu,05,,-12.45,-41.1
br,baixa do jatoba,Baixa do Jatobá,20,,-5.983333,-42.916667
br,baixa do jua,Baixa do Juá,06,,-4.65,-40.966667
br,baixa do jua,Baixa do Juá,22,,-5.151667,-36.850833
br,baixa do juazeiro,Baixa do Juazeiro,06,,-5.3,-40.783333
br,baixa do juazerio,Baixa do Juàzerio,22,,-5.573611,-37.396667
br,baixa do lago,Baixa do Lago,13,,-3.430556,-42.773611
br,baixa do lirio,Baixa do Lirio,20,,-6.916667,-40.55
br,baixa do maximo,Baixa do Máximo,05,,-10.45,-37.916667
br,baixa do meio,Baixa do Meio,06,,-7.166667,-40.383333
br,baixa do meio,Baixa do Meio,22,,-5.254167,-36.358333
br,baixa do mel,Baixa do Mel,20,,-6.816667,-41.483333
br,baixa do palmeira,Baixa do Palmeira,05,,-12.75,-39.216667
br,baixa do pau ferro,Baixa do Pau Ferro,30,,-8.766667,-36.616667
br,baixa do poco,Baixa do Poco,06,,-7.55,-38.766667
br,baixa do quinquim,Baixa do Quinquim,22,,-5.216667,-35.616667
br,baixa do quin-quin,Baixa do Quin-Quin,22,,-5.216667,-35.616667
br,baixa do recanto,Baixa do Recanto,30,,-7.9,-38.9
br,baixa do riachao,Baixa do Riachão,05,,-13.25,-39.116667
br,baixa do sapateiro,Baixa do Sapateiro,21,,-22.859849,-43.241578
br,baixa do serrote,Baixa do Serrote,02,,-9.033333,-37.516667
br,baixa do sitio,Baixa do Sítio,06,,-6.416667,-38.966667
br,baixa do tatu,Baixa do Tatu,22,,-5.649722,-37.289167
br,baixa do tingui,Baixa do Tingui,15,,-15.966667,-44.4
br,baixa escura,Baixa Escura,20,,-7.966667,-43.166667
br,baixa fechada,Baixa Fechada,06,,-6,-40.716667
br,baixa fechada,Baixa Fechada,22,,-5.684722,-37.367222
br,baixa feia,Baixa Feia,20,,-5.933333,-42.116667
br,baixa fria,Baixa Fria,06,,-3.35,-40.25
br,baixa fria,Baixa Fria,13,,-5.4,-43.333333
br,baixa fria,Baixa Fria,20,,-3.920833,-42.153889
br,baixa funda,Baixa Funda,02,,-9.333333,-36.266667
br,baixa funda,Baixa Funda,05,,-12.5,-41.25
br,baixa funda,Baixa Funda,13,,-3.088889,-42.318056
br,baixa funda,Baixa Funda,22,,-5.433333,-35.8
br,baixa grande,Baixa Grande,02,,-9.733333,-36.216667
br,baixa grande,Baixa Grande,05,7534,-11.95,-40.183333
br,baixa grande,Baixa Grande,06,,-2.869167,-40.6125
br,baixa grande,Baixa Grande,13,,-2.333333,-44.6
br,baixa grande,Baixa Grande,15,,-16.1,-42.116667
br,baixa grande,Baixa Grande,20,,-5.866667,-42.183333
br,baixa grande,Baixa Grande,22,,-5.71,-37.485833
br,baixa grande,Baixa Grande,30,,-7.583333,-39.683333
br,baixa nova,Baixa Nova,06,,-2.9,-40.4
br,baixao,Baixão,05,,-10.3,-43.55
br,baixao,Baixão,13,,-3.005556,-42.351111
br,baixao,Baixão,20,,-3.568611,-42.116111
br,baixao,Baixão,27,,-21.366667,-47.483333
br,baixao,Baixão,31,,-12.8,-46.633333
br,baixao da fortaleza,Baixão da Fortaleza,05,,-10.166667,-43.7
br,baixao da negra,Baixão da Negra,13,,-5.116667,-43.266667
br,baixao das cajazeiras,Baixão das Cajazeiras,20,,-3.4575,-42.078889
br,baixao de areia,Baixão de Areia,13,,-5.966667,-43.966667
br,baixao de forro,Baixão de Fôrro,05,,-11.15,-43.8
br,baixao do aniceto,Baixão do Aniceto,20,,-5.083333,-41.7
br,baixao do calista,Baixão do Calista,05,,-9.45,-43.183333
br,baixao do calisto,Baixão do Calisto,05,,-9.45,-43.183333
br,baixao do calixto,Baixão do Calixto,05,,-9.45,-43.183333
br,baixao do capim,Baixão do Capim,13,,-3.167222,-42.263889
br,baixao do cavalo velho,Baixão do Cavalo Velho,20,,-3.910833,-42.526944
br,baixao do cedro,Baixão do Cedro,13,,-3.296944,-42.258611
br,baixao do centro,Baixão do Centro,13,,-3.1125,-42.033889
br,baixao do chiqueiro,Baixão do Chiqueiro,20,,-3.755556,-42.269722
br,baixao do coco,Baixão do Coco,20,,-6.133333,-42.75
br,baixao do damasio,Baixão do Damásio,05,,-10.166667,-43.7
br,baixao do damazio,Baixão do Damázio,05,,-10.166667,-43.7
br,baixao do ferro,Baixão do Ferro,05,,-11.15,-43.8
br,baixao do lucas,Baixão do Lucas,05,,-12.433333,-44.9
br,baixao do melo,Baixão do Melo,20,,-6.066667,-41.033333
br,baixao do tubiba,Baixão do Tubiba,06,,-5.5,-41
br,baixao do umbuzeiro,Baixão do Umbuzeiro,20,,-7.233333,-41.883333
br,baixao grande,Baixão Grande,13,,-3.866944,-43.260278
br,baixao grande,Baixão Grande,20,,-3.547222,-42.116944
br,baixao i,Baixão I,13,,-3.088889,-43.4825
br,baixao i,Baixão I,20,,-3.400833,-42.047222
br,baixao ii,Baixão II,13,,-3.299444,-42.228889
br,baixao iii,Baixão III,13,,-3.101944,-43.114722
br,baixao iv,Baixão IV,13,,-3.135,-43.303056
br,baixao velho,Baixão Velho,05,,-10.1,-42.716667
br,baixa preta,Baixa Preta,06,,-4.283333,-38.516667
br,baixa quente,Baixa Quente,15,,-16.95,-42.066667
br,baixa redonda,Baixa Redonda,20,,-5.666667,-42.933333
br,baixa redonda,Baixa Redonda,30,,-7.916667,-38.783333
br,baixa sao miguel,Baixa São Miguel,22,,-5.516667,-35.7
br,baixas,Baixas,06,,-6.333333,-40.516667
br,baixas,Baixas,13,,-5.433333,-43.05
br,baixas,Baixas,17,,-7.383333,-38.016667
br,baixas,Baixas,20,,-8.266667,-43.133333
br,baixas,Baixas,22,,-6.066667,-38.433333
br,baixas,Baixas,30,,-7.833333,-38.9
br,baixa seca,Baixa Sêca,05,,-9.55,-43.066667
br,baixa seca,Baixa Sêca,15,,-16.733333,-40.766667
br,baixa velha,Baixa Velha,30,,-8.733333,-37.333333
br,baixa verde,Baixa Verde,01,,-9.883333,-67.616667
br,baixa verde,Baixa Verde,06,,-5.55,-37.95
br,baixa verde,Baixa Verde,17,,-6.383333,-38.433333
br,baixa verde,Baixa Verde,20,,-6.783333,-40.6
br,baixa verde,Baixa Verde,22,,-5.333333,-37.75
br,baixa verde,Baixa Verde,30,,-7.75,-34.833333
br,baixax,Baixax,20,,-6.95,-41.866667
br,baixinha,Baixinha,05,,-13.116667,-39.6
br,baixinha,Baixinha,06,,-2.909167,-40.561389
br,baixinha,Baixinha,13,,-3.811389,-43.063056
br,baixinha,Baixinha,20,,-3.685833,-42.007222
br,baixinha,Baixinha,22,,-5.25,-35.816667
br,baixinha da franca,Baixinha da Franca,22,,-5.25,-35.8
br,baixinha de todos os santos,Baixinha de Todos os Santos,15,,-17.866667,-41.683333
br,baixinha dos flores,Baixinha dos Flores,13,,-3.8025,-43.0475
br,baixinha i,Baixinha I,13,,-3.412778,-43.494722
br,baixinha ii,Baixinha II,13,,-3.286389,-43.188333
br,baixinho,Baixinho,06,,-7.25,-40.4
br,baixio,Baixio,05,,-12.133333,-37.7
br,baixio,Baixio,06,,-5.95,-39.483333
br,baixio,Baixio,14,,-14.65,-56.2
br,baixio,Baixio,17,,-6.516667,-38.266667
br,baixio,Baixio,20,,-7.966667,-40.683333
br,baixio,Baixio,22,,-5.966667,-37.883333
br,baixio,Baixio,30,,-7.683333,-39.783333
br,baixio da donona,Baixio da Donona,06,,-6.35,-39.683333
br,baixio da roca,Baixio da Roça,06,,-6.033333,-39.466667
br,baixio do boi,Baixio do Boi,06,,-7.516667,-39.016667
br,baixio do boi,Baixio do Boi,30,,-7.983333,-38.85
br,baixio do facundo,Baixio do Facundo,06,,-7.066667,-39.933333
br,baixio do fogo,Baixio do Fogo,22,,-6.333333,-38.3
br,baixio do fumo,Baixio do Fumo,30,,-7.75,-39.35
br,baixio do jua,Baixio do Juá,30,,-7.783333,-39.3
br,baixio do lima,Baixio do Lima,06,,-7.15,-40.166667
br,baixio do paulo,Baixio do Paulo,30,,-7.616667,-39.45
br,baixio dos campos,Baixio dos Campos,17,,-7.75,-38.216667
br,baixio dos curimatas,Baixío dos Curimatás,13,,-3.333333,-45.316667
br,baixio dos nicolaus,Baixio dos Nicolaus,30,,-7.633333,-38.716667
br,baixio dos ramos,Baixio dos Ramos,06,,-7.233333,-40.133333
br,baixios,Baixios,02,,-9.7,-37.2
br,baixio verde,Baixio Verde,06,,-6.716667,-39.2
br,baixo,Baixo,14,,-14.65,-56.2
br,baixo,Baixo,15,,-16.4,-41.033333
br,baixo,Baixo,17,,-6.733333,-36.366667
br,baixo,Baixo,20,,-7.616667,-41.15
br,baixo dos curimatas,Baixo dos Curimatás,13,,-3.333333,-45.316667
br,baixo do zacarias,Baixo do Zacarias,06,,-2.933333,-40.2
br,baixo feia,Baixo Feia,20,,-5.933333,-42.116667
br,baixo guandu,Baixo Guandu,08,19292,-19.516667,-41.016667
br,baixo longa,Baixo Longá,20,,-3.166667,-41.866667
br,baixo mearim,Baixo Mearim,13,,-3.466667,-44.883333
br,baixo palmital,Baixo Palmital,26,,-26.901785,-49.866444
br,baixo vere,Baixo Verê,18,,-25.866667,-52.95
br,baje,Baje,01,,-9.166667,-72.266667
br,baje,Bajé,23,,-31.333333,-54.1
br,balaco,Balacó,21,,-22.283333,-41.783333
br,balaiada,Balaiada,13,,-3.866667,-44.183333
br,balaieiro,Balaieiro,15,,-15.616667,-44.466667
br,balanca,Balanca,06,,-3.516667,-39.6
br,balanca,Balança,17,,-7.55,-38.016667
br,balanca,Balança,20,,-6.983333,-40.966667
br,balanca,Balanca,22,,-5.741667,-37.464444
br,balancas,Balanças,06,,-5.266667,-39.6
br,balanco,Balanco,17,,-6.966667,-36.833333
br,balanco,Balanço,20,,-6.333333,-42.816667
br,balanco,Balanço,31,,-11.55,-46.85
br,balao,Balão,06,,-5.966667,-39.516667
br,balao primeiro,Balão Primeiro,18,,-25.507952,-50.529375
br,balbina,Balbina,04,,-2.916667,-68.3
br,balbino,Balbino,15,,-16.233333,-40.483333
br,balbino dias,Balbino Dias,15,,-16.95,-40.716667
br,balbinos,Balbinos,27,,-21.916667,-49.35
br,balco,Balço,06,,-5.366667,-39.95
br,baldim,Baldim,15,,-9.283333,-43.95
br,balduino oliveira,Balduíno Oliveira,05,,-16.333333,-39.833333
br,baleia,Baleia,05,,-16.266667,-39.183333
br,balins,Balins,20,,-8.75,-44.433333
br,balisa,Balisa,03,,-.466667,-52.466667
br,balisa,Balisa,14,,-16.25,-52.433333
br,balisa,Balisa,23,,-27.583333,-52.15
br,balisa,Balisa,27,,-21.783333,-51.05
br,balisa,Balisa,29,,-16.25,-52.416667
br,baliza,Baliza,22,,-6.266667,-38
br,baliza,Baliza,23,,-27.583333,-52.15
br,baliza,Baliza,25,,.950278,-59.911111
br,baliza,Baliza,27,,-21.783333,-51.05
br,baliza,Baliza,29,,-16.25,-52.416667
br,balneario,Balneário,26,,-26.983333,-48.633333
br,balneario de camboriu,Balneário de Camboriú,26,,-26.983333,-48.633333
br,balneario de rincao,Balneário de Rincão,26,,-28.833333,-49.25
br,balneario do rincao,Balneário do Rincão,26,,-28.833333,-49.25
br,balneario itajuba,Balneário Itajubá,26,,-26.7047,-48.6843
br,balneario praia do leste,Balneário Praia do Leste,18,,-25.691667,-48.466667
br,balneario sao jose,Balneário São José,27,,-23.769578,-46.722904
br,balsa,Balsa,13,,-5.666667,-43.083333
br,balsa,Balsa,18,,-23.655676,-50.048953
br,balsamo,Bálsamo,06,,-7.133333,-38.733333
br,balsamo,Bálsamo,11,,-18.05,-54.05
br,balsamo,Balsamo,17,,-6.666667,-38.45
br,balsamo,Bálsamo,18,,-23.4,-50.8
br,balsamo,Bálsamo,21,,-20.916667,-41.733333
br,balsamo,Bálsamo,27,,-20.3,-47.3
br,balsa nova,Balsa Nova,18,,-25.6,-49.616667
br,balsas,Balsas,13,68058,-7.516667,-46.033333
br,balseiro,Balseiro,13,,-3.774167,-43.439722
br,baltasar,Baltasar,21,,-21.583333,-42.15
br,baltazar,Baltazar,11,,-20.266667,-55.1
br,baltazar,Baltazar,21,,-21.583333,-42.15
br,balthazar,Balthazar,21,,-21.583333,-42.15
br,baluarte,Baluarte,05,,-9.466667,-43.533333
br,baluarte,Baluarte,20,,-6.15,-42.05
br,bamba,Bamba,30,,-8.9,-36.2
br,bambu,Bambu,15,,-22.516667,-46.433333
br,bambuhy,Bambuhy,15,,-20.016667,-45.966667
br,bambui,Bambuí,15,18840,-20.016667,-45.966667
br,bambura,Bambura,30,,-8.433333,-35.4
br,banabuiu,Banabuiú,06,,-5.283333,-38.866667
br,banana,Banana,13,,-3.645278,-43.459444
br,banana,Banana,18,,-25.133333,-52.65
br,bananal,Bananal,04,,-3.033333,-68.166667
br,bananal,Bananal,05,,-11.95,-45.733333
br,bananal,Bananal,11,,-20.25,-56.05
br,bananal,Bananal,13,,-3.994167,-43.118333
br,bananal,Bananal,14,,-14.833333,-55.983333
br,bananal,Bananal,15,,-18.45,-42.25
br,bananal,Bananal,18,,-25.175826,-48.377379
br,bananal,Bananal,21,,-22.788788,-43.163603
br,bananal,Bananal,26,,-26.45,-49
br,bananal,Bananal,27,,-22.6079,-45.6939
br,bananal de cima,Bananal de Cima,15,,-19.383333,-41.833333
br,bananal do norte,Bananal do Norte,08,,-20.75,-41.3
br,bananal do rio,Bananal do Rio,15,,-19.366667,-41.85
br,bananal grande,Bananal Grande,27,,-24.583333,-48.166667
br,bananal nobre,Bananal Nobre,14,,-14.733333,-56.333333
br,bananalzinho,Bananalzinho,14,,-14.55,-54.883333
br,bananas,Bananas,06,,-6.116667,-40.15
br,bananas,Bananas,15,,-20.533333,-45.3
br,bananas,Bananas,18,,-25.383333,-51.283333
br,bananeira,Bananeira,04,,-7.110833,-70.763611
br,bananeira,Bananeira,05,,-10.616667,-40.316667
br,bananeira,Bananeira,06,,-4.116667,-41.15
br,bananeira,Bananeira,13,,-3.838611,-43.352778
br,bananeira,Bananeira,17,,-7.5,-36.966667
br,bananeira,Bananeira,18,,-23.40316,-49.830385
br,bananeira,Bananeira,20,,-4.716667,-41.85
br,bananeira,Bananeira,24,,-10.65,-65.383333
br,bananeira,Bananeira,30,,-8.183333,-36.733333
br,bananeiras,Bananeiras,02,,-9.816667,-36.616667
br,bananeiras,Bananeiras,04,,-7.916667,-66.866667
br,bananeiras,Bananeiras,05,,-10.8,-38.816667
br,bananeiras,Bananeiras,06,,-4.15,-38.333333
br,bananeiras,Bananeiras,17,7371,-6.75,-35.616667
br,bananeiras,Bananeiras,21,,-21.1,-41.95
br,bananeiras,Bananeiras,24,,-10.65,-65.383333
br,bananeiras,Bananeiras,29,,-18.016667,-49.366667
br,bananeiro,Bananeiro,06,,-5.05,-40.766667
br,banca,Banca,05,,-12.916667,-41.283333
br,bancarios,Bancários,21,,-22.791457,-43.179975
br,banco,Banco,05,,-13.166667,-39.333333
br,banco central,Banco Central,05,,-14.45,-39.416667
br,banco da areia,Banco da Areia,20,,-5.866667,-41.15
br,banco da vitoria,Banco da Vitória,05,,-14.8,-39.1
br,banco de pedra,Banco de Pedra,05,,-14.633333,-39.266667
br,banco do pedro,Banco do Pedro,05,,-14.633333,-39.266667
br,bancos,Bancos,22,,-5.75,-35.75
br,banda,Bandá,13,,-3.931667,-42.815
br,bandara,Bandara,30,,-8.25,-38.15
br,bandarra,Bandarra,05,,-11.033333,-43.333333
br,bandarra,Bandarra,06,,-5.733333,-40.733333
br,bandarra,Bandarra,17,,-6.55,-38.333333
br,bandeira,Bandeira,05,,-14.683333,-39.5
br,bandeira,Bandeira,06,,-4.583333,-39.466667
br,bandeira,Bandeira,13,,-3.503333,-42.995
br,bandeira,Bandeira,15,,-15.9,-40.566667
br,bandeira,Bandeira,18,,-25.066667,-53.116667
br,bandeira,Bandeira,22,,-5.783333,-35.966667
br,bandeira,Bandeira,27,,-21.883333,-47.666667
br,bandeira,Bandeira,30,,-7.383333,-37.383333
br,bandeira de mello,Bandeira de Mello,05,,-13.05,-40.833333
br,bandeira de melo,Bandeira de Melo,05,,-13.05,-40.833333
br,bandeira do almada,Bandeira do Almada,05,,-14.683333,-39.5
br,bandeira do colonia,Bandeira do Colônia,05,,-15.116667,-40.1
br,bandeira do sul,Bandeira do Sul,15,,-21.783333,-46.383333
br,bandeirante,Bandeirante,22,,-5.058056,-36.879167
br,bandeirante,Bandeirante,26,,-26.8,-53.633333
br,bandeirante,Bandeirante,29,,-13.683333,-50.8
br,bandeirantes,Bandeirantes,11,,-19.883333,-54.383333
br,bandeirantes,Bandeirantes,15,,-20.333333,-43.366667
br,bandeirantes,Bandeirantes,18,29495,-23.1,-50.35
br,bandeirantes,Bandeirantes,21,,-21.866667,-41.783333
br,bandeirantes,Bandeirantes,26,,-26.8,-53.633333
br,bandeirantes,Bandeirantes,27,,-20.6,-50.8
br,bandeirantes,Bandeirantes,29,,-13.683333,-50.8
br,bandeirinha,Bandeirinha,05,,-12.516667,-40.9
br,bandeirinha,Bandeirinha,15,,-21.05,-45.4
br,bandiacu,Bandiaçu,05,,-11.616667,-39.133333
br,bandidos da fumaca,Bandidos da Fumaça,14,,-15.966667,-58.3
br,bandolim,Bandolim,13,,-3.782778,-42.901667
br,bangu,Bangu,21,,-22.881957,-43.466771
br,bangue,Banguê,05,,-9.25,-39.816667
br,bangue,Bangüé,13,,-3.826944,-43.3425
br,bangue,Banguê,20,,-3.951389,-42.296944
br,bangue,Bangüê,22,,-5.485278,-36.943333
br,banguelo,Banguelo,20,,-3.657778,-42.489167
br,banhadinho,Banhadinho,18,,-23.9207,-50.1631
br,banho,Banho,18,,-25.507882,-50.234618
br,banja,Banja,20,,-8.416667,-45.716667
br,banque,Banquê,06,,-2.8,-40.333333
br,banquete,Banquete,21,,-22.166667,-42.466667
br,banunal,Banunal,04,,-3.033333,-68.166667
br,baptista botelho,Baptista Botelho,27,,-23.016667,-49.416667
br,baquirivu,Baquirivu,27,,-23.5,-46.433333
br,baracaja,Baracajá,11,,-23.066667,-54.983333
br,baracari,Baracari,04,,-3.9225,-62.880556
br,barada,Barada,16,,-2.433333,-51.266667
br,baraganca,Baragança,27,,-22.95,-46.566667
br,barandao,Barandão,17,,-6.466667,-36.016667
br,barao ataliba nogueira,Barão Ataliba Nogueira,27,,-22.4,-46.75
br,barao,Barão,21,,-22.902631,-43.343862
br,barao,Barão,23,,-29.383333,-51.533333
br,barao,Barão,27,,-21.7,-46.866667
br,barao da angra,Barão da Angra,21,,-22.133333,-43.25
br,barao de angra,Barão de Angra,21,,-22.133333,-43.25
br,barao de antonina,Barão de Antonina,27,,-23.616667,-49.566667
br,barao de aquino,Barão de Aquino,21,,-22.116667,-42.65
br,barao de aquiras,Barão de Aquirás,06,,-6.833333,-40.366667
br,barao de aquiraz,Barão de Aquiraz,06,,-6.833333,-40.366667
br,barao de bom jardim,Barão de Bom Jardim,05,,-12.283333,-38.6
br,barao de capanema,Barão de Capanema,14,,-13.316667,-57.866667
br,barao de cocais,Barão de Cocais,15,22387,-19.933333,-43.466667
br,barao de cotegipe,Barão de Cotegipe,23,,-27.616667,-52.383333
br,barao de geraldo,Barão de Geraldo,27,,-22.816667,-47.1
br,barao de grajahu,Barão de Grajahú,13,,-6.75,-43.016667
br,barao de grajau,Barão de Grajaú,13,,-6.75,-43.016667
br,barao de juparana,Barão de Juparanã,21,,-22.343889,-43.682778
br,barao de lucena,Barão de Lucena,18,,-23.166667,-52.416667
br,barao de melgaco,Barão de Melgaço,14,4353,-16.216667,-55.966667
br,barao de melgaco,Barão de Melgaço,24,,-11.85,-60.716667
br,barao de rio branco,Barão de Rio Branco,01,,-7.516667,-73.016667
br,barao de sao jose,Barão de São José,21,,-21.85,-41.2
br,barao de suacuna,Barão de Suaçuna,30,,-8.4,-35.266667
br,barao de suassuna,Barão de Suassuna,30,,-8.4,-35.266667
br,barao de tromai,Barão de Tromaí,13,,-1.483333,-45.6
br,barao de vassouras,Barão de Vassouras,21,,-22.370278,-43.690556
br,barao do amparo,Barão do Amparo,21,,-22.473889,-43.652222
br,barao do grajau,Barão do Grajaú,13,,-6.75,-43.016667
br,barao do guandu,Barão do Guandu,21,,-22.675,-43.42
br,barao do iriri,Barão do Iriri,21,,-22.681389,-43.094444
br,barao do orlando,Barão do Orlando,14,,-13.55,-57.45
br,barao do triunfo,Barão do Triunfo,23,,-30.383333,-51.733333
br,barao hirsch,Barão Hirsch,23,,-27.75,-52.466667
br,baratinha,Baratinha,05,,-13.333333,-40.983333
br,barauna,Baraúna,05,,-9.316667,-40.016667
br,barauna,Baraúna,15,,-19.25,-42.416667
br,barauna,Baraúna,17,,-6.633333,-36.25
br,barauna,Baraúna,22,14113,-5.083333,-37.616667
br,barauna ferrada,Baraúna Ferrada,06,,-6.616667,-40.566667
br,baraunas,Baraúnas,05,,-12.65,-41.8
br,baraunas,Baraúnas,06,,-7.116667,-40.1
br,baraunas,Baraunas,15,,-19.066667,-42.716667
br,baraunas,Baraúnas,30,,-7.766667,-38.7
br,barbacena,Barbacena,15,122213,-21.233333,-43.766667
br,barbacena,Barbacena,16,,-2.766667,-52.066667
br,barbadinha,Barbadinha,22,,-5.366667,-37.3
br,barbadinho,Barbadinho,22,,-5.366667,-37.3
br,barbado,Barbado,05,,-13.283333,-41.933333
br,barbado,Barbado,25,,.943889,-61.706389
br,barbalha,Barbalha,06,33944,-7.316667,-39.283333
br,barbara do oeste,Bárbara do Oeste,27,,-22.749167,-47.413611
br,barbatao,Barbatão,06,,-4.15,-38.633333
br,barboleta,Barboleta,27,,-20.933333,-49.466667
br,barborema,Barborema,17,,-6.8,-35.6
br,barbosa,Barbosa,05,,-12.15,-37.966667
br,barbosa,Barbosa,06,,-5,-38.866667
br,barbosa,Barbosa,15,,-20.55,-41.883333
br,barbosa,Barbosa,17,,-6.466667,-37.8
br,barbosa,Barbosa,18,,-23.7491,-49.8296
br,barbosa,Barbosa,22,,-5.716944,-36.631389
br,barbosa,Barbosa,25,,-2.75,-60.666667
br,barbosa,Barbosa,27,,-21.3,-49.95
br,barbosa ferraz,Barbosa Ferraz,18,9048,-24.066667,-52.05
br,barbosas,Barbosas,18,,-23.7491,-49.8296
br,barbozas,Barbozas,18,,-23.7491,-49.8296
br,barca,Barca,15,,-17.5,-41.283333
br,barcada,Barcada,14,,-15.316667,-56.466667
br,barcarena,Barcarena,16,29872,-1.5,-48.666667
br,barcarena nova,Barcarena Nova,16,,-1.5,-48.616667
br,barcelar,Barcelar,21,,-21.916667,-42.616667
br,barcellos,Barcellos,04,,-1,-62.966667
br,barcelona,Barcelona,22,,-5.933333,-35.933333
br,barcelos,Barcelos,04,7356,-1,-62.966667
br,barcelos,Barcelos,05,,-14.033333,-39
br,barcelos,Barcelos,21,,-21.716667,-41.183333
br,barcelos do sul,Barcelos do Sul,05,,-14.033333,-39
br,bar-da-barra,Bar-da-Barra,24,,-9.016667,-61.85
br,bardo,Bardo,22,,-6.266667,-36.233333
br,bargarim,Bargarim,08,,-20.5,-41.416667
br,barigui,Barigui,11,,-23.233333,-55.3
br,barigui,Barigui,18,,-25.533333,-49.383333
br,bariqui,Bariqui,11,,-23.233333,-55.3
br,bariri,Bariri,27,28802,-22.066667,-48.733333
br,bariry,Bariry,27,,-22.066667,-48.733333
br,baronesa,Baronêsa,05,,-18.1,-39.683333
br,baronesa,Baronesa,27,,-23.094059,-47.757579
br,baroneza,Baroneza,05,,-18.1,-39.683333
br,barra aberta,Barra Aberta,11,,-21.75,-55.05
br,barra alegre,Barra Alegre,08,,-21.15,-41.6
br,barra alegre,Barra Alegre,15,,-19.45,-42.616667
br,barra alegre,Barra Alegre,21,,-22.25,-42.3
br,barra alta,Barra Alta,20,,-9.633333,-45.616667
br,barra azul,Barra Azul,20,,-10.533333,-45
br,barra bairro do areado,Barra Bairro do Areado,27,,-24.6846,-48.3306
br,barra bali,Barra Bali,21,,-23.00897,-43.436166
br,barra,Barra,02,,-9.316667,-38.233333
br,barra,Barra,05,20895,-11.083333,-43.166667
br,barra,Barra,06,,-3.75,-41.966667
br,barra,Barra,11,,-19.766667,-56.8
br,barra,Barra,13,,-3.458056,-42.790278
br,barra,Barra,15,,-19.133333,-48.766667
br,barra,Barra,17,,-6.766667,-34.916667
br,barra,Barra,20,,-10.05,-45.05
br,barra,Barra,21,,-21.116667,-41.733333
br,barra,Barra,22,,-4.951111,-37.15
br,barra,Barra,26,,-28.5,-48.75
br,barra,Barra,27,,-21.133333,-47.716667
br,barra,Barra,28,,-11.15,-37.166667
br,barra,Barra,29,,-15.8,-50.283333
br,barra,Barra,30,,-7.733333,-37.633333
br,barra bonita,Barra Bonita,18,,-23.8469,-50.1886
br,barra bonita,Barra Bonita,26,,-27.266667,-51.916667
br,barra bonita,Barra Bonita,27,37247,-22.483333,-48.533333
br,barraca,Barraca,04,,-4.9,-60.15
br,barraca,Barraca,08,,-18.4,-39.866667
br,barraca,Barraca,15,,-20.416667,-43.233333
br,barraca,Barraca,16,,-6.133333,-57.55
br,barraca,Barraca,30,,-7.85,-38.8
br,barraca da barra,Barraca da Barra,24,,-9.016667,-61.85
br,barraca da boca,Barraca da Bôca,03,,-.6,-52.583333
br,barraca da forquilha,Barraca da Forquilha,13,,-3.383333,-42.7
br,barraca da vista,Barraca da Vista,04,,-7.852222,-58.3175
br,barraca de alfredo lopes,Barraca de Alfredo Lopes,16,,-6.95,-58.216667
br,barraca de maneco rondon,Barraca de Maneco Rondon,14,,-13.65,-58.3
br,barraca de seringueiro,Barraca de Seringueiro,04,,-7.033333,-58.816667
br,barraca do alianca,Barraca do Aliança,23,,-29.416667,-53.45
br,barraca do allianca,Barraca do Alliança,23,,-29.416667,-53.45
br,barraca do barra,Barraca do Barra,24,,-9.016667,-61.85
br,barraca do cabral,Barraca do Cabral,14,,-16.083333,-57.7
br,barraca do gustavo,Barraca do Gustavo,14,,-9.716667,-60.566667
br,barraca do lucas,Barraca do Lucas,14,,-9.683333,-60.55
br,barraca do maneco rondon,Barraca do Maneco Rondon,14,,-13.65,-58.3
br,barraca do peruano,Barraca do Peruano,04,,-8.133333,-58.25
br,barraca do surubim,Barraca do Surubim,25,,2.683333,-60.983333
br,barraca do tomas,Barraca do Tomás,25,,2.533333,-60.933333
br,barracao,Barracão,05,,-10.7,-38.533333
br,barracao,Barracão,06,,-3.733333,-40.983333
br,barracao,Barracão,13,,-4.083333,-43.016667
br,barracao,Barracão,14,,-15.916667,-57.15
br,barracao,Barracão,15,,-16.416667,-43.25
br,barracao,Barracão,18,,-26.25,-53.636389
br,barracao,Barracão,20,,-7.116667,-41.4
br,barracao,Barracão,23,,-27.666667,-51.416667
br,barracao,Barracão,24,,-11.95,-64.183333
br,barracao,Barracão,26,,-27.033333,-48.9
br,barracao,Barracão,30,,-8.933333,-37.366667
br,barracao barreto,Barracão Barreto,14,,-8.8,-58.4
br,barracao bello horisonte,Barracão Bello Horisonte,24,,-11.883333,-64.166667
br,barracao belo horizonte,Barracão Belo Horizonte,24,,-11.883333,-64.166667
br,barracao do barreto,Barracão do Barreto,14,,-8.8,-58.4
br,barracao do camaleao,Barracão do Camaleão,04,,-2.683333,-65.816667
br,barracao do curua,Barração do Curuá,16,,-5.4,-54.466667
br,barracao do liberato,Barracão do Liberato,24,,-12.633333,-60.966667
br,barracao do maia,Barracão do Maia,14,,-8.05,-58.233333
br,barracao floresta,Barracão Floresta,16,,-8.95,-57.266667
br,barracao joffre,Barracão Joffre,24,,-13.5,-60.55
br,barracao jofre,Barracão Jofre,24,,-13.5,-60.55
br,barracao queimado,Barracão Queimado,14,,-13.0875,-59.895833
br,barracao rio mar,Barracão Rio Mar,16,,-7.25,-53.7
br,barracao sao jose,Barracão São José,14,,-9.2,-57.066667
br,barracao triumpho,Barracão Triumpho,24,,-11.95,-64.216667
br,barracao triunfo,Barracão Triunfo,24,,-11.95,-64.216667
br,barraca pereirinha,Barraca Pereirinha,16,,-8.15,-57.583333
br,barraca rapirran,Barraca Rapirran,01,,-10.333333,-67.183333
br,barraca republica,Barraca Republica,14,,-13.5,-60.5
br,barracas,Barracas,06,,-6.966667,-38.883333
br,barracas,Barracas,28,,-10.416667,-37.1
br,barraca terra firme,Barraca Terra Firme,24,,-11.983333,-64.25
br,barra clara,Barra Clara,26,,-27.533333,-49.116667
br,barraco alto,Barraco Alto,24,,-13.033333,-61.433333
br,barraco,Barraco,04,,-7.1,-60.45
br,barra comprida,Barra Comprida,13,,-9.35,-46
br,barracon,Barracon,18,,-26.25,-53.636389
br,barra da agua branca,Barra da Água Branca,05,,-14.266667,-39.25
br,barra da anta,Barra da Anta,26,,-26.912634,-49.662623
br,barra da arara,Barra da Arara,30,,-7.716667,-38.733333
br,barra da areia,Barra da Areia,18,,-24.983333,-51.083333
br,barra da boa vista,Barra da Boa Vista,08,,-21.183333,-40.933333
br,barra da caida,Barra da Caída,20,,-8.566667,-42.133333
br,barra da campineira,Barra da Campineira,13,,-3.431389,-43.054722
br,barra da casa dos negros,Barra da Casa dos Negros,05,,-13.316667,-41.133333
br,barra da choca,Barra da Choça,05,,-14.85,-40.583333
br,barra da craibeira,Barra da Craibeira,22,,-6.816667,-36.583333
br,barra da cruz,Barra da Cruz,05,,-9.65,-41.533333
br,barra da cruz,Barra da Cruz,18,,-25.083333,-48.8
br,barra da cruz,Barra da Cruz,22,,-5.688056,-36.377222
br,barra da cruz,Barra da Cruz,27,,-25.016609,-48.514762
br,barra da estiva,Barra da Estiva,05,12618,-13.633333,-41.316667
br,barra da estiva,Barra da Estiva,13,,-3.116111,-42.365833
br,barra da figueira,Barra da Figueira,15,,-19.6,-41.566667
br,barra da forquilha,Barra da Forquilha,18,,-24.702787,-48.97872
br,barra da forquilha,Barra da Forquilha,30,,-7.716667,-39.483333
br,barra da imbuia,Barra da Imbuia,18,,-24.866667,-51
br,barra da jangada,Barra da Jangada,18,,-26.033333,-51.583333
br,barra da onca,Barra da Onça,13,,-3.4,-42.783333
br,barra da onca,Barra da Onça,22,,-5.8325,-36.647778
br,barra da pendencia,Barra da Pendencia,06,,-6.6,-38.916667
br,barra da prata,Barra da Prata,13,,-3.086944,-43.371667
br,barra da pratinha,Barra da Pratinha,26,,-26.681183,-49.834724
br,barra d'areia,Barra d'Areia,18,,-24.983333,-51.083333
br,barra da roca,Barra da Roça,13,,-8.238611,-45.655278
br,barra da salina,Barra da Salina,15,,-16.6,-42.316667
br,barra da salinas,Barra da Salinas,15,,-16.6,-42.316667
br,barra das antas,Barra das Antas,18,,-24.066667,-50.633333
br,barra da sao joao,Barra da São João,21,,-22.6,-42
br,barradas,Barradas,13,,-3.483333,-45.183333
br,barradas,Barradas,27,,-22.7362,-45.6582
br,barra das corvinas,Barra das Corvinas,13,,-3.628333,-42.636389
br,barra da silva,Barra da Silva,30,,-8.433333,-38.85
br,barra das pombas,Barra das Pombas,26,,-26.568414,-49.952401
br,barra da tijuca,Barra da Tijuca,21,,-23.008054,-43.340378
br,barra da varginha,Barra da Varginha,27,,-23.466667,-45.566667
br,barra da varzea,Barra da Várzea,20,,-8.533333,-41.933333
br,barra de alegre,Barra de Alegre,15,,-20.016667,-41.85
br,barra de arica,Barra de Aricá,14,,-15.983333,-55.9
br,barra de bacururu,Barra de Bacururu,04,,-3.75,-66.183333
br,barra de caravelas,Barra de Caravelas,05,,-17.733333,-39.183333
br,barra de cima,Barra de Cima,22,,-6.45,-35.616667
br,barra de espinharos,Barra de Espinharos,22,,-6.433333,-37.366667
br,barra de guaratiba,Barra de Guaratiba,21,,-23.0622,-43.564718
br,barra de itabapoana,Barra de Itabapoana,08,,-21.3,-40.966667
br,barra de jangada,Barra de Jangada,30,,-8.816667,-35.883333
br,barra de jequirico,Barra de Jequirico,05,,-13.2,-38.933333
br,barra del chuy,Barra del Chuy,23,,-33.733333,-53.383333
br,barra de miguel dias,Barra de Miguel Dias,05,,-10.05,-42.6
br,barra de misericordia,Barra de Misericórdia,30,,-7.883333,-37.583333
br,barra de piaba,Barra de Piaba,17,,-6.383333,-38.516667
br,barra de santana,Barra de Santana,18,,-25.916667,-52.866667
br,barra de santana,Barra de Santana,22,,-6.15,-37.116667
br,barra de santa rosa,Barra de Santa Rosa,17,,-6.716667,-36.066667
br,barra de santo antonio,Barra de Santo Antônio,02,,-9.4,-35.5
br,barra de santo antonio,Barra de Santo Antonio,17,,-7.683333,-36.15
br,barra de santo antonio grande,Barra de Santo Antônio Grande,02,,-9.4,-35.5
br,barra de sao bento,Barra de São Bento,17,,-6.733333,-38.383333
br,barra de sao francisco,Barra de São Francisco,08,20743,-18.75,-40.866667
br,barra de sao francisco,Barra de São Francisco,21,,-21.966667,-42.7
br,barra de sao joao,Barra de São João,21,,-22.6,-42
br,barra de sao joao,Barra de São João,30,,-8.416667,-35.666667
br,barra de sao manuel,Barra de São Manuel,16,,-7.328056,-58.085556
br,barra de sao mateus,Barra de São Mateus,08,,-18.583333,-39.75
br,barra de sao miguel,Barra de São Miguel,02,,-9.833333,-35.9
br,barra de sao miguel,Barra de São Miguel,17,,-7.733333,-36.316667
br,barra de sao pedro,Barra de São Pedro,30,,-7.833333,-40.316667
br,barra do acu,Barra do Açu,21,,-21.916667,-40.983333
br,barra do araquari,Barra do Araquari,26,,-26.4592,-48.6031
br,barra do arara,Barra do Arará,05,,-9.483333,-40.783333
br,barra do ararapira,Barra do Ararapira,18,,-25.29288,-48.099974
br,barra do arati,Barra do Arati,04,,-3.566667,-66.083333
br,barra do areado,Barra do Areado,27,,-24.605875,-48.990115
br,barra do arica,Barra do Aricá,14,,-15.983333,-55.9
br,barra do ariranha,Barra do Ariranha,15,,-18.666667,-41.1
br,barra do barao,Barra do Barão,05,,-10.466667,-38.15
br,barra do batatal,Barra do Batatal,08,,-20.616667,-40.766667
br,barra do batatal,Barra do Batatal,27,,-24.586279,-48.270775
br,barra do bode,Barra do Bode,22,,-5.300833,-36.576667
br,barra do bonito,Barra do Bonito,20,,-8.733333,-41.766667
br,barra do bonito,Barra do Bonito,30,,-7.916667,-38.316667
br,barra do braco,Barra do Braço,27,,-24.646833,-48.297547
br,barra do brejo,Barra do Brejo,05,,-9.95,-42.866667
br,barra do brejo,Barra do Brejo,20,,-10.033333,-45.366667
br,barra do breu de cima,Barra do Breu de Cima,04,,-3.833333,-66.316667
br,barra do bugres,Barra do Bugres,14,31313,-15.083333,-57.183333
br,barra do buriti,Barra do Buriti,29,,-14.316667,-46.033333
br,barra do cabure,Barra do Caburé,13,,-3.558333,-43.279444
br,barra do cafundo,Barra do Cafundó,26,,-28.3,-49.383333
br,barra do camaragibe,Barra do Camaragibe,02,,-9.316667,-35.433333
br,barra do canhoto,Barra do Canhoto,02,,-9.1,-36.066667
br,barra do carvalho,Barra do Carvalho,05,,-13.666667,-38.966667
br,barra do cedro,Barra do Cedro,13,,-3.576667,-43.327778
br,barra do cedro,Barra do Cedro,18,,-25.159167,-48.22
br,barra do cedro,Barra do Cedro,30,,-7.9,-39.55
br,barra do chapeu,Barra do Chapéu,27,,-24.466667,-49.033333
br,barra do chata,Barra do Chata,30,,-8.5,-35.933333
br,barra do cipo,Barra do Cipó,06,,-5.7,-39.15
br,barra do coelho,Barra do Coelho,18,,-24.822747,-48.912138
br,barra do comandante,Barra do Comandante,05,,-9.8,-43.433333
br,barra do corda,Barra do Corda,13,48902,-5.5,-45.25
br,barra do cordeiro,Barra do Cordeiro,18,,-25.168104,-48.528878
br,barra do corrego,Barra do Córrego,30,,-8.716667,-38.35
br,barra do cuiete,Barra do Cuieté,15,,-19.083333,-41.55
br,barra do cuite,Barra do Cuité,15,,-19.083333,-41.55
br,barra do dollman,Barra do Dollman,26,,-26.894773,-49.673667
br,barra do exu,Barra do Exu,30,,-8.066667,-38.45
br,barra do feijao,Barra do Feijão,06,,-5.416667,-38.1
br,barra do felix,Barra do Félix,14,,-16.466667,-55.933333
br,barra do figueiredo,Barra do Figueiredo,06,,-5.383333,-38.283333
br,barra do fogo,Barra do Fogo,05,,-14.666667,-40.05
br,barra do fogo,Barra do Fogo,15,,-15.233333,-42.066667
br,barra do funchal,Barra do Funchal,15,,-19.383333,-45.883333
br,barra do furado,Barra do Furado,05,,-15.25,-41.333333
br,barra do garapa,Barra do Garapa,13,,-5.45,-42.966667
br,barra do garcas,Barra do Garças,14,52402,-15.883333,-52.25
br,barra do garcez,Barra do Garcez,05,,-13.15,-38.833333
br,barra do garrote,Barra do Garrote,06,,-3.5,-39.9
br,barra do gentil,Barra do Gentil,05,,-13.9,-40.533333
br,barra do gil,Barra do Gil,05,,-13,-38.616667
br,barra do goncalo,Barra do Goncalo,22,,-5.616667,-35.883333
br,barra do imbui,Barra do Imbuí,21,,-22.400976,-42.976083
br,barra do inga,Barra do Ingá,13,,-5.516667,-43
br,barra do ipanema,Barra do Ipanema,02,,-9.866667,-37.25
br,barra do ipueira,Barra do Ipueira,06,,-5.116667,-39.533333
br,barra dois irmaos,Barra Dois Irmãos,05,,-10.15,-43.133333
br,barra-do-itabapoana,Barra-do-Itabapoana,08,,-21.3,-40.966667
br,barra do itapemirim,Barra do Itapemirim,08,,-21.016667,-40.8
br,barra do itapocu,Barra do Itapocu,26,,-26.5713,-48.6763
br,barra do itariri,Barra do Itariri,05,,-11.966667,-37.616667
br,barra do jacare,Barra do Jacaré,18,,-23.115321,-50.182551
br,barra do jacu,Barra do Jacu,13,,-3.559722,-43.150833
br,barra do jacupiranga,Barra do Jacupiranga,27,,-24.633333,-47.733333
br,barra do jacutinga,Barra do Jacutinga,26,,-27.090065,-49.741474
br,barra do jangada,Barra do Jangada,18,,-26.05,-51.5
br,barra do japo,Barra do Japó,04,,-3.533333,-66.083333
br,barra do jardim,Barra do Jardim,06,,-7.583333,-39.266667
br,barra do jardim,Barra do Jardim,30,,-8.35,-35.983333
br,barra do jenipapo,Barra do Jenipapo,30,,-9.283333,-37.033333
br,barra do jequirica,Barra do Jequiriçá,05,,-13.2,-38.933333
br,barra do jiquirica,Barra do Jiquiriça,05,,-13.2,-38.933333
br,barra do joaquim tavora,Barra do Joaquim Távora,08,,-19.316667,-40.583333
br,barra do jua,Barra do Juá,17,,-6.533333,-38.566667
br,barra do jua,Barra do Juá,30,,-8.45,-38.083333
br,barra do jundia,Barra do Jundiá,26,,-28.916667,-49.55
br,barra do limoeiro,Barra do Limoeiro,18,,-24.780486,-48.898393
br,barra do longa,Barra do Longá,20,,-3.15,-41.933333
br,barra do luis pereira,Barra do Luís Pereira,15,,-18.95,-44.116667
br,barra do macaco,Barra do Macaco,13,,-3.538056,-43.098333
br,barra do maraba,Barra do Marabá,08,,-21.2,-40.95
br,barra do marmelo,Barra do Marmelo,20,,-6.383333,-42.85
br,barra do maroba,Barra do Marobá,08,,-21.2,-40.95
br,barra do maxaranguape,Barra do Maxaranguape,22,,-5.516667,-35.266667
br,barra do mendes,Barra do Mendes,05,,-11.716667,-42.066667
br,barra do mendes,Barra do Mendes,18,,-24.761272,-48.884247
br,barra do mororo,Barra do Mororó,20,,-7.483333,-41.883333
br,barra do morro,Barra do Morro,14,,-15.583333,-55.85
br,barra do mosquito,Barra do Mosquito,20,,-9.483333,-45.783333
br,barra do mundo novo,Barra do Mundo Novo,05,,-11.866667,-40.516667
br,barra do nilo,Barra do Nilo,05,,-9.733333,-38.05
br,barra do novo brasil,Barra do Novo Brasil,08,,-19.116667,-40.416667
br,barra do oitis,Barra do Oitis,17,,-7.383333,-38.3
br,barra do olho d'agua,Barra do Olho d'Agua,20,,-6.983333,-40.866667
br,barra do ouro,Barra do Ouro,23,,-29.566667,-50.266667
br,barra do pajeu,Barra do Pajeú,30,,-8.9,-38.7
br,barra do palmeira,Barra do Palmeira,20,,-10.583333,-44.933333
br,barra do paraguacu,Barra do Paraguaçu,05,,-12.85,-38.8
br,barra do parai,Barra do Parai,21,,-22.466111,-43.827778
br,barra do paraopeba,Barra do Paraopeba,15,,-18.833333,-45.183333
br,barra do pindauva,Barra do Pindaúva,27,,-24.7579,-48.0694
br,barra do pinheiro,Barra do Pinheiro,26,,-27.333333,-51.75
br,barra do pirahy,Barra do Pirahy,21,,-22.466111,-43.827778
br,barra do pirai,Barra do Piraí,21,87669,-22.466111,-43.827778
br,barra do piraim,Barra do Piraim,14,,-16.483333,-56.333333
br,barra do pires,Barra do Pires,05,,-13.116667,-42.3
br,barra do poruquara,Barra do Poruquara,18,,-25.304528,-48.274031
br,barra do pote,Barra do Pote,05,,-13.033333,-38.633333
br,barra do prata,Barra do Prata,20,,-10.116667,-45.366667
br,barra do quarai,Barra do Quaraí,23,,-30.216667,-57.55
br,barra do riachao,Barra do Riachão,06,,-5.516667,-39.916667
br,barra do riachao,Barra do Riachão,13,,-3.554167,-43.389167
br,barra do riachinho,Barra do Riachinho,05,,-10.233333,-43.2
br,barra do riacho,Barra do Riacho,05,,-11.933333,-44.9
br,barra do riacho,Barra do Riacho,08,,-19.833333,-40.05
br,barra do riancho,Barra do Riancho,05,,-11.933333,-44.9
br,barra do ribeirao feio,Barra do Ribeirão Feio,27,,-24.797873,-48.434105
br,barra do ribeirao grande,Barra do Ribeirão Grande,27,,-24.9021,-48.4753
br,barra do ribeiro,Barra do Ribeiro,23,,-30.3,-51.3
br,barra do rico,Barra do Rico,15,,-16.616667,-41.116667
br,barra do rio abaixo,Barra do Rio Abaixo,27,,-23.383333,-47.116667
br,barra do rio azul,Barra do Rio Azul,18,,-25.723062,-50.686237
br,barra do rio azul,Barra do Rio Azul,23,,-27.383333,-52.433333
br,barra do rio,Barra do Rio,05,,-11.2,-46.133333
br,barra do rio,Barra do Rio,13,,-3.434167,-42.708611
br,barra do rio batatal,Barra do Rio Batatal,27,,-24.7148,-48.3367
br,barra do rio da prata,Barra do Rio da Prata,26,,-26.7033,-49.8293
br,barra do rio de contas,Barra do Rio de Contas,05,,-14.3,-39
br,barra do rio doce,Barra do Rio Doce,05,,-15.4,-39
br,barra do rio do cerro,Barra do Rio do Cerro,26,,-26.516667,-49.133333
br,barra do rio dos bugres,Barra do Rio dos Bugres,14,,-15.083333,-57.183333
br,barra do rio grande,Barra Do Rio Grande,05,,-11.083333,-43.166667
br,barra do rio jacupiranga,Barra do Rio Jacupiranga,27,,-24.633333,-47.733333
br,barra do rio negro,Barra do Rio Negro,04,,-3.113333,-60.025278
br,barra do rio pardo,Barra do Rio Pardo,27,,-24.670239,-48.681197
br,barra do rocha,Barra do Rocha,05,,-14.233333,-39.65
br,barra do roncador,Barra do Roncador,26,,-26.881276,-49.71475
br,barra do roque,Barra do Roque,20,,-6.6,-41.783333
br,barra do sabia,Barra do Sabiá,08,,-19.133333,-40.616667
br,barra do saco,Barra do Saco,02,,-9.7,-37.633333
br,barra do sai-guacu,Barra do Saí-Guaçu,18,,-25.9626,-48.598
br,barra dos andrades,Barra dos Andrades,18,,-25.663286,-50.606737
br,barra do santana,Barra do Santana,18,,-25.916667,-52.866667
br,barra do sao manuel,Barra do São Manuel,16,,-7.328056,-58.085556
br,barra do sao pedro,Barra do São Pedro,27,,-24.543839,-48.429994
br,barra dos bugres,Barra Dos Bugres,14,,-15.083333,-57.183333
br,barra dos cachimbos,Barra dos Cachimbos,30,,-8.883333,-36.4
br,barra dos coqueiros,Barra dos Coqueiros,28,28590,-10.9,-37.05
br,barra dos currais,Barra dos Currais,13,,-3.522222,-43.130278
br,barra dos dois rios,Barra dos Dois Rios,22,,-5.903056,-36.200556
br,barra do serrote,Barra do Serrote,30,,-8.433333,-39.3
br,barra do siemens,Barra do Siemens,18,,-25.6,-53.783333
br,barra do siemes,Barra do Siemes,18,,-25.6,-53.783333
br,barra do silva,Barra do Silva,30,,-8.433333,-38.85
br,barra do sirinhaem,Barra do Sirinhaém,30,,-8.6,-35.066667
br,barra do sitia,Barra do Sitiá,06,,-5.166667,-38.65
br,barra do sitio,Barra do Sítio,13,,-3.136111,-43.101944
br,barra dos melos,Barra dos Melos,06,,-3.9,-40.25
br,barra dos melos,Barra dos Melos,18,,-25.9582,-49.9061
br,barra do soldado,Barra do Soldado,27,,-24.734173,-48.050082
br,barra dos remedios,Barra dos Remédios,06,,-2.902222,-41.082778
br,barra do sul,Barra do Sul,26,,-26.4592,-48.6031
br,barra do superagui,Barra do Superagui,18,,-25.454499,-48.241236
br,barra do tapera,Barra do Tapera,13,,-5.433333,-42.933333
br,barra do taquara,Barra do Taquara,30,,-8.283333,-36.066667
br,barra do tarrachil,Barra do Tarrachil,05,,-8.783333,-38.966667
br,barra do teixeira,Barra do Teixeira,18,,-24.666667,-49.5
br,barra do triunfo,Barra do Triunfo,08,,-19.7,-40.5
br,barra do trombudo,Barra do Trombudo,26,,-27.25,-49.7
br,barra do trucu,Barra do Truçu,06,,-6.283333,-39.266667
br,barra do turvo,Barra do Turvo,27,,-24.757,-48.5059
br,barra dourada,Barra Dourada,27,,-20.783333,-49.566667
br,barra do varzea,Barra do Várzea,20,,-8.533333,-41.933333
br,barra do veado,Barra do Veado,26,,-27.383333,-51.933333
br,barra do xando,Barra do Xandó,17,,-7.583333,-36.416667
br,barra falsa,Barra Falsa,23,,-31.833333,-52.266667
br,barra feliz,Barra Feliz,15,,-19.966667,-43.45
br,barra funda,Barra Funda,02,,-9.7,-36.4
br,barra funda,Barra Funda,05,,-13.45,-40.383333
br,barra funda,Barra Funda,13,,-3.066667,-42.3
br,barragem,Barragem,05,,-10.733333,-43.5
br,barragem,Barragem,27,,-23.865658,-46.625292
br,barra grande,Barra Grande,02,,-9,-35.2
br,barra grande,Barra Grande,05,,-11.766667,-44.433333
br,barra grande,Barra Grande,13,,-3.3475,-43.476667
br,barra grande,Barra Grande,15,,-22.2,-45.333333
br,barra grande,Barra Grande,18,,-23.516498,-50.042443
br,barra grande,Barra Grande,20,,-2.911667,-41.410556
br,barra grande,Barra Grande,21,,-22.15,-42.316667
br,barra grande,Barra Grande,23,,-28.666667,-49.1
br,barra grande,Barra Grande,27,,-20.416667,-48.833333
br,barra grande,Barra Grande,30,,-8.116667,-36.516667
br,barra i,Barra I,13,,-3.393611,-43.430556
br,barra ii,Barra II,13,,-3.321111,-43.456111
br,barra logo,Barra Logo,13,,-8.216667,-45.75
br,barra longa,Barra Longa,15,,-20.283333,-43.05
br,barra luis pereira,Barra Luís Pereira,15,,-18.95,-44.116667
br,barra mansa,Barra Mansa,15,,-21.633333,-45.05
br,barra mansa,Barra Mansa,18,,-23.85231,-50.583984
br,barra mansa,Barra Mansa,21,164054,-22.533333,-44.183333
br,barra mansa,Barra Mansa,27,,-23.973624,-47.119444
br,barranca,Barranca,11,,-22.166667,-57.433333
br,barranca vermelha,Barranca Vermelha,11,,-19.933333,-56.766667
br,barranceira,Barranceira,26,,-28.416667,-48.8
br,barranco alto,Barranco Alto,11,,-18.216667,-55.483333
br,barranco alto,Barranco Alto,14,,-14.9,-56.45
br,barranco alto,Barranco Alto,15,,-21.183333,-45.966667
br,barranco alto,Barranco Alto,24,,-13.033333,-61.433333
br,barranco alto,Barranco Alto,27,,-25.214526,-48.06866
br,barranco arica,Barranco Aricá,14,,-15.983333,-55.9
br,barranco,Barranco,11,,-22.166667,-57.433333
br,barranco,Barranco,14,,-15.933333,-57.683333
br,barranco,Barranco,26,,-26.1,-48.783333
br,barranco branco,Barranco Branco,11,,-19.966667,-56.733333
br,barranco vermelho,Barranco Vermelho,11,,-19.933333,-56.766667
br,barrania,Barrânia,27,,-21.55,-46.533333
br,barra nova,Barra Nova,02,,-9.566667,-37.083333
br,barra nova,Barra Nova,05,,-16.666667,-39.616667
br,barra nova,Barra Nova,06,,-4.1,-38.15
br,barra nova,Barra Nova,08,,-18.9,-39.766667
br,barra nova,Barra Nova,13,,-3.876389,-43.326111
br,barranquinho,Barranquinho,11,,-18.216667,-55.483333
br,barrao de capanema,Barrão de Capanema,14,,-13.316667,-57.866667
br,barrao de melgaco,Barrão de Melgaco,24,,-11.85,-60.716667
br,barrao do grajahu,Barrao Do Grajahu,13,,-6.75,-43.016667
br,barrao orlando,Barrão Orlando,14,,-13.55,-57.45
br,barraos,Barraos,04,,-7.9,-60.683333
br,barra preta,Barra Preta,08,,-20.683333,-41.6
br,barraquinha,Barraquinha,18,,-24.283333,-51.566667
br,barra ribeirao da onca,Barra Ribeirão da Onça,26,,-27.0665,-49.7254
br,barra ribeirao do salto,Barra Ribeirão do Salto,27,,-24.8639,-48.1038
br,barra rio abaixo,Barra Rio Abaixo,27,,-23.383333,-47.116667
br,barra rio dona ema,Barra Rio Dona Ema,26,,-26.986507,-49.687686
br,barra rio manso,Barra Rio Manso,27,,-23.366667,-45.4
br,barras,Barras,18,,-24.066667,-52.45
br,barras,Barras,20,21597,-4.25,-42.3
br,barras do maratahoan,Barras do Maratahoan,20,,-4.25,-42.3
br,barras do maratoam,Barras do Maratoam,20,,-4.25,-42.3
br,barra seca,Barra Sêca,08,,-18.916667,-40.583333
br,barra seca,Barra Sêca,21,,-21.483333,-41.116667
br,barra sul,Barra Sul,21,,-23.007988,-43.427082
br,barratina,Barratina,11,,-20.016667,-55.116667
br,barra velha,Barra Velha,05,,-12.733333,-40.816667
br,barra velha,Barra Velha,06,,-5.9,-40.816667
br,barra velha,Barra Velha,13,,-3.534722,-43.314444
br,barra velha,Barra Velha,26,17665,-26.6406,-48.6851
br,barra velha,Barra Velha,30,,-8.1,-37.283333
br,barra verde,Barra Verde,05,,-12.766667,-40.816667
br,barra verde,Barra Verde,06,,-6.416667,-40.116667
br,barra verde,Barra Verde,22,,-6.333333,-37
br,barra verde,Barra Verde,30,,-7.866667,-39.25
br,barreado,Barreado,15,,-22.060833,-43.645
br,barreio de cima,Barreio de Cima,15,,-20.2488,-43.9922
br,barreios,Barreios,06,,-6.966667,-39.733333
br,barreira alta,Barreira Alta,03,,.55,-52.2
br,barreira alta,Barreira Alta,30,,-7.766667,-39.8
br,barreira,Barreira,04,,-4.1,-63.083333
br,barreira,Barreira,06,,-3.05,-40.116667
br,barreira,Barreira,11,,-18.033333,-54.166667
br,barreira,Barreira,13,,-3.470556,-42.534722
br,barreira,Barreira,15,,-16.283333,-41.183333
br,barreira,Barreira,16,,-4.066667,-55.883333
br,barreira,Barreira,20,,-6.483333,-41.916667
br,barreira,Barreira,21,,-21.083333,-41.866667
br,barreira,Barreira,27,,-23.116667,-49.033333
br,barreira,Barreira,30,,-8.15,-37.716667
br,barreira branca,Barreira Branca,04,,-4.315556,-67.345
br,barreira branca,Barreira Branca,20,,-3.896944,-42.215
br,barreira branca,Barreira Branca,31,,-6.7,-48.733333
br,barreira da aroeira,Barreira da Aroeira,20,,-7.766667,-41.566667
br,barreira da ema,Barreira da Ema,05,,-10.25,-42.366667
br,barreira do campo,Barreira do Campo,16,,-9.3,-50.1
br,barreira do jacuruna,Barreira do Jacuruna,05,,-13.066667,-38.85
br,barreira do mato,Barreira do Mato,13,,-5.933333,-43.45
br,barreira do matupiri,Barreira do Matupiri,04,,-5.570556,-61.106944
br,barreira dos vianas,Barreira dos Vianas,06,,-4.7,-37.766667
br,barreira do vasco,Barreira do Vasco,21,,-22.888855,-43.229272
br,barreira funda,Barreira Funda,13,,-3.879722,-43.411944
br,barreira funda,Barreira Funda,20,,-5.95,-42.866667
br,barreirao,Barreirão,05,,-11.816667,-41.95
br,barreirao,Barreirão,14,,-14.716667,-57.566667
br,barreirao,Barreirão,18,,-23.0802,-50.2295
br,barreiras,Barreiras,02,,-10.166667,-36.166667
br,barreiras,Barreiras,05,158296,-12.133333,-45
br,barreiras,Barreiras,06,,-3.416667,-39.133333
br,barreiras,Barreiras,13,,-2.868056,-41.9125
br,barreiras,Barreiras,15,,-17.533333,-43.016667
br,barreiras,Barreiras,17,,-6.866667,-34.9
br,barreiras,Barreiras,20,,-6.35,-41.933333
br,barreiras,Barreiras,21,,-21.083333,-41.866667
br,barreiras,Barreiras,22,,-5.083333,-36.5
br,barreiras,Barreiras,31,,-5.9,-47.45
br,barreiras de jacuruna,Barreiras de Jacuruna,05,,-13.066667,-38.85
br,barreiras de santo antonio,Barreiras de Santo Antônio,05,,-13.066667,-38.95
br,barreira vermelha,Barreira Vermelha,22,,-5.166667,-36.016667
br,barreirinha,Barreirinha,04,11257,-2.793611,-57.068889
br,barreirinha,Barreirinha,13,,-4.233333,-44.233333
br,barreirinha,Barreirinha,14,,-16.55,-56.116667
br,barreirinha,Barreirinha,15,,-21.333333,-45.516667
br,barreirinha,Barreirinha,16,,-9.466667,-50.133333
br,barreirinha,Barreirinha,20,,-7.166667,-40.933333
br,barreirinha,Barreirinha,27,,-22.866667,-47.7
br,barreirinhas,Barreirinhas,13,17437,-2.75,-42.833333
br,barreirinho,Barreirinho,15,,-21,-45.833333
br,barreirinho,Barreirinho,18,,-23.1297,-50.0098
br,barreirinho,Barreirinho,27,,-22.866667,-47.7
br,barreirinho d'oeste,Barreirinho d'Oeste,18,,-24.133333,-52.833333
br,barreirinho dos beltrao,Barreirinho dos Beltrão,18,,-25.730175,-50.652944
br,barreiro baixo,Barreiro Baixo,29,,-13.766667,-46.45
br,barreiro,Barreiro,05,,-10.75,-44.483333
br,barreiro,Barreiro,06,,-6.083333,-40.8
br,barreiro,Barreiro,07,,-15.983333,-47.483333
br,barreiro,Barreiro,11,,-18.033333,-54.166667
br,barreiro,Barreiro,13,,-3.210833,-43.335833
br,barreiro,Barreiro,14,,-14.483333,-55.8
br,barreiro,Barreiro,15,,-14.4,-43.866667
br,barreiro,Barreiro,18,,-24.15,-51
br,barreiro,Barreiro,20,,-10.75,-44.783333
br,barreiro,Barreiro,23,,-28.45,-53.95
br,barreiro,Barreiro,26,,-26.3,-50.533333
br,barreiro,Barreiro,27,,-22.216667,-47.116667
br,barreiro,Barreiro,30,,-7.733333,-37.833333
br,barreiro,Barreiro,31,,-5.7,-48.116667
br,barreiro branco,Barreiro Branco,04,,-7.183333,-58.116667
br,barreiro branco,Barreiro Branco,05,,-14.05,-43.683333
br,barreiro da ema,Barreiro da Ema,05,,-10.25,-42.366667
br,barreiro da ema,Barreiro da Ema,22,,-6.133333,-36.3
br,barreiro da raiz,Barreiro da Raiz,15,,-16.033333,-43.266667
br,barreiro de baixo,Barreiro de Baixo,27,,-22.616667,-44.4
br,barreiro de baixo,Barreiro de Baixo,29,,-13.766667,-46.45
br,barreiro de santo antonio,Barreiro de Santo Antônio,05,,-13.066667,-38.95
br,barreiro do jaiba,Barreiro do Jaíba,15,,-15.616667,-43.6
br,barreiro do nascimento,Barreiro do Nascimento,31,,-11.883333,-50.633333
br,barreiro dos cocos,Barreiro dos Cocós,20,,-3.638056,-42.163889
br,barreiro grande,Barreiro Grande,05,,-13.766667,-43.6
br,barreiro grande,Barreiro Grande,18,,-25.8999,-49.8913
br,barreiro iii,Barreiro III,20,,-3.520833,-42.033333
br,barreiro iv,Barreiro IV,20,,-3.5475,-42.127222
br,barreiro manecao,Barreiro Manecão,27,,-24.9374,-48.2948
br,barreiro novo,Barreiro Novo,05,,-13.833333,-41.133333
br,barreiro rico,Barreiro Rico,27,,-23.215193,-47.785318
br,barreiros,Barreiros,02,,-9.383333,-38.05
br,barreiros,Barreiros,05,,-15.6,-39.2
br,barreiros,Barreiros,06,,-4.416667,-38.65
br,barreiros,Barreiros,15,,-17.316667,-44.466667
br,barreiros,Barreiros,16,,-4.066667,-55.883333
br,barreiros,Barreiros,17,,-7.633333,-38.683333
br,barreiros,Barreiros,20,,-5.7,-43.066667
br,barreiros,Barreiros,22,,-5.083333,-36.5
br,barreiros,Barreiros,26,,-26.3,-50.533333
br,barreiros,Barreiros,27,,-24.7921,-48.4774
br,barreiros,Barreiros,30,35565,-8.816667,-35.2
br,barreiros de santo antonio,Barreiros de Santo Antônio,05,,-13.066667,-38.95
br,barreiros dos porcos,Barreiros dos Porcos,20,,-7.633333,-41.666667
br,barreiro vermelho,Barreiro Vermelho,15,,-15.366667,-43.283333
br,barrejirnha,Barrejirnha,20,,-5.766667,-42.6
br,barrelina,Barrelina,11,,-20.016667,-55.116667
br,barrenta,Barrenta,06,,-3.266667,-39.633333
br,barrento,Barrento,06,,-3.266667,-39.533333
br,barrento,Barrento,22,,-6,-36.05
br,barrento dos cesarios,Barrento dos Cesarios,17,,-6.9,-37.783333
br,barrera,Barrera,13,,-7.9,-47
br,barrero,Barrero,11,,-18.366667,-55.983333
br,barretina,Barretina,11,,-17.933333,-54.5
br,barreto,Barreto,08,,-20.633333,-41.233333
br,barreto,Barreto,14,,-8.8,-58.4
br,barreto,Barreto,15,,-20.05,-43.2
br,barreto,Barreto,21,,-22.333333,-41.75
br,barreto,Barreto,22,,-5.683333,-35.833333
br,barreto,Barreto,23,,-28.3,-51.466667
br,barreto,Barreto,30,,-7.55,-39.35
br,barretos,Barretos,15,,-20.283333,-43.166667
br,barretos,Barretos,27,101222,-20.55,-48.55
br,barretro ii,Barretro II,20,,-3.403889,-42.034444
br,barriero,Barriero,27,,-21.583333,-47.183333
br,barrieros,Barrieros,17,,-6.633333,-37.966667
br,barriga,Barriga,20,,-6.666667,-42.183333
br,barrigada,Barrigada,15,,-15.75,-46.35
br,barriguda,Barriguda,02,,-9.416667,-37.266667
br,barriguda,Barriguda,05,,-10.516667,-41.05
br,barriguda,Barriguda,06,,-5.6,-40.783333
br,barriguda,Barriguda,13,,-3.083333,-44.116667
br,barriguda,Barriguda,17,,-7.733333,-38.55
br,barriguda,Barriguda,28,,-10.833333,-38.133333
br,barriguda,Barriguda,30,,-7.666667,-38.783333
br,barriguda de cima,Barriguda de Cima,05,,-11.283333,-40.933333
br,barrigudinha,Barrigudinha,30,,-7.55,-39.55
br,barrigudo,Barrigudo,02,,-9.6,-36.9
br,barrigudo,Barrigudo,05,,-12.033333,-41.65
br,barrinha,Barrinha,05,,-10.233333,-43.5
br,barrinha,Barrinha,06,,-2.8,-40.35
br,barrinha,Barrinha,13,,-4.216667,-43.233333
br,barrinha,Barrinha,14,,-16.7,-56.833333
br,barrinha,Barrinha,18,,-24.714962,-48.995249
br,barrinha,Barrinha,20,,-2.915556,-41.385
br,barrinha,Barrinha,21,,-22.205833,-43.015278
br,barrinha,Barrinha,22,,-5.1675,-37.419722
br,barrinha,Barrinha,23,,-28.55,-52.666667
br,barrinha,Barrinha,27,26981,-21.2,-48.15
br,barrinha roteio,Barrinha Roteio,02,,-9.833333,-35.966667
br,barrinhas,Barrinhas,20,,-5.133333,-42.516667
br,barrinho,Barrinho,14,,-16.7,-56.833333
br,barrinho,Barrinho,27,,-23.116667,-45.366667
br,barrinia,Barrinia,27,,-21.55,-46.533333
br,barrio barra grande,Barrio Barra Grande,18,,-24.681411,-48.965007
br,barrio do costao,Barrio do Costão,18,,-25.289301,-48.328532
br,barrio do pavao,Barrio do Pavão,27,,-24.643945,-48.790037
br,barrio do rolado,Barrio do Rolado,18,,-24.716632,-48.930163
br,barrio ponta aguda,Barrio Ponta Aguda,26,,-26.9007,-49.0476
br,barrios,Barrios,11,,-21.783333,-53.55
br,barrio vocoroca,Barrio Voçoroca,18,,-25.8202,-49.0623
br,barris,Barris,05,,-12.985,-38.51
br,barroada,Barroada,15,,-19.016667,-42.65
br,barro alto,Barro Alto,04,,-3.883333,-63.766667
br,barro alto,Barro Alto,05,,-11.766667,-41.9
br,barro alto,Barro Alto,20,,-8.85,-45.383333
br,barro alto,Barro Alto,29,,-15.066667,-48.966667
br,barro,Barro,05,,-12.583333,-38.266667
br,barro,Barro,06,11519,-6.716667,-38.866667
br,barro,Barro,08,,-20.8,-41.716667
br,barro,Barro,13,,-3.083333,-45.416667
br,barro,Barro,15,,-16.333333,-41.483333
br,barro,Barro,17,,-6.983333,-38.05
br,barro,Barro,20,,-6.916667,-41.466667
br,barro,Barro,22,,-6.366667,-38.4
br,barro,Barro,23,,-27.566667,-52.05
br,barro,Barro,25,,4.283333,-60.833333
br,barro,Barro,27,,-23.716667,-45.4
br,barro,Barro,30,,-7.5,-39.8
br,barro branco,Barro Branco,06,,-7.75,-38.983333
br,barro branco,Barro Branco,08,,-20.55,-40.483333
br,barro branco,Barro Branco,13,,-3.805833,-43.3275
br,barro branco,Barro Branco,15,,-20.716667,-41.933333
br,barro branco,Barro Branco,16,,-1.333333,-47.8
br,barro branco,Barro Branco,18,,-25.225452,-48.872992
br,barro branco,Barro Branco,20,,-6.966667,-41.933333
br,barro branco,Barro Branco,21,,-21.5,-41.783333
br,barro branco,Barro Branco,25,,-3.366667,-60.25
br,barro branco,Barro Branco,26,,-28.416667,-49.416667
br,barro branco,Barro Branco,27,,-24.689223,-48.087762
br,barro branco,Barro Branco,30,,-8.483333,-36.866667
br,barro branco de baixo,Barro Branco de Baixo,18,,-25.699307,-50.48678
br,barro branco de cima,Barro Branco de Cima,18,,-25.68946,-50.48516
br,barroca,Barroca,05,,-10.9,-38.35
br,barroca,Barroca,13,,-3.712778,-43.188611
br,barroca,Barroca,17,,-7.166667,-36.433333
br,barroca,Barroca,20,,-3.337222,-42.019444
br,barroca,Barroca,22,,-5.986944,-36.768889
br,barroca,Barroca,27,,-21.416667,-47.283333
br,barroca,Barroca,29,,-15.35,-48.5
br,barroca,Barroca,30,,-8.25,-38.533333
br,barroca da onca,Barroca da Onça,20,,-3.576944,-42.171389
br,barroca da vaca,Barroca da Vaca,13,,-3.720278,-43.480556
br,barroca funda,Barroca Funda,20,,-7.916667,-43.116667
br,barroca funda,Barroca Funda,27,,-23.236,-47.9259
br,barrocao,Barrocão,05,,-9.833333,-42.116667
br,barrocao,Barrocão,13,,-3.123056,-43.095556
br,barrocao,Barrocão,15,,-16.416667,-43.25
br,barrocao,Barrocão,17,,-7.483333,-38.416667
br,barrocao,Barrocão,20,,-3.678056,-42.5125
br,barrocao,Barrocão,22,,-5.210278,-36.145556
br,barrocao,Barrocão,27,,-22.716667,-47.366667
br,barrocao de baixo,Barrocão de Baixo,13,,-3.682222,-43.35
br,barrocao de cima,Barrocão de Cima,13,,-3.675278,-43.361389
br,barrocao i,Barrocão I,13,,-3.409444,-43.138889
br,barrocao ii,Barrocão II,13,,-3.395,-43.133611
br,barrocas,Barrocas,05,,-11.516667,-39.066667
br,barrocas,Barrocas,20,,-7.883333,-43.1
br,barro da cabra,Barro da Cabra,17,,-6.966667,-38.133333
br,barro do prata,Barro do Prata,20,,-8.05,-44.466667
br,barro do saltinho,Barro do Saltinho,27,,-24.686267,-48.568586
br,barro duro,Barro Duro,13,,-2.866667,-42.283333
br,barro duro,Barro Duro,20,,-5.816667,-42.516667
br,barrolandia,Barrolândia,31,,-9.780833,-48.621944
br,barro preto,Barro Prêto,05,,-14.766667,-39.45
br,barro preto,Barro Prêto,15,,-20.383333,-43.016667
br,barro preto,Barro Preto,18,,-23.6396,-49.9692
br,barro preto,Barro Preto,20,,-5.283333,-42.65
br,barro preto,Barro Prêto,23,,-28.1,-53.7
br,barro preto,Barro Prêto,27,,-20.65,-47.333333
br,barro preto,Barro Prêto,30,,-8.566667,-37.116667
br,barroquinha,Barroquinha,06,12410,-2.983333,-41.083333
br,barroquinha,Barroquinha,13,,-3.362222,-42.719722
br,barroquinha dos fieis,Barroquinha dos Fiéis,06,,-2.979722,-41.134167
br,barro salto do meio,Barro Salto do Meio,18,,-25.806415,-48.990584
br,barros,Barros,17,,-7.3,-37.566667
br,barros,Barros,27,,-23.7378,-47.6138
br,barros cassal,Barros Cassal,23,,-29.1,-52.583333
br,barros filho,Barros Filho,21,,-22.830855,-43.369472
br,barroso,Barroso,02,,-9.366667,-37.233333
br,barroso,Barroso,14,,-15.9,-56.816667
br,barroso,Barroso,15,16975,-21.183333,-43.966667
br,barroso,Barroso,20,,-7.45,-41.566667
br,barroso,Barroso,27,,-21.416667,-47.283333
br,barro verde,Barro Verde,05,,-12.766667,-40.816667
br,barro verde,Barro Verde,17,,-7.2,-37.233333
br,barro vermelho,Barro Vermelho,05,,-10.966667,-44.166667
br,barro vermelho,Barro Vermelho,06,,-3.966667,-40.233333
br,barro vermelho,Barro Vermelho,13,,-3.701111,-42.743889
br,barro vermelho,Barro Vermelho,17,,-6.633333,-37.883333
br,barro vermelho,Barro Vermelho,20,,-3.367222,-42.085556
br,barro vermelho,Barro Vermelho,21,,-22.066667,-41.666667
br,barro vermelho,Barro Vermelho,22,,-5.476111,-36.140556
br,barro vermelho,Barro Vermelho,23,,-29.9277,-50.8876
br,barro vermelho,Barro Vermelho,26,,-28.9,-49.466667
br,barro vermelho,Barro Vermelho,30,,-7.466667,-37.35
br,barruada,Barruada,15,,-19.016667,-42.65
br,bar terra-firme,Bar Terra-Firme,24,,-12.033333,-64.283333
br,bartolomeu,Bartolomeu,06,,-3.6,-40.166667
br,bartolomeu,Bartolomeu,17,,-7.3,-38.466667
br,barueri,Barueri,27,251999,-23.516667,-46.883333
br,baruri,Baruri,04,,-.883333,-63.016667
br,base aerea do amapa,Base Aérea do Amapá,03,,2.066667,-50.866667
br,base das canoas,Base das Canoas,16,,-2.25,-55.95
br,basilio alves morango,Basílio Alves Morango,08,,-18.016667,-40.766667
br,basilio,Basílio,20,,-7.5,-40.833333
br,basilio,Basílio,23,,-31.883333,-53.016667
br,basilio gomes,Basílio Gomes,15,,-17.933333,-40.816667
br,bastiao,Bastião,05,,-12.816667,-41.7
br,bastioes,Bastiões,06,,-4.616667,-38.716667
br,bastioes,Bastiões,17,,-7.05,-35.55
br,bastister,Bastister,04,,-5.629444,-63.518333
br,bastos,Bastos,13,,-3.561389,-42.996111
br,bastos,Bastos,27,17681,-21.933333,-50.716667
br,basuca,Basuca,15,,-18.766667,-41.883333
br,bataguacu,Bataguaçu,11,,-21.7,-52.366667
br,bataipora,Bataiporã,11,7346,-22.333333,-53.283333
br,batal,Batal,20,,-6.1,-42.2
br,batalha,Batalha,02,11820,-9.683333,-37.133333
br,batalha,Batalha,05,,-10.866667,-45.3
br,batalha,Batalha,06,,-4.25,-38.616667
br,batalha,Batalha,08,,-21.1,-41.016667
br,batalha,Batalha,13,,-5.866667,-43.916667
br,batalha,Batalha,20,8744,-4.016667,-42.083333
br,batalha,Batalha,21,,-22.716667,-43.15
br,batalha,Batalha,27,,-21.916667,-49.25
br,batalha,Batalha,30,,-7.833333,-38.383333
br,batalha,Batalha,31,,-12.316667,-46.4
br,batalhao,Batalhão,17,,-7.2,-36.816667
br,batalha velha,Batalha Velha,13,,-6.383333,-43.216667
br,batan,Batan,21,,-22.862636,-43.424926
br,batataes,Batataes,27,,-20.883333,-47.616667
br,batatais,Batatais,27,51977,-20.883333,-47.616667
br,batatal,Batatal,02,,-9.4,-37.25
br,batatal,Batatal,14,,-14.466667,-57.35
br,batatal,Batatal,27,,-24.669195,-48.328014
br,batatas,Batatas,06,,-6.45,-40.2
br,batateira,Batateira,05,,-9.95,-41.366667
br,batateira,Batateira,30,,-8.55,-35.85
br,batateiro,Batateiro,30,,-8.65,-37.1
br,batatuba,Batatuba,27,,-23.066667,-46.416667
br,bate,Bate,20,,-8.866667,-43.15
br,batedor,Batedor,06,,-5.166667,-39.6
br,batedor,Batedor,30,,-7.566667,-39.816667
br,bateeiro,Bateeiro,05,,-12.383333,-40.916667
br,bateia,Bateia,15,,-22.366667,-45.833333
br,bateia,Batéia,18,,-24.800149,-48.868408
br,bateia,Bateia,26,,-26.966667,-48.916667
br,bateia do barracao,Batéia do Barracão,26,,-26.966667,-48.916667
br,bateias,Bateias,18,,-25.366667,-49.516667
br,bateias,Bateias,27,,-23.7035,-47.7711
br,bateias de baixo,Bateias de Baixo,26,,-26.0894,-49.2582
br,bateiro,Bateiro,05,,-12.383333,-40.916667
br,batelao,Batelão,21,,-21.316667,-41.033333
br,batel,Batel,18,,-25.4359,-48.738
br,batente,Batente,30,,-8.566667,-35.9
br,bateria,Bateria,04,,.966667,-66.833333
br,bate-vento,Bate-Vento,13,,-1.3,-44.9
br,batinga,Batinga,05,,-16.916667,-40.5
br,batinga,Batinga,15,,-21.683333,-45.05
br,batinga,Batinga,17,,-7.333333,-37.466667
br,batinga,Batinga,20,,-7.383333,-40.6
br,batinga,Batinga,30,,-7.633333,-37.8
br,batista,Batista,05,,-12.933333,-39.55
br,batista,Batista,27,,-24.9408,-48.0878
br,batista botelho,Batista Botelho,27,,-23.016667,-49.416667
br,batistini,Batistini,27,,-23.7564,-46.5654
br,batoque,Batoque,05,,-10.566667,-40.6
br,batoque,Batoque,06,,-4.4,-40.366667
br,batovi,Batovi,14,,-15.883333,-53.4
br,batuque,Batuque,05,,-15.1,-41.083333
br,baturite,Baturité,04,,-5.720833,-63.510278
br,baturite,Baturité,06,22417,-4.333333,-38.883333
br,baturite,Baturité,13,,-1.983333,-44.783333
br,batuva,Batuva,18,,-25.088056,-48.230278
br,baua,Baua,04,,1.033333,-66.816667
br,bau,Baú,06,,-6.4,-39.4
br,bau,Baú,11,,-18.316667,-53.166667
br,bau,Bau,13,,-5.25,-43.433333
br,bau,Baú,14,,-15.233333,-56.3
br,bau,Baú,15,,-17.916667,-43.083333
br,bau de cima,Baú de Cima,27,,-22.6952,-45.6205
br,bau do centro,Baú do Centro,27,,-22.7051,-45.6635
br,baunilha,Baunilha,08,,-19.566667,-40.5
br,baunilha,Baunilha,14,,-16.133333,-55.416667
br,bauru,Bauru,27,335031,-22.316667,-49.066667
br,baus,Baús,11,,-18.316667,-53.166667
br,bauta,Bautá,04,,.316667,-67.283333
br,bau velho,Baú Velho,27,,-22.7233,-45.6926
br,bauxi,Bauxi,14,,-15.166667,-56.65
br,bauzinho,Bauzinho,11,,-18.316667,-53.166667
br,baviera,Baviera,22,,-5.543889,-36.908889
br,baxio,Baxio,30,,-8.4,-40.183333
br,baxios,Baxios,02,,-9.7,-37.2
br,baxiu,Baxiú,22,,-5.166667,-35.8
br,bayeux,Bayeux,17,91058,-7.133333,-34.933333
br,b. da boiada,B. da Boiada,27,,-21.283333,-47.1
br,b. das rosas,B. das Rosas,27,,-22.766667,-46.783333
br,bebedor,Bebedor,02,,-9.116667,-37.666667
br,bebedor,Bebedor,11,,-21.316667,-55.866667
br,bebedor,Bebedor,20,,-5.05,-41.15
br,bebedouro,Bebedouro,02,,-9.65,-35.75
br,bebedouro,Bebedouro,05,,-10.283333,-37.933333
br,bebedouro,Bebedouro,06,,-6.633333,-40.116667
br,bebedouro,Bebedouro,13,,-2.466667,-44.766667
br,bebedouro,Bebedouro,15,,-19.233333,-48.383333
br,bebedouro,Bebedouro,17,,-6.883333,-38.416667
br,bebedouro,Bebedouro,27,71863,-20.933333,-48.466667
br,bebedouro,Bebedouro,30,,-8.45,-35.95
br,bebedouro de cima,Bebedouro de Cima,20,,-4.166667,-41.533333
br,bebedouro de sao pedro,Bebedouro de São Pedro,13,,-3.25,-42.116667
br,bebedouro i,Bebedouro I,13,,-3.473889,-43.381389
br,bebedouro ii,Bebedouro II,13,,-3.326667,-43.070833
br,bebedouro iii,Bebedouro III,13,,-3.555,-43.090833
br,bebedouuro de sao pedro,Bebedouuro de São Pedro,13,,-3.25,-42.116667
br,bebedovro,Bebedovro,13,,-6.166667,-43.1
br,beberibe,Beberibe,06,37028,-4.183333,-38.133333
br,bebida nova,Bebida Nova,13,,-3.932778,-42.976944
br,bebida velha,Bebida Velha,22,,-5.333333,-35.533333
br,beco,Beco,05,,-12.45,-41.683333
br,beco,Beco,06,,-5.466667,-40.85
br,becreio,Becreio,20,,-7.933333,-43.033333
br,beija-flor,Beija-Flor,05,,-12.683333,-38.95
br,beija-flor,Beija-Flor,11,,-21.383333,-56.233333
br,beija-flor,Beija-Flor,14,,-13.783333,-55.233333
br,beija-flor,Beija-Flor,15,,-21.133333,-43.083333
br,beija-flor,Beija-Flor,20,,-3.756111,-42.256389
br,beiju,Beiju,30,,-8.483333,-37.683333
br,beira alta,Beira Alta,08,,-20.7,-41.3
br,beira do rio,Beira do Rio,05,,-13.583333,-41.216667
br,beira do rio,Beira do Rio,06,,-6.416667,-38.883333
br,beira do tai,Beira do Taí,21,,-21.783333,-41.166667
br,beira mar,Beira Mar,23,,-32.166667,-52.116667
br,beirao,Beirão,15,,-16.833333,-41.8
br,beira serra,Beira Serra,15,,-20.2315,-43.9929
br,beja,Beja,16,,-1.6,-48.783333
br,bejamin novais,Bejamin Novais,23,,-28.666667,-53.6
br,beju,Beju,17,,-6.533333,-35.3
br,bela alianca,Bela Aliança,06,,-5.366667,-39.183333
br,bela alianca,Bela Aliança,26,,-27.1855,-49.5815
br,bela aurora,Bela Aurora,04,,.066667,-63.983333
br,bela aurora,Bela Aurora,06,,-4.866667,-40.266667
br,bela aurora,Bela Aurora,08,,-20.7,-41.416667
br,bela aurora,Bela Aurora,16,,-1.766667,-46.333333
br,bela aurora,Bela Aurora,28,,-9.816667,-37.5
br,bela brisa,Bela Brisa,04,,-6.883611,-62.637778
br,bela cruz,Bela Cruz,06,12715,-3.05,-40.183333
br,bela cruz,Bela Cruz,15,,-22.366667,-45.633333
br,bela estrela,Bela Estrêla,13,,-5.483333,-46.183333
br,bela fama,Bela Fama,15,,-20.0066,-43.8379
br,bela flor,Bela Flor,05,,-12.416667,-38.466667
br,bela floresta,Bela Floresta,27,,-20.6,-51.266667
br,belagua,Belágua,13,,-3.005,-42.448333
br,bela joana,Bela Joana,21,,-22.016667,-42.666667
br,bela lorena,Bela Lorena,15,,-15.216667,-46.016667
br,bela manha,Bela Manhã,18,,-23.466667,-50.35
br,belamento,Belamento,30,,-7.983333,-37.233333
br,belamino dias,Belamino Dias,15,,-17.016667,-46.75
br,bela monte,Bela Monte,16,,-3.083333,-51.766667
br,bela nova,Bela Nova,05,,-15.05,-39.683333
br,bela paisagem,Bela Paisagem,05,,-13.566667,-40.883333
br,bela pinta,Bela Pinta,05,,-14.716667,-40.633333
br,belarmino,Belarmino,18,,-25.133333,-52.816667
br,belarmino,Belarmino,29,,-17.516667,-52.7
br,belarmino ferreira,Belarmino Ferreira,15,,-16.883333,-43.2
br,belas aguas,Belas Águas,06,,-5.633333,-38.166667
br,belas artes,Belas Artes,11,,-21.683333,-54.383333
br,bela veneza,Bela Veneza,20,,-6.016667,-42.083333
br,bela vista,Bela Vista,02,,-9.283333,-36.233333
br,bela vista,Bela Vista,03,,2.266667,-50.75
br,bela vista,Bela Vista,04,,.133333,-67.9
br,bela vista,Bela Vista,05,,-11.833333,-41.016667
br,bela vista,Bela Vista,06,,-3.216667,-39.616667
br,bela vista,Bela Vista,08,,-19.533333,-40.416667
br,bela vista,Bela Vista,11,18127,-22.1,-56.516667
br,bela vista,Bela Vista,13,,-3.401389,-43.222778
br,bela vista,Bela Vista,14,,-16.05,-56.883333
br,bela vista,Bela Vista,15,,-15.733333,-44.55
br,bela vista,Bela Vista,16,,-3.366667,-51.833333
br,bela vista,Bela Vista,18,,-23.215593,-52.231562
br,bela vista,Bela Vista,20,,-3.916389,-42.227778
br,bela vista,Bela Vista,21,,-20.983333,-41.916667
br,bela vista,Bela Vista,22,,-5.161389,-36.496944
br,bela vista,Bela Vista,23,,-28.2,-52.633333
br,bela vista,Bela Vista,24,,-11.516667,-65.25
br,bela vista,Bela Vista,25,,1.933333,-60.983333
br,bela vista,Bela Vista,26,,-26.1784,-49.977
br,bela vista,Bela Vista,27,,-20.033333,-47.85
br,bela vista,Bela Vista,29,,-16.966667,-48.95
br,bela vista,Bela Vista,30,,-7.466667,-39.733333
br,bela vista,Bela Vista,31,,-8.211111,-49.222222
br,bela vista de goias,Bela Vista de Goiás,29,14289,-16.966667,-48.95
br,bela vista do meio,Bela Vista do Meio,05,,-14.933333,-40.233333
br,bela vista do paraiso,Bela Vista do Paraíso,18,14332,-23,-51.2
br,bela vista do sul,Bela Vista do Sul,26,,-26.1784,-49.977
br,bela vista do toldo,Bela Vista do Tôldo,26,,-26.266667,-50.5
br,belchior alto,Belchior Alto,26,,-26.8106,-49.0122
br,belchior baixo,Belchior Baixo,26,,-26.8544,-49.0348
br,belchior,Belchior,15,,-20.733333,-44.016667
br,belchior,Belchior,26,,-26.8544,-49.0348
br,belchior,Belchior,30,,-7.95,-39.9
br,beldroega,Beldroega,30,,-8.85,-36.75
br,beldroegas,Beldroegas,30,,-8.75,-38.166667
br,bele,Belé,06,,-5.283333,-39.15
br,belem,Belém,02,,-9.583333,-36.416667
br,belem,Belém,04,,-3.916667,-69.616667
br,belem,Belém,05,,-12.616667,-38.933333
br,belem,Belém,06,,-4.5,-39.616667
br,belem,Belém,08,,-18.466667,-39.733333
br,belem,Belém,13,,-1.85,-44.8
br,belem,Belem,15,,-18.9,-41.616667
br,belem,Belém,16,1407737,-1.45,-48.483333
br,belem,Belém,17,15550,-6.716667,-35.5
br,belem,Belém,18,,-23.433333,-51.733333
br,belem,Belém,20,,-4.216667,-42.983333
br,belem,Belém,21,,-22.966667,-44.283333
br,belem,Belém,22,,-5.35,-35.966667
br,belem,Belém,24,,-8.533333,-63.666667
br,belem,Belém,28,,-9.8,-37.333333
br,belem,Belém,30,,-7.916667,-38.466667
br,belem da cachoeira,Belém da Cachoeira,05,,-12.616667,-38.933333
br,belem de cabroba,Belém de Cabrobá,30,,-8.766667,-38.966667
br,belem de caicara,Belém de Caiçara,17,,-6.716667,-35.5
br,belem de maria,Belém de Maria,30,,-8.616667,-35.85
br,belem de sao francisco,Belém de São Francisco,30,11124,-8.766667,-38.966667
br,belem do bajados,Belém do Bajadós,04,,-3.494722,-62.547222
br,belem do brejo do cruz,Belém do Brejo do Cruz,17,,-6.183333,-37.533333
br,belem do candeia,Belém do Candeia,04,,-3.516667,-62.466667
br,belem do condeia,Belêm do Condeia,04,,-3.516667,-62.466667
br,belem do para,Belem do Pará,16,,-1.45,-48.483333
br,belem do sao francisco,Belém do São Francisco,30,,-8.766667,-38.966667
br,belem nova,Belem Nova,23,,-30.116667,-51.166667
br,belem novo,Belém Novo,23,,-30.2,-51.166667
br,belem santos,Belém Santos,15,,-18.9,-41.616667
br,belem velho,Belém Velho,23,,-30.116667,-51.166667
br,belexas,Belexas,04,,-2.55,-67.383333
br,beleza,Beleza,02,,-9.316667,-36.233333
br,beleza,Beleza,06,,-7.283333,-38.666667
br,beleza,Beleza,15,,-16.865833,-44.584722
br,beleza,Beleza,20,,-6.116667,-41.583333
br,beleza,Beleza,22,,-6.066667,-37.583333
br,beleza,Beleza,30,,-7.833333,-40.516667
br,belezas,Belezas,04,,-2.55,-67.383333
br,belford roxo,Belford Roxo,21,466101,-22.761995,-43.404654
br,belfort roxo,Belfort Roxo,21,,-22.761995,-43.404654
br,belisario,Belisário,15,,-20.916667,-42.466667
br,belisario,Belisário,23,,-28.466667,-53.433333
br,belisario benitez,Belisário Benítez,23,,-29.8,-56.85
br,belisario fulgencio,Belisário Fulgêncio,15,,-16.8,-41.666667
br,bella brisa,Bella Brisa,04,,-6.883611,-62.637778
br,bella monte,Bella Monte,16,,-3.083333,-51.766667
br,bella vista,Bella Vista,04,,.133333,-67.9
br,bella vista,Bella Vista,05,,-11.833333,-41.016667
br,bella vista,Bella Vista,11,,-22.1,-56.516667
br,bella vista,Bella Vista,15,,-16.258056,-44.159167
br,bella vista,Bella Vista,16,,-7.55,-53.3
br,bella vista,Bella Vista,18,,-25.433333,-54.583333
br,bella vista,Bella Vista,25,,1.933333,-60.983333
br,bella vista,Bella Vista,26,,-26.1784,-49.977
br,bella vista,Bella Vista,29,,-16.966667,-48.95
br,bello horizonte,Bello Horizonte,04,,-2.683333,-67.533333
br,bello horizonte,Bello Horizonte,15,,-19.916667,-43.933333
br,bello horizonte,Bello Horizonte,16,,-5.3,-52.866667
br,bello horizonte,Bello Horizonte,24,,-11.883333,-64.166667
br,bello jardim,Bello Jardim,30,,-8.333333,-36.433333
br,bello monte,Bello Monte,02,,-9.833333,-37.266667
br,bello monte,Bello Monte,04,,-8.116667,-70.333333
br,bello monte,Bello Monte,16,,-3.083333,-51.766667
br,bello monte,Bello Monte,27,,-23.35,-49.366667
br,bello valle,Bello Valle,15,,-20.416667,-44.016667
br,belmiro braga,Belmiro Braga,15,,-21.949722,-43.415833
br,belmont,Belmont,06,,-4.866667,-39.683333
br,belmont,Belmont,15,,-21.827222,-43.277778
br,belmont,Belmont,24,,-8.65,-63.833333
br,belmonte,Belmonte,01,,-9.333333,-72.2
br,belmonte,Belmonte,05,9806,-15.85,-38.9
br,belmonte,Belmonte,06,,-5.1,-40.683333
br,belmonte,Belmonte,14,,-7.65,-58.116667
br,belmonte,Belmonte,15,,-20,-41.45
br,belmonte,Belmonte,20,,-7.716667,-40.8
br,belmonte,Belmonte,21,,-22.35,-41.966667
br,belmonte,Belmonte,24,,-8.65,-63.833333
br,belmonte,Belmonte,27,,-21.466667,-47.516667
br,belmonte,Belmonte,30,,-7.866667,-38.766667
br,belo banho,Belo Banho,20,,-9.333333,-45.65
br,belo campo,Belo Campo,05,,-11.55,-41.616667
br,belo horizonte,Belo Horizonte,02,,-9.45,-37.266667
br,belo horizonte,Belo Horizonte,04,,-2.683333,-67.533333
br,belo horizonte,Belo Horizonte,05,,-10.15,-42.233333
br,belo horizonte,Belo Horizonte,06,,-7.083333,-40.083333
br,belo horizonte,Belo Horizonte,08,,-20.733333,-40.616667
br,belo horizonte,Belo Horizonte,11,,-19.8,-54.033333
br,belo horizonte,Belo Horizonte,15,2373255,-19.916667,-43.933333
br,belo horizonte,Belo Horizonte,16,,-5.3,-52.866667
br,belo horizonte,Belo Horizonte,17,,-6.866667,-38.35
br,belo horizonte,Belo Horizonte,20,,-6.883333,-40.6
br,belo horizonte,Belo Horizonte,22,,-5.456944,-36.410556
br,belo horizonte,Belo Horizonte,24,,-11.883333,-64.166667
br,belo horizonte,Belo Horizonte,27,,-21.416667,-47.933333
br,belo horizonte,Belo Horizonte,28,,-9.783333,-37.866667
br,belo horizonte,Belo Horizonte,31,,-10.483333,-49.516667
br,belo hurizonte,Belo Hurizonte,20,,-3.374167,-42.035556
br,belo jardim,Belo Jardim,02,,-9.416667,-37.316667
br,belo jardim,Belo Jardim,05,,-12.35,-40.083333
br,belo jardim,Belo Jardim,30,49923,-8.333333,-36.433333
br,belo monte,Belo Monte,01,,-10.033333,-66.8
br,belo monte,Belo Monte,02,,-9.833333,-37.266667
br,belo monte,Belo Monte,04,,-6.216667,-64.183333
br,belo monte,Belo Monte,05,,-13.666667,-40.316667
br,belo monte,Belo Monte,16,,-3.083333,-51.766667
br,belo monte,Belo Monte,20,,-6.5,-42.683333
br,belo monte,Belo Monte,27,,-23.35,-49.366667
br,belomonte,Belomonte,30,,-8.816667,-36.55
br,belo norte,Belo Norte,06,,-5.383333,-39.383333
br,belo oriente,Belo Oriente,15,18471,-19.233333,-42.466667
br,belo vale,Belo Vale,15,,-20.416667,-44.016667
br,belterra,Belterra,16,,-2.633333,-54.95
br,beluno,Beluno,23,,-29.45,-54.966667
br,belvedere,Belvedere,26,,-28.516667,-49.383333
br,bem-bom,Bem-Bom,05,,-9.716667,-41.8
br,bem-feita,Bem-Feita,02,,-9.283333,-37.566667
br,bemfica,Bemfica,15,,-21.683333,-43.433333
br,bemfica,Bemfica,16,,-1.3,-48.3
br,bem posta,Bem Posta,15,,-17.416667,-42.516667
br,bemposta,Bemposta,21,,-22.139722,-43.098056
br,bem-quer,Bem-Quer,13,,-6.683333,-42.983333
br,bem sera,Bem Será,31,,-8.923056,-48.106667
br,bem-te-vi,Bem-te-vi,04,,-5.417222,-63.041389
br,bem-te-vi,Bem-Te-Vi,18,,-24.2,-51.833333
br,bem-te-vi,Bem-Te-Vi,20,,-3.788889,-42.017778
br,bemtevi,Bemtevi,30,,-8.55,-35.583333
br,bene dias machado,Bené Dias Machado,15,,-20.4,-46.716667
br,benedicto leite,Benedicto Leite,13,,-7.216667,-44.566667
br,benedicto novo,Benedicto Novo,26,,-26.8,-49.416667
br,beneditinos,Beneditinos,20,5005,-5.45,-42.366667
br,benedito alves,Benedito Alves,26,,-26.665054,-49.518707
br,benedito,Benedito,04,,-6.35,-60.316667
br,benedito,Benedito,05,,-9.016667,-39.5
br,benedito,Benedito,15,,-14.583333,-44.216667
br,benedito dos santos,Benedito dos Santos,21,,-21.066667,-41.733333
br,benedito leite,Benedito Leite,13,,-7.216667,-44.566667
br,benedito medeiros,Benedito Medeiros,15,,-20.35,-46.816667
br,benedito novo,Benedito Novo,26,,-26.8,-49.416667
br,beneficente,Beneficente,04,,-7.133333,-60.483333
br,beneficiente,Beneficiente,04,,-7.133333,-60.483333
br,benevente,Benevente,08,,-20.8,-40.65
br,benevenuto,Benevenuto,25,,1.883333,-61.016667
br,benevides,Benevides,16,49813,-1.366667,-48.25
br,benevolencia,Benevolência,06,,-5.966667,-39.1
br,benfica,Benfica,01,,-8.316667,-73.283333
br,benfica,Benfica,04,,-6.765,-70.936389
br,benfica,Benfica,05,,-12.383333,-39.3
br,benfica,Benfica,15,,-16.783333,-41.083333
br,benfica,Benfica,16,,-1.3,-48.3
br,benfica,Benfica,21,,-22.891683,-43.24092
br,benfica,Benfica,22,,-4.972222,-37.068056
br,benfica,Benfica,23,,-29.9,-51.383333
br,benfica,Benfica,27,,-22.483333,-46.933333
br,bengala,Bengala,30,,-8.016667,-35.483333
br,bengalas,Bengalas,30,,-8.016667,-35.483333
br,benigno,Benigno,04,,-8,-60.85
br,benjamim,Benjamim,16,,-1.333333,-48.433333
br,benjamim,Benjamim,18,,-25.333333,-53.883333
br,benjamim,Benjamim,25,,-3.3,-61.016667
br,benjamim constant,Benjamim Constant,04,,-4.374444,-70.029722
br,benjamim de barros,Benjamim de Barros,29,,-17.566667,-51.916667
br,benjamim not,Benjamim Not,23,,-28.666667,-53.6
br,benjamim santana,Benjamim Santana,15,,-16.433333,-40.45
br,benjamin,Benjamin,16,,-1.333333,-48.433333
br,benjamin,Benjamin,18,,-25.333333,-53.883333
br,benjamin constant,Benjamin Constant,04,13998,-4.374444,-70.029722
br,benjamin constant,Benjamin Constant,15,,-21.95,-42.883333
br,benjamin constant,Benjamin Constant,16,,-1.066667,-46.6
br,benjamin constant,Benjamin Constant,23,,-27.733333,-52.983333
br,benjamin dos santos,Benjamin dos Santos,29,,-16.283333,-50.416667
br,benjamin novais,Benjamin Novais,23,,-28.666667,-53.6
br,benqueirao,Benqueirão,17,,-6.983333,-36.116667
br,benta,Benta,11,,-17.9,-54.583333
br,bentevi,Bentevi,30,,-8.55,-35.583333
br,bentinho,Bentinho,15,,-17.466667,-41.883333
br,bento alto,Bento Alto,26,,-28.666667,-49.566667
br,bento,Bento,05,,-13.3,-39.666667
br,bento,Bento,15,,-17.333333,-41.1
br,bento brasil,Bento Brasil,25,,2.816667,-60.616667
br,bento de abreu,Bento de Abreu,27,,-21.283333,-50.8
br,bento fernandes,Bento Fernandes,22,,-5.683333,-35.816667
br,bento gomes,Bento Gomes,11,,-18.066667,-54.533333
br,bento goncalves,Bento Gonçalves,23,90854,-29.166667,-51.516667
br,bento munhoz,Bento Munhoz,18,,-24.9,-53.133333
br,bento pereira,Bento Pereira,06,,-5.116667,-38.35
br,bentopolis,Bentópolis,18,,-22.933333,-51.716667
br,bento ribeiro,Bento Ribeiro,21,,-22.868014,-43.36207
br,bento rodrigues,Bento Rodrigues,15,,-20.233333,-43.416667
br,bento rogrigues,Bento Rogrigues,06,,-5.616667,-39.766667
br,bentos,Bentos,15,,-22.6324,-45.5363
br,bento simoes,Bento Simões,05,,-12.15,-38.733333
br,benvenuto,Benvenuto,16,,-2.583333,-51.016667
br,bequimao,Bequimão,13,5695,-2.433333,-44.783333
br,bergamasco,Bergamasco,27,,-22.416667,-47.3
br,berigui,Berigui,18,,-25.533333,-49.383333
br,berilo,Berilo,15,,-16.95,-42.483333
br,berimbau,Berimbau,04,,-5.590278,-60.806667
br,berimbau,Berimbau,06,,-5.416667,-40.466667
br,beriolino bitu,Beriolino Bitu,15,,-17.316667,-40.933333
br,beriquara,Beriquara,27,,-20.316667,-47.6
br,beritingas,Beritingas,05,,-11.6,-38.8
br,berlenga,Berlenga,20,,-7.266667,-40.583333
br,berlengas,Berlengas,20,,-6.4,-41.75
br,berlim,Berlim,04,,-3.438889,-63.416944
br,berlinque,Berlinque,05,,-13.1,-38.75
br,bernarda,Bernarda,06,,-7.2,-38.683333
br,bernarda,Bernarda,30,,-7.8,-38.166667
br,bernardes,Bernardes,29,,-16.5,-52.05
br,bernardino borges,Bernardino Borges,15,,-20.433333,-46.266667
br,bernardino de campos,Bernardino de Campos,27,9342,-23.033333,-49.483333
br,bernardo,Bernardo,02,,-9.083333,-37.466667
br,bernardo,Bernardo,05,,-13.133333,-39.016667
br,bernardo,Bernardo,13,,-4.616667,-44.916667
br,bernardo,Bernardo,16,,-5.283333,-55.866667
br,bernardo dias,Bernardo Dias,14,,-15.666667,-57.5
br,bernardo sayao,Bernardo Sayão,31,,-7.856944,-48.786389
br,bernardo vieira,Bernardo Vieira,30,,-7.933333,-38.55
br,berra bode,Berra Bode,06,,-3.9,-38.383333
br,berro,Berro,20,,-3.593333,-42.583056
br,berro de dentro,Berro de Dentro,20,,-3.583889,-42.570556
br,berta,Berta,11,,-17.9,-54.583333
br,bertanholi,Bertanholi,18,,-24.5,-49.516667
br,bertinho,Bertinho,16,,-2.783333,-50.4
br,bertioga,Bertioga,06,,-6.55,-38.85
br,bertioga,Bertioga,17,,-7.5,-36.283333
br,bertioga,Bertioga,27,,-23.85,-46.15
br,berto cirio,Berto Círio,23,,-29.8572,-51.2742
br,bertolinia,Bertolínia,20,,-7.633333,-43.95
br,bertolino pinheiro,Bertolino Pinheiro,15,,-17.3,-41.316667
br,bertopolis,Bertópolis,15,,-17.016667,-40.566667
br,beruri,Beruri,04,,-3.9,-61.366667
br,berury,Berury,04,,-3.9,-61.366667
br,besouro,Besouro,01,,-8.9825,-72.726389
br,besouro,Besouro,06,,-7.533333,-39.083333
br,besouro,Besouro,18,,-25.216667,-53.2
br,besta morta,Besta Morta,13,,-3.111389,-42.379722
br,betania,Betânia,05,,-13.083333,-42.866667
br,betania,Betânia,06,,-3.283333,-39.65
br,betania,Betânia,20,,-5.566667,-42.65
br,betania,Betânia,30,,-8.266667,-38.033333
br,betim,Betim,15,384038,-19.966667,-44.216667
br,betuli,Betuli,05,,-9.733333,-42.216667
br,bevenuto,Bevenuto,16,,-2.583333,-51.016667
br,beverine,Beverine,26,,-26.95,-53.216667
br,beverini,Beverini,26,,-26.95,-53.216667
br,beverni,Beverni,26,,-26.95,-53.216667
br,bevilacqua,Bevilacqua,23,,-29.1354,-51.0476
br,bexiga,Bexiga,13,,-3.7875,-42.7375
br,bexiga,Bexiga,23,,-29.966667,-52.683333
br,beya-flor,Beya-Flor,15,,-21.133333,-43.083333
br,bezerra,Bezerra,20,,-7.5,-41.516667
br,bezerras,Bezerras,17,,-6.816667,-37.85
br,bezerrinha,Bezerrinha,06,,-5.566667,-39.033333
br,bezerro manso,Bezerro Manso,20,,-5.233333,-42.516667
br,bezerro morto,Bezerro Morto,06,,-6.683333,-38.7
br,bezerro morto,Bezerro Morto,20,,-6.666667,-41.35
br,bezerros,Bezerros,20,,-3.753889,-42.515556
br,bezerros,Bezerros,30,51437,-8.233333,-35.75
br,bezouro,Bezouro,17,,-7.083333,-38.116667
br,biancardia,Biancárdia,08,,-19.416667,-40.333333
br,bias fortes,Bias Fortes,15,,-21.6,-43.766667
br,biaxao do coco,Biaxão do Côco,13,,-5.483333,-43.466667
br,biboca,Biboca,21,,-21.45,-41.633333
br,bica,Bica,06,,-5.316667,-38.033333
br,bica,Bica,13,,-3.955,-43.154444
br,bica de pedra,Bica de Pedra,27,,-22.233333,-48.683333
br,bicame,Bicame,27,,-23.090978,-47.865368
br,bicanca,Bicanca,08,,-20.216667,-40.216667
br,bicas,Bicas,15,12684,-21.716667,-43.066667
br,bicas do meio,Bicas do Meio,15,,-22.516667,-45.35
br,bicho de baixo,Bicho de Baixo,05,,-13.516667,-41.633333
br,bicho do meio,Bicho do Meio,05,,-13.5,-41.65
br,bicuda,Bicuda,05,,-13.95,-41.266667
br,bicudo,Bicudo,05,,-13.95,-41.266667
br,bicudo,Bicudo,27,,-21.216667,-51.05
br,bicudos,Bicudos,27,,-22.316667,-47.7
br,bicuiba,Bicuíba,13,,-3.4325,-42.598333
br,bicuiba,Bicuíba,15,,-20.083333,-42.366667
br,bicuida,Bicuída,13,,-3.287778,-42.333333
br,bifurcacao,Bifurcação,26,,-28.45,-48.783333
br,biguacu,Biguaçu,26,51993,-27.5,-48.666667
br,biguassu,Biguassú,26,,-27.5,-48.666667
br,biguazinho,Biguàzinho,27,,-24.283333,-47.533333
br,bilac,Bilac,27,,-21.4,-50.466667
br,bilheira,Bilheira,06,,-4.016667,-40.05
br,binga,Binga,04,,-7.7,-65.05
br,bioto guimaraes,Bioto Guimarães,15,,-17.716667,-41.583333
br,biquinha,Biquinha,21,,-22.223889,-43.686667
br,biquinhas,Biquinhas,15,,-18.783333,-45.5
br,biraba,Biraba,05,,-10.8,-42.833333
br,biriba,Biriba,05,,-12,-38.2
br,biribeira,Biribeira,05,,-12.633333,-38.2
br,biribiri,Biribiri,15,,-18.133333,-43.616667
br,birigui,Birigui,27,102279,-21.3,-50.316667
br,biriguy,Biriguy,27,,-21.3,-50.316667
br,biririca,Biririca,08,,-20.016667,-40.233333
br,biriricas,Biriricas,08,,-18.6,-40.1
br,biriricas de baixo,Biriricas de Baixo,08,,-20.35,-40.583333
br,biritiba-mirim,Biritiba-Mirim,27,26459,-23.583333,-46.033333
br,biritinga,Biritinga,05,,-11.6,-38.8
br,birra,Birra,31,,-11.666667,-47.066667
br,biscaia,Biscaia,18,,-25.066667,-49.9
br,bismarck,Bismarck,14,,-16.75,-57.05
br,bismarque,Bismarque,14,,-16.75,-57.05
br,bitaculo,Bitáculo,29,,-17.05,-49.783333
br,bitenambi,Bitenambi,05,,-13.583333,-39.5
br,bitencourt,Bitencourt,14,,-16.316667,-57.516667
br,bitencurt,Bitencurt,14,,-16.316667,-57.516667
br,bitumirim,Bitumirim,18,,-25.033333,-50.783333
br,bitupita,Bitupitá,06,,-2.895,-41.275833
br,bituri,Bituri,15,,-20.533333,-44.083333
br,bituruna,Bituruna,18,,-26.166667,-51.566667
br,bituva,Bituva,23,,-29.7818,-50.8024
br,bituva das campinas,Bituva das Campinas,18,,-25.5636,-50.4904
br,bituva do kuller,Bituva do Küller,18,,-25.539523,-50.483626
br,bituva dos lopes,Bituva dos Lopes,18,,-25.558158,-50.53029
br,bituva dos lucios,Bituva dos Lúcios,18,,-25.59849,-50.49416
br,bituva dos machados,Bituva dos Machados,18,,-25.60089,-50.44119
br,bituva dos saruva,Bituva dos Saruva,18,,-25.61541,-50.44665
br,bixopa,Bixopá,06,,-5,-38.2
br,bizarra,Bizarra,30,,-7.75,-35.466667
br,bizum,Bizum,20,,-7.783333,-41.633333
br,blau nunes,Blau Nunes,23,,-28.366667,-53.25
br,blauth,Blauth,23,,-29.233333,-51.45
br,blinha,Blinha,26,,-27.3,-49.366667
br,blumenau,Blumenau,26,293955,-26.9222,-49.0615
br,boa agua,Boa Água,06,,-4.833333,-38.616667
br,boa agua,Boa Água,17,,-6.4,-37.566667
br,boa agua,Boa Água,22,,-5.883333,-35.666667
br,boa agua dos leandros,Boa Água dos Leandros,22,,-6.183333,-37.883333
br,boa barreirinha,Boa Barreirinha,27,,-23.3,-47.166667
br,boa barrinha,Boa Barrinha,27,,-23.3,-47.166667
br,boa centro,Boa Centro,23,,-27.75,-54.5
br,boa chapeleira,Boa Chapeleira,27,,-23.233333,-47.083333
br,boacica,Boacica,22,,-5.283333,-35.566667
br,boacica,Boacica,27,,-24.9849,-48.0116
br,boacu,Boaçu,05,,-13.983333,-40.2
br,boa dadiva,Boa Dadiva,06,,-5.483333,-40.316667
br,boa eoperanca,Boa Eoperança,08,,-21.016667,-41.616667
br,boa espaernca,Bôa Espaernca,06,,-7.05,-38.783333
br,boa espera,Boa Espera,05,,-11.916667,-38.866667
br,boa esperanca,Boa Esperança,01,,-9.95,-66.683333
br,boa esperanca,Boa Esperança,02,,-9.316667,-37.95
br,boa esperanca,Boa Esperança,03,,2.833333,-51.05
br,boa esperanca,Boa Esperança,04,,-2.65,-67.483333
br,boa esperanca,Boa Esperança,05,,-10.616667,-43.5
br,boa esperanca,Boa Esperanca,06,,-4.066667,-39.266667
br,boa esperanca,Boa Esperança,08,9254,-18.55,-40.283333
br,boa esperanca,Boa Esperança,11,,-19.383333,-56.45
br,boa esperanca,Boa Esperança,13,,-1.683333,-44.966667
br,boa esperanca,Boa Esperança,15,33840,-21.083333,-45.566667
br,boa esperanca,Boa Esperança,16,,-1.25,-47.166667
br,boa esperanca,Boa Esperanca,17,,-6.233333,-37.483333
br,boa esperanca,Boa Esperança,18,,-24.383333,-49.566667
br,boa esperanca,Boa Esperança,20,,-10,-45.483333
br,boa esperanca,Boa Esperança,21,,-21.233333,-41.533333
br,boa esperanca,Boa Esperança,22,,-5.381111,-36.038611
br,boa esperanca,Boa Esperança,23,,-28.3,-53.65
br,boa esperanca,Boa Esperança,25,,3.35,-61.383333
br,boa esperanca,Boa Esperança,27,,-20.216667,-50.366667
br,boa esperanca,Boa Esperança,29,,-14.533333,-46.2
br,boa esperanca,Boa Esperança,30,,-8.4,-36.533333
br,boa esperanca,Boa Esperança,31,,-11.816667,-47.45
br,boa esperanca do mearim,Boa Esperança do Mearim,13,,-4.883333,-44.883333
br,boa esperanca do sul,Boa Esperança do Sul,27,10787,-21.983333,-48.383333
br,boa esperaranca,Boa Esperarança,20,,-9.25,-44.083333
br,boa estrela,Boa Estrêla,31,,-7.766667,-48.566667
br,boa familia,Bôa Familia,08,,-19.8,-40.85
br,boa familia,Boa Família,15,,-21.216667,-42.516667
br,boa fe,Boa Fé,04,,-5.646944,-64.205556
br,boa fe,Boa Fé,05,,-12.333333,-39.4
br,boa fe,Boa Fé,06,,-5.55,-40.516667
br,boa fe,Boa Fé,16,,-4.6,-56.2
br,boa fe,Boa Fé,17,,-6.883333,-38.7
br,boa fe,Boa Fé,20,,-6.066667,-41.916667
br,boa fe,Boa Fé,21,,-22.366667,-41.883333
br,boa fortuna,Boa Fortuna,21,,-21.05,-41.733333
br,boa guaxinduba,Boa Guaxinduba,27,,-23.283333,-47.016667
br,boa hora,Boa Hora,04,,-7.466667,-70.166667
br,boa hora,Boa Hora,05,,-10.966667,-38.533333
br,boa hora,Boa Hora,06,,-5.066667,-39.816667
br,boa hora,Boa Hora,13,,-3.2525,-42.938056
br,boa hora,Boa Hora,24,,-11.683333,-65.083333
br,boa ideia,Boa Idéia,30,,-8.15,-37.566667
br,boa lembranca,Boa Lembrança,04,,-4.475556,-59.866389
br,boa lembranca,Boa Lembrança,05,,-9.616667,-38.083333
br,boa memoria,Boa Memória,11,,-21.583333,-55.533333
br,boa morada,Boa Morada,13,,-7.766667,-45.5
br,boa morte,Boa Morte,15,,-20.433333,-43.983333
br,boa morte,Boa Morte,16,,-1.4,-48.666667
br,boa nova,Boa Nova,04,,-5.604444,-61.140833
br,boa nova,Boa Nova,05,,-12.25,-40.55
br,boa nova,Boa Nova,14,,-13.416667,-57.916667
br,boa nova,Boa Nova,20,,-6.066667,-42.716667
br,boapaba,Boapaba,08,,-19.65,-40.616667
br,boa paz,Boa Paz,05,,-12.15,-38.583333
br,boa paz,Boa Paz,22,,-5.291111,-36.502222
br,boa razao,Boa Razão,13,,-3.610556,-43.307778
br,boa-saude,Boa-Saúde,06,,-7.116667,-39.766667
br,boa-saude,Boa-Saúde,22,,-6.15,-35.583333
br,boa sentenca,Boa Sentença,05,,-13.583333,-41.716667
br,boa sombra,Boa Sombra,02,,-9.15,-37.733333
br,boa sorte,Boa Sorte,02,,-9.666667,-37.466667
br,boa sorte,Boa Sorte,04,,-2.75,-65.85
br,boa sorte,Boa Sorte,05,,-12.05,-40.183333
br,boa sorte,Boa Sorte,06,,-5.116667,-39.316667
br,boa sorte,Boa Sorte,08,,-20.2,-41.066667
br,boa sorte,Boa Sorte,11,,-19.35,-57.083333
br,boa sorte,Boa Sorte,13,,-8.55,-46.5
br,boa sorte,Boa Sorte,14,,-15.077778,-59.874722
br,boa sorte,Boa Sorte,15,,-15.833333,-40.983333
br,boa sorte,Boa Sorte,17,,-6.45,-36.35
br,boa sorte,Boa Sorte,21,,-21.85,-42.25
br,boa sorte,Boa Sorte,22,,-5.216667,-37.716667
br,boa sorte,Boa Sorte,30,,-8.566667,-36.9
br,boa sorte,Boa Sorte,31,,-9.630833,-46.533889
br,boa terra,Boa Terra,05,,-12.016667,-41
br,boa uniao,Boa União,04,,-4.393333,-59.991111
br,boa-uniao,Boa-União,04,,-7.55,-66.266667
br,boa uniao,Boa União,04,,-7.8,-68.516667
br,boa uniao,Boa União,05,,-12.183333,-38.5
br,boa uniao,Boa União,06,,-4.5,-38.75
br,boa uniao,Boa União,16,,-2.066667,-49.15
br,boa uniao,Boa União,30,,-8.466667,-40.433333
br,boa vargem grande,Boa Vargem Grande,27,,-23.35,-47.066667
br,boa ventura,Boa Ventura,06,,-4.9,-39.666667
br,boa ventura,Boa Ventura,15,,-21.616667,-43.35
br,boa ventura,Boa Ventura,17,,-7.416667,-38.216667
br,boa ventura,Boa Ventura,18,,-24.95,-51.616667
br,boa ventura,Boa Ventura,20,,-5.516667,-41.666667
br,boaventura,Boaventura,21,,-21.316667,-41.633333
br,boa ventura,Boa Ventura,29,,-16.45,-51.783333
br,boa ventura,Boa Ventura,30,,-7.783333,-37.533333
br,boa vereda,Boa Vereda,15,,-22.45,-46.233333
br,boa viagem,Boa Viagem,05,,-12.928333,-38.511667
br,boa viagem,Boa Viagem,06,24671,-5.116667,-39.733333
br,boa viagem,Boa Viagem,30,,-8.116667,-34.883333
br,boa vida,Boa Vida,15,,-19.683333,-41.6
br,boa vida,Boa Vida,21,,-22.3,-42.783333
br,boa vinha,Boa Vinha,27,,-22.6,-47.75
br,boa vista,Boa Vista,01,,-10.3,-67.25
br,boa vista,Boa Vista,02,,-9.366667,-36.066667
br,boa vista,Boa Vista,03,,-.35,-52.316667
br,boa vista,Boa Vista,04,,.25,-64.016667
br,boa vista,Boa Vista,05,,-10.066667,-39.7
br,boa-vista,Boa-Vista,05,,-12.666667,-40.616667
br,boa vista,Boa Vista,05,,-12.733333,-41.833333
br,boa vista,Boa Vista,06,,-3.733333,-40.233333
br,boa vista,Boa Vista,07,,-15.833333,-47.55
br,boa vista,Boa Vista,08,,-18.65,-39.75
br,boa vista,Boa Vista,11,,-17.85,-54.216667
br,boa vista,Boa Vista,13,,-3.216667,-42.566667
br,boa vista,Boa Vista,14,,-15.116667,-56.4
br,boa vista,Boa Vista,15,,-15.766667,-44.716667
br,boa vista,Boa Vista,16,,-1.066667,-50.333333
br,boa vista,Boa Vista,17,,-6.133333,-37.483333
br,boa vista,Boa Vista,18,,-23.983756,-50.519436
br,boa vista,Boa Vista,20,,-10.483333,-45.1
br,boa vista,Boa Vista,21,,-21.366667,-41.483333
br,boa vista,Boa Vista,22,,-5.064167,-37.470278
br,boa vista,Boa Vista,23,,-28.95,-53.15
br,boa vista,Boa Vista,24,,-10.733333,-62.133333
br,boa vista,Boa Vista,25,235159,2.816667,-60.666667
br,boa vista,Boa Vista,26,,-26.283333,-48.833333
br,boa vista,Boa Vista,27,,-20.166667,-47.7
br,boa vista,Boa Vista,28,,-10.433333,-37.133333
br,boa vista,Boa Vista,30,,-7.5,-37.533333
br,boa vista,Boa Vista,31,,-11.45,-47.883333
br,boa vista da malhada,Boa Vista da Malhada,30,,-8.066667,-36.083333
br,boa vista de baixo,Boa Vista de Baixo,27,,-23.686812,-47.46012
br,boa vista de colonia,Boa Vista de Colônia,30,,-7.666667,-37.7
br,boa vista de palmas,Boa Vista de Palmas,15,,-16.466667,-45.766667
br,boa vista de santo antonio,Boa Vista de Santo Antônio,05,,-10.516667,-42.55
br,boa vista de sao antonio,Boa Vista de São Antonio,05,,-10.516667,-42.55
br,boa vista de trairas e aracati,Boa Vista de Traíras e Aracati,29,,-16.183333,-49.083333
br,boa vista do bacuri,Boa Vista do Bacuri,04,,-4.833333,-62.566667
br,boa vista do burica,Boa Vista do Burica,23,,-27.633333,-54.166667
br,boa vista do capanema,Bóa Vista do Capanema,18,,-26.134444,-53.575833
br,boa vista do erechim,Bôa Vista do Erechim,23,,-27.633333,-52.283333
br,boa vista do helvecio,Boa Vista do Helvécio,05,,-13.716667,-40.616667
br,boa vista do jauato,Boa Vista do Jauató,04,,-2.766667,-65.283333
br,boa vista do lagamar,Boa Vista do Lagamar,05,,-11.833333,-43.316667
br,boa vista do marzagao,Bôa Vista do Marzagão,29,,-17.983333,-48.65
br,boa vista do matao,Boa Vista do Matão,29,,-16.216667,-49.2
br,boa vista do pichaui,Boa Vista do Pichauí,05,,-12.016667,-43.35
br,boa vista do pindare,Boa Vista do Pindaré,13,,-3.4,-44.983333
br,boa vista do pixaui,Boa Vista do Pixauí,05,,-12.016667,-43.35
br,boa vista do rio branco,Boa Vista do Rio Branco,25,,2.816667,-60.666667
br,boa vista dos cariocas,Boa Vista dos Cariocas,20,,-3.65,-42.15
br,boa vista dos inacios,Boa Vista dos Inácios,30,,-8.716667,-36.633333
br,boa vista dos nunes,Boa Vista dos Nunes,17,,-7.666667,-37.133333
br,boa vista dos oleos,Boa Vista dos Óleos,15,,-21.566667,-46.033333
br,boa vista do tocantins,Boa Vista Do Tocantins,31,,-6.324444,-47.277778
br,boa vista do tremendal,Boa Vista do Tremendal,15,,-15.15,-42.883333
br,boa vista esteires,Bôa Vista Esteires,05,,-10.516667,-42.55
br,boa vista lagamar,Bôa Vista Lagamar,05,,-11.833333,-43.316667
br,boa vista-lugarejo,Boa Vista-Lugarejo,27,,-24.583333,-47.633333
br,boa vitoria,Boa Vitória,15,,-22.5144,-45.6156
br,bobo,Bobo,15,,-20.783333,-45.133333
br,boca aripuana,Boca Aripuanã,04,,-5.120278,-60.378889
br,boca da barra ii,Boca da Barra II,13,,-3.444722,-42.394722
br,boca da caatinga,Boca da Caatinga,20,,-5.716667,-41.9
br,boca da catinga,Bôca da Catinga,20,,-10.016667,-44.433333
br,boca da estrada,Bôca da Estrada,25,,-2.116667,-61
br,boca da lagoa sao joao,Boca da Lagoa São Joaõ,04,,-1.917222,-61.405556
br,boca da mata,Bôca da Mata,02,17834,-9.6,-36.25
br,boca da mata,Bôca da Mata,05,,-12.766667,-39.466667
br,boca da mata,Boca da Mata,13,,-3.720833,-42.731111
br,boca-da-mata,Boca-da-Mata,17,,-7.516667,-34.9
br,boca da mata,Boca da Mata,20,,-3.777778,-42.492778
br,boca da mata,Bôca da Mata,25,,4.316667,-61.15
br,boca da mata de jose carneiro,Bôca da Mata de José Carneiro,02,,-9.566667,-36.366667
br,boca da peca,Bôca da Peça,30,,-7.983333,-37.7
br,boca da picada,Bôca da Picada,05,,-10.783333,-43.566667
br,boca da picada,Boca da Picada,13,,-3.41,-42.6
br,boca da serra,Boca da Serra,23,,-29.0879,-51.0308
br,boca da vargem,Boca da Várgem,22,,-5.320833,-36.725
br,boca da varzea,Bôca da Várzea,05,,-12.033333,-38.733333
br,boca de pauini,Boca de Pauiní,04,,-7.783333,-67.083333
br,boca de tauiry,Boca de Tauiry,16,,-5.133333,-49.3
br,boca de teuini,Boca de Teuiní,04,,-7.95,-67.233333
br,boca do acara,Bôca do Acará,04,,-6.229444,-62.254444
br,boca do acre,Bôca do Acre,04,14427,-8.75,-67.383333
br,boca do arara,Boca do Arará,01,,-8.945,-72.761667
br,boca do caju,Boca do Caju,04,,-3.369722,-62.591389
br,boca do campana,Bôca do Campana,04,,-5.866111,-61.693611
br,boca do campo,Bôca do Campo,05,,-12.25,-41.366667
br,boca do capana,Bôca do Capanã,04,,-5.866111,-61.693611
br,boca do carapanatuba,Boca do Carapanatuba,04,,-6.754167,-62.591667
br,boca do carrasco,Bôca do Carrasco,05,,-13.5,-40.833333
br,boca do centro,Bôca do Centro,20,,-3.75,-42.4
br,boca do coco,Boca do Coco,20,,-5.283333,-42.216667
br,boca do corrego,Bôca do Córrego,05,,-15.866667,-39.116667
br,boca do corrego,Boca do Côrrego,06,,-3.116667,-39.666667
br,boca do corvo,Boca do Corvo,06,,-4.666667,-37.816667
br,boca do curara,Boca do Curara,04,,-5.756111,-60.94
br,boca do curuquele,Bôca do Curuquelê,04,,-8.316667,-65.683333
br,boca do curuquete,Bôca do Curuquetê,04,,-8.316667,-65.683333
br,boca do fumaco,Boca do Fumaco,04,,-3.75,-59.05
br,boca do jaminaua,Boca do Jaminaua,01,,-9.133333,-70.833333
br,boca do jari,Bôca do Jari,16,,-1.116667,-51.966667
br,boca do mato,Bôca do Mato,05,,-12.766667,-39.466667
br,boca do mato,Boca do Mato,21,,-22.908466,-43.292974
br,boca do moaca,Boca do Moaca,04,,-7.683333,-68.283333
br,boca do moaco,Bôca do Moaco,04,,-7.683333,-68.283333
br,boca do monte,Bôca do Monte,23,,-29.633333,-53.916667
br,boca do munguba,Boca do Munguba,04,,-3.525278,-62.895556
br,boca do mutum,Bôca do Mutum,04,,-4.433333,-68.05
br,boca do pajeu,Bôca do Pajeú,30,,-8.916667,-38.7
br,boca do punus,Boca do Punus,04,,-3.683333,-61.45
br,boca do purus,Boca do Purus,04,,-3.683333,-61.45
br,boca do rapirra,Boca do Rapirra,01,,-10.333333,-67.183333
br,boca do rio,Bôca do Rio,05,,-13.033333,-38.65
br,boca do saco,Bôca do Saco,05,,-10.616667,-43.5
br,boca do saco,Boca do Saco,20,,-7.166667,-41.65
br,boca do taquari,Boca do Taquari,01,,-7.933333,-71.466667
br,boca do tauari,Boca do Tauari,01,,-7.933333,-71.466667
br,boca do tejo,Boca do Tejo,01,,-8.981389,-72.717778
br,boca do tucuma,Boca do Tucumã,04,,-5.092222,-59.365
br,bocaina,Bocaina,11,,-20.816667,-57.366667
br,bocaina,Bocaina,13,,-6.75,-44.066667
br,bocaina,Bocaina,14,,-15.266667,-56.75
br,bocaina,Bocaina,15,,-17.266667,-46.15
br,bocaina,Bocaina,18,,-25,-50.266667
br,bocaina,Bocaina,20,,-6.916667,-41.316667
br,bocaina,Bocâina,26,,-27.75,-49.95
br,bocaina,Bocaina,27,,-20.466667,-47.733333
br,bocaina de minas,Bocaina de Minas,15,,-22.166667,-44.4
br,bocaina do sul,Bocaina do Sul,26,,-27.75,-49.95
br,bocaine,Bocaine,20,,-6.716667,-42.666667
br,bocaiuva,Bocaiúva,11,,-21.983333,-54.533333
br,bocaiuva,Bocaiuva,14,,-14.65,-57.216667
br,bocaiuva,Bocaiúva,15,32115,-17.116667,-43.816667
br,bocaiuva,Bocaiúva,18,,-25.183333,-49.133333
br,bocaiuva,Bocaiúva,23,,-28.766667,-51.933333
br,bocaiuva,Bocaiuva,27,,-22.516667,-48.683333
br,bocaiuva do sul,Bocaiúva do Sul,18,,-25.183333,-49.133333
br,bocaiuval,Bocaiuval,14,,-14.65,-57.216667
br,bocaja,Bocajá,11,,-22.75,-55.216667
br,boca torta,Boca Torta,06,,-7.033333,-38.7
br,bocayuva,Bocayuva,15,,-17.116667,-43.816667
br,bocayuva,Bocayuva,18,,-25.183333,-49.133333
br,bocayuva,Bocayuva,27,,-22.516667,-48.683333
br,bocayuval,Bocayuval,14,,-14.65,-57.216667
br,bocca da maffa,Bocca da Maffa,02,,-9.683333,-36.183333
br,bocca de matta,Bocca de Matta,17,,-7.516667,-34.9
br,bocca do acre,Bocca do Acre,04,,-8.75,-67.383333
br,bocca do curua,Bocca do Curuá,16,,-5.4,-54.466667
br,bocca do fumaco,Bocca do Fumaco,04,,-3.75,-59.05
br,bocca do jaminaua,Bocca do Jaminaua,01,,-9.133333,-70.833333
br,bocca do moaco,Bocca do Moaco,04,,-7.683333,-68.283333
br,bocca do monte,Bocca do Monte,23,,-29.633333,-53.916667
br,bocca do mutum,Bocca do Mutum,04,,-4.433333,-68.05
br,bocca do pauhiny,Bocca do Pauhiny,04,,-7.783333,-67.083333
br,bocca do rapirra,Bocca do Rapirra,01,,-10.333333,-67.183333
br,bocca do tauary,Bocca do Tauary,01,,-7.933333,-71.466667
br,bocca do tauiry,Bocca do Tauiry,16,,-5.133333,-49.3
br,bocianowski,Bocianowski,26,,-28.633333,-49.316667
br,bocianwski,Bocianwski,26,,-28.633333,-49.316667
br,bocoroca,Boçoroca,05,,-9.466667,-41.1
br,bocoroca,Boçoroca,23,,-28.733333,-54.916667
br,bocuituba,Bocuituba,05,,-12.666667,-42.816667
br,bode,Bode,11,,-17.633333,-55.033333
br,bodega,Bodega,11,,-20.05,-55.783333
br,bodega,Bodega,15,,-20.316667,-43.1
br,bodo,Bodó,22,,-5.9875,-36.411944
br,bodoco,Bodocó,30,11329,-7.783333,-39.916667
br,bodocongo,Bodocongó,17,,-7.516667,-35.983333
br,bodote,Bodote,18,,-25.183333,-53.083333
br,bofete,Bofete,27,,-23.116667,-48.25
br,boia,Bóia,21,,-21.6,-41.933333
br,boiacanga,Boiacanga,04,,-.8625,-64.823333
br,boiacu,Boiaçu,25,,-.439722,-61.782222
br,boiada,Boiada,06,,-6.266667,-40.216667
br,boiada,Boiada,27,,-21.283333,-47.1
br,boiadeiro,Boiadeiro,27,,-22.603889,-47.495
br,boianga,Boiangá,21,,-21.633333,-41.433333
br,boiao,Boião,13,,-3.268611,-43.494167
br,boiaruca ponta,Boiaruca Ponta,04,,.066667,-67.783333
br,boi,Boi,06,,-4.033333,-40.083333
br,boi,Boi,13,,-3.028056,-43.034722
br,boi,Boi,20,,-3.546389,-42.346944
br,boi brave,Boi Brave,17,,-7.7,-36.25
br,boi de carro,Boi de Carro,14,,-16.233333,-56.583333
br,boi jagua,Boi Jagua,11,,-23.2,-54.85
br,boim,Boim,16,,-3.083333,-55.3
br,boi morto,Boi Morto,14,,-15.583333,-57.45
br,boi morto,Boi Morto,17,,-6.883333,-38.6
br,boi morto,Boi Morto,20,,-3.716111,-42.398611
br,boi pombo,Boi Pombo,17,,-7.85,-36.483333
br,boi raposo,Boi Rapôso,17,,-7.066667,-37.4
br,bois,Bois,06,,-3.55,-39.683333
br,bois,Bois,15,,-18.65,-49.283333
br,bois,Bois,20,,-9.433333,-44.766667
br,bois buritis,Bois Buritis,05,,-11.3,-42.816667
br,boiteuxburgo,Boiteuxburgo,26,,-27.416667,-49.216667
br,boituva,Boituva,27,48756,-23.283333,-47.683333
br,boiuna,Boiúna,21,,-22.905389,-43.412055
br,boi velho,Boi Velho,17,,-7.616667,-37.166667
br,boi velho,Boi Velho,22,,-5.133889,-36.465
br,bojuru,Bojuru,23,,-31.633333,-51.433333
br,bola,Bola,02,,-9.433333,-36.8
br,bola,Bola,06,,-4.5,-38.466667
br,bola,Bola,13,,-5.083333,-43.233333
br,bolacha,Bolacha,23,,-32.166667,-52.2
br,bolandeira,Bolandeira,05,,-12.8,-38.383333
br,bolandeira,Bolandeira,06,,-7.15,-40.283333
br,bolandeira,Bolandeira,17,,-6.35,-38.55
br,bolao,Bolão,17,,-7.616667,-36.35
br,bolaxa,Bolaxa,23,,-32.166667,-52.2
br,boliche,Boliche,11,,-20.266667,-54.5
br,boliche seco,Boliche Sêco,11,,-21.3,-55.95
br,bolicho,Bolicho,11,,-22.933333,-55.466667
br,bolicho seco,Bolicho Seco,11,,-21.3,-55.95
br,bolivia,Bolívia,04,,-7.366667,-58.033333
br,bolivia,Bolivia,06,,-4.666667,-40.033333
br,bolivia,Bolívia,20,,-3.982778,-42.039444
br,bom-assunto,Bom-Assunto,20,,-3.98,-42.2
br,bomba,Bomba,29,,-14.2,-46.3
br,bombas,Bombas,26,,-27.142222,-48.509444
br,bombinhas,Bombinhas,26,,-27.151111,-48.507778
br,bombonagem,Bombonagem,06,,-5.366667,-38.566667
br,bom caminho,Bom Caminho,05,,-12.433333,-40.35
br,bom centro,Bom Centro,23,,-27.75,-54.5
br,bom comercio,Bom Comércio,24,,-9.75,-65.9
br,bom conselho,Bom Conselho,02,,-9.7,-37.433333
br,bom conselho,Bom Conselho,05,,-10.6,-38.383333
br,bom conselho,Bom Conselho,30,27801,-9.166667,-36.683333
br,bom descanso,Bom Descanso,22,,-5.305833,-37.143333
br,bom desejo,Bom Desejo,05,,-12.616667,-41.666667
br,bom despacho,Bom Despacho,05,,-12.116667,-39.1
br,bom despacho,Bom Despacho,15,40304,-19.716667,-45.25
br,bom destino,Bom Destino,01,,-9.697778,-67.524167
br,bom destino,Bom Destino,03,,-.433333,-52.45
br,bom destino,Bom Destino,08,,-20.933333,-41.416667
br,bom destino,Bom Destino,11,,-22.966667,-55.283333
br,bom destino,Bom Destino,16,,-6.1,-54.116667
br,bom destino,Bom Destino,22,,-5.065278,-37.425833
br,bom destino,Bom Destino,30,,-9.1,-36.683333
br,bom dia,Bom Dia,11,,-23.383333,-55.133333
br,bom distracto,Bom Distracto,01,,-10.116667,-67.366667
br,bom distrato,Bom Distrato,01,,-10.116667,-67.366667
br,bomfim,Bomfim,05,,-10.45,-40.183333
br,bom fim,Bom Fim,05,,-12.233333,-39.216667
br,bomfim,Bomfim,05,,-9.166667,-39.066667
br,bom fim,Bom Fim,11,,-20.133333,-54.483333
br,bomfim,Bomfim,13,,-3.305278,-43.135278
br,bomfim,Bomfim,15,,-20.333333,-44.25
br,bom fim,Bom Fim,20,,-9.15,-42.85
br,bom fim,Bom Fim,21,,-22.516667,-43.5
br,bom fim,Bom Fim,27,,-21.266667,-47.816667
br,bomfim,Bomfim,29,,-16.666667,-48.6
br,bom fim,Bom Fim,31,,-11.853611,-47.534444
br,bomfim da feira,Bomfim da Feira,05,,-12.233333,-39.216667
br,bomfim paulista,Bomfim Paulista,27,,-21.266667,-47.816667
br,bom future,Bom Future,04,,-6.25,-62.216667
br,bom futuro,Bom Futuro,01,,-7.8,-72.066667
br,bom futuro,Bom Futuro,04,,-1.8,-65.65
br,bom futuro,Bom Futuro,05,,-16.416667,-39.283333
br,bom futuro,Bom Futuro,14,,-15.133333,-59.916667
br,bom futuro,Bom Futuro,16,,-6.416667,-54.783333
br,bom futuro,Bom Futuro,20,,-5.183333,-41.95
br,bom futuro,Bom Futuro,22,,-5.944722,-37.432778
br,bom futuro,Bom Futuro,24,,-9.583333,-64.983333
br,bom futuro,Bom Futuro,25,,3.533333,-61.1
br,bom garis,Bom Garis,05,,-11.266667,-38.95
br,bom gosto,Bom Gôsto,05,,-12.166667,-41.133333
br,bom gosto,Bom Gôsto,13,,-2.85,-43.95
br,bom gosto,Bom Gosto,20,,-5.016667,-42.583333
br,bom intento,Bom Intento,03,,.15,-52.05
br,bom intento,Bom Intento,04,,-5.638056,-63.776389
br,bom jardim,Bom Jardim,04,,-3.195833,-62.811111
br,bom jardim,Bom Jardim,05,,-10.266667,-43.5
br,bom jardim,Bom Jardim,06,,-4.566667,-38.633333
br,bom jardim,Bom Jardim,08,,-21.066667,-41.6
br,bom jardim,Bom Jardim,11,,-19.9,-54.2
br,bom jardim,Bom Jardim,13,12343,-7.016667,-46.483333
br,bom jardim,Bom Jardim,14,,-15.783333,-55.466667
br,bom jardim,Bom Jardim,15,,-15.916667,-40.55
br,bom jardim,Bom Jardim,16,,-1.116667,-50.95
br,bom jardim,Bom Jardim,17,,-6.933333,-38.716667
br,bom jardim,Bom Jardim,18,,-25.033333,-50.783333
br,bom jardim,Bom Jardim,20,,-3.528056,-42.428056
br,bom jardim,Bom Jardim,21,11716,-22.15,-42.416667
br,bom jardim,Bom Jardim,22,,-6.283333,-35.233333
br,bom jardim,Bom Jardim,23,,-29.6,-51.166667
br,bom jardim,Bom Jardim,24,,-10.333333,-62.333333
br,bom jardim,Bom Jardim,25,,3.433333,-61.25
br,bom jardim,Bom Jardim,26,,-28.333333,-49.633333
br,bom jardim,Bom Jardim,27,,-24.183333,-47.616667
br,bom jardim,Bom Jardim,29,,-16.283333,-52.116667
br,bom jardim,Bom Jardim,30,13523,-7.783333,-35.583333
br,bom jardim,Bom Jardim,31,,-12.441389,-47.7375
br,bom jardim da serra,Bom Jardim da Serra,26,,-28.333333,-49.633333
br,bom jardim de cima,Bom Jardim de Cima,15,,-20.583333,-42.883333
br,bom jardim de fora,Bom Jardim de Fora,05,,-12.266667,-43.3
br,bom jardim de goias,Bom Jardim de Goiás,29,,-16.283333,-52.116667
br,bom jardim de minas,Bom Jardim de Minas,15,,-21.95,-44.183333
br,bom jardin,Bom Jardin,15,,-21.95,-44.183333
br,bom jardin,Bom Jardin,18,,-25.033333,-50.783333
br,bom jesus,Bom Jesus,01,,-7.766667,-73.35
br,bom jesus,Bom Jesus,02,,-9.216667,-37.466667
br,bom jesus,Bom Jesus,04,,-2.883333,-68.483333
br,bom jesus,Bom Jesus,05,,-11.516667,-38.166667
br,bom jesus,Bom Jesus,06,,-3.45,-40.016667
br,bom jesus,Bom Jesus,08,,-19.833333,-41.116667
br,bom jesus,Bom Jesus,13,,-3.283611,-43.136111
br,bom jesus,Bom Jesus,15,,-18.683333,-42.633333
br,bom jesus,Bom Jesus,16,,-1.066667,-50.45
br,bom jesus,Bom Jesus,17,,-6.183333,-37.533333
br,bom jesus,Bom Jesus,18,,-26.0947,-49.7693
br,bom jesus,Bom Jesus,20,12666,-9.066667,-44.366667
br,bom jesus,Bom Jesus,22,,-5.235556,-36.191389
br,bom jesus,Bom Jesus,23,,-28.7,-50.4
br,bom jesus,Bom Jesus,24,,-10.330556,-62.241667
br,bom jesus,Bom Jesus,26,,-26.75,-52.35
br,bom jesus,Bom Jesus,29,,-18.2,-49.733333
br,bom jesus,Bom Jesus,30,,-7.6,-37.316667
br,bom jesus,Bom Jesus,31,,-11.85,-47.183333
br,bom jesus da cachoeira alegre,Bom Jesus da Cachoeira Alegre,15,,-21.266667,-42.4
br,bom jesus da cachoeira,Bom Jesus da Cachoeira,15,,-21.266667,-42.4
br,bom jesus da floresta,Bom Jesus da Floresta,15,,-18.833333,-41.25
br,bom jesus da lapa,Bom Jesus da Lapa,05,40693,-13.25,-43.416667
br,bom jesus da madeira,Bom Jesus da Madeira,15,,-20.716667,-42.416667
br,bom jesus da penha,Bom Jesus da Penha,15,,-21.016667,-46.533333
br,bom jesus da ribeira,Bom Jesus da Ribeira,27,,-24.716667,-47.55
br,bom jesus da serra,Bom Jesus da Serra,05,,-14.383333,-40.516667
br,bom jesus das pedras,Bom Jesus das Pedras,15,,-17.6,-44.983333
br,bom jesus da terra preta,Bom Jesus da Terra Preta,27,,-23.25,-46.6
br,bom jesus da vista alegre,Bom Jesus da Vista Alegre,15,,-19.066667,-41.75
br,bom jesus de itabapoana,Bom Jesus de Itabapoana,21,,-21.133333,-41.683333
br,bom jesus de itaueira,Bom Jesus de Itaueira,20,,-7.083333,-43.016667
br,bom jesus de lufa,Bom Jesus de Lufa,15,,-17.25,-41.916667
br,bom jesus de mantenas,Bom Jesus de Mantenas,15,,-18.666667,-41.4
br,bom jesus de quixelo,Bom Jesus de Quixelô,06,,-6.266667,-39.2
br,bom jesus de triunfo,Bom Jesus de Triunfo,23,,-29.933333,-51.716667
br,bom jesus do amparo,Bom Jesus do Amparo,15,,-19.7,-43.466667
br,bom jesus do bagre,Bom Jesus do Bagre,15,,-19.3,-42.5
br,bom jesus do barracao,Bom Jesus do Barracão,18,,-26.193333,-53.598611
br,bom jesus do carangola,Bom Jesus do Carangola,15,,-20.6,-42.283333
br,bom jesus do corrego,Bom Jesus do Córrego,15,,-22.633333,-46.033333
br,bom jesus do galho,Bom Jesus do Galho,15,12476,-19.833333,-42.316667
br,bom jesus do gurgueia,Bom Jesus do Gurgueia,20,,-9.066667,-44.366667
br,bom jesus do iguape,Bom Jesus do Iguapé,18,,-23.722138,-50.54571
br,bom jesus do itabapoana,Bom Jesus do Itabapoana,21,30822,-21.133333,-41.683333
br,bom jesus do itaveira,Bom Jesus do Itaveira,20,,-7.083333,-43.016667
br,bom jesus do lufa,Bom Jesús do Lufa,15,,-17.25,-41.916667
br,bom jesus do madeira,Bom Jesus do Madeira,15,,-20.716667,-42.416667
br,bom jesus do manhuacu,Bom Jesus do Manhuaçu,15,,-20.283333,-42.166667
br,bom jesus do munao novo,Bom Jesus do Munao Novo,15,,-19.75,-41.333333
br,bom jesus do mundo novo,Bom Jesus do Mundo Novo,15,,-19.75,-41.333333
br,bom jesus do norte,Bom Jesus do Norte,08,,-21.116667,-41.666667
br,bom jesus do pontal,Bom Jesus do Pontal,15,,-16.766667,-42.033333
br,bom jesus do quixelo,Bom Jesus do Quixelo,06,,-6.266667,-39.2
br,bom jesus do rio negro,Bom Jesus do Rio Negro,15,,-17.25,-40.933333
br,bom jesus dos meiras,Bom Jesús dos Meiras,05,,-14.216667,-41.666667
br,bom jesus dos perdoes,Bom Jesus dos Perdões,27,12422,-23.133333,-46.466667
br,bom jesus dos pobres,Bom Jesus dos Pobres,05,,-12.8,-38.783333
br,bom jesus do tabuleiro,Bom Jesus do Tabuleiro,15,,-18.25,-42.6
br,bom jesus do tocantins,Bom Jesus do Tocantins,31,,-8.913333,-48.011667
br,bom jesus do triunfo,Bom Jesus do Triunfo,23,,-29.933333,-51.716667
br,bom jesus i,Bom Jesus I,13,,-3.9525,-42.970278
br,bom jesus ii,Bom Jesus II,13,,-3.963611,-42.971667
br,bom jesus velho,Bom Jesus Velho,13,,-3.845556,-43.241111
br,bom lugar,Bom Lugar,04,,-2.7,-67.533333
br,bom lugar,Bom Lugar,13,,-4.333333,-44.966667
br,bom lugar,Bom Lugar,16,,-1.566667,-52.866667
br,bom lugar,Bom Lugar,20,,-3.646111,-42.383333
br,bom lugar,Bom Lugar,21,,-21.333333,-41.083333
br,bom lugar,Bom Lugar,22,,-5.133333,-37.083333
br,bom lugar,Bom Lugar,24,,-10.616667,-62.433333
br,bom negocio,Bom Negócio,22,,-5.656111,-36.754722
br,bom nome,Bom Nome,02,,-9.35,-37.966667
br,bom nome,Bom Nome,06,,-6.15,-39.4
br,bom nome,Bom Nome,17,,-7.333333,-36.983333
br,bom nome,Bom Nome,20,,-7.533333,-41.133333
br,bom nome,Bom Nome,30,,-7.983333,-38.616667
br,bom passar,Bom Passar,20,,-5.116667,-42.4
br,bom pastor,Bom Pastor,15,,-19.25,-41.05
br,bom pastor,Bom Pastor,30,,-8.616667,-36.783333
br,bompim da mata,Bompim da Mata,05,,-12.55,-38.3
br,bom principio,Bom Principio,06,,-3.216667,-40.066667
br,bom principio,Bom Princípio,13,,-3.250556,-43.118611
br,bom principio,Bom Princípio,20,,-3.908611,-42.536944
br,bom principio,Bom Princípio,22,,-6,-37.883333
br,bom principio,Bom Princípio,23,,-27.8,-54.766667
br,bom principio,Bom Principio,24,,-9.666667,-62.35
br,bom principio,Bom Princípio,31,,-7.75,-48.45
br,bom principip,Bom Principip,06,,-6.016667,-40.766667
br,bom recreio,Bom Recreio,23,,-28.25,-52.483333
br,bom repouso,Bom Repouso,15,,-22.466667,-46.15
br,bom retiro,Bom Retiro,05,,-12.95,-38.766667
br,bom retiro,Bom Retiro,08,,-20.616667,-40.85
br,bom retiro,Bom Retiro,15,,-18,-43.216667
br,bom retiro,Bom Retiro,18,,-25.204585,-48.989476
br,bom retiro,Bom Retiro,20,,-7.883333,-42.8
br,bom retiro,Bom Retiro,21,,-22.948002,-43.292054
br,bom retiro,Bom Retiro,23,,-27.333333,-52.15
br,bom retiro,Bom Retiro,24,,-10.491667,-62.425
br,bom retiro,Bom Retiro,26,,-27.116667,-51.466667
br,bom retiro,Bom Retiro,27,,-21.083333,-47.883333
br,bom retiro do guaiba,Bom Retiro do Guaíba,23,,-30.116667,-51.5
br,bom retiro do sul,Bom Retiro do Sul,23,,-29.616667,-51.933333
br,bom socego,Bom Socego,05,,-12.2,-42.666667
br,bom socego,Bom Socego,11,,-20.9,-55.516667
br,bom socorro,Bom Socorro,04,,-4.394167,-59.976389
br,bom sossego,Bom Sossêgo,05,,-12.2,-42.666667
br,bom sossego,Bom Sossêgo,06,,-5.133333,-40.866667
br,bom sossego,Bom Sossêgo,11,,-20.9,-55.516667
br,bom sossego,Bom Sossego,13,,-3.401111,-43.211944
br,bom sossego,Bom Sossêgo,17,,-6.683333,-37.55
br,bom sossego,Bom Sossêgo,30,,-7.866667,-37.783333
br,bom successo,Bom Successo,04,,-1.133333,-62.65
br,bom successo,Bom Successo,15,,-21.033333,-44.766667
br,bom successo,Bom Successo,21,,-22.860844,-43.25219
br,bom successo,Bom Successo,27,,-23.383333,-48.716667
br,bom sucesso,Bom Sucesso,04,,-1.133333,-62.65
br,bom sucesso,Bom Sucesso,05,,-12.083333,-37.966667
br,bom sucesso,Bom Sucesso,06,,-5.083333,-39.133333
br,bom sucesso,Bom Sucesso,07,,-15.516667,-47.633333
br,bom sucesso,Bom Sucesso,08,,-20.983333,-41.633333
br,bom sucesso,Bom Sucesso,11,,-19.316667,-55.2
br,bom sucesso,Bom Sucesso,13,,-3.6,-43.366667
br,bom sucesso,Bom Sucesso,14,,-15.716667,-56.116667
br,bom sucesso,Bom Sucesso,15,13786,-21.033333,-44.766667
br,bom sucesso,Bom Sucesso,16,,-6.833333,-53.833333
br,bom sucesso,Bom Sucesso,17,,-6.45,-37.933333
br,bom sucesso,Bom Sucesso,18,,-23.1403,-50.2403
br,bom sucesso,Bom Sucesso,20,,-3.867778,-42.590833
br,bom sucesso,Bom Sucesso,21,,-22.256618,-42.774422
br,bom sucesso,Bom Sucesso,26,,-26.700614,-49.688309
br,bom sucesso,Bom Sucesso,27,,-21.316667,-47.416667
br,bom sucesso,Bom Sucesso,29,,-15.583333,-49.433333
br,bom sucesso,Bom Sucesso,30,,-8.5,-36.716667
br,bom sucesso,Bom Sucesso,31,,-7.966667,-48.633333
br,bom sucesso dos milagres,Bom Sucesso dos Milagres,22,,-5.3,-37.683333
br,bom sueso,Bom Sueso,11,,-19.316667,-55.2
br,bom tempo,Bom Tempo,06,,-5.366667,-40.55
br,bom tempo,Bom Tempo,13,,-2.933333,-44.15
br,bom unico,Bom Único,04,,-2.85,-68.85
br,bom viver,Bom Viver,04,,-3.083333,-57.233333
br,bom viver,Bom Viver,05,,-12.716667,-38.433333
br,bom viver,Bom Viver,20,,-5.1,-41.983333
br,bom viver,Bom Viver,31,,-5.416667,-48.55
br,bonanca,Bonança,15,,-15.733333,-44.183333
br,bonassi,Bonassi,23,,-29.1507,-51.0277
br,bondade,Bondade,13,,-7.433333,-46.4
br,bonecu,Boneçu,05,,-12.633333,-38.483333
br,bonetes,Bonetes,26,,-26.216667,-50.633333
br,bonfim,Bonfim,02,,-9.266667,-37.333333
br,bonfim,Bonfim,04,,-.316667,-66.683333
br,bonfim,Bonfim,05,,-10.45,-40.183333
br,bonfim,Bonfim,06,,-3.583333,-40.483333
br,bonfim,Bonfim,11,,-18.266667,-57.466667
br,bonfim,Bonfim,13,,-2.983333,-45.2
br,bonfim,Bonfim,14,,-17.5,-54.75
br,bonfim,Bonfim,15,,-17.883333,-42.316667
br,bonfim,Bonfim,16,,-4.766667,-56.516667
br,bonfim,Bonfim,20,,-3.808056,-42.231111
br,bonfim,Bonfim,21,,-22.516667,-43.5
br,bonfim,Bonfim,22,,-5.184167,-36.506944
br,bonfim,Bonfim,24,,-10.316667,-62.331944
br,bonfim,Bonfim,25,5769,3.083333,-59.95
br,bonfim,Bonfim,27,,-22.966667,-45.25
br,bonfim,Bonfim,28,,-11.066667,-37.85
br,bonfim,Bonfim,29,,-16.666667,-48.6
br,bonfim,Bonfim,30,,-7.683333,-37.2
br,bonfim,Bonfim,31,,-11.853611,-47.534444
br,bonfim da feira,Bonfim da Feira,05,,-12.233333,-39.216667
br,bonfim da mata,Bonfim da Mata,05,,-12.55,-38.3
br,bonfim de arari,Bonfim de Arari,13,,-3.433333,-44.783333
br,bonfim do arari,Bonfim do Arari,13,,-3.433333,-44.783333
br,bonfim i,Bonfim I,13,,-3.686111,-43.395556
br,bonfim ii,Bonfim II,13,,-3.677222,-43.381111
br,bonfim iii,Bonfim III,13,,-3.914167,-43.305
br,bonfim paulista,Bonfim Paulista,27,,-21.266667,-47.816667
br,bonfinopolis,Bonfinópolis,29,,-16.633333,-48.966667
br,bonhu,Bonhu,06,,-4.833333,-38.166667
br,bonifacio,Bonifácio,02,,-9.516667,-36.566667
br,bonilha,Bonilha,13,,-5.2,-43.216667
br,bonim,Bonim,18,,-23.811631,-50.054001
br,boninal,Boninal,05,,-12.7,-41.833333
br,bonita,Bonita,05,,-12.45,-40.1
br,bonita,Bonita,06,,-6.6,-38.866667
br,bonitao,Bonitão,13,,-5.483333,-43.416667
br,bonitinho,Bonitinho,17,,-7.683333,-36
br,bonito,Bonito,02,,-10.383333,-36.333333
br,bonito,Bonito,05,,-11.683333,-37.633333
br,bonito,Bonito,06,,-4.383333,-39.466667
br,bonito,Bonito,11,14504,-21.133333,-56.466667
br,bonito,Bonito,13,,-5.383333,-43.233333
br,bonito,Bonito,15,,-15.333333,-44.766667
br,bonito,Bonito,16,,-1.416667,-47.316667
br,bonito,Bonito,17,,-7.283333,-36.95
br,bonito,Bonito,18,,-25.838713,-50.026299
br,bonito,Bonito,20,,-5.65,-41.8
br,bonito,Bonito,22,,-5.382778,-36.352778
br,bonito,Bonito,23,,-28.583333,-53.616667
br,bonito,Bonito,26,,-28.433333,-49.4
br,bonito,Bonito,27,,-25.201944,-48.131667
br,bonito,Bonito,29,,-18.383333,-52.116667
br,bonito,Bonito,30,8867,-8.483333,-35.733333
br,bonito,Bonito,31,,-6.133333,-47.466667
br,bonito de cima,Bonito de Cima,05,,-12.216667,-41.25
br,bonito de santa fe,Bonito de Santa Fé,17,,-7.316667,-38.516667
br,bonito velha,Bonito Velha,17,,-7.466667,-38.55
br,bonival,Bonival,05,,-12.7,-41.833333
br,bonotto,Bonotto,18,,-23.5766,-49.9358
br,bonsecesso,Bonsecesso,20,,-5.35,-42.566667
br,bons olhos,Bons Olhos,15,,-19.2,-48.3
br,bons olnos,Bons Olnos,15,,-19.2,-48.3
br,bonsucesso,Bonsucesso,05,,-12.65,-42.216667
br,bonsucesso,Bonsucesso,06,,-3.466667,-39.783333
br,bonsucesso,Bonsucesso,08,,-20.983333,-41.566667
br,bonsucesso,Bonsucesso,14,,-15.716667,-56.116667
br,bonsucesso,Bonsucesso,17,,-6.716667,-36.183333
br,bonsucesso,Bonsucesso,20,,-7.966667,-41.516667
br,bonsucesso,Bonsucesso,21,,-22.860844,-43.25219
br,bonsucesso,Bonsucesso,22,,-5.283333,-37.65
br,bonsucesso,Bonsucesso,26,,-26.700614,-49.688309
br,bonsucesso,Bonsucesso,30,,-7.9,-36.116667
br,bontim,Bontim,06,,-5.25,-38.5
br,bonzao,Bonzão,05,,-12.333333,-41.316667
br,boqueirao,Boqueirão,02,,-9.25,-37.866667
br,boqueirao,Boqueirão,05,,-11.333333,-43.85
br,boqueirao,Boqueirão,06,,-3.733333,-38.866667
br,boqueirao,Boqueirão,11,,-21.483333,-54.683333
br,boqueirao,Boqueirão,13,,-3.695,-43.4775
br,boqueirao,Boqueirão,15,,-16.566667,-41.283333
br,boqueirao,Boqueirão,17,11566,-7.483333,-36.116667
br,boqueirao,Boqueirão,18,,-25.483333,-49.25
br,boqueirao,Boqueirão,20,,-10.433333,-45.233333
br,boqueirao,Boqueirão,21,,-22.838612,-43.472046
br,boqueirao,Boqueirão,22,,-5.933333,-35.883333
br,boqueirao,Boqueirão,23,,-29.3,-52.433333
br,boqueirao,Boqueirão,30,,-7.716667,-39.85
br,boqueirao,Boqueirão,31,,-11.633333,-47.466667
br,boqueirao das parreiras,Boqueirao das Parreiras,05,,-14.566667,-42.95
br,boqueirao de fora,Boqueirão de Fora,20,,-6.816667,-40.716667
br,boqueirao de parreiras,Boqueirão de Parreiras,05,,-14.566667,-42.95
br,boqueirao do cesario,Boqueirão do Cesário,06,,-4.583333,-38.2
br,boqueirao do fabricio,Boqueirão do Fabrício,13,,-3.582222,-43.301944
br,boqueirao do gado,Boqueirão do Gado,13,,-3.577778,-43.255278
br,boqueirao do leao,Boqueirão do Leão,23,,-29.3,-52.433333
br,boqueirao dos cochos,Boqueirão dos Cochos,17,,-7.166667,-38.166667
br,boqueirao dos couxos,Boqueirao dos Couxos,17,,-7.166667,-38.166667
br,boqueirao dos filipes,Boqueirão dos Filipes,20,,-9.738611,-45.674167
br,boqueiraozinho,Boqueirãozinho,05,,-14.333333,-43.166667
br,boqueiraozinho,Boqueirãozinho,13,,-3.685278,-43.495833
br,boqueriao,Boquerião,23,,-29.3,-52.433333
br,boquim,Boquim,28,14848,-11.15,-37.616667
br,boquinha,Boquinha,11,,-18.4,-56.116667
br,boquira,Boquira,05,6306,-12.816667,-42.766667
br,bora,Borá,13,,-9.833333,-46.1
br,bora,Borá,27,,-22.283333,-50.55
br,boraceia,Boracéia,27,,-22.166667,-48.75
br,borba,Borba,04,11918,-4.388889,-59.593611
br,borba,Borba,23,,-30.966667,-52.383333
br,borba gato,Borba Gato,15,,-19.133333,-43
br,borboleta,Borboleta,18,,-24.7,-51.783333
br,borboleta,Borboleta,26,,-26.689823,-49.530043
br,borboleta,Borboleta,27,,-20.933333,-49.466667
br,borborema,Borborema,17,,-6.8,-35.6
br,borborema,Borborema,27,11759,-21.616667,-49.066667
br,borborema,Borborema,30,,-7.666667,-37.7
br,borda da mata,Borda da Mata,06,,-6.783333,-38.866667
br,borda da mata,Borda da Mata,15,12814,-22.266667,-46.166667
br,borda da mata,Borda da Mata,27,,-20.333333,-47.283333
br,borda da mata,Borda da Mata,28,,-10.566667,-37.266667
br,borda da matta,Borda da Matta,15,,-22.266667,-46.166667
br,borebi,Borebi,27,,-22.566667,-48.983333
br,boreby,Boreby,27,,-22.566667,-48.983333
br,borel,Borel,21,,-22.93768,-43.254174
br,borel,Borel,26,,-28.083333,-50.433333
br,borgado,Borgado,06,,-5.116667,-40.116667
br,borges,Borges,06,,-4.883333,-37.866667
br,borges,Borges,13,,-5.25,-43.116667
br,borges,Borges,14,,-15.6,-55.083333
br,borges,Borges,17,,-7.1,-36.566667
br,borges,Borges,25,,2.666667,-61.016667
br,borges,Borges,30,,-9.083333,-36.666667
br,borgheto,Borgheto,23,,-29.2,-51.5
br,borgueto,Borgueto,23,,-29.2,-51.5
br,borracha,Borracha,05,,-9.4,-39.633333
br,borracha,Borracha,13,,-3.083889,-42.296111
br,borrachudo,Borrachudo,06,,-3.55,-38.883333
br,borrachudo,Borrachudo,13,,-3.294444,-42.347778
br,borrachudo,Borrachudo,18,,-25.262411,-48.490596
br,borralho,Borralho,04,,-2.607222,-57.268056
br,borrazopolis,Borrazópolis,18,,-23.933333,-51.6
br,bosque,Bosque,20,,-6.633333,-41.75
br,bosque,Bosque,23,,-32.066667,-52.183333
br,bosque,Bosque,27,,-21.283333,-47.383333
br,bosque da inspiracao,Bosque da Inspiração,27,,-23.377521,-46.628864
br,bosque dos esquilos,Bosque dos Esquilos,21,,-22.962153,-43.331885
br,bossoroca,Bossoroca,05,,-9.466667,-41.1
br,bossoroca,Bossoroca,23,,-28.733333,-54.916667
br,botafogo,Botafogo,01,,-8.25,-73.166667
br,botafogo,Botafogo,04,,-3.45,-68.333333
br,botafogo,Botafogo,15,,-21.383333,-43.283333
br,botafogo,Botafogo,18,,-23.384729,-52.60926
br,botafogo,Botafogo,21,,-22.951497,-43.187988
br,botafogo,Botafogo,27,,-20.983333,-48.55
br,botelho,Botelho,25,,.648889,-61.675833
br,botelho,Botelho,27,,-21.35,-48.766667
br,botelhos,Botelhos,15,11073,-21.65,-46.4
br,botorobi,Botorobi,15,,-20.066667,-44.433333
br,botos,Botos,16,,-2.916667,-49.65
br,botucarai,Botucaraí,23,,-29.7,-52.9
br,botucatu,Botucatu,27,113865,-22.866667,-48.433333
br,botucudo,Botucudo,18,,-24.416667,-51.116667
br,botujuru,Botujuru,27,,-23.4599,-46.1089
br,botumirim,Botumirim,15,,-16.866667,-43.016667
br,botupora,Botuporã,05,,-13.35,-42.483333
br,botuquara,Botuquara,05,,-13.666667,-42.866667
br,botuquara,Botuquara,27,,-23.4902,-46.0532
br,boturana,Boturana,27,,-20.283333,-49.5
br,boturobi,Boturobi,15,,-20.066667,-44.433333
br,boturuma,Boturuma,27,,-20.283333,-49.5
br,boturuna,Boturuna,27,,-20.283333,-49.5
br,botuvera,Botuverá,26,,-27.216667,-49.083333
br,bouquinha,Bouquinha,20,,-5.25,-42.666667
br,braba,Braba,05,,-12.35,-39.05
br,brabo,Brabo,30,,-8.15,-37.433333
br,bracatinga,Bracatinga,18,,-25.371409,-48.984222
br,bracinho,Bracinho,11,,-19.6,-54.3
br,bracinho,Bracinho,13,,-3.31,-43.089722
br,bracinho,Bracinho,14,,-16.383333,-57.916667
br,braco,Braço,27,,-24.6379,-48.2924
br,braco do laurentino,Braço do Laurentino,26,,-27.2023,-49.7037
br,braco do norte,Braço do Norte,05,,-13.816667,-39.416667
br,braco do norte,Braço do Norte,26,22375,-28.283333,-49.183333
br,braco do pindauva,Braço do Pindaúva,27,,-24.804449,-48.061746
br,braco do pito,Braço do Pito,27,,-24.697295,-48.200288
br,braco do queimado,Braço do Queimado,27,,-24.7999,-48.2165
br,braco do trombudo,Braço do Trombudo,26,,-27.35,-49.9
br,braco feio,Braço Feio,27,,-24.878,-48.224
br,braco itoupava,Braço Itoupava,26,,-27.1405,-49.6458
br,braco paula ramos,Braço Paula Ramos,26,,-26.7532,-49.0045
br,braco torrete,Braço Torrete,26,,-26.998566,-49.956212
br,braga,Braga,06,,-4.633333,-39.766667
br,braga,Braga,17,,-7.183333,-38.616667
br,braga,Braga,23,,-27.616667,-53.75
br,braganca,Bragança,15,,-20.583333,-46.766667
br,braganca,Bragança,16,57164,-1.05,-46.766667
br,braganca,Bragança,23,,-30.7,-55.183333
br,braganca,Bragança,27,,-22.95,-46.566667
br,braganca paulista,Bragança Paulista,27,125099,-22.95,-46.566667
br,brajo faichada,Brajo Faichada,31,,-11.266667,-46.666667
br,branco,Branco,05,,-13.566667,-39.9
br,branco,Branco,14,,-15.983333,-58.616667
br,branco,Branco,21,,-21.5,-41.783333
br,brancos,Brancos,18,,-25.8483,-49.4494
br,brandao,Brandão,05,,-12.283333,-40.583333
br,brandao,Brandão,06,,-3.466667,-39.45
br,brandao,Brandão,30,,-8.666667,-39.183333
br,brandoes,Brandoes,22,,-6.516667,-37.2
br,brandura,Brandura,13,,-3.033333,-43.45
br,branquinha,Branquinha,02,,-9.25,-36.033333
br,brasabrantes,Brasabrantes,29,,-16.433333,-49.383333
br,brasa,Brasa,20,,-7.15,-40.9
br,bras,Brás,04,,-2.083333,-58.166667
br,bras,Brás,15,,-15.416667,-46.9
br,bras,Brás,27,,-23.55,-46.616667
br,bras cubas,Brás Cubas,27,,-23.533333,-46.216667
br,bras de pina,Brás de Pina,21,,-22.832978,-43.297667
br,brasilandia,Brasilândia,07,,-15.683333,-48.2
br,brasilandia do tocantins,Brasilândia do Tocantins,31,,-8.354722,-48.364444
br,brasilania,Brasilânia,29,,-16.016667,-49.533333
br,brasil,Brasil,04,,-4.516667,-71.616667
br,brasil,Brasil,11,,-22.366667,-54.566667
br,brasil,Brasil,27,,-21.583333,-46.9
br,brasileia,Brasiléia,01,11200,-11,-68.733333
br,brasileiras,Brasileiras,30,,-9.05,-36.566667
br,brasil gaudioso,Brasil Gaudioso,11,,-23.816667,-54.2
br,brasilia,Brasília,01,,-11,-68.733333
br,brasilia,Brasília,07,2207812,-15.783333,-47.916667
br,brasilia,Brasília,13,,-3.443889,-42.485556
br,brasilia,Brasília,15,,-16.211944,-44.430833
br,brasilia,Brasília,18,,-22.716667,-51.4
br,brasilia,Brasília,20,,-3.894722,-42.694167
br,brasilia,Brasília,22,,-5.370556,-36.329444
br,brasilia de minas,Brasília de Minas,15,14730,-16.211944,-44.430833
br,brasilia legal,Brasília Legal,16,,-3.816667,-55.6
br,brasilino,Brasilino,05,,-17.35,-40.433333
br,brasil leite,Brasil Leite,23,,-30.016667,-56.45
br,brasino,Brasino,23,,-30.633333,-51.733333
br,brasitania,Brasitânia,27,,-20.133333,-50.316667
br,braslandia,Braslândia,07,,-15.683333,-48.2
br,brasopolis,Brasópolis,15,,-22.466667,-45.616667
br,bras pires,Brás Pires,15,,-20.85,-43.25
br,brauna,Braúna,05,,-10.433333,-40.15
br,brauna,Braúna,06,,-6.866667,-39.45
br,brauna,Braúna,21,,-21.116667,-41.783333
br,brauna,Braúna,27,,-21.5,-50.3
br,brauna,Braúna,30,,-8.5,-36.383333
br,braunal,Braunal,11,,-22.1,-57.766667
br,braunas,Braúnas,15,,-19.066667,-42.716667
br,braunas,Braúnas,20,,-7.366667,-40.933333
br,braunas-de-guanhaes,Braúnas-de-Guanhães,15,,-19.066667,-42.716667
br,braunat,Braunat,11,,-22.1,-57.766667
br,brava,Brava,05,,-12.35,-39.05
br,bravo,Bravo,05,,-12.15,-39.466667
br,bravo,Bravo,06,,-6.483333,-39.3
br,bravo,Bravo,20,,-5.766667,-42.85
br,brazabrantes,Brazabrantes,29,,-16.433333,-49.383333
br,braz,Braz,27,,-23.55,-46.616667
br,brazilia,Brazilia,15,,-16.211944,-44.430833
br,brazlandia,Brazlândia,07,,-15.683333,-48.2
br,brazopolis,Brazópolis,15,,-22.466667,-45.616667
br,braz pires,Braz Pires,15,,-20.85,-43.25
br,brebe,Brebe,04,,-5.701111,-60.930833
br,brecha,Brecha,02,,-9.5,-36.833333
br,breiinho,Breiinho,30,,-9.1,-38.283333
br,breio de sao felix,Breio de São Félix,13,,-5.483333,-43.5
br,breio grande,Breio Grande,20,,-5.85,-41.35
br,breja,Breja,06,,-4.25,-38.933333
br,brejal,Brejal,18,,-25.116667,-49.483333
br,brejao,Brejão,05,,-10.938611,-45.714722
br,brejao,Brejão,11,,-20.266667,-55.533333
br,brejao,Brejão,13,,-3.512222,-43.195833
br,brejao,Brejão,14,,-15.316667,-57.316667
br,brejao,Brejão,15,,-18.666667,-47.333333
br,brejao,Brejão,18,,-23.0743,-50.0269
br,brejao,Brejão,20,,-5.766667,-42.55
br,brejao,Brejão,27,,-21.633333,-46.866667
br,brejao,Brejão,29,,-12.983333,-46.466667
br,brejao,Brejão,30,,-9.05,-36.483333
br,brejao,Brejão,31,,-7.033333,-48.1
br,brejao da caatinga,Brejão da Caatinga,05,,-10.433333,-40.866667
br,brejao da catinga,Brejão da Catinga,05,,-10.433333,-40.866667
br,brejao das grotas,Brejão das Grotas,05,,-10.65,-40.466667
br,brejao de santa cruz,Brejão de Santa Cruz,30,,-9.05,-36.483333
br,brejatuba,Brejatuba,18,,-25.916667,-48.583333
br,brejauba,Brejaúba,08,,-20.15,-41.283333
br,brejauba,Brejaúba,15,,-19.2,-43.15
br,brejaubinha,Brejaùbinha,15,,-18.75,-42.216667
br,brejauva,Brejaúva,27,,-23.833333,-47.95
br,brejetuba,Brejetuba,08,,-20.15,-41.283333
br,brejetubinha,Brejetubinha,08,,-20.166667,-41.283333
br,brejinhe,Brejinhe,06,,-7.7,-38.9
br,brejinho,Brejinho,05,,-11.366667,-43.3
br,brejinho,Brejinho,06,,-6.25,-39.033333
br,brejinho,Brejinho,13,,-3.511111,-43.178611
br,brejinho,Brejinho,17,,-6.333333,-37.8
br,brejinho,Brejinho,20,,-3.420556,-42.145278
br,brejinho,Brejinho,22,7625,-6.183333,-35.35
br,brejinho,Brejinho,30,,-7.383333,-37.316667
br,brejinho,Brejinho,31,,-11.014167,-48.505
br,brejinho das amethystas,Brejinho das Amethystas,05,,-14.283333,-42.516667
br,brejinho das ametistas,Brejinho das Ametistas,05,,-14.283333,-42.516667
br,brejinho da serra negra,Brejinho da Serra Negra,05,,-12.033333,-43
br,brejinho de nazare,Brejinho de Nazaré,31,,-11.014167,-48.505
br,brejinho do ismael,Brejinho do Ismael,13,,-5.883333,-43.283333
br,brejo alegre,Brejo Alegre,15,,-20.033333,-44.65
br,brejo alegre,Brejo Alegre,27,,-21.133333,-50.166667
br,brejo bonito,Brejo Bonito,15,,-19.016667,-46.6
br,brejo,Brejo,05,,-11.766667,-42.4
br,brejo,Brejo,06,,-3.766667,-41.25
br,brejo,Brejo,13,11636,-3.683333,-42.783333
br,brejo,Brejo,15,,-15.8,-45.866667
br,brejo,Brejo,17,,-6.916667,-37.966667
br,brejo,Brejo,20,,-3.971667,-42.090556
br,brejo,Brejo,22,,-5.166667,-35.75
br,brejo,Brejo,30,,-8.55,-35.333333
br,brejo comprido,Brejo Comprido,31,,-9.6625,-47.963611
br,brejo da canabrava,Brejo da Canabrava,05,,-12.15,-40.883333
br,brejo da cruz,Brejo da Cruz,17,,-6.35,-37.5
br,brejo da fama,Brejo da Fama,05,,-13.216667,-41.616667
br,brejo da fortaleza,Brejo da Fortaleza,20,,-6.75,-41.816667
br,brejo da madre de deus,Brejo da Madre de Deus,30,27370,-8.15,-36.366667
br,brejo da onca,Brejo da Onça,20,,-5.816667,-41.35
br,brejo da porta,Brejo da Porta,13,,-8.499722,-45.738056
br,brejo da roca,Brejo da Roça,13,,-8.238611,-45.655278
br,brejo das almas,Brejo das Almas,15,,-16.383333,-41.516667
br,brejo da serra,Brejo da Serra,05,,-10.1,-43.266667
br,brejo das freiras,Brejo das Freiras,17,,-6.7,-38.516667
br,brejo das missoes,Brejo das Missões,05,,-12.016667,-44.483333
br,brejo das pedras,Brejo das Pedras,20,,-7.4,-41.9
br,brejo da taboa,Brejo da Taboa,05,,-10.466667,-42.583333
br,brejo da tabua,Brejo da Tabua,05,,-10.466667,-42.583333
br,brejo de cima,Brejo de Cima,05,,-13.25,-41.583333
br,brejo de dentro,Brejo de Dentro,30,,-7.9,-37.666667
br,brejo de santo antonio,Brejo de Santo Antônio,30,,-7.483333,-40.116667
br,brejo de santo inacio,Brejo de Santo Inácio,20,,-7.416667,-41.916667
br,brejo de sao caetano,Brejo de São Caetano,15,,-14.65,-44.033333
br,brejo de sao felix,Brejo de São Félix,13,,-5.4,-43.4
br,brejo de sao joao,Brejo de São João,20,,-7.983333,-42.733333
br,brejo do amparo,Brejo do Amparo,15,,-15.433333,-44.4
br,brejo do borbo,Brejo do Borbo,05,,-9.35,-38.466667
br,brejo do burgo,Brejo do Burgo,05,,-9.35,-38.466667
br,brejo do buriti,Brejo do Buriti,05,,-10.016667,-43.033333
br,brejo do buriti,Brejo do Buriti,29,,-14.316667,-46.033333
br,brejo do coqueiro,Brejo do Coqueiro,20,,-6.483333,-41.366667
br,brejo do cruz,Brejo do Cruz,17,,-6.35,-37.5
br,brejo do espirito santo,Brejo do Espírito Santo,05,,-13.333333,-44.3
br,brejo do fogo,Brejo do Fogo,15,,-15.233333,-42.066667
br,brejo do gomes,Brejo do Gomes,05,,-9.433333,-42.166667
br,brejo do mata-fome,Brejo do Mata-Fome,15,,-14.883333,-44.216667
br,brejo do mutambal,Brejo do Mutambal,15,,-15.55,-44.016667
br,brejo do salto,Brejo do Salto,05,,-11.083333,-45.733333
br,brejo dos altos,Brejo dos Altos,20,,-8.933333,-44.283333
br,brejo dos cagados,Brejo dos Cágados,20,,-6.916667,-43.933333
br,brejo dos marques,Brejo dos Marques,05,,-12.766667,-41.8
br,brejo dos martires,Brejo dos Mártires,15,,-15.05,-43.116667
br,brejo dos santos,Brejo dos Santos,06,,-7.483333,-39
br,brejo dos santos,Brejo dos Santos,17,,-6.366667,-37.816667
br,brejo do tabuleiro,Brejo do Tabuleiro,20,,-7.1,-41.75
br,brejo do tucano,Brejo do Tucano,20,,-7.016667,-41.733333
br,brejoes,Brejões,05,,-11.033333,-41.416667
br,brejo escuro,Brejo Escuro,13,,-9,-46.016667
br,brejo espirito santo,Brejo Espírito Santo,05,,-13.333333,-44.3
br,brejo fechado,Brejo Fechado,31,,-11.266667,-46.666667
br,brejo feichado,Brejo Feichado,31,,-11.266667,-46.666667
br,brejo grande,Brejo Grande,05,,-10.033333,-38.533333
br,brejo grande,Brejo Grande,06,,-7.25,-39.766667
br,brejo grande,Brejo Grande,15,,-15.816667,-44.033333
br,brejo grande,Brejo Grande,20,,-5.85,-41.35
br,brejo grande,Brejo Grande,28,,-10.433333,-36.466667
br,brejo joao alves,Brejo João Alves,05,,-12.416667,-41.683333
br,brejo largo,Brejo Largo,31,,-11.033333,-46.533333
br,brejo logo,Brejo Logo,13,,-8.216667,-45.75
br,brejo luisa de brito,Brejo Luisa de Brito,05,,-12.85,-42.016667
br,brejo luiza de brito,Brejo Luiza de Brito,05,,-12.85,-42.016667
br,brejo novo,Brejo Novo,05,,-12.833333,-40.8
br,brejo novo,Brejo Nôvo,30,,-7.45,-37.516667
br,brejo piqui,Brejo Piqui,05,,-10.233333,-43.116667
br,brejo redondo,Brejo Redondo,13,,-6.45,-46.95
br,brejo santo,Brejo Santo,06,27384,-7.483333,-39
br,brejos das freiras,Brejos das Freiras,17,,-6.666667,-38.483333
br,brejos dos pachecos,Brejos dos Pachecos,06,,-3.5,-41.333333
br,brejo secco,Brejo Secco,06,,-7.2,-40.133333
br,brejo seco,Brejo Sêco,06,,-7.2,-40.133333
br,brejo seco,Brejo Seco,20,,-7.683333,-43.066667
br,brejo seco,Brejo Sêco,31,,-10.166667,-48.3
br,brejo velho,Brejo Velho,05,,-12.483333,-43.966667
br,brejo velho,Brejo Velho,20,,-6.483333,-41.366667
br,brejo velho,Brejo Velho,30,,-8.933333,-36.7
br,brejo verde,Brejo Verde,31,,-11.966944,-49.250556
br,brejui,Brejuí,22,,-6.316667,-36.55
br,brenha,Brenha,06,,-4.2,-38.816667
br,breozinho,Breozinho,04,,-3.483333,-66.383333
br,breta,Breta,05,,-12.5,-38.9
br,brete,Brete,11,,-18.65,-56.783333
br,breuzinho,Breuzinho,04,,-3.483333,-66.383333
br,breves,Breves,16,46474,-1.666667,-50.483333
br,brigada,Brigada,06,,-6.716667,-39.566667
br,brigadeiro tobias,Brigadeiro Tobias,27,,-23.507804,-47.366566
br,brigido,Brígido,15,,-21.416667,-46.216667
br,brigue,Brigue,03,,.85,-50.1
br,brilhante,Brilhante,06,,-5.416667,-40.25
br,brilhante,Brilhante,11,,-20.5,-54.2
br,brilhante,Brilhante,13,,-6.316667,-43.1
br,brilhante,Brilhante,14,,-15.966667,-55.066667
br,brilhante,Brilhante,15,,-18.5,-48.883333
br,brilhante,Brilhante,17,,-7.6,-38.066667
br,brilhantina,Brilhantina,16,,-1.2,-49.15
br,brinco,Brinco,21,,-21.916667,-41.75
br,brioso,Brioso,11,,-19.483333,-54.5
br,brioso,Brioso,27,,-20.8,-50.166667
br,britania,Britânia,29,,-15.233333,-51.15
br,brithante,Brithante,14,,-15.966667,-55.066667
br,brito,Brito,06,,-5.4,-39.233333
br,brito,Brito,17,,-7.066667,-36.933333
br,brito,Brito,28,,-10.6,-36.783333
br,brito,Brito,30,,-8.866667,-36.683333
br,broca,Broca,05,,-9.833333,-42.466667
br,broca,Broca,30,,-8.866667,-37.4
br,brocato,Brocató,30,,-7.816667,-38.166667
br,brocheirs,Brocheirs,23,,-29.516667,-51.633333
br,brochier,Brochier,23,,-29.516667,-51.633333
br,brococo,Brococo,30,,-7.883333,-38.1
br,brocos,Brocós,13,,-5.083333,-43.816667
br,brocoto,Brocotó,30,,-7.816667,-38.166667
br,brodosqui,Brodósqui,27,,-20.983333,-47.666667
br,brodovsky,Brodovsky,27,,-20.983333,-47.666667
br,brodowski,Brodowski,27,18366,-20.983333,-47.666667
br,brondo,Brondó,05,,-12.983333,-39.6
br,brotas,Brotas,05,,-12,-42.633333
br,brotas,Brotas,06,,-3.566667,-39.75
br,brotas,Brotas,14,,-15.2,-56.366667
br,brotas,Brotas,17,,-7.25,-37.95
br,brotas,Brotas,20,,-5.366667,-41.466667
br,brotas,Brotas,27,18690,-22.283333,-48.116667
br,brotas de macahubas,Brotas de Macahubas,05,,-12,-42.633333
br,brotas de macaubas,Brotas de Macaúbas,05,,-12,-42.633333
br,brudertahl,Brudertahl,26,,-26.797,-49.1716
br,brumadinho,Brumadinho,05,,-13.55,-41.9
br,brumadinho,Brumadinho,15,27564,-20.133333,-44.216667
br,brumado,Brumado,05,41990,-14.216667,-41.666667
br,brumado,Brumado,14,,-14.716667,-56.633333
br,brumado,Brumado,15,,-18.75,-49.416667
br,brumado do assurua,Brumado do Assuruá,05,,-11.55,-42.75
br,brumal,Brumal,15,,-19.983333,-43.45
br,brundue,Brundué,05,,-11.933333,-42.933333
br,bruno,Bruno,18,,-23.409548,-52.503786
br,brusque,Brusque,26,88286,-27.1,-48.933333
br,bruto,Bruto,06,,-4.8,-40.316667
br,buare,Buare,18,,-25.85,-52.45
br,buarque,Buarque,15,,-20.75,-43.8
br,buarque de macedo,Buarque de Macedo,15,,-20.75,-43.8
br,bu,Bu,05,,-12.05,-37.7
br,bubu,Bubu,08,,-20.283333,-40.4
br,bucuituba,Bucuituba,05,,-12.666667,-42.816667
br,buena hora,Buena Hora,14,,-15.6,-60.366667
br,buena hora,Buena Hora,24,,-11.683333,-65.083333
br,bueno andrade,Bueno Andrade,27,,-21.666667,-48.25
br,bueno brandao,Bueno Brandão,15,5777,-22.45,-46.35
br,bueno,Bueno,14,,-15.25,-56.2
br,bueno,Bueno,15,,-19.45,-41.5
br,bueno de andrada,Bueno de Andrada,27,,-21.666667,-48.25
br,bueno de andrade,Bueno de Andrade,21,,-22.366667,-43.416667
br,bueno de andrade,Bueno de Andrade,27,,-21.666667,-48.25
br,buenolandia,Buenolândia,29,,-15.8,-50.283333
br,buenopolis,Buenópolis,15,,-17.9,-44.183333
br,buenos aires,Buenos Aires,01,,-8.492778,-72.845556
br,buenos aires,Buenos Aires,05,,-9.6,-42.816667
br,buenos aires,Buenos Aires,06,,-6.366667,-38.866667
br,buenos-aires,Buenos-Aires,06,,-7.116667,-40.033333
br,buenos aires,Buenos Aires,08,,-20.583333,-40.533333
br,buenos aires,Buenos Aires,13,,-2.85,-43.533333
br,buenos aires,Buenos Aires,27,,-22.3,-50.216667
br,buenos aires,Buenos Aires,30,,-7.6,-37.233333
br,buenos,Buenos,27,,-23.728114,-47.518779
br,buerarama,Buerarama,05,,-14.95,-39.316667
br,buerarema,Buerarema,05,16002,-14.95,-39.316667
br,bufadeira,Bufadeira,18,,-23.983333,-51.2
br,bugi,Bugi,05,,-12.183333,-39.35
br,bugio,Bugio,18,,-25.667827,-50.67296
br,bugio,Bugio,27,,-24.8703,-48.4783
br,bugre,Bugre,15,,-19.416667,-42.25
br,bugre,Bugre,18,,-25.85,-52.45
br,bugre,Bugre,23,,-29.466667,-53.3
br,bugre,Bugre,26,,-26.066667,-50.233333
br,bugre,Bugre,27,,-22.6121,-45.6809
br,bugres,Bugres,26,,-26.066667,-50.233333
br,buicu,Buiçu,04,,-4.268611,-62.282222
br,buique,Buíque,30,20324,-8.616667,-37.15
br,buiu,Buiú,17,,-7.316667,-37.75
br,buiuio,Buiuio,29,,-14.733333,-49.083333
br,bujari,Bujari,01,,-9.838056,-67.937778
br,bujaru,Bujaru,16,6189,-1.516667,-48.016667
br,bulha,Bulha,27,,-24.544461,-48.049913
br,bulhoes,Bulhões,21,,-22.4606,-44.3453
br,bulhoes velho,Bulhões Velho,21,,-22.4606,-44.3453
br,bultrim,Bultrim,17,,-7.083333,-35.866667
br,bupeva,Bupeva,26,,-26.383333,-48.616667
br,buquim,Buquim,28,,-11.15,-37.616667
br,buquira,Buquira,27,,-22.966667,-45.833333
br,buracao,Buracão,05,,-12.183333,-44.95
br,buracao,Buracão,14,,-14.35,-55.766667
br,buracao,Buracão,15,,-16.25,-41.866667
br,buracao,Buracao,22,,-5.3,-37.733333
br,buracao,Buracão,27,,-21.016667,-47.633333
br,buracica,Buracica,05,,-12.266667,-38.516667
br,buraco d'agua,Buraco d'Agua,05,,-10.466667,-40.533333
br,buraco d' agua,Buraco d' Agua,05,,-12.083333,-39.283333
br,buraco d'agua,Buraco d'Agua,05,,-9.233333,-38.916667
br,buraco d'agua,Buraco d'Agua,20,,-6.066667,-41.75
br,buraco do bicho,Buraco do Bicho,15,,-20.666667,-43.916667
br,buraco seco,Buraco Sêco,22,,-5.45,-35.866667
br,buranhem,Buranhém,05,,-12.516667,-38.65
br,burarama,Burarama,08,,-20.7,-41.333333
br,burgo agricola,Burgo Agrícola,16,,-5.366667,-49.2
br,buri,Buri,05,,-12.2,-38
br,buri,Buri,27,15959,-23.8,-48.583333
br,buriche,Buriche,21,,-22.933333,-42.866667
br,burieta,Burietá,05,,-13.55,-39.466667
br,buris,Buris,05,,-13.55,-39.466667
br,buris dos lopes,Buris dos Lopes,05,,-13.516667,-39.466667
br,buritama,Buritama,27,12712,-21.05,-50.133333
br,buriti alegre,Buriti Alegre,29,9427,-18.15,-49.05
br,buriti bravo,Buriti Bravo,13,17693,-5.833333,-43.833333
br,buriti,Buriti,05,,-10.016667,-43.033333
br,buriti,Buriti,06,,-3.166667,-39.45
br,buriti,Buriti,11,,-18.25,-54.5
br,buriti,Buriti,13,7025,-3.916667,-42.95
br,buriti,Buriti,14,,-13.2,-58.633333
br,buriti,Buriti,15,,-17.283333,-46.466667
br,buriti,Buriti,18,,-25.366667,-52.666667
br,buriti,Buriti,20,,-3.830556,-42.495
br,buriti,Buriti,27,,-20.616667,-47.7
br,buriti,Buriti,29,,-14.383333,-46.2
br,buriti cercado,Buriti Cercado,13,,-6.533333,-43.083333
br,buriti compredo,Buriti Compredo,15,,-18.75,-45.15
br,buriti comprido,Buriti Comprido,20,,-6.666667,-41.616667
br,buriti congonhas,Buriti Congonhas,21,,-22.860244,-43.332003
br,buriti cortado,Buriti Cortado,13,,-5.183333,-43.1
br,buriti da aurora,Buriti da Aurora,13,,-3.991389,-43.435278
br,buriti da cachoeira,Buriti da Cachoeira,15,,-19.05,-44.533333
br,buriti das eguas,Buriti das Eguas,20,,-6.766667,-41.4
br,buriti de laje,Buriti de Laje,20,,-5.633333,-42.183333
br,buriti do castelo,Buriti do Castelo,20,,-5.95,-42.183333
br,buriti do estevam,Buriti do Estevam,13,,-3.783333,-43.259722
br,buriti dos correias,Buriti dos Correias,13,,-3.908056,-43.358889
br,buriti dos lopes,Buriti dos Lopes,20,9475,-3.166667,-41.866667
br,buriti dos montes,Buriti dos Montes,20,,-5.3,-41.1
br,buriti do tocantins,Buriti do Tocantins,31,,-5.296111,-48.094444
br,buriti fechado,Buriti Fechado,14,,-15.233333,-56.933333
br,buriti fechado,Buriti Fechado,20,,-5.25,-41.666667
br,buriti frio,Buriti Frio,13,,-5.333333,-43.45
br,buriti grande,Buriti Grande,13,,-6.166667,-47.166667
br,buriti grande,Buriti Grande,15,,-19.25,-44.583333
br,buriti grande,Buriti Grande,20,,-5.95,-42.9
br,buritirama,Buritirama,05,,-10.716667,-43.65
br,buritirana,Buritirana,13,,-3.368611,-43.021111
br,buritirana,Buritirana,20,,-6.183333,-42.85
br,buritirana,Buritirana,31,,-7.25,-48.216667
br,buriti redondo,Buriti Redondo,13,,-3.082778,-42.258333
br,buritis,Buritis,15,16243,-15.616667,-46.433333
br,buritis,Buritis,27,,-20.183333,-47.733333
br,buriti seco,Buriti Seco,13,,-3.266389,-42.788889
br,buriti-so,Buriti-Só,31,,-10.483333,-46.033333
br,buriti so,Buriti Só,31,,-6.983333,-48.066667
br,buriti torto,Buriti Torto,13,,-5.283333,-46.016667
br,buritizal,Buritizal,03,,.833333,-50.116667
br,buritizal,Buritizal,13,,-3.407222,-43.413333
br,buritizal,Buritizal,14,,-15.733333,-54.45
br,buritizal,Buritizal,16,,-5.183333,-56.6
br,buritizal,Buritizal,20,,-5.216667,-42.083333
br,buritizal,Buritizal,27,,-20.183333,-47.733333
br,buritizeiro,Buritizeiro,13,,-3.810278,-43.363333
br,buritizeiro,Buritizeiro,15,25104,-17.35,-44.966667
br,buritizinho,Buritizinho,05,,-12.216667,-44.433333
br,buritizinho,Buritizinho,06,,-7.333333,-38.783333
br,buritizinho,Buritizinho,13,,-2.95,-42.183333
br,buritizinho,Buritizinho,20,,-5.033333,-42.216667
br,buritizinho,Buritizinho,29,,-14.6,-47.616667
br,buritizinho i,Buritizinho I,13,,-3.570556,-43.088611
br,buritizinho ii,Buritizinho II,13,,-3.315833,-43.134722
br,buritizo,Buritizo,31,,-6.983333,-48.066667
br,burituranas,Burituranas,15,,-17.366667,-44.2
br,burity alegre,Burity Alegre,29,,-18.15,-49.05
br,burityba-mirim,Burityba-Mirim,27,,-23.583333,-46.033333
br,burity bravo,Burity Bravo,13,,-5.833333,-43.833333
br,burity,Burity,06,,-7.216667,-39.383333
br,burity,Burity,11,,-21.366667,-55.8
br,burity,Burity,13,,-3.916667,-42.95
br,burity,Burity,14,,-15.383333,-55.833333
br,burity,Burity,15,,-15.616667,-46.433333
br,burity dos lopes,Burity dos Lopes,20,,-3.166667,-41.866667
br,buritys,Buritys,27,,-20.183333,-47.733333
br,burityzal,Burityzal,16,,-5.183333,-56.6
br,buritzinho,Buritzinho,20,,-5.683333,-41.883333
br,burizeiro,Burizeiro,05,,-12.733333,-38.3
br,burrego,Burrego,30,,-7.933333,-36.483333
br,burriguda dos frades,Burriguda dos Frades,17,,-7.883333,-36.466667
br,burunhaem,Burunhaém,05,,-16.533333,-40.066667
br,bury,Bury,27,,-23.8,-48.583333
br,busca-vida,Busca-Vida,05,,-12.8,-38.2
br,busco vida,Busco Vida,05,,-12.883333,-38.266667
br,butanta,Butantã,05,,-12.866667,-39.333333
br,butia,Butiá,23,19494,-30.116667,-51.966667
br,butia,Butiá,26,,-27.116667,-50.833333
br,butia da lapa,Butiá da Lapa,18,,-26.0016,-49.9621
br,butia de cima,Butiá de Cima,18,,-25.990515,-50.014968
br,butia de cima,Butiá de Cima,26,,-26.2432,-49.9286
br,butia do lajeado,Butiá do Lajeado,26,,-26.2087,-49.9309
br,butia dos carvalho,Butiá dos Carvalho,26,,-26.2432,-49.9286
br,butia dos colaco,Butiá dos Colaço,18,,-25.982038,-50.016911
br,butia dos colacos,Butia dos Colaços,18,,-25.959069,-50.027663
br,butio,Butió,23,,-30.116667,-51.966667
br,buzebio oliveira,Buzebio Oliveira,18,,-23.8107,-50.3623
br,buzios,Búzios,21,,-22.75,-41.883333
br,caa-guacu,Caa-Guaçu,23,,-27.383333,-54.216667
br,caaguacu,Caaguaçu,27,,-23.277517,-47.854631
br,caaguacu de cima,Caaguaçu de Cima,27,,-23.308732,-47.851658
br,caamirim,Caamirim,17,,-7.1,-35.766667
br,caapiranga,Caapiranga,04,,-3.3,-61.216667
br,caapoa,Caapoã,17,,-6.766667,-36.8
br,caapora,Caaporã,17,13770,-7.516667,-34.9
br,caarapa,Caarapã,11,,-22.45,-55.016667
br,caarapo,Caarapó,11,13863,-22.633333,-54.8
br,caatiba,Caatiba,05,,-15,-40.4
br,caatinga,Caatinga,02,,-9.4,-37.45
br,caatinga,Caatinga,05,,-12.066667,-38.933333
br,caatinga,Caatinga,15,,-17.166667,-45.883333
br,caatinga,Caatinga,22,,-5.629167,-37.043611
br,caatinga de alecrim,Caatinga de Alecrim,17,,-6.966667,-38.216667
br,caatinga do atanasio,Caatinga do Atanásio,06,,-5.683333,-38.2
br,caatinga do gois,Caatinga do Góis,22,,-5.433333,-37.783333
br,caatinga do moura,Caatinga do Moura,05,,-10.966667,-40.75
br,caatinga do porco,Caatinga do Porco,20,,-7.8,-42.633333
br,caatingas,Caatingas,05,,-13.6,-39.983333
br,cabaca,Cabaça,05,,-10.133333,-43.466667
br,cabaca,Cabaca,22,,-5.453611,-36.355556
br,cabacais,Cabaçais,27,,-23.678882,-47.847069
br,cabacal,Cabaçal,14,,-16.016667,-58.533333
br,cabacas,Cabaças,05,,-9.216667,-38.383333
br,cabacas,Cabaças,06,,-5.066667,-40.866667
br,cabacas,Cabacas,17,,-7.55,-38.633333
br,cabacas,Cabaças,30,,-7.95,-38.7
br,cabaceira,Cabaceira,05,,-10.183333,-43.3
br,cabaceira,Cabaceira,06,,-7.033333,-40.416667
br,cabaceira,Cabaceira,20,,-6.883333,-40.516667
br,cabaceira,Cabaceira,28,,-9.933333,-37.116667
br,cabaceira,Cabaceira,30,,-8.45,-38.033333
br,cabaceiras,Cabaceiras,05,,-11.5,-42.333333
br,cabaceiras,Cabaceiras,17,,-7.016667,-37.45
br,cabaceiras,Cabaceiras,22,,-5.533333,-35.433333
br,cabaceiras,Cabaceiras,27,,-21.683333,-47.95
br,cabaceiras do paraguacu,Cabaceiras do Paraguaçu,05,,-12.55,-39.183333
br,cabaceiro,Cabaceiro,20,,-3.828333,-42.409444
br,cabaceiro,Cabaceiro,28,,-9.933333,-37.116667
br,cabana,Cabana,22,,-5.299444,-36.543889
br,cabana,Cabana,30,,-7.733333,-38.35
br,cabanas,Cabanas,30,,-8.416667,-36.35
br,cabana valente,Cabana Valente,18,,-25.5222,-49.8491
br,cabange,Cabange,02,,-9.266667,-36.466667
br,cabanha florestal,Cabanha Florestal,18,,-25.4129,-50.6895
br,cabaraquara,Cabaraquara,18,,-25.8304,-48.571
br,cabatam,Cabatam,06,,-3.133333,-39.733333
br,cabeca,Cabeça,02,,-9.966667,-36.066667
br,cabeca,Cabeça,13,,-5,-43.4
br,cabeca,Cabeça,27,,-22.35,-47.683333
br,cabeca,Cabeça,28,,-10.5,-36.416667
br,cabeca d'anta,Cabeca d'Anta,18,,-25.166667,-49.083333
br,cabeca danta,Cabeça Danta,20,,-3.3425,-42.089167
br,cabeca d'anta,Cabeca d'Anta,30,,-7.883333,-38
br,cabeca da onca,Cabeca da Onca,06,,-5.45,-40.916667
br,cabeca da onca,Cabeça da Onça,17,,-6.533333,-37.933333
br,cabeca de boi,Cabeça de Boi,05,,-8.933333,-38.533333
br,cabeca de boi,Cabeça de Boi,06,,-5.533333,-39.116667
br,cabeca de boi,Cabeça de Boi,17,,-7.266667,-37.783333
br,cabeca de boi,Cabeça de Boi,20,,-10.816667,-45.166667
br,cabeca de boi,Cabeca de Boi,22,,-5.783333,-35.55
br,cabeca de cavalo,Cabeça de Cavalo,17,,-7.8,-36.5
br,cabeca de joao ferino,Cabeca de João Ferino,20,,-6.033333,-36.55
br,cabeca de touro,Cabeça de Touro,14,,-16.116667,-59.95
br,cabeca de vaca,Cabeça de Vaca,06,,-5.25,-39.966667
br,cabeca de vaca,Cabeça de Vaca,22,,-5.841667,-37.154444
br,cabeca do cavalo,Cabeça do Cavalo,13,,-3.878889,-42.989444
br,cabeca-no-tempo,Cabeça-no-Tempo,20,,-9.7,-43.816667
br,cabeca quebrada,Cabeça Quebrada,13,,-3.695278,-42.785
br,cabecas,Cabeças,05,,-12.6,-39.033333
br,cabeceira alta do formoso,Cabeceira Alta do Formoso,29,,-18.1,-53.116667
br,cabeceira apa,Cabeceira Apa,11,,-22.016667,-55.766667
br,cabeceira bento gomes,Cabeceira Bento Gomes,14,,-15.8,-56.583333
br,cabeceira,Cabeceira,05,,-10.183333,-43.3
br,cabeceira,Cabeceira,11,,-18.75,-54.766667
br,cabeceira,Cabeceira,13,,-3.098056,-42.940833
br,cabeceira,Cabeceira,20,,-3.800833,-42.383611
br,cabeceira,Cabeceira,30,,-8.45,-38.033333
br,cabeceira da bacaba,Cabeceira da Bacaba,13,,-5.1,-43.15
br,cabeceira da boa hora,Cabeceira da Boa Hora,13,,-3.284722,-42.931944
br,cabeceira da laje,Cabeceira da Laje,13,,-7.216667,-47.25
br,cabeceira de pedras,Cabeceira de Pedras,05,,-12.166667,-46.216667
br,cabeceira do apa,Cabeceira do Apa,11,,-22.016667,-55.766667
br,cabeceira do corda,Cabeceira do Corda,13,,-6.55,-43.216667
br,cabeceira do frei pedro,Cabeceira do Frei Pedro,20,,-5.216667,-42.15
br,cabeceira do ipuita,Cabeceira do Ipuitã,11,,-23.433333,-55.266667
br,cabeceira do ouro,Cabeceira do Ouro,05,,-15.083333,-40.2
br,cabeceira do pindauva,Cabeceira do Pindaúva,27,,-24.7875,-48.054
br,cabeceira do riachao,Cabeceira do Riachão,20,,-9.071389,-45.72
br,cabeceira do ribeirao da cruz,Cabeceira do Ribeirão da Cruz,18,,-23.2777,-49.8034
br,cabeceira do utinga,Cabeceira do Utinga,05,,-12.016667,-41.05
br,cabeceira estrela,Cabeceira Estrela,11,,-22.3,-55.833333
br,cabeceira funda,Cabeceira Funda,11,,-20.933333,-55.033333
br,cabeceira grande,Cabeceira Grande,05,,-13.216667,-44.666667
br,cabeceira limpa,Cabeceira Limpa,11,,-21.583333,-53.85
br,cabeceira rio manso,Cabeceira Rio Manso,14,,-15.65,-55.316667
br,cabeceira sao domingos,Cabeceira São Domingos,11,,-22.083333,-55.016667
br,cabeceiras,Cabeceiras,20,,-4.483333,-42.316667
br,cabeceiras,Cabeceiras,29,,-15.8,-46.983333
br,cabeceiras de pedras,Cabeceiras de Pedras,05,,-12.166667,-46.216667
br,cabeceiras do paraguacu,Cabeceiras do Paraguaçu,05,,-12.55,-39.183333
br,cabeceira velha,Cabeceira Velha,05,,-11.316667,-46.333333
br,cabeceirinha,Cabeceirinha,13,,-6.983333,-47.05
br,cabeceirinha,Cabeceirinha,31,,-11.921667,-46.470278
br,cabe chico,Cabe Chico,04,,-5.766389,-59.35
br,cabecinha,Cabecinha,27,,-20.633333,-47.15
br,cabeco,Cabeço,20,,-6.95,-41.666667
br,cabeco,Cabeço,28,,-10.5,-36.416667
br,cabeco de joao ferino,Cabeço de João Ferino,22,,-5.974444,-36.616111
br,cabecote,Cabeçote,20,,-3.273611,-42.022222
br,cabeco vermelho,Cabeco Vermelho,22,,-5.466667,-35.983333
br,cabecudas,Cabeçudas,26,,-26.95,-48.633333
br,cabedello,Cabedello,17,,-6.966667,-34.833333
br,cabedelo,Cabedelo,17,54841,-6.966667,-34.833333
br,cabedelos,Cabedelos,17,,-6.966667,-34.833333
br,cabeleira,Cabeleira,22,,-5.3375,-36.965833
br,cabeleira,Cabeleira,30,,-8.45,-36.116667
br,cabelo,Cabelo,05,,-12.45,-38.266667
br,cabelo,Cabelo,30,,-7.766667,-37.733333
br,cabelo da cuia,Cabelo da Cuia,05,,-10.3,-39.066667
br,cabelo de negro,Cabelo de Negro,22,,-5.233333,-37.533333
br,cabiuna,Cabiúna,18,,-23.383333,-50.75
br,cabiunas,Cabiúnas,21,,-22.3,-41.716667
br,caboba,Caboba,13,,-3.009444,-42.577778
br,cabo,Cabo,30,,-8.283333,-35.033333
br,cabo chico,Cabo Chico,04,,-5.766389,-59.35
br,cabocio,Cabocio,20,,-7.533333,-41.283333
br,caboclinho,Caboclinho,06,,-2.95,-40.066667
br,caboclo,Caboclo,02,,-9.533333,-37.533333
br,caboclo,Caboclo,05,,-12.566667,-39
br,caboclo,Caboclo,06,,-6.5,-39.216667
br,caboclo,Caboclo,20,,-4.616667,-41.75
br,caboclo,Caboclo,30,,-8.516667,-41
br,caboclo de fora,Caboclo de Fora,05,,-13.2,-38.983333
br,caboclos,Caboclos,05,,-15.85,-39.766667
br,caboclos,Caboclos,20,,-7.283333,-40.583333
br,caboco,Cabocó,05,,-12.566667,-39
br,cabo do campo,Cabo do Campo,30,,-8.35,-36.883333
br,cabo do pasto,Cabo do Pasto,02,,-10.083333,-36.516667
br,cabo frio,Cabo Frio,21,108241,-22.883333,-42.016667
br,cabo joao,Cabo João,29,,-16.266667,-51.666667
br,cabore,Caboré,13,,-3.795833,-42.756944
br,cabore,Caboré,20,,-9.2,-42.516667
br,cabori,Cabori,04,,.366667,-67.283333
br,caboto,Caboto,05,,-12.75,-38.483333
br,cabo verde,Cabo Verde,15,,-21.466667,-46.4
br,cabo verde,Cabo Verde,20,,-3.819444,-42.326111
br,cabra,Cabra,20,,-5.416667,-42.916667
br,cabrais,Cabrais,17,,-7.2,-38.45
br,cabral,Cabral,14,,-15.5,-55.8
br,cabralia,Cabrália,05,,-12.816667,-41.8
br,cabralia,Cabrália,27,,-22.466667,-49.333333
br,cabralia paulista,Cabrália Paulista,27,,-22.466667,-49.333333
br,cabras,Cabras,04,,-5.168333,-58.588333
br,cabreiro,Cabreiro,05,,-13.4,-41.05
br,cabreiro,Cabreiro,06,,-4.616667,-37.833333
br,cabrestos,Cabrestos,15,,-20.383333,-46.266667
br,cabreuva,Cabreúva,27,38743,-23.3,-47.133333
br,cabritos,Cabritos,15,,-21.566667,-44.716667
br,cabrituba,Cabrituba,28,,-9.65,-37.933333
br,cabriuva,Cabriúva,27,,-23.3,-47.133333
br,cabrobo,Cabrobó,30,16096,-8.516667,-39.316667
br,cabucu,Cabuçu,05,,-12.016667,-38.1
br,cabucu,Cabuçu,21,,-22.783448,-43.547674
br,cabucu,Cabuçu,27,,-23.406329,-46.538228
br,cabuis,Cabuis,21,,-22.806128,-43.4062
br,cabura,Caburã,16,,-6.766667,-57.533333
br,cabure,Caburé,13,,-3.573889,-43.275833
br,cabure,Caburé,17,,-7.8,-38.233333
br,caburu,Caburu,15,,-21.1,-44.35
br,cabussu,Cabussú,21,,-22.833333,-42.916667
br,cacabu,Caçabu,05,,-10.666667,-38.916667
br,cacada,Caçada,18,,-25.142971,-48.363855
br,cacada real,Caçada Real,25,,3.133333,-60.716667
br,cacador,Caçador,18,,-24.940349,-48.885726
br,cacador,Caçador,21,,-22.764167,-43.791667
br,cacador,Caçador,26,63727,-26.783333,-51
br,cacador,Caçador,27,,-22.566667,-49.65
br,cacaia,Cacaia,16,,-1.4,-48.116667
br,cacal,Cacal,05,,-12.433333,-44.65
br,cacamba,Caçamba,02,,-9.383333,-36.266667
br,cacambinha,Caçambinha,02,,-9.216667,-36.25
br,cacapava,Caçapava,23,,-30.5,-53.5
br,cacapava,Caçapava,27,72341,-23.1,-45.716667
br,cacapava do sul,Caçapava do Sul,23,17522,-30.5,-53.5
br,cacapava velha,Caçapava Velha,27,,-23.116667,-45.65
br,cacaratiba,Caçaratiba,15,,-17.25,-43.033333
br,cacare,Cacaré,06,,-7.466667,-38.966667
br,cacare,Cacaré,17,,-6.783333,-38.566667
br,cacare varjota,Cacaré Varjota,17,,-7.1,-38.6
br,cacaria,Cacaria,30,,-7.866667,-39.466667
br,cacaroca,Caçaroca,08,,-20.4,-40.383333
br,cacarongonga,Caçarongonga,05,,-12.6,-38.616667
br,cacaual,Cacaual,13,,-1.983333,-46.216667
br,cacau,Cacáu,17,,-7.183333,-38.55
br,cacau,Cacaú,30,,-8.65,-35.266667
br,cacaueira,Cacaueira,06,,-5.133333,-40
br,cacaulandia,Cacaulândia,24,,-10.341667,-62.9
br,cacauzinho,Cacauzinho,04,,-4.425278,-59.830833
br,cacequi,Cacequi,23,12868,-29.883333,-54.816667
br,cacequy,Cacequy,23,,-29.883333,-54.816667
br,caceres,Cáceres,14,73431,-16.066667,-57.683333
br,cacete,Cacete,13,,-5.1,-45.333333
br,cachaco,Cachaço,06,,-6.616667,-38.816667
br,cachambi,Cachambi,21,,-22.889008,-43.279091
br,cachambu,Cachambú,26,,-27.166667,-52.866667
br,cachamorra,Cachamorra,21,,-22.951291,-43.56045
br,cacha pregos,Cacha Pregos,05,,-13.116667,-38.783333
br,cacha torta,Cacha Torta,15,,-20.333333,-42.483333
br,cacheado,Cacheado,06,,-5.266667,-40.833333
br,cache,Caché,20,,-8.45,-42.4
br,cacheiro,Cacheiro,08,,-20.85,-41.1
br,cachimba,Cachimba,18,,-25.616667,-49.35
br,cachimbo,Cachimbo,06,,-5.983333,-39.616667
br,cachimbo,Cachimbo,13,,-5.883333,-44.316667
br,cachimbo,Cachimbo,15,,-18.466667,-43.65
br,cachimbo,Cachimbo,16,,-8.95,-54.9
br,cachimbos,Cachimbos,15,,-18.466667,-43.65
br,cachimho,Cachimho,04,,-5.541944,-64.052222
br,cachinaoa,Cachinaoa,01,,-8.716667,-71.883333
br,cachinaua,Cachinaua,01,,-8.716667,-71.883333
br,cachirimbu,Cachirimbu,13,,-5.083333,-43.416667
br,cacho do moleiro,Cacho do Moleiro,05,,-12.416667,-41.633333
br,cacho dos moleiros,Cacho dos Moleiros,05,,-12.416667,-41.633333
br,cachoeira alegre,Cachoeira Alegre,08,,-20.716667,-41.833333
br,cachoeira alegre,Cachoeira Alegre,15,,-21.25,-42.3
br,cachoeira alegre,Cachoeira Alegre,21,,-21.25,-41.516667
br,cachoeira alta,Cachoeira Alta,29,,-18.8,-50.966667
br,cachoeira branca,Cachoeira Branca,11,,-20.9,-53.35
br,cachoeira,Cachoeira,01,,-8.6,-73.15
br,cachoeira,Cachoeira,02,,-9.6,-37.116667
br,cachoeira,Cachoeira,04,,-3.0425,-61.555278
br,cachoeira,Cachoeira,05,16145,-12.6,-38.966667
br,cachoeira,Cachoeira,06,,-3.166667,-40.216667
br,cachoeira,Cachoeira,08,,-20.35,-41.633333
br,cachoeira,Cachoeira,11,,-19.55,-54.583333
br,cachoeira,Cachoeira,13,,-3.057222,-42.755278
br,cachoeira,Cachoeira,14,,-14.416667,-55.65
br,cachoeira,Cachoeira,15,,-16.6,-41.483333
br,cachoeira,Cachoeira,16,,-1.016667,-48.966667
br,cachoeira,Cachoeira,17,,-6.316667,-37.566667
br,cachoeira,Cachoeira,18,,-25.2557,-48.7371
br,cachoeira,Cachoeira,20,,-3.744444,-42.526389
br,cachoeira,Cachoeira,21,,-21.2,-41.666667
br,cachoeira,Cachoeira,22,,-5.683056,-36.57
br,cachoeira,Cachoeira,23,,-30.033333,-52.9
br,cachoeira,Cachoeira,26,,-27.433333,-48.45
br,cachoeira,Cachoeira,27,,-20.2,-47.416667
br,cachoeira,Cachoeira,29,,-13.433333,-46.833333
br,cachoeira,Cachoeira,30,,-7.733333,-38.45
br,cachoeira,Cachoeira,31,,-7.216667,-48.383333
br,cachoeira da abadia,Cachoeira da Abadia,05,,-11.533333,-37.566667
br,cachoeira da abbadia,Cachoeira da Abbadia,05,,-11.533333,-37.566667
br,cachoeira da ema,Cachoeira da Ema,27,,-21.916667,-47.35
br,cachoeira da mata,Cachoeira da Mata,05,,-17.4,-40.05
br,cachoeira da moca,Cachoeira da Moça,06,,-3.766667,-40.15
br,cachoeira da moca,Cachoeira da Moça,17,,-6.583333,-38.383333
br,cachoeira de baixo,Cachoeira de Baíxo,06,,-4.3,-39.233333
br,cachoeira de cima,Cachoeira de Cima,18,,-25.22899,-48.750102
br,cachoeira de emas,Cachoeira de Emas,27,,-21.916667,-47.35
br,cachoeira de goias,Cachoeira de Goiás,29,,-16.733333,-50.633333
br,cachoeira de itabuna,Cachoeira de Itabuna,05,,-14.45,-39.416667
br,cachoeira de macacos,Cachoeira de Macacos,15,,-19.533333,-44.433333
br,cachoeira de manuel roque,Cachoeira de Manuel Roque,05,,-14.166667,-40.25
br,cachoeira de minas,Cachoeira de Minas,15,,-22.35,-45.783333
br,cachoeira de minas,Cachoeira de Minas,17,,-7.566667,-38.083333
br,cachoeira de sapo,Cachoeira de Sapo,22,,-5.75,-35.916667
br,cachoeira do arari,Cachoeira do Arari,16,,-1.016667,-48.966667
br,cachoeira do bom jesus,Cachoeira do Bom Jesus,26,,-27.433333,-48.45
br,cachoeira do brumado,Cachoeira do Brumado,15,,-20.383333,-43.283333
br,cachoeira do campo,Cachoeira do Campo,15,,-20.333333,-43.666667
br,cachoeira do funil,Cachoeira do Funil,21,,-22.116667,-43.55
br,cachoeira do ilanema,Cachoeira do Ilanema,18,,-25.7908,-49.4561
br,cachoeira do mato,Cachoeira do Mato,05,,-17.4,-40.05
br,cachoeira do pajehu,Cachoeira do Pajehu,15,,-15.95,-41.5
br,cachoeira do pajeu,Cachoeira do Pajeú,15,,-15.95,-41.5
br,cachoeira do repouso,Cachoeira do Repouso,04,,-4.359722,-70.933056
br,cachoeira do roberto,Cachoeira do Roberto,30,,-8.65,-41.133333
br,cachoeira do samuel,Cachoeira do Samuel,24,,-8.75,-63.466667
br,cachoeira do sapateiro,Cachoeira do Sapateiro,20,,-3.92,-42.097778
br,cachoeira do sapo,Cachoeira do Sapo,22,,-5.75,-35.916667
br,cachoeira dos domingos,Cachoeira dos Domingos,18,,-25.579476,-50.745684
br,cachoeira dos indios,Cachoeira dos Índios,17,,-6.933333,-38.683333
br,cachoeira do sul,Cachoeira do Sul,23,74696,-30.033333,-52.9
br,cachoeira dourada,Cachoeira Dourada,15,,-18.5,-49.5
br,cachoeira dourada,Cachoeira Dourada,29,,-18.5,-49.516667
br,cachoeira escura,Cachoeira Escura,15,,-19.3,-42.383333
br,cachoeira funda,Cachoeira Funda,11,,-20.933333,-55.033333
br,cachoeira grande,Cachoeira Grande,02,,-9.05,-37.516667
br,cachoeira grande,Cachoeira Grande,03,,2.15,-51.016667
br,cachoeira grande,Cachoeira Grande,05,,-13.583333,-41.266667
br,cachoeira grande,Cachoeira Grande,06,,-4.6,-41.2
br,cachoeira grande,Cachoeira Grande,30,,-8.483333,-36.166667
br,cachoeira lisa,Cachoeira Lisa,02,,-8.95,-36.05
br,cachoeirao,Cachoeirão,11,,-20.416667,-55.283333
br,cachoeira paulista,Cachoeira Paulista,27,,-22.666667,-45.016667
br,cachoeira pedra,Cachoeira Pedra,11,,-19.966667,-54.366667
br,cachoeira samuel,Cachoeira Samuel,24,,-8.75,-63.466667
br,cachoeiras,Cachoeiras,15,,-22.35,-45.783333
br,cachoeiras,Cachoeiras,17,,-6.983333,-37.266667
br,cachoeiras,Cachoeiras,21,,-22.316667,-41.883333
br,cachoeiras,Cachoeiras,30,,-7.683333,-39.366667
br,cachoeiras de macacu,Cachoeiras de Macacu,21,46178,-22.466667,-42.65
br,cachoeira seca,Cachoeira Seca,27,,-24.747248,-48.116356
br,cachoeira seca,Cachoeira Sêca,30,,-8.1,-36.033333
br,cachoeira torta,Cachoeira Torta,15,,-20.333333,-42.483333
br,cachoeirinha,Cachoeirinha,02,,-9.416667,-37.9
br,cachoeirinha,Cachoeirinha,04,,-3.124167,-60.009167
br,cachoeirinha,Cachoeirinha,05,,-12.65,-38.966667
br,cachoeirinha,Cachoeirinha,06,,-5.05,-40.4
br,cachoeirinha,Cachoeirinha,08,,-18.833333,-40.916667
br,cachoeirinha,Cachoeirinha,11,,-18.166667,-54.25
br,cachoeirinha,Cachoeirinha,13,,-1.616667,-45.1
br,cachoeirinha,Cachoeirinha,14,,-15.566667,-54.466667
br,cachoeirinha,Cachoeirinha,15,,-18.966667,-41.633333
br,cachoeirinha,Cachoeirinha,16,,-5.4,-54.3
br,cachoeirinha,Cachoeirinha,17,,-6.6,-35.533333
br,cachoeirinha,Cachoeirinha,18,,-24.166667,-49.666667
br,cachoeirinha,Cachoeirinha,21,,-21.533333,-41.916667
br,cachoeirinha,Cachoeirinha,23,121086,-29.95,-51.083333
br,cachoeirinha,Cachoeirinha,24,,-10.916667,-65.266667
br,cachoeirinha,Cachoeirinha,25,,2.333333,-60.866667
br,cachoeirinha,Cachoeirinha,27,,-21.666667,-47.533333
br,cachoeirinha,Cachoeirinha,30,12905,-8.483333,-36.233333
br,cachoeirinha,Cachoeirinha,31,,-6.0525,-47.749444
br,cachoeirinha de itauna,Cachoeirinha de Itaúna,08,,-18.833333,-40.916667
br,cachoeirinhas,Cachoeirinhas,15,,-21.95,-46.616667
br,cachoeiro,Cachoeiro,21,,-22.383333,-42.1
br,cachoeiro,Cachoeiro,30,,-8.216667,-36.033333
br,cachoeiro de itapemirim,Cachoeiro de Itapemirim,08,187023,-20.85,-41.1
br,cachoeiro de santa leopoldina,Cachoeiro de Santa Leopoldina,08,,-20.1,-40.533333
br,cachoeiro do itapemirim,Cachoeiro do Itapemirim,08,,-20.85,-41.1
br,cachoeiros,Cachoeiros,21,,-22.383333,-42.1
br,cachoera,Cachoera,06,,-5.6,-40.866667
br,cachoerinha,Cachoerinha,04,,-5.489444,-60.834444
br,cachoerinha,Cachoerinha,15,,-20.15,-41.733333
br,cachorrinho,Cachorrinho,03,,.716667,-51.916667
br,cachorro,Cachorro,14,,-15.25,-56.55
br,cachorro,Cachorro,17,,-6.683333,-37.7
br,cachos,Cachos,15,,-21.483333,-45.416667
br,cachucha,Cachucha,13,,-4.2,-44.516667
br,cacimba,Cacimba,05,,-10.2,-43.633333
br,cacimba,Cacimba,06,,-6.35,-38.733333
br,cacimba,Cacimba,14,,-15.8,-60.2
br,cacimba,Cacimba,17,,-7.05,-38.433333
br,cacimba,Cacimba,20,,-8.083333,-45.25
br,cacimba,Cacimba,30,,-7.983333,-36.016667
br,cacimba cercada,Cacimba Cercada,02,,-9.516667,-37.483333
br,cacimba da baixa,Cacimba da Baixa,22,,-5.078333,-36.492222
br,cacimba dantas,Cacimba Dantas,20,,-5.85,-41.216667
br,cacimba da onca,Cacimba da Onca,06,,-5.483333,-38.65
br,cacimba de areia,Cacimba de Areia,22,,-7.116667,-37.166667
br,cacimba de baixo,Cacimba de Baixo,05,,-9.583333,-39.083333
br,cacimba de baixo,Cacimba de Baixo,22,,-5.946944,-36.936111
br,cacimba de baixo,Cacimba de Baixo,30,,-7.783333,-38.35
br,cacimba de cima,Cacimba de Cima,17,,-7.533333,-36.333333
br,cacimba de cima,Cacimba de Cima,22,,-6.433333,-38.016667
br,cacimba de cima,Cacimba de Cima,30,,-8.9,-36.983333
br,cacimba de dentro,Cacimba de Dentro,17,7447,-6.65,-35.783333
br,cacimba de meio,Cacimba de Meio,17,,-7.55,-36.9
br,cacimba de pedra,Cacimba de Pedra,05,,-12.05,-39.1
br,cacimba de pedra riacho,Cacimba de Pedra Riacho,20,,-5.866667,-41.116667
br,cacimba do maia,Cacimba do Maia,05,,-9.116667,-38.983333
br,cacimba do meio,Cacimba do Meío,06,,-4.966667,-39.116667
br,cacimba do meio,Cacimba do Meio,20,,-6.633333,-41.716667
br,cacimba do meio,Cacimba do Meio,30,,-7.983333,-37.85
br,cacimba funda,Cacimba Funda,06,,-4.860833,-37.451667
br,cacimba grande,Cacimba Grande,20,,-5.85,-41.85
br,cacimba i,Cacimba I,13,,-3.941667,-43.041389
br,cacimba ii,Cacimba II,13,,-3.957222,-43.038333
br,cacimba nova,Cacimba Nova,06,,-5.1,-40.2
br,cacimba nova,Cacimba Nova,17,,-7.766667,-38.166667
br,cacimba nova,Cacimba Nova,30,,-8.116667,-38.8
br,cacimbao,Cacimbão,06,,-5.3,-40.95
br,cacimbao,Cacimbão,20,,-3.746944,-42.265
br,cacimbas,Cacimbas,02,,-9.516667,-37.616667
br,cacimbas,Cacimbas,05,,-12.016667,-37.683333
br,cacimbas,Cacimbas,06,,-2.85,-40.116667
br,cacimbas,Cacimbas,17,,-7.033333,-37.533333
br,cacimbas,Cacimbas,20,,-3.785,-42.008056
br,cacimbas,Cacimbas,21,,-21.65,-41.1
br,cacimbas,Cacimbas,30,,-8.066667,-35.266667
br,cacimbas dos cavalos,Cacimbas dos Cavalos,20,,-6.2,-41.333333
br,cacimba seca,Cacimba Sêca,30,,-9.033333,-38.25
br,cacimba velha,Cacimba Velha,20,,-6.9,-40.616667
br,cacimba velha,Cacimba Velha,22,,-5.1,-37.516667
br,cacimbinha,Cacimbinha,02,,-9.4,-37
br,cacimbinha,Cacimbinha,06,,-3.8,-40.2
br,cacimbinha,Cacimbinha,11,,-21.833333,-55.716667
br,cacimbinha,Cacimbinha,13,,-3.248056,-43.325
br,cacimbinha,Cacimbinha,17,,-6.65,-38.1
br,cacimbinha,Cacimbinha,20,,-7.183333,-41.3
br,cacimbinha,Cacimbinha,22,,-5.907778,-36.336667
br,cacimbinhas,Cacimbinhas,02,4618,-9.4,-37
br,cacimbinhas,Cacimbinhas,11,,-21.833333,-55.716667
br,cacimbinhas,Cacimbinhas,23,,-31.566667,-53.383333
br,cacimbo de dentro,Cacimbo de Dentro,17,,-6.65,-35.783333
br,cacimbu nova,Cacimbu Nova,17,,-6.7,-38.6
br,cacinum,Caçinum,05,,-12.683333,-38.866667
br,cacique doble,Cacique Doble,23,,-27.75,-51.616667
br,cacoal,Cacoal,04,,-4.148056,-60.548889
br,cacoal,Cacoal,13,,-1.983333,-46.216667
br,cacoal,Cacoal,24,55566,-11.5,-61.416667
br,cacoalinho,Cacoalinho,04,,-7.766667,-67.75
br,cacoci,Cacoci,06,,-6.416667,-40.5
br,cacoeirinha,Cacoeirinha,25,,3.033333,-60.433333
br,cacoeiro,Caçoeiro,18,,-25.429826,-48.427331
br,cacoes,Cações,05,,-13,-38.783333
br,cacomera,Cacomera,01,,-8.5,-71.15
br,caconda,Caconda,21,,-21.6,-41.583333
br,caconde,Caconde,27,12434,-21.55,-46.633333
br,cacozinho,Caçozinho,13,,-3.015278,-43.304444
br,cacual,Cacual,24,,-11.5,-61.416667
br,cacu,Caçu,29,10098,-18.616667,-51.066667
br,cacuia,Cacuia,21,,-22.809659,-43.19088
br,cacule,Caculé,05,12143,-14.5,-42.216667
br,cacununga,Caçununga,14,,-16.05,-53.633333
br,cacuria,Cacuriã,04,,-7.383333,-65.2
br,cadeado,Cadeado,23,,-28.566667,-53.816667
br,cadeia,Cadeia,23,,-29.7,-51.683333
br,cadete,Cadete,05,,-12.716667,-39.083333
br,cado bravo,Cado Bravo,05,,-10.733333,-40.566667
br,cadori,Cadori,26,,-27.25,-51.733333
br,cadoz,Cadoz,02,,-9.65,-35.766667
br,cadoz,Cadoz,06,,-3.616667,-41.366667
br,cadoz,Cadoz,13,,-3.444722,-42.723056
br,cadoz,Cadoz,20,,-6.033333,-41.516667
br,cadoz de baixo,Cadoz de Baixo,20,,-3.616667,-41.466667
br,cadozinho,Cadozinho,20,,-6.016667,-41.533333
br,cadu de baixo,Cadu de Baixo,05,,-13.45,-40.05
br,caem,Caém,05,,-11.1,-40.433333
br,caetano,Caetano,05,,-11.05,-39.683333
br,caetano,Caetano,06,,-3.066667,-39.55
br,caetano,Caetano,11,,-20.2,-55.266667
br,caetano,Caetano,13,,-9.566667,-45.85
br,caetano jose,Caetano José,15,,-20.266667,-44.216667
br,caetanopolis,Caetanópolis,15,,-19.3,-44.4
br,caete-acu,Caeté-Açu,05,,-12.716667,-41.483333
br,caete,Caeté,14,,-15.85,-58.283333
br,caete,Caeté,15,33232,-19.9,-43.666667
br,caete,Caeté,16,,-1.75,-48.666667
br,caete,Caetê,18,,-24.033333,-50.45
br,caete,Caeté,21,,-23.022693,-43.508827
br,caete,Caetê,27,,-23.598218,-47.067313
br,caetes,Caetés,15,,-21.813889,-43.263056
br,caetes,Caetés,30,5386,-8.766667,-36.633333
br,caetetuba,Caetetuba,27,,-23.136667,-46.643611
br,caetetu,Caetetú,20,,-5.483333,-42.983333
br,caetetu,Caetetu,27,,-23.5671,-47.8863
br,caetetu,Caetetu,31,,-11.966667,-47.266667
br,caetite,Caetité,05,27188,-14.066667,-42.483333
br,caetuba,Caetuba,27,,-23.136667,-46.643611
br,cafarnaum,Cafarnaum,05,12940,-11.7,-41.466667
br,cafeara,Cafeara,18,,-22.783333,-51.7
br,cafe,Café,08,,-20.883333,-41.566667
br,cafe,Café,15,,-19.066667,-41.866667
br,cafe,Café,18,,-23.890419,-50.185564
br,cafe,Café,27,,-20.45,-47.8
br,cafe,Café,29,,-14.516667,-49.066667
br,cafeeiros,Cafeeiros,18,,-23.75,-53.15
br,cafe jardim,Café Jardim,22,,-5.553611,-37.011111
br,cafelandia,Cafelandia,14,,-16.666667,-53.416667
br,cafelandia,Cafelândia,18,,-24.65,-53.333333
br,cafelandia,Cafelândia,21,,-21.5,-41.933333
br,cafelandia,Cafelândia,27,13777,-21.816667,-49.583333
br,cafelandia do leste matogrossense,Cafelândia do Leste Matogrossense,14,,-16.666667,-53.416667
br,cafemirim,Cafèmirim,15,,-19.316667,-41.983333
br,cafe ralo,Café Ralo,08,,-18.666667,-41
br,cafesal,Cafesal,18,,-23.883333,-53.5
br,cafesopolis,Cafesópolis,27,,-21.9,-49.75
br,cafezal,Cafèzal,13,,-3,-44.983333
br,cafezal,Cafèzal,15,,-21.316667,-46.5
br,cafezal,Cafèzal,16,,-1.55,-48.583333
br,cafezal,Cafèzal,18,,-23.883333,-53.5
br,cafezal,Cafezal,27,,-24.5923,-48.3234
br,cafezopolis,Cafezópolis,27,,-21.9,-49.75
br,cafofa,Cafofa,30,,-8.716667,-36.483333
br,cafongo,Cafongó,05,,-14.5,-39.6
br,cafuca,Cafuca,22,,-5.978333,-36.441111
br,cafufa,Cafufa,05,,-10.233333,-38.033333
br,cafuipe,Cafuipe,23,,-28.25,-54.016667
br,cafunda,Cafundá,21,,-22.9487,-43.488168
br,cafundo,Cafundó,02,,-9.433333,-36.383333
br,cafundo,Cafundó,05,,-11.233333,-42.6
br,cafundo,Cafundó,06,,-5.566667,-39.1
br,cafundo,Cafundó,13,,-3.605,-43.442778
br,cafundo,Cafundó,15,,-15.983333,-46.083333
br,cafundo,Cafundó,17,,-7.383333,-37.7
br,cafundo,Cafundo,20,,-5.85,-43.016667
br,cafundo,Cafundó,23,,-29.616667,-51.5
br,cafundo,Cafundó,27,,-21.616667,-47.716667
br,cafungo,Cafungó,05,,-14.5,-39.6
br,cafuta,Cafutá,05,,-10.233333,-38.033333
br,cagado,Cágado,02,,-8.966667,-37.783333
br,cagado,Cágado,05,,-13.116667,-39.516667
br,cagado,Cágado,13,,-5.883333,-43.966667
br,cagado,Cágado,20,,-3.474167,-42.005
br,cagado,Cágado,30,,-8.933333,-36.516667
br,cagados,Cágados,05,,-13.116667,-39.516667
br,cagados,Cágados,14,,-15.266667,-56.5
br,cagados,Cágados,20,,-5.55,-41.5
br,cahem,Cahem,05,,-11.1,-40.433333
br,cahen,Cahen,05,,-11.1,-40.433333
br,caiabu,Caiabu,27,,-22.016667,-51.233333
br,caiacanga,Caiacanga,26,,-27.716667,-48.583333
br,caiada,Caiada,22,,-6.016667,-35.633333
br,caiada de baixo,Caiada de Baixo,22,,-6.05,-35.683333
br,caiambe,Caiambé,04,,-3.566667,-64.383333
br,caiana,Caiana,13,,-3.835833,-42.829722
br,caiana,Caiana,15,,-20.7,-41.933333
br,caiana,Caiana,17,,-6.816667,-38.016667
br,caiana,Caiana,30,,-8.466667,-36.383333
br,caianana,Caianana,05,,-13.866667,-40.5
br,caiapo,Caiapó,15,,-21.683333,-42.416667
br,caiapo,Caiapó,16,,-8.338056,-50.464444
br,caiaponia,Caiapônia,29,10947,-16.95,-51.816667
br,caiary,Caiary,04,,-6.591667,-62.398889
br,caiaue,Caiaué,04,,-3.05,-65.966667
br,caibate,Caibaté,23,,-28.283333,-54.633333
br,caibe,Caibê,08,,-20.966667,-41.45
br,caibros,Caibros,17,,-6.9,-38.166667
br,cai,Caí,05,,-17,-39.183333
br,cai,Caí,23,,-29.583333,-51.383333
br,caicara,Caiçara,02,,-9.766667,-37.15
br,caicara,Caiçara,04,,-3.2,-64.833333
br,caicara,Caiçara,05,,-12.283333,-42.366667
br,caicara,Caiçara,06,,-2.866667,-40.4
br,caicara,Caiçara,15,,-17.25,-43.033333
br,caicara,Caiçara,16,,-1.15,-51.933333
br,caicara,Caiçara,17,,-6.483333,-36.416667
br,caicara,Caiçara,20,,-3.747222,-42.494444
br,caicara,Caiçara,22,,-5.066667,-36.05
br,caicara,Caiçara,23,,-27.266667,-53.416667
br,caicara,Caiçara,28,,-9.866667,-37.266667
br,caicara,Caiçara,29,,-15.566667,-50.2
br,caicara,Caicara,30,,-7.666667,-37.383333
br,caicara chico,Caicara Chico,17,,-7.4,-38.5
br,caicara de baixo,Caiçara de Baixo,05,,-10.433333,-41.616667
br,caicara de baixo,Caiçara de Baixo,17,,-6.716667,-37.65
br,caicara de baixo,Caicara de Baixo,22,,-5.883333,-35.983333
br,caicara de cima,Caiçara de Cima,17,,-6.766667,-37.616667
br,caicara do rio do vento,Caiçara do Rio do Vento,22,,-5.758889,-35.997778
br,caicara i,Caiçara I,20,,-3.270556,-42.086111
br,caicara ii,Caiçara II,20,,-3.273056,-42.046944
br,caicarinha,Caiçarinha,06,,-4.7,-39.266667
br,caicarinha,Caiçarinha,17,,-6.8,-37.816667
br,caicarinha,Caiçarinha,18,,-23.080059,-52.057597
br,caicarinha,Caicarinha,20,,-6.783333,-41.3
br,caicarinha da penha,Caiçarinha da Penha,30,,-7.833333,-38.55
br,caico,Caicó,17,,-6.733333,-37.5
br,caico,Caicó,22,54935,-6.45,-37.1
br,caida d'agua,Caida d'Agua,31,,-10.95,-47.483333
br,cai de baixo,Caí de Baixo,18,,-25.8816,-49.4394
br,cai de cima,Caí de Cima,18,,-25.856,-49.4545
br,caieira,Caieira,06,,-6.05,-38.916667
br,caieira,Caieira,13,,-4.9,-42.983333
br,caieira,Caieira,17,,-6.816667,-37.816667
br,caieira,Caieira,26,,-27.416667,-48.583333
br,caieira,Caieira,27,,-23.6439,-47.8452
br,caieira,Caieira,30,,-8.133333,-38.983333
br,caieiras,Caieiras,13,,-4.9,-42.983333
br,caieiras,Caieiras,18,,-25.8631,-48.5654
br,caieiras,Caieiras,27,87199,-23.36193,-46.740681
br,caieiras de sao pedro,Caieiras de São Pedro,22,,-5.451389,-36.587778
br,caietra,Caietra,15,,-21.216667,-43.916667
br,caimbombo,Caimbombo,05,,-12.7,-42.25
br,cainema,Cainema,05,,-13.1,-38.95
br,caioba,Caiobá,18,,-25.8395,-48.5406
br,caioca,Caioca,06,,-3.6,-40.25
br,caio prado,Caio Prado,06,,-4.666667,-38.966667
br,caipe,Caípe,05,,-12.7,-38.6
br,caipe,Caipê,08,,-20.966667,-41.45
br,caipira,Caipira,15,,-20.516667,-45.033333
br,caipora,Caipora,01,,-9.2,-72.683333
br,caipora,Caipora,30,,-8.316667,-40.566667
br,caipu,Caipu,06,,-6.633333,-39.316667
br,cairara,Cairara,06,,-5.15,-39.816667
br,cairari,Cairari,16,,-2.55,-49.116667
br,cairary,Cairary,16,,-2.55,-49.116667
br,caire,Cairé,23,,-28.116667,-53.083333
br,cairiri,Cairiri,31,,-11.888611,-49.073333
br,cairu,Cairu,04,,-7.233333,-65
br,cairu,Cairu,05,,-12.866667,-38.8
br,caisara,Caisara,04,,-4.274444,-59.409167
br,caissara,Caissara,04,,-3.2,-64.833333
br,caita,Caitá,18,,-25.783333,-50.583333
br,caitau,Caitaú,04,,-3.483333,-66.066667
br,caite,Caité,18,,-25.066667,-49.316667
br,caitete,Caiteté,05,,-14.066667,-42.483333
br,caitetu,Caitetu,02,,-9.716667,-37.266667
br,caitetu,Caitetu,05,,-9.866667,-42.583333
br,caitetu,Caitetú,20,,-5.483333,-42.983333
br,caiteuara,Caiteuara,13,,-1.683333,-45.5
br,caitinga,Caitinga,05,,-12.983333,-39.133333
br,caititu,Caititu,05,,-12.7,-41.766667
br,caititu,Caititu,06,,-6.116667,-39.616667
br,caititu,Caititu,17,,-7.15,-38.266667
br,caititu,Caititu,20,,-5.483333,-42.983333
br,caititu,Caititu,27,,-20.7,-47.366667
br,caititu,Caititu,30,,-7.966667,-38.7
br,caititu,Caititu,31,,-11.966667,-47.266667
br,caiua,Caiuá,27,,-21.883333,-51.966667
br,caiuas,Caiuás,11,,-21.8,-54.55
br,caiuba,Caiubá,18,,-25.8473,-48.5458
br,caiubi,Caiubi,05,,-15.916667,-39.883333
br,caiubi,Caiubi,27,,-22.741111,-47.471944
br,caiucara,Caiuçara,27,,-23.141667,-46.69
br,caixa da fartura,Caixa da Fartura,05,,-13.5,-39.966667
br,caixa d'agua,Caixa d'Agua,06,,-4.2,-40.466667
br,caixa d'agua,Caixa d'Agua,23,,-29.5,-54.633333
br,caixa furada,Caixa Furada,14,,-14.5,-56.183333
br,caixao,Caixão,02,,-9.3,-38.2
br,caixa prego,Caixa Prego,05,,-12.816667,-41.233333
br,caixa prego,Caixa Prego,18,,-25.466667,-52.333333
br,caixa-pregos,Caixa-Pregos,05,,-13.116667,-38.783333
br,caja,Cajá,02,,-9.666667,-36.216667
br,caja,Cajá,05,,-12.033333,-40.183333
br,caja,Cajá,06,,-5.933333,-38.6
br,caja,Cajá,17,,-7.183333,-35.383333
br,caja,Cajá,30,,-7.616667,-37.433333
br,caja de baixo,Caja de Baixo,30,,-7.516667,-37.6
br,cajaiba,Cajaíba,05,,-12.083333,-40.55
br,cajaibas,Cajaíbas,05,,-13.45,-39.1
br,cajamar,Cajamar,27,56684,-23.35,-46.883333
br,cajapio,Cajapió,22,,-2.966667,-44.8
br,cajari,Cajari,04,,-6.591667,-62.398889
br,cajari,Cajari,13,,-3.333333,-45.016667
br,cajati,Cajati,18,,-25.2,-53.166667
br,cajati,Cajati,27,27566,-24.728502,-48.110541
br,cajazeira,Cajàzeira,02,,-9.383333,-36.45
br,cajazeira,Cajàzeira,05,,-12.316667,-40.8
br,cajazeira,Cajazeira,06,,-3.3,-40.266667
br,cajazeira,Cajàzeira,13,,-5.05,-43.45
br,cajazeira,Cajázeira,17,,-7.2,-37.783333
br,cajazeira,Cajazeira,20,,-7.15,-41.6
br,cajazeira,Cajazeira,22,,-5.616667,-35.633333
br,cajazeira,Cajàzeira,30,,-8.116667,-36.4
br,cajazeiras,Cajàzeiras,05,,-10.766667,-38.433333
br,cajazeiras,Cajazeiras,06,,-3.4,-39.4
br,cajazeiras,Cajazeiras,13,,-3.117222,-42.376389
br,cajazeiras,Cajàzeiras,17,43914,-6.9,-38.566667
br,cajazeiras,Cajazeiras,20,,-3.666389,-42.586944
br,cajazeiras,Cajazeiras,22,,-6.483333,-38.1
br,cajazeiras dos lourencos,Cajàzeiras dos Lourenços,17,,-6.85,-37.916667
br,cajazeiro,Cajàzeiro,06,,-5.533333,-39.5
br,cajazeiro,Cajázeiro,17,,-7.283333,-37.7
br,cajazeiros,Cajazeiros,06,,-4.4,-40.366667
br,cajaziras,Cajaziras,06,,-7.15,-39.766667
br,cajiru,Cajiru,05,,-12.816667,-38.25
br,cajobi,Cajobi,27,,-20.883333,-48.816667
br,cajoby,Cajoby,27,,-20.883333,-48.816667
br,cajoeira,Cajoeira,16,,-5.616667,-54.45
br,cajual,Cajual,25,,-3.133333,-60.55
br,cajuapara,Cajuapara,16,,-4.283333,-47.416667
br,cajubeira,Cajubeira,20,,-3.64,-42.5525
br,cajubi,Cajubi,08,,-17.916667,-40.283333
br,cajubim,Cajubim,04,,-6.683333,-60.35
br,cajubim,Cajubim,16,,-.633333,-47.716667
br,cajubim,Cajubim,24,,-9.45,-62.633333
br,caju,Caju,04,,-3.395,-63.458889
br,caju,Caju,05,,-12.133333,-39.25
br,caju,Caju,13,,-6.066667,-43.183333
br,caju,Caju,14,,-14.366667,-56.433333
br,caju,Caju,15,,-16.35,-41.316667
br,caju,Caju,16,,-1.633333,-47.8
br,caju,Caju,21,,-22.877614,-43.21363
br,caju,Caju,25,,3.55,-59.883333
br,cajueira,Cajueira,16,,-7.8,-49.933333
br,cajueiras,Cajueiras,20,,-2.928889,-41.336111
br,cajueirinho,Cajueirinho,06,,-3.066667,-40.466667
br,cajueiro,Cajueiro,02,15063,-9.716667,-36.45
br,cajueiro,Cajueiro,05,,-10.833333,-39.433333
br,cajueiro,Cajueiro,06,,-3.866667,-40.383333
br,cajueiro,Cajueiro,13,,-2.833333,-46.566667
br,cajueiro,Cajueiro,16,,-3.233333,-51.833333
br,cajueiro,Cajueiro,17,,-7.7,-38.116667
br,cajueiro,Cajueiro,20,,-2.928889,-41.336111
br,cajueiro,Cajueiro,22,,-5.15,-35.533333
br,cajueiro,Cajueiro,24,,-10.85,-62.966667
br,cajueiro,Cajueiro,28,,-10.366667,-37.266667
br,cajueiro,Cajueiro,30,,-7.633333,-37.766667
br,cajueiro,Cajueiro,31,,-8.066667,-48.083333
br,cajueiro da estrema,Cajueiro da Estrema,05,,-17.833333,-39.266667
br,cajueiro do boi,Cajueiro do Boi,06,,-2.966667,-40.033333
br,cajueiro do sapucaia,Cajueiro do Sapucaia,02,,-9.75,-36.266667
br,cajueiros,Cajueiros,06,,-5.283333,-38.166667
br,cajui,Cajuí,05,,-10.05,-42.216667
br,cajui,Cajuí,13,,-3.715278,-43.458333
br,cajui,Cajui,17,,-6.6,-38.6
br,cajui,Cajuí,20,,-3.8725,-42.071944
br,cajuri,Cajuri,15,,-20.8,-42.8
br,cajuru,Cajuru,15,,-20.783333,-44.35
br,cajuru,Cajuru,23,,-29.25,-51.416667
br,cajuru,Cajuru,27,19540,-21.283333,-47.3
br,cajuru da cebolla,Cajuru da Cebolla,15,,-20.783333,-44.35
br,cajuru do sul,Cajuru do Sul,27,,-23.4,-47.383333
br,cajury,Cajury,15,,-20.8,-42.8
br,cajutino,Cajutino,04,,-.25,-66.766667
br,calabaca,Calabaça,06,,-6.816667,-39.233333
br,calabar,Calabar,13,,-3.354444,-43.421944
br,calabouco,Calabouço,13,,-1.866667,-44.7
br,calafibre,Calafibre,15,,-20.883333,-47.083333
br,calama,Calama,24,,-8.05,-62.883333
br,calamaque,Calamaque,04,,-.25,-64.066667
br,calambao,Calambão,15,,-20.766667,-43.2
br,calambau,Calambau,15,,-20.583333,-45.05
br,calanaque,Calanaque,04,,-.25,-64.066667
br,calcada,Calçada,05,,-12.083333,-38.033333
br,calcada,Calçada,06,,-6.066667,-40.716667
br,calcadinha,Calçadinha,18,,-24.8513,-48.5933
br,calcado,Calçado,30,,-8.7,-36.333333
br,calcados,Calçados,13,,-3.639167,-42.718333
br,cal,Cal,20,,-7.216667,-41.783333
br,calcamento,Calçamento,15,,-15.45,-44.466667
br,calcere,Calcere,05,,-14.633333,-42.816667
br,calc. muquy,Calç. Muquy,08,,-20.983333,-41.533333
br,calcoene,Calçoene,03,,2.5,-50.95
br,caldas,Caldas,06,,-7.4,-39.383333
br,caldas,Caldas,15,,-21.933333,-46.383333
br,caldas novas,Caldas Novas,29,63681,-17.75,-48.633333
br,caldeira,Caldeira,04,,-3.25,-60.25
br,caldeira,Caldeirã,20,,-8.366667,-42.6
br,caldeirao,Caldeirão,02,,-9.716667,-36.45
br,caldeirao,Caldeirão,04,,-3.25,-60.25
br,caldeirao,Caldeirão,05,,-10.2,-39.266667
br,caldeirao,Caldeirão,06,,-2.95,-40.316667
br,caldeirao,Caldeirão,13,,-3.675278,-43.448056
br,caldeirao,Caldeirão,17,,-6.966667,-38.533333
br,caldeirao,Caldeirão,20,,-4.316667,-41.716667
br,caldeirao,Caldeirão,22,,-5.666667,-35.766667
br,caldeirao,Caldeirão,24,,-9.316667,-64.583333
br,caldeirao,Caldeirão,30,,-7.733333,-37.883333
br,caldeirao da cruz,Caldeiraõ da Cruz,20,,-5.4,-41.266667
br,caldeirao da onca,Caldeirão da Onça,05,,-12.466667,-40.533333
br,caldeirao das queimadas,Caldeirão das Queimadas,05,,-13.6,-40.233333
br,caldeirao de baixo,Caldeirão de Baixo,13,,-3.961667,-43.345833
br,caldeirao de cima,Caldeirão de Cima,13,,-3.962222,-43.364722
br,caldeirao do bento,Caldeirão do Bento,05,,-12.633333,-40.266667
br,caldeirao do bravo gordo,Caldeirão do Bravo Gordo,20,,-6.266667,-41.233333
br,caldeirao do cachorro,Caldeirão do Cachorro,20,,-6.2,-40.916667
br,caldeirao do mato,Caldeirão do Mato,05,,-12.016667,-39.116667
br,caldeirao do onca,Caldeirão do Onça,05,,-12.466667,-40.533333
br,caldeirao do saco,Caldeirão do Saco,20,,-6.583333,-41.666667
br,caldeirao fejo,Caldeirão Fejo,20,,-6.333333,-40.8
br,caldeirao grande,Caldeirão Grande,05,,-11.016667,-40.283333
br,caldeirao grande,Caldeirão Grande,20,,-7.35,-40.633333
br,caldeiraosinho,Caldeirãosinho,20,,-9.166667,-42.983333
br,caldeirao verde,Caldeirão Verde,05,,-13.816667,-40.083333
br,caldeiraozinho,Caldeirãozinho,05,,-12.4,-40.833333
br,caldeiraozinho,Caldeirãozinho,20,,-7.316667,-40.666667
br,caldeiraozinho,Caldeirãozinho,30,,-7.766667,-39.65
br,caldeiras,Caldeiras,05,,-13.616667,-42.6
br,caldeiroes,Caldeirões,17,,-6.433333,-37.85
br,caldeiroes,Caldeirões,20,,-7.1,-41.5
br,caldeiroes,Caldeirões,30,,-9.233333,-36.583333
br,caldeiroes de cima,Caldeirões de Cima,02,,-9.333333,-36.65
br,calderao,Calderão,24,,-9.316667,-64.583333
br,calderoes,Calderões,30,,-9.233333,-36.583333
br,calembar,Calembar,05,,-12.666667,-38.85
br,calheiros,Calheiros,21,,-21.016667,-41.733333
br,california,Califórnia,05,,-13.883333,-39.616667
br,california,California,06,,-3.383333,-40.033333
br,california,Califórnia,08,,-20.216667,-40.6
br,california,Califórnia,13,,-3.216667,-46.766667
br,california,California,17,,-7.533333,-38.166667
br,california,Califórnia,18,,-23.633333,-51.366667
br,california,Califórnia,21,,-22.890465,-43.585594
br,california,Califórnia,22,,-5.783333,-35.4
br,california,Califórnia,27,,-21.65,-47.483333
br,calilea,Calilea,04,,-7.35,-62.933333
br,calisto,Calisto,04,,-4.680833,-59.438611
br,calisto,Calisto,31,,-11.466667,-46.75
br,calixto,Calixto,05,,-13.583333,-38.966667
br,calixto,Calixto,15,,-19.25,-41.15
br,calixto,Calixto,31,,-11.466667,-46.75
br,calmom,Calmom,26,,-26.6,-51.083333
br,calmon,Calmon,04,,-7.633333,-68.933333
br,calmon,Calmon,26,,-26.6,-51.083333
br,calogeras,Calógeras,18,,-23.983333,-49.816667
br,calogi,Calogi,08,,-20.066667,-40.433333
br,caloji,Caloji,08,,-20.066667,-40.433333
br,caloji,Caloji,15,,-20.933333,-45.066667
br,calpela do alto alegre,Calpéla do Alto Alégre,05,,-11.666667,-39.85
br,calsoene,Calsoene,03,,2.5,-50.95
br,caluc,Caluc,06,,-5.033333,-38.766667
br,calumbi,Calumbi,05,,-10.05,-39.15
br,calumbi,Calumbi,06,,-6.883333,-38.966667
br,calumbi,Calumbi,13,,-3.902222,-43.036389
br,calumbi,Calumbi,17,,-7.933333,-38.183333
br,calumbi,Calumbi,20,,-3.607778,-42.060278
br,calumbi,Calumbi,30,,-7.683333,-39.666667
br,calutino,Calutino,04,,-.25,-66.766667
br,calvario,Calvário,02,,-9.533333,-36.833333
br,camaca,Camacá,05,,-15.4,-39.5
br,camacagi,Camaçagi,05,,-10.1,-42.55
br,camacandi,Camaçandi,05,,-13.2,-39
br,camacaoca,Camacaoca,13,,-3.666667,-45.633333
br,camacari,Camaçari,02,,-10.116667,-36.283333
br,camacari,Camaçari,05,188763,-12.483333,-38.216667
br,camacari,Camaçari,13,,-5.266667,-47.366667
br,camacary,Camacary,05,,-12.683333,-38.333333
br,camacho,Camacho,15,,-20.633333,-45.166667
br,camacho,Camacho,26,,-28.616667,-48.866667
br,camacho,Camacho,27,,-22.333333,-47.216667
br,camacu,Camaçu,02,,-9.966667,-36.366667
br,cama galo,Cama Galo,06,,-6.4,-38.9
br,camaipi,Camaipi,03,,.016667,-51.933333
br,camalau,Camalaú,17,,-6.983333,-34.833333
br,camaleao,Camaleão,25,,.923056,-61.685833
br,camaleoes,Camaleões,03,,1.166667,-50.466667
br,camamu,Camamu,05,13924,-13.95,-39.116667
br,camanducaia,Camanducaia,15,17823,-22.766667,-46.15
br,camapoa,Camapoã,11,,-19.5,-54.083333
br,camapoan,Camapoan,11,,-19.5,-54.083333
br,camapua,Camapuã,11,11119,-19.5,-54.083333
br,camapua,Camapuã,15,,-20.533333,-43.983333
br,camapuan,Camapuan,11,,-19.5,-54.083333
br,camapuan,Camapuan,15,,-20.533333,-43.983333
br,camaqua,Camaquã,23,53170,-30.85,-51.816667
br,camaquam,Camaquam,11,,-19.5,-54.083333
br,camara,Camará,04,,-3.915833,-62.643056
br,camara,Camará,05,,-12.583333,-38.483333
br,camara,Camará,06,,-3.616667,-39.616667
br,camara,Camará,08,,-20.9,-41.266667
br,camara,Camará,16,,-3.033333,-55.3
br,camara,Câmara,21,,-21.4,-41.383333
br,camarada,Camarada,20,,-6.916667,-41.433333
br,camarada,Camarada,30,,-8.033333,-35.8
br,camaragibe,Camaragibe,30,148966,-8.016667,-34.966667
br,camarajibe,Camarajibe,30,,-8.016667,-34.966667
br,camarao,Camarão,02,,-9.216667,-37.766667
br,camarao,Camarão,04,,1.3,-68.716667
br,camarao,Camarão,22,,-5.942222,-36.515
br,camaratuba,Camaratuba,02,,-9.35,-36.05
br,camaratuba,Camaratuba,17,,-6.65,-35.133333
br,camaratuba,Camaratuba,30,,-8.416667,-35.85
br,camarazal,Camarazal,17,,-7.033333,-35.466667
br,camarazinho,Camarazinho,04,,-3.91,-62.691111
br,camargo,Camargo,18,,-23.35,-53.716667
br,camargo,Camargo,23,,-28.583333,-52.2
br,camargo,Camargo,27,,-20.833333,-48.75
br,camargo,Camargo,29,,-17.7,-52.3
br,camargo martins,Camargo Martins,29,,-15.8,-48.3
br,camargos,Camargos,15,,-20.266667,-43.4
br,camargos,Camargos,27,,-23.287796,-47.920502
br,camarinha,Camarinha,30,,-7.766667,-39.116667
br,camarinha d'agua,Camarinha d'Agua,30,,-7.933333,-36.533333
br,camarua,Camaruã,04,,-5.765,-64.394167
br,camaruji,Camaruji,05,,-13.516667,-39.1
br,camassandi,Camassandi,05,,-13.2,-39
br,camassari,Camassari,05,,-10.1,-42.55
br,camassary,Camassary,05,,-12.683333,-38.333333
br,camatei,Camateí,05,,-14.566667,-42.95
br,camaubau,Camaubau,22,,-5.25,-35.55
br,camauma,Camauma,04,,-5.449722,-62.853611
br,cambaiaba,Cambaiaba,21,,-21.783333,-41.25
br,cambajuva,Cambajuva,26,,-28.333333,-49.633333
br,cambaquara,Cambaquara,27,,-23.9,-45.45
br,cambara,Cambará,14,,-16.233333,-57.583333
br,cambara,Cambará,18,21380,-23.0411,-50.0741
br,cambara,Cambará,23,,-29.033333,-50.15
br,cambara,Cambará,26,,-26.910071,-49.805128
br,cambara,Cambará,27,,-21.716667,-49.016667
br,cambaras,Cambaras,14,,-16.55,-57.85
br,cambaratiba,Cambaratiba,27,,-21.716667,-49.016667
br,cambarazinho,Cambaràzinho,14,,-17.25,-57.2
br,cambaru,Cambarú,18,,-25.446563,-48.567797
br,cambauba,Cambaúba,31,,-10.833333,-47.516667
br,cambaubas,Cambaúbas,31,,-10.833333,-47.516667
br,cambe,Cambé,18,91081,-23.233333,-51.25
br,cambiasca,Cambiasca,21,,-21.716667,-41.933333
br,cambira,Cambira,18,,-23.6,-51.566667
br,cambirela,Cambirela,26,,-27.683333,-48.766667
br,cambito,Cambito,17,,-6.6,-38.516667
br,cambitos,Cambitos,30,,-7.45,-37.15
br,cambiu,Cambiú,18,,-23.864235,-50.417751
br,camboa,Camboa,06,,-3.4,-39.116667
br,camboapina,Camboapina,08,,-20.433333,-40.433333
br,camboata,Camboatá,05,,-12.4,-38.2
br,camboata,Camboatá,13,,-3.744444,-42.794167
br,camboata,Camboatá,28,,-10.683333,-37.216667
br,camboeiro,Camboeiro,20,,-6.466667,-41.783333
br,cambogue,Camboguê,05,,-12.383333,-37.966667
br,camboim,Camboim,02,,-9.366667,-36.033333
br,camboim de cima,Camboim de Cima,02,,-9.366667,-36.033333
br,camboriu,Camboriú,26,49101,-26.983333,-48.633333
br,cambota,Cambota,06,,-2.966667,-40.316667
br,cambota,Cambotá,13,,-3.448333,-42.7925
br,cambota,Cambota,21,,-22.214444,-43.683056
br,cambotas,Cambotas,13,,-3.133333,-42.75
br,cambotinha,Cambotinha,06,,-2.983333,-40.283333
br,cambuata,Cambuatá,28,,-10.683333,-37.216667
br,cambuca,Cambucá,30,,-7.833333,-35.883333
br,cambucas,Cambucás,05,,-13.933333,-41.15
br,cambuci,Cambuci,21,9395,-21.566667,-41.916667
br,cambucy,Cambucy,21,,-21.566667,-41.916667
br,cambuhy,Cambuhy,15,,-22.616667,-46.066667
br,cambui,Cambuí,05,,-10.166667,-43.283333
br,cambui,Cambuí,13,,-9.173889,-46.2
br,cambui,Cambuí,15,21247,-22.616667,-46.066667
br,cambui,Cambuí,18,,-23.864235,-50.417751
br,cambuii,Cambuíí,18,,-23.864235,-50.417751
br,cambuis,Cambuís,05,,-12.133333,-37.983333
br,cambuizal,Cambuizal,05,,-12.333333,-39.416667
br,cambuquira,Cambuquira,15,10116,-21.85,-45.3
br,cambuquira,Cambuquira,20,,-7.733333,-43.766667
br,camburi,Camburi,08,,-20.266667,-40.266667
br,camela,Camela,30,,-8.516667,-35.116667
br,cameleoes,Cameleões,03,,1.166667,-50.466667
br,camelia,Camélia,15,,-19.733333,-50.183333
br,camelleira,Camelleira,30,,-8.583333,-35.4
br,camelo,Camelo,22,,-5.682778,-36.980833
br,cameta,Cametá,04,,-6.166667,-64.233333
br,cameta,Cametá,13,,-8.983333,-46.4
br,cameta,Cametá,16,45202,-2.25,-49.5
br,cameta,Cametá,17,,-6.283333,-37.633333
br,camilo pereira,Camilo Pereira,15,,-16.85,-40.783333
br,caminas dos konkl,Caminas dos Konkl,26,,-26.1458,-49.9506
br,caminho alto,Caminho Alto,26,,-26.961235,-49.823244
br,caminho das areias,Caminho das Areias,26,,-26.9234,-49.218
br,caminho da serra,Caminho da Serra,21,,-22.839535,-43.518093
br,caminho do anum,Caminho do Anum,26,,-26.92689,-49.766712
br,caminho do job,Caminho do Job,21,,-22.819653,-43.380708
br,caminho do meio,Caminho do Meio,26,,-27.007,-49.5735
br,caminho do morro,Caminho do Morro,26,,-26.9609,-49.7951
br,caminho do pinhal,Caminho do Pinhal,26,,-26.985,-49.8624
br,caminho do posto,Caminho do Posto,26,,-26.965865,-49.689622
br,caminho dos cacadores,Caminho dos Caçadores,26,,-26.923717,-49.646533
br,caminho novo,Caminho Novo,13,,-3.8475,-42.766944
br,caminho novo,Caminho Nôvo,20,,-5.016667,-42.633333
br,caminho peroba,Caminho Peroba,26,,-27.1228,-49.7331
br,caminho tamandua,Caminho Tamanduá,26,,-27.0226,-49.6552
br,camiranga,Camiranga,16,,-1.8,-46.283333
br,camirim,Camirim,05,,-10.9,-41.6
br,camisa,Camisa,05,,-13.916667,-41.716667
br,camisao,Camisão,05,,-12.166667,-39.733333
br,camisao,Camisão,11,,-20.45,-55.716667
br,camobi,Camobi,23,,-29.7,-53.7
br,camoci,Camoci,20,,-2.916667,-41.55
br,camocim,Camocim,06,44482,-2.89,-40.846111
br,camocim,Camocim,30,,-8.35,-35.766667
br,camocim de sao felix,Camocim de São Félix,30,10270,-8.35,-35.766667
br,camocituba,Camocituba,30,,-8.35,-35.766667
br,camocochico,Camocochico,05,,-14.533333,-42.283333
br,camolengue,Camolengue,05,,-13.566667,-40.466667
br,camoroa,Camoroá,06,,-5.633333,-39.916667
br,campana,Campana,04,,-5.866111,-61.693611
br,campanario,Campanário,11,,-22.8,-55.05
br,campanario,Campanário,15,,-18.25,-41.733333
br,campanha,Campanha,11,,-21.1,-54.3
br,campanha,Campanha,13,,-3.483333,-44.45
br,campanha,Campanha,15,12507,-21.833333,-45.4
br,campanha,Campanha,21,,-22.305983,-42.80681
br,campeche,Campeche,26,,-27.666667,-48.483333
br,campeira,Campeira,13,,-9.492222,-45.931389
br,campeiro,Campeiro,11,,-21.183333,-54.75
br,campeiro,Campeiro,13,,-9.316667,-45.95
br,campelina,Campelina,05,,-13.65,-40.35
br,campelo,Campelo,13,,-4.433333,-44.7
br,campelo,Campelo,21,,-21.483333,-42.25
br,campeste,Campeste,06,,-3.35,-39.283333
br,campestre,Campestre,02,,-8.85,-35.583333
br,campestre,Campestre,05,,-12.35,-41.8
br,campestre,Campestre,06,,-4.283333,-37.966667
br,campestre,Campestre,11,,-22.216667,-56.1
br,campestre,Campestre,13,,-3.536667,-43.134722
br,campestre,Campestre,15,11802,-21.716667,-46.25
br,campestre,Campestre,17,,-7.416667,-37.716667
br,campestre,Campestre,18,,-25.05,-49.1
br,campestre,Campestre,20,,-3.826111,-42.29
br,campestre,Campestre,22,,-5.374722,-36.276111
br,campestre,Campestre,23,,-30.366667,-55.716667
br,campestre,Campestre,27,,-20.633333,-47.183333
br,campestre do biskai,Campestre do Biskai,18,,-25.7817,-49.3909
br,campestre dos borges,Campestre dos Borges,18,,-25.521053,-50.288722
br,campestre dos vieiras,Campestre dos Vieiras,18,,-25.513287,-50.295578
br,campestre grande,Campestre Grande,31,,-7.416667,-48.4
br,campestre i,Campestre I,13,,-3.8025,-42.780833
br,campestre ii,Campestre II,13,,-3.868611,-42.780278
br,campestrinho,Campestrinho,27,,-21.716667,-46.65
br,campimho,Campimho,30,,-7.783333,-37.633333
br,campina alta,Campina Alta,18,,-24.516667,-50.65
br,campina bonita,Campina Bonita,18,,-25.183333,-51.533333
br,campina,Campina,04,,-4.226111,-66.442778
br,campina,Campina,05,,-12.333333,-40.816667
br,campina,Campina,06,,-3.333333,-39.9
br,campina,Campina,13,,-7.65,-46.566667
br,campina,Campina,14,,-14.816667,-56.416667
br,campina,Campina,16,,-5.8,-48.333333
br,campina,Campina,17,,-6.766667,-38.416667
br,campina,Campina,18,,-24.3,-52.4
br,campina,Campina,23,,-27.983333,-54.85
br,campina,Campina,27,,-23.504146,-47.646744
br,campina,Campina,30,,-7.866667,-38.65
br,campina da lagoa,Campina da Lagoa,18,,-24.55,-52.716667
br,campina de pupos,Campina de Pupos,18,,-24.233333,-50.716667
br,campina do amoral,Campina do Amoral,18,,-24.3,-52.4
br,campina do monte alegre,Campina do Monte Alegre,27,,-23.583333,-48.466667
br,campina do norte,Campina do Norte,04,,-4.205556,-70.685
br,campina do paiol velho,Campina do Paiol Velho,18,,-25.4837,-50.5546
br,campina dos gomes,Campina dos Gomes,18,,-23.966667,-51.183333
br,campina dos indios,Campina dos Índios,18,,-25.966667,-52.333333
br,campina dos pupos,Campina dos Pupos,18,,-24.233333,-50.716667
br,campina do veado,Campina do Veado,27,,-24.116667,-48.9
br,campina grande,Campina Grande,03,,.183333,-51.116667
br,campina grande,Campina Grande,17,348943,-7.216667,-35.883333
br,campina grande,Campina Grande,18,,-25.3,-49.083333
br,campina grande do sul,Campina Grande do Sul,18,31112,-25.3,-49.083333
br,campinarama,Campinarama,05,,-15.25,-40.566667
br,campinarana,Campinarana,05,,-15.25,-40.566667
br,campinas belas,Campinas Belas,18,,-24.833333,-50.916667
br,campinas,Campinas,02,,-9.466667,-36.15
br,campinas,Campinas,04,,-3.283333,-61.1
br,campinas,Campinas,05,,-11.6,-41.033333
br,campinas,Campinas,06,,-5.183333,-39.083333
br,campinas,Campinas,11,,-19.016667,-56.35
br,campinas,Campinas,15,,-17.85,-43.366667
br,campinas,Campinas,16,,-5.8,-48.333333
br,campinas,Campinas,18,,-25.367941,-50.648109
br,campinas,Campinas,21,,-22.166667,-42.683333
br,campinas,Campinas,23,,-27.716667,-52.616667
br,campinas,Campinas,26,,-28.933333,-49.483333
br,campinas,Campinas,27,1031573,-22.9,-47.083333
br,campinas,Campinas,29,,-14.333333,-49.133333
br,campinas,Campinas,31,,-11.55,-47.316667
br,campinas do dias,Campinas do Dias,18,,-25.839744,-49.930653
br,campinas do piaui,Campinas do Piauí,20,,-7.65,-41.866667
br,campinas dos andrade,Campinas dos Andrade,18,,-26.0628,-49.8081
br,campinas do sul,Campinas do Sul,23,,-27.716667,-52.616667
br,campina verde,Campina Verde,15,13137,-19.516667,-49.466667
br,campina verde,Campina Verde,31,,-9.9325,-48.938333
br,campineira,Campineira,20,,-6.333333,-42.35
br,campineira i,Campineira I,13,,-3.43,-43.035278
br,camping das palmeiras,Camping das Palmeiras,26,,-26.7206,-48.6834
br,campinhas,Campinhas,05,,-11.6,-41.033333
br,campinho,Campinho,02,,-9.283333,-38.016667
br,campinho,Campinho,05,,-12.75,-41.1
br,campinho,Campinho,08,,-20.15,-40.283333
br,campinho,Campinho,13,,-1.716667,-46.216667
br,campinho,Campinho,16,,-1.533333,-52.3
br,campinho,Campinho,18,,-23.772678,-50.281099
br,campinho,Campinho,21,,-22.885594,-43.349477
br,campinho,Campinho,27,,-20.7,-47.716667
br,campinho,Campinho,30,,-7.816667,-39.1
br,campinho dos cavalos,Campinho dos Cavalos,20,,-3.951111,-42.576389
br,campinhos,Campinhos,05,,-12.416667,-38.783333
br,campinhos,Campinhos,30,,-8.933333,-38.516667
br,campininha,Campininha,27,,-23.23433,-47.878477
br,campinorte,Campinorte,29,,-14.333333,-49.133333
br,campins,Campins,20,,-6.183333,-41.8
br,campista,Campista,27,,-22.673911,-45.594034
br,campo alegra,Campo Alegra,31,,-10.7,-47.466667
br,campo alegre,Campo Alegre,02,16786,-9.8,-36.35
br,campo alegre,Campo Alegre,04,,-5.65,-61.033333
br,campo alegre,Campo Alegre,05,,-11.966667,-39.883333
br,campo alegre,Campo Alegre,06,,-5.65,-38.65
br,campo alegre,Campo Alegre,11,,-18.5,-56.5
br,campo alegre,Campo Alegre,13,,-5.283333,-46.366667
br,campo alegre,Campo Alegre,14,,-15.6,-56.6
br,campo alegre,Campo Alegre,15,,-15.583333,-44.1
br,campo alegre,Campo Alegre,18,,-23.368766,-49.822309
br,campo alegre,Campo Alegre,20,,-6.633333,-42.533333
br,campo alegre,Campo Alegre,21,,-22.2939,-44.4824
br,campo alegre,Campo Alegre,23,,-27.633333,-54.633333
br,campo alegre,Campo Alegre,26,,-26.1928,-49.2532
br,campo alegre,Campo Alegre,27,,-20.783333,-47.283333
br,campo alegre,Campo Alegre,29,,-18.566667,-51.95
br,campo alegre,Campo Alegre,30,,-7.816667,-38.933333
br,campo alegre,Campo Alegre,31,,-10.7,-47.466667
br,campo alegre de goias,Campo Alegre de Goiás,29,,-17.65,-47.75
br,campo alegre do mosquito,Campo Alegre do Mosquito,02,,-9.8,-36.35
br,campo alto,Campo Alto,26,,-26.75,-50.4
br,campo alto,Campo Alto,27,,-22.033333,-47.333333
br,campo arado,Campo Arado,14,,-15.766667,-55.4
br,campo azul,Campo Azul,15,,-16.504167,-44.808889
br,campo bello,Campo Bello,06,,-4.1,-39.05
br,campo bello,Campo Bello,15,,-20.883333,-45.266667
br,campo bello,Campo Bello,21,,-22.5,-44.566667
br,campo bello,Campo Bello,26,,-27.9,-50.75
br,campo belo,Campo Belo,03,,1.25,-50.8
br,campo belo,Campo Belo,15,,-16.504167,-44.808889
br,campo belo,Campo Belo,21,,-22.5,-44.566667
br,campo belo,Campo Belo,26,,-27.9,-50.75
br,campo belo do sul,Campo Belo do Sul,26,,-27.9,-50.75
br,campo bom,Campo Bom,23,54762,-29.683333,-51.05
br,campo bonito,Campo Bonito,18,,-25.016667,-52.983333
br,campo,Campo,11,,-20.483333,-56.133333
br,campo,Campo,13,,-5,-43.466667
br,campo,Campo,16,,-.866667,-46.866667
br,campo,Campo,25,,.803056,-61.660833
br,campo,Campo,27,,-21.383333,-47.05
br,campo comprido,Campo Comprido,13,,-3.966667,-42.884444
br,campo comprido,Campo Comprido,16,,-7.5,-57.5
br,campo comprido,Campo Comprido,17,,-6.516667,-36.133333
br,campo comprido,Campo Comprido,18,,-25.416667,-49.316667
br,campo comprido,Campo Comprido,20,,-3.746667,-42.566389
br,campo da bocaina,Campo da Bocaina,15,,-22.283333,-46.1
br,campo da lanca,Campo da Lança,26,,-26.1931,-49.8292
br,campo das flores,Campo das Flores,15,,-20.45,-46.3
br,campo de cima,Campo de Cima,05,,-10.683333,-44.416667
br,campo de diauarum,Campo de Diauarum,14,,-11.2,-53.233333
br,campo de fora,Campo de Fora,17,,-7.05,-37.866667
br,campo de pouso,Campo de Pouso,20,,-7.766667,-42.25
br,campo de pouso,Campo de Pouso,23,,-32.0995,-52.606
br,campo de sao cristovao,Campo de São Cristóvão,21,,-22.899292,-43.214988
br,campo de sao marcos,Campo de São Marcos,21,,-22.886555,-43.680647
br,campo de sementes,Campo de Sementes,22,,-5.066667,-37.4
br,campo do atanasio,Campo do Atanásio,27,,-22.933333,-45.383333
br,campo do baio,Campo do Baio,18,,-25.755321,-50.00932
br,campo do barreira,Campo do Barreira,16,,-9.3,-50.1
br,campo do brito,Campo do Brito,28,8011,-10.75,-37.5
br,campo do bugre,Campo do Bugre,18,,-25.45,-52.483333
br,campo do caju,Campo do Caju,15,,-20.633333,-45.116667
br,campo do careta,Campo do Careta,20,,-6.166667,-42.3
br,campo do cocho,Campo do Cocho,27,,-22.416667,-47.633333
br,campo do coelho,Campo do Coelho,21,,-22.25,-42.65
br,campo do coxo,Campo do Coxo,27,,-22.416667,-47.633333
br,campo do crioulo,Campo do Crioulo,28,,-10.966667,-37.766667
br,campo do cupinharol,Campo do Cupinharol,20,,-6.883333,-42.683333
br,campo do meio,Campo do Meio,05,,-10.466667,-40.166667
br,campo do meio,Campo do Meio,15,,-20.266667,-46.383333
br,campo do meio,Campo do Meio,23,,-28.316667,-52.05
br,campo do meio,Campo do Meio,27,,-23.5689,-47.7141
br,campo do monteiro,Campo do Monteiro,27,,-22.7081,-45.7256
br,campo do muro,Campo do Muro,17,,-6.883333,-36.25
br,campo do pires,Campo do Pires,15,,-20.0137,-43.8522
br,campo dora,Campo Dora,11,,-18.916667,-56.65
br,campo dos afonsos,Campo dos Afonsos,21,,-22.879221,-43.391562
br,campo do sampaio,Campo do Sampaio,15,,-18.066667,-43.85
br,campo dos bois,Campo dos Bois,14,,-16.583333,-57.05
br,campo dos cavalos,Campo dos Cavalos,05,,-9.566667,-40.65
br,campo dos pintos,Campo dos Pintos,26,,-26.4,-50.833333
br,campo do tenente,Campo do Tenente,18,,-25.979444,-49.681944
br,campo ere,Campo Erê,26,,-26.383333,-53.05
br,campo esperanca,Campo Esperança,01,,-10.416667,-67.633333
br,campo faxinal,Campo Faxinal,26,,-26.383333,-53.05
br,campo fechado,Campo Fechado,18,,-25.3,-51.75
br,campo florido,Campo Florido,15,,-19.783333,-48.583333
br,campo formosa,Campo Formosa,05,,-15.016667,-41.116667
br,campo formosa,Campo Formosa,29,,-17.05,-48.3
br,campo formoso,Campo Formoso,05,23921,-10.516667,-40.333333
br,campo formoso,Campo Formoso,11,,-19.433333,-55.25
br,campo formoso,Campo Formoso,15,,-19.783333,-48.583333
br,campo formoso,Campo Formoso,29,,-17.05,-48.3
br,campo frio,Campo Frio,05,,-14.483333,-43.383333
br,campo grande,Campo Grande,02,3054,-10.233333,-36.5
br,campo grande,Campo Grande,04,,-8.75,-59.266667
br,campo grande,Campo Grande,05,,-12.516667,-39.65
br,campo grande,Campo Grande,06,,-3.65,-40.05
br,campo grande,Campo Grande,08,,-20.35,-40.4
br,campo grande,Campo Grande,11,729164,-20.45,-54.616667
br,campo grande,Campo Grande,13,,-3.663056,-42.726111
br,campo grande,Campo Grande,14,,-16.166667,-56.266667
br,campo grande,Campo Grande,15,,-15.533333,-41.5
br,campo grande,Campo Grande,17,,-6.966667,-36.983333
br,campo grande,Campo Grande,18,,-25.057993,-48.958651
br,campo grande,Campo Grande,20,,-3.752222,-42.5875
br,campo grande,Campo Grande,21,,-22.906675,-43.552896
br,campo grande,Campo Grande,27,,-23.966667,-46.35
br,campo grande,Campo Grande,28,,-10.566667,-37.2
br,campo grande,Campo Grande,30,,-8.666667,-38.633333
br,campo grave,Campo Grave,06,,-5.283333,-38.75
br,campo largo,Campo Largo,05,,-10.4,-41.416667
br,campo largo,Campo Largo,13,,-6.383333,-44.983333
br,campo largo,Campo Largo,16,,-2.05,-49.616667
br,campo largo,Campo Largo,18,94669,-25.433333,-49.533333
br,campo largo,Campo Largo,20,,-3.833333,-42.65
br,campo largo,Campo Largo,27,,-23.190278,-46.702778
br,campo largo da roseira,Campo Largo da Roseira,18,,-25.7,-49.166667
br,campolide,Campolide,05,,-13.916667,-39.616667
br,campolide,Campolide,15,,-21.6,-43.883333
br,campo limpa,Campo Limpa,15,,-21.466667,-42.516667
br,campo limpo,Campo Limpo,05,,-10.133333,-42.116667
br,campo limpo,Campo Limpo,06,,-4.9,-40.283333
br,campo limpo,Campo Limpo,11,,-17.933333,-54.633333
br,campo limpo,Campo Limpo,14,,-15.366667,-55.25
br,campo limpo,Campo Limpo,21,,-22.310428,-42.912765
br,campo limpo,Campo Limpo,27,,-20.733333,-47.2
br,campo limpo,Campo Limpo,29,,-15.85,-48.283333
br,campo lindo,Campo Lindo,29,,-16.283333,-49.083333
br,campo magro,Campo Magro,18,,-25.35,-49.45
br,campo maior,Campo Maior,20,31740,-4.816667,-42.166667
br,campo maje,Campo Majé,30,,-8.516667,-36.75
br,campo mistico,Campo Místico,15,,-22.45,-46.35
br,campo mourao,Campo Mourão,18,,-24.05,-52.366667
br,campo mystico,Campo Mystico,15,,-22.45,-46.35
br,camponeza,Camponeza,20,,-6.483333,-41.3
br,campo novo,Campo Novo,06,,-3.816667,-40.233333
br,campo novo,Campo Novo,08,,-21.083333,-40.95
br,campo novo,Campo Novo,11,,-19.4,-57.616667
br,campo novo,Campo Novo,13,,-4.7,-45.9
br,campo novo,Campo Novo,14,,-15.9,-54.433333
br,campo novo,Campo Novo,15,,-17.9,-41.166667
br,campo novo,Campo Nôvo,17,,-6.816667,-36.25
br,campo novo,Campo Novo,18,,-25.116667,-49.116667
br,campo novo,Campo Novo,23,,-27.7,-53.8
br,campo novo,Campo Novo,26,,-26.0723,-49.8498
br,campo pequeno,Campo Pequeno,28,,-11.116667,-37.916667
br,campo real,Campo Real,18,,-25.45,-51.8
br,campo real,Campo Real,21,,-22.937244,-43.572851
br,campo redondo,Campo Redondo,13,,-3.703889,-43.378333
br,campo redondo,Campo Redondo,15,,-15.633333,-44
br,campo redondo,Campo Redondo,20,,-3.898056,-42.326111
br,campo redondo,Campo Redondo,22,,-6.233333,-36.183333
br,campo redondo,Campo Redondo,27,,-22.35,-47.983333
br,campos altos,Campos Altos,15,12648,-19.683333,-46.166667
br,campo santana,Campo Santana,01,,-8.3,-72.716667
br,campo santo,Campo Santo,05,,-12.995,-38.513333
br,campo santo,Campo Santo,30,,-8.966667,-36.533333
br,campos bellos,Campos Bellos,05,,-15.033333,-41.266667
br,campos bellos,Campos Bellos,29,,-13.05,-46.883333
br,campos belos,Campos Belos,06,,-4.1,-39.05
br,campos belos,Campos Belos,29,15583,-13.05,-46.883333
br,campos borges,Campos Borges,23,,-28.883333,-53
br,campos,Campos,05,,-12.283333,-39.033333
br,campos,Campos,06,,-4.566667,-39.35
br,campos,Campos,15,,-19.35,-45.933333
br,campos,Campos,20,,-3.183333,-41.5
br,campos,Campos,21,387422,-21.75,-41.3
br,campos,Campos,22,,-5.883333,-37.783333
br,campos,Campos,23,,-31.466667,-52.666667
br,campos,Campos,28,,-11.183333,-38.016667
br,campos,Campos,30,,-8.05,-36.45
br,campos da cantareira,Campos da Cantareira,27,,-23.378283,-46.643201
br,campos de cunha,Campos de Cunha,27,,-22.916667,-44.816667
br,campos de mairipora,Campos de Mairiporã,27,,-23.330821,-46.628445
br,campos de membeca,Campos de Membeca,21,,-22.4259,-44.4842
br,campos de sao benedito,Campos de São Benedito,27,,-23.309114,-46.653594
br,campos do jordao,Campos do Jordão,27,53320,-22.7264,-45.5792
br,campos do meio,Campos do Meio,15,,-21.116667,-45.833333
br,campos dos goitacazes,Campos dos Goitacazes,21,,-21.75,-41.3
br,campos eliseos,Campos Elíseos,11,,-20.95,-53.283333
br,campos elisios,Campos Elisios,11,,-20.95,-53.283333
br,campos elisios,Campos Elísios,21,,-22.4548,-44.4377
br,campos elyseos,Campos Elyseos,21,,-22.704444,-43.266111
br,campos geraes,Campos Geraes,15,,-21.233333,-45.766667
br,campos gerais,Campos Gerais,15,22260,-21.233333,-45.766667
br,campos lindos,Campos Lindos,31,,-7.996111,-46.706389
br,campos novas,Campos Novas,28,,-9.6,-38.033333
br,campos novos,Campos Novos,13,,-3.770278,-43.369722
br,campos novos,Campos Novos,18,,-25.45,-52.916667
br,campos novos,Campos Novos,21,,-22.7,-42.033333
br,campos novos,Campos Novos,22,,-5.431389,-36.559722
br,campos novos,Campos Novos,23,,-27.7,-53.8
br,campos novos,Campos Novos,26,22563,-27.4,-51.2
br,campos novos,Campos Novos,27,,-22.6,-50.016667
br,campos novos,Campos Novos,28,,-9.6,-38.033333
br,campos novos,Campos Novos,30,,-7.6,-37.466667
br,campos novos da cunha,Campos Novos da Cunha,27,,-22.916667,-44.816667
br,campos novos de cunha,Campos Novos de Cunha,27,,-22.916667,-44.816667
br,campos novos do paranapanema,Campos Novos do Paranapanema,27,,-22.6,-50.016667
br,campos novos paulista,Campos Novos Paulista,27,,-22.6,-50.016667
br,campos sales,Campos Sales,05,,-14.75,-40.533333
br,campos sales,Campos Sales,06,17648,-7.066667,-40.383333
br,campos sales,Campos Sales,20,,-4.016667,-42.083333
br,campos salles,Campos Salles,06,,-7.066667,-40.383333
br,campos segundo,Campos Segundo,20,,-6.316667,-41.916667
br,campo sujo,Campo Sujo,30,,-8.916667,-36.7
br,campos velhos,Campos Velhos,06,,-5.516667,-38
br,campo verde,Campo Verde,03,,.9,-51.633333
br,campo verde,Campo Verde,05,,-12.7,-39.316667
br,campo verde,Campo Verde,11,22842,-20.416667,-54.066667
br,campo verde,Campo Verde,17,,-7.55,-36.15
br,campo verde,Campo Verde,20,,-7.25,-40.75
br,campo vicente,Campo Vicente,23,,-29.633333,-50.9
br,camuca,Camuca,17,,-6.8,-35.6
br,camurcim,Camurcim,01,,-8.992222,-71.583333
br,camurim,Camurim,22,,-5.354444,-37.462778
br,camurugi,Camurugi,05,,-13.516667,-39.1
br,camurupim,Camurupim,20,,-3.083333,-41.45
br,camurupim,Camurupim,22,,-5.18,-36.323611
br,camurupim de baixo,Camurupim de Baixo,20,,-2.924167,-41.467778
br,camurupim de cima,Camurupim de Cima,20,,-3.083333,-41.45
br,camuscada,Camuscada,05,,-10.033333,-42.633333
br,camutanga,Camutanga,30,,-7.416667,-35.283333
br,camuti,Camuti,16,,-3.65,-56.55
br,camuxinga,Camuxinga,02,,-9.3,-37.216667
br,camuxinga de cima,Camuxinga de Cima,02,,-9.3,-37.216667
br,canaa,Canaã,04,,-4.134722,-60.964722
br,canaa,Canaã,06,,-3.216667,-39.366667
br,canaa,Canaã,15,,-20.683333,-42.616667
br,canaa,Canaã,17,,-6.516667,-38.416667
br,canaa,Canaã,22,,-5.18,-37.195278
br,canaa,Canaã,27,,-21.533333,-47.683333
br,canaa,Canaã,30,,-8.4,-36.783333
br,canaan,Canaán,06,,-5.833333,-39.6
br,canaan,Canaan,27,,-21.533333,-47.683333
br,canabarro,Canabarro,23,,-29.2,-52.033333
br,cana-brava,Cana-Brava,02,,-9.633333,-36.483333
br,canabrava,Canabrava,05,,-10.666667,-40.816667
br,cana brava,Cana Brava,05,,-12.05,-43.1
br,canabrava,Canabrava,05,,-12.116667,-43.9
br,cana brava,Cana Brava,05,,-13.3,-43.983333
br,canabrava,Canabrava,05,,-13.65,-41.583333
br,cana brava,Cana Brava,05,,-13.866667,-42.7
br,canabrava,Canabrava,05,,-9.75,-39.616667
br,cana brava,Cana Brava,05,,-9.783333,-38.316667
br,canabrava,Canabrava,06,,-6.433333,-40.716667
br,cana brava,Cana Brava,06,,-6.55,-40.616667
br,cana brava,Cana Brava,13,,-3.080556,-42.228333
br,cana-brava,Cana-Brava,13,,-6.283333,-42.983333
br,canabrava,Canabrava,15,,-16.316667,-43.416667
br,cana-brava,Cana-Brava,20,,-10.1,-45.183333
br,cana brava,Cana Brava,20,,-5.133333,-41.15
br,cana brava,Cana Brava,22,,-5.316667,-35.583333
br,cana brava,Cana Brava,28,,-10.416667,-37.216667
br,canabrava,Canabrava,28,,-9.583333,-37.983333
br,cana brava,Cana Brava,30,,-7.5,-39.833333
br,canabrava,Canabrava,30,,-8.666667,-39.183333
br,cana brava,Cana Brava,31,,-12.900833,-46.765833
br,canabrava da cruz,Canabrava da Cruz,05,,-12.183333,-38.716667
br,canabrava de santa teresa de jesus dos quiriris,Canabrava de Santa Teresa de Jesus dos Quiriris,05,,-10.833333,-38.533333
br,canabrava do boqueirao,Canabrava do Boqueirão,05,,-12.05,-43.1
br,canabrava do braga,Canabrava do Braga,05,,-12.216667,-43.1
br,canabrava do goncalo,Canabrava do Gonçalo,05,,-11.333333,-42.133333
br,canabrava do miranda,Canabrava do Miranda,05,,-11.683333,-41.766667
br,canabrava dos caldeiras,Canabrava dos Caldeiras,05,,-13.616667,-42.6
br,canabrava dos martins,Canabrava dos Martins,05,,-12.2,-41.683333
br,cana brava dos pais,Cana Brava dos Pais,02,,-9.633333,-36.483333
br,cana brava grande,Cana Brava Grande,06,,-7.516667,-38.783333
br,cana brava ii,Cana Brava II,13,,-3.0775,-42.280556
br,cana brava iii,Cana Brava III,13,,-3.205,-42.193333
br,canabravao,Canabravão,05,,-12.416667,-44.866667
br,canabravinha,Canabravinha,05,,-13.533333,-42.166667
br,cana bravinha,Cana Bravinha,06,,-7.616667,-38.733333
br,cana,Cana,06,,-6.266667,-40.65
br,cana,Cana,13,,-6.65,-42.916667
br,canaci,Canaci,05,,-13.5,-41.116667
br,canacuri,Canacuri,04,,-7.816667,-67.133333
br,canada,Canadá,05,,-12.283333,-40.433333
br,canada,Canada,06,,-4.2,-38.85
br,canada,Canadá,13,,-1.65,-44.9
br,cana do reino,Cana do Reino,15,,-21.783333,-45.833333
br,canafistinha,Canafistinha,06,,-2.933333,-40.283333
br,canafistula,Canafístula,02,,-9.4,-36.7
br,canafistula,Canafístula,05,,-12.433333,-43.283333
br,canafistula,Canafistula,06,,-3.966667,-39.233333
br,canafistula,Canafístula,13,,-3.866944,-43.468611
br,canafistula,Canafístula,17,,-6.983333,-35.566667
br,canafistula,Canafístula,20,,-3.897222,-42.277778
br,canafistula,Canafístula,22,,-5.380833,-36.533611
br,canafistula,Canafistula,30,,-7.883333,-37.5
br,canafistulas,Canafístulas,02,,-9.4,-36.7
br,canaguape,Canaguapé,04,,-6.964167,-71.414167
br,canaiba,Canaíba,05,,-14.033333,-42.783333
br,canais,Canais,05,,-11.833333,-41.583333
br,canal,Canal,05,,-11.35,-41.666667
br,canal de sao simao,Canal de São Simão,29,,-18.933333,-50.5
br,canama,Canamã,04,,-5.616667,-72.1
br,canamari,Canamari,01,,-10.133333,-69.266667
br,cananea,Cananéa,13,,-4.516667,-44.35
br,cananea,Cananéa,27,,-25.016667,-47.95
br,cananeia,Cananéia,13,,-4.516667,-44.35
br,cananeia,Cananéia,20,,-7.333333,-41.766667
br,cananeia,Cananéia,26,,-26.845,-49.021
br,cananeia,Cananéia,27,13786,-25.016667,-47.95
br,canapi,Canapi,02,4219,-9.133333,-37.5
br,canapolis,Canápolis,15,,-18.733333,-49.216667
br,canarama,Canarama,24,,-10.666667,-62.65
br,canarana,Canarana,05,10785,-11.683333,-41.766667
br,canarana,Canarana,13,,-4.45,-44.533333
br,canarias,Canárias,13,,-2.766944,-41.850833
br,canario,Canário,24,,-12.3125,-60.6375
br,canas,Canas,15,,-14.95,-42.7
br,canas,Canas,27,,-22.716667,-45.05
br,canastra,Canastra,02,,-8.933333,-35.85
br,canastra,Canastra,05,,-11.816667,-42.233333
br,canastrao,Canastrão,15,,-18.566667,-45.75
br,canasvieiras,Canasvieiras,26,,-27.433333,-48.5
br,canatiba,Canatiba,05,,-13.083333,-42.866667
br,canauani,Canauani,25,,2.8,-60.533333
br,canavani,Canavani,25,,2.8,-60.533333
br,cana verde,Cana Verde,15,,-21.016667,-45.183333
br,canavial,Canavial,04,,-2.75,-67.633333
br,canavial,Canavial,23,,-27.533333,-51.983333
br,canavieira,Canavieira,05,,-10.916667,-40.65
br,canavieira,Canavieira,20,,-7.066667,-41.883333
br,canavieiras,Canavieiras,05,26376,-15.65,-38.95
br,canavieiras,Canavieiras,18,,-25.6572,-48.8132
br,canca,Cancã,18,,-24.583333,-52.183333
br,cancela,Cancela,06,,-4.783333,-40.016667
br,cancela,Cancela,13,,-3.1,-42.978611
br,cancela,Cancela,14,,-14.716667,-56.283333
br,cancela,Cancela,15,,-16.316667,-42.716667
br,cancelao,Cancelão,23,,-31.383333,-53.116667
br,cancela preta,Cancela Preta,21,,-22.849816,-43.456318
br,canche,Canché,05,,-9.883333,-38.9
br,candeal,Candeal,05,,-11.816667,-39.116667
br,candeeiro,Candeeiro,20,,-3.505,-42.364722
br,candeia,Candeia,05,,-17.233333,-39.666667
br,candeia,Candeia,22,,-6.116667,-37.833333
br,candeia,Candeia,23,,-28.25,-51.55
br,candeia de baixo,Candeia de Baixo,06,,-4.283333,-38.85
br,candeial,Candeial,05,,-15.016667,-41.933333
br,candeias,Candeias,05,75673,-12.666667,-38.55
br,candeias,Candeias,13,,-3.101389,-42.092222
br,candeias,Candeias,15,,-20.783333,-45.266667
br,candeias,Candeias,30,,-8.233333,-34.933333
br,candeiras,Candeiras,05,,-12.583333,-39.366667
br,candelaria,Candelaria,15,,-22.5814,-45.6351
br,candelaria,Candelária,21,,-22.901199,-43.17823
br,candelaria,Candelária,23,15200,-29.666667,-52.8
br,candia,Cândia,27,,-20.9,-47.983333
br,candial,Candial,05,,-11.95,-42.3
br,candiba,Candiba,05,,-14.4,-42.866667
br,candido,Cândido,06,,-4.8,-39.383333
br,candido de abreu,Cândido de Abreu,18,5102,-24.583333,-51.333333
br,candido godoi,Cândido Godói,23,,-27.95,-54.766667
br,candido mendes,Cândido Mendes,13,8613,-1.45,-45.716667
br,candido mota,Cândido Mota,27,28145,-22.716667,-50.383333
br,candido motta,Candido Motta,27,,-22.716667,-50.383333
br,candido ramalho,Cândido Ramalho,15,,-17.033333,-41.983333
br,candido rodrigues,Cândido Rodrigues,27,,-21.333333,-48.633333
br,candiota,Candiota,23,,-31.6,-53.733333
br,candoca velha,Candoca Velha,16,,-6.95,-53.533333
br,candoi,Candói,18,,-25.716667,-52.183333
br,candomba,Candomba,05,,-14.133333,-40.6
br,candonga,Candonga,18,,-25.5855,-48.8042
br,candoso,Candoso,11,,-20.65,-57.65
br,candoy,Candoy,18,,-25.716667,-52.183333
br,candumba,Candumba,14,,-16.183333,-56.566667
br,canea,Canea,16,,-1.85,-54.933333
br,caneapo,Canéapo,16,,1.566667,-54.766667
br,caneiras,Caneiras,30,,-7.583333,-35.216667
br,canela,Canela,20,,-3.786667,-42.409167
br,canela,Canela,23,33529,-29.366667,-50.833333
br,canela,Canela,30,,-8.8,-37.366667
br,canela,Canela,31,,-10.22,-48.411111
br,canela da ema,Canela da Ema,20,,-6.883333,-41.25
br,canela de ema,Canela de Ema,05,,-14.2,-41.866667
br,canela oca,Canela Oca,27,,-24.9051,-48.021
br,canelinha,Canelinha,18,,-25.246487,-48.918742
br,canelinha,Canelinha,26,,-27.283333,-48.783333
br,canella,Canella,23,,-29.366667,-50.833333
br,canga,Canga,05,,-17.283333,-40.183333
br,canga,Canga,14,,-16.077222,-56.570556
br,cangaira,Cangaira,22,,-6.033333,-37.5
br,cangalha,Cangalha,11,,-19.55,-52.8
br,cangalha,Cangalha,15,,-17.666667,-47.183333
br,cangalha,Cangalha,30,,-7.433333,-37.083333
br,cangalhas,Cangalhas,15,,-17.666667,-47.183333
br,cangas,Cangas,14,,-16.077222,-56.570556
br,cangati,Cangati,06,,-5.433333,-39.183333
br,cangaty,Cangaty,06,,-5.566667,-38.983333
br,cangau,Cangaú,30,,-9.133333,-36.316667
br,cangica,Cangica,13,,-7.966667,-46.95
br,cangicas,Cangicas,27,,-23.271361,-46.599856
br,cangonhal,Cangonhal,15,,-21.933333,-45.816667
br,cangoritgo,Cangoritgo,05,,-11.95,-37.7
br,cangote,Cangote,13,,-3.733333,-44.966667
br,canguaretama,Canguaretama,22,18873,-6.383333,-35.133333
br,cangucu,Canguçu,23,20134,-31.4,-52.683333
br,cangueira,Cangueira,18,,-25.066667,-50.5
br,canguera,Canguera,27,,-23.5028,-47.7089
br,canguinha,Canguinha,14,,-14.7,-55.666667
br,canguinho,Canguinho,14,,-14.7,-55.666667
br,cangussu,Cangussú,23,,-31.4,-52.683333
br,canhoba,Canhoba,28,,-10.133333,-36.95
br,canhoe,Canhoé,04,,-6.041389,-63.372222
br,canhotinho,Canhotinho,06,,-5.35,-39.266667
br,canhotinho,Canhotinho,30,12074,-8.883333,-36.2
br,canhuma,Canhumã,04,,-3.916667,-59.15
br,canicora,Canicorã,11,,-21.616667,-55.416667
br,canicura,Canicurã,11,,-21.616667,-55.416667
br,caninae,Caninae,06,,-4.366667,-39.316667
br,caninana,Caninana,14,,-16.233333,-57.583333
br,caninde,Canindé,04,,-3.6375,-62.998611
br,caninde,Canindé,06,42721,-4.366667,-39.316667
br,caninde,Canindé,16,,-2.566667,-46.516667
br,caninde,Canindé,20,,-6.966667,-40.916667
br,caninde,Canindé,28,,-9.65,-37.8
br,caninde de sao francisco,Canindé de São Francisco,28,11053,-9.65,-37.8
br,canindezinho,Canindézinho,06,,-5.8,-38.2
br,canindezinho,Canindézinho,22,,-6.233333,-38.533333
br,caninha branca,Caninha Branca,15,,-20.116667,-41.55
br,caninha,Caninha,20,,-7.066667,-40.783333
br,canitar,Canitar,27,,-23.006204,-49.783344
br,canivete,Canivete,02,,-9.316667,-37.266667
br,canjarana,Canjarana,31,,-10.866667,-46.35
br,canjerana,Canjerana,31,,-10.866667,-46.35
br,canjica,Canjica,13,,-7.966667,-46.95
br,canjica,Canjica,15,,-17.983333,-43.466667
br,canjicas,Canjicas,15,,-17.983333,-43.466667
br,canna brava,Canna Brava,02,,-9.633333,-36.483333
br,canna brava,Canna Brava,05,,-10.833333,-38.533333
br,cannabrava,Cannabrava,05,,-13.616667,-42.6
br,canna brava,Canna Brava,05,,-9.2,-39.7
br,cannabrava,Cannabrava,15,,-17.4,-45.85
br,cannabrava,Cannabrava,30,,-7.5,-39.833333
br,canna brava do miranda,Canna Brava do Miranda,05,,-11.683333,-41.766667
br,cannabrava dos goncalves,Cannabrava dos Gonçalves,05,,-11.333333,-42.133333
br,canna,Canna,20,,-6.766667,-43.133333
br,cannarias,Cannarias,13,,-2.766944,-41.850833
br,cannasvieiras,Cannasvieiras,26,,-27.433333,-48.5
br,canna verde,Canna Verde,15,,-21.016667,-45.183333
br,cannavieiras,Cannavieiras,05,,-15.65,-38.95
br,canoa,Canôa,05,,-10.4,-40.116667
br,canoa,Canoa,13,,-5.283333,-43.083333
br,canoa,Canoã,14,,-14.55,-56.866667
br,canoa,Canoa,17,,-7.2,-38.116667
br,canoa grande,Canoa Grande,13,,-5.366667,-46.016667
br,canoana,Canoanã,31,,-11.927778,-49.899722
br,canoao,Canoão,05,,-11.516667,-42.016667
br,canoa quebrada,Canoa Quebrada,06,,-4.516667,-37.683333
br,canoas,Canoas,05,,-12.35,-37.883333
br,canoas,Canoas,15,,-16.37,-44.784722
br,canoas,Canoas,17,,-6.533333,-35.966667
br,canoas,Canoas,18,,-24.7755,-48.956055
br,canoas,Canoas,21,,-22.3972,-42.883849
br,canoas,Canoas,23,328297,-29.933333,-51.183333
br,canoas,Canoas,26,,-27.566667,-50.4
br,canoeira,Canoeira,14,,-16.9,-55.95
br,canoeiro,Canoeiro,18,,-25.803112,-50.187881
br,canoeiros,Canoeiros,15,,-18.033333,-45.516667
br,canoinhas,Canoinhas,18,,-24.016667,-49.95
br,canoinhas,Canoinhas,26,38863,-26.166667,-50.4
br,cansancao,Cansanção,05,9905,-10.683333,-39.516667
br,cansancao,Cansanção,20,,-7.083333,-40.966667
br,cansa-perna,Cansa-Perna,16,,-1.766667,-46.616667
br,cantagallo,Cantagallo,21,,-21.966667,-42.366667
br,canta-galo,Canta-Galo,04,,-4.448611,-58.114722
br,cantagalo,Cantagalo,04,,-8.233333,-60.7
br,canta-galo,Canta-Galo,04,,-8.35,-58.783333
br,cantagalo,Cantagalo,06,,-4.25,-38.683333
br,canta galo,Canta Galo,11,,-17.7,-56.016667
br,canta galo,Canta Galo,14,,-14.65,-57.216667
br,cantagalo,Cantagalo,15,,-18.533333,-42.616667
br,cantagalo,Cantagalo,18,,-25.111693,-48.898233
br,cantagalo,Cantagalo,21,14019,-21.966667,-42.366667
br,canta galo,Canta Galo,22,,-5.644722,-36.9325
br,cantagalo,Cantagalo,25,,3.533333,-61.1
br,cantagalo,Cantagalo,27,,-22.5959,-45.6852
br,cantanhede,Cantanhede,13,8344,-3.65,-44.4
br,cantao,Cantão,13,,-9.733333,-46.383333
br,cantareia,Cantareia,27,,-23.466667,-46.633333
br,cantareira,Cantareira,27,,-23.466667,-46.633333
br,cantarem,Cantarém,20,,-6.9,-42.65
br,canteiro,Canteiro,05,,-12.2,-38.9
br,canteiro,Canteiro,21,,-22.273333,-43.738889
br,cantinho,Cantinho,05,,-10.733333,-42.666667
br,cantinho,Cantinho,13,,-3.412778,-42.6325
br,cantinho,Cantinho,17,,-7.2,-37.916667
br,cantinho,Cantinho,20,,-3.474167,-42.345833
br,cantinho,Cantinho,22,,-5.481389,-36.478611
br,cantinho,Cantinho,30,,-8.166667,-37.233333
br,cantinho do boi,Cantinho do Boi,17,,-6.833333,-37.783333
br,cantinho ii,Cantinho II,13,,-3.856111,-43.274722
br,canto alegre,Canto Alegre,13,,-5.083333,-42.95
br,canto alegre,Canto Alegre,20,,-5.633333,-41.8
br,canto bom,Canto Bom,13,,-3.818056,-43.328333
br,canto bonito,Canto Bonito,31,,-8.920556,-48.370556
br,canto,Canto,05,,-12.883333,-39.35
br,canto,Canto,06,,-5.3,-40.566667
br,canto,Canto,15,,-22.512087,-45.521332
br,canto,Canto,20,,-6.266667,-42.5
br,canto,Canto,28,,-9.8,-37.416667
br,canto cercado,Canto Cercado,05,,-10.333333,-42.433333
br,canto comprido,Canto Comprido,22,,-5.2,-36.833333
br,canto da catinga,Canto da Catinga,20,,-3.467222,-42.358611
br,canto da folha,Canto da Folha,20,,-3.724722,-42.346111
br,canto da ilha de cima,Canto da Ilha de Cima,22,,-5.083333,-35.833333
br,canto da ipoeira,Canto da Ipoeira,20,,-6.483333,-41.983333
br,canto da lagoa,Canto da Lagoa,22,,-5.07,-36.842222
br,canto da moca,Canto da Moça,22,,-5.733333,-35.466667
br,canto da palmeira,Canto da Palmeira,20,,-3.845833,-42.111111
br,canto da parnaiba,Canto da Parnaíba,20,,-3.949167,-42.410278
br,canto da passagem,Canto da Passagem,13,,-3.364722,-42.464444
br,canto da porta,Canto da Porta,13,,-3.425556,-42.456944
br,canto da raposa,Canto da Raposa,20,,-3.294444,-42.058056
br,canto da roca,Canto da Roca,20,,-6.7,-42.416667
br,canto da serra,Canto da Serra,13,,-6.283333,-43.383333
br,canto da serra,Canto da Serra,20,,-7.316667,-41.583333
br,canto da varzea,Canto da Várzea,05,,-9.866667,-38.966667
br,canto de areia,Canto de Areia,13,,-7.366667,-44.8
br,canto de baixo,Canto de Baixo,20,,-3.423333,-42.1375
br,canto de baixo,Canto de Baixo,22,,-5.083333,-35.866667
br,canto de cima,Canto de Cima,20,,-3.413611,-42.151944
br,canto do araca,Canto do Araçá,13,,-3.731111,-43.410833
br,canto do boi,Canto do Boi,13,,-3.847222,-43.456389
br,canto do buriti,Canto do Buriti,20,11715,-8.116667,-42.966667
br,canto do caboclo,Canto do Caboclo,20,,-3.711389,-42.101667
br,canto do coco,Canto do Côco,20,,-9.883333,-45.166667
br,canto do engenho,Canto do Engenho,15,,-16.925,-44.1125
br,canto do feijao,Canto do Feijão,22,,-5.616667,-37.6
br,canto do ferreira,Canto do Ferreira,20,,-6.6,-42.15
br,canto do jateba,Canto do Jatebá,20,,-6.3,-42.566667
br,canto do juazeiro,Canto do Juàzeiro,06,,-6.1,-38.866667
br,canto do junco,Canto do Junco,22,,-5.15,-37.333333
br,canto do mangue,Canto do Mangue,22,,-5.068333,-36.780278
br,canto do mimoso,Canto do Mimoso,20,,-6.883333,-42.35
br,canto do morcego,Canto do Morcego,13,,-3.391111,-42.188333
br,canto do murici,Canto do Murici,20,,-3.331944,-42.094444
br,canto do riacho,Canto do Riacho,05,,-14.3,-43.6
br,canto do rio,Canto do Rio,04,,-3.912778,-62.661944
br,canto do rio verde,Canto do Rio Verde,05,,-10.416667,-42.316667
br,canto do rodeio,Canto do Rodeio,13,,-3.444444,-42.524444
br,canto do rumo,Canto do Rumo,05,,-12.05,-39.733333
br,canto do saco,Canto do Saco,13,,-3.146111,-42.019444
br,canto dos bezerra,Canto dos Bezerra,20,,-7.583333,-43.1
br,canto dos bois,Canto dos Bois,13,,-3.773889,-43.360278
br,canto dos cocos,Canto dos Côcos,20,,-5.983333,-42.466667
br,canto dos negros,Canto dos Negros,13,,-3.670278,-42.975
br,canto do travessao,Canto do Travessão,20,,-3.3475,-42.0175
br,canto do umari,Canto do Umarí,06,,-3.15,-40.116667
br,cantofe,Cantofe,20,,-7.566667,-41.516667
br,canto fundo,Canto Fundo,13,,-3.766111,-43.285
br,canto grande,Canto Grande,05,,-10.166667,-43.316667
br,canto grande,Canto Grande,13,,-3.105833,-43.425
br,canto grande,Canto Grande,20,,-3.584722,-42.5225
br,canto grande,Canto Grande,22,,-5.403611,-36.83
br,canto grande,Canto Grande,26,,-27.196389,-48.499444
br,canto grande i,Canto Grande I,20,,-3.306389,-42.064444
br,canto grande iii,Canto Grande III,13,,-3.271944,-42.238333
br,cantori,Cantori,04,,-9.333333,-66.633333
br,cantos,Cantos,22,,-6.066667,-37.883333
br,cantos dos cocos,Cantos dos Cocos,20,,-6.033333,-41.816667
br,canto serrote,Canto Serrote,22,,-5.172778,-37.224444
br,canto sujo,Canto Sujo,13,,-3.160556,-42.544722
br,canuana,Canuanã,31,,-11.927778,-49.899722
br,canudos,Canudos,02,,-9.583333,-36.483333
br,canudos,Canudos,04,,-7.253611,-58.103611
br,canudos,Canudos,05,,-11.533333,-45.083333
br,canudos,Canudos,06,,-5.483333,-38.516667
br,canudos,Canudos,17,,-7.466667,-37.7
br,canudos,Canudos,18,,-25.61564,-50.37611
br,canudos,Canudos,22,,-5.317778,-37.449722
br,canudos,Canudos,23,,-29.333333,-52.25
br,canudos,Canudos,30,,-7.833333,-38.616667
br,canudo velho,Canudo Velho,02,,-9.55,-36.483333
br,canuma,Canumã,04,,-4.038889,-59.0725
br,canutama,Canutama,04,6320,-6.533333,-64.333333
br,canxe,Canxé,05,,-9.883333,-38.9
br,canzianopolis,Canzianópolis,18,,-25.95,-53.7
br,canzionopolis,Canziónopolis,18,,-25.95,-53.7
br,caovi,Caovi,18,,-25.9541,-48.8264
br,caovi,Caovi,26,,-25.984996,-48.838127
br,capa,Capa,05,,-16.2,-39.366667
br,capacete,Capacete,04,,-4.3725,-69.749722
br,capacete,Capacete,18,,-25.383333,-54.316667
br,capa de cima,Capa de Cima,22,,-6.1,-38.166667
br,capado,Capado,11,,-19.916667,-54.35
br,capalazia,Capalazia,25,,2.866667,-60.833333
br,capana,Capana,04,,-1.866667,-68.95
br,capane,Capané,23,,-30.266667,-52.85
br,capanema,Capanema,05,,-12.85,-38.95
br,capanema,Capanema,14,,-13.316667,-57.866667
br,capanema,Capanema,16,47768,-1.2,-47.183333
br,capanema,Capanema,18,,-25.666667,-53.8
br,capanema,Capanema,27,,-23.509253,-47.708565
br,capanga,Capangá,15,,-20.183333,-46.083333
br,capao alto,Capão Alto,11,,-18.9,-54.866667
br,capao alto,Capão Alto,14,,-15.716667,-56.166667
br,capao alto,Capão Alto,18,,-24.75,-49.95
br,capao alto,Capão Alto,26,,-27.933333,-50.5
br,capao alto,Capão Alto,27,,-23.5719,-47.8233
br,capao bonito,Capão Bonito,11,,-21.233333,-54.783333
br,capao bonito,Capão Bonito,23,,-28.116667,-51.4
br,capao bonito,Capão Bonito,27,37207,-24.016667,-48.333333
br,capao,Capão,05,,-11.782778,-45.23
br,capao,Capão,13,,-3.8525,-43.086667
br,capao,Capão,15,,-16.566667,-41.95
br,capao,Capão,18,,-24.75,-49.95
br,capao,Capão,20,,-3.5075,-42.515556
br,capao,Capão,29,,-14.683333,-46.35
br,capao,Capão,31,,-11.65,-47.516667
br,capao comprido,Capão Comprido,05,,-13.566667,-41.85
br,capao da anta,Capão da Anta,14,,-15.4,-60.266667
br,capao da areia,Capão da Areia,05,,-13.416667,-43.433333
br,capao da areia,Capão da Areia,06,,-5.133333,-40.716667
br,capao da caca,Capão da Caça,15,,-19.35,-48.35
br,capao da canoa,Capão da Canoa,23,35896,-29.766667,-50.016667
br,capao da cruz,Capão da Cruz,27,,-21.433333,-47.9
br,capao da onca,Capão da Onça,05,,-12.066667,-41.45
br,capao da velha,Capão da Velha,20,,-7,-42.5
br,capao da volta,Capão da Volta,05,,-13.5,-41.466667
br,capao de heranca,Capão de Herança,23,,-28.366667,-51.016667
br,capao do boi,Capão do Boi,31,,-7.483333,-48.2
br,capao do cedro,Capão do Cedro,18,,-25.86426,-50.182205
br,capao do jeronimo,Capão do Jerônimo,05,,-11.866667,-45.766667
br,capao do jucu,Capão do Jucu,05,,-12.9,-41.683333
br,capao do leao,Capão do Leão,23,,-31.766667,-52.5
br,capao do mato,Capão do Mato,05,,-14.366667,-44.666667
br,capao do meio,Capão do Meio,05,,-12.1,-45.216667
br,capao do meio,Capão do Meio,23,,-29.6,-51.5
br,capao dos oleos,Capão dos Óleos,15,,-21.416667,-45.2
br,capao dos patos,Capão dos Patos,23,,-28.633333,-50.1
br,capao do tigre,Capão do Tigre,23,,-28.733333,-50.133333
br,capao frio,Capão Frio,05,,-13.033333,-41.633333
br,capao grande,Capão Grande,13,,-3.671944,-43.161667
br,capao grande,Capão Grande,14,,-15.716667,-56.166667
br,capao grande,Capão Grande,23,,-28.533333,-51.216667
br,capao i,Capão I,13,,-3.373889,-43.248056
br,capao ii,Capão II,13,,-3.329167,-42.975278
br,capao limpo,Capão Limpo,11,,-21.383333,-54.383333
br,capao nossa senhora de fatima,Capão Nossa Senhora de Fátima,23,,-27.933333,-51.05
br,capao preto,Capão Prêto,15,,-18.766667,-45.7
br,capao queimado,Capão Queimado,05,,-12.9,-41.583333
br,capao real,Capão Real,18,,-25.45,-51.8
br,capao redondo,Capão Redondo,05,,-12.283333,-41.4
br,capao redondo,Capão Redondo,15,,-16.666667,-45.4
br,capao redondo,Capão Redondo,27,,-23.612724,-47.634207
br,capao redondo de cima,Capão Redondo de Cima,27,,-23.640663,-47.631042
br,capao secco,Capão Secco,23,,-31.85,-52.333333
br,capao seco,Capão Sêco,11,,-21.066667,-54.683333
br,capao seco,Capão Sêco,23,,-31.85,-52.333333
br,capao seco,Capão Sêco,27,,-20.833333,-47.783333
br,capao taquarivahy,Capão Taquarivahy,27,,-23.933333,-48.683333
br,capao verde,Capão Verde,11,,-20.066667,-56.066667
br,capao vermelho,Capão Vermelho,05,,-12.9,-41.683333
br,capaozinho,Capãozinho,13,,-3.346111,-42.812778
br,caparao,Caparaó,08,,-20.533333,-41.666667
br,caparao,Caparaó,15,,-20.516667,-41.9
br,caparao velho,Caparaó Velho,15,,-20.433333,-41.866667
br,capas de anta,Cápas de Anta,14,,-15.4,-60.266667
br,capatana,Capatana,04,,-2.8,-67.066667
br,capatara,Capatará,01,,-10.283333,-67.816667
br,capatera,Capatera,01,,-10.283333,-67.816667
br,capa-vaca,Capa-Vaca,15,,-20.933333,-42.283333
br,capecu,Capeçu,04,,-2.716667,-66.85
br,capela,Capela,02,12520,-9.416667,-36.066667
br,capela,Capela,04,,-2.966667,-57.85
br,capela,Capela,05,,-13.75,-39.366667
br,capela,Capela,11,,-18.316667,-53.166667
br,capela,Capela,15,,-21.333333,-44.55
br,capela,Capela,22,,-5.6,-35.45
br,capela,Capela,23,,-29.7,-51.333333
br,capela,Capela,28,16024,-10.5,-37.066667
br,capela,Capela,29,,-16.366667,-49.85
br,capela da palma,Capela da Palma,05,,-13.133333,-39.15
br,capela de conceicao,Capela de Conceição,05,,-11.883333,-38.166667
br,capela de santana,Capela de Santana,23,,-29.7,-51.333333
br,capela de sao benedito,Capela de São Benedito,27,,-20.833333,-48.966667
br,capela de sao miguel,Capela de São Miguel,27,,-20.9,-48.85
br,capela do alto alegre,Capela do Alto Alegre,05,,-11.666667,-39.85
br,capela do alto,Capela do Alto,27,11991,-23.466667,-47.75
br,capela do bom jesus,Capela do Bom Jesus,27,,-23.3761,-46.1405
br,capela do jaguara,Capela do Jaguara,15,,-21.266667,-44.533333
br,capela do saco,Capela do Saco,15,,-21.333333,-44.55
br,capela ivapurunduva,Capela Ivapurunduva,27,,-24.55,-48.383333
br,capela nova,Capela Nova,15,,-20.916667,-43.616667
br,capela nova,Capela Nova,30,,-7.933333,-35.983333
br,capela nova das dores,Capela Nova das Dores,15,,-20.916667,-43.616667
br,capela sao benedito,Capela São Benedito,27,,-20.233333,-47.883333
br,capela sao miguel,Capela São Miguel,27,,-20.9,-48.85
br,capelinha,Capelinha,15,20074,-17.7,-42.516667
br,capelinha,Capelinha,27,,-24.844191,-48.231054
br,capelinha,Capelinha,29,,-15.716667,-49.8
br,capelinha do amparo,Capelinha do Amparo,21,,-22.133333,-41.716667
br,capelinha do chumbo,Capelinha do Chumbo,15,,-18.683333,-46.066667
br,capelinha do embirizal,Capelinha do Embirizal,15,,-22.033333,-45.433333
br,capella,Capella,04,,-2.966667,-57.85
br,capella,Capella,15,,-21.333333,-44.55
br,capella,Capella,23,,-29.7,-51.333333
br,capella,Capella,28,,-10.5,-37.066667
br,capella de sao benedicto,Capella de São Benedicto,27,,-20.233333,-47.883333
br,capella nova,Capella Nova,30,,-7.933333,-35.983333
br,capella nova das dores,Capella Nova das Dores,15,,-20.916667,-43.616667
br,capellinha,Capellinha,15,,-17.7,-42.516667
br,capernaum,Capernaúm,05,,-11.7,-41.466667
br,capetinga,Capetinga,15,,-20.6,-47.05
br,capetuva,Capetuva,15,,-21.866667,-46.35
br,capia,Capiá,02,,-9.183333,-37.433333
br,capianga,Capianga,05,,-12.016667,-38.216667
br,capia novo,Capiá Novo,02,,-9.233333,-37.483333
br,capibaribe,Capibaribe,30,,-7.95,-36.2
br,capim acu,Capim Açu,05,,-12.4,-39.7
br,capim-acu,Capim-Açu,13,,-3.116667,-43.666667
br,capim acu,Capim Açu,15,,-15.833333,-40.35
br,capim acu,Capim Açu,22,,-5.9,-35.516667
br,capim azedo,Capim Azêdo,27,,-23.666667,-47.216667
br,capim branco,Capim Branco,15,,-16.266667,-40.8
br,capim,Capim,02,,-9.516667,-37.183333
br,capim,Capim,15,,-18.833333,-41.833333
br,capim,Capim,16,,-1.683333,-47.783333
br,capim,Capim,17,,-6.933333,-35.183333
br,capim,Capim,20,,-7.316667,-41.216667
br,capim,Capim,24,,-12.266667,-60.6375
br,capim,Capim,30,,-8.466667,-38.25
br,capim,Capim,31,,-5.2,-48.483333
br,capim de planta,Capim de Planta,30,,-7.883333,-37.7
br,capim de roca,Capim de Roca,06,,-4.083333,-38.3
br,capim grosso,Capim Grosso,02,,-9.833333,-37.066667
br,capim grosso,Capim Grosso,05,17322,-11.366667,-40.016667
br,capim grosso,Capim Grosso,06,,-4.866667,-38.216667
br,capim grosso,Capim Grosso,20,,-10.666667,-44.85
br,capim grosso,Capim Grosso,30,,-8.566667,-38.783333
br,capim melado,Capim Melado,21,,-22.951881,-43.325423
br,capim planta,Capim Planta,30,,-9.1,-36.25
br,capim puba,Capim Puba,05,,-13.5,-40.933333
br,capim puba,Capim Puba,20,,-6,-42.233333
br,capim queimado,Capim Queimado,22,,-5.410833,-36.570278
br,capinal,Capinal,05,,-15,-40.8
br,capinal,Capinal,13,,-3.355278,-43.4775
br,capinal i,Capínal I,13,,-3.630556,-43.481944
br,capinal ii,Capínal II,13,,-3.826667,-43.217222
br,capinao,Capinão,05,,-12.75,-39.583333
br,capinas,Capinas,11,,-19.016667,-56.35
br,capinha,Capinha,05,,-12.233333,-39.05
br,capinopolis,Capinópolis,15,12486,-18.683333,-49.583333
br,capinzal,Capinzal,14,,-16.383333,-56.583333
br,capinzal,Capinzal,26,23799,-27.333333,-51.6
br,capipi,Capipi,16,,-7.55,-57.7
br,capi ponta,Capi Ponta,04,,1.216667,-69.133333
br,capira,Capira,13,,-3.136667,-43.393333
br,capistrano,Capistrano,06,,-4.466667,-38.916667
br,capitao bonito,Capitão Bonito,23,,-28.966667,-53.266667
br,capitao bras,Capitão Brás,27,,-24.707003,-48.163716
br,capitao,Capitão,04,,-3.116667,-58.15
br,capitao,Capitão,11,,-21.583333,-56.016667
br,capitao,Capitão,28,,-11.05,-38.033333
br,capitao carutu,Capitão Carutu,14,,-14.316667,-54.7
br,capitao de campo,Capitão de Campo,13,,-3.903333,-43.496389
br,capitao de campo,Capitão de Campo,20,,-4.466667,-41.95
br,capitao de campos,Capitão de Campos,13,,-3.989167,-42.873333
br,capitao de campos,Capitão de Campos,20,,-4.466667,-41.95
br,capitao de mato,Capitão de Mato,20,,-4.466667,-41.95
br,capitao do espirito santo,Capitão do Espírito Santo,15,,-21.583333,-44.466667
br,capitao-do-mato,Capitão-do-Mato,20,,-4.466667,-41.95
br,capitao filipe,Capitão Filipe,14,,-14.166667,-54.933333
br,capitao grande,Capitão Grande,08,,-18.633333,-40.1
br,capitao laurindo,Capitão Laurindo,30,,-7.4,-37.05
br,capitao mor,Capitão Mór,06,,-5.083333,-39.366667
br,capitao mor,Capitão Mor,17,,-8.216667,-37.066667
br,capitao poco,Capitão Poço,16,32704,-1.75,-47.083333
br,capitaozinho,Capitãozinho,17,,-7.15,-36.583333
br,capitari,Capitari,04,,-6.566667,-60.333333
br,capitari,Capitari,25,,.85,-61.4
br,capitary,Capitary,25,,.85,-61.4
br,capitoa,Capitoa,28,,-11.05,-38.033333
br,capitolio,Capitólio,15,,-20.6,-46.05
br,capituva,Capituva,15,,-21.083333,-45.35
br,capituva,Capituva,18,,-25.450182,-48.824928
br,capituva,Capituva,27,,-22.883333,-45.383333
br,capivara alta,Capivara Alta,26,,-28.133333,-49.316667
br,capivara,Capivara,02,,-9.766667,-37.016667
br,capivara,Capivara,05,,-12.3,-40.4
br,capivara,Capivara,06,,-5.683333,-39.75
br,capivara,Capivara,11,,-17.316667,-56.716667
br,capivara,Capivara,15,,-19.833333,-45.2
br,capivara,Capivara,20,,-3.766667,-41.65
br,capivara,Capivara,24,,-10.183333,-62.066667
br,capivaras,Capivaras,26,,-28.133333,-49.316667
br,capivari,Capivari,05,,-12.15,-40.366667
br,capivari,Capivari,15,,-18.433333,-43.416667
br,capivari,Capivari,18,,-25.139277,-48.856601
br,capivari,Capivari,21,,-22.083333,-41.333333
br,capivari,Capivari,23,,-29.75,-55.933333
br,capivari,Capìvari,26,,-28.266667,-49.016667
br,capivari,Capivari,27,35519,-23.000722,-47.502026
br,capivari da mata,Capivari da Mata,27,,-20.366667,-47.683333
br,capivarita,Capivarita,23,,-30.3,-52.166667
br,capivary,Capivary,05,,-12.15,-40.366667
br,capivary,Capivary,15,,-22.55,-45.916667
br,capivary,Capivary,21,,-22.65,-42.383333
br,capivary,Capivary,23,,-29.75,-55.933333
br,capivary,Capivary,26,,-28.266667,-49.016667
br,capivary,Capivary,27,,-23.000722,-47.502026
br,capo da canoa,Capo da Canoa,23,,-29.766667,-50.016667
br,capoeira,Capoeira,04,,-4.464167,-61.061111
br,capoeira,Capoeira,05,,-14.433333,-43.033333
br,capoeira,Capoeira,06,,-5.45,-38.7
br,capoeira,Capoeira,13,,-3.25,-45.25
br,capoeira,Capoeira,17,,-6.766667,-37.7
br,capoeira,Capoeira,20,,-5.216667,-41.216667
br,capoeira,Capoeira,22,,-5.983333,-35.516667
br,capoeira,Capoeira,27,,-23.4515,-46.0374
br,capoeira,Capoeira,30,,-8.2,-36.683333
br,capoeira,Capoeira,31,,-5.95,-48.216667
br,capoeiracu,Capoeiraçu,05,,-12.583333,-38.966667
br,capoeira de dentro,Capoeira de Dentro,05,,-13.3,-40.333333
br,capoeira do felipao,Capoeira do Felipão,05,,-14.533333,-45.283333
br,capoeira do filipao,Capoeira do Filipão,05,,-14.533333,-45.283333
br,capoeira do rei,Capoeira do Rei,03,,.95,-50.816667
br,capoeira do santo,Capoeira do Santo,17,,-7.4,-38.166667
br,capoeira grande,Capoeira Grande,13,,-1.85,-45.433333
br,capoeira grande,Capoeira Grande,15,,-16.533333,-41.3
br,capoeira grande,Capoeira Grande,21,,-22.980374,-43.618922
br,capoeiral,Capoeiral,20,,-6.383333,-42.216667
br,capoeirana,Capoeirana,15,,-19.7,-43.066667
br,capoeirao,Capoeirão,29,,-16.266667,-49.366667
br,capoeiras,Capoeiras,17,,-7.6,-38.516667
br,capoeiras,Capoeiras,27,,-24.416667,-48.85
br,capoeiras,Capoeiras,30,4861,-8.75,-36.633333
br,capoeira vermelha,Capoeira Vermelha,17,,-7.65,-38.416667
br,capoeirinha,Capoeirinha,04,,-3.788056,-60.424444
br,capoeirinha,Capoeirinha,18,,-25.433333,-54.033333
br,capoeirinha,Capoeirinha,23,,-28.633333,-51.783333
br,capoeiro,Capoeiro,06,,-7.483333,-38.966667
br,capoeirucu,Capoeiruçu,05,,-12.583333,-38.966667
br,capo-ere,Capo-Erê,23,,-27.75,-52.266667
br,caponga,Caponga,06,,-4.033333,-38.183333
br,caporanga,Caporanga,27,,-22.716667,-49.55
br,capote,Capote,06,,-3.35,-39.516667
br,capote,Capote,15,,-22.366667,-45.55
br,capote,Capote,20,,-3.932778,-42.304444
br,capricho,Capricho,04,,-5.6,-63.05
br,caprichos,Caprichos,04,,-5.6,-63.05
br,capuaba,Capuaba,08,,-20.325,-40.329167
br,capua,Capuá,22,,-5.8,-37.783333
br,capuan,Capuan,06,,-3.716667,-38.683333
br,capuava,Capuava,27,,-23.143036,-47.728758
br,capuavinha,Capuavinha,27,,-23.552,-47.9318
br,capuera do laborde,Capuera do Laborde,04,,-5.624722,-59.546944
br,capumba,Capumba,20,,-5.566667,-43.016667
br,capunga,Capunga,30,,-8.366667,-36.666667
br,caputera,Caputera,27,,-23.7,-48.9
br,caputira,Caputira,15,,-20.166667,-42.266667
br,capuxu,Capuxu,30,,-8.416667,-36.033333
br,caquende,Caquende,15,,-21.333333,-44.55
br,caraa,Caraá,23,,-29.8,-50.3
br,carababa,Carababá,04,,-5.116667,-67.116667
br,carabucu,Carabuçu,21,,-21.216667,-41.583333
br,caraca,Caraça,18,,-24.701271,-48.883256
br,caracanga,Caracanga,06,,-3.933333,-38.533333
br,cara,Cará,15,,-21.366667,-46.033333
br,cara-cara,Cará-Cará,18,,-25.166667,-50.133333
br,caracara,Caracará,20,,-3.7,-40.033333
br,caracarahy,Caracarahy,25,,1.833333,-61.133333
br,caracarai,Caracaraí,25,,1.833333,-61.133333
br,caracinha,Caracinha,18,,-24.697085,-48.869085
br,caracituba,Caracituba,30,,-8.35,-35.366667
br,caracois,Caracóis,15,,-19.783333,-44.233333
br,caracol,Caracol,11,,-22.016667,-57.033333
br,caracol,Caracol,15,,-15.3,-44.8
br,caracol,Caracol,17,,-7.3,-38.4
br,caracol,Caracol,20,,-9.283333,-43.333333
br,caracol,Caracol,23,,-30.183333,-53.033333
br,caracol,Caracol,24,,-9.116667,-64.3
br,caracol,Caracol,31,,-7.066667,-48.033333
br,caracu,Caracu,18,,-25.533333,-51.966667
br,caracuru,Caracuru,16,,-1.066667,-52.416667
br,caraguata,Caraguatá,05,,-13.35,-41.666667
br,caraguata,Caraguatá,26,,-26.783333,-50.833333
br,caraguatai,Caraguataí,05,,-13.35,-41.666667
br,caraguatai,Caraguataí,11,,-23.566667,-55.233333
br,caraguatal,Caraguatal,27,,-23.283333,-47.633333
br,caraguatatuba,Caraguatatuba,27,90844,-23.616667,-45.416667
br,caraguatehy,Caraguatehy,11,,-23.566667,-55.233333
br,caraguatei,Caraguateí,11,,-23.566667,-55.233333
br,carahubas,Carahubas,17,,-7.716667,-36.516667
br,carahubas,Carahubas,22,,-5.783333,-37.566667
br,carahyba,Carahyba,05,,-16.8,-39.15
br,carahyba,Carahyba,30,,-8.466667,-38.533333
br,carahybeira,Carahybeira,30,,-8.916667,-37.866667
br,carahy,Carahy,15,,-17.2,-41.7
br,caraiba,Caraíba,02,,-9.716667,-36.766667
br,caraiba,Caraíba,05,,-11.466667,-38.883333
br,caraiba,Caraíba,13,,-3.067222,-42.178333
br,caraiba,Caraíba,15,,-16.633333,-40.816667
br,caraiba,Caraiba,20,,-6.533333,-42.033333
br,caraiba,Caraíba,29,,-16.916667,-48.416667
br,caraiba,Caraíba,30,,-8.416667,-37.683333
br,caraiba de santa teresa,Caraíba de Santa Teresa,05,,-9.666667,-43.666667
br,caraiba de santa tereza,Caraíba de Santa Tereza,05,,-9.666667,-43.666667
br,caraiba do norte,Caraíba do Norte,13,,-6.416667,-42.883333
br,caraibal,Caraibal,15,,-15.5,-44.75
br,caraibas,Caraíbas,02,,-9.716667,-36.766667
br,caraibas,Caraíbas,05,,-10.466667,-40.433333
br,caraibas,Caraíbas,13,,-3.433333,-42.983333
br,caraibas,Caraíbas,15,,-14.966667,-44.366667
br,caraibas,Caraíbas,17,,-6.75,-37.833333
br,caraibas,Caraíbas,20,,-10.733333,-45.266667
br,caraibas,Caraíbas,30,,-8.716667,-36.983333
br,caraibeira,Caraìbeira,17,,-6.916667,-36.533333
br,caraibeira,Caraibeira,30,,-8.916667,-37.866667
br,caraibeiras,Caraibeiras,02,,-9.216667,-38.033333
br,caraibeiras,Caraìbeiras,17,,-6.766667,-36.083333
br,caraibeiras,Caraibeiras,30,,-8.15,-36.133333
br,caraibinha,Caraibinha,20,,-6.6,-42.083333
br,caraibuna,Caraibuna,05,,-13.75,-41.15
br,carai,Caraí,15,6750,-17.2,-41.7
br,caraipe,Caraípe,05,,-12.783333,-39.083333
br,caraiuva,Caraiuva,26,,-28.883333,-49.483333
br,caraiva,Caraíva,05,,-16.8,-39.15
br,caraja,Carajá,23,,-30.716667,-53.683333
br,caraja,Carajá,27,,-23.584952,-47.313239
br,carajari,Carajari,16,,-4.766667,-54.35
br,carajas,Carajás,16,,-2.95,-51.866667
br,carajazinho,Carajàzinho,23,,-28.55,-54.333333
br,carajuru,Carajuru,04,,.383333,-67.283333
br,carambehy,Carambehy,18,,-24.933333,-50.083333
br,carambei,Carambeí,18,13466,-24.933333,-50.083333
br,carambiu,Carambiú,18,,-25.565273,-48.830079
br,caramuru,Caramuru,18,,-23.366667,-51.216667
br,caramuru,Caramuru,27,,-21.416667,-50.816667
br,carana,Caraná,16,,-1.15,-46.933333
br,carana,Caraná,25,,3.116667,-60.95
br,caranahyba,Caranahyba,15,,-20.883333,-43.733333
br,caranaiba,Caranaíba,15,,-20.883333,-43.733333
br,caranandiua,Caranandiua,13,,-1.45,-46.15
br,carananduba,Carananduba,16,,-1.1,-48.416667
br,caranapatuba,Caranapatuba,04,,-6.754167,-62.591667
br,caranda,Carandá,11,,-20.116667,-55.583333
br,caranda dos veados,Carandá dos Veados,14,,-16.5,-57.1
br,caranda fundo,Carandá Fundo,14,,-16.75,-56.783333
br,carandahy,Carandahy,15,,-20.95,-43.8
br,carandai,Carandaí,15,16657,-20.933333,-43.916667
br,carandazal,Carandàzal,11,,-18.616667,-56.1
br,carandazinho,Carandàzinho,14,,-16.45,-56.133333
br,carandina,Carandina,27,,-22.624167,-47.261944
br,caranduba,Caranduba,16,,-1.1,-48.416667
br,carandububa,Carandububa,16,,-1.1,-48.416667
br,carangola,Carangola,15,25322,-20.733333,-42.033333
br,caranguejo,Caranguejo,21,,-22.901847,-43.43154
br,carao,Carao,13,,-6.45,-42.866667
br,carao,Carão,22,,-5.202222,-36.700556
br,carapai,Carapai,23,,-29.1751,-51.0449
br,carapajo,Carapajó,16,,-2.266667,-49.366667
br,carapana,Carapanã,04,,-4.496389,-62.013611
br,carapanatuba,Carapanatuba,04,,-6.754167,-62.591667
br,caraparu,Caraparu,16,,-1.366667,-48.133333
br,carapebas,Carapebas,20,,-2.966667,-41.45
br,carapebus,Carapebus,08,,-20.233333,-40.216667
br,carapebus,Carapebus,21,,-22.183333,-41.666667
br,carapeva,Carapeva,27,,-23.2,-45.4
br,carapia,Carapiá,21,,-22.968257,-43.578135
br,carapi,Carapi,02,,-9.133333,-37.5
br,carapi,Carapi,16,,-1.15,-47.516667
br,carapicuiba,Carapicuíba,27,361120,-23.516667,-46.833333
br,carapina,Carapina,08,,-20.233333,-40.283333
br,carapina,Carapina,29,,-18.366667,-51.15
br,carapo,Carapó,11,,-22.633333,-54.8
br,carapotos,Carapotós,30,,-8.133333,-36.066667
br,caraquatchi,Caraquatchi,11,,-23.566667,-55.233333
br,carara,Carará,04,,-6.727222,-62.513611
br,carara,Carará,13,,-3.5,-45.25
br,carara poco,Carara Poço,04,,1.066667,-67.533333
br,cararbas,Carárbas,06,,-5.05,-39.183333
br,caras,Caras,06,,-3.516667,-39.933333
br,carasinho,Carasinho,23,,-28.3,-52.8
br,carasinho,Caràsinho,26,,-28.333333,-50.483333
br,carataca,Caratacá,05,,-9.933333,-39.333333
br,caratateua,Caratateua,16,,-1.25,-48.433333
br,caratatiua,Caratatiua,13,,-1.633333,-45.233333
br,caratatiua,Caratatiua,16,,-4.166667,-47.183333
br,caratianas,Caratianas,24,,-9.483333,-63.1
br,caratinga,Caratinga,15,60067,-19.783333,-42.133333
br,caratuva,Caratuva,18,,-23.96602,-50.155993
br,carauari,Carauari,04,18059,-4.879167,-66.898056
br,carauary,Carauary,04,,-4.879167,-66.898056
br,carauba,Caraúba,06,,-2.833333,-40.1
br,carauba,Caraúba,20,,-3.456944,-42.046389
br,carauba,Caraúba,22,,-5.783333,-37.566667
br,caraubas,Caraúbas,06,,-3.633333,-38.85
br,caraubas,Caraúbas,17,,-7.716667,-36.516667
br,caraubas,Caraúbas,22,12462,-5.53,-37.303056
br,caravari,Caravari,04,,-4.879167,-66.898056
br,caravelas,Caravelas,05,11322,-17.75,-39.25
br,caravelas,Caravelas,20,,-5.1,-41.85
br,caravellas,Caravellas,05,,-17.75,-39.25
br,carazinho,Caràzinho,23,59418,-28.3,-52.8
br,carazinho,Caràzinho,26,,-28.333333,-50.483333
br,carbonita,Carbonita,15,,-17.533333,-43.016667
br,carcanha,Carcanha,05,,-13.3,-39.283333
br,carcara,Carçará,20,,-6.983333,-41
br,carcara,Carcara,30,,-7.866667,-39.583333
br,cardeal da silva,Cardeal da Silva,05,,-11.95,-37.95
br,cardosa,Cardosa,06,,-2.933333,-40.283333
br,cardoso,Cardoso,04,,-4.2,-59.25
br,cardoso,Cardoso,11,,-20.65,-57.65
br,cardoso,Cardoso,15,,-15.583333,-46.8
br,cardoso,Cardoso,17,,-7.433333,-38.05
br,cardoso,Cardoso,20,,-3.729722,-42.379722
br,cardoso,Cardoso,22,,-5.55,-35.95
br,cardoso,Cardoso,27,10556,-20.066667,-49.9
br,cardoso moreira,Cardoso Moreira,21,,-21.483333,-41.616667
br,cardoso moreiro,Cardoso Moreiro,21,,-21.483333,-41.616667
br,cardosos,Cardosos,21,,-22.899295,-43.47437
br,cardosos,Cardosos,22,,-6.016667,-37.7
br,cardozo,Cardozo,04,,-4.2,-59.25
br,careacu,Careaçu,15,,-22.033333,-45.7
br,care-assu,Caré-Assú,15,,-22.033333,-45.7
br,careiro,Careiro,04,6659,-3.2,-59.75
br,careiro da varzea,Careiro da Várzea,04,1092,-3.2,-59.75
br,caresalzinho,Caresalzinho,16,,-6.233333,-57.633333
br,cares,Cares,26,,-26.35,-48.85
br,careta,Careta,20,,-6.2,-42.3
br,cariaca,Cariaçá,05,,-10.516667,-40.133333
br,caria,Cariá,20,,-6.416667,-40.833333
br,cariacica,Cariacica,08,330681,-20.266667,-40.416667
br,cariacu,Cariaçu,15,,-22.033333,-45.7
br,cariatuba,Cariatuba,06,,-6.816667,-39.516667
br,caribas,Caribas,30,,-9.05,-37.266667
br,caribe,Caribé,05,,-13.233333,-38.983333
br,caricaca,Caricaca,31,,-7.883333,-48.15
br,carice,Caricé,30,,-7.533333,-35.083333
br,caridade,Caridade,06,,-4.216667,-39.2
br,caridade,Caridade,13,,-3.592222,-43.276389
br,caridade,Caridade,20,,-7.716667,-40.983333
br,carijo grande,Carijo Grande,18,,-25.773049,-50.227195
br,carijos,Carijós,18,,-25.9892,-49.3342
br,carima,Carimã,16,,-5.2,-56.783333
br,carima,Carimã,30,,-8.85,-35.333333
br,carin,Carin,04,,-3,-60.016667
br,carinhanha,Carinhanha,05,11408,-14.3,-43.783333
br,carioca,Carioca,13,,-3.057778,-42.425556
br,carioca,Carioca,15,,-19.816667,-44.75
br,carioca,Carioca,18,,-23.410899,-52.573522
br,carios,Carios,06,,-6.533333,-39.5
br,caripare,Cariparé,05,,-11.522222,-45.055556
br,caripe,Caripé,04,,-7.833333,-60.666667
br,caripi,Caripi,03,,3.75,-51.583333
br,caripi,Caripi,16,,-1.15,-47.516667
br,carira,Carira,28,10276,-10.35,-37.7
br,carire,Cariré,20,5488,-3.95,-40.45
br,caririacu,Caririaçu,06,12971,-7.033333,-39.283333
br,cariri,Cariri,02,,-9.716667,-36.133333
br,cariri,Cariri,04,,-3,-60.016667
br,cariri,Cariri,16,,-4.45,-57.1
br,cariri,Cariri,21,,-22.846273,-43.276752
br,cariri,Cariri,25,,3.85,-59.6
br,cariri do jovelino,Cariri do Jovelino,02,,-9.716667,-36.133333
br,cariri do tocantins,Cariri do Tocantins,31,,-11.888611,-49.073333
br,cariri mirim,Cariri Mirim,30,,-7.55,-39.483333
br,caririmirim,Caririmirim,30,,-7.633333,-39.533333
br,caririzinho,Caririzinho,30,,-7.633333,-39.533333
br,cariry,Cariry,04,,-3,-60.016667
br,caritianas,Caritianas,24,,-9.483333,-63.1
br,cariuacanga,Cariuacanga,04,,-5.016667,-62.8
br,cariucanga,Cariucanga,04,,-5.016667,-62.8
br,cariugonga,Cariugonga,04,,-5.016667,-62.8
br,carius,Cariús,06,,-6.533333,-39.5
br,cariutaba,Cariutaba,06,,-6.816667,-39.516667
br,carlopolis,Carlópolis,18,,-23.416667,-49.683333
br,carlos alves,Carlos Alves,15,,-21.616667,-43.116667
br,carlos barbosa,Carlos Barbosa,23,19106,-29.3,-51.5
br,carlos chagas,Carlos Chagas,15,13206,-17.716667,-40.75
br,carlos dahne,Carlos Dahne,26,,-26.733333,-52.633333
br,carlos dane,Carlos Dane,26,,-26.733333,-52.633333
br,carlos de vasconcelos,Carlos de Vasconcelos,15,,-18.833333,-48.45
br,carlos gomes,Carlos Gomes,06,,-6.533333,-38.9
br,carlos gomes,Carlos Gomes,21,,-21.983333,-41.8
br,carlos gomes,Carlos Gomes,23,,-27.7,-51.916667
br,carlos guinle,Carlos Guinle,21,,-22.452904,-42.967568
br,carlos lobato,Carlos Lobato,15,,-20.05,-45.583333
br,carlos pinheiro,Carlos Pinheiro,05,,-16.533333,-39.916667
br,carlos ribeiro,Carlos Ribeiro,15,,-21.016667,-45.833333
br,carlota,Carlota,11,,-19.783333,-57.65
br,carmelopolis,Carmelópolis,06,,-6.883333,-40.133333
br,carmesia,Carmésia,15,,-19.083333,-43.15
br,carmo,Carmo,03,,1.266667,-50.783333
br,carmo,Carmo,05,,-12.466667,-38.45
br,carmo,Carmo,13,,-3.483333,-44.833333
br,carmo,Carmo,15,,-19.5,-43.366667
br,carmo,Carmo,16,,-1.383333,-47.7
br,carmo,Carmo,21,11555,-21.933333,-42.616667
br,carmo,Carmo,28,,-10.65,-36.983333
br,carmo,Carmo,30,,-7.75,-38.716667
br,carmo,Carmo,31,,-5.4,-48.383333
br,carmo da cachoeira,Carmo da Cachoeira,15,,-21.466667,-45.216667
br,carmo da mata,Carmo da Mata,15,,-20.55,-44.883333
br,carmo da matta,Carmo da Matta,15,,-20.55,-44.883333
br,carmo de cajuru,Carmo de Cajurú,15,,-20.183333,-44.766667
br,carmo de minas,Carmo de Minas,15,,-22.116667,-45.133333
br,carmo de pains,Carmo de Pains,15,,-20.366667,-45.666667
br,carmo do cajuru,Carmo do Cajuru,15,16333,-20.183333,-44.766667
br,carmo do mata,Carmo do Mata,15,,-20.55,-44.883333
br,carmo do paranahyba,Carmo do Paranahyba,15,,-18.983333,-46.35
br,carmo do paranaiba,Carmo do Paranaíba,15,26537,-18.983333,-46.35
br,carmo do rio claro,Carmo do Rio Claro,15,14446,-20.966667,-46.116667
br,carmo do rio verde,Carmo do Rio Verde,29,6053,-15.35,-49.7
br,carmolandia,Carmolândia,31,,-7.000556,-48.254722
br,carmona,Carmona,05,,-12.466667,-41.583333
br,carmopolis,Carmópolis,28,,-10.65,-36.983333
br,carmopolis de minas,Carmópolis de Minas,15,,-20.55,-44.633333
br,carnahyba,Carnahyba,05,,-9.583333,-40.416667
br,carnahyba,Carnahyba,30,,-7.8,-37.816667
br,carnaiba,Carnaíba,05,,-10.566667,-38.916667
br,carnaiba,Carnaíba,13,,-3.934167,-43.429444
br,carnaiba,Carnaiba,20,,-6.683333,-41.7
br,carnaiba,Carnaíba,30,6019,-8.7,-37.666667
br,carnaiba de dento,Carnaíba de Dento,20,,-6.4,-41.4
br,carnaiba de flores,Carnaíba de Flôres,30,,-7.8,-37.816667
br,carnaiba do sertao,Carnaíba do Sertão,05,,-9.583333,-40.416667
br,carnaibas,Carnaíbas,05,,-13.633333,-42.166667
br,carnaibas,Carnaibas,20,,-7.083333,-43.116667
br,carnauba amarela,Carnaúba Amarela,13,,-3.861667,-43.326944
br,carnauba,Carnauba,06,,-2.983333,-39.95
br,carnauba,Carnaúba,13,,-3.619444,-42.985278
br,carnauba,Carnaúba,17,,-6.733333,-37.666667
br,carnauba,Carnaúba,20,,-3.574444,-42.333056
br,carnauba,Carnaúba,22,,-5.75,-35.683333
br,carnauba,Carnaúba,25,,3.4,-60.2
br,carnauba,Carnaúba,30,,-7.516667,-39.833333
br,carnauba comprida,Carnaúba Comprida,20,,-3.565,-42.046389
br,carnauba de forquilha,Carnaúba de Forquilha,20,,-7.25,-41.833333
br,carnauba de pedra,Carnaúba de Pedra,13,,-5.016667,-42.966667
br,carnauba dos dantas,Carnaúba dos Dantas,22,,-6.55,-36.583333
br,carnauba do soares,Carnaúba do Soares,20,,-5.3,-41.216667
br,carnauba enterrada,Carnaúba Enterrada,22,,-5.066667,-35.883333
br,carnaubais,Carnaúbais,22,,-5.347222,-36.832778
br,carnaubal,Carnaubal,06,,-4.166667,-40.95
br,carnaubal,Carnaubal,13,,-3.366667,-44.783333
br,carnaubal,Carnaubal,14,,-15.95,-59.033333
br,carnaubal,Carnaubal,17,,-7.366667,-36.95
br,carnaubal,Carnaubal,20,,-3.649444,-42.540833
br,carnaubal,Carnaubal,22,,-5.363333,-37.489444
br,carnaubalzinho,Carnaubalzinho,06,,-3.716667,-41.3
br,carnaubalzinho,Carnaubalzinho,20,,-5.5,-41.366667
br,carnaubas,Carnaúbas,06,,-5.633333,-39.55
br,carnaubas,Carnaúbas,20,,-3.6875,-42.100278
br,carnaubas,Carnaúbas,22,,-5.951389,-37.098056
br,carnauba seca,Carnaúba Sêca,17,,-7.566667,-38.316667
br,carnauba torta,Carnaúba Torta,20,,-3.833056,-42.068333
br,carnauba verde,Carnaúba Verde,22,,-5.1,-37.316667
br,carnaubeira,Carnaubeira,13,,-2.833611,-41.964167
br,carnaubeir,Carnaubeir,13,,-2.833611,-41.964167
br,carnaubeir,Carnaubeir,30,,-8.433333,-38.85
br,carnaubinha,Carnaùbinha,06,,-5.533333,-39.15
br,carnaubinha,Carnaubinha,13,,-3.775,-43.389167
br,carnaubinha,Carnaùbinha,17,,-6.75,-38.283333
br,carnaubinha,Carnaúbinha,20,,-3.511111,-42.419444
br,carnaubinha,Carnaúbinha,22,,-5.616944,-37.131944
br,carnaubinha,Carnaùbinha,30,,-7.816667,-38.516667
br,carnaubinhas,Carnaubinhas,06,,-4.516667,-37.883333
br,carne de vaca,Carne de Vaca,30,,-7.583333,-34.833333
br,carne gorda,Carne Gorda,22,,-5.468611,-37.022778
br,carneirinho,Carneirinho,15,,-19.7,-43.233333
br,carneirinhos,Carneirinhos,15,,-19.8,-43.183333
br,carneiro,Carneiro,15,,-19.183333,-41.75
br,carneiro,Carneiro,17,,-7.25,-37.916667
br,carneiro,Carneiro,22,,-5.916667,-37.766667
br,carneiro,Carneiro,30,,-7.883333,-38.533333
br,carneiro dos medeiro,Carneiro dos Medeiro,17,,-7.666667,-37.95
br,carneiros,Carneiros,02,,-9.5,-37.433333
br,carneiros,Carneiros,15,,-22.5283,-45.7455
br,carne seca,Carne Sêca,15,,-17.383333,-43.866667
br,carnica,Carniça,26,,-28.533333,-48.816667
br,carnica,Carniça,27,,-24.739801,-48.064333
br,carnobi,Carnobi,23,,-29.7,-53.7
br,carnoio,Carnoió,17,,-7.483333,-36.116667
br,caroa,Caroá,17,,-7.466667,-37.633333
br,caroa,Caroá,30,,-8.85,-37.766667
br,caroara,Caroará,13,,-4.133333,-43.3
br,carobas,Carobas,06,,-5.516667,-38.416667
br,carobinha,Carobinha,13,,-3.8125,-42.761944
br,carobinha,Carobinha,21,,-22.855311,-43.531893
br,caro custou,Caro Custou,06,,-6.15,-38.9
br,carolina belo,Carolina Belo,15,,-20.533333,-45.616667
br,carolina,Carolina,08,,-20.533333,-40.833333
br,carolina,Carolina,11,,-20.333333,-55.533333
br,carolina,Carolina,13,16229,-7.333333,-47.466667
br,carolino cruz,Carolino Cruz,15,,-15.166667,-43.35
br,carona,Carona,14,,-16.416667,-56.433333
br,caropanatuba,Caropanatuba,04,,-6.754167,-62.591667
br,caroveira,Caroveira,18,,-25.166667,-53.55
br,carovi,Carovi,23,,-28.933333,-54.833333
br,carpina,Carpina,20,,-3.033333,-41.35
br,carpina,Carpina,30,70690,-7.85,-35.25
br,carqueija,Carqueija,18,,-25.816675,-50.057254
br,carqueja,Carquêja,06,,-4.466667,-38.933333
br,carqueja,Carqueja,30,,-8.333333,-38.416667
br,carquejo,Carquejo,06,,-3.866667,-40.733333
br,carquejo,Carquejo,20,,-7.333333,-41
br,carranca,Carranca,13,,-3.911944,-43.033611
br,carrancas,Carrancas,15,,-21.5,-44.65
br,carranco,Carranco,17,,-7.666667,-38.316667
br,carrancudo,Carrancudo,20,,-6.8,-40.516667
br,carrapateira,Carrapateira,06,,-2.816667,-40.316667
br,carrapateira,Carrapateira,17,,-6.35,-36.35
br,carrapateira,Carrapateira,22,,-5.014444,-37.473611
br,carrapateira,Carrapateira,30,,-8.6,-36.866667
br,carrapateiras,Carrapateiras,06,,-5.95,-40.366667
br,carrapatinho,Carrapatinho,11,,-19.833333,-56.416667
br,carrapato,Carrapato,05,,-13.566667,-41.116667
br,carrapato,Carrapato,13,,-3.661389,-42.840833
br,carrapato,Carrapato,17,,-6.616667,-37.7
br,carrapato,Carrapato,20,,-6.066667,-42.683333
br,carrapato,Carrapato,21,,-21.4,-41.066667
br,carrapichel,Carrapichel,05,,-10.4,-40.183333
br,carrapicho,Carrapicho,06,,-4.9,-38.616667
br,carrapicho,Carrapicho,30,,-7.9,-36.333333
br,carrascao,Carrascão,13,,-3.636944,-43.225556
br,carrasco bonito,Carrasco Bonito,31,,-5.224444,-47.828611
br,carrasco,Carrasco,20,,-5.766667,-42.566667
br,carrasco,Carrasco,31,,-6.383333,-47.75
br,carrasco da faveira,Carrasco da Faveira,20,,-6.25,-42.483333
br,carrasquinho,Carrasquinho,20,,-3.9,-42.558333
br,carrazedo,Carrazedo,16,,-1.6,-51.9
br,carregador,Carregador,14,,-16.016667,-58.35
br,carreira,Carreira,20,,-3.491389,-42.483611
br,carreiras,Carreiras,15,,-20.583333,-43.733333
br,carreiras,Carreiras,20,,-7.55,-41.866667
br,carretao,Carretão,13,,-9.45,-46.166667
br,carretao,Carretão,17,,-6.616667,-38.516667
br,carretao,Carretão,20,,-5.9,-42.466667
br,carrocal,Carroçal,04,,-6.883333,-57.883333
br,carrocalzinho,Carroçalzinho,16,,-6.233333,-57.633333
br,carro,Carro,20,,-6.333333,-42.25
br,carro de boi,Carro de Boi,14,,-16.233333,-56.583333
br,carro quebrado,Carro Quebrado,20,,-7.25,-41.033333
br,carro quebrado,Carro Quebrado,22,,-4.962222,-37.180556
br,carro velho,Carro Velho,20,,-6.6,-42.316667
br,carsevenne,Carsevenne,03,,2.5,-50.95
br,cartucho,Cartucho,31,,-8.084722,-47.508056
br,carua,Caruá,05,,-15.966667,-39.8
br,carualina,Carualina,30,,-8.3,-37.583333
br,caruara,Caruara,13,,-3.4425,-42.652222
br,caruara,Caruara,23,,-29.25,-51.416667
br,caruaru,Caruaru,30,235375,-8.283333,-35.966667
br,caruatai,Caruataí,06,,-3.816667,-40.966667
br,caruauba,Caruaúba,06,,-5.083333,-38.916667
br,caru,Caru,26,,-27.666667,-50.583333
br,carumbe,Carumbé,11,,-21.983333,-55.016667
br,caruru,Caruru,13,,-4.766667,-46.583333
br,carutapera,Carutapera,13,12819,-1.216667,-46.016667
br,carvalho,Carvalho,16,,-2.283333,-51.45
br,carvalho,Carvalho,27,,-20.483333,-50.016667
br,carvalhos,Carvalhos,06,,-4.75,-40.433333
br,carvalhos,Carvalhos,15,,-22.083333,-44.483333
br,carvoal,Carvoal,14,,-16.383333,-56.3
br,carvoalzinho,Carvoalzinho,14,,-16.533333,-56.75
br,carvo branco,Carvo Branco,18,,-25.566667,-52.05
br,carvoeira,Carvoeira,04,,-1.383333,-62
br,carvoeiro,Carvoeiro,04,,-1.383333,-62
br,carvoeiro,Carvoeiro,06,,-3.2,-39.983333
br,carvoeiro,Carvoeiro,17,,-7.316667,-38.25
br,carvoeiro,Carvoeiro,22,,-6.116667,-37.733333
br,casa amarela,Casa Amarela,21,,-22.940569,-43.470067
br,casa branca,Casa Branca,05,,-12.183333,-41.183333
br,casa branca,Casa Branca,14,,-17.633333,-55.266667
br,casa branca,Casa Branca,15,,-20.1,-44.05
br,casa branca,Casa Branca,21,,-21.6,-41.716667
br,casa branca,Casa Branca,23,,-30.7074,-52.3773
br,casa branca,Casa Branca,25,,3.686579,-59.726517
br,casa branca,Casa Branca,27,22498,-21.766667,-47.083333
br,casa branca da serra,Casa Branca da Serra,21,,-22.474836,-42.996085
br,casa casimiro,Casa Casimiro,04,,-3.133333,-59.333333
br,casa da serra,Casa da Serra,05,,-12.116667,-39.35
br,casa das pedras,Casa das Pedras,18,,-25.335465,-48.908361
br,casa da telha,Casa da Telha,05,,-13.566667,-41.716667
br,casa de pedra,Casa de Pedra,20,,-4.233333,-41.116667
br,casa de pedra,Casa de Pedra,30,,-7.85,-39.616667
br,casa de telha,Casa de Telha,15,,-18.383333,-43.183333
br,casa do crispim,Casa do Crispim,31,,-10.501389,-49.700278
br,casa do rodrigues,Casa do Rodrigues,04,,-3.55,-60.883333
br,casa forte,Casa Forte,06,,-5.7,-39.733333
br,casa grande,Casa Grande,05,,-13.666667,-41.816667
br,casa grande,Casa Grande,15,,-20.783333,-43.933333
br,casa grande,Casa Grande,27,,-22.433333,-49.933333
br,casal,Casal,21,,-22.25,-43.5
br,casalvasco,Casalvasco,14,,-15.316667,-59.983333
br,casa morta,Casa Morta,05,,-9.95,-41.366667
br,casa nova,Casa Nova,05,33319,-9.416667,-41.133333
br,casa nova,Casa Nova,06,,-5.683333,-38.933333
br,casa nova,Casa Nova,11,,-21.166667,-54.05
br,casa nova,Casa Nova,20,,-5.766667,-42.15
br,casas velhas,Casas Velhas,17,,-6.566667,-38.466667
br,casa velha,Casa Velha,05,,-14.316667,-43.283333
br,casa verde,Casa Verde,05,,-9.883333,-42.7
br,casca,Casca,23,,-28.566667,-51.983333
br,casca de pedra,Casca de Pedra,15,,-17.833333,-40.816667
br,cascadura,Cascadura,21,,-22.886932,-43.330788
br,cascalho,Cascalho,05,,-14.766667,-43.566667
br,cascalho,Cascalho,21,,-22.991201,-43.498671
br,cascalho,Cascalho,26,,-27.2,-53.333333
br,cascalho rico,Cascalho Rico,15,,-18.566667,-47.866667
br,cascao,Cascão,15,,-18.466667,-48.166667
br,cascao,Cascão,17,,-6.933333,-37.85
br,cascata,Cascata,05,,-17.466667,-39.8
br,cascata,Cascata,13,,-8.709444,-46.720556
br,cascata,Cascata,23,,-27.533333,-54.25
br,cascatinha,Cascatinha,18,,-26.2,-51.366667
br,cascatinha,Cascatinha,21,,-22.481111,-43.155556
br,cascavel,Cascavel,05,,-13.166667,-41.4
br,cascavel,Cascavel,06,52357,-4.116667,-38.233333
br,cascavel,Cascavel,11,,-21.933333,-57.783333
br,cascavel,Cascavel,15,,-20.55,-47.066667
br,cascavel,Cascavel,17,,-8.166667,-36.766667
br,cascavel,Cascavel,18,257174,-24.95,-53.466667
br,cascavel,Cascavel,20,,-7.466667,-44.116667
br,cascavel,Cascavel,27,,-22.066667,-46.966667
br,cascavel de alexandre,Cascavel de Alexandre,17,,-8.15,-36.783333
br,cascavel do alexandre,Cascavel do Alexandre,17,,-8.15,-36.783333
br,cascudo,Cascudo,05,,-12.533333,-41.783333
br,cascudo,Cascudo,13,,-3.125,-43.388889
br,caseara,Caseara,31,,-9.211111,-49.836944
br,caseiros,Caseiros,23,,-28.266667,-51.7
br,casemiro,Casemiro,06,,-6.55,-40.65
br,casemiro costa,Casemiro Costa,15,,-20.85,-46.15
br,casemiro de abreu,Casemiro de Abreu,21,,-22.483333,-42.2
br,casimiro andrade,Casimiro Andrade,05,,-13.666667,-40.183333
br,casimiro,Casimiro,03,,.366667,-51.216667
br,casimiro,Casimiro,04,,-6.733333,-59.083333
br,casimiro de abreu,Casimiro de Abreu,21,19087,-22.483333,-42.2
br,casimiro pulgaz,Casimiro Pulgaz,23,,-29.0678,-51.0303
br,casinha,Casinha,16,,-1.083333,-49.65
br,casinha,Casinha,30,,-7.766667,-35.666667
br,casinhas,Casinhas,30,,-7.766667,-35.666667
br,casino,Casino,23,,-32.183333,-52.166667
br,casmorra,Casmorra,05,,-12.833333,-39.116667
br,casquilho,Casquilho,05,,-13.133333,-41.6
br,cassange,Cassange,14,,-16.8925,-56.720833
br,cassange,Cassange,15,,-21.2,-44.716667
br,cassarangongo,Cassarangongo,05,,-12.6,-38.616667
br,casseb,Casseb,16,,-.883333,-47.233333
br,cassia,Cassiã,04,,-5.915556,-64.5125
br,cassia,Cássia,15,14391,-20.6,-46.933333
br,cassia dos coqueiros,Cássia dos Coqueiros,27,,-21.283333,-47.166667
br,cassiana,Cassianã,04,,-7.266667,-64.883333
br,cassiano,Cassiano,15,,-18.883333,-42.25
br,cassiara,Cassiara,04,,-7.266667,-64.883333
br,cassilandia,Cassilândia,11,18498,-19.15,-51.75
br,cassino,Cassino,23,,-32.183333,-52.166667
br,cassinum,Cassinum,05,,-12.683333,-38.866667
br,cassio,Cássio,11,,-21.2,-56.333333
br,cassipore,Cassipore,03,,3.433333,-51.166667
br,cassique doble,Cassique Dóble,23,,-27.75,-51.616667
br,cassiterita,Cassiterita,15,,-21.116667,-44.466667
br,casso,Cassó,13,,-3.016667,-43.35
br,cassual,Cassual,18,,-25.494141,-48.327875
br,cassu,Cassú,29,,-18.616667,-51.066667
br,cassununga,Cassununga,14,,-16.05,-53.633333
br,cassurova,Cassurova,27,,-22.45,-47.983333
br,casta,Casta,20,,-3.95,-42.483333
br,castanarana,Castanarana,04,,-5.897778,-61.381389
br,castanha,Castanha,04,,-2.9,-57.55
br,castanha,Castanha,22,,-6.116667,-35.383333
br,castanha,Castanha,23,,-32.1168,-52.5985
br,castanha,Castanha,25,,.881111,-61.658333
br,castanha-cupu,Castanha-Cupu,04,,-4.167778,-60.825833
br,castanhal,Castanhal,04,,-2.9,-57.55
br,castanhal,Castanhal,16,137410,-1.3,-47.916667
br,castanhal,Castanhal,24,,-11.366667,-61.666667
br,castanhal grande,Castanhal Grande,16,,-1.716667,-55.433333
br,castanha-mirim,Castanha-mirim,04,,-4.2625,-62.193056
br,castanhao,Castanhão,05,,-13.95,-40.166667
br,castanhao,Castanhão,06,,-5.466667,-38.383333
br,castanheira,Castanheira,01,,-10.416667,-68.016667
br,castanheira,Castanheira,04,,-2.831111,-60.75
br,castanheira,Castanheira,05,,-11.8,-42.266667
br,castanheiro,Castanheiro,01,,-10.683333,-69.583333
br,castanheiro,Castanheiro,04,,-2.831111,-60.75
br,castanheiro,Castanheiro,20,,-9.816667,-45.283333
br,castanho,Castanho,04,,.35,-64.016667
br,castanho,Castanho,16,,-1.316667,-46.9
br,castanho,Castanho,17,,-6.533333,-37.666667
br,castelandia,Castelândia,20,,-5.85,-43.083333
br,castel,Castel,16,,-5.066667,-56.35
br,castelhano,Castelhano,20,,-2.816667,-40.35
br,castelhanos,Castelhanos,20,,-2.816667,-40.35
br,castelhanos de fora,Castelhanos de Fora,18,,-25.5722,-49.95
br,castell,Castell,16,,-5.066667,-56.35
br,castello,Castello,08,,-20.6,-41.2
br,castello,Castello,11,,-18.633333,-57.55
br,castello,Castello,20,,-5.333333,-41.55
br,castelo,Castelo,04,,-3.5,-68.3
br,castelo,Castelo,08,20198,-20.6,-41.2
br,castelo,Castelo,11,,-18.633333,-57.55
br,castelo,Castelo,13,,-3.5,-45.533333
br,castelo,Castelo,16,,-5.066667,-56.35
br,castelo,Castelo,17,,-6.783333,-37.533333
br,castelo,Castelo,18,,-25.166667,-53.266667
br,castelo,Castelo,20,,-5.333333,-41.55
br,castelo,Castelo,21,,-22.909151,-43.172155
br,castelo da torre,Castelo da Tôrre,05,,-12.583333,-38.033333
br,castelo do piaui,Castelo do Piauí,20,8589,-5.333333,-41.55
br,castelo novo,Castelo Novo,05,,-14.633333,-39.2
br,castilho,Castilho,27,11419,-20.866667,-51.483333
br,castilhos,Castilhos,23,,-27.816667,-52.166667
br,castolandia,Castolândia,06,,-5.516667,-38.266667
br,castrinopolis,Castrinópolis,29,,-15.35,-49.5
br,castro alres,Castro Alres,23,,-29.033333,-51.416667
br,castro alves,Castro Alves,04,,-3.366667,-68.95
br,castro alves,Castro Alves,05,13869,-12.75,-39.433333
br,castro alves,Castro Alves,23,,-29.033333,-51.416667
br,castro,Castro,02,,-10.216667,-36.783333
br,castro,Castro,18,42092,-24.783333,-50
br,castro,Castro,20,,-3.95,-42.483333
br,castro,Castro,22,,-6.516667,-38.166667
br,castro,Castro,28,,-11.416667,-37.416667
br,castrolandia,Castrolândia,29,,-15.35,-49.5
br,casualidade,Casualidade,11,,-22.05,-57
br,cata branca,Cata Branca,15,,-20.2224,-43.8477
br,catadupas,Catadupas,15,,-22.35,-45.783333
br,cataguarinho,Cataguarinho,15,,-21.3,-42.716667
br,cataguarino,Cataguarino,15,,-21.3,-42.716667
br,cataguases,Cataguases,15,62619,-21.4,-42.683333
br,cataguazes,Cataguazes,15,,-21.4,-42.683333
br,catajal,Catajal,15,,-16.633333,-40.333333
br,catakeamain,Catakeamain,24,,-11.133333,-63.15
br,catalao,Catalão,29,63545,-18.166667,-47.95
br,catana,Catana,05,,-12.083333,-38.083333
br,catanduva,Catanduva,23,,-29.766667,-50.566667
br,catanduva,Catanduva,27,109615,-21.133333,-48.966667
br,catanduva de dentro,Catanduva de Dentro,18,,-24.966667,-49.883333
br,catanduvas,Catanduvas,18,,-25.2,-53.133333
br,catanduvas,Catanduvas,26,,-27.066667,-51.683333
br,catangara,Catangara,26,,-26.9411,-49.9372
br,catao,Catão,05,,-12.3,-44.833333
br,cataqueama,Cataqueamã,24,,-11.133333,-63.15
br,cataquesma,Cataquesmã,24,,-11.133333,-63.15
br,catarina,Catarina,06,,-2.966667,-39.983333
br,catarina,Catarina,20,,-5.133333,-42.783333
br,catarina guimaraes,Catarina Guimarães,11,,-19.416667,-54.65
br,catarinenses,Catarinenses,18,,-23.033333,-52.666667
br,catas altas,Catas Altas,15,,-20.066667,-43.4
br,catas altas da noruega,Catas Altas da Noruega,15,,-20.683333,-43.5
br,cataua,Catauá,15,,-20.783333,-43.683333
br,catauara,Catauara,16,,-3.35,-56.383333
br,catavento,Catavento,06,,-6.216667,-38.666667
br,ca-te-espero,Cá-te-espero,04,,-5.859722,-61.326389
br,ca-te-espero,Cá-te-Espero,16,,-1.333333,-50.416667
br,catende,Catende,30,23323,-8.666667,-35.716667
br,catequese,Catequese,27,,-22.433333,-50.15
br,cateriangongo,Cateriangongo,15,,-16.083333,-41.5
br,catesporo,Catesporo,16,,-1.333333,-50.416667
br,catete,Catete,13,,-5.583333,-45.4
br,catete,Catete,15,,-21.166667,-46.366667
br,catete,Catete,21,,-22.925416,-43.181371
br,catete,Catete,26,,-26.730848,-49.543817
br,cateto,Cateto,18,,-23.2024,-50.2483
br,cateto,Cateto,27,,-20.4,-48.7
br,catetu,Catetu,06,,-2.85,-40.183333
br,cathe,Cathé,18,,-24.033333,-50.45
br,catiara,Catiara,15,,-19.166667,-46.666667
br,catigua,Catiguá,27,,-21.05,-49.05
br,catimbau,Catimbau,30,,-8.6,-37.25
br,catinga,Catinga,04,,-4.171944,-60.616944
br,catinga,Catinga,06,,-5.333333,-40.583333
br,catinga,Catinga,15,,-17.166667,-45.883333
br,catinga,Catinga,17,,-7.033333,-36.6
br,catinga da areia,Catinga da Areia,22,,-5.619722,-36.076667
br,catinga da moura,Catinga da Moura,05,,-10.966667,-40.75
br,catinga de cima,Catinga de Cima,05,,-11.283333,-38.033333
br,catinga de porco,Catinga de Porco,20,,-6.516667,-42.266667
br,catinga do moura,Catinga do Moura,05,,-10.966667,-40.75
br,catinga grande,Catinga Grande,06,,-6.483333,-39.05
br,catingal,Catingal,05,,-14.066667,-40.216667
br,catingas,Catingas,05,,-13.6,-39.983333
br,catingueira,Catingueira,06,,-6.3,-39.1
br,catingueira,Catingueira,17,,-6.5,-38.466667
br,catingueiro,Catingueiro,06,,-5.7,-40.633333
br,catingueiro,Catingueiro,20,,-10.65,-45.183333
br,catingueiro,Catingueiro,27,,-22.166667,-46.866667
br,catinguinha,Catinguinha,13,,-3.346944,-42.374167
br,catinqueiro,Catinqueiro,20,,-10.65,-45.183333
br,catipe,Catipé,13,,-3.119167,-42.403333
br,catipora,Catiporã,23,,-29.016667,-51.7
br,catirina,Catirina,06,,-7.233333,-38.966667
br,catita,Catita,05,,-9.516667,-43.016667
br,catita,Catita,15,,-19.55,-44.733333
br,catita,Catita,17,,-8.15,-36.983333
br,catole,Catolé,05,,-12.633333,-38.9
br,catole,Catolé,06,,-4.066667,-38.5
br,catole,Catolé,17,,-6.316667,-37.683333
br,catole,Catolé,22,,-5.383333,-35.466667
br,catole,Catolé,30,,-7.533333,-39.783333
br,catole do monte,Catolé do Monte,17,,-7.383333,-36.1
br,catole do rocha,Catolé do Rocha,17,19812,-6.35,-37.75
br,catole sabambaia de cima,Catolé Sabambaia de Cima,05,,-13.283333,-41.883333
br,catoles,Catolés,05,,-13.3,-41.833333
br,catolezinho,Catolèzinho,05,,-15.133333,-40.45
br,catolezinho,Catolézinho,06,,-5.783333,-39.65
br,catolezinho,Catolèzinho,17,,-6.483333,-37.95
br,catolezinho,Catolèzinho,22,,-6.2,-38.333333
br,catone,Catoné,15,,-20.916667,-42.15
br,catonho,Catonho,21,,-22.907192,-43.37531
br,catonho,Catonho,30,,-8.666667,-37.216667
br,catorze de julho,Catorze de Julho,21,,-22.764965,-43.289678
br,catrimani,Catrimani,25,,.45,-61.683333
br,catrimany,Catrimany,25,,.45,-61.683333
br,cattas altas,Cattas Altas,15,,-20.066667,-43.4
br,cattas altas de noruega,Cattas Altas de Noruega,15,,-20.683333,-43.5
br,catuana,Catuana,06,,-3.683333,-38.916667
br,catucaba,Catuçaba,27,,-23.25,-45.2
br,catu,Catu,05,39933,-12.35,-38.383333
br,catude,Catude,11,,-20.133333,-56.083333
br,catugi,Catugi,15,,-17.333333,-41.5
br,catugi,Catugi,18,,-23.766667,-51.416667
br,catu grande,Catu Grande,05,,-12.7,-38.866667
br,catuicara,Catuiçara,05,,-12.283333,-38.6
br,catuipe,Catuípe,23,,-28.25,-54.016667
br,catuira,Catuíra,26,,-27.666667,-49.4
br,catule,Catulé,05,,-14.916667,-45.433333
br,catule,Catulé,06,,-5.933333,-39.733333
br,catule-do-rocha,Catulé-do-Rocha,17,,-6.35,-37.75
br,catules,Catulés,05,,-13.3,-41.833333
br,catumbi,Catumbi,18,,-25.366667,-48.7
br,catumbi,Catumbi,21,,-22.916504,-43.193204
br,catunda,Catunda,06,,-4.533333,-40.2
br,catune,Catuné,15,,-20.916667,-42.15
br,catuni,Catuni,05,,-10.316667,-40.183333
br,catuni,Catuni,15,,-16.25,-43.316667
br,catu pequeno,Catu Pequeno,05,,-12.733333,-38.866667
br,catupi,Catupi,23,,-29.616667,-51.6
br,caturai,Caturaí,29,,-16.466667,-49.5
br,caturama,Caturama,05,,-13.316667,-42.3
br,catureta,Caturetã,23,,-29.166667,-51.633333
br,caturia,Caturiá,04,,-3.5,-68.416667
br,caturiaro,Caturiaro,20,,-6.933333,-41.733333
br,caturite,Caturité,17,,-7.4,-35.983333
br,caua,Cauã,22,,-5.833333,-35.983333
br,cauacu,Cauaçu,22,,-5.466667,-35.3
br,cauan,Cauan,22,,-5.083333,-35.8
br,cauarru,Cauarrú,13,,-5.1,-43.85
br,cauassu,Cauassu,17,,-6.883333,-37.483333
br,caubi,Caubi,05,,-14.2,-39.033333
br,caubi,Caubi,25,,2.533333,-60.933333
br,caucaia,Caucaia,06,275023,-3.7,-38.65
br,caucaia do alto,Caucaia do Alto,27,,-23.687967,-47.020966
br,cauhipe,Cauhipe,06,,-3.7,-38.85
br,cauipe,Cauípe,06,,-3.7,-38.85
br,cauitau,Cauitau,04,,-3.483333,-66.066667
br,cauna,Caúna,26,,-26.35,-50.883333
br,causa boa,Causa Boa,05,,-12.85,-41.316667
br,cavaca,Cavaca,13,,-3.826944,-43.419722
br,cava,Cava,15,,-21.516667,-45.333333
br,cava,Cava,21,,-22.684722,-43.459444
br,cavaco,Cavaco,05,,-13.05,-39.65
br,cavaco,Cavaco,15,,-21.166667,-46.1
br,cavaco,Cavaco,18,,-25.2,-52.116667
br,cavaco,Cavaco,30,,-8.383333,-37.7
br,cava da onca,Cava da Onça,05,,-12.283333,-38.35
br,cavagnoli,Cavagnoli,18,,-26.150278,-53.576944
br,cavaidaruco,Cavaidaruco,04,,1.1,-67.25
br,cavalaria,Cavalaria,06,,-4.183333,-38.516667
br,cavalcante,Cavalcante,24,,-8.4,-63.4
br,cavalcante,Cavalcante,29,3747,-13.8,-47.5
br,cavalcanti,Cavalcanti,21,,-22.87171,-43.315977
br,cavalcanti,Cavalcanti,24,,-8.4,-63.4
br,cavalcanti,Cavalcanti,29,,-13.8,-47.5
br,cavaleiro,Cavaleiro,20,,-9.1,-42.366667
br,cavaleiro,Cavaleiro,30,,-8.033333,-35.016667
br,cavalheiro,Cavalheiro,20,,-9.1,-42.366667
br,cavalheiro,Cavalheiro,29,,-17.25,-48.033333
br,cavalheiros,Cavalheiros,29,,-17.25,-48.033333
br,cavalin,Cavalin,27,,-22.233333,-50.15
br,cavalinho,Cavalinho,08,,-19.7,-40.383333
br,cavalo,Cavalo,14,,-14.683333,-56.283333
br,cavalo,Cavalo,28,,-9.983333,-37.066667
br,cavalo-de agua,Cavalo-de Água,27,,-24.799,-48.4848
br,cavalos,Cavalos,06,,-5.466667,-38.983333
br,cavalos,Cavalos,13,,-9.9,-46.316667
br,cavaru,Cavaru,21,,-22.250556,-43.346667
br,caveira,Caveira,30,,-7.583333,-35.216667
br,caveiras,Caveiras,05,,-12.8,-39.05
br,caveiras,Caveiras,23,,-30.85,-54.85
br,caverazinho,Caveràzinho,26,,-28.966667,-49.5
br,caverria do diabo,Caverria do Diabo,27,,-24.6355,-48.4044
br,caviana,Cavianã,13,,-3.561944,-42.705
br,cavi,Cavi,27,,-24.721006,-48.060175
br,caviuna,Caviúna,18,,-23.3,-51.366667
br,caviuna,Caviúna,27,,-24.579564,-48.960685
br,cavunge,Cavunge,05,,-12.25,-39.333333
br,cavuvu,Cavuvu,27,,-24.6854,-48.3839
br,caxambu,Caxambu,15,22701,-21.983333,-44.933333
br,caxambu,Caxambu,26,,-27.166667,-52.866667
br,caxanga,Caxangá,16,,-6.6,-51.983333
br,caxanga,Caxangá,30,,-8.483333,-35.4
br,caxe,Caxé,20,,-8.45,-42.4
br,caxeta,Caxeta,26,,-26.804474,-49.581785
br,caxias,Caxias,04,,-4.4825,-71.435556
br,caxias,Caxias,13,117393,-4.833333,-43.35
br,caxias,Caxias,21,,-22.790321,-43.325009
br,caxias,Caxias,23,,-29.1689,-51.1785
br,caxias das aldeias altas,Caxias das Aldeias Altas,13,,-4.833333,-43.35
br,caxias do sul,Caxias do Sul,23,381277,-29.1689,-51.1785
br,caximba,Caximba,27,,-21.25,-50.233333
br,caxinana,Caxinana,04,,-6.75,-70.133333
br,caxinaua,Caxinauá,01,,-8.716667,-71.883333
br,caxinaua,Caxinauá,04,,-6.75,-70.133333
br,caxinga,Caxingá,20,,-3.366667,-41.933333
br,caxingo,Caxingó,05,,-12.25,-42.766667
br,caxingo,Caxingó,13,,-3.620556,-42.655278
br,caxingo,Caxingó,20,,-10.498056,-45.220278
br,caxingo,Caxingó,30,,-8.9,-36.6
br,caxitore,Caxitoré,06,,-3.733333,-39.35
br,caxius do sul,Caxius do Sul,23,,-29.1689,-51.1785
br,caxixa,Caxixa,08,,-20.483333,-41.133333
br,caxumba,Caxumba,21,,-22.282132,-42.983321
br,cayambe,Cayambé,04,,-3.566667,-64.383333
br,cayaue,Cayaué,04,,-3.05,-65.966667
br,cayru,Cayru,05,,-13.5,-39.05
br,cayua,Cayuá,27,,-21.883333,-51.966667
br,caze,Cazé,06,,-5.1,-40.283333
br,cazuqui,Cazuqui,20,,-7.2,-41.85
br,cazuza,Cazuza,06,,-5.616667,-39.866667
br,cazuza ferreira,Cazuza Ferreira,23,,-28.95,-50.666667
br,ceara,Ceará,01,,-7.983333,-71.983333
br,ceara,Ceará,06,,-3.716667,-38.5
br,ceara,Ceará,13,,-1.75,-44.833333
br,ceara-mirim,Ceará-Mirim,22,32668,-5.633333,-35.433333
br,cearenses,Cearenses,08,,-19.666667,-40.833333
br,cebolao,Cebolão,18,,-23.466667,-50.883333
br,ceci,Ceci,11,,-23.45,-55
br,cecilia,Cecília,06,,-5.033333,-40.1
br,cedral,Cedral,27,,-20.9,-49.316667
br,cedro,Cedro,02,,-10,-36.6
br,cedro,Cedro,05,,-12.416667,-38.95
br,cedro,Cedro,06,14644,-6.6,-39.05
br,cedro,Cedro,11,,-18.283333,-55.033333
br,cedro,Cedro,13,,-2.533333,-43.55
br,cedro,Cedro,14,,-13.283333,-56.1
br,cedro,Cedro,15,,-16.666667,-43.916667
br,cedro,Cedro,16,,-5.6,-49.233333
br,cedro,Cedro,17,,-7.266667,-38.483333
br,cedro,Cedro,20,,-5.35,-42.466667
br,cedro,Cedro,21,,-22.938291,-43.458891
br,cedro,Cedro,27,,-24.9558,-48.4778
br,cedro,Cedro,28,,-10.25,-36.883333
br,cedro,Cedro,29,,-17.566667,-52.6
br,cedro,Cedro,30,,-7.5,-39.6
br,cedro de cima,Cedro de Cima,17,,-6.933333,-38.283333
br,cedro de sao joao,Cedro de São João,28,,-10.25,-36.883333
br,cedro do abaete,Cedro do Abaeté,15,,-19.133333,-45.7
br,cedrolandia,Cedrolândia,08,,-18.8,-40.683333
br,cedrolina,Cedrolina,29,,-14.5,-49.566667
br,cedro medio,Cedro Médio,26,,-28.716667,-49.583333
br,cego,Cego,16,,-.25,-50.033333
br,cego,Cego,30,,-8.483333,-36.816667
br,ceilandia,Ceilândia,07,,-15.816667,-48.116667
br,celina,Celina,08,,-20.766667,-41.583333
br,celso,Celso,03,,1.933333,-50.8
br,celsolandia,Celsolândia,06,,-2.983333,-40.116667
br,celso ramos,Celso Ramos,26,,-27.616667,-51.4
br,ceminario gravatai,Ceminário Gravataí,23,,-29.9399,-50.9888
br,cemiterio grande,Cemitério Grande,25,,-2.05,-61.016667
br,cemoaba,Cemoaba,06,,-3.433333,-39.433333
br,centenario,Centenário,11,,-19.466667,-56.05
br,centenario,Centenário,15,,-19.683333,-41.466667
br,centenario,Centenário,21,,-22.780661,-43.325123
br,centenario,Centenário,23,,-27.783333,-52.05
br,centenario,Centenário,31,,-8.894722,-47.183333
br,centenario do sul,Centenário do Sul,18,8358,-22.8,-51.616667
br,central,Central,03,,-.333333,-51.6
br,central,Central,05,,-11.133333,-42.133333
br,central,Central,06,,-6.066667,-40.433333
br,central,Central,18,,-25.470348,-48.837065
br,central de minas,Central de Minas,15,,-18.75,-41.333333
br,central de santa helena,Central de Santa Helena,15,,-18.75,-41.583333
br,centralina,Centralina,15,9497,-18.566667,-49.216667
br,centralito,Centralito,18,,-24.983333,-53.333333
br,central lupion,Central Lupion,18,,-25,-53.566667
br,central santa cruz,Central Santa Cruz,18,,-24.683333,-53.383333
br,centro,Centro,13,,-3.125278,-42.7375
br,centro,Centro,15,,-22.6323,-45.5495
br,centro,Centro,20,,-4.483333,-41.5
br,centro,Centro,21,,-22.891313,-43.126134
br,centro da agua branca,Centro da Água Branca,13,,-3.902778,-43.261944
br,centro da agua rica,Centro da Água Rica,13,,-3.7025,-43.036667
br,centro da barra,Centro da Barra,21,,-23.004838,-43.323975
br,centro da mata da varjota,Centro da Mata da Varjota,13,,-3.986389,-43.415278
br,centro da sabina,Centro da Sabina,13,,-3.428333,-43.143333
br,centro do diamante,Centro do Diamante,13,,-5.283333,-43.333333
br,centro dos abreus,Centro dos Abreus,13,,-3.671667,-42.847222
br,centro do santo antonio,Centro do Santo Antônio,13,,-3.375278,-43.495556
br,centro dos boas,Centro dos Boas,13,,-4.066667,-45.166667
br,centro dos ferreiras,Centro dos Ferreiras,13,,-3.639722,-43.443333
br,centro dos gomes,Centro dos Gomes,13,,-3.744722,-43.183611
br,centro dos jeronimos,Centro dos Jerônímos,13,,-3.656944,-43.311667
br,centro dos jonas,Centro dos Jonas,13,,-3.666389,-43.320556
br,centro dos lopes,Centro dos Lopes,13,,-3.673889,-43.330278
br,centro dos messias,Centro dos Messias,13,,-3.813333,-43.232778
br,centro dos monteis,Centro dos Montéis,13,,-3.7275,-43.196667
br,centro dos paulos,Centro dos Paulos,13,,-3.578611,-43.453056
br,centro dos teixeiras,Centro dos Teixeiras,13,,-3.643333,-42.931944
br,centro grande,Centro Grande,13,,-3.216667,-43.766667
br,centro i,Centro I,13,,-3.169167,-43.209722
br,centro ii,Centro II,13,,-3.183889,-43.229444
br,centro iii,Centro III,13,,-3.196944,-43.249167
br,centro jose machado,Centro José Machado,13,,-4.366667,-45.033333
br,centro novo,Centro Novo,13,,-4.233333,-44.183333
br,centro seco,Centro Sêco,13,,-3.216667,-43.8
br,centro velho,Centro Velho,13,,-3.775,-43.316944
br,ceo aberto,Céo Aberto,04,,-7.733333,-67.916667
br,cepa,Cepa,05,,-15.983333,-39.116667
br,cepa forte,Cepa Forte,05,,-11.566667,-37.7
br,cepo forte,Cepo Forte,05,,-11.566667,-37.7
br,ceraima,Ceraíma,05,,-14.266667,-42.75
br,ceramica,Cerámica,15,,-19.166667,-46.566667
br,ceramica,Cerâmica,20,,-5.016667,-42.8
br,cercadinho,Cercadinho,18,,-23.8,-49.883333
br,cercadinho,Cercadinho,27,,-23.468938,-47.761705
br,cercadinho,Cercadinho,31,,-9.805278,-48.723889
br,cercado,Cercado,05,,-10.7,-44.166667
br,cercado,Cercado,14,,-16.483333,-56.333333
br,cercado,Cercado,15,,-19.866667,-44.983333
br,cercado,Cercado,20,,-6.95,-41.4
br,cercado,Cercado,27,,-23.5869,-47.7549
br,cercado,Cercado,30,,-9.35,-36.966667
br,cercado do meio,Cercado do Meio,06,,-5.1,-37.95
br,cercado velho,Cercado Velho,06,,-3.816667,-41.283333
br,cerca velha,Cêrca Velha,15,,-19.516667,-46.083333
br,cerca velha,Cerca Velha,20,,-7.466667,-41.783333
br,cerco,Cêrco,05,,-12.166667,-41.4
br,cerco,Cêrco,22,,-5.868056,-36.670833
br,cerejeira,Cerejeira,08,,-21.133333,-41.316667
br,cerejeiras,Cerejeiras,27,,-23.3647,-46.06
br,ceres,Ceres,29,18760,-15.283333,-49.583333
br,ceresopolis,Ceresópolis,29,,-15.066667,-49.683333
br,cernambi,Cernambi,04,,-3.221944,-62.991389
br,cerne,Cerne,18,,-25.294916,-48.990693
br,cerobas,Cerobas,02,,-9.816667,-35.983333
br,cerqueira cesar,Cerqueira César,27,14261,-23.05,-49.166667
br,cerqueira cezar,Cerqueira Cezar,27,,-23.05,-49.166667
br,cerquilho,Cerquilho,27,35367,-23.201218,-47.757822
br,cerquilho velho,Cerquilho Velho,27,,-23.182753,-47.74501
br,cerquinha,Cerquinha,05,,-13.833333,-42.583333
br,cerquinha,Cerquinha,14,,-14.4,-55.966667
br,cerquinho,Cerquinho,30,,-8.883333,-36.266667
br,cerquinilha,Cerquinilha,30,,-8.883333,-36.266667
br,cerradinho,Cerradinho,18,,-23.8,-49.883333
br,cerrado,Cerrado,18,,-24.083333,-49.95
br,cerrado,Cerrado,27,,-23.5523,-47.8344
br,cerrito alegre,Cerrito Alegre,23,,-31.533333,-52.35
br,cerrito,Cerrito,11,,-23.516667,-54.483333
br,cerrito,Cerrito,23,,-31.85,-52.816667
br,cerrito,Cerrito,26,,-26.116667,-50.483333
br,cerro alegre,Cêrro Alegre,11,,-22.883333,-55.616667
br,cerro alegre,Cerro Alegre,18,,-25.758021,-50.049325
br,cerro azul,Cêrro Azul,18,,-24.833333,-49.25
br,cerro azul,Cerro Azul,23,,-28.15,-54.75
br,cerro azul,Cerro Azul,26,,-26.589107,-49.622915
br,cerro branco,Cêrro Branco,14,,-15.983333,-58.616667
br,cerro branco,Cêrro Branco,23,,-29.566667,-53.05
br,cerro,Cêrro,18,,-25.45,-50.916667
br,cerro chato,Cêrro Chato,23,,-29.683333,-53.266667
br,cerro-cora,Cerro-Corá,22,,-6.05,-36.35
br,cerro da arvore,Cêrro da Árvore,23,,-30.65,-52.75
br,cerro da canela,Cerro da Canela,23,,-30.6301,-52.2681
br,cerro da madeira,Cerro da Madeira,23,,-30.5574,-52.3656
br,cerro do leao,Cêrro do Leão,18,,-25.566667,-51.016667
br,cerro grande,Cêrro Grande,23,,-30.6,-51.75
br,cerro largo,Cêrro Largo,23,9548,-28.15,-54.75
br,cerro lindo,Cerro Lindo,18,,-25.936768,-50.231356
br,cerro negro,Cêrro Negro,26,,-27.8,-50.883333
br,cerro pelado,Cêrro Pelado,23,,-27.9,-55.133333
br,cerro pelado,Cêrro Pelado,26,,-26.45,-51.166667
br,cerros de mairipora,Cerros de Mairiporã,27,,-23.33699,-46.621906
br,cerro torim,Cêrro Torim,11,,-23.533333,-55.516667
br,cerro verde,Cêrro Verde,18,,-25.166667,-51.75
br,cerveja,Cerveja,06,,-7.233333,-40.5
br,cervinho,Cervinho,27,,-22.6,-50.45
br,cervo,Cervo,11,,-19.866667,-56.383333
br,cervo,Cervo,14,,-15.833333,-55.75
br,cervo,Cervo,15,,-21.2,-45.783333
br,cesarao,Cesarão,21,,-22.936075,-43.656877
br,cesar correia,César Correia,15,,-21.933333,-45.566667
br,cesar de pina,César de Pina,15,,-21.066667,-44.216667
br,cesarino afonso,Cesarino Afonso,27,,-22.25,-52.233333
br,cesario alvim,Cesário Alvim,21,,-22.666667,-42.45
br,cesario,Cesário,14,,-16.3,-57.433333
br,cesario lange,Cesário Lange,27,,-23.2247,-47.9536
br,ceu aberto,Céu Aberto,04,,-7.733333,-67.916667
br,ceu azul,Céu Azul,18,,-25.1502,-53.847
br,ceu azul,Céu Azul,27,,-24.741806,-48.075856
br,ceu azul,Céu Azul,30,,-8.016667,-38.916667
br,cevadongo,Cevadongo,11,,-21.15,-55.65
br,ceveira,Ceveira,27,,-22.366667,-47.983333
br,chacara acacia,Chácara Acácia,23,,-30.7035,-52.4737
br,chacara aurora,Chácara Aurora,04,,-2.755556,-60.0375
br,chacara,Chácara,15,,-15.733333,-42.35
br,chacara,Chácara,18,,-25.496667,-50.078333
br,chacara das palmeiras,Chácara das Palmeiras,15,,-20.45,-45.4
br,chacara do ceu,Chácara do Ceu,21,,-22.990771,-43.235058
br,chacara dois irmaos,Chácara Dois Irmãos,23,,-30.7235,-52.3502
br,chacara sao joao,Chácara São João,23,,-30.5465,-52.449
br,chacaras,Chácaras,14,,-15.2,-56.416667
br,chacaras coloniais village interlagos,Chácaras Coloniais Village Interlagos,27,,-23.357297,-46.659415
br,cha,Chã,02,,-9.616667,-36.45
br,chacorore,Chacororé,14,,-16.216667,-55.966667
br,cha da alegria,Chã da Alegria,30,,-7.983333,-35.216667
br,cha d'alegria,Cha d'Alegria,30,,-7.983333,-35.216667
br,cha de imbira,Chã de Imbira,02,,-9.833333,-36.266667
br,cha do estevam,Châ do Estevam,30,,-7.8,-35.05
br,cha do estevao,Chã do Estêvão,30,,-7.8,-35.05
br,cha do imbira,Chã do Imbira,02,,-9.833333,-36.266667
br,cha do rocha,Chã do Rocha,30,,-7.65,-35.566667
br,chafariz,Chafariz,20,,-3.733333,-41.416667
br,chafariz,Chafariz,22,,-5.462778,-37.303611
br,chagas,Chagas,01,,-10.883333,-68.483333
br,chagas,Chagas,23,,-31.9,-53.066667
br,chagas,Chagas,27,,-20.783333,-47.433333
br,chagas,Chagas,29,,-15.916667,-49.266667
br,cha grande,Chã Grande,30,17117,-8.233333,-35.45
br,chagu,Chagu,18,,-25.366667,-52.583333
br,chalana,Chalana,26,,-27.183333,-52.766667
br,chale,Chalé,15,,-20.033333,-41.7
br,chale,Chalé,21,,-21.6,-41.916667
br,chalet,Chalet,15,,-20.033333,-41.7
br,chambica,Chambica,31,,-6.4325,-48.381944
br,chambicazinho,Chambicazinho,31,,-6.4325,-48.381944
br,chambioa,Chambioá,16,,-6.4,-48.55
br,chambioazinho,Chambioàzinho,31,,-6.4325,-48.381944
br,chambivas,Chambivás,31,,-6.4325,-48.381944
br,chamego,Chamego,05,,-12.383333,-41.15
br,chamengo,Chamengo,05,,-12.383333,-41.15
br,champanha,Champanha,04,,-6.2,-59.033333
br,champs prives,Champs Privés,27,,-23.251874,-46.709593
br,chamu,Chamu,20,,-3.318056,-42.061944
br,chamuscada,Chamuscada,05,,-10.033333,-42.633333
br,chanaan,Chanaan,15,,-20.683333,-42.616667
br,chanchavocaia,Chanchavocaia,01,,-8.65,-73.183333
br,chane,Chané,11,,-18.15,-57.366667
br,changondo,Changondo,28,,-9.716667,-37.633333
br,chapada,Chapada,05,,-11.05,-41.966667
br,chapada,Chapada,06,,-4.283333,-38.8
br,chapada,Chapada,11,,-21.433333,-55.566667
br,chapada,Chapada,13,,-3.696389,-43.044722
br,chapada,Chapada,14,,-15.433333,-55.75
br,chapada,Chapada,15,,-17.116667,-42.533333
br,chapada,Chapada,17,,-7.566667,-37.75
br,chapada,Chapada,18,,-23.495605,-49.860886
br,chapada,Chapada,20,,-3.983611,-42.561944
br,chapada,Chapada,23,,-28.05,-53.066667
br,chapada,Chapada,31,,-10.598611,-49.108333
br,chapada da lagoa,Chapada da Lagoa,20,,-6.25,-42.533333
br,chapada da sinda,Chapada da Sindá,20,,-3.865833,-42.433611
br,chapada da vereda,Chapada da Vereda,20,,-6.283333,-42.616667
br,chapada de minas,Chapada de Minas,15,,-18.583333,-47.633333
br,chapada de sao francisco,Chapada de São Francisco,20,,-7.416667,-44.4
br,chapada de sao miguel,Chapada de São Miguel,20,,-3.795833,-42.092222
br,chapada do acude,Chapada do Acude,20,,-5.766667,-42.283333
br,chapada do arroz,Chapada do Arroz,05,,-12.4,-42.883333
br,chapada do barrocao,Chapada do Barrocão,20,,-3.682778,-42.5175
br,chapada do elias,Chapada do Elias,20,,-3.488056,-42.111667
br,chapada do guimaraes,Chapada do Guimarães,14,,-15.433333,-55.75
br,chapada do jatoba,Chapada do Jatobá,20,,-6.866667,-43.45
br,chapada do meio,Chapada do Meio,13,,-3.718889,-43.333333
br,chapada do pau de mel,Chapada do Pau de Mel,20,,-6.483333,-42.3
br,chapada do paulino,Chapada do Paulino,20,,-6.383333,-42.366667
br,chapada do severo,Chapada do Severo,30,,-7.85,-39.966667
br,chapada dos guimaraes,Chapada dos Guimarães,14,13657,-15.433333,-55.75
br,chapada i,Chapada I,20,,-3.413611,-42.088056
br,chapada ii,Chapada II,20,,-3.504722,-42.007778
br,chapada iii,Chapada III,13,,-3.385556,-42.4175
br,chapadao,Chapadão,13,,-3.688333,-43.369444
br,chapadao,Chapadão,14,,-15.1,-56.716667
br,chapadao,Chapadão,15,,-18.233333,-47.1
br,chapadao,Chapadão,21,,-22.822984,-43.373282
br,chapadao grande,Chapadão Grande,27,,-23.5901,-47.9755
br,chapadas,Chapadas,15,,-20.483333,-43.566667
br,chapada velha,Chapada Velha,05,,-12.016667,-42.166667
br,chapadinha,Chapadinha,13,40805,-3.733333,-43.35
br,chapadinha,Chapadinha,20,,-3.835278,-42.523333
br,chapadinha,Chapadinha,27,,-23.633333,-48.033333
br,chapadinho,Chapadinho,20,,-5.283333,-42.716667
br,chapeco,Chapecó,26,160160,-27.083333,-52.983333
br,chapelao,Chapelão,04,,-4.003333,-60.58
br,chapena,Chapena,11,,-20.466667,-56.383333
br,chapeo,Chapeo,23,,-28.9,-50.383333
br,chapeo,Chapeo,27,,-24.466667,-49.033333
br,chapeo,Chapeo,29,,-13.233333,-47.166667
br,chapeu,Chapéu,11,,-20.083333,-55.383333
br,chapeu,Chapéu,15,,-19.883333,-47.216667
br,chapeu,Chapéu,17,,-7.25,-38.516667
br,chapeu,Chapéu,18,,-23.55,-52.233333
br,chapeu,Chapéu,20,,-8.183333,-40.883333
br,chapeu,Chapéu,23,,-28.9,-50.383333
br,chapeu,Chapéu,29,,-13.233333,-47.166667
br,chapeu de couro,Chapéu de Couro,13,,-3.443056,-42.354722
br,chapeu de sol,Chapéu de Sol,02,,-9.766667,-36.433333
br,chapeu de sol,Chapéu de Sol,18,,-26.2,-50.733333
br,chapeu do gato,Chapéu do Gato,05,,-12.016667,-40.7
br,chapeu virado,Chapéu Virado,16,,-1.15,-48.466667
br,cha preta,Chã Preta,02,,-9.266667,-36.3
br,charco,Charco,05,,-13.466667,-40.233333
br,charco,Charco,15,,-18.216667,-46.25
br,charco do cedro,Charco do Cedro,05,,-13.6,-40.233333
br,charcos das pedras,Charcos das Pedras,05,,-13.216667,-40.05
br,chargeada,Chargeada,27,,-22.5,-47.766667
br,charitas,Charitas,06,,-4.6,-40.633333
br,charitas,Charitas,21,,-22.933333,-43.1
br,charito,Charito,06,,-4.6,-40.633333
br,charqueada,Charqueada,23,,-29.266667,-56.583333
br,charqueada,Charqueada,27,12391,-22.5,-47.766667
br,charqueada,Charqueada,31,,-8.879167,-48.440833
br,charqueadas,Charqueadas,23,32076,-29.966667,-51.633333
br,charqueadinha,Charqueadinha,27,,-22.516667,-47.75
br,charque,Charque,26,,-26.59425,-49.987357
br,charrua,Charrua,23,,-27.916667,-52.033333
br,chata,Chata,30,,-8.583333,-36.083333
br,chatelodo,Chatelodo,11,,-21.316667,-57.466667
br,chatinho,Chatinho,20,,-5.2,-42.583333
br,chatuba,Chatuba,21,,-22.809638,-43.434196
br,chaval,Chaval,06,,-3.033333,-41.25
br,chavantes,Chavantes,27,10306,-23.033333,-49.716667
br,chavantina,Chavantina,14,,-14.666667,-52.35
br,chave arpard,Chave Arpard,27,,-21.8,-52.033333
br,chave de ouro,Chave de Ouro,21,,-22.902696,-43.2967
br,chave do barao,Chave do Barão,27,,-22.766667,-47.633333
br,chave do faria,Chave do Faria,21,,-21.616667,-42.1
br,chaves,Chaves,04,,-4.204444,-60.723611
br,chaves,Chaves,06,,-5.283333,-40.683333
br,chaves,Chaves,14,,-16.166667,-56.366667
br,chaves,Chaves,16,,-.166667,-49.916667
br,chcara kabu,Chcara Kabu,27,,-23.3655,-46.158
br,chcaras boa vista,Chcaras Boa Vista,27,,-23.3546,-46.2402
br,chcaras eldorado,Chcaras Eldorado,27,,-23.2922,-46.2159
br,chcara tsurumatsu seki,Chcara Tsurumatsu Seki,27,,-23.3421,-46.0874
br,cherente,Cherente,31,,-9.529167,-48.319722
br,cheruan,Cheruan,04,,-6.05,-67.833333
br,chervor,Chervor,04,,-6.05,-67.833333
br,chiador,Chiador,15,,-22.004167,-43.058889
br,chiapeta,Chiapeta,23,,-27.916667,-53.95
br,chiapetta,Chiapetta,23,,-27.916667,-53.95
br,chiba,Chiba,18,,-24.983333,-53.5
br,chibiu,Chibiu,14,,-16.35,-53.583333
br,chicao,Chicão,11,,-18.433333,-56.283333
br,chicao,Chicão,30,,-8.116667,-35.95
br,chicaozinho,Chicãozinho,02,,-9.266667,-37.333333
br,chicha,Chichá,05,,-10.066667,-43.133333
br,chico antonio,Chico Antônio,14,,-14.7,-57.283333
br,chico baiano,Chico Baiano,29,,-16.216667,-52.266667
br,chico dias,Chico Dias,13,,-3.963889,-43.322222
br,chico herrera,Chico Herrera,04,,-.316667,-65.383333
br,chico lunia,Chico Lunia,23,,-29.966667,-50.6
br,chico magro,Chico Magro,30,,-7.816667,-38.566667
br,chico mana,Chico Mana,01,,-8.183333,-73.483333
br,chico manoel,Chico Manoel,13,,-3.828333,-43.450556
br,chico marques,Chico Marques,05,,-11.033333,-43.933333
br,chico martins,Chico Martins,16,,-8.616667,-49.666667
br,chico pirara,Chico Pirará,25,,3.05,-61.15
br,chico roberto,Chico Roberto,14,,-10.316667,-50.466667
br,chico til,Chico Til,15,,-15.8,-40.616667
br,chicuru,Chicuru,30,,-9.133333,-36.683333
br,chifre de veado,Chifre de Veado,11,,-21.716667,-54.933333
br,chile,Chile,15,,-16.4,-40.75
br,chilelandia,Chilelândia,11,,-21.816667,-54.316667
br,chimarrao,Chimarrão,23,,-28.633333,-51.5
br,china,China,11,,-21.133333,-56.7
br,chinela,Chinela,13,,-7.766667,-46.033333
br,chinesa,Chinesa,15,,-16.083333,-40.116667
br,chiniqua,Chiniquá,23,,-29.666667,-54.366667
br,chique-chique,Chique-Chique,05,,-10.833333,-42.733333
br,chique chique,Chique Chique,05,,-9.516667,-43.6
br,chique-chique,Chique-Chique,06,,-7.1,-40
br,chiqueirao,Chiqueirão,16,,-4.35,-49.55
br,chiqueirinho,Chiqueirinho,02,,-9.566667,-36.433333
br,chiqueiro,Chiqueiro,05,,-11.866667,-37.733333
br,chiqueiro,Chiqueiro,13,,-3.850278,-43.063056
br,chiqueiro velho,Chiqueiro Velho,30,,-7.85,-38.933333
br,chiquito,Chiquito,29,,-15.183333,-49.7
br,chiriri,Chiriri,25,,3.233333,-60.516667
br,chiruan,Chiruan,04,,-6.05,-67.833333
br,chiru,Chiru,11,,-21.183333,-56.433333
br,choa,Choa,15,,-19.75,-48.05
br,chocalho,Chocalho,06,,-3.583333,-40.1
br,chocolatal,Chocolatal,24,,-10.166667,-65.283333
br,chomm grande,Chomm Grande,15,,-18.633333,-42.05
br,chonim,Chonim,15,,-18.633333,-42.05
br,chonim de baixo,Chonim de Baixo,15,,-18.666667,-41.966667
br,chonin,Chonin,15,,-18.633333,-42.05
br,chopeiro,Chopeiro,20,,-6.816667,-40.45
br,chopim,Chopim,18,,-25.85,-52.5
br,chopin,Chopin,18,,-25.85,-52.5
br,chopinzinho,Chopinzinho,18,14142,-25.85,-52.5
br,chora,Chora,06,,-3.35,-40.116667
br,chorador,Chorador,06,,-4.783333,-41.083333
br,chora menino,Chora Menino,27,,-23.49961,-46.645202
br,chorao,Chorão,17,,-7.566667,-38.433333
br,chorao,Chorão,20,,-3.713333,-42.468333
br,chorao,Chorão,23,,-28.333333,-53.9
br,chorocho,Chorochó,05,,-8.983333,-39.1
br,choro,Choró,06,,-4.816667,-39.15
br,chorona,Chorona,08,,-20.9,-41.45
br,chorozinho,Choròzinho,06,9755,-4.3,-38.483333
br,chorrocho,Chorrochó,05,,-8.983333,-39.1
br,christiano ottoni,Christiano Ottoni,15,,-20.833333,-43.8
br,christina,Christina,15,,-22.216667,-45.266667
br,christovinho,Christovinho,20,,-7.05,-41.5
br,chui,Chuí,23,,-33.683333,-53.45
br,chumbado,Chumbado,13,,-6.116667,-43
br,chumbo,Chumbo,04,,-4.411667,-59.874167
br,chumbo,Chumbo,15,,-18.616667,-46.2
br,chumichuga,Chumichuga,20,,-7.366667,-41.233333
br,chuna,Chuna,16,,-1.033333,-46.3
br,chupador,Chupador,18,,-23.966667,-51.466667
br,chupador de baixo,Chupador de Baixo,06,,-4.883333,-40.116667
br,chupeiro,Chupeiro,20,,-8.05,-43.466667
br,chupeta,Chupeta,02,,-9.266667,-37.866667
br,chure,Churé,05,,-11.933333,-42.433333
br,chuva,Chuva,15,,-16.583333,-41.966667
br,chuvarada,Chuvarada,11,,-21.383333,-57.883333
br,chuy,Chuy,23,,-33.683333,-53.45
br,cianita,Cianita,15,,-21.483333,-44.333333
br,cianorte,Cianorte,18,55995,-23.616667,-52.616667
br,cibauma,Cibaúma,22,,-6.283333,-35.033333
br,cibrao,Cibrão,15,,-18.183333,-41.616667
br,cicaba,Cicaba,23,,-28.9,-50.383333
br,cicero dantas,Cícero Dantas,05,18317,-10.6,-38.383333
br,cidade alta,Cidade Alta,21,,-22.815596,-43.292357
br,cidade anico la da barra,Cidade Aníco la da Barra,30,,-9.016667,-38.266667
br,cidade avicola da barra,Cidade Avícola da Barra,30,,-9.016667,-38.266667
br,cidade,Cidade,05,,-10.716667,-42.183333
br,cidade,Cidade,22,,-6.1,-38.45
br,cidade das criancas,Cidade das Crianças,21,,-22.891808,-43.689904
br,cidade de deus,Cidade de Deus,21,,-22.948798,-43.364928
br,cidade de minas,Cidade de Minas,15,,-19.916667,-43.933333
br,cidade dos meninos,Cidade dos Meninos,21,,-22.670278,-43.327778
br,cidade felipea,Cidade Felipéa,17,,-7.116667,-34.866667
br,cidade gaucha,Cidade Gaúcha,18,,-23.366667,-53.016667
br,cidade industrial,Cidade Industrial,15,,-19.95,-44.016667
br,cidade jardim laranjeiras,Cidade Jardim Laranjeiras,21,,-22.93902,-43.193126
br,cidade jardim montevideo,Cidade Jardim Montevideo,21,,-22.619444,-43.3975
br,cidade martins,Cidade Martins,27,,-23.431488,-46.512137
br,cidade nova,Cidade Nova,13,,-3.688889,-43.286944
br,cidade nova,Cidade Nova,21,,-22.91164,-43.200492
br,cidade pirituba,Cidade Pirituba,27,,-23.475144,-46.724969
br,cidade progresso,Cidade Progresso,18,,-22.766667,-51.5
br,cidade satelite,Cidade Satélite,15,,-19.95,-44.016667
br,cidade universitaria,Cidade Universitária,21,,-22.856817,-43.226644
br,cidreira,Cidreira,20,,-7.35,-43.283333
br,cidreira,Cidreira,23,8965,-30.183333,-50.2
br,cigana,Cigana,05,,-11.6,-40.383333
br,cigana,Cigana,13,,-2.583333,-45.366667
br,cigana,Cigana,16,,-5.55,-48.966667
br,cigana,Cigana,20,,-7.35,-41.783333
br,cigana,Cigana,22,,-5.269444,-36.126389
br,cigano,Cigano,06,,-5.833333,-40.633333
br,cigano,Cigano,13,,-3.956389,-43.248611
br,cigano,Cigano,22,,-5.109722,-37.419444
br,cilandia,Cilândia,21,,-22.470833,-43.545833
br,cima da serra,Cima da Serra,05,,-8.95,-38.533333
br,cimbres,Cimbres,30,,-8.35,-36.85
br,cinco anzois,Cinco Anzóis,16,,-1.65,-54.9
br,cincoenta e um,Cincoenta e Um,08,,-19.266667,-40.483333
br,cinco folhas,Cinco Fôlhas,15,,-18.333333,-42.566667
br,cinco ilhas,Cinco Ilhas,15,,-18.333333,-42.566667
br,cinco ilhas,Cinco Ilhas,27,,-20.516667,-51.433333
br,cinco irmaos,Cinco Irmãos,14,,-17.166667,-57.7
br,cinco lagoas,Cinco Lagoas,21,,-22.483889,-43.733333
br,cinco pontoes,Cinco Pontões,08,,-19.816667,-40.933333
br,cinco portoes,Cinco Portões,08,,-19.816667,-40.933333
br,cinco rios,Cinco Rios,05,,-12.433333,-38.533333
br,cinco vargens,Cinco Vargens,05,,-11.733333,-40.566667
br,cinelandia,Cinelândia,21,,-22.912332,-43.175825
br,ciniaco,Ciniaco,23,,-28.35,-51.883333
br,cinquentenario,Cinquentenário,23,,-27.666667,-54.6
br,cintra,Cintra,04,,-5.570556,-64.141111
br,cintra,Cintra,16,,-.766667,-47.45
br,cintra pimenta,Cintra Pimenta,18,,-22.816667,-52.966667
br,cintra pimentel,Cintra Pimentel,18,,-22.816667,-52.966667
br,cinzas,Cinzas,18,,-23.45,-50.283333
br,cinzeiro,Cinzeiro,13,,-3.441111,-42.73
br,cipoal,Cipoal,04,,-1.783333,-65.85
br,cipoal,Cipoal,15,,-15.816667,-44.75
br,cipoal,Cipoal,16,,-1.716667,-55.433333
br,cipoal,Cipoal,20,,-7.95,-43.166667
br,cipo,Cipó,05,,-11.1,-38.516667
br,cipo,Cipó,06,,-4.5,-41.066667
br,cipo,Cipó,13,,-3.968056,-43.342778
br,cipo,Cipó,17,,-6.9,-37.533333
br,cipo,Cipó,20,,-3.664167,-42.501667
br,cipo,Cipó,30,,-8.183333,-38.3
br,cipo de fogo,Cipo de Fogo,13,,-3.325,-43.489722
br,cipo de leite,Cipó de Leite,05,,-10.2,-37.783333
br,cipo do assu,Cipó do Assú,05,,-12.266667,-38.05
br,cipo dos anjos,Cipó dos Anjos,06,,-5,-38.716667
br,cipo iii,Cipó III,13,,-3.658056,-43.405
br,cipolandia,Cipolândia,11,,-20.133333,-55.4
br,cipool,Cipool,04,,-6.433333,-61.733333
br,cipo poco,Cipó Poço,04,,1.215,-69.384444
br,cipotanea,Cipotânea,15,,-20.9,-43.366667
br,cipotania,Cipotânia,15,,-20.9,-43.366667
br,cipotuba,Cipotuba,14,,-15.866667,-57.616667
br,cipriano,Cipriano,06,,-6.716667,-38.883333
br,cipriano,Cipriano,15,,-19.75,-43.95
br,ciriaco,Ciríaco,23,,-28.35,-51.883333
br,cirilo,Cirilo,15,,-15.45,-45.066667
br,cirurgiao,Cirurgião,20,,-5.583333,-43.033333
br,cisneiros,Cisneiros,15,,-21.433333,-42.366667
br,cisterna,Cisterna,15,,-19.1,-48.316667
br,citrolandia,Citrolândia,21,,-22.6,-43.016667
br,ciza,Ciza,24,,-12.616667,-62.066667
br,clara aurora,Clara Aurora,04,,-5.686111,-64.351667
br,clara,Clara,23,,-29.55,-54.35
br,claraiba,Claraíba,26,,-27.2,-48.916667
br,clarana,Claranã,30,,-7.65,-39.933333
br,claraval,Claraval,15,,-20.4,-47.283333
br,claretiana,Claretiana,29,,-14.033333,-46.616667
br,clarimunda,Clarimunda,15,,-18.433333,-48.55
br,clarimundo,Clarimundo,15,,-16.3,-40.55
br,clarindo barbosa,Clarindo Barbosa,15,,-16.133333,-40.483333
br,clarinia,Clarínia,27,,-22.75,-49.45
br,claro,Claro,20,,-3.7425,-42.396667
br,claro de minas,Claro de Minas,15,,-17.916667,-47
br,claro dos pocoes,Claro dos Poções,15,,-17.083333,-44.266667
br,claros,Claros,15,,-17.083333,-44.266667
br,claudino machado,Claudino Machado,08,,-20.1,-41.333333
br,claudio,Cláudio,04,,-3.033611,-61.517778
br,claudio,Cláudio,13,,-3.531944,-42.975833
br,claudio,Cláudio,15,21825,-20.433333,-44.766667
br,claudio manoel,Claúdio Manoel,15,,-20.216667,-43.2
br,claudio manuel,Cláudio Manuel,15,,-20.216667,-43.2
br,clavasio alves da silva,Clavásio Alves da Silva,23,,-30.916667,-55.816667
br,clemente apolonio,Clemente Apolônio,15,,-17.633333,-41.633333
br,clemente argolo,Clemente Argôlo,23,,-28,-51.45
br,clemente branco,Clemente Branco,15,,-17.133333,-41.416667
br,clemente,Clemente,15,,-16.1,-42.816667
br,clemente,Clemente,18,,-23.899346,-50.181131
br,clemente,Clemente,28,,-9.983333,-37.816667
br,clemente neri,Clemente Néri,05,,-17.65,-40.266667
br,clemente rocha,Clemente Rocha,15,,-17.9,-41.016667
br,clementina,Clementina,27,,-21.533333,-50.45
br,clementino,Clementino,17,,-6.533333,-38.016667
br,clementino cunha,Clementino Cunha,15,,-20.05,-46.7
br,clementino leite,Clementino Leite,15,,-20.15,-46.3
br,clevelandia,Clevelandia,03,,3.816667,-51.866667
br,clevelandia,Clevelândia,18,14701,-26.4,-52.35
br,clevelandia do norte,Clevelândia do Norte,03,,3.816667,-51.866667
br,clevlandia,Clevlandia,03,,3.816667,-51.866667
br,climerio,Climério,05,,-17.516667,-39.266667
br,cliza,Cliza,24,,-12.616667,-62.066667
br,clodoaldo pedroso,Clodoaldo Pedroso,23,,-30.016667,-56.05
br,clodomiro de oliveira,Clodomiro de Oliveira,15,,-19.583333,-45.45
br,clovis,Clóvis,15,,-17.1,-40.633333
br,coadi,Coadi,04,,-3.016667,-64.983333
br,coady,Coady,04,,-3.016667,-64.983333
br,coapoa,Coapoã,17,,-6.766667,-36.8
br,coaraci,Coaraci,05,32345,-14.633333,-39.533333
br,coaracy nunes,Coaracy Nunes,03,,.883333,-51.283333
br,coari amazonas,Coari Amazonas,04,,-4.109444,-63.138889
br,coari,Coari,04,50491,-4.109444,-63.138889
br,coari,Coari,16,,-4.35,-49.533333
br,coary,Coary,04,,-4.109444,-63.138889
br,coassu,Coassu,06,,-2.85,-40.033333
br,coata,Coatá,04,,-6.983333,-70.833333
br,coataquara,Coataquará,16,,-5.45,-56.866667
br,coaticaba,Coatiçaba,29,,-13.983333,-46.516667
br,coatinga do porco,Coatinga do Porco,20,,-7.8,-42.633333
br,coatis,Coatis,05,,-13.583333,-41.966667
br,cobe,Cobé,05,,-11.833333,-40.45
br,cobica,Cobiça,15,,-21.466667,-45.333333
br,cobra,Cobra,06,,-7.066667,-39.85
br,cobra,Cobra,13,,-3.860556,-43.451389
br,cobra,Cobra,17,,-6.966667,-37.8
br,cobra,Cobra,25,,.741667,-61.676944
br,cobral,Cobral,27,,-24.8243,-48.5181
br,cobras,Cobras,14,,-7.9,-57.8
br,cobras,Cobras,26,,-27.333333,-49.616667
br,cocaes,Cocaes,15,,-19.85,-43.45
br,cocaina,Cocaína,27,,-20.5,-47.716667
br,cocais,Cocais,14,,-15.066667,-56.55
br,cocais,Cocais,15,,-19.85,-43.45
br,cocais,Cocais,27,,-23.669429,-47.70826
br,cocais da estrela,Cocais da Estrêla,15,,-19.45,-42.866667
br,cocala,Cocala,04,,-8.233333,-60.7
br,cocal,Cocal,03,,2.6,-50.866667
br,cocal,Cocal,05,,-12.183333,-42.266667
br,cocal,Cocal,13,,-1.983333,-46.216667
br,cocal,Cocal,15,,-21.25,-46.216667
br,cocal,Cocal,16,,-1.066667,-49.583333
br,cocal,Cocal,20,12097,-3.466667,-41.566667
br,cocal,Cocal,26,,-28.616667,-49.35
br,cocal,Cocal,29,,-14.666667,-48.683333
br,cocal da ilha,Cocal da Ilha,20,,-3.968056,-42.187222
br,cocal das montanhas,Cocal das Montanhas,20,,-3.789722,-42.089722
br,cocal dos alves,Cocal dos Alves,20,,-3.616667,-41.533333
br,cocal grande,Cocal Grande,13,,-3.796667,-43.337778
br,cocal i,Cocal I,13,,-3.259167,-43.473889
br,cocal ii,Cocal II,13,,-3.494722,-43.448056
br,cocal iii,Cocal III,13,,-3.531944,-43.475278
br,cocalinho,Cocalinho,13,,-3.109167,-43.438889
br,cocalinho,Cocalinho,14,,-14.366667,-51
br,cocalinho,Cocalinho,20,,-4.533333,-41.983333
br,cocal iv,Cocal IV,13,,-3.233611,-43.293611
br,cocal v,Cocal V,13,,-3.047222,-43.380278
br,cocal vi,Cocal VI,13,,-3.486944,-43.225833
br,cocal vii,Cocal VII,13,,-3.255556,-43.457222
br,cocalzinho,Cocalzinho,16,,-2.966667,-46.65
br,cocamera,Cocamera,01,,-8.5,-71.15
br,cocamora,Cocamorá,01,,-8.5,-71.15
br,cocao,Cocão,05,,-12.65,-39.066667
br,cocau,Cocaú,30,,-8.65,-35.266667
br,coceira,Coceira,13,,-3.390556,-42.940556
br,coceira de baixo,Coceira de Baixo,13,,-3.411667,-42.946111
br,cocelinho,Cocelinho,14,,-14.366667,-51
br,cocha,Cochá,06,,-7,-38.883333
br,cochichola,Cochichola,17,,-7.616667,-36.616667
br,cochilha,Cochilha,23,,-28.116667,-52.283333
br,cochilha grande,Cochilha Grande,23,,-28.233333,-50.983333
br,cochinho,Cochinho,23,,-28.566667,-52.75
br,cocho,Cocho,27,,-22.912951,-45.006106
br,cocho de agua,Cocho de Água,15,,-20.1309,-43.7609
br,cocho do malheiro,Cocho do Malheiro,05,,-12.416667,-41.633333
br,cocho grande,Cocho Grande,18,,-25.216667,-52.583333
br,cochos,Cochos,06,,-7.516667,-39.116667
br,cochos,Cochos,17,,-7.216667,-38.183333
br,cocoal,Cocoal,13,,-1.983333,-46.216667
br,cococa,Cococa,06,,-6.3,-40.483333
br,coco,Coco,06,,-3.766667,-38.5
br,coco,Côco,15,,-20.3,-44
br,coco,Côco,16,,-1.066667,-46.5
br,coco,Coco,20,,-5.85,-42.433333
br,coco,Côco,31,,-10.083333,-48.866667
br,coco do pote,Côco do Pote,08,,-20.616667,-40.716667
br,cocoi,Cocoi,06,,-6.416667,-40.5
br,cocos,Côcos,05,,-12.716667,-44.566667
br,cocos,Côcos,13,,-4.583333,-43.666667
br,cocos,Côcos,20,,-10.351667,-45.331389
br,cocota,Cocotá,21,,-22.806328,-43.181397
br,coculo,Coculo,05,,-14.05,-39.5
br,cocuruto,Cocuruto,23,,-32.05,-52.05
br,codajas,Codajás,04,11185,-3.833333,-62.083333
br,codajaz,Codajáz,04,,-3.833333,-62.083333
br,codia,Codia,06,,-5.633333,-39.333333
br,codo,Codó,13,83289,-4.483333,-43.883333
br,codoz,Codoz,06,,-3.616667,-41.366667
br,codoz do baixo,Codoz do Baixo,20,,-3.616667,-41.466667
br,codozinho,Codòzinho,13,,-4.766667,-44.166667
br,coelho,Coelho,05,,-11.2,-42.8
br,coelho,Coelho,15,,-18.55,-48.1
br,coelho,Coelho,17,,-6.9,-35.816667
br,coelho,Coelho,18,,-22.9,-51.05
br,coelho,Coelho,25,,2.35,-60.866667
br,coelho da rocha,Coelho da Rocha,21,,-22.773185,-43.382543
br,coelho neto,Coelho Neto,13,36185,-4.25,-43
br,coelho neto,Coelho Neto,21,,-22.834363,-43.352521
br,coimbra,Coimbra,11,,-18.3,-55.3
br,coimbra,Coimbra,15,,-20.866667,-42.8
br,coimbra,Coimbra,23,,-28.816667,-54.483333
br,coimbra,Coimbra,24,,-11.466667,-65.3
br,coimbra lara,Coimbra Lara,11,,-18.3,-55.3
br,coimelemong,Coimelemong,25,,4.216667,-60.816667
br,coipe,Coipé,05,,-12.7,-38.6
br,coite,Coité,02,,-9.633333,-36.583333
br,coite,Coité,05,,-11.466667,-39.183333
br,coite,Coité,06,,-7.433333,-38.716667
br,coite,Coite,20,,-6.583333,-42.3
br,coite,Coité,22,,-6.3,-38.416667
br,coite,Coité,30,,-8.8,-38.766667
br,coite comprido,Coité Comprido,06,,-7.366667,-38.633333
br,coite do noa,Coité do Noá,02,,-9.666667,-36.5
br,coite do noia,Coité do Nóia,02,2529,-9.666667,-36.5
br,coite do nois,Coite do Nois,02,,-9.666667,-36.5
br,coitezeira,Coitèzeira,06,,-7.666667,-39.116667
br,coitezeiro,Coitezeiro,06,,-3.2,-40.216667
br,coito de cima,Coito de Cima,05,,-13.15,-38.9
br,coitos de cima,Coitos de Cima,05,,-13.15,-38.9
br,coivara,Coivara,13,,-3.8,-44.583333
br,coivaras,Coivaras,20,,-5.083333,-42.216667
br,cojubim,Cojubim,24,,-9.2,-61.783333
br,colado,Colado,05,,-13.083333,-38.8
br,colares,Colares,04,,-5.518056,-60.836389
br,colares,Colares,16,,-1.016667,-48.266667
br,colatina,Colatina,08,101193,-19.533333,-40.616667
br,colectoria,Colectoria,23,,-27.45,-52.85
br,colegio agricola,Colégio Agricola,06,,-7.2,-39.45
br,colegio,Colégio,21,,-22.633333,-42.733333
br,colegio,Colégio,27,,-23.711314,-47.307933
br,colegio,Colégio,28,,-11.066667,-37.266667
br,coleta,Coleta,17,,-7.416667,-37.516667
br,coletoria do estado de mato grosso,Coletoria do Estado de Mato Grosso,04,,-7.333333,-58.1
br,coletoria do estado do mato grosso,Coletoria do Estado do Mato Grosso,04,,-7.333333,-58.1
br,coletoria do estado do para,Coletoria do Estado do Pará,16,,-7.316667,-58.05
br,colheres,Colheres,20,,-8.5,-44.866667
br,colina,Colina,05,,-13.85,-41.266667
br,colina,Colina,27,15042,-20.733333,-48.583333
br,colinas,Colinas,13,25159,-6.033333,-44.233333
br,colinas,Colinas,29,,-14.2,-48.05
br,colinas,Colinas,30,,-7.9,-39.683333
br,colinas de caucaia do alto,Colinas de Caucaia do Alto,27,,-23.661085,-47.019087
br,colinas do magarca,Colinas do Magarça,21,,-22.940904,-43.587982
br,colinas do tocantins,Colinas do Tocantins,31,,-8.021667,-48.378611
br,collares,Collares,16,,-.933333,-48.283333
br,collatina,Collatina,08,,-19.533333,-40.616667
br,collectoria,Collectoria,23,,-27.45,-52.85
br,collectoria do estado do para,Collectoria do Estado do Pará,16,,-7.316667,-58.05
br,collina,Collina,27,,-20.733333,-48.583333
br,colmeia,Colméia,31,,-8.679722,-48.647222
br,colocacao arati,Colocação Arati,04,,-9.316667,-67.466667
br,colocacao areal,Colocação Areal,01,,-9.983333,-67.15
br,colocacao areia branca,Colocação Areia Branca,04,,-9.4,-67.35
br,colocacao barra grande,Colocação Barra Grande,01,,-9.933333,-67.3
br,colocacao barro vermelho,Colocação Barro Vermelho,04,,-9.3,-67.466667
br,colocacao beija flor,Colocação Beija Flor,04,,-9.4,-67.283333
br,colocacao bela flor,Colocação Bela Flor,01,,-9.583333,-67.333333
br,colocacao bela vista,Colocação Bela Vista,04,,-9.408333,-67.283333
br,colocacao boa fe,Colocação Boa Fé,01,,-9.95,-67.083333
br,colocacao boa sorte,Colocação Boa Sorte,01,,-9.75,-67.433333
br,colocacao boa uniao,Colocação Boa União,01,,-9.933333,-67.416667
br,colocacao boa vista,Colocação Boa Vista,01,,-10.483333,-69.283333
br,colocacao bom jardim,Colocação Bom Jardim,04,,-9.266667,-67.466667
br,colocacao bom jesus,Colocação Bom Jesus,04,,-9.25,-67.283333
br,colocacao cafundo,Colocação Cafundo,04,,-9.4,-67.316667
br,colocacao cajui,Colocação Cajui,04,,-9.3,-67.316667
br,colocacao canaviera,Colocação Canaviera,01,,-10.366667,-69.416667
br,colocacao caruaru,Colocação Caruaru,01,,-9.8,-67.416667
br,colocacao centrinho,Colocação Centrinho,01,,-9.8,-67.183333
br,colocacao chave de ouro,Colocação Chave de Ouro,01,,-9.8,-67.416667
br,colocacao china,Colocação China,04,,-9.316667,-67.483333
br,colocacao combate,Colocacão Combate,24,,-10.266667,-62.968056
br,colocacao comitiu,Colocação Comitiú,04,,-9.05,-67.366667
br,colocacao dois de maio,Colocação Dois de Maio,01,,-9.933333,-67.483333
br,colocacao esperanca,Colocação Esperança,01,,-9.95,-67.1
br,colocacao expedito,Colocação Expedito,04,,-9.15,-67.2
br,colocacao extrema,Colocação Extrema,01,,-9.916667,-67.45
br,colocacao felisbino,Colocação Felisbino,04,,-9.033333,-67.333333
br,colocacao geral,Colocação Geral,04,,-9.15,-67.4
br,colocacao goianinha,Colocação Goianinha,04,,-9.383333,-67.283333
br,colocacao ico,Colocação Icó,04,,-9.05,-67.383333
br,colocacao independencia,Colocação Independência,04,,-9.416667,-67.283333
br,colocacao iquiri,Colocação Iquiri,01,,-9.8,-67.283333
br,colocacao jucelino,Colocação Jucelino,04,,-9.133333,-67.233333
br,colocacao lagoa de barro,Colocação Lagoa de Barro,01,,-9.75,-67.25
br,colocacao laranjeira,Colocação Laranjeira,04,,-9.15,-67.4
br,colocacao madeirinha,Colocação Madeirinha,04,,-9.05,-67.383333
br,colocacao mineiros,Colocação Mineiros,04,,-9.083333,-67.25
br,colocacao moco,Colocação Moco,04,,-9.15,-67.4
br,colocacao monte negro,Colocação Monte Negro,04,,-9.383333,-67.283333
br,colocacao montevideu,Colocação Montevidéu,01,,-9.7,-67.333333
br,colocacao moreira,Colocação Moreira,01,,-9.916667,-67.466667
br,colocacao natal,Colocação Natal,04,,-9.416667,-67.491667
br,colocacao nossa senhora aparecida,Colocação Nossa Senhora Aparecida,01,,-9.8,-67.416667
br,colocacao nossa senhora de fatima,Colocação Nossa Senhora de Fátima,01,,-9.833333,-67.416667
br,colocacao paquara,Colocação Paquara,01,,-10.483333,-69.2
br,colocacao pau mulato,Colocação Pau Mulato,04,,-9.083333,-67.416667
br,colocacao pelotas,Colocação Pelotas,04,,-9.35,-67.5
br,colocacao piratini,Colocação Piratini,01,,-10,-67.366667
br,colocacao porto alegre,Colocação Porto Alegre,04,,-9.25,-67.433333
br,colocacao porto rico,Colocação Porto Rico,04,,-9.25,-67.45
br,colocacao poti,Colocação Poti,01,,-10.416667,-69.366667
br,colocacao quatro bocas,Colocação Quatro Bocas,01,,-9.933333,-67.483333
br,colocacao quilometro 100,Colocação Quilômetro 100,04,,-9.516667,-67.3
br,colocacao quilometro 102,Colocação Quilômetro 102,04,,-9.516667,-67.283333
br,colocacao quilometro 105,Colocação Quilômetro 105,04,,-9.483333,-67.283333
br,colocacao quilometro 106,Colocação Quilômetro 106,04,,-9.45,-67.283333
br,colocacao quilometro 109,Colocação Quilômetro 109,04,,-9.433333,-67.283333
br,colocacao quilometro 110,Colocação Quilômetro 110,04,,-9.433333,-67.283333
br,colocacao quilometro 112,Colocação Quilômetro 112,04,,-9.416667,-67.283333
br,colocacao quilometro 117,Colocação Quilômetro 117,04,,-9.4,-67.283333
br,colocacao quilometro 121,Colocação Quilômetro 121,04,,-9.366667,-67.283333
br,colocacao quilometro 124,Colocação Quilômetro 124,04,,-9.333333,-67.283333
br,colocacao quilometro 133,Colocação Quilômetro 133,04,,-9.3,-67.316667
br,colocacao quilometro 140,Colocação Quilômetro 140,04,,-9.233333,-67.283333
br,colocacao quilometro 48,Colocação Quilômetro 48,04,,-9.166667,-67.283333
br,colocacao quilometro 84,Colocação Quilômetro 84,01,,-9.65,-67.333333
br,colocacao quilometro 87,Colocação Quilômetro 87,01,,-9.633333,-67.333333
br,colocacao quilometro 98,Colocação Quilômetro 98,04,,-9.533333,-67.3
br,colocacao raposa,Colocação Raposa,04,,-9.25,-67.35
br,colocacao redencao nova,Colocação Redenção Nova,04,,-9.316667,-67.483333
br,colocacao redencao velha,Colocação Redenção Velha,04,,-9.35,-67.483333
br,colocacao ribeira,Colocação Ribeira,04,,-9.1,-67.4
br,colocacao salvacao,Colocação Salvação,04,,-9.266667,-67.466667
br,colocacao samauma,Colocação Samauma,04,,-9.133333,-67.4
br,colocacao santa cruz,Colocação Santa Cruz,01,,-9.866667,-67.05
br,colocacao santa luzia,Colocação Santa Luzia,04,,-9.05,-67.383333
br,colocacao santa maria,Colocação Santa Maria,01,,-9.95,-67.483333
br,colocacao santa rosa,Colocação Santa Rosa,01,,-9.883333,-67.45
br,colocacao santa teresinha,Colocação Santa Teresinha,01,,-9.95,-67.483333
br,colocacao santo antonio,Colocação Santo Antônio,01,,-9.816667,-67.416667
br,colocacao sao francisco,Colocação São Francisco,01,,-9.8,-67.366667
br,colocacao sao francisco,Colocação São Francisco,04,,-9.116667,-67.266667
br,colocacao sao jose,Colocação São José,01,,-9.85,-67.25
br,colocacao sao jose,Colocação São José,04,,-9.25,-67.45
br,colocacao sao luis,Colocação São Luís,01,,-9.7,-67.333333
br,colocacao sao pedro,Colocação São Pedro,01,,-9.916667,-67.466667
br,colocacao sao sebastiao,Colocação São Sebastião,01,,-9.816667,-67.333333
br,colocacao sapucaia,Colocação Sapucaia,01,,-9.866667,-67.433333
br,colocacao saudade,Colocação Saudade,04,,-9.383333,-67.333333
br,colocacao segredo,Colocação Segredo,04,,-9.083333,-67.416667
br,colocacao tabatinga,Colocação Tabatinga,04,,-9.05,-67.35
br,colocacao tapiri,Colocação Tapiri,04,,-9.4,-67.491667
br,colocacao veneza,Colocação Veneza,01,,-9.933333,-67.116667
br,colocacao vertente,Colocação Vertente,01,,-9.983333,-67.2
br,colocacao xavier,Colocação Xavier,01,,-9.916667,-67.466667
br,cologne,Cologne,03,,3.216667,-51.233333
br,colombia,Colômbia,27,,-20.166667,-48.666667
br,colombina,Colombina,05,,-13.1,-40.233333
br,colombo,Colombo,18,217002,-25.283333,-49.233333
br,colombo,Colombo,23,,-28.933333,-51.866667
br,colombo,Colombo,27,,-20.816667,-49.783333
br,colona,Colôna,30,,-7.583333,-39.733333
br,colonia 15 de novembro,Colonia 15 de Novembro,23,,-28.75,-53.1
br,colonia agricola do matapi,Colônia Agrícola do Matapi,03,,.683333,-51.433333
br,colonia agricola knust,Colônia Agricola Knust,21,,-22.491525,-42.829268
br,colonia agro-pecuaria brasilandia,Colônia Agro-Pecuária Brasilândia,15,,-16.983333,-46.15
br,colonia amelia,Colônia Amélia,13,,-1.7,-45.583333
br,colonia argollo,Colonia Argollo,23,,-28,-51.45
br,colonia augusta vitoria,Colônia Augusta Vitória,18,,-24.25,-50.85
br,colonia barao do triumpho,Colonia Barão do Triumpho,23,,-30.383333,-51.733333
br,colonia barreiro terra indigena kaingang,Colônia Barreiro Terra Indígena Kaingang,18,,-23.822714,-50.967949
br,colonia bela vista,Colônia Bela Vista,18,,-25.2595,-50.7042
br,colonia benfica,Colônia Benfica,18,,-25.370833,-50.018611
br,colonia boa vista,Colônia Boa Vista,04,,-3.298611,-60.441944
br,colonia bom retiro,Colonia Bom Retiro,23,,-30.116667,-51.5
br,colonia braga,Colônia Braga,27,,-20.633333,-50.3
br,colonia bromado,Colônia Bromado,18,,-25.665351,-50.207006
br,colonia cachoeira,Colônia Cachoeira,18,,-25.661926,-50.723685
br,colonia campina,Colônia Campina,18,,-25.833333,-52.55
br,colonia campo esperimenta,Colonia Campo Esperimenta,04,,-3.25,-60.25
br,colonia campo novo,Colônia Campo Novo,18,,-25.45,-52.916667
br,colonia castelhanos,Colônia Castelhanos,18,,-25.796266,-48.915261
br,colonia cocalinho,Colônia Cocalinho,16,,-7.816667,-49.266667
br,colonia,Colônia,01,,-8.366667,-73.166667
br,colonia,Colônia,04,,-4.425833,-62.182778
br,colonia,Colônia,05,,-13.8,-44.45
br,colonia,Colônia,11,,-19.333333,-57.316667
br,colonia,Colônia,15,,-17.883333,-41.066667
br,colonia,Colônia,16,,-1.433333,-47.533333
br,colonia,Colônia,18,,-25.383333,-50.1
br,colonia,Colônia,20,,-6.733333,-41.333333
br,colonia,Colônia,21,,-21.383333,-41.416667
br,colonia,Colônia,22,,-5.452778,-37.326111
br,colonia,Colônia,23,,-29.7,-53.7
br,colonia,Colônia,27,,-23.526104,-47.651825
br,colonia,Colônia,30,,-8.733333,-35.816667
br,colonia correa grande,Colônia Correa Grande,14,,-16.616667,-55.266667
br,colonia correctional,Colonia Correctional,21,,-21.666667,-41.9
br,colonia corrego grande,Colônia Córrego Grande,14,,-16.616667,-55.266667
br,colonia correz grande,Colónia Correz Grande,14,,-16.616667,-55.266667
br,colonia da barra,Colonia da Barra,23,,-28.35,-54.266667
br,colonia da boa vista,Colônia da Boa Vista,30,,-7.666667,-37.7
br,colonia da fazenda boa vista,Colônia da Fazenda Boa Vista,15,,-22.5281,-45.6143
br,colonia da fazenda imaculada conceicao,Colônia da Fazenda Imaculada Conceição,15,,-22.5089,-45.6169
br,colonia da fazenda sao bernardo,Colônia da Fazenda São Bernardo,27,,-23.036946,-47.531699
br,colonia da fepasa,Colônia da FEPASA,27,,-23.8552,-46.6958
br,colonia daltro,Colônia Daltro,23,,-29.95,-51.566667
br,colonia dantas,Colônia Dantas,18,,-24.1,-50.283333
br,colonia de bracinho,Colônia de Bracinho,11,,-18.583333,-56.716667
br,colonia de ferias,Colônia de Férias,27,,-24.35,-47.166667
br,colonia de ferias securitario,Colônia de Férias Securitário,15,,-21.946389,-43.319722
br,colonia de itapua,Colônia de Itapuã,23,,-30.35,-51
br,colonia de matapi,Colônia de Matapi,03,,.683333,-51.433333
br,colonia de mineracao,Colonia de Mineraçao,14,,-14.666667,-56.716667
br,colonia do educandario,Colônia do Educandário,21,,-22.30219,-42.812852
br,colonia do lago,Colônia do Lago,18,,-25.351667,-50.042222
br,colonia do miranda,Colonia do Miranda,11,,-21.683333,-56.083333
br,colonia do pescadores,Colônia do Pescadores,03,,2.133333,-50.5
br,colonia do piagui,Colônia do Piagui,27,,-22.754634,-45.180821
br,colonia do prata,Colônia do Prata,16,,-1.3,-47.6
br,colonia dos mineiros,Colônia dos Mineiros,14,,-13.901389,-59.755556
br,colonia dos pescadores,Colônia dos Pescadores,03,,2.133333,-50.5
br,colonia dos tachos,Colônia dos Tachos,14,,-15.55,-53
br,colonia fernando costa,Colônia Fernando Costa,25,,-2.5,-60.9
br,colonia ferreira gomes,Colonia Ferreira Gomes,03,,.8,-51.133333
br,colonia itapeti,Colonia Itapeti,27,,-23.3633,-46.1732
br,colonia jatahy,Colonia Jatahy,18,,-23.216667,-50.966667
br,colonia joanisdorf,Colônia Joanisdorf,18,,-25.738611,-49.765278
br,colonia joao vigesimo terceiro,Colônia João Vigésimo Terceiro,18,,-24.9952,-48.8041
br,colonia joao xxiii,Colônia João XXIII,18,,-24.9952,-48.8041
br,colonia juliano moreira,Colônia Juliano Moreira,21,,-22.936841,-43.397508
br,colonia lagoa,Colônia Lagoa,18,,-25.722808,-50.153567
br,colonia lago do caldeirao,Colônia Lago do Caldeirão,04,,-3.255278,-60.225556
br,colonia lago do limao,Colônia Lago do Limão,04,,-3.186667,-60.341944
br,colonia lajedo,Colonia Lajedo,22,,-5,-37.65
br,colonia langare,Colônia Langare,23,,-28.133333,-52.133333
br,colonia langaro,Colônia Langaro,23,,-28.133333,-52.133333
br,colonia leopoldina,Colônia Leopoldina,02,11914,-8.95,-35.65
br,colonia maciel,Colônia Maciel,18,,-25.428611,-50.136389
br,colonia mallet,Colonia Mallet,18,,-25.4,-52.233333
br,colonia manuel ribas,Colônia Manuel Ribas,18,,-24.516667,-51.65
br,colonia marcelino,Colônia Marcelino,18,,-25.772401,-49.233989
br,colonia marechal floriano,Colonia Marechal Floriano,29,,-15.95,-51.25
br,colonia maria brava,Colônia Maria Brava,18,,-25.256025,-50.612368
br,colonia marques de abrantes,Colônia Marquês de Abrantes,18,,-24.978733,-48.928895
br,colonia merure,Colônia Meruré,14,,-15.55,-53.083333
br,colonia militar,Colônia Militar,13,,-1.8,-46.2
br,colonia militar pedro ii,Colônia Militar Pedro II,03,,.966667,-50.966667
br,colonia mineira,Colonia Mineira,18,,-23.6877,-49.8295
br,colonia miranda,Colônia Miranda,11,,-21.683333,-56.083333
br,colonia moema,Colônia Moema,18,,-25.1,-50.283333
br,colonia municipal bela vista,Colônia Municipal Bela Vista,11,,-22.016667,-56.083333
br,colonia municipal,Colônia Municipal,18,,-25.840426,-49.75598
br,colonia murici,Colônia Murici,18,,-25.583333,-49.083333
br,colonia nova,Colônia Nova,26,,-26.133333,-50.066667
br,colonia nova varsovia,Colônia Nova Varsóvia,18,,-23.5287,-49.9429
br,colonia osorio,Colônia Osório,13,,-1.816667,-46.1
br,colonia ozorio,Colonia Ozorio,13,,-1.816667,-46.1
br,colonia padre paulo,Colônia Padre Paulo,18,,-26.0992,-49.3105
br,colonia paricatuba,Colônia Paricatuba,04,,-3.083333,-60.236389
br,colonia pascal,Colônia Pascal,18,,-24.833333,-53.116667
br,colonia pascoal,Colônia Pascoal,18,,-24.833333,-53.116667
br,colonia passo lancare,Colonia Passo Lancare,23,,-28.133333,-52.133333
br,colonia paulista,Colônia Paulista,27,,-23.8466,-46.6992
br,colonia pequeri,Colônia Pequeri,18,,-24.95,-51.933333
br,colonia piagui,Colônia Piaguí,27,,-22.754634,-45.180821
br,colonia piquiri,Colônia Piquiri,18,,-24.95,-51.933333
br,colonia piragara,Colonia Piragara,14,,-16.983333,-56.433333
br,colonia polidoro,Colônia Polidoro,26,,-26.783333,-51.083333
br,colonia queiroz,Colonia Queiroz,18,,-25.4,-52.233333
br,colonia residencial da fazenda sao pedro,Colônia Residencial da Fazenda São Pedro,27,,-23.5567,-47.5025
br,colonia residencial secretaria da agricultura do estado de sao paulo,Colônia Residencial Secretaria da Agricultura do Estado de São Paulo,27,,-23.0025,-47.7272
br,colonia retiro,Colônia Retiro,18,,-25.7884,-49.2794
br,colonia rio branco,Colonia Rio Branco,04,,-2.766667,-67.666667
br,colonia rio do pinto,Colônia Rio do Pinto,18,,-25.5077,-48.8203
br,colonia rio-jana,Colonia Rio-Jana,04,,-2.766667,-67.666667
br,colonia rio matapi,Colônia Rio Matapi,03,,.683333,-51.433333
br,colonia rodrigues alves,Colônia Rodrigues Alves,01,,-7.716667,-72.65
br,colonia sagrado coracao de jesus,Colônia Sagrado Coração de Jesus,14,,-15.55,-53
br,colonia santa cruz numero 1,Colônia Santa Cruz Número 1,18,,-25.303611,-50.090278
br,colonia santa cruz numero 2,Colônia Santa Cruz Número 2,18,,-25.283889,-50.113056
br,colonia santa cruz numero dois,Colônia Santa Cruz Número Dois,18,,-25.283889,-50.113056
br,colonia santa cruz numero um,Colônia Santa Cruz Número Um,18,,-25.303611,-50.090278
br,colonia santa isabel,Colônia Santa Isabel,14,,-16.683333,-56.016667
br,colonia santa rosa,Colônia Santa Rosa,20,,-6.8,-42.283333
br,colonia santa teresinha,Colônia Santa Teresinha,18,,-25.9,-52.65
br,colonia santa terezinha,Colônia Santa Terezinha,18,,-25.9,-52.65
br,colonia sao bras,Colônia São Brás,26,,-28.466667,-48.866667
br,colonia sao feliciano,Colonia São Feliciano,23,,-30.7,-52.116667
br,colonia sao joao,Colônia São João,18,,-24.95,-53.233333
br,colonia sao jose,Colonia São José,04,,-9.4,-67.083333
br,colonia sao lourenco,Colônia São Lourenço,18,,-25.508614,-50.590901
br,colonia sao miguel,Colônia São Miguel,18,,-23.494279,-49.987652
br,colonia sao paulo,Colonia São Paulo,23,,-29.416667,-53.133333
br,colonia sao pedro,Colônia São Pedro,18,,-24.933333,-53.183333
br,colonia sapucai,Colônia Sapucaí,18,,-24.883333,-53.266667
br,colonia sarandy,Colonia Sarandy,23,,-27.933333,-52.916667
br,colonia sebastiao,Colônia Sebastião,24,,-11.05,-65.333333
br,colonia serrinha,Colônia Serrinha,05,,-16.366667,-39.616667
br,colonia sete barras,Colônia Sete Barras,18,,-24.678901,-48.809952
br,colonia suruqua,Colônia Suruquá,18,,-23.297619,-52.630529
br,colonia taquaral,Colônia Taquaral,18,,-25.833333,-50.366667
br,colonia tavares bastos,Colônia Tavares Bastos,18,,-25.15,-50.2
br,colonia thesoura,Colonia Thesoura,23,,-28.116667,-53.083333
br,colonia velha,Colônia Velha,27,,-24.8486,-48.0277
br,colonia vieira,Colônia Vieira,26,,-26.316667,-50.333333
br,colonia vieiras,Colonia Vieiras,26,,-26.316667,-50.333333
br,colonia vilinha,Colônia Vilinha,18,,-25.388611,-50.102222
br,colono miranda,Colono Miranda,11,,-21.683333,-56.083333
br,colorado,Colorado,18,20449,-22.816667,-51.95
br,colorado,Colorado,23,,-28.516667,-53
br,columna,Columna,15,,-18.233333,-42.833333
br,coluna,Coluna,06,,-4.05,-38.5
br,coluna,Coluna,15,,-18.233333,-42.833333
br,comandahy,Comandahy,23,,-28.133333,-54.566667
br,comandante arbues,Comandante Arbues,27,,-21.15,-51.1
br,comandante,Comandante,05,,-9.8,-43.433333
br,comandante,Comandante,20,,-6.783333,-42.233333
br,comandatuba,Comandatuba,05,,-15.35,-39
br,comandulli,Comandulli,23,,-29.1886,-51.0576
br,combate,Combate,24,,-10.266667,-62.968056
br,combuti,Combuti,13,,-3.147222,-43.114167
br,come-assado,Come-Assado,31,,-10.6,-46.3
br,comendador gomes,Comendador Gomes,15,,-19.683333,-49.083333
br,comendador levy gasparian,Comendador Levy Gasparian,21,,-22.028333,-43.206944
br,comendador soares,Comendador Soares,21,,-22.742165,-43.479066
br,comendador venancio,Comendador Venâncio,21,,-21.183333,-42.1
br,comendador viana,Comendador Viana,15,,-19.433333,-45.783333
br,comercinho,Comercinho,05,,-13.966667,-39.75
br,comercinho,Comercinho,15,,-16.316667,-41.783333
br,comercinho de palha,Comercinho de Palha,08,,-18.133333,-40.35
br,comercio,Comércio,05,,-12.85,-39.216667
br,comercio,Comércio,23,,-29.366667,-53.366667
br,comercio de fora,Comércio de Fora,05,,-13.016667,-41.433333
br,cometa,Cometa,13,,-8.960833,-46.302222
br,commandador gomes,Commandador Gomes,15,,-19.683333,-49.083333
br,commandatuba,Commandatuba,05,,-15.35,-39
br,commercinho,Commercinho,15,,-16.316667,-41.783333
br,comocoxico,Comocoxico,05,,-14.533333,-42.283333
br,comodoro,Comodoro,14,,-13.656944,-59.788889
br,comoxatiba,Comoxatiba,05,,-17.1,-39.183333
br,companhia,Companhia,08,,-21.166667,-41.2
br,companhia,Companhia,15,,-22.833333,-46.133333
br,companugamu,Companugamu,25,,4.316667,-60.133333
br,compasso,Compasso,22,,-6.183333,-37.783333
br,compensa,Compensa,04,,-3.106667,-60.058056
br,compestre,Compestre,15,,-21.716667,-46.25
br,compestre,Compestre,20,,-7.016667,-44.233333
br,compina,Compina,16,,-5.8,-48.333333
br,comportas,Comportas,30,,-8.183333,-34.95
br,compos das almas,Compos das Almas,17,,-7.733333,-36.15
br,comum,Comum,22,,-6.133333,-35.416667
br,comunidade piraque,Comunidade Piraquê,21,,-22.993853,-43.610348
br,comunidade zepelin,Comunidade Zepelin,21,,-22.924515,-43.700494
br,comur,Comur,18,,-23.05,-53.05
br,conani,Conaní,03,,2.866667,-51.1
br,concaicao do jacuipe,Concaição do Jacuípe,05,,-12.3,-38.766667
br,concao,Concão,06,,-7.2,-39.733333
br,concecao do pompeo,Conceção do Pompeó,15,,-19.2,-44.983333
br,conceicao coite,Conceicao Coite,05,,-11.55,-39.266667
br,conceicao,Conceição,03,,1.15,-50.833333
br,conceicao,Conceição,04,,-3.05,-57.783333
br,conceicao,Conceição,05,,-11.15,-43.166667
br,conceicao,Conceição,06,,-3.783333,-40.2
br,conceicao,Conceição,08,,-19.416667,-40.7
br,conceicao,Conceição,11,,-19.383333,-55.8
br,conceicao,Conceição,13,,-2.75,-43.666667
br,conceicao,Conceição,14,,-15.016667,-58.016667
br,conceicao,Conceição,15,,-16.183333,-40.45
br,conceicao,Conceição,16,,-1.05,-52.516667
br,conceicao,Conceição,17,,-6.266667,-37.683333
br,conceicao,Conceição,18,,-24.45,-50.55
br,conceicao,Conceição,20,,-6.55,-42.866667
br,conceicao,Conceição,21,,-21.766667,-41.6
br,conceicao,Conceicão,22,,-5.479444,-36.401389
br,conceicao,Conceição,23,,-28.9,-53.783333
br,conceicao,Conceição,24,,-10.316667,-61.85
br,conceicao,Conceição,25,,2.183333,-60.966667
br,conceicao,Conceição,26,,-26.25,-53.333333
br,conceicao,Conceição,27,,-21.5,-46.766667
br,conceicao,Conceição,30,,-7.883333,-34.833333
br,conceicao,Conceicão,31,,-11.468889,-46.7825
br,conceicao da aparecida,Conceição da Aparecida,15,,-21.1,-46.2
br,conceicao da apparecida,Conceiçao da Apparecida,15,,-21.1,-46.2
br,conceicao da barra,Conceição da Barra,08,19310,-18.583333,-39.75
br,conceicao da barra,Conceição da Barra,15,,-21.116667,-44.466667
br,conceicao da boa vista,Conceição da Boa Vista,15,,-21.55,-42.433333
br,conceicao da feira,Conceição da Feira,05,15838,-12.5,-39
br,conceicao da ibitipoca,Conceição da Ibitipoca,15,,-21.716667,-43.916667
br,conceicao da pedra,Conceição da Pedra,15,,-22.15,-45.45
br,conceicao da pedras,Conceicão da Pedras,15,,-22.15,-45.45
br,conceicao das alagoas,Conceição das Alagoas,15,16152,-19.916667,-48.383333
br,conceicao das creoulas,Conceição das Creoulas,30,,-8.316667,-38.933333
br,conceicao das crioulas,Conceição das Crioulas,30,,-8.316667,-38.933333
br,conceicao das laranjeiras,Conceição das Laranjeiras,15,,-18.866667,-41.4
br,conceicao da vargem,Conceição da Vargem,15,,-16.163056,-44.773611
br,conceicao de baixo,Conceição de Baixo,18,,-25.709265,-50.580079
br,conceicao de bento simoes,Conceição de Bento Simões,05,,-12.15,-38.733333
br,conceicao de campinas,Conceição de Campinas,27,,-22.9,-47.083333
br,conceicao de ipanema,Conceição de Ipanema,15,,-19.916667,-41.683333
br,conceicao de itagua,Conceição de Itaguá,15,,-20.133333,-44.233333
br,conceicao de jacarei,Conceição de Jacareí,21,,-23.033333,-44.15
br,conceicao de jacuipe,Conceição de Jacuípe,05,,-12.3,-38.766667
br,conceicao de macabu,Conceição de Macabu,21,,-22.066667,-41.866667
br,conceicao de monte alegre,Conceição de Monte Alegre,27,,-22.466667,-50.633333
br,conceicao de muquy,Conceição de Muquy,08,,-20.983333,-41.533333
br,conceicao de piracicaba,Conceição de Piraciçaba,15,,-19.9,-43.1
br,conceicao de salinas,Conceição de Salinas,05,,-12.866667,-38.783333
br,conceicao de tronqueiras,Conceição de Tronqueiras,15,,-18.683333,-42.25
br,conceicao de vera cruz,Conceição de Vera Cruz,05,,-13.033333,-38.633333
br,conceicao do almeida,Conceição do Almeida,05,9691,-12.8,-39.2
br,conceicao do araguaia,Conceição do Araguaia,16,27116,-8.241111,-49.296111
br,conceicao do araguaya,Conceicao Do Araguaya,16,,-8.241111,-49.296111
br,conceicao do araxa,Conceição do Araxá,15,,-19.35,-47.283333
br,conceicao do arroio,Conceição do Arroio,23,,-29.9,-50.266667
br,conceicao do arrolo,Conceicao do Arrolo,23,,-29.9,-50.266667
br,conceicao do barreiro,Conceição do Barreiro,15,,-17.316667,-44.466667
br,conceicao do bento simoes,Conceição do Bento Simões,05,,-12.15,-38.733333
br,conceicao do caninde,Conceição do Canindé,20,,-7.9,-41.566667
br,conceicao do capim,Conceição do Capim,15,,-19.566667,-41.233333
br,conceicao do cariri,Conceição do Cariri,06,,-7.516667,-39.116667
br,conceicao do castello,Conceicao do Castello,08,,-20.366667,-41.25
br,conceicao do castelo,Conceição do Castelo,08,4538,-20.366667,-41.25
br,conceicao do coite,Conceição do Coité,05,35172,-11.55,-39.266667
br,conceicao do formoso,Conceição do Formoso,15,,-21.416667,-43.35
br,conceicao do gato,Conceição do Gato,05,,-12.533333,-41.533333
br,conceicao do ibitipoca,Conceição do Ibitipoca,15,,-21.716667,-43.916667
br,conceicao do mato dentro,Conceição do Mato Dentro,15,10708,-19.016667,-43.416667
br,conceicao do mau,Conceição do Maú,25,,3.583333,-59.883333
br,conceicao do monte alegre,Conceição do Monte Alegre,27,,-22.466667,-50.633333
br,conceicao do muqui,Conceição do Muqui,08,,-20.983333,-41.533333
br,conceicao do mutum,Conceição do Mutum,08,,-19.416667,-40.7
br,conceicao do norte,Conceição do Norte,08,,-20.35,-41.4
br,conceicao do norte,Conceição do Norte,31,,-11.468889,-46.7825
br,conceicao do orico,Conceicão do Orico,05,,-14.3,-39.366667
br,conceicao do para,Conceição do Pará,15,,-19.733333,-44.9
br,conceicao do paraiba,Conceição do Paraíba,02,,-9.416667,-36.066667
br,conceicao do paraiba,Conceição do Paraíba,15,,-21.75,-42.383333
br,conceicao do rio acima,Conceição do Rio Acima,15,,-20.083333,-43.583333
br,conceicao do rio verde,Conceição do Rio Verde,15,12402,-21.883333,-45.083333
br,conceicao dos ganchos,Conceição dos Ganchos,04,,-4.735556,-59.929167
br,conceicao dos ouros,Conceição dos Ouros,15,,-22.416667,-45.783333
br,conceicao do tocantins,Conceição do Tocantins,31,,-12.264444,-47.232778
br,conceicao do turvo,Conceição do Turvo,15,,-20.916667,-43.1
br,conceicao ii,Conceição II,13,,-3.778056,-43.298056
br,conceicao velha,Conceição Velha,05,,-12.783333,-39.083333
br,conceicao velha,Conceição Velha,20,,-7.883333,-41.55
br,conceicas,Conceicas,04,,-7.033333,-60.333333
br,concha,Concha,11,,-21.183333,-57.633333
br,concha,Concha,14,,-16.05,-58.8
br,concha,Concha,21,,-21.4,-41.416667
br,conchal,Conchal,11,,-21.183333,-57.633333
br,conchal,Conchal,27,21351,-22.333333,-47.166667
br,conchas,Conchas,11,,-21.183333,-57.633333
br,conchas,Conchas,14,,-16.05,-58.8
br,conchas,Conchas,18,,-25.066667,-50.366667
br,conchas,Conchas,27,14301,-23.016667,-48
br,concordia,Concórdia,04,,-4.596111,-66.623611
br,concordia,Concórdia,15,,-21.6,-45.766667
br,concordia,Concórdia,16,,-1.85,-49.033333
br,concordia,Concórdia,18,,-25.933333,-51.1
br,concordia,Concórdia,20,,-4.633333,-41.633333
br,concordia,Concórdia,23,,-29.55,-51.766667
br,concordia,Concórdia,26,55368,-27.233333,-52.016667
br,concordia,Concórdia,27,,-20.483333,-47.35
br,concordia de mucuri,Concórdia de Mucuri,15,,-17.7,-41.866667
br,concordia do mucuri,Concórdia do Mucuri,15,,-17.7,-41.866667
br,condado,Condado,06,,-5.433333,-39.066667
br,condado,Condado,17,,-6.35,-37.55
br,condado,Condado,20,,-6.766667,-40.65
br,condado,Condado,25,,-3.2,-60.45
br,condado,Condado,30,19585,-7.583333,-35.1
br,condado do norte,Condado do Norte,15,,-15.8,-44.05
br,conde,Conde,05,15565,-11.816667,-37.616667
br,conde,Conde,16,,-1.566667,-48.75
br,conde,Conde,17,,-7.266667,-34.9
br,conde,Condé,23,,-29.966667,-51.8
br,conde de araruama,Conde de Araruama,21,,-22.066667,-41.616667
br,condeixa,Condeixa,16,,-.9,-48.6
br,conde sao miguel,Conde São Miguel,27,,-20.9,-48.85
br,condessa,Condessa,17,,-5.85,-35.85
br,condessa,Condêssa,22,,-5.883333,-35.783333
br,condeuba,Condeúba,05,,-14.883333,-41.983333
br,condicao,Condição,25,,2.183333,-60.966667
br,condominio blue,Condomínio Blue,21,,-23.00198,-43.429575
br,condominio eldorado,Condomínio Eldorado,21,,-22.943697,-43.32727
br,condominio floresta,Condomínio Floresta,21,,-22.970149,-43.32733
br,condominio mansoes,Condomínio Mansões,21,,-23.002547,-43.415793
br,condominio pedra de itauna,Condomínio Pedra de Itauna,21,,-23.005901,-43.422357
br,condominio pitocos,Condomínio Pitocos,21,,-22.935312,-43.632396
br,condominio urussanga,Condomínio Urussanga,21,,-22.946977,-43.331828
br,condor,Condor,23,,-28.2,-53.483333
br,conduru,Conduru,08,,-20.683333,-41.2
br,conduru,Conduru,13,,-5.566667,-44.333333
br,conego joao pio,Cônego João Pio,15,,-20,-42.933333
br,conego lira,Cônego Lira,30,,-8.133333,-36.316667
br,conego marinho,Cônego Marinho,15,,-15.233333,-44.45
br,confins,Confins,15,,-19.616667,-43.983333
br,confusao,Confusão,15,,-20.483333,-46.25
br,confusao,Confusão,27,,-20.616667,-47.166667
br,congo,Congo,05,,-12.216667,-37.8
br,congo,Congo,17,,-7.8,-36.666667
br,congonha,Congonha,06,,-3.766667,-41.283333
br,congonhal,Congonhal,15,,-21.933333,-45.816667
br,congonhal de baixo,Congonhal de Baixo,27,,-23.42117,-47.81096
br,congonhal de cima,Congonhal de Cima,27,,-23.451476,-47.822305
br,congonhas,Congonhas,15,48819,-20.5,-43.866667
br,congonhas,Congonhas,18,,-23.2,-50.733333
br,congonhas,Congonhas,27,,-23.7274,-47.8115
br,congonhas do campo,Congonhas do Campo,15,,-20.5,-43.866667
br,congonhas do norte,Congonhas do Norte,15,,-18.8,-43.666667
br,congonhas do norte,Congonhas do Norte,18,,-23.2,-50.733333
br,congonhinhas,Congonhinhas,18,,-23.5495,-50.5508
br,congorito,Congorito,05,,-11.95,-37.7
br,conjugado habitacao companhia de habitacao do parana,Conjugado Habitação Companhia de Habitação do Paraná,18,,-23.029809,-50.072102
br,conjugado habitacao vila mariana,Conjugado Habitação Vila Mariana,26,,-26.229515,-49.508792
br,conjuncao residencial bela vista,Conjunção Residencial Bela Vista,26,,-26.1997,-49.271
br,conjunto alvorada,Conjunto Alvorada,21,,-22.898991,-43.721623
br,conjunto antares,Conjunto Antares,21,,-22.919768,-43.659831
br,conjunto bandeirantes,Conjunto Bandeirantes,21,,-22.934367,-43.366297
br,conjunto boa esperanca,Conjunto Boa Esperança,21,,-22.905622,-43.667552
br,conjunto campinho,Conjunto Campinho,21,,-22.880717,-43.613711
br,conjunto dom jamie camara,Conjunto Dom Jamie Camara,21,,-22.86623,-43.458205
br,conjunto dom pedro i,Conjunto Dom Pedro I,21,,-22.884654,-43.426888
br,conjunto esperanca,Conjunto Esperança,21,,-22.809421,-43.31989
br,conjunto habitacao doutor jamidas,Conjunto Habitação Doutor Jamidas,18,,-23.280591,-50.069796
br,conjunto habitacao joaquim c. da silveira,Conjunto Habitação Joaquim C. da Silveira,18,,-23.305579,-50.089423
br,conjunto habitacional companhia de habitacao do parana ii,Conjunto Habitacional Companhia de Habitação do Paraná II,18,,-25.802709,-48.538626
br,conjunto habitacional pica-pau-amarelo,Conjunto Habitacional Pica-Pau-Amarelo,18,,-25.796,-48.5255
br,conjunto joao saldanha,Conjunto João Saldanha,21,,-22.846278,-43.462129
br,conjunto joao xxiii,Conjunto João XXIII,21,,-22.904824,-43.702702
br,conjunto manguariba,Conjunto Manguariba,21,,-22.872372,-43.632949
br,conjunto miecimo da silva,Conjunto Miécimo da Silva,21,,-22.900895,-43.712587
br,conjunto missoes,Conjunto Missoes,21,,-22.805243,-43.283028
br,conjunto nelson mandela,Conjunto Nelson Mandela,21,,-22.885388,-43.246687
br,conjunto nova pavuna,Conjunto Nova Pavuna,21,,-22.811076,-43.35424
br,conjunto nova sepetiba,Conjunto Nova Sepetiba,21,,-22.958043,-43.686956
br,conjunto nova sepetiba ii,Conjunto Nova Sepetiba II,21,,-22.968354,-43.661587
br,conjunto novo mundo,Conjunto Novo Mundo,21,,-22.896302,-43.72172
br,conjunto otacilio camara,Conjunto Otacílio Camará,21,,-22.936075,-43.656877
br,conjunto pro-morar,Conjunto Pró-Morar,21,,-22.851688,-43.397404
br,conjunto quafa,Conjunto Quafá,21,,-22.850834,-43.488401
br,conjunto residencia frei joao e. couto,Conjunto Residência Frei João E. Couto,18,,-23.298246,-50.061345
br,conjunto rio grande,Conjunto Rio Grande,21,,-22.893601,-43.716405
br,conjunto sao fernando,Conjunto São Fernando,21,,-22.888717,-43.719521
br,conjunto sargento joao lima,Conjunto Sargento João Lima,21,,-22.923599,-43.540615
br,conjunto sargento miguel filho,Conjunto Sargento Miguel Filho,21,,-22.858071,-43.481218
br,conjunto seis de novembro,Conjunto Seis de Novembro,21,,-22.852625,-43.457107
br,conjunto taquaral,Conjunto Taquaral,21,,-22.867211,-43.486262
br,conjunto urucania,Conjunto Urucânia,21,,-22.912618,-43.654748
br,conori ponta,Conori Ponta,04,,.05,-67.633333
br,conquista,Conquista,05,,-11.816667,-41.516667
br,conquista,Conquista,15,,-19.733333,-44.6
br,conquista,Conquista,21,,-22.283333,-42.683333
br,conquista,Conquista,26,,-26.3796,-48.6685
br,conquista,Conquista,27,,-21.166667,-47.883333
br,conrado,Conrado,21,,-22.535556,-43.567222
br,conselheiro josino,Conselheiro Josino,21,,-21.5,-41.35
br,conselheiro joviano,Conselheiro Joviano,21,,-21.5,-41.35
br,conselheiro jovino,Conselheiro Jovino,21,,-21.5,-41.35
br,conselheiro lafaiete,Conselheiro Lafaiete,15,111598,-20.666667,-43.8
br,conselheiro lafayete,Conselheiro Lafayete,15,,-20.666667,-43.8
br,conselheiro mairinck,Conselheiro Mairinck,18,,-23.622841,-50.169004
br,conselheiro mairink,Conselheiro Mairink,18,,-23.622841,-50.169004
br,conselheiro mata,Conselheiro Mata,15,,-18.283333,-43.983333
br,conselheiro matta,Conselheiro Matta,15,,-18.283333,-43.983333
br,conselheiro mayrinck,Conselheiro Mayrinck,18,,-23.622841,-50.169004
br,conselheiro paulino,Conselheiro Paulino,21,,-22.216667,-42.516667
br,conselheiro pena,Conselheiro Pena,15,12735,-19.166667,-41.5
br,conselheiro zacarias,Conselheiro Zacarias,18,,-23.412809,-49.979241
br,conselho,Conselho,05,,-12.816667,-40.383333
br,conselho,Conselho,06,,-7.133333,-38.683333
br,conselho,Conselho,22,,-5.933333,-38.116667
br,conselho martim francisco,Conselho Martim Francisco,27,,-22.516667,-46.95
br,consequencia,Consequência,14,,-16.916667,-57.583333
br,conserva,Conserva,13,,-2.75,-44.95
br,conservatoria,Conservatória,21,,-22.288333,-43.931111
br,consolacao,Consolação,15,,-22.55,-45.916667
br,consolacao,Consolação,20,,-9.883333,-43.816667
br,consolacao,Consolação,21,,-22.912313,-43.26992
br,consolacao,Consolação,25,,3.933333,-60.983333
br,consolacao,Consolação,31,,-10.5,-49.616667
br,constancia,Constância,21,,-22.367056,-42.93291
br,constancia,Constância,27,,-21.383333,-46.8
br,constancia das baetas,Constãncia das Baetas,04,,-6.2,-62.216667
br,constancia dos baetas,Constância dos Baetas,04,,-6.2,-62.216667
br,constantcia baetas,Constantcia Baetas,04,,-6.2,-62.216667
br,constant,Constant,15,,-21.95,-42.883333
br,constantina,Constantina,23,,-27.733333,-52.983333
br,constituinte,Constituinte,06,,-3.683333,-39.566667
br,consuelo,Consuelo,24,,-12.65,-62.133333
br,consulado,Consulado,21,,-22.955574,-43.573667
br,contagem,Contagem,15,627132,-19.916667,-44.1
br,contao,Contão,25,,3.933333,-60.416667
br,contenda,Contenda,05,,-13.7,-41.2
br,contenda,Contenda,18,,-25.7,-49.533333
br,contenda,Contenda,22,,-5.65,-35.233333
br,contendas,Contendas,05,,-11.55,-39.7
br,contendas,Contendas,06,,-3,-39.95
br,contendas,Contendas,15,,-21.9,-45.016667
br,contendas,Contendas,17,,-6.433333,-37.583333
br,contendas,Contendas,20,,-3.666667,-42.45
br,contendas,Contendas,27,,-20.683333,-47.183333
br,contendas,Contendas,29,,-18.05,-47.35
br,contendas do sincora,Contendas do Sincorá,05,,-13.75,-41.033333
br,contentes,Contentes,20,,-6.083333,-41.4
br,contestado,Contestado,01,,-8.358056,-72.72
br,contiguiba,Contiguiba,28,,-10.866667,-37.116667
br,continente,Continente,27,,-24.61609,-48.450426
br,contracto,Contracto,20,,-9.65,-45.183333
br,contrato,Contrato,20,,-9.65,-45.183333
br,contrato,Contrato,23,,-29.083333,-50.766667
br,contria,Contria,15,,-18.233333,-44.516667
br,convenfo,Convenfo,05,,-14.066667,-39.566667
br,convento,Convento,05,,-14.066667,-39.566667
br,convento,Convento,06,,-5.3,-40.716667
br,convento,Convento,18,,-25.569444,-48.312778
br,convento,Convento,28,,-11.466667,-37.45
br,convento da penha,Convento da Penha,08,,-20.333333,-40.283333
br,conventos,Conventos,26,,-28.95,-49.383333
br,convivencia,Convivência,21,,-21.6,-41.016667
br,copacabana,Copacabana,14,,-15.416667,-59.983333
br,copacabana,Copacabana,21,,-22.968532,-43.185787
br,copaiba,Copaíba,04,,-4.325833,-65.278889
br,copasa,Copasa,18,,-25.840943,-48.787556
br,copatana,Copatana,04,,-2.8,-67.066667
br,copichaba,Copichaba,05,,-11.316667,-43.166667
br,copioba,Copioba,05,,-12.116667,-37.966667
br,copixaba,Copixaba,05,,-11.316667,-43.166667
br,coque,Coque,06,,-5.366667,-39.133333
br,coqueiral,Coqueiral,15,,-21.183333,-45.45
br,coqueiral,Coqueiral,30,,-8.083333,-34.966667
br,coqueiral cupe,Coqueiral Cupe,02,,-9.4,-35.5
br,coqueiralzinho,Coqueiralzinho,18,,-23.1156,-50.1363
br,coqueirinho,Coqueirinho,17,,-6.766667,-34.95
br,coqueiro,Coqueiro,02,,-9.383333,-37.266667
br,coqueiro,Coqueiro,05,,-11.483333,-37.4
br,coqueiro,Coqueiro,06,,-7.183333,-40.483333
br,coqueiro,Coqueiro,11,,-21.216667,-56.383333
br,coqueiro,Coqueiro,13,,-3.083333,-42.4
br,coqueiro,Coqueiro,14,,-15.8,-57.2
br,coqueiro,Coqueiro,15,,-19.65,-44.833333
br,coqueiro,Coqueiro,18,,-26.174167,-53.562222
br,coqueiro,Coqueiro,20,,-2.907778,-41.578889
br,coqueiro,Coqueiro,21,,-21.216667,-41.433333
br,coqueiro,Coqueiro,22,,-5.026667,-37.461944
br,coqueiro,Coqueiro,25,,3.083333,-61.183333
br,coqueiro,Coqueiro,30,,-7.6,-37.583333
br,coqueiro grande,Coqueiro Grande,05,,-12.716667,-38.55
br,coqueiros,Coqueiros,05,,-12.15,-38.683333
br,coqueiros,Coqueiros,08,,-20.866667,-41.25
br,coqueiros,Coqueiros,14,,-15.416667,-57
br,coqueiros,Coqueiros,15,,-19.133333,-42.616667
br,coqueiros,Coqueiros,21,,-21.883333,-41.3
br,coqueiros,Coqueiros,23,,-28.116667,-52.783333
br,coqueiros,Coqueiros,26,,-26.4878,-48.7241
br,coqueiros,Coqueiros,27,,-21,-47.833333
br,coqueiros,Coqueiros,31,,-7.883333,-48.583333
br,coqueiro secco,Coqueiro Secco,02,,-9.633333,-35.783333
br,coqueiro seco,Coqueiro Sêco,02,,-9.633333,-35.783333
br,coquiero,Coquiero,14,,-15.716667,-57.033333
br,coquinhos,Coquinhos,05,,-14.55,-40.683333
br,coracao,Coração,26,,-26.95,-51.766667
br,coracao de jesus,Coração de Jesus,05,,-11.65,-39.933333
br,coracao de jesus,Coração de Jesus,15,12410,-16.684444,-44.360278
br,coracao de jesus,Coração de Jesus,26,,-27.283333,-51.1
br,coracao de maria,Coração de Maria,05,6571,-12.233333,-38.75
br,coracao de maria,Coração de Maria,30,,-8.616667,-40.916667
br,coracao do brasil,Coração do Brasil,15,,-17.85,-40.866667
br,corbelia,Corbélia,18,12725,-24.8,-53.3
br,corcovado,Corcovado,15,,-15.966667,-41.1
br,corcovado,Corcovado,16,,-1.65,-50.533333
br,corcunda,Corcunda,06,,-5.2,-38.283333
br,corcunda,Corcunda,11,,-19.983333,-56.45
br,corda,Corda,31,,-6.466667,-48.216667
br,cordeiro,Cordeiro,02,,-9.383333,-38.116667
br,cordeiro,Cordeiro,21,15602,-22.033333,-42.366667
br,cordeiro,Cordeiro,22,,-5.583333,-35.266667
br,cordeiro,Cordeiro,27,,-22.483333,-47.466667
br,cordeiro,Cordeiro,30,,-8.866667,-36.95
br,cordeiropolis,Cordeirópolis,27,25639,-22.483333,-47.466667
br,cordeiros,Cordeiros,05,,-15.016667,-41.933333
br,cordeiros,Cordeiros,17,,-7.383333,-36.8
br,cordeiros,Cordeiros,21,,-22.866667,-42.95
br,cordeiros,Cordeiros,27,,-22.483333,-47.466667
br,cordilheira,Cordilheira,23,,-30.3,-52.85
br,cordilheiras,Cordilheiras,23,,-30.7,-53.816667
br,cordisburgo,Cordisburgo,15,,-19.116667,-44.35
br,cordovil,Cordovil,21,,-22.825276,-43.303956
br,cordozal,Cordozal,04,,.625,-62.651111
br,coreahu,Coreahú,06,,-3.366667,-40.766667
br,coreau,Coreaú,06,12383,-3.55,-40.65
br,coremas,Coremas,17,11897,-7.016667,-37.966667
br,corguinho,Córguinho,06,,-2.816667,-40.4
br,corguinho,Corguinho,11,,-19.883333,-54.866667
br,corguinho dos agostinho,Corguinho dos Agostinho,06,,-2.916667,-40.05
br,coribe,Coribe,05,,-13.833333,-44.466667
br,corimbata,Corimbatá,11,,-21.25,-57.266667
br,corimbota,Corimbota,11,,-21.25,-57.266667
br,corina,Corina,11,,-22.216667,-56.183333
br,corintho,Corintho,15,,-18.35,-44.45
br,corinto,Corinto,15,22121,-18.35,-44.45
br,coripos,Coripós,30,,-8.816667,-39.816667
br,corisco,Corisco,13,,-3.308056,-42.335833
br,corisco,Corisco,27,,-23.289665,-46.506647
br,corityba,Corityba,18,,-25.416667,-49.25
br,corixa,Corixa,14,,-16.466667,-55.666667
br,corixao,Corixão,11,,-18.333333,-56.533333
br,corixo,Corixo,14,,-16.25,-56.166667
br,cornelio procopio,Cornélio Procópio,18,45031,-23.133333,-50.65
br,cornelios,Cornélios,23,,-29.65,-50.05
br,cornelius,Cornélius,23,,-29.65,-50.05
br,cornicha,Cornicha,05,,-13.366667,-40.216667
br,coroaci,Coroaci,15,,-18.583333,-42.283333
br,coroa,Coroa,05,,-13.016667,-38.616667
br,coroa,Coroa,14,,-17.2,-54.5
br,coroacy,Coroacy,15,,-18.583333,-42.283333
br,coroa da ilha,Coroa da Ilha,05,,-9.683333,-41.966667
br,coroa da onca,Coroa da Onça,08,,-20.95,-40.883333
br,coroados,Coroados,27,,-21.366667,-50.25
br,coroa grande,Coroa Grande,06,,-2.816667,-40.15
br,coroa grande,Coroa Grande,21,,-22.898611,-43.863889
br,coroa grande,Coroa Grande,22,,-5.734722,-36.811667
br,coroas,Coroas,15,,-21.016667,-44.233333
br,coroas,Coroas,21,,-22.129444,-43.735
br,coroas limpas,Coroas Limpas,22,,-5.89,-36.581389
br,coroata,Coroatá,06,,-4.4,-40.883333
br,coroata,Coroatá,13,34130,-4.133333,-44.133333
br,coroata,Coroatá,20,,-6.983333,-41.516667
br,corocoro,Corocoró,04,,.066667,-67.683333
br,coromandel,Coromandel,15,22648,-18.466667,-47.216667
br,corona,Corona,11,,-22.45,-55.016667
br,coronel albuquerque,Coronel Albuquerque,14,,-15.45,-54.783333
br,coronel barros,Coronel Barros,11,,-18.65,-56.783333
br,coronel barros,Coronel Barros,23,,-28.383333,-54.066667
br,coronel bicaco,Coronel Bicaco,23,,-27.716667,-53.716667
br,coronel cardoso,Coronel Cardoso,21,,-22.15,-43.966667
br,coronel carlos,Coronel Carlos,22,,-5.381111,-36.938056
br,coronel,Coronel,05,,-12.566667,-41.816667
br,coronel ezequiel,Coronel Ezequiel,22,,-6.383333,-36.183333
br,coronel fabriciano,Coronel Fabriciano,15,100918,-19.516667,-42.633333
br,coronel flores,Coronel Flores,23,,-30.4,-55.9
br,coronel freitas,Coronel Freitas,26,,-26.933333,-52.783333
br,coronel galvao,Coronel Galvão,11,,-18.933333,-54.866667
br,coronel goulart,Coronel Goulart,27,,-22.2,-51.533333
br,coronel joao pessoa,Coronel João Pessoa,22,,-6.266667,-38.483333
br,coronel macedo,Coronel Macedo,27,,-23.633333,-49.316667
br,coronel maia,Coronel Maia,17,,-6.233333,-37.733333
br,coronel manoel viana,Coronel Manoel Viana,23,,-29.583333,-55.483333
br,coronel martim francisco,Coronel Martim Francisco,27,,-22.516667,-46.95
br,coronel martins,Coronel Martins,26,,-26.5,-52.65
br,coronel murta,Coronel Murta,15,,-16.616667,-42.183333
br,coronel pacheco,Coronel Pacheco,15,,-21.583333,-43.266667
br,coronel passos maia,Coronel Passos Maia,26,,-26.816667,-52.016667
br,coronel pilar,Coronel Pilar,23,,-29.266667,-51.683333
br,coronel ponce,Coronel Ponce,14,,-15.566667,-55.016667
br,coronel retiro,Coronel Retiro,14,,-15.366667,-56.633333
br,coronel soares,Coronel Soares,05,,-14.383333,-39.6
br,coronel teixeira,Coronel Teixeira,23,,-27.4,-52.033333
br,coronel vivida,Coronel Vivida,18,15841,-25.966667,-52.566667
br,coronzo,Coronzó,06,,-6.516667,-40.65
br,coroveira,Coroveira,18,,-25.166667,-53.55
br,cor procopio,Cor Procopio,18,,-23.133333,-50.65
br,corralinho,Corralinho,20,,-6.883333,-42.7
br,correa pinto,Correa Pinto,26,,-27.583333,-50.4
br,corredeira,Corredeira,14,,-16.033333,-56.716667
br,corredeira,Corredeira,20,,-3.5775,-42.342222
br,corredeira,Corredeira,27,,-22.05,-49.633333
br,corredor,Corredor,17,,-6.5,-37.483333
br,corredor,Corredor,22,,-6.183333,-37.95
br,corredores,Corredores,06,,-5.35,-40.783333
br,corredores,Corredores,13,,-5.783333,-43.116667
br,corredores,Corredores,20,,-5.116667,-42.033333
br,corrego alto,Córrego Alto,08,,-20.65,-41.033333
br,corrego azul,Córrego Azul,27,,-21.083333,-50.533333
br,corrego,Córrego,05,,-13.066667,-39.566667
br,corrego,Córrego,06,,-3.033333,-40.35
br,corrego,Córrego,22,,-5.320278,-36.352778
br,corrego da bica,Córrego da Bica,18,,-24.8948,-48.5953
br,corrego da foice,Córrego da Foice,15,,-22.6457,-45.7477
br,corrego da forquiha,Córrego da Forquiha,06,,-2.883333,-40.483333
br,corrego d'agua,Corrego d'Agua,08,,-19.7,-40.266667
br,corrego da lavra,Córrego da Lavra,27,,-24.9582,-48.5495
br,corrego da luz,Córrego da Luz,21,,-22.433333,-42.25
br,corrego d'anta,Corrego d'Anta,08,,-20.45,-41.7
br,corrego danta,Córrego Danta,15,,-19.8,-45.9
br,corrego d'anta,Corrego d'Anta,15,,-20.616667,-45.8
br,corrego dantas,Córrego Dantas,08,,-20.45,-41.7
br,corrego da onca,Córrego da Onça,27,,-24.8119,-48.5265
br,corrego da poeira,Córrego da Poeira,06,,-2.916667,-40.25
br,corrego da prata,Córrego da Prata,21,,-21.866667,-42.466667
br,corrego das panelas,Córrego das Panelas,06,,-2.866667,-40.433333
br,corrego das varas,Córrego das Varas,06,,-2.95,-40.116667
br,corrego da valeria,Córrego da Valéria,15,,-20.016667,-43.366667
br,corrego de dentro,Córrego de Dentro,06,,-2.922778,-40.524167
br,corrego de ouro,Córrego de Ouro,29,,-16.3,-50.533333
br,corrego do barro,Córrego do Barro,27,,-24.749686,-48.464115
br,corrego do bom jesus,Córrego do Bom Jesus,15,,-22.633333,-46.033333
br,corrego do cafe,Córrego do Café,08,,-18.85,-40.75
br,corrego do diamante,Córrego do Diamante,18,,-24.7622,-48.932
br,corrego do franco,Córrego do Franco,18,,-24.7995,-48.5512
br,corrego do franco,Córrego do Franco,27,,-24.8075,-48.5531
br,corrego do frango,Córrego do Frango,18,,-24.7948,-48.5599
br,corrego do meio,Córrego do Meio,06,,-2.933333,-40.466667
br,corrego do meio,Córrego do Meio,08,,-20.266667,-41.1
br,corrego do monjolo,Córrego do Monjolo,18,,-24.76337,-48.909082
br,corrego do ouro,Córrego do Ouro,15,,-21.366667,-45.783333
br,corrego do prata,Corrego do Prata,21,,-21.866667,-42.466667
br,corrego do retiro,Córrego do Retiro,06,,-4.6,-37.65
br,corrego do rodrigues,Córrego do Rodrigues,06,,-4.55,-37.716667
br,corrego dos ana,Córrego dos Ana,06,,-2.866667,-40.383333
br,corrego dos borges,Córrego dos Borges,06,,-2.916667,-40.45
br,corrego dos manos,Córrego dos Manos,08,,-20.883333,-41.2
br,corrego do soldado,Córrego do Soldado,15,,-20.183333,-44.566667
br,corrego fundo,Córrego Fundo,06,,-3.35,-39.35
br,corrego fundo,Córrego Fundo,08,,-19.85,-40.283333
br,corrego fundo,Córrego Fundo,15,,-20.466667,-45.533333
br,corrego fundo,Córrego Fundo,27,,-21.616667,-47.35
br,corrego grande,Córrego Grande,05,,-16.25,-39.683333
br,corrego grande,Córrego Grande,06,,-2.906111,-40.629444
br,corrego grande,Córrego Grande,08,,-18.583333,-40.616667
br,corrego grande,Córrego Grande,18,,-24.755585,-48.863228
br,corrego grande do bee,Corrego Grande do Beé,06,,-2.966667,-39.966667
br,corrego novo,Córrego Novo,06,,-3.016667,-39.966667
br,corrego novo,Córrego Novo,15,,-19.833333,-42.383333
br,corrego rico,Córrego Rico,27,,-21.283333,-48.266667
br,corrego rico,Córrego Rico,29,,-15.233333,-47.8
br,corrego salgado,Córrego Salgado,06,,-4.966667,-37.716667
br,corrego sao luis,Córrego São Luís,26,,-26.9825,-49.9563
br,corregos,Córregos,15,,-18.883333,-43.516667
br,corregos,Córregos,17,,-6.433333,-37.883333
br,corregos,Córregos,18,,-24.962731,-48.96971
br,corrego seco,Córrego Sêco,27,,-20.966667,-48.933333
br,corrego sujo,Córrego Sujo,21,,-22.253698,-42.832709
br,corregozinho,Còrregozinho,15,,-20.166667,-45.266667
br,corregozinho,Corregozinho,18,,-24.804947,-48.916128
br,correia,Correia,14,,-16.8,-53.05
br,correia,Correia,15,,-20.816667,-45.666667
br,correia de almeida,Correia de Almeida,15,,-21.283333,-43.633333
br,correia pinto,Correia Pinto,26,13752,-27.583333,-50.4
br,correias,Correias,15,,-21.75,-46.333333
br,correias,Correias,18,,-25.51641,-50.217265
br,correias,Correias,22,,-5,-37.5
br,correias,Correias,27,,-23.492979,-47.796283
br,correntao,Correntão,13,,-7.916667,-45.983333
br,corrente,Corrente,05,,-11.9,-45.066667
br,corrente,Corrente,06,,-7.666667,-39.15
br,corrente,Corrente,11,,-17.616667,-54.983333
br,corrente,Corrente,13,,-3.5725,-42.849722
br,corrente,Corrente,20,13967,-10.436111,-45.131111
br,correntes,Correntes,11,,-17.616667,-54.983333
br,correntes,Correntes,26,,-27.083333,-50.633333
br,correntes,Correntes,30,,-9.133333,-36.316667
br,correnteza,Correnteza,04,,-3.5,-68.316667
br,correnteza,Correnteza,21,,-21.283333,-41.316667
br,correntezas,Correntezas,21,,-22.5,-42.516667
br,correntina,Correntina,05,13187,-13.333333,-44.65
br,correntinho,Correntinho,15,,-18.666667,-42.783333
br,corrida,Corrida,15,,-19.5,-42.833333
br,cortada,Cortada,17,,-7.133333,-37.933333
br,corta dedos,Corta Dedos,17,,-7.133333,-36.166667
br,cortado,Cortado,20,,-6.633333,-41.783333
br,cortado,Cortado,23,,-29.75,-52.966667
br,cortado,Cortado,27,,-21.533333,-47.766667
br,corta mao,Corta Mão,05,,-13.05,-39.5
br,corte,Côrte,23,,-30.15,-55.066667
br,corte,Corte,30,,-9.083333,-38.05
br,corte de pedra,Corte de Pedra,05,,-13.5,-39.45
br,corte grande,Corte Grande,05,,-12.066667,-37.766667
br,corte grande,Corte Grande,27,,-22.15,-47.4
br,cortes,Cortês,30,,-8.466667,-35.55
br,cortez,Cortez,30,,-8.466667,-35.55
br,cortume,Cortume,20,,-5.133333,-41.166667
br,cortume,Cortume,30,,-8.183333,-36
br,coruja,Coruja,15,,-17.25,-41.7
br,coruja,Coruja,30,,-8.066667,-36.6
br,corujas,Corujas,06,,-6.016667,-38.95
br,corujas,Corujás,26,,-28.266667,-49.116667
br,corumba,Corumbá,05,,-15.916667,-48.8
br,corumba,Corumbá,11,96523,-19.016667,-57.65
br,corumba,Corumbá,27,,-23.243056,-46.563333
br,corumba de goias,Corumbá de Goiás,05,,-15.916667,-48.8
br,corumbahyba,Corumbahyba,29,,-18.15,-48.566667
br,corumbaiba,Corumbaíba,29,,-18.15,-48.566667
br,corumbalina,Corumbalina,29,,-17.316667,-48.5
br,corumbatahy,Corumbatahy,27,,-22.233333,-47.633333
br,corumbatai,Corumbataí,27,,-22.233333,-47.633333
br,corumbazul,Corumbazul,29,,-18.25,-48.866667
br,corunha,Corunhã,11,,-18.4,-56.183333
br,corupa,Corupá,26,,-26.433333,-49.233333
br,coruripe,Coruripe,02,31428,-10.133333,-36.166667
br,corveta,Corveta,26,,-26.4504,-48.7445
br,corvo branco,Corvo Branco,18,,-25.566667,-52.05
br,corvo,Corvo,23,,-29.366667,-51.866667
br,cosme de brito,Cosme de Brito,17,,-6.45,-38.55
br,cosme pais,Cosme Pais,06,,-5.433333,-39.133333
br,cosme pinto,Cosme Pinto,17,,-7.133333,-36.8
br,cosme velho,Cosme Velho,21,,-22.933333,-43.2
br,cosmo,Cosmo,21,,-22.90862,-43.616594
br,cosmo,Cosmo,22,,-5.566667,-35.383333
br,cosmopolis,Cosmópolis,27,47773,-22.633333,-47.2
br,cosmorama,Cosmorama,27,,-20.466667,-49.783333
br,cosmos,Cosmos,21,,-22.90862,-43.616594
br,cosso,Cossó,13,,-3.016667,-43.35
br,cosstana,Cosstanã,04,,-7.266667,-64.883333
br,costa barros,Costa Barros,21,,-22.816916,-43.373008
br,costa,Costa,14,,-16.533333,-57.516667
br,costa,Costa,20,,-7.75,-42.25
br,costa,Costa,21,,-22.716667,-41.983333
br,costa da cadeia,Costa da Cadeia,23,,-29.7,-51.683333
br,costa da praia,Costa da Praia,26,,-28.766667,-49.166667
br,costa da salvacao,Costa da Salvação,04,,-3.843611,-62.530556
br,costa do anajatuba,Costa do Anajatuba,04,,-2.801944,-60.771667
br,costa do ipiranga,Costa do Ipiranga,04,,-4.3625,-59.764167
br,costa do julio,Costa do Júlio,04,,-3.873056,-62.6425
br,costa do ladrao,Costa do Ladrão,23,,-30.6692,-52.3272
br,costa do mamuripana,Costa do Mamuripana,25,,-.338611,-61.859722
br,costa dos rosa,Costa Dos Rosa,04,,-4.5775,-62.061944
br,costa do trocaris,Costa do Trocaris,04,,-3.900278,-62.863611
br,costa machado,Costa Machado,27,,-22.2,-51.833333
br,costa marques,Costa Marques,24,,-12.466667,-64.266667
br,costaneira,Costaneira,20,,-7.333333,-41.766667
br,costao,Costão,23,,-29.45,-51.966667
br,costa pereira,Costa Pereira,08,,-20.416667,-40.65
br,costa rica,Costa Rica,11,12931,-23.416667,-54.65
br,costa rica,Costa Rica,27,,-23.013779,-47.657411
br,costas,Costas,06,,-6,-40.533333
br,costas,Costas,15,,-19.966667,-44.933333
br,costas,Costas,27,,-23.75,-48.65
br,costa sena,Costa Sena,15,,-18.666667,-43.666667
br,costeira,Costeira,26,,-26.4465,-48.6248
br,costela,Costela,05,,-12.966667,-41.466667
br,costello,Costello,20,,-5.333333,-41.55
br,costinha,Costinha,17,,-6.965556,-34.858333
br,cotaxe,Cotaxé,08,,-18.2,-40.716667
br,cotegipe,Cotegipe,05,,-12.033333,-44.25
br,cotegipe,Cotegipe,23,,-27.616667,-52.383333
br,cotejipe,Cotejipe,05,,-12.033333,-44.25
br,cotia,Cotia,02,,-9.016667,-37.5
br,cotia,Cotia,08,,-20.75,-41.55
br,cotia,Cotia,13,,-9.533333,-46.116667
br,cotia,Cotia,14,,-15.966667,-56.466667
br,cotia,Cotia,27,174832,-23.616667,-46.933333
br,cotia fria,Cotia Fria,13,,-2.533333,-45.35
br,cotia gorda,Cotia Gorda,13,,-3.548056,-43.375
br,cotias,Cotias,13,,-3.166667,-45.166667
br,coti,Coti,05,,-11.95,-38.1
br,cotingo,Cotingo,25,,3.9,-60.483333
br,cotinguiba,Cotingüiba,28,,-10.866667,-37.116667
br,cotipora,Cotiporã,23,,-29.016667,-51.7
br,coto,Cotó,13,,-3.266667,-42.883333
br,coto,Cotó,22,,-5.383333,-37.066667
br,cotovello,Cotovello,22,,-5.933333,-35.933333
br,cotovelo,Cotovelo,04,,.8575,-62.584444
br,cotovelo,Cotovêlo,15,,-17.883333,-44.683333
br,cotovelo,Cotovêlo,22,,-6.166667,-38.3
br,cotovelo,Cotovêlo,23,,-31.666667,-52.1
br,counani,Counani,03,,2.866667,-51.1
br,courao,Courão,05,,-12.833333,-39.233333
br,couro d'anta,Couro d'Anta,30,,-8.316667,-36.766667
br,couro de porco,Couro de Porco,20,,-3.966944,-42.251111
br,coutinho,Coutinho,06,,-5.816667,-40.683333
br,coutinho,Coutinho,27,,-24.871569,-48.11342
br,couto de magalhaes,Couto de Magalhães,15,,-18.066667,-43.416667
br,couto de magalhaes,Couto de Magalhães,31,,-8.310278,-49.056944
br,couto magalhaes,Couto Magalhães,31,,-8.310278,-49.056944
br,coutos,Coutos,05,,-14.85,-39.066667
br,couves,Couves,20,,-9.316667,-44.516667
br,cova,Cóva,05,,-10.866667,-40.016667
br,cova da onca,Cova da Onça,05,,-12.283333,-38.35
br,cova de mandioca,Cova de Mandioca,05,,-14.683333,-42.666667
br,cova donga,Cova Donga,20,,-6.583333,-40.75
br,covanca,Covanca,15,,-20.233333,-43.016667
br,covanca,Covanca,21,,-22.914716,-43.339308
br,covao,Covão,05,,-12.466667,-38.066667
br,covas,Covas,05,,-10.866667,-40.016667
br,covas,Covas,30,,-8.516667,-36.6
br,coxilha butia,Coxilha Butiá,23,,-30.6013,-52.2767
br,coxilha,Coxilha,23,,-28.116667,-52.283333
br,coxilha do meio,Coxilha do Meio,18,,-25.65,-50.25
br,coxilha grande,Coxilha Grande,23,,-28.233333,-50.983333
br,coxilhao das ameixeiras,Coxilhão das Ameixeiras,18,,-25.708293,-50.179952
br,coxilhao de cima,Coxilhão de Cima,18,,-25.652592,-50.23711
br,coxilhao de santa rosa,Coxilhão de Santa Rosa,18,,-25.643074,-50.254351
br,coxilhao do meio,Coxilhão do Meio,18,,-25.676943,-50.236377
br,coxilhao vitoriano,Coxilhão Vitoriano,18,,-25.668815,-50.24242
br,coxilha rica,Coxilha Rica,18,,-26,-52.833333
br,coxim,Coxim,11,30216,-18.5,-54.75
br,coxinhos,Coxinhos,18,,-25.4116,-50.595
br,coxipo-acu,Coxipó-Açu,14,,-15.35,-56.15
br,coxipo,Coxipo,14,,-15.633333,-56.066667
br,coxipo da ponte,Coxipó da Ponte,14,,-15.633333,-56.066667
br,coxipo do ouro,Coxipó do Ouro,14,,-15.45,-55.983333
br,coxipozinho,Coxipozinho,14,,-15.35,-56.15
br,coxixola,Coxixola,17,,-7.616667,-36.616667
br,coxo,Coxo,06,,-3.766667,-40.2
br,coxo,Coxo,20,,-5.7,-43.033333
br,coxos,Coxos,18,,-25.735616,-50.634748
br,craiba,Craíba,02,,-9.716667,-36.766667
br,crajau,Crajaú,21,,-22.921111,-43.262222
br,cramerim,Cramerim,08,,-20.05,-40.35
br,craolandia,Craolândia,31,,-8.106944,-47.012222
br,crasto,Crasto,28,,-11.416667,-37.416667
br,crateus,Crateús,06,52934,-5.166667,-40.666667
br,cratheus,Cratheús,06,,-5.166667,-40.666667
br,crato,Crato,04,,-7.483333,-63.033333
br,crato,Crato,06,88788,-7.233333,-39.383333
br,crauna,Craúna,20,,-5.983333,-42.516667
br,crauta,Crautá,30,,-8.15,-37.316667
br,cravaio,Cravaio,28,,-9.983333,-37.066667
br,cravari,Cravari,14,,-12.466667,-57.966667
br,cravinhos,Cravinhos,27,32690,-21.316667,-47.75
br,cravo,Cravo,04,,-5.273889,-59.953611
br,cravo,Cravo,16,,-1.783333,-47.983333
br,creca,Creca,22,,-5.966667,-37.328611
br,creciuna,Creciúna,26,,-28.666667,-49.383333
br,creolas,Creolas,14,,-15.116667,-57.6
br,creoulas,Creoulas,05,,-13.516667,-42.016667
br,crescencio,Crescêncio,11,,-23.266667,-55.533333
br,cresciuma,Cresciuma,18,,-25.433333,-54.4
br,cresciuma,Cresciúma,26,,-28.666667,-49.383333
br,crichas,Crichas,29,,-14.45,-49.966667
br,criciuma,Criciuma,08,,-20.216667,-41.616667
br,criciuma,Criciúma,18,,-24.74859,-48.941097
br,criciuma,Criciúma,26,161957,-28.666667,-49.383333
br,criciumal,Criciumal,23,,-27.5,-54.116667
br,crimea,Criméa,13,,-5.416667,-43.4
br,crimeia,Criméia,13,,-5.416667,-43.4
br,criminoso,Criminoso,13,,-3.981111,-42.94
br,crioli,Crioli,13,,-4.35,-43.35
br,crioli,Crioli,20,,-3.512778,-42.454722
br,crioli da sinha,Crioli da Sinhá,13,,-5.4,-44.366667
br,crioulas,Crioulas,05,,-13.516667,-42.016667
br,crioulas,Crioulas,06,,-6.966667,-38.95
br,crisanto,Crisanto,06,,-5.266667,-39.183333
br,crisciuma,Crisciúma,08,,-20.216667,-41.616667
br,crisciuma,Crisciuma,18,,-26.016667,-51.716667
br,crisolia,Crisólia,15,,-22.25,-46.416667
br,crisolita,Crisólita,15,,-17.25,-40.933333
br,crisopolis,Crisópolis,05,,-11.516667,-38.166667
br,crispim jaques,Crispim Jaques,15,,-17.833333,-41.133333
br,crissiumal,Crissiumal,23,,-27.5,-54.116667
br,cristai,Cristai,27,,-23.3,-46.816667
br,cristais,Cristais,05,,-14,-41.45
br,cristais,Cristais,06,,-4.5,-38.35
br,cristais,Cristais,15,,-20.866667,-45.516667
br,cristais,Cristais,27,,-20.4,-47.416667
br,cristalandia,Cristalândia,05,,-14,-41.45
br,cristalandia,Cristalândia,31,,-10.598611,-49.108333
br,cristal,Cristal,05,,-11.816667,-41.316667
br,cristal,Cristal,08,,-18.083333,-40.1
br,cristal,Cristal,11,,-21.5,-54.366667
br,cristal,Cristal,23,,-27.9,-54.95
br,cristal,Cristal,31,,-13.011389,-47.788333
br,cristalia,Cristália,15,,-16.716667,-42.866667
br,cristalia,Cristália,30,,-8.816667,-40.366667
br,cristalina,Cristalina,08,,-18.783333,-40.583333
br,cristalina,Cristalina,29,36532,-16.75,-47.6
br,cristalino,Cristalino,08,,-18.783333,-40.583333
br,cristalino,Cristalino,29,,-16.75,-47.6
br,cristiano guimaraes,Cristiano Guimarães,15,,-17.733333,-41.583333
br,cristiano otoni,Cristiano Otoni,15,,-20.833333,-43.8
br,cristianopolis,Cristianópolis,28,,-11.483333,-37.766667
br,cristianopolis,Cristianópolis,29,,-17.216667,-48.75
br,cristina,Cristina,15,,-22.216667,-45.266667
br,cristina,Cristina,28,,-11.483333,-37.766667
br,cristinapolis,Cristinápolis,28,6430,-11.483333,-37.766667
br,cristino castro,Cristino Castro,20,,-8.816667,-44.216667
br,cristino,Cristino,11,,-22.05,-52.75
br,cristino valadao,Cristino Valadão,15,,-20.366667,-45.383333
br,cristo rei,Cristo Rei,18,,-22.7,-52.466667
br,cristovao,Cristóvão,06,,-7.633333,-38.883333
br,cristovao,Cristóvão,15,,-16.233333,-40.583333
br,cristovao,Cristóvão,20,,-7.35,-41.816667
br,cristovao,Cristóvão,22,,-4.928333,-36.961944
br,cristovao,Cristóvão,27,,-23.5772,-47.5778
br,criuva,Criúva,23,,-28.983333,-50.966667
br,crixalandia,Crixalândia,29,,-15.3,-47.25
br,crixas,Crixás,29,11560,-14.45,-49.966667
br,crixas,Crixás,31,,-11.105833,-48.846111
br,croa,Croá,17,,-6.65,-38.533333
br,croas,Croás,30,,-7.85,-38.133333
br,croata(1),Croatá(1),06,,-4.033333,-38.5
br,croata,Croatá,06,,-3.65,-39.116667
br,croata,Croatá,13,,-4.133333,-44.133333
br,croata,Croatá,17,,-6.916667,-38.4
br,croata,Croata,20,,-6.883333,-42.45
br,croata,Croatá,22,,-6.166667,-37.7
br,croata,Croata,30,,-7.633333,-39.6
br,croata,Croatá,31,,-5.7,-48.033333
br,crobas,Crobas,14,,-14.9,-56.533333
br,crominia,Cromínia,29,,-17.283333,-49.35
br,cruangi,Cruangi,30,,-7.566667,-35.35
br,cruangy,Cruangy,30,,-7.566667,-35.35
br,cruanji,Cruanji,30,,-7.566667,-35.35
br,crubixa,Crubixá,08,,-20.65,-40.85
br,crubixais,Crubixais,21,,-22.233333,-42.066667
br,crubixe,Crubixê,08,,-20.65,-40.85
br,crucilandia,Crucilândia,15,,-20.383333,-44.35
br,crueira,Crueira,05,,-12.35,-37.966667
br,cruena,Cruena,05,,-12.35,-37.966667
br,crumatai,Crumataí,05,,-12.416667,-39.183333
br,crupiara,Crupiara,15,,-17.083333,-41.9
br,crussai,Crussaí,05,,-12.833333,-39.383333
br,cruxati,Cruxati,06,,-3.266667,-39.633333
br,cruxeiro,Cruxeiro,06,,-4.333333,-38.116667
br,cruzalia,Cruzália,27,,-22.75,-50.8
br,cruz alta,Cruz Alta,11,,-19.066667,-55.566667
br,cruz alta,Cruz Alta,23,65276,-28.65,-53.6
br,cruz alta,Cruz Alta,27,,-22.75,-50.8
br,cruz altinha,Cruz Altinha,23,,-28.283333,-51.95
br,cruzamento,Cruzamento,22,,-5.283333,-35.75
br,cruz,Cruz,05,,-12.6,-38.483333
br,cruz,Cruz,06,8723,-2.95,-40.183333
br,cruz,Cruz,13,,-3.696667,-42.976944
br,cruz,Cruz,15,,-20.6,-45.933333
br,cruz,Cruz,20,,-10.35,-45.066667
br,cruz,Cruz,22,,-6.183333,-38.516667
br,cruz,Cruz,30,,-8.9,-36.583333
br,cruz da esperanca,Cruz da Esperança,27,,-21.3,-47.416667
br,cruz das almas,Cruz das Almas,05,47846,-12.666667,-39.1
br,cruz das almas,Cruz das Almas,20,,-6.1,-42.8
br,cruz das almas,Cruz das Almas,21,,-21.783333,-41.3
br,cruz das almas,Cruz das Almas,27,,-22.733333,-46.85
br,cruz das posses,Cruz das Posses,27,,-21.083333,-47.9
br,cruz da teresa,Cruz da Teresa,17,,-7.016667,-37.916667
br,cruz de almas,Cruz de Almas,22,,-6.066667,-37.766667
br,cruz de malta,Cruz de Malta,30,,-8.25,-40.333333
br,cruz de palhano,Cruz de Palhano,06,,-4.733333,-37.95
br,cruz descoberta,Cruz Descoberta,27,,-22.75,-46.8
br,cruz do amapa,Cruz do Amapá,03,,2.183333,-50.8
br,cruz do bento joao,Cruz do Bento João,15,,-22.316667,-45.816667
br,cruz do espirito santo,Cruz do Espírito Santo,17,5812,-7.133333,-35.1
br,cruz do meio,Cruz do Meio,02,,-9.5,-36.833333
br,cruz do palhano,Cruz do Palhano,06,,-4.733333,-37.95
br,cruz dos anjos,Cruz dos Anjos,05,,-12.933333,-39.366667
br,cruzeirinho,Cruzeirinho,06,,-6.283333,-38.733333
br,cruzeiro bondao,Cruzeiro Bondão,05,,-12.016667,-38.966667
br,cruzeiro bordao,Cruzeiro Bordão,05,,-12.016667,-38.966667
br,cruzeiro,Cruzeiro,04,,-5.24,-60.55
br,cruzeiro,Cruzeiro,06,,-4.333333,-38.116667
br,cruzeiro,Cruzeiro,15,,-18.833333,-42.916667
br,cruzeiro,Cruzeiro,18,,-23.438879,-49.861796
br,cruzeiro,Cruzeiro,20,,-3.940278,-42.426111
br,cruzeiro,Cruzeiro,21,,-21.433333,-41.9
br,cruzeiro,Cruzeiro,23,,-27.866667,-54.433333
br,cruzeiro,Cruzeiro,25,,3.45,-61.2
br,cruzeiro,Cruzeiro,26,,-27.166667,-51.5
br,cruzeiro,Cruzeiro,27,72023,-22.566667,-44.966667
br,cruzeiro da fortaleza,Cruzeiro da Fortaleza,15,,-18.966667,-46.666667
br,cruzeiro da paz,Cruzeiro da Paz,05,,-11.9,-38.2
br,cruzeiro das araras,Cruzeiro das Araras,04,,-5.24,-60.55
br,cruzeiro da tapera d'agua limpa,Cruzeiro da Tapera d'Agua Limpa,11,,-19.35,-54.7
br,cruzeiro do norte,Cruzeiro do Norte,06,,-5.516667,-39.15
br,cruzeiro do norte,Cruzeiro do Norte,18,,-23.233333,-50.85
br,cruzeiro do oeste,Cruzeiro do Oeste,18,15058,-23.766667,-53.066667
br,cruzeiro dos peixotos,Cruzeiro dos Peixotos,15,,-18.7,-48.366667
br,cruzeiro do sul,Cruzeiro do Sul,01,45517,-7.633333,-72.6
br,cruzeiro do sul,Cruzeiro do Sul,05,,-14.1,-39.083333
br,cruzeiro do sul,Cruzeiro do Sul,18,,-22.933333,-52.15
br,cruzeiro do sul,Cruzeiro do Sul,23,,-29.516667,-51.983333
br,cruzeiro do sul,Cruzeiro do Sul,26,,-27.166667,-51.5
br,cruzeiro do sul,Cruzeiro do Sul,30,,-8.583333,-36.3
br,cruzeiros dos peixotos,Cruzeiros dos Peixotos,15,,-18.7,-48.366667
br,cruzes,Cruzes,02,,-9.65,-36.483333
br,cruzes,Cruzes,27,,-21.533333,-50.533333
br,cruzes,Cruzes,30,,-8.65,-36.116667
br,cruzeta,Cruzeta,06,,-6.566667,-39.866667
br,cruzeta,Cruzeta,22,,-6.416667,-36.783333
br,cruz i,Cruz I,13,,-3.454722,-43.102778
br,cruz ii,Cruz II,13,,-3.450833,-43.088611
br,cruzilandia,Cruzilândia,13,,-3.920833,-43.376389
br,cruzilha,Cruzilha,15,,-21.833333,-44.8
br,cruzilia,Cruzília,15,12816,-21.833333,-44.8
br,cruzinha,Cruzinha,20,,-5.466667,-42.766667
br,cruzinha,Cruzinha,22,,-5.937778,-36.148056
br,cruzinha,Cruzinha,23,,-28.266667,-52.95
br,cruz machado,Cruz Machado,18,,-26.016667,-51.35
br,cruz muchado,Cruz Muchado,18,,-26.016667,-51.35
br,cruz velha,Cruz Velha,11,,-20.916667,-57.166667
br,cruz vera,Cruz Vera,15,,-22.5037,-45.6866
br,crystaes,Crystaes,15,,-20.866667,-45.516667
br,crystaes,Crystaes,27,,-20.4,-47.416667
br,crystallina,Crystallina,15,,-16.716667,-42.866667
br,crystallina,Crystallina,29,,-16.75,-47.6
br,cta. alegre,Cta. Alegre,20,,-5.033333,-42.216667
br,cuapiacu,Cuapiaçu,27,,-20.833333,-49.216667
br,cubas,Cubas,15,,-18.583333,-43.666667
br,cubatao,Cubatão,18,,-25.826451,-48.77469
br,cubatao,Cubatão,26,,-26.216667,-48.833333
br,cubatao,Cubatão,27,118412,-23.883333,-46.416667
br,cubataozinho,Cubatáozinho,18,,-25.7238,-48.7616
br,cubati,Cubati,17,,-6.85,-36.35
br,cubiculo,Cubículo,26,,-28.566667,-49.1
br,cucaia,Cucaia,16,,-1.4,-48.116667
br,cucau,Cucaú,30,,-8.65,-35.266667
br,cucui,Cucuí,04,,1.178333,-66.841111
br,cucuy,Cucuy,04,,1.178333,-66.841111
br,cue,Cuê,11,,-22.9,-55.316667
br,cueira,Cueira,05,,-13.6,-38.883333
br,cuiaba,Cuiabá,04,,-2.533333,-66.383333
br,cuiaba,Cuiabá,14,521961,-15.583333,-56.083333
br,cuiaba,Cuiabá,15,,-18.416667,-43.816667
br,cuiaba,Cuiabá,27,,-22.3,-52.1
br,cuiaba,Cuiabá,28,,-9.7,-37.816667
br,cuiaba da larga,Cuiabá da Larga,14,,-14.35,-55.35
br,cuiaba de larga,Cuiabá de Larga,14,,-14.35,-55.35
br,cuiaba paulista,Cuiabá Paulista,27,,-22.3,-52.1
br,cuiabazinho,Cuiabazinho,14,,-14.383333,-55.616667
br,cuiari,Cuiari,04,,1.5,-68.183333
br,cuiary,Cuiary,04,,.216667,-67.266667
br,cuinarana,Cuinarana,16,,-.783333,-47.583333
br,cuipeua,Cuipeua,16,,-1.883333,-54.883333
br,cuipiranga,Cuipiranga,06,,-4.809722,-37.273056
br,cuite,Cuité,05,,-11.55,-39.266667
br,cuite,Cuité,15,,-19.083333,-41.55
br,cuite,Cuité,17,11392,-6.483333,-36.15
br,cuite,Cuité,20,,-4.3,-42.75
br,cuite,Cuité,22,,-6.416667,-35.3
br,cuitegi,Cuitegi,17,,-6.9,-35.533333
br,cuiteji,Cuiteji,17,,-6.9,-35.533333
br,cuitelos,Cuitelos,27,,-22.116667,-47.1
br,cujubim,Cujubim,04,,-5.777222,-60.946944
br,cujubim,Cujubim,24,,-8.966667,-64.05
br,cujubina,Cujubina,24,,-9.45,-62.633333
br,cujuhim,Cujuhim,04,,-1.686389,-69.163611
br,cujuim,Cujuim,04,,-1.686389,-69.163611
br,culuim,Culuim,04,,-1.686389,-69.163611
br,culuvana,Culuvana,16,,-4.1,-55.833333
br,cuma,Cumã,13,,-2.116667,-44.6
br,cumanda,Cumandá,25,,.836667,-61.661667
br,cumari,Cumari,29,,-18.266667,-48.183333
br,cumaru,Cumaru,04,,-1.116667,-62.733333
br,cumaru,Cumaru,16,,-1.716667,-47.816667
br,cumaru,Cumaru,25,,2.9,-60.9
br,cumaru,Cumaru,30,6861,-8,-35.7
br,cumaruma,Cumarumã,03,,-3.416667,-50.3
br,cumary,Cumary,29,,-18.266667,-48.183333
br,cumati,Cumati,17,,-6.816667,-36.366667
br,cumbaru,Cumbaru,14,,-15.516667,-56.933333
br,cumbauba,Cumbaúba,20,,-8.8,-45.383333
br,cumbaubas,Cumbaúbas,20,,-8.8,-45.383333
br,cumbe,Cumbe,05,,-10.516667,-39.016667
br,cumbe,Cumbe,06,,-4.483333,-37.75
br,cumbe,Cumbe,13,,-3.878889,-43.396389
br,cumbe,Cumbe,20,,-5.516667,-41.466667
br,cumbe,Cumbe,22,,-6.05,-37.933333
br,cumbe,Cumbe,28,,-10.35,-37.233333
br,cumbe,Cumbe,30,,-7.633333,-37.733333
br,cumbica,Cumbica,27,,-23.466667,-46.466667
br,cumbre,Cumbre,13,,-3.867222,-43.285278
br,cumiranga,Cumiranga,16,,-1.8,-46.283333
br,cumuruxatiba,Cumuruxatiba,05,,-17.1,-39.183333
br,cunani,Cunani,03,,2.866667,-51.1
br,cunany,Cunany,03,,2.866667,-51.1
br,cuncas,Cuncas,06,,-7.1,-38.733333
br,cunha,Cunha,15,,-20.333333,-42.966667
br,cunha,Cunha,26,,-26.9,-53.15
br,cunha,Cunha,27,11896,-23.083333,-44.966667
br,cunha,Cunha,30,,-8.116667,-36.533333
br,cunhambebe,Cunhambebe,21,,-23,-44.333333
br,cunhamporanga,Cunhamporanga,18,,-24.716667,-49.8
br,cunhangi,Cunhangi,05,,-13.133333,-39.15
br,cunhanji,Cunhanji,05,,-13.133333,-39.15
br,cunha pora,Cunha Porã,26,,-26.9,-53.15
br,cunhaporanga,Cunhaporanga,18,,-24.716667,-49.8
br,cunhas,Cunhas,15,,-20.333333,-42.966667
br,cunhupuca,Cunhupuca,04,,-9.083333,-68.116667
br,cunuman,Cunuman,04,,-6.133333,-60.166667
br,cuo,Cuó,22,,-5.571389,-36.873889
br,cupaoba,Cupaoba,17,,-6.683333,-35.433333
br,cuparaque,Cuparaque,15,,-18.966667,-41.1
br,cupatana,Cupatana,04,,-2.8,-67.066667
br,cupe,Cupê,02,,-9.4,-35.5
br,cupe,Cupe,30,,-8.466667,-34.983333
br,cupia,Cupia,04,,-4.910556,-62.632222
br,cupicara,Cupiçara,17,,-7.483333,-34.95
br,cupido,Cupido,08,,-19.066667,-40.133333
br,cupido,Cupido,17,,-7.416667,-37.6
br,cupijo,Cupijó,16,,-2.166667,-49.616667
br,cupim,Cupim,04,,-4.568056,-59.862778
br,cupim,Cupim,14,,-16.266667,-56.833333
br,cupim,Cupim,21,,-21.8,-41.4
br,cupim,Cupim,28,,-9.866667,-37.616667
br,cupim grande,Cupim Grande,13,,-6.416667,-43.116667
br,cupins,Cupins,05,,-12.7,-44.683333
br,cupins,Cupins,11,,-19.85,-51.05
br,cupins,Cupins,20,,-4.4,-42.6
br,cupira,Cupira,02,,-9.483333,-37.283333
br,cupira,Cupira,30,19096,-8.616667,-35.95
br,cupiras de cima,Cupiras de Cima,17,,-6.966667,-37.483333
br,cupissara,Cupissara,17,,-7.483333,-34.95
br,cupixi,Cupixi,03,,.633333,-51.783333
br,cupuba,Cupuba,05,,-17.966667,-39.566667
br,cupyra,Cupyra,30,,-8.616667,-35.95
br,curaca,Curaçá,05,11858,-8.983333,-39.9
br,cura,Cura,13,,-3.0725,-42.220556
br,curador,Curador,02,,-5.25,-44.5
br,curaja,Curajá,16,,-.783333,-47.95
br,curapaity,Curapaity,04,,-5.609722,-63.328611
br,curaru,Curaru,04,,-4.187778,-60.641111
br,curati,Curati,25,,2.516667,-60.033333
br,curatis,Curatis,06,,-4.983333,-40.266667
br,curato do santa catarina,Curato do Santa Catarina,21,,-22.116667,-41.8
br,cure,Curé,11,,-21.416667,-56.416667
br,curema,Curema,17,,-7.016667,-37.966667
br,curiango,Curiango,15,,-16.166667,-40.65
br,curiau,Curiaú,03,,.133333,-51.066667
br,curicaca,Curicaca,30,,-7.45,-39.816667
br,curicaca,Curicaca,31,,-7.883333,-48.15
br,curicha,Curicha,16,,1.933333,-54.916667
br,curicica,Curicica,21,,-22.953823,-43.38384
br,curima,Curima,16,,-2.183333,-49.466667
br,curima,Curimã,22,,-5.866667,-35.766667
br,curima,Curimã,30,,-8.85,-35.333333
br,curimata,Curimatá,04,,-4.816667,-66.766667
br,curimata,Curimata,06,,-4.183333,-38.533333
br,curimata,Curimatá,20,,-10.033333,-44.283333
br,curimatai,Curimataí,15,,-17.85,-43.983333
br,curimatau,Curimataú,17,,-6.716667,-35.5
br,curinhaem,Curinhãem,17,,-7.133333,-35.45
br,curio,Curió,06,,-5.533333,-39.833333
br,curipare,Curiparé,05,,-11.522222,-45.055556
br,curipi,Curipi,03,,3.716667,-51.683333
br,curitbanos,Curitbanos,27,,-22.883333,-46.55
br,curite,Curite,16,,-4.1,-55.666667
br,curitiba,Curitiba,18,1718433,-25.416667,-49.25
br,curitibanos,Curitibanos,26,32142,-27.3,-50.6
br,curituba,Curituba,28,,-9.65,-37.8
br,curituba velha,Curituba Velha,28,,-9.65,-37.933333
br,curityba,Curityba,18,,-25.416667,-49.25
br,curitybanos,Curitybanos,26,,-27.3,-50.6
br,curiuva,Curiúva,18,,-24.033333,-50.45
br,curraes novos,Curraes Novos,22,,-6.25,-36.516667
br,currahnho,Currahnho,06,,-4.133333,-38.366667
br,currais barauna,Currais Baraúna,22,,-5.083333,-37.616667
br,currais,Currais,05,,-9.566667,-43.233333
br,currais,Currais,13,,-3.383056,-42.503611
br,currais,Currais,15,,-16.066667,-40.816667
br,currais,Currais,17,,-7.083333,-38.616667
br,currais,Currais,20,,-8.966667,-44.416667
br,currais novo,Currais Novo,22,,-5.971667,-36.788056
br,currais novos,Currais Novos,22,35561,-6.25,-36.516667
br,currais velhos,Currais Velhos,06,,-2.85,-40.083333
br,curral alto,Curral Alto,23,,-32.95,-52.766667
br,curral,Curral,05,,-13.383333,-38.966667
br,curral,Curral,17,,-6.833333,-38.2
br,curral,Curral,20,,-6.816667,-40.516667
br,curral da igreja,Curral da Igreja,13,,-3.383333,-44.8
br,curral das minas,Curral das Minas,26,,-26.9906,-49.0145
br,curral das pedras,Curral das Pedras,05,,-12.366667,-39.533333
br,curral das pedras,Curral das Pedras,28,,-9.966667,-37.083333
br,curral da vaizea,Curral da Vaizea,06,,-6.55,-39.116667
br,curral de arame,Curral de Arame,11,,-22.35,-54.966667
br,curral de baixo,Curral de Baixo,22,,-5.322222,-37.2875
br,curral de dentro,Curral de Dentro,15,,-15.95,-41.883333
br,curral de pedra,Curral de Pedra,17,,-6.716667,-37.816667
br,curral de pedra,Curral de Pedra,23,,-30.416667,-54.85
br,curral de pedra,Curral de Pedra,27,,-20.216667,-47.566667
br,curral de pedras,Curral de Pedras,05,,-13.616667,-44.516667
br,curral de pedras,Curral de Pedras,20,,-6.116667,-40.9
br,curral de pedro,Curral de Pedro,23,,-30.416667,-54.85
br,curral de vacas,Curral de Vacas,31,,-7.416667,-48.266667
br,curral de varas,Curral de Varas,15,,-15.65,-41.733333
br,curral do gato,Curral do Gato,05,,-10.883333,-38.966667
br,curral do meio,Curral do Meio,02,,-9.533333,-37.7
br,curral do meio,Curral do Meio,05,,-12.6,-39.616667
br,curral do meio,Curral do Meio,13,,-3.108611,-42.11
br,curral do mel,Curral do Mel,05,,-12.6,-39.616667
br,curral do pedra,Curral do Pedra,23,,-30.416667,-54.85
br,curral dos bois,Curral dos Bois,05,,-9.183333,-38.3
br,curral dos bois,Curral dos Bois,28,,-10.516667,-36.916667
br,curral dos bois,Curral dos Bois,30,,-8.95,-38.15
br,curral falso,Curral Falso,05,,-12.416667,-39.233333
br,curral falso,Curral Falso,21,,-22.935509,-43.667128
br,curral feio,Curral Feio,05,,-10.7,-38.566667
br,curral grande,Curral Grande,06,,-3.533333,-39.166667
br,curral grande,Curral Grande,20,,-3.815833,-42.34
br,curralinha,Curralinha,05,,-11.15,-43.166667
br,curralinha,Curralinha,20,,-3.803333,-42.396667
br,curralinho,Curralinho,02,,-9.266667,-37.783333
br,curralinho,Curralinho,04,,-3.866667,-59.216667
br,curralinho,Curralinho,05,,-10.866667,-41.633333
br,curralinho,Curralinho,06,,-3.233333,-39.383333
br,curralinho,Curralinho,11,,-21.383333,-54.766667
br,curralinho,Curralinho,13,,-3.342222,-42.178611
br,curralinho,Curralinho,15,,-21.516667,-43.866667
br,curralinho,Curralinho,16,,-1.8,-49.783333
br,curralinho,Curralinho,17,,-6.266667,-37.6
br,curralinho,Curralinho,18,,-24.433333,-50.05
br,curralinho,Curralinho,20,,-3.473889,-42.1825
br,curralinho,Curralinho,22,,-5.856389,-37.109167
br,curralinho,Curralinho,28,,-10.75,-36.833333
br,curralinho,Curralinho,30,,-8.65,-38.6
br,curralinho de baixo,Curralinho de Baixo,05,,-13.933333,-41.516667
br,curralinho de cima,Curralinho de Cima,05,,-13.95,-41.533333
br,curralinho do meio,Curralinho do Meio,05,,-13.95,-41.516667
br,curralinho dos paula,Curralinho dos Paula,15,,-20.8,-44.25
br,curralinho dos paulas,Curralinho dos Paulas,15,,-20.8,-44.25
br,curralito,Curralito,11,,-23.183333,-55.383333
br,curral moreira,Curral Moreira,15,,-20.483333,-44.166667
br,curral novo,Curral Novo,05,,-10.4,-43.166667
br,curral novo,Curral Nôvo,06,,-6.033333,-38.833333
br,curral novo,Curral Nôvo,20,,-7.816667,-40.9
br,curral novo,Curral Novo,22,,-5.971667,-36.788056
br,curral novo,Curral Novo,23,,-32.95,-52.766667
br,curral preto,Curral Prêto,22,,-5.433333,-35.633333
br,curral queimado,Curral Queimado,20,,-5.233333,-42.483333
br,curral queimado,Curral Queimado,22,,-6.766667,-37.2
br,curral queimado,Curral Queimado,30,,-8.733333,-36.633333
br,curral taquara,Curral Taquara,11,,-20.433333,-56.3
br,curral varo santana,Curral Varo Santana,13,,-6.066667,-43.183333
br,curral velho,Curral Velho,05,,-10.5,-43.5
br,curral velho,Curral Velho,06,,-5.05,-40.7
br,curral velho,Curral Velho,13,,-3.567778,-42.605833
br,curral velho,Curral Velho,16,,-1.850833,-56.243611
br,curral velho,Curral Velho,17,,-7.533333,-38.2
br,curral velho,Curral Velho,20,,-3.948056,-42.081389
br,curral velho,Curral Velho,30,,-7.683333,-37.683333
br,curral velho dos ramos,Curral Velho dos Ramos,30,,-7.716667,-37.683333
br,curralzinho,Curralzinho,13,,-4.25,-43
br,curriola,Curriola,18,,-25.083333,-49.4
br,curro velho,Curro Velho,16,,-1.430556,-48.488889
br,currupira,Currupira,21,,-23.031682,-43.497123
br,curtume,Curtume,06,,-7.283333,-38.616667
br,curtume,Curtume,17,,-7.166667,-37.666667
br,curtume,Curtume,20,,-5.133333,-41.166667
br,curua,Curuá,16,,-1.85,-55.15
br,curuai,Curuaí,16,,-2.283333,-55.483333
br,curuay,Curuay,16,,-2.283333,-55.483333
br,curuca,Curuçá,04,,-3.483333,-57.966667
br,curuca,Curuçá,16,10843,-.716667,-47.833333
br,curucambaba,Curuçambaba,16,,-2.133333,-49.3
br,curucambabiaca,Curuçambabiaca,16,,-1.616667,-48.566667
br,curucamba,Curuçambá,16,,-1.616667,-48.566667
br,curucazinho,Curuçàzinho,16,,-.633333,-47.466667
br,curuci,Curuci,16,,-4.35,-57.033333
br,curucurana,Curucurana,26,,-27.1258,-49.7137
br,curu,Curú,06,,-3.666667,-39.233333
br,curucutu,Curucutu,27,,-23.8892,-46.6125
br,curumatahy,Curumatahy,15,,-17.85,-43.983333
br,curumu,Curumu,16,,-1.016667,-51.05
br,curundundum,Curundundum,05,,-9.733333,-39.266667
br,curunhu,Curunhu,06,,-4.116667,-40.866667
br,curupa,Curupá,13,,-9.9,-45.9
br,curupai,Curupai,11,,-22.833333,-53.883333
br,curupaiti,Curupaiti,04,,-3.4,-68.866667
br,curupari,Curupari,16,,-1.616667,-52.7
br,curupasse,Curupasse,20,,-5.316667,-41.15
br,curupati,Curupati,04,,-5.609722,-63.328611
br,curupati,Curupati,17,,-7.833333,-36.966667
br,curupeba,Curupeba,05,,-12.716667,-38.6
br,curupira,Curupira,04,,-.2,-63.2
br,curupira,Curupira,06,,-4.516667,-38.533333
br,curupira,Curupira,14,,-15.116667,-56.8
br,cururipe,Cururipe,02,,-10.133333,-36.166667
br,cururu,Cururú,04,,1.233333,-68.983333
br,cururu,Cururu,13,,-4.766667,-46.583333
br,cururu,Cururu,16,,-7.2,-58.066667
br,cururu,Cururu,22,,-6.133333,-35.1
br,cururupe,Cururupe,05,,-14.85,-39.033333
br,cururu poco,Cururu Poço,04,,1.233333,-68.983333
br,cururupu,Cururupu,13,22007,-1.833333,-44.866667
br,curutari,Curutari,04,,-3.921111,-62.930833
br,curuzu,Curuzu,04,,-6.1,-64.283333
br,curuzu,Curuzu,13,,-3.333333,-43.583333
br,curva,Curva,13,,-1.366667,-45.933333
br,curva grande,Curva Grande,13,,-2.616667,-45.45
br,curvello,Curvello,15,,-18.75,-44.416667
br,curvelo,Curvelo,11,,-21.183333,-56.733333
br,curvelo,Curvelo,15,63955,-18.75,-44.416667
br,curytiba,Curytiba,18,,-25.416667,-49.25
br,custodia,Custodia,06,,-4.983333,-39.183333
br,custodia,Custódia,30,18107,-8.116667,-37.65
br,custodia freire,Custódia Freire,01,,-9.917222,-67.8875
br,custodio,Custódio,05,,-17.7,-40.25
br,custodio,Custódio,06,,-4.983333,-39.183333
br,custodio,Custódio,13,,-3.200833,-43.329444
br,custodio,Custódio,14,,-16.316667,-55.966667
br,custodio,Custódio,15,,-17.15,-41.8
br,custodio lima,Custódio Lima,13,,-3.25,-42.066667
br,cutade,Cutade,11,,-20.133333,-56.083333
br,cutape,Cutape,11,,-20.133333,-56.083333
br,cutia,Cutia,04,,-1.609722,-61.401111
br,cutia,Cutia,06,,-3.2,-40.283333
br,cutia,Cutia,13,,-1.883333,-45.45
br,cutia,Cutia,20,,-2.790833,-41.8
br,cutia,Cutia,26,,-27.1901,-49.5258
br,cutia,Cutia,27,,-23.616667,-46.933333
br,cutia fria,Cutia Fria,13,,-2.533333,-45.35
br,cutias,Cutias,13,,-3.166667,-45.166667
br,cutiu,Cutiú,16,,-1.416667,-46.166667
br,cuyaba,Cuyabá,14,,-15.583333,-56.083333
br,cuyaba,Cuyabá,15,,-19.85,-43.733333
br,cuyaba de larga,Cuyaba de Larga,14,,-14.35,-55.35
br,cuyaba larga,Cuyabá Larga,14,,-14.35,-55.35
br,cuyabasinho,Cuyabasinho,14,,-14.383333,-55.616667
br,cuyari,Cuyarí,04,,1.5,-68.183333
br,dacabi,Dacabi,25,,3.533333,-60.383333
br,dairel,Dairel,15,,-18.4,-45.333333
br,daiva,Daiva,04,,-7.343333,-71.022222
br,dalbergia,Dalbérgia,26,,-27.0324,-49.5906
br,daltro filho,Daltro Filho,23,,-29.333333,-51.766667
br,damasio,Damásio,13,,-2.916667,-42.383333
br,damasio sapa,Damásio Sapa,15,,-18.2,-43.716667
br,damianopolis,Damianópolis,29,,-14.55,-46.166667
br,damolandia,Damolândia,29,,-16.266667,-49.366667
br,dandanho,Dandanho,25,,2.666667,-60.95
br,danguer,Danguer,13,,-6.133333,-43.766667
br,daniel,Daniel,03,,.4,-51.266667
br,daniel de queiroz,Daniel de Queiroz,06,,-4.816667,-38.983333
br,dansia,Dansia,16,,1.85,-56.216667
br,dantas,Dantas,18,,-24.1,-50.283333
br,dao dos ferros,Dao Dos Ferros,22,,-6.116667,-38.166667
br,darabi,Darabi,04,,.766667,-66.883333
br,darcilandia,Darcilândia,27,,-21.433333,-51.65
br,darcilena,Darcilena,28,,-10.25,-36.883333
br,darcynopolis,Darcynópolis,31,,-6.704444,-47.588889
br,dario,Dário,04,,-2.833333,-68.7
br,dario,Dário,08,,-19.266667,-40.5
br,das flores,Das Flôres,15,,-22.079167,-43.556667
br,data,Data,05,,-12.7,-42.216667
br,datas,Datas,15,,-18.416667,-43.666667
br,dattas,Dattas,15,,-18.416667,-43.666667
br,davi caldas,Davi Caldas,20,,-4.75,-42.933333
br,davinopolis,Davinópolis,29,,-15.966667,-50.133333
br,d. cruz,D. Cruz,23,,-27.416667,-53.416667
br,deca martins,Deca Martins,23,,-31.166667,-55.166667
br,decana,Deçana,04,,.05,-67.333333
br,decarle,Decarle,27,,-22.708333,-47.326944
br,dedo grosso,Dedo Grosso,26,,-26.366667,-48.95
br,defunto,Defunto,06,,-7.066667,-40.233333
br,defuntos,Defuntos,22,,-6.166667,-35.266667
br,degredo,Degrêdo,13,,-4.266667,-43.166667
br,deixe-comigo,Deixe-Comigo,05,,-12.716667,-39.166667
br,del castilho,Del Castilho,21,,-22.882215,-43.272891
br,deleite,Deleite,06,,-5.916667,-39.15
br,delfim moreira,Delfim Moreira,15,,-22.5,-45.283333
br,delfina,Delfina,16,,-1.183333,-56.05
br,delfino,Delfino,05,,-10.45,-41.2
br,delfino,Delfino,23,,-30.6633,-52.4773
br,delfinopolis,Delfinópolis,15,,-20.333333,-46.85
br,delgado,Delgado,29,,-14.5,-48.733333
br,deliberador,Deliberador,18,,-23.1,-51.116667
br,delicia,Delícia,04,,-5.579167,-60.955556
br,deliciosa,Deliciosa,20,,-5.5,-41.333333
br,del lago residencias,Del Lago Residencias,21,,-22.990745,-43.380515
br,delmiro,Delmiro,02,,-9.383333,-37.983333
br,delmiro gouveia,Delmiro Gouveia,02,35260,-9.383333,-37.983333
br,delvino,Delvino,15,,-16.516667,-41.4
br,demanda,Demanda,25,,3.833333,-61.4
br,demarcacao,Demarcação,30,,-8.45,-35.4
br,demerval lobao,Demerval Lobão,20,15285,-5.366667,-42.683333
br,demetrio lemos,Demétrio Lemos,22,,-6.216667,-37.9
br,demetrio ribeiro,Demétrio Ribeiro,21,,-22.395,-43.734722
br,demetrio ribeiro,Demétrio Ribeiro,23,,-29.566667,-54.55
br,democracia,Democracia,04,,-5.801389,-61.434722
br,dende,Dendê,05,,-12.9,-38.75
br,dende,Dendê,21,,-22.800668,-43.185845
br,dende grande,Dendê Grande,05,,-12.95,-39.366667
br,dendes,Dendês,05,,-12.9,-38.75
br,dendezeiro,Dendezeiro,05,,-12.4,-38.05
br,dendezeiros,Dendezeiros,05,,-12.4,-38.05
br,dentro,Dentro,22,,-6.1,-37.916667
br,deocleciano faria,Deocleciano Faria,15,,-16.133333,-41.266667
br,deodoro da fonseca,Deodoro da Fonseca,22,,-5.45,-36.85
br,deodoro,Deodoro,21,,-22.844483,-43.389646
br,deodoro,Deodoro,23,,-29.466667,-52.266667
br,deposito,Depósito,01,,-8.366667,-73.316667
br,deposito,Depósito,25,,3.2,-60.583333
br,dermeval lobao,Dermeval Lobão,20,,-5.366667,-42.683333
br,derradeira varzea,Derradeira Várzea,30,,-8.133333,-38.65
br,derramado,Derramado,11,,-18.55,-56.683333
br,derribadinha,Derribadinha,15,,-18.85,-41.833333
br,derrubada,Derrubada,16,,-.733333,-47.416667
br,derrubadas,Derrubadas,23,,-27.25,-53.866667
br,derrubadinha,Derrubadinha,15,,-17.85,-40.733333
br,desagua,Deságua,05,,-12.033333,-40.766667
br,desamparada,Desamparada,05,,-14.616667,-41.783333
br,desapregados,Desapregados,06,,-7.316667,-40.15
br,descalvado,Descalvado,14,,-16.75,-57.7
br,descalvado,Descalvado,18,,-24.85,-50.533333
br,descalvado,Descalvado,27,26138,-21.9,-47.616667
br,descalvados,Descalvados,14,,-16.75,-57.7
br,descampadinho,Descampadinho,18,,-24.804947,-48.916128
br,descampado,Descampado,18,,-24.715815,-48.946491
br,descanco,Descanço,11,,-18.383333,-54.966667
br,descanco,Descanço,13,,-6.1,-43.05
br,descanso da vida,Descanso da Vida,27,,-24.659183,-48.247192
br,descanso,Descanso,11,,-18.383333,-54.966667
br,descanso,Descanso,13,,-6.1,-43.05
br,descanso,Descanso,20,,-6.266667,-42.8
br,descanso,Descanso,26,,-26.833333,-53.583333
br,descanso,Descanso,30,,-7.833333,-39.933333
br,descarocador,Descaroçador,15,,-21.166667,-45.816667
br,descarreiro,Descarreiro,31,,-5.65,-47.466667
br,descarreto,Descarreto,31,,-5.65,-47.466667
br,descida da boiada,Descida da Boiada,20,,-7.633333,-41.866667
br,descoberta,Descoberta,05,,-12.183333,-43.416667
br,descoberta,Descoberta,16,,.45,-53.283333
br,descoberta,Descoberta,20,,-5.9,-42.116667
br,descoberta,Descoberta,22,,-5.8,-35.516667
br,descoberto,Descoberto,05,,-13.983333,-44.3
br,descoberto,Descoberto,15,,-21.45,-42.966667
br,descoberto,Descoberto,18,,-25.907,-48.7116
br,descoberto,Descoberto,29,,-13.433333,-49.166667
br,desegnano,Desegnano,04,,-6.7,-70.416667
br,desejo,Desejo,21,,-21.45,-41.316667
br,desembargador otoni,Desembargador Otoni,15,,-17.716667,-43.3
br,desemboque,Desemboque,15,,-20.016667,-47.05
br,desempenho,Desempenho,05,,-12.566667,-38.65
br,desengano,Desengano,04,,-6.7,-70.416667
br,desengano,Desengano,08,,-19.616667,-40.283333
br,desengano,Desengano,14,,-13.4,-57.483333
br,desengano,Desengano,18,,-24.95,-52.783333
br,desengano,Desengano,21,,-22.343889,-43.682778
br,deserto,Deserto,04,,-7.083333,-58.833333
br,deserto,Deserto,06,,-3.5,-39.483333
br,deserto,Deserto,13,,-2.516667,-44.8
br,deserto,Deserto,16,,-3.366667,-51.8
br,deserto,Deserto,20,,-3.65,-41.6
br,deserto i,Deserto I,13,,-3.472778,-43.475833
br,deserto ii,Deserto II,13,,-3.081667,-43.428611
br,designio,Desígnio,20,,-4.333333,-42.966667
br,despenhado,Despenhado,15,,-20.116667,-46.233333
br,destacamenta das pedras,Destacamenta das Pedras,24,,-12.85,-62.9
br,destacamento da corixa,Destacamento da Corixa,14,,-16.333333,-58.316667
br,destampina,Destampina,05,,-14.283333,-39.85
br,destero,Destero,26,,-27.583333,-48.566667
br,desterro de entre rios,Destêrro de Entre Rios,15,,-20.666667,-44.333333
br,desterro,Destêrro,05,,-12.166667,-38.15
br,desterro,Desterro,06,,-2.816667,-40.4
br,desterro,Destêrro,17,5374,-7.283333,-37.1
br,desterro,Destêrro,22,,-5.862222,-36.847222
br,desterro,Desterro,26,,-27.583333,-48.566667
br,desterro do mello,Desterro do Mello,15,,-21.15,-43.516667
br,desterro do melo,Destêrro do Melo,15,,-21.15,-43.516667
br,destino,Destino,13,,-3.904722,-43.48
br,desvio erval,Desvio Erval,23,,-31.883333,-53.166667
br,desvio machado,Desvio Machado,23,,-29.266667,-51.466667
br,desvio ribas,Desvio Ribas,18,,-25.2,-50.066667
br,desvio rizzo,Desvio Rizzo,23,,-29.1935,-51.2348
br,de tras da serra,De Trás da Serra,20,,-7.783333,-41.666667
br,deus andou,Deus Andou,05,,-12.516667,-39.85
br,deus-me-guia,Deus-me-Guia,22,,-5.423056,-36.3225
br,deus-me-livre,Deus-me-Livre,05,,-12.25,-45.033333
br,deus quer,Deus Quer,13,,-3.418889,-43.476667
br,dez de maio,Dez de Maio,18,,-24.7,-53.916667
br,dez de novembro,Dez de Novembro,26,,-27.033333,-50.933333
br,dez de novembro,Dez de Novembro,27,,-21.216667,-51.35
br,dezessete,Dezessete,24,,-11.083333,-61.916667
br,diadema,Diadema,27,390641,-23.7,-46.616667
br,diamante de uba,Diamante de Ubá,15,,-21.2,-42.916667
br,diamante,Diamante,05,,-11.166667,-42.566667
br,diamante,Diamante,06,,-4.216667,-38.733333
br,diamante,Diamante,17,,-7.416667,-38.25
br,diamante,Diamante,18,,-24.916667,-54.133333
br,diamantina,Diamantina,04,,-4.5425,-60.488611
br,diamantina,Diamantina,13,,-4.216667,-44.2
br,diamantina,Diamantina,15,37344,-18.25,-43.6
br,diamantina,Diamantina,16,,-.466667,-49.1
br,diamantina,Diamantina,17,,-7.466667,-38.316667
br,diamantina,Diamantina,18,,-25.5122,-50.45927
br,diamantina,Diamantina,20,,-6.183333,-41.95
br,diamantina,Diamantina,22,,-5.383611,-36.399167
br,diamantino,Diamantino,14,15700,-14.416667,-56.45
br,diamentina,Diamentina,14,,-14.416667,-56.45
br,dianopolis,Dianópolis,31,12896,-11.700556,-46.743056
br,dias coelho,Dias Coelho,05,,-11.516667,-40.783333
br,dias d'avila,Dias d'Avila,05,,-12.6,-38.3
br,dias,Dias,15,,-22.466667,-45.566667
br,dias,Dias,27,,-23.816667,-47.583333
br,dico leopoldina,Dico Leopoldina,20,,-5.85,-41.35
br,dilermando aguiar,Dilermando Aguiar,23,,-2.7,-54.2
br,dilermanno aguiar,Dilermanno Aguiar,23,,-2.7,-54.2
br,dinarie m. perreira,Dinarie M. Perreira,23,,-31.066667,-55.3
br,dinisia,Dinísia,27,,-21.416667,-49.8
br,diogenes sampaio,Diógenes Sampaio,05,,-12.95,-39.583333
br,diogo de vasconcelos,Diogo de Vasconcelos,15,,-20.483333,-43.2
br,diogo,Diogo,06,,-4.183333,-38.05
br,diogo lopes,Diogo Lopes,22,,-5.084722,-36.454444
br,diogo lopez,Diogo Lopez,22,,-5.084722,-36.454444
br,dionisio cerqueira,Dionísio Cerqueira,26,,-26.25,-53.633333
br,dionisio,Dionísio,05,,-12.183333,-40.816667
br,dionisio,Dionísio,15,,-19.816667,-42.75
br,diorama,Diorama,29,,-16.35,-51.233333
br,dique,Dique,13,,-6.183333,-42.95
br,dirceu barcelos,Dirceu Barcelos,23,,-30.066667,-56.983333
br,dirceu,Dirceu,16,,-1.366667,-46.55
br,dirceu,Dirceu,27,,-22.116667,-49.916667
br,direcao,Direcão,20,,-6.05,-41.85
br,divina pastora,Divina Pastora,05,,-11.95,-37.95
br,divina pastora,Divina Pastôra,28,,-10.666667,-37.15
br,divineia,Divinéia,21,,-22.922996,-43.639509
br,divininho,Divininho,15,,-20.75,-41.883333
br,divino das laranjeiras,Divino das Laranjeiras,15,,-18.766667,-41.483333
br,divino de guanhaes,Divino de Guanhães,15,,-18.8,-42.633333
br,divino de laranjeiras,Divino de Laranjeiras,15,,-18.766667,-41.483333
br,divino de uba,Divino de Ubá,15,,-20.983333,-43.016667
br,divino de virginopolis,Divino de Virginópolis,15,,-18.8,-42.633333
br,divino,Divino,08,,-21,-41.6
br,divino,Divino,11,,-18.75,-55.483333
br,divino,Divino,15,,-19.083333,-41.966667
br,divino do carangola,Divino do Carangola,15,,-20.616667,-42.15
br,divino do cibrao,Divino do Cibrão,15,,-18.266667,-41.35
br,divino espirito santo,Divino Espírito Santo,04,,-5.769167,-61.282222
br,divino espirito santo do carangola,Divino Espirito Santo do Carangola,15,,-20.616667,-42.15
br,divinolandia,Divinolândia,27,,-21.666667,-46.666667
br,divinopolis,Divinópolis,15,193835,-20.15,-44.9
br,divinopolis,Divinópolis,17,,-6.85,-38.416667
br,divinopolis,Divinopolis,22,,-5.983333,-37.816667
br,divinopolis,Divinópolis,29,,-18.183333,-49.35
br,divinopolis do tocantins,Divinópolis do Tocantins,31,,-9.742778,-49.119722
br,divisa,Divisa,06,,-5.016667,-39.95
br,divisa,Divisa,08,,-19.116667,-40.566667
br,divisa,Divisa,11,,-21.5,-54.666667
br,divisa,Divisa,15,,-15.7,-41.333333
br,divisa,Divisa,18,,-26.066667,-50.35
br,divisa,Divisa,21,,-22.4509,-44.3024
br,divisa,Divisa,27,,-21.116667,-50.933333
br,divisa nova,Divisa Nova,15,,-21.516667,-46.2
br,divisao,Divisão,06,,-5.283333,-38.283333
br,divisao,Divisão,20,,-6.15,-41.933333
br,divisao,Divisão,22,,-5.25,-35.966667
br,divisopolis,Divisópolis,15,,-15.733333,-41
br,dix-sept rosado,Dix-sept Rosado,22,,-5.45,-37.516667
br,djalma coutinho,Djalma Coutinho,08,,-20.033333,-40.45
br,djalma dutra,Djalma Dutra,05,,-11.433333,-40.6
br,dobrada,Dobrada,27,,-21.516667,-48.383333
br,dobrado,Dobrado,27,,-21.516667,-48.383333
br,doce fino,Doce Fino,18,,-25.9691,-49.4413
br,doce grande,Doce Grande,18,,-26.0235,-49.4287
br,docimhas,Docimhas,06,,-5.15,-40.4
br,d'oeste esperanca,d'Oeste Esperanca,27,,-21.866667,-51.216667
br,dogelo,Dogelo,26,,-27.7,-50.55
br,dois amigos,Dois Amigos,20,,-3.362778,-42.111389
br,dois buritis,Dois Buritis,05,,-11.3,-42.816667
br,dois caminhos,Dois Caminhos,17,,-6.65,-38.166667
br,dois corregos,Dois Córregos,14,,-15.05,-56.366667
br,dois corregos,Dois Córregos,27,20975,-22.366667,-48.35
br,dois de abril,Dois de Abril,15,,-16.833333,-40.3
br,dois de julho,Dois de Julho,05,,-12.066667,-38.966667
br,dois de maio,Dois de Maio,18,,-25.1,-53.75
br,dois de novembro,Dois de Novembro,24,,-8.85,-62.283333
br,dois irma,Dois Irma,18,,-24.866667,-54.366667
br,dois irmaos do goias,Dois Irmãos do Goiás,31,,-9.211389,-48.955556
br,dois irmaos,Dois Irmãos,05,,-10.15,-43.133333
br,dois irmaos,Dois Irmãos,08,,-20.716667,-40.75
br,dois irmaos,Dois Irmãos,11,,-21.683333,-57.166667
br,dois irmaos,Dois Irmãos,13,,-6.583333,-43.45
br,dois irmaos,Dois Irmãos,18,,-24.1,-49.983333
br,dois irmaos,Dois Irmãos,20,,-5.45,-42.283333
br,dois irmaos,Dois Irmãos,22,,-5.714722,-36.19
br,dois irmaos,Dois Irmãos,23,28926,-29.583333,-51.083333
br,dois irmaos,Dois Irmãos,26,,-27.283333,-51.866667
br,dois irmaos,Dois Irmãos,30,,-8.833333,-35.95
br,dois irmaos,Dois Irmãos,31,,-11.75,-48.633333
br,dois irmaos do tocantins,Dois Irmãos do Tocantins,31,,-9.211389,-48.955556
br,dois lageados,Dois Lageados,23,,-28.983333,-51.85
br,dois lajeados,Dois Lajeados,23,,-28.983333,-51.85
br,dois mil reis,Dois Mil Réis,11,,-18.4,-55.866667
br,dois paus,Dois Paus,13,,-3.146389,-43.462778
br,dois riachos,Dois Riachos,02,4565,-9.383333,-37.083333
br,dois riachos,Dois Riachos,06,,-6.333333,-38.8
br,dois riachos,Dois Riachos,17,,-6.233333,-37.616667
br,dois riachos,Dois Riachos,30,,-7.666667,-37.633333
br,dois rios de baixo,Dois Rios de Baixo,06,,-5.533333,-39.566667
br,dois rios,Dois Rios,21,,-23.183333,-44.183333
br,dois serrotes,Dois Serrotes,30,,-8.216667,-37.3
br,dois vizinhos,Dois Vizinhos,18,22766,-25.783333,-53.083333
br,dolcinopolis,Dolcinópolis,27,,-20.133333,-50.516667
br,dolearina,Dolearina,15,,-18.683333,-47.8
br,dom alexandre,Dom Alexandre,14,,-8.7,-58.016667
br,dom basilio,Dom Basílio,05,,-13.733333,-41.766667
br,dom bosco,Dom Bosco,15,,-16.616667,-41.916667
br,dom cavati,Dom Caváti,15,,-19.383333,-42.1
br,domelia,Domélia,27,,-22.733333,-49.3
br,dom eugenio marculino,Dom Eugênio Marculino,22,,-5.45,-35.383333
br,dom feliciano,Dom Feliciano,23,,-30.7,-52.116667
br,domiciano lemos,Domiciano Lemos,15,,-20.333333,-46.916667
br,domiciano ribeiro,Domiciano Ribeiro,29,,-16.933333,-47.766667
br,domicilio,Domicilio,20,,-5.1,-42.633333
br,domingada,Domingada,27,,-22.683333,-47.916667
br,domingao,Domingão,05,,-12.05,-38.1
br,domingo,Domingo,15,,-15.8,-46.65
br,domingo joao,Domingo João,06,,-5.7,-38.316667
br,domingos campos,Domingos Campos,15,,-21.3,-46.05
br,domingos da costa,Domingos da Costa,06,,-5.15,-39.666667
br,domingos,Domingos,13,,-3.058056,-43.4875
br,domingos,Domingos,15,,-15.8,-46.65
br,domingos martins,Domingos Martins,08,6061,-20.366667,-40.666667
br,dominguinho,Dominguinho,20,,-7.883333,-41.866667
br,dom joaquim,Dom Joaquim,15,,-18.95,-43.266667
br,dom lara,Dom Lara,15,,-19.766667,-42.216667
br,dom leme,Dom Leme,06,,-7.3,-39.766667
br,dom macedo costa,Dom Macedo Costa,05,,-12.9,-39.15
br,dom mauricio,Dom Maurício,06,,-4.9,-39.15
br,dom pedrito,Dom Pedrito,23,36912,-30.983333,-54.666667
br,dom pedrito,Dom Pedrito,24,,-11.433333,-65.316667
br,dom pedro,Dom Pedro,13,15570,-4.483333,-44.45
br,dom quintino,Dom Quintino,06,,-7.083333,-39.483333
br,dom rodrigo,Dom Rodrigo,18,,-25.416667,-49.616667
br,dom silverio do bonfim,Dom Silvério do Bonfim,15,,-20.383333,-44.35
br,dom silverio,Dom Silvério,15,,-20.15,-42.966667
br,dom vicoso,Dom Viçoso,15,,-20.766667,-42.516667
br,dom vital,Dom Vital,06,,-7.25,-39.65
br,dona amelia,Dona Amélia,27,,-22.733333,-49.3
br,dona america,Dona América,08,,-21.166667,-41.433333
br,dona candida,Dona Cândida,27,,-21.416667,-46.883333
br,dona carlota,Dona Carlota,11,,-19.783333,-57.65
br,dona,Dona,02,,-9.45,-37.133333
br,dona ema,Dona Ema,26,,-26.985465,-49.724472
br,dona eusebia,Dona Eusébia,15,,-21.3,-42.8
br,dona euzebia,Dona Euzébia,15,,-21.3,-42.8
br,dona francisca,Dona Francisca,23,,-29.616667,-53.35
br,dona ines,Dona Inês,17,3709,-6.616667,-35.633333
br,dona isabel,Dona Isabel,15,,-18.15,-43.166667
br,dona josefa,Dona Josefa,11,,-21.4,-54.366667
br,dona luisa,Dona Luísa,27,,-23.119612,-47.79444
br,dona mariana,Dona Mariana,21,,-22.166667,-42.633333
br,donatos,Donatos,30,,-7.983333,-38.75
br,don rodrigo,Don Rodrigo,18,,-25.416667,-49.616667
br,doradoquara,Doradoquara,15,,-18.416667,-47.616667
br,dorandia,Dorândia,21,,-22.464444,-43.945278
br,dores da boa esperanca,Dores da Boa Esperanca,15,,-21.083333,-45.566667
br,dores da victoria,Dores da Victoria,15,,-21.1,-42.6
br,dores da vitoria,Dores da Vitória,15,,-21.1,-42.6
br,dores de campo,Dores de Campo,15,,-21.1,-44.033333
br,dores de campos,Dores de Campos,15,,-21.1,-44.033333
br,dores de guanhaes,Dores de Guanhães,15,,-19.05,-42.95
br,dores de macabu,Dores de Macabu,21,,-21.983333,-41.516667
br,dores de santa juliana,Dôres de Santa Juliana,15,,-19.316667,-47.533333
br,dores do indaia,Dores do Indaiá,15,12859,-19.45,-45.6
br,dores do indaya,Dôres do Indayá,15,,-19.45,-45.6
br,dores do paraibuna,Dores do Paraibuna,15,,-21.516667,-43.65
br,dores,Dôres,15,,-19.05,-42.95
br,dores,Dores,29,,-14.433333,-46.133333
br,dores do rio preto,Dores do Rio Preto,08,,-20.7,-41.833333
br,dores do turvo,Dores do Turvo,15,,-20.966667,-43.183333
br,dorgelo,Dorgelo,26,,-27.7,-50.55
br,doris irmaos,Doris Irmãos,06,,-4.883333,-38.466667
br,dorizon,Dorizon,18,,-25.933333,-50.866667
br,dormida,Dormida,25,,-2.55,-60.116667
br,dormideira,Dormideira,06,,-5.883333,-40.633333
br,dos macacos,Dos Macacos,16,,-1.183333,-50.466667
br,dos macaos,Dos Macaos,16,,-1.183333,-50.466667
br,do tapaua boca,do Tapauá Bôca,04,,-5.620278,-63.1875
br,dourada,Dourada,02,,-9.383333,-36.266667
br,douradao,Douradão,11,,-20.666667,-55.4
br,douradinho,Douradinho,11,,-21.583333,-53.266667
br,douradinho,Douradinho,15,,-18.55,-47.2
br,douradinho,Douradinho,23,,-30.4,-51.2
br,dourado,Dourado,06,,-5.033333,-38.3
br,dourado,Dourado,15,,-22.05,-45.966667
br,dourado,Dourado,18,,-23.1299,-50.0641
br,dourado,Dourado,20,,-6.016667,-41.833333
br,dourado,Dourado,23,,-27.416667,-52.233333
br,dourado,Dourado,26,,-27.083333,-53.683333
br,dourado,Dourado,27,,-22.116667,-48.3
br,douradoquara,Douradoquara,15,,-18.416667,-47.616667
br,dourados,Dourados,06,,-4.083333,-38.733333
br,dourados,Dourados,11,162206,-22.216667,-54.8
br,dourados,Dourados,15,,-19.9,-48.7
br,dourados,Dourados,27,,-20.65,-47.716667
br,dous corregos,Dous Corregos,27,,-22.366667,-48.35
br,doutor almeida,Doutor Almeida,05,,-12.316667,-41.066667
br,doutor bozano,Doutor Bozano,23,,-28.4,-53.7
br,doutor camargo,Doutor Camargo,18,,-23.55,-52.233333
br,doutor elias,Doutor Elias,21,,-22.2,-42.233333
br,doutor faria,Doutor Faria,15,,-16.016667,-41.533333
br,doutor loreti,Doutor Loréti,21,,-22.033333,-41.983333
br,doutor luna,Doutor Luna,15,,-19.65,-44.016667
br,doutor lund,Doutor Lund,15,,-19.65,-44.016667
br,doutor mauricio cardoso,Doutor Maurício Cardoso,23,,-27.5,-54.366667
br,doutor paula sousa,Doutor Paula Sousa,29,,-17.966667,-51.65
br,doutor pedrinho,Doutor Pedrinho,26,,-26.742619,-49.573393
br,doutor pestana,Doutor Pestana,23,,-28.516667,-53.983333
br,doutor sampaio,Doutor Sampaio,20,,-6.866667,-42.866667
br,doutor seabra,Doutor Seabra,05,,-12.416667,-41.766667
br,doutor severiano,Doutor Severiano,22,,-6.083333,-38.366667
br,dracena,Dracena,27,37940,-21.533333,-51.483333
br,duarte dias,Duarte Dias,30,,-8.066667,-35.4
br,duartina,Duartina,27,11903,-22.4,-49.416667
br,duas antas,Duas Antas,13,,-1.8,-45.733333
br,duas antas,Duas Antas,18,,-25.22199,-48.987322
br,duas barras do morro,Duas Barras do Morro,05,,-11.966667,-40.966667
br,duas barras,Duas Barras,05,,-11.966667,-40.966667
br,duas barras,Duas Barras,08,,-20.75,-40.883333
br,duas barras,Duas Barras,15,,-16.483333,-40.633333
br,duas barras,Duas Barras,17,,-7.666667,-38.166667
br,duas barras,Duas Barras,20,,-8.95,-42.1
br,duas barras,Duas Barras,21,,-21.383333,-41.65
br,duas barras,Duas Barras,22,,-6.216667,-37.216667
br,duas barras,Duas Barras,27,,-21.183333,-50.266667
br,duas barras,Duas Barras,30,,-7.883333,-37.433333
br,duas bocas,Duas Bôcas,08,,-20.283333,-40.466667
br,duas e dois,Duas e Dois,05,,-14.9,-42.333333
br,duas estradas,Duas Estradas,17,,-6.683333,-35.416667
br,duas fontes,Duas Fontes,31,,-11.585556,-46.531389
br,duas lagoas,Duas Lagoas,05,,-13.183333,-40.233333
br,duas lagoas,Duas Lagoas,20,,-7.233333,-43.416667
br,duas lagoas,Duas Lagoas,31,,-11.933333,-47.65
br,duas palmeiras,Duas Palmeiras,05,,-12.116667,-41.116667
br,duas passagens,Duas Passagens,05,,-12.6,-41.8
br,duas passagens,Duas Passagens,22,,-5.6,-35.516667
br,duas ponies,Duas Ponìes,08,,-20.583333,-40.833333
br,duas pontas,Duas Pontas,31,,-11.585556,-46.531389
br,duas pontes,Duas Pontes,08,,-20.583333,-40.833333
br,duas pontes,Duas Pontes,31,,-11.585556,-46.531389
br,duas serras,Duas Serras,05,,-10.416667,-38.383333
br,duas umburanas,Duas Umburanas,05,,-13.566667,-40.983333
br,duere,Dueré,31,,-11.359444,-49.187222
br,dula,Dula,13,,-3.7425,-43.278611
br,dulcinopolis,Dulcinópolis,27,,-20.133333,-50.516667
br,dumont,Dumont,27,,-21.15,-47.966667
br,dumontina,Dumontina,27,,-22.283333,-51.666667
br,dunas,Dunas,23,,-31.75,-52.283333
br,duplo ceu,Duplo Céu,27,,-20.216667,-49.566667
br,duque bacelar,Duque Bacelar,13,,-4.15,-42.95
br,duque de caxias,Duque de Caxias,05,,-17.616667,-40.033333
br,duque de caxias,Duque de Caxias,21,818339,-22.790321,-43.325009
br,durande,Durandé,15,,-20.2,-41.8
br,duro,Duro,05,,-12.95,-38.616667
br,duros,Duros,20,,-6.116667,-40.966667
br,duro velho,Duro Velho,15,,-18.083333,-41.216667
br,dutra,Dutra,27,,-21.85,-47.566667
br,dyonisio cerqueira,Dyonisio Cerqueira,26,,-26.25,-53.633333
br,eao,Eão,23,,-29.2,-52.233333
br,echapora,Echaporã,27,,-22.433333,-50.2
br,eclerica,Eclérica,29,,-16.166667,-51.75
br,ecoporanga,Ecoporanga,08,13774,-18.383333,-40.833333
br,edeia,Edéia,29,9838,-17.3,-49.916667
br,eden,Éden,21,,-22.794028,-43.398897
br,eden,Éden,27,,-23.416667,-47.416667
br,edgard verneque,Edgard Verneque,30,,-8.083333,-34.933333
br,edgar werneck,Edgar Werneck,30,,-8.083333,-34.933333
br,edil,Edil,06,,-5.25,-40.616667
br,edmundo abreu,Edmundo Abreu,23,,-30.25,-50.85
br,edmundo,Edmundo,15,,-18.516667,-48.066667
br,edson passos,Edson Passos,21,,-22.791208,-43.419659
br,eduardo dietrich,Eduardo Dietrich,08,,-20.283333,-40.65
br,eduardo dietrick,Eduardo Dietrick,08,,-20.283333,-40.65
br,eduardo,Eduardo,15,,-20.683333,-45.2
br,eduardo gomes,Eduardo Gomes,22,,-5.916667,-35.25
br,eduardo luis,Eduardo Luís,11,,-19.066667,-54.033333
br,educandario,Educandário,27,,-23.9205,-46.7359
br,educandos,Educandos,04,,-3.144167,-60.008056
br,edu,Edu,22,,-5.966667,-38.066667
br,efigenia,Efigênia,15,,-19.883333,-41.516667
br,ega,Ega,04,,-3.366667,-64.7
br,egidio,Egidio,06,,-5.15,-37.8
br,egidio,Egídio,27,,-22.883333,-46.983333
br,egon,Egon,18,,-26.214722,-53.609722
br,eirunepe,Eirunepé,04,17175,-6.666667,-69.866667
br,el aduar,El Aduar,23,,-32.283333,-52.25
br,elba,Elba,04,,-5.166667,-62.85
br,el borach,El Borach,24,,-9.666667,-65.433333
br,eldorado,Eldorado,05,,-12.833333,-40.916667
br,eldorado,Eldorado,11,9075,-23.783333,-54.316667
br,eldorado,Eldorado,18,,-23.766667,-51.25
br,eldorado,Eldorado,27,,-24.522222,-48.1075
br,eldorado paulista,Eldorado Paulista,27,,-24.522222,-48.1075
br,elela,Elela,25,,4.416667,-60.15
br,elesbao,Elesbão,21,,-21.966667,-41.65
br,elesbao veloso,Elesbão Veloso,20,10430,-6.216667,-42.133333
br,elesboa veloso,Elesboa Veloso,20,,-6.216667,-42.133333
br,eletra,Eletra,23,,-2.333333,-50.683333
br,eleuterio,Eleutério,27,,-22.316667,-46.716667
br,elias fausto,Elias Fausto,27,12705,-23.033333,-47.383333
br,eliseu brasus,Eliseu Brasus,18,,-25.066667,-53.683333
br,eliseu martins,Eliseu Martins,20,,-8.216667,-43.7
br,elisiario,Elisiário,27,,-21.166667,-49.116667
br,elisiario lemos,Elisiário Lemos,15,,-20.716667,-46.316667
br,eliziario,Eliziário,27,,-21.166667,-49.116667
br,el marco,El Marco,04,,-4.233611,-69.941667
br,eloi,Elói,14,,-8.766667,-58.266667
br,eloi mendes,Elói Mendes,15,19413,-21.616667,-45.566667
br,el oso,El Oso,14,,-15.25,-60.35
br,eloy mendes,Eloy Mendes,15,,-21.616667,-45.566667
br,el ozo,El Ozo,14,,-15.25,-60.35
br,elvira,Elvira,04,,-5.166667,-72.85
br,ema,Ema,05,,-11.683333,-43.266667
br,ema,Ema,06,,-2.95,-40.25
br,ema,Ema,13,,-3.945833,-43.408611
br,ema,Ema,20,,-6.966667,-41.133333
br,ema,Ema,30,,-8.283333,-38.383333
br,emanuel ramalho,Emanuel Ramalho,15,,-17.5,-40.783333
br,emas,Emas,05,,-11.683333,-43.266667
br,emas,Emas,17,,-7.1,-37.716667
br,emas,Emas,22,,-6.366667,-38.25
br,ematuba,Ematuba,06,,-5.133333,-40.216667
br,embare,Embaré,27,,-21.95,-47.933333
br,embauba,Embaúba,27,,-20.983333,-48.833333
br,embauba,Embaúba,31,,-11.683333,-48.766667
br,embaubas,Embaúbas,15,,-17.616667,-42.8
br,embaubas,Embaúbas,21,,-22.6,-42.416667
br,embau,Embaú,21,,-22.6,-42.416667
br,embauval,Embauval,14,,-16.316667,-55.35
br,embira,Embira,05,,-12.65,-39.15
br,embiral,Embiral,13,,-5.383333,-47.716667
br,embirizal,Embirizal,15,,-22.033333,-45.433333
br,embirucu,Embiruçu,15,,-19.383333,-44.316667
br,emboaba,Emboaba,23,,-29.3,-51.333333
br,emboabas,Emboabas,15,,-21.3,-44.133333
br,emboracica,Emboracica,05,,-12.6,-38.133333
br,emborahy,Emborahy,16,,-1.15,-46.5
br,emborai,Emboraí,16,,-1.15,-46.5
br,emboscaba,Emboscaba,11,,-22.8,-55.55
br,emboscada,Emboscada,11,,-22.8,-55.55
br,embosoada,Embosoada,11,,-22.8,-55.55
br,embu,Embu,27,221737,-23.65,-46.85
br,embu-guacu,Embu-Guaçu,27,76311,-23.816667,-46.8
br,embu-mirim,Embu-Mirim,27,,-22.533333,-45.083333
br,embura,Embura,27,,-23.8824,-46.7422
br,emburana,Emburana,22,,-5.2,-35.8
br,emburrado,Emburrado,05,,-14.116667,-39.75
br,eme,Eme,15,,-19.166667,-41.3
br,emendada,Emendada,02,,-9.616667,-37.566667
br,e. mendes,E. Mendes,11,,-20.566667,-56.116667
br,emericiano borges,Emericiano Borges,15,,-16.9,-41.65
br,emidio da rosa,Emídio da Rosa,11,,-19.6,-55.033333
br,emidio,Emídio,15,,-17.366667,-40.833333
br,emidios,Emídios,15,,-21.133333,-42.666667
br,emile,Emile,04,,-4.166667,-59.216667
br,emilianopolis,Emilianópolis,27,,-21.816667,-51.483333
br,emilia paulista,Emília Paulista,27,,-21.75,-50.966667
br,emilio,Emílio,04,,-4.166667,-59.216667
br,emilio magno,Emílio Magno,05,,-16.133333,-38.966667
br,emilio meier,Emílio Méier,23,,-30.266667,-50.516667
br,emilio meyer,Emílio Meyer,23,,-30.266667,-50.516667
br,emilio ribas,Emílio Ribas,27,,-22.716667,-45.566667
br,emilio timo,Emílio Timo,15,,-17.916667,-41.75
br,emmas,Emmas,05,,-11.683333,-43.266667
br,emparedado,Emparedado,20,,-6.4,-42.666667
br,empatacao,Empatação,13,,-6.383333,-42.983333
br,empocado,Empoçado,18,,-26.166667,-51.45
br,empresa,Emprêsa,01,,-9.966667,-67.8
br,empresa,Emprêsa,05,,-13.633333,-39.5
br,emvira,Emvira,04,,-7.3,-70.216667
br,enamuna,Enamuna,25,,4.65,-60.116667
br,enamung,Enamung,25,,4.65,-60.116667
br,encano alto,Encano Alto,26,,-26.9894,-49.1716
br,encano baixo,Encano Baixo,26,,-26.8862,-49.1855
br,encano central,Encano Central,26,,-26.9266,-49.1938
br,encano do norte,Encano do Norte,26,,-26.8674,-49.2024
br,encano,Encano,26,,-26.9901,-49.1107
br,encantado,Encantado,06,,-5.483333,-39.3
br,encantado,Encantado,13,,-6.15,-43.15
br,encantado,Encantado,21,,-22.892808,-43.299706
br,encantado,Encantado,23,16637,-29.25,-51.883333
br,encanto,Encanto,06,,-5.516667,-39.3
br,encanto,Encanto,22,,-6.116667,-38.316667
br,encanto,Encanto,25,,2.9,-60.95
br,encarnacao,Encarnação,05,,-12.933333,-38.75
br,enchente,Enchente,16,,-1.15,-49.633333
br,enchu,Enchu,31,,-7.166667,-48.233333
br,encosta,Encosta,17,,-6.2,-37.4
br,encruzilhada do sul,Encruzilhada do Sul,23,17199,-30.533333,-52.516667
br,encruzilhada,Encruzilhada,05,4711,-15.516667,-40.9
br,encruzilhada,Encruzilhada,06,,-4.516667,-37.95
br,encruzilhada,Encruzilhada,13,,-3.411944,-42.414444
br,encruzilhada,Encruzilhada,15,,-21.833333,-44.8
br,encruzilhada,Encruzilhada,18,,-25.45,-50.233333
br,encruzilhada,Encruzilhada,21,,-22.183889,-43.296389
br,encruzilhada,Encruzilhada,23,,-28.25,-51.383333
br,encruzilhada,Encruzilhada,26,,-26.216667,-50.466667
br,encruzilhada,Encruzilhada,30,,-7.533333,-39.416667
br,encruzo,Encruzo,05,,-12.616667,-39.15
br,encruzo,Encruzo,08,,-19.983333,-40.433333
br,eneida,Eneida,27,,-21.916667,-51.3
br,enemutu,Enemutu,25,,4.033333,-59.883333
br,enganado,Enganado,05,,-17.033333,-39.566667
br,engano,Engano,04,,-4.816667,-72.366667
br,engano,Engano,06,,-4.933333,-39.116667
br,engano,Engano,08,,-20.616667,-40.85
br,engano,Engano,11,,-21.383333,-54.066667
br,engano,Engano,20,,-3.968056,-42.398889
br,engano,Engano,21,,-22.32263,-42.987556
br,engano,Engano,26,,-27.083333,-52.133333
br,engasgo,Engasgo,20,,-8.633333,-45.4
br,engenheiro avidos,Engenheiro Ávidos,17,,-6.983333,-38.466667
br,engenheiro balduino,Engenheiro Balduíno,27,,-20.683333,-49.7
br,engenheiro beltrao,Engenheiro Beltrão,18,14955,-23.783333,-52.25
br,engenheiro blei,Engenheiro Blei,18,,-25.616667,-49.75
br,engenheiro bley,Engenheiro Bley,18,,-25.616667,-49.75
br,engenheiro carvalhais,Engenheiro Carvalhais,21,,-22.033333,-43.4
br,engenheiro correia,Engenheiro Correia,15,,-20.333333,-43.783333
br,engenheiro dodt,Engenheiro Dodt,20,,-8.816667,-45.933333
br,engenheiro franca,Engenheiro Franca,05,,-13.25,-39.766667
br,engenheiro gutierrez,Engenheiro Gutierrez,18,,-25.517425,-50.655997
br,engenheiro joao tome,Engenheiro João Tomé,06,,-4.6,-40.633333
br,engenheiro jose lopes,Engenheiro José Lopes,06,,-5.666667,-39.433333
br,engenheiro junqueira,Engenheiro Junqueira,21,,-22.983333,-44.05
br,engenheiro leal,Engenheiro Leal,21,,-22.877436,-43.322833
br,engenheiro lisboa,Engenheiro Lisboa,15,,-19.8,-47.6
br,engenheiro marsilac,Engenheiro Marsilac,27,,-23.9078,-46.7069
br,engenheiro morbech,Engenheiro Morbech,14,,-16.05,-53.633333
br,engenheiro morbeck,Engenheiro Morbeck,14,,-16.05,-53.633333
br,engenheiro morsing,Engenheiro Morsing,21,,-22.540556,-43.776944
br,engenheiro navarro,Engenheiro Navarro,15,,-17.283333,-43.95
br,engenheiro nicanor pereira,Engenheiro Nicanor Pereira,21,,-22.624167,-43.705278
br,engenheiro nobrega,Engenheiro Nóbrega,21,,-22.451389,-43.700556
br,engenheiro passos,Engenheiro Passos,21,,-22.5,-44.683333
br,engenheiro paulo de frontin,Engenheiro Paulo de Frontin,21,,-22.549444,-43.683889
br,engenheiro pedreira,Engenheiro Pedreira,21,,-22.689444,-43.595556
br,engenheiro pontes,Engenheiro Pontes,05,,-13.066667,-39.45
br,engenheiro raul goncalves,Engenheiro Raul Gonçalves,29,,-17.85,-48.183333
br,engenheiro schmidt,Engenheiro Schmidt,27,,-20.866667,-49.3
br,engenheiro schnoor,Engenheiro Schnoor,15,,-17.166667,-42.083333
br,engenheiro taveira,Engenheiro Taveira,27,,-21.15,-50.5
br,engenheiro vieira cortes,Engenheiro Vieira Cortês,21,,-22.183333,-43.433333
br,engenhinho,Engenhinho,14,,-14.983333,-56.633333
br,engenho agua branca,Engenho Água Branca,05,,-12,-40.666667
br,engenho araci,Engenho Araci,16,,-1.116667,-48.266667
br,engenhoca,Engenhoca,05,,-12.616667,-38.866667
br,engenhoca,Engenhoca,13,,-3.860556,-43.361111
br,engenho central,Engenho Central,13,,-3.616667,-45.35
br,engenho central,Engenho Central,21,,-22.416667,-44.3
br,engenho d'agua,Engenho d'Agua,05,,-12.616667,-38.633333
br,engenho d'agua,Engenho d'Agua,13,,-4.866667,-43.183333
br,engenho d'agua,Engenho d'Agua,21,,-21.566667,-41.883333
br,engenho da rainha,Engenho da Rainha,21,,-22.866943,-43.289473
br,engenho da raiz,Engenho da Raiz,05,,-12.25,-38.85
br,engenho da serra,Engenho da Serra,21,,-22.921749,-43.321663
br,engenho de dentro,Engenho de Dentro,21,,-22.899791,-43.295592
br,engenho de quicama,Engenho de Quiçamã,21,,-22.1,-41.55
br,engenho de quissama,Engenho de Quissamã,21,,-22.1,-41.55
br,engenho de serra,Engenho de Serra,15,,-21.3,-45.15
br,engenho doldt,Engenho Doldt,20,,-8.816667,-45.933333
br,engenho do mato,Engenho do Mato,21,,-22.865,-43.296944
br,engenho do ribeiro,Engenho do Ribeiro,15,,-19.566667,-45.283333
br,engenho,Engenho,05,,-10.7,-44.25
br,engenho,Engenho,06,,-6.183333,-38.566667
br,engenho,Engenho,13,,-4.416667,-44.75
br,engenho,Engenho,14,,-14.816667,-56.333333
br,engenho,Engenho,15,,-15.183333,-43.083333
br,engenho,Engenho,16,,-1.016667,-47.366667
br,engenho,Engenho,21,,-22.918056,-43.834444
br,engenho,Engenho,22,,-6.033333,-38.083333
br,engenho gutierrez,Engenho Gutierrez,18,,-25.535033,-50.65262
br,engenho irmaos gnatta marumbi,Engenho Irmãos Gnatta Marumbi,18,,-25.5078,-48.8764
br,engenho novo,Engenho Novo,05,,-12.183333,-38.583333
br,engenho novo,Engenho Novo,15,,-21.814722,-43.002222
br,engenho novo,Engenho Novo,21,,-22.905921,-43.265956
br,engenho novo,Engenho Novo,30,,-8.783333,-36.016667
br,engenho pederneiras,Engenho Pederneiras,27,,-22.970556,-47.792222
br,engenho prata,Engenho Prata,02,,-9.916667,-36.116667
br,engenho sao jose,Engenho São José,22,,-5.616667,-35.333333
br,engenho velho,Engenho Velho,05,,-10.7,-44.25
br,engenho velho,Engenho Velho,13,,-3.266667,-42.683333
br,engenho velho,Engenho Velho,14,,-15.783333,-56.133333
br,engenho velho,Engenho Velho,15,,-20.95,-46.6
br,engenho velho,Engenho Velho,17,,-7.666667,-38.616667
br,engenho velho,Engenho Velho,21,,-22.922038,-43.217414
br,engenho velho,Engenho Velho,30,,-7.666667,-37.75
br,enio pepino,Ênio Pepino,27,,-22.483333,-52.333333
br,enjeitado,Enjeitado,13,,-2.89,-41.903611
br,enjeitado,Enjeitado,20,,-3.718611,-42.465833
br,enseada da cutia,Enseada da Cutia,04,,-3.476111,-63.350833
br,enseada de brito,Enseada de Brito,26,,-27.779167,-48.633889
br,enseada de fogo,Enseada de Fogo,02,,-9.966667,-36.083333
br,enseada do brito,Enseada do Brito,26,,-27.779167,-48.633889
br,enseada do fogo,Enseada do Fogo,02,,-9.966667,-36.083333
br,enseada do jenipapo,Enseada do Jenipapo,04,,-4.945556,-62.424444
br,enseada do membeca,Enseada do Membeca,04,,-5.549444,-63.106389
br,enseada,Enseada,20,,-9.953889,-45.122222
br,enseada,Enseada,27,,-23.483333,-45.083333
br,enseadinha,Enseadinha,30,,-8.083333,-38.466667
br,entrada da torre,Entrada da Torre,27,,-24.578575,-48.106114
br,entrada do campo,Entrada do Campo,23,,-28.733333,-51.266667
br,entrada,Entrada,20,,-8.666667,-41.883333
br,entre folhas,Entre Fôlhas,15,,-19.616667,-42.233333
br,entre ijuis,Entre Ijuís,23,,-28.35,-54.266667
br,entre monte,Entre Monte,02,,-9.666667,-37.65
br,entremontes,Entremontes,02,,-9.666667,-37.65
br,entre montes,Entre Montes,06,,-4.633333,-40.25
br,entremontes,Entremontes,30,,-7.55,-39.816667
br,entre morros,Entre Morros,20,,-3.474444,-42.074722
br,entrepelado,Entrepelado,23,,-29.733333,-50.75
br,entre rios de minas,Entre Rios de Minas,15,,-20.683333,-44.066667
br,entre rios,Entre Rios,04,,-4.116667,-58.7
br,entre rios,Entre Rios,05,27406,-11.933333,-38.083333
br,entre rios,Entre Rios,06,,-4.2,-40.433333
br,entre rios,Entre Rios,11,,-19.366667,-54.15
br,entre rios,Entre Rios,15,,-20.683333,-44.066667
br,entre-rios,Entre-Rios,16,,-5.4,-54.466667
br,entre rios,Entre Rios,17,,-6.866667,-35.6
br,entre rios,Entre Rios,18,,-25.274167,-50.235556
br,entre-rios,Entre-Rios,20,,-6.783333,-42.483333
br,entre-rios,Entre-Rios,21,,-22.111389,-43.207778
br,entre rios,Entre Rios,23,,-28.55,-51.35
br,entre rios,Entre Rios,27,,-20.65,-51.383333
br,entre val,Entre Val,05,,-12.583333,-39.25
br,entroncamento,Entroncamento,05,,-13.2,-40.333333
br,entroncamento,Entroncamento,22,,-5.265278,-36.801944
br,entroncamento,Entroncamento,23,,-29.85,-54.933333
br,entronoamento,Entronoamento,06,,-6.333333,-38.916667
br,entupido,Entupido,27,,-22.5,-44.85
br,envira,Envira,04,9957,-7.3,-70.216667
br,envira,Envira,14,,-4.370278,-58.1225
br,enxofre,Enxofre,27,,-23.000393,-47.615523
br,enxovia de baixo,Enxovia de Baixo,27,,-23.387749,-47.937582
br,enxovia do meio,Enxovia do Meio,27,,-23.397363,-47.975034
br,enxu,Enxu,05,,-11.05,-39.683333
br,enxu,Enxu,13,,-3.320833,-42.587222
br,enxu,Enxu,20,,-8.5,-44.583333
br,epaminondas matos,Epaminondas Matos,15,,-16.783333,-41.3
br,epaminondas otoni,Epaminondas Otoni,15,,-17.916667,-41.016667
br,epaminondas ridiculo,Epaminondas Ridículo,05,,-16.216667,-39.833333
br,ephigenia,Ephigenia,15,,-19.883333,-41.516667
br,epigua,Epiguá,27,,-20.65,-49.366667
br,epistasio,Epistasio,22,,-5.520278,-36.383611
br,epistasio pessoa,Epistasio Pessôa,22,,-5.520278,-36.383611
br,epitacio pessoa,Epitácio Pessoa,18,,-24.7814,-48.983
br,epitacio pessoa,Epitácio Pessoa,22,,-5.520278,-36.383611
br,equador,Equador,22,,-6.95,-36.716667
br,era crista rio branco,Era Cristã Rio Branco,27,,-22.742222,-47.329167
br,era,Era,05,,-13.1,-40.8
br,erale,Eralé,16,,.3,-54.016667
br,era nova,Era Nova,15,,-18.933333,-41.933333
br,ercilio vargas,Ercílio Vargas,11,,-23.366667,-54.65
br,erebango,Erebango,23,,-27.883333,-52.3
br,erecatu,Erecatú,23,,-28.316667,-52.05
br,erechim,Erechim,23,96089,-27.633333,-52.283333
br,eremetu,Eremetú,25,,4.033333,-59.883333
br,eremita,Eremita,15,,-21.583333,-45.066667
br,eremiterio santa clara,Eremitério Santa Clara,18,,-25.819,-49.144
br,erequim,Erequim,23,,-27.633333,-52.283333
br,erere,Ererê,06,,-6.033333,-38.35
br,erere,Ererê,16,,-1.983333,-54.166667
br,eretreia,Eretréia,27,,-22.3,-47.583333
br,erexim,Erexim,23,,-27.633333,-52.283333
br,erial,Erial,15,,-19.783333,-47.666667
br,eritima,Eritimã,16,,.583333,-54.116667
br,eritiman,Eritiman,16,,.583333,-54.116667
br,eritrea,Eritréa,27,,-22.3,-47.583333
br,eritreia,Eritréia,27,,-22.3,-47.583333
br,ermenegildo,Ermenegildo,14,,-10.283333,-60.516667
br,ermida,Ermida,27,,-23.2,-47
br,ermidinha,Ermidinha,15,,-16.432222,-44.090556
br,ermo,Ermo,06,,-6.516667,-38.616667
br,ernestina,Ernestina,23,,-28.5,-52.566667
br,ernesto alves,Ernesto Alves,23,,-29.366667,-54.733333
br,ernesto guedes,Ernesto Guedes,04,,-3,-65.95
br,ernesto machado,Ernesto Machado,21,,-21.633333,-41.65
br,ernesto pires,Ernesto Pires,15,,-20.55,-45.466667
br,erva de rato,Erva de Rato,02,,-9.283333,-36.233333
br,erva,Erva,15,,-18.466667,-45.45
br,erva,Erva,18,,-25.15,-49.666667
br,erval de baixo,Erval de Baixo,18,,-24.516667,-50.85
br,erval d'oeste,Erval d'Oeste,26,,-27.216667,-51.566667
br,erval,Erval,01,,-8.133333,-72.833333
br,erval,Erval,11,,-20.733333,-56.283333
br,erval,Erval,15,,-20.833333,-42.65
br,erval,Erval,18,,-23.782994,-50.047793
br,erval,Erval,23,,-31.883333,-53.166667
br,erval,Erval,26,,-26.883333,-53.433333
br,erval grande,Erval Grande,23,,-27.383333,-52.583333
br,ervalia,Ervália,15,,-20.833333,-42.65
br,erval seco,Erval Sêco,23,,-27.55,-53.5
br,erval velho,Erval Velho,26,,-27.283333,-51.433333
br,ervalzinho,Ervalzinho,18,,-24.833333,-51.283333
br,ervania,Ervânia,11,,-21.716667,-55.533333
br,erveira,Erveira,18,,-25.25,-52.533333
br,erveira,Erveira,26,,-26.1784,-49.977
br,erveiras,Erveiras,23,,-29.45,-52.65
br,esburacado,Esburacado,15,,-15.4,-42.75
br,escada,Escada,20,,-5.933333,-40.966667
br,escada,Escada,30,48084,-8.366667,-35.233333
br,escadilha,Escadilha,22,,-5.25,-35.933333
br,escadinha,Escadinha,30,,-8.083333,-38.466667
br,escalvado,Escalvado,15,,-20.233333,-42.816667
br,escalvado,Escalvado,20,,-8.216667,-42.416667
br,escalvado,Escalvado,26,,-26.833333,-48.733333
br,escalvados,Escalvados,13,,-3.751944,-42.706667
br,escapole,Escapole,06,,-2.866667,-40.016667
br,escaramuca,Escaramuça,27,,-20.733333,-47.683333
br,escola de pesca,Escola de Pesca,21,,-23.066667,-43.983333
br,escolvado,Escolvado,20,,-8.216667,-42.416667
br,escondido,Escondido,05,,-17.083333,-39.516667
br,escondido,Escondido,06,,-5.616667,-41.016667
br,escondido,Escondido,13,,-3.2725,-43.205
br,escondido,Escondido,16,,-5.45,-54.316667
br,escondido,Escondido,20,,-7.8,-41.65
br,escondido,Escondido,22,,-5.355278,-36.922778
br,escondido,Escondido,24,,-13.583333,-60.933333
br,escondido,Escondido,29,,-14.583333,-46.2
br,escondido,Escondido,30,,-7.783333,-37.583333
br,esconso,Esconso,05,,-12.3,-41.55
br,escravado,Escravado,20,,-5.6,-41.816667
br,escrivao,Escrivão,16,,-3.05,-55.183333
br,escura,Escura,15,,-19.3,-42.383333
br,escurial,Escurial,21,,-22.7,-42.816667
br,escurinho,Escurinho,26,,-28.083333,-50.483333
br,escuro,Escuro,01,,-8.148611,-72.857222
br,escuta,Escuta,13,,-3.255,-43.424167
br,esfola-carneiro,Esfola-Carneiro,05,,-12.883333,-38.783333
br,eslivas,Eslivas,13,,-4.666667,-45.75
br,esmaratita,Esmaratita,23,,-28.05,-51.2
br,esmeralda,Esmeralda,04,,-.616667,-63.35
br,esmeralda,Esmeralda,15,,-19.333333,-42.866667
br,esmeralda,Esmeralda,23,,-28.05,-51.2
br,esmeraldas,Esmeraldas,04,,-.616667,-63.35
br,esmeraldas,Esmeraldas,15,103252,-19.75,-44.316667
br,esmeril,Esmeril,15,,-20.5,-43.966667
br,esmeril,Esmeril,27,,-20.816667,-47.3
br,espacinha,Espacinha,06,,-4.666667,-40.483333
br,espadim,Espadim,11,,-23.833333,-55.366667
br,espalhado,Espalhado,27,,-22.531389,-47.255556
br,espalha,Espalha,22,,-5.993889,-37.395278
br,espanha,Espanha,16,,-5.933333,-57.466667
br,espanha,Espanha,30,,-8.933333,-37.516667
br,espera d'anta,Espera d'Anta,05,,-11.85,-40.783333
br,espera,Espera,05,,-12.183333,-40.3
br,espera,Espera,20,,-7.283333,-41.55
br,espera feliz,Espera Feliz,15,13031,-20.65,-41.916667
br,esperana,Esperana,27,,-23.315,-46.0112
br,esperanca,Esperança,01,,-8.25,-72.733333
br,esperanca,Esperança,04,,-3.513889,-62.911389
br,esperanca,Esperança,06,,-4.633333,-39.366667
br,esperanca,Esperança,11,,-17.966667,-54.466667
br,esperanca,Esperança,13,,-9.783333,-46.283333
br,esperanca,Esperança,14,,-17.55,-55
br,esperanca,Esperança,16,,-1.75,-52.466667
br,esperanca,Esperança,17,18817,-7.016667,-35.85
br,esperanca,Esperança,18,,-23.565508,-49.778472
br,esperanca,Esperança,20,,-10.8,-45.15
br,esperanca,Esperança,22,,-5.508611,-36.207778
br,esperanca,Esperança,23,,-27.366667,-53.966667
br,esperanca,Esperança,30,,-7.75,-38.616667
br,esperanca feliz,Esperança Feliz,04,,-5.572222,-61.0675
br,esperancinha,Esperancinha,05,,-13.733333,-40.25
br,esperantina,Esperantina,20,21044,-3.9,-42.233333
br,esperantina,Esperantina,31,,-5.313056,-48.415
br,esperantinopolis,Esperantinópolis,13,13167,-4.883333,-44.883333
br,esperanza,Esperanza,04,,-4.408889,-69.864167
br,esperito santo,Esperito Santo,04,,-3.433333,-68.716667
br,espero-em-deus,Espero-em-Deus,04,,-6.35,-60.316667
br,espigao alto,Espigão Alto,18,,-25.416667,-52.8
br,espigao alto,Espigão Alto,23,,-27.6,-51.433333
br,espigao bonito,Espigão Bonito,18,,-23.947229,-50.385707
br,espigao comprido,Espigão Comprido,27,,-24.634741,-48.102103
br,espigao das antas,Espigão das Antas,18,,-25.904756,-49.329364
br,espigao do bugre,Espigão do Bugre,26,,-26.1827,-49.813
br,espigao d'oeste,Espigao d'Oeste,24,,-11.583333,-60.983333
br,espigao,Espigão,15,,-14.983333,-42.583333
br,espigao,Espigão,18,,-25.944214,-50.139135
br,espigao,Espigão,27,,-22.183333,-51.366667
br,espingarda,Espingarda,13,,-3.866944,-43.051111
br,espinhaco,Espinhaço,13,,-3.993611,-43.371667
br,espinha de peixe,Espinha de Peixe,15,,-19.133333,-45.966667
br,espinha do peixe,Espinha do Peixe,15,,-19.133333,-45.966667
br,espinha,Espinha,05,,-13.883333,-40.533333
br,espinha,Espinha,13,,-7.4,-44.7
br,espinhal,Espinhal,16,,-5.583333,-48.183333
br,espinharas,Espinharas,17,,-7.133333,-37.05
br,espinheirinho,Espinheirinho,22,,-5.456944,-37.161944
br,espinheiro de baixo,Espinheiro de Baixo,22,,-5.834722,-37.101389
br,espinheiro,Espinheiro,06,,-7.1,-40.066667
br,espinheiro,Espinheiro,14,,-15.366667,-56.316667
br,espinheiro,Espinheiro,20,,-7.833333,-41.933333
br,espinheiro,Espinheiro,22,,-5.219444,-36.625833
br,espinheiro,Espinheiro,30,,-8.533333,-38.233333
br,espinheiros,Espinheiros,26,,-26.25,-48.8
br,espinho,Espinho,05,,-13.333333,-41.366667
br,espinho,Espinho,06,,-3.266667,-40.15
br,espinhos,Espinhos,20,,-8.183333,-42.483333
br,espinilho,Espinilho,23,,-28.15,-53.733333
br,espinilho,Espinilho,26,,-27.283333,-51.1
br,espinosa,Espinosa,15,17549,-14.933333,-42.833333
br,espirito santo de vitoria,Espírito Santo de Vitória,08,,-20.333333,-40.283333
br,espirito santo do dourado,Espírito Santo do Dourado,15,,-22.05,-45.966667
br,espirito santo do pinhal,Espirito Santo do Pinhal,27,37449,-22.2,-46.75
br,espirito santo do rio pardo,Espírito Santo do Rio Pardo,27,,-23.1,-48.383333
br,espirito santo do turvo,Espírito Santo do Turvo,27,,-22.716667,-49.416667
br,espirito santo,Espírito Santo,03,,1.25,-50.9
br,espirito santo,Espírito Santo,04,,-2.716667,-67.55
br,espirito santo,Espírito Santo,05,,-13.716667,-41.3
br,espirito santo,Espírito Santo,06,,-4.05,-41.183333
br,espirito santo,Espírito Santo,08,,-20.333333,-40.283333
br,espirito santo,Espírito Santo,11,,-21.4,-56.233333
br,espirito santo,Espírito Santo,13,,-6.983333,-44.233333
br,espirito santo,Espirito Santo,14,,-16.766667,-55.533333
br,espirito santo,Espírito Santo,15,,-17.65,-45.933333
br,espirito santo,Espírito Santo,16,,-1.1,-49.216667
br,espirito santo,Espírito Santo,17,,-6.45,-38.4
br,espirito santo,Espírito Santo,18,,-23.366667,-51.216667
br,espirito santo,Espírito Santo,20,,-5.016667,-41.233333
br,espirito santo,Espírito Santo,21,,-21.266667,-41.416667
br,espirito santo,Espírito Santo,22,,-6.316667,-35.283333
br,espirito santo,Espírito Santo,23,,-30.158333,-51.220833
br,espirito santo,Espírito Santo,24,,-10.066667,-62.566667
br,espirito santo,Espírito Santo,25,,3.800941,-59.623694
br,espirito santo,Espírito Santo,28,,-11.533333,-37.516667
br,espirito santo,Espirito Santo,30,,-7.583333,-37.55
br,espirito santo quartal geral,Espirito Santo Quartal Geral,15,,-19.3,-45.55
br,espiritu santo,Espiritu Santo,04,,-2.95,-69.116667
br,esplanada,Esplanada,05,18859,-11.783333,-37.95
br,esplanada,Esplanada,26,,-28.7,-49.183333
br,espraiado,Espraiado,20,,-5.4,-42.75
br,espuma,Espuma,17,,-7.483333,-37.583333
br,espumoso,Espumoso,23,10340,-28.733333,-52.85
br,esquerda,Esquerda,20,,-7.4,-41.816667
br,esquia,Esquia,13,,-3.990278,-43.023889
br,esquina gaucha,Esquina Gaúcha,23,,-27.45,-54.2
br,esquina tucunduva,Esquina Tucunduva,23,,-27.633333,-54.416667
br,esquina tudunduva,Esquina Tudunduva,23,,-27.633333,-54.416667
br,esse,Esse,06,,-4.35,-39.333333
br,estacada,Estacada,17,,-6.766667,-35.316667
br,estacadinha,Estacadinha,22,,-5.250556,-36.771111
br,estacado,Estacado,05,,-12.033333,-38.65
br,estaca,Estaca,11,,-20.3,-54.533333
br,estacao de condado,Estacão de Condado,17,,-6.866667,-37.583333
br,estacao de coronel ponce,Estacão de Coronel Ponce,14,,-15.566667,-55.016667
br,estacao de nhambiquaras,Estacão de Nhambiquaras,14,,-12.75,-59.8
br,estacao de utiarity,Estacão de Utiarity,14,,-13.033333,-58.283333
br,estacao do juruena,Estaçao do Juruena,14,,-12.85,-58.933333
br,estacao engenheiro cornelio,Estação Engenheiro Cornélio,30,,-7.9,-38.083333
br,estacao,Estação,04,,-7,-73.75
br,estacao,Estação,20,,-8.283333,-42.2
br,estacao,Estação,27,,-23.029702,-47.950508
br,estacao experimental,Estação Experimental,18,,-25.4527,-50.5836
br,estacao mandaguari,Estacão Mandaguarí,27,,-22.183333,-51.366667
br,estacao nova,Estação Nova,25,,3.7,-61.383333
br,estacao parecis,Estação Parecis,14,,-14.15,-56.933333
br,estacao rio vermelho,Estação Rio Vermelho,26,,-26.283333,-49.333333
br,estacao telegrafica torno largo,Estação Telegráfica Torno Largo,24,,-9.116667,-63.25
br,estaca zero,Estaca Zero,05,,-10.166667,-39.116667
br,estaca zero,Estaca Zero,20,,-5.716667,-42.633333
br,estacio,Estácio,21,,-22.918166,-43.20397
br,estados unidos,Estados Unidos,04,,-3.293333,-62.923333
br,estaleiro de cima,Estaleiro de Cima,05,,-12.25,-38.633333
br,estaleiro,Estaleiro,05,,-12.25,-38.633333
br,estaleiro,Estaleiro,18,,-25.9016,-48.7679
br,estalo,Estalo,21,,-22.566667,-44.466667
br,estancia da lagoa,Estância da Lagoa,23,,-29.216667,-56.083333
br,estancia da pedreira,Estância da Pedreira,23,,-29.633333,-56.3
br,estancia da saudade,Estância da Saudade,23,,-29.55,-56.816667
br,estancia da sociedade,Estância da Sociedade,23,,-30.266667,-55.65
br,estancia de lima,Estância de Lima,23,,-31.016667,-55.85
br,estancia do castello,Estancia do Castello,08,,-20.6,-41.2
br,estancia do cati,Estância do Cati,23,,-30.483333,-56.15
br,estancia do cerro,Estância do Cêrro,23,,-30.733333,-55.75
br,estancia do junco,Estância do Junco,23,,-30.033333,-56.816667
br,estancia do mormaco,Estância do Mormaço,23,,-30.733333,-55.85
br,estancia do serro,Estância do Serro,23,,-30.733333,-55.75
br,estancia entroncamento,Estância Entroncamento,23,,-29.85,-54.933333
br,estancia esperanca,Estância Esperança,23,,-31,-55.833333
br,estancia,Estância,28,55655,-11.266667,-37.433333
br,estancia funchal,Estância Funchal,23,,-30.566667,-55.883333
br,estancia grande,Estância Grande,23,,-29.9984,-50.9499
br,estancia ibira,Estância Ibira,23,,-30.45,-55.766667
br,estancia jaguareta,Estância Jaguareta,23,,-29.9,-55.083333
br,estancia jose benitez,Estância José Benítez,23,,-30.933333,-55.866667
br,estancia jose m. de sousa,Estância José M. de Sousa,23,,-30.55,-56.2
br,estancia lagoa brilhante,Estância Lagoa Brilhante,14,,-14.451389,-59.6875
br,estancia lula vilela,Estância Lula Vilela,23,,-29.95,-56.633333
br,estancia manoel fialho,Estância Manoel Fialho,23,,-31.066667,-55.983333
br,estancia mimosa,Estância Mimosa,23,,-29.783333,-56.383333
br,estancia mirante,Estância Mirante,23,,-29.633333,-52.083333
br,estancia nova,Estância Nova,23,,-30.133333,-54.516667
br,estancia querencia,Estância Querência,27,,-23.794775,-46.681455
br,estancia rancho felicio,Estância Rancho Felício,23,,-30.716667,-56.033333
br,estancia rosario,Estância Rosário,23,,-30.25,-55.566667
br,estancia santa luisa,Estância Santa Luísa,23,,-29.683333,-54.633333
br,estancia santa rosa,Estância Santa Rosa,23,,-29.283333,-55.8
br,estancia santa ursula,Estância Santa Úrsula,23,,-30.483333,-55.9
br,estancia santo inacio,Estancia Santo Inácio,23,,-30.666667,-55.15
br,estancia sao domingos,Estância São Domingos,23,,-28.966667,-55.6
br,estancia sao francisco,Estância São Francisco,23,,-29.55,-56.433333
br,estancia sao vicente,Estância São Vicente,23,,-29.05,-56.283333
br,estancia sarandi,Estância Sarandi,23,,-30.666667,-55.95
br,estancia teixeira,Estância Teixeira,23,,-30.366667,-54.066667
br,estancia teresinha,Estância Teresinha,23,,-30.016667,-56.583333
br,estancia tres capoes,Estância Três Capões,23,,-29.6,-56.1
br,estancia vargem grande,Estância Vargem Grande,27,,-23.35,-47.066667
br,estancia varola,Estância Varola,23,,-30.716667,-55.7
br,estancia velha,Estância Velha,23,40391,-29.65,-51.183333
br,estancia vista alegre,Estância Vista Alegre,23,,-30.116667,-57.216667
br,estandarte,Estandarte,13,,-1.433333,-45.533333
br,estanhado,Estanhado,15,,-16.116667,-40.4
br,estanhado,Estanhado,20,,-3.95,-42.766667
br,estefania,Estefânia,23,,-29.25,-52.033333
br,esteio,Esteio,11,,-21.45,-54.65
br,esteio,Esteio,23,84903,-29.85,-51.166667
br,esteios,Esteios,15,,-19.916667,-45.733333
br,esteira,Esteira,23,,-28.633333,-50.85
br,esteirao,Esteirão,04,,-7.283333,-58.6
br,esteja quiteo,Esteja Quiteo,20,,-6.616667,-42.5
br,estela maris,Estela Maris,23,,-32.216667,-52.2
br,estendeslau,Estendeslau,06,,-4.033333,-41.15
br,esterao,Esterão,04,,-7.283333,-58.6
br,estevao de araujo,Estêvão de Araújo,15,,-20.616667,-42.533333
br,estevao,Estêvão,06,,-4.9,-39.15
br,estevao,Estevão,20,,-6.716667,-41.433333
br,estevao,Estêvão,26,,-27.583333,-48.716667
br,estevao velho,Estêvão Velho,05,,-12.4,-39.15
br,esteves junior,Esteves Júnior,26,,-27.433333,-51.85
br,estirao comprido,Estirão Comprido,14,,-16.316667,-55.966667
br,estirao do itapuvucu,Estirão do Itapuvuçu,27,,-24.71097,-48.025893
br,estirao do tejo,Estirão do Tejo,01,,-8.989444,-72.733056
br,estirao do tiro,Estirão do Tiro,04,,-5.861667,-64.947222
br,estirao,Estirão,04,,.366667,-64.05
br,estiro,Estiro,13,,-6.383333,-44.983333
br,estiva da gangati,Estiva da Gangati,13,,-3.183611,-43.451389
br,estiva da josefa,Estiva da Josefa,13,,-3.121667,-43.419722
br,estiva de baixo,Estiva de Baixo,05,,-15,-39.583333
br,estiva do arraial,Estiva do Arraial,13,,-2.75,-44.333333
br,estivado,Estivado,05,,-13.85,-39.583333
br,estivado,Estivado,14,,-14.516667,-56.166667
br,estiva dos vidal,Estiva dos Vidal,18,,-25.73844,-50.45512
br,estiva,Estiva,02,,-9.483333,-36.183333
br,estiva,Estiva,05,,-11.9,-40.566667
br,estiva,Estiva,11,,-19.516667,-51.85
br,estiva,Estiva,13,,-2.75,-44.333333
br,estiva,Estiva,14,,-15.5,-55.5
br,estiva,Estiva,15,,-15.366667,-42.616667
br,estiva,Estiva,17,,-7.183333,-38.266667
br,estiva,Estiva,18,,-23.983333,-51.383333
br,estiva,Estiva,20,,-3.92,-42.673333
br,estiva,Estiva,23,,-29.866667,-53.266667
br,estiva,Estiva,26,,-26.35,-50.05
br,estiva,Estiva,27,,-22.716667,-47.916667
br,estiva,Estiva,29,,-13.733333,-46.683333
br,estiva,Estiva,31,,-10.7,-47.483333
br,estiva i,Estiva I,13,,-3.144722,-42.380556
br,estiva ii,Estiva II,13,,-3.113056,-42.429167
br,estivas,Estivas,02,,-9.066667,-36.166667
br,estivas,Estivas,05,,-13.716667,-42.9
br,estivas,Estivas,13,,-6.233333,-35.216667
br,estivas,Estivas,20,,-6.2,-41.366667
br,estivas,Estivas,22,,-5.666667,-35.25
br,estncia guarema,Estncia Guarema,27,,-23.4081,-46.0089
br,estncia zelia,Estncia Zelia,27,,-23.2989,-46.2245
br,estocado,Estocado,20,,-7.016667,-43.116667
br,estopa,Estopa,13,,-3.150833,-42.953889
br,estrada carolina,Estrada Carolina,26,,-26.7872,-49.0658
br,estrada da mulde,Estrada da Mulde,26,,-26.8247,-49.2355
br,estrada do 29,Estrada do 29,16,,-1.1,-46.516667
br,estrada do campo,Estrada do Campo,23,,-28.733333,-51.266667
br,estrada do quinze,Estrada do Quinze,26,,-25.9917,-48.7077
br,estrada,Estrada,05,,-12.283333,-37.816667
br,estrada,Estrada,06,,-6.366667,-39.15
br,estrada nova,Estrada Nova,21,,-21.8,-42.116667
br,estradaozinho,Estradãozinho,23,,-27.516667,-54.166667
br,estrangeiro,Estrangeiro,17,,-7.233333,-37.5
br,estreito,Estreito,05,,-11.166667,-44.15
br,estreito,Estreito,06,,-3.35,-40.183333
br,estreito,Estreito,11,,-21.45,-54.65
br,estreito,Estreito,13,17647,-5.783333,-43.25
br,estreito,Estreito,16,,-6.983333,-51.716667
br,estreito,Estreito,17,,-7.05,-37.916667
br,estreito,Estreito,18,,-24.669599,-48.86494
br,estreito,Estreito,20,,-3.924722,-42.04
br,estreito,Estreito,22,,-5.233333,-37.35
br,estreito,Estreito,23,,-31.833333,-51.733333
br,estreito,Estreito,30,,-7.983333,-38.2
br,estrela da jordania,Estrêla da Jordânia,15,,-15.866667,-40.283333
br,estrela dalva,Estrêla Dalva,15,,-21.75,-42.433333
br,estrela de jordania,Estrêla de Jordânia,15,,-15.866667,-40.283333
br,estrela d'oeste,Estrela d'Oeste,27,,-20.3,-50.416667
br,estrelado,Estrelado,08,,-20.583333,-41.3
br,estrela do garrafao,Estrêla do Garrafão,15,,-20.65,-46.683333
br,estrela do indaia,Estrêla do Indaiá,15,,-19.516667,-45.783333
br,estrela do leste,Estrêla do Leste,14,,-16.283333,-53.566667
br,estrela do norte,Estrêla do Norte,05,,-14.783333,-39.15
br,estrela do norte,Estrêla do Norte,27,,-22.466667,-51.65
br,estrela do norte,Estrêla do Norte,29,,-13.816667,-49.066667
br,estrela do oeste,Estrêla do Oeste,27,,-20.3,-50.416667
br,estrela do rio,Estrela do Rio,05,,-14.783333,-39.15
br,estrela dos anapurus,Estrêla dos Anapurus,13,,-3.666667,-43.1
br,estrela do sul,Estrela do Sul,11,,-18.4,-53.6
br,estrela do sul,Estrêla do Sul,15,,-18.766667,-47.7
br,estrela,Estrêla,05,,-13.016667,-40.25
br,estrela,Estrela,06,,-3.233333,-39.3
br,estrela,Estrêla,11,,-18.4,-53.6
br,estrela,Estrela,13,,-3.891389,-43.479444
br,estrela,Estrela,14,,-15.333333,-57.233333
br,estrela,Estrêla,15,,-19.516667,-45.783333
br,estrela,Estrela,17,,-6.683333,-37.9
br,estrela,Estrêla,21,,-22.627778,-43.199167
br,estrela,Estrêla,23,24200,-28.1,-50.983333
br,estrela,Estrêla,27,,-21.75,-47.333333
br,estrela matutina,Estrela Matutina,04,,-4.485556,-59.8825
br,estrella,Estrella,15,,-19.516667,-45.783333
br,estrella,Estrella,21,,-22.627778,-43.199167
br,estrella,Estrella,23,,-29.483333,-51.966667
br,estrellas,Estrellas,01,,-8.45,-71.683333
br,estrello do sul,Estrello do Sul,15,,-18.766667,-47.7
br,estrema de pixuna,Estrema de Pixuna,04,,-7.25,-72.233333
br,estrema,Estrema,01,,-10.666667,-70.1
br,estrema,Estrema,04,,-7.316667,-71.166667
br,estrema,Estrema,05,,-10.666667,-42.933333
br,estrema,Estrema,06,,-5.45,-39.9
br,estrema,Estrema,13,,-6.283333,-43.966667
br,estrema,Estrema,15,,-15.4,-44.266667
br,estrema,Estrema,17,,-6.533333,-38.433333
br,estrema,Estrema,20,,-7.866667,-40.7
br,estrema,Estrema,22,,-5.933333,-35.966667
br,estrema,Estrema,23,,-28.183333,-51.133333
br,estrema,Estrema,30,,-7.966667,-38.5
br,estrema,Estrema,31,,-11.081389,-48.159167
br,estreminha,Estreminha,01,,-9.746111,-67.566667
br,estremo,Estremo,17,,-7.8,-38.283333
br,estremoz,Estremoz,22,,-5.7,-35.3
br,estrenta,Estrenta,05,,-13.866667,-40.633333
br,estrias,Estrias,13,,-4.666667,-45.75
br,estrondadeira,Estrondadeira,22,,-5.153611,-36.7925
br,estulandia,Estulândia,29,,-17.266667,-49.05
br,estulania,Estulânia,29,,-17.266667,-49.05
br,etiquira,Etiquira,14,,-17.2,-54.116667
br,eubanque,Eubanque,15,,-21.55,-43.5
br,euca,Euca,03,,2.683333,-50.85
br,eucalipto,Eucalipto,23,,-27.8,-51.4
br,euclidelandia,Euclidelândia,21,,-21.9,-42.266667
br,euclides da cunha,Euclides da Cunha,05,29739,-10.516667,-39.016667
br,euclides da cunha,Euclides da Cunha,21,,-21.766667,-42.366667
br,euclydes malta,Euclydes Malta,02,,-9.416667,-36.066667
br,eucum,Eucum,05,,-13.983333,-41.2
br,eufrasina,Eufrasina,18,,-25.4599,-48.5761
br,eufrosina,Eufrosina,18,,-25.933333,-50.683333
br,eufrozina,Eufrozina,18,,-25.933333,-50.683333
br,eugenio de castro,Eugênio de Castro,23,,-28.55,-54.05
br,eugenio de melo,Eugênio de Melo,27,,-23.15,-45.783333
br,eugenio,Eugênio,02,,-9.666667,-36.383333
br,eugenio penzo,Eugênio Penzo,11,,-22.216667,-55.883333
br,eugenopolis,Eugenópolis,15,,-21.1,-42.183333
br,eureca,Eureca,15,,-16.633333,-41.1
br,euriapolis,Euriápolis,05,,-16.366667,-39.583333
br,eurico almeida,Eurico Almeida,15,,-16.633333,-41.033333
br,eurico lima,Eurico Lima,25,,3.433333,-60.633333
br,eurico martelete,Eurico Martelete,23,,-28.6,-53.516667
br,eurico martellet,Eurico Martellet,23,,-28.6,-53.516667
br,europa,Europa,01,,-10.616667,-68.266667
br,europa,Europa,04,,-3.26,-62.856667
br,eusebio de oliveira,Eusébio de Oliveira,18,,-23.8107,-50.3623
br,eusebio,Eusébio,06,37290,-3.9,-38.45
br,eusebio,Eusébio,15,,-15.3,-43.083333
br,euxenita,Euxenita,15,,-18.55,-43.033333
br,euzebio oliveira,Euzebio Oliveira,18,,-23.8107,-50.3623
br,euziebio,Euziebio,06,,-3.9,-38.45
br,eva,Eva,25,,3.15,-59.933333
br,evangelista de sousa,Evangelista de Sousa,27,,-23.9263,-46.6505
br,evangelista,Evangelista,23,,-28.633333,-51.95
br,evangelista,Evangelista,27,,-23.9114,-46.6602
br,evarista,Evarista,23,,-31.2,-52.233333
br,evaristo,Evaristo,15,,-21.933333,-45.766667
br,evaristo,Evaristo,23,,-31.2,-52.233333
br,evaristo melo,Evaristo Melo,15,,-21.166667,-46.75
br,ewbank,Ewbank,15,,-21.55,-43.5
br,exaltacao,Exaltação,04,,-5.646667,-61.156389
br,exapora,Exaporã,27,,-22.433333,-50.2
br,excelsior,Excelsior,21,,-22.941133,-43.271241
br,ex. novo treviso,Ex. Novo Treviso,23,,-28.95,-51.383333
br,expedicionario alicio,Expedicionário Alício,15,,-19.65,-41.3
br,expedito lopes,Expedito Lopes,20,,-6.95,-41.633333
br,extracao,Extração,15,,-18.283333,-43.533333
br,extraccao,Extracção,15,,-18.283333,-43.533333
br,extrato,Extrato,04,,-4.1725,-60.596389
br,extrema de pixuma,Extrema de Pixuma,04,,-7.25,-72.233333
br,extrema,Extrema,01,,-10.666667,-70.1
br,extrema,Extrema,04,,-6.876389,-70.671111
br,extrema,Extrema,05,,-10.666667,-42.933333
br,extrema,Extrema,06,,-6.816667,-39.233333
br,extrema,Extrema,13,,-6.283333,-43.966667
br,extrema,Extrema,15,13682,-22.85,-46.316667
br,extrema,Extrema,20,,-3.604444,-42.294167
br,extrema,Extrema,23,,-28.183333,-51.133333
br,extrema,Extrema,31,,-11.081389,-48.159167
br,extremas,Extremas,20,,-5.166667,-41.816667
br,extremos,Extremós,22,,-5.7,-35.3
br,extremoz,Extremóz,22,17992,-5.7,-35.3
br,exu,Exú,06,,-5.133333,-38.333333
br,exu,Exu,17,,-6.8,-38.283333
br,exu,Exu,20,,-5.216667,-41.633333
br,exu,Exu,22,,-6.133333,-37.8
br,exu,Exu,30,10875,-7.516667,-39.716667
br,exu velho,Exu Velho,30,,-7.416667,-39.716667
br,ezeperio almeida,Ezepério Almeida,15,,-16.266667,-41.683333
br,ezequiel,Ezequiel,11,,-22.05,-54.45
br,ezequiel,Ezequiel,16,,-1.95,-49.1
br,fabiao,Fabião,15,,-15.133333,-44.233333
br,fabrica das chitas,Fábrica das Chitas,21,,-22.929977,-43.230479
br,fabrica,Fábrica,30,,-8.75,-36.866667
br,fabrica santa barbara,Fábrica Santa Bárbara,15,,-18.016667,-44.116667
br,facao,Facão,06,,-5.483333,-39.183333
br,facao,Fação,13,,-3.421389,-42.961944
br,facao,Facão,14,,-16.166667,-57.6
br,facao,Facão,26,,-26.874552,-49.760227
br,facao,Facão,27,,-22.4,-47.383333
br,fachinal do sao joao,Fachinal do Sao Joao,26,,-27.033333,-51.716667
br,fachinal dos guedes,Fachinal dos Guedes,26,,-26.883333,-52.233333
br,fachinal,Fachinal,23,,-28.066667,-53.7
br,fagundes,Fagundes,06,,-3.95,-38.35
br,fagundes,Fagundes,08,,-18.7,-40.816667
br,fagundes,Fagundes,16,,-7.216667,-55.633333
br,fagundes,Fagundes,17,5137,-7.35,-35.8
br,fagundes,Fagundes,22,,-5.228611,-36.277222
br,fagundes varela,Fagundes Varela,23,,-28.883333,-51.7
br,faisca,Faisca,06,,-4.283333,-38.783333
br,faisqueirinha,Faisqueirinha,18,,-25.423675,-48.767161
br,faisqueiro,Faisqueiro,17,,-6.933333,-38.45
br,falas,Falas,17,,-7.616667,-37.85
br,fala verdade,Fala Verdade,11,,-19.666667,-54.783333
br,falcao,Falcão,21,,-22.2877,-44.2531
br,falcao,Falcão,31,,-5.933333,-48.216667
br,falha,Falha,13,,-9.283333,-46.801389
br,fama,Fama,15,,-21.4,-45.833333
br,fama,Fama,30,,-8.783333,-36.433333
br,faminha,Faminha,08,,-20.216667,-41.733333
br,fangueira,Fangueira,15,,-20.366667,-44.633333
br,fangueiro,Fangueiro,15,,-20.366667,-44.633333
br,fao,Fão,23,,-29.2,-52.233333
br,faral,Faral,05,,-18.083333,-39.633333
br,faria,Faria,02,,-9.616667,-37.333333
br,faria,Faria,30,,-8.133333,-36.3
br,faria lemas,Faria Lemas,15,,-20.8,-42.016667
br,faria lemos,Faria Lemos,15,,-20.8,-42.016667
br,faria lemos,Faria Lemos,23,,-29.2454,-51.075
br,farias brito,Farias Brito,06,11288,-6.916667,-39.566667
br,farias,Farias,06,,-7.35,-39.4
br,farias,Farias,15,,-19.016667,-42.833333
br,farias,Farias,17,,-7.383333,-36.666667
br,farias,Farias,22,,-5.183333,-35.883333
br,farias,Farias,23,,-29.066667,-52.733333
br,farias,Farias,28,,-9.866667,-37.25
br,faria velho,Faria Velho,15,,-17.733333,-44.65
br,farinha,Farinha,05,,-13.85,-44.816667
br,farinha,Farinha,17,,-7.116667,-36.133333
br,farinha,Farinha,18,,-23.35,-51
br,farinha,Farinha,23,,-27.183333,-53.133333
br,farinha molhada,Farinha Molhada,05,,-13.6,-41.4
br,farinhas,Farinhas,17,,-7.083333,-37.233333
br,farinhas,Farinhas,23,,-27.183333,-53.133333
br,farinna,Farinna,17,,-7.116667,-36.133333
br,faro,Faro,04,,-7.316667,-62.933333
br,faro,Faro,16,,-2.133333,-56.733056
br,farol,Farol,05,,-12.566667,-38
br,farol,Farol,18,,-24.1,-52.666667
br,farpao,Farpão,15,,-16.1,-40.05
br,farroupilha,Farroupilha,23,57651,-29.233333,-51.35
br,fartura,Fartura,17,,-7.55,-38.4
br,fartura,Fartura,18,,-25.5208,-48.9002
br,fartura,Fartura,20,,-2.994722,-41.315278
br,fartura,Fartura,26,,-26.166667,-50.55
br,fartura,Fartura,27,12405,-23.383333,-49.5
br,fasola,Fasola,30,,-8.633333,-37.133333
br,fatima,Fátima,05,,-10.65,-38.233333
br,fatima,Fátima,14,,-16.183333,-54.966667
br,fatima,Fátima,17,,-6.95,-38.716667
br,fatima,Fatima,21,,-22.917491,-43.188336
br,fatima,Fátima,30,,-7.95,-37.783333
br,fatima,Fátima,31,,-10.764444,-48.827222
br,fau,Faú,27,,-24.25,-47.45
br,faustino,Faustino,05,,-12.883333,-40.083333
br,faustino,Faustino,06,,-7.05,-39.516667
br,faustino,Faustino,13,,-3.359167,-43.432778
br,faustino,Faustino,17,,-7.583333,-38.466667
br,fausto,Fausto,03,,.316667,-51.7
br,fausto,Fausto,21,,-22.433333,-41.95
br,fava,Fava,20,,-7.466667,-41.666667
br,faval,Faval,14,,-15.65,-56.783333
br,favania,Favânia,16,,-3.116667,-51.8
br,faveira,Faveira,02,,-8.966667,-37.766667
br,faveira,Faveira,06,,-4.05,-40.983333
br,faveira,Faveira,13,,-3.787222,-42.822778
br,faveira,Faveira,20,,-3.888611,-42.398333
br,faveirinha,Faveirinha,13,,-3.136667,-42.203889
br,favela,Favela,05,,-10.15,-41.65
br,favela,Favela,06,,-6.3,-40.3
br,favela,Favela,17,,-6.983333,-37.183333
br,favela,Favela,22,,-5.2925,-37.170833
br,favela,Favela,24,,-10.283333,-63.7
br,favela,Favela,30,,-8.216667,-37.516667
br,favelas,Favelas,20,,-7.116667,-40.683333
br,favella,Favella,05,,-10.15,-41.65
br,favella,Favella,24,,-10.283333,-63.7
br,favera,Favera,20,,-6.3,-42.3
br,favinhas,Favinhas,30,,-7.766667,-39.483333
br,favonio,Favónio,04,,-2.683333,-66.75
br,favorita,Favorita,05,,-17.95,-40.033333
br,faxina,Faxina,27,,-20.55,-47.783333
br,faxinal chagu,Faxinal Chagu,18,,-25.366667,-52.583333
br,faxinal da pinguela,Faxinal da Pinguela,18,,-24.95,-50.55
br,faxinal de catanduva,Faxinal de Catanduva,18,,-24.633333,-51.266667
br,faxinal de catanduvas,Faxinal de Catanduvas,18,,-24.633333,-51.266667
br,faxinal de santo antonio,Faxinal de Santo Antônio,18,,-24.716667,-51.283333
br,faxinal de sao joao,Faxinal de São João,18,,-25.116667,-52.733333
br,faxinal de sao pedro,Faxinal de São Pedro,18,,-25.710737,-50.727342
br,faxinal de sao pedro,Faxinal de São Pedro,26,,-26.833333,-50.9
br,faxinal de sao sebastiao,Faxinal de São Sebastião,18,,-23.983333,-51.366667
br,faxinal do louro,Faxinal do Louro,18,,-25.5882,-50.2713
br,faxinal do rio-acima,Faxinal do Rio-Acima,27,,-23.7425,-47.9107
br,faxinal dos andrades,Faxinal dos Andrades,18,,-25.722389,-50.244229
br,faxinal dos augustos,Faxinal dos Augustos,18,,-25.27,-50.6539
br,faxinal dos castilhos,Faxinal dos Castilhos,18,,-25.85,-49.85
br,faxinal dos correas,Faxinal dos Corrêas,18,,-25.7129,-49.9362
br,faxinal dos elois,Faxinal dos Elóis,18,,-24.016667,-51.433333
br,faxinal dos fabricios,Faxinal dos Fabrícios,18,,-25.58318,-50.39106
br,faxinal dos fereiras,Faxinal dos Fereiras,18,,-25.699723,-50.234788
br,faxinal dos ferreiras,Faxinal dos Ferreiras,18,,-25.690917,-50.27085
br,faxinal dos francos,Faxinal dos Francos,18,,-25.596448,-50.587398
br,faxinal dos guedes,Faxinal dos Guedes,26,,-26.883333,-52.233333
br,faxinal do silva,Faxinal do Silva,18,,-25.583333,-50.166667
br,faxinal dos indios,Faxinal dos Índios,18,,-25.166667,-52.433333
br,faxinal dos lapeanos,Faxinal dos Lapeanos,18,,-25.661897,-50.484193
br,faxinal dos mineiros de baixo,Faxinal dos Mineiros de Baixo,18,,-25.52475,-50.39104
br,faxinal dos mineiros de cima,Faxinal dos Mineiros de Cima,18,,-25.5366,-50.41288
br,faxinal dos mineiros,Faxinal dos Mineiros,18,,-25.53873,-50.33662
br,faxinal do soturno,Faxinal do Soturno,23,,-29.616667,-53.433333
br,faxinal dos paula,Faxinal dos Paula,18,,-25.692752,-50.737806
br,faxinal dos pintos,Faxinal dos Pintos,18,,-25.54389,-50.44975
br,faxinal dos pretos,Faxinal dos Pretos,18,,-25.855482,-49.791815
br,faxinal dos quartins,Faxinal dos Quartins,18,,-25.55826,-50.115547
br,faxinal dos rodrigues,Faxinal dos Rodrigues,18,,-25.721388,-50.221168
br,faxinal dos santos,Faxinal dos Santos,18,,-26.283333,-51.5
br,faxinal dos silvas,Faxinal dos Silvas,18,,-25.573102,-50.13258
br,faxinal dos vieiras,Faxinal dos Vieiras,18,,-25.62945,-50.612387
br,faxinal,Faxinal,18,13464,-23.983333,-51.366667
br,faxinal,Faxinal,23,,-29.616667,-53.433333
br,faxinal,Faxinal,26,,-26.8814,-49.9338
br,faxinal,Faxinal,27,,-23.7034,-47.8336
br,faxinal preto,Faxinal Prêto,23,,-28.516667,-49.883333
br,faxinal sao joao,Faxinal São João,18,,-25.116667,-52.733333
br,faxinalsinho,Faxinalsinho,18,,-24,-51.45
br,faxinalzinho do rio azul,Faxinalzinho do Rio Azul,18,,-24,-51.45
br,faxinalzinho,Faxinalzinho,18,,-24,-51.45
br,faz dos fojos,Faz dos Fojos,18,,-25.761369,-50.089885
br,fazenda acude das melancias,Fazenda Açude das Melancias,06,,-4.75,-38.283333
br,fazenda acude grande,Fazenda Açude Grande,06,,-4.95,-40.333333
br,fazenda acude velho,Fazenda Açude Velho,06,,-3.6,-39.366667
br,fazenda acudinho,Fazenda Açudinho,06,,-3.483333,-39.266667
br,fazenda agostinho,Fazenda Agostinho,27,,-21.4,-47.55
br,fazenda agua boa,Fazenda Água Boa,06,,-3.466667,-39.3
br,fazenda agua branca,Fazenda Água Branca,06,,-3.866667,-39.766667
br,fazenda agua branca,Fazenda Agua Branca,14,,-16.25,-55.466667
br,fazenda agua branca,Fazenda Água Branca,15,,-16.183333,-41.466667
br,fazenda agua clara,Fazenda Água Clara,11,,-21.9,-54.183333
br,fazenda agua limpa,Fazenda Agua Limpa,11,,-21.716667,-52.666667
br,fazenda agua limpa,Fazenda Agua Limpa,29,,-17.1,-50.116667
br,fazenda agua limpo,Fazenda Agua Limpo,11,,-19.35,-54.7
br,fazenda alavanca,Fazenda Alavanca,11,,-21.55,-54.3
br,fazenda alceu garcia,Fazenda Alceu Garcia,15,,-18.55,-49.183333
br,fazenda alcidio,Fazenda Alcídio,27,,-21.033333,-47.55
br,fazenda aldeia,Fazenda Aldeia,11,,-18.2,-55.166667
br,fazenda alegre,Fazenda Alegre,06,,-4.233333,-40.383333
br,fazenda alegre,Fazenda Alegre,11,,-17.633333,-56.966667
br,fazenda alegre,Fazenda Alegre,14,,-15.247778,-59.833333
br,fazenda alegrete,Fazenda Alegrete,11,,-21.383333,-54.75
br,fazenda alegria,Fazenda Alegria,06,,-3.833333,-39.516667
br,fazenda alianca,Fazenda Aliança,06,,-3.866667,-39.266667
br,fazenda alianca,Fazenda Aliança,11,,-19.05,-55.266667
br,fazenda alivio,Fazenda Alívio,06,,-4.55,-39.033333
br,fazenda alto da copeira,Fazenda Alto da Copeira,20,,-7.083333,-42.233333
br,fazenda amazonas,Fazenda Amazonas,25,,3.366667,-61.383333
br,fazenda amescla,Fazenda Amescla,20,,-5.6,-42.583333
br,fazenda angical,Fazenda Angical,20,,-5.066667,-41.933333
br,fazenda angico,Fazenda Angico,27,,-21.783333,-47.733333
br,fazenda angicos,Fazenda Angicos,06,,-4.316667,-39.566667
br,fazenda angico torto,Fazenda Angico Torto,06,,-3.75,-40.383333
br,fazenda anhumas,Fazenda Anhumas,14,,-16.933333,-54.666667
br,fazenda ano novo,Fazenda Ano Novo,11,,-21.8,-54.15
br,fazenda ano nuevo,Fazenda Ano Nuevo,11,,-21.8,-54.15
br,fazenda antonio bastos,Fazenda Antônio Bastos,23,,-29.983333,-56.883333
br,fazenda antonio,Fazenda Antônio,20,,-6.1,-41.916667
br,fazenda antonio lima,Fazenda Antônio Lima,15,,-17.366667,-40.666667
br,fazenda antonio rocha,Fazenda Antônio Rocha,15,,-17.416667,-41.5
br,fazenda aparecida,Fazenda aparecida,27,,-20.25,-47.8
br,fazenda apertada,Fazenda Apertada,06,,-4.633333,-38.05
br,fazenda aprazivel,Fazenda Aprazível,06,,-4.183333,-40.266667
br,fazenda aqua limpo,Fazenda Aqua Limpo,11,,-19.35,-54.7
br,fazenda aquario,Fazenda Aquário,01,,-9.983333,-67.15
br,fazenda aquiri,Fazenda Aquiri,06,,-4.383333,-40.116667
br,fazenda arapuca,Fazenda Arapuca,14,,-16.316111,-56.558056
br,fazenda arara,Fazenda Arara,01,,-9.916667,-67.45
br,fazenda arara,Fazenda Arara,30,,-7.35,-39.733333
br,fazenda argentina,Fazenda Argentina,27,,-21.85,-47.8
br,fazenda arirao,Fazenda Arirão,06,,-4.3,-39.466667
br,fazenda armador,Fazenda Armador,06,,-3.916667,-39.85
br,fazenda aroeira,Fazenda Aroeira,06,,-3.516667,-39.35
br,fazenda arraia,Fazenda Arraia,06,,-3.766667,-39.883333
br,fazenda arroz,Fazenda Arroz,20,,-7.083333,-41.3
br,fazenda artur,Fazenda Artur,18,,-23.314702,-52.713582
br,fazenda asa os pedra,Fazenda Asa os Pedra,06,,-4.25,-39.016667
br,fazenda atalho,Fazenda Atalho,06,,-3.833333,-40.3
br,fazenda atalho,Fazenda Atalho,21,,-22.55,-42.15
br,fazenda atanacio,Fazenda Atanácio,11,,-21.516667,-54.15
br,fazenda aterradinho,Fazenda Aterradinho,27,,-23.45,-48.566667
br,fazenda aurora,Fazenda Aurora,11,,-19.2,-55.483333
br,fazenda bacuri,Fazenda Bacuri,20,,-5.7,-42.8
br,fazenda baia dos passaros,Fazenda Baia dos Passaros,14,,-16.6,-55.866667
br,fazenda baile,Fazenda Baile,11,,-22.233333,-53.4
br,fazenda baixa dos lucas,Fazenda Baixa dos Lucas,06,,-4.566667,-38.133333
br,fazenda baixa grande,Fazenda Baixa Grande,06,,-3.45,-40.3
br,fazenda baixa grande,Fazenda Baixa Grande,20,,-6,-42.666667
br,fazenda baixao,Fazenda Baixão,27,,-21.366667,-47.483333
br,fazenda balbino dias,Fazenda Balbino Dias,15,,-16.95,-40.716667
br,fazenda balsamo,Fazenda Balsamo,11,,-18.05,-54.05
br,fazenda bandeira,Fazenda Bandeira,27,,-21.883333,-47.666667
br,fazenda barao,Fazenda Barão,27,,-21.7,-46.866667
br,fazenda barbada,Fazenda Barbada,06,,-3.3,-39.85
br,fazenda baronesa,Fazenda Baronesa,06,,-3.583333,-40.216667
br,fazenda barra das tabocas,Fazenda Barra das Tabocas,06,,-3.4,-39.75
br,fazenda barra do jua,Fazenda Barra do Juá,06,,-4.316667,-39.866667
br,fazenda barra do leao,Fazenda Barra do Leão,06,,-4.3,-39.066667
br,fazenda barra do lemos,Fazenda Barra do Lemos,06,,-3.933333,-39.05
br,fazenda barragem,Fazenda Barragem,06,,-3.15,-40.283333
br,fazenda barragem,Fazenda Barragem,30,,-7.966667,-39.833333
br,fazenda barranco alto,Fazenda Barranco Alto,11,,-18.216667,-55.483333
br,fazenda barra nova,Fazenda Barra Nova,06,,-3.983333,-39.35
br,fazenda barreira,Fazenda Barreira,06,,-3.7,-39.716667
br,fazenda barreiro,Fazenda Barreiro,11,,-18.033333,-54.166667
br,fazenda barreiro,Fazenda Barreiro,27,,-22.216667,-47.116667
br,fazenda barrero,Fazenda Barrero,11,,-18.366667,-55.983333
br,fazenda barreto,Fazenda Barreto,06,,-3.866667,-39.083333
br,fazenda barriga,Fazenda Barriga,06,,-3.516667,-39.266667
br,fazenda barrigas,Fazenda Barrigas,06,,-4.55,-39.75
br,fazenda barrinha,Fazenda Barrinha,06,,-4.033333,-39.116667
br,fazenda barro prato,Fazenda Barro Prato,27,,-20.816667,-47.8
br,fazenda barro vermelho,Fazenda Barro Vermelho,06,,-4,-40.016667
br,fazenda batalha,Fazenda Batalha,08,,-21.1,-41.016667
br,fazenda batalho,Fazenda Batalho,08,,-21.1,-41.016667
br,fazenda batatal,Fazenda Batatal,27,,-24.605435,-48.294138
br,fazenda batedor,Fazenda Batedor,06,,-3.483333,-39.85
br,fazenda bebedouro,Fazenda Bebedouro,27,,-20.233333,-47.733333
br,fazenda belas artes,Fazenda Belas Artes,11,,-21.683333,-54.383333
br,fazenda bela vista,Fazenda Bela Vista,15,,-22.05,-46.15
br,fazenda bela vista,Fazenda Bela Vista,23,,-30.933333,-55.7
br,fazenda bela vista,Fazenda Bela Vista,27,,-20.033333,-47.85
br,fazenda belmonte,Fazenda Belmonte,20,,-7.066667,-41.183333
br,fazenda belo horizonte,Fazenda Belo Horizonte,11,,-21.316667,-54.483333
br,fazenda belo horizonte,Fazenda Belo Horizonte,20,,-5.066667,-42.216667
br,fazenda benita,Fazenda Benita,11,,-17.9,-54.583333
br,fazenda benlo,Fazenda Benlo,11,,-17.983333,-54.9
br,fazenda benta,Fazenda Benta,11,,-17.9,-54.583333
br,fazenda betania,Fazenda Betânia,06,,-3.733333,-39.85
br,fazenda betania,Fazenda Betãnia,20,,-7,-42.05
br,fazenda bete,Fazenda Bete,06,,-4.116667,-39.05
br,fazenda beto horizonte,Fazenda Beto Horizonte,06,,-4.366667,-40
br,fazenda boa esperanca,Fazenda Boa Esperanca,06,,-4.383333,-38.266667
br,fazenda boa esperanca,Fazenda Boa Esperanca,30,,-7.766667,-40.166667
br,fazenda boa sorte,Fazenda Bôa Sorte,11,,-21.633333,-57.166667
br,fazenda boa sorte,Fazenda Boa Sorte,14,,-15.077778,-59.874722
br,fazenda boa vista,Fazenda Boa Vista,04,,-9.15,-67.3
br,fazenda boa vista,Fazenda Boa Vista,06,,-3.4,-40.216667
br,fazenda boa vista,Fazenda Bõa Vista,11,,-17.85,-54.216667
br,fazenda boa vista,Fazenda Boa Vista,15,,-17.85,-40.833333
br,fazenda boa vista,Fazenda Boa Vista,20,,-5.483333,-42.166667
br,fazenda boa vista,Fazenda Boa Vista,27,,-20.166667,-47.7
br,fazenda boa vista,Fazenda Boa Vista,30,,-7.933333,-39.716667
br,fazenda bocaina,Fazenda Bocaina,27,,-20.5,-47.716667
br,fazenda bofe,Fazenda Bofe,30,,-7.333333,-39.75
br,fazenda boleri,Fazenda Boleri,11,,-22.616667,-55.583333
br,fazenda bolery,Fazenda Bolery,11,,-22.616667,-55.583333
br,fazenda bom fim,Fazenda Bom Fim,27,,-22.966667,-46.116667
br,fazenda bom futuro,Fazenda Bom Futuro,06,,-3.816667,-40.3
br,fazenda bom jardim,Fazenda Bom Jardim,06,,-4.716667,-38.266667
br,fazenda bom jesus,Fazenda Bom Jesus,14,,-16.343056,-56.913611
br,fazenda bom jesus,Fazenda Bom Jesus,20,,-5.583333,-41.6
br,fazenda bom lugar,Fazenda Bom Lugar,06,,-4.116667,-39.6
br,fazenda bom sucesso,Fazenda Bom sucesso,27,,-21.55,-47.016667
br,fazenda bonanca,Fazenda Bonança,06,,-3.533333,-39.05
br,fazenda bonfim,Fazenda Bonfim,06,,-3.5,-39.783333
br,fazenda bonito,Fazenda Bonito,20,,-5.316667,-42.033333
br,fazenda boqueirao,Fazenda Boqueirão,06,,-4.383333,-40.25
br,fazenda boqueirao,Fazenda Boqueirão,20,,-5.1,-41.55
br,fazenda boquirao,Fazenda Boquirão,20,,-5.033333,-42.133333
br,fazenda borery,Fazenda Borery,11,,-22.616667,-55.583333
br,fazenda brasil,Fazenda Brasil,27,,-21.583333,-46.9
br,fazenda brejao,Fazenda Brejao,11,,-21.666667,-54.6
br,fazenda brilhante,Fazenda Brilhante,11,,-20.5,-54.2
br,fazenda brioso,Fazenda Brioso,11,,-19.483333,-54.5
br,fazenda brito,Fazenda Brito,06,,-3.833333,-39.85
br,fazenda bueno,Fazenda Bueno,06,,-3.616667,-39.85
br,fazenda buriti,Fazenda Burití,11,,-18.25,-54.5
br,fazenda buriti,Fazenda Buriti,14,,-15.383333,-55.833333
br,fazenda buritizinho,Fazenda Buritizinho,20,,-6.083333,-41.966667
br,fazenda burity,Fazenda Burity,14,,-15.383333,-55.833333
br,fazenda cabaceiras,Fazenda Cabaceiras,06,,-4.45,-40.183333
br,fazenda cabeca do bol,Fazenda Cabeça do Bol,06,,-3.866667,-39.833333
br,fazenda cacadas,Fazenda Caçadas,06,,-4.7,-39.066667
br,fazenda cachoeira do umbelino,Fazenda Cachoeira do Umbelino,06,,-4.55,-39.516667
br,fazenda cachoeira,Fazenda Cachoeira,06,,-3.65,-40.366667
br,fazenda cachoeira,Fazenda Cachoeira,11,,-19.55,-54.583333
br,fazenda cachoeira,Fazenda Cachoeira,14,,-15.9,-58.366667
br,fazenda cachoeira,Fazenda Cachoeira,15,,-20.666667,-45.3
br,fazenda cachoeira,Fazenda Cachoeira,27,,-20.2,-47.416667
br,fazenda cachoeirinha,Fazenda Cachoeirinha,06,,-3.683333,-39.816667
br,fazenda cachoeirinha,Fazenda Cachoeirinha,11,,-18.166667,-54.25
br,fazenda cachoeirinha,Fazenda Cachoeirinha,24,,-10.268056,-62.698611
br,fazenda cachoeirinha,Fazenda Cachoeirinha,27,,-21.666667,-47.533333
br,fazenda cacimba de cima,Fazenda Cacimba de Cima,06,,-4.033333,-39.316667
br,fazenda cacimba,Fazenda Cacimba,06,,-3.733333,-39.383333
br,fazenda cacimba funda,Fazenda Cacimba Funda,06,,-3.95,-39.583333
br,fazenda cacimba salgada,Fazenda Cacimba Salgada,06,,-3.766667,-39.833333
br,fazenda cacimbas,Fazenda Cacimbas,20,,-5.033333,-42.083333
br,fazenda cacimbinha,Fazenda Cacimbinha,06,,-3.45,-40.383333
br,fazenda cadro,Fazenda Cadro,06,,-3.783333,-39.1
br,fazenda cafe,Fazenda Café,27,,-20.45,-47.8
br,fazenda cafundo,Fazenda Cafundó,06,,-6.083333,-39.516667
br,fazenda caicara,Fazenda Caiçara,06,,-4.8,-39.8
br,fazenda caicarinha,Fazenda Caiçarinha,06,,-3.95,-39.833333
br,fazenda cajazeira,Fazenda Cajazeira,06,,-3.116667,-39.916667
br,fazenda cajazeira,Fazenda Cajazeira,24,,-10.091667,-62.969444
br,fazenda cajazeiras,Fazenda Cajazeiras,06,,-3.733333,-40.183333
br,fazenda cajueiro grande,Fazenda Cajueiro Grande,06,,-3.15,-39.95
br,fazenda cajui,Fazenda Cajuí,06,,-7.05,-39.066667
br,fazenda calabria,Fazenda Calábria,06,,-4.1,-39.083333
br,fazenda calcadinho,Fazenda Calcadinho,20,,-7.183333,-42.85
br,fazenda caldeirao,Fazenda Caldeirão,20,,-7.3,-41.133333
br,fazenda california,Fazenda Califórnia,27,,-21.65,-47.483333
br,fazenda calumbi,Fazenda Calumbi,20,,-5.433333,-41.916667
br,fazenda caminhadeira,Fazenda Caminhadeira,06,,-3.983333,-39.883333
br,fazenda campeira,Fazenda Campeira,13,,-9.492222,-45.931389
br,fazenda campestre,Fazenda Campestre,06,,-4.8,-39.016667
br,fazenda campina,Fazenda Campina,06,,-4.566667,-40.183333
br,fazenda campinas,Fazenda Campinas,06,,-3.766667,-39.716667
br,fazenda campinho,Fazenda Campinho,27,,-20.7,-47.716667
br,fazenda campo alegre,Fazenda Campo Alegre,11,,-19.766667,-55.016667
br,fazenda campo alegre,Fazenda Campo Alegre,14,,-16.116667,-58.05
br,fazenda campo alegre,Fazenda Campo Alegre,27,,-21.05,-47.866667
br,fazenda campo limpo,Fazenda Campo Limpo,11,,-17.933333,-54.633333
br,fazenda campo lindo,Fazenda Campo Lindo,20,,-7.9,-40.85
br,fazenda canaa,Fazenda Canaã,20,,-6,-41.95
br,fazenda canabrava,Fazenda Canabrava,06,,-4.75,-40.2
br,fazenda cana brava,Fazenda Cana Brava,06,,-6.983333,-39.55
br,fazenda canafistula,Fazenda Canafistula,06,,-3.4,-40.1
br,fazenda canafistula,Fazenda Canafistula,30,,-7.383333,-39.633333
br,fazenda cangalha,Fazenda Cangalha,11,,-19.55,-52.8
br,fazenda canto alegre,Fazenda Canto Alegre,20,,-5,-42.25
br,fazenda capao alto,Fazenda capão Alto,11,,-18.9,-54.866667
br,fazenda capao bonito,Fazenda Capão Bonito,11,,-21.366667,-54.4
br,fazenda capao da cruz,Fazenda capão da Cruz,27,,-21.433333,-47.9
br,fazenda capao seco,Fazenda Capão Seco,27,,-20.833333,-47.783333
br,fazenda capao verde,Fazenda Capão Verde,14,,-16.134444,-56.746111
br,fazenda capela,Fazenda Capela,11,,-20.466667,-56.383333
br,fazenda capella,Fazenda Capella,11,,-20.466667,-56.383333
br,fazenda capim acu,Fazenda Capim Açu,06,,-3.7,-39.766667
br,fazenda capim grande,Fazenda Capim Grande,20,,-7.483333,-40.833333
br,fazenda capim grosso,Fazenda Capim Grosso,30,,-7.916667,-40.1
br,fazenda capivari,Fazenda Capivari,27,,-23.939,-46.6811
br,fazenda cara,Fazenda Cara,06,,-3.816667,-39.45
br,fazenda caraibas,Fazenda Caraibas,20,,-5.483333,-42.4
br,fazenda caraibas,Fazenda Caraibas,30,,-7.966667,-40.116667
br,fazenda carao,Fazenda Carão,06,,-3.316667,-40.383333
br,fazenda cararu,Fazenda Cararu,06,,-3.783333,-38.883333
br,fazenda carauba,Fazenda Caraúba,06,,-3.333333,-39.983333
br,fazenda caraubas,Fazenda Caraúbas,06,,-4.083333,-39.5
br,fazenda carima,Fazenda Carima,06,,-4.483333,-40.433333
br,fazenda carius,Fazenda Cariús,06,,-4.483333,-38.166667
br,fazenda carlos lobato,Fazenda Carlos Lobato,15,,-20.05,-45.583333
br,fazenda carnaiba,Fazenda Carnaiba,20,,-5.7,-42.1
br,fazenda carnauba,Fazenda Carnaúba,06,,-3.816667,-39.3
br,fazenda carnaubinha,Fazenda Carnaubinha,06,,-4.216667,-39.983333
br,fazenda carneiro,Fazenda Carneiro,06,,-4.2,-39.033333
br,fazenda carreiras,Fazenda Carreiras,06,,-4.2,-40.233333
br,fazenda caruaru,Fazenda Caruaru,06,,-4.116667,-40.333333
br,fazenda casa de pedra,Fazenda Casa de Pedra,06,,-3.75,-39.066667
br,fazenda casa nova,Fazenda Casa Nova,06,,-4.25,-39.133333
br,fazenda casa nova,Fazenda Casa Nova,20,,-7.75,-42.8
br,fazenda cascavel,Fazenda Cascavel,06,,-4.516667,-40.45
br,fazenda casinha,Fazenda Casinha,06,,-3.666667,-40.083333
br,fazenda castanheiro,Fazenda Castanheiro,20,,-7.583333,-42.083333
br,fazenda castro,Fazenda Castro,06,,-4.766667,-39.366667
br,fazenda catalao,Fazenda Catalão,06,,-4.45,-38.266667
br,fazenda catuana,Fazenda Catuana,06,,-4.633333,-40.2
br,fazenda caua,Fazenda Cauã,06,,-3.833333,-39.65
br,fazenda central,Fazenda Central,06,,-4.9,-39.5
br,fazenda cervo,Fazenda Cervo,11,,-21.683333,-54.966667
br,fazenda chagu,Fazenda Chagu,18,,-25.366667,-52.583333
br,fazenda chaparral,Fazenda Chaparral,04,,-9.066667,-67.25
br,fazenda chapena,Fazenda chapena,11,,-20.466667,-56.383333
br,fazenda chapeu,Fazenda Chapéu,06,,-4.883333,-39.35
br,fazenda chaves,Fazenda Chaves,06,,-4.266667,-38.483333
br,fazenda chupeiro,Fazenda Chupeiro,20,,-7.5,-40.866667
br,fazenda cidade,Fazenda Cidade,06,,-3.916667,-39.866667
br,fazenda cipo,Fazenda Cipó,06,,-3.916667,-39.2
br,fazenda coite,Fazenda Coité,06,,-3.616667,-38.783333
br,fazenda colombia,Fazenda Colômbia,27,,-21.883333,-47.85
br,fazenda conceicao,Fazenda Conceição,06,,-3.5,-39.4
br,fazenda conceicao,Fazenda Conceição,15,,-16.416667,-40.633333
br,fazenda conceicao,Fazenda Conceição,27,,-21.5,-46.766667
br,fazenda constanca,Fazenda Constança,27,,-21.383333,-46.8
br,fazenda consulta,Fazenda Consulta,06,,-3.833333,-39.666667
br,fazenda contenda,Fazenda Contenda,06,,-4.933333,-38.583333
br,fazenda contentes,Fazenda Contentes,22,,-4.983333,-37.316667
br,fazenda coqueiros,Fazenda Coqueiros,21,,-22.873016,-43.491258
br,fazenda coqueiros,Fazenda Coqueiros,27,,-21,-47.833333
br,fazenda corixao,Fazenda Corixão,11,,-18.333333,-56.533333
br,fazenda correa,Fazenda Corrêa,14,,-16.8,-53.05
br,fazenda corrente,Fazenda Corrente,06,,-4.116667,-39.966667
br,fazenda criacao de aporema,Fazenda Criação de Aporema,03,,1.2,-50.883333
br,fazenda crianco,Fazenda Criancó,06,,-4.283333,-38.6
br,fazenda criolinhas,Fazenda Criolinhas,06,,-3.2,-39.583333
br,fazenda cristal,Fazenda Cristal,11,,-21.5,-54.366667
br,fazenda cristino,Fazenda Cristino,11,,-22.05,-52.75
br,fazenda cruz alta,Fazenda Cruz Alta,11,,-19.066667,-55.566667
br,fazenda cruzeiro,Fazenda Cruzeiro,27,,-20.7,-47.333333
br,fazenda cuiaba,Fazenda Cuiabá,27,,-22.3,-52.1
br,fazenda cupim,Fazenda Cupim,06,,-4.666667,-39.833333
br,fazenda curiango,Fazenda Curiango,15,,-16.166667,-40.65
br,fazenda curicaca,Fazenda Curicaca,06,,-3.816667,-40.483333
br,fazenda curral de pedra,Fazenda Curral de Pedra,27,,-20.216667,-47.566667
br,fazenda curralinho,Fazenda Curralinho,11,,-21.383333,-54.766667
br,fazenda curralinho,Fazenda Curralinho,18,,-24.433333,-50.05
br,fazenda curral velho,Fazenda Curral Velho,06,,-3.483333,-39.233333
br,fazenda currupati,Fazenda Currupati,06,,-4.6,-38.9
br,fazenda curu,Fazenda Curu,06,,-3.816667,-39.333333
br,fazenda da aldeia,Fazenda da Aldeia,11,,-18.2,-55.166667
br,fazenda da alianca,Fazenda da Aliança,27,,-20.5,-47.316667
br,fazenda da baia do felix,Fazenda da Baía do Felix,14,,-16.466667,-55.933333
br,fazenda da baia dos passaros,Fazenda da Baía dos Pássaros,14,,-16.6,-55.866667
br,fazenda da baia sao folis,Fazenda da Baia São Folis,14,,-16.466667,-55.933333
br,fazenda da bandeirinha,Fazenda da Bandeirinha,15,,-21.05,-45.4
br,fazenda da barra,Fazenda da Barra,11,,-21.316667,-53.083333
br,fazenda da barra,Fazenda da Barra,15,,-18.45,-47.716667
br,fazenda da bica,Fazenda da Bica,21,,-22.891621,-43.332845
br,fazenda da boa sorte,Fazenda da Boa Sorte,11,,-21.633333,-57.166667
br,fazenda da bocaina,Fazenda da Bocaina,15,,-21.4,-45.033333
br,fazenda da capoeira grande,Fazenda da Capoeira Grande,15,,-16.533333,-41.3
br,fazenda da conceicao,Fazenda da Conceição,11,,-20.833333,-55.783333
br,fazenda da concordia,Fazenda da Concórdia,27,,-20.483333,-47.35
br,fazenda da corixa,Fazenda da Corixa,14,,-16.466667,-55.666667
br,fazenda da estrela,Fazenda da Estrela,14,,-16.283333,-53.566667
br,fazenda da faiza,Fazenda da Faiza,06,,-3.716667,-39.266667
br,fazenda da galha,Fazenda da Galha,15,,-16.1,-40.133333
br,fazenda da ilha,Fazenda da Ilha,11,,-18.483333,-56.416667
br,fazenda da lagoa,Fazenda da Lagoa,20,,-7.683333,-42.716667
br,fazenda da lagoinha,Fazenda da Lagoinha,11,,-20.716667,-54.766667
br,fazenda da mata,Fazenda da Mata,08,,-20.633333,-41.383333
br,fazenda da mata,Fazenda da Mata,14,,-17.033333,-53.9
br,fazenda da matta,Fazenda da Matta,14,,-17.033333,-53.9
br,fazenda da mazela,Fazenda da Mazela,15,,-16.066667,-41.666667
br,fazenda da moata,Fazenda da Moata,14,,-17.033333,-53.9
br,fazenda da pedra,Fazenda da Pedra,27,,-21.2,-47.616667
br,fazenda da ponte,Fazenda da Ponte,15,,-19.3,-44.4
br,fazenda da prata,Fazenda da Prata,27,,-20.283333,-47.866667
br,fazenda da serra,Fazenda da Serra,27,,-21.366667,-47.216667
br,fazenda das palmeiras,Fazenda das Palmeiras,11,,-18.966667,-57.066667
br,fazenda das pedras,Fazenda das Pedras,27,,-23.466667,-45.9
br,fazenda das pombas,Fazenda das Pombas,20,,-7.433333,-40.866667
br,fazenda das violas,Fazenda das Violas,18,,-25.269167,-50.407222
br,fazenda de affonso,Fazenda de Affonso,14,,-14.5,-57
br,fazenda de antonio carmo,Fazenda de Antônio Carmo,15,,-17.266667,-41.916667
br,fazenda de baixo,Fazenda de Baixo,02,,-9.35,-37.066667
br,fazenda de cima,Fazenda de Cima,14,,-15.933333,-56.616667
br,fazenda de dentro,Fazenda de Dentro,05,,-10.2,-38.133333
br,fazenda de fora,Fazenda de Fora,05,,-9.666667,-41.716667
br,fazenda degredo,Fazenda Degredo,20,,-7.433333,-42.3
br,fazenda delicia,Fazenda Delícia,06,,-3.883333,-40.483333
br,fazenda de mosta,Fazenda de Mosta,14,,-17.033333,-53.9
br,fazenda desterro,Fazenda Desterro,06,,-4.066667,-39.1
br,fazenda d. francesa,Fazenda D. Francesa,15,,-16.033333,-39.95
br,fazenda divino,Fazenda Divino,11,,-18.75,-55.483333
br,fazenda divisa,Fazenda Divisa,11,,-21.5,-54.666667
br,fazenda do amazonas,Fazenda do Amazonas,25,,3.366667,-61.383333
br,fazenda do arranca dento,Fazenda do Arranca Dento,15,,-16.166667,-40.983333
br,fazenda do balbino,Fazenda do Balbino,15,,-16.233333,-40.483333
br,fazenda do balsamo,Fazenda do Bálsamo,27,,-20.3,-47.3
br,fazenda do banco,Fazenda do Banco,27,,-23.473005,-47.969966
br,fazenda do barata,Fazenda do Barata,21,,-22.901853,-43.444829
br,fazenda do barranca vermelha,Fazenda do Barranca Vermelha,11,,-19.933333,-56.766667
br,fazenda do barranca vermelho,Fazenda do Barranca Vermelho,11,,-19.933333,-56.766667
br,fazenda do barreiro,Fazenda do Barreiro,15,,-16.283333,-41.183333
br,fazenda do brejao,Fazenda do Brejão,27,,-21.633333,-46.866667
br,fazenda do caete,Fazenda do Caeté,14,,-15.85,-58.283333
br,fazenda do caethe,Fazenda do Caethé,14,,-15.85,-58.283333
br,fazenda do campeiro,Fazenda do Campeiro,11,,-21.183333,-54.75
br,fazenda do campo,Fazenda do Campo,27,,-21.383333,-47.05
br,fazenda do canal,Fazenda do Canal,05,,-11.35,-41.666667
br,fazenda do carlos,Fazenda do Carlos,06,,-3.933333,-39.6
br,fazenda do caxitore,Fazenda do Caxitoré,06,,-3.816667,-39.5
br,fazenda do corregozinho,Fazenda do Còrregozinho,15,,-20.166667,-45.266667
br,fazenda do engenho,Fazenda do Engenho,27,,-24.665705,-48.925795
br,fazenda do fundo,Fazenda do Fundo,15,,-20.333333,-45.966667
br,fazenda dois irmaos,Fazenda Dois Irmãos,01,,-9.816667,-67.416667
br,fazenda dois mil rels,Fazenda Dois Mil Rels,11,,-18.4,-55.866667
br,fazenda do japao,Fazenda do Japão,27,,-20.383333,-47.566667
br,fazenda do jardin,Fazenda do Jardin,15,,-21.833333,-46.083333
br,fazenda do jatoba,Fazenda do Jatobá,11,,-20.05,-55.966667
br,fazenda do macaco,Fazenda do Macaco,11,,-19.116667,-54.25
br,fazenda do mari,Fazenda do Mari,06,,-4.95,-38.05
br,fazenda do meio,Fazenda do Meio,20,,-8.4,-44.7
br,fazenda do melado,Fazenda do Melado,27,,-20.783333,-47.816667
br,fazenda do morias,Fazenda do Morias,14,,-17.516667,-55.083333
br,fazenda do morrinho,Fazenda do Morrinho,14,,-16.283333,-55.7
br,fazenda dona candida,Fazenda Dona Cândida,27,,-21.416667,-46.883333
br,fazenda dona josepha,Fazenda Dona Josepha,11,,-21.4,-54.366667
br,fazenda do paraiso,Fazenda do Paraíso,14,,-17.616667,-55.166667
br,fazenda do pires,Fazenda do Pires,27,,-22.133333,-47.233333
br,fazenda do rio das pedras,Fazenda do Rio das Pedras,27,,-22.183333,-47.016667
br,fazenda do rio nova,Fazenda do Rio Nova,11,,-21.333333,-56.75
br,fazenda do rio novo,Fazenda do Rio Novo,11,,-21.333333,-56.75
br,fazenda do salto,Fazenda do Salto,11,,-20.583333,-54.683333
br,fazenda dos barrosos,Fazenda dos Barrosos,06,,-3.633333,-39.15
br,fazenda dos palmares,Fazenda dos Palmares,15,,-16.333333,-40.766667
br,fazenda dos prazeres,Fazenda dos Prazeres,15,,-17.816667,-41.6
br,fazenda dos quincas-tua,Fazenda dos Quincas-Tua,15,,-16.133333,-40.283333
br,fazenda do tabuleiro,Fazenda do Tabuleiro,20,,-7.05,-42.2
br,fazenda do trapia,Fazenda do Trapiá,06,,-4.75,-40.366667
br,fazenda douradinho,Fazenda Douradinho,11,,-21.583333,-53.266667
br,fazenda dourados,Fazenda Dourados,27,,-20.65,-47.716667
br,fazenda dutra,Fazenda Dutra,27,,-21.85,-47.566667
br,fazenda eduardo,Fazenda Eduardo,15,,-20.683333,-45.2
br,fazenda emerenciano borges,Fazenda Emerenciano Borges,15,,-16.9,-41.65
br,fazenda encantado,Fazenda Encantado,06,,-3.766667,-39.333333
br,fazenda entre rios,Fazenda Entre Rios,11,,-19.366667,-54.15
br,fazenda escaramuca,Fazenda Escaramuça,27,,-20.733333,-47.683333
br,fazenda espera,Fazenda Espera,06,,-3.683333,-39.05
br,fazenda esperanca,Fazenda Esperança,06,,-3.483333,-39.416667
br,fazenda esperanca,Fazenda Esperança,11,,-17.966667,-54.466667
br,fazenda esperantina,Fazenda Esperantina,06,,-3.233333,-40.466667
br,fazenda espinheiro,Fazenda Espinheiro,06,,-4.783333,-39.633333
br,fazenda espirito santo,Fazenda Espírito Santo,04,,-9.266667,-67.016667
br,fazenda espirito santo,Fazenda Espírito Santo,06,,-4.933333,-39.05
br,fazenda estaca,Fazenda Estaca,11,,-20.3,-54.533333
br,fazenda estiva,Fazenda Estiva,06,,-3.966667,-40.05
br,fazenda estreito,Fazenda Estreito,11,,-21.45,-54.65
br,fazenda estreito,Fazenda Estreito,20,,-7.016667,-42
br,fazenda estrela d'oeste,Fazenda Estrela d'Oeste,27,,-21.433333,-47.716667
br,fazenda estrela,Fazenda Estrêla,06,,-4.8,-38.183333
br,fazenda estrela,Fazenda Estrela,11,,-18.4,-53.6
br,fazenda estrella,Fazenda Estrella,14,,-16.283333,-53.566667
br,fazenda eucalipto,Fazenda Eucalipto,30,,-7.416667,-40.383333
br,fazenda extrema,Fazenda Extrema,06,,-3.583333,-39.966667
br,fazenda extremas,Fazenda Extremas,06,,-3.966667,-39.4
br,fazenda famosa,Fazenda Famosa,06,,-3.75,-39.283333
br,fazenda faustina,Fazenda Faustina,06,,-7.05,-39.166667
br,fazenda fechado,Fazenda Fechado,06,,-4.016667,-39.55
br,fazenda feijao,Fazenda Feijão,06,,-3.6,-39.016667
br,fazenda feijiao bravo,Fazenda Feijião Bravo,06,,-4.6,-38.233333
br,fazenda felicio,Fazenda Felício,15,,-16.433333,-40.233333
br,fazenda ferino,Fazenda Ferino,06,,-3.633333,-39.783333
br,fazenda ferrao,Fazenda Ferrão,06,,-3.716667,-39.916667
br,fazenda ferrugem,Fazenda Ferrugem,06,,-3.1,-39.85
br,fazenda fialho,Fazenda Fialho,23,,-29.7756,-50.908
br,fazenda fidelis,Fazenda Fidélis,06,,-4.116667,-39.8
br,fazenda figueira branca,Fazenda Figueira Branca,27,,-21.75,-47.766667
br,fazenda figueira,Fazenda Figueira,11,,-17.666667,-56.85
br,fazenda firme,Fazenda Firme,11,,-19.266667,-57.016667
br,fazenda flamengo,Fazenda Flamengo,06,,-3.933333,-40.35
br,fazenda flor da amescla,Fazenda Flor da Amescla,20,,-5.433333,-41.916667
br,fazenda flores,Fazenda Flores,06,,-4.25,-40.3
br,fazenda floresta,Fazenda Floresta,27,,-20.55,-47.95
br,fazenda fofa,Fazenda Fofa,06,,-3.85,-39.266667
br,fazenda formiga,Fazenda Formiga,11,,-21.3,-54.433333
br,fazenda formosa,Fazenda Formosa,14,,-15.065,-59.875
br,fazenda formosa,Fazenda Formosa,20,,-7.3,-41.15
br,fazenda formosa,Fazenda Formosa,27,,-24.538931,-48.092747
br,fazenda forquilha,Fazenda Forquilha,06,,-6.566667,-39.966667
br,fazenda forquilha,Fazenda Forquilha,11,,-21.566667,-53.616667
br,fazenda forquima,Fazenda Forquima,06,,-3.566667,-39.916667
br,fazenda fortaleza,Fazenda Fortaleza,11,,-18.55,-54.766667
br,fazenda frade,Fazenda Frade,20,,-6.983333,-41.916667
br,fazenda franquilino marques,Fazenda Franquilino Marques,15,,-16.533333,-41.766667
br,fazenda frio,Fazenda Frio,06,,-3.666667,-39.266667
br,fazenda furnas,Fazenda Furnas,06,,-4.316667,-39.133333
br,fazenda galiza,Fazenda Galiza,20,,-7.583333,-40.85
br,fazenda gameleira,Fazenda Gameleira,11,,-20.616667,-54.633333
br,fazenda gameleira,Fazenda Gameleira,14,,-14.830556,-59.801389
br,fazenda gangorra,Fazenda Gangorra,06,,-4.25,-40
br,fazenda gangorrinha,Fazenda Gangorrinha,15,,-17.166667,-41.7
br,fazenda gaviao,Fazenda Gavião,06,,-3.533333,-39.383333
br,fazenda gemeleira,Fazenda Gemeleira,06,,-4.683333,-37.583333
br,fazenda gomalina,Fazenda Gomalina,14,,-15.816667,-59.802778
br,fazenda goncalinho,Fazenda Goncalinho,13,,-6.766667,-43.383333
br,fazenda goncalves,Fazenda Gonçalves,27,,-20.266667,-47.616667
br,fazenda gondim,Fazenda Gondim,06,,-4.783333,-38.25
br,fazenda grande,Fazenda Grande,05,,-11.866667,-43.333333
br,fazenda grossos,Fazenda Grossos,06,,-4.533333,-39.15
br,fazenda grupiara,Fazenda Grupiara,15,,-17.083333,-41.9
br,fazenda gueda,Fazenda Gueda,11,,-18.883333,-55.65
br,fazenda gurupa,Fazenda Gurupa,06,,-4.5,-39.083333
br,fazenda henrique rocha,Fazenda Henrique Rocha,15,,-16.716667,-41.816667
br,fazenda hermes silva,Fazenda Hermes Silva,15,,-18.333333,-41.766667
br,fazenda ibitira,Fazenda Ibitira,27,,-21.85,-47.633333
br,fazenda ilidio rocha,Fazenda Ilídio Rocha,11,,-19.6,-55.033333
br,fazenda imbirussu,Fazenda Imbirussú,11,,-20.6,-54.75
br,fazenda inez,Fazenda Inêz,06,,-3.483333,-40.25
br,fazenda inga,Fazenda Ingá,06,,-4.25,-39.533333
br,fazenda inga,Fazenda Ingá,20,,-5.183333,-41.5
br,fazenda inhare,Fazenda Inharé,06,,-3.983333,-39.3
br,fazenda inhuma,Fazenda Inhuma,13,,-8.746667,-45.968889
br,fazenda invejada,Fazenda Invejada,20,,-5.35,-42.266667
br,fazenda ipe,Fazenda Ipê,01,,-9.833333,-67.433333
br,fazenda ipiranga,Fazenda Ipiranga,06,,-3.766667,-40.283333
br,fazenda ipiranga,Fazenda Ipiranga,27,,-21.8,-47.7
br,fazenda ipueira da areia,Fazenda Ipueira da Areia,06,,-4.483333,-39.133333
br,fazenda ipueira da pedra,Fazenda Ipueira da Pedra,06,,-4.85,-39.75
br,fazenda ipueira da vaca,Fazenda Ipueira da Vaca,06,,-4.55,-39.233333
br,fazenda ipueiras,Fazenda Ipueiras,06,,-3.266667,-40.35
br,fazenda ipueirinha,Fazenda Ipueirinha,06,,-4.983333,-39.55
br,fazenda iracema,Fazenda Iracema,06,,-4.466667,-38.35
br,fazenda itataia,Fazenda Itataia,06,,-4.583333,-39.8
br,fazenda ivai,Fazenda Ivaí,18,,-23.066667,-52.466667
br,fazenda jaburu,Fazenda Jaburu,06,,-3.816667,-39.35
br,fazenda jaburu,Fazenda Jaburu,14,,-16.137222,-56.991944
br,fazenda jabuta,Fazenda Jabuta,20,,-6.683333,-42.3
br,fazenda jacare,Fazenda Jacaré,20,,-5.35,-42.033333
br,fazenda jacutinga,Fazenda Jacutinga,15,,-16.616667,-40.933333
br,fazenda jaguarandi,Fazenda Jaguarandí,27,,-21.85,-47.7
br,fazenda jandira,Fazenda Jandira,06,,-4,-39.083333
br,fazenda japonesa,Fazenda Japonesa,06,,-4.416667,-38.716667
br,fazenda jatoba,Fazenda Jatobá,06,,-3.9,-40.083333
br,fazenda jatoba,Fazenda Jatobá,11,,-20.05,-55.966667
br,fazenda jau,Fazenda Jaú,11,,-20.983333,-55.133333
br,fazenda jauru,Fazenda Jaurú,11,,-21.666667,-53.833333
br,fazenda joao gil,Fazenda João Gil,15,,-16.35,-41.5
br,fazenda joao gualberto,Fazenda João Gualberto,15,,-17.233333,-40.733333
br,fazenda joao pereira,Fazenda João Pereira,06,,-3.883333,-39.966667
br,fazenda joaquim nogueira,Fazenda Joaquim Nogueira,15,,-17.8,-40.8
br,fazenda jocelin bitencourt,Fazenda Jocelin Bitencourt,11,,-21.5,-54.266667
br,fazenda jori,Fazenda Jori,06,,-4.366667,-38.45
br,fazenda jose antonio,Fazenda José Antônio,15,,-16.25,-39.933333
br,fazenda jose bravo,Fazenda José Bravo,15,,-17.066667,-41.116667
br,fazenda jose mendes,Fazenda José Mendes,15,,-17.95,-41.7
br,fazenda jose nino,Fazenda José Nino,11,,-19.683333,-53.75
br,fazenda jose quito,Fazenda José Quito,14,,-17.966667,-53.683333
br,fazenda jose ramos,Fazenda José Ramos,15,,-18.383333,-41.416667
br,fazenda jose rosa,Fazenda José Rosa,27,,-21.4,-47.133333
br,fazenda jua,Fazenda Juá,06,,-4.283333,-39.95
br,fazenda juca,Fazenda Jucá,06,,-3.333333,-40.35
br,fazenda juliao,Fazenda Julião,06,,-3.616667,-40.366667
br,fazenda juliao,Fazenda Julião,30,,-7.916667,-40.316667
br,fazenda juncal,Fazenda Juncal,06,,-3.45,-39.75
br,fazenda junco,Fazenda Junco,06,,-3.866667,-39.716667
br,fazenda jurema,Fazenda Jurema,06,,-3.95,-39.183333
br,fazenda jurema,Fazenda Jurema,20,,-5.433333,-42.15
br,fazenda jurujuba,Fazenda Jurujuba,06,,-4.383333,-40.066667
br,fazenda juventino rocha,Fazenda Juventino Rocha,15,,-17.95,-40.383333
br,fazenda lacerda,Fazenda Lacerda,27,,-21.966667,-47.833333
br,fazenda ladeira vermelha,Fazenda Ladeira Vermelha,06,,-3.316667,-40.3
br,fazenda lageado,Fazenda Lageado,11,,-20.95,-54.783333
br,fazenda lageado,Fazenda Lageado,27,,-21.966667,-47.716667
br,fazenda lagoa achada,Fazenda Lagoa Achada,06,,-4.45,-38.1
br,fazenda lagoa cercada,Fazenda Lagoa Cercada,06,,-3.8,-39.733333
br,fazenda lagoa da cruz,Fazenda Lagoa da Cruz,06,,-3.766667,-39.133333
br,fazenda lagoa da onca,Fazenda Lagoa da Onca,06,,-4.116667,-40.3
br,fazenda lagoa da vereda,Fazenda Lagoa da Vereda,20,,-7.783333,-42.983333
br,fazenda lagoa de dentro,Fazenda Lagoa de Dentro,06,,-3.816667,-39.383333
br,fazenda lagoa de pedra,Fazenda Lagoa de Pedra,06,,-3.9,-40.433333
br,fazenda lagoa desalsa,Fazenda Lagoa Desalsa,22,,-4.85,-37.283333
br,fazenda lagoa do barro,Fazenda Lagoa do Barro,20,,-5.183333,-41.616667
br,fazenda lagoa do felix,Fazenda Lagoa do Félix,06,,-3.816667,-39.133333
br,fazenda lagoa do mato,Fazenda Lagoa do Mato,06,,-3.583333,-39.083333
br,fazenda lagoa do mato,Fazenda Lagoa do Mato,20,,-7.95,-42.783333
br,fazenda lagoa do porco,Fazenda Lagoa do Porco,06,,-4.766667,-39.683333
br,fazenda lagoa dos bois,Fazenda Lagoa dos Bois,06,,-4.85,-39.366667
br,fazenda lagoa dos mouras,Fazenda Lagoa dos Mouras,20,,-5.883333,-42.083333
br,fazenda lagoa dos pintos,Fazenda Lagoa dos Pintos,06,,-3.5,-39.316667
br,fazenda lagoa do trapia,Fazenda Lagoa do Trapiá,06,,-4.583333,-39.966667
br,fazenda lagoa nova,Fazenda Lagoa Nova,06,,-4.416667,-38.433333
br,fazenda lagoa queimada,Fazenda Lagoa Queimada,06,,-3.7,-39.216667
br,fazenda lagoa tapada,Fazenda Lagoa Tapada,06,,-4.833333,-40.433333
br,fazenda lagoa verde,Fazenda Lagoa Verde,06,,-4.55,-39.283333
br,fazenda lagoinha,Fazenda Lagoinha,06,,-3.716667,-39.65
br,fazenda lajeado,Fazenda Lajeado,11,,-20.95,-54.783333
br,fazenda lajes,Fazenda Lajes,06,,-3.933333,-39.383333
br,fazenda lajes,Fazenda Lajes,20,,-6.05,-41.933333
br,fazenda lama podre,Fazenda Lama Podre,06,,-7.2,-39.583333
br,fazenda lama preta,Fazenda Lama Preta,21,,-22.866028,-43.659774
br,fazenda lameiro,Fazenda Lameiro,06,,-7.15,-39.166667
br,fazenda lanceiro,Fazenda Lanceiro,11,,-20.983333,-53.716667
br,fazenda latada,Fazenda Latada,06,,-3.55,-40.016667
br,fazenda laura,Fazenda Laura,06,,-3.816667,-39.583333
br,fazenda lava pes,Fazenda Lava Pés,15,,-17.2,-41.816667
br,fazenda lavorinha,Fazenda Lavorinha,27,,-20.166667,-47.633333
br,fazenda lavrinha,Fazenda Lavrinha,18,,-24.716667,-50.133333
br,fazenda lazaro,Fazenda Lázaro,06,,-4.9,-39.783333
br,fazenda limeira,Fazenda Limeira,27,,-20.95,-47.916667
br,fazenda limoeiro,Fazenda Limoeiro,11,,-21.666667,-54.15
br,fazenda limoeiro,Fazenda Limoeiro,27,,-21.566667,-47.7
br,fazenda linda,Fazenda Linda,06,,-4.583333,-39.6
br,fazenda livramento,Fazenda Livramento,06,,-3.7,-39.383333
br,fazenda lobo,Fazenda Lôbo,14,,-17.766667,-53.783333
br,fazenda lunarda,Fazenda Lunarda,20,,-5.233333,-41.55
br,fazenda lundu,Fazenda Lundu,06,,-3.566667,-39.783333
br,fazenda luzitania,Fazenda Luzitania,11,,-18.416667,-55.783333
br,fazenda macaco,Fazenda Macaco,06,,-3.683333,-39.866667
br,fazenda macaco,Fazenda Macaco,11,,-19.116667,-54.25
br,fazenda macambira,Fazenda Macambira,06,,-4.616667,-39
br,fazenda macapa,Fazenda Macapá,06,,-4.05,-40.016667
br,fazenda macaubas,Fazenda Macaúbas,27,,-21.666667,-46.966667
br,fazenda madrigal,Fazenda Madrigal,20,,-7.166667,-42.516667
br,fazenda madrilha,Fazenda Madrilha,20,,-7.1,-42.45
br,fazenda malhada bonita,Fazenda Malhada Bonita,20,,-7.583333,-40.9
br,fazenda malhada do pau branco,Fazenda Malhada do Pau Branco,06,,-4.966667,-39.9
br,fazenda malhada vermelha,Fazenda Malhada Vermelha,20,,-7.166667,-42.016667
br,fazenda malhadinha,Fazenda Malhadinha,20,,-7.533333,-41.116667
br,fazenda maloca,Fazenda Maloca,01,,-9.75,-67.366667
br,fazenda mamueira,Fazenda Mamueira,20,,-5.583333,-42.616667
br,fazenda mandacaru,Fazenda Mandacaru,06,,-3.833333,-39.783333
br,fazenda mandibira,Fazenda Mandibira,06,,-7.3,-39.616667
br,fazenda manuino,Fazenda Manuíno,30,,-7.733333,-40.1
br,fazenda maraba,Fazenda Maraba,27,,-22.1,-51.933333
br,fazenda maracata,Fazenda Maracatá,06,,-3.933333,-39.55
br,fazenda maracuja,Fazenda Maracujá,06,,-4.683333,-39.733333
br,fazenda maraja,Fazenda Marajá,06,,-4.533333,-40.45
br,fazenda marajo,Fazenda Marajó,06,,-4.65,-39.066667
br,fazenda maranata,Fazenda Maranata,06,,-3.983333,-39.1
br,fazenda mares,Fazenda Marés,06,,-4.433333,-39.066667
br,fazenda marfim,Fazenda Marfim,27,,-20.633333,-47.466667
br,fazenda maria luiza,Fazenda Maria Luíza,18,,-25.6436,-48.7976
br,fazenda mariano guedes,Fazenda Mariano Guedes,15,,-16.85,-41.266667
br,fazenda maria preta,Fazenda María Preta,20,,-7.533333,-41.316667
br,fazenda mari,Fazenda Mari,06,,-3.833333,-39.4
br,fazenda marizeira,Fazenda Marizeira,06,,-3.633333,-39.2
br,fazenda marrecas,Fazenda Marrecas,06,,-3.4,-39.883333
br,fazenda maruai,Fazenda Maruai,25,,3.733333,-60.466667
br,fazenda massape,Fazenda Massapê,06,,-3.933333,-39.333333
br,fazenda massape grande,Fazenda Massapê Grande,06,,-4.916667,-39.383333
br,fazenda mata da chuva,Fazenda Mata da Chuva,27,,-20.916667,-47.95
br,fazenda mata fresca,Fazenda Mata Fresca,06,,-3.533333,-39.25
br,fazenda mateus,Fazenda Mateus,06,,-4,-40.2
br,fazenda matinha,Fazenda Matinha,27,,-20.15,-47.733333
br,fazenda mato grande,Fazenda Mato Grande,14,,-16.2,-56.333333
br,fazenda mato grosso,Fazenda Mato Grosso,06,,-4.666667,-39.45
br,fazenda matoso,Fazenda Matoso,06,,-4.9,-40.383333
br,fazenda melancia dos costas,Fazenda Melancia dos Costas,06,,-3.7,-39.2
br,fazenda melancia,Fazenda Melancia,06,,-3.75,-39.133333
br,fazenda melancias,Fazenda Melancias,06,,-3.6,-39.216667
br,fazenda mel,Fazenda Mel,06,,-4.083333,-39.116667
br,fazenda miguelito,Fazenda Miguelito,11,,-23.716667,-55.4
br,fazenda milagres,Fazenda Milagres,20,,-5.733333,-41.233333
br,fazenda mimosa,Fazenda Mimosa,06,,-3.583333,-39.216667
br,fazenda mimoso,Fazenda Mimoso,11,,-19.683333,-55.1
br,fazenda mina,Fazenda Mina,06,,-4.6,-39.083333
br,fazenda mirador,Fazenda Mirador,06,,-4.166667,-39.866667
br,fazenda miranda,Fazenda Miranda,06,,-3.933333,-39.7
br,fazenda mirante,Fazenda Mirante,27,,-20.116667,-47.716667
br,fazenda missi,Fazenda Missi,06,,-3.45,-39.85
br,fazenda mista,Fazenda Mista,27,,-20.466667,-47.766667
br,fazenda mocambo,Fazenda Mocambo,06,,-4.333333,-39.966667
br,fazenda mocambo,Fazenda Mocambo,20,,-7.433333,-42.466667
br,fazenda mocambo,Fazenda Mocambo,30,,-7.95,-39.766667
br,fazenda modelo,Fazenda Modelo,21,,-22.99267,-43.589936
br,fazenda moises quito,Fazenda Moisés Quito,14,,-17.933333,-53.566667
br,fazenda monte alegre,Fazenda Monte Alegre,06,,-4.85,-38.033333
br,fazenda monte alegre,Fazenda Monte Alegre,27,,-20.333333,-47.816667
br,fazenda monte alvernes,Fazenda Monte Alvernes,06,,-3.866667,-39.366667
br,fazenda monte belo,Fazenda Monte Belo,27,,-20.783333,-47.966667
br,fazenda monte castelo,Fazenda Monte Castelo,06,,-4.816667,-39.3
br,fazenda monte castelo,Fazenda Monte Castelo,20,,-5.083333,-42.016667
br,fazenda monte,Fazenda Monte,06,,-3.6,-39.283333
br,fazenda monte lima,Fazenda Monte Lima,06,,-4.8,-39.15
br,fazenda monte santos,Fazenda Monte Santos,20,,-7.083333,-41.45
br,fazenda montes claros,Fazenda Montes Claros,06,,-3.633333,-39.466667
br,fazenda monte verde,Fazenda Monte Verde,11,,-18.25,-54.683333
br,fazenda moreira,Fazenda Moreira,06,,-3.733333,-39.433333
br,fazenda mororo,Fazenda Mororó,06,,-3.95,-39.016667
br,fazenda morrinhos,Fazenda Morrinhos,06,,-4.566667,-39.833333
br,fazenda morrinhos,Fazenda Morrinhos,20,,-5.15,-42.516667
br,fazenda morro alto,Fazenda Morro Alto,27,,-21.783333,-47.416667
br,fazenda mucungu,Fazenda Mucungu,30,,-7.75,-40.433333
br,fazenda mufumbo,Fazenda Mufumbo,06,,-4.916667,-38.2
br,fazenda mulungu,Fazenda Mulungu,06,,-3.733333,-39.583333
br,fazenda mundo novo,Fazenda Mundo Novo,06,,-4.65,-38.133333
br,fazenda mundubin,Fazenda Mundubin,06,,-3.633333,-39.1
br,fazenda munuo novo,Fazenda Munuo Novo,06,,-3.616667,-39.4
br,fazenda muquem,Fazenda Muquem,20,,-7.566667,-40.833333
br,fazenda mutum,Fazenda Mutum,11,,-20.483333,-53.266667
br,fazenda nascente,Fazenda Nascente,11,,-21.566667,-54.216667
br,fazenda natal,Fazenda Natal,11,,-18.95,-55.716667
br,fazenda nazare,Fazenda Nazaré,06,,-2.966667,-40.116667
br,fazenda nazare,Fazenda Nazaré,27,,-22.116667,-51.966667
br,fazenda nazareno,Fazenda Nazareno,06,,-4.05,-39.816667
br,fazenda nossa senhora aparecida,Fazenda Nossa Senhora Aparecida,01,,-9.25,-70.316667
br,fazenda nossa senhora da conceicao,Fazenda Nossa Senhora da Conceição,14,,-16.307778,-56.530556
br,fazenda nossa senhora da guia,Fazenda Nossa Senhora da Guia,14,,-16.216667,-56.45
br,fazenda nova,Fazenda Nova,02,,-9.45,-37.166667
br,fazenda nova,Fazenda Nova,06,,-3.1,-40.2
br,fazenda nova,Fazenda Nova,11,,-21.866667,-56.533333
br,fazenda nova,Fazenda Nova,13,,-3.9525,-43.006667
br,fazenda nova,Fazenda Nôva,17,,-7.9,-36.833333
br,fazenda nova,Fazenda Nova,20,,-7.25,-41.633333
br,fazenda nova,Fazenda Nova,22,,-5.36,-36.643611
br,fazenda nova,Fazenda Nova,29,,-16.183333,-50.8
br,fazenda nova,Fazenda Nova,30,,-7.916667,-40
br,fazenda nova horizonte,Fazenda Nova Horizonte,11,,-19.266667,-54.75
br,fazenda nova india,Fazenda Nova India,21,,-22.861005,-43.636319
br,fazenda nova olinda,Fazenda Nova Olinda,06,,-4.5,-39.033333
br,fazenda nova veneza,Fazenda Nova Veneza,27,,-22.783333,-47.25
br,fazenda novo axioma,Fazenda Novo Axioma,04,,-9.583333,-67.316667
br,fazenda nuno pereira,Fazenda Nuno Pereira,15,,-16.783333,-40.766667
br,fazenda observacao,Fazenda Observação,20,,-7.6,-40.933333
br,fazenda oiticica,Fazenda Oiticica,06,,-3.633333,-39.683333
br,fazenda oiticica rasteira,Fazenda Oiticica Rasteira,06,,-4.716667,-40.4
br,fazenda oiticiquinha,Fazenda Oiticiquinha,06,,-4.083333,-39.333333
br,fazenda olaria,Fazenda Olaria,15,,-20.1,-46.1
br,fazenda olho d'agua das pedras,Fazenda Olho d'Agua das Pedras,20,,-5.45,-42.033333
br,fazenda olho d'agua do jirau,Fazenda Olho d'Agua do Jirau,06,,-4.933333,-39.85
br,fazenda olho d'agua,Fazenda Olho d'Agua,06,,-3.883333,-39.583333
br,fazenda olho d'agua,Fazenda Olho d'Agua,20,,-6.983333,-41.566667
br,fazenda olhos d'agua,Fazenda Olhos d'Agua,11,,-20.833333,-54.8
br,fazenda olinda,Fazenda Olinda,06,,-4.033333,-39.883333
br,fazenda oriacao da regiao dos lagos,Fazenda Oriação da Região dos Lagos,03,,1.75,-50.8
br,fazenda ouro branco,Fazenda Ouro Branco,06,,-4.8,-39.25
br,fazenda ouro preto,Fazenda Ouro Prêto,01,,-9.783333,-67.366667
br,fazenda ouro preto,Fazenda Ouro Preto,06,,-3.7,-40.05
br,fazenda ouro verde,Fazenda Ouro Verde,11,,-21.9,-54.35
br,fazenda ozorio,Fazenda Ozório,15,,-20.25,-45.766667
br,fazenda pacova,Fazenda Pacova,06,,-3.7,-39.133333
br,fazenda padre cicero,Fazenda Padre Cícero,01,,-9.933333,-67.483333
br,fazenda pae joao,Fazenda Pae João,26,,-28.183333,-50.633333
br,fazenda palestina,Fazenda Palestina,06,,-4.5,-39.133333
br,fazenda palestina,Fazenda Palestina,20,,-5.133333,-42.2
br,fazenda palmeira,Fazenda Palmeira,11,,-21.483333,-54.45
br,fazenda palmeiras,Fazenda Palmeiras,15,,-20.233333,-46.166667
br,fazenda palmeiras,Fazenda Palmeiras,27,,-21.8,-47.633333
br,fazenda palmitos,Fazenda Palmitos,27,,-20.683333,-47.9
br,fazenda palotina,Fazenda Palotina,04,,-9.55,-67.183333
br,fazenda papagaio,Fazenda Papagaio,20,,-7.1,-42.866667
br,fazenda papel,Fazenda Papel,06,,-4.2,-39.55
br,fazenda paracati,Fazenda Paracati,20,,-7.166667,-42.566667
br,fazenda parada,Fazenda Parada,06,,-4.466667,-39.066667
br,fazenda paraiso,Fazenda Paraiso,06,,-4.716667,-38.6
br,fazenda parana,Fazenda Paraná,04,,-9.483333,-67.283333
br,fazenda parana,Fazenda Paraná,15,,-16.216667,-41.316667
br,fazenda passagem,Fazenda Passagem,06,,-4.466667,-40.433333
br,fazenda passagem franca,Fazenda Passagem Franca,06,,-3.966667,-38.733333
br,fazenda passarinha,Fazenda Passarinha,06,,-4.866667,-40.466667
br,fazenda pastos bons,Fazenda Pastos Bons,06,,-4.183333,-39.9
br,fazenda patos,Fazenda Patos,06,,-4.666667,-40.45
br,fazenda patos,Fazenda Patos,20,,-7.6,-40.816667
br,fazenda pau branco,Fazenda Pau Branco,06,,-3.3,-40.466667
br,fazenda pau d'arco,Fazenda Pau d'Arco,06,,-3.633333,-39.066667
br,fazenda pau darco,Fazenda Pau Darco,30,,-7.983333,-39.883333
br,fazenda pau ferrado,Fazenda Pau Ferrado,30,,-7.35,-39.933333
br,fazenda pau ferro,Fazenda Pau Ferro,06,,-4.983333,-39.483333
br,fazenda pau ferro,Fazenda Pau Ferro,30,,-7.716667,-40.033333
br,fazenda pau fincado,Fazenda Pau Fincado,06,,-3.616667,-40.016667
br,fazenda pauliceia,Fazenda Pauliceia,11,,-21.683333,-54.45
br,fazenda paulista,Fazenda Paulista,04,,-9.05,-67.25
br,fazenda pe da serra,Fazenda Pé da Serra,06,,-3.55,-39.7
br,fazenda pedra amarela,Fazenda Pedra Amarela,06,,-4.15,-40.05
br,fazenda pedra arrancada,Fazenda Pedra Arrancada,06,,-3.033333,-39.9
br,fazenda pedra branca,Fazenda Pedra Branca,06,,-3.8,-39.166667
br,fazenda pedra d'agua,Fazenda Pedra d'Agua,06,,-4.833333,-38.3
br,fazenda pedra de amolar,Fazenda Pedra de Amolar,06,,-4.066667,-39.816667
br,fazenda pedra redonda,Fazenda Pedra Redonda,06,,-3.283333,-40.3
br,fazenda pedra vermelha,Fazenda Pedra Vermelha,06,,-3.783333,-40.066667
br,fazenda pedro gomes,Fazenda Pedro Gomes,11,,-18.066667,-54.533333
br,fazenda pedro neto,Fazenda Pedro Neto,27,,-21.266667,-47.616667
br,fazenda pedro severo,Fazenda Pedro Severo,11,,-17.666667,-54.033333
br,fazenda pensilvania,Fazenda Pensilvânia,06,,-4.4,-40.233333
br,fazenda pequiri,Fazenda Pequirí,11,,-17.7,-54.15
br,fazenda perdicao,Fazenda Perdicão,06,,-4.433333,-39.1
br,fazenda pereiro,Fazenda Pereiro,20,,-7.85,-40.916667
br,fazenda piabas,Fazenda Piabas,06,,-4.233333,-39.866667
br,fazenda picada,Fazenda Picada,06,,-3.35,-40.383333
br,fazenda picos,Fazenda Picos,06,,-4.183333,-40.25
br,fazenda pindahyval,Fazenda Pindahyval,14,,-17.15,-56.083333
br,fazenda pindaival,Fazenda Pindaìval,14,,-17.15,-56.083333
br,fazenda pingador,Fazenda Pingador,14,,-15.066667,-55.8
br,fazenda pinhador,Fazenda Pinhador,14,,-15.066667,-55.8
br,fazenda pinheiro,Fazenda Pinheiro,18,,-25.416,-50.5498
br,fazenda pintada,Fazenda Pintada,06,,-4.216667,-39.883333
br,fazenda pintada,Fazenda Pintada,20,,-7.733333,-43.116667
br,fazenda pio ix,Fazenda Pio IX,20,,-6.916667,-41.9
br,fazenda piquizeiro,Fazenda Piquizeiro,20,,-5.566667,-42.516667
br,fazenda pirai,Fazenda Pirai,06,,-3.716667,-40.45
br,fazenda piratininga,Fazenda Piratininga,01,,-9.933333,-67.066667
br,fazenda piripiri,Fazenda Piripiri,20,,-5.433333,-42.166667
br,fazenda pitombeira,Fazenda Pitombeira,06,,-3.8,-40.3
br,fazenda pitombeira,Fazenda Pitombeira,20,,-5.016667,-41.3
br,fazenda pitombeiras,Fazenda Pitombeiras,06,,-3.933333,-39.483333
br,fazenda pituba,Fazenda Pituba,27,,-21.966667,-46.833333
br,fazenda pocinhos,Fazenda Pocinhos,06,,-3.9,-40.183333
br,fazenda poco comprido,Fazenda Poço Comprido,06,,-3.783333,-40.416667
br,fazenda poco da onca,Fazenda Poço da Onça,06,,-3.616667,-39.883333
br,fazenda poco da pedra,Fazenda Poço da Pedra,06,,-4,-39.433333
br,fazenda poco,Fazenda Poço,27,,-20.416667,-47.866667
br,fazenda poco redondo,Fazenda Poço Redondo,06,,-3.816667,-39.516667
br,fazenda poco verde,Fazenda Poço Verde,06,,-3.616667,-39.983333
br,fazenda ponta da mata,Fazenda Ponta da Mata,27,,-20.8,-47.916667
br,fazenda ponta da serra,Fazenda Ponta da Serra,06,,-4.966667,-39.8
br,fazenda ponta,Fazenda Ponta,27,,-21.466667,-46.683333
br,fazenda pontal,Fazenda Pontal,11,,-19.683333,-52.816667
br,fazenda ponte alta,Fazenda Ponte Alta,14,,-15.416667,-55.283333
br,fazenda ponte novo,Fazenda Ponte Novo,15,,-20.05,-46.133333
br,fazenda porteiras,Fazenda Porteiras,20,,-7.083333,-42.7
br,fazenda porto alegre,Fazenda Pôrto Alegre,11,,-18.283333,-55.433333
br,fazenda porto de pedra,Fazenda Pôrto de Pedra,11,,-21.2,-52.966667
br,fazenda posto dourado,Fazenda Posto Dourado,14,,-16.475,-56.2
br,fazenda pouso alto,Fazenda Pouso Alto,11,,-19.416667,-54.133333
br,fazenda prates,Fazenda Prates,08,,-20.916667,-41.016667
br,fazenda pratinha,Fazenda Pratinha,27,,-21.433333,-47.533333
br,fazenda primavera,Fazenda Primavera,18,,-24.872129,-48.74213
br,fazenda primavera,Fazenda Primavera,20,,-7.1,-41.183333
br,fazenda promissao,Fazenda Promissão,11,,-18.65,-55.7
br,fazenda providencia,Fazenda Providéncia,06,,-3.966667,-39.133333
br,fazenda pui,Fazenda Puí,01,,-9.783333,-67.333333
br,fazenda pulador,Fazenda Pulador,11,,-19.05,-54.816667
br,fazenda puruna,Fazenda Puruna,06,,-3.133333,-40.3
br,fazenda quandu,Fazenda Quandu,06,,-4.066667,-38.65
br,fazenda queimada,Fazenda Queimada,30,,-7.566667,-40.1
br,fazenda queimadas,Fazenda Queimadas,06,,-4.85,-37.55
br,fazenda queiroz,Fazenda Queiroz,11,,-20.3,-51.45
br,fazenda quicim,Fazenda Quicim,06,,-3.733333,-39.2
br,fazenda raizama,Fazenda Raizama,11,,-17.85,-54.466667
br,fazenda rajada,Fazenda Rajada,06,,-3.533333,-39.466667
br,fazenda rancharia,Fazenda Rancharia,11,,-18.916667,-55.7
br,fazenda rancho alegre,Fazenda Rancho Alegre,06,,-4.516667,-38.183333
br,fazenda rancho dos mocos,Fazenda Rancho dos Moços,06,,-4.016667,-39.066667
br,fazenda rapina,Fazenda Rapina,06,,-4.033333,-39.6
br,fazenda recife,Fazenda Recife,06,,-4.25,-40.2
br,fazenda recreio,Fazenda Recreio,11,,-17.65,-56.75
br,fazenda recreio,Fazenda Recreio,27,,-21.8,-46.683333
br,fazenda redoma,Fazenda Redoma,20,,-5.416667,-42.133333
br,fazenda reforma,Fazenda Reforma,22,,-4.933333,-37.383333
br,fazenda rejeitado,Fazenda Rejeitado,06,,-4.85,-40.45
br,fazenda remanso,Fazenda Remanso,27,,-20.366667,-47.833333
br,fazenda requejao,Fazenda Requejão,15,,-16.5,-41.583333
br,fazenda resfriado,Fazenda Resfriado,27,,-20.833333,-47.833333
br,fazenda retiro da ponte,Fazenda Retiro da Ponte,27,,-20.8,-47.8
br,fazenda retiro,Fazenda Retiro,06,,-3.8,-39.866667
br,fazenda retiro,Fazenda Retiro,20,,-7.2,-42.516667
br,fazenda retiro seguro,Fazenda Retiro Seguro,11,,-18.2,-56.75
br,fazenda retriano,Fazenda Retriano,06,,-4.416667,-38.166667
br,fazenda riachao,Fazenda Riachão,13,,-6.2,-43.25
br,fazenda riacho cipo,Fazenda Riacho Cipó,05,,-12.35,-39.3
br,fazenda riacho da areia,Fazenda Riacho da Areia,20,,-7.133333,-42.233333
br,fazenda riacho das umburanas,Fazenda Riacho das Umburanas,06,,-4.616667,-38.183333
br,fazenda riacho do divertido,Fazenda Riacho do Divertido,06,,-4.7,-38.166667
br,fazenda riacho do mato,Fazenda Riacho do Mato,06,,-4.283333,-40.433333
br,fazenda riacho do meio,Fazenda Riacho do Meio,06,,-3.716667,-39
br,fazenda riacho do onca,Fazenda Riacho do Onca,20,,-5.583333,-41.866667
br,fazenda riacho dos bois,Fazenda Riacho dos Bois,06,,-4.383333,-39.966667
br,fazenda riacho,Fazenda Riacho,06,,-3.266667,-40.483333
br,fazenda riacho fundo,Fazenda Riacho Fundo,06,,-3.55,-39.266667
br,fazenda riacho novo,Fazenda Riacho Novo,06,,-4.166667,-40.233333
br,fazenda ribeirao claro,Fazenda Ribeirão Claro,11,,-21.05,-53.633333
br,fazenda rio alegre,Fazenda Rio Alegre,14,,-16.8925,-56.720833
br,fazenda rio branco,Fazenda Rio Branco,11,,-21.666667,-53.9
br,fazenda rio branco,Fazenda Rio Branco,20,,-6.9,-42.9
br,fazenda rio manso,Fazenda Rio Manso,27,,-23.366667,-45.4
br,fazenda rio verde,Fazenda Rio Verde,11,,-18.933333,-54.866667
br,fazenda rocilandia,Fazenda Rocilândia,06,,-4.233333,-39.666667
br,fazenda romualda,Fazenda Romualda,15,,-16.066667,-40.466667
br,fazenda rosaria,Fazenda Rosária,27,,-21.466667,-46.783333
br,fazenda rosendo pinto,Fazenda Rosendo Pinto,15,,-17.833333,-41.516667
br,fazenda sabonete,Fazenda Sabonete,06,,-4.033333,-39.7
br,fazenda saco do adelino,Fazenda Saco do Adelino,20,,-5.25,-41.1
br,fazenda saco do vento,Fazenda Saco do Vento,06,,-4,-39.816667
br,fazenda saco,Fazenda Saco,06,,-3.216667,-39.95
br,fazenda saco,Fazenda Saco,20,,-5.183333,-41.183333
br,fazenda saco,Fazenda Saco,30,,-7.916667,-40.216667
br,fazenda saco verde,Fazenda Saco Verde,06,,-3.733333,-39.683333
br,fazenda salgadinho,Fazenda Salgadinho,06,,-3.466667,-40.166667
br,fazenda salgado,Fazenda Salgado,06,,-3.866667,-39.4
br,fazenda salto,Fazenda Salto,11,,-20.583333,-54.683333
br,fazenda salto,Fazenda Salto,27,,-21.8,-47.766667
br,fazenda salvacao,Fazenda Salvação,06,,-4.983333,-39.15
br,fazenda santa alda,Fazenda Santa Alda,11,,-20.716667,-53.766667
br,fazenda santa ambrosina,Fazenda Santa Ambrosina,27,,-21.666667,-47.466667
br,fazenda santa ana,Fazenda Santa Ana,06,,-3.816667,-39.466667
br,fazenda santa barbara,Fazenda Santa Barbara,11,,-22.25,-53.483333
br,fazenda santa barbara,Fazenda Santa Barbara,14,,-15.65,-59.15
br,fazenda santa catarina,Fazenda Santa Catarina,11,,-23.116667,-55.533333
br,fazenda santa catarina,Fazenda Santa Catarina,20,,-5.016667,-42.016667
br,fazenda santa cruz,Fazenda Santa Cruz,11,,-18.533333,-57.2
br,fazenda santa escolastica,Fazenda Santa Escolástica,27,,-21.85,-47.233333
br,fazenda santa etelvina,Fazenda Santa Etelvina,27,,-21.766667,-47.633333
br,fazenda santa eugenia,Fazenda Santa Eugênia,27,,-21.816667,-47.35
br,fazenda santa fe,Fazenda Santa Fé,06,,-3.616667,-39.3
br,fazenda santa fe,Fazenda Santa Fé,11,,-18.633333,-55.8
br,fazenda santa fe,Fazenda Santa Fé,20,,-7.1,-42.25
br,fazenda santa fe,Fazenda Santa Fé,22,,-4.95,-37.266667
br,fazenda santa fe,Fazenda Santa Fé,27,,-20.933333,-47.883333
br,fazenda santa gertrudes,Fazenda Santa Gertrudes,27,,-20.65,-47.866667
br,fazenda santa helena,Fazenda Santa Helena,06,,-4.4,-39.133333
br,fazenda santa helena,Fazenda Santa Helena,20,,-7.15,-41.166667
br,fazenda santa herminia,Fazenda Santa Hermínia,27,,-20.266667,-47.533333
br,fazenda santa ilidia,Fazenda Santa Ilídia,11,,-22.416667,-52.883333
br,fazenda santa ines,Fazenda Santa Inês,27,,-20.566667,-47.816667
br,fazenda santa luzia,Fazenda Santa Lúzia,06,,-3.833333,-39.25
br,fazenda santa luzia,Fazenda Santa Luzia,11,,-17.933333,-54.95
br,fazenda santa luzia,Fazenda Santa Luzia,20,,-5.683333,-41.483333
br,fazenda santa luzia,Fazenda Santa Luzia,27,,-21.816667,-47.733333
br,fazenda santa maria,Fazenda Santa Maria,06,,-3.316667,-39.916667
br,fazenda santa maria,Fazenda Santa Maria,14,,-16.716667,-56.033333
br,fazenda santa maria,Fazenda Santa Maria,18,,-23.183333,-50.533333
br,fazenda santa maria,Fazenda Santa Maria,27,,-20.416667,-47.716667
br,fazenda santa marta,Fazenda Santa Marta,30,,-7.35,-39.633333
br,fazenda santana do piraim,Fazenda Santana do Piraim,14,,-16.333333,-56.083333
br,fazenda santana,Fazenda Santana,06,,-3.85,-40.016667
br,fazenda santa naria,Fazenda Santa Naria,27,,-20.65,-47.416667
br,fazenda santa paz,Fazenda Santa Paz,20,,-7.3,-42.166667
br,fazenda santa rita,Fazenda Santa Rita,14,,-16.313611,-56.590278
br,fazenda santa rita,Fazenda Santa Rita,20,,-7.733333,-42.716667
br,fazenda santa rita,Fazenda Santa Rita,23,,-30.916667,-51.483333
br,fazenda santa rosa,Fazenda Santa Rosa,06,,-3.466667,-38.983333
br,fazenda santa rosa,Fazenda Santa Rosa,20,,-7.05,-41.25
br,fazenda santa sofia,Fazenda Santa Sofia,11,,-21.583333,-53.733333
br,fazenda santa sophia,Fazenda Santa Sophia,11,,-21.583333,-53.733333
br,fazenda santa teresa,Fazenda Santa Teresa,06,,-4.15,-39.1
br,fazenda santa teresinha,Fazenda Santa Teresinha,27,,-21.216667,-47.683333
br,fazenda santa tereza,Fazenda Santa Tereza,20,,-7.166667,-42.066667
br,fazenda santa victoria,Fazenda Santa Victória,20,,-5.15,-42.3
br,fazenda santiago,Fazenda Santiago,27,,-20.6,-47.766667
br,fazenda santo antonia,Fazenda Santo Antonia,22,,-5.45,-36.883333
br,fazenda santo antonio,Fazenda Santo Antõnio,06,,-2.95,-40.116667
br,fazenda santo antonio,Fazenda Santo Antonio,08,,-20.45,-41.266667
br,fazenda santo antonio,Fazenda Santo Antônio,11,,-21.533333,-56.983333
br,fazenda santo antonio,Fazenda Santo Antônio,27,,-20.616667,-47.85
br,fazenda santo expedito,Fazenda Santo Expedito,20,,-5.116667,-41.433333
br,fazenda santo isidio,Fazenda Santo Isídio,06,,-3.116667,-40.25
br,fazenda santos dumont,Fazenda Santos Dumont,06,,-4.85,-40.366667
br,fazenda sao antonio,Fazenda São Antônio,06,,-4.166667,-39.066667
br,fazenda sao benedito,Fazenda São Benedito,11,,-18.8,-56.016667
br,fazenda sao bento,Fazenda São Bento,11,,-17.983333,-54.9
br,fazenda sao bernardo,Fazenda São Bernardo,06,,-3.833333,-40.066667
br,fazenda sao claudio,Fazenda São Claúdio,06,,-3.866667,-40.1
br,fazenda sao clemente,Fazenda São Clemente,27,,-21.816667,-47.583333
br,fazenda sao cristovao,Fazenda São Cristovão,06,,-4.1,-39.1
br,fazenda sao domingos,Fazenda São Domingos,11,,-18.45,-55.966667
br,fazenda sao felix,Fazenda São Felix,11,,-19.95,-55.2
br,fazenda sao fernandes,Fazenda São Fernandes,06,,-3.616667,-40.3
br,fazenda sao francisco do venacio,Fazenda São Francisco do Venâcio,06,,-3.75,-39.516667
br,fazenda sao francisco,Fazenda São Francisco,06,,-3.716667,-40.416667
br,fazenda sao francisco,Fazenda São Francisco,11,,-19.75,-55.216667
br,fazenda sao francisco,Fazenda São Francisco,20,,-5.45,-41.1
br,fazenda sao geraldo,Fazenda São Geraldo,27,,-21.05,-47.75
br,fazenda sao gregorio,Fazenda São Gregório,06,,-4.533333,-39.133333
br,fazenda sao guilherme,Fazenda São Guilherme,27,,-22,-47.633333
br,fazenda sao jeronimo,Fazenda São Jerônimo,27,,-20.866667,-47.916667
br,fazenda sao joao da boa vista,Fazenda São João da Boa Vista,06,,-3.783333,-38.716667
br,fazenda sao joao do guapore,Fazenda São João do Guaporé,14,,-15.977778,-59.886111
br,fazenda sao joao,Fazenda São João,06,,-3.283333,-39.616667
br,fazenda sao joao,Fazenda São João,11,,-18.333333,-55.883333
br,fazenda sao joao,Fazenda São João,14,,-14.429167,-59.566667
br,fazenda sao joao,Fazenda São João,27,,-20.983333,-47.833333
br,fazenda sao joaquim,Fazenda São Joaquim,06,,-3.733333,-39.7
br,fazenda sao joaquim,Fazenda São Joaquim,11,,-20.666667,-53.9
br,fazenda sao joaquim,Fazenda São Joaquim,27,,-21.016667,-47.85
br,fazenda sao jose do rio negrinho,Fazenda São José do Rio Negrinho,11,,-19.1,-54.916667
br,fazenda sao jose,Fazenda São José,04,,-9.25,-67.033333
br,fazenda sao jose,Fazenda São José,06,,-3.483333,-39.166667
br,fazenda sao jose,Fazenda São José,11,,-21.4,-54.366667
br,fazenda sao jose,Fazenda São José,27,,-20.2,-47.383333
br,fazenda sao luis,Fazenda São Luís,06,,-4.466667,-39.966667
br,fazenda sao luis,Fazenda São Luis,22,,-4.966667,-37.433333
br,fazenda sao luis,Fazenda São Luís,27,,-20.383333,-47.616667
br,fazenda sao manoel,Fazenda São Manoel,11,,-20.766667,-53
br,fazenda sao mateus,Fazenda São Mateus,06,,-4.516667,-38.266667
br,fazenda sao miguel,Fazenda São Miguel,14,,-16.866667,-56.633333
br,fazenda sao miguel,Fazenda São Miguel,27,,-21.05,-47.95
br,fazenda sao natalia,Fazenda São Natalia,11,,-18.366667,-55.983333
br,fazenda sao paulo,Fazenda São Paulo,06,,-4.25,-39.416667
br,fazenda sao pedro,Fazenda São Pedro,06,,-4.266667,-40.2
br,fazenda sao pedro,Fazenda São Pedro,11,,-21.833333,-53.6
br,fazenda sao rafael,Fazenda São Rafael,11,,-22.166667,-53.266667
br,fazenda sao raimundo,Fazenda São Raimundo,06,,-4.166667,-38.183333
br,fazenda sao roberto,Fazenda São Roberto,27,,-21.85,-47.966667
br,fazenda sao romao,Fazenda São Romão,06,,-3.466667,-39.883333
br,fazenda sao romao,Fazenda São Romão,11,,-18.55,-54.45
br,fazenda sao roque,Fazenda São Roque,11,,-21.7,-54.25
br,fazenda sao salvador,Fazenda São Salvador,06,,-4.6,-39.316667
br,fazenda sao sebastiao,Fazenda São Sebastião,27,,-22.133333,-49.2
br,fazenda sao vicente,Fazenda São Vicente,06,,-4.15,-39.016667
br,fazenda sao vicente,Fazenda São Vicente,15,,-16.533333,-41.033333
br,fazenda sao vicente,Fazenda São Vicente,20,,-5.083333,-42.333333
br,fazenda sapucaiba,Fazenda Sapucaiba,06,,-4.6,-39.916667
br,fazenda saquinho,Fazenda Saquinho,06,,-3.516667,-39.7
br,fazenda sebastiao ferreira,Fazenda Sebastião Ferreira,15,,-17.466667,-41.433333
br,fazenda sebastiao goncalves,Fazenda Sebastião Gonçalves,15,,-16.916667,-41.733333
br,fazenda serra branca,Fazenda Serra Branca,06,,-4.333333,-39.55
br,fazenda serra branca,Fazenda Serra Branca,20,,-6.066667,-41.45
br,fazenda serrinha,Fazenda Serrinha,06,,-3.6,-39.733333
br,fazenda serrinha,Fazenda Serrinha,27,,-21.483333,-47.516667
br,fazenda serrote do meio,Fazenda Serrote do Meio,06,,-3.8,-39.483333
br,fazenda serrote,Fazenda Serrote,06,,-4.283333,-39.116667
br,fazenda sibirol,Fazenda Sibirol,06,,-4.933333,-39.8
br,fazenda sipauba,Fazenda Sipaúba,30,,-7.55,-40.333333
br,fazenda sitio,Fazenda Sítio,06,,-4.083333,-39.75
br,fazenda soares,Fazenda Soares,06,,-3.233333,-40.233333
br,fazenda sobrado,Fazenda Sobrado,27,,-20.283333,-47.683333
br,fazenda sombreira,Fazenda Sombreira,06,,-4.083333,-39.1
br,fazenda sororo,Fazenda Sororô,06,,-3.466667,-39.633333
br,fazenda sousa,Fazenda Sousa,23,,-29.1236,-51.0127
br,fazenda souza,Fazenda Souza,23,,-29.1236,-51.0127
br,fazenda souze,Fazenda Souze,06,,-4.05,-39.55
br,fazenda s romao,Fazenda S Romão,22,,-4.9,-37.4
br,fazenda sta alice,Fazenda Sta Alice,20,,-5.266667,-41.566667
br,fazenda sta madalena,Fazenda Sta Madalena,20,,-5.016667,-41.833333
br,fazenda sta maria,Fazenda Sta Maria,20,,-5.3,-41.883333
br,fazenda sucuarana,Fazenda Suçuarana,06,,-3.3,-40.383333
br,fazenda sucuri,Fazenda Sucuri,11,,-21.766667,-54.75
br,fazenda sumare,Fazenda Sumaré,06,,-3.666667,-39.166667
br,fazenda taboca,Fazenda Taboca,06,,-3.65,-39.016667
br,fazenda tabocas,Fazenda Tabocas,06,,-3.416667,-39.716667
br,fazenda tabuleiro,Fazenda Tabuleiro,20,,-5.516667,-41.716667
br,fazenda tabuleiro grande,Fazenda Tabuleiro Grande,20,,-6.916667,-42.45
br,fazenda tagy,Fazenda Tagy,20,,-8.733333,-45.866667
br,fazenda talhada,Fazenda Talhada,20,,-6.883333,-41.966667
br,fazenda tanque do aquino,Fazenda Tanque do Aquino,14,,-16.011667,-56.539167
br,fazenda tanque,Fazenda Tanque,06,,-4.683333,-39.966667
br,fazenda tapera,Fazenda Tapera,06,,-4.583333,-39.8
br,fazenda taquaperi,Fazenda Taquaperi,11,,-23.216667,-55.416667
br,fazenda taquara,Fazenda Taquara,11,,-19.283333,-54.933333
br,fazenda taquara,Fazenda Taquara,18,,-24.683333,-50.166667
br,fazenda taquaral,Fazenda Taquaral,27,,-21.683333,-47.566667
br,fazenda tarao,Fazenda Tarão,20,,-7.15,-42.866667
br,fazenda tatajuba,Fazenda Tatajuba,06,,-4.7,-39.833333
br,fazenda taubate,Fazenda Taubaté,06,,-3.983333,-38.983333
br,fazenda tejuba,Fazenda Tejubá,06,,-3.366667,-39.9
br,fazenda telemaco guimaraes,Fazenda Telêmaco Guimarães,15,,-17.65,-41.916667
br,fazenda telha,Fazenda Telha,06,,-3.733333,-40.3
br,fazenda terere,Fazenda Tererê,11,,-21.316667,-55.033333
br,fazenda terra seca,Fazenda Terra Sêca,06,,-4.083333,-38.666667
br,fazenda timbauba,Fazenda Timbaúba,06,,-4.15,-39.566667
br,fazenda toca,Fazenda Toca,27,,-21.3,-47.533333
br,fazenda todos os santos,Fazenda Todos os Santos,06,,-4.283333,-39.566667
br,fazenda torroes,Fazenda Torrões,20,,-7.083333,-41.216667
br,fazenda torto,Fazenda Torto,06,,-3.2,-39.8
br,fazenda touro,Fazenda Touro,06,,-3.95,-39.533333
br,fazenda trairas,Fazenda Traíras,06,,-4.483333,-40.2
br,fazenda trapia,Fazenda Trapiá,06,,-4.2,-39.066667
br,fazenda tremendal,Fazenda Tremendal,20,,-7.116667,-42.45
br,fazenda tres barras,Fazenda Três Barras,11,,-22.066667,-52.933333
br,fazenda tres barras,Fazenda Três Barras,27,,-20.733333,-47.816667
br,fazenda tres lagoas,Fazenda Três Lagoas,06,,-3.766667,-39.2
br,fazenda tres marias,Fazenda Três Marias,06,,-4.3,-39.866667
br,fazenda tres marias,Fazenda Três Marias,11,,-19.133333,-55.35
br,fazenda triangulo,Fazenda Triângulo,11,,-21.683333,-54.866667
br,fazenda triunfo,Fazenda Triunfo,11,,-21.433333,-56.483333
br,fazenda trombas,Fazenda Trombas,06,,-3.766667,-39.616667
br,fazenda trombucas,Fazenda Trombucas,27,,-20.216667,-47.3
br,fazenda tubiba,Fazenda Tubiba,06,,-4.316667,-40.416667
br,fazenda turvo,Fazenda Turvo,11,,-21.35,-55.516667
br,fazenda umari,Fazenda Umari,01,,-4.416667,-38.033333
br,fazenda umburana,Fazenda Umburana,06,,-3.916667,-39.783333
br,fazenda urubu,Fazenda Urubu,06,,-3.883333,-39.883333
br,fazenda vaca branca,Fazenda Vaca Branca,06,,-4.116667,-39.566667
br,fazenda vaca serrada,Fazenda Vaca Serrada,06,,-4.833333,-39.45
br,fazenda vale do guapore,Fazenda Vale do Guaporé,14,,-14.420833,-59.595833
br,fazenda valparaiso,Fazenda Valparaiso,06,,-4.233333,-39.816667
br,fazenda vargem comprida,Fazenda Vargem Comprida,14,,-16.316667,-56.7175
br,fazenda vargem da cruz,Fazenda Vargem da Cruz,06,,-4.066667,-40.083333
br,fazenda varginha,Fazenda Varginha,27,,-21.45,-46.9
br,fazenda varjota,Fazenda Varjota,06,,-3.65,-39.416667
br,fazenda varzea da pedra,Fazenda Varzea da Pedra,06,,-3.733333,-40.416667
br,fazenda varzea de ema,Fazenda Várzea de Ema,06,,-4.316667,-39.283333
br,fazenda varzea do jua,Fazenda Várzea do Juá,06,,-4.25,-40.216667
br,fazenda varzea dos bois,Fazenda Varzea dos Bois,06,,-3.516667,-39.866667
br,fazenda varzea grande,Fazenda Várzea Grande,06,,-4.8,-40.416667
br,fazenda varzea queimada,Fazenda Várzea Queimada,06,,-4.616667,-39.016667
br,fazenda varzea redonda,Fazenda Várzea Redonda,06,,-4.35,-40.4
br,fazenda varzera de cima,Fazenda Várzera de Cima,06,,-4.366667,-39.933333
br,fazenda varzinha,Fazenda Varzinha,06,,-4.866667,-38.866667
br,fazenda vazantes do barroso,Fazenda Vazantes do Barroso,06,,-4.466667,-39.766667
br,fazenda vazea grande,Fazenda Vázea Grande,06,,-4.866667,-38.25
br,fazenda velha,Fazenda Velha,05,,-10.416667,-42.316667
br,fazenda velha,Fazenda Velha,14,,-15.783333,-56.133333
br,fazenda velha,Fazenda Velha,20,,-7.566667,-42.466667
br,fazenda velha,Fazenda Velha,27,,-21.366667,-47.083333
br,fazenda venancio,Fazenda Venâncio,06,,-3.783333,-39.566667
br,fazenda vencaia,Fazenda Vencaia,18,,-23.066667,-50.55
br,fazenda veneza,Fazenda Veneza,06,,-4.566667,-39.166667
br,fazenda vento frio,Fazenda Vento Frio,15,,-16.6,-40.466667
br,fazenda vereda grande,Fazenda Vereda Grande,20,,-7.266667,-42.983333
br,fazenda vertedouro,Fazenda Vertedouro,06,,-4.783333,-38.433333
br,fazenda vertente,Fazenda Vertente,13,,-5.25,-42.833333
br,fazenda vila popular,Fazenda Vila Popular,06,,-3.6,-40.016667
br,fazenda vista alegre,Fazenda Vista Alegre,04,,-4.896944,-60.022222
br,fazenda vista alegre,Fazenda Vista Alegre,06,,-4.633333,-40.2
br,fazenda vista alegre,Fazenda Vista Alegre,11,,-19.816667,-53.966667
br,fazenda vitoria,Fazenda Vitória,06,,-4.583333,-40.166667
br,fazenda vitoriano,Fazenda Vitoriano,06,,-4.45,-38.166667
br,fazenda volta do mateus,Fazenda Volta do Mateus,06,,-4.916667,-38.333333
br,fazenda volta,Fazenda Volta,06,,-4.633333,-40.433333
br,fazenda xinuaque,Fazenda Xinuaquê,06,,-4.266667,-39.633333
br,fazenda xixa,Fazenda Xixá,06,,-3.666667,-39.45
br,fazenda zeca botelho,Fazenda Zeca Botelho,15,,-16.533333,-40.416667
br,fazenda ziemer e companhia,Fazenda Ziemer e Companhia,15,,-17.133333,-41.35
br,fazendinha,Fazendinha,02,,-9.55,-37.116667
br,fazendinha,Fazendinha,04,,-3.5,-58.833333
br,fazendinha,Fazendinha,05,,-11.716667,-38.6
br,fazendinha,Fazendinha,06,,-3.066667,-39.733333
br,fazendinha,Fazendinha,11,,-19.266667,-52.616667
br,fazendinha,Fazendinha,13,,-5.316667,-42.833333
br,fazendinha,Fazendinha,14,,-14.483333,-56.6
br,fazendinha,Fazendinha,15,,-20.5,-41.85
br,fazendinha,Fazendinha,16,,-.933333,-47.733333
br,fazendinha,Fazendinha,18,,-25.433333,-53.016667
br,fazendinha,Fazendinha,20,,-8.083333,-41.066667
br,fazendinha,Fazendinha,21,,-22.766667,-41.966667
br,fazendinha,Fazendinha,22,,-5.166667,-35.65
br,fazendinha,Fazendinha,27,,-21.033333,-50.7
br,fazendinha,Fazendinha,28,,-10.4,-36.616667
br,fazendinha,Fazendinha,30,,-8.883333,-36.983333
br,faz exu,Faz Exu,14,,-16.901667,-56.607778
br,faz ichu,Faz Ichu,14,,-16.901667,-56.607778
br,fechadinho,Fechadinho,17,,-6.633333,-37.583333
br,fechado,Fechado,06,,-5.833333,-39.35
br,fechados,Fechados,15,,-18.816667,-43.866667
br,fedegoso,Fedegoso,05,,-11.466667,-40.883333
br,federacao,Federação,23,,-32.233333,-53.083333
br,feia,Feia,15,,-14.883333,-44.9
br,feijao branco,Feijão Branco,06,,-3.183333,-40.416667
br,feijao,Feijão,04,,-3.088889,-63.124722
br,feijao,Feijão,05,,-9.466667,-43.533333
br,feijao,Feijão,06,,-4.05,-39.716667
br,feijao,Feijão,13,,-3.492222,-43.485556
br,feijao,Feijão,17,,-6.683333,-38.583333
br,feijao,Feijão,20,,-3.645833,-42.256667
br,feijao,Feijão,22,,-5.816667,-35.666667
br,feijao,Feijão,30,,-7.766667,-38.733333
br,feijao gordo,Feijão Gordo,06,,-6.233333,-40.683333
br,feijao sem sal,Feijão Sem Sal,14,,-15.383333,-57.7
br,feijoal,Feijoal,04,,-4.317778,-69.561111
br,feijo,Feijó,01,12951,-8.15,-70.35
br,feijo,Feijo,17,,-7,-36.016667
br,feijo,Feijó,23,,-32.516667,-53.1
br,feira de saint anna,Feira De Saint Anna,05,,-12.25,-38.95
br,feira de santana,Feira de Santana,05,481926,-12.25,-38.95
br,feira de sant'anna,Feira de Sant'Anna,05,,-12.25,-38.95
br,feira do pau,Feira do Páu,05,,-9.166667,-39.066667
br,feira,Feira,05,,-12.25,-38.95
br,feira,Feira,13,,-6.45,-44.2
br,feira grande,Feira Grande,02,3985,-9.85,-36.7
br,feira nova,Feira Nova,30,12985,-7.933333,-35.35
br,feira sucupira,Feira Sucupira,13,,-6.45,-44.2
br,feirinha,Feirinha,05,,-14.1,-40.383333
br,feital,Feital,18,,-24.723816,-48.838678
br,feital,Feital,27,,-24.503077,-48.158547
br,feiticeira,Feiticeira,06,,-5.966667,-38.816667
br,feiticeiro,Feiticeiro,06,,-5.966667,-38.816667
br,feitoria da guariroba,Feitoria da Guariroba,14,,-13.483333,-58.333333
br,feitoria do guariroba,Feitoria do Guariroba,14,,-13.483333,-58.333333
br,feitoria do orlando,Feitoria do Orlando,14,,-14.166667,-57.166667
br,feitoria do papagaio,Feitoria do Papagaio,14,,-13.466667,-58.35
br,feitoria dos 3 jacus,Feitoria dos 3 Jacús,14,,-13.75,-57.333333
br,feitoria dos tres jacus,Feitoria dos Três Jacus,14,,-13.75,-57.333333
br,feitoria do tine,Feitoria do Tiné,14,,-14.583333,-51.516667
br,feitoria,Feitoria,13,,-4.216667,-43.233333
br,feitoria,Feitoria,20,,-2.909167,-41.774722
br,feitoria,Feitoria,30,,-7.533333,-40.1
br,feitoria gelo,Feitoria Gelo,16,,-9.233333,-57.05
br,feitoria pedro pinheiro,Feitoria Pedro Pinheiro,14,,-13.35,-58.416667
br,feitosa,Feitosa,06,,-6.966667,-39.183333
br,feixes,Feixes,16,,-4.8,-56.45
br,feixinho de criviuma,Feixinho de Criviúma,18,,-24.744893,-48.939514
br,fejos,Fejós,06,,-7.566667,-39.133333
br,feliciano,Feliciano,15,,-21.833333,-45.166667
br,feliciano,Feliciano,30,,-8.166667,-37.25
br,feliciano trindade,Feliciano Trindade,11,,-19.083333,-54.233333
br,felicidade,Felicidade,04,,-4.424722,-59.811389
br,felicidade,Felicidade,05,,-16.2,-38.983333
br,felicidade,Felicidade,06,,-7.633333,-38.85
br,felicidade,Felicidade,16,,-.966667,-49.316667
br,felicidale,Felicidale,06,,-5.6,-38.55
br,felicina,Felicina,15,,-19.083333,-42.4
br,felicio dos santos,Felício dos Santos,15,,-18.083333,-43.183333
br,felicio,Felicio,15,,-16.433333,-40.233333
br,felicio peixoto,Felício Peixoto,15,,-20.583333,-46.233333
br,felinto,Felinto,04,,-6.95,-58.283333
br,felipe camarao,Felipe Camarão,13,,-2.85,-46.6
br,felipe camarao,Felipe Camarão,22,,-5.766667,-35.333333
br,felipe,Felipe,06,,-4.833333,-40.2
br,felipe guerra,Felipe Guerra,22,,-5.6,-37.683333
br,felipe schmidt,Felipe Schmidt,26,,-26.2,-50.7
br,felisberto caldeira,Felisberto Caldeira,15,,-18.016667,-43.316667
br,felisberto caldeiro,Felisberto Caldeiro,15,,-18.016667,-43.316667
br,felisberto,Felisberto,13,,-3.514167,-42.720556
br,felisburgo,Felisburgo,15,,-16.65,-40.766667
br,felix,Félix,15,,-18.55,-48.533333
br,felix,Felix,27,,-23.1042,-47.9896
br,felixlandia,Felixlândia,15,,-18.783333,-44.916667
br,felizardo,Felizardo,06,,-6.85,-38.75
br,felizburgo,Felizburgo,15,,-16.65,-40.766667
br,feliz deserto,Feliz Deserto,02,,-10.283333,-36.3
br,feliz do deserto,Feliz do Deserto,02,,-10.283333,-36.3
br,feliz,Feliz,15,,-20.616667,-44.85
br,feliz,Feliz,23,,-29.45,-51.3
br,feliz vitoria,Feliz Vitória,04,,-5.653056,-61.197778
br,fellipe dos santos,Fellipe dos Santos,15,,-20.516667,-43.8
br,fenix,Fênix,18,,-23.9,-51.95
br,ferias,Férias,13,,-3.242778,-42.208889
br,fermao dias,Fermão Dias,27,,-22.35,-49.516667
br,fernanda,Fernanda,05,,-13.683333,-41.583333
br,fernandes belo,Fernandes Belo,16,,-1.116667,-46.316667
br,fernandes,Fernandes,05,,-11.883333,-41.533333
br,fernandes,Fernandes,06,,-4.883333,-39.85
br,fernandes,Fernandes,13,,-3.529167,-43.399444
br,fernandes,Fernandes,15,,-19.333333,-42.716667
br,fernandes,Fernandes,22,,-5.066667,-37.333333
br,fernandes figueira,Fernandes Figueira,21,,-22.116667,-43.866667
br,fernandes pinheiro,Fernandes Pinheiro,18,,-25.416,-50.5498
br,fernandes pinheiro,Fernandes Pinheiro,21,,-22.066667,-43.216667
br,fernandes ribeiro,Fernandes Ribeiro,21,,-22.066667,-43.216667
br,fernando bueno,Fernando Bueno,15,,-20.8,-46.166667
br,fernando,Fernando,20,,-3.865278,-42.710278
br,fernando pedrosa,Fernando Pedrosa,22,,-5.697222,-36.530556
br,fernando pedroza,Fernando Pedroza,22,,-5.697222,-36.530556
br,fernandopolis,Fernandópolis,27,61932,-20.266667,-50.233333
br,fernando prestes,Fernando Prestes,27,,-21.266667,-48.716667
br,fernao dias,Fernão Dias,15,,-16.378889,-44.443333
br,fernao dias,Fernão Dias,18,,-23.1,-51.733333
br,fernao dias,Fernão Dias,27,,-22.35,-49.516667
br,fernao,Fernão,27,,-22.35,-49.516667
br,fernao velho,Fernão Velho,02,,-9.583333,-35.766667
br,ferquim,Ferquim,15,,-20.366667,-43.216667
br,ferradas,Ferradas,05,,-14.85,-39.35
br,ferrao,Ferrão,06,,-5.9,-40.35
br,ferrao,Ferrão,27,,-23.462474,-46.74787
br,ferrao,Ferrão,30,,-8.75,-37.066667
br,ferraria,Ferraria,18,,-25.45,-49.4
br,ferraz de vasconcelos,Ferraz de Vasconcelos,27,167904,-23.533333,-46.366667
br,ferraz,Ferraz,27,,-23.640301,-47.46586
br,ferraz,Ferraz,30,,-7.816667,-36.016667
br,ferreira,Ferreira,04,,-3.457222,-62.620278
br,ferreira,Ferreira,05,,-12.7,-40.983333
br,ferreira,Ferreira,15,,-20.9,-46.616667
br,ferreira,Ferreira,23,,-29.983333,-52.983333
br,ferreira gomes,Ferreira Gomes,03,,.8,-51.133333
br,ferreiro,Ferreiro,30,,-7.716667,-39.966667
br,ferreiropolis,Ferreirópolis,15,,-15.983333,-42.216667
br,ferreiros,Ferreiros,15,,-19.133333,-43
br,ferreiros,Ferreiros,21,,-22.416667,-43.566667
br,ferreiros,Ferreiros,30,,-7.4,-35.266667
br,ferreiro torto,Ferreiro Torto,22,,-5.85,-35.316667
br,ferrete,Ferrete,05,,-8.833333,-39.75
br,ferro novo,Ferro Novo,04,,-3.666667,-68.75
br,ferros,Ferros,15,,-19.233333,-43.033333
br,ferro velho,Ferro Velho,04,,-3.783333,-68.766667
br,ferrugem,Ferrugem,05,,-12.05,-40.766667
br,ferrugem,Ferrugem,13,,-9.883333,-46.466667
br,ferrugem,Ferrugem,23,,-29.9166,-50.7932
br,ferrugen,Ferrugen,13,,-9.883333,-46.466667
br,ferruginha,Ferruginha,15,,-18.9,-41.233333
br,fervedouro,Fervedouro,05,,-12.133333,-41.35
br,fervedouro,Fervedouro,15,,-20.733333,-42.283333
br,fetoria,Fetoria,16,,-9.233333,-57.05
br,fiacao,Fiação,02,,-9.866667,-37.05
br,fibuzanto sousa,Fibuzanto Sousa,15,,-17.116667,-41.416667
br,fidalga,Fidalga,06,,-5.483333,-40.183333
br,fidalgo,Fidalgo,15,,-19.516667,-43.966667
br,fidalgo,Fidalgo,22,,-6.316667,-38.066667
br,fidelandia,Fidelândia,15,,-18.2,-41.25
br,fidelis,Fidelis,04,,-5.574167,-59.569167
br,fidelo,Fidelo,22,,-5.124167,-36.061389
br,fidie,Fidié,13,,-4.283333,-43.316667
br,figueira branca,Figueira Branca,27,,-21.75,-47.766667
br,figueira da bandeira,Figueira da Bandeira,14,,-16.983333,-58.333333
br,figueira de braco,Figueira de Braço,18,,-25.422064,-48.817991
br,figueira de santa joanna,Figueira de Santa Joanna,08,,-19.85,-40.866667
br,figueira,Figueira,08,,-19.85,-40.866667
br,figueira,Figueira,11,,-17.666667,-56.85
br,figueira,Figueira,14,,-16.066667,-56.833333
br,figueira,Figueira,15,,-18.85,-41.933333
br,figueira,Figueira,17,,-7.666667,-38.283333
br,figueira,Figueira,18,,-23.334954,-50.234613
br,figueira,Figueira,23,,-28.416667,-53.266667
br,figueira,Figueira,26,,-28.466667,-49.366667
br,figueira,Figueira,27,,-20.833333,-48.75
br,figueiras,Figueiras,21,,-22.88896,-43.571589
br,figueiras,Figueiras,23,,-28.416667,-53.266667
br,figueiredo,Figueiredo,05,,-13.366667,-39.2
br,figueiredo,Figueiredo,06,,-6.333333,-40.066667
br,figueiredo,Figueiredo,22,,-6.266667,-37.833333
br,figueiredo,Figueiredo,30,,-7.916667,-39.266667
br,figueirinha,Figueirinha,11,,-19.083333,-57.2
br,figueropolis,Figuerópolis,31,,-12.141389,-49.114444
br,filadelfia,Filadélfia,24,,-9.1,-61.666667
br,filadelfia,Filadélfia,31,,-7.327222,-47.366111
br,filadeli,Filadéli,05,,-10.733333,-40.133333
br,filho d'anta,Filho d'Anta,05,,-11.5,-40.433333
br,filho dantas,Filho Dantas,05,,-11.5,-40.433333
br,filipe camarao,Filipe Camarão,13,,-2.85,-46.6
br,filipe camarao,Filipe Camarão,22,,-5.766667,-35.333333
br,filipe,Filipe,05,,-13.35,-40.15
br,filipe,Filipe,11,,-20.2,-55.383333
br,filipe,Filipe,15,,-14.516667,-44.633333
br,filipe turco,Filipe Turco,15,,-20.633333,-46.616667
br,filipson,Filipson,23,,-29.533333,-53.766667
br,filomena,Filomena,06,,-5.2,-40.8
br,fim do campo,Fim do Campo,27,,-23.120556,-46.715
br,fim do pasto,Fim do Pasto,13,,-3.555,-42.991944
br,final feliz,Final Feliz,21,,-22.825584,-43.387766
br,fineza,Fineza,20,,-6.6,-41.266667
br,finlandia,Finlândia,06,,-5.6,-40.05
br,fino,Finô,15,,-16.266667,-40.033333
br,fiocruz,Fiocruz,21,,-22.941072,-43.404278
br,fioduardo,Fioduardo,23,,-30.25,-55.116667
br,fioropolis,Fiorópolis,18,,-22.7,-52.15
br,firmamento,Firmamento,22,,-5.5175,-36.194444
br,firme,Firme,11,,-19.266667,-57.016667
br,firmiano,Firmiano,15,,-16.216667,-40.833333
br,firmino alves,Firmino Alves,05,,-14.983333,-39.933333
br,firmino candido,Firmino Cândido,15,,-20.35,-46.7
br,firmino ferraz,Firmino Ferraz,15,,-16.266667,-40.466667
br,firmino,Firmino,05,,-12.133333,-41.05
br,firmino,Firmino,06,,-5.2,-38.25
br,firmino,Firmino,15,,-16.066667,-40.366667
br,firmino grifo,Firmino Grifo,08,,-20.2,-41.3
br,firmino grilo,Firmino Grilo,08,,-20.2,-41.3
br,firminopolis,Firminópolis,29,9084,-16.666667,-50.316667
br,firmo,Firmo,15,,-14.65,-44.55
br,fiuza,Fiúza,06,,-4.316667,-38.516667
br,flambuaia,Flambuaiã,27,,-23.155818,-46.530213
br,flamenga,Flamenga,06,,-2.953889,-40.886667
br,flamengo,Flamengo,05,,-12.933333,-38.333333
br,flamengo,Flamengo,06,,-5.983333,-39.633333
br,flamengo,Flamengo,20,,-3.538889,-42.150833
br,flamengo,Flamengo,21,,-22.929422,-43.171882
br,flamenguinho,Flamenguinho,18,,-23.8248,-50.195
br,flarania,Flarânia,22,,-5.143889,-37.369444
br,flecha,Flecha,17,,-6.916667,-38.1
br,flechal,Flechal,16,,-2.083333,-46.483333
br,flechas,Flechas,05,,-12.383333,-38.566667
br,flechas,Flechas,11,,-19.25,-56.583333
br,flechas,Flechas,14,,-16.033333,-57.25
br,flechas,Flechas,22,,-6.316667,-38.233333
br,flecheira,Flecheira,06,,-7.25,-38.75
br,flecheira,Flecheira,13,,-3.530556,-42.72
br,flecheira,Flecheira,17,,-7.416667,-38.4
br,flecheira,Flecheira,21,,-22.935833,-43.908333
br,flecheiras,Flecheiras,02,,-9.266667,-35.716667
br,flecheiras,Flecheiras,28,,-10.716667,-36.9
br,flexal,Flexal,16,,-1.933333,-55.283333
br,flexal,Flexal,25,,3.416667,-60.3
br,flexas,Flexas,11,,-21.05,-56.083333
br,flexas,Flexas,14,,-16.033333,-57.25
br,flexeiras,Flexeiras,02,7783,-9.266667,-35.716667
br,floduardo,Floduardo,23,,-30.25,-55.116667
br,flora,Flora,01,,-8.633333,-72.866667
br,flora,Flora,04,,-3.45,-62.65
br,flora,Flora,16,,-.666667,-47.35
br,florai,Floraí,18,,-23.283333,-52.316667
br,floralia,Florália,15,,-19.95,-43.3
br,floralio,Floralio,15,,-19.95,-43.3
br,florania,Florânia,22,,-6.133333,-36.816667
br,flora rica,Flora Rica,27,,-21.666667,-51.383333
br,flor branca,Flor Branca,06,,-5.266667,-40.216667
br,flor da mata,Flor da Mata,08,,-20.233333,-41.7
br,flor da serra,Flor da Serra,20,,-6.283333,-41.35
br,flor de cipo,Flor de Cipó,05,,-13.033333,-40.55
br,flor de maio,Flor de Maio,11,,-22.883333,-55.266667
br,flor de maio,Flor de Maio,23,,-27.8,-54.2
br,flor do gaucho,Flor do Gaúcho,16,,-4.016667,-51.866667
br,flor do tempo,Flor do Tempo,13,,-5.7,-43.2
br,floreal,Floreal,27,,-20.683333,-50.116667
br,florencio ferreira,Florêncio Ferreira,15,,-20.766667,-45.45
br,florencio,Florêncio,15,,-17.266667,-41.05
br,florencio,Florêncio,25,,1.85,-61.083333
br,florescencia,Florescência,01,,-9.716667,-68.75
br,florescente,Florescente,30,,-8.733333,-35.5
br,flores da aparecida,Flores da Aparecida,08,,-20.633333,-41.533333
br,flores da cunha,Flores da Cunha,23,20522,-29.0287,-51.1822
br,flores da ribeira de pajeu,Flôres da Ribeira de Pajeú,30,,-7.85,-37.983333
br,flores do piaui,Flores do Piauí,20,,-7.833333,-42.933333
br,flores,Flores,04,,-3.9,-61.866667
br,flores,Flores,05,,-11.433333,-40.983333
br,flores,Flores,06,,-4.733333,-38.45
br,flores,Flores,13,,-3.6,-44.866667
br,flores,Flores,14,,-9.716667,-60.566667
br,flores,Flores,15,,-20.666667,-45.166667
br,flores,Flôres,17,,-6.833333,-37.833333
br,flores,Flores,20,,-5.466667,-41.516667
br,flores,Flores,22,,-5.555833,-36.531667
br,flores,Flôres,23,,-28.9,-51.566667
br,flores,Flores,24,,-9.6,-62.333333
br,flores,Flores,26,,-26.333333,-53.4
br,flores,Flores,29,,-14.566667,-47.066667
br,flores,Flores,30,11694,-7.85,-37.983333
br,flores novas,Flores Novas,06,,-5.95,-39.066667
br,floresta azul,Floresta Azul,05,,-14.85,-39.683333
br,floresta da tijuca,Floresta da Tijuca,21,,-22.957572,-43.292843
br,floresta do sul,Floresta do Sul,27,,-21.983333,-51.283333
br,floresta,Floresta,01,,-8.116667,-72.783333
br,floresta,Floresta,04,,-1.75,-65.95
br,floresta,Floresta,05,,-12.55,-38.466667
br,floresta,Floresta,06,,-4.683333,-38.583333
br,floresta,Floresta,08,,-20.7,-41.333333
br,floresta,Floresta,11,,-23.816667,-54.2
br,floresta,Floresta,14,,-9.583333,-60.55
br,floresta,Floresta,15,,-15.466667,-45.8
br,floresta,Floresta,16,,-2.05,-46.45
br,floresta,Floresta,17,,-7.533333,-36.233333
br,floresta,Floresta,18,,-23.633333,-52.116667
br,floresta,Floresta,20,,-6.133333,-41.533333
br,floresta,Floresta,21,,-21.916667,-42.433333
br,floresta,Floresta,23,,-30.020833,-51.2
br,floresta,Floresta,27,,-20.433333,-47.766667
br,floresta,Floresta,28,,-9.766667,-37.433333
br,floresta,Floresta,30,18100,-8.6,-38.566667
br,floresta,Floresta,31,,-11.616667,-46.966667
br,florestal,Florestal,06,,-5.233333,-39.166667
br,florestal,Florestal,15,,-19.9,-44.45
br,florestal,Florestal,18,,-25.45,-50.583333
br,floresta nova,Floresta Nova,17,,-7.033333,-36.833333
br,floresta sao joao,Floresta São João,18,,-25.704789,-50.085443
br,florestina,Florestina,15,,-18.483333,-48.016667
br,florestopolis,Florestópolis,18,13250,-22.866667,-51.4
br,flor,Flor,22,,-6.316667,-35.066667
br,floriano,Floriano,02,,-9.416667,-35.833333
br,floriano,Floriano,06,,-5.083333,-39.033333
br,floriano,Floriano,18,,-23.516667,-52.05
br,floriano,Floriano,20,48588,-6.783333,-43.016667
br,floriano,Floriano,21,,-22.4509,-44.3024
br,floriano hesseman,Floriano Hesseman,08,,-20.3,-40.916667
br,floriano huve,Floriano Huve,08,,-20.316667,-40.9
br,floriano huver,Floriano Huver,08,,-20.316667,-40.9
br,floriano peixoto,Floriano Peixoto,02,,-9.533333,-35.6
br,floriano peixoto,Floriano Peixoto,04,,-9.05,-67.4
br,floriano peixoto,Floriano Peixoto,06,,-5.083333,-39.033333
br,floriano peixoto,Floriano Peixoto,23,,-27.866667,-52.083333
br,florianopolis,Florianópolis,01,,-7.7,-72.65
br,florianopolis,Florianópolis,04,,-2.516667,-67.333333
br,florianopolis,Florianópolis,26,412731,-27.583333,-48.566667
br,florianos,Florianos,18,,-25.464167,-50.360833
br,floriano simmer,Floriano Simmer,08,,-20.333333,-40.866667
br,florida,Florida,11,,-21.966667,-57.616667
br,florida,Flórida,18,,-23.066667,-51.95
br,florida,Flórida,21,,-21.383333,-41.7
br,florida,Flórida,23,,-29.25,-54.6
br,florinea,Florínea,27,,-22.933333,-50.716667
br,florinia,Florínia,27,,-22.933333,-50.716667
br,florsina,Florsina,15,,-17.05,-40.633333
br,fluviopolis,Fluviópolis,18,,-26.016667,-50.616667
br,f. nova,F. Nova,11,,-19.516667,-51.85
br,fojo,Fojo,05,,-13.3,-39.533333
br,fojos,Fojos,18,,-24.966667,-50.783333
br,folego do sergio,Fôlego do Sérgio,05,,-14.1,-44.4
br,folha branca,Fôlha Branca,05,,-11.65,-40.883333
br,folha,Fôlha,05,,-12.4,-44.55
br,folha larga,Fôlha Larga,05,,-12.4,-44.55
br,folha larga,Folha Larga,13,,-3.96,-43.216111
br,folha larga,Fôlha Larga,15,,-17.066667,-42.983333
br,folha miuda,Fôlha Miúda,02,,-9.716667,-36.816667
br,folheta,Folheta,05,,-13.516667,-41.966667
br,fomento,Fomento,20,,-6.983333,-42.066667
br,fonceca,Foncêca,11,,-21.883333,-53.466667
br,fonceca,Fonceca,15,,-20.166667,-43.3
br,fonseca,Fonseca,06,,-5.466667,-39.066667
br,fonseca,Fonseca,15,,-20.166667,-43.3
br,fonseca,Fonseca,21,,-22.85,-42.05
br,fontainha,Fontainha,06,,-4.616667,-37.6
br,fonte boa,Fonte Boa,04,16060,-2.533333,-66.016667
br,fonte boa,Fonte Boa,24,,-10.733333,-62.6
br,fonte da saudade,Fonte da Saudade,21,,-22.965234,-43.199947
br,fonte de fatima,Fonte de Fátima,20,,-7.7,-42.35
br,fonte do pau-d'agua,Fonte do Pau-d'Agua,14,,-14.516667,-58.583333
br,fontela,Fontela,21,,-22.994557,-43.444213
br,fonte limpa,Fonte Limpa,08,,-20.2,-40.3
br,fonte nova,Fonte Nova,05,,-14.4,-43.683333
br,fonte rica,Fonte Rica,13,,-5.966667,-43.35
br,fontes,Fontes,21,,-22.7,-43.883333
br,fontes,Fontes,22,,-6.1,-35.4
br,fonte termal agua quente,Fonte Termal Água Quente,14,,-13.833333,-58.8
br,fonte thermal agua quente,Fonte Thermal Agua Quente,14,,-13.833333,-58.8
br,fontinha,Fontinha,13,,-3.131944,-42.478056
br,fontoura,Fontoura,31,,-11.300833,-50.605556
br,fontoura xavier,Fontoura Xavier,23,,-28.983333,-52.35
br,forca,Fôrca,15,,-18.55,-47.3
br,fordlandia,Fordlândia,16,,-3.8,-55.483333
br,forges,Forges,15,,-14.85,-44.383333
br,formate,Formate,08,,-20.366667,-40.483333
br,formiga,Formiga,05,,-13.316667,-39.266667
br,formiga,Formiga,11,,-21.3,-54.433333
br,formiga,Formiga,13,,-2.316667,-45.383333
br,formiga,Formiga,14,,-16.083333,-56.866667
br,formiga,Formiga,15,56405,-20.45,-45.416667
br,formiga,Formiga,18,,-25.133333,-53.016667
br,formiga,Formiga,20,,-6.683333,-42.366667
br,formiga,Formiga,23,,-27.95,-52.116667
br,formiga,Formiga,27,,-22.383333,-51.3
br,formiga,Formiga,30,,-7.95,-39.116667
br,formigas,Formigas,18,,-25.133333,-53.016667
br,formigueiro,Formigueiro,17,,-6.75,-38.55
br,formigueiro,Formigueiro,20,,-5.116667,-41.366667
br,formigueiro,Formigueiro,23,,-30,-53.433333
br,formigueiro,Formigueiro,30,,-7.733333,-37.533333
br,formosa de cima,Formosa de Cima,05,,-9.533333,-42.133333
br,formosa do rio preto,Formosa do Rio Prêto,05,14365,-11.046389,-45.059722
br,formosa,Formosa,05,,-10.383333,-43.466667
br,formosa,Formosa,06,,-4.083333,-38.983333
br,formosa,Formosa,11,,-19.233333,-56.783333
br,formosa,Formosa,13,,-3.436667,-42.393333
br,formosa,Formosa,14,,-15.065,-59.875
br,formosa,Formosa,17,,-7.3,-37.833333
br,formosa,Formosa,20,,-3.683333,-42.533333
br,formosa,Formosa,21,,-21.633333,-41.633333
br,formosa,Formosa,22,,-6.366667,-35
br,formosa,Formosa,23,,-29.6,-52.616667
br,formosa,Formosa,27,,-23.783333,-45.35
br,formosa,Formosa,29,81233,-15.533333,-47.333333
br,formosa,Formosa,30,,-8.783333,-36.583333
br,formos,Formos,06,,-2.8,-40.4
br,formoso de cima,Formoso de Cima,05,,-13.866667,-41.3
br,formoso do araguaia,Formoso do Araguaia,31,14963,-11.796667,-49.466944
br,formoso,Formoso,15,,-14.95,-46.233333
br,formoso,Formoso,27,,-22.633333,-44.516667
br,formoso,Formoso,29,,-13.616667,-48.9
br,formoso,Formoso,31,,-11.796667,-49.466944
br,forninho,Forninho,13,,-3.768611,-42.791944
br,forno velho,Forno Velho,22,,-5.183333,-37.4
br,forqueta,Forqueta,23,,-29.183333,-51.25
br,forquilha de cima,Forquilha de Cima,17,,-7.5,-36.45
br,forquilha,Forquilha,05,,-10.2,-38.616667
br,forquilha,Forquilha,06,,-2.9,-40.533333
br,forquilha,Forquilha,11,,-20.733333,-56.016667
br,forquilha,Forquilha,13,,-3.621389,-42.821667
br,forquilha,Forquilha,14,,-14.95,-56.366667
br,forquilha,Forquilha,15,,-16.583333,-40.816667
br,forquilha,Forquilha,18,,-24.755902,-48.776489
br,forquilha,Forquilha,20,,-3.460556,-42.197778
br,forquilha,Forquilha,22,,-5.228611,-37.321389
br,forquilha,Forquilha,26,,-27.516667,-48.783333
br,forquilha,Forquilha,27,,-24.799734,-48.397958
br,forquilha,Forquilha,30,,-8.8,-37.716667
br,forquilhas,Forquilhas,26,,-27.516667,-48.783333
br,forquilhinha,Forquilhinha,26,23021,-28.75,-49.466667
br,forquilinho,Forquilinho,26,,-28.75,-49.466667
br,forquilna,Forquilna,06,,-5.033333,-38.466667
br,forrambamba,Forrambambá,04,,-7.083333,-69.683333
br,forromeco,Forromeco,23,,-29.45,-51.4
br,fortaleza de abuna,Fortaleza de Abunã,24,,-9.783333,-65.5
br,fortaleza de a nilo coutinho,Fortaleza de A Nilo Coutinho,06,,-5.7,-40.8
br,fortaleza de ituxi,Fortaleza de Ituxi,04,,-7.483333,-66.333333
br,fortaleza de nova braganca,Fortaleza de Nova Bragança,06,,-3.716667,-38.5
br,fortaleza do abuna,Fortaleza do Abunã,24,,-9.783333,-65.5
br,fortaleza do ituxi,Fortaleza do Ituxi,04,,-7.483333,-66.333333
br,fortaleza dos nogueira,Fortaleza dos Nogueira,13,,-6.9,-46.15
br,fortaleza dos nogueiras,Fortaleza dos Nogueiras,13,,-6.9,-46.15
br,fortaleza dos valos,Fortaleza dos Valos,23,,-28.8,-53.216667
br,fortaleza do tabocao,Fortaleza do Tabocão,31,,-9.012778,-48.431944
br,fortaleza,Fortaleza,01,,-10.083333,-69.083333
br,fortaleza,Fortaleza,03,,-.366667,-52.45
br,fortaleza,Fortaleza,04,,-1.816667,-65.616667
br,fortaleza,Fortaleza,05,,-10.55,-38.6
br,fortaleza,Fortaleza,06,2416920,-3.316667,-41.416667
br,fortaleza,Fortaleza,08,,-20.366667,-41.4
br,fortaleza,Fortaleza,11,,-18.55,-54.766667
br,fortaleza,Fortaleza,13,,-1.316667,-46.133333
br,fortaleza,Fortaleza,15,,-16.016667,-41.266667
br,fortaleza,Fortaleza,16,,-1.066667,-52.533333
br,fortaleza,Fortaleza,18,,-24.3,-50.3
br,fortaleza,Fortaleza,20,,-5.533333,-41.9
br,fortaleza,Fortaleza,23,,-27.483333,-53.4
br,fortaleza,Fortaleza,24,,-11.916667,-63.4
br,fortaleza,Fortaleza,26,,-26.8681,-49.0537
br,fortaleza,Fortaleza,31,,-5.983333,-48.133333
br,fortaleza nova de braganca,Fortaleza Nova de Bragança,06,,-3.716667,-38.5
br,forte ambe,Forte Ambé,16,,-3.183333,-52.166667
br,forte coimbra,Forte Coimbra,11,,-19.916667,-57.783333
br,forte colmbra,Forte Colmbra,11,,-19.916667,-57.783333
br,forte da graca,Forte da Graça,04,,-3.616667,-66.1
br,forte de coimbra,Forte de Coimbra,11,,-19.916667,-57.783333
br,forte do abuna,Forte do Abunã,24,,-9.783333,-65.5
br,forte do coimbra,Forte do Coimbra,11,,-19.916667,-57.783333
br,forte do mato,Forte do Mato,22,,-5.774444,-37.46
br,forte do principe da beira,Forte do Principe Da Beira,24,,-12.416667,-64.416667
br,forte,Forte,04,,-3.483333,-62.616667
br,forte,Forte,16,,-3.183333,-52.166667
br,forte,Forte,20,,-6.65,-41.866667
br,forte,Forte,29,,-14.266667,-47.283333
br,forteleza,Forteleza,15,,-16.016667,-41.266667
br,forte militao formiga,Forte Militão Formiga,20,,-8.116667,-45.583333
br,forte principe da beira,Forte Principe da Beira,24,,-12.416667,-64.416667
br,forte veneza,Forte Veneza,16,,-5.083333,-52.933333
br,fortim,Fortim,06,,-4.45,-37.783333
br,fortinho,Fortinho,06,,-4.45,-37.783333
br,forto nacional,Forto Nacional,31,,-10.708056,-48.308333
br,fortuna,Fortuna,06,,-6.9,-39.45
br,fortuna,Fortuna,13,9755,-5.683333,-44.066667
br,fortuna,Fortuna,15,,-19.116667,-43.816667
br,fortuna,Fortuna,18,,-24.216667,-53.066667
br,fortuna,Fortuna,22,,-5.723056,-37.054167
br,fortuna,Fortuna,27,,-22.3,-50.283333
br,foveiros,Foveiros,20,,-7.183333,-40.866667
br,foz de iguassu,Foz de Iguassu,18,,-25.55,-54.583333
br,foz do acre,Foz do Acre,04,,-8.75,-67.383333
br,foz do aripuana,Foz do Aripuanã,04,,-5.120278,-60.378889
br,foz do ati,Foz do Ati,04,,-7.4,-70.4
br,foz do aty,Foz do Aty,04,,-7.4,-70.4
br,foz do breu,Foz do Breu,01,,-9.4,-72.7
br,foz do cahy,Foz do Cahy,05,,-17,-39.183333
br,foz do copea,Foz do Copeá,04,,-3.866667,-63.316667
br,foz do embira,Foz do Embira,04,,-7.333333,-70.233333
br,foz do iguacu,Foz do Iguaçu,18,293526,-25.55,-54.583333
br,foz do iguassu,Foz do Iguassú,18,,-25.55,-54.583333
br,foz do itucuman,Foz do Itucuman,04,,-6.983333,-69.833333
br,foz do jamari,Foz do Jamari,24,,-8.45,-63.483333
br,foz do jordao,Foz do Jordão,01,,-9.383333,-71.933333
br,foz do jutahy,Foz do Jutahy,04,,-2.733333,-66.95
br,foz do jutai,Foz do Jutaí,04,,-2.733333,-66.95
br,foz do macacoari,Foz do Macacoari,03,,.316667,-50.7
br,foz do mamoria,Foz do Mamoriá,04,,-2.45,-66.533333
br,foz do muru,Foz do Muru,01,,-8.166667,-70.766667
br,foz do pauini,Foz do Pauini,04,,-7.783333,-67.083333
br,foz do rio jau,Foz do Rio Jaú,04,,-1.9025,-61.436667
br,foz do riozinho,Foz do Riozinho,04,,-7.181389,-71.817222
br,foz do tarauaca,Foz do Tarauacá,04,,-6.7,-69.783333
br,foz do tauari,Foz do Tauari,01,,-7.933333,-71.466667
br,foz macacoari,Foz Macacoari,03,,.316667,-50.7
br,foz pauini,Foz Pauini,04,,-7.783333,-67.083333
br,foz riosinho,Foz Riosinho,04,,-7.181389,-71.817222
br,foz riozinho,Foz Riozinho,04,,-7.181389,-71.817222
br,frade,Frade,06,,-5.616667,-38.766667
br,frade,Frade,08,,-20.9,-40.983333
br,frade,Frade,17,,-7.3,-38.266667
br,frade,Frade,20,,-5.7,-41.25
br,frade,Frade,21,,-22.95,-44.433333
br,frades de baixo,Frades de Baixo,27,,-22.596111,-47.268889
br,frades de cima,Frades de Cima,27,,-22.576389,-47.286389
br,frades,Frades,13,,-5.216667,-47.966667
br,frades,Frades,21,,-22.334069,-42.754247
br,frades,Frades,27,,-22.557778,-47.26
br,fraga,Fraga,20,,-7.433333,-41.016667
br,fragas,Fragas,27,,-23.324389,-47.895741
br,fragoso,Fragoso,05,,-12.4,-38.5
br,fragoso,Fragoso,21,,-22.588889,-43.183333
br,fragosos,Fragosos,26,,-26.16,-49.3839
br,franca,França,05,,-10.183333,-42.533333
br,franca,França,06,,-6.216667,-38.733333
br,franca,Franca,27,305047,-20.533333,-47.4
br,francelino,Francelino,23,,-29.9262,-50.8211
br,francesa,Francesa,15,,-16.033333,-39.95
br,frances dos carvalhos,Francês dos Carvalhos,15,,-22.083333,-44.483333
br,frances,Francês,02,,-9.733333,-35.816667
br,frances,Francês,16,,-4.8,-56.583333
br,francez dos carvalhos,Francez dos Carvalhos,15,,-22.083333,-44.483333
br,francez,Francez,02,,-9.733333,-35.816667
br,francez,Francez,16,,-4.8,-56.583333
br,francinopolis,Francinópolis,20,,-6.433333,-42.266667
br,francisco afonso,Francisco Afonso,15,,-20.583333,-45.216667
br,francisco aires,Francisco Aires,20,,-6.616667,-42.683333
br,francisco alves,Francisco Alves,18,,-24.066667,-53.833333
br,francisco araujo,Francisco Araújo,15,,-21.4,-45.616667
br,francisco badaro,Francisco Badaró,15,,-17.05,-42.316667
br,francisco beltrao,Francisco Beltrão,18,57618,-26.083333,-53.066667
br,francisco caetano,Francisco Caetano,15,,-20.516667,-46.633333
br,francisco d'abila,Francisco d'Abila,23,,-31.883333,-53.166667
br,francisco dantas,Francisco Dantas,22,,-6.066667,-38.166667
br,francisco da rocha,Francisco da Rocha,27,,-23.324821,-46.718475
br,francisco das chagas,Francisco das Chagas,29,,-15.916667,-49.266667
br,francisco de sales,Francisco de Sales,15,,-19.866667,-49.766667
br,francisco de salles,Francisco de Salles,15,,-19.866667,-49.766667
br,francisco dos reis,Francisco dos Reis,03,,.316667,-51.15
br,francisco esteves,Francisco Esteves,15,,-21.466667,-45.916667
br,francisco ferreira,Francisco Ferreira,15,,-17.55,-41.35
br,francisco,Francisco,04,,-5.667222,-60.269722
br,francisco,Francisco,15,,-15.8,-43.95
br,francisco,Francisco,16,,-5.633333,-57.05
br,francisco,Francisco,17,,-7.2,-38.333333
br,francisco gonzaga,Francisco Gonzaga,29,,-17.4,-48.7
br,francisco graciano,Francisco Graciano,15,,-21.866667,-46.2
br,francisco lourenco,Francisco Lourenço,15,,-17.066667,-41.45
br,francisco lucas,Francisco Lucas,14,,-13.116667,-55.933333
br,francisco luciano,Francisco Luciano,08,,-20.033333,-41.25
br,francisco miranda,Francisco Miranda,15,,-21.366667,-45.683333
br,francisco monteiro,Francisco Monteiro,15,,-15.25,-44.933333
br,francisco morato,Francisco Morato,27,171793,-23.279503,-46.746403
br,francisco narciso,Francisco Narciso,04,,1.566667,-67.183333
br,francisco pereira,Francisco Pereira,15,,-21.466667,-46.233333
br,francisco pinho,Francisco Pinho,15,,-20.8,-45.65
br,francisco quirino,Francisco Quirino,27,,-23.083333,-47.166667
br,francisco rodrigues,Francisco Rodrigues,15,,-20.35,-45.533333
br,francisco sa,Francisco Sá,15,13997,-16.466667,-43.5
br,francisco sales,Francisco Sales,15,,-21.7,-44.45
br,francisco santos,Francisco Santos,20,,-6.983333,-41.133333
br,francisco silva,Francisco Silva,15,,-21.283333,-45.8
br,francisco sousa,Francisco Sousa,05,,-13.85,-41.05
br,francisco tuta,Francisco Tuta,15,,-21.1,-47.1
br,francisco venancio,Francisco Venâncio,15,,-20.5,-45.116667
br,francisco vilela,Francisco Vilela,15,,-20.983333,-46.333333
br,francisco xavier,Francisco Xavier,15,,-17.616667,-41.4
br,francisco xavier,Francisco Xavier,29,,-17.133333,-48.6
br,franco da rocha,Franco da Rocha,27,108860,-23.324821,-46.718475
br,franco,Franco,06,,-3.133333,-39.45
br,frango,Frango,30,,-8.1,-36.616667
br,franklin,Franklin,04,,-5.4,-71.95
br,franquilino marques,Franquilino Marques,15,,-16.533333,-41.766667
br,franquinha,Franquinha,03,,.9,-50.083333
br,franquinho,Franquinho,03,,.9,-50.083333
br,frazao,Frazão,11,,-21.483333,-56.816667
br,frechal,Frechal,04,,-4.3,-59.5
br,frechal,Frechal,16,,-5.333333,-56.816667
br,frecheira,Frecheira,13,,-2.633333,-43.55
br,frecheira,Frecheira,17,,-6.383333,-38.55
br,frecheira,Frecheira,20,,-3.366667,-41.6
br,frecheira,Frecheira,30,,-8.983333,-36.566667
br,frecheiras,Frecheiras,02,,-9.266667,-35.716667
br,frecheiras,Frecheiras,05,,-12.55,-38.066667
br,frecheiras,Frecheiras,06,,-2.927222,-40.688611
br,frecheiras,Frecheiras,08,,-21.033333,-41.183333
br,frecheiras,Frecheiras,13,,-2.633333,-43.55
br,frecheiras,Frecheiras,20,,-3.366667,-41.6
br,frecheiras,Frecheiras,30,,-8.4,-35.316667
br,frecheirinha,Frecheirinha,06,,-3.766667,-40.8
br,frecheirinhas,Frecheirinhas,06,,-3.766667,-40.8
br,fredericia,Frederícia,17,,-7.116667,-34.866667
br,frederico westphalen,Frederico Westphalen,23,20896,-27.366667,-53.4
br,freguesia da lagoa,Freguesia da Lagoa,26,,-27.483333,-51.4
br,freguesia do andira,Freguesia do Andirá,04,,-2.85,-57
br,freguesia do o,Freguesia do O,27,,-23.498773,-46.699829
br,freguesia,Freguesia,18,,-25.176839,-48.992474
br,freguesia,Freguesia,21,,-22.789548,-43.172061
br,freguezia da lagoa,Freguezia da Lagoa,26,,-27.483333,-51.4
br,freguezia do andira,Freguezia do Andirá,04,,-2.85,-57
br,freguezia,Freguezia,04,,-2.85,-57
br,freguezia,Freguezia,21,,-22.789548,-43.172061
br,freicheiras,Freicheiras,13,,-2.85,-42.083333
br,freicheiras,Freicheiras,20,,-3.366667,-41.6
br,frei eustaquio,Frei Eustáquio,15,,-21.183333,-45.366667
br,frei gaspar,Frei Gaspar,15,,-18.083333,-41.433333
br,frei gonzaga,Frei Gonzaga,15,,-17.433333,-41.183333
br,frei inocencio,Frei Inocêncio,15,,-18.55,-41.9
br,frei jorge,Frei Jorge,06,,-2.95,-40.333333
br,frei manuel,Frei Manuel,14,,-15.766667,-56.666667
br,frei martinho,Frei Martinho,17,,-6.4,-36.45
br,frei matins,Frei Matins,17,,-7.133333,-37.716667
br,frei miguelinho,Frei Miguelinho,30,,-7.95,-35.95
br,frei orlando,Frei Orlando,15,,-18.833333,-45.266667
br,frei paulo,Frei Paulo,28,7047,-10.55,-37.533333
br,freire alemao,Freire Alemão,29,,-16.816667,-49.916667
br,freire,Freire,23,,-31.666667,-52.816667
br,frei rogerio,Frei Rogério,26,,-27.183333,-50.833333
br,frei serafim,Frei Serafim,15,,-18.15,-41.85
br,freitas,Freitas,05,,-9.283333,-38.266667
br,freitas,Freitas,18,,-23.6854,-49.7532
br,freitas,Freitas,23,,-29.9935,-50.9788
br,freitas,Freitas,30,,-7.816667,-35.566667
br,frei timoteo,Frei Timóteo,18,,-23.216667,-50.95
br,frei timotheo,Frei Timótheo,18,,-23.216667,-50.95
br,freixal,Freixal,16,,-1.183333,-51.133333
br,freixeira,Freixeira,13,,-2.633333,-43.55
br,freixeiras,Freixeiras,05,,-12.55,-38.066667
br,freixeiras,Freixeiras,13,,-2.85,-42.083333
br,frexeiras,Frexeiras,05,,-12.55,-38.066667
br,frexeiras,Frexeiras,06,,-3.233333,-39.266667
br,frexeiras,Frexeiras,30,,-8.4,-35.316667
br,frexeirinha,Frexeirinha,06,,-3.766667,-40.8
br,fria,Fria,05,,-12.433333,-40.683333
br,frial,Frial,15,,-19.783333,-47.666667
br,frias,Frias,26,,-26.233333,-48.716667
br,friburgo,Friburgo,04,,-7.783333,-70.983333
br,frigideira,Frigideira,13,,-3.977222,-43.214722
br,frigorifico,Frigorífico,27,,-20.583333,-48.533333
br,frio,Frio,05,,-12.4,-42.9
br,frio,Frio,30,,-8.866667,-35.633333
br,frois,Fróis,15,,-16.566667,-45.933333
br,fronteira,Fronteira,01,,-10.583333,-68.966667
br,fronteiras,Fronteiras,20,,-7.083333,-40.616667
br,fronteiro,Fronteiro,06,,-7.283333,-38.866667
br,frozinho,Frozinho,20,,-5.6,-41.216667
br,fructal,Fructal,15,,-20.033333,-48.916667
br,fructuoso,Fructuoso,24,,-13.533333,-60.95
br,fruta de leite,Fruta de Leite,15,,-16.133333,-42.483333
br,frutal do campo,Frutal do Campo,27,,-22.85,-50.516667
br,frutal,Frutal,15,40931,-20.033333,-48.916667
br,frutal,Frutal,27,,-20.583333,-47.316667
br,fruteira,Fruteira,20,,-5.5,-42.9
br,fruteiras,Fruteiras,21,,-22.416667,-41.866667
br,frutuoso,Frutuoso,24,,-13.533333,-60.95
br,fua,Fua,17,,-6.85,-38.466667
br,fuba,Fubá,21,,-22.889219,-43.333018
br,fugida,Fugida,13,,-3.35,-44.416667
br,fugido,Fugido,13,,-3.45,-46.083333
br,fulu,Fulu,15,,-19.1,-41.3
br,fumaca,Fumaça,04,,-5.488889,-59.651111
br,fumaca,Fumaça,05,,-13.283333,-40.566667
br,fumaca,Fumaça,13,,-2.733333,-46.65
br,fumaca,Fumaça,14,,-15.966667,-58.3
br,fumaca,Fumaça,21,,-22.298,-44.3099
br,fuma,Fuma,27,,-20.8,-47.283333
br,fumal,Fumal,20,,-6.45,-41.8
br,fumo,Fumo,06,,-5.8,-40.533333
br,funchal,Funchal,15,,-19.283333,-45.9
br,fundacao,Fundação,18,,-24.997619,-48.816866
br,fundao de zombaria,Fundão de Zombaria,30,,-8.266667,-36.65
br,fundao,Fundão,05,,-12.016667,-38.1
br,fundao,Fundão,06,,-5.966667,-39.55
br,fundao,Fundão,08,12715,-19.916667,-40.4
br,fundao,Fundão,15,,-20.333333,-45.966667
br,fundao,Fundão,17,,-6.35,-38.483333
br,fundao,Fundão,18,,-25.8,-51.983333
br,fundao,Fundão,27,,-23.5743,-47.9331
br,fundao,Fundão,30,,-8.333333,-36.6
br,fundinho,Fundinho,14,,-16.716667,-56.666667
br,fundinho,Fundinho,17,,-7.35,-37.516667
br,fundo do canoas,Fundo do Canoas,26,,-27.198345,-49.674522
br,fundoes,Fundoes,30,,-7.433333,-37.45
br,fundo,Fundo,05,,-11.883333,-43.133333
br,fundo,Fundo,11,,-20.983333,-53.316667
br,fundo,Fundo,13,,-6.1,-44.25
br,fundo leao,Fundo Leão,26,,-27.037,-49.6861
br,funilandia,Funilândia,15,,-19.35,-44.05
br,funil de baixo,Funil de Baixo,27,,-23.729811,-47.472912
br,funil,Funil,01,,-9.116667,-70.166667
br,funil,Funil,05,,-11.4,-44.316667
br,funil,Funil,13,,-3.448611,-42.866667
br,funil,Funil,15,,-19.35,-44.05
br,funil,Funil,21,,-21.616667,-42.083333
br,funil,Funil,27,,-23.724324,-47.454727
br,funil,Funil,31,,-9.683333,-48.383333
br,funis,Funis,05,,-12.95,-41.916667
br,furada,Furada,06,,-5.75,-40.866667
br,furada,Furada,08,,-19.983333,-40.333333
br,furado,Furado,05,,-11.883333,-43.133333
br,furado olhos d'agua,Furado Olhos d'Agua,15,,-16.4,-43.933333
br,furados,Furados,05,,-12.85,-41.7
br,furna da onca,Furna da Onça,20,,-6.866667,-40.966667
br,furna,Furna,05,,-13.466667,-41.766667
br,furna,Furna,15,,-22.183333,-45.816667
br,furnas,Furnas,05,,-13.466667,-41.766667
br,furnas,Furnas,06,,-4.55,-38.666667
br,furnas,Furnas,13,,-3.664444,-43.245278
br,furnas,Furnas,18,,-24.783333,-49.033333
br,furnas,Furnas,20,,-3.539722,-42.494722
br,furnas,Furnas,22,,-5.766667,-35.783333
br,furnas,Furnas,27,,-22.366667,-47.433333
br,furnas,Furnas,30,,-8.133333,-36.316667
br,furquim,Furquim,15,,-20.366667,-43.216667
br,furquim,Furquim,27,,-21.066667,-51.283333
br,furtado de campos,Furtado de Campos,15,,-21.45,-43.066667
br,furtados,Furtados,15,,-20.766667,-45.15
br,furtainha,Furtaínha,22,,-5.866667,-35.95
br,furta-lhe-a-vida,Furta-lhe-a-Vida,20,,-6.75,-41.85
br,furundum,Furundum,27,,-23.506556,-47.163849
br,futrica,Futrica,20,,-6.95,-41.083333
br,futuro,Futuro,02,,-9.716667,-37.433333
br,futuro,Futuro,20,,-7.116667,-41.533333
br,futuro,Futuro,22,,-5.468056,-36.98
br,fuzil,Fuzil,13,,-7.833333,-46.983333
br,gabiarra,Gabiarra,05,,-16.25,-39.683333
br,gabiroba,Gabiroba,26,,-28.15,-49.016667
br,gabriel emilio,Gabriel Emílio,15,,-18.783333,-40.983333
br,gabriel franco,Gabriel Franco,15,,-21.8,-46.233333
br,gabriel,Gabriel,05,,-11.233333,-41.883333
br,gabriel lobo,Gabriel Lôbo,15,,-17.033333,-40.816667
br,gabriel martins,Gabriel Martins,15,,-21.15,-46.583333
br,gabriel monteiro,Gabriel Monteiro,27,,-21.533333,-50.566667
br,gabriel piza,Gabriel Piza,27,,-23.561194,-47.123849
br,gadelha,Gadelha,06,,-6.416667,-39.283333
br,gado brabo,Gado Brabo,13,,-6.316667,-42.966667
br,gado bravo,Gado Bravo,05,,-10.733333,-40.566667
br,gado bravo,Gado Bravo,06,,-5.283333,-40.566667
br,gado bravo,Gado Bravo,17,,-6.75,-37.7
br,gado bravo,Gado Bravo,20,,-5.583333,-42.883333
br,gado bravo,Gado Bravo,22,,-4.868056,-37.244444
br,gado bravo,Gado Bravo,28,,-10.516667,-37.283333
br,gado bravo,Gado Bravo,30,,-8.5,-36.316667
br,gado,Gado,06,,-4.133333,-38.9
br,gado manso,Gado Manso,13,,-3.881389,-43.494444
br,gado velhaco,Gado Velhaco,20,,-6.516667,-42.033333
br,gafanhoto,Gafanhoto,06,,-4.616667,-38.083333
br,gafanhoto,Gafanhoto,15,,-20.083333,-44.883333
br,gaivota,Gaivota,03,,.4,-51.783333
br,gaivota,Gaivota,04,,-5.61,-63.439167
br,gaivotas,Gaivotas,04,,-5.61,-63.439167
br,gajueiro,Gajueiro,16,,-7.8,-49.933333
br,galante,Galante,17,,-7.116667,-38.433333
br,galao,Galão,31,,-10.616667,-46.25
br,galdino,Galdino,05,,-16.366667,-39.85
br,galdino,Galdino,06,,-4.85,-39.9
br,galdino,Galdino,15,,-16.416667,-41.483333
br,galeao,Galeão,05,,-13.4,-39.016667
br,galeao,Galeão,21,,-22.804054,-43.239784
br,galena,Galena,15,,-18.35,-46.116667
br,galeno,Galeno,05,,-13.4,-39.016667
br,galha,Galha,15,,-16.1,-40.133333
br,galhao,Galhão,31,,-10.616667,-46.25
br,galhardo,Galhardo,05,,-12.2,-38.2
br,galheiro,Galheiro,05,,-14.266667,-44.583333
br,galheiro,Galheiro,11,,-18.966667,-51.95
br,galheiro,Galheiro,13,,-9.45,-45.8
br,galheiro,Galheiro,29,,-13.3,-46.416667
br,galheiros,Galheiros,29,,-13.3,-46.416667
br,galia,Gália,27,,-22.3,-49.566667
br,galileia,Galiléia,04,,-7.35,-62.933333
br,galileia,Galiléia,06,,-5.9,-40.583333
br,galileia,Galiléia,13,,-9.983333,-46.366667
br,galileia,Galiléia,15,,-19,-41.55
br,galinha,Galinha,02,,-9.583333,-36.933333
br,galinha,Galinha,15,,-17.616667,-46.033333
br,galinhas,Galinhas,06,,-6.25,-38.566667
br,galinhos,Galinhos,05,,-18.016667,-39.983333
br,galinhos,Galinhos,22,,-5.090556,-36.273611
br,gallia,Gallia,27,,-22.3,-49.566667
br,gallinhos,Gallinhos,22,,-5.090556,-36.273611
br,galo de ouro,Galo de Ouro,27,,-23.170213,-47.725132
br,galopolis,Galópolis,23,,-29.2307,-51.1573
br,galos,Galos,22,,-5.0975,-36.242222
br,galuado,Galuado,17,,-6.9,-37.733333
br,galvao,Galvão,26,,-26.466667,-52.65
br,galvao,Galvão,27,,-22.8,-47.416667
br,gama,Gama,07,,-15.95,-48.083333
br,gama,Gama,15,,-19.133333,-42.383333
br,gama,Gama,20,,-5.316667,-41.7
br,gamaleira,Gamaleira,31,,-10.85,-47.7
br,gambas,Gambás,05,,-11.016667,-42.466667
br,gamboa,Gamboa,05,,-12.983333,-38.6
br,gamboa,Gamboa,06,,-2.978056,-40.8375
br,gamboa,Gamboa,21,,-22.897567,-43.192758
br,gamboa,Gamboa,26,,-26.334,-48.6311
br,gamboa grande,Gamboa Grande,05,,-13.05,-38.8
br,gamela,Gamela,02,,-9.016667,-35.216667
br,gamela,Gamela,13,,-3.974167,-42.927222
br,gamela,Gamela,30,,-8.666667,-35.083333
br,gamelao,Gamelão,14,,-15.483333,-57.833333
br,gameleina,Gameleina,16,,-8.666667,-49.583333
br,gameleira da lagoa do boi,Gameleira da Lagoa do Boi,05,,-11.916667,-41.716667
br,gameleira da lapa,Gameleira da Lapa,05,,-12.866667,-43.366667
br,gameleira de aurelio,Gameleira de Aurélio,05,,-14.3,-41.283333
br,gameleira de sao sebastiao,Gameleira de São Sebastião,06,,-7.25,-39.116667
br,gameleira do assurua,Gameleira do Assuruá,05,,-11.3,-42.666667
br,gameleira dos machados,Gameleira dos Machados,05,,-14.25,-41.45
br,gameleira,Gameleira,02,,-10.083333,-36.483333
br,gameleira,Gameleira,05,,-10.183333,-42.716667
br,gameleira,Gameleira,06,,-3.766667,-39.966667
br,gameleira,Gameleira,11,,-20.616667,-54.633333
br,gameleira,Gameleira,13,,-3.883333,-45.183333
br,gameleira,Gameleira,15,,-16.183333,-41.566667
br,gameleira,Gameleira,16,,-7.75,-49.9
br,gameleira,Gameleira,17,,-6.616667,-37.916667
br,gameleira,Gameleira,20,,-2.95,-41.7
br,gameleira,Gameleira,22,,-5.65,-35.566667
br,gameleira,Gameleira,28,,-10.416667,-37.683333
br,gameleira,Gameleira,30,19424,-8.583333,-35.4
br,gameleira,Gameleira,31,,-10.85,-47.7
br,gameleira i,Gameleira I,13,,-3.807222,-42.813333
br,gameleira ii,Gameleira II,13,,-3.886389,-42.805278
br,gameleiras,Gameleiras,05,,-13.6,-40.233333
br,gameleiras,Gameleiras,15,,-15.083333,-43.133333
br,gameleiras,Gameleiras,20,,-6.083333,-41.483333
br,gameleira velha,Gameleira Velha,20,,-3.536667,-42.429444
br,gameleiro,Gameleiro,05,,-11.683333,-40.833333
br,gamelera,Gameléra,16,,-7.75,-49.9
br,gamelleira do assurua,Gamelleira do Assuruá,05,,-11.3,-42.666667
br,gamelleira,Gamelleira,05,,-11.816667,-42.65
br,gamelleira,Gamelleira,06,,-4.016667,-41.2
br,gamelleira,Gamelleira,15,,-15.083333,-43.133333
br,gamelleira,Gamelleira,16,,-7.75,-49.9
br,gamelleira,Gamelleira,30,,-8.583333,-35.4
br,gamelleira,Gamelleira,31,,-10.85,-47.7
br,gamelleiras,Gamelleiras,15,,-15.083333,-43.133333
br,gancho,Gancho,13,,-3.516667,-44.766667
br,gancho,Gancho,31,,-11.668889,-48.935556
br,ganchos,Ganchos,26,,-27.316667,-48.583333
br,gandhi,Gandhi,18,,-22.95,-51.15
br,gandu da laje,Gandú da Laje,05,,-13.75,-39.566667
br,gandu do laje,Gandu do Laje,05,,-13.75,-39.566667
br,gandu,Gandu,05,21566,-13.75,-39.5
br,gangorra,Gangorra,06,,-3.683333,-40.166667
br,gangorra,Gangorra,15,,-16.9,-42.616667
br,gangorra,Gangorra,17,,-6.5,-37.833333
br,gangorra,Gangorra,20,,-5.383333,-41.8
br,gangorra,Gangorra,22,,-4.933889,-37.301389
br,gangorra,Gangorra,30,,-7.816667,-37.616667
br,gangorras,Gangorras,15,,-16.9,-42.616667
br,gangorrinha,Gangorrinha,15,,-17.166667,-41.7
br,gangua,Gangúa,17,,-7.8,-38.266667
br,gangu,Gangu,05,,-11.6,-38.1
br,ganupo cervo,Ganupo Cervo,14,,-17.083333,-54.016667
br,garacui,Garacuí,18,,-24.713033,-48.677725
br,garaiber,Garaiber,13,,-6.416667,-42.866667
br,garajuba,Garajuba,05,,-12.7,-38.1
br,garajuva,Garajuva,26,,-28.883333,-49.483333
br,garamparo,Garamparo,13,,-3.048611,-43.397222
br,garancia,Garancia,04,,-7.1,-64.683333
br,garanhane,Garanhane,18,,-23.6188,-49.7763
br,garanhuns,Garanhuns,30,110087,-8.9,-36.483333
br,garapa de cima,Garapa de Cima,06,,-4.216667,-38.633333
br,garapa,Garapa,05,,-11.233333,-41.35
br,garapa,Garapa,06,,-4.216667,-38.633333
br,garapa,Garapa,25,,4.216667,-60.816667
br,garapua,Garapuá,05,,-13.483333,-38.916667
br,garapuava,Garapuava,15,,-16.1,-46.55
br,gararu,Gararu,28,2977,-9.966667,-37.083333
br,garatuba,Garatuba,18,,-25.35,-49.15
br,garauna,Garauna,06,,-5.3,-39.433333
br,garca,Garça,03,,1.866667,-50.8
br,garca,Garça,06,,-5.283333,-40.683333
br,garca,Garça,27,41303,-22.233333,-49.616667
br,garces,Garcês,23,,-29.666667,-51.25
br,garcez,Garcez,23,,-29.666667,-51.25
br,garcia,Garcia,02,,-9.383333,-37.083333
br,garcia,Garcia,15,,-18.616667,-49
br,garcia,Garcia,26,,-27.483333,-48.983333
br,garcia,Garcia,27,,-23.057457,-47.741826
br,garcia,Garcia,30,,-8.983333,-37.016667
br,garcias,Garcias,11,,-20.566667,-52.216667
br,garcias,Garcias,15,,-20.8,-45.2
br,gardenia azul,Gardênia Azul,21,,-22.960505,-43.35033
br,gardenia,Gardenia,15,,-19.25,-48.683333
br,gardenia,Gardênia,27,,-22.616667,-50.9
br,garganta do leandro,Garganta do Leandro,18,,-24.866667,-49.05
br,garganta do lizandro,Garganta do Lizandró,18,,-24.866667,-49.05
br,garganta,Garganta,02,,-9.5,-37.016667
br,garganta,Garganta,05,,-14.733333,-39.55
br,garganta,Garganta,20,,-6.683333,-40.533333
br,garganta,Garganta,22,,-6.1,-36.9
br,gargau,Gargaú,17,,-7.05,-34.9
br,gargau,Gargaú,21,,-21.583333,-41.066667
br,gargorra,Gargorra,20,,-8.783333,-42.2
br,garhaubinha,Garhaubinha,06,,-6.466667,-38.783333
br,garibaldi,Garibaldi,23,30364,-29.25,-51.533333
br,garibaldi,Garibaldi,26,,-26.533333,-49.216667
br,garibe,Garibé,05,,-13.133333,-38.983333
br,garimpo cervo,Garimpo Cervo,14,,-17.083333,-54.016667
br,garimpo das canoas,Garimpo das Canoas,15,,-20.4,-47.283333
br,garimpo do jenipapo,Garimpo do Jenipapo,31,,-7.616667,-49.2
br,garimpo,Garimpo,15,,-18.816667,-47.666667
br,garimpo luciano,Garimpo Luciano,29,,-16.833333,-52.966667
br,garimpo novo,Garimpo Novo,14,,-15.9,-54.433333
br,garimpo rio claro,Garimpo Rio Claro,14,,-13.75,-56.683333
br,gariroba,Gariroba,18,,-23.839628,-50.856787
br,garopaba,Garopaba,26,,-28.066667,-48.666667
br,garopada,Garopada,26,,-28.066667,-48.666667
br,garrafa,Garrafa,11,,-19.733333,-56.383333
br,garrafao,Garrafão,08,,-20.15,-40.816667
br,garrafao,Garrafão,20,,-3.924722,-42.398889
br,garrafao,Garrafão,21,,-21.233333,-41.4
br,garra,Garra,17,,-7.316667,-37.616667
br,garrota morta,Garrota Morta,22,,-6.2,-37.966667
br,garrote,Garrote,02,,-9.416667,-37.15
br,garrote,Garrote,06,,-3.533333,-39.916667
br,garrote,Garrote,13,,-8.25,-46.083333
br,garrote,Garrote,17,,-7.083333,-38.033333
br,garrote,Garrote,30,,-8.866667,-36.2
br,garrotes,Garrotes,17,,-7.383333,-38
br,garruchos,Garruchos,23,,-28.183333,-55.65
br,garuba,Garuba,26,,-26.9477,-49.012
br,garuva acima,Garuva Acima,26,,-25.983324,-48.878435
br,garuva,Garuva,26,,-25.983324,-48.878435
br,gaspar alto,Gaspar Alto,26,,-26.9967,-49.0371
br,gaspar alves,Gaspar Alves,15,,-17.383333,-41.283333
br,gaspar do meio,Gaspar do Meio,26,,-26.9942,-49.0467
br,gaspar,Gaspar,26,33659,-26.933333,-48.966667
br,gaspar martins,Gaspar Martins,23,,-29.4,-51.95
br,gaspar simoes,Gaspar Simões,23,,-30.5081,-52.3265
br,gastao vidigal,Gastão Vidigal,27,,-20.8,-50.166667
br,gata branca,Gata Branca,15,,-20.2224,-43.8477
br,gate espeto,Gaté Espeto,16,,-1.333333,-50.416667
br,gatinho,Gatinho,14,,-14.5,-56.516667
br,gato,Gato,02,,-9.166667,-37.75
br,gato,Gato,05,,-9.116667,-38.4
br,gato,Gato,06,,-6.116667,-40.683333
br,gato,Gato,13,,-5.95,-46.416667
br,gato,Gato,20,,-8.966667,-42.2
br,gatos,Gatos,05,,-12.316667,-39.3
br,gatos,Gatos,17,,-7.383333,-38.416667
br,gaturamo,Gaturamo,27,,-21.266667,-47.816667
br,gauacu,Gauacu,22,,-5.55,-35.9
br,gaucha,Gaúcha,14,,-15.066667,-58.566667
br,gaucha,Gaúcha,18,,-25.35,-54.166667
br,gaucho,Gaúcho,15,,-20.566667,-45.1
br,gauer,Gauer,23,,-27.716667,-52.283333
br,gaujiru,Gaujiru,22,,-5.066667,-35.866667
br,gaupo,Gaupó,13,,-2.833333,-45.033333
br,gaurama,Gaurama,23,,-27.566667,-52.05
br,gavao peixoto,Gavão Peixoto,27,,-21.833333,-48.5
br,gavea,Gávea,21,,-22.975777,-43.234925
br,gavea pequena,Gávea Pequena,21,,-22.976366,-43.277772
br,gaviao,Gavião,02,,-9.466667,-36.7
br,gaviao,Gavião,04,,-4.839167,-66.854444
br,gaviao,Gavião,05,,-11.466667,-39.783333
br,gaviao,Gavião,06,,-6.083333,-40.666667
br,gaviao,Gavião,13,,-3.080833,-42.287222
br,gaviao,Gavião,17,,-7.6,-37.766667
br,gaviao,Gavião,20,,-9.316667,-43.316667
br,gaviao,Gavião,22,,-5.907778,-37.189722
br,gaviao,Gavião,30,,-7.7,-38.316667
br,gaviao peixoto,Gavião Peixoto,27,,-21.833333,-48.5
br,gaviaosinho,Gaviãosinho,04,,-6.516667,-68.833333
br,gaviaozinho,Gaviãozinho,04,,-6.516667,-68.833333
br,gavioes,Gaviões,21,,-22.566667,-42.55
br,gaza,Gaza,05,,-11.233333,-42.083333
br,gazea,Gázea,06,,-4.683333,-40.733333
br,geada,Geada,27,,-22.576944,-47.473333
br,gelo,Gêlo,16,,-9.233333,-57.05
br,gemilina,Gemilina,18,,-25.497431,-48.836272
br,general brito,General Brito,26,,-26.026784,-49.990317
br,general camara,General Câmara,23,,-29.9,-51.766667
br,general carneiro,General Carneiro,14,,-15.7,-52.75
br,general carneiro,General Carneiro,15,,-19.833333,-43.833333
br,general carneiro,General Carneiro,18,,-26.466667,-51.416667
br,general daltro filho,General Daltro Filho,23,,-29.333333,-51.766667
br,general daltro filho,General Daltro Filho,26,,-26.866667,-51.05
br,general dutra,General Dutra,26,,-26.5,-51.083333
br,general osorio,General Osório,18,,-24.716667,-53.9
br,general osorio,General Osório,21,,-22.203056,-43.724167
br,general rondon,General Rondon,14,,-16.5,-54.633333
br,general rondon,General Rondon,18,,-24.566667,-54.066667
br,general salgado,General Salgado,27,8084,-20.65,-50.366667
br,general sampaio,General Sampaio,06,,-4.033333,-39.483333
br,general tiburcio,General Tibúrcio,06,,-3.5,-41.116667
br,general vargas,General Vargas,23,,-29.7,-54.666667
br,generica,Genérica,25,,4.483333,-60
br,generoso,Generoso,18,,-26.116667,-52.183333
br,generoso gomes,Generoso Gomes,15,,-17.683333,-41.633333
br,generoso ponce,Generoso Ponce,24,,-9.25,-64.383333
br,genesio,Genêsio,15,,-17.166667,-41.433333
br,genesio molin,Genésio Molin,23,,-29.0575,-51.034
br,genipapeiro,Genipapeiro,06,,-3.966667,-38.4
br,genipapeiro,Genipapeiro,20,,-7.816667,-41.666667
br,genipapo,Genipapo,04,,-5.571944,-60.921111
br,genipapo,Genipapo,05,,-11.616667,-45.016667
br,genipapo,Genipapo,16,,-.7,-49.166667
br,genipapo,Genipapo,20,,-6.033333,-41.383333
br,genipapo,Genipapo,30,,-8.283333,-36.566667
br,gentileza,Gentileza,28,,-9.8,-37.433333
br,gentil,Gentil,06,,-7.8,-39.05
br,gentil,Gentil,16,,-2.033333,-49.25
br,gentil,Gentil,18,,-24.416667,-49.633333
br,gentil,Gentil,20,,-7.183333,-41.683333
br,gentio de ouro,Gentio de Ouro,05,,-11.416667,-42.5
br,gentio do meio,Gentio do Meio,05,,-12.983333,-39.616667
br,gentio do ouro,Gentio do Ouro,05,,-11.416667,-42.5
br,gentio,Gentio,05,,-12.866667,-41.733333
br,gentio,Gentio,15,,-15.7,-45.233333
br,gentio,Gentio,18,,-24.416667,-49.633333
br,genuino,Genuíno,04,,-4.016667,-59.083333
br,genuino sampaio,Genuino Sampaio,23,,-29.65,-51.183333
br,geolandia,Geolândia,05,,-12.55,-39.083333
br,geomelino garcia,Geomelino Garcia,15,,-21.15,-46.933333
br,gequi,Gequí,30,,-8.65,-39.65
br,gerais,Gerais,05,,-13.1,-41.866667
br,geraizinho,Geraizinho,15,,-15.833333,-41.383333
br,geraldino,Geraldino,15,,-21.833333,-45.816667
br,geraldo lemos,Geraldo Lemos,15,,-20.783333,-46.633333
br,geral,Geral,20,,-6.666667,-40.6
br,gercinopolis,Gercinópolis,29,,-17.1,-49.633333
br,gerdau cosigua,Gerdau Cosigua,21,,-22.901391,-43.747547
br,geremeira,Geremeira,05,,-12.183333,-39.35
br,geremoabo,Geremoabo,05,,-10.066667,-38.35
br,geremuniz,Geremuniz,27,,-23.342,-46.2221
br,gerencia,Gerência,05,,-13.283333,-40.333333
br,gererau,Gereraú,06,,-3.916667,-38.55
br,gergelim,Gergelim,30,,-7.8,-40.483333
br,geriba,Geribá,15,,-16.833333,-40.383333
br,geribues,Geribués,13,,-3.510278,-43.135
br,gericino,Gericinó,21,,-22.845527,-43.480843
br,gerimatalha,Gerimatalha,02,,-9.666667,-37
br,gerinaldo,Gerinaldo,04,,-3.783333,-58.383333
br,germania,Germânia,01,,-10.333333,-69.55
br,germano,Germano,14,,-15.566667,-54.466667
br,gerno,Gernó,30,,-7.883333,-39.9
br,geroki,Geroki,11,,-22.966667,-54.566667
br,geroky,Geroky,11,,-22.966667,-54.566667
br,geroma,Geroma,20,,-6.8,-42.25
br,gertrudes,Gertrudes,11,,-21.516667,-55.433333
br,gertrudes,Gertrudes,27,,-23.349688,-46.746322
br,geru,Gerú,28,,-11.383333,-37.85
br,gesteira,Gesteira,15,,-20.266667,-43.133333
br,geti,Getí,20,,-10.033333,-44.283333
br,getulandia,Getulândia,21,,-22.666667,-44.1
br,getulina,Getulina,27,,-21.75,-49.916667
br,getulio d. vargas,Getúlio D. Vargas,23,,-29.25,-55.966667
br,getulio vargas,Getúlio Vargas,16,,-.966667,-47.866667
br,getulio vargas,Getúlio Vargas,22,,-6.016667,-37.416667
br,getulio vargas,Getúlio Vargas,23,13695,-27.833333,-52.266667
br,getulio vargas,Getúlio Vargas,26,,-27.0448,-49.6224
br,giboia,Gibóia,05,,-9.25,-41.583333
br,gibraltar,Gibraltar,01,,-7.433333,-73.6
br,gicaras,Giçaras,15,,-15.783333,-41.216667
br,gigante,Gigante,15,,-16.216667,-41.216667
br,giguy,Giguy,06,,-4.766667,-37.766667
br,gijoca dos bento,Gijoca dos Bento,06,,-2.85,-40.95
br,gijoca dos lula,Gijoca dos Lula,06,,-2.833333,-40.433333
br,gilbert,Gilbert,27,,-22.65,-47.7
br,gilboes,Gilboés,20,,-9.838889,-45.353889
br,gilbues,Gilbués,20,4920,-9.838889,-45.353889
br,gil,Gil,20,,-5.583333,-42.616667
br,gil,Gil,23,,-29.8,-51.633333
br,gimirim,Gimirim,15,,-21.8,-45.966667
br,ginete,Ginete,15,,-16.083333,-41.466667
br,ginga,Ginga,20,,-3.671667,-42.211944
br,ginimbu,Ginimbu,02,,-9.916667,-36.133333
br,giovani,Giovani,15,,-19.066667,-48.016667
br,giqui de baixo,Giqui de Baixo,06,,-4.75,-37.766667
br,giqui,Giqui,05,,-13.45,-41.1
br,giqui,Giqui,06,,-4.766667,-37.766667
br,gira,Gira,05,,-12.783333,-41.2
br,giral do ponceano,Giral do Ponceano,02,,-9.916667,-36.833333
br,giramundo de baixo,Giramundo de Baixo,13,,-3.013611,-42.825
br,giramundo,Giramundo,13,,-3.013333,-42.861111
br,gira mundo,Gira Mundo,20,,-5.15,-42.45
br,girando,Girando,08,,-20.866667,-41.416667
br,girao de pedra,Girão de Pedra,05,,-13.683333,-40.283333
br,girao,Girão,06,,-5.8,-39.416667
br,girao,Girao,14,,-15.233333,-57.7
br,girao,Girão,24,,-9.433333,-64.683333
br,girau do ponciano,Girau do Ponciano,02,9606,-9.916667,-36.833333
br,girau,Giráu,05,,-10.216667,-43.466667
br,girau,Girau,06,,-5.8,-39.416667
br,girau,Girau,24,,-9.433333,-64.683333
br,girua,Giruá,23,12170,-28.033333,-54.35
br,giru,Giru,15,,-16.833333,-41.066667
br,gissaras,Gissaras,15,,-15.783333,-41.216667
br,gitirana,Gitirana,20,,-7.35,-40.65
br,glaucilandia,Glaucilândia,15,,-16.85,-43.716667
br,glaura,Glaura,15,,-20.3,-43.633333
br,gleba 4,Gleba 4,18,,-23.9549,-50.9311
br,gleba brauna,Gleba Braúna,14,,-14.666667,-56.95
br,gleba capem,Gleba Capem,14,,-13.5,-55.1
br,gleba finch,Gleba Finch,21,,-23.02691,-43.478501
br,gleba i,Gleba I,27,,-23.333729,-46.638687
br,gleba ii,Gleba II,27,,-23.333121,-46.62723
br,gleba primeiro,Gleba Primeiro,27,,-23.333729,-46.638687
br,gleba quatro,Gleba Quatro,18,,-23.9549,-50.9311
br,gleba rio ferro,Gleba Rio Ferro,14,,-12.466667,-54.8
br,gleba segunda,Gleba Segunda,27,,-23.333121,-46.62723
br,glibert,Glibert,27,,-22.65,-47.7
br,glicerio,Glicério,21,,-22.233333,-42.066667
br,glicerio,Glicério,27,,-21.4,-50.183333
br,glicerio,Glicério,30,,-8.933333,-36.1
br,gloria de goyta,Gloria de Goytá,30,,-8.016667,-35.283333
br,gloria do goita,Glória do Goitá,30,17374,-8.016667,-35.283333
br,gloria,Glória,01,,-9.739722,-67.617222
br,gloria,Glória,05,,-9.183333,-38.3
br,gloria,Glória,08,,-20.366667,-40.516667
br,gloria,Glória,11,,-22.35,-54.216667
br,gloria,Glória,13,,-1.683333,-46.216667
br,gloria,Gloria,15,,-18.466667,-44.2
br,gloria,Glória,18,,-23.3625,-50.0108
br,gloria,Glória,21,,-22.92104,-43.180498
br,gloria,Glória,23,,-30.075,-51.2
br,glorinha,Glorinha,23,,-29.88,-50.7957
br,glucinio,Glucínio,15,,-18.216667,-42.35
br,glyceria,Glyceria,27,,-21.4,-50.183333
br,glycerio,Glycerio,21,,-22.233333,-42.066667
br,glycerio,Glycerio,30,,-8.933333,-36.1
br,gnadental,Gnadental,26,,-26.9503,-49.887
br,goaiz,Goaiz,29,,-15.933333,-50.133333
br,godinho,Godinho,21,,-21.266667,-41.6
br,godinho,Godinho,23,,-31.783333,-52.966667
br,godinho,Godinho,27,,-23.72738,-47.372327
br,godofredo,Godofredo,11,,-22.066667,-56.15
br,godofredo goncalves,Godofredo Gonçalves,11,,-22.066667,-56.15
br,goes,Goes,15,,-22.55,-46.3
br,goes,Góes,22,,-5.416667,-37.783333
br,goiaba,Goiaba,27,,-22.3,-47.966667
br,goiabal,Goiabal,04,,-3.116667,-68.05
br,goiabal,Goiabal,15,,-19.933333,-42.7
br,goiabal,Goiabal,16,,-1.05,-49.6
br,goiabas,Goiabas,15,,-19.1,-43.166667
br,goiabeira,Goiabeira,02,,-8.966667,-36.116667
br,goiabeira,Goiabeira,05,,-14.866667,-39.35
br,goiabeira,Goiabeira,06,,-7.583333,-38.75
br,goiabeira,Goiabeira,15,,-18.983333,-41.2
br,goiabeira,Goiabeira,17,,-7.083333,-37.5
br,goiabeira,Goiabeira,20,,-4.483333,-41.466667
br,goiabeira,Goiabeira,26,,-27.633333,-50.85
br,goiabeiras,Goiabeiras,08,,-20.266667,-40.3
br,goiabeiras,Goiabeiras,15,,-18.983333,-41.2
br,goiaberira,Goiaberira,20,,-5.35,-41.866667
br,goialina,Goialina,29,,-16.1,-49.333333
br,goiana,Goianá,15,,-21.533333,-43.2
br,goiana,Goianã,27,,-23.567958,-47.190678
br,goiana,Goianá,29,,-16.666667,-49.266667
br,goiana,Goiana,30,43981,-7.55,-34.983333
br,goianapolis,Goianápolis,29,10114,-16.5,-49.016667
br,goianases,Goianases,15,,-20.7,-47.033333
br,goianas,Goianás,29,,-16.383333,-49.316667
br,goianazes,Goianazes,15,,-20.7,-47.033333
br,goiandira,Goiandira,29,,-18.133333,-48.1
br,goiandira,Goiandira,31,,-11.766667,-46.666667
br,goianesia,Goianésia,29,47884,-15.3,-49.116667
br,goiania,Goiânia,29,1171209,-16.666667,-49.266667
br,goianinha,Goianinha,06,,-7.4,-39.116667
br,goianinha,Goianinha,22,13434,-6.266667,-35.2
br,goianinha,Goianinha,30,,-7.583333,-35.1
br,goianira,Goianira,29,22823,-16.5,-49.433333
br,goianopolis,Goianópolis,29,,-16.5,-49.016667
br,goianorte,Goianorte,31,,-8.726389,-48.817778
br,goiapabo-acu,Goiapabo-Açu,08,,-19.9,-40.45
br,goias,Goiás,29,19098,-15.933333,-50.133333
br,goiatins,Goiatins,31,,-7.700833,-47.179444
br,goiatuba,Goiatuba,29,25925,-18.016667,-49.366667
br,goiaz,Goiaz,29,,-15.933333,-50.133333
br,goio en,Goio En,26,,-27.283333,-52.683333
br,goio-ere,Goio-Erê,18,,-24.2,-53.016667
br,goio-ere,Goio-Erè,26,,-27.283333,-52.683333
br,goioxim,Goioxim,18,,-25.233333,-52.016667
br,goioxin,Goioxin,18,,-25.233333,-52.016667
br,goipaba acu,Goipaba Açu,08,,-19.9,-40.45
br,goipabo,Goipabo,08,,-19.9,-40.45
br,gois artigas,Góis Artigas,18,,-25.45,-51.25
br,gois calmon,Góis Calmon,05,,-12.783333,-38.35
br,gois,Góis,15,,-22.55,-46.3
br,gois,Gois,30,,-7.783333,-37.75
br,goitacases,Goitacases,21,,-21.816667,-41.266667
br,goitacazes,Goitacazes,21,,-21.816667,-41.266667
br,goitacaz,Goitacaz,04,,-3.564722,-62.963056
br,goitizeiro,Goitizeiro,08,,-20.2,-40.366667
br,golabeira,Golabeira,26,,-27.633333,-50.85
br,golden green,Golden Green,21,,-23.007568,-43.3486
br,golfo,Golfo,05,,-13.65,-40.133333
br,golfos,Golfos,05,,-14.133333,-43.433333
br,golfos,Golfos,17,,-6.416667,-38.483333
br,gomalina,Gomalina,14,,-15.816667,-59.802778
br,gomaticum,Gomaticum,05,,-12.366667,-38.516667
br,gomes,Gomes,06,,-7.483333,-38.733333
br,gomes,Gomes,17,,-7.466667,-38.1
br,gomes,Gomes,18,,-23.2817,-50.2265
br,gomes,Gomes,23,,-29.7,-51.966667
br,goncalo alves,Gonçalo Alves,20,,-3.756944,-42.081111
br,goncalo,Gonçalo,05,,-11.183333,-40.266667
br,goncalo,Gonçalo,13,,-3.044167,-43.231667
br,goncalo,Goncalo,17,,-7.55,-36.1
br,goncalo,Gonçalo,24,,-9.75,-62.266667
br,goncalo,Gonçalo,30,,-9.1,-37.15
br,goncalo martins,Gonçalo Martins,05,,-14.166667,-42.666667
br,goncalves dias,Gonçalves Dias,13,,-3.666667,-45.783333
br,goncalves ferreira,Gonçalves Ferreira,30,,-8.266667,-35.883333
br,goncalves,Gonçalves,15,,-22.666667,-45.85
br,goncalves,Gonçalves,27,,-20.266667,-47.616667
br,goncalves junior,Gonçalves Júnior,18,,-25.483333,-50.783333
br,gongogi,Gongogi,05,,-14.316667,-39.483333
br,gongonhas,Gongonhas,15,,-20.5,-43.866667
br,gonzaga,Gonzaga,15,,-18.8,-42.5
br,gopouva,Gopoúva,27,,-23.466667,-46.533333
br,goqay,Goqay,30,,-8.65,-39.65
br,gordo,Gordo,02,,-9.983333,-37.033333
br,gordura,Gordura,15,,-20.035,-43.9433
br,gorgulho,Gorgulho,31,,-8.923056,-48.106667
br,gorino,Gorino,29,,-14.633333,-49.733333
br,gorita,Gorita,15,,-21.066667,-46.483333
br,gororos,Gororós,15,,-18.783333,-43.25
br,gorutuba,Gorutuba,15,,-15.833333,-43.2
br,gostoso,Gostoso,22,,-5.116667,-35.633333
br,gotas,Gotas,17,,-7.266667,-37.616667
br,gotovelo,Gotovelo,22,,-5.933333,-35.933333
br,goulart,Goulart,27,,-21.133333,-50.283333
br,gouvea,Gouvêa,15,,-18.45,-43.733333
br,gouveia,Gouveia,15,,-18.45,-43.733333
br,governador amaral peixoto,Governador Amaral Peixoto,21,,-22.633333,-43.566667
br,governador archer,Governador Archer,13,,-4.933333,-44.383333
br,governador dix-sept rosado,Governador Dix-Sept Rosado,22,6626,-5.466667,-37.516667
br,governador lindemberg,Governador Lindemberg,08,,-19.266667,-40.483333
br,governador lindenberg,Governador Lindenberg,08,,-19.266667,-40.483333
br,governador lupion,Governador Lupion,18,,-25.983333,-53.716667
br,governador portela,Governador Portela,21,,-22.471389,-43.4925
br,governador portella,Governador Portella,21,,-2.483333,-43.5
br,governador valadares,Governador Valadares,15,250882,-18.85,-41.933333
br,goyandira,Goyandira,29,,-18.133333,-48.1
br,goyania,Goyania,29,,-16.666667,-49.266667
br,goyaninha,Goyaninha,06,,-7.4,-39.116667
br,goyanna,Goyanna,30,,-7.55,-34.983333
br,goyanninha,Goyanninha,22,,-6.266667,-35.2
br,goyanninha,Goyanninha,30,,-7.583333,-35.1
br,goyaz,Goyaz,29,,-15.933333,-50.133333
br,goy en,Goy En,26,,-27.283333,-52.683333
br,goyo-en,Goyo-En,26,,-27.283333,-52.683333
br,goza,Goza,05,,-11.233333,-42.083333
br,graca aranha,Graça Aranha,08,,-19.25,-40.6
br,graca aranha,Graça Aranha,13,,-5.383333,-44.333333
br,graca de deus,Graça de Deus,11,,-22.433333,-55.416667
br,graca,Graça,05,,-12.996667,-38.525
br,graca,Graça,06,,-4.05,-40.75
br,gracas a deus,Graças a Deus,05,,-14.7,-39.7
br,gracas,Graças,06,,-3.15,-39.75
br,graca torta,Graça Torta,02,,-9.583333,-35.666667
br,gracho cardoso,Gracho Cardoso,28,,-10.233333,-37.2
br,gracianopolis,Gracianópolis,27,,-21.433333,-51.6
br,graciosa,Graciosa,05,,-13.266667,-40.016667
br,graciosa,Graciosa,20,,-6.783333,-41.266667
br,gracismoes,Gracismões,06,,-4.983333,-37.933333
br,gradaus,Gradaús,16,,-7.716667,-51.183333
br,gragoata,Gragoatá,21,,-22.89899,-43.129635
br,grajahu,Grajahú,13,,-5.816667,-46.133333
br,grajau,Grajaú,05,,-9.733333,-39.033333
br,grajau,Grajaú,13,30218,-5.816667,-46.133333
br,grajau,Grajaú,20,,-7.833333,-41.283333
br,grajau,Grajaú,21,,-22.920256,-43.264573
br,gralau,Gralau,13,,-5.816667,-46.133333
br,gralha,Gralha,27,,-22.45,-49.516667
br,gramacho,Gramacho,21,,-22.755929,-43.293393
br,gramacno,Gramacno,21,,-22.755929,-43.293393
br,gramadinho,Gramadinho,18,,-25.083333,-52.183333
br,gramadinho,Gramadinho,27,,-23.766667,-48.116667
br,gramado,Gramado,18,,-23.6593,-49.8287
br,gramado,Gramado,23,26365,-29.4,-50.9
br,gramado,Gramado,26,,-26.966667,-51.083333
br,gramado,Gramado,27,,-23.283333,-45.5
br,gramados,Gramados,18,,-26.066667,-52.5
br,gramado xavier,Gramado Xavier,23,,-29.25,-52.483333
br,grama,Grama,05,,-13.9,-40.033333
br,grama,Grama,15,,-16.85,-42.15
br,grama,Grama,20,,-6.016667,-42.3
br,grama,Grama,21,,-22.75,-43.266667
br,grama,Grama,23,,-29.4,-50.9
br,grama,Grama,27,,-21.716667,-46.816667
br,gramal,Gramal,21,,-21.816667,-42.166667
br,gramame,Gramame,17,,-7.216667,-34.883333
br,gramas,Gramas,26,,-26.683333,-52.166667
br,graminea,Gramínea,15,,-22.166667,-46.633333
br,granada,Granada,15,,-20.183333,-42.4
br,granada,Granada,18,,-23.216667,-51.733333
br,grand degrad,Grand Dégrad,03,,2.316667,-51.533333
br,grande,Grande,05,,-11.5,-43.766667
br,grande,Grande,23,,-28.966667,-53.9
br,grande,Grande,30,,-9.3,-38.25
br,grande nova,Grande Nova,05,,-12.416667,-39.283333
br,grande oriente,Grande Oriente,04,,-7.666667,-67.666667
br,grandes rios,Grandes Rios,18,,-24.133333,-51.516667
br,grande vista,Grande Vista,05,,-12.25,-40.75
br,grand pegrand,Grand Pégrand,03,,2.316667,-51.533333
br,granito,Granito,30,,-7.716667,-39.6
br,granja beira mar,Granja Beira Mar,22,,-5.966667,-35.15
br,granja comari,Granja Comari,21,,-22.447949,-42.977514
br,granja do silencio,Granja do Silêncio,23,,-29.833333,-57.1
br,granja,Granja,04,,-2.883333,-57.066667
br,granja,Granja,06,25880,-3.1,-40.833333
br,granja santa fe,Granja Santa Fé,20,,-5.066667,-42.633333
br,granjas,Granjas,15,,-17.3,-42.233333
br,granjas,Granjas,18,,-23.197107,-52.209457
br,granja silvia,Granja Silvia,23,,-32.416667,-53.416667
br,granja velha,Granja Velha,18,,-25.7764,-49.7589
br,granjeiro,Granjeiro,06,,-6.883333,-39.2
br,granzato,Granzato,23,,-27.883333,-51.783333
br,granzoto,Granzoto,23,,-27.883333,-51.783333
br,graolandia,Graolândia,31,,-8.106944,-47.012222
br,grao mogol,Grão Mogol,15,3278,-16.566667,-42.9
br,grao para,Grão Pará,26,,-28.166667,-49.216667
br,grato,Grato,04,,-7.483333,-63.033333
br,grauna,Grauna,17,,-6.2,-37.3
br,gravata de fora,Gravata de Fora,17,,-7.433333,-36.3
br,gravata de jaburu,Gravatá de Jaburu,30,,-7.866667,-36.1
br,gravata do ibiapina,Gravatá do Ibiapina,30,,-7.866667,-36.1
br,gravata,Gravatá,02,,-9.066667,-36.133333
br,gravata,Gravatá,05,,-13.35,-41.666667
br,gravata,Gravata,06,,-7.333333,-38.683333
br,gravata,Gravata,15,,-17.483333,-41.883333
br,gravata,Gravatá,17,,-6.983333,-35.483333
br,gravata,Gravata,20,,-10.55,-45.266667
br,gravata,Gravatá,26,,-28.333333,-49.033333
br,gravata,Gravatá,30,61250,-8.2,-35.566667
br,gravatahy,Gravatahy,23,,-29.9426,-50.9933
br,gravatai,Gravataí,23,238782,-29.9426,-50.9933
br,gravatal,Gravatal,26,,-28.333333,-49.033333
br,gravier,Gravier,06,,-4.796389,-37.347222
br,gravi,Gravi,11,,-20.283333,-56.35
br,greenwood park,Greenwood Park,21,,-22.983239,-43.307458
br,gref,Gref,11,,-22.5,-55.4
br,gregorio barbosa,Gregório Barbosa,15,,-18.166667,-41.5
br,gregorio,Gregório,05,,-11.4,-43.9
br,gregorio,Gregorio,06,,-3.5,-40.25
br,gregorio,Gregório,11,,-20.766667,-56.716667
br,gregorio,Gregorio,14,,-10.366667,-60.483333
br,gregorio jordes,Gregorio Jordes,14,,-14.116667,-56.683333
br,gregorio marinho,Gregório Marinho,14,,-10.366667,-60.483333
br,gremios,Grêmios,06,,-4.466667,-39.633333
br,grilo,Grilo,27,,-23.7737,-47.0319
br,grilos,Grilos,30,,-7.666667,-40.05
br,grimeia,Griméia,13,,-5.416667,-43.4
br,grimpas,Grimpas,29,,-16.966667,-49.233333
br,gritos,Gritos,30,,-7.666667,-40.05
br,groairas,Groaíras,06,,-3.883333,-40.383333
br,grossas,Grossas,22,,-4.986111,-37.151944
br,grosso,Grosso,02,,-9.616667,-37.483333
br,grosso,Grosso,05,,-12.95,-39.266667
br,grosso,Grosso,17,,-6.8,-38.25
br,grosso,Grosso,30,,-7.533333,-37.283333
br,grossos,Grossos,06,,-3.533333,-40.05
br,grossos,Grossos,13,,-3.065556,-42.168611
br,grossos,Grossos,20,,-3.512222,-42.055833
br,grossos,Grossos,22,,-4.986111,-37.151944
br,grota do tobi,Grota do Tobi,05,,-12.2,-40.933333
br,grota do tubi,Grota do Tubi,05,,-12.2,-40.933333
br,grota funda,Grota Funda,20,,-6.833333,-40.533333
br,grota funda,Grota Funda,21,,-23.008498,-43.5182
br,grota funda,Grota Funda,26,,-26.333333,-49.133333
br,grota funda,Grota Funda,27,,-23.37847,-47.813178
br,grota,Grota,05,,-10.85,-43.35
br,grota,Grota,06,,-5.233333,-40.733333
br,grota,Grota,15,,-20.4,-42.566667
br,grota,Grota,21,,-22.8593,-43.270526
br,grotao,Grotão,02,,-9.783333,-36.366667
br,grotao,Grotão,05,,-10.816667,-41.683333
br,grotao,Grotão,15,,-21.633333,-45.283333
br,grotao,Grotão,17,,-6.75,-37.9
br,grotao,Grotão,20,,-5.633333,-42.666667
br,grotao,Grotão,21,,-22.8472,-43.292041
br,grotao,Grotão,29,,-14.616667,-46.183333
br,grotao,Grotão,30,,-8.7,-36.683333
br,grotas,Grotas,14,,-14.9,-56.533333
br,grota verde,Grota Verde,06,,-4.866667,-40.116667
br,grumari,Grumari,21,,-23.042408,-43.536149
br,grumarim,Grumarim,21,,-21.6,-41.783333
br,gruna,Gruna,05,,-10.683333,-41.516667
br,grupiara,Grupiara,15,,-18.5,-47.733333
br,grupo escolar,Grupo Escolar,29,,-15.216667,-49.583333
br,grussai,Grussaí,21,,-21.7,-41.033333
br,gruta funda,Gruta Funda,13,,-6.183333,-46.933333
br,gruta funda,Gruta Funda,22,,-5.966667,-35.8
br,gruta,Gruta,23,,-29.2402,-51.0375
br,gruzeiro,Gruzeiro,23,,-27.866667,-54.433333
br,guabiju,Guabiju,23,,-28.533333,-51.7
br,guabiraba,Guabiraba,06,,-2.9,-39.933333
br,guabiraba,Guabiraba,13,,-3.530833,-43.233333
br,guabiraba,Guabiraba,20,,-3.265,-42.111389
br,guabiraba,Guabiraba,30,,-8.416667,-35.666667
br,guabirabas,Guabirabas,20,,-6.883333,-41.783333
br,guabiroba,Guabiroba,18,,-23.618,-49.786
br,guabiroba,Guabiroba,23,,-29.8868,-50.8158
br,guacho,Guacho,23,,-28.733333,-51.083333
br,guachos,Guachos,27,,-21.966667,-51.033333
br,guaci,Guaci,06,,-4.25,-38.833333
br,guacoce,Guaçocê,06,,-6.266667,-39.016667
br,guacoce,Guaçoce,17,,-7.55,-38.516667
br,guacu,Guaçu,11,,-22.183333,-54.516667
br,guacu,Guaçu,27,,-23.502765,-47.132564
br,guacui,Guaçuí,08,19787,-20.766667,-41.683333
br,guacupi,Guaçupi,23,,-29.266667,-53.666667
br,guadalupe,Guadalupe,13,,-3.545,-43.253889
br,guadalupe,Guadalupe,20,,-6.733333,-43.783333
br,guadalupe,Guadalupe,21,,-22.841158,-43.372279
br,guadalupe,Guadalupe,30,,-8.683333,-35.1
br,guadiana,Guadiana,18,,-23.316667,-52.116667
br,guadrapas,Guadrapas,06,,-3.366667,-39.283333
br,guagiru,Guagiru,06,,-3.633333,-38.733333
br,guagiru,Guagiru,17,,-7.533333,-34.833333
br,guahyba,Guahyba,23,,-3.1,-51.316667
br,guahyra,Guahyra,18,,-24.066667,-54.25
br,guahyra,Guahyra,27,,-20.316667,-48.3
br,guaiaca dos pretos,Guaiaca dos Pretos,18,,-25.634072,-50.187003
br,guaiaca,Guaiaca,18,,-25.619969,-50.20077
br,guaianases,Guaianases,27,,-23.55,-46.416667
br,guaianas,Guaianás,27,,-22.3,-48.883333
br,guaianazes,Guaianazes,27,,-23.55,-46.416667
br,guaianeses,Guaianeses,27,,-23.55,-46.416667
br,guaiba,Guaíba,23,101026,-30.111111,-51.316667
br,guaibal,Guaibal,04,,-3.116667,-68.05
br,guaibim,Guaibim,05,,-13.283333,-38.966667
br,guaibinha,Guaibinha,05,,-12.7,-38.9
br,guaicara,Guaiçara,27,,-21.633333,-49.783333
br,guaicuhy,Guaicuhy,15,,-17.2,-44.816667
br,guaicui,Guaicuí,15,,-17.2,-44.816667
br,guaicurus,Guaicurus,23,,-28.966667,-51.4
br,guai,Guaí,05,,-12.85,-38.95
br,guaimbe,Guaimbê,27,,-21.916667,-49.9
br,guaipava,Guaipava,15,,-21.666667,-45.716667
br,guaiquica,Guaiquica,27,,-22.416667,-47.233333
br,guairaca,Guairacá,18,,-22.783333,-51.7
br,guaira,Guaíra,18,27611,-24.066667,-54.25
br,guaira,Guaíra,27,34093,-20.316667,-48.3
br,guairas,Guaíras,05,,-12.65,-44.266667
br,guaiuba,Guaiuba,06,21822,-4.033333,-38.633333
br,guajara-acu,Guajará-Açu,16,,-1.633333,-48.116667
br,guajara acu,Guajará Açu,24,,-10.766667,-65.366667
br,guajara-assu,Guajará-Assu,24,,-10.766667,-65.366667
br,guajara,Guajará,04,9776,-2.966667,-57.666667
br,guajara,Guajará,16,,-.516667,-49.033333
br,guajara-miri,Guajará-Miri,16,,-1.483333,-48.283333
br,guajara-mirim,Guajará-Mirim,24,34123,-10.8,-65.366667
br,guajara sinha,Guajara Sinha,04,,-5.635556,-63.371111
br,guajaratuba,Guajaratuba,04,,-5.115833,-62.732222
br,guajaratubinha,Guajaratubinha,04,,-5.635556,-63.371111
br,guajarauna,Guajaraúna,16,,-1.766667,-48.7
br,guajarraa,Guajarraã,04,,-7.7,-66.883333
br,guajeru,Guajeru,05,,-14.566667,-41.933333
br,guajiru,Guajiru,06,,-4.786389,-37.296389
br,guajiru,Guajirú,17,,-7.533333,-34.833333
br,guajiru,Guajiru,22,,-5.116667,-35.633333
br,guajuru,Guajuru,08,,-20.301389,-40.291667
br,guajuvira de cima,Guajuvira de Cima,18,,-25.633333,-49.466667
br,guajuvira,Guajuvira,18,,-25.6,-49.533333
br,gualibade,Gualibade,21,,-22.215278,-43.710278
br,guama,Guamá,16,,-1.616667,-47.45
br,guamare,Guamaré,22,,-5.106389,-36.320278
br,guamiranga,Guamiranga,18,,-25.2,-50.783333
br,guamiranga,Guamiranga,26,,-26.516667,-48.9
br,guamirim,Guamirim,18,,-25.566667,-50.866667
br,guampara,Guampará,18,,-25.033333,-52.183333
br,guanabara,Guanabara,01,,-10.4,-69.583333
br,guanabara,Guanabara,04,,-7.666667,-67.333333
br,guanabara,Guanabara,06,,-4.866667,-38.4
br,guanabara,Guanabara,22,,-5.983333,-35.933333
br,guanabara,Guanabara,27,,-20.766667,-51.183333
br,guanaces,Guanacés,06,,-4.15,-38.333333
br,guanambi,Guanambi,05,58879,-14.216667,-42.783333
br,guanamby,Guanamby,05,,-14.216667,-42.783333
br,guananazes,Guananazes,23,,-29.516667,-50.066667
br,guanandi,Guanandi,11,,-18.866667,-56.266667
br,guando,Guandó,08,,-20.25,-41.083333
br,guandu do sena,Guandu do Sena,21,,-22.838658,-43.504509
br,guandu,Guandu,02,,-9.233333,-37.233333
br,guandu,Guandu,08,,-20.25,-41.083333
br,guandu,Guandu,21,,-21.55,-41.333333
br,guanhaes,Guanhães,15,23728,-18.783333,-42.95
br,guanipa-mirim,Guanipá-Mirim,05,,-15.7,-39.65
br,guanumbi,Guanumbi,30,,-8.65,-37.033333
br,guapaba,Guapaba,17,,-7.6,-35.583333
br,guape,Guapé,15,,-20.783333,-45.916667
br,guapiacu,Guapiaçu,27,14065,-20.833333,-49.216667
br,guapiara,Guapiara,27,,-24.166667,-48.533333
br,guapiassu,Guapiassú,27,,-20.833333,-49.216667
br,guapicu,Guapicu,18,,-25.37404,-48.316321
br,guapimirim,Guapimirim,21,31202,-22.533333,-42.983333
br,guapira,Guapira,05,,-12.8,-39.05
br,guapirama,Guapirama,18,,-23.516498,-50.042443
br,guapiranga,Guapiranga,27,,-21.516667,-49.633333
br,guapiuru,Guapiuru,27,,-24.166667,-48.533333
br,guapo,Guapó,29,10075,-16.816667,-49.533333
br,guaporanga,Guaporanga,26,,-27.45,-48.633333
br,guapore,Guaporé,23,20050,-28.85,-51.9
br,guapore,Guapore,24,,-11.9,-64.983333
br,guaporema,Guaporema,18,,-23.316667,-52.833333
br,guaporema,Guaporema,23,,-28.85,-51.9
br,guapua,Guapuã,27,,-20.4,-47.416667
br,guapy-mirim,Guapy-mirim,21,,-22.533333,-42.983333
br,guarabira,Guarabira,17,43750,-6.85,-35.483333
br,guarabira,Guarabira,20,,-6.266667,-41.766667
br,guarabu,Guarabu,21,,-22.807151,-43.193964
br,guaracai,Guaraçaí,27,,-21.033333,-51.133333
br,guaraciaba d'oeste,Guaraciaba d'Oeste,27,,-21.383333,-51.666667
br,guaraciaba do norte,Guaraciaba do Norte,06,18186,-4.166667,-40.766667
br,guaraciaba,Guaraciaba,15,,-20.566667,-43
br,guaraciaba,Guaraciaba,26,,-26.583333,-53.583333
br,guaraciama,Guaraciama,15,,-17.05,-43.683333
br,guaraci,Guaraci,05,,-14.633333,-39.533333
br,guaraci,Guaraci,18,,-22.95,-51.666667
br,guaraci,Guaraci,27,,-20.483333,-48.95
br,guaracy,Guaracy,27,,-20.483333,-48.95
br,guaragi,Guaragi,18,,-25.274167,-50.235556
br,guara,Guará,07,,-15.833333,-47.9
br,guara,Guará,18,,-25.383333,-51.283333
br,guara,Guará,27,18263,-20.433333,-47.833333
br,guara,Guará,31,,-8.811667,-48.391667
br,guarai,Guaraí,31,,-8.811667,-48.391667
br,guaraipo,Guaraipo,18,,-24.933333,-49.3
br,guaraipos,Guaraipos,18,,-24.933333,-49.3
br,guaraji,Guaraji,18,,-25.274167,-50.235556
br,guarajuba,Guarajúba,05,,-12.7,-38.1
br,guarakessaba,Guarakessaba,18,,-25.296382,-48.32361
br,guarama,Guarama,23,,-27.566667,-52.05
br,guaramano,Guaramano,23,,-28.133333,-54.566667
br,guaramiranga,Guaramiranga,06,,-4.25,-38.933333
br,guaramirim,Guaramirim,26,21422,-26.45,-49
br,guarana,Guaraná,08,,-19.666667,-40.25
br,guarana,Guaraná,16,,-4.366667,-56.433333
br,guarandi,Guarandi,14,,-16.9,-56.966667
br,guaranesia,Guaranésia,15,17437,-21.3,-46.8
br,guaraniacu,Guaraniaçu,18,7781,-25.1,-52.866667
br,guarani das missoes,Guarani das Missões,23,,-28.133333,-54.566667
br,guarani d'oeste,Guarani d'Oeste,27,,-20.083333,-50.333333
br,guarani,Guarani,01,,-10.116667,-67.8
br,guarani,Guarani,05,,-12.7,-41.833333
br,guarani,Guarani,06,,-3.483333,-39.533333
br,guarani,Guarani,08,,-20.3,-41.366667
br,guarani,Guarani,11,,-19.9,-55.8
br,guarani,Guarani,14,,-17.033333,-56.116667
br,guarani,Guarani,15,,-16.316667,-40.816667
br,guarani,Guarani,16,,-6.45,-57.966667
br,guarani,Guarani,18,,-23.087207,-52.115217
br,guarani,Guarani,20,,-6.866667,-41.766667
br,guarani,Guarani,21,,-22.444317,-42.986268
br,guarani,Guarani,23,,-28.133333,-54.566667
br,guarani,Guarani,27,,-21.666667,-50.883333
br,guarani,Guarani,30,,-7.483333,-39.7
br,guaraniuva,Guaraniúva,27,,-21.566667,-51.283333
br,guaranta,Guarantã,27,,-21.883333,-49.566667
br,guarantan,Guarantan,27,,-21.883333,-49.566667
br,guarany,Guarany,06,,-4.166667,-38.466667
br,guarany,Guarany,15,,-21.366667,-43.05
br,guarapara,Guarapará,27,,-21.5,-48.033333
br,guarapari,Guarapari,08,100927,-20.666667,-40.5
br,guarapary,Guarapary,08,,-20.666667,-40.5
br,guarapiranga,Guarapiranga,27,,-21.966667,-48.25
br,guarapo,Guarapó,27,,-23.272038,-47.891223
br,guarapua,Guarapuã,27,,-22.233333,-48.366667
br,guarapuava,Guarapuava,18,150851,-25.383333,-51.45
br,guarapuavinha,Guarapuavinha,18,,-25.516667,-51.133333
br,guaraquecaba,Guaraqueçaba,18,,-25.296382,-48.32361
br,guaraquicaba,Guaraquiçaba,18,,-25.296382,-48.32361
br,guarara,Guarará,15,,-21.716667,-43.033333
br,guararapes,Guararapes,23,,-28.966667,-51.4
br,guararapes,Guararapes,27,27558,-21.25,-50.633333
br,guararapes,Guararapes,30,,-8.15,-34.95
br,guarare,Guararé,23,,-28.466667,-52.35
br,guararema,Guararema,08,,-18.833333,-40.716667
br,guararema,Guararema,18,,-23.316667,-52.833333
br,guararema,Guararema,27,23572,-23.4132,-46.0377
br,guararu,Guararu,06,,-3.716667,-38.866667
br,guarataia,Guarataia,15,,-18.333333,-41.866667
br,guarataia,Guarataia,21,,-21.45,-41.4
br,guaratiba,Guaratiba,21,,-22.966667,-42.8
br,guaratinga,Guaratinga,05,9519,-16.566667,-39.566667
br,guaratingueta,Guaratinguetá,27,105882,-22.815156,-45.18923
br,guaratuba,Guaratuba,18,18906,-25.8804,-48.571
br,guarau,Guaraú,27,,-23.216667,-47.316667
br,guarau ii,Guaraú II,27,,-24.723498,-48.061022
br,guarauna,Guaraúna,18,,-25.324722,-50.360556
br,guarau segunda,Guaraú Segunda,27,,-24.723498,-48.061022
br,guaravera,Guaravera,18,,-23.533333,-51.183333
br,guaraxaim,Guaraxaim,23,,-31.05,-51.516667
br,guarda do embau,Guarda do Embaú,26,,-27.9025,-48.591111
br,guardado,Guardado,05,,-12.5,-38.85
br,guardado,Guardado,22,,-6.133333,-38.4
br,guarda dos ferreiros,Guarda dos Ferreiros,15,,-19.366667,-46.133333
br,guarda,Guarda,26,,-28.45,-49.083333
br,guarda mor,Guarda Mor,15,,-17.783333,-47.1
br,guarda mor,Guarda Mor,26,,-27.15,-50.7
br,guarda nova,Guarda Nova,23,,-32.016667,-53.45
br,guardinha,Guardinha,15,,-20.916667,-47.166667
br,guardinha,Guardinha,27,,-23.324629,-47.843784
br,guarehy,Guarehy,27,,-23.366667,-48.166667
br,guarei,Guareí,27,,-23.366667,-48.166667
br,guareta,Guaretá,18,,-24.066667,-51.75
br,guariba,Guariba,04,,-5.685556,-61.268611
br,guariba,Guariba,05,,-13.166667,-40.216667
br,guariba,Guariba,13,,-6.283333,-43.116667
br,guariba,Guariba,15,,-16.666667,-41.166667
br,guariba,Guariba,16,,-4.016667,-56.6
br,guariba,Guariba,27,30464,-21.35,-48.233333
br,guariba,Guariba,30,,-8.8,-36.216667
br,guaribas,Guaribas,02,,-9.4,-37.016667
br,guaribas,Guaribas,05,,-11.183333,-45.383333
br,guaribas,Guaribas,06,,-4.683333,-40.25
br,guaribas,Guaribas,13,,-5.216667,-45.916667
br,guaribas,Guaribas,15,,-16.75,-41.25
br,guaribas,Guaribas,17,,-7.033333,-38.3
br,guaribas,Guaribas,20,,-3.865833,-42.683333
br,guaribas,Guaribas,30,,-8.716667,-36.966667
br,guaribas i,Guaribas I,13,,-3.163333,-43.245
br,guaribas ii,Guaribas II,13,,-3.148889,-43.211944
br,guaribas iii,Guaribas III,13,,-3.115278,-43.191944
br,guaribas iv,Guaribas IV,13,,-3.083611,-43.146667
br,guaribas v,Guaribas V,13,,-3.440833,-43.370833
br,guaricanga,Guaricanga,27,,-21.516667,-49.633333
br,guariguari,Guariguari,13,,-3.316667,-45.833333
br,guarima grande,Guarimã Grande,13,,-3.1725,-42.850278
br,guarima,Guarimã,13,,-3.331111,-43.388611
br,guarimanzal,Guarimanzal,13,,-3.090278,-43.494167
br,guarimazinho,Guarimãzinho,13,,-3.149722,-42.795833
br,guarinos,Guarinos,29,,-14.633333,-49.733333
br,guarirama,Guariramã,13,,-2.366667,-45.5
br,guariroba,Guariroba,15,,-21.533333,-46.1
br,guariroba,Guariroba,27,,-21.45,-48.616667
br,guariroba,Guariroba,31,,-11.766667,-48.183333
br,guarita do sul,Guaritá do Sul,18,,-24.116667,-51.5
br,guarita,Guarita,17,,-7.333333,-35.383333
br,guarita,Guaritá,18,,-24.066667,-51.75
br,guarituba,Guarituba,18,,-23.430099,-52.532017
br,guariuba,Guariúba,04,,-5.217778,-60.390833
br,guarizinho,Guarizinho,27,,-23.7,-48.85
br,guarnesia,Guarnesia,15,,-21.3,-46.8
br,guartela,Guartela,18,,-24.633333,-50.3
br,guaruja,Guarujá,11,,-21.883333,-54.033333
br,guaruja,Guarujá,23,,-30.1625,-51.225
br,guaruja,Guarujá,26,,-26.433333,-53.566667
br,guaruja,Guarujá,27,310430,-24,-46.266667
br,guarulhas,Guarulhas,21,,-21.733333,-41.333333
br,guarulhos,Guarulhos,21,,-21.733333,-41.333333
br,guarulhos,Guarulhos,27,1169601,-23.45068,-46.526175
br,guarus,Guarus,21,,-21.733333,-41.333333
br,guary,Guary,27,,-23.7,-48.85
br,guassi,Guassi,06,,-4.25,-38.833333
br,guassosse,Guassossê,06,,-6.266667,-39.016667
br,guatacaba,Guataçaba,29,,-13.85,-46.95
br,guata,Guatá,26,,-28.383333,-49.5
br,guatambu,Guatambu,26,,-27.116667,-52.75
br,guatapara,Guatapará,27,,-21.483333,-47.983333
br,guatapora,Guataporá,27,,-21.483333,-47.983333
br,guataquara,Guataquara,16,,-5.45,-56.866667
br,guatuba,Guatuba,06,,-4.033333,-38.616667
br,guavira,Guavira,11,,-20.5,-56.583333
br,guaxi,Guaxi,11,,-19.266667,-55.5
br,guaxima,Guaxima,15,,-19.866667,-47.566667
br,guaxindiba,Guaxindiba,08,,-18.566667,-39.733333
br,guaxindiba,Guaxindiba,21,,-22.783333,-42.966667
br,guaxinduba,Guaxinduba,27,,-23.283333,-47.016667
br,guaxinduva,Guaxinduva,27,,-23.109722,-46.511944
br,guaxupe,Guaxupé,15,47890,-21.3,-46.7
br,guayanas,Guayanas,27,,-22.3,-48.883333
br,guayanazes,Guayanazes,15,,-20.7,-47.033333
br,guayaramerim,Guayaramerim,24,,-10.8,-65.366667
br,guaycara,Guaycára,27,,-21.633333,-49.783333
br,guayra,Guayrá,18,,-24.066667,-54.25
br,guayra,Guayra,27,,-20.316667,-48.3
br,guayuba,Guayuba,06,,-4.033333,-38.633333
br,gueda,Guedá,11,,-18.883333,-55.65
br,guedes,Guedes,08,,-18.433333,-39.95
br,guedes,Guedes,15,,-20.366667,-44.266667
br,guedes,Guedes,17,,-7.833333,-37.066667
br,guedeslandia,Guedeslândia,06,,-5.766667,-38.85
br,guerem,Guerém,05,,-13.283333,-39.233333
br,guerengue,Guerenguê,21,,-22.927805,-43.37915
br,gueriroba,Gueriroba,31,,-11.766667,-48.183333
br,guerreira,Guerreira,13,,-5.65,-43.35
br,gueto,Gueto,06,,-5.033333,-39.516667
br,guia de pacobaiba,Guia de Pacobaíba,21,,-22.7025,-43.131944
br,guia de pacopahyba,Guia de Pacopahyba,21,,-22.7025,-43.131944
br,guia,Guia,04,,.433333,-67.333333
br,guia,Guia,06,,-4.9,-39.8
br,guia,Guia,14,,-15.366667,-56.233333
br,guia lopes da laguna,Guia Lopes da Laguna,11,9681,-21.433333,-56.116667
br,guia lopes,Guia Lopes,11,,-21.433333,-56.116667
br,guia lopes,Guia Lopes,15,,-20.25,-46.366667
br,guianuba,Güianuba,23,,-29.833333,-51.15
br,guida,Guida,22,,-5.466667,-37.533333
br,guido,Guido,13,,-9.2,-46.566667
br,guidoval,Guidoval,15,,-21.15,-42.8
br,guigos,Guigós,05,,-11.466667,-42.2
br,guilherme da silveira,Guilherme da Silveira,21,,-22.872778,-43.457222
br,guilherme paster,Guilherme Paster,08,,-20.183333,-40.766667
br,guilherme plastes,Guilherme Plastes,08,,-20.183333,-40.766667
br,guilhermino,Guilhermino,15,,-20.233333,-46.883333
br,guimaraes carneiro,Guimarães Carneiro,18,,-23.258527,-50.013306
br,guimaraes,Guimarães,06,,-3.883333,-40.383333
br,guimaraes,Guimarães,13,,-2.133333,-44.6
br,guimarania,Guimarânia,15,,-18.85,-46.783333
br,guinda,Guinda,15,,-18.25,-43.683333
br,guine de baixo,Guiné de Baixo,05,,-12.733333,-41.533333
br,guine,Guiné,05,,-12.8,-41.5
br,guiomar,Guiomar,08,,-20.65,-40.933333
br,guira barreiro,Guirá Barreiro,11,,-23.7,-54.666667
br,guirapa,Guirapá,05,,-14.4,-42.616667
br,guiratinga,Guiratinga,14,10938,-16.35,-53.75
br,guiricema,Guiricema,15,,-21,-42.716667
br,guivotas,Guivotas,04,,-5.61,-63.439167
br,guloso,Guloso,05,,-14.033333,-39.683333
br,gundiana,Gundiana,18,,-23.316667,-52.116667
br,gunza,Gunza,21,,-22.947402,-43.476753
br,guraituba,Guraituba,18,,-25.35,-49.15
br,gurgel,Gurgel,06,,-4.933333,-37.683333
br,gurgel,Gurgel,16,,-7.883333,-49.866667
br,gurgueia,Gurguéia,06,,-6.366667,-40.55
br,gurgueia,Gurgueia,20,,-6.233333,-42.616667
br,gurihu,Gurihú,06,,-2.850556,-40.589167
br,gurijuba,Gurijuba,03,,.866667,-50.316667
br,gurim,Gurim,21,,-22.833333,-42
br,gurinhata,Gurinhatã,15,,-19.233333,-49.8
br,gurinhem,Gurinhém,17,5618,-7.133333,-35.45
br,guriri,Guriri,21,,-21.916667,-41.466667
br,guriu,Guriú,06,,-2.850556,-40.589167
br,gurjao,Gurjão,17,,-7.25,-36.483333
br,gurjao,Gurjão,30,,-7.516667,-35.316667
br,gurupa,Gurupá,16,6461,-1.416667,-51.65
br,gurupa,Gurupá,27,,-21.65,-50.016667
br,gurupai,Gurupai,11,,-23.183333,-54.4
br,gurupa mirim,Gurupá Mirim,05,,-15.7,-39.65
br,gurupay,Gurupay,11,,-23.183333,-54.4
br,gurupi,Gurupí,16,,-1.6,-46.216667
br,gurupi,Gurupi,31,64790,-11.7225,-49.012222
br,gurupy,Gurupy,16,,-1.6,-46.216667
br,gussolandia,Gussolândia,27,,-20.65,-50.683333
br,gustavo bento,Gustavo Bento,15,,-21.033333,-46.616667
br,gustavo berthier,Gustavo Berthier,23,,-27.866667,-51.55
br,gustavo bertier,Gustavo Bertier,23,,-27.866667,-51.55
br,gustavo,Gustavo,05,,-14.633333,-45.5
br,gustavo,Gustavo,14,,-10.1,-60.516667
br,gustavo prates,Gustavo Prates,15,,-16.183333,-40.633333
br,gustavo richard,Gustavo Richard,26,,-26.983333,-49.75
br,guveia,Guveia,15,,-18.45,-43.733333
br,guzolandia,Guzolândia,27,,-20.65,-50.683333
br,gymirim,Gymirim,15,,-21.8,-45.966667
br,ha-mais-tempo,Há-Mais-Tempo,13,,-4.333333,-44.933333
br,hamburgberg,Hamburgberg,23,,-29.666667,-51.1
br,hamburgo,Hamburgo,04,,-2.716667,-66.783333
br,hamburgo velho,Hamburgo Velho,23,,-29.666667,-51.1
br,hammonia,Hammonia,26,,-27.0546,-49.516
br,hamond,Hamond,27,,-21.416667,-48.25
br,hamonia,Hamônia,26,,-27.0546,-49.516
br,hansa,Hansa,26,,-26.433333,-49.233333
br,haras farroupilhas,Haras Farroupilhas,27,,-22.756681,-45.146013
br,haras mondesir,Haras Mondesir,27,,-22.764651,-45.111524
br,harmonia,Harmonia,08,,-21.1,-41.45
br,harmonia,Harmonia,15,,-19.183333,-48.233333
br,harmonia,Harmonia,18,,-24.3,-50.616667
br,harmonia,Harmonia,23,,-29.533333,-51.433333
br,hauer,Hauer,18,,-23.366667,-50.333333
br,havai,Havai,06,,-3.916667,-40.283333
br,havre,Havre,04,,-7.883333,-71.466667
br,hebron,Hebron,06,,-6.266667,-39.583333
br,heimat,Heimat,26,,-26.716667,-49.55
br,heimtal,Heimtal,18,,-23.25,-51.15
br,helena de goias,Helena de Goiás,29,,-17.716667,-50.583333
br,heliodora gomes,Heliodora Gomes,15,,-18.75,-48.5
br,heliodora,Heliodora,15,,-22.066667,-45.533333
br,heliopolis,Heliópolis,05,,-10.683333,-38.283333
br,helvecia,Helvécia,05,,-17.833333,-39.666667
br,helvetia,Helvetia,05,,-17.833333,-39.666667
br,hematita,Hematita,15,,-19.516667,-43.033333
br,henrique dias,Henrique Dias,30,,-8.283333,-37.183333
br,henrique escostegui,Henrique Escostegui,23,,-30.583333,-55.95
br,henrique lage,Henrique Lage,26,,-28.233333,-48.666667
br,henrique laje,Henrique Laje,26,,-28.233333,-48.666667
br,henrique rocha,Henrique Rocha,15,,-16.716667,-41.816667
br,henrique zambao,Henrique Zambão,08,,-20.25,-41.116667
br,heraclito,Heráclito,15,,-18.616667,-48.416667
br,hercilio luz,Hercílio Luz,26,,-28.75,-49.366667
br,herciliopolis,Herciliópolis,26,,-26.8,-51.5
br,herculandia,Herculândia,27,,-22.016667,-50.366667
br,herculanea,Herculânea,11,,-18.5,-54.75
br,herculania,Herculânia,11,,-18.5,-54.75
br,herculania,Herculânia,27,,-22.016667,-50.366667
br,herculano,Herculano,30,,-7.8,-38.35
br,herculanopolis,Herculanópolis,13,,-3.680556,-42.682778
br,hercules coradi,Hércules Coradi,23,,-29.0162,-51.1969
br,hermenegildo,Hermenegildo,14,,-10.283333,-60.516667
br,hermenegildo,Hermenegildo,23,,-33.65,-53.283333
br,hermes silva,Hermes Silva,15,,-18.333333,-41.766667
br,hermilo alves,Hermilo Alves,15,,-21.016667,-43.8
br,hermogenes,Hermógenes,15,,-18.433333,-48.283333
br,hersiliopolis,Hersiliópolis,26,,-26.8,-51.5
br,herva,Herva,15,,-18.466667,-45.45
br,herval,Herval,11,,-20.733333,-56.283333
br,herval,Herval,15,,-20.833333,-42.65
br,herval,Herval,18,,-24.933333,-51.2
br,herval,Herval,23,,-30.6537,-52.2658
br,herval,Herval,26,,-27.216667,-51.566667
br,herval seco,Herval Sêco,23,,-27.55,-53.5
br,herval velho,Herval Velho,26,,-27.283333,-51.433333
br,herveira,Herveira,18,,-25.25,-52.533333
br,herveiras,Herveiras,23,,-29.45,-52.65
br,hevelandia,Hevelândia,04,,-5.203889,-61.835833
br,hibiraiaras,Hibiraiaras,23,,-28.366667,-51.65
br,hibiraires,Hibiraires,23,,-28.366667,-51.65
br,hidrolandia,Hidrolândia,06,10299,-4.416667,-40.416667
br,hidrolandia,Hidrolândia,29,7082,-16.966667,-49.233333
br,hidrolina,Hidrolina,29,,-14.616667,-49.416667
br,higienopolis,Higienópolis,21,,-22.876087,-43.263984
br,higienopolis,Higienópolis,23,,-30.016667,-51.183333
br,hilario vieira,Hilário Vieira,15,,-21.45,-46.183333
br,himalaia,Himalaia,02,,-9.566667,-36.216667
br,hipolito baldina,Hipólito Baldina,15,,-20.883333,-46.133333
br,hipolito cardoso,Hipólito Cardoso,15,,-21.216667,-46.05
br,holanda,Holanda,06,,-4.733333,-40.333333
br,holofote,Holofote,21,,-22.1,-42.416667
br,honorato,Honorato,14,,-10.283333,-60.516667
br,honorio bicalho,Honório Bicalho,15,,-20.023,-43.8214
br,honorio da fonseca,Honório da Fonseca,15,,-20.866667,-46.583333
br,honorio gurgel,Honório Gurgel,21,,-22.843435,-43.354
br,honorio sa,Honório Sá,15,,-16.9,-41.766667
br,horacio alves,Horácio Alves,15,,-20.616667,-46.183333
br,horacio,Horácio,11,,-21.9,-55.35
br,horacio,Horácio,15,,-17.3,-40.816667
br,hora,Hora,13,,-3.15,-44.05
br,horizonte,Horizonte,01,,-8.9425,-72.579722
br,horizonte,Horizonte,05,,-13.7,-40.266667
br,horizonte,Horizonte,06,51181,-4.083333,-38.483333
br,horizonte,Horizonte,21,,-22.483333,-44.516667
br,horizonte,Horizonte,22,,-5.386667,-37.401389
br,horizonte,Horizonte,23,,-27.616667,-54.316667
br,horizonte,Horizonte,30,,-8.783333,-36.033333
br,horizonte novo,Horizonte Novo,05,,-10.35,-39.716667
br,horizontina,Horizontina,23,14416,-27.616667,-54.316667
br,horma,Horma,18,,-22.666667,-51.983333
br,horror,Horror,13,,-3.6,-45.016667
br,hortencias,Hortências,21,,-22.906322,-43.536554
br,hortencio dutra,Hortêncio Dutra,23,,-28.75,-50.233333
br,horto florestal,Horto Florestal,27,,-23.40948,-46.679794
br,horto,Horto,21,,-22.968216,-43.237217
br,hortolandia,Hortolândia,27,203537,-22.85,-47.25
br,hosana,Hosana,22,,-6.266667,-37.75
br,hospedaria,Hospedaria,21,,-22.351461,-42.94053
br,huitanaa,Huitanaã,04,,-7.666667,-65.766667
br,hulha negra,Hulha Negra,23,,-31.4,-53.883333
br,humaita,Humaitá,01,,-8.033333,-71.5
br,humaita,Humaitá,04,31207,-7.516389,-63.022222
br,humaita,Humaitá,05,,-12.35,-40.683333
br,humaita,Humaitá,06,,-3.733333,-40.266667
br,humaita,Humaitá,15,,-16.766667,-41.816667
br,humaita,Humaitá,16,,-2.05,-48.3
br,humaita,Humaitá,17,,-7.416667,-38.566667
br,humaita,Humaitá,21,,-22.957414,-43.199222
br,humaita,Humaitá,23,,-27.566667,-53.966667
br,humayta,Humaytá,01,,-8.261111,-72.738889
br,humayta,Humaytá,04,,-7.516389,-63.022222
br,humayta,Humaytá,05,,-9.116667,-39.716667
br,humayta,Humayta,15,,-21.769722,-43.490278
br,humberto de campos,Humberto de Campos,13,5208,-2.616667,-43.45
br,humberto,Humberto,15,,-20.4,-46.8
br,humberto monte,Humberto Monte,06,,-3.666667,-40.233333
br,humildes,Humildes,05,,-12.366667,-38.883333
br,humildes,Humildes,20,,-5.25,-42.2
br,hungria,Hungria,27,,-23.1923,-47.7318
br,hydrolandia,Hydrolandia,29,,-16.966667,-49.233333
br,hyutanaha,Hyutanahã,04,,-7.666667,-65.766667
br,iabara,Iabará,31,,-10.773611,-47.4075
br,iabete,Iabetê,31,,-10.773611,-47.4075
br,iacaia,Iacaiá,15,,-20.433333,-46.733333
br,iacanga,Iacanga,27,,-21.9,-49.016667
br,iacara,Iaçara,29,,-14.15,-46.666667
br,iaciara,Iaciara,29,9207,-14.15,-46.666667
br,iaci,Iaci,27,,-20.883333,-49.55
br,iacri,Iacri,27,,-21.85,-50.65
br,iacu,Iaçu,05,22665,-12.75,-40.216667
br,ianatama,Ianatama,16,,-1.35,-47.75
br,iandara,Iandara,27,,-21.45,-51.616667
br,iandrinopolis,Iandrinópolis,18,,-23.683333,-51.083333
br,iapi,Iapi,06,,-5.683333,-40.333333
br,iapu,Iapu,15,,-19.433333,-42.216667
br,iaquirana,Iaquirana,04,,1.066667,-67.566667
br,iara,Iara,06,,-7.05,-38.783333
br,iara maria,Iara Maria,23,,-29.4,-54.733333
br,iaramonta,Iaramonta,25,,4.6,-60.4
br,iaras,Iaras,27,,-22.883333,-49.183333
br,iaripo,Iaripo,16,,1.183333,-54.666667
br,iassiara,Iassiara,29,,-14.15,-46.666667
br,iassi,Iassí,05,,-12.75,-40.216667
br,iata,Iata,24,,-10.583333,-65.383333
br,iateca,Iatecá,30,,-8.983333,-36.7
br,iati,Iati,30,,-9.05,-36.85
br,iatoba,Iatobá,30,,-7.95,-37.883333
br,iauacana,Iauacaná,04,,.516667,-67.383333
br,iauare,Iauaré,16,,-7.883333,-57.183333
br,iauarete,Iauaretê,04,,.45,-67.366667
br,iaunari,Iaunari,04,,-.516667,-64.833333
br,iaurare,Iauraré,16,,-7.883333,-57.183333
br,iaurete,Iauretê,04,,.633333,-69.2
br,iavarete,Iavaretê,04,,.633333,-69.2
br,iavarite,Iavarite,04,,-4.383333,-57.266667
br,ibacatu,Ibacatú,23,,-29.416667,-53.133333
br,ibaete,Ibaeté,27,,-21.95,-48
br,ibaiti,Ibaiti,18,25797,-23.8469,-50.1886
br,ibarama,Ibarama,23,,-29.416667,-53.133333
br,ibare,Ibaré,23,,-30.816667,-54.266667
br,ibaretama,Ibaretama,06,,-4.866667,-38.883333
br,ibateguara,Ibateguara,02,8755,-8.983333,-35.866667
br,ibate,Ibaté,27,29550,-21.95,-48
br,ibatequara,Ibatequara,02,,-8.983333,-35.866667
br,ibatiba,Ibatiba,08,12265,-20.233333,-41.516667
br,ibatuba,Ibatuba,15,,-22.066667,-45.05
br,ibatui,Ibatuí,05,,-11.883333,-38.166667
br,ibera peteby,Ibera Peteby,11,,-23.15,-55.033333
br,ibera petehy,Ibera Petehy,11,,-23.15,-55.033333
br,ibera petei,Ibera Petei,11,,-23.15,-55.033333
br,ibera petel,Iberá Petel,11,,-23.15,-55.033333
br,ibertioga,Ibertioga,15,,-21.416667,-43.966667
br,ibiaca,Ibiaçá,23,,-28.05,-51.866667
br,ibiacema,Ibiacema,05,,-11.316667,-43.166667
br,ibiaci,Ibiaci,18,,-22.916667,-51.016667
br,ibiacuce,Ibiaçucê,05,,-14.283333,-42.283333
br,ibiacu,Ibiaçu,05,,-14.266667,-39.25
br,ibia'e,Ibia'e,23,,-28.5,-50.933333
br,ibiagui,Ibiagui,05,,-13.05,-44.2
br,ibiahy,Ibiahy,15,,-16.858889,-44.910556
br,ibia,Ibiá,15,18428,-19.483333,-46.533333
br,ibia,Ibiã,26,,-27.166667,-51.233333
br,ibiai,Ibiaí,15,,-16.858889,-44.910556
br,ibiajara,Ibiajara,05,,-13,-42.2
br,ibiam,Ibiam,26,,-27.166667,-51.233333
br,ibiapaba,Ibiapaba,06,,-5.05,-40.916667
br,ibiapaba,Ibiapaba,08,,-20.233333,-40.416667
br,ibiapina,Ibiapina,06,9205,-3.916667,-40.9
br,ibiapina,Ibiapina,17,,-7.533333,-37.816667
br,ibiapina,Ibiapina,30,,-7.833333,-35.883333
br,ibiapinopolis,Ibiapinópolis,17,,-7.05,-36.35
br,ibiapora,Ibiaporã,05,,-12.05,-40.816667
br,ibiaporan,Ibiaporan,05,,-12.05,-40.816667
br,ibiara,Ibiara,17,,-7.5,-38.416667
br,ibiassuce,Ibiassucê,05,,-14.283333,-42.283333
br,ibiatiara,Ibiatiara,06,,-7.1,-40
br,ibicaba,Ibicaba,08,,-19.966667,-41.216667
br,ibicaba,Ibicaba,27,,-21.5,-47.4
br,ibicacu,Ibicacú,23,,-29.533333,-50.8
br,ibicarai,Ibicaraí,05,19575,-14.85,-39.6
br,ibicare,Ibicaré,26,,-27.116667,-51.4
br,ibicatu,Ibicatu,06,,-6.9,-39.45
br,ibicoara,Ibicoara,05,,-13.4,-41.3
br,ibicua,Ibicuã,06,,-5.933333,-39.416667
br,ibicui,Ibicuí,05,10640,-14.85,-39.983333
br,ibicui,Ibicuí,21,,-22.9606,-44.0266
br,ibicuipeba,Ibicuipeba,06,,-5.25,-38.116667
br,ibicuitaba,Ibicuitaba,06,,-4.738889,-37.312222
br,ibicuitanga,Ibicuitanga,06,,-4.983333,-38.65
br,ibicuitinga,Ibicuitinga,06,,-4.983333,-38.65
br,ibicuituba,Ibicuituba,13,,-2.55,-43.233333
br,ibimirim,Ibimirim,30,12272,-8.533333,-37.7
br,ibiopaba,Ibiopaba,08,,-20.233333,-40.416667
br,ibipeba,Ibipeba,05,14444,-11.633333,-42.033333
br,ibipetuba,Ibipetuba,05,,-11,-44.533333
br,ibipetum,Ibipetum,05,,-11.816667,-42.65
br,ibipeuara,Ibipeuara,13,,-2.966667,-44.85
br,ibipira,Ibipira,13,,-6.516667,-44.633333
br,ibipitanga,Ibipitanga,05,,-12.9,-42.5
br,ibipora,Ibiporã,18,44791,-23.283333,-51.05
br,ibiporanga,Ibiporanga,05,,-14.583333,-40.016667
br,ibiporanga,Ibiporanga,27,,-20.416667,-49.6
br,ibiquara,Ibiquara,05,,-13.4,-41.3
br,ibiquera,Ibiquera,05,,-12.633333,-40.95
br,ibiquitaba,Ibiquitaba,06,,-4.738889,-37.312222
br,ibiraba biraba,Ibiraba Biraba,05,,-10.8,-42.833333
br,ibiracatu,Ibiracatu,15,,-15.65,-44.183333
br,ibiraci,Ibiraci,15,,-20.466667,-47.133333
br,ibiracu,Ibiraçu,08,8921,-19.833333,-40.366667
br,ibiracy,Ibiracy,15,,-20.466667,-47.133333
br,ibiraiaras,Ibiraiaras,23,,-28.366667,-51.65
br,ibira,Ibirá,27,,-21.1,-49.283333
br,ibiraja,Ibirajá,05,,-17.2,-40.15
br,ibirajuba,Ibirajuba,30,,-8.583333,-36.183333
br,ibirama,Ibirama,26,18360,-27.0546,-49.516
br,ibiranete,Ibiranete,11,,-23.15,-55.033333
br,ibiranga,Ibiranga,30,,-7.4,-35.233333
br,ibiranhem,Ibiranhém,05,,-17.833333,-40.166667
br,ibira peteym,Ibira Peteym,11,,-23.15,-55.033333
br,ibirapitanga,Ibirapitanga,05,7302,-14.15,-39.366667
br,ibirapitinga,Ibirapitinga,05,,-14.15,-39.366667
br,ibirapua,Ibirapuã,05,,-17.65,-40.116667
br,ibirapuera,Ibirapuera,27,,-21.75,-50.966667
br,ibirapuita,Ibirapuitã,23,,-28.566667,-52.55
br,ibirarema,Ibirarema,27,,-22.816667,-50.1
br,ibirataia,Ibirataia,05,19943,-14.066667,-39.633333
br,ibiratanha,Ibiratanha,06,,-5.3,-38.05
br,ibiratinga,Ibiratinga,30,,-8.533333,-35.266667
br,ibirite,Ibirité,15,141376,-20.033333,-44.066667
br,ibirituba,Ibirituba,05,,-12.416667,-38.116667
br,ibirocaim,Ibirocaim,05,,-13.733333,-41.766667
br,ibiruba,Ibirubá,23,14180,-28.633333,-53.1
br,ibirucucu,Ibirucuçu,11,,-20.6,-54.75
br,ibitiara,Ibitiara,05,,-12.65,-42.216667
br,ibitiara,Ibitiara,06,,-7.1,-40
br,ibitiguacu,Ibitiguaçu,21,,-21.5,-42.083333
br,ibitiguaia,Ibitiguaia,15,,-21.949722,-43.415833
br,ibitiguassu,Ibitiguassú,21,,-21.5,-42.083333
br,ibitiguira,Ibitiguira,05,,-13.2,-40.15
br,ibiti,Ibiti,27,,-22.666667,-46.683333
br,ibitinema,Ibitinema,21,,-21.55,-42.283333
br,ibitinga,Ibitinga,27,49075,-21.75,-48.816667
br,ibitioca,Ibitioca,21,,-21.816667,-41.466667
br,ibitipoca,Ibitipoca,15,,-21.55,-43.916667
br,ibitipora,Ibitiporã,21,,-21.466667,-42.033333
br,ibitiquira,Ibitiquira,05,,-13.2,-40.15
br,ibitira,Ibitira,05,,-14.116667,-42.083333
br,ibitira,Ibitira,15,,-19.45,-45.133333
br,ibitira,Ibitira,27,,-21.85,-47.633333
br,ibitira,Ibitira,30,,-7.4,-35.233333
br,ibitirama,Ibitirama,08,,-20.533333,-41.666667
br,ibitiranga,Ibitiranga,30,,-7.666667,-37.7
br,ibitirucu,Ibitiruçu,17,,-7.25,-38.316667
br,ibitiruna,Ibitiruna,15,,-21.15,-44.75
br,ibitiruna,Ibitiruna,27,,-22.774167,-47.962778
br,ibitita,Ibititá,05,,-11.566667,-41.966667
br,ibitiuna,Ibitiuna,27,,-22.774167,-47.962778
br,ibitiura,Ibitiúra,15,,-22.066667,-46.433333
br,ibitiuva,Ibitiúva,27,,-20.983333,-48.333333
br,ibituba,Ibituba,08,,-19.683333,-41.016667
br,ibitu,Ibitu,27,,-20.616667,-48.75
br,ibitunane,Ibitunane,05,,-11.55,-42.75
br,ibitupa,Ibitupã,05,,-14.516667,-39.866667
br,ibituporanga,Ibituporanga,21,,-22.764167,-43.791667
br,ibitura,Ibitura,27,,-20.983333,-48.333333
br,ibituruma,Ibituruma,15,,-21.15,-44.75
br,ibituruna,Ibituruna,15,,-21.15,-44.75
br,ibiuna,Ibiúna,27,22860,-23.654522,-47.218152
br,ibo,Ibó,05,,-12.283333,-37.933333
br,ibo,Ibó,30,,-8.616667,-39.733333
br,iborepi,Iborepi,06,,-6.833333,-38.983333
br,iboti,Iboti,27,,-20.816667,-49.633333
br,ibotim,Ibotim,29,,-16.283333,-52.116667
br,ibotirama,Ibotirama,05,17354,-12.183333,-43.216667
br,ibrahy,Ibrahy,15,,-16.858889,-44.910556
br,ibuaco,Ibuaço,06,,-4.75,-39.8
br,ibuacu,Ibuaçu,06,,-4.75,-39.8
br,ibuguacu,Ibuguaçu,06,,-3.366667,-41.266667
br,ibuira,Ibuira,05,,-14.166667,-40.25
br,ibura,Ibura,17,,-7.216667,-37.75
br,ibutinga,Ibutinga,26,,-26.922936,-49.905742
br,icacu,Icaçu,23,,-28.2,-53.483333
br,icaicara,Icaiçara,30,,-8.083333,-39.783333
br,icaicarinha,Icaiçarinha,30,,-8.116667,-38.133333
br,icaissara,Icaissara,30,,-8.083333,-39.783333
br,icaissarinha,Icaissarinha,30,,-8.116667,-38.133333
br,icana,Içana,04,,1.083333,-67.583333
br,icapui,Icapuí,06,,-4.7125,-37.353333
br,icara,Içara,15,,-20.816667,-44.483333
br,icara,Içara,18,,-23.166667,-51.616667
br,icara,Içara,26,51455,-28.7,-49.3
br,icarai,Icaraí,06,,-3.05,-39.633333
br,icarai,Icarai,15,,-16.217778,-44.906667
br,icarai,Icaraí,21,,-22.9,-43.099722
br,icaraima,Icaraíma,18,,-23.383333,-53.683333
br,icatuacu,Icatuaçu,27,,-23.733333,-46.4
br,icatu,Icatu,05,,-10.8,-42.833333
br,icatu,Icatu,13,3648,-2.766667,-44.066667
br,icaturama,Icaturama,27,,-21.466667,-47.35
br,icem,Icém,27,,-20.35,-49.2
br,ichu,Ichu,05,,-11.766667,-39.2
br,icoaraci,Icoaraci,16,,-1.3,-48.466667
br,ico da mundinha,Icó da Mundinha,30,,-9.1,-40.583333
br,ico de d. mundinha,Icó de D. Mundinha,30,,-9.1,-40.583333
br,ico,Icó,05,,-10,-42.5
br,ico,Icó,06,28323,-6.4,-38.85
br,ico,Icó,30,,-7.983333,-38.216667
br,iconha,Iconha,08,5051,-20.8,-40.8
br,icoraci,Icoraci,16,,-1.3,-48.466667
br,icozeira,Icòzeira,05,,-8.933333,-39.316667
br,icozinho,Icòzinho,06,,-6.516667,-38.666667
br,ida,Ida,11,,-20.95,-55.916667
br,idaiolanda,Idaiolanda,27,,-20.7,-49.866667
br,ideal,Ideal,06,,-4.433333,-38.683333
br,ideal,Ideal,27,,-20.8,-51.1
br,ideao,Ideão,16,,-5.716667,-54.216667
br,idiolanda,Idiolanda,27,,-20.7,-49.866667
br,idiolandia,Idiolândia,27,,-20.7,-49.866667
br,ielmo marinho,Ielmo Marinho,22,,-5.816667,-35.55
br,iepe,Iepê,27,,-22.666667,-51.083333
br,iequiri,Iequiri,30,,-8.666667,-36.45
br,igacaba,Igaçaba,27,,-20.133333,-47.45
br,igaci,Igaci,02,6480,-9.55,-36.633333
br,igapira,Igapira,27,,-20.316667,-49.9
br,igapo,Igapo,05,,-13.766667,-42.716667
br,igapo,Igapó,22,,-5.766667,-35.25
br,igapo,Igapó,30,,-9.2,-36.433333
br,igapora,Igaporã,05,,-13.766667,-42.716667
br,igaracu do tiete,Igaraçu do Tietê,27,22758,-22.516667,-48.566667
br,igaracu,Igaraçu,05,,-13.4,-41.3
br,igaracu,Igaraçu,30,,-7.833333,-34.9
br,igarahy,Igarahy,27,,-21.416667,-46.816667
br,igara,Igara,05,,-10.4,-40.116667
br,igarai,Igaraí,05,,-11.616667,-43.4
br,igarai,Igaraí,27,,-21.416667,-46.816667
br,igarapava,Igarapava,27,29058,-20.05,-47.783333
br,igarape-acu,Igarapé-açu,04,,-5.726944,-61.270833
br,igarape-acu,Igarapé-Açu,16,24136,-1.116667,-47.616667
br,igarape-acu,Igarapé-açú,30,,-8.683333,-35.916667
br,igarape-assu,Igarapé-Assú,16,,-1.116667,-47.616667
br,igarapeassu,Igarapeassu,30,,-8.683333,-35.916667
br,igarape azul,Igarapé Azul,25,,-2.216667,-60.966667
br,igarapeba,Igarapeba,30,,-8.816667,-35.883333
br,igarape do lago,Igarapé do Lago,03,,.083333,-51.466667
br,igarape grande,Igarapé Grande,03,,-.133333,-51.866667
br,igarape grande,Igarapé Grande,13,,-4.683333,-44.966667
br,igarape,Igarapé,15,21427,-20.066667,-44.3
br,igarape-miri,Igarapé-Miri,16,26838,-1.983333,-48.966667
br,igarape-mirim,Igarapé-Mirim,16,,-1.983333,-48.966667
br,igarape miry,Igarapé Miry,16,,-1.983333,-48.966667
br,igarape viroa,Igarapé Viroa,25,,.961111,-61.374444
br,igarapiuna,Igarapiuna,05,,-13.816667,-39.15
br,igarassu,Igarassú,05,,-13.4,-41.3
br,igarassu,Igarassú,27,,-22.516667,-48.566667
br,igarassu,Igarassu,30,86458,-7.833333,-34.9
br,igarata,Igaratá,27,,-23.2,-46.116667
br,igarate,Igaraté,27,,-23.2,-46.116667
br,igaratinga,Igaratinga,15,,-19.933333,-44.7
br,igarau,Igaraú,13,,-2.75,-44.316667
br,igarite,Igarité,05,,-11.616667,-43.4
br,igaroi,Igarói,06,,-6.366667,-39.05
br,igataua,Igatauá,30,,-9.133333,-36.466667
br,igatemi,Igatemí,11,,-23.666667,-54.566667
br,igatiquira,Igatiquira,05,,-13.35,-39.816667
br,igatu,Igatu,05,,-12.883333,-41.3
br,ignacio uchoa,Ignacio Uchoa,27,,-20.933333,-49.216667
br,igrapiuna,Igrapiúna,05,,-13.816667,-39.15
br,igreja,Igreja,05,,-13,-38.666667
br,igreja nova,Igreja Nova,02,4606,-10.116667,-36.65
br,igreja nova,Igreja Nova,05,,-12.183333,-38.5
br,igreja nova,Igreja Nova,15,,-18.25,-41.733333
br,igreja nova,Igreja Nova,22,,-5.816667,-35.45
br,igreja nova,Igreja Nova,28,,-10.933333,-38.066667
br,igrejinha,Igrejinha,23,28680,-29.566667,-50.8
br,iguaba grande,Iguaba Grande,21,11104,-22.833333,-42.216667
br,iguaba,Iguaba,05,,-10.283333,-37.933333
br,iguacu,Iguaçu,05,,-12.933333,-41.066667
br,iguacu,Iguaçu,15,,-19.716667,-42.35
br,iguacu,Iguaçu,18,,-25.416667,-52.416667
br,iguacu,Iguaçu,21,,-22.65,-43.416667
br,iguacu,Iguaçu,22,,-5.583333,-35.916667
br,iguacuzinho,Iguaçuzinho,18,,-24.583333,-53.283333
br,iguaiba,Iguaíba,13,,-2.466667,-44.1
br,igua,Iguá,05,,-14.95,-40.933333
br,iguai,Iguaí,05,14218,-14.75,-40.066667
br,igualdade,Igualdade,04,,-1.8,-68.5
br,iguape,Iguape,05,,-12.683333,-38.85
br,iguape,Iguape,06,,-3.933333,-38.266667
br,iguape,Iguape,08,,-20.583333,-40.483333
br,iguape,Iguape,27,23785,-24.716667,-47.55
br,iguaraci,Iguaraci,30,,-7.833333,-37.533333
br,iguaracu,Iguaraçu,18,,-23.183333,-51.833333
br,iguaracu,Iguaraçu,30,,-7.833333,-34.9
br,iguarassu,Iguarassú,30,,-7.833333,-34.9
br,iguaratinga,Iguaratinga,13,,-6.25,-42.866667
br,iguassu,Iguassu,06,,-4.616667,-39.383333
br,iguassu,Iguassú,18,,-25.416667,-52.416667
br,iguatama,Iguatama,15,,-20.166667,-45.7
br,iguatemi,Iguatemi,05,,-13.916667,-41.9
br,iguatemi,Iguatemi,11,9619,-23.666667,-54.566667
br,iguatemi,Iguatemi,18,,-23.35,-52.066667
br,iguatemi,Iguatemi,23,,-31.166667,-52.533333
br,iguati,Iguati,05,,-11.616667,-39.45
br,iguatu,Iguatu,06,70381,-6.366667,-39.3
br,iguatu,Iguatu,13,,-6.133333,-43.766667
br,iguira,Iguira,05,,-10.6,-42.65
br,iguita,Iguita,15,,-21.466667,-46.3
br,iguitu,Iguitu,05,,-11.466667,-42.2
br,ihum,Ihum,11,,-21.583333,-56.716667
br,ihun,Ihun,11,,-21.583333,-56.716667
br,ii-tapuia,Ii-Tapuia,04,,.3,-68.65
br,ijaci,Ijaci,15,,-21.166667,-44.933333
br,ijucapirama,Ijucapirama,23,,-29.4,-54.633333
br,ijuhy,Ijuhy,23,,-28.383333,-53.916667
br,ijui,Ijuí,23,71203,-28.383333,-53.916667
br,ildefonso,Ildefonso,14,,-14.716667,-57.65
br,ilerneiras,Ilerneiras,23,,-29.45,-52.65
br,ilhabela,Ilhabela,27,32903,-23.783333,-45.35
br,ilha brava,Ilha Brava,15,,-18.883333,-42.016667
br,ilha caratirimani,Ilha Caratirimani,25,,.45,-61.683333
br,ilha danta,Ilha Danta,20,,-3.266667,-42.062222
br,ilha das cotias,Ilha das Cotias,04,,-2.233333,-56.716667
br,ilha das cutias,Ilha das Cutias,04,,-2.233333,-56.716667
br,ilha das flores,Ilha das Flores,28,,-10.45,-36.55
br,ilha das pecas,Ilha das Peças,18,,-25.458578,-48.334891
br,ilha de itaoca,Ilha de Itaoca,21,,-22.783333,-43.066667
br,ilha de monte cristo,Ilha de Monte Cristo,04,,-5.587222,-61.124444
br,ilha dentro,Ilha Dentro,05,,-15.1,-41.683333
br,ilha do catrimani,Ilha do Catrimani,25,,.45,-61.683333
br,ilha do ferro,Ilha do Ferro,02,,-9.733333,-37.516667
br,ilha do ouro,Ilha do Ouro,28,,-9.9,-37.25
br,ilha do pao,Ilha do Pão,15,,-16.333333,-40.883333
br,ilha grande,Ilha Grande,02,,-9.133333,-37.95
br,ilha grande,Ilha Grande,04,,-.45,-65.033333
br,ilha grande,Ilha Grande,05,,-13.833333,-40.716667
br,ilha grande,Ilha Grande,06,,-5.5,-38.466667
br,ilha grande,Ilha Grande,18,,-25.066667,-48.75
br,ilha grande,Ilha Grande,26,,-27.55,-49.483333
br,ilha grande,Ilha Grande,28,,-11.066667,-37.2
br,ilha,Ilha,02,,-9.733333,-37.516667
br,ilha,Ilha,05,,-10.616667,-38.283333
br,ilha,Ilha,06,,-2.967222,-41.094444
br,ilha,Ilha,11,,-18.483333,-56.416667
br,ilha,Ilha,13,,-6.633333,-42.866667
br,ilha,Ilha,20,,-7.25,-41.9
br,ilha,Ilha,21,,-23.005133,-43.538387
br,ilha,Ilha,27,,-23.688257,-47.661139
br,ilha,Ilha,28,,-11.233333,-37.916667
br,ilha,Ilha,30,,-8.25,-37.216667
br,ilha luz,Ilha Luz,27,,-23.516667,-45.8
br,ilha rasa,Ilha Rasa,18,,-25.331405,-48.398951
br,ilha redonda,Ilha Redonda,03,,.1,-51.133333
br,ilha redonda,Ilha Redonda,05,,-8.633333,-39.55
br,ilha sao vicente,Ilha São Vicente,31,,-5.616667,-48.133333
br,ilhas,Ilhas,26,,-28.9,-49.333333
br,ilha solteira,Ilha Solteira,27,,-20.383333,-51.333333
br,ilha verde,Ilha Verde,11,,-18.516667,-57.416667
br,ilheos do prata,Ilhéos do Prata,15,,-20,-42.783333
br,ilheos,Ilhéos,05,,-14.816667,-39.033333
br,ilheos,Ilheos,15,,-20,-42.783333
br,ilheus do prata,Ilhéus do Prata,15,,-20,-42.783333
br,ilheus,Ilhéus,05,155503,-14.816667,-39.033333
br,ilhinha,Ilhinha,31,,-6.666667,-48.466667
br,ilho dos bois,Ilho dos Bois,28,,-10.45,-36.55
br,ilhota de cima,Ilhota de Cima,05,,-12.883333,-39.133333
br,ilhota,Ilhota,26,,-26.9,-48.866667
br,ilicinea,Ilicinea,15,,-20.933333,-45.833333
br,ilicinia,Ilicínia,15,,-20.933333,-45.833333
br,ilidio rocha,Ilídio Rocha,11,,-19.6,-55.033333
br,ilio-se,Iliô-Sê,14,,-14.45,-58.466667
br,illidio da rosa,Illidio da Rosa,11,,-19.6,-55.033333
br,illio-se,Illio-sé,14,,-14.45,-58.466667
br,ilopolis,Ilópolis,23,,-28.933333,-52.116667
br,imaculada,Imaculada,17,4014,-7.366667,-37.5
br,imaruhy,Imaruhy,26,,-28.35,-48.816667
br,imarui,Imaruí,26,,-28.35,-48.816667
br,imbahu,Imbahú,18,,-24.383333,-50.65
br,imbare,Imbaré,27,,-21.95,-47.933333
br,imbarie,Imbariê,21,,-22.627778,-43.199167
br,imbauba,Imbaúba,30,,-8.583333,-37.133333
br,imbaubas,Imbaúbas,15,,-17.616667,-42.8
br,imbau,Imbaú,18,,-24.383333,-50.65
br,imbau,Imbaú,21,,-22.6,-42.416667
br,imbauzinho,Imbaùzinho,18,,-24.266667,-50.666667
br,imbe,Imbé,05,,-12.05,-38.016667
br,imbe,Imbé,15,,-19.6,-41.966667
br,imbe,Imbé,21,,-21.983333,-41.883333
br,imbe,Imbé,30,,-7.966667,-37.366667
br,imbetiba,Imbetiba,21,,-22.383333,-41.783333
br,imbiacaba,Imbiaçaba,30,,-7.833333,-40.316667
br,imbiara,Imbiara,05,,-12.666667,-38.9
br,imbiara,Imbiara,21,,-22.8,-42.566667
br,imbiassaba,Imbiassaba,30,,-7.833333,-40.316667
br,imbiricu,Imbiriçu,15,,-19.383333,-44.316667
br,imbirituba,Imbirituba,05,,-12.416667,-38.116667
br,imbirucu,Imbiruçu,11,,-20.6,-54.75
br,imbirucu,Imbiruçu,15,,-19.383333,-44.316667
br,imbirucu,Imbiruçu,20,,-9.5,-42.8
br,imbituba,Imbituba,26,43450,-28.233333,-48.666667
br,imbituva,Imbituva,18,19671,-25.2,-50.583333
br,imblassaba,Imblassaba,30,,-7.833333,-40.316667
br,imboiara,Imboiara,05,,-9.233333,-41.95
br,imbozal,Imbòzal,16,,-1,-52.533333
br,imbu-guacu,Imbu-Guaçu,27,,-23.816667,-46.8
br,imbuia,Imbuia,18,,-24.866667,-51
br,imbuial,Imbuial,18,,-25.183333,-49.133333
br,imbui,Imbuí,08,,-20.616667,-41.683333
br,imbui,Imbuí,21,,-22.320594,-42.866862
br,imbu,Imbu,18,,-24.916667,-50.283333
br,imbu,Imbu,27,,-23.65,-46.85
br,imbuira,Imbuíra,05,,-14.166667,-40.25
br,imbura,Imbura,21,,-22.183333,-41.8
br,imbura,Imbura,30,,-8.6,-37.15
br,imburana,Imburana,06,,-2.867778,-40.785278
br,imburana,Imburana,20,,-3.711667,-42.485833
br,imburana,Imburana,31,,-12.883333,-46.683333
br,imburanas,Imburanas,06,,-4.85,-41.016667
br,imburi,Imburi,21,,-21.483333,-41.283333
br,imburo,Imburo,21,,-22.183333,-41.8
br,imbute,Imbute,21,,-21.383333,-41.65
br,immaculada,Immaculada,17,,-7.366667,-37.5
br,imoroti,Imoroti,17,,-7.516667,-37.666667
br,imperatriz,Imperatriz,04,,-5.3,-67.183333
br,imperatriz,Imperatriz,06,,-3.5,-39.583333
br,imperatriz,Imperatriz,13,218110,-5.533333,-47.483333
br,imperatriz,Imperatriz,15,,-21.233333,-47.066667
br,imperio,Império,08,,-19.533333,-39.816667
br,impertinencia,Impertinência,22,,-6.116667,-38.016667
br,imprensa,Imprensa,30,,-9.266667,-37.166667
br,inacio goncalves,Inácio Gonçalves,15,,-19.016667,-48.1
br,inacio,Inácio,15,,-15.416667,-44.866667
br,inacio,Inácio,17,,-7.666667,-36.05
br,inacio martins,Inácio Martins,18,,-25.516667,-51.133333
br,inacio pires,Inácio Pires,05,,-13.816667,-41.083333
br,inacios,Inácios,15,,-21.366667,-46.983333
br,inacio uchoa,Inácio Uchoa,27,,-20.933333,-49.216667
br,inaja,Inajá,04,,-4.448611,-58.114722
br,inaja,Inajá,18,,-22.733333,-52.233333
br,inaja,Inajá,29,,-17.583333,-48.2
br,inaja,Inajá,30,,-8.9,-37.816667
br,inajaroba,Inajaroba,28,,-11.333333,-37.5
br,inambu,Inambu,04,,-3.583333,-62.433333
br,incaibro,Incaibro,05,,-9.533333,-41.15
br,inchu,Inchu,05,,-11.766667,-39.2
br,inconfidencia,Inconfidência,21,,-22.238611,-43.204167
br,inconfidencia,Inconfidência,23,,-28.65,-54.133333
br,inconfidentes,Inconfidentes,15,,-22.333333,-46.333333
br,indaiabira,Indaiabira,15,,-15.483333,-42.216667
br,indaia do aguapei,Indaiá do Aguapeí,27,,-21.4,-51.133333
br,indaia,Indaiá,05,,-13.683333,-39.583333
br,indaia,Indaiá,11,,-20.033333,-55.033333
br,indaia,Indaiá,15,,-15.983333,-42.383333
br,indaia,Indaiá,20,,-8.816667,-43.016667
br,indaial de baixo,Indaial de Baixo,26,,-28.4,-49
br,indaial de cima,Indaial de Cima,26,,-28.35,-48.933333
br,indaial,Indaial,26,44360,-26.8922,-49.2239
br,indaiatuba,Indaiatuba,27,179914,-23.083333,-47.233333
br,indai,Indaí,05,,-12.016667,-40.566667
br,indaval,Indaval,26,,-26.8922,-49.2239
br,indaya,Indayá,15,,-19.45,-45.6
br,independencia,Independência,01,,-10.566667,-68.216667
br,independencia,Independência,04,,-7.269722,-70.973333
br,independencia,Independência,05,,-16.516667,-39.416667
br,independencia,Independência,06,11113,-5.383333,-40.316667
br,independencia,Independência,08,,-19.183333,-40.5
br,independencia,Independência,11,,-21.683333,-56.383333
br,independencia,Independência,13,,-4.35,-44.333333
br,independencia,Independência,14,,-13.566667,-53.95
br,independencia,Independência,15,,-19.166667,-41.25
br,independencia,Independência,16,,-2.866667,-52.016667
br,independencia,Independencia,22,,-5.263333,-36.723056
br,independencia,Independência,23,,-27.85,-54.2
br,inderere,Indererê,05,,-13.183333,-41.6
br,india,Índia,15,,-15.9,-41.55
br,indiana,Indiana,15,,-17.833333,-40.35
br,indiana,Indiana,27,,-22.183333,-51.233333
br,indianopolis,Indianópolis,15,,-19.033333,-47.916667
br,indianopolis,Indianópolis,18,,-23.479941,-52.697173
br,indiapora,Indiaporã,27,,-19.95,-50.283333
br,indiarana,Indiarana,11,,-19.4,-56.216667
br,indiaroba,Indiaroba,28,4989,-11.533333,-37.516667
br,indio brasil,Indio Brasil,04,,-6.783333,-62.383333
br,indio brazil,Indio Brazil,04,,-6.783333,-62.383333
br,indio do brasil,Índio do Brasil,04,,-6.783333,-62.383333
br,indio,Índio,18,,-25.5,-54.233333
br,indio,Índio,29,,-16.016667,-49.133333
br,indios guarani,Índios Guarani,18,,-25.783333,-51.333333
br,indios,Indios,18,,-25.5,-54.233333
br,indios,Índios,26,,-27.783333,-50.2
br,indubrasil,Indubrasil,11,,-20.466667,-54.733333
br,industrial,Industrial,08,,-18.733333,-40.266667
br,inema,Inema,05,,-14.5,-39.5
br,inema,Inema,21,,-22.194167,-43.311667
br,ines,Inês,13,,-3.65,-45.366667
br,ines,Inês,18,,-22.65,-51.916667
br,inesopolis,Inesópolis,17,,-7.616667,-38.55
br,infernao,Infernão,04,,-8.483333,-60.783333
br,inferninho,Inferninho,27,,-24.742598,-48.226301
br,infinito,Infinito,13,,-5.333333,-43.333333
br,ingahy,Ingahy,15,,-21.4,-44.916667
br,ingahy,Ingahy,27,,-20.416667,-49.366667
br,ingai,Ingaí,15,,-21.4,-44.916667
br,ingai,Ingaí,27,,-20.416667,-49.366667
br,inga,Ingá,05,,-17.333333,-39.3
br,inga,Ingá,06,,-3.916667,-40.183333
br,inga,Ingá,11,,-19.8,-56.566667
br,inga,Ingá,13,,-3.216667,-43.716667
br,inga,Ingá,17,8751,-7.283333,-35.6
br,inga,Ingá,18,,-23.0488,-50.2276
br,inga,Ingá,21,,-22.903473,-43.125181
br,inga,Ingá,22,,-5.533333,-35.583333
br,inga,Ingá,30,,-7.833333,-39.266667
br,ingaseiro,Ingaseiro,11,,-22.083333,-57.916667
br,ingas,Ingás,27,,-20.416667,-49.366667
br,ingazeira,Ingàzeira,05,,-11,-44.933333
br,ingazeira,Ingàzeira,06,,-6.333333,-38.7
br,ingazeira,Ingazeira,20,,-3.865,-42.078056
br,ingazeira,Ingàzeira,21,,-22.2,-41.733333
br,ingazeira,Ingàzeira,30,,-7.666667,-37.466667
br,ingazeiras,Ingàzeiras,06,,-7.1,-39
br,ingazeiras,Ingàzeiras,30,,-7.733333,-39.533333
br,ingazeiro,Ingàzeiro,11,,-22.083333,-57.916667
br,ingazeiro,Ingàzeiro,30,,-8.633333,-38.65
br,inglesa,Inglesa,21,,-22.470833,-43.2225
br,ingles de souza,Inglês de Souza,16,,-1.933333,-54.166667
br,ingleses do rio vermelho,Inglêses do Rio Vermelho,26,,-27.45,-48.366667
br,ingleses,Inglêses,26,,-27.45,-48.366667
br,ingrata,Ingrata,18,,-24.533333,-50.366667
br,inhacora,Inhacorá,23,,-27.883333,-54.033333
br,inhahy,Inhahy,15,,-17.933333,-43.616667
br,inhai,Inhaí,15,,-17.933333,-43.616667
br,inhambupe,Inhambupe,05,14735,-11.783333,-38.35
br,inhambupo,Inhambupo,05,,-11.783333,-38.35
br,inhame,Inhame,15,,-20.35,-45.883333
br,inhames,Inhames,15,,-20.35,-45.883333
br,inhamuns,Inhamuns,06,,-5.866667,-39.85
br,inhandava,Inhandava,23,,-29.616667,-51.933333
br,inhangapi,Inhangapi,16,,-1.416667,-47.883333
br,inhangapy,Inhangapy,16,,-1.416667,-47.883333
br,inhapi,Inhapi,02,7550,-9.233333,-37.75
br,inhapim,Inhapim,15,13594,-19.55,-42.116667
br,inhare,Inharé,06,,-5.416667,-40.55
br,inhare,Inharé,20,,-6.816667,-40.566667
br,inhata,Inhatá,05,,-12.416667,-38.683333
br,inhauma,Inhaúma,02,,-9.15,-36.116667
br,inhauma,Inhaúma,06,,-7.166667,-39.75
br,inhauma,Inhaúma,15,,-19.483333,-44.366667
br,inhauma,Inhaúma,21,,-22.871944,-43.280556
br,inhauma,Inhaúma,30,,-8.316667,-36.4
br,inhaumas,Inhaúmas,05,,-13.033333,-44.616667
br,inhema,Inhema,27,,-22.016667,-49.8
br,inhoaiba,Inhoaíba,21,,-22.905259,-43.603237
br,inhoan,Inhôan,21,,-22.916667,-42.95
br,inhobim,Inhobim,05,,-15.25,-40.95
br,inhomirim,Inhomirim,21,,-22.569167,-43.182778
br,inhucu,Inhuçu,06,,-4.116667,-40.866667
br,inhuma,Inhuma,05,,-12.216667,-38.5
br,inhuma,Inhuma,11,,-20.3,-55.916667
br,inhuma,Inhuma,20,7056,-6.666667,-41.7
br,inhumas,Inhumas,13,,-8.746667,-45.968889
br,inhumas,Inhumas,15,,-16.15,-41.4
br,inhumas,Inhumas,20,,-6.666667,-41.7
br,inhumas,Inhumas,29,44058,-16.366667,-49.5
br,inhuporanga,Inhuporanga,06,,-4.1,-39.05
br,inhussu,Inhussu,06,,-4.116667,-40.866667
br,inhuvera,Inhuverá,23,,-29.466667,-52.083333
br,inimutaba,Inimutaba,15,,-18.75,-44.366667
br,ininga,Ininga,20,,-3.646389,-42.618333
br,inoa,Inoá,04,,-3.981667,-62.837222
br,inoa,Inoã,21,,-22.916667,-42.95
br,inocencia,Inocência,11,,-19.783333,-51.8
br,inojosa,Inojosa,06,,-4.416667,-39.25
br,instituto do prata,Instituto do Prata,16,,-1.3,-47.6
br,insua,Insua,14,,-17.616667,-57.716667
br,insu,Insú,14,,-17.616667,-57.716667
br,inta,Intã,06,,-4.016667,-40.183333
br,inta,Intã,30,,-7.766667,-37.716667
br,intas,Intãs,05,,-9.65,-41.433333
br,intendencia,Intendência,05,,-9.85,-43.05
br,interlagos de itauna,Interlagos de Itaúna,21,,-22.996398,-43.412117
br,interlandia,Interlândia,29,,-16.2,-49.033333
br,inubia,Inúbia,05,,-13.116667,-41.983333
br,inubia,Inubia,27,,-21.75,-50.966667
br,inubia paulista,Inúbia Paulista,27,,-21.75,-50.966667
br,inveia,Inveia,22,,-5.983333,-37.65
br,invejada,Invejada,20,,-5.266667,-41.866667
br,inveja,Inveja,04,,-4.634444,-59.427222
br,inveja,Inveja,05,,-8.9,-38.833333
br,inveja,Inveja,15,,-20.816667,-45.55
br,invernada da cruz,Invernada da Cruz,18,,-25.872729,-50.2279
br,invernada do machado,Invernada do Machado,18,,-26.1,-52.283333
br,invernada grande,Invernada Grande,18,,-24.997851,-48.873815
br,invernada,Invernada,15,,-18.733333,-50.15
br,invernada,Invernada,18,,-25.152863,-48.913933
br,invernada,Invernada,27,,-22.816667,-47.483333
br,invernadinha,Invernadinha,11,,-22.15,-55.066667
br,invernadinha,Invernadinha,27,,-23.347622,-47.920527
br,iomere,Iomerê,26,,-27.016667,-51.25
br,ione,Ione,18,,-23.416667,-50.383333
br,ipaarana,Ipaarana,17,,-7.15,-35.85
br,ipaba,Ipaba,15,15971,-19.383333,-42.416667
br,ipaguacu,Ipaguaçu,06,,-3.5,-40.266667
br,ipameri,Ipameri,20,,-10.366667,-45.133333
br,ipameri,Ipameri,29,18607,-17.716667,-48.15
br,ipamery,Ipamery,29,,-17.716667,-48.15
br,ipanana,Ipanana,15,,-19.4,-42.683333
br,ipane,Ipané,13,,-9.55,-46.15
br,ipane,Ipane,15,,-19.4,-42.683333
br,ipanema de cima,Ipanema de Cima,27,,-23.510047,-47.540333
br,ipanema do meio,Ipanema do Meio,27,,-23.5032,-47.5489
br,ipanema do norte,Ipanema do Norte,18,,-25.204515,-48.330452
br,ipanema,Ipanema,05,,-11.383333,-41.6
br,ipanema,Ipanema,15,11705,-19.783333,-41.733333
br,ipanema,Ipanema,18,,-24.728399,-48.997064
br,ipanema,Ipanema,21,,-22.980968,-43.202864
br,ipanema,Ipanema,23,,-30.133333,-51.229167
br,ipanema,Ipanema,27,,-23.433333,-47.6
br,ipanema,Ipanema,30,,-8.4,-36.816667
br,ipanguacu,Ipanguaçu,22,4624,-5.497778,-36.853333
br,ipanore,Ipanoré,04,,.25,-68.466667
br,ipaobi,Ipaobi,27,,-21.9,-47.033333
br,ipaporanga,Ipaporanga,06,,-4.9,-40.766667
br,iparanga,Iparanga,27,,-23.6,-46.616667
br,ipatinga,Ipatinga,15,228750,-19.5,-42.533333
br,ipatinga,Ipatinga,27,,-23.5028,-47.5289
br,ipauanema,Ipauanema,04,,-4.359167,-62.1175
br,ipaua-pixuna,Ipauá-Pixuna,04,,-2.75,-65.783333
br,ipauarana,Ipauarana,17,,-7.15,-35.85
br,ipaucu,Ipauçu,27,,-23.05,-49.616667
br,ipauguacu,Ipauguaçu,22,,-5.497778,-36.853333
br,ipaumirim,Ipaumirim,06,,-6.783333,-38.716667
br,ipaussu,Ipaussú,27,10833,-23.05,-49.616667
br,ipecaeta,Ipecaetá,05,,-12.316667,-39.3
br,ipecu,Ipeçu,04,,-6.516667,-64.266667
br,ipeguari,Ipeguari,29,,-17.716667,-50.583333
br,ipegue,Ipegue,11,,-20.2,-56.05
br,ipe,Ipê,08,,-20.633333,-40.9
br,ipe,Ipé,11,,-21.016667,-55.816667
br,ipe,Ipê,21,,-22.695833,-43.870556
br,ipe,Ipé,23,,-28.833333,-51.283333
br,iperaba,Iperaba,26,,-26.216667,-48.583333
br,iperoba,Iperoba,26,,-26.216667,-48.583333
br,ipero,Iperó,27,17161,-23.35,-47.683333
br,ipes,Ipês,27,,-21.65,-50.016667
br,ipeuate,Ipeuate,05,,-13.166667,-40.1
br,ipeuna,Ipeúna,27,,-22.433333,-47.716667
br,ipiabanha,Ipiabanha,31,,-9.542222,-48.228056
br,ipiabas,Ipiabas,21,,-22.376944,-43.873056
br,ipiacu,Ipiaçu,15,,-18.7,-49.95
br,ipiau,Ipiaú,05,38763,-14.133333,-39.733333
br,ipigua,Ipiguá,27,,-20.65,-49.366667
br,ipiiba,Ipiíba,21,,-22.866667,-42.95
br,ipiraca,Ipiraca,18,,-24.633333,-53.9
br,ipira,Ipirá,05,27218,-12.166667,-39.733333
br,ipira,Ipira,26,,-27.433333,-51.816667
br,ipiranga do piaui,Ipiranga do Piauí,20,,-6.816667,-41.75
br,ipiranga,Ipiranga,04,,-2.931944,-69.695
br,ipiranga,Ipiranga,05,,-12.466667,-38.716667
br,ipiranga,Ipiranga,06,,-7.066667,-39.816667
br,ipiranga,Ipiranga,08,,-19.516667,-40.283333
br,ipiranga,Ipiranga,11,,-20.933333,-55.583333
br,ipiranga,Ipiranga,15,,-16.716667,-41
br,ipiranga,Ipiranga,18,,-25.016667,-50.583333
br,ipiranga,Ipiranga,20,,-5,-42.083333
br,ipiranga,Ipiranga,21,,-22.894254,-43.573518
br,ipiranga,Ipiranga,22,,-5.45,-35.683333
br,ipiranga,Ipiranga,23,,-27.933333,-52.416667
br,ipiranga,Ipiranga,24,,-10.016667,-62.866667
br,ipiranga,Ipiranga,26,,-26.485,-48.7336
br,ipiranga,Ipiranga,27,,-21.8,-47.7
br,ipiranguinha,Ipiranguinha,18,,-23.8869,-50.124
br,ipiranguinha,Ipiranguinha,27,,-25.0325,-48.1364
br,ipitanga,Ipitanga,05,,-12.9,-38.333333
br,ipitangas,Ipitangas,21,,-22.75,-42.75
br,ipiuna,Ipiúna,05,,-13.583333,-39.866667
br,ipixuna,Ipixuna,04,5814,-7.051667,-71.693056
br,ipixuna,Ipixuna,13,,-4.366667,-44.566667
br,ipixuna,Ipixuna,16,,-.75,-50.966667
br,ipoeira,Ipoeira,14,,-16.4,-56.983333
br,ipoeiras,Ipoeiras,30,,-7.816667,-39.483333
br,ipoema,Ipoema,15,,-19.6,-43.433333
br,ipojuca,Ipojuca,27,,-22.433333,-47.716667
br,ipojuca,Ipojuca,30,53365,-8.4,-35.066667
br,ipomeia,Ipoméia,26,,-26.95,-51.1
br,ipora,Iporã,18,10353,-23.983333,-53.616667
br,ipora,Iporã,26,,-26.983333,-53.583333
br,ipora,Iporá,29,28254,-16.466667,-51.116667
br,iporanga,Iporanga,27,,-24.584536,-48.592226
br,ipuacu,Ipuaçu,05,,-12.283333,-39.016667
br,ipuacu,Ipuaçu,26,,-26.566667,-52.45
br,ipua,Ipuã,27,12370,-20.45,-48.033333
br,ipuarana,Ipuarana,17,,-7.15,-35.85
br,ipuassu,Ipuassú,05,,-12.35,-39.083333
br,ipubi,Ipubi,30,16424,-7.65,-40.116667
br,ipucaba,Ipuçaba,05,,-12.116667,-42.966667
br,ipuca,Ipuca,21,,-21.633333,-41.733333
br,ipueira cercada,Ipueira Cercada,06,,-5.25,-40.266667
br,ipueira de ze manezinho,Ipueira de Zé Manezinho,06,,-7.183333,-38.916667
br,ipueira do macio,Ipueira do Macio,05,,-10.366667,-39.95
br,ipueira do peixe,Ipueíra do Peixe,17,,-6.816667,-37.6
br,ipueira do velho,Ipueira do Velho,06,,-3.533333,-40.083333
br,ipueira,Ipueira,02,,-9.666667,-37.366667
br,ipueira,Ipueira,05,,-12.383333,-39.55
br,ipueira,Ipueira,06,,-3.95,-38.9
br,ipueira,Ipueira,17,,-7.166667,-36.516667
br,ipueira,Ipueira,20,,-6.95,-40.616667
br,ipueira,Ipueira,22,,-5.366667,-37.5
br,ipueira,Ipueira,30,,-7.483333,-37.183333
br,ipueiras,Ipueiras,02,,-9.333333,-37.05
br,ipueiras,Ipueiras,06,17111,-7.033333,-40.45
br,ipueiras,Ipueiras,20,,-7.066667,-41.45
br,ipueiras,Ipueiras,30,,-7.833333,-39.45
br,ipueiras,Ipueiras,31,,-11.232778,-48.351944
br,ipueirinha,Ipueirinha,06,,-3.65,-40.133333
br,ipueirinha,Ipueirinha,17,,-7.65,-36.433333
br,ipueirinha,Ipueirinha,30,,-8.133333,-37.416667
br,ipuera,Ipuera,30,,-7.816667,-39.483333
br,ipu,Ipu,06,26679,-4.333333,-40.7
br,ipu,Ipu,13,,-5.783333,-45.983333
br,ipuiuna,Ipuiúna,15,,-22.1,-46.183333
br,ipu marzagao,Ipu Marzagão,06,,-3.533333,-39.516667
br,ipumirim,Ipumirim,26,,-27.083333,-52.133333
br,ipupiara,Ipupiara,05,,-11.816667,-42.616667
br,ipura,Ipurá,26,,-26.983333,-53.583333
br,ipuyuna,Ipuyuna,15,,-22.1,-46.183333
br,ipuzinho,Ipuzinho,06,,-5.116667,-40.15
br,ipyra,Ipyra,26,,-27.433333,-51.816667
br,iracema,Iracema,01,,-10.533333,-68.15
br,iracema,Iracema,04,,-4.435833,-59.800833
br,iracema,Iracema,05,,-13.283333,-40.966667
br,iracema,Iracema,06,,-3.983333,-40.083333
br,iracema,Iracema,24,,-11.25,-61.933333
br,iracema,Iracema,26,,-26.45,-50.016667
br,iracemapolis,Iracemápolis,27,16796,-22.583333,-47.533333
br,iraceminha,Iraceminha,26,,-26.833333,-53.25
br,iracemopolis,Iracemopolis,27,,-22.583333,-47.533333
br,iraci,Iraci,30,,-8.816667,-35.95
br,iraguacu,Iraguaçu,30,,-7.766667,-38.033333
br,irahy,Irahy,15,,-18.983333,-47.466667
br,irahy,Irahy,23,,-27.183333,-53.25
br,irai de minas,Iraí de Minas,15,,-18.983333,-47.466667
br,irai,Iraí,05,,-11.933333,-38.616667
br,irai,Iraí,15,,-18.983333,-47.466667
br,irai,Iraí,23,,-27.183333,-53.25
br,iraiti,Iraiti,04,,.216667,-69.45
br,iraiti,Iraití,23,,-29.616667,-53.35
br,irajai,Irajaí,30,,-7.933333,-37.45
br,iraja,Irajá,06,,-4.366667,-40.433333
br,iraja,Irajá,13,,-5.133333,-43.1
br,iraja,Irajá,21,,-22.836453,-43.320873
br,irajuba,Irajuba,05,,-13.266667,-40.083333
br,iramaia,Iramaia,05,,-12.366667,-41.366667
br,iranduba grande,Iranduba Grande,04,,-3.25,-60.1
br,iranduba,Iranduba,04,11181,-3.25,-60.1
br,irani,Irani,06,,-5.216667,-39
br,irani,Irani,26,,-27.016667,-51.9
br,iraobi,Iraobí,31,,-11.014167,-48.505
br,irape,Irapé,27,,-23.066667,-49.733333
br,irapicuru,Irapicuru,05,,-12.1,-39.116667
br,irapiranga,Irapiranga,28,,-10.983333,-37.3
br,irapoa,Irapoá,20,,-3.583333,-42.383333
br,iraporanga,Iraporanga,05,,-12.3,-41.55
br,irapua de baixo,Irapuá de Baixo,06,,-5.166667,-40.45
br,irapua de cima,Irapuá de Cima,06,,-5.15,-40.433333
br,irapua,Irapuá,06,,-4.716667,-40.45
br,irapua,Irapuá,20,,-3.583333,-42.383333
br,irapua,Irapuá,23,,-27.5,-53.7
br,irapua,Irapuã,27,,-21.3,-49.4
br,irapuan,Irapuan,27,,-21.3,-49.4
br,irapuazinho,Irapuãzinho,05,,-13.5,-41.75
br,irapuru,Irapuru,16,,-2.45,-50.933333
br,irapuru,Irapuru,27,,-21.566667,-51.35
br,iraputa,Iraputã,26,,-26.45,-50.016667
br,iraquara,Iraquara,05,5131,-12.25,-41.6
br,iraquara,Iraquara,16,,-.916667,-47.216667
br,irara,Irará,05,9099,-12.033333,-38.766667
br,irara,Irara,11,,-19.333333,-51.3
br,iratama,Iratama,30,,-9.033333,-36.416667
br,irateua,Irateua,16,,-1.833333,-52.283333
br,irati,Irati,18,45831,-25.4735,-50.6524
br,irati,Irati,27,,-24.672521,-48.024675
br,iratim,Iratim,18,,-26.466667,-51.416667
br,iratinga,Iratinga,06,,-3.733333,-39.533333
br,irati velho,Irati Velho,18,,-25.5,-50.666667
br,iraty,Iraty,18,,-25.4735,-50.6524
br,iratym,Iratym,18,,-26.466667,-51.416667
br,iraucuba,Irauçuba,06,13282,-3.75,-39.783333
br,irece,Irecê,05,63628,-11.3,-41.866667
br,irere,Irerê,18,,-23.516667,-51.083333
br,iretama,Iretama,18,,-24.283333,-52.033333
br,irgang,Irgang,23,,-28.25,-53.883333
br,iriri,Iriri,08,,-20.833333,-40.7
br,iriri,Iriri,21,,-22.333333,-41.95
br,iriritiba,Iriritiba,08,,-20.7,-40.75
br,irituia,Irituia,16,5602,-1.766667,-47.433333
br,irmaos valadares,Irmãos Valadares,11,,-18.283333,-55.416667
br,irmaos ziemel,Irmãos Ziemel,15,,-17.133333,-41.35
br,iroi,Iroí,18,,-23.25,-52.15
br,irundiara,Irundiara,05,,-14.283333,-42.283333
br,irundi,Irundi,08,,-19.916667,-40.433333
br,irupi,Irupi,08,3505,-20.35,-41.633333
br,isaac portilho,Isaac Portilho,29,,-16.783333,-51.7
br,isabel dos santos,Isabel dos Santos,29,,-16.1,-51.7
br,isabel,Isabel,08,,-20.383333,-40.633333
br,isaias coelho,Isaías Coelho,20,,-7.733333,-41.666667
br,isalino,Isalino,15,,-16.433333,-40.3
br,isidoro alves,Isidoro Alves,11,,-20.15,-55.2
br,isidoro,Isidoro,06,,-6.133333,-39.25
br,isidro alves,Isidro Alves,11,,-20.15,-55.2
br,isidro,Isidro,11,,-21.466667,-55.25
br,israelandia,Israelândia,29,,-16.35,-50.9
br,itaara,Itaara,23,,-29.583333,-53.783333
br,itabaiana,Itabaiana,04,,-6.05,-67.833333
br,itabaiana,Itabaiana,05,,-13.45,-39.45
br,itabaiana,Itabaiana,17,19101,-7.333333,-35.333333
br,itabaiana,Itabaiana,28,63889,-10.683333,-37.433333
br,itabaianinha,Itabaianinha,28,17149,-11.266667,-37.783333
br,itabai,Itabaí,05,,-15.416667,-40.45
br,itabapoana,Itabapoana,08,,-21.2,-41.466667
br,itabapoana,Itabapoana,21,,-21.3,-40.966667
br,itabayanna,Itabayanna,04,,-6.05,-67.833333
br,itabayanna,Itabayanna,17,,-7.333333,-35.333333
br,itabayanninha,Itabayanninha,28,,-11.266667,-37.783333
br,itaberaba,Itaberaba,05,47303,-12.533333,-40.3
br,itaberahy,Itaberahy,29,,-16.033333,-49.8
br,itaberai,Itaberaí,29,22130,-16.033333,-49.8
br,itabera,Itaberá,27,11098,-23.85,-49.15
br,itabi,Itabi,28,,-10.133333,-37.1
br,itabira de mato dentro,Itabira de Mato Dentro,15,,-19.616667,-43.216667
br,itabira de matto dentro,Itabira de Matto Dentro,15,,-19.616667,-43.216667
br,itabira,Itabira,05,,-11.55,-40.45
br,itabira,Itabira,15,99498,-19.616667,-43.216667
br,itabirinha de mantena,Itabirinha de Mantena,15,,-18.533333,-41.25
br,itabirinha,Itabirinha,15,,-18.533333,-41.25
br,itabirito,Itabirito,15,39453,-20.2484,-43.8063
br,itaboca grande,Itaboca Grande,16,,-4.483333,-49.55
br,itaboca,Itaboca,04,,-4.883611,-62.682778
br,itaboca,Itaboca,15,,-22.05,-44.083333
br,itaboca,Itaboca,16,,-4.271389,-52.662222
br,itaborahy,Itaborahy,21,,-22.75,-42.866667
br,itaborahy,Itaborahy,29,,-16.033333,-49.8
br,itaborai,Itaboraí,21,182500,-22.75,-42.866667
br,itabuna,Itabuna,05,205666,-14.8,-39.266667
br,itacaja,Itacajá,31,,-8.359444,-47.628333
br,itacamba,Itacambá,17,,-6.55,-37.8
br,itacambira,Itacambira,15,,-17.066667,-43.3
br,itacara,Itacara,05,,-12.183333,-38.816667
br,itacarambi,Itacarambi,15,13084,-15.016667,-44.05
br,itacaramby,Itacaramby,15,,-15.083333,-44.116667
br,itacare,Itacaré,05,12490,-14.3,-39
br,itacatu,Itacatu,30,,-8.8,-36.466667
br,itacava,Itacava,05,,-12.183333,-38.816667
br,itaccuce,Itaçcucê,05,,-14.233333,-44.3
br,itace,Itacê,29,,-14.683333,-49.45
br,itacema,Itacêma,23,,-29.933333,-51.9
br,itachama,Itachama,05,,-13.033333,-39.716667
br,itaci,Itaci,15,,-20.933333,-45.95
br,itacima,Itacima,06,,-4.133333,-38.716667
br,itacira,Itacira,05,,-12.266667,-41.183333
br,itacoara,Itacoára,21,,-21.683333,-42.066667
br,itacoatiara,Itacoatiara,04,45995,-3.05,-58.433333
br,itacoatiara,Itacoatiara,05,,-11.35,-43.216667
br,itacoatiara,Itacoatiara,21,,-22.969722,-43.031667
br,itacoatiara,Itacoatiara,30,,-8.916667,-36.65
br,itacolomi,Itacolomi,14,,-16.283333,-55.45
br,itacolomi,Itacolomi,15,,-19.016667,-43.566667
br,itacolomi,Itacolomi,21,,-22.811817,-43.232401
br,itacolomi,Itacolomí,23,,-28.7,-50.4
br,itacu,Itaçu,08,,-19.683333,-40.783333
br,itacuruba,Itacuruba,30,,-8.833333,-38.716667
br,itacurubi,Itacurubi,23,,-29.033333,-55.033333
br,itacuruca,Itacuruçá,05,,-12.233333,-39.216667
br,itacuruca,Itacuruçá,21,,-22.925833,-43.912222
br,itacuru,Itacuru,15,,-19.4,-43.316667
br,itacurussa,Itacurussá,05,,-12.233333,-39.216667
br,itacurussa,Itacurussá,21,,-22.925833,-43.912222
br,itacurussu,Itacurussú,21,,-22.925833,-43.912222
br,itacutiara,Itacutiara,05,,-8.966667,-38.516667
br,itacutiu,Itacutiú,16,,-1.416667,-46.166667
br,itacy,Itacy,15,,-21.016667,-46.016667
br,itaete,Itaetê,05,,-12.983333,-40.966667
br,itagiba,Itagibá,05,9843,-14.283333,-39.85
br,itagi,Itagi,05,10534,-14.166667,-40.016667
br,itaguacu,Itaguaçu,05,,-13.6,-41.9
br,itaguacu,Itaguaçu,08,8202,-19.8,-40.85
br,itaguai,Itaguaí,21,100756,-22.870755,-43.773821
br,itagua,Itaguá,06,,-6.95,-40.35
br,itaguaje,Itaguajé,18,,-22.616667,-51.983333
br,itaguan,Itaguan,18,,-22.816667,-52.683333
br,itaguara,Itaguará,13,,-5.183333,-42.916667
br,itaguara,Itaguara,15,,-20.383333,-44.483333
br,itaguaru,Itaguaru,29,,-15.733333,-49.616667
br,itaguassu,Itaguassú,08,,-19.8,-40.85
br,itaguassu,Itaguassú,30,,-8.483333,-36.166667
br,itaguatins,Itaguatins,31,,-5.728056,-47.33
br,itaguira,Itaguira,05,,-15.15,-39.733333
br,itahu,Itahú,22,,-5.833333,-37.983333
br,itahum,Itahum,11,,-22,-55.333333
br,itahy,Itahy,27,,-23.4,-49.1
br,itahype,Itahypé,15,,-17.416667,-41.683333
br,itaiacoca,Itaiacoca,18,,-25.133333,-49.916667
br,itaia,Itaiá,05,,-14.883333,-39.916667
br,itaiapaba,Itaiapaba,06,,-4.183333,-38.583333
br,itaiba,Itaíba,05,,-12.433333,-40.3
br,itaiba,Itaíba,30,13870,-8.95,-37.416667
br,itaibo,Itaibó,05,,-13.966667,-39.733333
br,itaicaba,Itaiçaba,06,,-4.666667,-37.85
br,itaici,Itaici,08,,-20.533333,-41.516667
br,itaici,Itaici,14,,-16.066667,-55.9
br,itai de fora,Itaí de Fora,05,,-12.133333,-43.333333
br,itai,Itaí,05,,-12.15,-43.283333
br,itai,Itaí,21,,-22.716667,-42.183333
br,itai,Itaí,23,,-28.333333,-53.966667
br,itai,Itaí,27,22305,-23.4,-49.1
br,itailubesinho,Itailubesinho,23,,-28.633333,-50.566667
br,itaimbe,Itaimbé,08,,-19.666667,-40.833333
br,itaimbesinho,Itaimbésinho,23,,-28.633333,-50.566667
br,itaimbezinho,Itaimbèzinho,23,,-28.633333,-50.566667
br,itaim,Itaim,15,,-22.4,-45.883333
br,itaina,Itaina,18,,-22.716667,-52.9
br,itainopolis,Itainópolis,20,2922,-7.4,-41.516667
br,itainzinho,Itainzinho,20,,-7.95,-41
br,itaiobaia,Itaiobaia,08,,-20.133333,-40.366667
br,itaio,Itaió,26,,-26.466667,-49.866667
br,itaiopolis,Itaiópolis,26,,-26.333333,-49.933333
br,itaipaba,Itaipaba,06,,-4.183333,-38.616667
br,itaipava,Itaipava,05,,-14.333333,-39.333333
br,itaipava,Itaipava,08,,-20.533333,-41.516667
br,itaipava,Itaipava,13,,-5.1,-45.816667
br,itaipava,Itaipava,16,,-6.766667,-48.966667
br,itaipava,Itaipava,21,,-22.375833,-43.131389
br,itaipavas,Itaipavas,05,,-17.466667,-40.066667
br,itaipe,Itaípe,05,,-14.75,-39.05
br,itaipe,Itaipé,15,,-17.416667,-41.683333
br,itaipuacu,Itaipuaçu,21,,-22.966667,-43.016667
br,itaipu,Itaipu,06,,-6.95,-40.35
br,itaipu,Itaipu,21,,-22.965556,-43.047778
br,itaipu,Itaipu,25,,3.6,-60.833333
br,itaipu,Itaipu,27,,-22.600278,-47.363056
br,ita,Itá,26,,-27.266667,-52.316667
br,itaitinga,Itaitinga,06,30939,-3.966667,-38.516667
br,itaituba,Itaituba,04,,-5.4,-60.75
br,itaituba,Itaituba,16,64758,-4.216667,-56.016667
br,itaitu,Itaitu,05,,-11.333333,-40.4
br,itaiuba,Itaiúba,27,,-20.733333,-49.883333
br,itaiu,Itaiú,29,,-15.966667,-50.05
br,itajahi,Itajahi,26,,-26.883333,-48.65
br,itajahy do sul,Itajahy do Sul,26,,-27.2181,-49.6436
br,itajahy,Itajahy,26,,-26.883333,-48.65
br,itajai,Itajai,05,,-13.766667,-39.15
br,itajai,Itajaí,26,155719,-26.883333,-48.65
br,itaja,Itajá,29,,-19.116667,-51.616667
br,itajara,Itajara,21,,-21.316667,-42.016667
br,itajiba,Itajibá,05,,-14.283333,-39.85
br,itaji,Itaji,05,,-14.166667,-40.016667
br,itajobi,Itajobi,04,,-5.194444,-60.351389
br,itajobi,Itajobi,27,12231,-21.316667,-49.066667
br,itajoby,Itajoby,27,,-21.316667,-49.066667
br,itajuba,Itajubá,15,81871,-22.433333,-45.45
br,itajuba,Itajubá,26,,-26.6998,-48.6826
br,itajuba,Itajubá,29,,-16.466667,-51.116667
br,itajubaquara,Itajubaquara,05,,-11.4,-42.683333
br,itajubatiba,Itajubatiba,17,,-7.016667,-37.666667
br,itajuby,Itajuby,27,,-21.316667,-49.066667
br,itajui,Itajuí,05,,-11.046389,-45.059722
br,itajuipe,Itajuípe,05,15517,-14.683333,-39.366667
br,itaju,Itaju,27,,-22,-48.8
br,itajuru,Itajuru,05,,-14.083333,-39.933333
br,itajutiba,Itajutiba,15,,-19.433333,-41.9
br,itajuva,Itajuva,26,,-26.840133,-49.588065
br,italia,Itália,20,,-7.25,-41.983333
br,italva,Italva,21,,-21.416667,-41.7
br,itamacari,Itamacari,16,,-2.166667,-46.316667
br,itamaraca,Itamaracá,01,,-9.716667,-72.016667
br,itamaraca,Itamaracá,02,,-9.35,-35.866667
br,itamaraca,Itamaracá,30,14969,-7.75,-34.833333
br,itamaraju,Itamaraju,05,47630,-17.066667,-39.533333
br,itamarandiba,Itamarandiba,15,18064,-17.85,-42.85
br,itamarandyba,Itamarandyba,15,,-17.85,-42.85
br,itamarati,Itamarati,04,,-6.4,-68.25
br,itamarati,Itamarati,05,,-15.933333,-39.533333
br,itamarati,Itamarati,15,,-15.616667,-41.4
br,itamarati,Itamarati,16,,-1.133333,-51.183333
br,itamaraty,Itamaraty,15,,-21.416667,-42.816667
br,itamaraty,Itamaraty,20,,-4.416667,-41.466667
br,itamari,Itamari,05,,-13.783333,-39.616667
br,itamatare,Itamataré,13,,-2.266667,-46.4
br,itamauari,Itamauari,16,,-2.166667,-46.316667
br,itambacuri,Itambacuri,15,14652,-18.016667,-41.7
br,itambacury,Itambacury,15,,-18.016667,-41.7
br,itambaraca,Itambaracá,18,,-23.016667,-50.433333
br,itambe do mato dentro,Itambé do Mato Dentro,15,,-19.4,-43.316667
br,itambe,Itambé,05,23559,-15.25,-40.616667
br,itambe,Itambé,15,,-18.5,-43.3
br,itambe,Itambe,17,,-7.333333,-35.333333
br,itambe,Itambé,18,,-23.466667,-51.416667
br,itambe,Itambé,27,,-20.616667,-48.75
br,itambe,Itambé,30,,-7.416667,-35.1
br,itambi,Itambi,21,,-22.733333,-42.966667
br,itamby,Itamby,21,,-22.733333,-42.966667
br,itamembe,Itamembé,15,,-20.366667,-44.683333
br,itamira,Itamira,05,,-11.716667,-38.233333
br,itamirim,Itamirim,05,,-14.983333,-39.933333
br,itamirim,Itamirim,15,,-14.766667,-42.883333
br,itamoabo,Itamoabo,05,,-12.8,-38.533333
br,itamoata,Itamoata,02,,-9.883333,-37.183333
br,itamogi,Itamogi,15,,-21.083333,-47.05
br,itamoji,Itamoji,15,,-21.083333,-47.05
br,itamonte,Itamonte,15,,-22.283333,-44.883333
br,itamorotinga,Itamorotinga,17,,-7.483333,-36.666667
br,itamotinga,Itamotinga,05,,-14.583333,-39.616667
br,itamuri,Itamuri,15,,-20.983333,-42.366667
br,itanage,Itanagé,05,,-13.716667,-42.033333
br,itanagra,Itanagra,05,,-12.266667,-38.05
br,itanaje,Itanajé,05,,-13.716667,-42.033333
br,itanara,Itanarã,11,,-23.783333,-55.45
br,itangua,Itangua,27,,-23.524667,-47.5049
br,itanhaem,Itanhaém,27,90387,-24.183333,-46.783333
br,itanhaen,Itanhaen,27,,-24.183333,-46.783333
br,itanhandu,Itanhandu,15,11274,-22.3,-44.95
br,itanhanga,Itanhangá,21,,-22.99431,-43.307353
br,itanhem,Itanhém,05,13517,-17.15,-40.333333
br,itanhenga,Itanhenga,08,,-20.266667,-40.383333
br,itanhi,Itanhi,05,,-11.533333,-37.566667
br,itanhomi,Itanhomi,15,,-19.166667,-41.866667
br,itanhomy,Itanhomy,15,,-19.166667,-41.866667
br,itanhum,Itanhum,11,,-22,-55.333333
br,itans,Itans,06,,-4.6,-38.866667
br,itans,Itans,20,,-7,-40.65
br,itaobi,Itaobí,31,,-11.014167,-48.505
br,itaobim,Itaobim,15,15938,-16.566667,-41.833333
br,itaoca,Itaoca,08,,-20.733333,-41.116667
br,itaoca,Itaoca,21,,-21.816667,-41.466667
br,itaoca,Itaoca,27,,-24.640643,-48.840119
br,itaocara,Itaocara,21,16763,-21.683333,-42.066667
br,itapaci,Itapaci,29,16118,-14.95,-49.566667
br,itapacura,Itapacurá,16,,-4.383333,-56.1
br,itapage,Itapagé,06,32043,-3.683333,-39.566667
br,itapagipe,Itapagipe,05,,-12.916667,-38.5
br,itapagipe,Itapagipe,15,,-19.9,-49.366667
br,itapai,Itapai,06,,-4.25,-38.716667
br,itapaje,Itapajé,06,,-3.683333,-39.566667
br,itapajipe,Itapajipe,15,,-19.9,-49.366667
br,itapanhoacanga,Itapanhoacanga,15,,-18.783333,-43.45
br,itapaoa,Itapaoa,26,,-28.95,-49.583333
br,itapara,Itapará,18,,-25.466667,-51.05
br,itapare,Itaparé,15,,-16.616667,-42.183333
br,itaparica,Itaparica,05,20101,-12.9,-38.7
br,itaparica,Itaparica,08,,-20.383333,-40.316667
br,itaparica,Itaparica,30,,-9.083333,-38.3
br,itapeacu,Itapeaçú,04,,-2.967778,-58.058056
br,itapeba,Itapeba,21,,-22.816667,-41.983333
br,itapebi,Itapebi,05,,-15.933333,-39.533333
br,itapebucu,Itapebuçu,06,,-4.016667,-38.916667
br,itapebussu,Itapebussu,06,,-4.016667,-38.916667
br,itapecerica da serra,Itapecerica da Serra,27,151608,-23.716667,-46.833333
br,itapecerica,Itapecerica,05,,-13.2,-38.9
br,itapecerica,Itapecerica,15,16684,-20.466667,-45.116667
br,itapecerica,Itapecerica,27,,-23.716667,-46.833333
br,itapechinga,Itapechinga,27,,-22.966667,-46.583333
br,itapecoa,Itapecoá,08,,-20.9,-40.983333
br,itapeco,Itapecô,30,,-8.416667,-35.666667
br,itapecuru-mirim,Itapecuru-Mirim,13,,-3.4,-44.333333
br,itapeim,Itapeim,06,,-4.333333,-38.116667
br,itapeipu,Itapeipu,05,,-11.316667,-40.35
br,itape,Itapé,05,,-14.9,-39.433333
br,itape,Itapé,23,,-30.6,-51.566667
br,itapejara,Itapejara,18,,-25.966667,-52.816667
br,itapema,Itapema,05,,-12.7,-38.75
br,itapema,Itapema,18,,-25.4539,-48.6867
br,itapema,Itapema,26,36766,-27.099167,-48.615
br,itapema,Itapema,27,,-23.229824,-47.754668
br,itapemirim,Itapemirim,08,17536,-21.016667,-40.833333
br,itapemirim,Itapemirim,30,,-7.683333,-39.366667
br,itapera,Itapera,13,,-2.533333,-43.783333
br,itaperapoan,Itaperapoan,29,,-15.866667,-50.6
br,itaperapua,Itaperapuã,29,,-15.866667,-50.6
br,itaperapuan,Itaperapuan,29,,-15.866667,-50.6
br,itaperarama,Itaperarama,08,,-20.733333,-40.75
br,itaperiu,Itaperiú,26,,-26.583333,-48.733333
br,itaperorama,Itaperorama,08,,-20.733333,-40.75
br,itaperucu,Itaperuçu,18,26552,-25.2,-49.366667
br,itaperu,Itaperu,20,,-5.466667,-43.016667
br,itaperuna,Itaperuna,21,84436,-21.2,-41.9
br,itapeteiu,Itapeteiú,21,,-22.9,-42.783333
br,itapeti do salto,Itapeti do Salto,27,,-23.405707,-46.115047
br,itapeti,Itapeti,27,,-23.389269,-46.168376
br,itapetim,Itapetim,30,,-7.366667,-37.183333
br,itapetinga,Itapetinga,05,59723,-15.25,-40.25
br,itapetinga,Itapetinga,27,,-23.204444,-46.546389
br,itapetingui,Itapetingui,05,,-12.45,-38.733333
br,itapetininga,Itapetininga,27,126245,-23.6,-48.05
br,itapetininga,Itapetininga,30,,-7.366667,-37.183333
br,itapeua,Itapeua,04,,-3.966667,-57.283333
br,itapeua,Itapeua,13,,-2.883333,-44.683333
br,itapeua,Itapeua,16,,-2.5,-48.666667
br,itapeuna,Itapeúna,13,,-2.883333,-44.683333
br,itapeuna,Itapeúna,27,,-24.596435,-48.217785
br,itapeva,Itapevá,04,,-3.966667,-57.283333
br,itapeva,Itapeva,15,,-18.3,-43.433333
br,itapeva,Itapeva,23,,-29.5,-50.1
br,itapeva,Itapeva,27,62958,-23.966667,-48.866667
br,itapevi,Itapevi,27,201752,-23.55,-46.933333
br,itapevy,Itapevy,27,,-23.55,-46.933333
br,itapiche,Itapiché,15,,-21.166667,-45.933333
br,itapiciru,Itapiciru,05,,-13.616667,-40.85
br,itapicuma,Itapiçuma,30,,-7.766667,-34.9
br,itapicuri,Itapicuri,05,,-12.1,-39.116667
br,itapicuru de baixo,Itapicuru de Baixo,05,,-11.816667,-37.616667
br,itapicuru,Itapicuru,02,,-9.533333,-36.783333
br,itapicuru,Itapicuru,05,4994,-11.316667,-38.25
br,itapicuru,Itapicuru,13,,-3.4,-44.333333
br,itapicuru,Itapicurú,16,,-1.716667,-48.383333
br,itapicuru-mirim,Itapicurú-Mirim,13,,-3.4,-44.333333
br,itapina,Itapina,08,,-19.533333,-40.833333
br,itapinima,Itapinima,04,,-5.410556,-60.727222
br,itapinimba,Itapinimba,04,,-5.410556,-60.727222
br,itapipoca,Itapipoca,06,55785,-3.5,-39.583333
br,itapira,Itapira,05,,-14.3,-39.333333
br,itapira,Itapira,27,61608,-22.433333,-46.833333
br,itapira,Itapira,30,,-7.65,-35.566667
br,itapirama,Itapirama,24,,-11.033333,-62.116667
br,itapiranga,Itapiranga,04,,-2.666667,-58.023889
br,itapiranga,Itapiranga,13,,-2.2,-44.633333
br,itapiranga,Itapiranga,26,,-27.133333,-53.716667
br,itapirapua,Itapirapuã,27,,-24.583333,-49.166667
br,itapirapua,Itapirapuã,29,7923,-15.866667,-50.6
br,itapirapuan,Itapirapuan,29,,-15.866667,-50.6
br,itapirarema,Itapirarema,06,,-3.766667,-40.9
br,itapiratins,Itapiratins,31,,-8.321111,-47.955278
br,itapirucu,Itapiruçu,15,,-21.466667,-42.366667
br,itapiru,Itapiru,05,,-11.483333,-39.016667
br,itapiru,Itapiru,15,,-16.5,-40.416667
br,itapiruna,Itapiruna,15,,-19.166667,-42.083333
br,itapirussu,Itapirussu,15,,-21.466667,-42.366667
br,itapissuma,Itapissuma,30,16674,-7.766667,-34.9
br,itapitacai,Itapitacai,23,,-29.883333,-57.133333
br,itapitanga,Itapitanga,05,,-13.116667,-38.883333
br,itapitanga,Itapitanga,08,,-20.85,-40.75
br,itapitocahy,Itapitocahy,23,,-29.883333,-57.133333
br,itapitocai,Itapitocaí,23,,-29.883333,-57.133333
br,itapiuna,Itapiúna,06,,-4.55,-38.95
br,itapival,Itapival,13,,-5.1,-45.816667
br,itapixe,Itapixé,15,,-21.166667,-45.933333
br,itapixuna,Itapixuna,16,,-1.166667,-46.333333
br,itapoa,Itapoã,05,,-12.95,-38.366667
br,itapoa,Itapoã,15,,-16.333333,-40.566667
br,itapoa,Itapoã,16,,-.75,-48.108333
br,itapoa,Itapoã,23,,-30.266667,-51.016667
br,itapoama,Itapoama,08,,-20.866667,-40.933333
br,itapoan,Itapoan,05,,-12.95,-38.366667
br,itapoca,Itapoca,08,,-20.316667,-40.416667
br,itapocu,Itapocu,08,,-20.066667,-40.433333
br,itapocu,Itapocu,26,,-26.5481,-48.7138
br,itapo,Itapó,06,,-4.033333,-38.4
br,itapolis,Itápolis,27,34553,-21.583333,-48.766667
br,itapopo,Itapopó,11,,-23.783333,-54.316667
br,itapora do tocantins,Itaporã do Tocantins,31,,-8.529722,-48.572222
br,itapora,Itaporã,05,,-12.616667,-39.166667
br,itapora,Itaporã,11,12327,-22.016667,-54.9
br,itapora,Itaporã,31,,-8.529722,-48.572222
br,itaporanga d'ajuda,Itaporanga d'Ajuda,28,,-10.983333,-37.3
br,itaporanga,Itaporanga,02,,-10.233333,-36.566667
br,itaporanga,Itaporanga,04,,-2.666667,-58.023889
br,itaporanga,Itaporanga,15,,-20.483333,-42.483333
br,itaporanga,Itaporanga,17,15890,-7.3,-38.166667
br,itaporanga,Itaporanga,21,,-22.033333,-41.483333
br,itaporanga,Itaporanga,27,10202,-23.7,-49.483333
br,itaporanga,Itaporanga,28,,-10.983333,-37.3
br,itapore,Itaporé,15,,-16.616667,-42.183333
br,itapore,Itaporé,31,,-10.579722,-49.189167
br,itapororoca,Itapororoca,17,,-6.85,-35.25
br,itapororocas,Itapororocas,05,,-12.133333,-39.033333
br,itapua,Itapuã,04,,-3.4,-66.05
br,itapua,Itapuã,05,,-12.95,-38.366667
br,itapua,Itapuã,18,,-22.816667,-52.683333
br,itapua,Itapua,21,,-22.066667,-41.933333
br,itapua,Itapuã,23,,-30.266667,-51.016667
br,itapuan,Itapuan,18,,-22.816667,-52.683333
br,itapuan,Itapuan,23,,-30.266667,-51.016667
br,itapuca,Itapuca,21,,-23.037545,-43.557548
br,itapuca,Itapuca,23,,-28.9,-52.05
br,itapucu,Itapucu,08,,-20.066667,-40.433333
br,itapui,Itapuí,05,,-15.116667,-40.1
br,itapui,Itapuí,26,,-27.116667,-51.4
br,itapui,Itapuí,27,10445,-22.233333,-48.683333
br,itapu,Itapu,05,,-13.4,-38.983333
br,itapura,Itapura,05,,-11.55,-40.45
br,itapura,Itapura,27,,-20.666667,-51.516667
br,itapuranga,Itapuranga,04,,-5.2525,-60.220278
br,itapuranga,Itapuranga,29,19709,-15.583333,-49.983333
br,itapuvucu,Itapuvuçu,27,,-24.700879,-48.04082
br,itaquadiba,Itaquadiba,08,,-19.983333,-40.333333
br,itaquahy,Itaquahy,21,,-22.870755,-43.773821
br,itaqua,Itaquá,26,,-27.3,-49.45
br,itaquandiba,Itaquandiba,08,,-19.983333,-40.333333
br,itaquaquecetuba,Itaquaquecetuba,27,336686,-23.483333,-46.35
br,itaquaquicetuba,Itaquaquicetuba,27,,-23.483333,-46.35
br,itaquarai,Itaquaraí,05,,-14.05,-41.733333
br,itaquara,Itaquara,05,,-13.45,-39.95
br,itaquarari,Itaquarari,05,,-14.05,-41.733333
br,itaquari,Itaquari,05,,-12.85,-39.533333
br,itaquari,Itaquari,08,,-20.333333,-40.366667
br,itaquatiara,Itaquatiara,04,,-3.05,-58.433333
br,itaquatiara,Itaquatiara,05,,-11.35,-43.216667
br,itaquatiara,Itaquatiara,21,,-22.969722,-43.031667
br,itaquatiara,Itaquatiara,30,,-8.916667,-36.65
br,itaquena,Itaquena,05,,-16.633333,-39.083333
br,itaquera,Itaquera,27,,-23.533333,-46.45
br,itaquere,Itaquere,27,,-21.666667,-48.25
br,itaqueri da serra,Itaqueri da Serra,27,,-22.35,-47.9
br,itaqueri velho,Itaqueri Velho,27,,-22.3,-47.85
br,itaquery da serra,Itaquery da Serra,27,,-22.35,-47.9
br,itaqui,Itaqui,04,,-7.428333,-63.011389
br,itaqui,Itaqui,13,,-2.6,-44.35
br,itaqui,Itaqui,18,,-25.3177,-48.5207
br,itaqui,Itaqui,23,34631,-29.133333,-56.55
br,itaquira,Itaquira,21,,-22.15,-41.65
br,itaquitinga,Itaquitinga,30,11128,-7.666667,-35.1
br,itaquy,Itaquy,04,,-7.428333,-63.011389
br,itaquy,Itaquy,23,,-29.133333,-56.55
br,itaracema,Itaracema,06,,-3.766667,-40.9
br,itaracu,Itaraçu,27,,-23.480849,-47.755322
br,itarana,Itarana,08,4060,-19.85,-40.866667
br,itarantim,Itarantim,05,12370,-15.65,-40.05
br,itarare,Itararé,13,,-9.2,-46.166667
br,itarare,Itararé,26,,-27.733333,-50.55
br,itarare,Itararé,27,44810,-24.116667,-49.333333
br,itarema,Itarema,06,10330,-2.9,-39.966667
br,itaretama,Itaretama,22,,-5.698889,-36.245278
br,itariri,Itariri,27,,-24.3,-47.166667
br,itariry,Itariry,27,,-24.3,-47.166667
br,itaroquem,Itaroquém,23,,-28.483333,-55.35
br,itaruma,Itarumã,29,,-18.7,-51.416667
br,itas,Itãs,06,,-4.616667,-38.883333
br,itassuce,Itassucê,05,,-14.233333,-44.3
br,itatapua,Itatapua,16,,-.616667,-51.2
br,itatiaia,Itatiaia,15,,-19.333333,-41.433333
br,itatiaia,Itatiaia,21,12571,-22.5,-44.566667
br,itatiaiucu,Itatiaiuçu,15,,-20.2,-44.416667
br,itatiaiussu,Itatiaiussu,15,,-20.2,-44.416667
br,itatiayassu,Itatiayassú,15,,-20.2,-44.416667
br,itatiba,Itatiba,23,,-27.366667,-52.45
br,itatiba,Itatiba,27,67935,-23,-46.85
br,itaticui,Itaticuí,23,,-30.3,-52.166667
br,itati,Itati,05,,-14.316667,-39.283333
br,itati,Itati,23,,-29.5,-50.1
br,itatinga,Itatinga,05,,-15.25,-40.25
br,itatinga,Itatinga,27,15696,-23.116667,-48.6
br,itatingui,Itatingui,05,,-15.2,-39.433333
br,itatira,Itatira,06,,-4.5,-39.616667
br,itatitu,Itatitu,27,,-23.566,-47.7758
br,itatuba,Itatuba,04,,-5.683333,-63.410556
br,itatuba,Itatuba,17,4951,-7.383333,-35.65
br,itatupa,Itatupã,16,,-.616667,-51.2
br,itauba,Itaúba,04,,-4.009444,-59.809722
br,itauba,Itaúba,20,,-7.1,-41.983333
br,itauba,Itaúba,23,,-29.2,-53.183333
br,itaubal,Itaúbal,03,,1.683333,-51
br,itaucu,Itauçu,29,,-16.216667,-49.616667
br,itau de minas,Itaú de Minas,15,14888,-20.716667,-46.75
br,itaueira,Itaueira,20,5146,-7.6,-43.033333
br,itau,Itaú,22,,-5.833333,-37.983333
br,itaum,Itaum,11,,-22,-55.333333
br,itauna do sul,Itaúna do Sul,18,,-22.666667,-52.816667
br,itauna,Itaúna,04,,-2.983333,-65.983333
br,itauna,Itaúna,05,,-13.05,-40.066667
br,itauna,Itauna,06,,-4.55,-38.95
br,itauna,Itauna,13,,-6.2,-43.266667
br,itauna,Itaúna,15,77402,-20.066667,-44.566667
br,itauna,Itaúna,18,,-22.666667,-52.816667
br,itauna,Itaúna,20,,-6.083333,-41.85
br,itauna,Itaúna,21,,-22.775556,-43.031944
br,itauna,Itaúna,27,,-24.596435,-48.217785
br,itauna,Itaúna,30,,-8.166667,-36.133333
br,itaunas,Itaúnas,08,,-18.416667,-39.7
br,itauninha,Itauninha,15,,-19.7,-43.033333
br,itauninhas,Itauninhas,08,,-18.5,-40.083333
br,itavera,Itaverá,21,,-22.716667,-44.15
br,itaverava,Itaverava,15,,-20.666667,-43.6
br,itaxama,Itaxama,05,,-13.033333,-39.716667
br,itayopolis,Itayopolis,26,,-26.333333,-49.933333
br,iterere,Itererê,13,,-1.566667,-45.1
br,itibira,Itibira,15,,-19.45,-45.133333
br,itinga da serra,Itinga da Serra,05,,-10.566667,-40.283333
br,itinga ii,Itinga II,26,,-26.652,-48.7377
br,itinga,Itinga,05,,-10.566667,-40.283333
br,itinga,Itinga,15,4917,-16.6,-41.783333
br,itinga,Itinga,18,,-25.427678,-48.51946
br,itinga,Itinga,27,,-23.5673,-47.5726
br,itiquira,Itiquira,14,,-17.083333,-54.933333
br,itira,Itira,15,,-16.766667,-42.033333
br,itirapina,Itirapina,27,13229,-22.25,-47.816667
br,itirapua,Itirapuã,27,,-20.633333,-47.233333
br,itirapuan,Itirapuan,27,,-20.633333,-47.233333
br,itirucu,Itiruçu,05,11499,-13.516667,-40.15
br,itirussu,Itirussu,05,,-13.516667,-40.15
br,itiuba,Itiúba,05,9410,-10.716667,-39.85
br,itiuca,Itiúca,05,,-13.583333,-39
br,itobi,Itobi,27,,-21.733333,-46.966667
br,itoby,Itoby,27,,-21.733333,-46.966667
br,itonia,Itônia,18,,-23.933333,-53.9
br,itororo do paranapanema,Itororó do Paranapanema,27,,-22.6,-51.716667
br,itororo,Itororó,05,17325,-15.116667,-40.1
br,itororo,Itororó,26,,-27.216667,-51.533333
br,itoupava alto,Itoupava Alto,26,,-26.7623,-49.0551
br,itoupava central,Itoupava Central,26,,-26.8191,-49.0797
br,itoupava,Itoupava,26,,-26.8191,-49.0797
br,itoupava rega,Itoupava Rega,26,,-26.75,-49.083333
br,itoupava regia,Itoupava Régia,26,,-26.75,-49.083333
br,itoupavazinha,Itoupavazinha,26,,-26.851,-49.1216
br,ituacu,Ituaçu,05,4891,-13.816667,-41.3
br,ituassu,Ituassú,05,,-13.816667,-41.3
br,itubera,Ituberá,05,19888,-13.733333,-39.15
br,itucuma,Itucumã,04,,-6.983333,-69.833333
br,itueta,Itueta,15,,-19.383333,-41.183333
br,ituguacu,Ituguaçu,30,,-8.483333,-36.166667
br,ituguassu,Ituguassú,30,,-8.483333,-36.166667
br,itui,Ituí,15,,-21.533333,-42.916667
br,ituim,Ituim,23,,-28.55,-51.35
br,itu,Itu,27,137589,-23.266667,-47.316667
br,ituiutaba,Ituiutaba,15,85346,-18.966667,-49.466667
br,itumbiara,Itumbiara,29,79583,-18.416667,-49.216667
br,itumirim,Itumirim,05,,-10.2,-40.216667
br,itumirim,Itumirim,15,,-21.316667,-44.883333
br,itupava,Itupava,18,,-25,-49.35
br,itupava,Itupava,26,,-26.8191,-49.0797
br,itupeva,Itupeva,05,25183,-17.416667,-40.483333
br,itupeva,Itupeva,15,,-18.3,-43.433333
br,itupeva,Itupeva,27,,-23.15,-47.066667
br,itupiranga,Itupiranga,16,21301,-5.15,-49.333333
br,ituporanga,Ituporanga,26,13038,-27.416667,-49.6
br,ituquara,Ituquara,16,,-1.016667,-51.05
br,ituqui,Ituqui,04,,-7.428333,-63.011389
br,iturama,Iturama,15,29805,-19.733333,-50.183333
br,itutinga,Itutinga,15,,-21.3,-44.666667
br,ituverava,Ituverava,27,36998,-20.333333,-47.783333
br,ituxi,Ituxi,04,,-6.962778,-71.282222
br,ituyutaba,Ituyutaba,15,,-18.966667,-49.466667
br,ityrapina,Ityrapina,27,,-22.25,-47.816667
br,iuarete,Iuaretê,04,,.633333,-69.2
br,iubatinga,Iubatinga,27,,-21.916667,-51.2
br,iucai,Iucaí,04,,-.216667,-66.816667
br,iuira,Iuira,04,,1.05,-67.533333
br,iuitepora,Iuiteporã,30,,-8.516667,-35.8
br,iuiu,Iuiú,05,,-14.4,-43.583333
br,ium,Ium,11,,-21.583333,-56.716667
br,iuna,Iúna,08,14290,-20.35,-41.533333
br,iupeba,Iupeba,27,,-23.683333,-46.366667
br,iu-rapecuma,Iú-Rapecuma,04,,.1,-67.716667
br,iuru,Iuru,08,,-21.15,-41.6
br,iuru,Iurú,23,,-28.8,-51.033333
br,ius,Ius,20,,-3.881667,-42.008056
br,iutica,Iutica,04,,1.078889,-69.476389
br,ivagaci,Ivagaci,23,,-27.633333,-54.166667
br,ivahy,Ivahy,18,,-25.016667,-50.866667
br,ivai,Ivaí,11,,-17.333333,-56.633333
br,ivai,Ivaí,18,,-25.016667,-50.866667
br,ivai,Ivaí,23,,-29.016667,-53.8
br,ivaipora,Ivaiporã,18,23316,-24.25,-51.75
br,ivaitinga,Ivaítinga,18,,-23.166667,-52.416667
br,ivanheima,Ivanheima,11,,-22.166667,-53.616667
br,ivape,Ivapé,29,,-17.333333,-53.2
br,ivaporunduvinha,Ivaporunduvinha,27,,-24.525056,-48.300158
br,ivatuba,Ivatuba,18,,-23.616667,-52.216667
br,ivatuva,Ivatuva,18,,-23.616667,-52.216667
br,iviti,Iviti,31,,-7.802222,-47.798611
br,ivo,Ivo,14,,-14.983333,-56.366667
br,ivolandia,Ivolândia,29,,-16.566667,-50.85
br,ivora,Ivorá,23,,-29.533333,-53.566667
br,ivoti,Ivoti,23,18172,-29.6,-51.166667
br,ixu,Ixu,05,,-11.766667,-39.2
br,ixu,Ixu,14,,-16.8,-56.533333
br,ixuzinho,Ixuzinho,14,,-16.9,-56.616667
br,izidoria,Izidoria,20,,-6.533333,-41.833333
br,izidoro,Izidoro,06,,-6.133333,-39.25
br,izidro alves,Izidro Alves,11,,-20.15,-55.2
br,jabaete,Jabaeté,08,,-20.383333,-40.5
br,jabaquara,Jabaquara,08,,-20.7,-40.666667
br,jabaquara,Jabaquara,27,,-23.03696,-47.751127
br,jabitaca,Jabitacá,30,,-7.833333,-37.383333
br,jaboatao,Jaboatão,28,,-10.333333,-36.8
br,jaboatao,Jaboatão,30,702632,-8.116667,-35.016667
br,jabora,Jaborá,26,,-27.183333,-51.733333
br,jaborandi,Jaborandi,18,,-23.016667,-50.433333
br,jaborandi,Jaborandi,27,,-20.666667,-48.416667
br,jaborandy,Jaborandy,27,,-20.666667,-48.416667
br,jabota ferrado,Jabotá Ferrado,13,,-6.516667,-42.966667
br,jabota,Jabotá,05,,-12.65,-41.8
br,jabota,Jabotá,11,,-17.516667,-54.05
br,jaboticaba,Jaboticaba,05,,-12.7,-41.283333
br,jaboticaba,Jaboticaba,15,,-16.2,-41.666667
br,jaboticaba,Jaboticaba,30,,-8.35,-36.15
br,jaboticabal,Jaboticabal,27,69396,-21.266667,-48.316667
br,jaboticabas,Jaboticabas,15,,-19.5,-43.75
br,jaboticabeira,Jaboticabeira,26,,-28.566667,-48.983333
br,jaboticatubas,Jaboticatubas,15,,-19.5,-43.75
br,jaboti,Jaboti,04,,-4.133333,-58.633333
br,jaboti,Jaboti,05,,-12.45,-40.583333
br,jaboti,Jaboti,08,,-20.65,-40.566667
br,jaboti,Jaboti,11,,-20.8,-56.383333
br,jaboti,Jaboti,13,,-3.227778,-42.584444
br,jaboti,Jaboti,18,,-23.742125,-50.073954
br,jaboti,Jaboti,20,,-2.972222,-41.62
br,jaboty,Jaboty,04,,-4.133333,-58.633333
br,jaboty,Jaboty,18,,-23.766667,-50.066667
br,jaburu,Jaburu,01,,-8,-72.75
br,jaburu,Jaburu,02,,-9.066667,-37.583333
br,jaburu,Jaburu,04,,-5.565833,-64.190833
br,jaburu,Jaburu,05,,-12.966667,-38.6
br,jaburu,Jaburú,06,,-3.383333,-40.2
br,jaburu,Jaburu,13,,-5.55,-46.2
br,jaburu,Jaburu,14,,-16.137222,-56.991944
br,jaburu,Jaburu,16,,-.966667,-47.2
br,jaburu,Jaburu,17,,-8.15,-36.933333
br,jaburu,Jaburu,20,,-3.398056,-42.130278
br,jaburuna,Jaburuna,06,,-3.866667,-40.983333
br,jaburuzinho,Jaburuzinho,16,,-1.4,-50.666667
br,jabuticaba,Jabuticaba,23,,-27.616667,-53.266667
br,jabuticabal,Jabuticabal,27,,-21.266667,-48.316667
br,jabuticabeira,Jabuticabeira,21,,-22.486667,-43.713611
br,jabuticabeira,Jabuticabeira,26,,-28.566667,-48.983333
br,jabuti,Jabuti,04,,-3.088889,-61.706111
br,jabuti,Jabuti,05,,-12.45,-40.583333
br,jabuti,Jabuti,08,,-20.65,-40.566667
br,jabuti,Jabuti,11,,-20.8,-56.383333
br,jabuti,Jabuti,13,,-3.030556,-42.114167
br,jabuti,Jabuti,18,,-23.766667,-50.066667
br,jabuti,Jabuti,20,,-2.972222,-41.62
br,jacama,Jacama,13,,-4.533333,-43.816667
br,jacampari,Jacampari,06,,-4.733333,-39.933333
br,jacana,Jaçanã,22,,-6.4,-36.233333
br,jacana velho,Jaçanã Velho,04,,-5.621389,-63.998611
br,jacaqua,Jacaquá,23,,-29.683333,-55.2
br,jacaracanga,Jacaracanga,05,,-12.733333,-38.483333
br,jacaraci,Jacaraci,05,,-14.85,-42.433333
br,jacaracy,Jacaracy,05,,-14.85,-42.433333
br,jacaraipe,Jacaraípe,08,,-20.15,-40.183333
br,jacaranda,Jacarandá,05,,-12.883333,-39.15
br,jacaranda,Jacarandá,06,,-2.989722,-40.9125
br,jacaranda,Jacarandá,08,,-21.05,-40.833333
br,jacaranda,Jacarandá,13,,-3.606667,-43.080278
br,jacaranda,Jacarandá,15,,-20.133333,-44.683333
br,jacarandira,Jacarandira,15,,-20.766667,-44.416667
br,jacarara,Jacarará,17,,-8.066667,-36.866667
br,jacarau,Jacaraú,17,7035,-6.666667,-35.3
br,jacareacanga,Jacareacanga,16,1402,-6.266667,-57.65
br,jacare catinga,Jacaré Catinga,27,,-20.883333,-50.833333
br,jacarecica,Jacarecica,02,,-9.616667,-35.7
br,jacareci,Jacareci,05,,-15.3,-39.583333
br,jacarecoara,Jacarécoara,06,,-4.116667,-38.2
br,jacare dos homens,Jacaré dos Homens,02,,-9.65,-37.2
br,jacare grande,Jacaré Grande,05,,-11.75,-44.033333
br,jacare grande,Jacaré Grande,15,,-15.433333,-43.3
br,jacarehy,Jacarehy,21,,-23.033333,-44.15
br,jacarehy,Jacarehy,27,,-23.316667,-45.966667
br,jacarehype,Jacarehype,08,,-20.15,-40.183333
br,jacarei,Jacareí,11,,-23.816667,-54.583333
br,jacarei,Jacareí,21,,-23.033333,-44.15
br,jacarei,Jacareí,27,213114,-23.316667,-45.966667
br,jacareipe,Jacareipe,08,,-20.15,-40.183333
br,jacare,Jacaré,02,,-9.316667,-37.583333
br,jacare,Jacaré,04,,-2.4,-66.116667
br,jacare,Jacaré,05,,-10.8,-41.65
br,jacare,Jacaré,08,,-18.883333,-40.083333
br,jacare,Jacaré,13,,-2.766667,-44.7
br,jacare,Jacaré,14,,-15.266667,-57.7
br,jacare,Jacare,15,,-15.083333,-44.116667
br,jacare,Jacaré,16,,-1.283333,-49.933333
br,jacare,Jacaré,20,,-3.617222,-42.072222
br,jacare,Jacaré,21,,-22.891724,-43.25476
br,jacare,Jacaré,22,,-6.35,-37.233333
br,jacare,Jacaré,23,,-29.216667,-51.883333
br,jacare,Jacare,24,,-10.216667,-64.183333
br,jacare,Jacaré,27,,-23.11907,-47.915394
br,jacare,Jacaré,30,,-8.083333,-40.216667
br,jacare mirim,Jacaré Mirim,22,,-5.783333,-35.35
br,jacarepagua,Jacarepaguá,21,,-22.959706,-43.369083
br,jacarequara,Jacarequará,06,,-4.116667,-38.2
br,jacarequara,Jacarèquara,13,,-2.016667,-44.583333
br,jacarequara,Jacarèquara,16,,-1.533333,-47.05
br,jacaresinho,Jacaresinho,18,,-23.171382,-49.991453
br,jacaretinga,Jacaretinga,04,,-5.400278,-60.425556
br,jacare velho,Jacaré Velho,15,,-21.333333,-42.85
br,jacarezinho,Jacarezinho,04,,-4.456111,-59.838056
br,jacarezinho,Jacarèzinho,16,,-5.983333,-48.333333
br,jacarezinho,Jacarezinho,18,35090,-23.171382,-49.991453
br,jacarezinho,Jacarèzinho,23,,-29.216667,-51.883333
br,jacarini,Jacarini,15,,-21.933333,-45.933333
br,jacata,Jacatã,16,,-6.466667,-54.05
br,jacauna,Jacaúna,06,,-3.933333,-38.266667
br,jaceaba,Jaceaba,15,,-20.533333,-43.983333
br,jaceguai,Jaceguai,06,,-3.133333,-40.466667
br,jaceruba,Jaceruba,21,,-22.590833,-43.551667
br,jacetuba,Jacetuba,21,,-22.590833,-43.551667
br,jaciaba,Jaciaba,18,,-24.933333,-51.2
br,jaciara,Jaciara,14,22739,-15.983333,-54.95
br,jacigua,Jaciguá,08,,-20.7,-41.016667
br,jaci,Jaci,27,,-20.883333,-49.55
br,jacinto,Jacinto,15,,-16.166667,-40.283333
br,jacinto,Jacinto,18,,-23.4865,-49.7532
br,jacinto machado,Jacinto Machado,26,,-29,-49.766667
br,jacioba,Jacioba,02,,-9.766667,-36.9
br,jaciparana,Jaciparaná,24,,-9.25,-64.383333
br,jacipora,Jaciporã,27,,-21.6,-51.583333
br,jacirema,Jacirema,27,,-20.2,-47.95
br,jacirendi,Jacirendi,27,,-21.75,-47.333333
br,jacitara,Jacitara,04,,-2.016667,-65.15
br,jacobina,Jacobina,02,,-9.9,-37.116667
br,jacobina,Jacobina,05,47639,-11.183333,-40.516667
br,jacobina,Jacobina,14,,-16.25,-57.566667
br,jacobina,Jacobina,15,,-15.916667,-40.1
br,jacobina,Jacobina,20,,-7.933333,-41.2
br,jacobina,Jacobina,22,,-5.816667,-35.366667
br,jacoca de beixo,Jacoca de Beixo,22,,-5.666667,-35.433333
br,jacoca,Jacoca,17,,-7.266667,-34.9
br,jacohy,Jacohy,23,,-29.916667,-53.133333
br,jacone,Jaconé,21,,-22.916667,-42.633333
br,jacri,Jacrí,27,,-21.85,-50.65
br,jacruaru,Jacruaru,04,,-.383333,-64.5
br,jacuarana,Jacuarana,04,,-4.691111,-59.431111
br,jacuba,Jacuba,15,,-20.266667,-44.6
br,jacuba,Jacuba,27,,-22.05,-49.016667
br,jacucuara,Jacucuara,16,,-5.566667,-56.883333
br,jacuda,Jacuda,31,,-7.116667,-48.2
br,jacuecanga,Jacuecanga,21,,-22.9967,-44.245
br,jacueira,Jacueira,30,,-8.733333,-35.816667
br,jacuhy,Jacuhy,15,,-21.033333,-46.75
br,jacuhype,Jacuhype,02,,-8.85,-35.466667
br,jacu iii,Jacu III,13,,-3.622778,-43.148611
br,jacu ii,Jacu II,13,,-3.632778,-43.1525
br,jacui,Jacuí,08,,-20.233333,-40.316667
br,jacu i,Jacu I,13,,-3.648333,-43.151667
br,jacui,Jacuí,15,,-21.033333,-46.75
br,jacui,Jacuí,23,,-29.916667,-53.133333
br,jacuipe,Jacuípe,01,,-9.2,-71.55
br,jacuipe,Jacuípe,02,,-8.85,-35.466667
br,jacuipe,Jacuípe,05,,-12.483333,-38.45
br,jacuisinho,Jacuisinho,23,,-29.033333,-53.066667
br,jacuitiba,Jacuitiba,27,,-23.927482,-47.069543
br,jacuitinga,Jacuitinga,02,,-8.85,-35.466667
br,jacu iv,Jacu IV,13,,-3.600833,-43.147778
br,jacuizinho,Jacuìzinho,23,,-29.033333,-53.066667
br,jacu,Jacu,05,,-11.9,-37.683333
br,jacu,Jacú,06,,-3.933333,-39.4
br,jacu,Jacu,08,,-20.433333,-40.333333
br,jacu,Jacu,13,,-3.914722,-43.098889
br,jacu,Jacu,15,,-16.113611,-44.586667
br,jacu,Jacu,17,,-6.616667,-36.15
br,jacu,Jacu,20,,-3.7475,-42.376667
br,jacu,Jacu,22,,-6.066667,-38.033333
br,jacu,Jacu,27,,-23.316667,-48.533333
br,jacu,Jacu,30,,-7.75,-39.833333
br,jacu,Jacu,31,,-11.75,-47.2
br,jacuma,Jacumã,17,,-7.266667,-34.8
br,jacuma,Jacumã,22,,-5.583333,-35.233333
br,jacuman,Jacuman,17,,-7.266667,-34.8
br,jacunda,Jacundá,14,,-15.666667,-56.65
br,jacunda,Jacundá,16,,-4.55,-49.466667
br,jacunda,Jacundá,24,,-9.1,-62.816667
br,jacundazinho,Jacundàzinho,16,,-4.45,-49.5
br,jacupema,Jacupema,27,,-24.666463,-48.34902
br,jacupemba,Jacupemba,08,,-19.583333,-40.2
br,jacupemba,Jacupemba,26,,-26.988026,-49.892625
br,jacupiranga,Jacupiranga,27,11149,-24.700034,-48.004445
br,jacupiranguinha,Jacupiranguinha,27,,-24.7788,-48.1942
br,jacuquara,Jacuquara,16,,-5.566667,-56.883333
br,jacurici,Jacurici,05,,-10.85,-39.716667
br,jacuricy,Jacuricy,05,,-10.85,-39.716667
br,jacuri,Jacuri,22,,-5.218611,-37.495278
br,jacuripe,Jacuripe,01,,-9.2,-71.55
br,jacuruna,Jacuruna,05,,-13.066667,-38.85
br,jacurutu,Jacurutu,05,,-12.216667,-42.95
br,jacurutu,Jacurutu,30,,-8.583333,-38.716667
br,jacus,Jacus,20,,-6.95,-41.816667
br,jacutinga dos binos,Jacutinga dos Binos,18,,-23.2221,-50.1086
br,jacutinga,Jacutinga,05,,-17.983333,-39.6
br,jacutinga,Jacutinga,11,,-20.333333,-56.566667
br,jacutinga,Jacutinga,14,,-16.166667,-58.166667
br,jacutinga,Jacutinga,15,18470,-22.283333,-46.616667
br,jacutinga,Jacutinga,18,,-25.066667,-51.95
br,jacutinga,Jacutinga,23,,-27.733333,-52.5
br,jacutinga,Jacutinga,26,,-26.3103,-48.6642
br,jacu vii,Jacu VII,13,,-3.049444,-43.245556
br,jacu vi,Jacu VI,13,,-3.573056,-43.148333
br,jacu v,Jacu V,13,,-3.588611,-43.144444
br,jacy parana,Jacy Paraná,24,,-9.25,-64.383333
br,jader pinto,Jáder Pinto,21,,-21.166667,-41.583333
br,jadibaru,Jadibaru,04,,-6.45,-64.25
br,jaeteua,Jaeteua,04,,-3.25,-60.816667
br,jaguapita,Jaguapitã,18,,-23.116667,-51.55
br,jaguaquara,Jaguaquara,05,41499,-13.533333,-39.966667
br,jaguaraci,Jaguaraci,05,,-10.9,-41.566667
br,jaguaracu,Jaguaraçu,15,,-19.666667,-42.75
br,jaguara,Jaguara,05,,-12.116667,-39.1
br,jaguara,Jaguará,15,,-19.733333,-44.666667
br,jaguarandi,Jaguarandi,27,,-21.85,-47.7
br,jaguarao,Jaguarão,06,,-4.466667,-38.75
br,jaguarao,Jaguarão,15,,-16.366667,-40.233333
br,jaguarao,Jaguarão,23,29613,-32.566667,-53.383333
br,jaguarari,Jaguarari,05,22617,-10.266667,-40.2
br,jaguarary,Jaguarary,05,,-10.266667,-40.2
br,jaguarasinho,Jaguarasinho,14,,-14.866667,-55.916667
br,jaguaras,Jaguaras,15,,-19.733333,-44.666667
br,jaguarembe,Jaguarembé,21,,-21.783333,-42.033333
br,jaguaretama,Jaguaretama,06,,-5.616667,-38.766667
br,jaguarete,Jaguaretê,11,,-22.75,-55.45
br,jaguariahyva,Jaguariahyva,18,,-24.25,-49.7
br,jaguariaiva,Jaguariaíva,18,30627,-24.25,-49.7
br,jaguaribara,Jaguaribara,06,,-5.666667,-38.616667
br,jaguaribe,Jaguaribe,05,,-13.116667,-38.9
br,jaguaribe,Jaguaribe,06,23066,-5.883333,-38.616667
br,jaguaribe,Jaguaribe,15,,-15.633333,-42.35
br,jaguaribe,Jaguaribe,17,,-7.133333,-34.883333
br,jaguaribe,Jaguaribe,30,,-7.733333,-34.816667
br,jaguaribe mirim,Jaguaribe Mirim,06,,-5.883333,-38.616667
br,jaguaricatu,Jaguaricatu,18,,-24.5,-49.516667
br,jaguari,Jaguari,23,,-29.5,-54.683333
br,jaguari,Jaguari,27,,-22.669444,-47.296667
br,jaguarinho,Jaguarinho,15,,-20.033333,-47.3
br,jaguaripe,Jaguaripe,05,,-12.916667,-39.166667
br,jaguarisinho,Jaguarisinho,23,,-29.483333,-54.783333
br,jaguarita,Jaguarita,23,,-27.466667,-52.25
br,jaguaritira,Jaguaritira,15,,-17.9,-42.233333
br,jaguariuna,Jaguariúna,27,31718,-22.683333,-46.983333
br,jaguarizinho,Jaguarizinho,23,,-29.483333,-54.783333
br,jaguaruana,Jaguaruana,06,21791,-4.833333,-37.783333
br,jaguaruna,Jaguaruna,15,,-19.716667,-44.8
br,jaguaruna,Jaguaruna,26,11650,-28.6,-49.033333
br,jaguary,Jaguary,15,,-22.766667,-46.15
br,jaguary,Jaguary,23,,-29.5,-54.683333
br,jaguary,Jaguary,27,,-22.683333,-46.983333
br,jaguatei,Jaguateí,27,,-21.1,-48.783333
br,jaguatirica,Jaguatirica,18,,-24.166667,-50.633333
br,jahu,Jahú,27,,-22.3,-48.55
br,jaiba,Jaíba,05,,-12.2,-38.883333
br,jaibaras,Jaibaras,06,,-3.766667,-40.483333
br,jaicos,Jaicós,20,7107,-7.35,-41.133333
br,jaime ferreira,Jaime Ferreira,15,,-20.333333,-45.2
br,jaime guimaraes,Jaime Guimarães,15,,-16.383333,-40.766667
br,jaiobazinho,Jaiobazinho,31,,-11.433333,-46.633333
br,jaiu,Jaiú,05,,-9.383333,-38.516667
br,jajuaryahiva,Jajuaryahiva,18,,-24.25,-49.7
br,jalbaras,Jalbaras,06,,-3.766667,-40.483333
br,jales,Jales,27,43832,-20.266667,-50.55
br,jamacaru,Jamacaru,06,,-7.4,-39.116667
br,jamaica,Jamaica,27,,-21.516667,-51.616667
br,jamanacari,Jamanacari,04,,-2.983333,-69.033333
br,jamandua,Jamanduá,04,,-6.383333,-64.283333
br,jamapara,Jamapará,21,,-21.916667,-42.716667
br,jamari,Jamari,04,,-4.374167,-62.062778
br,jamari,Jamari,05,,-12.516667,-40.466667
br,jamari,Jamari,24,,-8.75,-63.45
br,jamary,Jamary,24,,-8.45,-63.483333
br,jambeiro,Jambeiro,15,,-18.083333,-42.766667
br,jambeiro,Jambeiro,27,,-23.266667,-45.683333
br,jaminaua,Jaminauá,01,,-9.353056,-71.024722
br,jamitine,Jamitine,04,,-5.704722,-63.642778
br,jampruca,Jampruca,15,,-18.45,-41.8
br,janari,Janari,04,,-.333333,-64.216667
br,j. anastacio,J. Anastácio,15,,-21.05,-45.216667
br,janauba,Janaúba,15,56573,-15.8,-43.316667
br,jandaia do sul,Jandaia do Sul,18,17507,-23.6,-51.65
br,jandaia,Jandaia,18,,-23.6,-51.65
br,jandaia,Jandaia,29,,-17.1,-50.116667
br,jandaira,Jandaíra,05,,-11.566667,-37.7
br,jandaira,Jandaíra,22,,-5.356111,-36.126111
br,jandia,Jandiá,04,,-5.089167,-59.393333
br,jandira,Jandira,27,113607,-23.516667,-46.9
br,jandiritiua,Jandiritiua,13,,-2.1,-44.633333
br,jandoim,Jandoim,06,,-5.266667,-38.266667
br,jandui,Janduí,22,,-5.445556,-37.061944
br,janduis,Janduís,22,,-6.016667,-37.416667
br,jangada do sul,Jangada do Sul,18,,-26.366667,-51.25
br,jangada,Jangada,05,,-11.683333,-43.266667
br,jangada,Jangada,14,,-15.233333,-56.483333
br,jangada,Jangada,15,,-22.05,-45.966667
br,jangada,Jangada,18,,-23.316667,-50.85
br,jangada,Jangada,26,,-26.5,-51.233333
br,jangada,Jangada,27,,-21.416667,-50.683333
br,jangada nova,Jangada Nova,27,,-21.45,-50.733333
br,jangadinha,Jangadinha,26,,-26.5,-51.233333
br,jangadinha,Jangadinha,27,,-21.416667,-50.55
br,janga,Janga,30,,-7.933333,-34.816667
br,jango,Jango,11,,-20.45,-55.483333
br,jangraguera,Jangraguera,06,,-5.483333,-40.2
br,jangueta,Janguetá,08,,-20.016667,-40.266667
br,jansen,Jansen,23,,-29.133333,-51.383333
br,janua coeli,Jánua Coeli,16,,-1.966667,-49.383333
br,januaria de acaracu,Januária de Acaracu,06,,-3.7,-40.35
br,januaria,Januária,15,34812,-15.483333,-44.366667
br,januario cicco,Januário Cicco,22,,-6.15,-35.583333
br,januario cico,Januário Cico,22,,-6.15,-35.583333
br,januario,Januário,05,,-12.366667,-39.1
br,januario,Januário,13,,-1.833333,-46.2
br,jao,Jaó,27,,-21.166667,-47.283333
br,jaozeiro,Jaozeiro,06,,-7.2,-39.333333
br,japao,Japão,06,,-5.566667,-39.216667
br,japao,Japão,13,,-3.483333,-42.6
br,japao,Japão,14,,-14.983333,-57.333333
br,japao,Japão,15,,-20.55,-44.633333
br,japao,Japão,27,,-20.383333,-47.566667
br,japaraiba,Japaraíba,15,,-20.133333,-45.5
br,japaratinga,Japaratinga,02,,-9.083333,-35.25
br,japaratuba,Japaratuba,02,,-9.083333,-35.25
br,japaratuba,Japaratuba,28,8092,-10.583333,-36.95
br,japecanaa,Japecanaa,17,,-6.216667,-37.466667
br,japecanga,Japecanga,30,,-8.233333,-36.133333
br,japeco,Japeco,13,,-5.15,-42.933333
br,japerica,Japerica,16,,-.833333,-47.083333
br,japeri,Japeri,21,95102,-22.648889,-43.658611
br,japiim,Japiim,01,,-7.616667,-72.9
br,japi,Japi,22,,-6.45,-35.933333
br,japira,Japira,05,,-17.483333,-39.75
br,japira,Japira,18,,-23.8115,-50.139
br,japiuba,Japiúba,27,,-20.65,-50.5
br,japoata,Japoatã,28,4207,-10.333333,-36.8
br,japo,Japó,04,,-3.533333,-66.083333
br,japomirim,Japomirim,05,,-14.15,-39.733333
br,japuiba,Japuíba,21,,-22.583333,-42.7
br,japuim,Japuim,01,,-7.616667,-72.9
br,japu,Japu,05,,-14.916667,-39.2
br,japura,Japurá,04,1908,-1.395833,-69.422222
br,japura,Japurá,18,,-23.470661,-52.554346
br,japyra,Japyra,18,,-23.8115,-50.139
br,jaquari,Jaquari,27,,-24.183333,-48.966667
br,jaquaritinga do norte,Jaquaritinga do Norte,30,,-7.9,-36.05
br,jaquary,Jaquary,27,,-22.683333,-46.983333
br,jaqueira,Jaqueira,02,,-9.383333,-37.166667
br,jaqueira,Jaqueira,05,,-12.366667,-41.15
br,jaqueira,Jaqueira,13,,-3.326667,-43.120556
br,jaqueira,Jaqueira,30,,-8.733333,-35.816667
br,jaques,Jaques,17,,-7.75,-36.4
br,jaquhype,Jaquhype,05,,-12.483333,-38.633333
br,jaquipe,Jaquípe,05,,-12.483333,-38.633333
br,jaquirana,Jaquirana,23,,-28.9,-50.383333
br,jaraaua,Jaraauá,27,,-23.454785,-46.742034
br,jaracatia,Jaracatiá,08,,-20.783333,-40.783333
br,jaracatia,Jaracatiá,18,,-24.183333,-52.9
br,jaracatia,Jaracatiá,30,,-8.133333,-36.35
br,jaracatiara,Jaracatiara,08,,-20.783333,-40.783333
br,jaracutia,Jaracutiá,30,,-8.133333,-36.35
br,jaragua do sul,Jaraguá do Sul,26,130132,-26.483333,-49.066667
br,jaragua,Jaraguá,11,,-20.466667,-54.766667
br,jaragua,Jaraguá,14,,-17.15,-55.083333
br,jaragua,Jaraguá,15,,-21.15,-45.733333
br,jaragua,Jaraguá,26,,-26.483333,-49.066667
br,jaragua,Jaraguá,27,,-23.454785,-46.742034
br,jaragua,Jaraguá,29,30410,-15.75,-49.333333
br,jaraguari,Jaraguari,11,,-20.15,-54.416667
br,jaraguary,Jaraguary,11,,-20.15,-54.416667
br,jaramataia,Jaramatáia,02,,-9.666667,-37
br,jaramataia,Jaramataia,17,,-7.533333,-37.983333
br,jarametais,Jarametais,02,,-9.666667,-37
br,jarandragoeira,Jarandragoeira,06,,-3.8,-38.733333
br,jaraqui,Jaraqui,04,,-2.953611,-60.438889
br,jararaca,Jararaca,25,,2.933333,-60.533333
br,jararaca,Jararaca,26,,-26.2,-50.7
br,jararaca,Jararaca,27,,-21.95,-47.883333
br,jararaita,Jararaitá,13,,-2.416667,-44.683333
br,jardim 25 de agosto,Jardim 25 de Agosto,21,,-22.793058,-43.300357
br,jardim acacia,Jardim Acacia,21,,-22.802155,-43.63629
br,jardim acacio,Jardim Acácio,27,,-23.413867,-46.510778
br,jardim agua branca,Jardim Água Branca,21,,-22.85932,-43.441392
br,jardim aguas lindas,Jardim Águas Lindas,21,,-22.847854,-43.761216
br,jardim alecrim,Jardim Alecrim,21,,-22.892835,-43.486203
br,jardim alegre,Jardim Alegre,18,,-24.15,-51.716667
br,jardim alexandrino,Jardim Alexandrino,21,,-22.884949,-43.663487
br,jardim almeida,Jardim Almeida,27,,-23.267277,-46.568012
br,jardim alto da cantareira,Jardim Alto da Cantareira,27,,-23.386919,-46.606195
br,jardim america,Jardim América,21,,-22.736976,-43.41834
br,jardim america,Jardim América,27,,-23.302734,-46.552188
br,jardim arnaldo eugenio,Jardim Arnaldo Eugênio,21,,-22.913263,-43.536194
br,jardim aurea,Jardim Áurea,21,,-22.890932,-43.448857
br,jardim bangu,Jardim Bangu,21,,-22.849494,-43.466021
br,jardim barra da tijuca,Jardim Barra da Tijuca,21,,-22.99696,-43.341392
br,jardim bela vista,Jardim Bela Vista,21,,-22.883689,-43.59257
br,jardim bela vista,Jardim Bela Vista,27,,-23.433333,-46.516667
br,jardim boiuna,Jardim Boiuna,21,,-22.910025,-43.39452
br,jardim bom pastor,Jardim Bom Pastor,21,,-22.753943,-43.372396
br,jardim borba gato,Jardim Borba Gato,27,,-23.801,-46.6333
br,jardim botanico,Jardim Botãnico,21,,-22.962103,-43.223313
br,jardim botujuru,Jardim Botujuru,27,,-24.712285,-48.004843
br,jardim brasilia,Jardim Brasilia,27,,-23.441833,-46.708345
br,jardim brisa de atibaia,Jardim Brisa de Atibaia,27,,-23.227778,-46.590278
br,jardim brogota,Jardim Brogotá,27,,-23.093333,-46.540278
br,jardim campinas barro branco,Jardim Campinas Barro Branco,27,,-23.7705,-46.7064
br,jardim campinho,Jardim Campinho,21,,-22.89364,-43.587549
br,jardim campo belo,Jardim Campo Belo,21,,-22.871192,-43.584966
br,jardim capuacu,Jardim Capuaçu,21,,-22.772981,-43.524402
br,jardim carioca,Jardim Carioca,21,,-22.802916,-43.192492
br,jardim centenario,Jardim Centenário,27,,-23.173056,-46.573333
br,jardim central,Jardim Central,21,,-22.795564,-43.640988
br,jardim cidade pirituba,Jardim Cidade Pirituba,27,,-23.481371,-46.721903
br,jardim cinco lagos de santa maria,Jardim Cinco Lagos de Santa Maria,27,,-23.310842,-46.534106
br,jardim cincos marias,Jardim Cincos Marias,21,,-22.974416,-43.6413
br,jardim clarice,Jardim Clarice,21,,-22.868567,-43.496941
br,jardim cristina,Jardim Cristina,27,,-23.320984,-46.209231
br,jardim da barra,Jardim da Barra,21,,-22.993018,-43.30017
br,jardim da lagoa,Jardim da Lagoa,27,,-23.251846,-46.609341
br,jardim dalila,Jardim Dalila,27,,-23.5144,-47.6042
br,jardim da luz,Jardim da Luz,21,,-22.887524,-43.563792
br,jardim da luz,Jardim da Luz,29,,-17.866667,-49.133333
br,jardim damasceno,Jardim Damasceno,27,,-23.447669,-46.6977
br,jardim da posse,Jardim da Posse,21,,-22.732733,-43.453351
br,jardim da represa,Jardim da Represa,27,,-23.7597,-46.596
br,jardim das acacias,Jardim Das Acácias,21,,-22.728734,-43.424298
br,jardim das accias,Jardim das Accias,27,,-23.3148,-46.2042
br,jardim da saude,Jardim da Saude,27,,-23.6,-46.633333
br,jardim das bandeiras,Jardim das Bandeiras,27,,-23.63475,-47.576458
br,jardim da serra,Jardim da Serra,27,,-23.338096,-46.544179
br,jardim das flores,Jardim das Flores,27,,-23.252028,-46.545072
br,jardim das oliveiras,Jardim das Oliveiras,23,,-29.9035,-50.8793
br,jardim das pedrinhas,Jardim das Pedrinhas,21,,-22.919087,-43.673656
br,jardim de abril,Jardim de Abril,27,,-23.449478,-46.748266
br,jardim de alah,Jardim de Alah,21,,-22.983379,-43.212576
br,jardim de angicos,Jardim de Angicos,22,,-5.65,-35.983333
br,jardim de areias,Jardim de Areias,06,,-4.233333,-38.883333
br,jardim de mairipora gleba ii,Jardim de Mairiporã Gleba II,27,,-23.310899,-46.634021
br,jardim de mairipora gleba i,Jardim de Mairiporã Gleba I,27,,-23.320817,-46.632506
br,jardim de mairipora gleba segunda,Jardim de Mairiporã Gleba Segunda,27,,-23.310899,-46.634021
br,jardim de mairipora gleba um,Jardim de Mairiporã Gleba Um,27,,-23.320817,-46.632506
br,jardim de pirahanas,Jardim de Pirahanas,22,,-6.366667,-37.333333
br,jardim de piranhas,Jardim de Piranhas,22,10935,-6.366667,-37.333333
br,jardim do saco,Jardim do Saco,21,,-22.852778,-43.298386
br,jardim dos afonsos,Jardim dos Afonsos,27,,-23.426989,-46.529113
br,jardim do serido,Jardim do Seridó,22,9520,-6.583333,-36.766667
br,jardim dos jacarandas,Jardim dos Jacarandas,27,,-23.257667,-46.598586
br,jardim embaixador,Jardim Embaixador,27,,-22.7074,-45.5613
br,jardim espirito santo,Jardim Espírito Santo,21,,-22.942551,-43.691985
br,jardim estancia brasil,Jardim Estância Brasil,27,,-23.156667,-46.594722
br,jardim estrela d'alva,Jardim Estrela D'Alva,21,,-22.908183,-43.669201
br,jardim europa,Jardim Europa,27,,-22.70174,-45.5568
br,jardim fortaleza,Jardim Fortaleza,06,,-3.783333,-38.7
br,jardim frutuoso,Jardim Frutuoso,21,,-22.871039,-43.668243
br,jardim gaver,Jardim Gaver,21,,-22.991907,-43.258183
br,jardim gibeon,Jardim Gibeon,27,,-23.255711,-46.588286
br,jardim gramacho,Jardim Gramacho,21,,-22.757371,-43.280747
br,jardim guanabara,Jardim Guanabara,21,,-22.81559,-43.199759
br,jardim guanabara,Jardim Guanabara,27,,-22.725556,-47.338056
br,jardim guaratiba,Jardim Guaratiba,21,,-22.976087,-43.652047
br,jardim helena,Jardim Helena,27,,-22.733056,-47.306111
br,jardim heliana,Jardim Heliana,27,,-23.755804,-46.669761
br,jardim iaraqua,Jardim Iaraqua,21,,-22.921519,-43.560105
br,jardim imperial,Jardim Imperial,21,,-22.785931,-43.540761
br,jardim ipiranga,Jardim Ipiranga,27,,-22.7475,-47.358056
br,jardim ipora,Jardim Iporã,27,,-23.783141,-46.703551
br,jardimirim,Jardimirim,06,,-7.6,-39.183333
br,jardim ita,Jardim Ita,21,,-22.914253,-43.694581
br,jardim itapeva,Jardim Itapeva,27,,-23.479096,-46.744996
br,jardim iv centenario,Jardim IV Centenario,27,,-23.7986,-46.5866
br,jardim,Jardim,06,7189,-3.433333,-39.016667
br,jardim,Jardim,07,,-15.983333,-47.4
br,jardim,Jardim,08,,-21.116667,-41.666667
br,jardim,Jardim,11,23781,-21.466667,-56.15
br,jardim,Jardim,13,,-3.922222,-43.435833
br,jardim,Jardim,15,,-17.516667,-43.333333
br,jardim,Jardim,17,,-7.733333,-35.85
br,jardim,Jardim,20,,-5.266667,-42.283333
br,jardim,Jardim,22,,-5.55,-35.966667
br,jardim,Jardim,26,,-26.7,-52.116667
br,jardim,Jardim,27,,-20.7,-47.65
br,jardim,Jardim,30,,-7.666667,-39.966667
br,jardim,Jardim,31,,-11.815,-46.628056
br,jardim joana,Jardim Joana,27,,-23.441508,-46.54395
br,jardim joari,Jardim Joari,21,,-22.910144,-43.551035
br,jardim julia miguel,Jardim Júlia Miguel,21,,-22.917602,-43.629564
br,jardim lagoa mar,Jardim Lagoa Mar,21,,-22.997594,-43.333852
br,jardim laranjeiras,Jardim Laranjeiras,21,,-22.773809,-43.537012
br,jardim leal,Jardim Leal,21,,-22.760474,-43.322785
br,jardim luana,Jardim Luana,21,,-22.970551,-43.628557
br,jardim lucia,Jardim Lúcia,27,,-23.269879,-46.573036
br,jardim magarca,Jardim Magarça,21,,-22.936529,-43.58056
br,jardim marambaia,Jardim Marambaia,21,,-22.642778,-43.405278
br,jardim marapendi,Jardim Marapendi,21,,-23.007126,-43.354456
br,jardim maravilha,Jardim Maravilha,21,,-22.962417,-43.605376
br,jardim maria antonia,Jardim Maria Antônia,27,,-23.313123,-46.548285
br,jardim maria da gloria,Jardim Maria da Glória,27,,-23.511,-47.6194
br,jardim maria paula,Jardim Maria Paula,21,,-22.877222,-43.024722
br,jardim maric,Jardim Maric,27,,-23.4954,-46.1814
br,jardim master,Jardim Master,27,,-23.505,-47.603
br,jardim mendanha,Jardim Mendanha,21,,-22.853576,-43.540102
br,jardim miramar,Jardim Miramar,27,,-23.8113,-46.5793
br,jardim mirassol,Jardim Mirassol,21,,-22.768307,-43.526496
br,jardim monte alto,Jardim Monte Alto,27,,-23.400411,-46.535857
br,jardim monteiro,Jardim Monteiro,21,,-22.939236,-43.572136
br,jardim monte santo,Jardim Monte Santo,21,,-22.910958,-43.535833
br,jardim moreira,Jardim Moreira,27,,-23.438048,-46.544225
br,jardim morro azul,Jardim Morro Azul,27,,-22.5575,-47.428333
br,jardim natal,Jardim Natal,27,,-23.8634,-46.6693
br,jardim nautico da cantareira,Jardim Nautico Da Cantareira,27,,-23.321917,-46.565265
br,jardim nazare,Jardim Nazaré,21,,-22.858889,-43.017778
br,jardim nossa senhora das gracas,Jardim Nossa Senhora das Graças,21,,-22.851977,-43.530159
br,jardim nova europa,Jardim Nova Europa,27,,-22.581111,-47.417778
br,jardim nova guaratiba,Jardim Nova Guaratiba,21,,-22.924333,-43.564477
br,jardim nova italia,Jardim Nova Italia,27,,-22.556667,-47.415
br,jardim nova suica,Jardim Nova Suica,27,,-22.568889,-47.390833
br,jardim novo dem,Jardim Novo dem,27,,-23.3375,-46.2243
br,jardim novo mundo,Jardim Novo Mundo,21,,-22.945912,-43.366722
br,jardim novo murido,Jardim Novo Murido,27,,-23.5365,-47.511
br,jardim novo realengo,Jardim Novo Realengo,21,,-22.886456,-43.421007
br,jardim oceanico,Jardim Oceánico,21,,-23.008997,-43.31058
br,jardim okinaga,Jardim Okinaga,27,,-23.29516,-46.609842
br,jardim okinagua,Jardim Okinagua,27,,-23.288624,-46.59943
br,jardim olavo bilac,Jardim Olavo Bilac,21,,-22.775623,-43.338883
br,jardim olimpio,Jardim Olímpio,27,,-23.251484,-46.592467
br,jardim oriental,Jardim Oriental,21,,-22.890624,-43.520244
br,jardim palmares,Jardim Palmares,21,,-22.885328,-43.641309
br,jardim palmira,Jardim Palmira,27,,-23.347301,-47.82114
br,jardim paraiso,Jardim Paraíso,21,,-22.896417,-43.578894
br,jardim paraiso,Jardim Paraíso,27,,-23.414929,-46.503447
br,jardim paulista,Jardim Paulista,21,,-22.882618,-43.559894
br,jardim paulista,Jardim Paulista,27,,-23.4364,-46.550721
br,jardim pedregoso,Jardim Pedregoso,21,,-22.869189,-43.562994
br,jardim perlamar,Jardim Perlamar,27,,-23.5171,-47.612
br,jardim pernambuco,Jardim Pernambuco,21,,-22.981663,-43.230026
br,jardim piai,Jardim Piaí,21,,-22.97783,-43.664689
br,jardim pinheiro,Jardim Pinheiro,27,,-23.434013,-46.523018
br,jardim portugal,Jardim Portugal,27,,-23.3035,-46.2156
br,jardim praca seca,Jardim Praça Seca,21,,-22.89935,-43.360216
br,jardim presidente,Jardim Presidente,27,,-23.243889,-46.606111
br,jardim primavera,Jardim Primavera,27,,-23.1825,-46.5775
br,jardim prince,Jardim Prince,27,,-23.318843,-46.601489
br,jardim progresso,Jardim Progresso,27,,-23.8405,-46.7254
br,jardim quarto centenario,Jardim Quarto Centenario,21,,-22.950817,-43.377515
br,jardim recanto do sol,Jardim Recanto do Sol,27,,-23.807,-46.6466
br,jardim redentor,Jardim Redentor,21,,-22.752885,-43.352962
br,jardim regina,Jardim Regina,27,,-23.473582,-46.742598
br,jardim represa,Jardim Represa,27,,-23.773,-46.5875
br,jardim rincao,Jardim Rincão,27,,-23.433735,-46.719491
br,jardim roseiral,Jardim Roseiral,27,,-23.152222,-46.7075
br,jardim rossi,Jardim Rossi,27,,-23.442033,-46.52387
br,jardim salaco,Jardim Salaco,21,,-22.370508,-42.97434
br,jardim sandra,Jardim Sandra,27,,-23.299064,-46.567056
br,jardim santa cruz,Jardim Santa Cruz,21,,-22.901307,-43.677692
br,jardim santa cruz,Jardim Santa Cruz,27,,-23.5083,-47.5996
br,jardim santa emilia,Jardim Santa Emilia,27,,-23.421602,-46.511361
br,jardim santa helena,Jardim Santa Helena,21,,-22.726082,-43.344217
br,jardim santa lucrecia,Jardim Santa Lucrécia,27,,-23.441587,-46.747402
br,jardim santana,Jardim Santana,27,,-23.326457,-46.57574
br,jardim santa teresa,Jardim Santa Teresa,27,,-23.7969,-46.6358
br,jardim sao bernardo,Jardim São Bernardo,27,,-23.7641,-46.7029
br,jardim sao domingos,Jardim São Domingos,21,,-22.919562,-43.637839
br,jardim sao gabriel das jabuticabeiras,Jardim São Gabriel das Jabuticabeiras,27,,-23.320527,-46.612706
br,jardim sao geraldo,Jardim São Geraldo,21,,-22.866117,-43.557795
br,jardim sao jeronimo,Jardim São Jerônimo,21,,-22.914079,-43.527179
br,jardim sao luis,Jardim São Luis,27,,-22.574722,-47.381667
br,jardim sao manuel,Jardim São Manuel,27,,-22.582222,-47.405278
br,jardim sao nicolau,Jardim São Nicolau,27,,-23.125,-46.548889
br,jardim sao paulo,Jardim São Paulo,27,,-23.43682,-46.5263
br,jardim sao pedro,Jardim São Pedro,21,,-22.87455,-43.558657
br,jardim sao rafael,Jardim São Rafael,27,,-23.758239,-46.7157
br,jardim sao vicente,Jardim São Vicente,21,,-22.715505,-43.443721
br,jardim sete de abril,Jardim Sete de Abril,21,,-22.928877,-43.637188
br,jardim shangrila,Jardim Shangrilá,21,,-22.914801,-43.406639
br,jardim sulacap,Jardim Sulacap,21,,-22.889059,-43.386159
br,jardim sun valley,Jardim Sun Valley,27,,-23.246667,-46.563333
br,jardim terra nova,Jardim Terra Nova,21,,-22.941667,-43.059167
br,jardim toquio,Jardim Tóquio,27,,-23.325673,-47.824945
br,jardim tupa,Jardim Tupa,27,,-23.7949,-46.5458
br,jardim uberlandia,Jardim Uberlândia,21,,-22.884783,-43.416375
br,jardim valaria simone,Jardim Valaria Simone,27,,-23.30359,-46.53489
br,jardim vila mar de guaratiba,Jardim Vila Mar de Guaratiba,21,,-22.991224,-43.623368
br,jardim vinte cinco de agosto,Jardim Vinte Cinco de Agosto,21,,-22.793058,-43.300357
br,jardim vista alegre,Jardim Vista Alegre,26,,-26.248237,-49.539618
br,jardim vitoria,Jardim Vitoria,21,,-22.913385,-43.637687
br,jardim vitoria-regia,Jardim Vitória-Régia,27,,-23.188333,-46.613889
br,jardineira,Jardineira,05,,-13.9,-41.266667
br,jardineiro,Jardineiro,17,,-6.816667,-38.55
br,jardinesia,Jardinésia,15,,-19.25,-48.683333
br,jardin,Jardin,31,,-11.815,-46.628056
br,jardinocolis,Jardinocolis,05,,-12.183333,-43.216667
br,jardinopolis,Jardinópolis,05,,-12.183333,-43.216667
br,jardinopolis,Jardinópolis,15,,-17.666667,-41.416667
br,jardinopolis,Jardinópolis,27,34719,-21.033333,-47.766667
br,jari,Jari,04,,-4.75,-62.157778
br,jari,Jari,23,,-29.283333,-54.216667
br,jarina,Jarina,04,,-4.391389,-71.238889
br,jarinu,Jarinu,27,17905,-23.103333,-46.728056
br,jarquari,Jarquarí,11,,-20.15,-54.416667
br,j. artur,J. Artur,05,,-12.416667,-40.633333
br,jarudore,Jarudore,14,,-16.2,-54.3
br,jaru,Jaru,24,28018,-10.433333,-62.45
br,jary,Jary,23,,-29.283333,-54.216667
br,jataba,Jatabá,20,,-9.516667,-44.966667
br,jatahy,Jatahy,29,,-17.883333,-51.716667
br,jatahy,Jatahy,30,,-8.65,-40.233333
br,jatahy parana,Jatahy Paraná,18,,-23.216667,-50.966667
br,jatai,Jataí,14,,-13.391667,-59.8625
br,jatai,Jataí,15,,-18.883333,-44.166667
br,jatai,Jataí,18,,-23.216667,-50.966667
br,jatai,Jataí,27,,-23.5,-48.983333
br,jatai,Jataí,29,76548,-17.883333,-51.716667
br,jataizinho,Jataìzinho,18,10014,-23.216667,-50.966667
br,jatal,Jatal,18,,-23.216667,-50.966667
br,jatauba,Jataúba,30,,-7.966667,-36.483333
br,jatei,Jatei,11,,-22.383333,-54.316667
br,jati,Jati,06,,-7.683333,-39.016667
br,jatimana,Jatimana,05,,-13.7,-39.05
br,jatina de baixo,Jatinã de Baixo,30,,-8.783333,-38.783333
br,jatina,Jatinã,30,,-8.766667,-38.966667
br,jatiuca,Jatiúca,30,,-7.883333,-38.216667
br,jatoba,Jatobá,04,,-4.562778,-59.408056
br,jatoba,Jatobá,05,,-11.9,-38.5
br,jatoba,Jatobá,06,,-2.981667,-40.973333
br,jatoba,Jatobá,11,,-20.05,-55.966667
br,jatoba,Jatobá,13,,-3.133056,-42.191667
br,jatoba,Jatobá,14,,-16.733333,-57.616667
br,jatoba,Jatobá,15,,-15.3,-44.166667
br,jatoba,Jatobá,16,,-4.833611,-52.783889
br,jatoba,Jatobá,17,,-6.833333,-37.65
br,jatoba,Jatoba,20,,-6.433333,-42.366667
br,jatoba,Jatobá,22,,-5.934444,-36.971111
br,jatoba,Jatobá,27,,-21.633333,-50.233333
br,jatoba,Jatobá,30,,-7.55,-37.583333
br,jatoba,Jatobá,31,,-5.2,-48.333333
br,jatobal,Jatobal,16,,-4.5,-49.533333
br,jatobazal,Jatobazal,20,,-3.982778,-42.005556
br,jatobazinho,Jatobàzinho,06,,-5.266667,-40.366667
br,jatobazinho,Jatobàzinho,30,,-7.983333,-38.166667
br,jatobazinho,Jatobàzinho,31,,-11.433333,-46.633333
br,jatobe,Jatobe,30,,-9.083333,-38.3
br,jatuarama,Jatuarama,24,,-9.416667,-62.533333
br,jatuarana,Jatuarana,04,,-5.806389,-63.536944
br,jatuarana,Jatuarana,24,,-9.133333,-62.25
br,jatuaraninha,Jatuaraninha,04,,-5.268611,-60.383611
br,jaua,Jauá,05,,-12.766667,-38.166667
br,jauaraita,Jauaraitá,13,,-2.416667,-44.683333
br,jauaraua,Jauarauá,04,,-2.883333,-64.916667
br,jauarete,Jauareté,04,,.633333,-69.2
br,jauarete poco,Jauaretê Poço,04,,1.203611,-69.297778
br,jauari,Jauari,04,,-.333333,-64.216667
br,jauato,Jauató,04,,-2.733333,-65.383333
br,jaudilandia,Jaudilândia,14,,-13.6,-55.966667
br,jau do tocantins,Jaú do Tocantins,31,,-12.6525,-48.54
br,jau,Jaú,11,,-20.983333,-55.133333
br,jau,Jaú,14,,-16.1,-55.866667
br,jau,Jaú,27,119209,-22.3,-48.55
br,jaupaci,Jaupaci,29,,-16.3,-50.9
br,jauquara,Jauquara,14,,-15.183333,-57.083333
br,jauquarinha,Jauquarinha,14,,-15.2,-56.933333
br,jauru,Jauru,11,,-18.583333,-54.283333
br,jauzinho,Jaùzinho,31,,-10.033333,-48.333333
br,java,Javá,05,,-12.766667,-38.166667
br,javari,Javari,04,,-4.333333,-57.266667
br,javari quixaba,Javari Quixaba,22,,-5.351667,-37.251667
br,j. bernardes,J. Bernardes,29,,-16.5,-52.05
br,jeceaba,Jeceaba,15,,-20.533333,-43.983333
br,jeguy,Jeguy,30,,-8.65,-39.65
br,jeitosa,Jeitosa,13,,-5.95,-46.55
br,jejui,Jejuí,28,,-4.433333,-45.283333
br,jenesare,Jenesaré,06,,-6.9,-40.033333
br,jenipabu,Jenipabu,05,,-12.7,-38.116667
br,jenipabu,Jenipabu,22,,-5.7,-35.204167
br,jenipapeiro,Jenipapeiro,06,,-5.916667,-39.083333
br,jenipapeiro,Jenipapeiro,17,,-6.516667,-37.616667
br,jenipapeiro,Jenipapeiro,20,,-3.645556,-42.466389
br,jenipapeiro,Jenipapeiro,22,,-5.716667,-42.283333
br,jenipapo iii,Jenipapo III,13,,-3.101389,-42.328056
br,jenipapo ii,Jenipapo II,13,,-3.381111,-42.448611
br,jenipapo i,Jenipapo I,20,,-3.546111,-42.053333
br,jenipapo,Jenipapo,02,,-9.133333,-36.1
br,jenipapo,Jenipapo,03,,3.8,-51.183333
br,jenipapo,Jenipapo,04,,-5.571944,-60.921111
br,jenipapo,Jenipapo,05,,-11.5,-44.95
br,jenipapo,Jenipapo,06,,-3.55,-39.183333
br,jenipapo,Jenipapo,11,,-21.4,-55.233333
br,jenipapo,Jenipapo,13,,-3.8175,-43.299444
br,jenipapo,Jenipapo,15,,-16.633333,-41.866667
br,jenipapo,Jenipapo,16,,-5.35,-48.883333
br,jenipapo,Jenipapo,17,,-6.766667,-37.766667
br,jenipapo,Jenipapo,20,,-7.366667,-43.95
br,jenipapo,Jenipapo,28,,-10.466667,-36.583333
br,jenipapo,Jenipapo,30,,-8.283333,-36.566667
br,jenipapo,Jenipapo,31,,-7.516667,-49.3
br,jenipauba,Jenipaúba,13,,-2.066667,-44.566667
br,jequeri,Jequeri,15,,-20.45,-42.666667
br,jequery,Jequery,15,,-20.45,-42.666667
br,jequia do cabuta,Jequiá do Cabuta,02,,-9.616667,-36.433333
br,jequia,Jequia,27,,-22.416667,-51.666667
br,jequiazinho,Jequiàzinho,02,,-10.016667,-36.033333
br,jequie,Jequié,05,127479,-13.85,-40.083333
br,jequie,Jequié,27,,-22.416667,-51.666667
br,jequi,Jequi,05,,-13.05,-40.066667
br,jequi,Jequi,22,,-5.916667,-35.166667
br,jequi,Jequi,30,,-8.65,-39.65
br,jequirica,Jequirica,05,,-13.233333,-39.6
br,jequiri,Jequiri,13,,-3.406944,-42.562222
br,jequiri,Jequiri,30,,-8.666667,-36.45
br,jequitahy,Jequitahy,15,,-17.25,-44.466667
br,jequitaia,Jequitaia,05,,-17.25,-39.5
br,jequitaia,Jequitaia,27,,-21.1,-51.05
br,jequitai,Jequitaí,15,,-17.25,-44.466667
br,jequitiba,Jequitibá,05,,-12.933333,-39.266667
br,jequitiba,Jequitiba,08,,-20.05,-40.666667
br,jequitiba,Jequitibá,15,,-19.266667,-44.016667
br,jequitinhonha,Jequitinhonha,15,15463,-16.433333,-41
br,jereba,Jereba,02,,-9.45,-36.316667
br,jeremias,Jeremias,18,,-25,-51.35
br,jeremoabo,Jeremoabo,05,16642,-10.066667,-38.35
br,jeriba,Jeribá,15,,-16.833333,-40.383333
br,jeribatuba,Jeribatuba,05,,-13.066667,-38.783333
br,jericoacoara,Jericoacoara,06,,-2.793333,-40.515833
br,jericoaquara,Jericoaquara,06,,-2.8,-40.483333
br,jerico,Jericó,06,,-5.183333,-40.366667
br,jerico,Jericó,17,,-6.55,-37.8
br,jerico,Jericó,22,,-5.37,-36.096944
br,jerico,Jericó,30,,-7.766667,-38.033333
br,jerimatalho,Jerimatalho,02,,-9.666667,-37
br,jerimum,Jerimum,06,,-5.383333,-39.45
br,jerimum,Jerimum,17,,-6.566667,-38.216667
br,jeriquaquara,Jeriquaquara,06,,-2.8,-40.483333
br,jeriquara,Jeriquara,27,,-20.316667,-47.6
br,jeritaco,Jeritacó,30,,-8.4,-37.633333
br,jeriva,Jerivá,27,,-23.233918,-47.723666
br,jeroaquara,Jeroaquara,29,,-15.383333,-50.416667
br,jeromenha,Jeromenha,20,,-7.083333,-43.5
br,jeronimo barcelos,Jerônimo Barcelos,11,,-19.666667,-54.633333
br,jeronimo belmonte,Jerônimo Belmonte,11,,-23.366667,-55.016667
br,jeronimo,Jerônimo,04,,-4.320278,-59.247778
br,jeronimo,Jerônimo,05,,-12.616667,-41.666667
br,jeronimo,Jerônimo,11,,-18.016667,-54.65
br,jeronimo,Jerônimo,15,,-20.183333,-41.616667
br,jeronimo,Jerônimo,20,,-6.75,-42.25
br,jeronimo monteiro,Jerônimo Monteiro,08,6902,-20.783333,-41.4
br,jeronimo r. oliveira,Jerônimo R. Oliveira,15,,-17.633333,-41.65
br,jeru,Jeru,28,,-11.383333,-37.85
br,jerumenha,Jerumenha,20,,-7.083333,-43.5
br,jerumirim,Jerumirim,21,,-22.933333,-44.316667
br,jerusalem,Jerusalém,22,,-5.366667,-35.533333
br,jerusalem,Jerusalém,24,,-12.783333,-60.091667
br,jerusalem,Jerusalem,30,,-7.783333,-38.916667
br,jesenopolis,Jesenópolis,15,,-16.566667,-42.533333
br,jesuania,Jesuânia,15,,-22,-45.3
br,jesuino da costa,Jesuíno da Costa,15,,-16.666667,-40.433333
br,jesuino,Jesuíno,15,,-17.166667,-41.033333
br,jesuino pereira,Jesuíno Pereira,15,,-17.5,-41.7
br,jesuitas,Jesuítas,21,,-22.875485,-43.67888
br,jetiba,Jetibá,08,,-20.05,-40.666667
br,jiboia,Jibóia,04,,1.226667,-69.502778
br,jiboia,Jibóia,05,,-13.016667,-40.466667
br,jiboia,Jibóia,13,,-3.513611,-42.979722
br,jiboia,Jibóia,15,,-16.0925,-44.85
br,jiboia,Jibóia,17,,-7.033333,-38.066667
br,jiboia,Jibóia,27,,-22.825833,-47.8675
br,jiboia,Jibóia,28,,-9.966667,-37.183333
br,jiju,Jiju,04,,-4.986944,-59.336389
br,jiju,Jiju,25,,3.533333,-60.366667
br,jimirim,Jimirim,15,,-21.8,-45.966667
br,jinipabu,Jinipabú,05,,-12.7,-38.116667
br,ji-parana,Ji-Paraná,24,100128,-10.833333,-61.966667
br,jipuba,Jipuba,13,,-2.066667,-44.666667
br,jiquia do cabuta,Jiquiá do Cabutã,02,,-9.616667,-36.433333
br,jiquia do fogo,Jiquiá do Fogo,02,,-9.883333,-36.166667
br,jiquia do sal,Jiquiá do Sal,02,,-9.683333,-36.383333
br,jiquia do sul,Jiquiá do Sul,02,,-9.683333,-36.383333
br,jiquiazinho,Jiquiàzinho,02,,-9.716667,-36.366667
br,jiquie,Jiquié,05,,-13.85,-40.083333
br,jiqui,Jiqui,05,,-13.45,-41.1
br,jiqui,Jiqui,22,,-5.916667,-35.183333
br,jiquirica,Jiquiriçá,05,,-13.233333,-39.6
br,jiquirica,Jiquiricá,06,,-4.65,-38.9
br,jiquiri,Jiquiri,13,,-3.056111,-42.140556
br,jiquitaia,Jiquitaia,05,,-17.25,-39.5
br,jirau de capim,Jirau de Capim,17,,-7.283333,-37
br,jirau de pedra,Jirau de Pedra,05,,-13.683333,-40.283333
br,jirau do ponciano,Jirau do Ponciano,02,,-9.916667,-36.833333
br,jirau,Jirau,05,,-10.216667,-43.466667
br,jirau,Jirau,06,,-4.65,-37.95
br,jirau,Jirau,17,,-7.433333,-36.566667
br,jirau,Jirau,20,,-8.85,-42.266667
br,jirau,Jirau,24,,-9.433333,-64.683333
br,jirau,Jirau,30,,-8.85,-37.383333
br,jiribatuba,Jiribatuba,05,,-13.066667,-38.783333
br,jirua,Jiruá,23,,-28.033333,-54.35
br,jitauna,Jitaúna,05,11619,-14.016667,-39.95
br,jitirana,Jitirana,02,,-9.866667,-36.216667
br,jitirana,Jitirana,05,,-12.133333,-41.35
br,jitirana,Jitirana,06,,-6.666667,-39.9
br,jitirana,Jitirana,17,,-7.233333,-38.033333
br,jitirana,Jitirana,20,,-6.366667,-41.4
br,jitirana,Jitirana,30,,-8.916667,-36.2
br,j. m. peres,J. M. Péres,29,,-18.233333,-51.966667
br,joacaba,Joaçaba,26,23516,-27.166667,-51.5
br,joaci,Joaci,04,,-7.733333,-70.433333
br,joacuba,Joaçuba,08,,-18.4,-40.633333
br,joacy,Joacy,04,,-7.733333,-70.433333
br,joahyma,Joahyma,15,,-16.65,-41.033333
br,joaia,Joáia,26,,-27.266667,-48.7
br,joaima,Joaíma,15,9022,-16.65,-41.033333
br,joa,Joá,06,,-3.883333,-39.883333
br,joa,Joá,18,,-23.4107,-49.8662
br,joa,Joá,21,,-23.003951,-43.288897
br,joa,Joá,30,,-8.316667,-39.4
br,joana coeli,Joana Coeli,16,,-1.966667,-49.383333
br,joana fraga,Joana Fraga,11,,-23.75,-54.833333
br,joana,Joana,05,,-13.066667,-39.766667
br,joana m. fraga,Joana M. Fraga,11,,-23.75,-54.833333
br,joana peres,Joana Peres,16,,-3.3,-49.7
br,joanesia,Joanésia,15,,-19.2,-42.666667
br,joanes,Joanes,16,,-.85,-48.516667
br,joaneta,Joaneta,23,,-29.45,-51.116667
br,joanica,Joanica,05,,-17.5,-39.45
br,joanico,Joanico,04,,-3.866667,-66.383333
br,joanina,Joanina,05,,-14.7,-41.666667
br,joaninha,Joaninha,05,,-14.7,-41.666667
br,joaninha,Joaninha,06,,-5.816667,-40.433333
br,joaninha,Joaninha,13,,-3.136667,-43.341944
br,joanisdorf,Joanisdorf,18,,-25.738611,-49.765278
br,joannes,Joannes,16,,-.85,-48.516667
br,joannopolis,Joannopolis,27,,-22.933333,-46.283333
br,joanopolis,Joanópolis,05,,-14.6,-41.133333
br,joanopolis,Joanópolis,27,11772,-22.933333,-46.283333
br,joa nova,Joá Nova,05,,-11.966667,-43.366667
br,joa novo,Joá Novo,05,,-11.966667,-43.366667
br,joan ribeiro,Joãn Ribeiro,31,,-10.583333,-46.166667
br,joao alfredo,João Alfredo,27,,-22.666667,-47.75
br,joao alfredo,João Alfredo,30,11334,-7.866667,-35.583333
br,joao alves,João Alves,29,,-15.7,-48.35
br,joao amaro,João Amaro,05,,-12.766667,-40.366667
br,joao anacleto diniz,João Anacleto Diniz,15,,-18.55,-48.966667
br,joao arregui,João Arrégui,23,,-29.483333,-56.716667
br,joao arreguy,João Arreguy,23,,-29.483333,-56.716667
br,joao augusto,João Augusto,04,,-7.2,-72.166667
br,joao balbino,João Balbino,15,,-21.083333,-46.6
br,joao batista,João Batista,08,,-20.383333,-40.75
br,joao bento,João Bento,30,,-7.533333,-39.5
br,joao betti,João Betti,23,,-29.0506,-51.1519
br,joao borges,João Borges,23,,-30.016667,-50.966667
br,joao branco,João Branco,29,,-17.466667,-52.233333
br,joao camara,João Câmara,22,19886,-5.533333,-35.8
br,joao carmelo,João Carmelo,29,,-17.716667,-51.883333
br,joao carmo,João Carmo,15,,-20.75,-47.1
br,joao cavalcanti,João Cavalcânti,16,,-5.616667,-54.45
br,joao chaves,João Chaves,15,,-16.75,-41.016667
br,joao-coelho,João-Coelho,16,,-1.266667,-48.183333
br,joao correa,João Correa,23,,-29.6,-50.916667
br,joao correia,João Correia,02,,-8.966667,-37.716667
br,joao correia,João Correia,05,,-13.266667,-41.616667
br,joao correia,João Correia,23,,-29.6,-50.916667
br,joao cristal,João Cristal,15,,-16.033333,-40.166667
br,joao da costa,João da Costa,06,,-6.333333,-40.666667
br,joao de deus,João de Deus,20,,-3.457222,-42.124722
br,joao de melo,João de Melo,15,,-20.4,-46.233333
br,joao de morais,João de Morais,11,,-17.55,-55.35
br,joao de sousa,João de Sousa,13,,-3.942222,-42.996944
br,joao de souza,João de Souza,20,,-6.8,-41.366667
br,joao dias,João Dias,05,,-13.55,-41.8
br,joao dias,João Dias,22,,-6.266667,-37.8
br,joao diogo,João Diogo,18,,-25.233333,-53.083333
br,joao do couro,João do Couro,30,,-7.583333,-37.283333
br,joao domingos,João Domingos,29,,-16.216667,-51.766667
br,joao dorneles,João Dorneles,23,,-30.15,-56.35
br,joao dutra,João Dutra,29,,-16.3,-48.8
br,joao estevao,João Estêvão,13,,-5.583333,-43.333333
br,joao eugenio,João Eugênio,18,,-25.6,-49.616667
br,joao fabricio,João Fabrício,15,,-20.566667,-46.516667
br,joao fagundes,João Fagundes,23,,-30.066667,-57.333333
br,joao felipe,João Felipe,24,,-10.65,-63.65
br,joao felipe rodriguez,João Felipe Rodriguez,16,,-2.883333,-50.366667
br,joao fereira,João Fereira,17,,-7.516667,-37.666667
br,joao ferreira,João Ferreira,14,,-14.266667,-57.333333
br,joao filipe,João Filipe,24,,-10.65,-63.65
br,joao franca,João França,16,,-2.25,-53.583333
br,joao gama,João Gama,25,,4.116667,-60.716667
br,joao gil,João Gil,15,,-16.35,-41.5
br,joao gomes,João Gomes,11,,-19.9,-54.45
br,joao gomes,João Gomes,22,,-6.266667,-38.1
br,joao goncalo,João Gonçalo,20,,-5.666667,-41.133333
br,joao goncalves,João Goncalves,06,,-4.683333,-38.783333
br,joao grama,João Grama,25,,4.116667,-60.716667
br,joao gualberto,João Gualberto,15,,-17.233333,-40.733333
br,joao guaxo,João Guaxo,15,,-16.8,-40.733333
br,joao gurgel,João Gurgel,16,,-7.883333,-49.866667
br,joao honorio,João Honório,21,,-22.1525,-43.988333
br,joao,João,15,,-19.833333,-43.133333
br,joao,João,16,,-2.766667,-50.983333
br,joao-joao,João-João,17,,-7.316667,-37.35
br,joao jorge,João Jorge,02,,-9.333333,-37.266667
br,joao leite,João Leite,14,,-15.933333,-56.7
br,joao lisboa,João Lisboa,13,,-5.366667,-47.283333
br,joao lopes de baixo,João Lopes de Baixo,06,,-5.033333,-40.05
br,joao lopes de cima,João Lopes de Cima,06,,-5.016667,-40.083333
br,joao lopes,João Lopes,03,,-.4,-52.45
br,joao lopes,João Lopes,15,,-17.15,-41.216667
br,joao manuel,João Manuel,15,,-15.5,-43.683333
br,joao marques,João Marques,27,,-21.1,-47.483333
br,joao martins,João Martins,06,,-3.566667,-40.25
br,joao martins,João Martins,15,,-17.266667,-44.583333
br,joao martins,João Martins,20,,-7.15,-43.9
br,joao martins,João Martins,23,,-30.3,-55.583333
br,joao massape,João Massapê,06,,-5.866667,-40.5
br,joao matheus,João Matheus,16,,-5.65,-48.166667
br,joao matos,João Matos,15,,-16.766667,-41.183333
br,joao miguel,João Miguel,15,,-20.783333,-46.783333
br,joao monlevade,João Monlevade,15,72763,-19.833333,-43.133333
br,joao morais,João Morais,11,,-17.55,-55.35
br,joao nea,João Néa,18,,-23.218066,-49.86396
br,joao neiva,João Neiva,08,10383,-19.75,-40.4
br,joao pais,João Pais,11,,-18.483333,-56.483333
br,joao para,João Pará,05,,-13.733333,-41.766667
br,joao paulo,João Paulo,14,,-14.583333,-56.733333
br,joao paz,João Paz,20,,-3.893889,-42.120833
br,joao pedro,João Pedro,05,,-13.883333,-40.6
br,joao pedro,João Pedro,14,,-16.35,-55.95
br,joao pedro,João Pedro,15,,-21.066667,-46.95
br,joao pedro,João Pedro,22,,-5.516667,-37.833333
br,joao pereira,João Pereira,06,,-6.466667,-38.833333
br,joao peres,João Peres,13,,-2.929167,-41.910833
br,joao pessoa,João Pessôa,04,,-6.666667,-69.866667
br,joao pessoa,João Pessoa,05,,-12.25,-41.6
br,joao pessoa,João Pessoa,08,,-21.066667,-41.366667
br,joao pessoa,João Pessoa,16,,-1.116667,-47.616667
br,joao pessoa,João Pessoa,17,650896,-7.116667,-34.866667
br,joao pessoa,João Pessoa,20,,-3.9,-42.7
br,joao pessoa,João Pessoa,26,,-26.45,-48.933333
br,joao pinheiro,João Pinheiro,14,,-10.916667,-51.383333
br,joao pinheiro,João Pinheiro,15,28083,-17.75,-46.166667
br,joao pinto,João Pinto,15,,-19.266667,-41.516667
br,joao pires,João Pires,15,,-16.35,-40.6
br,joao queiroz,João Queiroz,11,,-20.5,-56.116667
br,joao queiroz,João Queiroz,15,,-18.566667,-49.283333
br,joao ramalho,João Ramalho,27,,-22.266667,-50.8
br,joao ramos,João Ramos,15,,-18.383333,-41.416667
br,joao rezende,João Rezende,11,,-19.75,-54.383333
br,joao ribeiro,João Ribeiro,05,,-11.116667,-45.466667
br,joao ribeiro,João Ribeiro,14,,-15.616667,-54.083333
br,joao ribeiro,João Ribeiro,15,,-20.683333,-44.066667
br,joao ribeiro,João Ribeiro,31,,-10.583333,-46.166667
br,joao rodigues,João Rodigues,23,,-29.933333,-52.2
br,joao rodrigues,João Rodrigues,05,,-12.4,-44.85
br,joao rodrigues,João Rodrigues,20,,-5.633333,-42.316667
br,joao rodrigues,João Rodrigues,23,,-29.933333,-52.2
br,joao roque,João Roque,15,,-17.433333,-41.25
br,joao silva,João Silva,31,,-10.516667,-49.166667
br,joao soares,João Soares,05,,-9.283333,-41.65
br,joao soares,João Soares,06,,-5.283333,-40.533333
br,joao surra,João Surrá,18,,-24.702208,-48.660124
br,joao s. vieira,João S. Vieira,23,,-31.116667,-55.116667
br,joao vaqueiro,João Vaqueiro,15,,-16.233333,-40.616667
br,joao vaz,João Vaz,05,,-13.7,-41.6
br,joao viana,João Viana,15,,-14.666667,-44.15
br,joao vieira de castro,João Vieira de Castro,15,,-19.066667,-48.7
br,joao vieira,João Vieira,02,,-9.866667,-36.516667
br,joao vieira,João Vieira,05,,-11.133333,-38.983333
br,joao vieira,João Vieira,06,,-7.65,-39.083333
br,joao vieira,João Vieira,11,,-19.766667,-54.733333
br,joaozinho,Joãozinho,06,,-7.65,-38.966667
br,joaquim adolfo,Joaquim Adolfo,15,,-20.933333,-46.066667
br,joaquim alves,Joaquim Alves,15,,-20.766667,-45.633333
br,joaquim antonio,Joaquim Antônio,14,,-10.433333,-60.45
br,joaquim antonio,Joaquim Antônio,15,,-20.683333,-45.55
br,joaquim cardoso,Joaquim Cardoso,25,,3.766667,-60.466667
br,joaquim coimbra,Joaquim Coimbra,15,,-17.266667,-41.316667
br,joaquim correia,Joaquim Correia,22,,-6.116667,-38.316667
br,joaquim davi,Joaquim Davi,15,,-19.483333,-49.866667
br,joaquim de jesus,Joaquim de Jesus,14,,-17.833333,-53.2
br,joaquim egidio,Joaquim Egídio,27,,-22.883333,-46.983333
br,joaquim felicio,Joaquim Felício,15,,-17.766667,-44.15
br,joaquim ferreira,Joaquim Ferreira,15,,-17.466667,-41.7
br,joaquim gomes,Joaquim Gomes,02,14808,-9.2,-35.716667
br,joaquim hilario,Joaquim Hilário,15,,-20.583333,-46.416667
br,joaquim i. de jesus,Joaquim I. de Jesus,14,,-17.833333,-53.2
br,joaquim,Joaquim,06,,-5.05,-39.55
br,joaquim j. riria,Joaquim J. Riria,14,,-17.85,-53.2
br,joaquim leite,Joaquim Leite,21,,-22.35,-44.25
br,joaquim luis,Joaquim Luís,15,,-19,-48.483333
br,joaquim messias,Joaquim Messias,15,,-20.65,-46.133333
br,joaquim murtinho,Joaquim Murtinho,18,,-24.4,-49.866667
br,joaquim nabuco,Joaquim Nabuco,30,9965,-8.616667,-35.533333
br,joaquim nogueira,Joaquim Nogueira,15,,-17.8,-40.8
br,joaquim norberto,Joaquim Norberto,15,,-20.25,-45.316667
br,joaquim nunes,Joaquim Nunes,08,,-20.3,-41.583333
br,joaquim pereira dos santos,Joaquim Pereira dos Santos,15,,-17.066667,-41.383333
br,joaquim pires,Joaquim Pires,20,,-3.509167,-42.195556
br,joaquim prata,Joaquim Prata,15,,-16.8,-40.616667
br,joaquim rocha,Joaquim Rocha,15,,-16.533333,-40.633333
br,joaquim rodrigues,Joaquim Rodrigues,15,,-20.2,-45.433333
br,joaquim soares,Joaquim Soares,15,,-21.133333,-46.9
br,joaquim tavora,Joaquim Távora,06,,-5.966667,-38.816667
br,joaquim tavora,Joaquim Távora,18,,-23.504,-49.9239
br,joaquim tavora,Joaquim Távora,20,,-3.466667,-42.366667
br,joaquim tropeiro,Joaquim Tropeiro,15,,-17.433333,-40.516667
br,joaquim wolff,Joaquim Wolff,26,,-26.433333,-49.066667
br,joaquim wolf,Joaquim Wolf,26,,-26.433333,-49.066667
br,joaquina,Joaquina,05,,-12.333333,-37.916667
br,joarizinho,Joarizinho,16,,-4.366667,-49.516667
br,joatinga,Joatinga,21,,-23.013009,-43.288439
br,joatuba,Joatuba,08,,-19.866667,-40.95
br,joazeirinho,Joàzeirinho,17,,-7.066667,-36.583333
br,joazeiro,Joazeiro,05,,-9.5,-40.5
br,joazeiro,Joázeiro,06,,-4.45,-38.416667
br,joazeiro,Joazeiro,17,,-7.066667,-36.583333
br,jocelin bitencourt,Jocelin Bitencourt,11,,-21.5,-54.266667
br,jococa,Jococa,16,,-7.25,-53.683333
br,joeirana,Joeirana,15,,-18.05,-41.1
br,joe,Joé,15,,-21.016667,-46.366667
br,joelho,Joelho,27,,-24.669352,-48.144771
br,joffily,Jóffily,17,,-7.066667,-36.05
br,joffre,Joffre,24,,-13.5,-60.55
br,jofre,Jofre,24,,-13.5,-60.55
br,jogo da bola,Jôgo da Bola,05,,-12.933333,-39.216667
br,johannisdorf,Johannisdorf,18,,-25.738611,-49.765278
br,joia,Jóia,23,,-28.65,-54.133333
br,joinvile,Joinvile,26,,-26.3,-48.833333
br,joinville,Joinville,26,461312,-26.3,-48.833333
br,jojoca,Jojoca,16,,-7.25,-53.683333
br,jonas carvalho,Jonas Carvalho,14,,-16.15,-54
br,jonatas,Jônatas,11,,-20.05,-54.75
br,jonia,Jônia,26,,-27.266667,-48.7
br,jordania,Jordânia,15,,-15.9,-40.183333
br,jordao,Jordão,05,,-11.816667,-42.616667
br,jordao,Jordão,06,,-3.666667,-40.483333
br,jordao,Jordão,16,,-1.483333,-47.35
br,jordao,Jordão,22,,-5.966667,-37.566667
br,jordao,Jordão,26,,-26.993,-49.0627
br,jordaozinho,Jordãozinho,18,,-25.483333,-51.45
br,jorge cury,Jorge Cury,14,,-16.283333,-54.95
br,jorge,Jorge,17,,-7.55,-37.766667
br,jorge,Jorge,20,,-7.8,-41
br,jorge lima,Jorge Lima,15,,-21.083333,-46.85
br,jorge mendes,Jorge Mendes,15,,-17.366667,-41.35
br,jorge teixeira,Jorge Teixeira,15,,-21.966667,-45.866667
br,jose alencar,José Alencar,06,,-6.4,-39.15
br,jose alexandre,José Alexandre,05,,-14.716667,-41.383333
br,jose alves da silva,José Alves da Silva,15,,-20.066667,-41.6
br,jose alves de cima,José Alves de Cima,05,,-14.233333,-41.25
br,jose alves,José Alves,15,,-16.05,-40.5
br,jose amaro,José Amaro,21,,-21.866667,-41.216667
br,jose antonio,José Antônio,15,,-16.25,-39.933333
br,jose augusto,José Augusto,15,,-20.9,-45.633333
br,jose baio,José Baio,05,,-12.2,-40.883333
br,jose barbosa,José Barbosa,15,,-20.233333,-46.366667
br,jose batista,José Batista,15,,-21.366667,-46.35
br,jose bento,José Bento,11,,-22.033333,-53.233333
br,jose bento,José Bento,30,,-8.483333,-36.4
br,jose boiteux,José Boiteux,26,,-26.959168,-49.627078
br,jose bonifacio,José Bonifacio,23,,-27.633333,-52.283333
br,jose bonifacio,José Bonifácio,24,,-12.166667,-60.083333
br,jose bonifacio,José Bonifácio,27,27828,-21.05,-49.683333
br,jose borba,José Borba,15,,-21.016667,-46.616667
br,jose borges,José Borges,15,,-16.333333,-40.566667
br,jose bravo,José Bravo,15,,-17.066667,-41.116667
br,jose calhau,José Calhau,15,,-17.666667,-41.95
br,jose candido,José Cândido,15,,-20.266667,-45.4
br,jose cardoso,José Cardoso,08,,-20.6,-41.6
br,jose cariango,José Cariango,15,,-16.483333,-40.716667
br,jose carmelitano,José Carmelitano,15,,-20.383333,-46.833333
br,jose carvalho,José Carvalho,14,,-17.183333,-54.116667
br,jose ceconelo,José Ceconelo,23,,-29.0916,-51.1846
br,jose coco,José Coco,31,,-7.516667,-48.383333
br,jose coimbra,José Coimbra,15,,-17.016667,-41.483333
br,jose coroado,José Coroado,14,,-14.3,-54.7
br,jose costa,José Costa,15,,-20.6,-45.85
br,jose couto,José Couto,05,,-16.4,-39.866667
br,jose da costa,José da Costa,30,,-8.766667,-35.316667
br,jose da cruz,José da Cruz,14,,-16.25,-58.583333
br,jose da lage,José da Lage,02,,-9,-36.033333
br,jose da penha,José da Penha,22,,-6.316667,-38.283333
br,jose da silva,José da Silva,17,,-7.316667,-36.483333
br,jose de alencar,José de Alencar,06,,-6.4,-39.15
br,jose de almeida,José de Almeida,15,,-20.0008,-43.8869
br,jose de cruz,José de Cruz,14,,-16.25,-58.583333
br,jose de delmiro,José de Delmiro,15,,-16.9,-40.733333
br,jose de freitas,José de Freitas,20,21433,-4.75,-42.583333
br,jose dias,José Dias,31,,-6.516667,-48.633333
br,jose eusebio,José Eusébio,15,,-20.816667,-46.883333
br,jose fernandes,José Fernandes,18,,-24.75,-48.983333
br,jose ferreira,José Ferreira,15,,-20.95,-46.8
br,jose figueiredo,José Figueiredo,15,,-15.616667,-46.633333
br,jose figueredo,José Figueredo,15,,-15.616667,-46.633333
br,jose filipe rodrigues,José Filipe Rodrigues,16,,-2.883333,-50.366667
br,josefina,Josefina,15,,-15.183333,-46.583333
br,jose franca,José França,11,,-18.866667,-54.433333
br,jose geraldo,José Geraldo,05,,-16.15,-39.233333
br,jose gomez,José Gomez,30,,-7.516667,-39.566667
br,jose goncalo,José Gonçalo,05,,-10.783333,-43.716667
br,jose goncalves,José Gonçalves,05,,-14.733333,-40.716667
br,jose goncalves,José Goncalves,15,,-16.016667,-40.416667
br,jose grande,José Grande,15,,-17.816667,-40.816667
br,jose gregorio,José Gregório,29,,-16.116667,-51.75
br,jose grosso,José Grosso,05,,-16.183333,-39.833333
br,jose hipolito,José Hipólito,15,,-21.166667,-46.933333
br,jose inacio,José Inácio,15,,-21.566667,-45.566667
br,jose jacinto,José Jacinto,15,,-21.9,-45.85
br,jose,José,04,,.816667,-66.883333
br,jose,José,29,,-17.583333,-52.85
br,jose,José,30,,-8.483333,-36.4
br,jose lacerda,José Lacerda,18,,-24.516667,-50.85
br,joselandia,Joselândia,14,,-16.533333,-56.2
br,joselandia,Joselândia,15,,-20.8,-43.583333
br,jose laurentino,José Laurentino,04,,-8.633333,-60.683333
br,jose laurindo,José Laurindo,04,,-8.633333,-60.683333
br,jose lopes,José Lopes,15,,-17.216667,-41.6
br,jose lourenco,José Lourenço,17,,-6.683333,-38.233333
br,jose luis,José Luís,20,,-9.3,-42.9
br,jose luiz,José Luiz,20,,-9.3,-42.9
br,jose marcelino,José Marcelino,05,,-13,-39.383333
br,jose maria de freitas,José Maria de Freitas,23,,-29.966667,-57.3
br,jose maria,José Maria,23,,-29.966667,-57.3
br,jose mariano,José Mariano,15,,-17.783333,-40.25
br,jose mariano,José Mariano,21,,-22.866667,-42.95
br,jose mariano,José Mariano,30,,-8.483333,-35.4
br,jose marques,José Marques,15,,-17.466667,-41.266667
br,jose martins,José Martins,15,,-16.95,-40.866667
br,jose meira,José Meira,15,,-16.583333,-40.416667
br,jose mendes,José Mendes,15,,-17.95,-41.7
br,jose menino,José Menino,11,,-19.2,-54.283333
br,jose miguel,José Miguel,15,,-21.483333,-46.466667
br,jose mineiro,José Mineiro,31,,-7.2,-48.333333
br,jose moco,José Mocó,05,,-16.383333,-39.85
br,jose moreira,José Moreira,06,,-5.883333,-39.15
br,jose moreira,José Moreira,15,,-17.516667,-41.316667
br,jose moronga,José Moronga,15,,-20.65,-45.366667
br,jose murtinho,José Murtinho,18,,-24.4,-49.866667
br,jose nino,José Nino,11,,-19.683333,-53.75
br,josenopolis,Josenópolis,15,,-16.566667,-42.533333
br,jose nunes,José Nunes,15,,-17.65,-41.966667
br,jose nunes,José Nunes,17,,-7.116667,-36.45
br,jose olimpio de castro,José Olímpio de Castro,15,,-20.6,-45.55
br,jose oliveira,José Oliveira,15,,-17.366667,-41.066667
br,jose otavio,José Otávio,23,,-31.283333,-54.116667
br,jose pacheco,José Pacheco,15,,-17.916667,-41.75
br,jose paulo,José Paulo,13,,-3.533333,-42.716667
br,jose pedro,José Pedro,15,,-21.583333,-46.383333
br,jose pedro,José Pedro,29,,-12.883333,-49.05
br,jose pereira,José Pereira,11,,-23.4,-55.25
br,jose pinheiro,José Pinheiro,27,,-20.216667,-47.633333
br,jose pinto,José Pinto,28,,-10.566667,-37.733333
br,jose pio,José Pio,15,,-21.35,-46.316667
br,jose preto,José Prêto,08,,-18,-40.8
br,jose primo,José Primo,15,,-15.05,-42.9
br,jose quito,José Quito,14,,-17.966667,-53.683333
br,jose raydam,José Raydam,15,,-18.2,-42.5
br,jose ribeiro,José Ribeiro,05,,-16.35,-39.166667
br,jose rocha,José Rocha,15,,-20.333333,-46.483333
br,jose rodrigues,José Rodrigues,06,,-4.416667,-37.816667
br,jose rodrigues,José Rodrigues,17,,-6.783333,-37.783333
br,jose roiz,Josê Roiz,05,,-17.366667,-40.333333
br,jose roque,José Roque,15,,-20.316667,-45.466667
br,jose rosa,José Rosa,15,,-20.7,-45.55
br,jose rosa,José Rosa,27,,-21.4,-47.133333
br,jose santos,José Santos,05,,-12.183333,-41.116667
br,jose santos,José Santos,16,,-5.583333,-56.033333
br,jose sebastiao,José Sebastião,15,,-20.266667,-46.933333
br,jose sebastiao leite,José Sebastião Leite,15,,-20.666667,-46.233333
br,jose segundo,José Segundo,15,,-20.75,-45.65
br,jose silva,José Silva,15,,-21.5,-45.983333
br,jose tartano,José Tartano,23,,-28.916667,-54.65
br,jose teixeira,José Teixeira,15,,-20.433333,-45.566667
br,jose teodoro,José Teodoro,27,,-22.183333,-51.2
br,jose theodoro,José Theodoro,27,,-22.183333,-51.2
br,jose tome,José Tomé,15,,-17.183333,-41.1
br,jose valerio,José Valério,15,,-20.5,-45.4
br,jose velho,José Velho,02,,-9.55,-36.516667
br,jose vermelho,José Vermelho,15,,-16.95,-40.883333
br,jose viana,José Viana,15,,-16.433333,-41.933333
br,jose vicente,José Vicente,15,,-20.583333,-45.116667
br,jose vieira,José Vieira,22,,-5.666667,-35.6
br,jose vilela,José Vilela,27,,-21.05,-47.216667
br,josezinho,Josèzinho,16,,-2.816667,-51.9
br,josias,Josias,15,,-18.35,-48.6
br,josina,Josina,03,,.166667,-51.216667
br,jos,Jós,15,,-21.016667,-46.366667
br,jouquarinha,Jouquarinha,14,,-15.2,-56.933333
br,joviania,Joviânia,29,,-17.816667,-49.5
br,joviniano santos,Joviniano Santos,15,,-15.3,-43.333333
br,jovino sales,Jovino Sales,15,,-21.4,-46.033333
br,j. quirino,J. Quirino,29,,-18.7,-51.816667
br,j.r. vilela,J.R. Vilela,14,,-15.616667,-54.083333
br,juaba,Juaba,16,,-2.383333,-49.55
br,juacaba,Juaçaba,26,,-27.166667,-51.5
br,juacema,Juacema,05,,-10.2,-40.216667
br,jua,Juá,05,,-11.166667,-43.166667
br,jua,Juá,06,,-3.883333,-39.883333
br,jua,Juá,17,,-7.116667,-37.016667
br,jua,Juá,18,,-23.4107,-49.8662
br,jua,Juá,20,,-6.683333,-41.45
br,jua,Juá,23,,-29.133333,-50.866667
br,jua,Juá,27,,-21.166667,-47.283333
br,jua,Juá,30,,-7.95,-39.216667
br,juan borges,Juan Borges,03,,-.383333,-52.233333
br,jua novo,Juá Novo,05,,-11.966667,-43.366667
br,juarana,Juarana,05,,-17.716667,-39.566667
br,juarez,Juarez,17,,-6.816667,-38.183333
br,juarez tavora,Juarez Távora,17,,-7.166667,-35.566667
br,juari,Juari,04,,-3.0475,-61.589444
br,juarina,Juarina,31,,-8.080556,-49.018333
br,juataba,Juataba,06,,-5.083333,-39.033333
br,juatama,Juatama,06,,-5.083333,-39.033333
br,juatuba,Juatuba,08,,-19.866667,-40.95
br,juatuba,Juatuba,15,24551,-19.95,-44.333333
br,juazeirinho,Juàzeirinho,17,8846,-6.816667,-38.05
br,juazeirinho,Juàzeirinho,22,,-5.192222,-37.368333
br,juazeiro de baixo,Juàzeiro de Baixo,06,,-5.133333,-38.566667
br,juazeiro do norte,Juazeiro do Norte,06,225234,-7.2,-39.333333
br,juazeiro grande,Juàzeiro Grande,20,,-8.05,-41.35
br,juazeiro grande,Juàzeiro Grande,30,,-7.65,-38.716667
br,juazeiro,Juàzeiro,05,137807,-9.5,-40.5
br,juazeiro,Juazeiro,06,,-2.95,-40.383333
br,juazeiro,Juàzeiro,15,,-19.75,-44.8
br,juazeiro,Juàzeiro,17,,-7.066667,-36.583333
br,juazeiro,Juazeiro,20,,-5.1,-41.083333
br,juazeiro,Juazeiro,22,,-5.478611,-36.520556
br,juazeiro,Juàzeiro,30,,-7.55,-37.416667
br,jubaia,Jubaia,06,,-4.05,-38.716667
br,jubai,Jubaí,15,,-19.9,-47.683333
br,jubim,Jubim,16,,-.85,-48.516667
br,jubiro,Jubiro,16,,-.8,-48.533333
br,juca azevedo,Juca Azevêdo,21,,-21.166667,-41.733333
br,juca batista,Juca Batista,18,,-23.2277,-50.2474
br,juca,Juca,04,,1.116667,-67.233333
br,juca,Jucá,06,,-4.066667,-40.4
br,juca,Jucá,13,,-3.566667,-44.583333
br,juca,Juca,17,,-7.133333,-37.616667
br,juca,Jucá,22,,-5.320278,-36.314167
br,juca,Jucá,30,,-7.75,-35.366667
br,juca leite,Juca Leite,27,,-23.049955,-47.635238
br,jucal,Jucal,22,,-5.209444,-37.398056
br,juca modesto,Juca Modesto,15,,-20.733333,-45.616667
br,juca neto,Juca Neto,21,,-21.35,-41.95
br,juca novo,Jucá Novo,05,,-11.966667,-43.366667
br,juca pedroso,Juca Pedroso,15,,-20.466667,-46.816667
br,juca pereira,Juca Pereira,15,,-17.566667,-41.7
br,juca polvora,Juca Pólvora,15,,-10.566667,-48.066667
br,jucara,Juçara,13,,-1.883333,-44.65
br,jucara,Juçara,18,,-23.6,-52.466667
br,jucara,Juçara,29,,-15.883333,-50.85
br,jucaral,Juçaral,13,,-3.040833,-43.053333
br,jucaral,Juçaral,21,,-22.95,-44.266667
br,jucaral,Juçaral,30,,-8.266667,-35.066667
br,jucari,Juçari,05,,-15.2,-39.533333
br,juca santana,Juca Santana,15,,-18.7,-48.116667
br,jucas,Jucás,06,14822,-6.533333,-39.533333
br,jucas,Jucás,13,,-9.966667,-46.1
br,jucatan,Jucatán,16,,-6.466667,-54.05
br,jucati,Jucati,30,,-8.7,-36.45
br,juca vaqueiro,Juca Vaqueiro,15,,-17.216667,-41.1
br,jucazinho,Jucàzinho,17,,-7.766667,-36
br,jucelandia,Jucelândia,29,,-15.333333,-51.316667
br,juciape,Juciape,05,,-13.516667,-41.6
br,jucuitiba,Jucuitiba,27,,-23.927482,-47.069543
br,jucu,Jucu,08,,-20.433333,-40.333333
br,jucuma,Jucumã,06,,-2.816667,-40.416667
br,jucuri,Jucuri,22,,-5.133333,-35.966667
br,jucurucu,Jucurucu,05,,-16.866667,-39.916667
br,jucurupava,Jucurupava,27,,-23.596273,-47.518138
br,jucururi,Jucururi,22,,-5.218611,-37.495278
br,jucururu,Jucururu,05,,-16.866667,-39.916667
br,jucurutu,Jucurutu,05,,-12.216667,-42.95
br,jucurutu,Jucurutu,22,11535,-6.033333,-37.016667
br,jucurutu,Jucurutu,30,,-8.583333,-38.716667
br,juerana,Juerana,05,,-12.95,-38.616667
br,juie,Juié,04,,-4.791111,-62.452222
br,juiracu,Juiraçu,15,,-19.916667,-42.8
br,juissara,Juissara,13,,-5.933333,-46.25
br,juiz de fora,Juiz de Fora,15,470201,-21.751667,-43.352778
br,juiz,Juiz,06,,-7.066667,-38.983333
br,julia,Júlia,04,,-1.616667,-67.983333
br,juliana,Juliana,06,,-3.016667,-40.116667
br,juliania,Juliânia,27,,-21.866667,-50.366667
br,juliao,Julião,05,,-17.35,-40.4
br,juliao,Julião,20,,-8.766667,-41.616667
br,juliao,Julião,30,,-7.95,-40.516667
br,juliapolis,Juliapolis,27,,-21.866667,-50.366667
br,julio costa,Júlio Costa,15,,-20.666667,-46.516667
br,julio de carvalho,Júlio de Carvalho,15,,-16.55,-40.133333
br,julio de castilhos,Júlio de Castilhos,23,16487,-29.233333,-53.683333
br,julio do castilhos,Julio do Castilhos,23,,-29.233333,-53.683333
br,julio ferreira,Júlio Ferreira,15,,-17.883333,-40.5
br,julio,Júlio,23,,-29.9564,-50.788
br,julio laender,Júlio Laender,15,,-17.883333,-41.383333
br,julio marques,Júlio Marques,23,,-29.1,-54.166667
br,julio mesquita,Júlio Mesquita,27,,-22.016667,-49.8
br,jumas,Jumas,04,,-6.887778,-62.679444
br,jumas quadros,Jumas Quadros,04,,-6.887778,-62.679444
br,jumirim,Jumirim,18,,-23.716667,-52.25
br,jumirim,Jumirim,27,,-23.084831,-47.78461
br,juncal,Juncal,15,,-22.75,-46.266667
br,juncal,Juncal,27,,-23.67952,-47.525306
br,juncao,Junção,23,,-32.066667,-52.15
br,junccao,Juncção,23,,-32.066667,-52.15
br,junco do serido,Junco do Seridó,17,,-7,-36.716667
br,junco grande,Junco Grande,05,,-13.85,-42.433333
br,junco,Junco,05,,-11.333333,-40.15
br,junco,Junco,06,,-2.983333,-40.033333
br,junco,Junco,17,,-7,-36.716667
br,junco,Junco,20,,-3.9325,-42.265
br,junco,Junco,22,,-6.066667,-37.5
br,junco,Junco,28,,-10.933333,-38.183333
br,junco,Junco,30,,-7.883333,-35.916667
br,jundiacanga,Jundiacanga,27,,-23.549963,-47.693806
br,jundiahy,Jundiahy,27,,-23.183333,-46.866667
br,jundiai do sul,Jundiaí do Sul,18,,-23.4363,-50.2474
br,jundiai,Jundiaí,18,,-23.45,-50.283333
br,jundiai,Jundiaí,22,,-5.883333,-35.366667
br,jundiai,Jundiaí,27,321595,-23.183333,-46.866667
br,jundiaizinho,Jundiaizinho,27,,-23.231111,-46.613611
br,jundia,Jundiá,02,2267,-8.933333,-35.566667
br,jundia,Jundiá,05,,-12.466667,-40.583333
br,jundia,Jundiá,06,,-5.433333,-39.3
br,jundia,Jundiá,08,,-18.416667,-40.216667
br,jundia,Jundiá,15,,-16.033333,-41.583333
br,jundia,Jundiá,25,,-.183333,-60.683333
br,jundia,Jundia,30,,-7.983333,-36.583333
br,jundiapeba,Jundiapeba,27,,-23.55,-46.25
br,jundiaquara,Jundiaquara,27,,-23.55075,-47.5788
br,junqueira,Junqueira,27,,-20.833333,-49.866667
br,junqueiro,Junqueiro,02,7740,-9.933333,-36.483333
br,junqueiropolis,Junqueirópolis,27,13965,-21.533333,-51.433333
br,jupagua,Jupaguá,05,,-11.816667,-44.333333
br,jupara,Jupará,11,,-23.516667,-55.033333
br,juparana,Juparana,21,,-22.343889,-43.682778
br,jupariteua,Jupariteua,16,,-1.716667,-48.383333
br,jupia,Jupiá,26,,-26.45,-52.666667
br,jupi,Jupi,30,,-8.716667,-36.416667
br,jupiranga,Jupiranga,17,,-7.366667,-35.233333
br,jupy,Jupy,30,,-8.716667,-36.416667
br,juqueri,Juqueri,27,,-23.322628,-46.586935
br,juqueri-mirim,Juqueri-Mirim,27,,-23.35,-46.616667
br,juquery,Juquery,27,,-23.322628,-46.586935
br,juquia-grande,Juquiá-Grande,27,,-23.9771,-47.2265
br,juquia-guacu,Juquia-Guaçu,27,,-23.948401,-47.193355
br,juquia,Juquiá,18,,-25.233333,-52.016667
br,juquia,Juquiá,27,13497,-24.316667,-47.633333
br,juquiquera,Juquiquera,04,,-.283333,-63
br,juquiratiba,Juquiratiba,27,,-22.95,-48.1
br,juquitiba,Juquitiba,27,13961,-23.927482,-47.069543
br,jurace,Juracê,27,,-21.066667,-47.75
br,juraci,Juraci,05,,-13.2,-40.883333
br,juracy,Juracy,05,,-13.2,-40.883333
br,juramento,Juramento,15,,-16.833333,-43.583333
br,juranda,Juranda,18,,-23.264674,-52.505798
br,juraraita,Juraraitá,13,,-2.416667,-44.683333
br,jurara,Jurará,04,,-6.616667,-62.366667
br,jurece,Jurecê,27,,-21.066667,-47.75
br,jureia,Juréia,15,,-21.283333,-46.366667
br,jurema do toco preto,Jurema do Toco Preto,20,,-7.016667,-42.683333
br,jurema,Jurema,02,,-9.466667,-37.916667
br,jurema,Jurema,05,,-11.533333,-38.416667
br,jurema,Jurema,06,,-2.983333,-39.983333
br,jurema,Jurema,15,,-18.616667,-42.783333
br,jurema,Jurema,16,,-6.166667,-49.4
br,jurema,Jurema,17,,-6.733333,-37.7
br,jurema,Jurema,18,,-23.033333,-52.833333
br,jurema,Jurema,20,,-7.066667,-40.966667
br,jurema,Jurema,22,,-5.033333,-37.25
br,jurema,Jurema,27,,-21.366667,-48.566667
br,jurema,Jurema,30,,-7.633333,-37.5
br,juremal,Juremal,05,,-9.733333,-40.366667
br,juremal,Juremal,06,,-5.4,-38.216667
br,juremal,Juremal,22,,-5.083333,-37.55
br,jurema seca,Jurema Sêca,22,,-5.116944,-37.242778
br,jurema torta,Jurema Torta,06,,-6.7,-38.783333
br,jureminha,Jureminha,17,,-7.55,-36.75
br,juriassu,Juriassu,04,,-5.083333,-62.85
br,juricia,Jurícia,04,,-7.8,-65.1
br,juricia labrea,Juricia Labrea,04,,-7.8,-65.1
br,juripiranga,Juripiranga,17,8320,-7.366667,-35.233333
br,juritianha,Juritianha,06,,-2.9,-40
br,juriti,Juriti,15,,-21.6,-45.383333
br,juritis,Juritis,27,,-21.166667,-50.166667
br,juriti velho,Jurití Velho,16,,-2.4,-56.3
br,jurity novo,Jurity Novo,16,,-2.166667,-56.066667
br,jurity velho,Jurity Velho,16,,-2.4,-56.3
br,juruaia,Juruaia,15,,-21.25,-46.583333
br,jurua,Juruá,01,,-7.633333,-72.6
br,jurua,Juruá,04,,-3.45,-66.05
br,jurua,Juruá,22,,-6.016667,-37.483333
br,jurua puca,Juruá Puca,04,,-2.833333,-65.916667
br,jurua-puka,Jurua-Puka,04,,-2.833333,-65.916667
br,jurubeba,Jurubeba,13,,-3.129722,-43.043611
br,jurubeba,Jurubeba,20,,-8.833333,-42.566667
br,jurubeba,Jurubeba,30,,-8.533333,-36.833333
br,juruce,Jurucê,27,,-21.066667,-47.75
br,juruena,Juruena,14,,-12.85,-58.933333
br,jurujuba,Jurujuba,21,,-22.933333,-43.116667
br,juru,Juru,17,3796,-7.533333,-37.816667
br,juru,Juru,24,,-10.433333,-62.45
br,juruma,Juruma,23,,-30.266667,-50.516667
br,jurumirim,Jurumirim,15,,-20.133333,-42.683333
br,jurumirim,Jurumirim,21,,-22.933333,-44.316667
br,jurumirim,Jurumirim,27,,-23.183333,-49.266667
br,jurupara do meio,Jurupará do Meio,27,,-23.647385,-47.446322
br,jurupara,Jurupará,27,,-23.690935,-47.36966
br,jurupari,Jurupari,04,,-2.583333,-57.6
br,jurupeba,Jurupeba,27,,-20.3,-49.333333
br,jurupema,Jurupema,27,,-21.366667,-48.566667
br,juruti,Juruti,16,13333,-2.166667,-56.066667
br,juruti velho,Juruti Velho,16,,-2.4,-56.3
br,juruty novo,Juruty Novo,16,,-2.166667,-56.066667
br,juruty velho,Juruty Velho,16,,-2.4,-56.3
br,juruvalho,Juruvalho,05,,-13.333333,-39.366667
br,jurvena,Jurvena,14,,-12.85,-58.933333
br,juscelandia,Juscelândia,29,,-15.333333,-51.316667
br,jussara,Jussara,13,,-6.683333,-44.283333
br,jussara,Jussara,18,16374,-23.6,-52.466667
br,jussara,Jussara,29,,-15.883333,-50.85
br,jussaral,Jussaral,13,,-3.515,-43.326667
br,jussaral,Jussaral,21,,-22.95,-44.266667
br,jussaral,Jussaral,30,,-8.266667,-35.066667
br,jussari,Jussari,05,,-15.2,-39.533333
br,jussiape,Jussiape,05,,-13.516667,-41.6
br,jussipe,Jussipe,05,,-13.516667,-41.6
br,justina,Justina,05,,-13.033333,-39.683333
br,justiniano de serpa,Justiniano de Serpa,06,,-4.016667,-38.483333
br,justiniano,Justiniano,15,,-16.016667,-40.333333
br,justiniano serpa,Justiniano Serpa,06,,-4.016667,-38.483333
br,justino hipolito,Justino Hipólito,15,,-21.016667,-46.75
br,justino,Justino,31,,-8.45,-49.416667
br,justinopolis,Justinópolis,15,,-19.8,-44.016667
br,justino polvora,Justino Pólvora,15,,-18.616667,-48.216667
br,jutahy,Jutahy,04,,-2.733333,-66.95
br,jutahy,Jutahy,16,,-5.6,-57
br,jutai,Jutaí,04,5863,-5.183333,-68.9
br,jutai,Jutaí,16,,-1.166667,-46.816667
br,jutai,Jutaí,30,,-8.65,-40.233333
br,jutica,Jutica,04,,-3.65,-64.266667
br,juti,Juti,11,,-22.866667,-54.616667
br,juturnaiba,Juturnaíba,21,,-22.633333,-42.3
br,juvenal correia,Juvenal Correia,15,,-17.1,-45.666667
br,juvencio,Juvêncio,11,,-21.766667,-53.433333
br,juvencio,Juvêncio,16,,-.966667,-46.616667
br,juventina,Juventina,25,,3.266667,-61.316667
br,juventino,Juventino,15,,-18.45,-48.25
br,juventino,Juventino,18,,-25.916667,-52.733333
br,juventino pinto,Juventino Pinto,15,,-20.433333,-45.483333
br,juventino rocha,Juventino Rocha,15,,-17.95,-40.383333
br,juvi,Juvi,11,,-20.383333,-56.466667
br,juye,Juyé,04,,-4.791111,-62.452222
br,kadan,Kadan,18,,-25.498848,-48.831351
br,kalore,Kaloré,18,,-23.783333,-51.65
br,kapiting,Kapiting,16,,2.183333,-55.316667
br,keteroco,Ketéroco,14,,-14.45,-58.116667
br,kilometer 11,Kilometer 11,27,,-20.8,-51.1
br,kilometer 170,Kilometer 170,23,,-29.05,-55.033333
br,kilometer 29,Kilometer 29,16,,-1.1,-46.516667
br,kilometer 41,Kilometer 41,05,,-14.85,-39.55
br,kilometer sete,Kilometer Sete,27,,-22.033333,-50.466667
br,korff,Korff,23,,-28.8,-51.033333
br,kraemer,Kraemer,23,,-29.3,-55.35
br,kramer,Kramer,23,,-29.3,-55.35
br,krauel central,Krauel Central,26,,-26.948072,-49.724089
br,kroeff,Kroeff,23,,-29.733333,-51.133333
br,krooberger,Krooberger,26,,-26.9738,-49.0765
br,kumparuyamon,Kumparuyamon,25,,4.316667,-60.133333
br,kumparuymou,Kumparuymou,25,,4.316667,-60.133333
br,labirinto,Labirinto,06,,-5.783333,-39.7
br,labrea,Lábrea,04,19624,-7.266667,-64.783333
br,l. abreu,L. Abreu,23,,-29.933333,-57.183333
br,lacerda,Lacerda,06,,-5.433333,-39.35
br,lacerda,Lacerda,27,,-21.966667,-47.833333
br,lacerdopolis,Lacerdópolis,26,,-27.25,-51.566667
br,lacio,Lácio,27,,-22.25,-49.883333
br,lacos,Laços,05,,-14.016667,-41.283333
br,lacos,Lacos,11,,-21.35,-57.45
br,lacre,Lacre,13,,-3.863611,-43.416111
br,ladainha,Ladainha,15,,-17.65,-41.733333
br,ladario,Ladário,11,15790,-19.016667,-57.583333
br,ladeira 1,Ladeira 1,18,,-25.69265,-50.30681
br,ladeira 2,Ladeira 2,18,,-25.70138,-50.30981
br,ladeira da esperanca,Ladeira da Esperanca,06,,-4.7,-39.316667
br,ladeira dois,Ladeira Dois,18,,-25.70138,-50.30981
br,ladeira grande,Ladeira Grande,02,,-9.8,-36.433333
br,ladeira grande,Ladeira Grande,06,,-3.966667,-38.7
br,ladeira grande,Ladeira Grande,22,,-5.833333,-35.466667
br,ladeira,Ladeira,05,,-12.116667,-38.283333
br,ladeira,Ladeira,13,,-2.3,-44.616667
br,ladeira,Ladeira,18,,-25.69633,-50.33388
br,ladeira,Ladeira,28,,-10.433333,-36.766667
br,ladeira,Ladeira,31,,-11.866667,-46.533333
br,ladeirao,Ladeirão,31,,-11.866667,-46.533333
br,ladeira um,Ladeira Um,18,,-25.69265,-50.30681
br,ladeira vermelha,Ladeira Vermelha,13,,-5.6,-47.166667
br,ladeira vermelha,Ladeira Vermelha,20,,-3.649167,-42.584167
br,laeti,Laeti,05,,-13.466667,-41
br,lafaiete,Lafaiete,15,,-20.666667,-43.8
br,lagama,Lagama,22,,-5.789444,-37.05
br,lagamar do caboclo,Lagamar do Cabôclo,22,,-4.921389,-37.178056
br,lagamar,Lagamar,06,,-4.2,-38.4
br,lagamar,Lagamar,15,,-18.216667,-46.8
br,lagamar,Lagamar,17,,-7.433333,-36.483333
br,lagamar,Lagamar,30,,-8.35,-38.516667
br,lagao,Lagão,27,,-23.733333,-47.983333
br,lagarta,Lagarta,05,,-10.616667,-40.133333
br,lagartixa,Lagartixa,06,,-7.1,-38.933333
br,lagartixa,Lagartixa,30,,-8.683333,-36.566667
br,lagarto,Lagarto,21,,-21.416667,-41.65
br,lagarto,Lagarto,28,45446,-10.9,-37.683333
br,lageadinho,Lageadinho,11,,-20.916667,-55.416667
br,lageado bonita,Lageado Bonita,18,,-24.166667,-50.783333
br,lageado bonito,Lageado Bonito,18,,-24.166667,-50.783333
br,lageado,Lageado,13,,-7.75,-46.883333
br,lageado,Lageado,14,,-16.35,-53.75
br,lageado,Lageado,23,,-29.45,-51.966667
br,lageado,Lageado,26,,-26.75,-50.35
br,lageado liso,Lageado Liso,18,,-23.945099,-50.564365
br,lagedinha,Lagedinha,05,,-12.333333,-40.85
br,lagedinho,Lagedinho,05,,-12.333333,-40.85
br,lagedo,Lagedo,06,,-4.016667,-38.916667
br,lagedo,Lagedo,30,,-8.666667,-36.316667
br,lage do muriahe,Lage do Muriahé,21,,-21.2,-42.116667
br,lage grande,Lage Grande,30,,-8.6,-35.716667
br,lage,Lage,05,,-10.183333,-40.966667
br,lage,Lage,08,,-19.533333,-40.833333
br,lage,Lage,14,,-15.833333,-55.6
br,lage,Lage,16,,-5.566667,-56.966667
br,lage,Lage,30,,-7.883333,-38.216667
br,lages,Lages,05,,-10.933333,-42.35
br,lages,Lages,06,,-6.1,-39.45
br,lages,Lages,15,,-16.566667,-45.933333
br,lages,Lages,22,,-5.698889,-36.245278
br,lages,Lages,24,,-10.416667,-65.383333
br,lages,Lages,26,164679,-27.8,-50.316667
br,lages,Lages,29,,-14.05,-48.133333
br,laginha,Laginha,13,,-7.783333,-47.033333
br,laginha,Laginha,15,,-20.15,-41.616667
br,laginha,Laginha,22,,-5.504722,-37.228611
br,laginhas,Laginhas,22,,-6.25,-37.05
br,lagoa amarela,Lagoa Amarela,13,,-3.562222,-43.429444
br,lagoa arcada,Lagoa Arcada,20,,-9.583333,-45.65
br,lagoa azul,Lagoa Azul,18,,-25.1398,-53.8972
br,lagoa azul,Lagoa Azul,27,,-21.7,-50.833333
br,lagoa boaventura,Lagoa Boaventura,05,,-13.916667,-41.1
br,lagoa bonita,Lagoa Bonita,05,,-12.433333,-40.866667
br,lagoa bonita,Lagoa Bonita,13,,-4.8,-44.883333
br,lagoa bonita,Lagoa Bonita,15,,-19.116667,-44.2
br,lagoa branca,Lagoa Branca,05,,-12.85,-41.983333
br,lagoa branca,Lagoa Branca,23,,-30.166667,-55.583333
br,lagoa branca,Lagoa Branca,27,,-21.9,-47.033333
br,lagoa brilhante,Lagoa Brilhante,14,,-14.451389,-59.6875
br,lagoa capim,Lagoa Capim,30,,-8.883333,-37.416667
br,lagoa cercada,Lagoa Cercada,06,,-7.033333,-40.233333
br,lagoa cercada,Lagoa Cercada,20,,-9.133333,-44.25
br,lagoa cercada,Lagoa Cercada,30,,-8.366667,-38.216667
br,lagoa cerrada,Lagoa Cerrada,06,,-6.416667,-40.666667
br,lagoa clara,Lagoa Clara,05,,-12.683333,-44.4
br,lagoa comprida,Lagoa Comprida,05,,-12.4,-40.55
br,lagoa comprida,Lagoa Comprida,13,,-4.366667,-45.916667
br,lagoa comprida,Lagoa Comprida,20,,-9.35,-42.6
br,lagoa comprida,Lagoa Comprida,28,,-9.85,-37.866667
br,lagoa da areia,Lagoa da Areia,02,,-9.616667,-36.75
br,lagoa da barra,Lagoa da Barra,05,,-12.283333,-39.8
br,lagoa da brauna,Lagoa da Braúna,30,,-8.633333,-36.366667
br,lagoa da caicara,Lagoa da Caiçara,06,,-6.733333,-39.666667
br,lagoa da cajazeira,Lagoa da Cajàzeira,02,,-9.516667,-36.883333
br,lagoa da canabrava,Lagoa da Canabrava,05,,-11.283333,-42
br,lagoa da canoa,Lagoa da Canoa,02,13428,-9.816667,-36.766667
br,lagoa da comprida,Lagoa da Comprida,20,,-3.86,-42.489167
br,lagoa da confusao,Lagoa da Confusão,31,,-10.789722,-49.553889
br,lagoa da cruz,Lagoa da Cruz,05,,-10.633333,-43.5
br,lagoa da cruz,Lagoa da Cruz,18,,-25.889657,-50.223641
br,lagoa da cruz,Lagoa da Cruz,20,,-3.673056,-42.227222
br,lagoa da cruz,Lagoa da Cruz,30,,-7.716667,-37.916667
br,lagoa da eugenia,Lagoa da Eugênia,05,,-10.683333,-39.65
br,lagoa d'agua,Lagoa d'Agua,13,,-3.598889,-43.063889
br,lagoa da iha,Lagoa da Iha,17,,-7.716667,-36.716667
br,lagoa da laje,Lagoa da Laje,30,,-8.433333,-36.233333
br,lagoa da lapinha,Lagoa da Lapinha,05,,-12.45,-40.966667
br,lagoa da laurencia,Lagoa da Laurência,05,,-13.8,-42.016667
br,lagoa da legua,Lagoa da Légua,17,,-8.133333,-37.066667
br,lagoa da manga,Lagoa da Manga,06,,-3.933333,-40.1
br,lagoa da mata,Lagoa da Mata,05,,-12.316667,-39.55
br,lagoa da mata,Lagoa da Mata,06,,-4.866667,-40.433333
br,lagoa da moita,Lagoa da Moita,06,,-5.866667,-39.65
br,lagoa da montanha,Lagôa da Montanha,22,,-6.483333,-35.3
br,lagoa danta,Lagoa Danta,05,,-12.4,-39.616667
br,lagoa d'anta,Lagoa d'Anta,06,,-6.483333,-39.333333
br,lagoa da onca,Lagoa da Onça,05,,-15.5,-40.833333
br,lagoa da onca,Lagoa da Onça,13,,-3.290278,-42.405833
br,lagoa da pedra,Lagoa da Pedra,02,,-9.4,-36.816667
br,lagoa da pedra,Lagoa da Pedra,05,,-10.616667,-39.4
br,lagoa da pedra,Lagoa da Pedra,06,,-6.466667,-38.966667
br,lagoa da pedra,Lagoa da Pedra,30,,-8.266667,-36.05
br,lagoa da porta,Lagoa da Porta,02,,-9.95,-36.383333
br,lagoa da porta,Lagoa da Porta,06,,-5.7,-39.566667
br,lagoa da porta,Lagoa da Porta,30,,-8.816667,-36.45
br,lagoa da prata,Lagoa da Prata,15,43119,-20.016667,-45.55
br,lagoa da queimada,Lagoa da Queimada,05,,-12.983333,-39.6
br,lagoa da redinha,Lagoa da Redinha,17,,-6.883333,-36.8
br,lagoa da roca,Lagôa da Róca,05,,-10.5,-40.4
br,lagoa das almas,Lagoa das Almas,20,,-5.2,-41.833333
br,lagoa da salsa,Lagoa da Salsa,06,,-4.95,-37.733333
br,lagoa das cabras,Lagoa das Cabras,05,,-13.033333,-40.066667
br,lagoa das carnaubas,Lagoa Das Carnaubas,06,,-5.066667,-38.583333
br,lagoa das chaves,Lagoa das Chaves,30,,-8.366667,-36.516667
br,lagoa das cobras,Lagoa das Cobras,05,,-13.033333,-40.066667
br,lagoa das covas,Lagoa das Covas,05,,-13.933333,-41.133333
br,lagoa das dores,Lagoa das Dôres,30,,-7.833333,-38.183333
br,lagoa da serra,Lagoa da Serra,20,,-3.955833,-42.166111
br,lagoa das mocas,Lagoa das Moças,30,,-8.4,-36.516667
br,lagoa das pedras,Lagoa das Pedras,02,,-9.35,-37.883333
br,lagoa das pedras,Lagoa das Pedras,06,,-2.974444,-41.021944
br,lagoa das pedras,Lagoa das Pedras,20,,-6.916667,-41.233333
br,lagoa das vacas,Lagoa das Vacas,05,,-13.45,-40.733333
br,lagoa das vacas,Lagoa das Vacas,20,,-9.183333,-42.616667
br,lagoa da vaca,Lagoa da Vaca,05,,-12.016667,-41.6
br,lagoa da velha,Lagoa da Velha,05,,-12.916667,-39.533333
br,lagoa da vila,Lagoa da Vila,04,,-.116667,-63.95
br,lagoa da volta,Lagoa da Volta,06,,-2.816667,-40.25
br,lagoa de arroz,Lagoa de Arroz,17,,-7.05,-38.533333
br,lagoa de baixo,Lagoa de Baixo,06,,-2.883333,-40.216667
br,lagoa de baixo,Lagoa de Baixo,20,,-9.75,-45.45
br,lagoa de barro,Lagoa de Barro,05,,-13.4,-40.466667
br,lagoa de boi,Lagoa de Boi,22,,-5.9,-35.383333
br,lagoa de canoa,Lagôa de Canôa,02,,-9.816667,-36.766667
br,lagoa de casca,Lagoa de Casca,06,,-5.116667,-37.916667
br,lagoa de cima,Lagoa de Cima,13,,-5.766667,-43.783333
br,lagoa de cima,Lagoa de Cima,17,,-7.6,-36.683333
br,lagoa de cima,Lagoa de Cima,22,,-5.0675,-37.331667
br,lagoa de cima,Lagoa de Cima,30,,-8.333333,-36.75
br,lagoa de dentro ii,Lagoa de Dentro II,20,,-3.33,-42.012778
br,lagoa de dentro i,Lagoa de Dentro I,20,,-3.478611,-42.072222
br,lagoa de dentro,Lagoa de Dentro,05,,-11.616667,-39.616667
br,lagoa de dentro,Lagoa de Dentro,06,,-3.25,-39.266667
br,lagoa de dentro,Lagoa de Dentro,13,,-5.616667,-43.283333
br,lagoa de dentro,Lagoa de Dentro,17,,-7.8,-38.1
br,lagoa de dentro,Lagoa de Dentro,20,,-3.67,-42.247778
br,lagoa de dentro,Lagoa de Dentro,22,,-6.133333,-38.433333
br,lagoa de dentro,Lagoa de Dentro,28,,-9.95,-37.2
br,lagoa de dentro,Lagoa de Dentro,30,,-7.666667,-40.516667
br,lagoa de fora,Lagoa de Fora,08,,-21.266667,-41
br,lagoa de jose alves,Lagoa de José Alves,05,,-9.133333,-39.333333
br,lagoa de junco,Lagoa de Junco,20,,-5.583333,-41.15
br,lagoa dentro,Lagoa Dentro,30,,-8.783333,-36.2
br,lagoa de onca,Lagoa de Onça,06,,-7.333333,-38.8
br,lagoa de ouro,Lagôa de Ouro,30,,-9.133333,-36.466667
br,lagoa de pedra,Lagoa de Pedra,02,,-9.783333,-36.716667
br,lagoa de pedra,Lagoa de Pedra,05,,-12.65,-40.583333
br,lagoa de pedra,Lagoa de Pedra,13,,-4.333333,-45.166667
br,lagoa de pedra,Lagoa de Pedra,17,,-7,-37.316667
br,lagoa de pedra,Lagoa de Pedra,22,,-6.166667,-35.433333
br,lagoa de pedra,Lagoa de Pedra,30,,-7.95,-36.166667
br,lagoa de pedras,Lagoa de Pedras,05,,-12.666667,-40.816667
br,lagoa de santana,Lagoa de Santana,06,,-3.016667,-39.916667
br,lagoa de sao goncalo,Lagoa de São Gonçalo,15,,-18.45,-45.7
br,lagoa de sao joao,Lagoa de São João,30,,-7.75,-37.95
br,lagoa de sao joaquim,Lagoa de São Joaquim,20,,-5.616667,-41.416667
br,lagoa de sao jose,Lagoa de São José,30,,-9.266667,-36.65
br,lagoa de todos-os-santos,Lagoa de Todos-os-Santos,20,,-5.75,-42.8
br,lagoa do alegre,Lagoa do Alegre,05,,-9.1,-41.666667
br,lagoa do alto,Lagoa do Alto,05,,-13.166667,-40.1
br,lagoa do anjo,Lagoa do Anjo,06,,-6.133333,-40.65
br,lagoa do arroz,Lagoa do Arroz,05,,-12.2,-40.733333
br,lagoa do arroz,Lagoa do Arroz,20,,-9.716667,-43.85
br,lagoa do arroz,Lagoa do Arroz,30,,-8.433333,-36.05
br,lagoa do atalho,Lagoa do Atalho,05,,-12.233333,-39.683333
br,lagoa do baixao,Lagoa do Baixão,20,,-9.416667,-43.783333
br,lagoa do banco,Lagoa do Banco,20,,-8.55,-41.516667
br,lagoa do barro,Lagoa do Barro,13,,-3.088889,-42.288611
br,lagoa do barro,Lagoa do Barro,20,,-3.918056,-42.186111
br,lagoa do barro,Lagoa do Barro,30,,-7.766667,-40.366667
br,lagoa do boia,Lagoa do Boia,17,,-7.833333,-36.85
br,lagoa do boi,Lagoa do Boi,05,,-11.916667,-41.716667
br,lagoa do boi,Lagoa do Boi,06,,-5.833333,-40.283333
br,lagoa do boi,Lagoa do Boi,17,,-7.233333,-36.8
br,lagoa do buraco,Lagoa do Buraco,05,,-12.4,-40.9
br,lagoa do burro,Lagoa do Burro,30,,-7.816667,-40.366667
br,lagoa do cabore,Lagoa do Caboré,20,,-9.2,-42.516667
br,lagoa do caminho,Lagoa do Caminho,02,,-9.333333,-36.866667
br,lagoa do canto,Lagoa do Canto,05,,-13.966667,-42.466667
br,lagoa do canto,Lagoa do Canto,20,,-6.966667,-41.316667
br,lagoa do canto,Lagoa do Canto,22,,-5.75,-35.483333
br,lagoa do capim,Lagoa do Capim,05,,-12.333333,-40.083333
br,lagoa do carneiro,Lagoa do Carneiro,06,,-3.033333,-40.05
br,lagoa do caroa,Lagoa do Caroá,30,,-7.883333,-37.733333
br,lagoa do carro,Lagoa do Carro,06,,-6.783333,-40.35
br,lagoa do carro,Lagoa do Carro,30,,-7.85,-35.333333
br,lagoa do carvao,Lagoa do Carvão,28,,-10.6,-37.283333
br,lagoa do casco,Lagoa do Casco,13,,-3.066667,-43.35
br,lagoa do chiquinho,Lagoa do Chiquinho,05,,-13.55,-40.2
br,lagoa do cigano,Lagoa do Cigano,13,,-3.482778,-43.080833
br,lagoa do cipo,Lagoa do Cipó,05,,-12.083333,-40.2
br,lagoa do coco,Lagoa do Côco,02,,-8.966667,-37.516667
br,lagoa do crasto,Lagoa do Crasto,06,,-2.816667,-40.2
br,lagoa do curral,Lagoa do Curral,02,,-9.666667,-37.633333
br,lagoa do curral,Lagoa do Curral,30,,-8.666667,-37.016667
br,lagoa do dionisio,Lagoa do Dionísio,05,,-12.466667,-42.233333
br,lagoa do emidio,Lagôa do Emídio,30,,-9.2,-36.433333
br,lagoa do emigdio,Lagôa do Emigdio,30,,-9.2,-36.433333
br,lagoa do enchu,Lagoa do Enchu,13,,-4.016667,-43.05
br,lagoa do engano,Lagoa do Engano,06,,-4.65,-38.1
br,lagoa do espinho,Lagoa do Espinho,22,,-5.233333,-35.95
br,lagoa do estevam,Lagoa do Estevam,05,,-14.6,-42.066667
br,lagoa do estevao,Lagoa do Estêvão,05,,-14.6,-42.066667
br,lagoa do exu,Lagoa do Exu,13,,-4.016667,-43.05
br,lagoa do fabio,Lagoa do Fábio,05,,-10.816667,-43.466667
br,lagoa do felix,Lagoa do Félix,30,,-8.216667,-36.6
br,lagoa do forno,Lagoa do Forno,17,,-6.8,-38.166667
br,lagoa do garrote,Lagoa do Garrote,02,,-9.35,-37.266667
br,lagoa do gato,Lagoa do Gato,20,,-7.55,-41.833333
br,lagoa do gaviao,Lagoa do Gavião,02,,-9.466667,-37.633333
br,lagoa do gouveia,Lagoa do Gouveia,15,,-18.816667,-45.833333
br,lagoa do itaenga,Lagoa do Itaenga,30,17719,-7.916667,-35.333333
br,lagoa do jacare,Lagoa do Jacaré,15,,-19.133333,-44.533333
br,lagoa do jacu,Lagoa do Jacu,30,,-9.15,-36.683333
br,lagoa do jatoba,Lagoa do Jatobá,05,,-9.85,-43.65
br,lagoa do junco,Lagoa do Junco,06,,-4.65,-37.9
br,lagoa do junco,Lagoa do Junco,17,,-7.15,-37.966667
br,lagoado,Lagoado,20,,-10.533333,-44.066667
br,lagoa do lajedo,Lagoa do Lajedo,05,,-12.916667,-40.2
br,lagoa do lajeiro,Lagoa do Lajeiro,02,,-9.45,-37.616667
br,lagoa do laje,Lagoa do Laje,30,,-8.433333,-36.233333
br,lagoa do laurencio,Lagoa do Laurêncio,05,,-13.8,-42.016667
br,lagoa do leite,Lagoa do Leite,05,,-12.483333,-42.5
br,lagoa do limoeiro,Lagoa do Limoeiro,17,,-7.233333,-38.6
br,lagoa do manuel,Lagoa do Manuel,05,,-13.533333,-40.983333
br,lagoa do maracaja,Lagoa do Maracajá,06,,-6.1,-38.966667
br,lagoa do marco,Lagoa do Marco,13,,-5.366667,-42.966667
br,lagoa do marizeiro,Lagoa do Marizeiro,22,,-5.413889,-37.326944
br,lagoa do martins,Lagoa do Martins,05,,-12.116667,-40.733333
br,lagoa do mata,Lagoa do Mata,28,,-10.2,-36.983333
br,lagoa do mato,Lagoa do Mato,05,,-12.35,-39.75
br,lagoa do mato,Lagoa do Mato,06,,-4.583333,-37.633333
br,lagoa do mato,Lagoa do Mato,13,,-2.466667,-44.7
br,lagoa do mato,Lagoa do Mato,20,,-3.858889,-42.516944
br,lagoa do mato,Lagoa do Mato,22,,-5.183333,-35.5
br,lagoa do mato,Lagoa do Mato,28,,-10.2,-36.983333
br,lagoa do meio,Lagoa do Meio,05,,-13.2,-42.533333
br,lagoa do meio,Lagoa do Meio,17,,-6.933333,-36.9
br,lagoa do meio,Lagoa do Meío,20,,-7.116667,-41.966667
br,lagoa do meio,Lagoa do Meio,22,,-5.833333,-35.583333
br,lagoa do mel,Lagoa do Mel,02,,-9.633333,-36.75
br,lagoa do mel,Lagoa do Mel,22,,-5.95,-35.6
br,lagoa do mocambo,Lagoa do Mocambo,20,,-6.066667,-42.233333
br,lagoa do morto,Lagoa do Morto,05,,-13.1,-39.966667
br,lagoa do mucambo,Lagoa do Mucambo,20,,-6.066667,-42.233333
br,lagoa do neco,Lagoa do Neco,20,,-8.816667,-42.916667
br,lagoa do ouro,Lagoa do Ouro,05,,-10.45,-38.566667
br,lagoa do ouro,Lagoa do Ouro,30,,-9.133333,-36.466667
br,lagoa do padre,Lagoa do Padre,30,,-8.716667,-36.15
br,lagoa do pai goncalo,Lagoa do Pai Gonçalo,02,,-9.616667,-36.833333
br,lagoa do peixe,Lagoa do Peixe,05,,-12.466667,-41.183333
br,lagoa do pendao,Lagoa do Pendão,22,,-5.483333,-35.716667
br,lagoa do poco,Lagoa do Poço,05,,-11.25,-43.933333
br,lagoa do pore,Lagoa do Poré,22,,-5.386111,-36.881111
br,lagoa do prata,Lagoa do Prata,22,,-5.333333,-35.5
br,lagoa do rancho,Lagoa do Rancho,02,,-9.416667,-37.383333
br,lagoa do rancho,Lagoa do Rancho,28,,-9.966667,-37.433333
br,lagoa do remigio,Lagôa do Remigio,17,,-6.933333,-35.783333
br,lagoa do riacho,Lagoa do Riacho,30,,-9.133333,-36.7
br,lagoa do rocha,Lagoa do Rocha,06,,-5.216667,-37.816667
br,lagoa do sabino,Lagoa do Sabino,06,,-5.05,-37.833333
br,lagoa do saco,Lagoa do Saco,20,,-6.183333,-41.883333
br,lagoa do sapo,Lagoa do Sapo,22,,-5.766667,-35.916667
br,lagoa dos bastos,Lagoa dos Bastos,05,,-12.833333,-41.95
br,lagoa dos bois,Lagoa dos Bois,06,,-4.616667,-38.833333
br,lagoa dos bois,Lagoa dos Bois,14,,-16.233333,-56.966667
br,lagoa dos caboclos,Lagoa dos Caboclos,02,,-9.483333,-36.783333
br,lagoa dos cavalos,Lagoa dos Cavalos,20,,-3.521667,-42.090278
br,lagoa dos claudios,Lagoa dos Cláudios,20,,-4.383333,-41.616667
br,lagoa dos cocos da edmundo,Lagoa dos Cocos da Edmundo,20,,-6.066667,-42.566667
br,lagoa dos couros,Lagoa dos Couros,30,,-8.266667,-36.316667
br,lagoa dos currais,Lagoa dos Currais,22,,-5.883333,-35.516667
br,lagoa do serrote,Lagoa do Serrote,05,,-12.366667,-39.35
br,lagoa do serrote,Lagoa do Serrote,06,,-3.5,-40.1
br,lagoa dos filos,Lagoa dos Filos,06,,-4.783333,-39.866667
br,lagoa dos gatos,Lagoa dos Gatos,30,,-8.65,-35.9
br,lagoa do silvestre,Lagoa do Silvestre,13,,-3.594167,-43.175
br,lagoa do sitio,Lagoa do Sítio,20,,-6.5,-41.566667
br,lagoa dos macacos,Lagoa dos Macacos,20,,-3.909444,-42.305556
br,lagoa dos marinheiros,Lagoa dos Marinheiros,06,,-6.15,-38.55
br,lagoa dos martins,Lagoa dos Martins,15,,-20.366667,-45.95
br,lagoa dos mil homens,Lagoa dos Mil Homens,06,,-6.266667,-38.6
br,lagoa dos negros,Lagoa dos Negros,05,,-10.316667,-43.516667
br,lagoa dos neris,Lagoa dos Neris,06,,-5.666667,-40.816667
br,lagoa do sousa,Lagoa do Sousa,30,,-8.7,-35.933333
br,lagoa do souza,Lagoa do Souza,30,,-8.7,-35.933333
br,lagoa dos patos,Lagoa dos Patos,11,,-22.9,-55.5
br,lagoa dos patos,Lagoa dos Patos,13,,-3.4075,-42.631111
br,lagoa dos patos,Lagoa dos Patos,15,,-16.977222,-44.576111
br,lagoa dos perreira,Lagoa dos Perreira,18,,-25.9373,-49.3262
br,lagoa dos pioes,Lagoa dos Piões,17,,-7.8,-36.9
br,lagoa dos porcos,Lagoa dos Porcos,05,,-12.016667,-40.966667
br,lagoa dos tocos,Lagoa dos Tocos,06,,-6.383333,-39.2
br,lagoa do sul,Lagoa do Sul,26,,-26.366667,-50.333333
br,lagoa dos veados,Lagoa dos Veados,05,,-10.316667,-43.75
br,lagoa do tabuleiro,Lagoa do Tabuleiro,20,,-8.466667,-42.15
br,lagoa do tamboata,Lagoa do Tamboata,06,,-3.383333,-40.083333
br,lagoa do tapuio,Lagoa do Tapuio,06,,-5.116667,-38.6
br,lagoa do timoteo,Lagoa do Timoteo,05,,-13.85,-42.183333
br,lagoa do tocantins,Lagoa do Tocantins,31,,-10.379722,-47.423611
br,lagoa do umbuzeiro,Lagoa do Umbuzeiro,20,,-6.95,-40.966667
br,lagoa dourada,Lagoa Dourada,15,,-20.916667,-44.083333
br,lagoa do vaqueiro,Lagoa do Vaqueiro,05,,-13.483333,-40.816667
br,lagoa do vau,Lagoa do Vau,05,,-11.133333,-45.366667
br,lagoa do velho,Lagoa do Velho,22,,-6,-35.866667
br,lagoa do vicente,Lagoa do Vicente,02,,-9.6,-37.483333
br,lagoa do vieira,Lagoa do Vieira,30,,-7.733333,-38.566667
br,lagoa encantada,Lagoa Encantada,05,,-12.6,-41.3
br,lagoa escondida,Lagoa Escondida,05,,-9.816667,-39.383333
br,lagoa estrema,Lagoa Estrema,30,,-7.833333,-38.816667
br,lagoa fechada,Lagoa Fechada,30,,-8.916667,-37.916667
br,lagoa feia,Lagoa Feia,21,,-23.040005,-43.520522
br,lagoa felix,Lagoa Félix,20,,-9.483333,-45.383333
br,lagoa formosa,Lagoa Formosa,05,,-14.983333,-41.05
br,lagoa formosa,Lagoa Formosa,15,11822,-18.783333,-46.4
br,lagoa formosa,Lagoa Formosa,27,,-21.916667,-46.933333
br,lagoa funda,Lagoa Funda,05,,-13.85,-41.95
br,lagoa funda,Lagoa Funda,06,,-7.333333,-38.633333
br,lagoa funda,Lagoa Funda,28,,-10.016667,-36.95
br,lagoa grande de cima,Lagoa Grande de Cima,17,,-7.95,-37.033333
br,lagoa grande,Lagoa Grande,02,,-10.333333,-36.7
br,lagoa grande,Lagoa Grande,05,,-12.366667,-41.033333
br,lagoa grande,Lagoa Grande,06,,-2.902222,-40.655556
br,lagoa grande,Lagoa Grande,13,,-2.916667,-43.466667
br,lagoa grande,Lagoa Grande,15,,-15.7,-41.816667
br,lagoa grande,Lagoa Grande,17,,-6.966667,-38.1
br,lagoa grande,Lagoa Grande,18,,-25.683333,-49.45
br,lagoa grande,Lagoa Grande,20,,-6.85,-41.3
br,lagoa grande,Lagoa Grande,22,,-5.85,-35.633333
br,lagoa grande,Lagoa Grande,30,,-9,-40.283333
br,lagoa jose luis,Lagoa José Luís,05,,-15.016667,-40.95
br,lagoa jurema,Lagoa Jurema,20,,-9.35,-43.3
br,lagoa,Lagoa,02,,-9.383333,-37.266667
br,lagoa,Lagoa,05,,-11.05,-42.316667
br,lagoa,Lagoa,06,,-3.166667,-39.733333
br,lagoa,Lagoa,08,,-19.166667,-40.183333
br,lagoa,Lagoa,11,,-20.55,-54.083333
br,lagoa,Lagoa,13,,-3.305833,-43.212778
br,lagoa,Lagoa,14,,-14.133333,-55.333333
br,lagoa,Lagoa,15,,-16.3,-42.45
br,lagoa,Lagoa,17,,-6.583333,-37.916667
br,lagoa,Lagoa,18,,-23.833333,-51.116667
br,lagoa,Lagoa,20,,-3.535278,-42.016944
br,lagoa,Lagoa,21,,-22.968276,-43.204229
br,lagoa,Lagoa,22,,-5.316667,-35.766667
br,lagoa,Lagoa,26,,-27.616667,-48.483333
br,lagoa,Lagôa,27,,-21.9,-47.033333
br,lagoa,Lagoa,30,,-7.9,-36.366667
br,lagoa limpa,Lagoa Limpa,30,,-8.366667,-36.15
br,lagoa nova,Lagoa Nova,05,,-12.75,-40.9
br,lagoa nova,Lagoa Nova,06,,-3.45,-39.35
br,lagoa nova,Lagoa Nova,13,,-5.616667,-44.416667
br,lagoa nova,Lagoa Nova,20,,-5.45,-42.866667
br,lagoa nova,Lagoa Nova,21,,-22.801885,-43.792225
br,lagoa nova,Lagoa Nova,22,5984,-6.1,-36.483333
br,lagoa nova,Lagoa Nova,27,,-23.4948,-46.0121
br,lagoa nova,Lagoa Nova,30,,-7.716667,-39.566667
br,lagoao,Lagoão,15,,-16.283333,-45.783333
br,lagoao,Lagoão,18,,-26.45,-52.016667
br,lagoao,Lagoão,23,,-28.55,-53.466667
br,lagoao,Lagoão,27,,-23.671336,-47.577358
br,lagoa pau da embira,Lagoa Pau da Embira,17,,-7.5,-36.033333
br,lagoa pedro borges,Lagoa Pedro Borges,30,,-8.333333,-36.733333
br,lagoa preta,Lagoa Preta,05,,-10.65,-37.9
br,lagoa preta,Lagoa Preta,06,,-7.783333,-39.066667
br,lagoa queimada,Lagoa Queimada,05,,-12.683333,-40.433333
br,lagoa queimada,Lagoa Queimada,17,,-7.233333,-36.85
br,lagoa rasa,Lagoa Rasa,22,,-5.716667,-37.75
br,lagoa real,Lagoa Real,05,,-14.033333,-42.133333
br,lagoa redonda,Lagoa Redonda,05,,-11.933333,-38.016667
br,lagoa redonda,Lagoa Redonda,20,,-5.416667,-41.75
br,lagoa redonda,Lagoa Redonda,22,,-5.966667,-35.65
br,lagoa salgada,Lagoa Salgada,22,,-6.116667,-35.483333
br,lagoa santa,Lagoa Santa,15,38487,-19.633333,-43.883333
br,lagoa santo antonio,Lagoa Santo Antonio,15,,-17.133333,-46.816667
br,lagoa seca,Lagoa Sêca,02,,-9.833333,-36.616667
br,lagoa seca,Lagoa Sêca,05,,-12.5,-41.333333
br,lagoa seca,Lagoa Sêca,13,,-5.783333,-46.1
br,lagoa seca,Lagoa Sêca,17,8103,-7.15,-35.85
br,lagoa seca,Lagoa Sêca,18,,-25.45,-51.866667
br,lagoa seca,Lagoa Seca,20,,-5.333333,-42.733333
br,lagoa seca,Lagoa Sêca,22,,-5.653333,-37.175
br,lagoa seca,Lagoa Sêca,27,,-20.966667,-49.9
br,lagoa seca,Lagoa Sêca,30,,-7.666667,-35.183333
br,lagoa secca,Lagôa Secca,30,,-7.666667,-35.183333
br,lagoas,Lagoas,05,,-10.633333,-40.866667
br,lagoas,Lagoas,11,,-20.8,-51.716667
br,lagoas,Lagoas,20,,-5.4,-41.916667
br,lagoas,Lagoas,30,,-7.5,-39.833333
br,lagoa solgada,Lagoa Solgada,06,,-5.016667,-38.15
br,lagoas queimadas,Lagoas Queimadas,06,,-5.483333,-40.566667
br,lagoa tapada,Lagoa Tapada,06,,-6.666667,-38.733333
br,lagoa torta,Lagoa Torta,05,,-14.6,-41.1
br,lagoa velha,Lagoa Velha,06,,-5.433333,-39.816667
br,lagoa velha,Lagoa Velha,15,,-16.166667,-41.6
br,lagoa velha,Lagoa Velha,20,,-6.55,-41.283333
br,lagoa velha,Lagoa Velha,30,,-8.983333,-38.216667
br,lagoa verde,Lagoa Verde,05,,-12.333333,-38.1
br,lagoa verde,Lagoa Verde,13,,-4.066667,-45.733333
br,lagoa verde,Lagoa Verde,14,,-16.283333,-58.466667
br,lagoa verde,Lagoa Verde,18,,-25.896686,-49.516236
br,lagoa vermelha,Lagoa Vermelha,02,,-8.966667,-36.066667
br,lagoa vermelha,Lagoa Vermelha,06,,-4.916667,-37.8
br,lagoa vermelha,Lagoa Vermelha,23,24446,-28.216667,-51.533333
br,lagoa vermelha,Lagoa Vermelha,27,,-23.609,-47.9189
br,lago da pedra,Lago da Pedra,13,25996,-4.333333,-45.166667
br,lago de cima,Lago de Cima,13,,-5.766667,-43.783333
br,lago de santo antonio,Lago de Santo Antônio,04,,-6.75,-62.35
br,lagoinas,Lagoinas,30,,-7.933333,-38.433333
br,lagoinha de baixo,Lagoinha de Baixo,14,,-15.35,-55.55
br,lagoinha de cima,Lagoinha de Cima,14,,-15.466667,-55.55
br,lagoinha,Lagoinha,02,,-10.166667,-36.383333
br,lagoinha,Lagoinha,04,,-7.516667,-73.5
br,lagoinha,Lagoinha,05,,-10.316667,-38.9
br,lagoinha,Lagoinha,06,,-2.966667,-40.433333
br,lagoinha,Lagoinha,07,,-15.816667,-47.4
br,lagoinha,Lagoinha,08,,-20.233333,-41.066667
br,lagoinha,Lagoinha,11,,-20.716667,-54.766667
br,lagoinha,Lagoinha,13,,-3.783333,-43.833333
br,lagoinha,Lagoinha,14,,-14.183333,-56.8
br,lagoinha,Lagoinha,15,,-16.1,-42.416667
br,lagoinha,Lagoinha,17,,-6.616667,-38.35
br,lagoinha,Lagoinha,20,,-10.15,-45.283333
br,lagoinha,Lagoinha,21,,-22.939317,-43.207387
br,lagoinha,Lagoinha,22,,-5.067222,-37.406389
br,lagoinha,Lagoinha,23,,-30.283333,-55.733333
br,lagoinha,Lagoinha,27,,-22.35,-47.45
br,lagoinha,Lagoinha,28,,-9.916667,-37.166667
br,lagoinha,Lagoinha,30,,-7.333333,-37.333333
br,lagoinhas,Lagoinhas,05,,-12.6,-41.016667
br,lagoinhas,Lagoinhas,22,,-6.25,-37.05
br,lagoinhas,Lagoinhas,30,,-7.933333,-38.433333
br,lago izidoro,Lago Izidoro,01,,-8.056944,-72.655278
br,lago,Lago,05,,-8.883333,-40.916667
br,lago,Lago,18,,-24.966667,-49.916667
br,lagolandia,Lagolândia,29,,-15.616667,-49.033333
br,lago novo,Lago Novo,03,,1.4,-50.633333
br,lago verde,Lago Verde,04,,-7.733333,-70.433333
br,lago verde,Lago Verde,24,,-8.483333,-62.55
br,lago vermelha,Lago Vermelha,17,,-6.75,-38.5
br,lagradouro,Lagradouro,30,,-8.733333,-37.333333
br,laguinho,Laguinho,13,,-4.283333,-44.5
br,laguna carapa,Laguna Carapã,11,,-22.45,-55.016667
br,laguna,Laguna,11,,-22.533333,-55.216667
br,laguna,Laguna,26,39712,-28.483333,-48.783333
br,laguna pora,Laguna Porã,11,,-23.2,-54.85
br,laguna vera,Laguna Verá,11,,-23.416667,-55.2
br,lagunita,Lagunita,11,,-22.6,-55.383333
br,lajao,Lajão,15,,-19.166667,-41.5
br,laje 146,Laje 146,24,,-10.416667,-65.383333
br,lajeadinha,Lajeadinha,27,,-23.37351,-47.881518
br,lajeadinho,Lajeadinho,11,,-20.916667,-55.416667
br,lajeadinho,Lajeadinho,18,,-25.166667,-50.916667
br,lajeadinho,Lajeadinho,23,,-28.6,-49.916667
br,lajeado bonita,Lajeado Bonita,18,,-24.166667,-50.783333
br,lajeado bonito,Lajeado Bonito,17,,-7.683333,-37.9
br,lajeado bonito,Lajeado Bonito,18,,-24.166667,-50.783333
br,lajeado bonito,Lajeado Bonito,23,,-27.35,-54.166667
br,lajeado branco,Lajeado Branco,06,,-5.683333,-41.016667
br,lajeado de baixo,Lajeado de Baixo,22,,-5.466667,-35.8
br,lajeado do cacador,Lajeado do Caçador,18,,-26.139104,-49.51109
br,lajeado do salto,Lajeado do Salto,27,,-23.75,-47.283333
br,lajeado dos pintos,Lajeado dos Pintos,26,,-27.183333,-51.983333
br,lajeado grande,Lajeado Grande,18,,-25.833333,-52.916667
br,lajeado grande,Lajeado Grande,23,,-27.416667,-53.95
br,lajeado grande,Lajeado Grande,26,,-26.95,-52.6
br,lajeado,Lajeado,05,,-12.466667,-41.666667
br,lajeado,Lajeado,11,,-20.233333,-55.466667
br,lajeado,Lajeado,13,,-7.75,-46.883333
br,lajeado,Lajeado,14,,-16.35,-53.75
br,lajeado,Lajeado,15,,-19.166667,-48.7
br,lajeado,Lajeado,16,,-4.947778,-53.698333
br,lajeado,Lajeado,18,,-23.316667,-50.3
br,lajeado,Lajeado,20,,-4.283333,-41.983333
br,lajeado,Lajeado,22,,-5.45,-35.783333
br,lajeado,Lajeado,23,65408,-29.45,-51.966667
br,lajeado,Lajeado,26,,-26.658663,-49.635506
br,lajeado,Lajeado,27,,-21.916667,-47.466667
br,lajeado,Lajeado,29,,-17.1,-51.766667
br,lajeado,Lajeado,31,,-9.725833,-48.218056
br,lajeado liso,Lajeado Liso,18,,-23.945099,-50.564365
br,lajeados,Lajeados,27,,-20.683333,-47.15
br,laje cento e quarenta e seis,Laje Cento e Quarenta e Seis,24,,-10.416667,-65.383333
br,laje da oiticica,Laje da Oiticica,06,,-5.433333,-37.933333
br,lajedao,Lajedão,05,,-17.583333,-40.366667
br,laje da onca,Laje da Onça,17,,-7.566667,-37.8
br,laje d' aqua,Laje d' Aqua,17,,-7.516667,-37.7
br,laje de muriae,Laje de Muriaé,21,,-21.2,-42.116667
br,lajedinho,Lajedinho,05,,-11.383333,-40.633333
br,lajedinho,Lajedinho,15,,-14.933333,-43.7
br,lajedo alto,Lajedo Alto,05,,-12.083333,-40.4
br,laje do banco,Laje do Banco,05,,-14.35,-39.416667
br,laje do capia,Laje do Capiá,02,,-9.283333,-36.566667
br,laje do carrapato,Laje do Carrapato,30,,-7.883333,-38.216667
br,lajedo de baixo,Lajedo de Baixo,05,,-12.483333,-40.416667
br,lajedo de cima,Lajedo de Cima,05,,-12.516667,-40.45
br,lajedo de mel,Lajedo de Mel,06,,-5.083333,-37.733333
br,lajedo do batista,Lajedo do Batista,05,,-12.933333,-39.55
br,lajedo do carrapicho,Lajedo do Carrapicho,30,,-8.466667,-36.716667
br,lajedo do cedro,Lajedo do Cedro,30,,-8.3,-35.966667
br,lajedo do souza,Lajedo do Souza,06,,-5.383333,-37.933333
br,lajedo do tabocal,Lajedo do Tabocal,05,,-13.466667,-40.216667
br,lajedo do trapia,Lajedo do Trapiá,02,,-9.283333,-36.566667
br,lajedo grande,Lajedo Grande,15,,-16.316667,-41.566667
br,lajedo grande,Lajedo Grande,23,,-27.483333,-53.983333
br,lajedo,Lajedo,05,,-12.583333,-39.2
br,lajedo,Lajedo,17,,-6.816667,-36.133333
br,lajedo,Lajedo,22,,-5.041944,-37.423333
br,lajedo,Lajedo,30,25262,-8.383333,-36.55
br,laje do meio,Laje do Meio,22,,-5.5,-37.9
br,lajedo novo,Lajedo Novo,05,,-12.183333,-40.583333
br,laje do rio preto,Laje do Rio Prêto,05,,-13.6,-39.5
br,lajedo seco,Lajedo Sêco,05,,-12.866667,-40.4
br,laje do tabocal,Laje do Tabocal,05,,-13.466667,-40.216667
br,laje funda,Laje Funda,05,,-12.233333,-40.983333
br,laje grande,Laje Grande,13,,-3.8,-45.233333
br,laje grande,Laje Grande,30,,-8.6,-35.716667
br,lajeiro alto,Lajeiro Alto,30,,-8.95,-36.933333
br,lajeiro,Lajeiro,02,,-9.333333,-36.766667
br,lajeiro,Lajeiro,20,,-3.591944,-42.092222
br,lajeiro,Lajeiro,22,,-4.983333,-37.65
br,laje,Laje,02,,-9.433333,-37.666667
br,laje,Laje,05,4708,-10.183333,-40.966667
br,laje,Laje,08,,-19.533333,-40.833333
br,laje,Laje,13,,-3.016667,-46.65
br,laje,Laje,14,,-15.833333,-55.6
br,laje,Laje,15,,-21.05,-45.25
br,laje,Laje,16,,-5.566667,-56.966667
br,laje,Laje,20,,-9.2,-42.166667
br,laje,Laje,21,,-21.2,-42.116667
br,laje,Laje,24,,-9.8,-62.433333
br,laje,Laje,27,,-20.466667,-47.833333
br,laje,Laje,30,,-8.083333,-36.05
br,lajes dos lopes,Lajes dos Lopes,06,,-4.766667,-39.783333
br,lajes,Lajes,02,,-9.533333,-36.2
br,lajes,Lajes,05,,-10.933333,-42.35
br,lajes,Lajes,06,,-4.016667,-38.7
br,lajes,Lajes,13,,-6.466667,-42.983333
br,lajes,Lajes,14,,-15.833333,-55.6
br,lajes,Lajes,15,,-16.35,-46.183333
br,lajes,Lajes,17,,-7.3,-36.2
br,lajes,Lajes,20,,-5.266667,-41.766667
br,lajes,Lajes,22,,-5.283889,-37.289722
br,lajes,Lajes,26,,-27.8,-50.316667
br,lajes,Lajes,29,,-14.05,-48.133333
br,lajes,Lajes,30,,-7.45,-37.216667
br,lajes pintadas,Lajes Pintadas,22,,-6.15,-36.116667
br,laje velha,Laje Velha,30,,-7.833333,-37.833333
br,lajinha do mutum,Lajinha do Mutum,15,,-19.716667,-41.416667
br,lajinha do sebastiao inacio,Lajinha do Sebastiao Inácio,22,,-5.421667,-36.421944
br,lajinha,Lajinha,02,,-9.183333,-36.05
br,lajinha,Lajinha,05,,-12.183333,-40.6
br,lajinha,Lajinha,07,,-15.816667,-47.4
br,lajinha,Lajinha,08,,-19.166667,-40.766667
br,lajinha,Lajinha,13,,-7.783333,-47.033333
br,lajinha,Lajinha,15,,-19.716667,-41.416667
br,lajinha,Lajinha,18,,-25.655779,-50.127056
br,lajinha,Lajinha,20,,-5.583333,-42.816667
br,lajinha,Lajinha,22,,-6.15,-37.733333
br,lajinha,Lajinha,30,,-7.85,-38.433333
br,lajinha,Lajinha,31,,-7.683333,-48.083333
br,lalima,Lâlima,11,,-20.566667,-56.233333
br,lamacre,Lamacré,15,,-17.466667,-42.066667
br,lama,Lama,05,,-16.466667,-39.933333
br,lamarao do passe,Lamarão do Passé,05,,-12.6,-38.4
br,lamarao,Lamarão,05,,-10.65,-43.616667
br,lamarao,Lamarão,06,,-7.266667,-39.45
br,lamarao,Lamarão,08,,-20.95,-40.95
br,lamarao,Lamarão,17,,-6.666667,-36.266667
br,lambari de baixo,Lambari de Baixo,27,,-22.783333,-47.483333
br,lambari do meio,Lambari do Meio,27,,-22.8,-47.483333
br,lambari,Lambari,08,,-20.6,-41.466667
br,lambari,Lambari,11,,-21.716667,-54.083333
br,lambari,Lambari,15,14516,-21.966667,-45.35
br,lambari,Lambari,18,,-23.854763,-50.661714
br,lambari,Lambari,27,,-23.32767,-46.109887
br,lambary,Lambary,15,,-21.966667,-45.35
br,lambedor,Lambedor,06,,-5.75,-39.866667
br,lambedor,Lambedor,18,,-26.066667,-52.583333
br,lambedor,Lambedor,30,,-7.683333,-38.65
br,lambedouro,Lambedouro,06,,-3.583333,-41.016667
br,lamberio,Lamberio,15,,-16.333333,-40.35
br,lameirao,Lameirão,05,,-10.65,-43.616667
br,lameirao,Lameirão,08,,-20.683333,-40.533333
br,lameiro,Lameiro,05,,-13.55,-41.116667
br,lameiro,Lameiro,06,,-7.266667,-39.45
br,lameiro,Lameiro,13,,-3.6325,-42.684722
br,lameiro,Lameiro,20,,-3.745833,-42.646944
br,lamim,Lamim,15,,-20.783333,-43.466667
br,lamina,Lâmina,11,,-20.566667,-56.233333
br,lamin,Lamin,15,,-20.783333,-43.466667
br,lamins,Lamins,15,,-22.383333,-44.916667
br,lamounier,Lamounier,15,,-20.466667,-45.033333
br,lanacre,Lanacrê,15,,-17.466667,-42.066667
br,lanca,Lança,18,,-25.116667,-49.233333
br,lanca,Lança,26,,-26.3,-51
br,lancante,Lançante,05,,-12.283333,-40.55
br,lanceiro,Lanceiro,11,,-20.983333,-53.716667
br,lanchinha,Lanchinha,22,,-5.9925,-37.491111
br,landi,Landi,15,,-16.15,-41.683333
br,landim,Landim,05,,-12.6,-45.3
br,landin,Landin,05,,-12.6,-45.3
br,landri sales,Landri Sales,20,,-7.266667,-43.916667
br,landulfo,Landulfo,08,,-20.566667,-41.7
br,languiru,Languiru,23,,-29.466667,-51.8
br,lanza velha,Lanza Velha,06,,-2.85,-40.1
br,lapa de fogo,Lapa de Fogo,15,,-16.083333,-41.266667
br,lapa,Lapa,05,,-12.4,-38.733333
br,lapa,Lapa,18,25621,-25.765833,-49.717778
br,lapa,Lapa,20,,-4.35,-41.133333
br,lapa,Lapa,21,,-22.913993,-43.182011
br,lapa,Lapa,27,,-23.533333,-46.7
br,lapa,Lapa,30,,-7.5,-35.166667
br,lapao,Lapão,05,11253,-11.4,-41.833333
br,la paz,La Paz,24,,-9.75,-65.866667
br,lapeira dantas,Lapeira Dantas,30,,-7.366667,-37.316667
br,lapela,Lapela,13,,-3.733333,-44.75
br,lapinha,Lapinha,05,,-12.1,-41.283333
br,lapinha,Lapinha,15,,-19.566667,-43.966667
br,la preta,La Preta,05,,-11.766667,-41.033333
br,larangeiras,Larangeiras,06,,-5.283333,-38.866667
br,larangeiras,Larangeiras,17,,-7.066667,-35.766667
br,larangeiras,Larangeiras,21,,-21.766667,-42.166667
br,larangeiras,Larangeiras,27,,-20.283333,-48.716667
br,larangeiras,Larangeiras,28,,-10.8,-37.166667
br,laranginha,Laranginha,18,,-25.416667,-52.416667
br,larania da terra,Larania da Terra,08,,-20.216667,-41.683333
br,laranja azeda,Laranja Azeda,18,,-25.128934,-48.346437
br,laranja-azeda,Laranja-Azeda,27,,-23.128889,-46.505
br,laranja azeda,Laranja Azeda,27,,-23.2008,-47.9369
br,laranja-azeda,Laranja-Azeda,27,,-23.216325,-47.829697
br,laranja da terra,Laranja da Terra,08,3016,-19.9,-41.066667
br,laranjais,Laranjais,21,,-21.766667,-42.166667
br,laranja,Laranja,04,,-3.9,-63.416667
br,laranja,Laranja,05,,-13.883333,-41.833333
br,laranja,Laranja,20,,-5.05,-42.683333
br,laranjal,Laranjal,04,,-3.483333,-68.266667
br,laranjal,Laranjal,14,,-14.5,-55.133333
br,laranjal,Laranjal,15,,-21.366667,-42.466667
br,laranjal,Laranjal,18,,-24.73586,-48.978322
br,laranjal,Laranjal,23,,-28.5,-51.55
br,laranjal,Laranjal,26,,-28.65,-48.983333
br,laranjal,Laranjal,27,,-21.683333,-47.85
br,laranjal paulista,Laranjal Paulista,27,21264,-23.048726,-47.835968
br,laranjeira,Laranjeira,06,,-5.283333,-38.866667
br,laranjeira,Laranjeira,08,,-19.966667,-40.25
br,laranjeira,Laranjeira,14,,-15.9,-56.683333
br,laranjeira,Laranjeira,17,,-6.9,-37.283333
br,laranjeira,Laranjeira,18,,-24.7,-51.066667
br,laranjeira,Laranjeira,23,,-28.05,-54.683333
br,laranjeira,Laranjeira,26,,-27.583333,-50.883333
br,laranjeiras do sul,Laranjeiras do Sul,18,18425,-25.416667,-52.416667
br,laranjeiras ii,Laranjeiras II,13,,-3.493889,-43.418889
br,laranjeiras i,Laranjeiras I,13,,-3.496111,-43.383056
br,laranjeiras,Laranjeiras,05,,-12.566667,-38.533333
br,laranjeiras,Laranjeiras,06,,-5.283333,-38.866667
br,laranjeiras,Laranjeiras,08,,-19.966667,-40.25
br,laranjeiras,Laranjeiras,11,,-18.566667,-57.45
br,laranjeiras,Laranjeiras,13,,-3.506389,-43.390556
br,laranjeiras,Laranjeiras,14,,-15.15,-56.316667
br,laranjeiras,Laranjeiras,15,,-21.2,-45.433333
br,laranjeiras,Laranjeiras,16,,-5.7,-54.216667
br,laranjeiras,Laranjeiras,17,,-7.066667,-35.766667
br,laranjeiras,Laranjeiras,18,,-24.366667,-52.333333
br,laranjeiras,Laranjeiras,21,,-21.633333,-41.683333
br,laranjeiras,Laranjeiras,23,,-28.05,-54.683333
br,laranjeiras,Laranjeiras,26,,-26.2922,-48.6806
br,laranjeiras,Laranjeiras,27,,-20.283333,-48.716667
br,laranjeiras,Laranjeiras,28,23251,-10.8,-37.166667
br,laranjeiras,Laranjeiras,31,,-7.116667,-48.3
br,laranjeirinha,Laranjeirinha,24,,-10.866667,-65.3
br,laranjinha,Laranjinha,18,,-23.4,-50.3
br,laranjo,Laranjo,20,,-3.646944,-42.6475
br,laras,Laras,27,,-22.925,-47.896111
br,lar das meninas,Lar das Meninas,26,,-27.217377,-49.612074
br,larga,Larga,14,,-16.616667,-57.616667
br,larga,Larga,29,,-14.55,-46.216667
br,largao,Largão,11,,-20.266667,-56.85
br,largo da freguesia,Largo da Freguesia,21,,-22.942258,-43.342881
br,largo da taquara,Largo da Taquara,21,,-22.922289,-43.369654
br,largo do aarao,Largo do Aarão,21,,-22.949733,-43.678987
br,largo do bicao,Largo do Bicão,21,,-22.838337,-43.309894
br,largo do forno,Largo do Forno,13,,-5.3,-43.233333
br,largo do mariano,Largo do Mariano,13,,-5.3,-43.45
br,largo do queimado,Largo do Queimado,27,,-24.8027,-48.2595
br,largo do tanque,Largo do Tanque,21,,-22.916805,-43.356529
br,largo,Largo,05,,-11.8,-40.783333
br,largo,Largo,06,,-7.083333,-38.85
br,largo,Largo,14,,-16.616667,-57.616667
br,largo,Largo,20,,-10.016667,-44
br,largo,Largo,22,,-5.699167,-37.453333
br,lascadinha,Lascadinha,26,,-26.666667,-50.266667
br,las palmas,Las Palmas,03,,1.233333,-50.916667
br,lassance,Lassance,15,,-17.9,-44.566667
br,lastro,Lastro,22,,-6.516667,-38.166667
br,lau alves,Laú Alves,15,,-16.216667,-40.083333
br,laudja,Laudjá,11,,-21.083333,-56.7
br,laura,Laura,21,,-21.3,-41.566667
br,laurentino,Laurentino,26,,-27.218,-49.7323
br,laurindo antonio de jesus,Laurindo Antônio de Jesus,23,,-28.966667,-54.55
br,laurindo,Laurindo,16,,-7.933333,-57.516667
br,laurindo matos,Laurindo Matos,15,,-17.25,-41.833333
br,laurinha,Laurinha,15,,-19.516667,-41.383333
br,laurinha,Laurinha,18,,-24.716667,-50.133333
br,lauro,Lauro,21,,-21.3,-41.566667
br,lauro muller,Lauro Müller,26,10036,-28.4,-49.383333
br,lauro penteado,Lauro Penteado,27,,-21.5,-50.466667
br,lauro sodre,Lauro Sodré,04,,-3.858056,-62.584444
br,lauro sodre,Lauro Sodré,16,,-.75,-47.266667
br,lauzane paulista,Lauzane Paulista,27,,-23.478607,-46.64304
br,lava cara,Lava Cara,14,,-16.266667,-56.066667
br,lavadeira de baixo,Lavadeira de Baixo,05,,-14.016667,-40.666667
br,lavadeira,Lavadeira,13,,-3.183333,-45.016667
br,lavadeira,Lavadeira,17,,-7.25,-37.766667
br,lavador,Lavador,05,,-15.1,-41.766667
br,lavador,Lavador,18,,-25.98115,-50.063117
br,lavagem dos patos,Lavagem dos Patos,13,,-3.035278,-42.132222
br,lavagem,Lavagem,06,,-3.233333,-39.366667
br,lavagem,Lavagem,15,,-14.85,-43.95
br,lavaginha,Lavaginha,06,,-3.2,-39.383333
br,lavajinha,Lavajinha,13,,-3.027222,-42.081111
br,lava-pes,Lava-Pés,05,,-13.466667,-40.4
br,lava-pes,Lava-Pés,15,,-17.2,-41.816667
br,laviai,Laviaí,11,,-20.55,-55.983333
br,lavinia,Lavínia,27,,-21.15,-51.033333
br,lavoura,Lavoura,14,,-14.983333,-56.783333
br,lavouras,Lavouras,14,,-14.983333,-56.783333
br,lavrador,Lavrador,05,,-15.1,-41.766667
br,lavra,Lavra,27,,-24.532194,-48.985463
br,lavras da mangabeira,Lavras da Mangabeira,06,17221,-6.75,-38.95
br,lavras do batovi,Lavras do Batoví,23,,-30.816667,-53.916667
br,lavras do sul,Lavras do Sul,23,,-30.816667,-53.916667
br,lavras,Lavras,06,,-6.75,-38.95
br,lavras,Lavras,13,,-5.166667,-43.433333
br,lavras,Lavras,14,,-15.3,-59.233333
br,lavras,Lavras,15,81474,-21.233333,-45
br,lavras,Lavras,18,,-24.483333,-50.483333
br,lavras,Lavras,23,,-30.816667,-53.916667
br,lavras,Lavras,27,,-22.925,-47.896111
br,lavras novas,Lavras Novas,15,,-20.483333,-43.516667
br,lavrinha,Lavrinha,14,,-15.95,-56.966667
br,lavrinha,Lavrinha,18,,-23.89279,-50.0665
br,lavrinha,Lavrinha,27,,-20.166667,-47.633333
br,lavrinhas,Lavrinhas,14,,-15.95,-56.966667
br,lavrinhas,Lavrinhas,27,,-22.583333,-44.9
br,lavrinhas,Lavrinhas,29,,-14.966667,-49.3
br,lazarino vieira,Lazarino Vieira,15,,-21.066667,-46.833333
br,leandro ferreira,Leandro Ferreira,15,,-19.7,-45.033333
br,leandro,Leandro,13,,-3.8425,-43.487778
br,leao,Leão,23,,-30.116667,-52.05
br,leao,Leão,26,,-27.3,-51.566667
br,leblon,Leblon,11,,-19.983333,-56.016667
br,leblon,Leblon,21,,-22.98094,-43.220489
br,lebon regis,Lebon Régis,26,,-26.933333,-50.7
br,leda,Leda,11,,-19.333333,-56.95
br,ledo,Lêdo,17,,-7.25,-36.233333
br,leitao da cunha,Leitão da Cunha,21,,-22.083333,-42.016667
br,leite de sousa,Leite de Sousa,21,,-22.216667,-44.05
br,leite,Leite,27,,-23.697607,-47.496278
br,leites,Leites,27,,-23.6936,-47.5011
br,leiu,Leiú,20,,-7.216667,-43.15
br,lejeado,Lejeado,20,,-7.833333,-41.6
br,lembranca,Lembrança,08,,-20.55,-41.366667
br,lembranca,Lembrança,20,,-4.183333,-42.716667
br,leme do prado,Leme do Prado,15,,-17.083333,-42.666667
br,leme,Leme,21,,-22.961147,-43.165816
br,leme,Leme,27,81211,-22.2,-47.4
br,lemes,Lemes,15,,-22.5593,-45.5549
br,lemos do prado,Lemos do Prado,15,,-17.083333,-42.666667
br,lemos,Lemos,03,,.216667,-51.166667
br,lemos,Lemos,15,,-20.533333,-46.8
br,lemos,Lemos,27,,-23.983333,-48.5
br,lencoes,Lençóes,05,,-12.566667,-41.383333
br,lencoes,Lenções,15,,-20.816667,-45.083333
br,lencoes,Lençoes,27,,-22.6,-48.783333
br,lencois,Lençóis,05,,-12.566667,-41.383333
br,lencois,Lençóis,15,,-20.816667,-45.083333
br,lencois,Lençóis,27,,-22.6,-48.783333
br,lencois paulista,Lençóis Paulista,27,57376,-22.6,-48.783333
br,lencol,Lençol,27,,-24.624127,-48.030623
br,leocadia,Leocádia,05,,-16.55,-39.916667
br,leocadio boa vista,Leocadio Bôa Vista,14,,-17.116667,-56.583333
br,leodorio,Leodório,05,,-12.15,-38.083333
br,leoes,Leões,23,,-30.983333,-54.466667
br,leonardo,Leonardo,15,,-17.083333,-41.716667
br,leonardo,Leonardo,22,,-5.388333,-36.568611
br,leoncio clemente ferreira,Leôncio Clemente Ferreira,15,,-17.266667,-40.85
br,leoncio,Leoncio,18,,-23.317013,-52.579009
br,leonel barbosa,Leonel Barbosa,11,,-21.633333,-55.533333
br,leonel,Leonel,08,,-21.083333,-41.083333
br,leoni,Leoni,18,,-22.783333,-53.2
br,leopoldina,Leopoldina,02,,-8.95,-35.65
br,leopoldina,Leopoldina,15,46743,-21.533333,-42.633333
br,leopoldina,Leopoldina,29,,-14.9,-51.083333
br,leopoldina,Leopoldina,30,,-8.083333,-39.566667
br,leopoldo bulhoes,Leopoldo Bulhões,29,,-16.616667,-48.766667
br,leopoldo de bulhoes,Leopoldo de Bulhões,29,,-16.616667,-48.766667
br,leopoldo goulart,Leopoldo Goulart,15,,-18.583333,-48.233333
br,leopolis,Leópolis,18,,-23.066667,-50.75
br,leque,Leque,13,,-2.516667,-45.183333
br,leria,Leria,04,,-2.516667,-67.333333
br,lesbando goncalves,Lesbando Gonçalves,15,,-17.133333,-41.233333
br,lessa,Lessa,05,,-12.216667,-41.133333
br,letonia,Letônia,27,,-22.116667,-50.616667
br,letras,Letras,30,,-7.833333,-39.9
br,letreiro de cima,Letreiro de Cima,20,,-5.933333,-41.016667
br,letreiro,Letreiro,06,,-7.4,-38.666667
br,letreiro,Letreiro,22,,-6.35,-38.316667
br,letreiro,Letreiro,30,,-8.55,-36.083333
br,leucadia,Leucádia,29,,-18.05,-51.85
br,levergeria,Levergéria,11,,-21.133333,-55.8
br,leverger,Leverger,14,,-15.866667,-56.083333
br,levinopolis,Levinópolis,15,,-15.266667,-44.283333
br,libanesa,Libanesa,04,,-3.783333,-62.15
br,liberata,Liberata,26,,-27.033333,-50.933333
br,liberato gomes,Liberato Gomes,15,,-14.8,-42.883333
br,liberato,Liberato,13,,-4.75,-44.533333
br,liberato,Liberato,24,,-12.633333,-60.966667
br,liberato salzano,Liberato Salzano,23,,-27.583333,-53.066667
br,liberdade,Liberdade,01,,-9.383333,-71.116667
br,liberdade,Liberdade,04,,-2.566667,-67.383333
br,liberdade,Liberdade,05,,-11.966667,-45.016667
br,liberdade,Liberdade,06,,-6.133333,-38.583333
br,liberdade,Liberdade,08,,-20.583333,-41.55
br,liberdade,Liberdade,15,,-22.016667,-44.316667
br,liberdade,Liberdade,17,,-7.133333,-37.216667
br,liberdade,Liberdade,20,,-4.383333,-42.933333
br,liberdade,Liberdade,21,,-21.216667,-41.583333
br,liberdade,Liberdade,23,,-28.2,-53.483333
br,liberdade,Liberdade,27,,-23.55,-46.633333
br,liberdade,Liberdade,30,,-7.616667,-37.35
br,liberia,Libéria,29,,-14.2,-49.15
br,liberlandi,Liberlândi,22,,-5.157778,-37.35
br,licania,Licania,06,,-3.45,-40.2
br,licinio de almeida,Licínio de Almeida,05,,-14.683333,-42.5
br,licuri,Licuri,05,,-12.083333,-41.6
br,lidia rezende,Lidia Rezende,11,,-19.883333,-54.633333
br,lidice,Lídice,21,,-22.833,-44.1921
br,lidio lima,Lídio Lima,11,,-23.25,-54.983333
br,lido,Lido,21,,-22.965431,-43.177858
br,ligeiro alto,Ligeiro Alto,30,,-8.95,-36.933333
br,ligeiro de cima,Ligeiro de Cima,17,,-7.533333,-36.65
br,ligeiro,Ligeiro,18,,-25.753704,-50.144944
br,ligeiro,Ligeiro,20,,-7.8,-41.9
br,light,Light,27,,-23.594078,-47.358195
br,lima campos,Lima Campos,06,,-6.416667,-38.95
br,lima duarte,Lima Duarte,15,13480,-21.85,-43.8
br,lima,Lima,04,,-4.393611,-63.51
br,lima,Lima,06,,-5.25,-38.216667
br,lima,Lima,18,,-25.437586,-48.743244
br,lima,Lima,25,,2.616667,-60.666667
br,limao do curua,Limão do Curuá,03,,.75,-50.15
br,limao,Limão,03,,.75,-50.15
br,limao,Limão,04,,-4.248333,-59.939722
br,limao,Limão,06,,-4.466667,-39.05
br,limao,Limão,08,,-20.95,-40.95
br,limao,Limão,16,,-1.033333,-49.55
br,limao,Limão,17,,-7.483333,-37.7
br,limao,Limão,22,,-5.3,-35.866667
br,limao,Limão,25,,3.933333,-60.5
br,limao,Limão,27,,-23.492827,-46.669158
br,limboso,Limboso,14,,-15.183333,-57
br,limeira,Limeira,05,,-10.85,-40.233333
br,limeira,Limeira,11,,-18.566667,-56.7
br,limeira,Limeira,15,,-18.583333,-41.166667
br,limeira,Limeira,18,,-24.7,-51.65
br,limeira,Limeira,21,,-21.216667,-41.366667
br,limeira,Limeira,23,,-29.35,-53.35
br,limeira,Limeira,26,,-27.05,-48.866667
br,limeira,Limeira,27,289671,-22.561667,-47.402778
br,limeirinha,Limeirinha,27,,-21.416667,-46.683333
br,limeiro,Limeiro,26,,-27.05,-48.866667
br,limirio,Limírio,15,,-18.416667,-48.383333
br,limitao,Limitão,30,,-8.9,-36.8
br,limoeira,Limoeira,02,,-9.733333,-36.516667
br,limoeira,Limoeira,04,,-7.25,-71.05
br,limoeira,Limoeira,11,,-18.566667,-56.7
br,limoeiro de anadia,Limoeiro de Anadia,02,2770,-9.733333,-36.516667
br,limoeiro do ajuru,Limoeiro do Ajuru,16,4109,-1.916667,-49.383333
br,limoeiro do norte,Limoeiro do Norte,06,29450,-5.133333,-38.083333
br,limoeiro,Limoeiro,02,,-9.733333,-36.516667
br,limoeiro,Limoeiro,04,,-6.881944,-71.144444
br,limoeiro,Limoeiro,05,,-10.583333,-39.116667
br,limoeiro,Limoeiro,06,,-5.133333,-38.083333
br,limoeiro,Limoeiro,11,,-21.666667,-54.15
br,limoeiro,Limoeiro,13,,-3.070278,-42.186667
br,limoeiro,Limoeiro,14,,-14.266667,-55.283333
br,limoeiro,Limoeiro,15,,-16.033333,-40.833333
br,limoeiro,Limoeiro,16,,-1.916667,-49.383333
br,limoeiro,Limoeiro,18,,-24.783544,-48.930205
br,limoeiro,Limoeiro,20,,-3.862778,-42.253611
br,limoeiro,Limoeiro,21,,-21.3,-41.966667
br,limoeiro,Limoeiro,22,,-5.475278,-36.052778
br,limoeiro,Limoeiro,24,,-12.333333,-63.316667
br,limoeiro,Limoeiro,27,,-21.566667,-47.7
br,limoeiro,Limoeiro,30,42329,-7.866667,-35.45
br,limoeirro,Limoeirro,27,,-22.583333,-49.35
br,limondeua,Limondeua,16,,-1.166667,-46.233333
br,limo verde,Limo Verde,08,,-20.55,-41.716667
br,limpeza,Limpeza,13,,-6.7,-43.666667
br,limpeza,Limpeza,20,,-3.863889,-42.322778
br,limpo branco,Limpo Branco,17,,-7.966667,-37.05
br,limpo dos mocos,Limpo dos Mocós,28,,-9.833333,-37.65
br,limpo grande,Limpo Grande,14,,-15.733333,-56.166667
br,limpo grande,Limpo Grande,20,,-5.816667,-42.05
br,linda flor,Linda Flor,22,,-5.493056,-36.903333
br,linda,Linda,22,,-5.189167,-36.0975
br,linda rosa,Linda Rosa,05,,-13.383333,-39.75
br,linda vista,Linda Vista,04,,-3.182778,-62.827778
br,linda vista,Linda Vista,29,,-17.05,-49.783333
br,lindeza,Lindeza,06,,-5.483333,-40.3
br,lindeza,Lindeza,20,,-5.3,-42.466667
br,lindoia,Lindóia,11,,-20.883333,-55.166667
br,lindoia,Lindoia,20,,-6.116667,-41.916667
br,lindoia,Lindóia,27,,-22.516667,-46.65
br,lindolfo almeida,Lindolfo Almeida,15,,-20.433333,-46.55
br,lindolfo,Lindolfo,15,,-18.466667,-48.416667
br,lindo,Lindo,06,,-6.166667,-39.733333
br,lingua de vaca,Lingua de Vaca,22,,-5.471667,-36.813333
br,linha 3,Linha 3,23,,-28.4,-53.95
br,linha 40,Linha 40,23,,-29.1084,-51.2175
br,linha 5,Linha 5,18,,-25.4325,-50.7463
br,linha 60,Linha 60,23,,-29.0816,-51.2169
br,linha 7,Linha 7,23,,-29.466667,-53.4
br,linha 8,Linha 8,23,,-28.366667,-53.8
br,linha binda,Linha Binda,26,,-27.25,-51.833333
br,linha blinda,Linha Blinda,26,,-27.25,-51.833333
br,linha cerqueira,Linha Cerqueira,26,,-26.523195,-49.9514
br,linha cinco,Linha Cinco,18,,-25.4325,-50.7463
br,linha c,Linha C,18,,-25.440562,-50.74636
br,linha de austin,Linha de Austin,21,,-22.890501,-43.63236
br,linha floresta,Linha Floresta,18,,-25.1131,-53.8626
br,linha formosa,Linha Formosa,18,,-26.162222,-53.656111
br,linha gaucha,Linha Gaúcha,18,,-26.125278,-53.512778
br,linha goncalves,Linha Gonçalves,18,,-25.971492,-50.19098
br,linha jansen,Linha Jansen,23,,-29.133333,-51.383333
br,linha munhoz,Linha Munhoz,18,,-25.996136,-50.19694
br,linha oitenta,Linha Oitenta,23,,-29.0548,-51.2182
br,linha oito leste,Linha Oito Leste,23,,-28.366667,-53.8
br,linha oito,Linha Oito,23,,-28.366667,-53.8
br,linha ouro verde,Linha Ouro Verde,18,,-25.1107,-53.8867
br,linha panison,Linha Panison,18,,-25.2347,-53.9122
br,linha quarenta,Linha Quarenta,23,,-29.1084,-51.2175
br,linhares,Linhares,08,105084,-19.416667,-40.066667
br,linha santo andrade,Linha Santo Andrade,18,,-25.994955,-50.164396
br,linha sao roque,Linha São Roque,26,,-27.3,-52.016667
br,linha sarandi,Linha Sarandi,18,,-26.037222,-53.512778
br,linha sertaneja,Linha Sertaneja,18,,-26.198333,-53.6125
br,linha sessenta,Linha Sessenta,23,,-29.0816,-51.2169
br,linha sete,Linha Sete,23,,-29.466667,-53.4
br,linha torrens,Linha Torrens,26,,-28.633333,-49.266667
br,linha tres ceste,Linha Três Ceste,23,,-28.4,-53.95
br,linha tres,Linha Três,23,,-28.4,-53.95
br,linha tres oeste,Linha Três Oeste,23,,-28.4,-53.95
br,linha treze,Linha Treze,18,,-25.152,-53.8688
br,linha vacaria,Linha Vacaria,18,,-25.1731,-53.9839
br,lins do vasconcelos,Lins do Vasconcelos,21,,-22.913977,-43.277668
br,lins,Lins,27,69452,-21.666667,-49.75
br,lipo,Lipo,26,,-27.583333,-50.816667
br,lira,Lira,06,,-6.616667,-40.616667
br,lisbao,Lisbão,04,,-.316667,-63.016667
br,lisieh,Lisieh,06,,-4,-40.183333
br,liso,Liso,13,,-6.6,-44.566667
br,literlade,Literlade,20,,-4.383333,-42.933333
br,liveramento do brumado,Liveramento do Brumado,05,,-13.65,-41.833333
br,liverpool,Liverpool,03,,-.333333,-52.366667
br,livramento de tiuma,Livramento de Tiúma,30,,-7.466667,-35.4
br,livramento do brumado,Livramento do Brumado,05,,-13.65,-41.833333
br,livramento do tiuma,Livramento do Tiúma,30,,-7.466667,-35.4
br,livramento,Livramento,02,,-9.25,-36.483333
br,livramento,Livramento,03,,1.066667,-50.766667
br,livramento,Livramento,04,,-7.420833,-58.623889
br,livramento,Livramento,05,19914,-13.65,-41.833333
br,livramento,Livramento,06,,-3.7,-38.966667
br,livramento,Livramento,11,,-19.25,-55.95
br,livramento,Livramento,13,,-1.266667,-45.866667
br,livramento,Livramento,14,,-15.8,-56.366667
br,livramento,Livramento,15,,-21.083333,-44.05
br,livramento,Livramento,16,,-5.233333,-56.183333
br,livramento,Livramento,17,,-7.05,-34.9
br,livramento,Livramento,20,,-4.75,-42.583333
br,livramento,Livramento,22,,-5.616667,-37.55
br,livramento,Livramento,23,,-30.883333,-55.516667
br,livramento,Livramento,25,,3.333333,-61.316667
br,lizarda,Lizarda,31,,-9.630833,-46.533889
br,loanda,Loanda,18,18233,-22.9,-53.166667
br,lobao,Lobão,20,,-5.366667,-42.683333
br,lobato,Lobato,05,,-12.916667,-38.483333
br,lobato,Lobato,18,,-22.983333,-51.95
br,lobo leite,Lôbo Leite,15,,-20.516667,-43.8
br,lobo,Lôbo,11,,-17.9,-54.033333
br,lobo,Lôbo,13,,-1.716667,-45.233333
br,lobo,Lôbo,14,,-14.683333,-54.783333
br,lobo,Lôbo,15,,-15.883333,-46.766667
br,lobo,Lôbo,16,,-8.583333,-49.666667
br,lobo,Lôbo,27,,-22.283333,-47.916667
br,local da antiga colonia sao lourenco,Local da Antiga Colônia São Lourenço,14,,-16.666667,-55.25
br,locos,Locos,11,,-21.35,-57.45
br,lodo,Lôdo,05,,-12.533333,-38.1
br,logoa branca,Logoa Branca,06,,-5.583333,-40.683333
br,logoa da pedra,Logoa da Pedra,30,,-7.783333,-37.8
br,logrador,Logrador,15,,-19.3,-45.1
br,logrador,Logrador,17,,-6.466667,-37.5
br,logradoura,Logradoura,06,,-6.033333,-39.7
br,logradouro de cima,Logradouro de Cima,05,,-9.916667,-39.616667
br,logradouro,Logradouro,06,,-4.166667,-40.1
br,logradouro,Logradouro,15,,-19.3,-45.1
br,logradouro,Logradouro,17,,-6.783333,-38.183333
br,logradouro,Logradouro,22,,-5.333333,-37.5
br,logradouro,Logradouro,28,,-10.266667,-37.316667
br,logradouro,Logradouro,30,,-8.366667,-38.1
br,loiro paco,Loiro Paço,04,,1.366667,-68.683333
br,loiro poco,Loiro Poço,04,,1.366667,-68.683333
br,loisiania,Loisiânia,21,,-22.033333,-41.516667
br,lomba grande,Lomba Grande,23,,-29.75,-51.016667
br,lombardia,Lombardia,05,,-15.883333,-39.533333
br,lombas,Lombas,23,,-29.916667,-50.5
br,lombo,Lombo,27,,-23.102463,-47.844494
br,lona,Lona,11,,-21.25,-56.65
br,londrina,Londrina,18,471836,-23.3,-51.15
br,longos,Longos,20,,-7.483333,-41.65
br,lontra,Lontra,05,,-12.25,-38
br,lontra,Lontra,11,,-20.8,-54.016667
br,lontra,Lontra,15,,-15.9,-44.3
br,lontra,Lontra,16,,-4.383333,-51.916667
br,lontra,Lontra,31,,-7.16,-48.0575
br,lontrao,Lontrão,18,,-25.3271,-50.6609
br,lontras,Lontras,06,,-4.566667,-40.966667
br,lontras,Lontras,26,,-27.1697,-49.5411
br,lopei,Lopeí,18,,-24.8,-53.6
br,lopes,Lopes,05,,-12.083333,-41.1
br,lopes,Lopes,11,,-21.833333,-53.6
br,lopes,Lopes,15,,-20.166667,-41.883333
br,lopes,Lopes,23,,-32.091667,-52.166667
br,lopes,Lopes,27,,-22.383333,-47.483333
br,loredo,Loredo,06,,-6.466667,-38.933333
br,lorena,Lorena,04,,-7.683333,-71.45
br,lorena,Lorena,15,,-16.666667,-41
br,lorena,Lorena,16,,-4.733333,-56.35
br,lorena,Lorena,22,,-5.521944,-37.476944
br,lorena,Lorena,24,,-9.6,-62.116667
br,lorena,Lorena,27,76507,-22.733333,-45.133333
br,lorenz,Lorenz,03,,2.5,-51.666667
br,loreto,Loreto,04,,-.316667,-65.9
br,loreto,Loreto,05,,-12.766667,-38.616667
br,loreto,Loreto,13,,-7.083333,-45.15
br,loreto,Loreto,22,,-6.783333,-37.116667
br,loreto,Loreto,23,,-29.716667,-54.966667
br,loteamento chcaras cruzeiro do sul,Loteamento Chcaras Cruzeiro do Sul,27,,-23.4623,-46.1309
br,loteamento chcaras mont clair,Loteamento Chcaras Mont Clair,27,,-23.2547,-46.2306
br,loteamento colina ii de mainpora,Loteamento Colina II de Mainporã,27,,-23.271237,-46.590608
br,loteamento colina segunda de maripora,Loteamento Colina Segunda de Mariporã,27,,-23.271237,-46.590608
br,loteamento ijal,Loteamento Ijal,27,,-23.2915,-46.0456
br,loteamento jardim ana maria,Loteamento Jardim Ana Maria,27,,-24.736787,-48.118844
br,loteamento jardim barigui,Loteamento Jardim Barigui,18,,-25.432,-48.7461
br,loteamento joao wesley,Loteamento João Wesley,21,,-22.908085,-43.588621
br,loteamento mapua,Loteamento Mapua,21,,-22.926003,-43.387179
br,loteamento miguel cruz,Loteamento Miguel Cruz,27,,-23.2746,-46.2304
br,loteamento parque leao,Loteamento Parque Leão,18,,-25.431538,-48.728292
br,loteamento pontal da saxonia,Loteamento Pontal da Saxônia,26,,-26.9079,-49.062
br,loteamento sao miguel,Loteamento São Miguel,21,,-22.931636,-43.70299
br,loteamento vila imperial,Loteamento Vila Imperial,23,,-29.9081,-50.9042
br,loteamento vila incomager,Loteamento Vila Incomager,27,,-24.515823,-48.111101
br,lote ideal,Lote Ideal,11,,-23.5,-54.566667
br,loucas,Louças,20,,-10.1,-43.9
br,lourdes,Lourdes,11,,-18.65,-56.616667
br,lourdes,Lourdes,26,,-27.083333,-51
br,lourdes,Lourdes,27,,-20.966667,-50.216667
br,lourenco cue,Lourenço Cuê,11,,-23.216667,-55.533333
br,lourenco,Lourenço,02,,-9.366667,-36.366667
br,lourenco,Lourenço,03,,2.5,-51.666667
br,lourenco,Lourenço,15,,-16.016667,-41.65
br,lourenco,Lourenço,18,,-25.829902,-50.1887
br,lourenco preo,Lourenço Prêo,15,,-14.933333,-42.95
br,lourenco velho,Lourenço Velho,15,,-22.366667,-45.316667
br,lourival nunes,Lourival Nunes,08,,-20.216667,-40.333333
br,louro muller,Louro Müller,26,,-28.4,-49.383333
br,louro poco,Louro Poço,04,,.45,-69.05
br,louveira,Louveira,27,31098,-23.066667,-46.966667
br,lovat,Lovat,18,,-23.533333,-51.7
br,loyola,Loyola,08,,-20.783333,-40.6
br,lua cheia,Lua Cheia,01,,-10.566667,-68.816667
br,lua cheia,Lua Cheia,15,,-16.15,-40.25
br,luanda,Luanda,15,,-21.733333,-45.216667
br,luanda,Luanda,18,,-22.9,-53.166667
br,luanda,Luanda,30,,-7.7,-38.383333
br,lua nova,Lua Nova,05,,-14.3,-39.716667
br,lua nova,Lua Nova,16,,-5.116667,-56.633333
br,lucaia,Lucaia,05,,-14.75,-40.533333
br,lucania,Lucânia,01,,-8.116667,-72.816667
br,lucas borges,Lucas Borges,16,,-.883333,-56.083333
br,lucas costa,Lucas Costa,15,,-17.566667,-41.466667
br,lucaslandia,Lucaslândia,14,,-13.254167,-59.8625
br,lucas,Lucas,14,,-13.116667,-55.933333
br,lucas,Lucas,21,,-22.2895,-42.778508
br,lucas,Lucas,23,,-31.783333,-53.366667
br,lucelia,Lucélia,27,14807,-21.733333,-51.016667
br,lucena,Lucena,13,,-9.283333,-45.916667
br,lucena,Lucena,17,,-6.9,-34.866667
br,luciano dos chicotes,Luciano dos Chicotes,06,,-7.433333,-38.85
br,luciano,Luciano,02,,-9.333333,-38.033333
br,luciano,Luciano,15,,-20.683333,-46.583333
br,lucianopolis,Lucianópolis,27,,-22.45,-49.516667
br,lucianos,Lucianos,15,,-19.8,-50.15
br,lucianos,Lucianos,27,,-24.033333,-47.983333
br,luciano veras,Luciano Veras,22,,-5.656667,-37.484722
br,luciara,Luciára,14,,-10.75,-50.566667
br,lucio da luz,Lúcio da Luz,14,,-10.6,-51.35
br,lucios,Lúcios,15,,-22.6002,-45.6345
br,lucios,Lúcios,21,,-22.279682,-42.785642
br,lucrecia,Lucrécia,22,,-6.116667,-37.816667
br,lucrecia,Lucrécia,23,,-30.633333,-53.15
br,lucrecio,Lucrecio,23,,-30.633333,-53.15
br,lucuri,Lucuri,05,,-12.233333,-40.783333
br,ludgero,Ludgéro,30,,-7.483333,-39.8
br,ludovico,Ludovico,21,,-22.116667,-41.316667
br,lufa,Lufa,15,,-17.25,-41.916667
br,lugar combate,Lugar Combate,24,,-10.266667,-62.968056
br,lugar grande,Lugar Grande,20,,-5.883333,-41.133333
br,lugar tira serveja,Lugar Tira Serveja,24,,-10.333333,-62.933333
br,luisa,Luisa,22,,-6.216667,-36.683333
br,luis alves,Luís Alves,26,,-26.733333,-48.95
br,luisana,Luisana,18,,-24.3,-52.283333
br,luis antonio,Luís Antônio,27,,-21.5,-47.683333
br,luis antonio,Luís Antônio,30,,-8.933333,-38.166667
br,luisburgo,Luisburgo,15,,-20.433333,-42.1
br,luis candido,Luis Cândido,15,,-20.583333,-45.383333
br,luis carlos,Luis Carlos,27,,-23.4573,-46.0584
br,luis carlos manso,Luís Carlos Manso,15,,-21.45,-46.066667
br,luis correia,Luís Correia,20,11480,-2.883611,-41.666111
br,luis domingues,Luís Domingues,13,,-1.3,-45.866667
br,luis eugenio,Luís Eugênio,15,,-16.05,-40.233333
br,luis flores,Luís Flores,24,,-11.283333,-65.366667
br,luis gomes,Luís Gomes,22,,-6.416667,-38.383333
br,luis goncalves,Luís Gonçalves,16,,-5.616667,-50.416667
br,luis gonzaga,Luís Gonzaga,15,,-20.233333,-46
br,luisiana,Luisiana,18,,-24.3,-52.283333
br,luisiania,Luisiânia,21,,-22.033333,-41.516667
br,luisiania,Luisiânia,27,,-21.683333,-50.283333
br,luislandia,Luislândia,15,,-16.071667,-44.659167
br,luis,Luís,15,,-14.633333,-44.733333
br,luis,Luís,30,,-8.8,-36.233333
br,luis paulino,Luís Paulino,15,,-21.483333,-45.966667
br,luis pereira,Luís Pereira,30,,-7.95,-38.683333
br,luis pires de minas,Luís Pires de Minas,15,,-16.491667,-44.267778
br,luis prado,Luís Prado,15,,-21.483333,-45.7
br,luis tanare,Luís Tanare,15,,-16.2,-41.516667
br,luis tristao,Luís Tristão,15,,-21.45,-46.516667
br,luis viana,Luís Viana,05,,-8.75,-41.233333
br,luiza de brito,Luíza de Brito,05,,-12.85,-42.016667
br,luiz alves,Luiz Alves,26,,-26.733333,-48.95
br,luiz antonio,Luiz Antonio,27,,-21.55,-47.716667
br,luiz antonio,Luiz Antonio,30,,-8.933333,-38.166667
br,luizburgo,Luizburgo,15,,-20.433333,-42.1
br,luiz correia,Luiz Correia,20,,-2.883611,-41.666111
br,luiz duranto,Luiz Duranto,23,,-29.0339,-51.0466
br,luiz flores,Luiz Flõres,24,,-11.283333,-65.366667
br,luiz gomes,Luiz Gomes,22,,-6.416667,-38.383333
br,luiziania,Luiziânia,27,,-21.683333,-50.283333
br,luiz menegon,Luiz Menegon,23,,-29.0486,-51.1372
br,luiz pereira,Luiz Pereira,15,,-18.95,-44.116667
br,luiz viana,Luiz Viana,05,,-8.75,-41.233333
br,lumiar,Lumiar,21,,-22.366667,-42.2
br,luminarias,Luminárias,15,,-21.5,-44.9
br,luminosa,Luminosa,15,,-22.5814,-45.6351
br,luna,Luna,06,,-5.95,-39.366667
br,lunardelli,Lunardelli,18,,-23,-51.2
br,lunburgo,Lunburgo,15,,-20.433333,-42.1
br,lundiacanga,Lundiacanga,27,,-23.5716,-47.6827
br,lundiara,Lundiara,05,,-14.283333,-42.283333
br,lupercio,Lupércio,27,,-22.416667,-49.8
br,lupionopolis,Lupionópolis,18,,-22.733333,-51.666667
br,lupuna,Lupuna,01,,-9.65,-72.033333
br,lurdes,Lurdes,15,,-21.763889,-43.328889
br,lusitania,Lusitânia,11,,-18.416667,-55.783333
br,lusitania,Lusitânia,24,,-9.133333,-64.183333
br,lusitania,Lusitânia,27,,-21.133333,-48.266667
br,lustosa,Lustosa,05,,-12.2,-38.6
br,lustosa,Lustosa,20,,-4.333333,-42.55
br,lutecia,Lutécia,27,,-22.333333,-50.383333
br,luzeia,Luzéia,04,,-3.316667,-57.666667
br,luzeiro,Luzeiro,01,,-7.833333,-72.683333
br,luzelia,Luzélia,04,,-3.316667,-57.666667
br,luzerna,Luzerna,26,,-27.116667,-51.466667
br,luzia,Luzia,06,,-5.066667,-40.133333
br,luzia,Luzia,20,,-7.416667,-40.8
br,luzia,Lúzia,30,,-7.983333,-36.533333
br,luziania,Luziânia,29,143601,-16.25,-47.933333
br,luzilandia,Luzilândia,08,,-18.7,-40.533333
br,luzilandia,Luzilândia,20,12999,-3.466667,-42.366667
br,luzilandia,Luzilândia,29,,-16.25,-47.933333
br,luzio,Luzio,15,,-18.7,-48.283333
br,luzitana,Luzitana,21,,-21.866667,-41.466667
br,luzitania,Luzitania,11,,-18.416667,-55.783333
br,luzitania,Luzitânia,24,,-9.133333,-64.183333
br,luzitania,Luzitânia,27,,-20.866667,-50.25
br,luz,Luz,15,14396,-19.8,-45.666667
br,luz,Luz,16,,-8.7,-49.616667
br,luz,Luz,23,,-28.633333,-50.85
br,lyndoya,Lyndoya,27,,-22.516667,-46.65
br,mabederi,Mabederi,04,,-7.266667,-65.116667
br,mabedery,Mabedery,04,,-7.266667,-65.116667
br,mabe,Mabé,04,,.75,-67.216667
br,mabideri,Mabideri,04,,-7.266667,-65.116667
br,macabu,Macabu,21,,-22.066667,-41.866667
br,macabuzinho,Macabuzinho,21,,-22.083333,-41.733333
br,macacaria,Macacaria,20,,-3.5675,-42.142222
br,macacas,Macacas,18,,-24.633333,-52.333333
br,macaco branco,Macaco Branco,27,,-24.686856,-48.223039
br,macacoeira,Macacoeira,13,,-1.216667,-46.1
br,macaco,Macaco,05,,-12.466667,-38.833333
br,macaco,Macaco,06,,-3.433333,-39.233333
br,macaco,Macaco,11,,-19.116667,-54.25
br,macaco,Macaco,13,,-10.05,-45.95
br,macaco,Macaco,14,,-14.8,-56.666667
br,macaco,Macaco,15,,-16.183333,-40.8
br,macaco,Macaco,17,,-7.2,-38.166667
br,macaco,Macaco,18,,-25.135212,-48.356261
br,macaco,Macaco,20,,-5.116667,-42.316667
br,macaco,Macaco,21,,-21.983333,-42.266667
br,macaco,Macaco,22,,-6.5,-38.1
br,macaco,Macaco,23,,-32.125,-52.108333
br,macaco-prego,Macaco-Prego,04,,-4.168056,-60.841944
br,macacos,Macacos,02,,-9.666667,-36.683333
br,macacos,Macacos,05,,-12.6,-39.783333
br,macacos,Macacos,06,,-5.316667,-39.883333
br,macacos,Macacos,14,,-14.8,-56.666667
br,macacos,Macacos,15,,-16.316667,-41.9
br,macacos,Macacos,17,,-7.033333,-36.9
br,macacos,Macacos,18,,-24.633333,-52.333333
br,macacos,Macacos,20,,-3.888056,-42.526667
br,macacos,Macacos,22,,-6.35,-37.933333
br,macacos,Macacos,29,,-15.75,-48.316667
br,macacos,Macacos,30,,-7.833333,-37.533333
br,macacu,Macacu,08,,-19.3,-40.133333
br,macado,Macado,25,,1.066667,-60.916667
br,macae,Macaé,21,143031,-22.383333,-41.783333
br,macahe,Macahé,21,,-22.383333,-41.783333
br,macahubas,Macahubas,05,,-13.033333,-42.7
br,macahubas,Macahubas,27,,-20.8,-49.95
br,macahyba,Macahyba,22,,-5.85,-35.35
br,macaia,Macaia,15,,-21.133333,-44.916667
br,macaiba,Macaíba,22,40016,-5.716667,-35.55
br,macajatuba,Macajatuba,13,,-2.483333,-44.15
br,macajuba,Macajuba,05,,-12.15,-40.366667
br,macajuba,Macajuba,13,,-3.97,-43.154444
br,macambara,Macambará,21,,-22.325,-43.526389
br,macambara,Maçambará,23,,-29.133333,-56.05
br,macambira,Macambira,05,,-11.583333,-44.15
br,macambira,Macambira,06,,-3.8,-39.983333
br,macambira,Macambira,13,,-3.074722,-43.415278
br,macambira,Macambira,17,,-7.216667,-38.083333
br,macambira,Macambira,20,,-4.1,-41.95
br,macambira,Macambira,22,,-6.283333,-37.983333
br,macambira,Macambira,28,,-10.666667,-37.533333
br,macambira,Macambira,30,,-8.6,-36.433333
br,macambira velha,Macambira Velha,06,,-4.55,-41.116667
br,macanare,Macanaré,04,,.416667,-67.3
br,macaoca,Macaoca,06,,-4.666667,-39.433333
br,macao,Macao,04,,-7.45,-70.55
br,macao,Macáo,22,,-5.116667,-36.633333
br,macapa,Macapá,03,338937,.033333,-51.05
br,macapa,Macapá,04,,-9.483333,-67.5
br,macapa,Macapá,06,,-7.683333,-39.016667
br,macapa,Macapá,13,,-2.633333,-44.9
br,macapa,Macapá,16,,-1.366667,-47.883333
br,macapa,Macapá,17,,-6.933333,-37.5
br,macapa,Macapá,20,,-2.908889,-41.442222
br,macapa,Macapá,30,,-7.566667,-35.45
br,macaparana,Macaparana,06,,-7.683333,-39.016667
br,macaparana,Macaparana,30,14363,-7.566667,-35.45
br,macapuana,Macapuana,04,,-3.6,-69.366667
br,macarandiba,Maçarandiba,17,,-7.183333,-35.766667
br,macaranduba,Maçaranduba,05,,-13.766667,-39.35
br,macaranduba,Maçaranduba,06,,-7.716667,-38.883333
br,macaranduba,Maçaranduba,17,,-7.183333,-35.766667
br,macaranduba,Maçaranduba,20,,-2.961111,-41.426389
br,macaranduba,Macaranduba,22,,-5.733333,-35.366667
br,macaranduba,Maçaranduba,26,,-26.6,-49.016667
br,macaranduba,Maçaranduba,30,,-8.483333,-36.466667
br,macarani,Macarani,05,,-15.55,-40.4
br,macarau,Macaraú,06,,-4.2,-40.433333
br,macario,Macário,03,,.066667,-51.233333
br,macaroca,Maçaroca,05,,-9.883333,-40.283333
br,macaroca,Maçaroca,25,,-.678889,-61.955833
br,macata,Macatá,26,,-26.366667,-50.983333
br,macatuba,Macatuba,27,16180,-22.516667,-48.683333
br,macauari,Macauari,04,,-4.133333,-57.583333
br,macauary,Macauary,04,,-4.133333,-57.583333
br,macaubal,Macaubal,27,,-20.8,-49.95
br,macauba,Macaúba,05,,-12.65,-40.4
br,macauba,Macauba,11,,-22.183333,-54.516667
br,macauba,Macaúba,13,,-6.65,-44.016667
br,macauba,Macaúba,16,,-5.566667,-49.933333
br,macauba,Macaúba,27,,-20.583333,-49.5
br,macauba,Macaúba,31,,-10.516667,-50.5
br,macaubas,Macaúbas,05,14598,-13.033333,-42.7
br,macaubas,Macaúbas,15,,-20.816667,-46.25
br,macaubas,Macaúbas,27,,-20.583333,-49.5
br,macau,Macau,04,,-7.45,-70.55
br,macau,Macau,22,18070,-5.116667,-36.633333
br,macauzinho,Macáuzinho,22,,-5.148611,-36.605278
br,macava,Macava,06,,-5.166667,-37.916667
br,macavari,Macavari,04,,-4.133333,-57.583333
br,macayo,Maçayó,02,,-9.666667,-35.716667
br,macazeira,Macazeira,05,,-12.333333,-41.033333
br,macedina,Macedina,14,,-16.066667,-52.4
br,macedonia,Macedônia,27,,-20.116667,-50.216667
br,maceio,Maceió,01,,-8.333333,-70.916667
br,maceio,Maceió,02,955010,-9.666667,-35.716667
br,maceio,Maceió,06,,-2.8825,-40.956944
br,maceio,Maceió,21,,-22.91,-43.064167
br,macena,Macena,02,,-9.35,-36.266667
br,macenas,Macenas,13,,-3.869722,-42.803056
br,machacalis,Machacalis,15,,-17.083333,-40.75
br,machadinha,Machadinha,21,,-22.033333,-41.5
br,machadinho,Machadinho,23,,-27.566667,-51.666667
br,machado,Machado,04,,-4.923611,-62.3625
br,machado,Machado,05,,-13.366667,-39.7
br,machado,Machado,06,,-7.516667,-38.683333
br,machado,Machado,08,,-20.1,-41.333333
br,machado,Machado,13,,-6.35,-44.4
br,machado,Machado,15,33003,-21.683333,-45.933333
br,machado,Machado,25,,1.066667,-60.916667
br,machado,Machado,30,,-8.35,-36.583333
br,machado mineiro,Machado Mineiro,15,,-15.616667,-41.4
br,machado portela,Machado Portela,05,,-13.15,-40.75
br,machados,Machados,15,,-21.033333,-47.066667
br,machados,Machados,27,,-21.883333,-51.366667
br,machados,Machados,30,,-7.7,-35.516667
br,machorra,Machorra,11,,-21.983333,-56.483333
br,macieira,Macieira,26,,-26.85,-51.383333
br,maciel,Maciel,04,,-7.4825,-71.154722
br,maciel,Maciel,05,,-12.45,-39.1
br,maciel,Maciel,16,,-4.666667,-56.3
br,maciel,Maciel,18,,-25.561111,-48.405
br,macinha,Macinha,05,,-12.316667,-39
br,macios,Macios,17,,-7.266667,-38.016667
br,macoapani,Macoapani,04,,-2.533333,-65.5
br,maconha,Maconha,17,,-8.166667,-37.133333
br,macoto,Macotó,21,,-21.85,-41.683333
br,macuari,Macuari,04,,-4.133333,-57.583333
br,macuco,Macuco,05,,-14.95,-39.316667
br,macuco,Macuco,15,,-19.633333,-43.5
br,macuco,Macuco,21,,-21.983333,-42.266667
br,macuco,Macuco,27,,-21.716667,-50.133333
br,macucos,Macucos,27,,-21.716667,-50.133333
br,macucu,Macucu,08,,-19.3,-40.133333
br,macuge,Macugê,30,,-7.5,-35.166667
br,macuim,Maçuim,05,,-12.616667,-38.483333
br,macuje,Macujê,30,,-7.5,-35.166667
br,macujo,Macujó,05,,-13.033333,-38.866667
br,macu,Macu,04,,.416667,-69.25
br,macuna,Macunã,13,,-3.632778,-43.093056
br,macunam,Macunam,06,,-3.833333,-38.633333
br,macupim,Macupim,16,,-1.133333,-46.2
br,macuripari,Macuripari,04,,-5.569444,-63.978333
br,macuripary,Macuripary,04,,-5.569444,-63.978333
br,macuripe,Macuripe,04,,-7.083333,-64.6
br,macurure,Macururé,05,,-9.166667,-39.066667
br,macuxiri,Macuxiri,04,,1.766667,-67.283333
br,macuxixiri,Macuxixiri,04,,1.766667,-67.283333
br,madail,Madail,13,,-4.933333,-45.616667
br,madalena,Madalena,02,,-9.716667,-35.9
br,madalena,Madalena,06,,-4.533333,-40.2
br,madalena,Madalena,13,,-4.15,-44.883333
br,madalena,Madalena,27,,-21.066667,-50.933333
br,madeira cortada iii,Madeira Cortada III,13,,-3.293889,-42.320833
br,madeira cortada i,Madeira Cortada I,13,,-3.271667,-42.26
br,madeira cortada,Madeira Cortada,13,,-3.819444,-43.305556
br,madeira,Madeira,13,,-4.3,-42.983333
br,madeira,Madeira,17,,-7.6,-36
br,madeira,Madeira,20,,-3.486389,-42.5075
br,madeira,Madeira,24,,-10.333333,-65.333333
br,madeira,Madeira,30,,-7.583333,-37.55
br,madeira ricota,Madeira Ricota,27,,-22.616,-45.5756
br,madeireira dal'pai,Madeireira Dal'Pai,18,,-25.5186,-48.9995
br,madeiro,Madeiro,06,,-3.483333,-40.266667
br,maderia nova,Maderia Nova,30,,-7.666667,-39.583333
br,madia,Madia,04,,.916667,-67.05
br,madiberi,Madiberi,04,,-7.266667,-65.116667
br,madragoa,Madragoa,13,,-1.783333,-45.05
br,madre de deus de minas,Madre de Deus de Minas,15,,-21.483333,-44.333333
br,madre de deus,Madre de Deus,05,14887,-12.733333,-38.616667
br,madre-de-deus,Madre-de-Deus,13,,-6.566667,-43.45
br,madre de deus,Madre de Deus,15,,-21.483333,-44.333333
br,madre de deus,Madre de Deus,30,,-8.15,-36.366667
br,madre deus,Madre Deus,13,,-6.566667,-43.45
br,madre,Madre,13,,-6.383333,-43.383333
br,madre,Madre,26,,-28.533333,-48.933333
br,madresilva,Madresilva,21,,-22.233333,-41.95
br,madressilva,Madressilva,21,,-22.233333,-41.95
br,madrugao,Madrugão,05,,-14.25,-43.75
br,madureira,Madureira,21,,-22.881023,-43.339976
br,mae d'agua,Mae d'Agua,17,,-7.25,-37.433333
br,mae d'agua,Mae d'Agua,30,,-8.566667,-38.5
br,mae de deus,Mãe de Deus,06,,-5.55,-40.8
br,mae dos homens,Mãe dos Homens,15,,-18.116667,-42.616667
br,mae luzia,Mãe Luzia,26,,-28.7,-49.466667
br,mae maria,Mãe Maria,16,,-5.25,-48.916667
br,maetinga,Maetinga,05,,-14.666667,-41.5
br,mafra,Mafra,26,40081,-26.1181,-49.8023
br,mafrense,Mafrense,20,,-8.35,-41.066667
br,magalhaes bastos,Magalhães Bastos,21,,-22.876001,-43.410665
br,magalhaes de almeida,Magalhães de Almeida,13,,-3.4,-42.2
br,magalhaes,Magalhães,05,,-12.35,-38.966667
br,magalhaes,Magalhães,11,,-18.233333,-55.35
br,magarai,Magaraí,08,,-20.166667,-40.466667
br,magarca,Magarça,21,,-22.944358,-43.599244
br,magdalena,Magdalena,06,,-4.533333,-40.2
br,magda,Magda,27,,-20.616667,-50.233333
br,mage,Magé,21,216342,-22.655556,-43.015278
br,magno,Magno,21,,-22.872778,-43.34
br,magro,Magro,20,,-6.6,-41.8
br,maguari,Maguari,04,,-5.666389,-63.847778
br,maguari,Maguari,16,,-6.633333,-51.883333
br,maguari,Maguari,17,,-7.133333,-35.1
br,magu,Magu,13,,-2.929167,-41.910833
br,maia,Maia,06,,-6.516667,-38.7
br,maia,Maia,17,,-6.8,-35.566667
br,maia,Maia,27,,-22,-52.066667
br,maiauata,Maiauatá,16,,-1.85,-49.033333
br,mailasqui,Mailasqui,27,,-23.552919,-47.086823
br,mainart,Mainart,15,,-20.466667,-43.333333
br,maioba,Maioba,06,,-2.5,-44.15
br,maiquinique,Maiquinique,05,,-15.616667,-40.25
br,mairi,Mairi,05,13475,-11.716667,-40.133333
br,mairinque,Mairinque,15,,-17.783333,-40.483333
br,mairinque,Mairinque,27,44457,-23.544368,-47.183467
br,mairipora,Mairiporã,27,54417,-23.322628,-46.586935
br,mairipotaba,Mairipotaba,29,,-17.3,-49.466667
br,majaratuba,Majaratuba,04,,.183333,-63.966667
br,maje,Majé,21,,-22.655556,-43.015278
br,maje,Majé,30,,-8.533333,-36.733333
br,major ezequiel,Major Ezequiel,15,,-20.016667,-42.983333
br,major isidoro,Major Isidoro,02,8507,-9.533333,-37
br,major izidoro,Major Izidoro,02,,-9.533333,-37
br,majorlandia,Majorlândia,06,,-4.55,-37.666667
br,major,Major,26,,-27.416667,-48.95
br,major moreno,Major Moreno,14,,-15.533333,-54.8
br,major oscar,Major Oscar,18,,-24.616667,-53.283333
br,major prado,Major Prado,27,,-20.883333,-50.5
br,major sales,Major Sales,22,,-6.4,-38.316667
br,major simplicio,Major Simplício,06,,-4.566667,-40.4
br,major vieira,Major Vieira,26,,-26.316667,-50.333333
br,majuria,Majuriã,04,,-7.5,-64.916667
br,malacacheta,Malacacheta,15,8820,-17.833333,-42.083333
br,malacacheta,Malacacheta,25,,2.7,-60.383333
br,malaquias,Malaquias,22,,-5.333333,-37.533333
br,malaquias,Malaquias,25,,3.333333,-61.1
br,malaripo,Malaripo,16,,.766667,-54.25
br,mal assombrado,Mal Assombrado,22,,-5.9175,-36.114167
br,mal da areia,Mal da Areia,05,,-14.65,-43.333333
br,male,Malé,18,,-25.916667,-50.833333
br,maleta,Maleta,05,,-9.35,-41.9
br,maleta,Maleta,11,,-19.433333,-57.6
br,malet,Malét,18,,-25.916667,-50.833333
br,malhada alta,Malhada Alta,20,,-7.233333,-40.75
br,malhada bonita,Malhada Bonita,05,,-12.016667,-39.683333
br,malhada bonita,Malhada Bonita,30,,-7.783333,-39.633333
br,malhada branca,Malhada Branca,30,,-8.516667,-37.1
br,malhada cortada,Malhada Cortada,30,,-7.983333,-38.3
br,malhada da areia,Malhada da Areia,05,,-13.216667,-41.783333
br,malhada da areia,Malhada da Areia,06,,-7.633333,-38.816667
br,malhada da areia,Malhada da Areia,20,,-3.828889,-42.603611
br,malhada da aroeira,Malhada da Aroeira,06,,-6.533333,-38.766667
br,malhada da aroeira,Malhada da Aroeira,17,,-7.483333,-38.6
br,malhada da barra,Malhada da Barra,20,,-10.416667,-45.216667
br,malhada da cruz,Malhada da Cruz,22,,-6.15,-36.383333
br,malhada da espera,Malhada da Espera,17,,-7.2,-37.583333
br,malhada da jurema,Malhada da Jurema,05,,-10.3,-38.583333
br,malhada da laje,Malhada da Laje,02,,-9.066667,-37.383333
br,malhada da onca,Malhada da Onça,05,,-13.816667,-44.433333
br,malhada d'areia,Malhada d'Areia,06,,-5.383333,-39.15
br,malhada da roca,Malhada da Roca,17,,-7.3,-36.383333
br,malhada das pedras,Malhada das Pedras,13,,-3.558056,-42.587222
br,malhada de areia,Malhada de Areia,22,,-6.65,-37.483333
br,malhada de baixo,Malhada de Baixo,20,,-3.623611,-42.231944
br,malhada de boi,Malhada de Boi,06,,-6.166667,-38.616667
br,malhada de cima,Malhada de Cima,20,,-7.433333,-40.816667
br,malhada de fora,Malhada de Fora,20,,-6.55,-42.7
br,malhada de padre,Malhada de Padre,05,,-14.383333,-41.9
br,malhada de pedra,Malhada de Pedra,05,,-14.383333,-41.9
br,malhada de pedra,Malhada de Pedra,20,,-3.988889,-42.381944
br,malhada de pedra,Malhada de Pedra,30,,-8.6,-38.483333
br,malhada do angico,Malhada do Angico,22,,-6.7,-36.75
br,malhada do bezerro,Malhada do Bezerro,17,,-6.633333,-38.25
br,malhada do boi,Malhada do Boi,30,,-8.25,-38.016667
br,malhada do cordeiro,Malhada do Cordeiro,22,,-6.383333,-36.616667
br,malhada do cosme,Malhada do Cosme,30,,-8.833333,-36.733333
br,malhada do ferreira,Malhada do Ferreira,20,,-6.9,-41.45
br,malhada do juazeiro,Malhada do Juàzeiro,22,,-5.568889,-36.707778
br,malhada do meio,Malhada do Meio,20,,-3.8,-42.05
br,malhada do rio,Malhada do Rio,22,,-5.974444,-36.009167
br,malhada do saco,Malhada do Saco,30,,-9.05,-37.966667
br,malhada do sal,Malhada do Sal,05,,-8.966667,-38.533333
br,malhada dos bois,Malhada dos Bois,28,,-10.333333,-36.933333
br,malhada dos cavalos,Malhada dos Cavalos,13,,-3.434444,-42.571111
br,malhada funda,Malhada Funda,06,,-6.933333,-38.866667
br,malhada grande,Malhada Grande,05,,-12.85,-41.85
br,malhada grande,Malhada Grande,06,,-4.083333,-40.316667
br,malhada grande,Malhada Grande,17,,-7.316667,-38.15
br,malhada grande,Malhada Grande,20,,-3.8525,-42.653333
br,malhada grande,Malhada Grande,30,,-7.716667,-38.85
br,malhada,Malhada,02,,-9.483333,-37.966667
br,malhada,Malhada,05,,-12.083333,-38.783333
br,malhada,Malhada,06,,-3.833333,-38.833333
br,malhada,Malhada,13,,-3.470833,-42.471389
br,malhada,Malhada,17,,-6.633333,-38.416667
br,malhada,Malhada,20,,-3.672222,-42.395278
br,malhada,Malhada,29,,-13.3,-46.583333
br,malhada,Malhada,30,,-7.583333,-39.633333
br,malhada,Malhada,31,,-11.933333,-47.066667
br,malhada nova,Malhada Nova,05,,-10.083333,-38.066667
br,malhada real,Malhada Real,20,,-7.083333,-41.983333
br,malhada redonda,Malhada Redonda,17,,-7.083333,-38.433333
br,malhada redonda,Malhada Redonda,22,,-5.943889,-37.256389
br,malhadas,Malhadas,05,,-12.516667,-38.016667
br,malhada vermelha,Malhada Vermelha,05,,-10.683333,-37.866667
br,malhada vermelha,Malhada Vermelha,06,,-5.566667,-38.533333
br,malhada vermelha,Malhada Vermelha,20,,-6.416667,-42.866667
br,malhada vermelha,Malhada Vermelha,30,,-7.933333,-38.1
br,malhada vermerha,Malhada Vermerha,17,,-7.55,-38.133333
br,malhadinha,Malhadinha,05,,-13.983333,-41.766667
br,malhadinha,Malhadinha,06,,-5.283333,-39.533333
br,malhadinha,Malhadinha,13,,-3.246111,-42.08
br,malhadinha,Malhadinha,17,,-7.866667,-37.033333
br,malhadinha,Malhadinha,20,,-6.533333,-42.116667
br,malhadinha,Malhadinha,22,,-5.611944,-36.044722
br,malhadinha,Malhadinha,30,,-8.133333,-37.35
br,malhadinho,Malhadinho,15,,-15.35,-44.15
br,malhado,Malhado,20,,-3.394722,-42.104167
br,malhador,Malhador,28,5574,-10.65,-37.3
br,malha fria,Malha Fria,30,,-7.716667,-38.633333
br,malheiro,Malheiro,22,,-5.435278,-36.845278
br,malibu,Malibu,21,,-23.00296,-43.396932
br,malica,Maliça,20,,-7.983333,-45.233333
br,malicia,Malícia,13,,-4.466667,-43.283333
br,malicia,Malícia,15,,-15.9,-40.933333
br,malicia,Malícia,20,,-3.736944,-42.485
br,malicia,Malicia,30,,-7.916667,-39.083333
br,mallet,Mallet,18,,-25.4,-52.233333
br,maloca acai,Maloca Acaí,04,,1.115556,-69.777778
br,maloca apuii,Maloca Apuií,04,,-.1,-65.083333
br,maloca arara,Maloca Arara,04,,1.065,-69.301667
br,maloca araripira,Maloca Araripirá,04,,1.184444,-69.086944
br,maloca barrigudo,Maloca Barrigudo,04,,-4.126111,-61.127778
br,maloca caiabi,Maloca Caiabi,14,,-11.366667,-55.533333
br,maloca capi ponta,Maloca Capi Ponta,04,,1.216667,-69.133333
br,maloca caruru,Maloca Caruru,04,,1.089167,-69.313056
br,maloca cipo poco,Maloca Cipó Poço,04,,1.215,-69.384444
br,maloca de barro,Maloca de Barro,25,,4.216667,-60.8
br,maloca de indios,Maloca de Índios,14,,-11.883333,-58.2
br,maloca de indios,Maloca de Índios,24,,-9.683333,-64.983333
br,maloca de indius,Maloca de Indius,24,,-9.683333,-64.983333
br,maloca do baruema,Maloca do Baruema,16,,-8.083333,-54.616667
br,maloca do barurema,Maloca do Barurema,16,,-8.083333,-54.616667
br,maloca do borurema,Maloca do Borurema,16,,-8.083333,-54.616667
br,maloca do capieri,Maloca do Capieri,16,,-7.366667,-57.4
br,maloca do chico,Maloca do Chico,04,,.15,-69
br,maloca do goncalo,Maloca do Gonçalo,16,,-7.766667,-57.433333
br,maloca do joao ferreira,Maloca do João Ferreira,14,,-14.266667,-57.333333
br,maloca dos indios,Maloca dos Índios,24,,-9.683333,-64.983333
br,maloca dos urucuianas,Maloca dos Urucuianas,03,,1.45,-53.916667
br,maloca foz do querari,Maloca Foz do Querari,04,,1.090278,-69.833889
br,maloca iauarete ponta,Maloca Iauaretê Ponta,04,,1.122778,-69.73
br,maloca jacare,Maloca Jacaré,04,,1.042778,-69.4125
br,maloca jandu,Maloca Jandu,04,,1.079444,-69.306667
br,maloca japu,Maloca Japu,04,,1.0575,-69.279444
br,maloca jauarete poco,Maloca Jauaretê Poço,04,,1.203611,-69.297778
br,maloca jiboia,Maloca Jibóia,04,,1.226667,-69.502778
br,maloca jibora,Maloca Jibóra,04,,1.226667,-69.502778
br,maloca ji ponta,Maloca Ji Ponta,04,,1.0425,-69.235278
br,maloca joao ferreira,Maloca João Ferreira,14,,-14.266667,-57.333333
br,maloca jutica,Maloca Jutica,04,,1.078889,-69.476389
br,maloca macu,Maloca Macu,25,,3.683333,-63.716667
br,maloca,Maloca,01,,-10.8,-70.133333
br,maloca,Maloca,04,,-3.35,-63.55
br,maloca,Maloca,16,,-3.783333,-51.783333
br,maloca,Maloca,24,,-9.683333,-64.983333
br,maloca mandurucu,Maloca Mandurucu,04,,-7.6,-58.15
br,maloca matapi,Maloca Matapi,04,,1.086111,-69.363056
br,maloca mundurucu,Maloca Mundurucu,14,,-8.15,-58.216667
br,maloca mundurucus,Maloca Mundurucus,14,,-8.15,-58.216667
br,maloca pacu,Maloca Pacu,04,,1.083889,-69.827222
br,maloca periquito,Maloca Periquito,04,,1.059444,-69.253611
br,maloca ponta de bacaba,Maloca Ponta de Bacaba,04,,1.085,-69.804167
br,maloca puraque ponta,Maloca Puraque Ponta,04,,1.083889,-69.307222
br,maloca repy,Maloca Repy,03,,1.766667,-54.266667
br,maloca salamaim,Maloca Salamaim,24,,-12.25,-61.066667
br,malocas caiabi,Malocas Caiabí,14,,-11.366667,-55.533333
br,maloca taina,Maloca Taina,04,,1.1025,-69.648333
br,maloca tapira geral,Maloca Tapira Geral,04,,1.085278,-69.372222
br,maloca taracua,Maloca Taracuá,04,,1.096389,-69.6625
br,maloca timbe,Maloca Timbé,16,,-3.033333,-47.75
br,maloca uacari,Maloca Uacari,04,,1.224444,-69.350556
br,maloca velha,Maloca Velha,16,,-3.783333,-51.783333
br,malta,Malta,17,,-6.9,-37.516667
br,maltide,Maltide,08,,-20.566667,-40.816667
br,malu,Malu,18,,-23.666667,-52.383333
br,ma luz,Má Luz,26,,-26.166667,-48.533333
br,malva do meio,Malva do Meio,13,,-3.739444,-43.257778
br,malva,Malva,13,,-3.723889,-43.270833
br,mamanguape,Mamanguape,17,27607,-6.833333,-35.116667
br,mamao,Mamão,20,,-7,-40.633333
br,mamata,Mamata,05,,-12.2,-39.65
br,mambai,Mambaí,29,,-14.466667,-46.116667
br,mambore,Mamborê,18,,-24.3,-52.533333
br,mambucaba,Mambucaba,21,,-23.016667,-44.516667
br,mambuquinha,Mambuquinha,15,,-20.85,-46.6
br,mambure,Mamburê,18,,-24.3,-52.533333
br,mamede,Mamede,13,,-3.135,-42.901944
br,mamede,Mamede,20,,-4.416667,-42.866667
br,mameluca,Mameluca,06,,-7.683333,-39.25
br,mameluco,Mameluco,30,,-7.666667,-38.7
br,mamia,Mamiá,04,,-4.016667,-62.866667
br,mamoeiro,Mamoeiro,06,,-6.6,-39.45
br,mamoeiro,Mamoeiro,20,,-4.433333,-41.5
br,mamoeiro,Mamoeiro,30,,-7.966667,-38.933333
br,mamoes,Mamões,20,,-7.066667,-40.566667
br,mamoi,Mamoí,13,,-5.55,-45.533333
br,mamona,Mamona,15,,-15.516667,-41.783333
br,mamonas,Mamonas,05,,-12.766667,-42.733333
br,mamonas,Mamonas,15,,-15.133333,-42.966667
br,mamonas,Mamonas,20,,-7.683333,-41.116667
br,mamoneiro,Mamoneiro,15,,-14.8,-44.483333
br,mamore,Mamoré,24,,-10.816667,-65.333333
br,mamoria,Mamoriá,04,,-7.516667,-66.3
br,mamoria petropolis,Mamoriá Petropolis,04,,-7.516667,-66.3
br,mamori,Mamori,04,,-1.633333,-67.666667
br,mamory,Mamory,04,,-1.633333,-67.666667
br,mamparra,Mamparra,27,,-24.25,-47.916667
br,mamui,Mamuí,13,,-3.023889,-42.347222
br,manaca,Manacá,06,,-6.333333,-40.533333
br,manaca,Manacá,30,,-7.833333,-40.316667
br,manacanau,Manacanaú,22,,-5.372222,-37.1475
br,manacapuru,Manacapuru,04,52460,-3.3,-60.616667
br,manairama,Manairama,22,,-6.7,-36.95
br,manaira,Manaíra,17,4913,-7.7,-38.166667
br,mana,Mana,02,,-9.416667,-36.633333
br,manaos,Manáos,04,,-3.113333,-60.025278
br,manaparu,Manaparu,20,,-6.883333,-40.983333
br,manaquery,Manaquery,04,,-3.316667,-60.35
br,manaquiri,Manaquiri,04,9986,-3.316667,-60.35
br,manari,Manari,30,,-8.816667,-37.583333
br,manauara,Manauara,03,,.283333,-51.166667
br,manaus,Manaus,04,1598227,-3.113333,-60.025278
br,manaus,Manaus,13,,-1.5,-46.183333
br,mancio lima,Mâncio Lima,01,7086,-7.616667,-72.9
br,mandacaia,Mandaçaia,18,,-24.333333,-50.666667
br,mandacaia,Mandaçaia,30,,-8.1,-36.266667
br,mandacaio,Mandaçaio,18,,-24.333333,-50.666667
br,mandacaru,Mandacaru,02,,-9.483333,-36.733333
br,mandacaru,Mandacaru,05,,-12.55,-39.683333
br,mandacaru,Mandacaru,06,,-6.266667,-38.8
br,mandacaru,Mandacaru,13,,-2.6,-42.7
br,mandacaru,Mandacaru,17,,-7.466667,-38.333333
br,mandacaru,Mandacarú,20,,-6.35,-41.4
br,mandacaru,Mandacaru,22,,-6.366667,-38.15
br,mandacaru,Mandacaru,30,,-8.333333,-38.016667
br,mandaguacu,Mandaguaçu,18,14866,-23.333333,-52.083333
br,mandaguari,Mandaguari,15,,-21.516667,-45.716667
br,mandaguari,Mandaguari,18,38239,-23.533333,-51.7
br,mandai,Mandai,13,,-4.933333,-45.616667
br,mandala,Mandala,21,,-23.003497,-43.375252
br,mandani,Mandani,11,,-23.233333,-55.45
br,mandante,Mandante,30,,-7.916667,-38.566667
br,mandassaia,Mandassáia,30,,-8.1,-36.266667
br,mandembo,Mandembo,15,,-20.666667,-45.883333
br,mandicununga,Mandiçununga,27,,-23.102881,-47.69301
br,mandihy,Mandihy,04,,-4.4625,-59.796111
br,mandii,Mandii,04,,-4.4625,-59.796111
br,mandim,Mandim,30,,-9.166667,-38.083333
br,mandiocal,Mandiocal,05,,-13.25,-41.616667
br,mandioca,Mandioca,04,,-4.029722,-60.613056
br,mandioca,Mandioca,13,,-3.905,-43.468889
br,mandiqueira,Mandiqueira,21,,-22.083333,-41.566667
br,mandiquera,Mandiquera,21,,-22.083333,-41.566667
br,mandiqui,Mandiqui,04,,-4.4625,-59.796111
br,mandira,Mandira,27,,-24.993569,-48.038232
br,mandirituba,Mandirituba,18,,-25.7792,-49.3258
br,mandiroba,Mandiroba,05,,-14.483333,-43.05
br,mandiu,Mandiú,27,,-20.65,-47.55
br,manduca,Manduca,25,,2.099914,-59.75621
br,manducas,Manducas,18,,-25.55186,-50.555967
br,manduira,Manduira,18,,-25.462032,-48.798487
br,mandu,Mandu,05,,-12.033333,-38.583333
br,mandu,Mandu,17,,-7.083333,-38.233333
br,mandu,Mandu,27,,-22.9,-45.516667
br,manduri,Manduri,27,,-23.016667,-49.316667
br,mandury,Mandury,27,,-23.016667,-49.316667
br,maneco,Maneco,11,,-22.1,-55.983333
br,maneco rondon,Maneco Rondon,14,,-13.65,-58.3
br,mangaba,Mangaba,11,,-20.216667,-56.116667
br,mangaba,Mangaba,13,,-3.861667,-43.272222
br,mangaba,Mangaba,20,,-3.812778,-42.525833
br,mangabeira,Mangabeira,05,,-12.05,-41.9
br,mangabeira,Mangabeira,06,,-3.666667,-38.75
br,mangabeira,Mangabeira,13,,-3.818611,-43.412778
br,mangabeira,Mangabeira,16,,-2.55,-49.583333
br,mangabeira,Mangabeira,20,,-5.7,-42.683333
br,mangabeira,Mangabeira,22,,-5.833333,-35.3
br,mangabeira,Mangabeira,29,,-13.2,-46.566667
br,mangabeira,Mangabeira,31,,-11.9,-47.233333
br,mangabeira velha,Mangabeira Velha,13,,-3.194167,-43.147778
br,mangabeirinha,Mangabeirinha,13,,-3.299167,-43.228333
br,mangai,Mangaí,11,,-23.233333,-55.45
br,mangalatiba,Mangalatiba,21,,-22.9609,-44.0402
br,mangalo,Mangalô,05,,-12.15,-38.466667
br,manga,Manga,03,,3.733333,-51.716667
br,manga,Manga,05,,-11.466667,-44
br,manga,Manga,06,,-4.416667,-38.883333
br,manga,Manga,11,,-19.233333,-57.216667
br,manga,Manga,13,,-3.516667,-43.95
br,manga,Manga,14,,-15.233333,-59.966667
br,manga,Manga,15,11733,-14.766667,-43.933333
br,manga,Manga,20,,-6.783333,-43.016667
br,manga nova,Manga Nova,30,,-7.033333,-37.716667
br,mangarahy,Mangarahy,08,,-20.166667,-40.466667
br,mangarai,Mangaraí,08,,-20.166667,-40.466667
br,mangarataia,Mangarataia,04,,1.086389,-66.841111
br,mangarate,Mangarate,27,,-20.4,-49.316667
br,mangaratiba,Mangaratiba,21,27023,-22.9609,-44.0402
br,mangaratu,Mangaratu,27,,-20.4,-49.316667
br,mangola,Mangolã,05,,-12.15,-38.466667
br,manguaba,Manguaba,02,,-9.6,-35.933333
br,manguairinha,Manguairinha,18,,-25.95,-52.15
br,manguari,Manguari,04,,-1.605556,-69.268889
br,manguba,Manguba,06,,-3.916667,-38.616667
br,mangueira da pedra,Mangueira da Pedra,23,,-30.216667,-51.8
br,mangueira de pedra,Mangueira de Pedra,23,,-30.516667,-55.616667
br,mangueira ii,Mangueira II,13,,-3.158056,-43.122222
br,mangueira i,Mangueira I,13,,-3.226111,-43.417778
br,mangueiral,Mangueiral,21,,-22.86437,-43.480205
br,mangueira,Mangueira,05,,-12.05,-38.3
br,mangueira,Mangueira,11,,-21.716667,-53.5
br,mangueira,Mangueira,13,,-3.033333,-43.033333
br,mangueira,Mangueira,15,,-15.2,-41.75
br,mangueira,Mangueira,16,,-2.883333,-46.65
br,mangueira,Mangueira,20,,-5.35,-42.316667
br,mangueira,Mangueira,21,,-21.316667,-41.416667
br,mangueira,Mangueira,23,,-32.183333,-52.166667
br,mangueira,Mangueira,25,,3.733333,-60.466667
br,mangueiras,Mangueiras,16,,-5.333333,-49.133333
br,mangueiras,Mangueiras,21,,-22.549167,-43.578611
br,mangueirinha,Mangueirinha,18,,-25.95,-52.15
br,mangue,Mangue,13,,-1.15,-45.716667
br,mangue,Mangue,30,,-9.1,-36.266667
br,manguenza caboclo,Manguenza Cabôclo,17,,-7.483333,-37.983333
br,manguenza,Manguenza,17,,-7.366667,-37.733333
br,mangue secco,Mangue Secco,05,,-11.466667,-37.366667
br,mangue seco,Mangue Sêco,05,,-11.466667,-37.366667
br,mangue seco,Mangue Seco,06,,-2.841944,-40.553611
br,manguinho,Manguinho,05,,-11.316667,-41.1
br,manguinhos,Manguinhos,05,,-12.9,-38.616667
br,manguinhos,Manguinhos,08,,-20.183333,-40.2
br,manguinhos,Manguinhos,21,,-21.4,-41
br,mangunca,Mangunça,13,,-1.616667,-44.666667
br,manha,Manhã,31,,-12.716667,-46.4
br,manhuacu,Manhuaçu,15,54608,-20.25,-42.033333
br,manhuassu,Manhuassú,15,,-20.1,-41.933333
br,manhumirim,Manhumirim,15,15544,-20.366667,-41.95
br,maniacu,Maniaçu,05,,-13.85,-42.433333
br,maniassu,Maniassu,05,,-13.85,-42.433333
br,manibu,Manibu,06,,-4.8,-37.266667
br,manica,Manica,05,,-11.416667,-38.45
br,manicobal,Maniçobal,30,,-7.866667,-38.766667
br,manicoba,Maniçoba,05,,-9.25,-40.3
br,manicoba,Maniçoba,06,,-6.1,-38.55
br,manicoba,Maniçoba,17,,-6.65,-37.816667
br,manicoba,Manicoba,22,,-5.659444,-36.063333
br,manicoba,Maniçoba,30,,-8.25,-36.1
br,manicore,Manicoré,04,14058,-5.812222,-61.2975
br,manimbu,Manimbu,02,,-10.166667,-36.366667
br,manino,Manino,20,,-7.2,-40.633333
br,manissobal,Manissobal,30,,-7.866667,-38.766667
br,manituba,Manituba,06,,-5.216667,-39.566667
br,maniva,Maniva,21,,-21.4,-41.15
br,manjericao,Manjericão,22,,-5.941389,-36.329444
br,manoel alexandre,Manoel Alexandre,06,,-6.483333,-39.183333
br,manoel borba,Manoel Borba,30,,-7.583333,-35.5
br,manoel branco,Manoel Branco,14,,-8.266667,-58.1
br,manoel cardozo,Manoel Cardozo,16,,-7.666667,-57.583333
br,manoel da hora,Manoel da Hora,30,,-8.366667,-36.15
br,manoel de freitas,Manoel de Freitas,23,,-29.05,-55.033333
br,manoel de mato,Manoel de Mato,17,,-7.65,-37.916667
br,manoel dias,Manoel Dias,06,,-3.75,-40.016667
br,manoel dias,Manoel Dias,22,,-6.016667,-36.733333
br,manoel dos santos,Manoel dos Santos,16,,-1.183333,-47.016667
br,manoel ferreira,Manoel Ferreira,06,,-5.916667,-40.65
br,manoel lopes,Manoel Lopes,13,,-3.8675,-43.393056
br,manoel,Manoel,24,,-11.283333,-65.366667
br,manoel mororo,Manoel Mororó,06,,-7.35,-38.933333
br,manoel pereira,Manoel Pereira,05,,-12.316667,-40.533333
br,manoel raimundo,Manoel Raimundo,20,,-3.799167,-42.653333
br,manoelsinho,Manoelsinho,16,,-7.35,-54.85
br,manoel soares,Manoel Soares,30,,-7.633333,-37.683333
br,manoel thomaz,Manoel Thomaz,14,,-8.85,-60.7
br,manoel tomaz,Manoel Tomaz,14,,-8.85,-60.7
br,manoel urbano,Manoel Urbano,01,4066,-8.883333,-69.3
br,manoel viana,Manoel Viana,23,,-29.583333,-55.483333
br,manos,Manos,04,,-3.113333,-60.025278
br,mansao,Mansão,05,,-13.483333,-40.633333
br,mansidao,Mansidão,05,,-10.716667,-44.033333
br,mansinha,Mansinha,06,,-7.066667,-38.866667
br,mansinha,Mansinha,30,,-8.366667,-37.616667
br,mansinha,Mansinha,31,,-9.476667,-47.172778
br,mansinho,Mansinho,06,,-5.8,-38.133333
br,mansinho,Mansinho,13,,-3.716111,-42.778611
br,mansinho,Mansinho,20,,-5.133333,-41.683333
br,manso,Manso,13,,-5.55,-45.533333
br,mantena,Mantena,15,18715,-18.783333,-40.983333
br,manteninha,Manteninha,08,,-18.883333,-41.1
br,mantenopolis,Mantenópolis,08,8547,-18.883333,-41.1
br,mantiqueira,Mantiqueira,15,,-21.95,-44.183333
br,mantiqueira,Mantiqueira,21,,-22.083333,-41.566667
br,mantiqueira,Mantiqueira,27,,-23.371654,-46.573866
br,mantiqueira,Mantiqueira,30,,-7.75,-38.833333
br,manto,Manto,06,,-6.35,-40.016667
br,manuel andrade,Manuel Andrade,15,,-20.733333,-46.65
br,manuel barros,Manuel Barros,15,,-18.266667,-41.666667
br,manuel-borba,Manuel-Borba,30,,-7.583333,-35.5
br,manuel botelho,Manuel Botelho,31,,-10.733333,-48.95
br,manuel branco,Manuel Branco,14,,-8.266667,-58.1
br,manuel cabra,Manuel Cabra,05,,-13.566667,-41.85
br,manuel cantulina,Manuel Cantulina,15,,-16.483333,-41.816667
br,manuel cardoso,Manuel Cardoso,16,,-7.666667,-57.583333
br,manuel costa,Manuel Costa,23,,-30.85,-55.7
br,manuel da hora,Manuel da Hora,30,,-8.366667,-36.15
br,manuel da silva,Manuel da Silva,15,,-20.866667,-46.666667
br,manuel de morais,Manuel de Morais,21,,-22.033333,-42.15
br,manuel domingues,Manuel Domingues,15,,-21.25,-46.05
br,manuel duarte,Manuel Duarte,21,,-22.085,-43.558611
br,manuel dutra,Manuel Dutra,15,,-21.016667,-46.8
br,manuel emidio,Manuel Emídio,15,,-16.233333,-40.05
br,manuel emidio,Manuel Emídio,20,,-8.033333,-43.833333
br,manuel feliciano,Manuel Feliciano,05,,-16.266667,-39.85
br,manuel fialho,Manuel Fialho,23,,-31.066667,-55.983333
br,manuel freitas,Manuel Freitas,23,,-29.05,-55.033333
br,manuel gomes,Manuel Gomes,15,,-17.533333,-41.483333
br,manuel lopes,Manuel Lopes,15,,-20.116667,-45.5
br,manuel,Manuel,24,,-11.283333,-65.366667
br,manuel parreira,Manuel Parreira,15,,-17.983333,-41.7
br,manuel pereira,Manuel Pereira,05,,-11.4,-39.766667
br,manuel pereira,Manuel Pereira,15,,-20.483333,-45.583333
br,manuel ribas,Manuel Ribas,18,,-24.516667,-51.65
br,manuelsinho,Manuelsinho,16,,-7.35,-54.85
br,manuel tavares,Manuel Tavares,21,,-22.133333,-41.6
br,manuel tomas,Manuel Tomás,14,,-8.85,-60.7
br,manuel urbano,Manuel Urbano,01,,-8.883333,-69.3
br,manuel viana,Manuel Viana,23,,-29.583333,-55.483333
br,manuelzinho,Manuelzinho,04,,-4.947778,-62.371389
br,manuelzinho,Manuelzinho,16,,-7.35,-54.85
br,mao beijada,Mão Beijada,17,,-8.1,-37.016667
br,mao direita,Mão Direita,06,,-6.066667,-38.933333
br,mao,Mao,21,,-20.816667,-41.75
br,mapada,Mapada,04,,-2.375,-61.09
br,mapana,Mapana,04,,-2.933333,-67.883333
br,mapangapa,Mapangapa,04,,-8.7,-67.383333
br,mapaugapa,Mapaugapa,04,,-8.7,-67.383333
br,mapele,Mapele,05,,-12.783333,-38.433333
br,mapia,Mapiá,04,,-8.45,-67.45
br,mapireme,Mapireme,16,,.516667,-54.083333
br,mapirunga,Mapirunga,06,,-7.233333,-40.283333
br,mapixi,Mapixi,04,,-5.617222,-64.012778
br,mapixy,Mapixy,04,,-5.666667,-63.883333
br,mapua,Mapuá,06,,-5.983333,-38.666667
br,maquina,Máquina,21,,-21.283333,-41.066667
br,maquine,Maquiné,05,,-11.05,-42.35
br,maquine,Maquiné,20,,-5.966667,-41.383333
br,maquine,Maquiné,23,,-29.683333,-50.183333
br,maquinista maino,Maquinista Maino,23,,-28.166667,-52.35
br,maquinista severo,Maquinista Severo,23,,-28.25,-53.133333
br,maraa,Maraã,04,4387,-1.833333,-65.366667
br,maraba,Marabá,16,145865,-5.35,-49.116667
br,maraba paulista,Marabá Paulista,27,,-22.1,-51.933333
br,marabitana,Marabitana,16,,-.883333,-47.983333
br,marabitanas,Marabitanas,04,,.966667,-66.85
br,marabitanas,Marabitanas,16,,-.883333,-47.983333
br,maracachita,Maracachita,25,,2.533333,-60.6
br,maracacume,Maracaçumé,13,14164,-2.05,-45.933333
br,maracahy,Maracahy,27,,-22.6,-50.65
br,maracaibo,Maracaibo,24,,-12.404167,-60.604167
br,maracai,Maracaí,27,11960,-22.6,-50.65
br,maracaja,Maracajá,04,,.233333,-69.65
br,maracaja,Maracajá,06,,-4.516667,-41.066667
br,maracaja,Maracajá,13,,-3.916667,-44.2
br,maracaja,Maracajá,16,,-4.147778,-50.220278
br,maracaja,Maracajá,22,,-5.568889,-36.392778
br,maracaja,Maracajá,26,,-28.85,-49.466667
br,maracajatuba,Maracajatuba,16,,-3.416667,-56.45
br,maracajau,Maracajaú,22,,-5.416667,-35.316667
br,maracaju,Maracaju,11,22690,-21.633333,-55.15
br,maraca,Maracá,03,,2.133333,-50.5
br,maraca,Maracá,30,,-7.916667,-36.116667
br,maracana ll,Maracanã ll,23,,-29.7717,-50.7695
br,maracana,Maracanã,04,,-.366667,-66.466667
br,maracana,Maracanã,05,,-12.75,-38.533333
br,maracana,Maracanã,06,,-2.933333,-40.033333
br,maracana,Maracanã,16,12683,-.766667,-47.45
br,maracana,Maracanã,21,,-22.91453,-43.225466
br,maracana,Maracanã,23,,-29.7719,-50.778
br,maracana,Maracanã,30,,-7.733333,-38.85
br,maracanau,Maracanaú,06,193532,-3.866667,-38.633333
br,maracani,Maracaní,25,,3.183333,-61.183333
br,maracanu,Maracanu,25,,3.183333,-61.183333
br,maracas,Maracás,05,22050,-13.433333,-40.45
br,maracassume,Maracassumé,13,,-2.05,-45.933333
br,maracoeira,Maracoeira,13,,-1.233333,-46.1
br,maraconahu,Maraconahú,06,,-3.866667,-38.633333
br,maracuja,Maracujá,01,,-8.383333,-71
br,maracuja,Maracujá,05,,-11.516667,-40.216667
br,maracuja,Maracujá,06,,-3.516667,-40.05
br,maracuja,Maracujá,17,,-7.083333,-37.5
br,maracuja,Maracujá,20,,-3.691389,-42.420278
br,marada,Marada,16,,-5.35,-49.116667
br,marafi,Marafi,04,,-5.716667,-67.783333
br,maragao,Maragão,06,,-4.5,-38.783333
br,maragassume,Maragassume,13,,-2.05,-45.933333
br,maragipinho,Maragipinho,05,,-13.05,-38.95
br,maragogi,Maragogi,02,16066,-9.016667,-35.216667
br,maragogipe,Maragogipe,05,20921,-12.766667,-38.916667
br,maragogipinho,Maragogipinho,05,,-13.05,-38.95
br,maragogy,Maragogy,02,,-9.016667,-35.216667
br,maragoji,Maragoji,02,,-9.016667,-35.216667
br,maragojipe,Maragojipe,05,,-12.766667,-38.916667
br,maragojipinho,Maragojipinho,05,,-13.05,-38.95
br,maragua,Maraguá,06,,-7.3,-38.55
br,marahu,Marahú,05,,-14.1,-39
br,maraial,Maraial,30,14204,-8.783333,-35.833333
br,maraja,Marajá,04,,-4.983333,-60.233333
br,maraja,Marajá,13,,-3.046389,-43.436111
br,maraja,Marajá,20,,-3.793889,-42.421111
br,marajazinho,Marajazinho,04,,-5.014444,-60.265
br,marajoara,Marajoara,27,,-21.583333,-51.233333
br,marala,Marala,23,,-29.55,-51.566667
br,mar a mar,Mar A Mar,21,,-23.024172,-43.496663
br,marambaia,Marambaia,15,,-17.183333,-41.583333
br,marambaia,Marambaia,21,,-22.766667,-42.933333
br,marambainha,Marambainha,15,,-17.183333,-41.583333
br,maranduva,Maranduva,27,,-24.566513,-48.110699
br,maranga,Marangá,21,,-23.004287,-43.451551
br,marangatu,Marangatu,21,,-21.616667,-42.216667
br,marangua,Maranguá,21,,-22.866667,-42.6
br,maranguape,Maranguape,01,,-9.166667,-72.266667
br,maranguape,Maranguape,06,,-3.883333,-38.666667
br,maranhao,Maranhão,13,,-2.516667,-44.266667
br,maranhao,Maranhão,17,,-7.766667,-36.116667
br,marapanim,Marapanim,16,10236,-.7,-47.7
br,marape,Marapé,08,,-20.216667,-41.3
br,marape,Marape,14,,-12,-56.65
br,marapica,Marapica,21,,-22.804191,-43.58837
br,marapicu,Marapicu,21,,-22.804191,-43.58837
br,marapoama,Marapoama,27,,-21.25,-49.133333
br,marapuama,Marapuama,27,,-21.25,-49.133333
br,marari,Marari,04,,-5.716667,-67.783333
br,marari,Marari,26,,-27.216667,-51.083333
br,mararupa,Mararupá,06,,-7.25,-38.766667
br,marary,Marary,04,,-5.716667,-67.783333
br,marasco,Marasco,23,,-32.816667,-53.066667
br,marataizes,Marataízes,08,25397,-21.033333,-40.833333
br,marata,Maratá,23,,-29.55,-51.566667
br,marata,Maratá,26,,-26.366667,-50.983333
br,maraua,Marauá,04,,-2.9,-66.083333
br,marau,Maraú,05,2956,-14.1,-39
br,marau,Marau,16,,-4.05,-56.883333
br,marau,Marau,23,26860,-28.45,-52.2
br,maravilha,Maravilha,02,,-9.233333,-37.3
br,maravilha,Maravilha,04,,-2.512778,-60.786111
br,maravilha,Maravilha,05,,-10.866667,-40.016667
br,maravilha,Maravilha,14,,-15.816667,-56.066667
br,maravilha,Maravilha,15,,-17.583333,-41.116667
br,maravilha,Maravilha,20,,-8.133333,-42.133333
br,maravilha,Maravilha,25,,3.433333,-60.183333
br,maravilha,Maravilha,26,13865,-26.783333,-53.15
br,maravilha,Maravilha,30,,-8.55,-40.65
br,maravilha nova,Maravilha Nova,30,,-8.55,-40.65
br,maravilhas,Maravilhas,15,,-18.033333,-43.216667
br,maraxo paia,Maraxó Paiá,16,,1.433333,-56.533333
br,marayal,Marayal,30,,-8.783333,-35.833333
br,marazinho,Marazinho,04,,-4.997778,-60.253889
br,marcacac,Marcaçâc,22,,-5.749444,-36.921667
br,marcacao,Marcação,02,,-10,-36.95
br,marcacao,Marcação,17,,-6.566667,-36.366667
br,marcal ii,Marçal II,13,,-3.363611,-43.348333
br,marcal i,Marçal I,13,,-3.326667,-43.459722
br,marcal,Marçal,13,,-3,-45.083333
br,marcal,Marçal,20,,-6.916667,-40.533333
br,marcelina,Marcelina,17,,-6.233333,-37.683333
br,marcelino,Marcelino,04,,-1.75,-66.35
br,marcelino,Marcelino,05,,-17.666667,-40.266667
br,marcelino,Marcelino,15,,-16.166667,-40.9
br,marcelino,Marcelino,18,,-25.772401,-49.233989
br,marcelino ramos,Marcelino Ramos,23,,-27.466667,-51.9
br,marcelino vieira,Marcelino Vieira,22,,-6.3,-38.166667
br,marcellino,Marcellino,04,,.9,-67.2
br,marcellino ramos,Marcellino Ramos,23,,-27.466667,-51.9
br,marcelo guedes,Marcelo Guedes,15,,-18.15,-41.683333
br,marcelo,Marcelo,05,,-10.816667,-40.316667
br,marcelo,Marcelo,16,,-2.2,-51.45
br,marcelo,Marcelo,17,,-7.233333,-38.3
br,marcia b. serpa,Márcia B. Serpa,23,,-30.1,-56.2
br,marcianopolis,Marcianópolis,18,,-26.110278,-53.646667
br,marcilio dias,Marcílio Dias,26,,-26.133333,-50.383333
br,marco brasileiro,Marco Brasileiro,24,,-10.4,-65.366667
br,marco do riachao,Marco do Riachão,20,,-6.133333,-42.966667
br,marcolandia,Marcolândia,30,,-7.433333,-40.65
br,marcolino,Marcolino,05,,-12.15,-41
br,marcolino,Marcolino,15,,-17.25,-41.3
br,marcolino moco,Marcolino Moco,15,,-17.166667,-40.883333
br,marcolino moura,Marcolino Moura,05,,-13.583333,-41.716667
br,marco,Marco,06,14502,-3.133333,-40.15
br,marco,Marco,13,,-3.951944,-43.329167
br,marco,Marco,20,,-3.351389,-42.113333
br,marcondesia,Marcondésia,27,,-20.866667,-48.7
br,marcondes,Marcondes,15,,-18.55,-48.1
br,marco prete,Marco Prete,06,,-3.983333,-40.316667
br,marco rondon,Marco Rondon,24,,-12.016667,-60.866667
br,marcos-de-honorato,Marcos-de-Honorato,14,,-10.283333,-60.516667
br,marcos,Marcos,05,,-13.3,-40.383333
br,marcos,Marcos,13,,-3.734722,-42.792222
br,marcos,Marcos,17,,-6.516667,-35.133333
br,marcos parente,Marcos Parente,20,,-7.25,-43.7
br,marcos rocha,Marcos Rocha,15,,-15.033333,-46.8
br,marco zero,Marco Zero,18,,-22.95,-53.416667
br,mar de espanha,Mar de Espanha,15,9678,-21.865833,-43.009722
br,mar de hespanha,Mar de Hespanha,15,,-21.865833,-43.009722
br,mar d'hespanha,Mar d'Hespanha,15,,-21.865833,-43.009722
br,marecas,Marecas,04,,-4.820556,-62.603889
br,marechal al floriano,Marechal al Floriano,18,,-25.316667,-53.883333
br,marechal bormann,Marechal Bormann,26,,-27.233333,-52.6
br,marechal candido rondon,Marechal Cândido Rondon,18,43028,-24.566667,-54.066667
br,marechal deodoro,Marechal Deodoro,02,50579,-9.716667,-35.9
br,marechal floriano,Marechal Floriano,02,,-9.45,-37.766667
br,marechal floriano,Marechal Floriano,08,5869,-20.416667,-40.666667
br,marechal floriano,Marechal Floriano,29,,-15.95,-51.25
br,marechal hermes,Marechal Hermes,21,,-22.864928,-43.370145
br,marechal jardim,Marechal Jardim,21,,-22.483333,-44.516667
br,marechal luz,Marechal Luz,26,,-26.166667,-48.533333
br,marechal mallet,Marechal Mallet,18,,-25.916667,-50.833333
br,marechal taumaturgo,Marechal Taumaturgo,01,,-8.948333,-72.786944
br,marema,Marema,26,,-26.833333,-52.716667
br,mare,Maré,21,,-22.856039,-43.243481
br,marequeta,Marequetá,06,,-5.033333,-39.483333
br,maresias,Maresias,27,,-23.8,-45.55
br,mares,Marés,06,,-4.433333,-39.066667
br,marezinha,Marèzinha,06,,-5.766667,-39.183333
br,marfim,Marfim,06,,-4.683333,-40.433333
br,marfim,Marfim,13,,-4,-44.783333
br,marfim,Marfim,20,,-3.619722,-42.150556
br,marfim,Marfim,27,,-20.633333,-47.466667
br,marfinzeiro,Marfinzeiro,13,,-3.800833,-43.429444
br,margarida,Margarida,11,,-21.683333,-56.733333
br,margarida,Margarida,18,,-24.65,-54.116667
br,margarida,Margarida,21,,-21.45,-41.45
br,margaride,Margaride,22,,-6.283333,-36.833333
br,mar grande,Mar Grande,05,,-12.95,-38.616667
br,maria boneca,Maria Boneca,05,,-16.116667,-39.4
br,maria carlota,Maria Carlota,15,,-21.133333,-45.166667
br,maria cruger,Maria Cruger,08,,-20.2,-40.533333
br,maria cruz,Maria Cruz,05,,-16.483333,-39.966667
br,maria da fe,Maria da Fé,15,,-22.3,-45.383333
br,maria da graca,Maria da Graça,21,,-22.883498,-43.265481
br,maria farinha,Maria Farinha,30,,-7.85,-34.833333
br,maria gorda,Maria Gorda,18,,-24.816667,-49.016667
br,maria helena,Maria Helena,18,,-23.583333,-53.166667
br,maria isabel,Maria Isabel,05,,-10.033333,-41.366667
br,maria joana,Maria Joana,14,,-14.366667,-56.983333
br,marialva,Marialva,18,26001,-23.466667,-51.7
br,maria,Maria,23,,-28.533333,-52.15
br,maria melo,Maria Melo,17,,-7.416667,-35.416667
br,mariana,Mariana,05,,-9.5,-38.95
br,mariana,Mariana,08,,-20.35,-41.55
br,mariana,Mariana,15,45237,-20.383333,-43.416667
br,mariana,Mariana,18,,-25.356153,-48.425234
br,mariana,Mariana,30,,-8.983333,-37.616667
br,mariana pimentel,Mariana Pimentel,23,,-30.333333,-51.583333
br,mariana velha,Mariana Velha,16,,-1.7,-46.25
br,marianna,Marianna,15,,-20.383333,-43.416667
br,marianna,Marianna,30,,-8.983333,-37.616667
br,mariano flores,Mariano Flores,15,,-14.8,-44.133333
br,mariano guedes,Mariano Guedes,15,,-16.85,-41.266667
br,mariano,Mariano,05,,-12.333333,-39.233333
br,mariano,Mariano,08,,-20.35,-41.55
br,marianopolis do tocantins,Marianópolis do Tocantins,31,,-9.742778,-49.566944
br,marianopolis,Marianópolis,13,,-4.783333,-44.633333
br,marianopolis,Marianópolis,15,,-16.416667,-42.533333
br,mariano ramos,Mariano Ramos,14,,-15.95,-57.083333
br,mariante,Mariante,23,,-29.7,-51.966667
br,maria paes,Maria Paes,17,,-6.733333,-37.316667
br,maria pereira,Maria Pereira,20,,-5.75,-39.633333
br,mariapolis,Mariápolis,27,,-21.8,-51.183333
br,mariapopolis,Mariapópolis,15,,-16.416667,-42.533333
br,maria preta,Maria Preta,26,,-26.4,-53.583333
br,maria quiteria,Maria Quitéria,05,,-12.133333,-39.033333
br,maria rosa,Maria Rosa,05,,-16.216667,-39.433333
br,maria soares,Maria Soares,17,,-7.533333,-38.45
br,marias pretas,Marias Pretas,02,,-9.5,-36.816667
br,marias pretas,Marias Pretas,17,,-6.933333,-38.566667
br,marias pretas,Marias Pretas,30,,-7.85,-37.783333
br,maria viera,Maria Viera,29,,-16.35,-51.416667
br,maribondo,Maribondo,02,10475,-9.583333,-36.233333
br,maribondo,Maribondo,05,,-13.583333,-40.216667
br,maribondo,Maribondo,20,,-3.950278,-42.277222
br,maribondo,Maribondo,31,,-7.383333,-48.066667
br,maribu,Maribu,05,,-13.95,-39.75
br,marica,Maricá,21,79552,-22.916667,-42.816667
br,marica,Maricá,23,,-30.5,-53.833333
br,maricoabo,Maricoabo,05,,-13.416667,-39.066667
br,maricota,Maricota,30,,-7.9,-34.883333
br,mari da sombra,Mari da Sombra,22,,-5.645278,-36.006389
br,mariental,Mariental,18,,-25.733333,-49.65
br,marilac,Marilac,15,,-18.466667,-42.183333
br,marila,Marilá,18,,-23.483333,-52.116667
br,marilandia do sul,Marilândia do Sul,18,,-23.75,-51.316667
br,marilandia,Marilândia,08,3900,-19.416667,-40.516667
br,marilandia,Marilândia,15,,-18.616667,-49.116667
br,marilandia,Marilândia,18,,-23.75,-51.316667
br,marilandia,Marilândia,29,,-15.683333,-49.183333
br,marilena,Marilena,18,,-22.716667,-53.05
br,marilia,Marília,27,212222,-22.216667,-49.933333
br,mariluz,Mariluz,18,,-24.033333,-53.216667
br,mari,Mari,04,,-7.083333,-64.566667
br,mari,Mari,05,,-9.216667,-38.316667
br,mari,Mari,06,,-5.833333,-39.833333
br,mari,Mari,17,17535,-7.083333,-35.333333
br,mari,Mari,22,,-5.131389,-37.26
br,mari,Mari,30,,-9.266667,-38.25
br,marimba,Marimba,20,,-8.85,-44.383333
br,marimbas,Marimbas,17,,-7.016667,-38.666667
br,marimbas,Marimbas,20,,-8.85,-44.383333
br,marimbondo,Marimbondo,02,,-9.583333,-36.233333
br,marimbondo,Marimbondo,18,,-23.55,-49.7
br,marimbondo,Marimbondo,21,,-21.416667,-41.616667
br,marimbondo,Marimbondo,27,,-22.383333,-47.216667
br,marimbondo,Marimbondo,28,,-10.683333,-36.916667
br,marineiro,Marineiro,27,,-20.316667,-49.9
br,marineli,Marinéli,21,,-22.116667,-42.616667
br,maringa,Maringá,18,311727,-23.416667,-51.916667
br,marinheiro,Marinheiro,17,,-7.583333,-36.533333
br,marinheiro,Marinheiro,20,,-6.9,-41.366667
br,marinheiros,Marinheiros,06,,-3.183333,-39.466667
br,marinho,Marinho,17,,-7.583333,-36.166667
br,marinopolis,Marinopolis,13,,-4.783333,-44.633333
br,marinopolis,Marinópolis,27,,-20.45,-50.833333
br,marins,Marins,15,,-22.45