#charset "us-ascii"
#include <adv3.h>
#include <en_us.h>

/* Remote Viewing Adjustments */

/*
This file contains a modify statement to make Thing behave
better under remote viewing conditions. These changes should be
integrated into the library itself in version 3.0.9, so use this
file only to make CSC work with 3.0.8.  Mike Roberts has pointed
out some obscure cases where this implementation could cause bugs,
but these are unlikely to occur and will be resolved in the final
library implementation.

The meat of these changes is that the "remote" version of description
methods will be chosen based on the location of the *actor* instead of
the location of the pov.  This more correctly aligns with the player's
assumption that objects are being described according to the actor's 
perceptions, rather than whatever device is serving as the point-of-view.

(The pov may be a video camera or walkie-talkie or something of that
sort which allows an actor to perceive things from a distant location.
In such a situation, the actor still perceives these objects as "remote"
to his/her location even though they may be "local" to the pov object.)
*/

/*
The following methods are now being passed the actor instead of
the pov.  They are usually the same object, but not always.
In particular, the BasicWindow and Window classes use the window
itself as the pov during a LOOK THROUGH command.
TADS 3 doesn't care what a method's parameters are labeled, so you
probably won't need to change any existing code.  Just be aware
of this change when working with CSC, which will probably have you
using these methods a lot more.

remoteSpecialDesc
remoteInitSpecialDesc
showRemoteSpecialDesc
showSpecialDescInContents
showObscuredSpecialDescInContents
showDistantSpecialDescInContents
showRemoteSpecialDescInContents
*/

modify Thing
    adjustLookAroundTable(tab, pov, actor)
    {
        /* Remove the pov object */
        tab.removeElement(pov);
        /*
         *   Don't remove the actor, because it's perfectly logical for
         *   the actor to be in the pov's field of view when they're not
         *   the same object.
         */
        //tab.removeElement(actor);

        /*
         *   To compensate for not removing the actor by default,
         *   let the actor and pov have some control over the field
         *   of view as well.
         */
        if(self != pov && self != actor)
        {
            pov.adjustLookAroundTable(tab, pov, actor);
            actor.adjustLookAroundTable(tab, pov, actor);
        }
    }

    showSpecialDescWithInfo(info, pov)
    {
        /*
         *   Determine what to show, based on the point-of-view location
         *   and the sense path.  If the point of view isn't in the same
         *   top-level location as 'self', use the remote special
         *   description; otherwise, select the obscured, distant, or
         *   basic special description according to the transparency of
         *   the sense path.  
         */
        if (getOutermostRoom() != gActor.getOutermostRoom() || gActor != pov)
        {
            /* different top-level rooms - use the remote description */
            showRemoteSpecialDesc(gActor);
        }
        else if (info.trans == obscured)
        {
            /* we're obscured, so show our obscured special description */
            showObscuredSpecialDesc();
        }
        else if (info.trans == distant)
        {
            /* we're at a distance, so use our distant special description */
            showDistantSpecialDesc();
        }
        else if (canDetailsBeSensed(sight, info, pov))
        {
            /* 
             *   we're not obscured or distant, and our details can be
             *   sensed, so show our fully-visible special description 
             */
            showSpecialDesc();
        }
    }
    
    showSpecialDescInContentsWithInfo(info, pov, cont)
    {
        /* determine what to show, based on the location and sense path */
        if (getOutermostRoom() != gActor.getOutermostRoom() || gActor != pov)
            showRemoteSpecialDescInContents(gActor, cont);
        else if (info.trans == obscured)
            showObscuredSpecialDescInContents(gActor, cont);
        else if (info.trans == distant)
            showDistantSpecialDescInContents(gActor, cont);
        else if (canDetailsBeSensed(sight, info, pov))
            showSpecialDescInContents(gActor, cont);
    }

    lookAroundWithinDesc(actor, illum)
    {
        local pov = getPOV();
        if(pov == nil) pov = actor;

        /* 
         *   check for illumination - we must have at least dim ambient
         *   lighting (level 2) to see the room's long description 
         */
        if (illum > 1)
        {
            /* 
             *   Display the normal description of the room - use the
             *   roomRemoteDesc if the actor isn't in the same room; use
             *   the firstDesc if this is the first time in the room; and
             *   otherwise the basic roomDesc.  
             */
            if (!actor.isIn(self) || actor != pov)
            {
                /* we're viewing the room remotely */
                roomRemoteDesc(actor);
            }
            else if (actor.hasSeen(self))
            {
                /* we've seen it already - show the basic room description */
                roomDesc;
            }
            else
            {
                /* 
                 *   we've never seen this location before - show the
                 *   first-time description 
                 */
                roomFirstDesc;
            }
        }
        else
        {
            /* display the in-the-dark description of the room */
            roomDarkDesc;
        }
    }

    lookAroundWithinContents(actor, illum, infoTab)
    {
        local lst;
        local lister;
        local remoteLst;
        local outer;
        local pov;

        /* get our outermost enclosing room */
        outer = getOutermostRoom();
        
        /* get the pov, use the actor if there's no other */
        pov = getPOV();
        if(pov == nil) pov = actor;

        /* mark everything visible from the room as having been seen */
        setAllSeenBy(infoTab, actor);
        
        /* 
         *   if the illumination is less than 'dim' (level 2), display
         *   only self-illuminating items 
         */
        if (illum != nil && illum < 2)
        {
            /* 
             *   We're in the dark - list only those objects that the actor
             *   can sense via the sight-like senses, which will be the
             *   list of self-illuminating objects.  Don't include items
             *   being carried by the actor, though, since those don't
             *   count as part of the actor's surroundings.  (To produce
             *   this list, simply make a list consisting of all of the
             *   objects in the sense info table that aren't contained in
             *   the actor; since the table naturally includes only objects
             *   that are illuminated, the entire contents of the table
             *   will give us the visible objects.)  
             */
            lst = senseInfoTableSubset(
                infoTab, {obj, info: !obj.isIn(actor)});
            
            /* use my dark contents lister */
            lister = darkRoomContentsLister;

            /* there are no remote-room lists to generate */
            remoteLst = nil;
        }
        else
        {
            /* start with my contents list */
            lst = contents;

            /* always remove the actor from the list */
            lst -= actor;

            /* use the normal (lighted) lister */
            /*
             *   Even though we're the local room to the pov, we might
             *   be remote to the actor, in which case we should use
             *   the remote lister for clarity.
             */
            lister = (actor.isIn(self) && actor == pov) ?
                roomContentsLister : remoteRoomContentsLister(self);

            /*
             *   Generate a list of all of the *other* top-level rooms
             *   with contents visible from here.  To do this, build a
             *   list of all of the unique top-level rooms, other than our
             *   own top-level room, that contain items in the visible
             *   information table.  
             */
            remoteLst = new Vector(5);
            infoTab.forEachAssoc(new function(obj, info)
            {
                /* if this object isn't in our top-level room, note it */
                if (obj != outer && !obj.isIn(outer))
                {
                    local objOuter;
                    
                    /* 
                     *   it's not in our top-level room, so find its
                     *   top-level room 
                     */
                    objOuter = obj.getOutermostRoom();

                    /* if this one isn't in our list yet, add it */
                    if (remoteLst.indexOf(objOuter) == nil)
                        remoteLst.append(objOuter);
                }
            });
        }

        /* add a paragraph before the room's contents */
        "<.p>";

        /* show the contents */
        lister.showList(actor, self, lst, ListRecurse, 0, infoTab, nil);

        /* 
         *   if we can see anything in remote top-level rooms, show the
         *   contents of each other room with visible contents 
         */
        if (remoteLst != nil)
        {
            /* show each remote room's contents */
            for (local i = 1, local len = remoteLst.length() ; i <= len ; ++i)
            {
                local cur;
                local cont;

                /* get this item */
                cur = remoteLst[i];

                /* start with the direct contents of this remote room */
                cont = cur.contents;

                /* 
                 *   Remove any objects that are also contents of the
                 *   local room or of any other room we've already listed.
                 *   It's possible that we have a MultiLoc that's in one
                 *   or more of the visible top-level locations, so we
                 *   need to make sure we only list each one once. 
                 */
                cont = cont.subset({x: !x.isIn(outer)});
                for (local j = 1 ; j < i ; ++j)
                    cont = cont.subset({x: !x.isIn(remoteLst[j])});

                /*
                 *   list this remote room's contents using our remote
                 *   lister for this remote room 
                 */
                outer.remoteRoomContentsLister(cur).showList(
                    actor, cur, cont, ListRecurse, 0, infoTab, nil);
            }
        }
    }
;

/*
 *   The following modifications to the style tags are necessary when
 *   using lookAroundWithin to generate descriptions of rooms for other
 *   purposes than LOOK AROUNDs.  In particular, the old definitions
 *   caused spurious line breaks and blank lines when lookAroundWithin
 *   was called without the LookRoomName flag.
 */
replace roomnameStyleTag: StyleTag 'roomname' '\n<b>' '</b><br>\n';
replace roomdescStyleTag: StyleTag 'roomdesc' '' '';

