########################################################################
# housekeeping
########################################################################

use v6.d;

unit module ProcStats:ver<0.1.0>:auth<CPAN:lembark>;

########################################################################
# subroutines
########################################################################

sub dump-rusage
(
    Bool()      :$final = False,
    Bool()      :$first = $final,
    Bool()      :$force = $final,
    Stringy()   :$label = $final ?? 'Final' !! '',
)
is export( :DEFAULT )
{
    my $wtime       = now.Num;

    constant FIELDS =
    <
        maxrss  ixrss   idrss   isrss    
        minflt  majflt  nswap   
        inblock oublock msgsnd
        msgrcv  nsignals nvcsw   nivcsw
    >;  
    constant IGNORE =
    <
        ixrss idrss isrss
        nswap
        msgsnd msgrcv nsignals
    >;  
    constant COMPARE =
    <
        maxrss
        majflt  minflt
        inblock oublock
    >;
    state $passes   = 0;

    use nqp;
    nqp::getrusage( my int @raw );

    my ( $user_s, $user_us, $syst_s, $syst_us )
    = splice @raw, 0, 4;

    my %sample = FIELDS Z=> @raw;
    %sample{ IGNORE } :delete;

    my $utime   = ( $user_s + $user_us / 1_000_000  );
    my $stime   = ( $syst_s + $syst_us / 1_000_000  );

    # %prior is full sample on the first pass
    # regardless of $first setting.

    state %last  =
    state %first =
    (
        |%sample,
        :$wtime,
        :$utime,
        :$stime,
    );

    my %prior
    = $first
    ?? %first 
    !! %last
    ;

    my %curr
    = ( $force || ! $passes )
    ?? %sample
    !! do
    {
        my @diffs
        = COMPARE.grep( { %sample{$_} != %prior{$_} } );

        @diffs Z=> %sample{ @diffs }
    }
    ;

    sub write-stat ( Pair $p )  
    {
        note
        sprintf
            '%-*s : %s',
            once {FIELDS».chars.max},
            $p.key,
            $p.value
        ;
        return
    }
    
    sub write-diff ( Pair $p )
    {
        my $k   = $p.key;
        my $v   = $p.value - %first{ $k };

        write-stat $k => $v;
    }

    state &write    = &write-stat;

    for %curr.sort -> $stat
    {
        FIRST
        {
            note '---';
            write-stat ( output => $++  );
            write-stat ( :$passes       );
            write-stat ( :$label        ) if $label;

            # compose pairs before passing the values
            # as parameters. write( ... ) without the 
            # space treats them as named parameters.

            write (:$wtime) ;
            write (:$utime) ;
            write (:$stime) ;
        }

        # stat is already a pair, no parens needed.

        write $stat ;
    }

    once { &write = &write-diff };

    ++$passes;

    %last = %sample;

    return
}

=finish

=begin pod

=head1 NAME

ProcStats - dump rusage process statistics with optional label.

=head1 SYNOPSIS

    # print rusage output to stderr.
    # "sample" is incremented with each output.
    #
    # first pass ("sample 0") dumps all stats.
    # successive ones only list changes.
    #
    # system time (stime) invariably increases due to 
    # rusage call and is only output if another field
    # causes output.
    #
    # format is YAML hash with one document per sample.


    dump-rusage( label => "$*PROGRAM" );

    ---
    sample   : 0
    label    : rand-dictonary
    stime    : 724655
    idrss    : 129224
    inblock  : 0
    isrss    : 0
    ixrss    : 39741
    majflt   : 0
    maxrss   : 0
    minflt   : 0
    msgrcv   : 48
    msgsnd   : 0
    nivcsw   : 0
    nsignals : 0
    nswap    : 32466
    nvcsw    : 0
    oublock  : 0
    utime    : 0

    # force output even if there are no changes.

    dump-rusage( label => 'Initial users', :force );
    ---
    sample   : 1
    label    : Initial users
    stime    : 748975
    idrss    : 129620
    nswap    : 32639

    dump-rusage;

    ---
    sample   : 2
    stime    : 769168
    idrss    : 129864
    nswap    : 32732

    dump-rusage( :force, :start );

    ---
    sample   : 92
    label    : First
    stime    : 126712
    idrss    : 136936
    inblock  : 0
    isrss    : 0
    ixrss    : 70682
    majflt   : 0
    maxrss   : 0
    minflt   : 0
    msgrcv   : 48
    msgsnd   : 0
    nivcsw   : 0
    nsignals : 0
    nswap    : 34532
    nvcsw    : 0
    oublock  : 0
    utime    : 1



=head2 Notes

    man 2 rusage;


=head1 SEE ALSO

=item getrusage(3) getrusage(3p)

getrusage is POSIX.

    struct rusage 
    {
       struct timeval ru_utime; /* user CPU time used */
       struct timeval ru_stime; /* system CPU time used */
       long   ru_maxrss;        /* maximum resident set size */
       long   ru_ixrss;         /* integral shared memory size */
       long   ru_idrss;         /* integral unshared data size */
       long   ru_isrss;         /* integral unshared stack size */
       long   ru_minflt;        /* page reclaims (soft page faults) */
       long   ru_majflt;        /* page faults (hard page faults) */
       long   ru_nswap;         /* swaps */
       long   ru_inblock;       /* block input operations */
       long   ru_oublock;       /* block output operations */
       long   ru_msgsnd;        /* IPC messages sent */
       long   ru_msgrcv;        /* IPC messages received */
       long   ru_nsignals;      /* signals received */
       long   ru_nvcsw;         /* voluntary context switches */
       long   ru_nivcsw;        /* involuntary context switches */
    };

