#! /usr/bin/env false

use v6.d;

use JSON::Fast;
use Log;
use Log::Level;

#| A simple implementation of Log.
unit class Log::JSON is Log;

#| An array of hashes. Each has must have a handle key, with an IO::Handle to
#| print messages to. A level key with an integer to signify a Log::Level is
#| also required.
has @!outputs;

#| A supplier to emit messages to.
has Supplier $!messages;

#| Set to true to pretty-print the JSON.
has Bool:D $.pretty = False;

#| Send an emergency message.
multi method emergency (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('emergency', $message, |%meta),
		level => 0,
	})
}

#| Send a formatted emergency message.
multi method emergency (
	#| A printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('emergency', $format.sprintf(|@args), |%meta),
		level => 0,
	})
}

#| Send an alert message.
multi method alert (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('alert', $message, |%meta),
		level => 1,
	})
}

#| Send a formatted alert message.
multi method alert (
	#| A printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('alert', $format.sprintf(|@args), |%meta),
		level => 1,
	})
}

#| Send a critical message.
multi method critical (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('critical', $message, |%meta),
		level => 2,
	})
}

#| Send a formatted critical message.
multi method critical (
	#| A printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('critical', $format.sprintf(|@args), |%meta),
		level => 2,
	})
}

#| Send an error message.
multi method error (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('error', $message, |%meta),
		level => 3,
	})
}

#| Send a formatted error message.
multi method error (
	#| A printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('error', $format.sprintf(|@args), |%meta),
		level => 3,
	})
}

#| Send a warning message.
multi method warning (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('warning', $message, |%meta),
		level => 4,
	})
}

#| Send a formatted warning message.
multi method warning (
	#| A printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('warning', $format.sprintf(|@args), |%meta),
		level => 4,
	})
}

#| Send a notice message. This is the "normal" level to publish log entries on.
multi method notice (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('notice', $message, |%meta),
		level => 5,
	})
}

#| Send a formatted notice message. This is the "normal" level to publish log
#| entries on.
multi method notice (
	#| A printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('notice', $format.sprintf(|@args), |%meta),
		level => 5,
	})
}

#| Send an informational message.
multi method info (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('info', $message, |%meta),
		level => 6,
	})
}

#| Send a formatted informational message.
multi method info (
	#| The printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('info', $format.sprintf(|@args), |%meta),
		level => 6,
	})
}

#| Send a debug message.
multi method debug (
	#| The message to print.
	Str:D $message,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('debug', $message, |%meta),
		level => 7,
	})
}

#| Send a formatted debug message.
multi method debug (
	#| A printf-style format string.
	Str:D $format,

	#| Arguments to substitute into the format string placeholders.
	*@args,

	#| Any additional fields of metadata to include.
	*%meta,
) {
	$!messages.emit({
		message => self!message('debug', $format.sprintf(|@args), |%meta),
		level => 7,
	})
}

#| Add an output to the logger.
multi method add-output (
	#| The IO::Handle to write log messages to.
	IO::Handle:D $handle,

	#| The log level for this particular stream.
	Int() $level where Log::Level::Emergency ≤ * ≤ Log::Level::Debug,

	#| A filter function to apply to any messages sent to the output.
	Callable :$filter?,
) {
	@!outputs.push({
		:$handle,
		:$level,
		:$filter,
	});
}

#| Convenience method to get the callee name. This is the package name if one
#| exists, or the source file name.
method !callee (
	#| The depth of which to get the callframe.
	Int:D $depth = 3,

	--> Str
) {
	my $file = callframe($depth).annotations<file>;

	return $file unless $file.ends-with(')');

	$file.words.tail.substr(1, *-1);
}

#| Convenience method for formatting the message as desired.
method !message (
	#| A human readable identifier for the level.
	Str:D $level,

	#| The original message to be logged.
	Str:D $message,

	#| Any additional fields for metadata on the log message.
	*%meta,

	--> Str
) {
	to-json({
		time => DateTime.now.utc,
		hostname => $*KERNEL.hostname,
		origin => self!callee,
		:$level,
		:$message,
		|%meta,
	}, :$!pretty);
}

submethod TWEAK
{
	# Set up the Supply for messages.
	$!messages .= new;
	$!messages.Supply.tap(sub (%payload) {
		@!outputs
			.race
			.grep(%payload<level> ≤ *<level>)
			.grep({ $_<filter> ?? $_<filter>(%payload).so !! True })
			.map(*<handle>.say(%payload<message>))
			;
	});
}

=begin pod

=NAME    Log::JSON
=VERSION 0.1.0
=AUTHOR  Patrick Spek <p.spek@tyil.nl>

=begin LICENSE
Copyright © 2020

This program is free software: you can redistribute it and/or modify it under
the terms of the GNU Lesser General Public License as published by the Free
Software Foundation, version 3.

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE.  See the GNU Lesser General Public License for more
details.

You should have received a copy of the GNU Lesser General Public License along
with this program.  If not, see http://www.gnu.org/licenses/.
=end LICENSE

=end pod

# vim: ft=raku noet sw=8 ts=8
