# Edit system config files -- services, inetd.conf.
# Might also benefit from data hiding with itcl.

set sysconfig_begin   "#begin-kerbnet-additions"
set sysconfig_end     "#end-kerbnet-additions"
set sysconfig_disable "#disabled-for-kerbnet#"

proc sysconfig_read { file } {
    # Read file and return list of lines.

    # Let errors propagate.
    set f [open $file r]
    set contents [read -nonewline $f]
    close $f

    split $contents "\n"
}

proc sysconfig_remove_kerbnet { contents } {
    global sysconfig_begin sysconfig_end sysconfig_disable

    set result {}
    set omit 0
    foreach x $contents {
	if ![string compare $sysconfig_begin $x] { set omit 1 }
	if !$omit {
	    regsub "^$sysconfig_disable" $x "" x
	    lappend result $x
	}
	if ![string compare $sysconfig_end $x] { set omit 0 }
    }
    if $omit {
	error "end-kerbnet-additions marker not found"
    }
    return $result
}

proc foreach_counted { counter varname list body } {
    upvar 1 $varname v
    upvar 1 $counter idx
    set limit [llength $list]
    for {set idx 0} {$idx < $limit} {incr idx} {
	set v [lindex $list $idx]
	uplevel 1 $body
    }
}

# Specific to /etc/services....

# file contents - lines
set etc_services {}
# new entries to be added in kerbnet section -
#  list of { { name ?name...? } port/proto comment }
set new_services {}

# I don't want to change any lines I don't have to modify.  So I
# don't convert back and forth.  (Converting but keeping a mapping
# from the parsed form back to the original might help.  Try it,
# later...)

# Cache speeds it up a little.
proc parse_service_line { line } {
    global service_parse_cache
    if ![info exists service_parse_cache($line)] {
	set split1 [split $line "#"]
	set data [lindex $split1 0]
	regsub -all "\[ \t\]+" [string trim $data] " " data
	set comment [join [lrange $split1 1 end] "#"]

	set fields [split $data " "]
	if [llength $fields]==0 { return [list {} {} $comment] }
	set port [lindex $fields 1]
	set names [lreplace $fields 1 1]

	set service_parse_cache($line) [list $names $port $comment]
    }
    return $service_parse_cache($line)
}

proc unparse_service_line { data } {
    set names [lindex $data 0]
    if [llength $names] {
	set name [lindex $names 0]
	if [string length $name]>7 {
	    set sep "\t"
	} else {
	    set sep "\t\t"
	}
	if [string length [lindex $data 1]]>7 {
	    set sep2 "\t"
	} else {
	    set sep2 "\t\t"
	}
	set othernames [lrange $names 1 end]
	if [string length $othernames]>7 {
	    set sep3 " "
	} else {
	    set sep3 "\t"
	}
	set cmt [lindex $data 2]
	if [string length $cmt] { set cmt "$sep3#$cmt" }
	return "$name$sep[lindex $data 1]$sep2$othernames$cmt"
    } else {
	# comment only
	return "#[lindex $data 2]"
    }
}

proc add_service { names port comment } {
    global etc_services new_services
    set proto [lindex [split $port /] 1]
    if [llength $names] {
	# If matching entries exists (ignoring comments and additional names)
	# don't add them again.  Saves time in the remove_service passes.
	foreach x $etc_services {
	    set parsed [parse_service_line $x]
	    set port2 [lindex $parsed 1]
	    if [string compare $port $port2] continue
	    foreach name [lindex $parsed 0] {
		set idx [lsearch -exact $names $name]
		if $idx>=0 { set names [lreplace $names $idx $idx] }
	    }
	}
	if ![llength $names] { return }
	foreach name $names { remove_service $name $proto }
    }
    lappend new_services [list $names $port $comment]
}

proc remove_service { name proto } {
    global etc_services
    global new_services
    global sysconfig_disable

    for {set i 0} {$i < [llength $etc_services]} {incr i} {
	# This repeated parsing is wasteful, but I don't want to
	# change (by reassembly) any lines I'm not modifying.
	set line [parse_service_line [lindex $etc_services $i]]
	if [string compare $proto [lindex [split [lindex $line 1] /] 1]] {
	    continue
	}
	set idx [lsearch -exact [lindex $line 0] $name]
	if $idx>=0 {
	    set etc_services \
		    [lreplace $etc_services $i $i "$sysconfig_disable[lindex $etc_services $i]"]
	    set names [lindex $line 0]
	    if [llength $names]>1 {
		set names [lreplace $names $idx $idx]
		add_service $names [lindex $line 1] [lindex $line 2]
	    }
	}
    }
    for {set i 0} {$i < [llength $new_services]} {incr i} {
	set line [lindex $new_services $i]
	if [string compare $proto [lindex [split [lindex $line 1] /] 1]] {
	    continue
	}
	set idx [lsearch -exact [lindex $line 0] $name]
	if $idx>=0 {
	    set new_services [lreplace $new_services $i $i]
	    incr i -1
	    set names [lindex $line 0]
	    if [llength $names]>1 {
		set names [lreplace $names $idx $idx]
		add_service $names [lindex $line 1] [lindex $line 2]
	    }
	}
    }
}

proc get_service { name } {
    set result {}

    global etc_services new_services
    foreach x $etc_services {
	set y [parse_service_line $x]
	if [lsearch -exact [lindex $y 0] $name]>=0 {
	    lappend result [lindex $y 1]
	}
    }
    foreach y $new_services {
	if [lsearch -exact [lindex $y 0] $name]>=0 {
	    lappend result [lindex $y 1]
	}
    }
    return $result
}
proc have_service { name } { return [llength [get_service $name]]>0 }


set etc_services [sysconfig_read /etc/services]
set etc_services [sysconfig_remove_kerbnet $etc_services]

set krb5_services [list \
	{klogin 543/tcp "Kerberos authenticated rlogin"}		\
	{{kshell kcmd} 544/tcp "Kerberos remote shell"} 		\
	{kerberos-adm 749/tcp "Kerberos 5 admin/changepw"}		\
	{kerberos-adm 749/udp "Kerberos 5 admin/changepw"}		\
	{krb5_prop 754/tcp "Kerberos slave propagation"}		\
	{eklogin 2105/tcp "Kerberos auth/encrypted rlogin"}		\
	{krb524 4444/tcp "Kerberos 5 to 4 ticket conv"}]
proc set_up_extra_services_list {} {
    # xxx special, for krb4 compat
    global krb5_services
    if [have_service kerberos] {
	set kerberos_name kerberos-sec
    } else {
	set kerberos_name {kerberos kdc}
    }
    lappend krb5_services [list $kerberos_name 88/udp "Kerberos V5 KDC"]
    lappend krb5_services [list $kerberos_name 88/tcp "Kerberos V5 KDC"]
}

proc show_needed_services_entries {} {
    global krb5_services

    puts "\nThese are the entries you need to ensure are present in"
    puts "your /etc/services file:\n"
    foreach x $krb5_services {
	puts "[unparse_service_line $x]"
    }
    puts "\n"
    pause
}

proc update_services { } {
    global krb5_services
    global insttype
    set_up_extra_services_list

    global pkg
    puts "$pkg requires various entries to be present in /etc/services."
    puts -nonewline "They can be added now, or you can add them later."
    if [info exists insttype] { set x $insttype } else { set x foo }
    if [string compare $x appclient] {
	puts "  (But note that"
	puts "installing an application server or KDC may require that these"
	puts -nonewline "entries exist during the installation process.)"
    }
    puts ""
    if [ask-yesno "Update /etc/services now?" y] {
	global etc_services new_services
	global sysconfig_begin sysconfig_end

	puts -nonewline "Working..." ; flush stdout

	foreach x $krb5_services {
	    eval add_service $x
	}

	set newfile /etc/services.new
	if [catch {
	    set f [open $newfile w]
	    foreach x $etc_services {
		puts $f $x
	    }
	    if [llength $new_services] {
		puts $f $sysconfig_begin
		foreach x $new_services {
		    puts $f [unparse_service_line $x]
		}
		puts $f $sysconfig_end
	    }
	    close $f
	    file delete -force /etc/services.old
	    exec ln /etc/services /etc/services.old
	    file rename -force $newfile /etc/services
	    puts "done.  (Old version in /etc/services.old.)"
	} err] {
	    puts "error updating /etc/services:\n$err\n"
	    puts "Abort this script, fix the error and start again, or"
	    puts "install the changes yourself."
	    show_needed_services_entries
	}

	global service_parse_cache
	unset service_parse_cache
    } else {
	show_needed_services_entries
    }
}

# Specific to /etc/inetd.conf....

# list of lines
set etc_inetd {}
# list of { service stream/dgram tcp/udp nowait/wait user pathname args }
#      or { comment-including-# }
set new_inetd {}

proc parse_inetd_line { line } {
    if [regexp "^#" $line] {
	return [list $line]
    }
    regsub -all "\[ \t\]+" [string trim $line] " " line
    set fields [split $line]
    if [llength $fields]<7 {
	error "too few fields in /etc/inetd line ($line)?"
    }
    lreplace $fields 6 end [lrange $fields 6 end]
}
proc unparse_inetd_line { data } {
    if [llength $data]>1 {
	return "[lindex $data 0]\t[lindex $data 1] [lindex $data 2] [lindex $data 3] [lindex $data 4]\t[lindex $data 5]\t[lindex $data 6]";
    } else {
	return [lindex $data 0]
    }
}

proc add_krb_daemon { service cmd } {
    set args $cmd

    global krb5_path generic_prefix
    global etc_inetd new_inetd
    global sysconfig_disable

    foreach_counted i x $etc_inetd {
	if [regexp "^$service\[ \t\]" $x] {
	    set etc_inetd [lreplace $etc_inetd $i $i "$sysconfig_disable[lindex $etc_inetd $i]"]
	    break
	}
    }
    foreach_counted i x $new_inetd {
	if ![string compare [lindex $x 0] $service] {
	    error "already adding inetd service $service"
	}
    }
    if [info exists generic_prefix] {
	set progname "$generic_prefix/sbin/[lindex $args 0]"
    } else {
	set progname "$krb5_path(prefix)/sbin/[lindex $args 0]"
    }
    lappend new_inetd [list $service stream tcp nowait root $progname $args]
}

set krb5_daemons {}

if 1 {
    # insecure version
    set krb5_daemon_invocation(ftp)	{ftpd}
    set krb5_daemon_invocation(klogin)	{klogind -k -i}
    set krb5_daemon_invocation(eklogin)	{klogind -k -i -e}
    set krb5_daemon_invocation(kshell)	{kshd -k -i -A}
    set krb5_daemon_invocation(telnet)	{telnetd -a none}
    set krb5_daemon_invocation(krb5_prop) {kpropd}
} else {
    # secure version
    set krb5_daemon_invocation(ftp)	{ftpd -a}
    set krb5_daemon_invocation(klogin)	{klogind -5 -c}
    set krb5_daemon_invocation(eklogin)	{klogind -5 -c -e}
    set krb5_daemon_invocation(kshell)	{kshd -5 -c -A}
    set krb5_daemon_invocation(telnet)	{telnetd -a valid}
    set krb5_daemon_invocation(krb5_prop) {kpropd}
}

proc show_new_inetd { } {
    global new_inetd
    puts "\nThese are the entries you need to put in /etc/inetd.conf:\n"
    foreach x $new_inetd { puts [unparse_inetd_line $x] }
    puts ""
    puts "Then send a hangup (HUP) signal to inetd, to get it to reread the"
    puts "config file.\n"
    pause
}
proc update_inetd { } {
    global sysconfig_begin sysconfig_end
    global etc_inetd new_inetd pkg
    global krb5_daemons krb5_daemon_invocation

    if ![llength $krb5_daemons] return

    set etc_inetd [sysconfig_read /etc/inetd.conf]
    set etc_inetd [sysconfig_remove_kerbnet $etc_inetd]

    foreach d $krb5_daemons {
	add_krb_daemon $d $krb5_daemon_invocation($d)
    }

    if [ask-yesno "Update /etc/inetd.conf with new services now?" y] {
	set newfile /etc/inetd.conf.new
	if [catch {
	    set f [open $newfile w]
	    foreach x $etc_inetd {
		puts $f $x
	    }
	    if [llength $new_inetd] {
		puts $f $sysconfig_begin
		foreach x $new_inetd {
		    puts $f [unparse_inetd_line $x]
		}
		puts $f $sysconfig_end
	    }
	    close $f
	    file delete -force /etc/inetd.conf.old
	    exec ln /etc/inetd.conf /etc/inetd.conf.old
	    file rename -force $newfile /etc/inetd.conf
	    puts "\nDone.  (Old version in /etc/inetd.conf.old.)\n"
	    puts "You'll need to send a hangup (HUP) signal to the inetd"
	    puts "process when you've finished configuring $pkg, to get it to"
	    puts "reread inetd.conf.\n"
	    pause
	} err] {
	    puts "Error updating /etc/inetd.conf:\n$err\n"
	    puts "Abort this script, fix the error and start again, or"
	    puts "install the changes yourself."
	    show_new_inetd
	}
    } else {
	show_new_inetd
    }
}
update_inetd
