/*
 * kdc/kdc_preauth.c
 *
 * Copyright 1995 by the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * Export of this software from the United States of America may
 *   require a specific license from the United States Government.
 *   It is the responsibility of any person or organization contemplating
 *   export to obtain such a license before exporting.
 * 
 * WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
 * distribute this software and its documentation for any purpose and
 * without fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of M.I.T. not be used in advertising or publicity pertaining
 * to distribution of the software without specific, written prior
 * permission.  M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 * 
 * Preauthentication routines for the KDC.
 */

#include "k5-int.h"
#include "kdc_util.h"
#include "kdc5_err.h"
#include "extern.h"
#include <stdio.h>

static krb5_error_code get_enc_timestamp
    KRB5_PROTOTYPE((krb5_context, krb5_kdc_req *request,
		    krb5_db_entry *client, krb5_key_data *client_key,
		    krb5_db_entry *server, krb5_pa_data *data));
static krb5_error_code verify_enc_timestamp
    KRB5_PROTOTYPE((krb5_context, krb5_db_entry *client,
		    krb5_kdc_req *request,
		    krb5_enc_tkt_part * enc_tkt_reply, krb5_pa_data *data));

static krb5_error_code get_etype_info
    KRB5_PROTOTYPE((krb5_context, krb5_kdc_req *request,
		    krb5_db_entry *client, krb5_key_data *client_key,
		    krb5_db_entry *server, krb5_pa_data *data));

static krb5_error_code get_pw_salt
    KRB5_PROTOTYPE((krb5_context, krb5_kdc_req *request,
		    krb5_db_entry *client, krb5_key_data *client_key,
		    krb5_db_entry *server, krb5_pa_data *data));
static krb5_error_code return_pw_salt
    KRB5_PROTOTYPE((krb5_context, krb5_pa_data * padata, 
		    krb5_db_entry *client,
		    krb5_kdc_req *request, krb5_kdc_rep *reply,
		    krb5_key_data *client_key,
		    krb5_keyblock *encrypting_key,
		    krb5_pa_data **send_pa));

/* SAM preauth support */
static krb5_error_code verify_sam_response
    KRB5_PROTOTYPE((krb5_context, krb5_db_entry *client,
		    krb5_kdc_req *request,
		    krb5_enc_tkt_part * enc_tkt_reply, krb5_pa_data *data));

static krb5_error_code get_sam_edata
    KRB5_PROTOTYPE((krb5_context, krb5_kdc_req *request,
		    krb5_db_entry *client, krb5_key_data *client_key,
		    krb5_db_entry *server, krb5_pa_data *data));

static krb5_error_code get_x9_edata
    KRB5_PROTOTYPE((krb5_context, krb5_kdc_req *request,
		    krb5_db_entry *client, krb5_db_entry *server,
		    krb5_pa_data *data));
static krb5_error_code get_securid_edata
    KRB5_PROTOTYPE((krb5_context, krb5_kdc_req *request,
		    krb5_db_entry *client, krb5_db_entry *server,
		    krb5_pa_data *data));

typedef krb5_error_code (*verify_proc)
    KRB5_PROTOTYPE((krb5_context, krb5_db_entry *client,
		    krb5_kdc_req *request,
		    krb5_enc_tkt_part * enc_tkt_reply, krb5_pa_data *data));

typedef krb5_error_code (*edata_proc)
    KRB5_PROTOTYPE((krb5_context, krb5_kdc_req *request,
		    krb5_db_entry *client, krb5_key_data *client_key,
		    krb5_db_entry *server, krb5_pa_data *data));

typedef krb5_error_code (*return_proc)
    KRB5_PROTOTYPE((krb5_context, krb5_pa_data * padata, 
		    krb5_db_entry *client,
		    krb5_kdc_req *request, krb5_kdc_rep *reply,
		    krb5_key_data *client_key,
		    krb5_keyblock *encrypting_key,
		    krb5_pa_data **send_pa));

typedef struct _krb5_preauth_systems {
    int		type;
    int		flags;
    edata_proc	get_edata;
    verify_proc	verify_padata;
    return_proc return_padata;
} krb5_preauth_systems;

/*
 * Preauth property flags
 */
#define PA_HARDWARE	0x00000001
#define PA_SOFTWARE	0x00000002
#define PA_REQUIRED	0x00000004
#define PA_SUFFICIENT	0x00000008

static krb5_preauth_systems preauth_systems[] = {
    {
        KRB5_PADATA_ENC_TIMESTAMP,
        PA_SOFTWARE,
        get_enc_timestamp,
	verify_enc_timestamp,
	0
    },
    {
	KRB5_PADATA_ETYPE_INFO,
	PA_SOFTWARE|PA_HARDWARE,
	get_etype_info,
	0,
	0
    },
    {
	KRB5_PADATA_PW_SALT,
	PA_SOFTWARE|PA_HARDWARE,
	get_pw_salt, 
	0,
	return_pw_salt
    },
    {
	KRB5_PADATA_SAM_RESPONSE,
	0,
	0,
	verify_sam_response,
	0
    },
    {
	KRB5_PADATA_SAM_CHALLENGE,
	PA_HARDWARE,		/* causes get_preauth_hint_list to use this */
	get_sam_edata,
	0,
	0
    },
    { -1,}
};

#define MAX_PREAUTH_SYSTEMS (sizeof(preauth_systems)/sizeof(preauth_systems[0]))

static krb5_error_code
find_pa_system(type, preauth)
    int			type;
    krb5_preauth_systems	**preauth;
{
    krb5_preauth_systems 	*ap = preauth_systems;
    
    while ((ap->type != -1) && (ap->type != type))
	ap++;
    if (ap->type == -1)
	return(KRB5_PREAUTH_BAD_TYPE);
    *preauth = ap;
    return 0;
} 

const char *missing_required_preauth(client, server, enc_tkt_reply)
    krb5_db_entry *client, *server;
    krb5_enc_tkt_part *enc_tkt_reply;
{
#if 0
    /*
     * If this is the pwchange service, and the pre-auth bit is set,
     * allow it even if the HW preauth would normally be required.
     * 
     * Sandia national labs wanted this for some strange reason... we
     * leave it disabled normally.
     */
    if (isflagset(server->attributes, KRB5_KDB_PWCHANGE_SERVICE) &&
	isflagset(enc_tkt_reply->flags, TKT_FLG_PRE_AUTH))
	return 0;
#endif
    
    if (isflagset(client->attributes, KRB5_KDB_REQUIRES_HW_AUTH)) {
	if (!isflagset(enc_tkt_reply->flags, TKT_FLG_HW_AUTH))
	    return "NEEDED_HW_PREAUTH";
    } else if (isflagset(client->attributes, KRB5_KDB_REQUIRES_PRE_AUTH)) {
	if (!isflagset(enc_tkt_reply->flags, TKT_FLG_PRE_AUTH))
	    return "NEEDED_PREAUTH";
    }

    return 0;
}

void get_preauth_hint_list(request, client, client_key, server, e_data)
    krb5_kdc_req *request;
    krb5_db_entry *client, *server;
    krb5_key_data *client_key;
    krb5_data *e_data;
{
    int hw, sw;
    krb5_preauth_systems *ap;
    krb5_pa_data **pa_data, **pa;
    krb5_data *edat;
    krb5_error_code retval;
    
    /* Zero these out in case we need to abort */
    e_data->length = 0;
    e_data->data = 0;
    
    hw = isflagset(client->attributes, KRB5_KDB_REQUIRES_HW_AUTH);
    sw = (!hw) && (isflagset(client->attributes, KRB5_KDB_REQUIRES_PRE_AUTH));
    pa_data = malloc(sizeof(krb5_pa_data *) * (MAX_PREAUTH_SYSTEMS+1));
    if (pa_data == 0)
	return;
    memset(pa_data, 0, sizeof(krb5_pa_data *) * (MAX_PREAUTH_SYSTEMS+1));
    pa = pa_data;

    for (ap = preauth_systems; ap->type != -1; ap++) {
	if (((hw && (ap->flags & PA_HARDWARE)) ||
	     (sw && (ap->flags & PA_SOFTWARE))) &&
	    (ap->get_edata)) {

	    *pa = malloc(sizeof(krb5_pa_data));
	    if (*pa == 0)
		goto errout;
	    memset(*pa, 0, sizeof(krb5_pa_data));
	    (*pa)->magic = KV5M_PA_DATA;
	    (*pa)->pa_type = ap->type;
	    retval = (ap->get_edata)(kdc_context, request, client, client_key,
				     server, *pa);
	    if (retval) {
		/* just failed on this type, continue */
		free(*pa);
		*pa = 0;
		continue;
	    }
	    pa++;
	}
    }
    retval = encode_krb5_padata_sequence((const krb5_pa_data **) pa_data,
					 &edat);
    if (retval)
	goto errout;
    *e_data = *edat;
    free(edat);

errout:
    krb5_free_pa_data(kdc_context, pa_data);
    return;
}

/*
 * This routine is called to verify the preauthentication information
 * for a V5 request.
 * 	
 * Returns 0 if the pre-authentication is valid, non-zero to indicate
 * an error code of some sort.
 */

krb5_error_code
check_padata (context, client, request, enc_tkt_reply)
    krb5_context	context;
    krb5_db_entry *	client;
    krb5_kdc_req *	request;
    krb5_enc_tkt_part * enc_tkt_reply;
{
    krb5_error_code retval;
    krb5_pa_data **padata;
    krb5_preauth_systems *pa_sys;

    if (request->padata == 0)
	return 0;

    for (padata = request->padata; *padata; padata++) {
	if (find_pa_system((*padata)->pa_type, &pa_sys))
	    continue;
	if (pa_sys->verify_padata == 0)
	    continue;
	retval = pa_sys->verify_padata(context, client, request,
				       enc_tkt_reply, *padata);
	if (retval) {
	    if (pa_sys->flags & PA_REQUIRED)
		break;
	} else {
	    if (pa_sys->flags & PA_SUFFICIENT)
		break;
	}
    }
    if (retval)
	com_err("krb5kdc", retval, "pa verify failure");

    return retval;
}

/*
 * return_padata creates any necessary preauthentication
 * structures which should be returned by the KDC to the client
 */
krb5_error_code
return_padata(context, client, request, reply,
	      client_key, encrypting_key)
    krb5_context	context;
    krb5_db_entry *	client;
    krb5_kdc_req *	request;
    krb5_kdc_rep *	reply;
    krb5_key_data *	client_key;
    krb5_keyblock *	encrypting_key;
{
    krb5_error_code		retval;
    krb5_pa_data **		padata;
    krb5_pa_data **		send_pa_list;
    krb5_pa_data **		send_pa;
    krb5_pa_data *		pa = 0;
    krb5_preauth_systems *	ap;
    int 			size = 0;

    for (ap = preauth_systems; ap->type != -1; ap++) {
	if (ap->return_padata)
	    size++;
    }

    if ((send_pa_list = malloc((size+1) * sizeof(krb5_pa_data *))) == NULL)
	return ENOMEM;

    send_pa = send_pa_list;
    *send_pa = 0;
    
    for (ap = preauth_systems; ap->type != -1; ap++) {
	if (ap->return_padata == 0)
	    continue;
	pa = 0;
	if (request->padata) {
	    for (padata = request->padata; *padata; padata++) {
		if ((*padata)->pa_type == ap->type) {
		    pa = *padata;
		    break;
		}
	    }
	}
	if ((retval = ap->return_padata(context, pa, client, request, reply,
					client_key, encrypting_key, send_pa)))
	    goto cleanup;

	if (*send_pa)
	    send_pa++;
	*send_pa = 0;
    }
    
    retval = 0;

    if (send_pa_list[0]) {
	reply->padata = send_pa_list;
	send_pa_list = 0;
    }
    
cleanup:
    if (send_pa_list)
	krb5_free_pa_data(context, send_pa_list);
    return (retval);
}

static krb5_error_code
get_enc_timestamp(context, request, client, client_key, server, data)
     krb5_context context;
     krb5_kdc_req *request;
     krb5_db_entry *client;
     krb5_key_data *client_key;
     krb5_db_entry *server;
     krb5_pa_data *data;
{ 
    /* the data is empty, but it needs to be here */

    data->contents = NULL;
    data->length = 0;

    return(0);
}

static krb5_error_code
verify_enc_timestamp(context, client, request, enc_tkt_reply, pa)
    krb5_context	context;
    krb5_db_entry *	client;
    krb5_kdc_req *	request;
    krb5_enc_tkt_part * enc_tkt_reply;
    krb5_pa_data *	pa;
{
    krb5_pa_enc_ts *		pa_enc = 0;
    krb5_error_code		retval;
    krb5_data			scratch;
    krb5_data			enc_ts_data;
    krb5_enc_data 		*enc_data = 0;
    krb5_keyblock		key;
    krb5_key_data *		client_key;
    krb5_int32			start;
    krb5_timestamp		timenow;
    
    enc_ts_data.data = 0;
    scratch.data = pa->contents;
    scratch.length = pa->length;
    
    if ((retval = decode_krb5_enc_data(&scratch, &enc_data)) != 0) {
	retval = KRB5KDC_ERR_PREAUTH_FAILED;
	goto cleanup;
    }

    start = 0;
    while (1) {
	if ((retval = krb5_dbe_search_enctype(context, client,
					      &start, enc_data->enctype,
					      -1, 0, &client_key))) {
	    if (retval == ENOENT)
		retval = KRB5KRB_AP_ERR_BAD_INTEGRITY;
	    else
		retval = KRB5KDC_ERR_PREAUTH_FAILED;
	    goto cleanup;
	}

	if ((retval = krb5_dbekd_decrypt_key_data(context, &master_encblock, 
						  client_key, &key, NULL))) {
	    retval = KRB5KDC_ERR_PREAUTH_FAILED;
	    goto cleanup;
	}
	key.enctype = enc_data->enctype;

	retval = krb5_decrypt_data(context, &key, 0, enc_data, &enc_ts_data);
	memset((char *)key.contents, 0, key.length);
	krb5_xfree(key.contents);

	if (retval == 0)
	    break;
    }

    if ((retval = decode_krb5_pa_enc_ts(&enc_ts_data, &pa_enc)) != 0) {
	retval = KRB5KDC_ERR_PREAUTH_FAILED;
	goto cleanup;
    }

    if ((retval = krb5_timeofday(context, &timenow)) != 0) {
	retval = KRB5KDC_ERR_PREAUTH_FAILED;
	goto cleanup;
    }
    
    if (labs(timenow - pa_enc->patimestamp) > context->clockskew) {
	retval = KRB5KRB_AP_ERR_SKEW;
	goto cleanup;
    }

    setflag(enc_tkt_reply->flags, TKT_FLG_PRE_AUTH);

    retval = 0;
    
cleanup:
    if (enc_data) {
	if (enc_data->ciphertext.data)
	    krb5_xfree(enc_data->ciphertext.data);
	free(enc_data);
    }
    if (enc_ts_data.data)
	krb5_xfree(enc_ts_data.data);
    if (pa_enc)
	krb5_xfree(pa_enc);
    return retval;
}

/*
 * This function returns the etype information for a particular
 * client, to be passed back in the preauth list in the KRB_ERROR
 * message.
 */
static krb5_error_code
get_etype_info(context, request, client, xxx, server, pa_data)
    krb5_context 	context;
    krb5_kdc_req *	request;
    krb5_db_entry *	client;
    krb5_key_data *	xxx;
    krb5_db_entry *	server;
    krb5_pa_data *	pa_data;
{
    krb5_etype_info_entry **	entry = 0;
    krb5_key_data		*client_key;
    krb5_error_code		retval;
    krb5_data			salt;
    krb5_data *			scratch;
    krb5_enctype		db_etype;
    int 			i = 0;
    int 			start = 0;

    salt.data = 0;

    entry = malloc((client->n_key_data * 2 + 1) * sizeof(krb5_etype_info_entry *));
    if (entry == NULL)
	return ENOMEM;
    entry[0] = NULL;

    while (1) {
	retval = krb5_dbe_search_enctype(context, client, &start, -1,
					 -1, 0, &client_key);
	if (retval == ENOENT)
	    break;
	if (retval)
	    goto cleanup;
	db_etype = client_key->key_data_type[0];
	if (db_etype == ENCTYPE_DES_CBC_MD4 || db_etype == ENCTYPE_DES_CBC_MD5)
	    db_etype = ENCTYPE_DES_CBC_CRC;
	
	while (1) {
	    if ((entry[i] = malloc(sizeof(krb5_etype_info_entry))) == NULL) {
		retval = ENOMEM;
		goto cleanup;
	    }
	    entry[i+1] = 0;
	    entry[i]->magic = KV5M_ETYPE_INFO_ENTRY;
	    entry[i]->etype = db_etype;
	    entry[i]->length = -1;
	    entry[i]->salt = 0;
	    retval = get_salt_from_key(context, request->client,
				       client_key, &salt);
	    if (retval)
		goto cleanup;
	    if (salt.length >= 0) {
		entry[i]->length = salt.length;
		entry[i]->salt = salt.data;
		salt.data = 0;
	    }
	    i++;
	    /*
	     * If we have a DES_CRC key, it can also be used as a
	     * DES_MD5 key.
	     */
	    if (db_etype == ENCTYPE_DES_CBC_CRC)
		db_etype = ENCTYPE_DES_CBC_MD5;
	    else
		break;
	}
    }
    retval = encode_krb5_etype_info((const krb5_etype_info_entry **) entry,
				    &scratch);
    if (retval)
	goto cleanup;
    pa_data->contents = scratch->data;
    pa_data->length = scratch->length;

    retval = 0;

cleanup:
    if (entry)
	krb5_free_etype_info(context, entry);
    if (salt.data)
	krb5_xfree(salt.data);
    return retval;
}

static krb5_error_code
make_pw_salt(context, client, client_key, padata)
     krb5_context context;
     krb5_principal client;
     krb5_key_data *client_key;
     krb5_pa_data *padata;
{ 
    krb5_error_code retval;
    krb5_data *		scratch;
    krb5_data		salt_data;

    if (client_key->key_data_ver == 1 ||
	client_key->key_data_type[1] == KRB5_KDB_SALTTYPE_NORMAL) {
	/* the caller will ignore errors in get_edata functions,
	   and not produce a padata, which is what we want. */
	return -1;
    }

    switch (client_key->key_data_type[1]) {
    case KRB5_KDB_SALTTYPE_V4:
	/* send an empty (V4) salt */
	padata->contents = 0;
	padata->length = 0;
	break;
    case KRB5_KDB_SALTTYPE_NOREALM:
	if ((retval = krb5_principal2salt_norealm(kdc_context, client,
						  &salt_data)))
	    goto cleanup;
	padata->contents = (krb5_octet *)salt_data.data;
	padata->length = salt_data.length;
	break;
    case KRB5_KDB_SALTTYPE_AFS3:
	/* send an AFS style realm-based salt */
	/* for now, just pass the realm back and let the client
	   do the work. In the future, add a kdc configuration
	   variable that specifies the old cell name. */
	padata->pa_type = KRB5_PADATA_AFS3_SALT;
	/* it would be just like ONLYREALM, but we need to pass the 0 */
	scratch = krb5_princ_realm(kdc_context, client);
	if ((padata->contents = malloc(scratch->length+1)) == NULL) {
	    retval = ENOMEM;
	    goto cleanup;
	}
	memcpy(padata->contents, scratch->data, scratch->length);
	padata->length = scratch->length+1;
	padata->contents[scratch->length] = 0;
	break;
    case KRB5_KDB_SALTTYPE_ONLYREALM:
	scratch = krb5_princ_realm(kdc_context, client);
	if ((padata->contents = malloc(scratch->length)) == NULL) {
	    retval = ENOMEM;
	    goto cleanup;
	}
	memcpy(padata->contents, scratch->data, scratch->length);
	padata->length = scratch->length;
	break;
    case KRB5_KDB_SALTTYPE_SPECIAL:
	if ((padata->contents = malloc(client_key->key_data_length[1]))
	    == NULL) {
	    retval = ENOMEM;
	    goto cleanup;
	}
	memcpy(padata->contents, client_key->key_data_contents[1],
	       client_key->key_data_length[1]);
	padata->length = client_key->key_data_length[1];
	break;
    default:
	free(padata);
	return 0;
    }

    return 0;
    
cleanup:
    free(padata);
    return retval;
}

static krb5_error_code
get_pw_salt(context, request, client, client_key, server, data)
     krb5_context context;
     krb5_kdc_req *request;
     krb5_db_entry *client;
     krb5_key_data *client_key;
     krb5_db_entry *server;
     krb5_pa_data *data;
{
    return(make_pw_salt(context, client->princ, client_key, data));
}

static krb5_error_code
return_pw_salt(context, in_padata, client, request, reply, client_key,
	       encrypting_key, send_pa)
    krb5_context	context;
    krb5_pa_data *	in_padata;
    krb5_db_entry *	client;
    krb5_kdc_req *	request;
    krb5_kdc_rep *	reply;
    krb5_key_data *	client_key;
    krb5_keyblock *	encrypting_key;
    krb5_pa_data **	send_pa;
{
    krb5_error_code ret;
    krb5_pa_data *	padata;

    if ((padata = malloc(sizeof(krb5_pa_data))) == NULL)
	return ENOMEM;
    padata->magic = KV5M_PA_DATA;
    padata->pa_type = KRB5_PADATA_PW_SALT;

    if (ret = make_pw_salt(context, request->client, client_key, padata)) {
	/* if no pw_salt was made, return nothing to the client */
	if (ret == -1)
	    return(0);
	free(padata);
	return(ret);
    }

    *send_pa = padata;
    return(0);
}

static krb5_error_code
get_sam_edata(context, request, client, client_key, server, pa_data)
    krb5_context 	context;
    krb5_kdc_req *	request;
    krb5_db_entry *	client;
    krb5_key_data *	client_key;
    krb5_db_entry *	server;
    krb5_pa_data *	pa_data;
{
    krb5_error_code		retval;
    retval = get_x9_edata(context, request, client, server, pa_data);
    if (!retval) return retval;	/* success */
#ifdef WITH_PREDICTIVE_SECURID
    retval = get_securid_edata(context, request, client, server, pa_data);
    if (!retval) return retval;	/* success */
#endif /* WITH_PREDICTIVE_SECURID */
    /* insert other attempts here */
    return retval;		/* report last failure */
}

#define KADMIN_TL_X9_KEY	0x0101
typedef struct _kadmin_tl_x9_key {
    krb5_int16		sam_type;
    krb5_keyblock	key;
} kadmin_tl_x9_key;

krb5_error_code
kadmin_parse_x9_info(context, tld, x9key)
    krb5_context	context;
    krb5_tl_data       *tld;
    kadmin_tl_x9_key   *x9key;
{
    krb5_octet	       *ptr;
    krb5_int16		tmp;

    ptr = tld->tl_data_contents;
    krb5_kdb_decode_int16(ptr, x9key->sam_type);	ptr += 2;
    krb5_kdb_decode_int16(ptr, tmp);			ptr += 2;
    x9key->key.enctype = tmp;
    krb5_kdb_decode_int16(ptr, tmp);			ptr += 2;
    x9key->key.length = tmp;
#if 0
com_err("krb5kdc", 0, "type %d enctype %d len %d", 
	x9key->sam_type, x9key->key.enctype, x9key->key.length);
#endif
    x9key->key.contents = malloc(x9key->key.length);
    if (!x9key->key.contents) return ENOMEM;
    memcpy(x9key->key.contents, ptr, x9key->key.length);
#if 0
com_err("krb5kdc", 0,
	"starting fixup of key %02x%02x%02x%02x %02x%02x%02x%02x",
	ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
#endif
    if (x9key->key.enctype == ENCTYPE_DES_CBC_RAW)
	mit_des_fixup_keyblock_parity(&x9key->key);
#if 0
com_err("krb5kdc", 0, 
	"fixup done: %02x%02x%02x%02x %02x%02x%02x%02x",
	ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5], ptr[6], ptr[7]);
#endif
    return 0;
}


static krb5_error_code
get_x9_edata(context, request, client, server, pa_data)
    krb5_context 	context;
    krb5_kdc_req *	request;
    krb5_db_entry *	client;
    krb5_db_entry *	server;
    krb5_pa_data *	pa_data;
{
    krb5_error_code		retval;
    krb5_sam_challenge		sc;
    krb5_predicted_sam_response	psr;
    krb5_data *			scratch;
    int 			i = 0;
    int 			start = 0;
    krb5_encrypt_block		eblock;
    krb5_keyblock encrypting_key;
    char response[9];
    char inputblock[8];
    krb5_data predict_response;

    /*
      Done correctly, instead of probing for seperate keys we simply
      find the KADMIN_TL_X9_KEY record in the tl_data.

      We currently assume only record per principal, and the sam_type
      field indicates which type of card.
     */
    krb5_tl_data client_tld;
    kadmin_tl_x9_key x9key;
    char *device_type_name;
    char *device_vendor_name;

    client_tld.tl_data_type = KADMIN_TL_X9_KEY;
    retval = krb5_dbe_lookup_tl_data(context, client, &client_tld);
    if (retval) return retval;
    if (!client_tld.tl_data_length) return KDC5_PREAUTH_TL_NOTFOUND;
    retval = kadmin_parse_x9_info(context, &client_tld, &x9key);
    if (retval) return retval;

    sc.sam_type = x9key.sam_type;
    /* not stored encrypted, per marc's advice */
    /* encrypting_key.contents is the copy that gets freed on exit */
    encrypting_key = x9key.key;

    sc.magic = KV5M_SAM_CHALLENGE;
    sc.sam_flags = KRB5_SAM_USE_SAD_AS_KEY;

    switch (sc.sam_type) {
    case PA_SAM_TYPE_ACTIVCARD_DEC:
    case PA_SAM_TYPE_ACTIVCARD_HEX:
	device_type_name = "activcard";
	device_vendor_name = "Activcard";
	break;
    case PA_SAM_TYPE_DIGI_PATH:
    case PA_SAM_TYPE_DIGI_PATH_HEX:
	device_type_name = "snk4";
	device_vendor_name = "Digital Pathways";
	break;
    default:
	device_type_name = "x9.9";
	device_vendor_name = "generic x9.9";
	break;
    }


    switch (sc.sam_type) {
    case PA_SAM_TYPE_GRAIL:
      sc.sam_type_name.data = "Experimental System";
      sc.sam_type_name.length = strlen(sc.sam_type_name.data);
      sc.sam_challenge_label.data = "experimental challenge label";
      sc.sam_challenge_label.length = strlen(sc.sam_challenge_label.data);
      sc.sam_challenge.data = "12345";
      sc.sam_challenge.length = strlen(sc.sam_challenge.data);

      psr.magic = KV5M_PREDICTED_SAM_RESPONSE;
      /* string2key on sc.sam_challenge goes in here */
      /* eblock is just to set the enctype */
      {
	const krb5_enctype type = ENCTYPE_DES_CBC_MD5;
	if (!valid_enctype(type)) return KRB5_PROG_ETYPE_NOSUPP;
	krb5_use_enctype(context, &eblock, type);
	retval = krb5_string_to_key(context, &eblock, 
				    &psr.sam_key, &sc.sam_challenge, 
				    0 /* salt */);
	retval = encode_krb5_predicted_sam_response(&psr, &scratch);
	if (retval) goto cleanup;
	
	{
	  krb5_enc_data tmpdata;
	  retval = krb5_encrypt_data(context, master_encblock.key, 0, 
				     scratch, &tmpdata);
	  sc.sam_track_id = tmpdata.ciphertext;
	}
	if (retval) goto cleanup;
      }

      sc.sam_response_prompt.data = "response prompt";
      sc.sam_response_prompt.length = strlen(sc.sam_response_prompt.data);
      sc.sam_pk_for_sad.length = 0;
      sc.sam_nonce = 0;
      /* Generate checksum */
      /*krb5_checksum_size(context, ctype)*/
      /*krb5_calculate_checksum(context,ctype,in,in_length,seed,
	seed_length,outcksum) */
      /*krb5_verify_checksum(context,ctype,cksum,in,in_length,seed,
	seed_length) */
      sc.sam_cksum.contents = (krb5_octet *)
	malloc(krb5_checksum_size(context, CKSUMTYPE_RSA_MD5_DES));
      if (sc.sam_cksum.contents == NULL) return(ENOMEM);

      retval = krb5_calculate_checksum(context, CKSUMTYPE_RSA_MD5_DES,
				       sc.sam_challenge.data,
				       sc.sam_challenge.length,
				       psr.sam_key.contents, /* key */
				       psr.sam_key.length, /* key length */
				       &sc.sam_cksum);
      if (retval) { free(sc.sam_cksum.contents); return(retval); }
      
      retval = encode_krb5_sam_challenge(&sc, &scratch);
      if (retval) goto cleanup;
      pa_data->magic = KV5M_PA_DATA;
      pa_data->pa_type = KRB5_PADATA_SAM_CHALLENGE;
      pa_data->contents = scratch->data;
      pa_data->length = scratch->length;
      
      retval = 0;
      break;

    case PA_SAM_TYPE_ACTIVCARD_DEC:
    case PA_SAM_TYPE_ACTIVCARD_HEX:
    case PA_SAM_TYPE_DIGI_PATH:
    case PA_SAM_TYPE_DIGI_PATH_HEX:
      sc.sam_type_name.data = device_vendor_name;
      sc.sam_type_name.length = strlen(sc.sam_type_name.data);
#if 1
      sc.sam_challenge_label.data = "Enter the following on your keypad";
      sc.sam_challenge_label.length = strlen(sc.sam_challenge_label.data);
#endif
      /* generate digit string, take it mod 1000000 (six digits.) */
      {
	int j;
	krb5_encrypt_block eblock;
	krb5_keyblock *session_key = 0;
	char outputblock[8];
	int i;
	memset(inputblock, 0, 8);
	krb5_use_enctype(kdc_context, &eblock, ENCTYPE_DES_CBC_CRC);
	retval = krb5_random_key(kdc_context, &eblock, 
				 krb5_enctype_array[ENCTYPE_DES_CBC_CRC]->random_sequence,
				 &session_key);
	if (retval) {
	  /* random key failed */
	  com_err("krb5kdc", retval,"generating random challenge for preauth");
	  return retval;
	}
	/* now session_key has a key which we can pick bits out of */
	/* we need six decimal digits. Grab 6 bytes, div 2, mod 10 each. */
	if (session_key->length != 8) {
	  com_err("krb5kdc", retval = KRB5KDC_ERR_ETYPE_NOSUPP,
		  "keytype didn't match code expectations");
	  return retval;
	}
	for(i = 0; i<6; i++) {
	  inputblock[i] = '0' + ((session_key->contents[i]/2) % 10);
	}
	if (session_key)
	  krb5_free_keyblock(kdc_context, session_key);

	/* retval = krb5_finish_key(kdc_context, &eblock); */
	/* now we have inputblock containing the 8 byte input to DES... */
	sc.sam_challenge.data = inputblock;
	sc.sam_challenge.length = 6;

	krb5_use_enctype(kdc_context, &eblock, ENCTYPE_DES_CBC_RAW);
	encrypting_key.enctype = ENCTYPE_DES_CBC_RAW;
	/* do any necessary key pre-processing */
	retval= krb5_process_key(kdc_context, &eblock, &encrypting_key);

	if (retval) {
	  com_err("krb5kdc", retval, "%s processing key", device_type_name);
	  return retval;
	}

	{
	  char ivec[8];
	  memset(ivec,0,8);
	  retval = krb5_encrypt(kdc_context, inputblock, outputblock, 8,
				&eblock, ivec);
	}
	if (retval) {
	  com_err("krb5kdc", retval, "%s response generation failed", 
		  device_type_name);
	  return retval;
	}
	/* now output block is the raw bits of the response; convert it
	   to display form */
	for (j=0; j<4; j++) {
	  char n[2];
	  int k;
	  n[0] = outputblock[j] & 0xf;
	  n[1] = (outputblock[j]>>4) & 0xf;
	  for (k=0; k<2; k++) {
	    switch (sc.sam_type) {
	    case PA_SAM_TYPE_DIGI_PATH:	/* fold abc->2, def->3 */
	      /* This is equivalent to:
		 if(n[k]>=0xa && n[k]<=0xc) n[k] = 2;
		 if(n[k]>=0xd && n[k]<=0xf) n[k] = 3;
	       */
	      if(n[k] > 9) n[k] = ((n[k]-1)>>2);
	      break;
	    default:		/* straight hex */
	    case PA_SAM_TYPE_DIGI_PATH_HEX:
	    case PA_SAM_TYPE_ACTIVCARD_HEX:
	      if(n[k] > 9) n[k] += 'a' - '9' + 1;
	      break;
	    case PA_SAM_TYPE_ACTIVCARD_DEC:
	      if(n[k] > 9) n[k] -= 10; /* a=0, f=5 */
	      break;
	    }
	  }
	  /* for v4, we keygen: *(j+(char*)&key1) = (n[1]<<4) | n[0]; */
	  /* for v5, we just generate a string */
	  response[2*j+0] = '0' + n[1];
	  response[2*j+1] = '0' + n[0];
	  /* and now, response has what we work with. */
	}
	response[8] = 0;
	predict_response.data = response;
	predict_response.length = 8;
#if 0				/* for debugging, hack the output too! */
sc.sam_challenge_label.data = response;
sc.sam_challenge_label.length = strlen(sc.sam_challenge_label.data);
#endif
      }

      psr.magic = KV5M_PREDICTED_SAM_RESPONSE;
      /* string2key on sc.sam_challenge goes in here */
      /* eblock is just to set the enctype */
      {
	const krb5_enctype type = ENCTYPE_DES_CBC_MD5;
	if (!valid_enctype(type)) return KRB5_PROG_ETYPE_NOSUPP;
	krb5_use_enctype(context, &eblock, type);
	retval = krb5_string_to_key(context, &eblock, 
				    &psr.sam_key, &predict_response, 
				    0 /* salt */);
	retval = encode_krb5_predicted_sam_response(&psr, &scratch);
	if (retval) goto cleanup;
	
	{
	  krb5_enc_data tmpdata;
	  retval = krb5_encrypt_data(context, master_encblock.key, 0, 
				     scratch, &tmpdata);
	  sc.sam_track_id = tmpdata.ciphertext;
	}
	if (retval) goto cleanup;
      }

      switch (sc.sam_type) {
      case PA_SAM_TYPE_DIGI_PATH:
      case PA_SAM_TYPE_ACTIVCARD_DEC:
	sc.sam_response_prompt.data = "Enter the displayed response";
	break;
      default:
      case PA_SAM_TYPE_DIGI_PATH_HEX:
      case PA_SAM_TYPE_ACTIVCARD_HEX:
	sc.sam_response_prompt.data = "Enter the displayed hex response (in lower case)";
	break;
      }
      sc.sam_response_prompt.length = strlen(sc.sam_response_prompt.data);
      sc.sam_pk_for_sad.length = 0;
      sc.sam_nonce = 0;
      /* Generate checksum */
      /*krb5_checksum_size(context, ctype)*/
      /*krb5_calculate_checksum(context,ctype,in,in_length,seed,
	seed_length,outcksum) */
      /*krb5_verify_checksum(context,ctype,cksum,in,in_length,seed,
	seed_length) */
      sc.sam_cksum.contents = (krb5_octet *)
	malloc(krb5_checksum_size(context, CKSUMTYPE_RSA_MD5_DES));
      if (sc.sam_cksum.contents == NULL) return(ENOMEM);

      retval = krb5_calculate_checksum(context, CKSUMTYPE_RSA_MD5_DES,
				       sc.sam_challenge.data,
				       sc.sam_challenge.length,
				       psr.sam_key.contents, /* key */
				       psr.sam_key.length, /* key length */
				       &sc.sam_cksum);
      if (retval) { free(sc.sam_cksum.contents); return(retval); }
      
      retval = encode_krb5_sam_challenge(&sc, &scratch);
      if (retval) goto cleanup;
      pa_data->magic = KV5M_PA_DATA;
      pa_data->pa_type = KRB5_PADATA_SAM_CHALLENGE;
      pa_data->contents = scratch->data;
      pa_data->length = scratch->length;
      
      retval = 0;
      break;

    default:
      retval = KRB5KDC_ERR_PADATA_TYPE_NOSUPP;
      com_err("krb5kdc", 0, "unknown SAM edata response type");
      break;

    }

cleanup:
    memset((char *)encrypting_key.contents, 0, encrypting_key.length);
    krb5_xfree(encrypting_key.contents);
    return retval;
}

static krb5_error_code
verify_sam_response(context, client, request, enc_tkt_reply, pa)
    krb5_context	context;
    krb5_db_entry *	client;
    krb5_kdc_req *	request;
    krb5_enc_tkt_part * enc_tkt_reply;
    krb5_pa_data *	pa;
{
    krb5_error_code		retval;
    krb5_data			scratch;
    krb5_sam_response		*sr = 0;
    krb5_predicted_sam_response	*psr = 0;
    krb5_enc_sam_response_enc	*esre = 0;
    krb5_timestamp		timenow;

    scratch.data = pa->contents;
    scratch.length = pa->length;
    
    retval = decode_krb5_sam_response(&scratch, &sr);
    if (retval) com_err("krb5kdc", retval, "decode_krb5_sam_response failed");
    if (retval) goto cleanup;

    {
      krb5_enc_data tmpdata;
      tmpdata.ciphertext = sr->sam_track_id;
      retval = krb5_decrypt_data(context, master_encblock.key, 0, 
				 &tmpdata, &scratch);
      if (retval) com_err("krb5kdc", retval, "decrypt track_id failed");
    }
    if (retval) goto cleanup;
    retval = decode_krb5_predicted_sam_response(&scratch, &psr);
    if (retval) com_err("krb5kdc", retval, "decode_krb5_predicted_sam_response failed");
    if (retval) goto cleanup;
    {
      /* now psr.sam_key is what we said to use... */
      retval = krb5_decrypt_data(context, &psr->sam_key, 0, 
				 &sr->sam_enc_nonce_or_ts, &scratch);
      if (retval) com_err("krb5kdc", retval, "decrypt nonce_or_ts failed");
    }
    if (retval) goto cleanup;
    retval = decode_krb5_enc_sam_response_enc(&scratch, &esre);
    if (retval) com_err("krb5kdc", retval, "decode_krb5_enc_sam_response_enc failed");
    if (retval) goto cleanup;
    if (esre->sam_timestamp != sr->sam_patimestamp) {
      retval = KRB5KDC_ERR_PREAUTH_FAILED;
      goto cleanup;
    }
    retval = krb5_timeofday(context, &timenow);
    if (retval) goto cleanup;
    
    if (labs(timenow - sr->sam_patimestamp) > context->clockskew) {
	retval = KRB5KRB_AP_ERR_SKEW;
	goto cleanup;
    }

    setflag(enc_tkt_reply->flags, TKT_FLG_HW_AUTH);
  cleanup:
    if (retval) com_err("krb5kdc", retval, "sam verify failure");
    if (sr) krb5_xfree(sr);
    if (psr) krb5_xfree(psr);
    if (esre) krb5_xfree(esre);

    return retval;
}

#ifdef WITH_PREDICTIVE_SECURID
#define KADMIN_TL_SECURID_TAG	0x0102
typedef struct _kadmin_tl_securid_tag {
    krb5_int16		sam_type;
    krb5_data		securid_tag; /* use this tag name not the principal */
} kadmin_tl_securid_tag;

krb5_error_code
kadmin_parse_securid_tag(context, tld, stag)
    krb5_context	context;
    krb5_tl_data       *tld;
    kadmin_tl_securid_tag   *stag;
{
    krb5_octet	       *ptr;
    krb5_int16		tmp;

    ptr = tld->tl_data_contents;
    krb5_kdb_decode_int16(ptr, stag->sam_type);	ptr += 2;
    krb5_kdb_decode_int16(ptr, tmp);		ptr += 2;
    stag->securid_tag.length = tmp;
    stag->securid_tag.data = malloc(stag->securid_tag.length);
    if (!stag->securid_tag.data) return ENOMEM;
    memcpy(stag->securid_tag.data, ptr, stag->securid_tag.length);
    return 0;
}



#ifdef SECURID_TEST
#define GOODUSER "test"
#define EXPIREDUSER "oldtest"
#define PASSCODE "userpin123456"

int get_ace_passcode (username, passcode, n)
char *username, *passcode;
int n;
{
  if (strcmp(username, GOODUSER) == 0) {
    strncpy(passcode, PASSCODE, n);
    return(1);
  } else if (strcmp(username, EXPIREDUSER) == 0) {
    return(-1);
  } else {
    return(0);
  }
}
#endif /* SECURID_TEST */

static krb5_error_code
get_securid_edata(context, request, client, server, pa_data)
    krb5_context 	context;
    krb5_kdc_req *	request;
    krb5_db_entry *	client;
    krb5_db_entry *	server;
    krb5_pa_data *	pa_data;
{
    krb5_error_code		retval;
    krb5_sam_challenge		sc;
    krb5_predicted_sam_response	psr;
    krb5_data *			scratch;
    int 			i = 0;
    int 			start = 0;
    krb5_encrypt_block		eblock;
    char response[9];
    char inputblock[8];
    krb5_data predict_response;

    /*
      Done correctly, instead of probing for seperate keys we simply
      find the KADMIN_TL_SECURID_TAG record in the tl_data.

      We currently assume only record per principal, and the sam_type
      field indicates which type of card.
     */
    krb5_tl_data client_tld;
    kadmin_tl_securid_tag stag;
    char *device_type_name;
    char *device_vendor_name;

    predict_response.length = 0;

    client_tld.tl_data_type = KADMIN_TL_SECURID_TAG;
    retval = krb5_dbe_lookup_tl_data(context, client, &client_tld);
    if (retval) return retval;
    if (!client_tld.tl_data_length) return KDC5_PREAUTH_TL_NOTFOUND;
    retval = kadmin_parse_securid_tag(context, &client_tld, &stag);
    if (retval) return retval;

    sc.sam_type = stag.sam_type;
    /* here, feed stag.securid_tag to database. */
    {
	int rv; char passcode[128];
	rv = get_ace_passcode(stag.securid_tag.data, 
			      passcode, sizeof passcode);
	if (rv != 1) {
	    /* RETURNS:
	     * -1 card deactivated
	     *  0 user not database
	     *  1 user found in database, passcode value set
	     */
	    return (!rv)?KDC5_PREAUTH_DB_NOTFOUND:KDC5_PREAUTH_DB_DISABLED;
	}
	predict_response.data = malloc(strlen(passcode)+1);
	if (!predict_response.data) return ENOMEM;
	strcpy(predict_response.data, passcode);
	predict_response.length = strlen(passcode);
    }

    sc.magic = KV5M_SAM_CHALLENGE;
    sc.sam_flags = KRB5_SAM_USE_SAD_AS_KEY;

    switch (sc.sam_type) {
    case PA_SAM_TYPE_SECURID_PREDICT:
	device_type_name = "SecurID";
	device_vendor_name = "Security Dynamics";
	break;
    default:
	device_type_name = "unknown predicted-string token";
	device_vendor_name = "unknown predicted-string vendor";
	break;
    }


    switch (sc.sam_type) {
    case PA_SAM_TYPE_SECURID_PREDICT:
      sc.sam_type_name.data = device_vendor_name;
      sc.sam_type_name.length = strlen(sc.sam_type_name.data);

#if 0
      sc.sam_challenge_label.data = "Enter the current displayed token value";
      sc.sam_challenge_label.length = strlen(sc.sam_challenge_label.data);
#else
      sc.sam_challenge_label.data = NULL;
      sc.sam_challenge_label.length = 0;
#endif

      psr.magic = KV5M_PREDICTED_SAM_RESPONSE;
      /* string2key on sc.sam_challenge goes in here */
      /* eblock is just to set the enctype */
      {
	const krb5_enctype type = ENCTYPE_DES_CBC_MD5;
	if (!valid_enctype(type)) return KRB5_PROG_ETYPE_NOSUPP;
	krb5_use_enctype(context, &eblock, type);
	retval = krb5_string_to_key(context, &eblock, 
				    &psr.sam_key, &predict_response, 
				    0 /* salt */);
	retval = encode_krb5_predicted_sam_response(&psr, &scratch);
	if (retval) goto cleanup;
	
	{
	  krb5_enc_data tmpdata;
	  retval = krb5_encrypt_data(context, master_encblock.key, 0, 
				     scratch, &tmpdata);
	  sc.sam_track_id = tmpdata.ciphertext;
	}
	if (retval) goto cleanup;
      }

      switch (sc.sam_type) {
      case PA_SAM_TYPE_SECURID_PREDICT:
	sc.sam_response_prompt.data = "Enter the current displayed token value";
	break;
      default:
	sc.sam_response_prompt.data = "Enter generic token value";
	break;
      }
      sc.sam_response_prompt.length = strlen(sc.sam_response_prompt.data);
      sc.sam_pk_for_sad.length = 0;
      sc.sam_nonce = 0;
      /* Generate checksum */
      sc.sam_cksum.contents = (krb5_octet *)
	malloc(krb5_checksum_size(context, CKSUMTYPE_RSA_MD5_DES));
      if (sc.sam_cksum.contents == NULL) return(ENOMEM);

      /* just checksum the response_prompt since that's all we send */
      retval = krb5_calculate_checksum(context, CKSUMTYPE_RSA_MD5_DES,
				       sc.sam_response_prompt.data,
				       sc.sam_response_prompt.length,
				       psr.sam_key.contents, /* key */
				       psr.sam_key.length, /* key length */
				       &sc.sam_cksum);
      if (retval) { free(sc.sam_cksum.contents); return(retval); }

      sc.sam_challenge.data = "";
      sc.sam_challenge.length = 0;
      sc.sam_challenge_label.data = "";
      sc.sam_challenge_label.length = 0;
      
      retval = encode_krb5_sam_challenge(&sc, &scratch);
      if (retval) goto cleanup;
      pa_data->magic = KV5M_PA_DATA;
      pa_data->pa_type = KRB5_PADATA_SAM_CHALLENGE;
      pa_data->contents = scratch->data;
      pa_data->length = scratch->length;
      
      retval = 0;
      break;

    default:
      retval = KRB5KDC_ERR_PADATA_TYPE_NOSUPP;
      com_err("krb5kdc", 0, "unknown SAM edata response type");
      break;

    }

cleanup:
    if (predict_response.length) {
	memset((char *)predict_response.data, 0, predict_response.length);
	krb5_xfree(predict_response.data);
    }
    return retval;
}

#endif /* WITH_PREDICTIVE_SECURID */
