/*
 * src/lib/krb5/asn.1/asn1_encode.c
 * 
 * Copyright 1994 by the Massachusetts Institute of Technology.
 * All Rights Reserved.
 *
 * Export of this software from the United States of America may
 *   require a specific license from the United States Government.
 *   It is the responsibility of any person or organization contemplating
 *   export to obtain such a license before exporting.
 * 
 * WITHIN THAT CONSTRAINT, permission to use, copy, modify, and
 * distribute this software and its documentation for any purpose and
 * without fee is hereby granted, provided that the above copyright
 * notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that
 * the name of M.I.T. not be used in advertising or publicity pertaining
 * to distribution of the software without specific, written prior
 * permission.  M.I.T. makes no representations about the suitability of
 * this software for any purpose.  It is provided "as is" without express
 * or implied warranty.
 */

/* ASN.1 primitive encoders */

#include "asn1_encode.h"
#include "asn1_make.h"

asn1_error_code asn1_encode_integer(buf, val, retlen)
     asn1buf * buf;
     const long val;
     int * retlen;
{
  asn1_error_code retval;
  int length = 0, partlen;
  long valcopy;
  int digit;
  
  valcopy = val;
  do {
    digit = (int) (valcopy&0xFF);
    retval = asn1buf_insert_octet(buf,(asn1_octet) digit);
    if(retval) return retval;
    length++;
    valcopy = valcopy >> 8;
  } while (valcopy != 0 && valcopy != ~0);

  if((val > 0) && ((digit&0x80) == 0x80)) { /* make sure the high bit is */
    retval = asn1buf_insert_octet(buf,0); /* of the proper signed-ness */
    if(retval) return retval;
    length++;
  }else if((val < 0) && ((digit&0x80) != 0x80)){
    retval = asn1buf_insert_octet(buf,0xFF);
    if(retval) return retval;
    length++;
  }

  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_INTEGER,length, &partlen); 
  if(retval) return retval;
  length += partlen;

  *retlen = length;
  return 0;
}

asn1_error_code asn1_encode_unsigned_integer(buf, val, retlen)
     asn1buf * buf;
     const unsigned long val;
     int * retlen;
{
  asn1_error_code retval;
  int length = 0, partlen;
  unsigned long valcopy;
  int digit;
  
  valcopy = val;
  do {
    digit = (int) (valcopy&0xFF);
    retval = asn1buf_insert_octet(buf,(asn1_octet) digit);
    if(retval) return retval;
    length++;
    valcopy = valcopy >> 8;
  } while (valcopy != 0 && valcopy != ~0);

  if(digit&0x80) {		          /* make sure the high bit is */
    retval = asn1buf_insert_octet(buf,0); /* of the proper signed-ness */
    if(retval) return retval;
    length++;
  }

  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_INTEGER,length, &partlen); 
  if(retval) return retval;
  length += partlen;

  *retlen = length;
  return 0;
}

asn1_error_code asn1_encode_octetstring(buf, len, val, retlen)
     asn1buf * buf;
     const int len;
     const asn1_octet * val;
     int * retlen;
{
  asn1_error_code retval;
  int length;

  retval = asn1buf_insert_octetstring(buf,len,val);
  if(retval) return retval;
  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_OCTETSTRING,len,&length);
  if(retval) return retval;

  *retlen = len + length;
  return 0;
}

asn1_error_code asn1_encode_charstring(buf, len, val, retlen)
     asn1buf * buf;
     const int len;
     const char * val;
     int * retlen;
{
  asn1_error_code retval;
  int length;

  retval = asn1buf_insert_charstring(buf,len,val);
  if(retval) return retval;
  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_OCTETSTRING,len,&length);
  if(retval) return retval;

  *retlen = len + length;
  return 0;
}

asn1_error_code asn1_encode_null(buf, retlen)
     asn1buf * buf;
     int * retlen;
{
  asn1_error_code retval;
  
  retval = asn1buf_insert_octet(buf,0x00);
  if(retval) return retval;
  retval = asn1buf_insert_octet(buf,0x05);
  if(retval) return retval;

  *retlen = 2;
  return 0;
}

asn1_error_code asn1_encode_printablestring(buf, len, val, retlen)
     asn1buf * buf;
     const int len;
     const char * val;
     int * retlen;
{
  asn1_error_code retval;
  int length;

  retval = asn1buf_insert_charstring(buf,len,val);
  if(retval) return retval;
  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_PRINTABLESTRING,len,			 &length);
  if(retval) return retval;

  *retlen = len + length;
  return 0;
}

asn1_error_code asn1_encode_ia5string(buf, len, val, retlen)
     asn1buf * buf;
     const int len;
     const char * val;
     int * retlen;
{
  asn1_error_code retval;
  int length;

  retval = asn1buf_insert_charstring(buf,len,val);
  if(retval) return retval;
  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_IA5STRING,len,			 &length);
  if(retval) return retval;

  *retlen = len + length;
  return 0;
}

#ifdef _MACINTOSH
#define EPOCH ((66 * 365 * 24 * 60 * 60) + (17 *  24 * 60 * 60) + (getTimeZoneOffset() * 60 * 60))
#else
#define EPOCH (0)
#endif


static const char int_to_02d[100][2] = {
  {'0','0'}, {'0','1'}, {'0','2'}, {'0','3'}, {'0','4'}, {'0','5'}, {'0','6'}, {'0','7'}, {'0','8'}, {'0','9'},
  {'1','0'}, {'1','1'}, {'1','2'}, {'1','3'}, {'1','4'}, {'1','5'}, {'1','6'}, {'1','7'}, {'1','8'}, {'1','9'},
  {'2','0'}, {'2','1'}, {'2','2'}, {'2','3'}, {'2','4'}, {'2','5'}, {'2','6'}, {'2','7'}, {'2','8'}, {'2','9'},
  {'3','0'}, {'3','1'}, {'3','2'}, {'3','3'}, {'3','4'}, {'3','5'}, {'3','6'}, {'3','7'}, {'3','8'}, {'3','9'},
  {'4','0'}, {'4','1'}, {'4','2'}, {'4','3'}, {'4','4'}, {'4','5'}, {'4','6'}, {'4','7'}, {'4','8'}, {'4','9'},
  {'5','0'}, {'5','1'}, {'5','2'}, {'5','3'}, {'5','4'}, {'5','5'}, {'5','6'}, {'5','7'}, {'5','8'}, {'5','9'},
  {'6','0'}, {'6','1'}, {'6','2'}, {'6','3'}, {'6','4'}, {'6','5'}, {'6','6'}, {'6','7'}, {'6','8'}, {'6','9'},
  {'7','0'}, {'7','1'}, {'7','2'}, {'7','3'}, {'7','4'}, {'7','5'}, {'7','6'}, {'7','7'}, {'7','8'}, {'7','9'},
  {'8','0'}, {'8','1'}, {'8','2'}, {'8','3'}, {'8','4'}, {'8','5'}, {'8','6'}, {'8','7'}, {'8','8'}, {'8','9'},
  {'9','0'}, {'9','1'}, {'9','2'}, {'9','3'}, {'9','4'}, {'9','5'}, {'9','6'}, {'9','7'}, {'9','8'}, {'9','9'},
};

#define fmt2d(buf,off,val) {\
   buf[off]   = int_to_02d[val][0];  \
   buf[off+1] = int_to_02d[val][1]; }

#define ok2d(val) (((val) >= 0) && ((val) <= 99))

#define CACHE_TIME_STRING

#ifdef CACHE_TIME_STRING
/* This code is not thread-safe.  Don't enable it for production use
   until we've moved the cache into some per-context location.  */

/* The minimum cache size appears to be about 3.  We seem to get
   passed a time_t of 0 pretty often, so cache that as well.  Caching
   it separately happens to make the check for uninitialized cache
   entries simple.  */
#define CACHE_SIZE 4
struct timecache {
  time_t t;
  int lru;
  char s[16];
};
static struct timecache timecache[CACHE_SIZE];
/* this will get filled in on first use */
static struct timecache zerotime = { -1 };
static int count;

static struct timecache *check_timecache (val)
     time_t val;
{
  struct timecache *oldest = 0;
  int i;
  if (val == 0)
    return &zerotime;
  for (i = 0; i < CACHE_SIZE; i++) {
    if (val == timecache[i].t) {
      oldest = &timecache[i];
      break;
    }
    if (oldest == NULL || oldest->lru > timecache[i].lru)
      oldest = &timecache[i];
  }
  oldest->lru = ++count;
  if (count < 0) {
    int delta = count - CACHE_SIZE - 1;
    for (i = 0; i < CACHE_SIZE; i++)
      timecache[i].lru -= delta;
    count -= delta;
    if (count < 0)
      abort ();
  }
  return oldest;
}
#endif

asn1_error_code asn1_encode_generaltime(buf, val, retlen)
     asn1buf * buf;
     const time_t val;
     int * retlen;
{
  asn1_error_code retval;
  struct tm *gtime;
  char sbuf[16];
  char *s = sbuf;
  int length, sum=0;
  time_t gmt_time;
  int tmp2d;

#ifdef CACHE_TIME_STRING
  struct timecache *cache;

  cache = check_timecache (val);
  s = cache->s;
  if (cache->t == val) {
    goto do_it;
  } else {
    cache->t = val;
  }
#endif

  gmt_time = val + EPOCH;
  gtime = gmtime(&gmt_time);

#if 0
  /* Time encoding: YYYYMMDDhhmmssZ */
  sprintf(s, "%04d%02d%02d%02d%02d%02dZ",
	  1900+gtime->tm_year, gtime->tm_mon+1, gtime->tm_mday,
	  gtime->tm_hour, gtime->tm_min, gtime->tm_sec);
#else

/* could return ASN1_MISSING_FIELD instead, but it shouldn't actually happen */
#define safe2d(offset, val) {\
  tmp2d = val; \
  if(ok2d(tmp2d)) { fmt2d(s, offset, tmp2d); } else return ASN1_OVERFLOW; }


  /* We're trying to avoid division/remainder here, since on the SPARC
     they turn into function calls, and this function is called a lot.  */
  if (gtime->tm_year < 0 || gtime->tm_year > 199) {
    safe2d(0, gtime->tm_year/100 + 19);
    safe2d(2, gtime->tm_year % 100);
  } else if (gtime->tm_year < 100) {
    s[0] = '1', s[1] = '9';
    safe2d(2, gtime->tm_year);
  } else {
    s[0] = '2', s[1] = '0';
    safe2d(2, gtime->tm_year - 100);
  }
  /* the others are easy */
  safe2d(4, gtime->tm_mon+1);
  safe2d(6, gtime->tm_mday);
  safe2d(8, gtime->tm_hour);
  safe2d(10, gtime->tm_min);
  safe2d(12, gtime->tm_sec);
  s[14] = 'Z';
  s[15] = 0;

#endif

#ifdef CACHE_TIME_STRING
do_it:
#endif

  retval = asn1buf_insert_charstring(buf,15,s);
  if(retval) return retval;
  sum = 15;

  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_GENERALTIME,sum,&length);
  if(retval) return retval;
  sum += length;

  *retlen = sum;
  return 0;
}

asn1_error_code asn1_encode_generalstring(buf, len, val, retlen)
     asn1buf * buf;
     const int len;
     const char * val;
     int * retlen;
{
  asn1_error_code retval;
  int length;

  retval = asn1buf_insert_charstring(buf,len,val);
  if(retval) return retval;
  retval = asn1_make_tag(buf,UNIVERSAL,PRIMITIVE,ASN1_GENERALSTRING,len,
			 &length);
  if(retval) return retval;

  *retlen = len + length;
  return 0;
}
