#include <windows.h>
#include <stdio.h>
#include <winwlx.h>

#include "krb5.h"
#include "com_err.h"

#include "kerbgina.h"

/*
 * Location of the real msgina.
 */

#define REALGINA_PATH   TEXT("MSGINA.DLL")


/*
 * winlogon function dispatch table
 */

PWLX_DISPATCH_VERSION_1_0 g_pWinlogon;

/*
 * Functions pointers to the real msgina which we will call.
 */

PGWLXNEGOTIATE GWlxNegotiate;
PGWLXINITIALIZE GWlxInitialize;
PGWLXDISPLAYSASNOTICE GWlxDisplaySASNotice;
PGWLXLOGGEDOUTSAS GWlxLoggedOutSAS;
PGWLXACTIVATEUSERSHELL GWlxActivateUserShell;
PGWLXLOGGEDONSAS GWlxLoggedOnSAS;
PGWLXDISPLAYLOCKEDNOTICE GWlxDisplayLockedNotice;
PGWLXWKSTALOCKEDSAS GWlxWkstaLockedSAS;
PGWLXISLOCKOK GWlxIsLockOk;
PGWLXISLOGOFFOK GWlxIsLogoffOk;
PGWLXLOGOFF GWlxLogoff;
PGWLXSHUTDOWN GWlxShutdown;

/*
 * NEW for version 1.1
 */

PGWLXSTARTAPPLICATION GWlxStartApplication;
PGWLXSCREENSAVERNOTIFY GWlxScreenSaverNotify;

static HANDLE hGlobalWlx;     /* Handle to tell winlogon who's calling */
static PWLX_DISPATCH_VERSION_1_0   pWlxFuncs;      /* Ptr to table of functions */

static BOOL global_do_debugprints = TRUE;

typedef struct _kerb_info {
	WCHAR *userName;
	WCHAR *userDomain;
	WCHAR *userPassword;
	WCHAR *oldUserPassword;
	BOOL excluded_user;
	HANDLE hUserToken;
	DWORD require_kerberos_logon;
	WCHAR *ccache_file;
	WCHAR *extra_env_vars;
	WCHAR *ccache_dir;
	WCHAR *ccachename;
	WCHAR *kerbnet_home;
	WCHAR *user_exclude_list;
} kerb_info;

typedef enum _errortype {
	GINA_ERROR_MESSAGE, GINA_INFO_MESSAGE, GINA_WARNING_MESSAGE
} errortype;

static kerb_info global_kinfo;
static BOOL global_kinit_succeeded = FALSE;
static BOOL global_done_wsastartup = FALSE;

void destroyKerbInfo(kerb_info *kinfo);
BOOL doKinit(kerb_info *,PVOID,PWSTR);

#define KERBNET_MAJOR_VERSION L"1"
#define DEFAULT_EXCLUDED_USERS L"*\\Administrator"
#define KERBNET_HOME L"KERBNET_HOME="
#define KRB5_SUBPATH L"etc\\krb5.conf"
#define KRB5_CONFIG L"KRB5_CONFIG="
#define KRB5CCNAME L"KRB5CCNAME="
#define REALMMAP_NAME L"REALMS"
#define OUTPUT_BUFFER_SIZE 1024
#define CYG_KERBBASE_REG_NAME L"SOFTWARE\\Cygnus Solutions\\Kerbnet\\"
#define CYG_GINA_REG_NAME L"\\Gina"
#define CEDENTIALS_CACHE_REG_NAME L"CredentialsCacheDir"
#define REQUIRE_KERBEROS_LOGON_REG_NAME L"RequireKerberosLogon"
#define EXCLUDED_USERS_REG_NAME L"ExcludedUsers"
#define KERBNET_HOME_REG_NAME L"KERBNET_HOME"  
#define DEFAULT_PREFIX L"FILE:"

#if defined(_DEBUG)

#define debug_print(x) do_debug_print((x))
#define DEBUG_FILE_NAME L"C:\\TMP\\GINADEBUG.OUT"
/*
 * Debug print to file function.
 */
static void do_debug_print(char *message)
{
	static HANDLE fh = INVALID_HANDLE_VALUE;
	DWORD written;

	if(global_do_debugprints == FALSE)
		return;

	if(fh == INVALID_HANDLE_VALUE) {
		fh = CreateFileW(DEBUG_FILE_NAME, 
						GENERIC_WRITE,
						FILE_SHARE_DELETE|FILE_SHARE_READ|FILE_SHARE_WRITE,
						0,
						CREATE_ALWAYS,
						FILE_ATTRIBUTE_NORMAL,
						0);
	}
	WriteFile(fh, message, lstrlenA(message), &written,0);
	FlushFileBuffers(fh);
}

/* Debug wrapper for LocalAlloc/LocalFree */
static void *MyLocalAlloc(DWORD flags, DWORD size)
{
	char buffer[200];
	unsigned char *ret;
	DWORD newsize = 4+(size * 2);

	ret = LocalAlloc(flags, newsize);
	if(ret != 0) {
		*((DWORD *)ret) = size;
		ret += 4;
		FillMemory(ret, newsize - 4, 0xCC);
	}
	
	sprintf(buffer, "LocalAlloc, size = %d, actual size = %d, return = %x\n", size, 
		LocalSize(ret-4), ret);
	do_debug_print(buffer);
	return ret;
}

static void MyLocalFree(HLOCAL val)
{
	char buffer[200];
	DWORD i;
	DWORD size; 
	unsigned char *orig_val = (unsigned char *)val;
	BOOL overrun = FALSE;
	val = (((unsigned char *)val) - 4);

	size = *((DWORD *)val);

	for(i = size; i < (size*2); ++i)
		if(orig_val[i] != 0xCC) {
			overrun = TRUE;
			break;
		}

	sprintf(buffer, 
		"LocalFree, size = %d, actual size = %d, val = %x, overrun = %d at %d (newval = %d)\n", 
		size, LocalSize(val), orig_val, overrun, i, orig_val[i]);

	do_debug_print(buffer);
	LocalFree(val);
}

#if 0
#define LocalAlloc MyLocalAlloc
#define LocalFree MyLocalFree
#endif

#else /* _DEBUG */

#define debug_print(x)

#endif /* defined(_DEBUG) */

/*
 * Utility function to convert a unicode string to an ascii string.
 */

static BOOL ascii_string( const WCHAR *uni_str, char **a_str)
{
	/* First Calculate the space needed. */
	DWORD size;

	size = WideCharToMultiByte(CP_ACP, 0, uni_str, -1, 0, 0, 0, 0);
	*a_str = (char *)LocalAlloc(LMEM_FIXED, size + 1);
	if(*a_str == 0) {
		return TRUE;
	}
	WideCharToMultiByte(CP_ACP, 0, uni_str, -1, *a_str, size, 0, 0);
	return FALSE;
}

/* 
 * Utility function to copy the kerberos info.
 */

static BOOL copyKerbInfo(kerb_info *kinfo, const HANDLE htoken, 
				   const WCHAR *userName, const WCHAR *userDomain,
				   const WCHAR *userPassword, const WCHAR *oldUserPassword)
{
	if(userName != 0) {
		kinfo->userName = (WCHAR *)LocalAlloc(LMEM_FIXED, (lstrlenW(userName) + 1)*sizeof(WCHAR));
		if(kinfo->userName == 0)
			return TRUE;
		lstrcpyW(kinfo->userName, userName);
	} else 
		kinfo->userName = 0;
	if(userDomain != 0) {
		kinfo->userDomain = (WCHAR *)LocalAlloc(LMEM_FIXED, (lstrlenW(userDomain) + 1)*sizeof(WCHAR));
		if(kinfo->userDomain == 0) {
			destroyKerbInfo(kinfo);
			return TRUE;
		}
		lstrcpyW(kinfo->userDomain, userDomain);
	} else
		kinfo->userDomain = 0;
	if(userPassword != 0) {
		kinfo->userPassword = (WCHAR *)LocalAlloc(LMEM_FIXED, (lstrlenW(userPassword) + 1)*sizeof(WCHAR));
		if(kinfo->userPassword == 0) {
			destroyKerbInfo(kinfo);
			return TRUE;
		}
		lstrcpyW(kinfo->userPassword, userPassword);
	} else
		kinfo->userPassword = 0;
	if(oldUserPassword != 0) {
		kinfo->oldUserPassword = (WCHAR *)LocalAlloc(LMEM_FIXED, (lstrlenW(oldUserPassword) + 1)*sizeof(WCHAR));
		if(kinfo->oldUserPassword == 0) {
			destroyKerbInfo(kinfo);
			return TRUE;
		}
		lstrcpyW(kinfo->oldUserPassword, oldUserPassword);
	} else
		kinfo->oldUserPassword = 0;

	/* Set up the globals */
	kinfo->ccache_file = 0;
	kinfo->excluded_user = FALSE;
	kinfo->hUserToken = htoken;
	kinfo->require_kerberos_logon = FALSE;
	kinfo->extra_env_vars = 0;
	kinfo->ccache_dir = 0;
	kinfo->ccachename = 0;
	kinfo->kerbnet_home = 0;
	kinfo->user_exclude_list = 0;

	return FALSE;
}

/*
 * Utility function to destroy the kerberos info.
 */

void destroyKerbInfo(kerb_info *kinfo)
{
	if(kinfo->userName != 0) {
		LocalFree((HLOCAL)kinfo->userName);
		kinfo->userName = 0;
	}
	if(kinfo->userDomain != 0) {
		LocalFree((HLOCAL)kinfo->userDomain);
		kinfo->userDomain = 0;
	}
	if(kinfo->ccache_file != 0) {
		DeleteFileW(kinfo->ccache_file);
		LocalFree((HLOCAL)kinfo->ccache_file);
		kinfo->ccache_file = 0;
	}
	if(kinfo->userPassword != 0) {
		ZeroMemory(kinfo->userPassword, lstrlenW(kinfo->userPassword)*sizeof(WCHAR));
		LocalFree((HLOCAL)kinfo->userPassword);
		kinfo->userPassword = 0;
	}
	if(kinfo->oldUserPassword != 0) {
		ZeroMemory(kinfo->oldUserPassword, lstrlenW(kinfo->oldUserPassword)*sizeof(WCHAR));
		LocalFree((HLOCAL)kinfo->oldUserPassword);
		kinfo->oldUserPassword = 0;
	}
	kinfo->excluded_user = FALSE;
	/* 
	 * We don't delete the token handle here as the GINA code
	 * that created it and also WinLogon.exe take care of this.
	 */
	kinfo->require_kerberos_logon = FALSE;
	if(kinfo->extra_env_vars != 0) {
		LocalFree((HLOCAL)kinfo->extra_env_vars);
		kinfo->extra_env_vars = 0;
	}
	if(kinfo->ccache_dir != 0) {
		LocalFree((HLOCAL)kinfo->ccache_dir);
		kinfo->ccache_dir = 0;
	}
	if(kinfo->ccachename != 0) {
		LocalFree((HLOCAL)kinfo->ccachename);
		kinfo->ccachename = 0;
	}
	if(kinfo->kerbnet_home != 0) {
		LocalFree((HLOCAL)kinfo->kerbnet_home);
		kinfo->kerbnet_home = 0;
	}
	if(kinfo->user_exclude_list != 0) {
		LocalFree((HLOCAL)kinfo->user_exclude_list);
		kinfo->user_exclude_list = 0;
	}
}

/*
 * hook into the real GINA.
 */

BOOL
MyInitialize( void )
{
    HINSTANCE hDll;

    /*
     * Load MSGINA.DLL.
     */
    if( !(hDll = LoadLibrary( REALGINA_PATH )) ) {
        return FALSE;
    }

    /*
     * Get pointers to all of the WLX functions in the real MSGINA.
     */
    GWlxNegotiate = (PGWLXNEGOTIATE)GetProcAddress( hDll, "WlxNegotiate" );
    if( !GWlxNegotiate ) {
        return FALSE;
    }

    GWlxInitialize = (PGWLXINITIALIZE)GetProcAddress( hDll, "WlxInitialize" );
    if( !GWlxInitialize ) {
        return FALSE;
    }

    GWlxDisplaySASNotice =
        (PGWLXDISPLAYSASNOTICE)GetProcAddress( hDll, "WlxDisplaySASNotice" );
    if( !GWlxDisplaySASNotice ) {
        return FALSE;
    }

    GWlxLoggedOutSAS =
        (PGWLXLOGGEDOUTSAS)GetProcAddress( hDll, "WlxLoggedOutSAS" );
    if( !GWlxLoggedOutSAS ) {
        return FALSE;
    }

    GWlxActivateUserShell =
        (PGWLXACTIVATEUSERSHELL)GetProcAddress( hDll, "WlxActivateUserShell" );
    if( !GWlxActivateUserShell ) {
        return FALSE;
    }

    GWlxLoggedOnSAS =
        (PGWLXLOGGEDONSAS)GetProcAddress( hDll, "WlxLoggedOnSAS" );
    if( !GWlxLoggedOnSAS ) {
        return FALSE;
    }

    GWlxDisplayLockedNotice =
        (PGWLXDISPLAYLOCKEDNOTICE)GetProcAddress(
                                        hDll,
                                        "WlxDisplayLockedNotice" );
    if( !GWlxDisplayLockedNotice ) {
        return FALSE;
    }

    GWlxIsLockOk = (PGWLXISLOCKOK)GetProcAddress( hDll, "WlxIsLockOk" );
    if( !GWlxIsLockOk ) {
        return FALSE;
    }

    GWlxWkstaLockedSAS =
        (PGWLXWKSTALOCKEDSAS)GetProcAddress( hDll, "WlxWkstaLockedSAS" );
    if( !GWlxWkstaLockedSAS ) {
        return FALSE;
    }

    GWlxIsLogoffOk = (PGWLXISLOGOFFOK)GetProcAddress( hDll, "WlxIsLogoffOk" );
    if( !GWlxIsLogoffOk ) {
        return FALSE;
    }

    GWlxLogoff = (PGWLXLOGOFF)GetProcAddress( hDll, "WlxLogoff" );
    if( !GWlxLogoff ) {
        return FALSE;
    }

    GWlxShutdown = (PGWLXSHUTDOWN)GetProcAddress( hDll, "WlxShutdown" );
    if( !GWlxShutdown ) {
        return FALSE;
    }

    /*
     * we don't check for failure here because these don't exist for
     * gina's implemented prior to Windows NT 4.0
     */

    GWlxStartApplication = (PGWLXSTARTAPPLICATION) GetProcAddress( hDll, "WlxStartApplication" );
    GWlxScreenSaverNotify = (PGWLXSCREENSAVERNOTIFY) GetProcAddress( hDll, "WlxScreenSaverNotify" );

    /*
     * Everything loaded ok.  Return success.
     */
    return TRUE;
}


BOOL
WINAPI
WlxNegotiate(
    DWORD       dwWinlogonVersion,
    DWORD       *pdwDllVersion)
{
    if( !MyInitialize() )
        return FALSE;

    return GWlxNegotiate( dwWinlogonVersion, pdwDllVersion );
}


BOOL
WINAPI
WlxInitialize(
    LPWSTR      lpWinsta,
    HANDLE      hWlx,
    PVOID       pvReserved,
    PVOID       pWinlogonFunctions,
    PVOID       *pWlxContext)
{

    /* Save the handle for future use. */
    hGlobalWlx = hWlx;
    pWlxFuncs = (PWLX_DISPATCH_VERSION_1_0) pWinlogonFunctions;

    return GWlxInitialize(
                lpWinsta,
                hWlx,
                pvReserved,
                pWinlogonFunctions,
                pWlxContext
                );
}


VOID
WINAPI
WlxDisplaySASNotice(
    PVOID   pWlxContext)
{
    GWlxDisplaySASNotice( pWlxContext );
}

VOID doErrorMessage(const errortype type, const WCHAR *message, DWORD lastError, int krb_error)
{
	WCHAR *newmsg = (WCHAR *)message;
	WCHAR *title;
	DWORD icon;

	/* Add a text description of the error. */

	if(lastError != 0) {
		WCHAR *lpMsgBuf;
		DWORD newlen = lstrlenW(message) + 1;

		if(FormatMessage( 
			FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
			NULL,
			lastError,
			MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* Default language */
			(WCHAR *)&lpMsgBuf,
			0,
			NULL 
		) == 0)
			return;

		/* Concatenate the string. */
		newmsg = (WCHAR *)LocalAlloc(LMEM_FIXED, (newlen + lstrlenW(lpMsgBuf) + 1)*sizeof(WCHAR));
		if(newmsg == 0) {
			LocalFree((HLOCAL)lpMsgBuf );
			return;
		}

		lstrcpyW(newmsg, message);
		lstrcatW(newmsg, lpMsgBuf);

		/* Free the buffer. */
		LocalFree( (HLOCAL)lpMsgBuf );
	} else if( krb_error != 0) {
		WCHAR *lpMsgBuf;
		const char *err_str = error_message(krb_error);
		DWORD newlen = lstrlenW(message) + 1;
		DWORD size;

		size = MultiByteToWideChar( CP_ACP, 0, err_str, -1, 0, 0);
		newmsg = (WCHAR *)LocalAlloc(LMEM_FIXED, (lstrlenW(message) + size + 1)*sizeof(WCHAR));
		if(newmsg == 0) {
			return;
		}

		lstrcpyW(newmsg, message);
		lpMsgBuf = &newmsg[lstrlenW(message)];
		MultiByteToWideChar( CP_ACP, 0, err_str, -1, lpMsgBuf, size+1);
	}

	switch((int)type) {
	case GINA_ERROR_MESSAGE:
		title = L"Cygnus Kerbnet Gina Error";
		icon = MB_ICONSTOP;
		break;
	case GINA_INFO_MESSAGE:
		title = L"Cygnus Kerbnet Gina Information";
		icon = MB_ICONINFORMATION;
		break;
	case GINA_WARNING_MESSAGE:
		title = L"Cygnus Kerbnet Gina Warning";
		icon = MB_ICONEXCLAMATION;
		break;
	}

	pWlxFuncs->WlxMessageBox(hGlobalWlx, 0, 
				newmsg,
				title,
				MB_SYSTEMMODAL|MB_SETFOREGROUND|MB_SERVICE_NOTIFICATION|	
				icon|MB_OK);

	if(newmsg != message)
		LocalFree((HLOCAL)newmsg);
}

/*
 * Function called when system in logout state and someone
 * typed CTRL-ALT-DEL
 */

int
WINAPI
WlxLoggedOutSAS(
    PVOID           pWlxContext,
    DWORD           dwSasType,
    PLUID           pAuthenticationId,
    PSID            pLogonSid,
    PDWORD          pdwOptions,
    PHANDLE         phToken,
    PWLX_MPR_NOTIFY_INFO    pMprNotifyInfo,
    PVOID           *pProfile)
{
    int iRet;
	DWORD ret;
	BOOL was_excluded_user = TRUE;
	BOOL require_kerberos_logon = FALSE;

	debug_print("WlxLoggedOutSAS 1\n");

    iRet = GWlxLoggedOutSAS(
                pWlxContext,
                dwSasType,
                pAuthenticationId,
                pLogonSid,
                pdwOptions,
                phToken,
                pMprNotifyInfo,
                pProfile
                );

    if(iRet == WLX_SAS_ACTION_LOGON) {
		__try {

			debug_print("WlxLoggedOutSAS 2\n");

			/*
			 * copy token and pMprNotifyInfo for later use
			 */
			if(copyKerbInfo(&global_kinfo,
							*phToken,
							pMprNotifyInfo->pszUserName,
							pMprNotifyInfo->pszDomain,
							pMprNotifyInfo->pszPassword,
							0) == TRUE)	{
				doErrorMessage(GINA_ERROR_MESSAGE,
						L"Kerberos Gina: Failed to allocate memory for Kerberos structure.",
						ERROR_NOT_ENOUGH_MEMORY, 0); 
				return WLX_SAS_ACTION_NONE;	/* Logoff immediately */
			}
			debug_print("WlxLoggedOutSAS 3\n");
			
			global_kinit_succeeded = FALSE;
			global_kinfo.excluded_user = FALSE;

			if(global_done_wsastartup == FALSE) {
				WSADATA wsadata;
				DWORD err;
				if((err = WSAStartup(MAKEWORD(1,1), &wsadata)) == 0 ) {
					global_done_wsastartup = TRUE;
				} else {
					doErrorMessage(GINA_ERROR_MESSAGE, L"Kerberos Gina: WSAStartup failed.",
						err, 0);
				}
			}

			ret = doKinit(&global_kinfo, pWlxContext,0);

			was_excluded_user = global_kinfo.excluded_user;
			require_kerberos_logon = global_kinfo.require_kerberos_logon;

			debug_print("WlxLoggedOutSAS 4\n");

			if(ret == FALSE	) {
				global_kinit_succeeded = TRUE;
				/*
				 * Destroy just the password now, we have finished with it.
				 */
				if(global_kinfo.userPassword != 0) {
					ZeroMemory(global_kinfo.userPassword, 
							   lstrlenW(global_kinfo.userPassword)*sizeof(WCHAR));
					LocalFree((HLOCAL)global_kinfo.userPassword);
					global_kinfo.userPassword = 0;
				}
			} else {
				WCHAR *save_extra_env_vars;
				/*
				 * Save the extra environment variables - the user may
				 * use these to do a manual kinit even though the auto kinit
				 * failed.
				 */
				save_extra_env_vars = global_kinfo.extra_env_vars;
				global_kinfo.extra_env_vars = 0;
				/*
				 * We can destroy all the saved info now, we didn't get
				 * a ticket.
				 */
				destroyKerbInfo(&global_kinfo);
				/* Restore extra_env_vars. */
				global_kinfo.extra_env_vars = save_extra_env_vars;
			}

			debug_print("WlxLoggedOutSAS 5\n");

			if((require_kerberos_logon == TRUE) && (was_excluded_user == FALSE) && (ret == TRUE))
				return WLX_SAS_ACTION_NONE; /* Logoff immediately */

		} __except ( TRUE ) {
			doErrorMessage(GINA_ERROR_MESSAGE, L"Kerberos Gina: GPF(1).",0, 0);
		}

    }

    return iRet;
}

/*
 * Function to return the KERBNET_HOME key. As a side effect
 * sets global_require_kerberos_logon if it is set in the registry.
 * WCHAR * returned must be de-allocated with LocalFree().
 */

static BOOL getKerbNetHome(const WCHAR *kerbnet_major_version, 
						   HKEY *gkey,
						   kerb_info *kinfo)
{
	HKEY hkey = (HKEY)INVALID_HANDLE_VALUE;
	HKEY gina_key = (HKEY)INVALID_HANDLE_VALUE;
	WCHAR *subkey_name = 0;
	WCHAR *ret = 0;
	DWORD subkey_len;
	DWORD type, datasize = 0;
	DWORD err, str_len, i;


	*gkey = INVALID_HANDLE_VALUE;
	kinfo->kerbnet_home = 0;
	kinfo->ccache_dir = 0;
	kinfo->user_exclude_list = 0;

	/*
	 * Allocate the space needed for the keyname (used for Gina keyname also).
	 */
	subkey_len = lstrlenW(CYG_KERBBASE_REG_NAME) + lstrlenW(kerbnet_major_version) + 
				 lstrlenW(CYG_GINA_REG_NAME) + 1;
	if((subkey_name = (WCHAR *)LocalAlloc(LMEM_FIXED, subkey_len*sizeof(WCHAR))) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Unable to allocate memory for registry key name.", 0, 0);
		goto cleanup;
	}
	lstrcpyW(subkey_name, CYG_KERBBASE_REG_NAME);
	lstrcatW(subkey_name, kerbnet_major_version);

	if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, subkey_name, 0, KEY_READ, &hkey)) !=
				ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to open registry key for configuration.",
			err, 0);
		goto cleanup;
	}

	/*
	 * Now open the gina key.
	 */
	lstrcpyW(subkey_name, CYG_KERBBASE_REG_NAME);
	lstrcatW(subkey_name, kerbnet_major_version);
	lstrcatW(subkey_name, CYG_GINA_REG_NAME);

	if((err = RegOpenKeyEx(HKEY_LOCAL_MACHINE, subkey_name, 0, KEY_READ, &gina_key)) !=
				ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to open Gina registry key for configuration.",
			err, 0);
		goto cleanup;
	}

	/* Try and get the KERBNET_HOME value size. */
	if((err = RegQueryValueEx(hkey, KERBNET_HOME_REG_NAME, 0, &type, 0, &datasize)) != ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to get size for KERBNET_HOME value.", err, 0);
		goto cleanup;
	}

	/* Add one WCHAR for a possibly missing '\' character */
	datasize += sizeof(WCHAR);

	/* Allocate the memory for the value */
	if((kinfo->kerbnet_home = (WCHAR *)LocalAlloc(LMEM_FIXED, datasize)) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate memory for KERBNET_HOME value.", 0, 0);
		goto cleanup;
	}

	/* Now get the value */
	if((err = RegQueryValueEx(hkey, KERBNET_HOME_REG_NAME, 0, &type, (LPBYTE)kinfo->kerbnet_home, 
														&datasize)) != ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to get value for KERBNET_HOME.", err, 0);
		goto cleanup;
	}

	/* Ensure it ends in a '\' */
	if(kinfo->kerbnet_home[lstrlenW(kinfo->kerbnet_home)-1] != L'\\')
		lstrcatW(kinfo->kerbnet_home, L"\\");

	datasize = 0;

	/* Try and get the CredentialsCacheDir value size. */
	if((err = RegQueryValueEx(hkey, CEDENTIALS_CACHE_REG_NAME, 0, &type, 0, &datasize)) != ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to get size for CredentialsCacheDir value.", err, 0);
		goto cleanup;
	}

	/* Add one WCHAR for a possibly missing '\' character */
	datasize += sizeof(WCHAR);
	/* Add 20 byte padding. We may want to add FILE:, MEM: or REGISTRY: prefixes here. */
	datasize += (20*sizeof(WCHAR));

	/* Allocate the memory for the value */
	if((kinfo->ccache_dir = (WCHAR *)LocalAlloc(LMEM_FIXED, datasize)) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate memory for CredentialsCacheDir value.", 0, 0);
		goto cleanup;
	}

	/* Now get the value */
	if((err = RegQueryValueEx(hkey, CEDENTIALS_CACHE_REG_NAME, 0, &type, 
						(LPBYTE)kinfo->ccache_dir, &datasize)) != ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to get value for CredentialsCacheDir.", err, 0);
		goto cleanup;
	}

	/* Ensure it ends in a '\' */
	if(kinfo->ccache_dir[lstrlenW(kinfo->ccache_dir)-1] != L'\\')
		lstrcatW(kinfo->ccache_dir, L"\\");

	/* Check that the value has a valid prefix. Add the default prefix if not. */
	str_len = lstrlenW(	kinfo->ccache_dir );
	for(i = 0; i < str_len; i++ ) {
		if(	kinfo->ccache_dir[i] == L':' )
			break;
	}
	if( i == 1) {
		/* Filename like C:. Add FILE: */
		MoveMemory( &kinfo->ccache_dir[6], kinfo->ccache_dir, (str_len+1)*sizeof(WCHAR));
		lstrcpy( kinfo->ccache_dir, L"FILE:");
		MoveMemory( &kinfo->ccache_dir[5], &kinfo->ccache_dir[6], (str_len+1)*sizeof(WCHAR)); 
	} else if( i == (DWORD)lstrlenW( kinfo->ccache_dir )) {
		/* No ':'  add the default prefix. */
		DWORD def_prefix_len = lstrlenW(DEFAULT_PREFIX);
		MoveMemory( &kinfo->ccache_dir[def_prefix_len+1], kinfo->ccache_dir, 
					(str_len+1)*sizeof(WCHAR));
		lstrcpy( kinfo->ccache_dir, DEFAULT_PREFIX);
		MoveMemory( &kinfo->ccache_dir[def_prefix_len], 
					&kinfo->ccache_dir[def_prefix_len+1], (str_len+1)*sizeof(WCHAR)); 
	}
		
	/* Get the RequireKerberosLogon value - FALSE if it doesn't exist. */
	datasize = sizeof(DWORD);
	if((err = RegQueryValueEx(gina_key, REQUIRE_KERBEROS_LOGON_REG_NAME, 0, &type, 
				(LPBYTE)&kinfo->require_kerberos_logon, &datasize)) != ERROR_SUCCESS) {
		kinfo->require_kerberos_logon = FALSE;
	}

	/* Get the excluded user list - one entry, '*\Administrator' if it doesn't exist. */
	datasize = 0;

	/* Try and get the ExcludedUsers value size. */
	if((err = RegQueryValueEx(gina_key, EXCLUDED_USERS_REG_NAME, 0, &type, 0, &datasize)) != ERROR_SUCCESS) {
		datasize = (lstrlenW(DEFAULT_EXCLUDED_USERS) + 2) * sizeof(WCHAR);
	}

	/* Allocate the memory for the value */
	if((kinfo->user_exclude_list = (WCHAR *)LocalAlloc(LMEM_FIXED, datasize)) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate memory for ExcludedUsers value.", 0, 0);
		goto cleanup;
	}

	if((err = RegQueryValueEx(hkey, EXCLUDED_USERS_REG_NAME, 0, &type, 
						(LPBYTE)kinfo->user_exclude_list, &datasize)) != ERROR_SUCCESS) {
		lstrcpyW(kinfo->user_exclude_list, DEFAULT_EXCLUDED_USERS);
		/* Add an extra null termination */
		(kinfo->user_exclude_list)[lstrlenW(kinfo->user_exclude_list)+1] = L'\0';
	}

	LocalFree((HLOCAL)subkey_name);
	RegCloseKey(hkey);
	*gkey = gina_key;
	return FALSE;

cleanup:

	if(subkey_name != 0)
		LocalFree((HLOCAL)subkey_name);
	if(kinfo->kerbnet_home != 0) {
		LocalFree((HLOCAL)kinfo->kerbnet_home);
		kinfo->kerbnet_home = 0;
	}
	if(kinfo->ccache_dir != 0) {
		LocalFree((HLOCAL)kinfo->ccache_dir);
		kinfo->ccache_dir = 0;
	}
	if(kinfo->user_exclude_list != 0) {
		LocalFree((HLOCAL)kinfo->user_exclude_list);
		kinfo->user_exclude_list = 0;
	}
	if(hkey != (HKEY)INVALID_HANDLE_VALUE)
		RegCloseKey(hkey);
	if(gina_key != (HKEY)INVALID_HANDLE_VALUE)
		RegCloseKey(gina_key);
	return TRUE;
}

/*
 * Function to determine the environment variables
 * needed for the user shell.
 * This will include the variables
 * KERBNET_HOME=<kerbnet_home>,
 * KRB5_CONFIG=%KERBNET_HOME%\etc\krb5.conf.
 * and KRB5CCNAME=FILE:%ccahcedir%\DOMAIN.user
 */

static WCHAR *setupNewEnv(kerb_info *kinfo)
{
	DWORD extraSpace;
	DWORD ccache_space;
	WCHAR *p = 0;
	WCHAR *newEnv = 0;
	WCHAR computername[MAX_COMPUTERNAME_LENGTH+1];
	DWORD comp_name_size = sizeof(computername)/sizeof(WCHAR);

	/*
	 * Get the computername.
	 */
	if(GetComputerName(computername, &comp_name_size) == 0) {
	  doErrorMessage(GINA_ERROR_MESSAGE,
			 L"Kerberos Gina: Failed to get computer name.",GetLastError(), 0);
	  return 0;
	}

	/*
	 * Calculate the size of KERBNET_HOME=<kerbnet_home>,
	 * KRB5_CONFIG=%KERBNET_HOME%\etc\krb5.conf.
	 * and KRB5CCNAME=FILE:%ccahcedir%\DOMAIN.user
	 */

	/*
	 * NB. Kerbnet_home, ccache_dir, should be filtered for '=' characters.
	 */

	extraSpace = /* KERBNET_HOME=<kerbnet_home> */
				 lstrlenW(KERBNET_HOME) + lstrlenW(kinfo->kerbnet_home) + 1 +
				 /* KRB5_CONFIG=%KERBNET_HOME%\etc\krb5.conf */ 
				 lstrlenW(KRB5_CONFIG) + lstrlenW(kinfo->kerbnet_home) +
				 lstrlenW(KRB5_SUBPATH) + 1 +
				 /* KRB5CCNAME=%ccachdir%\MACHINE.user */
				 lstrlenW(KRB5CCNAME) + lstrlenW(kinfo->ccache_dir) +
				 lstrlenW(computername) + 1 +
				 lstrlenW(kinfo->userName) + 1 +
				 1; /* For extra L'\0' */

#if 0
	/* DEBUG !! */
	/* Add size for strace=1,/tmp/kinit.out */
	extraSpace += 30;
	/* END DEBUG */
#endif
	
	if((newEnv = (WCHAR *)LocalAlloc(LMEM_FIXED, 
					 extraSpace * sizeof(WCHAR))) == 0) {
	  doErrorMessage(GINA_ERROR_MESSAGE,
			 L"Kerberos Gina: Unable to allocate memory for new environment.",0, 0);
	  return 0;
	}

	p = newEnv;

	/* Set KERBNET_HOME=<kerbnet_home>*/
	lstrcpyW(p, KERBNET_HOME);
	lstrcatW(p, kinfo->kerbnet_home);
	p += lstrlenW(p) + 1;
	/* Set KRB5_CONFIG=%KERBNET_HOME%\etc\krb5.conf*/
	lstrcpyW(p, KRB5_CONFIG);
	lstrcatW(p, kinfo->kerbnet_home);
	lstrcatW(p, KRB5_SUBPATH);
	p += lstrlenW(p) + 1;
	/* Set KRB5CCNAME=FILE:%ccahcedir%\MACHINE.user */
	lstrcpyW(p, KRB5CCNAME);
	lstrcatW(p, kinfo->ccache_dir);
	lstrcatW(p, computername);
	lstrcatW(p, L".");
	lstrcatW(p, kinfo->userName);
	p += lstrlenW(p) + 1;

	/* Save the KRB5CCNAME name. */
	ccache_space = lstrlenW(kinfo->ccache_dir) +
				 lstrlenW(computername) + 1 +
				 lstrlenW(kinfo->userName) + 1;
	if((kinfo->ccache_file = (WCHAR *)LocalAlloc(LMEM_FIXED, 
						     ccache_space * sizeof(WCHAR))) == 0) {
	  doErrorMessage(GINA_ERROR_MESSAGE,
			 L"Kerberos Gina: Unable to allocate memory for ccache name.",0, 0);
	  LocalFree((HLOCAL)kinfo->extra_env_vars);
	  kinfo->extra_env_vars = 0;
	  return 0;
	}
	lstrcpyW(kinfo->ccache_file, kinfo->ccache_dir);
	lstrcatW(kinfo->ccache_file, computername);
	lstrcatW(kinfo->ccache_file, L".");
	lstrcatW(kinfo->ccache_file, kinfo->userName);

#if 0
	/* DEBUG !! */
	/* Add strace=1,/tmp/kinit.out */
	lstrcpyW(p, L"strace=1,c:\\tmp\\kinit.out");
	p += lstrlenW(p) + 1;
#endif

	/* Add the extra terminating L'\0' */
	*p = L'\0';

	return newEnv;
}

/*
 * Function to add the environment variables into this process (WinLogon.exe).
 * Needed so the Kerberos shared libraries can find their config files correctly.
 * Return FALSE on fail, TRUE on success.
 */

BOOL setupEnvVars(WCHAR *extra_env_vars)
{
	WCHAR *p;
	WCHAR *pe;

	if(extra_env_vars == 0)
		return TRUE;

	for( p = extra_env_vars; *p; p += (lstrlenW(p) + 1)) {
		for( pe = p; *pe && (*pe != L'='); pe++)
			;
		if(*pe == 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Malformed environment string.",0, 0);
			return FALSE;
		}
		/* Null terminate at the '=' */
		*pe = 0;
		/* Now do the setenv */
		if(SetEnvironmentVariable(p, &pe[1]) == FALSE) {
			*pe = L'='; /* Reset */
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to set environment variable.",GetLastError(), 0);
			return FALSE;
		}
		*pe = L'='; /* Reset */
	}
	return TRUE;
}

/*
 * Function to delete the extra environment variables from this process (WinLogon.exe).
 */


void deleteEnvVars(WCHAR *extra_env_vars)
{
	WCHAR *p;
	WCHAR *pe;

	if(extra_env_vars == 0)
		return;

	for( p = extra_env_vars; *p; p += (lstrlenW(p) + 1)) {
		for( pe = p; *pe && (*pe != L'='); pe++)
			;
		if(*pe == 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Malformed environment string.",0, 0);
			return;
		}
		/* Null terminate at the '=' */
		*pe = 0;
		/* Now do the setenv - we ignore errors as if setEnvVars
                   was never done (eg. for an excluded user) then no 
                   environment variables will have seen set. */
		SetEnvironmentVariable(p, 0);
		*pe = L'='; /* Reset */
	}
	return;
}


/*
 * Utility function for mapNTUserToKerbUser.
 */

static WCHAR *useDefault(const WCHAR *name)
{
	WCHAR *ret = (WCHAR *)LocalAlloc(LMEM_FIXED, (lstrlenW(name) + 1)*sizeof(WCHAR));
	if(ret != 0)
		lstrcpyW(ret, name);
	else
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate memory for username.", 0, 0);
	return ret;
}

/*
 * Function to map an NT domain name to a kerberos realm.
 * If no map given then just use the username with no realm.
 */

static WCHAR *mapNTUsertoKerbUser(const kerb_info *kinfo, HKEY *hkey)
{
	WCHAR *ret = 0;
	WCHAR *p;
	WCHAR *user = kinfo->userName;
	WCHAR *map;
	DWORD type;
	DWORD datasize = 0;
	DWORD err;

	if((err = RegQueryValueEx(hkey, REALMMAP_NAME, 0, &type, 0, &datasize))!=ERROR_SUCCESS) {
		/* Realm map not there, use default realm */
		return useDefault(user);
	}

	map = (WCHAR *)LocalAlloc(LMEM_FIXED, datasize);
	if(map == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate memory for REALMS value.", 0, 0);
		return 0;
	}
	if((err = RegQueryValueEx(hkey, REALMMAP_NAME, 0, &type, (LPBYTE)map, &datasize))!=ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to get REALMS value.", err, 0);
		goto cleanup;
	}

	/* 
	 * Now search through the strings looking for the users logged
	 * on domain. If not found use the default.
	 */
	for( p = map; *p; p += lstrlenW(map) + 1) {
		if(CompareString(LOCALE_SYSTEM_DEFAULT, 0, 
			             p, lstrlenW(kinfo->userDomain),
						 kinfo->userDomain, lstrlenW(kinfo->userDomain)) == 2) {
			/* Move past the NT domain name and the following '=' character. */
			p += lstrlenW(kinfo->userDomain) + 1;
			ret = (WCHAR *)LocalAlloc(LMEM_FIXED, 
						( lstrlenW(p) + 1 + lstrlenW(user) + 1 ) * sizeof(WCHAR));
			if(ret == 0) {
				doErrorMessage(GINA_ERROR_MESSAGE,
					L"Kerberos Gina: Unable to allocate memory for username.", 0, 0);
				goto cleanup;
			}
			lstrcpyW(ret, user);
			lstrcatW(ret, L"@");
			lstrcatW(ret, p);
			LocalFree((HLOCAL)map);
			return ret;
		}
	}

	LocalFree((HLOCAL)map);
	/* Realm entry not there for this domain, use default realm */
	return useDefault(user);

cleanup:

	if(map != 0)
		LocalFree((HLOCAL)map);
	return ret;
}

/*
 * Funstion to check if the DOMAIN\user is on a list of users who should not
 * attempt to get tickets.
 */

static BOOL is_user_on_excluded_list(kerb_info *kinfo)
{
	WCHAR *p;
	WCHAR *user_portion;
	BOOL domain_match = FALSE;
	BOOL user_match = FALSE;

	for( p = kinfo->user_exclude_list; *p; p += (lstrlenW(p) + 1)) {
		for( user_portion = p; *user_portion && *user_portion != L'\\'; ++user_portion)
			;
		if(*user_portion == 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Malformed ExcludedUsers list :missing '\\' in name.", 0, 0);
			return TRUE;
		}
		/* Move past the '\' */
		++user_portion;

		if(*p == L'*') {
			domain_match = TRUE;
		} else {
			/* Cheat by setting the '\' to '\0' and using lstrcmpW */
			user_portion[-1] = L'\0';
			domain_match = (lstrcmpiW( p, kinfo->userDomain) == 0);
			user_portion[-1] = L'\\';
		}
		if(*user_portion == L'*') {
			user_match = TRUE;
		} else {
			user_match = (lstrcmpiW( user_portion, kinfo->userName) == 0);
		}
		if((domain_match == TRUE) && (user_match == TRUE))
			return TRUE;
	}
	return FALSE;
}

/*
 * Function to call kinit with username. Calls new Kerberos 5
 * init API. Return TRUE if failure, FALSE if not.
 */

static BOOL doKinit(
	kerb_info       *kinfo,
	PVOID           pWlxContext,
    PWSTR           pszDesktopName)
{
	BOOL ret = FALSE;
	HKEY gina_key = (HKEY)INVALID_HANDLE_VALUE;
	WCHAR *userspec = 0;
	char *username = 0;
	char *password = 0;
	char *ccache_name = 0;
	int err;
	krb5_context kcontext = 0;
	krb5_get_init_creds_opt opts;
	krb5_ccache ccache = 0;
	krb5_principal princ = 0;
	krb5_creds creds;
	DWORD exit_code = 1;
	BOOL impersonating_user = FALSE;

	__try {

		debug_print("doKinit 1\n");

		/* 
		 * Get the KERBNET_HOME value from the registry.
		 * Has the side effect of setting kinfo->require_kerberos_logon.
		 */
		if(getKerbNetHome(KERBNET_MAJOR_VERSION, &gina_key, kinfo)) {
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 2\n");

		/* Setup new environment */
		if((kinfo->extra_env_vars = setupNewEnv(kinfo)) == 0) {
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 3\n");

		/*
		 * If the user is explicitly excluded from getting a ticket, just return.
		 */
		if(is_user_on_excluded_list(kinfo) == TRUE) {
			kinfo->excluded_user = TRUE;
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 4\n");

		/* Get the user@REALM we will use */
		if((userspec = mapNTUsertoKerbUser(kinfo, gina_key)) == 0) {
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 5\n");

		/* Set the environment variables we need in WinLogon.exe. */
		if(setupEnvVars(kinfo->extra_env_vars) == FALSE) {
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 5a\n");

		if(ImpersonateLoggedOnUser(kinfo->hUserToken) == 0) {
			doErrorMessage(GINA_ERROR_MESSAGE, 
						L"Kerberos Gina: Unable to impersonate user.", GetLastError(), 0);
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 6\n");

		impersonating_user = TRUE;

		/* 
		 * Initialize the Kerberos context.
		 */
		if((err = krb5_init_context(&kcontext)) != 0) {
		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to initialize Kerberos context.",
				0, 0);
			kcontext = 0;
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 7\n");

		/*
		 * Initialize the Kerberos error messages.
		 */
		krb5_init_ets(kcontext);

		debug_print("doKinit 8\n");

		/*
		 * Initialize the krb5 opts.
		 */
		krb5_get_init_creds_opt_init(&opts);


		debug_print("doKinit 9\n");

		/*
		 * convert the unicode credential cache path to an ASCII string.
		 */
		if(ascii_string( kinfo->ccache_file, &ccache_name) != 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to convert credential cache to string.",
				GetLastError(), 0);
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 10\n");

		/*
		 * Resolve a credentials cache.
		 */
		if((err = krb5_cc_resolve(kcontext, ccache_name, &ccache)) != 0) {
		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to resolve Kerberos credendial cache.",
				0, err);
			ccache = 0;
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 11\n");

		/*
		 * convert the unicode user name to an ASCII string.
		 */
		if(ascii_string( userspec, &username) != 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to convert user name to string.",
				GetLastError(), 0);
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 12\n");

		/* 
		 * Parse the username into a Kerberos principal.
		 */
		if((err = krb5_parse_name( kcontext, username, &princ)) != 0) {
		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to parse Kerberos user name.",
				0, err);
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 13\n");

		/*
		 * Convert the unicode password to an ascii string.
		 */
		if(ascii_string( kinfo->userPassword, &password) != 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to convert password to string.",
				GetLastError(), 0);
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 14\n");

		/*
		 * Get the ticket.
		 */
		if((err = krb5_get_init_creds_password( kcontext,
												&creds,
												princ,
												password,
												0,
												0,
												0,
												0,
												&opts)) != 0) {
			if(password != 0)
				ZeroMemory( password, lstrlenA(password));
 		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to get Kerberos ticket.",
				0, err);
			ret = TRUE;
			goto cleanup;
		}

		debug_print("doKinit 15\n");

		/* Wipe the password. */
		if(password != 0)
			ZeroMemory( password, lstrlenA(password));

		/*
		 * Now store the creds in the ccache.
		 */
		if((err = krb5_cc_initialize( kcontext, ccache, princ)) != 0) {
 		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to initialize Kerberos credential cache.",
				0, err);
			ret = TRUE;
			goto cleanup;
		}
		if((err = krb5_cc_store_cred( kcontext, ccache, &creds)) != 0) {
 		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to store Kerberos ticket in credential cache.",
				0, err);
			ret = TRUE;
			goto cleanup;
		}

	} __except ( TRUE ) {
		doErrorMessage(GINA_ERROR_MESSAGE, L"Kerberos Gina: GPF(3).",0, 0);
		ret = TRUE;
	}

cleanup:

	debug_print("doKinit 16\n");

	if((kcontext != 0) && (princ != 0)) {
		krb5_free_principal(kcontext, princ);
	}
	if((kcontext != 0) && (ccache != 0)) {
		krb5_cc_close(kcontext, ccache);
	}
	if(kcontext != 0) {
		krb5_free_context(kcontext);
	}
	if(impersonating_user == TRUE) {
		RevertToSelf();
	}
	if(gina_key != (HKEY)INVALID_HANDLE_VALUE)
		RegCloseKey(gina_key);
	if(userspec != 0)
		LocalFree((HLOCAL)userspec);
	if(username != 0)
		LocalFree((HLOCAL)username);
	if(password != 0) {
		ZeroMemory( password, lstrlenA(password));
		LocalFree((HLOCAL)password);
	}
	if(ccache_name != 0)
		LocalFree((HLOCAL)ccache_name);

	return ret;
}

/*
 * Code to convert a SID into a textural form to look up a value
 * in HKEY_USERS. If len is too small set size and return.
 * NB. This code based on Microsoft Knowledgebase code.
 */

BOOL GetTextualSid( PSID psid, WCHAR *sid_text, DWORD *len)
{
    SID_IDENTIFIER_AUTHORITY *psia;
    DWORD sub_auth_count;
	DWORD i;
    DWORD sid_size;
 
    /* Get SidIdentifierAuthority */
    psia = GetSidIdentifierAuthority(psid);
 
    /* Get sidsubauthority count */
    sub_auth_count = (DWORD)*GetSidSubAuthorityCount(psid);
 
    /*
     * compute buffer length
     * S-SID_REVISION- + identifierauthority- + subauthorities- + NULL
     */
    sid_size = (15 + 12 + (12 * sub_auth_count) + 1) * sizeof(WCHAR);
 
    /*
     * check provided buffer length.
     * If not large enough, indicate proper size and setlasterror
     */
    if (*len < sid_size) {
        *len = sid_size;
        SetLastError(ERROR_INSUFFICIENT_BUFFER);
        return FALSE;
    }
 
    /*
     *  prepare S-SID_REVISION-
     */
    sid_size = wsprintf(sid_text, L"S-%lu-", SID_REVISION );
 
    /*
     * prepare SidIdentifierAuthority
     */
    if ( (psia->Value[0] != 0) || (psia->Value[1] != 0) ) {
        sid_size += wsprintf(sid_text + lstrlenW(sid_text),
                    L"0x%02hx%02hx%02hx%02hx%02hx%02hx",
                    (USHORT)psia->Value[0],
                    (USHORT)psia->Value[1],
                    (USHORT)psia->Value[2],
                    (USHORT)psia->Value[3],
                    (USHORT)psia->Value[4],
                    (USHORT)psia->Value[5]);
    } else {
        sid_size += wsprintf(sid_text + lstrlenW(sid_text),
                    L"%lu",
                    (ULONG)(psia->Value[5]      )   +
                    (ULONG)(psia->Value[4] <<  8)   +
                    (ULONG)(psia->Value[3] << 16)   +
                    (ULONG)(psia->Value[2] << 24)   );
    }
 
    /*
     * loop through SidSubAuthorities
     */
    for (i = 0; i < sub_auth_count; ++i) {
        sid_size += wsprintf(sid_text + sid_size, L"-%lu",
							*GetSidSubAuthority(psid, i) );
    }
 
    return TRUE;
}
 

WCHAR *getStringSidFromToken( HANDLE htoken)
{
	TOKEN_USER *userval = 0 ;
	DWORD size = 0;
	WCHAR *sid_text = 0;

	/* First get the size needed */
	GetTokenInformation( htoken, TokenUser,(LPVOID)userval, 0, &size);

	/* Now allocate a size buffer */
	if((userval = (TOKEN_USER *)LocalAlloc(LMEM_FIXED, size)) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate TokenUser buffer.", GetLastError(), 0);
		return 0;
	}

	if(GetTokenInformation( htoken, TokenUser,(LPVOID)userval, 
				size, &size) == FALSE) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to get token information.", GetLastError(), 0);
		LocalFree((HLOCAL)userval);
		return 0;
	}

	/* 
	 * First get the size.
	 */
	size = 0;
	GetTextualSid( userval->User.Sid, sid_text, &size);

	/* Now allocate a size buffer */
	if((sid_text = (WCHAR *)LocalAlloc(LMEM_FIXED, size)) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate SID buffer.", GetLastError(), 0);
		LocalFree((HLOCAL)userval);
		return 0;
	}

	/*
	 * Now get the SID as text.
	 */
	GetTextualSid( userval->User.Sid, sid_text, &size);

	LocalFree((HLOCAL)userval);
	return sid_text;
}

BOOL addEnvironmentToUserEnv(WCHAR *newEnv, HANDLE htoken)
{
	HKEY hkey = (HKEY)INVALID_HANDLE_VALUE;
	WCHAR *p;
	WCHAR *eqp;
	WCHAR *sid_text = 0;
	WCHAR *key_name = 0;
	DWORD key_name_len;
	DWORD err;
	BOOL ret = FALSE;

    if(ImpersonateLoggedOnUser(htoken) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to become user to set environment.", GetLastError(), 0);
		goto cleanup;
	}

	if((sid_text = getStringSidFromToken(htoken)) == 0)
		goto cleanup;

	key_name_len = lstrlenW(sid_text) + lstrlenW(L"\\Environment") + 1;
	if((key_name = (WCHAR *)LocalAlloc(LMEM_FIXED, key_name_len * sizeof(WCHAR))) == 0) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to allocate memory for environment key name.", 
			GetLastError(), 0);
		goto cleanup;
	}

	lstrcpyW(key_name, sid_text);
	lstrcatW(key_name, L"\\Environment");

	if((err = RegOpenKeyEx(HKEY_USERS, key_name, 0, 
				KEY_READ | KEY_SET_VALUE, &hkey)) !=
				ERROR_SUCCESS) {
		doErrorMessage(GINA_ERROR_MESSAGE,
			L"Kerberos Gina: Unable to open registry key 'Environment' for configuration.",
			err, 0);
		goto cleanup;
	}
	/*
	 * Now go through all the extra values we want to
	 * add and temporarily replace the '=' character with
	 * '\0'.
	 */
 	for(p = (WCHAR *)newEnv; *p; p += (lstrlenW(p) + 1)) {
		for(eqp = p; *eqp != L'=' && &eqp; eqp++)
			;
		/* 
		 * We should be pointing at a L'=', if not we
		 * have a corrupt entry, just continue.
		 */
		if(*eqp == L'\0')
			continue;
		*eqp = L'\0';
		if((err = RegSetValueEx( hkey, p, 0, REG_SZ, (const BYTE *)&eqp[1], 
								sizeof(WCHAR)*lstrlenW(&eqp[1]))) !=
			ERROR_SUCCESS) {
			/* Restore the '='. */
			*eqp = L'=';
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Unable to set user environment.",
							err, 0);
			goto cleanup;
		}
		/* Restore the '='. */
		*eqp = L'=';
	}

	ret = TRUE;

cleanup:

	RevertToSelf();
	if(hkey != (HKEY)INVALID_HANDLE_VALUE)
		RegCloseKey(hkey);
	if(sid_text != 0)
		LocalFree((HLOCAL)sid_text);
	if(key_name != 0)
		LocalFree((HLOCAL)key_name);

	return ret;
}

BOOL
WINAPI
WlxActivateUserShell(
    PVOID           pWlxContext,
    PWSTR           pszDesktopName,
    PWSTR           pszMprLogonScript,
    PVOID           pEnvironment)
{

	BOOL ret = FALSE;

	__try {
		/*
		 * At this point the users profile has been loaded
		 * as HKEY_USERS\<SID>. Add our new variables to
		 * the Environment subkey.
		 */

		if(global_kinfo.extra_env_vars != 0)
			addEnvironmentToUserEnv(global_kinfo.extra_env_vars, global_kinfo.hUserToken);

		ret = GWlxActivateUserShell(
					pWlxContext,
					pszDesktopName,
					pszMprLogonScript,
					pEnvironment
					);
	} __except ( TRUE ) {
		doErrorMessage(GINA_ERROR_MESSAGE, L"Kerberos Gina: GPF(4).",0, 0);
		ret = FALSE;
	}
	return ret;
}


int
WINAPI
WlxLoggedOnSAS(
    PVOID           pWlxContext,
    DWORD           dwSasType,
    PVOID           pReserved)
{
    return GWlxLoggedOnSAS( pWlxContext, dwSasType, pReserved );
}

VOID
WINAPI
WlxDisplayLockedNotice(
    PVOID           pWlxContext )
{
    GWlxDisplayLockedNotice( pWlxContext );
}


BOOL
WINAPI
WlxIsLockOk(
    PVOID           pWlxContext)
{
    return GWlxIsLockOk( pWlxContext );
}


int
WINAPI
WlxWkstaLockedSAS(
    PVOID           pWlxContext,
    DWORD           dwSasType )
{
    return GWlxWkstaLockedSAS( pWlxContext, dwSasType );
}

/*
 * Function to delete tickets on logout.
 */

BOOL doKdestroy(
	kerb_info       *kinfo,
	PVOID           pWlxContext,
    PWSTR           pszDesktopName)
{
	BOOL ret = FALSE;
	int err;
	krb5_context kcontext = 0;
	krb5_ccache ccache = 0;
	BOOL impersonating_user = FALSE;
	char *ccache_name;

	__try {
		if(ImpersonateLoggedOnUser(kinfo->hUserToken) == 0) {
			doErrorMessage(GINA_ERROR_MESSAGE, 
						L"Kerberos Gina: Unable to impersonate user.", GetLastError(), 0);
			ret = TRUE;
			goto cleanup;
		}

		impersonating_user = TRUE;

		if((err = krb5_init_context(&kcontext)) != 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to initialize Kerberos.",
				0, 0);
		}

		krb5_init_ets(kcontext);

		/*
		 * convert the unicode credential cache path to an ASCII string.
		 */
		if(ascii_string( kinfo->ccache_file, &ccache_name) != 0) {
			doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to convert credential cache to string.",
				GetLastError(), 0);
			ret = TRUE;
			goto cleanup;
		}

		if((err = krb5_cc_resolve(kcontext, ccache_name, &ccache)) != 0) {
		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to resolve Kerberos credendial cache.",
				0, err);
			ccache = 0;
			ret = TRUE;
			goto cleanup;
		}

		if((err = krb5_cc_destroy(kcontext, ccache)) != 0) {
		    doErrorMessage(GINA_ERROR_MESSAGE,
				L"Kerberos Gina: Failed to destroy Kerberos credendial cache.",
				0, err);
			ret = TRUE;
			goto cleanup;
		}

	} __except ( TRUE ) {
		doErrorMessage(GINA_ERROR_MESSAGE, L"Kerberos Gina: GPF(5).",0, 0);
		ret = TRUE;
		return ret;
	}

cleanup:

	if((kcontext != 0) && (ccache != 0)) {
		krb5_cc_close(kcontext, ccache);
	}
	if(kcontext != 0) {
		krb5_free_context(kcontext);
	}

	if(impersonating_user == TRUE) {
		RevertToSelf();
	}
	if(ccache_name != 0)
		LocalFree((HLOCAL)ccache_name);
	return ret;
}

BOOL
WINAPI
WlxIsLogoffOk(
    PVOID pWlxContext
    )
{
    BOOL bSuccess;
	DWORD ret;

    bSuccess = GWlxIsLogoffOk( pWlxContext );

    if(bSuccess) {
		__try {
			/*
			 * if it's ok to logoff, finish with the stored credentials
			 * and scrub the buffers
			 */
			debug_print("WlxIsLogoffOk 1\n");

			if(global_kinit_succeeded == TRUE) /* kdestroy here.... */
				ret = doKdestroy(&global_kinfo, pWlxContext,0);

			debug_print("WlxIsLogoffOk 2\n");
		} __except ( TRUE ) {
			doErrorMessage(GINA_ERROR_MESSAGE,L"Kerberos Gina: GPF(6).",0, 0);
		}
    }

    return bSuccess;
}


VOID
WINAPI
WlxLogoff(
    PVOID pWlxContext
    )
{
	/* Remove any extra environment variables from Winlogon.exe */
	deleteEnvVars(global_kinfo.extra_env_vars);
	/* Destroy any global resources used for this logon */
	destroyKerbInfo(&global_kinfo);
	/* Shutdown sockets for Winlogon.exe */
	WSACleanup();
	global_done_wsastartup = FALSE;
    GWlxLogoff( pWlxContext );
}


VOID
WINAPI
WlxShutdown(
    PVOID pWlxContext,
    DWORD ShutdownType
    )
{
    GWlxShutdown( pWlxContext, ShutdownType );
}


/*
 * NEW for version 1.1
 */

BOOL
WINAPI
WlxScreenSaverNotify(
    PVOID                   pWlxContext,
    BOOL *                  pSecure
    )
{
    if(GWlxScreenSaverNotify != NULL)
        return GWlxScreenSaverNotify( pWlxContext, pSecure );

    /*
     * if not exported, return something intelligent
     */

    *pSecure = TRUE;

    return TRUE;
}

BOOL
WINAPI
WlxStartApplication(
    PVOID                   pWlxContext,
    PWSTR                   pszDesktopName,
    PVOID                   pEnvironment,
    PWSTR                   pszCmdLine
    )
{
    if(GWlxStartApplication != NULL)
        return GWlxStartApplication(
            pWlxContext,
            pszDesktopName,
            pEnvironment,
            pszCmdLine
            );

    /*
     * if not exported, return something intelligent
     */
	return FALSE;
}

