/*
 * Boilerplate service starter.
 */

#if defined(_WIN32) && defined(__CYGWIN32__)

#include <windows.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <limits.h>
#include <string.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/times.h>
#include <syslog.h>

#include "win32service.h"

#ifdef __CYGWIN32__
#define NULLDEVICE "/dev/null"
#endif /* __CYGWIN32__ */

/* 
 * Functions defined inside real service code.
 * - need to be prototyped in win32service.h.
 *
 * extern int do_service_install(int *argc, char **argv);
 * extern int do_service_delete(int *argc, char **argv);
 * extern int do_service_main(struct args *);
 * extern int request_service_stop(SERVICE_STATUS *);
 * extern int do_service_usage(int argc, char **argv);
 */

/* Handle to talk to Service manager. */
SERVICE_STATUS_HANDLE global_ssh;
SERVICE_STATUS global_sstatus;
HANDLE global_service_thread;
BOOL global_debug_flag;
BOOL global_log_to_stderr_flag;
char global_service_name[] = CYGWIN_SERVICE_NAME;

/*
 * Convert system error to char. Returns 
 * memory allocated with LocalAlloc.
 */

char *error_to_string(DWORD error)
{
  char *msgbuf;
  
  if(FormatMessage(
       FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
       NULL,
       error,
       MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT), /* Default language */
       (char *)&msgbuf,
       0,
       NULL
       ) == 0)
    return 0;
  return msgbuf;
}

/*
 * Return a pointer to a string describing an os error.
 * error_to_string returns a pointer to LocalAlloc'ed
 * memory. Cache it and release when the next one is
 * requested.
 */

char *str_oserr(DWORD err)
{
  static char *lastmsg = 0;

  if(lastmsg)
    LocalFree((HLOCAL)lastmsg);

  lastmsg = error_to_string(err);
  return lastmsg;
}

/*
 * va_list function to log an error message.
 */

void log_message_va(int sev, const char *fmt, va_list ap)
{
  static FILE *fp = 0;
  int length;
  char *p;

  if(global_log_to_stderr_flag) {
    vfprintf(stderr, fmt, ap);
    fprintf(stderr, "\n");
    return;
  }

  if(fp == 0) {
    fp = fopen(NULLDEVICE, "w");
    if(fp == 0) {
      syslog(LOG_ERR, "Unable to open /dev/null for log.");
      exit(-1);
    }
  }

  length = vfprintf(fp, fmt, ap);
  p = (char *)malloc(length+1);
  if( p == 0) {
    syslog(LOG_ERR, "malloc fail in log_message.");
    return;
  }
  (void)vsprintf(p, fmt, ap);
  syslog( sev, p);
  free(p);
}
	
/*
 * varags function to log an error message.
 */

void log_message(int sev, const char *fmt, ...)
{
  va_list ap;
  va_start( ap, fmt);

  log_message_va(sev, fmt, ap);

  va_end( ap );
}

/*
 * Utility function to get allocate a SID from a name.
 * Looks on local machine. SID is allocated with LocalAlloc
 * and must be freed by the caller.
 * Returns TRUE on success, FALSE on fail.
 */

BOOL get_sid(const char *name, SID **ppsid)
{
  SID_NAME_USE sid_use;
  DWORD sid_size = 0;
  DWORD dom_size = 0;
  char *domain;

  *ppsid = 0;
  if(LookupAccountName(0, name, 0, &sid_size, 0, &dom_size, &sid_use) == 0) {
    if(GetLastError() != ERROR_INSUFFICIENT_BUFFER) {
      log_message( LOG_ERR, "get_sid: LookupAccountName for size on name %s failed. Error was %s",
            name, str_oserr(GetLastError()));
      return FALSE;
    }
  }

  *ppsid = (SID *)LocalAlloc( LMEM_FIXED, sid_size);
  domain = (char *)LocalAlloc( LMEM_FIXED, dom_size);
  if( *ppsid == 0 || domain == 0) {
    log_message( LOG_ERR, "get_sid: LocalAlloc failed. Error was %s",
                 str_oserr(GetLastError()));
    if(*ppsid)
      LocalFree((HLOCAL)*ppsid);
    if(domain)
      LocalFree((HLOCAL)domain);
    *ppsid = 0;
    return FALSE;
  }

  if(LookupAccountName(0, name, *ppsid, &sid_size, domain, &dom_size, &sid_use) == 0) {
    log_message( LOG_ERR, 
         "get_sid: LookupAccountName failed for name %s. Error was %s",
         name, str_oserr(GetLastError()));
    LocalFree((HLOCAL)*ppsid);
    LocalFree((HLOCAL)domain);
    *ppsid = 0;
    return FALSE;
  }

  LocalFree((HLOCAL)domain);
  return TRUE;
}

/*
 * Utility function to setup a security descriptor
 * from a varargs list of char *name followed by a DWORD access
 * mask. The access control list is allocated with LocalAlloc
 * and must be freed by the caller.
 * returns TRUE on success, FALSE on fail.
 */

BOOL create_sd_from_list( SECURITY_DESCRIPTOR *sdout, int num, ...)
{
  va_list ap;
  SID **sids = 0;
  char *name;
  DWORD amask;
  DWORD acl_size;
  PACL pacl = 0;
  int i;

  if((sids = (SID **)calloc(1,sizeof(SID *)*num)) == 0) {
    log_message(LOG_ERR, "create_sd_from_list: calloc fail.");
    return FALSE;
  }

  acl_size = num * (sizeof(ACL) +
             sizeof(ACCESS_ALLOWED_ACE) +
             sizeof(DWORD));

  /* Collect all the SID's */
  va_start( ap, num);
  for( i = 0; i < num; i++) {
    name = va_arg( ap, char *);
    amask = va_arg(ap, DWORD);
    if(get_sid( name, &sids[i]) == FALSE)
      goto cleanup;
    acl_size += GetLengthSid(sids[i]);
  }
  va_end(ap);
  if((pacl = (PACL)LocalAlloc( LMEM_FIXED, acl_size)) == 0) {
    log_message( LOG_ERR, "create_sd_from_list: LocalAlloc fail. Error was %s",
            str_oserr(GetLastError()));
    goto cleanup;
  }

  if(InitializeSecurityDescriptor( sdout, SECURITY_DESCRIPTOR_REVISION) == FALSE) {
    log_message( LOG_ERR, "create_sd_from_list: InitializeSecurityDescriptor fail. Error was %s",
                 str_oserr(GetLastError()));
    goto cleanup;
  }
  if(InitializeAcl( pacl, acl_size, ACL_REVISION) == FALSE) {
    log_message( LOG_ERR, "create_sd_from_list: InitializeAcl fail. Error was %s",
                 str_oserr(GetLastError()));
    goto cleanup;
  }
  va_start(ap, num);
  for( i = 0; i < num; i++) {
    ACE_HEADER *ace_p;
    name = va_arg( ap, char *);
    amask = va_arg( ap, DWORD);
    if(AddAccessAllowedAce( pacl, ACL_REVISION, amask, sids[i]) == FALSE) {
      log_message( LOG_ERR, "create_sd_from_list: AddAccessAllowedAce fail. Error was %s",
                 str_oserr(GetLastError()));
      goto cleanup;
    }
    /* Make sure the ACE is inheritable */
    if(GetAce( pacl, 0, (LPVOID *)&ace_p) == FALSE) {
      log_message( LOG_ERR, "create_sd_from_list: GetAce fail. Error was %s",
                 str_oserr(GetLastError()));
      goto cleanup;
    }
    ace_p->AceFlags |= ( CONTAINER_INHERIT_ACE | OBJECT_INHERIT_ACE);
  }

  /* Add the ACL into the sd. */
  if(SetSecurityDescriptorDacl( sdout, TRUE, pacl, FALSE) == FALSE) {
    log_message( LOG_ERR, "create_sd_from_list: SetSecurityDescriptorDacl fail. Error was %s",
               str_oserr(GetLastError()));
    goto cleanup;
  }
  for( i = 0; i < num; i++)
    if(sids[i] != 0)
      LocalFree((HLOCAL)sids[i]);
  free(sids);

  return TRUE;

cleanup:

  if(sids != 0) {
    for( i = 0; i < num; i++)
      if(sids[i] != 0)
        LocalFree((HLOCAL)sids[i]);
    free(sids);
  }
  if(pacl != 0)
    LocalFree((HLOCAL)pacl);
  return FALSE;
}

/*
 * Utility function to copy a file into a directory, setting the given
 * SECURITY_DESCRIPTOR on the new file. Solves race conditions with doing
 * CopyFile followed by SetFileSecurity.
 * Returns FALSE on fail, TRUE if successful.
 */

#define COPY_BUF_SIZE (50*1024)

BOOL copy_file_with_sd( const char *src, const char *dst, BOOL fail_if_exists, SECURITY_DESCRIPTOR *sd)
{
  HANDLE hSrc, hDst;
  SECURITY_ATTRIBUTES sa;
  char *buffer = 0;

  if((fail_if_exists == TRUE) && (GetFileAttributes(dst) != 0xFFFFFFFF)) {
    log_message( LOG_ERR, "copy_file_with_sd: File %s already exists.",
                 dst);
    return FALSE;
  }

  if((hSrc = CreateFile( src, GENERIC_READ, 0, 0, OPEN_EXISTING, 0, 0)) 
                        == INVALID_HANDLE_VALUE) {
    log_message(LOG_ERR, 
              "copy_file_with_sd: Failed to open source file %s. Error was %s",
              src, str_oserr(GetLastError()));
    return FALSE;
  }

  sa.nLength = sizeof(SECURITY_ATTRIBUTES);
  sa.lpSecurityDescriptor = sd;
  sa.bInheritHandle = 0;

  if((hDst = CreateFile( dst, GENERIC_WRITE, 0, &sa, CREATE_NEW, 
                      FILE_ATTRIBUTE_NORMAL, 0)) == INVALID_HANDLE_VALUE) {
    log_message(LOG_ERR, 
         "copy_file_with_sd: Failed to open destination file %s. Error was %s",
         dst, str_oserr(GetLastError()));
    CloseHandle(hSrc);
    return FALSE;
  }

  if((buffer = (char *)LocalAlloc( LMEM_FIXED, COPY_BUF_SIZE)) == 0) {
    log_message(LOG_ERR, 
         "copy_file_with_sd: LocalAlloc failed. Error was %s",
         str_oserr(GetLastError()));
    CloseHandle(hSrc);
    CloseHandle(hDst);
    return FALSE;
  }

  while(TRUE) {
    DWORD numread;
    DWORD numwritten;
    if(ReadFile( hSrc, buffer, COPY_BUF_SIZE, &numread, 0) == 0) {
      log_message(LOG_ERR,
                  "copy_file_with_sd: ReadFile on file %s failed. Error was %s",
                  src, str_oserr(GetLastError()));
      LocalFree((HLOCAL)buffer);
      CloseHandle(hSrc);
      CloseHandle(hDst);
      return FALSE;
    }
    /* Check for EOF */
    if(numread == 0)
      break;
    if(WriteFile( hDst, buffer, numread, &numwritten, 0) == 0) {
      log_message(LOG_ERR,
                  "copy_file_with_sd: WriteFile on file %s failed. Error was %s",
                  dst, str_oserr(GetLastError()));
      LocalFree((HLOCAL)buffer);
      CloseHandle(hSrc);
      CloseHandle(hDst);
      return FALSE;
    }
  }
  LocalFree((HLOCAL)buffer);
  CloseHandle(hSrc);
  CloseHandle(hDst);
  return TRUE;
}

/*
 * Utility function to cause a registry tree to be created.
 * If the supplied tree doesn't exist it will walk down the
 * tree creating the branches.
 */

int create_registry_tree( HKEY key_start, const char *reg_path, SECURITY_DESCRIPTOR *sd)
{
  /* First see if the key already exists */
  HKEY key;
  DWORD err;
  DWORD disp;
  SECURITY_ATTRIBUTES sa, *psa = 0;
  char *cur_path = 0;
  char *p;

  if((err = RegOpenKeyEx( key_start, reg_path, 0, KEY_READ, &key)) == ERROR_SUCCESS) {
    /* Key exists */
    RegCloseKey(key);
    return 0;
  }

  if((cur_path = malloc(strlen(reg_path) + 1)) == 0) {
    log_message( LOG_ERR, "create_registry_tree: malloc failed.");
    return -1;
  }

  if(sd != 0) {
    sa.nLength = sizeof(SECURITY_ATTRIBUTES);
    sa.lpSecurityDescriptor = sd;
    sa.bInheritHandle = FALSE;
    psa = &sa;
  }

  p = strchr(reg_path, '\\');
  while(1) {
    if( p == 0 ) {
      strcpy(cur_path, reg_path);
    } else {
      strncpy( cur_path, reg_path, p - reg_path);
      cur_path[p-reg_path] = '\0';
    }

    if((err = RegCreateKeyEx( key_start, cur_path, 0, 0, REG_OPTION_NON_VOLATILE,
                               KEY_READ, psa, &key, &disp)) != ERROR_SUCCESS) {
      free(cur_path);
      log_message(LOG_ERR, 
            "create_registry_tree: RegCreateKeyEx failed for path %s. Error was %s.", 
            cur_path, str_oserr(err));
    }
    RegCloseKey(key);
    if(p == 0)
      break;
    p++;
    p = strchr(p, '\\');
  }
  return 0;
}

/*
 * Utility function to remove a registry tree. It starts at the
 * leaf node and deletes up while the keys it examines have no
 * sub-keys.
 */

int delete_registry_tree( HKEY key_start, const char *reg_path)
{
  DWORD err;
  HKEY key;
  char *cur_path;
  DWORD num_subkeys;
 
  char *p = (char *)-1;

  if((cur_path = strdup( reg_path)) == 0) {
    log_message( LOG_ERR, "delete_registry_tree: malloc failed.");
    return -1;
  }

  while(p) {
    /*
     * Open the current key.
     */
    if((err = RegOpenKeyEx( key_start, cur_path, 0, KEY_READ|KEY_WRITE, 
                     &key)) != ERROR_SUCCESS) {
      log_message(LOG_ERR, "delete_registry_tree: RegOpenKeyEx failed for key %s\
. Error was %s", cur_path, str_oserr(err));
      free(cur_path);
      return -1;
    }

    /*
     * Find out how many sub-keys it has.
     */
    if((err = RegQueryInfoKey( key, 0, 0, 0, &num_subkeys, 
                             0, 0, 0, 0, 0, 0, 0)) != ERROR_SUCCESS) {
      log_message(LOG_ERR, "delete_registry_tree: RegQueryInfoKey failed for key %s. Error was %s", cur_path, str_oserr(err));
      free(cur_path);
      RegCloseKey(key);
      return -1;
    }

    if(num_subkeys != 0) {
      free(cur_path);
      RegCloseKey(key);
      return 0;
    }

    /* Key has no subkeys - close and delete it. */
    RegCloseKey(key);
    if((err = RegDeleteKey(key_start, cur_path)) != ERROR_SUCCESS) {
      log_message(LOG_ERR, "delete_registry_tree: RegDeleteKey failed for key %
s. Error was %s", cur_path, str_oserr(err));
      free(cur_path);
      return -1;
    }

    /* Strip the last component of the path. */
    p = strrchr( cur_path, '\\');
    if( p != 0)
      *p = 0;
  }

  free(cur_path);
  return 0;
}

/*
 * Utility function to install an argv array as a REG_MULTI_SZ into
 * a given Value in a given registry key. Used to setup unix command
 * lines for NT services.
 */

int setup_service_args( HKEY base_key, const char *reg_path, const char *reg_value,
                        int argc, char **argv)
{
  int i;
  char *value;
  char *p;
  int str_size = 0;
  DWORD err;
  HKEY hkey;

  for(i = 0; i < argc; i++)
    str_size += strlen(argv[i]) + 1;
  str_size++;

  if((value = malloc( str_size)) == 0) {
    log_message(LOG_ERR, "setup_service_args: malloc fail.");
    return -1;
  }

  for( i = 0, p = value; i < argc; i++) {
    strcpy(p, argv[i]);
    p += (strlen(argv[i]) + 1);
  }
  *p = '\0';

  if((err = RegOpenKeyEx( base_key, reg_path, 0, 
                         KEY_SET_VALUE, &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "setup_service_args: RegOpenKeyEx failed for key %s. Error was %s",
                reg_path, str_oserr(err));
    free(value);
    return -1;
  }
  if((err = RegSetValueEx( hkey, reg_value, 
                      0, REG_MULTI_SZ, value, str_size)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "setup_service_args: RegSetValueEx failed for value %s. Error was %s",
                reg_value, str_oserr(err));
    free(value);
    RegCloseKey(hkey);
    return -1;
  }
  free(value);
  RegCloseKey(hkey);
  return 0;
}

#ifdef KERBNET
/*
 * Kerbnet specific code to create the base registry entries
 * needed for all Kerbnet services.
 */

#define KERBNET_REG_BASE "SOFTWARE\\Cygnus Solutions\\Kerbnet\\1"
#define REG_KERBNET_HOME_VALUE "KERBNET_HOME"
#define REG_KERBNET_CCACHEDIR "CredentialsCacheDir"

int check_kerbnet_base_args()
{
  char name[2048];
  char *p;
  DWORD err;
  HKEY hkey;

  if(GetModuleFileName(0, name, sizeof(name)) == FALSE) {
    log_message(LOG_ERR, 
                "check_kerbnet_base_args: GetModuleFileName failed. Error was %s",
                str_oserr(GetLastError()));
    return -1;
  }

  p = strrchr(name, '\\');
  if( p == 0) {
    log_message( LOG_ERR, "check_kerbnet_base_args: GetModuleFileName returned invalid name. Name was %s", name);
    return -1;
  }

  /* Go up one more level to remove the sbin directory. */
  *p = 0;
  p = strrchr(name, '\\');
  if( p == 0 || (strcasecmp( p, "\\sbin")!=0)) {
    log_message( LOG_ERR, "check_kerbnet_base_args: service installed in incorrect location (needs to be in sub-directory sbin). Cannot obtain KERBNET_BASE value.");
    return -1;
  }

  *p = 0;
  /* Set this as the KERBNET_HOME value. */
  if((err = RegOpenKeyEx( HKEY_LOCAL_MACHINE, KERBNET_REG_BASE, 0, KEY_SET_VALUE,
                         &hkey)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "check_kerbnet_base_args: RegOpenKeyEx failed for key %s. Error was %s",
                KERBNET_REG_BASE, str_oserr(err));
    return -1;
  }
  if((err = RegSetValueEx( hkey, REG_KERBNET_HOME_VALUE, 0, 
                          REG_SZ, name, strlen(name)+1)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "check_kerbnet_base_args: RegSetValueEx failed on value %s. Error was %s",
                REG_KERBNET_HOME_VALUE, str_oserr(err));
    RegCloseKey(hkey);
    return -1;
  }
  strcat( name, "\\ccache");
  if((err = RegSetValueEx( hkey, REG_KERBNET_CCACHEDIR, 0, 
                     REG_SZ, name, strlen(name)+1)) != ERROR_SUCCESS) {
    log_message(LOG_ERR, "check_kerbnet_base_args: RegSetValueEx failed on value %s. Error was %s",
                REG_KERBNET_CCACHEDIR, str_oserr(err));
    RegCloseKey(hkey);
    return -1;
  }

  RegCloseKey(hkey);
  return 0;
}
#endif /* KERBNET */

/*
 * Report status to ServiceControlManager.
 */

BOOL TellServiceManager(DWORD state, DWORD accept, DWORD error, DWORD timeout)
{
  static DWORD checkpoint = 1;
  BOOL ret;

  if(global_debug_flag == TRUE)
    return TRUE;
    
  global_sstatus.dwControlsAccepted = accept; 
  global_sstatus.dwCurrentState = state;
  global_sstatus.dwWin32ExitCode = error;
  global_sstatus.dwWaitHint = timeout;

  if((state == SERVICE_RUNNING) || (state == SERVICE_STOPPED))
    global_sstatus.dwCheckPoint = 0;
  else
    global_sstatus.dwCheckPoint = checkpoint++;

  if((ret = SetServiceStatus(global_ssh, &global_sstatus))==FALSE) {
    log_message(LOG_ERR, "Failed to SetServiceStatus : Error was %s.", str_oserr(GetLastError()));
  }
  return ret;
}

/*
 * Service control handler.
 */

VOID WINAPI service_ctrl(DWORD ctrlCode)
{
  switch(ctrlCode)
    {
      case SERVICE_CONTROL_STOP:
      case SERVICE_CONTROL_SHUTDOWN:
        /*
         * Admin has asked us to shut down.
         * Tell SCM we are doing so then call
         * service and ask it to stop.
         */
	TellServiceManager(SERVICE_STOP_PENDING, 0, 0, 0);
	request_service_stop(&global_sstatus);
        break;
      case SERVICE_CONTROL_PAUSE:
	TellServiceManager(SERVICE_PAUSE_PENDING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
        request_service_pause(&global_sstatus);
        break;
      case SERVICE_CONTROL_CONTINUE:
	TellServiceManager(SERVICE_CONTINUE_PENDING, CYGWIN_SERVICE_ACCEPT_ALL, 0, 0);
        request_service_continue(&global_sstatus);
        break;
      default:
	/* Just tell the SCM we're swimming along nicely... */
	TellServiceManager(global_sstatus.dwCurrentState, CYGWIN_SERVICE_ACCEPT_ALL,
			   0, 0);
	break;
     }
}

/*
 * Handler function to stop user logoff events terminating
 * us with Cygwin32.
 */

BOOL handler_routine( DWORD event )
{
  return TRUE;
}

/*
 * Entry point for our service.
 */

void service_main(int argc, char *argv[])
{
  struct argc_argv args;
  DWORD threadid;
 
  openlog( global_service_name, LOG_PID, LOG_DAEMON);

  /* Set handler to ignore control signals. for
     some reason a Service using Cygwin32 needs this
     so it won't exit on user logout.
   */

  if(SetConsoleCtrlHandler( handler_routine, TRUE) == FALSE) {
    log_message(LOG_ERR, "Failed to register CtrlHandler. Error was %s",
                str_oserr(GetLastError()));
    goto cleanup;
  }

  /*
   * Register function to handle control requests.
   */

  if((global_debug_flag == FALSE) && ((global_ssh = RegisterServiceCtrlHandler(global_service_name,
                                       (LPHANDLER_FUNCTION)service_ctrl))==0)) {
    log_message(LOG_ERR, "Failed to register ServiceCtrlHandler. Error was %s", 
                str_oserr(GetLastError()));
    goto cleanup;
  }

  /*
   * Initialize the service status structure.
   */
  global_sstatus.dwServiceType = SERVICE_WIN32_OWN_PROCESS;
  global_sstatus.dwServiceSpecificExitCode = 0;

  /*
   * Tell service manager we are starting.
   */
  if(TellServiceManager(SERVICE_START_PENDING, 0, 0, 5000)==0)
    goto cleanup;

  /*
   * Run the service in a separate thread.
   * We do this so we can destroy it if it fails
   * to shut down.
   */

  args.argc = argc;
  args.argv = argv;

  if((global_service_thread = CreateThread(0, 0, (LPTHREAD_START_ROUTINE)do_service_main, 
			    (LPVOID)&args, 0, &threadid)) == 0) {
    log_message(LOG_ERR, "Failed to create service thread. Error was : %s", str_oserr(GetLastError()));
    goto cleanup;
  }

  /* We terminate when this returns... */
  WaitForSingleObject(global_service_thread, INFINITE);

cleanup:

  TellServiceManager(SERVICE_STOPPED, 0, 0, 0);

  closelog();

  return;
}  

/*
 * Get a username or password argument. Argument searchfor must be in argv[2] and
 * end in :. Anything after the : is returned in buf if there is room.
 * Returns 1 if no arg given, 0 if success, -1 on fail.
 */

int get_upassarg(int argc, char *argv[], const char *searchfor, char *buf, int bsize)
{
  if(argc < 3)
    return 1;
    
  if(((argv[2][0] == '-') || (argv[2][0] == '/')) &&
     (strncasecmp(&argv[2][1], searchfor, strlen(searchfor)) == 0)) {
    char *p = &argv[2][1 + strlen(searchfor)];
    if(strlen(p)+1 > bsize) {
      fprintf(stderr, "%s: String following %s parameter is too long.\n",
	      argv[0], searchfor);
      return -1;
    }
    strcpy(buf, p);
    return 0;
  }
  return -1;
}

/*
 * Print out a separate install message.
 * Different from the usage message.
 */

void do_install_message(const char *argv0)
{
  fprintf(stderr, "%s: To install service %s type :\n%s /install [/username:DOMAIN\\user] [/password:password]\n",
	  argv0, global_service_name, argv0);
}

/*
 * Function to delete a service. Calls service
 * supplied function first, and only deletes if
 * this returns zero. This allows any registry
 * deletes etc. to be done.
 */

int service_delete(int argc, char *argv[])
{
  int ret;
  SC_HANDLE hservice;
  SC_HANDLE hsManager;
  DWORD last_err;
  SERVICE_STATUS sstatus;
  int cont = TRUE;

  /*
   * This function must eat all args it uses, and leave the
   * correct values for args remaining in argc, argv.
   */
  if(ret = do_service_delete(&argc, argv, &cont)) {
     return ret;
  }

  if(cont == FALSE)
    return 0;

  if((hsManager = OpenSCManager(NULL, NULL, SC_MANAGER_ALL_ACCESS)) == 0) {
    last_err = GetLastError();
    fprintf(stderr, "%s: Failed to open the ServiceManager.\n", argv[0]);
    fprintf(stderr, " LastError was %d, %s\n",last_err, error_to_string(last_err));
    return -1;
  }

  if((hservice = OpenService(hsManager, global_service_name, SERVICE_ALL_ACCESS)) == 0) {
    last_err = GetLastError();
    fprintf(stderr, "%s: Failed to open the Service %s.\n", argv[0], global_service_name);
    fprintf(stderr, " LastError was %d, %s\n",last_err, error_to_string(last_err));
    CloseServiceHandle(hsManager);
    return -1;
  }

  /*
   * Ensure service is in stopped state.
   */
  if( QueryServiceStatus( hservice, &sstatus) == FALSE) {
    last_err = GetLastError();
    fprintf(stderr, "%s: Failed to query the service status for service %s. Error was  %d, %s\n",
             argv[0], global_service_name, 
             last_err, error_to_string(last_err));
    CloseServiceHandle(hservice);
    CloseServiceHandle(hsManager);
    return -1;
  }

  if( sstatus.dwCurrentState != SERVICE_STOPPED ) {
    printf("%s: Service %s is not currently in the STOPPED state. Stop the service\nBy typing 'NET STOP %s' and retry this operation\n",
             argv[0], global_service_name, global_service_name);
    CloseServiceHandle(hservice);
    CloseServiceHandle(hsManager);
    return -1;
  }

  if( DeleteService(hservice) ) {
    printf("%s: Service %s removed successfully.\n", argv[0], global_service_name);
    ret = 0;
  } else {
    last_err = GetLastError();
    fprintf(stderr, "%s: Failed to remove the Service %s.\n", argv[0], global_service_name);
    fprintf(stderr, " LastError was %d, %s\n",last_err, error_to_string(last_err));
    ret = -1;
  }
  CloseServiceHandle(hservice);
  CloseServiceHandle(hsManager);
  return 0;
}

/*
 * Function to install a service. Calls service
 * supplied function first, and only installs if
 * this returns zero. This allows any registry
 * updates etc. to be done.
 */

int service_install(int argc, char *argv[])
{
  int ret;
  DWORD last_err;
  char pathname[PATH_MAX];
  char username[129];
  char password[129];
  char *uname, *upass;
  SC_HANDLE sch = 0;
  SC_HANDLE hserv = 0;
  int cont = TRUE;

  /*
   * This function must eat all args it uses, and leave the
   * correct values for args remaining in argc, argv.
   */
  if(ret = do_service_install(&argc, argv, &cont))
    return ret;

  if(cont == FALSE)
    return 0;

  /*
   * Get the pathname we are invoked as.
   */
  if(GetModuleFileName(0, pathname, sizeof(pathname))==0) {
    last_err = GetLastError();
    fprintf(stderr, "%s: Failed to get the pathname from which this program was invoked.",
	    argv[0]);
    fprintf(stderr, " LastError was %d, %s\n",last_err, error_to_string(last_err));
    return -1;
  }

  /*
   * Get the username and password the service is to be installed
   * under, or use LocalSystem if no user name was given.
   */

  *username = 0;
  *password = 0;
  uname = 0;
  upass = 0;

  ret = get_upassarg(argc, argv, "username:", username, sizeof(username));
  if(ret == -1) {
    /* 
     * Malformed username arg.
     */
    do_install_message(argv[0]);
    return -1;
  }

  if(ret == 0) {
    /*
     * Username arg was ok, get password arg.
     */
    ret = get_upassarg(argc, argv, "password:", password, sizeof(password));
    if(ret != 0) {
      do_install_message(argv[0]);
      return -1;
    }
    uname = username;
    upass = password;
  }

  if((sch = OpenSCManager(0, 0, SC_MANAGER_CREATE_SERVICE)) == 0) {
    last_err = GetLastError();
    fprintf(stderr, "%s: Failed to open service database.\n", argv[0]);
    fprintf(stderr, "LastError was %d, %s\n",last_err, error_to_string(last_err));
    exit(-1);
  }

  hserv = CreateService(sch, global_service_name, global_service_name, 
		SERVICE_ALL_ACCESS, SERVICE_WIN32_OWN_PROCESS,
		SERVICE_AUTO_START, SERVICE_ERROR_NORMAL,
		pathname, 0, 0,
		CYGWIN_SERVICE_SZDEPENDENCIES,
		uname, upass);
  
  if(hserv == 0) {
    last_err = GetLastError();
    fprintf(stderr, "%s: Failed to create the service %s.\n", argv[0], global_service_name);
    fprintf(stderr, "LastError was %d, %s\n",last_err, error_to_string(last_err));
    return -1;
  }

  CloseServiceHandle(hserv);
  CloseServiceHandle(sch);

  printf("%s: Service %s installed successfully.\n", argv[0], global_service_name);
  return 0;
}
  
int main(int argc, char *argv[])
{
  SERVICE_TABLE_ENTRY serviceTable[] = {
    { CYGWIN_SERVICE_NAME, (LPSERVICE_MAIN_FUNCTION)service_main},
    { 0, 0}
  };

  /*
   * Check to see if we are being asked to install or 
   * de-install ourselves.
   */

  if(argc > 1) {
    if((argv[1][0] == '-') || (argv[1][0] == '/')) {
      if(strcasecmp(&argv[1][1], "install") == 0) {
	global_log_to_stderr_flag = TRUE;
        exit(service_install(argc, argv));
      } else if (strcasecmp(&argv[1][1], "delete") == 0) {
	global_log_to_stderr_flag = TRUE;
        exit(service_delete(argc, argv));
      } else if ((strcasecmp(&argv[1][1], "help") == 0) || 
                 (argv[1][1] == '?')) {
        global_log_to_stderr_flag = TRUE;
        exit(do_service_usage(argc, argv));
      } else if (strcasecmp(&argv[1][1], "debug") == 0) {
	global_debug_flag = TRUE;
	global_log_to_stderr_flag = TRUE;
	service_main(argc, argv);
	exit(0);
      }
    }
  }

  /* Try to start the service. */
  if(!StartServiceCtrlDispatcher(serviceTable)) {
    exit(do_service_usage(argc, argv));  
  }
  return 0;
}
#endif /* _WIN32 && __CYGWIN32__ */

