/*********************************************************************
**
**     File name:               ssh_presetup.c
**
**                              Copyright 1998 Tadayoshi Kohno.
**				All rights reserved.
**                              See the LICENSE file.
**
**     Purpose:                 handles pre-connection setup
**
**     Author/Date:             Tadayoshi Kohno, 21 February 1998
**
**     Notes:
**	The function(s) in this file are used to setup the client
**	(and perhaps someday the server).  These functions
**	should be called before any other ssh_* function.
**
**     Functions:
**	ssh_presetup_client	pre-setup for client side of ssh
**
*********************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/libssh/RCS/ssh_presetup.c,v 3.9 1998/05/02 03:35:24 kohno Exp $";
#endif

#include "ssh_presetup.h"

/******************************************************************************
**
**     File/Function name:	ssh_presetup_client
**
**     Purpose:			handles initialization stuff for client
**
**     Preconditions:		none
**
**     Parameters:		ssh_info	info about to-be connection
**				debug_func	function to print debug msg
**				debug_mode	SSH_{YES,NO}_DEBUG
**
**     Exit (post) conditions:	S_GOOD
**
**     Error conditions:	S_BAD		not used
**				ssh_errno propogated
**
**     Side effects:		debug mode set to debug_mode,
**				debug_func set as function to do debugging
**
**				Eric Young's random number generator seeded
**
**				internal stdout/stderr buffer initialized
**
**     Author/Date:		Tadayoshi Kohno, 21 February 1998
**     Modified:		Tadayoshi Kohno, 14 March 1998
**					(call to random number seeder)
**				Tadayoshi Kohno, 4 April 1998
**					(initialize compress_info)
**
**     References:
**
**     Notes:
**	valid arguements for debug_mode are based off
**	ssh_debug_activate_new.  Namely, SSH_YES_DEBUG and SSH_NO_DEBUG
**	are both valid.
**
**	This function should be called before all other ssh_* 
**	functions.
**
**	The debug_func can be NULL, in which can debugging is turned off
**
******************************************************************************/

int ssh_presetup_client
(
	struct ssh_struct * ssh_info,	/* infomation about to-be connection */

	SSHDebugPrinter debug_func,	/* function to display debug info */
	int debug_mode			/* debug mode */
)
{
	if (ssh_info == (struct ssh_struct *) NULL)
	{
		ssh_errno_set(SSH_ERRNO_NULL_POINTER);
		return(S_BAD);
	}

/*
**	Make sure Eric Young's SSLeay libraries are using the "correct"
**	malloc (moved from ssh_connect_client())
*/
	CRYPTO_malloc_init();

/*
**	initialize debugger
*/
	ssh_debug_activate_new(ssh_info, debug_mode);
	ssh_debug_setfunc_new(&(ssh_info->debug_info), debug_func);

/*
**	initialize random number
*/
	if (ssh_random_seed_simple() == S_BAD)
	{
		/* propogate ssh_errno */
		return(S_BAD);
	}
	ssh_debugger_new(&(ssh_info->debug_info),
		"seeded random number generator",
		"ssh_presetup_client");

/*
**	initialize compression (initially no compression);
*/
	if (ssh_compression_none(ssh_info) == S_BAD)
	{
		/* propagate ssh_errno */
		return(S_BAD);
	}

	return(S_GOOD);
}

