/******************************************************************************
**
**     File/Function name:	unix_ssh
**				
**				Copyright 1998 Tadayoshi Kohno.
**				All rights reserved.
**				See the LICENSE file.
**
**     Purpose:			demonstrate the use of libSSH on Unix
**				platforms
**
**     Preconditions:		libSSH compiled, makefile points to it
**
**     Author/Date:		Tadayoshi Kohno, 15 Feb 1998
**
**     References:
**
**     Notes:
**	timeouts	(SIGALRM)
**	interrupts	(SIGPIPE, SIGINT, SIGALRM)
**	disconnections	(??)
**	...
**	perror
**
******************************************************************************/

#ifndef lint
static char *RCSid="$Header: /home/kohno/LibSSH/libssh.0.0.1beta/unix_ssh/RCS/unix_ssh.c,v 1.24 1998/05/21 16:22:01 kohno Exp $";
#endif

#include <netinet/in.h>
#include <netdb.h>
#include <sys/socket.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/wait.h>
#include <signal.h>
#include <unistd.h>
#include <termios.h>

#include <strings.h>
#include <string.h>
#include <ctype.h>

#include <stdlib.h>
#include <stdio.h>

#include <ssh.h>

#include "unbuf_io.h"

#define DEBUG

#define MAX_SERVER_NAME		100		/* length of server name */
#define FIELD_LENGTHS		500		/* login, password lengths */
#define ANY_PROTOCOL		0		/* protocol family */

/*
**	some error conditions
*/
#define ERR_SOCKET		-1		/* error with socket() */
#define ERR_GETHBNAME		-2		/* error with gethostbyname() */
#define ERR_CONNECT		-3		/* error with connect() */
#define ERR_ARGC		-4		/* wrong number of arguments */
#define ERR_SERVLEN		-5		/* server lname length wrong */
#define ERR_LOGINREAD		-6		/* error reading login/passwd */
#define ERR_PREFREAD		-7		/* error reading preferences */


/*
**	connect to server
*/
int connect_server(char * server, int port);

/*
**	print debugging information
*/
int print_debug_msg(const char * category, const char * body);

/*
**	get login and password
*/
int get_login_passwd(char * login, char * passwd, int lengths);
int get_preferences(uint8_t * cipher_type, int * auth_mode,
	char * identity_file, int identity_length);

/*
**	handle interactive stuff
*/
int interactive_session(int sockfd, struct ssh_struct * ssh_info);


/*
**	socket to server
*/
int SockFD;

/*
**	info on SSH connection
*/
struct ssh_struct SSHData;


/******************************************************************************
**
**     File/Function name:	main
**
**     Purpose:			driver to handle ssh connection to a server
**
**     Preconditions:		none
**
**     Parameters:		server		server to connect to
**
**     Exit (post) conditions:	0		success
**
**     Error conditions:	< 0		error
**				#defines above describe these better
**
**     Side effects:		connected to server using ssh protocol
**
**     Author/Date:		Tadayoshi Kohno, 15 Feb 1998
**
**     References:
**	Please see the libssh package
**
**     Notes:
**	This is a tester program for libSSH on Unix boxes.  Doesn't really
**	do much, but should (eventually) be able to test all the desired
**	features of libSSH.
**
**	Right now doesn't really do much.  Basically connects with the server
**	then starts the ssh connection itself.
**
******************************************************************************/

int main
(
	int argc,
	char * argv[]
)
{
	char server_name[MAX_SERVER_NAME];	/* server to connect to */
	char ssh_error_msg[SSH_ERRNO_LEN];	/* error messages */

	char passwd[FIELD_LENGTHS];		/* user password */
	char login[FIELD_LENGTHS];		/* user login */

	uint8_t cipher_type;			/* choice of cipher */
	int auth_mode;				/* choice of auth modes */
	char identity_file[FIELD_LENGTHS];	/* identity file */

	int error_code;				/* error code for int. ses */
	int return_value;			/* misc return value */

/*
**	first grab any useful information (server, ...)
*/
	if (argc <= 1)
	{
		fprintf(stderr, "Usage: %s <Server Machine> [parameters]\n",
			argv[0]);
		return(ERR_ARGC);
	}

	if (strlen(argv[1]) > MAX_SERVER_NAME - 1)
	{
		fprintf(stderr, "Server name too long\n");
		return(ERR_SERVLEN);
	}

	(void) strcpy(server_name, argv[1]);

/*
**	get login and password
*/
	if (get_login_passwd(login, passwd, FIELD_LENGTHS))
	{
		fprintf(stderr, "Error reading login, passwd\n");
		return(ERR_LOGINREAD);
	}
	
/*
**	get preferences if we have another parameter
*/
	if (argc >= 3)
	{
		if (get_preferences(&cipher_type, &auth_mode, identity_file,
			FIELD_LENGTHS))
		{
			fprintf(stderr, "Error reading preferences\n");
			return(ERR_PREFREAD);
		}
	} else
	{
		cipher_type = SSH_CIPHER_IDEA;
		auth_mode = SSH_AUTH_PASSWORD;
		strcpy(identity_file, "");
	}

/*
**	connect to server
*/
	/* xxx catch SIGINT and alarm */
	if ((SockFD = connect_server(server_name, SSH_PORT)) <= 0)
	{
		fprintf(stderr, "Error connecting to %s on port %d\n",
			server_name, SSH_PORT);
		return(SockFD);
	}


	fprintf(stderr, "connected to server step 1\n");

/*
**	initialze ssh connection
*/
/*
	if (ssh_presetup_client(&SSHData, (SSHDebugPrinter) print_debug_msg,
*/
	if (ssh_presetup_client(&SSHData, print_debug_msg,
		SSH_YES_DEBUG))
	{
		ssh_errno_to_str(ssh_error_msg);

		fprintf(stderr, "error opening SSH connection: %s\n",
			ssh_error_msg);
		
		close(SockFD);
		return(ssh_errno_get());
	}

/*
**	start ssh connection
*/
	if (ssh_connect_client(SockFD, &SSHData, "libSSHSampleClient-0.0.0",
		login, auth_mode, passwd, cipher_type, identity_file))
	{
		ssh_errno_to_str(ssh_error_msg);

		fprintf(stderr, "error opening SSH connection: %s\n",
			ssh_error_msg);
		
		close(SockFD);
		return(ssh_errno_get());
	}
/*
**	wipe out the passwd
*/
	bzero((void *) passwd, FIELD_LENGTHS);

	fprintf(stderr, "connected to server step 2\n");
/*
**	request compression
*/
	if ((return_value = ssh_request_compression(SockFD, &SSHData, 6))
		== S_BAD)
	{
		ssh_errno_to_str(ssh_error_msg);

		fprintf(stderr, "error requesting compression: %s\n",
			ssh_error_msg);
			
		close(SockFD);
		return(ssh_errno_get());
	}

	if (return_value == SSH_COMPRESSION_STARTED)
	{
		printf("COMPRESSION:  started compression\n");
	} else if (return_value == SSH_COMPRESSION_NOT_STARTED)
	{
		printf("COMPRESSION:  NOT started compression\n");
	} else
	{
		printf("COMPRESSION:  weird return value\n");
	}

/*
**	temporarily turn off debugging
*/
	ssh_debug_activate_new(&SSHData, SSH_NO_DEBUG);
	fprintf(stderr, "debugging turned off for pty request\n");


/*
**	request a pty
*/
	if (ssh_request_pty(SockFD, &SSHData, "vt100", 24, 80,
		0, 0, (uint8_t *) "\0", 1))
	{
		ssh_errno_to_str(ssh_error_msg);

		fprintf(stderr, "error requesting pty: %s\n",
			ssh_error_msg);
			
		close(SockFD);
		return(ssh_errno_get());
	}

/*
**	turn debugging back on
*/
	ssh_debug_activate_new(&SSHData, SSH_YES_DEBUG);
	fprintf(stderr, "debugging turned on for request shell\n");


/*
**	request a shell
*/
	if (ssh_request_exec_shell(SockFD, &SSHData))
	{
		ssh_errno_to_str(ssh_error_msg);

		fprintf(stderr, "error requesting shell: %s\n",
			ssh_error_msg);

		close(SockFD);
		return(ssh_errno_get());
	}

	fprintf(stderr, "connected to server\n");

/*
**	connected, now switch to an interactive session
*/
	if ((error_code = interactive_session(SockFD, &SSHData)))
	{
		return(error_code);
	}
	
/*
**	cleanup after connection.  probably a call to finalize
*/
	if (ssh_final_client(&SSHData))
	{
		ssh_errno_to_str(ssh_error_msg);

		fprintf(stderr, "error finalizing connection: %s\n",
			ssh_error_msg);

		close(SockFD);
		return(ssh_errno_get());
	}

/*
**	close SSH connection?  how are we going to handle this?? xxx
*/

/*
**	close the connection
*/
	(void) close(SockFD);

	return(0);
}


/******************************************************************************
**
**     File/Function name:	connect_server
**
**     Purpose:			connect to server (just connect, no SSH)
**
**     Preconditions:		none
**
**     Parameters:		server		server name or ip address
**				port		port on server to connect to
**
**     Exit (post) conditions:	sockfd		socket file descriptor
**
**     Error conditions:	< 0
**				ERR_SOCKET	error opening socket
**				ERR_GETHBNAME	error with gethostbyname()
**				ERR_CONNECT	error connection to server
**
**     Side effects:		client connects to server on specified port
**
**     Author/Date:		Tadayoshi Kohno, 15 Feb 1998
**
**     Notes:
**	After calling this function, the client should be able to
**	communicate with the server using the returned socket.
**
******************************************************************************/

int connect_server
(
	char * server,		/* name/addr of server to connect to */
	int port		/* port to connect to */
)
{
	struct sockaddr_in serv_addr;	/* server's socket address */
	struct hostent * serv_host;	/* information about the server */

	int sockfd;			/* file descriptor of socket opened */

/*
** 	open sockfd to be used to talk to server
*/
	if ((sockfd = socket(AF_INET, SOCK_STREAM, ANY_PROTOCOL)) == -1)
	{
		perror("error connecting with server");
		return(ERR_SOCKET);
	}

/*
**	setup serv_addr for connect()
*/
	bzero(&serv_addr, (int) sizeof(serv_addr));
	serv_addr.sin_family = AF_INET;
	serv_addr.sin_port = htons(port);

	if (isalpha(server[0]))
	{
		fprintf(stderr, "connecting to hostname %s\n", server);
		if ((serv_host = gethostbyname(server))
			== (struct hostent *) NULL)
		{
			fprintf(stderr, "error with gethostbyname\n");
			return(ERR_GETHBNAME);
		}

		bcopy(serv_host->h_addr, &serv_addr.sin_addr,
			serv_host->h_length);
		
	} else
	{
		fprintf(stderr, "connecting to IP address %s\n", server);
		if((serv_addr.sin_addr.s_addr = inet_addr(server))
			== INADDR_NONE)
		{
			fprintf(stderr, "error converting ip addr\n");
			return(ERR_GETHBNAME);
		}
	}


/*
**	connect() to server
*/
	/* xxx alarm/timeout */
	if (connect(sockfd, (struct sockaddr *) &serv_addr,
		(int) sizeof(serv_addr)) == -1)
	{
		perror("error connecting to server");
		return(ERR_CONNECT);
	}
	/* xxx unset alarm */

	return(sockfd);
}

/******************************************************************************
**
**     File/Function name:	xxx
**
**     Purpose:			xxx
**
**     Preconditions:		xxx
**
**     Parameters:		xxx
**
**     Exit (post) conditions:	xxx
**
**     Error conditions:	xxx
**
**     Side effects:		xxx
**
**     Author/Date:		xxx
**
**     References:
**
**     Notes:
**
******************************************************************************/

int print_debug_msg
(
	const char * category,
	const char * body
)
{
	fprintf(stderr, "DEBUG: %s -- %s (libSSH)\n", category, body);
	fflush(stderr);
	return(0);
}


/******************************************************************************
**
**     File/Function name:	get_login_passwd
**
**     Purpose:			get the login and password/passphrase from user
**
**     Preconditions:		none
**
**     Parameters:		login		location to store login
**				passwd		location to store passwd
**				lengths		lengths of login, password strs
**
**     Exit (post) conditions:	0
**
**     Error conditions:	-1
**
**     Side effects:		user prompted for login and password.
**				login read (echo'd) and password read
**				(not echo'd).
**
**     Author/Date:		Tadayoshi Kohno, 15 Feb 1998
**
**     References:
**
**     Notes:
**
******************************************************************************/

int get_login_passwd
(
	char * login,		/* login to get */
	char * passwd,		/* password to read */
	int lengths		/* lengths of login, password */
)
{
	int index;

/*
**	read the login
*/
	printf("enter login: ");
	if (fgets(login, lengths, stdin) == (char *) NULL)
	{
		return(-1);
	}

/*
**	make sure we read a valid string (not to short of long) and
**	chop off the newline character
*/
	if (strlen(login) < 1 || login[strlen(login) - 1] != '\n')
	{
		return(-1);
	}
	login[strlen(login) - 1] = 0;

/*
**	read the password (no echo)
*/
	printf("enter password/passphrase: ");
	for (index = 0; index < lengths - 1; index++)
	{
		passwd[index] = unbuf_noecho_getc();
		if (passwd[index] == '\n')
		{
			passwd[index] = 0;

			putchar('\n');
			return(0);
		}
	}

	putchar('\n');
	return(-1);
}

/******************************************************************************
**
**     File/Function name:	get_preferences
**
**     Purpose:			get user preferences (cipher type, ...)
**
**     Preconditions:		none
**
**     Parameters:		*cipher_type	cipher type to use
**				*auth_mode	authentication mode to use
**				*identity_file	location of identity file
**
**				identity_length	length of *identity_file string
**
**     Exit (post) conditions:	0
**
**     Error conditions:	-1
**
**     Side effects:		*cipher_type, *auth_mode and identity_file set
**
**     Author/Date:		Tadayoshi Kohno, 15 Feb 1998
**
**     Notes:
**	the resulting string in identity_file doesn't matter if we're
**	using SSH_AUTH_PASSWORD authentication.
**
******************************************************************************/

int get_preferences
(
	uint8_t * cipher_type,	/* cipher type to use */
	int * auth_mode,	/* authentication method to use */
	char * identity_file,	/* identity file to use */
	int identity_length	/* length of identity_file string */
)
{
	int choice;		/* cipher, auth mode choice */

	printf("please select cipher type: ");
	printf("SSH_CIPHER_NONE (%d), ", SSH_CIPHER_NONE);
	printf("SSH_CIPHER_IDEA (%d), ", SSH_CIPHER_IDEA);
	printf("SSH_CIPHER_BLOWFISH (%d), ", SSH_CIPHER_BLOWFISH);
	printf("SSH_CIPHER_DES (%d), ", SSH_CIPHER_DES);
	printf("SSH_CIPHER_3DES (%d): ", SSH_CIPHER_3DES);
	scanf("%d", &choice);

	switch (choice)
	{
		case SSH_CIPHER_NONE:
			*cipher_type = SSH_CIPHER_NONE;
			break;
		case SSH_CIPHER_DES:
			*cipher_type = SSH_CIPHER_DES;
			break;
		case SSH_CIPHER_3DES:
			*cipher_type = SSH_CIPHER_3DES;
			break;
		case SSH_CIPHER_IDEA:
			*cipher_type = SSH_CIPHER_IDEA;
			break;
		case SSH_CIPHER_BLOWFISH:
			*cipher_type = SSH_CIPHER_BLOWFISH;
			break;
		default:
			*cipher_type = SSH_CIPHER_IDEA;
	}

	printf("please select authentication method: ");
	printf("SSH_AUTH_PASSWORD (%d), ", SSH_AUTH_PASSWORD);
	printf("SSH_AUTH_RSA (%d): ", SSH_AUTH_RSA);
	scanf("%d", &choice);

	switch (choice)
	{
		case SSH_AUTH_RSA:
			*auth_mode = SSH_AUTH_RSA;

			(void) fgetc(stdin);

			printf("enter dentity file: ");
			if (fgets(identity_file, identity_length,
				stdin) == (char *) NULL)
			{
				return(-1);
			}

			/* chop off newline and make sure string valid */
			if (strlen(identity_file) < 1
				|| identity_file[strlen(identity_file) - 1]
				!= '\n')
			{
				return(-1);
			}
			identity_file[strlen(identity_file) - 1] = 0;

			break;

		case SSH_AUTH_PASSWORD:
		default:
			*auth_mode = SSH_AUTH_PASSWORD;
			strcpy(identity_file, "");
			break;
	}

	printf("cipher_type: %d, authmode: %d, identity file: %s.\n",
		*cipher_type, *auth_mode, identity_file);

	return(0);
}

/******************************************************************************
**
**     File/Function name:	interactive_session
**
**     Purpose:			perform the interactive ssh session
**
**     Preconditions:		ssh connection established
**
**     Parameters:		sockfd		socket to server
**				ssh_info	current ssh connection info
**
**     Exit (post) conditions:	0
**
**     Error conditions:	-1
**
**     Side effects:		interactive ssh session:
**					send stdin data to server
**					read stdout/stderr data from server
**
**     Author/Date:		Tadayoshi Kohno, 15 Feb 1998
**
**     References:
**
**     Notes:
**	Currently yucky.
**
**	Note the need to reset the timers after the calls to select (for
**	linux boxes at least)
**
**	This should provide a quick and dirty example of how to use libSSH
**	
**	I think I really need to think about how to finalize a connection xxx
**
******************************************************************************/

int interactive_session
(
	int sockfd,			/* socket to server */
	struct ssh_struct * ssh_info	/* ssh connection information */
)
{
	char ssh_error_msg[SSH_ERRNO_LEN];	/* error messages */
	fd_set read_set_stdin;		/* read set for stdin */
	fd_set read_set_ssh;		/* read set for ssh */

	struct timeval wait_time;	/* time to wait for select() */

	int max_fd_stdin;		/* max file descriptor for stdin */
	int max_fd_ssh;			/* max file descriptor for ssh */

	int ret_val_select;		/* return value from select()  */
	uint32_t ret_val_ssh;		/* return value from ssh_read_... */

	uint8_t buf[SSH_MAX_PACKET];	/* buffer to read from stdin/ssh */

/*
**	initialize our variables for select()
*/
	max_fd_ssh = sockfd + 1;
	max_fd_stdin = STDIN_FILENO + 1;

	FD_ZERO(&read_set_stdin);
	FD_SET(STDIN_FILENO, &read_set_stdin);

	FD_ZERO(&read_set_ssh);
	FD_SET(sockfd, &read_set_ssh);

	wait_time.tv_sec = 0;
	wait_time.tv_usec = 500;

/*
**	start unbuffered input
*/
	start_unbuf_io();

	while (1)
	{
	/*
	**	clear everything.  is this necessary?
	*/
		FD_ZERO(&read_set_stdin);
		FD_SET(STDIN_FILENO, &read_set_stdin);

		FD_ZERO(&read_set_ssh);
		FD_SET(sockfd, &read_set_ssh);

	/*
	**	see if we have any data waiting for us on stdin
	*/
		ret_val_select = select(max_fd_stdin, &read_set_stdin, NULL,
			NULL, &wait_time);
		wait_time.tv_sec = 0;
		wait_time.tv_usec = 500;

		if (ret_val_select == -1)
		{
			return(-1);
		}

	/*
	**	grab the waiting data and send it to the server
	*/
		if (ret_val_select)
		{
		/*
			buf[0] = noecho_getc();
			buf[1] = 0;
			if (ssh_write_stdin(sockfd, ssh_info, buf) != S_GOOD)
			{
		*/
			buf[0] = noecho_getc();
			if (ssh_write_stdin_n(sockfd, ssh_info, buf, 1)
				!= S_GOOD)
			{
				ssh_errno_to_str(ssh_error_msg);

				fprintf(stderr, "error writing stdin: %s\n",
					ssh_error_msg);
		
				return(ssh_errno_get());
			}
		}

	/*
	**	see if there is any stdout/stderr data from the server
	*/
		ret_val_select = select(max_fd_ssh, &read_set_ssh, NULL, NULL,
			&wait_time);
		wait_time.tv_sec = 0;
		wait_time.tv_usec = 500;

		if (ret_val_select == -1)
		{
			return(-1);
		}

	/*
	**	grab the data and display it
	*/
		if (ret_val_select)
		{
			switch (ssh_read_interactive(sockfd, ssh_info,
				(uint8_t *) buf, &ret_val_ssh, 
				SSH_MAX_PACKET))
			{
				case SSH_SMSG_STDOUT_DATA:
				case SSH_SMSG_STDERR_DATA:

					buf[ret_val_ssh] = 0;

					fputs(buf, stdout);
					fflush(stdout);

					break;
					
				case SSH_SMSG_EXITSTATUS:
					(void) ssh_disconnect_client_confirm(
						sockfd, ssh_info);
					return(0);

				case SSH_MSG_DISCONNECT:
					return(0);

				case SSH_MSG_ERROR:
					ssh_errno_to_str(ssh_error_msg);
	
					fprintf(stderr, "error reading: %s\n",
					ssh_error_msg);
		
					return(-1);

				case SSH_MSG_NOTYPE:
				default:
					/*
					** no (understood) message
					*/
					break;
					
			}
			/*
			if ((ret_val_ssh = ssh_read_merge(sockfd, ssh_info,
				(uint8_t *) buf)) == -1)
			{
				if (ssh_errno_get() == SSH_ERRNO_DISCONNECTED)
				{
					(void) ssh_disconnect_client_confirm(
						sockfd, ssh_info);
					return(0);
				}
			}

			null terminate it before we try to display it
			buf[ret_val_ssh] = 0;

			fputs(buf, stdout);
			fflush(stdout);
			*/
		}

	}

	return(0);
}

