/*
-- ----------------------------------------------------------------------------
--
--  Object name : linkio.c
--  Revision    : 1.0
--
--  Copyright (c) Inmos Ltd, 1987.
--  All Rights Reserved.
--
--  DESCRIPTION
--    TDS Server  INMOS B004 Link Adaptor Driver
--
--  NOTES
--  Some of these routines are provided in assembler code in quickio.c.
--  HISTORY
--     ibm-pc   03-Apr-86  CREATION.
--     ibm-pc   17-Feb-88  RJO - Addition of new Timeout facility to the
--              simple byte operations.Server13 upgrade.
--     ibm-pc   03-Mar-88  RJO Added new booting flags reset etc.
--     ibm-pc   04-Mar-88  RJO Added FindBoard feature - quite useful that one.
-- ----------------------------------------------------------------------------
*/
#include "inmos.h"
#include "srvconst.h"
#include <stdio.h>
#include <linux/b004.h>
#include <sys/stat.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include "setjmp.h"
#include <linux/types.h>
#include <linux/termios.h>
#define T_PEEK    1
#define T_POKE    0
#define TESTPATTERN 0x1234567
#define BOOT_BUFFER_LENGTH RECORD_LENGTH*4

/*
-- ----------------------------------------------------------------------------
-- Address Of IBM-PC Machine specific byte.
-- ----------------------------------------------------------------------------
*/

#define ID_ADDRESS                0xffff000e
#define PC_ID                     -1
/*
-- ----------------------------------------------------------------------------
-- Peek And Poke Constants.
-- ----------------------------------------------------------------------------
*/

#define MOSTNEG_INT 0x80000000l
#define T_POKE 0
#define T_PEEK 1
/*
-- ----------------------------------------------------------------------------
-- External variables
-- ----------------------------------------------------------------------------
*/
extern BOOL Running;                     /* Defined in Server.c module   */
extern BOOL BootFlag;                    /* defined in server.c          */
extern BOOL InitialBootFlag;             /* defined in server.c          */
extern BOOL ReBoot;                      /* defined in server.c          */
extern BOOL AnalyseFlag;                 /* defined in server.c          */
int Link;
extern struct termios OrgMode, CurMode;

/*
-- ----------------------------------------------------------------------------
-- void InitRoot - Set up all the Link addresses.
--
-- Input Parameters:
--  None.
-- Output Parameters:
--  None.
-- ----------------------------------------------------------------------------
*/
void InitRoot ()
{
  ioctl(0, TCGETS, &OrgMode);
  ioctl(0, TCGETS, &CurMode);
  CurMode.c_iflag &= ~ICRNL;
  CurMode.c_lflag &= ~(ICANON | ECHO);
  CurMode.c_cc[VTIME] = 0;
}
/*
-- ----------------------------------------------------------------------------
-- int LinkInTest - Test if we have a byte on the Link.
-- Return TRUE if a byte is available on the link, FALSE otherwise.
--
-- Input Parameters:
--  None.
-- Output Parameters:
--  (BOOL) Byte or not !
-- ----------------------------------------------------------------------------
*/
int LinkInTest ()
{
  return ( ioctl(Link, B004READABLE, 0 ) );
}
/*
-- ----------------------------------------------------------------------------
-- int ByteFromLink
-- Read a byte from the link adaptor.
--
-- Input Parameters:
--  None.
--
-- Output Parameters:
--  (BOOL) Byte or not !
--
-- ----------------------------------------------------------------------------
*/
int ByteFromLink ()
{
  char Byte;
  read(Link, &Byte, 1);
  return ( (int) Byte );
}
/*
-- ----------------------------------------------------------------------------
-- void ByteToLink
-- Write a byte to the link adaptor.
--
-- Input Parameters:
--   (int) ch - Byte to send
--
-- Output Parameters:
--
-- ----------------------------------------------------------------------------
*/
void ByteToLink (ch)
int ch;
{
  char Byte;
  Byte  = (char) ch;
  write(Link, &Byte, 1);
}
/*
-- ----------------------------------------------------------------------------
-- int ByteToLinkT
-- Write a byte to the link adaptor within a specified time, or timeout.
-- Return TRUE if the byte was NOT written, FALSE if the byte was written.
--
-- Input Parameters:
--  (int) ch - Byte to send
-- Output Parameters:
--  (BOOL) Byte written or not !
-- ----------------------------------------------------------------------------
*/
int ByteToLinkT (ch)
int ch;
{
  char Byte;
  Byte = (char) ch;
  if ( write(Link, &Byte, 1) != 1)
    return TRUE;
  else
    return FALSE;
}
/*
-- ----------------------------------------------------------------------------
-- int ByteFromLinkT - Timed byte from Link.
--
-- Input Parameters:
--    (int *) Ch - Pointer to byte obtained from the Link.
-- Output Parameters:
--    (BOOL) Return result.
-- ----------------------------------------------------------------------------
*/
BOOL ByteFromLinkT (Ch)
int *Ch;
{
  char Byte;
  int  Result;
  Result = read(Link, &Byte, 1);
  *Ch = (int) Byte;
  if (Result != 1)
    return( TRUE );
  else
    return( FALSE );
}
/*
-- ----------------------------------------------------------------------------
-- int ErrorTest
-- Test if an Error is present in the transputer system.
--
-- Input Parameters:
--  None
-- Output Parameters:
--  (BOOL) Return TRUE or FALSE.
-- ----------------------------------------------------------------------------
*/
int ErrorTest ()
{
  return (  ioctl( Link, B004ERROR, 0)  );
}
/*
-- ----------------------------------------------------------------------------
-- void ResetRoot - Reset the root transputer.
-- ----------------------------------------------------------------------------
*/
void ResetRoot ()
{
  ioctl( Link, B004RESET, 0 );
}
/*
-- ----------------------------------------------------------------------------
-- void ResetAnalyseRoot - Reset the root transputer into Analyse mode.
-- ----------------------------------------------------------------------------
*/
void ResetAnalyseRoot ()
{
  ioctl( Link, B004ANALYSE, 0 );
}
/*
-- ----------------------------------------------------------------------------
-- int WordFromLink - Read a word from the link.
--
-- Input Parameters:
--  None.
-- Output Parameters:
--  (int) The word from the Link
--
-- NOTES
-- This routine is host- and transputer- word-length dependent.
-- Here we assume a host int length of two bytes, and a transputer int length
-- of four bytes.
-- ----------------------------------------------------------------------------
*/
int WordFromLink ()
{
  short Word, Dummy;
  read(Link, &Word, 2);
  read(Link, &Dummy, 2);
  return( (int) Word );
}
/*
-- ----------------------------------------------------------------------------
-- void WordToLink -
-- Write a word to the link, least significant byte first.
--
-- Input Parameters:
--  (int) w - Word to be wriiten.
-- Output Parameters:
--  None.
--
-- NOTES
-- This routine is host- and transputer- word-length dependent.
-- Here we assume a host int length of two bytes, and a transputer int length
-- of four bytes.
-- ----------------------------------------------------------------------------
*/
void WordToLink (w)
int w;
{
  write(Link, &w, 4);
}
/*
-- ----------------------------------------------------------------------------
-- void SliceFromLink -  Read a slice from the link.
--
-- Input Parameters:
--  (int *) Length - Length of slice
--  (BYTE *) Slice - Slice itself
--
-- Output Parameters:
--  None.
--
-- NOTES
-- The slice consists of a four-byte length followed by a sequence of bytes.
-- ----------------------------------------------------------------------------
*/
void SliceFromLink (Length, Slice)
int *Length;
BYTE *Slice;
{
  int  real_len;
  real_len = WordFromLink();
  *Length = real_len;
#ifdef REV_A_FIX
  if (real_len == 1) real_len = 2;
#endif
  read(Link, Slice, real_len);
}
/*
-- ----------------------------------------------------------------------------
-- int TruncSliceFromLink
-- Read a slice from the link whose maximum length is MaxLength.
--
-- Input Parameters:
--  (int) MaxLength - Max length of slice
--  (BYTE *) Str    - The slice string
--
-- Output Parameters:
--  Return the length.
--
-- NOTES
-- If the length Read in is longer than this, the remainder of the slice is
-- Read in, but ignored. Return number of bytes Read in.
-- ----------------------------------------------------------------------------
*/
int TruncSliceFromLink (MaxLength, Str)
int MaxLength;
BYTE * Str;
{
  int length;
  int RealLen, i;
  char Dummy;
  length = RealLen = WordFromLink();
#ifdef REV_A_FIX
  if (RealLen == 1) RealLen = 2;
#endif
  if (RealLen > MaxLength)
    {
      read(Link, Str, MaxLength);
      for (i = MaxLength; i < RealLen; i++)
	read(Link, &Dummy, 1);
      length = MaxLength;
    }
  else
    {
      read(Link, Str, RealLen);
    }
  return (length);
}
/*
-- ----------------------------------------------------------------------------
-- void SliceToLink - Write a Slice to the link.
--
-- Input Parameters:
--  (int) Length - Length of the Slice.
--  (BYTE) *Slice - The Slice itself.
--
-- Output Parameters:
--  Return the Length.
--
-- NOTES
-- The Slice consists of a four-byte Length followed by a sequence of bytes.
-- ----------------------------------------------------------------------------
*/
void SliceToLink (Length, Slice)
int Length;
BYTE *Slice;
{
  register int i, RealLen;
  RealLen = Length;
  WordToLink (RealLen);
#ifdef REV_A_FIX
  if (RealLen == 1) RealLen = 2;
#endif
  write(Link, Slice, RealLen);
}
/*
-- ----------------------------------------------------------------------------
-- void LongWordToLink - Write a long word to the link.
--
-- Input Parameters:
--  (int) w - Word to send to Link.
--
-- Output Parameters:
--  None.
--
-- NOTES
-- This routine is host- and transputer- word-length dependent.
-- Here we assume a host long int length of at least four bytes,
-- and a transputer int length of four bytes.
-- ----------------------------------------------------------------------------
*/
void LongWordToLink (w)
long int w;
{
  int LongWord;
  LongWord = w;
  write(Link, &w, 4);
}
/*
-- ----------------------------------------------------------------------------
-- long LongWordFromLink
--
-- Input Parameters:
--  None.
--
-- Output Parameters:
--  Return the Long Word
--
-- NOTES
-- This routine is host- and transputer- word-length dependent.
-- Here we assume a host long int length of at least four bytes,
-- and a transputer int length of four bytes.
-- ----------------------------------------------------------------------------
*/
long LongWordFromLink ()
{
  int LongWord;
  read(Link, &LongWord, 4);
  return( (long) LongWord );
}
/*
-- ----------------------------------------------------------------------------
-- BOOL LongWordFromLinkT
--
-- Input Parameters:
--     (long int) Lval - Long word to be returned.
-- Output Parameters:
--     (BOOL) Return result.
-- ----------------------------------------------------------------------------
*/
BOOL LongWordFromLinkT (Lval)
long int *Lval;
{
  int LongWord;
  int Result;
  Result = read(Link, &LongWord, 4);
  *Lval = LongWord;
  if (Result == 4)
    return(FALSE);
  else
    return(TRUE);
}
/*
-- ----------------------------------------------------------------------------
-- BOOL LongWordToLinkT
--
-- Input Parameters:
--     (long int) W - Long word to be sent.
-- Output Parameters:
--     (BOOL) Return result.
-- ----------------------------------------------------------------------------
*/
BOOL LongWordToLinkT (W)
long int W;
{
  int LongWord;
  LongWord = W;

  if ( write(Link, &LongWord, 4) != 4 )
    return( TRUE );
  else
    return( FALSE );
}
/*
-- ----------------------------------------------------------------------------
-- void SafeByteToLink
--
-- Input Parameters:
--  (int) b - The 'safe byte' to send.
--
-- Output Parameters:
--   None.
-- NOTES
-- ----------------------------------------------------------------------------
*/
void SafeByteToLink ( b )
int b;
{
  ByteToLink ( b );
#ifdef REV_A_FIX
  ByteToLink ( b );
#endif
}
/*
-- ----------------------------------------------------------------------------
-- int SafeByteFromLink
--
-- Input Parameters:
--   None.
--
-- Output Parameters:
--   Return the Safe Byte.
-- ----------------------------------------------------------------------------
*/
int SafeByteFromLink ()
{
  int B, Padding;
  B = ByteFromLink ();
#ifdef REV_A_FIX
  Padding = ByteFromLink ();
#endif
  return (B);
}
/*
-- ----------------------------------------------------------------------------
-- void PokeTransputerWord
--
-- Input Parameters:
--   (long int) PokeVal   - Value to poke
--   (long int) PokeThang -
-- Output Parameters:
--   None.
-- ----------------------------------------------------------------------------
*/
void PokeTransputerWord (PokeVal, PokeThang)
long int PokeVal;
long int PokeThang;
{
  ByteToLink (T_POKE);
  LongWordToLink (PokeVal);
  LongWordToLink (PokeThang);
}
/*
-- ----------------------------------------------------------------------------
-- long PeekTransputerWord
--
-- Input Parameters:
--   (int) i - Place to peek.
-- Output Parameters:
--   (long) - Long word from the Link.
-- ----------------------------------------------------------------------------
*/
long PeekTransputerWord (i)
int i;
{
  ByteToLinkT (T_PEEK);
  LongWordToLink (MOSTNEG_INT + (long)(i<<2));
  return (LongWordFromLink ());
}
/*
-- ----------------------------------------------------------------------------
-- long PeekWord(i) - Read the contents of the i'th word above MOSTNEG_INT
--
-- Input Parameters:
--  (int) i - The i'th word.
-- Output Parameters:
--  (int) The long word from the Link
--
-- ----------------------------------------------------------------------------
*/
long PeekWord(i)
int i;
{
  ByteToLink(T_PEEK);
  LongWordToLink(MOSTNEG_INT + (long)(i << 2));
  return(LongWordFromLink());
}
/*
-- ----------------------------------------------------------------------------
-- BOOL FindBoard - See if there is a transputer around.
--
-- Input Parameters:
--     None.
-- Output Parameters:
--     (BOOL) Result - TRUE if there is board, FALSE otherwise.
-- ----------------------------------------------------------------------------
*/
BOOL FindBoard ()
{
  long int N;

  if (ByteToLinkT (T_PEEK))
    return (FALSE);                     /* No Board at all */
  if (LongWordToLinkT (MOSTNEG_INT))
    return (FALSE);                     /* No board        */
  if (LongWordFromLinkT (&N))
    return (FALSE);                     /* No Board        */
  PokeTransputerWord (MOSTNEG_INT, TESTPATTERN);
  if (PeekTransputerWord (MOSTNEG_INT) == TESTPATTERN)
    {
      PokeTransputerWord (MOSTNEG_INT, N);
      return (TRUE);                    /* A Board         */
    };
  return (FALSE);                       /* No Board        */
}
/*
-- ----------------------------------------------------------------------------
-- int BootRoot
-- Boot the root transputer, ie. Reset the root transputer, open the boot file
-- and Write this down the link adaptor as a stream of bytes.
--
-- Input Parameters:
--  (BYTE *) BootFileName - The name of the Boot file.
--
-- Output Parameters:
--  int Return result for booting
--
-- ----------------------------------------------------------------------------
*/
int BootRoot (BootFileName)
BYTE *BootFileName;
{
  int Result;
  int BootFile;

  if (BootFlag)
    {
      if (InitialBootFlag || ReBoot)
        {
          if (ReBoot || AnalyseFlag)
            ResetAnalyseRoot();
          else
            ResetRoot();
        }

      if (FindBoard())                /* Is there a board there ??? */
        {
          PeekBaseMemory();

          if ((BootFile = open (BootFileName, O_RDWR )) >= 1 )
            {
              register int count = 0;
              BYTE FileBuffer [BOOT_BUFFER_LENGTH];
              BOOL DoBoot = TRUE;

              while (DoBoot)
                {
                  count = read (  BootFile, FileBuffer, BOOT_BUFFER_LENGTH );
                  if (count > 0)
                    {
                      DoBoot =  write( Link, FileBuffer, count ) ;
                      if ( DoBoot != count)
                        Result = F_BOOT_TIMEOUT;
                    }
                  else
                    {
                      DoBoot = FALSE;
                      if (close (BootFile) >= 0 )
                        Result = F_OK;
                      else
                        Result = F_CANNOT_CLOSE_FILE;
                    }
                }
            }
          else
              Result = F_NO_SUCH_FILE;
        }
      else
        Result = T_NO_BOARD;
    }
  else
    Result = T_OK;
  return (Result);
}
