/*
 * hostname	This file contains an implementation of the command
 *		that either displays or sets a system's host (and/or
 *		domain) name.  If no name is given on the command
 *		line, this program will look for it in a file called
 *		"/etc/HOSTNAME".  This is nice for multiple machines
 *		on which only the /etc directory contains the system-
 *		dependent info files.  In addition to setting the
 *		specified host name, this program will also set the
 *		machine's main IP address.
 *
 * Usage:	hostname [-finvS] [name]
 *
 * Version:	@(#)hostname.c	2.03	05/24/93
 *
 * Author:	Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 */
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <netinet/in.h>
#include <errno.h>
#include <netdb.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <stdio.h>


#define HOSTNAME	"/etc/HOSTNAME"


char *Version = "@(#) hostname 2.03 (05/25/93)";


int opt_f = 0;				/* set the FULL DOMAIN name	*/
int opt_v = 0;				/* debugging output flag	*/


static void get_host(char *host);
static struct in_addr get_addr(char *host);
static void set_host(struct in_addr in);


static void get_host(host)
char *host;
{
  char buff[128];
  register FILE *fp;
  register char *sp;

  strcpy(host, "(unknown)");
  if ((fp = fopen(HOSTNAME, "r")) == (FILE *)NULL) return;
  if (fgets(buff, 128, fp) == (char *)NULL) {
	(void) fclose(fp);
	return;
  }
  if ((sp = strchr(buff, '\n')) != (char *)NULL) *sp = '\0';
  strcpy(host, buff);
}


static struct in_addr get_addr(host)
char *host;
{
  struct hostent *hp;
  struct in_addr in;

  if ((hp = gethostbyname(host)) == (struct hostent *)NULL) {
	fprintf(stderr, "hostname: unknown host \"%s\"\n", host);
	exit(-1);
  }
  memcpy((char *) &in, (char *) hp->h_addr_list[0], hp->h_length);
  if (opt_f == 1) strcpy(host, hp->h_name);
  return(in);
}


static void set_host(in)
struct in_addr in;
{
  printf("hostname: setting IP ADDRESS to %s\n", inet_ntoa(in));
}


static void usage()
{
  fprintf(stderr, "Usage: hostname [-finvS] [name]\n");
  exit(-1);
}


void main(argc, argv)
int argc;
char *argv[];
{
  char hostname[128];
  struct in_addr in;
  unsigned long net;
  register int c, opt_S, opt_i, opt_n;
  extern int getopt(), optind, opterr;

  /* Fetch the command-line arguments. */
  opterr = 0;
  opt_S = 0;
  opt_i = 0;
  opt_n = 0;
  while ((c = getopt(argc, argv, "finvS")) != EOF) switch(c) {
	case 'f':
		opt_f = 1;
		break;
	case 'i':
		opt_i = 1;
		break;
	case 'n':
		opt_n = 1;
		break;
	case 'v':
		opt_v = 1;
		break;
	case 'S':
		opt_S = 1;
		break;
	default:
		usage();
  }

  /* Do we have to SHOW the current setting? */
  if (opt_S == 0) {
	if (optind != argc) usage();
	if (gethostname(hostname, 128) < 0) {
		perror("gethostname");
		exit(-1);
	}
	if (opt_i == 1) {
		in = get_addr(hostname);
		printf("%s", inet_ntoa(in));
	} else if (opt_n == 1) {
		in = get_addr(hostname);
		net = inet_netof(in);
		printf("%d.%d.%d.%d",
			((net >> 8) & 255),
			(net & 255),
			((net >> 24) & 255),
			((net >> 16) & 255));
#if 0
		printf("%s", inet_ntoa(*(struct in_addr *) &net));
#endif
	} else printf("%s", hostname);
	printf("\n");
	exit(0);
  }

  /* No.  Did we give a name? */
  if (optind == argc) get_host(hostname);
    else strcpy(hostname, argv[optind]);
  if (opt_v == 1) fprintf(stderr, "HOSTNAME: \"%s\"\n", hostname);
  in = get_addr(hostname);
  if (sethostname(hostname, strlen(hostname)) < 0) {
	perror("sethostname");
	exit(-1);
  }
  if (opt_i == 1) {
	if (opt_v == 1) fprintf(stderr, "IPADDR is %s\n", inet_ntoa(in));
	set_host(in);
  }

  exit(0);
}
