/* 
 *	FIG : Facility for Interactive Generation of figures
 *
 *	Copyright (c) 1985 by Supoj Sutanthavibul (supoj@sally.UTEXAS.EDU)
 *	January 1985.
 *	1st revision : Aug 1985.
 *
 *	Change function implemented by Frank Schmuck (schmuck@svax.cs.cornell.edu)
 *	July 1988.
 *
 *	%W%	%G%
*/
#include <stdio.h>
#include <math.h>
#include <suntool/sunview.h>
#include <suntool/panel.h>
#include "alloc.h"
#include "const.h"
#include "font.h"
#include "func.h"
#include "object.h"
#include "paintop.h"

#define			TOLERANCE	7

extern char		*sprintf();
extern char		*calloc();

extern			(*canvas_kbd_proc)();
extern			(*canvas_locmove_proc)();
extern			(*canvas_leftbut_proc)();
extern			(*canvas_middlebut_proc)();
extern			(*canvas_rightbut_proc)();
extern			null_proc();
extern			set_popupmenu();
extern F_line		*line_search(),		*copy_line();
extern F_arc		*arc_search(),		*copy_arc();
extern F_ellipse	*ellipse_search(),	*copy_ellipse();
extern F_text		*text_search(),		*copy_text();
extern F_spline		*spline_search(),	*copy_spline();
extern F_compound	*compound_search(),	*copy_compound();
extern F_arrow		*forward_arrow(),	*backward_arrow();
extern			zoomed_x(), zoomed_y(), unzoomed_x(), unzoomed_y();
extern			zscaled(), unzscaled();

extern F_compound	objects;

extern int              SIDERULER_WIDTH, PANL_WID;
extern int              RHS_PANL;
extern struct pixwin	*canvas_pixwin;
extern int		canvas_swfd;
extern struct cursor	null_cursor;
extern struct cursor	pick15_cursor;
extern struct cursor	*cur_cursor, wait_cursor;
extern int		background_color, foreground_color;
extern double		cur_dashlength;
extern double		cur_dotgap;
extern Frame		base_frame;

extern int		manhattan_mode;
extern int		mountain_mode;
extern int		autoforwardarrow_mode;
extern int		autobackwardarrow_mode;
extern int		latexline_mode;
extern int		latexarrow_mode;
extern int		magnet_mode;
extern int		line_thickness;
extern int		pen_size;
extern int		pen_type;
extern int		flip_axis;
extern int		rotate_angle;
extern int		cur_line_style;
extern double		cur_dashlength;
extern double		cur_dotgap;
extern double		cur_styleval;
extern double		cur_angle;
extern int		cur_color;
extern int		cur_textflags;
extern int		cur_textfont;
extern int		cur_textsize;
extern int		cur_texttype;
extern int		cur_depth;
extern double		cur_textangle;
extern int		cur_area_fill;
extern int		cur_pen;

			change_selected();
			init_change();
			done_compound();
			done_line();
			done_text();
			done_arc();
			done_ellipse();
			done_spline();
int			make_window_defaults();
int			done_defaults();
static int		defaults = 0;

static  F_line		*new_line;
static  F_ellipse	*new_ellipse;
static  F_text		*new_text;
static  F_spline	*new_spline;
static  F_arc		*new_arc;
static  F_compound	*new_compound;

static  Frame		change_frame;
static  Panel		change_panel;
static  Panel_item	subtype_panel;
static  Panel_item	thickness_panel;
static  Panel_item	color_panel;
static  Panel_item	depth_panel;
static	Panel_item	radius_panel;
static  Panel_item	area_fill_panel;
static  Panel_item	style_panel;
static  Panel_item	style_val_panel;
static  Panel_item	for_arrow_panel;
static  Panel_item	back_arrow_panel;
static  Panel_item	text_panel;
static  Panel_item	x1_panel, y1_panel;
static  Panel_item	x2_panel, y2_panel;
static  Panel_item	x3_panel, y3_panel;
static  Panel_item	font_panel;
static  Panel_item	font_size_panel;
static  Panel_item	rigid_panel;
static  Panel_item	special_panel;
static  Panel_item	psfont_panel;
static	Panel_item	hidden_text_panel;
static	Panel_item	compound_item;
static  Panel_item	angle_panel;
static	Panel_item	dotgap_panel;
static	Panel_item	dashlength_panel;
static	Attr_avlist	font_names;
static	Attr_avlist	psfont_names;
static	Attr_avlist	fill_names;

#define	MAX_POINTS	100
static  Panel_item	px_panel[MAX_POINTS];
static  Panel_item	py_panel[MAX_POINTS];

static	int		ellipse_flag;
static  int		(*done_proc)();
static  int		button_result;
#define CANCEL		0
#define DONE		1
#define APPLY		2

#define UNZM		unzoomed_x, unzoomed_y
#define ZOOM		zoomed_x, zoomed_y
#define ZSCL		zscaled, zscaled
#define UNZS		unzscaled, unzscaled

static struct {
	int			thickness;
	int			color;
	int			depth;
	int			style;
	double			style_val;
	int			pen;
	int 			area_fill;
	int			for_arrow;
	int			back_arrow;
} generic_vals;

#define put_generic_vals(x) \
	generic_vals.thickness	= x->thickness; \
	generic_vals.color	= x->color; \
	generic_vals.depth	= x->depth; \
	generic_vals.style	= x->style; \
	generic_vals.style_val	= x->style_val; \
	generic_vals.pen	= x->pen; \
	generic_vals.area_fill	= x->area_fill

#define put_generic_arrows(x) \
	generic_vals.for_arrow	= (x->for_arrow != NULL); \
	generic_vals.back_arrow	= (x->back_arrow != NULL)

#define get_generic_vals(x) \
	new_generic_values(NULL, NULL); \
	x->thickness	= generic_vals.thickness; \
	x->color	= generic_vals.color; \
	x->depth	= generic_vals.depth; \
	x->style	= generic_vals.style; \
	x->style_val	= generic_vals.style_val; \
	x->pen		= generic_vals.pen; \
	x->area_fill	= generic_vals.area_fill

#define get_generic_arrows(x) \
	if (x->for_arrow) \
		free((char*)x->for_arrow); \
	if (x->back_arrow) \
		free((char*)x->back_arrow); \
	x->for_arrow  = (generic_vals.for_arrow)?  forward_arrow(): NULL; \
	x->back_arrow = (generic_vals.back_arrow)? backward_arrow(): NULL

make_window_line(l)
F_line	*l;
{
	struct f_point	p1, p2;
        char buf[32];

	new_line = copy_line(l);
	new_line->next = l;
	put_generic_vals(new_line);
	switch (new_line->type) {
	    case T_POLYLINE:
		put_generic_arrows(new_line);
		generic_window("POLYLINE", "Polyline", done_line, 1, 1);
		points_panel(new_line->points, 0, 12, UNZM);
		break;
	    case T_POLYGON:
		put_generic_arrows(new_line);
		generic_window("POLYLINE", "Polygon", done_line, 1, 1);
		points_panel(new_line->points, 1, 12, UNZM);
		break;
	    case T_BOX:
	case T_ARC_BOX:
		generic_window("POLYLINE", "Box", done_line, 1, 0);
		p1 = *new_line->points;
		p2 = *new_line->points->next->next;
		xy_panel(p1.x, p1.y, "Corner", 10, &x1_panel, &y1_panel, UNZM);
		xy_panel(p2.x, p2.y, "Corner", 11, &x2_panel, &y2_panel, UNZM);
		radius_panel = panel_create_item(change_panel, PANEL_TEXT,
			PANEL_LABEL_X,		ATTR_COL(2),
			PANEL_LABEL_Y,		ATTR_ROW(12),
			PANEL_VALUE_X,		ATTR_COL(20),
			PANEL_VALUE_Y,		ATTR_ROW(12),
			PANEL_LABEL_STRING,	"Radius:",
			PANEL_VALUE,		sprintf(buf, "%d", new_line->radius),
			PANEL_VALUE_DISPLAY_LENGTH, 10,
			0);
		break;
	    }
	}

static
get_new_line_values()
{
	struct f_point	p1, p2, *p;

	get_generic_vals(new_line);
	if (new_line->type != T_BOX && new_line->type != T_ARC_BOX) {
	    get_generic_arrows(new_line);
	    get_points(new_line->points, (new_line->type == T_POLYGON), ZOOM);
	    }
	else {
	    p1.x = zoomed_x(atoi(panel_get_value(x1_panel)));
	    p1.y = zoomed_y(atoi(panel_get_value(y1_panel)));
	    p2.x = zoomed_x(atoi(panel_get_value(x2_panel)));
	    p2.y = zoomed_y(atoi(panel_get_value(y2_panel)));
	    p = new_line->points;
	    p->x = p1.x;  p->y = p1.y;  p = p->next;
	    p->x = p2.x;  p->y = p1.y;  p = p->next;
	    p->x = p2.x;  p->y = p2.y;  p = p->next;
	    p->x = p1.x;  p->y = p2.y;  p = p->next;
	    p->x = p1.x;  p->y = p1.y;
	    new_line->radius = atoi(panel_get_value(radius_panel));
	    if (new_line->radius > 0)
		    new_line->type = T_ARC_BOX;
	    else
		    new_line->type = T_BOX;
	    }
	}

done_line()
{
	F_line	*old_line;

	pw_batch_on(canvas_pixwin);
	draw_line(new_line, ERASE);
	old_line = new_line->next;
	switch (button_result) {
	    case APPLY:
		get_new_line_values();
		draw_line(new_line, PAINT);
		break;
	    case DONE:
		clean_up();
		get_new_line_values();
		set_action_object(F_CHANGE, O_POLYLINE);
		new_line->next = NULL;
		delete_line(&objects.lines, old_line);
		insert_line(&objects.lines, new_line);
		set_latestline(old_line);
		old_line->next = new_line;
		change_selected();
		draw_line(new_line, PAINT);
		show_pointmarker();
		set_modifiedflag();
		break;
	    case CANCEL:
		clean_up();
		new_line->next = NULL;
		free_line(&new_line);
		change_selected();
		draw_line(old_line, PAINT);
		show_pointmarker();
		break;
	    }
	pw_batch_off(canvas_pixwin);
	}

#define 	round(x)	((int) ((x) + ((x >= 0)? 0.5: -0.5)))

new_font_names(ip, ie)
Panel_item	ip;
Event		*ie;
{
	panel_set(font_panel, ATTR_LIST, ((int)panel_get_value(ip))
		  ? psfont_names : font_names, 0);
}

make_window_text(t)
F_text	*t;
{
        char buf[32];

	new_text = copy_text(t);
	new_text->next = t;
	generic_window("TEXT", NULL, done_text, 0, 0);
	subtype_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(2),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(2),
	    PANEL_LABEL_STRING,		"Text alignment:",
	    PANEL_CHOICE_STRINGS,	"left justified", "center justified",
	    				   "right justified", 0,
	    PANEL_VALUE,		new_text->type,
	    0);
	psfont_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(3),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(3),
	    PANEL_LABEL_STRING,		"PS Font:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		psfont_text(new_text),
	    PANEL_NOTIFY_PROC,		new_font_names,
	    0);
	font_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(4),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(4),
	    PANEL_LABEL_STRING,		"Font:",
	    ATTR_LIST,			psfont_text(new_text)
						? psfont_names: font_names,
	    PANEL_VALUE,		new_text->font,
	    0);
	rigid_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(5),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(5),
	    PANEL_LABEL_STRING,		"Rigid:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		rigid_text(new_text),
	    0);
	special_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(6),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(6),
	    PANEL_LABEL_STRING,		"Special:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		special_text(new_text),
	    0);
	color_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(7),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(7),
	      PANEL_LABEL_STRING,	"Color:",
	      PANEL_VALUE,		sprintf(buf, "%d", new_text->color),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      0);
	depth_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(8),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(8),
	      PANEL_LABEL_STRING,	"Depth:",
	      PANEL_VALUE,		sprintf(buf, "%d", new_text->depth),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      0);
	angle_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(9),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(9),
	      PANEL_LABEL_STRING,	"Angle:",
	      PANEL_VALUE,		sprintf(buf, "%d", round(180/M_PI * new_text->angle)),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      0);
	font_size_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(10),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(10),
	      PANEL_LABEL_STRING,	"Size:",
	      PANEL_VALUE,		sprintf(buf, "%d", new_text->size),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      0);
	xy_panel(new_text->base_x, new_text->base_y, "Origin", 8,
	    &x1_panel, &y1_panel, UNZM);
	text_panel = panel_create_item(change_panel, PANEL_TEXT,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(13),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(13),
	    PANEL_LABEL_STRING,		"Text:",
	    PANEL_VALUE,		new_text->cstring,
	    PANEL_VALUE_DISPLAY_LENGTH, 40,
	    0);
	hidden_text_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(14),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(14),
	    PANEL_LABEL_STRING,		"Hidden:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		hidden_text(new_text),
	    0);
	}

static
get_new_text_values()
{
	char		*s;
	struct pr_size	size;

	new_text->type =   (int)panel_get_value(subtype_panel);
	new_text->font =   (int)panel_get_value(font_panel);
	new_text->size =   atoi(panel_get_value(font_size_panel));
	new_text->flags = 
	  	  (panel_get_value(rigid_panel)		? RIGID_TEXT	: 0)
  		| (panel_get_value(special_panel) 	? SPECIAL_TEXT	: 0)
  		| (panel_get_value(hidden_text_panel) 	? HIDDEN_TEXT	: 0)
 		| (panel_get_value(psfont_panel)	? PSFONT_TEXT	: 0);
	new_text->color =  atoi(panel_get_value(color_panel));
	new_text->depth =  atoi(panel_get_value(depth_panel));
	new_text->angle =  M_PI/180*atoi(panel_get_value(angle_panel));
	new_text->base_x = zoomed_x(atoi(panel_get_value(x1_panel)));
	new_text->base_y = zoomed_y(atoi(panel_get_value(y1_panel)));
	if (new_text->cstring)
		    cfree(new_text->cstring);
	s = panel_get_value(text_panel);
	new_text->cstring = calloc((unsigned)(strlen(s)+1),sizeof(char));
	strcpy(new_text->cstring, s);
	size = pf_textwidth(strlen(s), canvas_font, s);
	new_text->height = size.y * 0.7;
	new_text->length = size.x;
	}

done_text()
{
	F_text	*old_text;

	pw_batch_on(canvas_pixwin);
	draw_text(new_text, INV_PAINT);
	old_text = new_text->next;
	switch (button_result) {
	    case APPLY:
		get_new_text_values();
		break;
	    case DONE:
		clean_up();
		get_new_text_values();
		set_action_object(F_CHANGE, O_TEXT);
		new_text->next = NULL;
		delete_text(&objects.texts, old_text);
		insert_text(&objects.texts, new_text);
		set_latesttext(old_text);
		old_text->next = new_text;
		change_selected();
		show_pointmarker();
		set_modifiedflag();
		break;
	    case CANCEL:
		clean_up();
		new_text->next = NULL;
		free_text(&new_text);
		new_text = old_text;
		change_selected();
		show_pointmarker();
		break;
	    }
	    draw_text(new_text, PAINT);
	    pw_batch_off(canvas_pixwin);
	}

make_window_ellipse(e)
F_ellipse *e;
{
	char		*s1, *s2;

	new_ellipse = copy_ellipse(e);
	new_ellipse->next = e;
	switch (new_ellipse->type) {
	    case T_ELLIPSE_BY_RAD:
		s1 = "ELLIPSE";
		s2 = "specified by radius";
		ellipse_flag = 1;
		break;
	    case T_ELLIPSE_BY_DIA:
		s1 = "ELLIPSE";
		s2 = "specified by diameter";
		ellipse_flag = 1;
		break;
	    case T_CIRCLE_BY_RAD:
		s1 = "CIRCLE";
		s2 = "specified by radius";
		ellipse_flag = 0;
		break;
	    case T_CIRCLE_BY_DIA:
		s1 = "CIRCLE";
		s2 = "specified by diameter";
		ellipse_flag = 0;
		break;
	    }
	put_generic_vals(new_ellipse);
	generic_window(s1, s2, done_ellipse, 1, 0);
	if (ellipse_flag) {
	    f_pos_panel(&new_ellipse->center,   "Center  ", 10,
	      &x1_panel, &y1_panel, UNZM);
	    f_pos_panel(&new_ellipse->radiuses, "Radiuses", 11,
	      &x2_panel, &y2_panel, UNZS);
	    }
	else {
	    f_pos_panel(&new_ellipse->center,"Center", 10,
	      &x1_panel, &y1_panel, UNZM);
	    int_panel(new_ellipse->radiuses.x, "Radius", 11,
	      &x2_panel, unzscaled);
	    }
	}

static
get_new_ellipse_values()
{
	F_pos	old_center;
	F_pos	old_radiuses;

#	define	adjust_ref(s,f) \
		s.f = new_ellipse->center.f + \
		((s.f - old_center.f)*new_ellipse->radiuses.f)/old_radiuses.f

	get_generic_vals(new_ellipse);
	old_center =   new_ellipse->center;
	old_radiuses = new_ellipse->radiuses;
	get_f_pos(&new_ellipse->center,   x1_panel, y1_panel, ZOOM);
	if (ellipse_flag) 
	    get_f_pos(&new_ellipse->radiuses, x2_panel, y2_panel, ZSCL);
	else
	    new_ellipse->radiuses.x = new_ellipse->radiuses.y =
	      zscaled(atoi(panel_get_value(x2_panel)));
	adjust_ref(new_ellipse->start, x);
	adjust_ref(new_ellipse->start, y);
	adjust_ref(new_ellipse->end, x);
	adjust_ref(new_ellipse->end, y);
	}

done_ellipse()
{
	F_ellipse	*old_ellipse;

	pw_batch_on(canvas_pixwin);
	draw_ellipse(new_ellipse, background_color);
	old_ellipse = new_ellipse->next;
	switch (button_result) {
	    case APPLY:
		get_new_ellipse_values();
		draw_ellipse(new_ellipse, foreground_color);
		break;
	    case DONE:
		clean_up();
		get_new_ellipse_values();
		set_action_object(F_CHANGE, O_ELLIPSE);
		new_ellipse->next = NULL;
		delete_ellipse(&objects.ellipses, old_ellipse);
		insert_ellipse(&objects.ellipses, new_ellipse);
		set_latestellipse(old_ellipse);
		old_ellipse->next = new_ellipse;
		change_selected();
		draw_ellipse(new_ellipse, foreground_color);
		show_pointmarker();
		set_modifiedflag();
		break;
	    case CANCEL:
		clean_up();
		new_ellipse->next = NULL;
		free_ellipse(&new_ellipse);
		change_selected();
		draw_ellipse(old_ellipse, foreground_color);
		show_pointmarker();
		break;
	    }
	pw_batch_off(canvas_pixwin);
	}

make_window_arc(a)
F_arc *a;
{
	new_arc = copy_arc(a);
	new_arc->next = a;
	put_generic_vals(new_arc);
	put_generic_arrows(new_arc);
	generic_window("ARC", "Specified by 3 points", done_arc, 1, 1);
	f_pos_panel(&new_arc->point[0], "p1", 12, &x1_panel, &y1_panel, UNZM);
	f_pos_panel(&new_arc->point[1], "p2", 13, &x2_panel, &y2_panel, UNZM);
	f_pos_panel(&new_arc->point[2], "p3", 14, &x3_panel, &y3_panel, UNZM);
	}

static
get_new_arc_values()
{
	F_pos	p0, p1, p2;
	double	cx, cy;

	get_generic_vals(new_arc);
	get_generic_arrows(new_arc);
	get_f_pos(&p0, x1_panel, y1_panel, ZOOM);
	get_f_pos(&p1, x2_panel, y2_panel, ZOOM);
	get_f_pos(&p2, x3_panel, y3_panel, ZOOM);
	if (compute_arccenter(p0, p1, p2, &cx, &cy)) {
	    new_arc->point[0] = p0;
	    new_arc->point[1] = p1;
	    new_arc->point[2] = p2;
	    new_arc->center.x = cx;
	    new_arc->center.y = cy;
	    new_arc->direction = compute_direction(p0, p1, p2);
	    }
	else
	    put_msg("Invalid ARC points!");
	}

done_arc()
{
	F_arc	*old_arc;

	pw_batch_on(canvas_pixwin);
	draw_arc(new_arc, background_color);
	old_arc = new_arc->next;
	switch (button_result) {
	    case APPLY:
		get_new_arc_values();
		draw_arc(new_arc, foreground_color);
		break;
	    case DONE:
		clean_up();
		get_new_arc_values();
		set_action_object(F_CHANGE, O_ARC);
		new_arc->next = NULL;
		delete_arc(&objects.arcs, old_arc);
		insert_arc(&objects.arcs, new_arc);
		set_latestarc(old_arc);
		old_arc->next = new_arc;
		change_selected();
		draw_arc(new_arc, foreground_color);
		show_pointmarker();
		set_modifiedflag();
		break;
	    case CANCEL:
		clean_up();
		new_arc->next = NULL;
		free_arc(&new_arc);
		change_selected();
		draw_arc(old_arc, foreground_color);
		show_pointmarker();
		break;
	    }
	pw_batch_off(canvas_pixwin);
	}

make_window_spline(s)
F_spline *s;
{
	new_spline = copy_spline(s);
	new_spline->next = s;
	put_generic_vals(new_spline);
	put_generic_arrows(new_spline);
	switch (new_spline->type) {
	    case T_OPEN_NORMAL:
		generic_window("SPLINE", "Normal open", done_spline, 1, 1);
		points_panel(new_spline->points, 0, 12, UNZM);
		break;
	    case T_CLOSED_NORMAL:
		generic_window("SPLINE", "Normal closed", done_spline, 1, 1);
		points_panel(new_spline->points, 1, 12, UNZM);
		break;
	    case T_OPEN_INTERPOLATED:
		generic_window("SPLINE", "Interpolated open", done_spline, 1, 1);
		points_panel(new_spline->points, 0, 12, UNZM);
		break;
	    case T_CLOSED_INTERPOLATED:
		generic_window("SPLINE","Interpolated closed",done_spline, 1,1);
		points_panel(new_spline->points, 1, 12, UNZM);
		break;
	    }
	}

done_spline()
{
	F_spline	*old_spline;

	pw_batch_on(canvas_pixwin);
	draw_spline(new_spline, ERASE);
	old_spline = new_spline->next;
	switch (button_result) {
	    case APPLY:
		get_generic_vals(new_spline);
		get_generic_arrows(new_spline);
		get_points(new_spline->points, closed_spline(new_spline), ZOOM);
		if (int_spline(new_spline))
		    remake_control_points(new_spline);
		draw_spline(new_spline, PAINT);
		break;
	    case DONE:
		clean_up();
		get_generic_vals(new_spline);
		get_generic_arrows(new_spline);
		get_points(new_spline->points, closed_spline(new_spline), ZOOM);
		set_action_object(F_CHANGE, O_SPLINE);
		if (int_spline(new_spline))
		    remake_control_points(new_spline);
		new_spline->next = NULL;
		delete_spline(&objects.splines, old_spline);
		insert_spline(&objects.splines, new_spline);
		set_latestspline(old_spline);
		old_spline->next = new_spline;
		change_selected();
		draw_spline(new_spline, PAINT);
		show_pointmarker();
		set_modifiedflag();
		break;
	    case CANCEL:
		clean_up();
		new_spline->next = NULL;
		free_spline(&new_spline);
		change_selected();
		draw_spline(old_spline, PAINT);
		show_pointmarker();
		break;
	    }
	pw_batch_off(canvas_pixwin);
	}

property_proc()
{
	put_msg("Close the CHANGE window first!");
	}

property_selected()
{
	canvas_kbd_proc = null_proc;
	canvas_locmove_proc = null_proc;
	canvas_leftbut_proc = property_proc;
	canvas_middlebut_proc = property_proc;
	canvas_rightbut_proc = property_proc;
	}

change_selected()
{
	canvas_kbd_proc = null_proc;
	canvas_locmove_proc = null_proc;
	canvas_leftbut_proc = init_change;
	canvas_middlebut_proc = make_window_defaults;
	canvas_rightbut_proc = set_popupmenu;
	set_cursor(&pick15_cursor);
	reset_action_on();
	}

init_change(x, y, xm, ym)
int	x, y, xm, ym;
{
	F_line		*l;
	F_ellipse	*e;
	F_text		*t;
	F_spline	*s;
	F_arc		*a;
	F_compound	*c;
	int		px, py;

	win_setcursor(canvas_swfd, &wait_cursor);
	if ((c = compound_search(xm, ym, TOLERANCE, &px, &py)) != NULL)
	    make_window_compound(c);
	else if ((l = line_search(xm, ym, TOLERANCE, &px, &py)) != NULL)
	    make_window_line(l);
	else if ((t = text_search(xm, ym)) != NULL)
	    make_window_text(t);
	else if ((e = ellipse_search(xm, ym, TOLERANCE, &px, &py)) != NULL)
	    make_window_ellipse(e);
	else if ((a = arc_search(xm, ym, TOLERANCE, &px, &py)) != NULL)
	    make_window_arc(a);
	else if ((s = spline_search(xm, ym, TOLERANCE, &px, &py)) != NULL)
	    make_window_spline(s);
	else {
		win_setcursor(canvas_swfd, cur_cursor);
		return;
	}
	set_action_on();
	property_selected();
	erase_pointmarker();
	(void)window_fit(change_panel);
	(void)window_fit(change_frame);
	(void)window_set(change_frame, WIN_SHOW, TRUE, 0);
	win_setcursor(canvas_swfd, cur_cursor);
	}

/*
 *  The following routines
 *	change_line, change_arc, change_ellipse, change_spline, 
 *	change_text, change_compound
 *  are called by the UNDO function (undo.c).
 *  saved_objects.xxxx contains a pointer to the original, unchanged version
 *  of the object.  saved_objects.xxxx->next contains a pointer to the new,
 *  changed version of the object.
 */
 
change_line(lold, lnew)
F_line	*lold, *lnew;
{
	lnew->next = NULL;
	delete_line(&objects.lines, lold);
	insert_line(&objects.lines, lnew);
	set_latestline(lold);
	lold->next = lnew;
	pw_batch_on(canvas_pixwin);
	draw_line(lold, ERASE);
	draw_line(lnew, PAINT);
	pw_batch_off(canvas_pixwin);
	}
	
change_arc(aold, anew)
F_arc	*aold, *anew;
{
	anew->next = NULL;
	delete_arc(&objects.arcs, aold);
	insert_arc(&objects.arcs, anew);
	set_latestarc(aold);
	aold->next = anew;
	pw_batch_on(canvas_pixwin);
	draw_arc(aold, background_color);
	draw_arc(anew, foreground_color);
	pw_batch_off(canvas_pixwin);
	}

change_ellipse(eold, enew)
F_ellipse	*eold, *enew;
{
	enew->next = NULL;
	delete_ellipse(&objects.ellipses, eold);
	insert_ellipse(&objects.ellipses, enew);
	set_latestellipse(eold);
	eold->next = enew;
	pw_batch_on(canvas_pixwin);
	draw_ellipse(eold, background_color);
	draw_ellipse(enew, foreground_color);
	pw_batch_off(canvas_pixwin);
	}

change_spline(sold, snew)
F_spline	*sold, *snew;
{
	snew->next = NULL;
	delete_spline(&objects.splines, sold);
	insert_spline(&objects.splines, snew);
	set_latestspline(sold);
	sold->next = snew;
	pw_batch_on(canvas_pixwin);
	draw_spline(sold, ERASE);
	draw_spline(snew, PAINT);
	pw_batch_off(canvas_pixwin);
	}

change_text(told, tnew)
F_text	*told, *tnew;
{
	tnew->next = NULL;
	delete_text(&objects.texts, told);
	insert_text(&objects.texts, tnew);
	set_latesttext(told);
	told->next = tnew;
	pw_batch_on(canvas_pixwin);
	draw_text(told, INV_PAINT);
	draw_text(tnew, PAINT);
	pw_batch_off(canvas_pixwin);
	}

change_compound(cold, cnew)
F_compound	*cold, *cnew;
{
	/* 
	 * This is needed to undo "align" even tho it is not used for 
	 * the "change" operation.
	 */
	cnew->next = NULL;
	delete_compound(&objects.compounds, cold);
	insert_compound(&objects.compounds, cnew);
	set_latestcompound(cold);
	cold->next = cnew;
	pw_batch_on(canvas_pixwin);
	erase_compound(cold);
	draw_compoundbox(cold, ERASE);
	draw_compound(cnew);
	draw_compoundbox(cnew, PAINT);
	pw_batch_off(canvas_pixwin);
	}



static
new_generic_values(ip, ie)
Panel_item	ip;
Event		*ie;
{
    generic_vals.thickness = atoi(panel_get_value(thickness_panel));
    generic_vals.color =     atoi(panel_get_value(color_panel));
    generic_vals.depth =     atoi(panel_get_value(depth_panel));
    generic_vals.area_fill=  (int)panel_get_value(area_fill_panel);
}

static
new_arrow_values(ip, ie)
Panel_item	ip;
Event		*ie;
{
    generic_vals.for_arrow  = (int)panel_get_value(for_arrow_panel);
    generic_vals.back_arrow = (int)panel_get_value(back_arrow_panel);
}

static char *
style_val_string(s, v)
int   s;
double v;
{
    static char buf[64];
    switch(s) {
	case SOLID_LINE:
	    return sprintf(buf, " ");
 	case DASH_LINE:
	    return sprintf(buf, "dash length = %3.1f", v);
 	case DOTTED_LINE:
	    return sprintf(buf, "dot gap  =    %3.1f", v);
	}
    return NULL;
    }

static
new_style_values(ip, ie)
Panel_item	ip;
Event		*ie;
{
    int   s, v;
    double f;

    s = (int)panel_get_value(style_panel);
    v = (int)panel_get_value(style_val_panel);
    f = (double)v / 10.0;
    if (s != generic_vals.style) {
	switch(s) {
	    case SOLID_LINE:
		f = 0.0;
		break;
	    case DASH_LINE:
		f = cur_dashlength;
		break;
	    case DOTTED_LINE:
		f = cur_dotgap;
		break;
	    }
	panel_set(style_val_panel, PANEL_VALUE, (int)(10*f), 0);
	}
    if (s != generic_vals.style  ||  f != generic_vals.style_val)
	panel_set(style_val_panel,
	  PANEL_LABEL_STRING, style_val_string(s, f),
	  0);
    generic_vals.style = s;
    generic_vals.style_val = f;
}

static
done_button(panel_local,item, event)
Panel		panel_local;
Panel_item	item;
Event		*event;
{
    button_result = DONE;
    done_proc();
    (void)window_set(change_frame, WIN_SHOW, FALSE, 0);
    (void)window_destroy(change_panel);
}

static
apply_button(panel_local,item, event)
Panel		panel_local;
Panel_item	item;
Event		*event;
{
    button_result = APPLY;
    done_proc();
}

static
cancel_button(panel_local,item, event)
Panel		panel_local;
Panel_item	item;
Event		*event;
{
    button_result = CANCEL;
    done_proc();
    (void)window_set(change_frame, WIN_SHOW, FALSE, 0);
    (void)window_destroy(change_panel);
}

make_change_frame()
{
	font_names = attr_create_list(
	    PANEL_CHOICE_STRINGS,      "Default",
 				       "Roman",
 				       "Bold",
 				       "Italic",
 				       "Modern",
 				       "Typewriter",
 				       0,
				      0);

	psfont_names = attr_create_list(
	    PANEL_CHOICE_STRINGS,      
			"Times-Roman", "Times-Roman",	/* default */
			"Times-Italic",			/* italic */
			"Times-Bold",			/* bold */
			"Times-BoldItalic",
			"AvantGarde",
			"AvantGarde-BookOblique",
			"AvantGarde-Demi",
			"AvantGarde-DemiOblique",
			"Bookman-Light",
			"Bookman-LightItalic",
			"Bookman-Demi",
			"Bookman-DemiItalic",
			"Courier",	
			"Courier-Oblique",
			"Courier-Bold",
			"Courier-BoldItalic",
			"Helvetica",
			"Helvetica-Oblique",
			"Helvetica-Bold",
			"Helvetica-BoldOblique",
			"Helvetica-Narrow",
			"Helvetica-Narrow-Oblique",
			"Helvetica-Narrow-Bold",
			"Helvetica-Narrow-BoldOblique",
			"NewCenturySchlbk-Roman",
			"NewCenturySchlbk-Italic",
			"NewCenturySchlbk-Bold",
			"NewCenturySchlbk-BoldItalic",
			"Palatino-Roman",
			"Palatino-Italic",
			"Palatino-Bold",
			"Palatino-BoldItalic",
			"Symbol",
			"ZapfChancery-MediumItalic",
			"ZapfDingbats",
			0,
			0);

	fill_names = attr_create_list(
	    PANEL_CHOICE_STRINGS,      
			"Unfilled",
			"White fill (0.0)",
			"Gray fill (0.05)",
			"Gray fill (0.10)",
			"Gray fill (0.15)",
			"Gray fill (0.20)",
			"Gray fill (0.25)",
			"Gray fill (0.30)",
			"Gray fill (0.35)",
			"Gray fill (0.40)",
			"Gray fill (0.45)",
			"Gray fill (0.50)",
			"Gray fill (0.55)",
			"Gray fill (0.60)",
			"Gray fill (0.65)",
			"Gray fill (0.70)",
			"Gray fill (0.75)",
			"Gray fill (0.80)",
			"Gray fill (0.85)",
			"Gray fill (0.90)",
			"Gray fill (0.95)",
			"Black fill (1.0)",
			0,
			0);

	change_frame = window_create(base_frame, FRAME,
	    WIN_SHOW,		FALSE,
	    FRAME_LABEL,	"Change",
	    FRAME_SHOW_LABEL,	TRUE,
	    0);
}


static
generic_window(object_type, sub_type, d_proc, generics, arrows)
char	*object_type, *sub_type;
int	(*d_proc)();
int	generics, arrows;
{
	char buf[64];

	done_proc = d_proc;
	(void)window_set(change_frame, FRAME_DONE_PROC, done_button, 0);

	change_panel = window_create(change_frame, PANEL, 0);

	(void)panel_create_item(change_panel, PANEL_MESSAGE, 
		PANEL_LABEL_X,		ATTR_COL(0),
		PANEL_LABEL_Y,		ATTR_ROW(0),
		PANEL_LABEL_STRING,	object_type,
		PANEL_LABEL_BOLD,	TRUE,
		0);
	if (sub_type)
	    (void)panel_create_item(change_panel, PANEL_MESSAGE, 
		PANEL_LABEL_X,		ATTR_COL(2),
		PANEL_LABEL_Y,		ATTR_ROW(1),
		PANEL_LABEL_STRING,	sub_type,
		0);
	(void)panel_create_item(change_panel, PANEL_BUTTON, 
		PANEL_LABEL_X,		ATTR_COL(30),
		PANEL_LABEL_Y,		ATTR_ROW(0),
		PANEL_LABEL_IMAGE,	panel_button_image(change_panel,
					  "Done", 6, (Pixfont *)0),
		PANEL_NOTIFY_PROC,	done_button,
		0);
	(void)panel_create_item(change_panel, PANEL_BUTTON, 
		PANEL_LABEL_X,		ATTR_COL(40),
		PANEL_LABEL_Y,		ATTR_ROW(0),
		PANEL_LABEL_IMAGE,	panel_button_image(change_panel,
					  "Apply", 6, (Pixfont *)0),
		PANEL_NOTIFY_PROC,	apply_button,
		0);
	(void)panel_create_item(change_panel, PANEL_BUTTON, 
		PANEL_LABEL_X,		ATTR_COL(50),
		PANEL_LABEL_Y,		ATTR_ROW(0),
		PANEL_LABEL_IMAGE,	panel_button_image(change_panel,
					  "Cancel", 6, (Pixfont *)0),
		PANEL_NOTIFY_PROC,	cancel_button,
		0);

	if (generics) {
	    thickness_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(3),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(3),
	      PANEL_LABEL_STRING,	"Thickness:",
	      PANEL_VALUE,		sprintf(buf,"%d",generic_vals.thickness),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      PANEL_NOTIFY_PROC,	new_generic_values,
	      0);
	    color_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(4),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(4),
	      PANEL_LABEL_STRING,	"Color:",
	      PANEL_VALUE,		sprintf(buf, "%d", generic_vals.color),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      PANEL_NOTIFY_PROC,	new_generic_values,
	      0);
	    depth_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(5),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(5),
	      PANEL_LABEL_STRING,	"Depth:",
	      PANEL_VALUE,		sprintf(buf, "%d", generic_vals.depth),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      PANEL_NOTIFY_PROC,	new_generic_values,
	      0);
	    area_fill_panel = panel_create_item(change_panel, PANEL_CYCLE,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(6),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(6),
	      PANEL_LABEL_STRING,	"Area fill:",
	      ATTR_LIST,		fill_names,
	      PANEL_VALUE,		generic_vals.area_fill,
	      PANEL_NOTIFY_PROC,	new_generic_values,
	      0);
	    if (defaults == 0) {
	    style_panel = panel_create_item(change_panel, PANEL_CYCLE,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(7),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(7),
	      PANEL_LABEL_STRING,	"Line style:",
	      PANEL_CHOICE_STRINGS,	"solid line", "dashed line",
					  "dotted line", 0,
	      PANEL_VALUE,		generic_vals.style,
	      PANEL_NOTIFY_PROC,	new_style_values,
	      0);
	    style_val_panel =  panel_create_item(change_panel, PANEL_SLIDER, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(8),
	      PANEL_LABEL_STRING,	style_val_string(
					  generic_vals.style,
					  generic_vals.style_val),
	      PANEL_MIN_VALUE,		0,
	      PANEL_MAX_VALUE,		300,
	      PANEL_SLIDER_WIDTH,	300,
	      PANEL_VALUE,		(int)(10*generic_vals.style_val),
	      PANEL_SHOW_RANGE,		FALSE,
	      PANEL_LAYOUT,		PANEL_VERTICAL,
	      PANEL_NOTIFY_PROC,	new_style_values,
	      0);
    }
	    }
	if (arrows) {
	    for_arrow_panel = panel_create_item(change_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(10),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(10),
	      PANEL_LABEL_STRING,	"Forward arrow:",
	      PANEL_VALUE,		generic_vals.for_arrow,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      PANEL_NOTIFY_PROC,	new_arrow_values,
	      0);
	    back_arrow_panel = panel_create_item(change_panel, PANEL_TOGGLE, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(11),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(11),
	      PANEL_LABEL_STRING,	"Backward arrow:",
	      PANEL_VALUE,		generic_vals.back_arrow,
	      PANEL_FEEDBACK,		PANEL_MARKED,
	      PANEL_NOTIFY_PROC,	new_arrow_values,
	      0);
	    }
	}

static
int_panel(x, label, row, pi_x, zf)
int		x;
char		*label;
int		row;
Panel_item	*pi_x;
int		(*zf)();
{
	char buf[32];

	*pi_x = panel_create_item(change_panel, PANEL_TEXT,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(row),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(row),
	    PANEL_LABEL_STRING,		label,
	    PANEL_VALUE,		sprintf(buf, "%d", zf(x)),
	    PANEL_VALUE_DISPLAY_LENGTH, 5,
	    0);
	}

static
xy_panel(x, y, label, row, pi_x, pi_y, zfx, zfy)
int		x, y;
char		*label;
int		row;
Panel_item	*pi_x, *pi_y;
int		(*zfx)(), (*zfy)();
{
	char buf[32];
	char bufxy[32];

	*pi_x = panel_create_item(change_panel, PANEL_TEXT,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(row),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(row),
	    PANEL_LABEL_STRING,		sprintf(buf, "%s  x, y:", label),
	    PANEL_VALUE,		sprintf(bufxy, "%d", zfx(x)),
	    PANEL_VALUE_DISPLAY_LENGTH, 5,
	    0);
	*pi_y = panel_create_item(change_panel, PANEL_TEXT,
	    PANEL_VALUE_X,		ATTR_COL(27),
	    PANEL_VALUE_Y,		ATTR_ROW(row),
	    PANEL_VALUE,		sprintf(bufxy, "%d", zfy(y)),
	    PANEL_VALUE_DISPLAY_LENGTH, 5,
	    0);
	}

static
f_pos_panel(fp, label, row, pi_x, pi_y, zfx, zfy)
F_pos		*fp;
char		*label;
int		row;
Panel_item	*pi_x, *pi_y;
int		(*zfx)(), (*zfy)();
{
	char buf[32];
	char bufxy[32];

	*pi_x = panel_create_item(change_panel, PANEL_TEXT,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(row),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(row),
	    PANEL_LABEL_STRING,		sprintf(buf, "%s  x, y:", label),
	    PANEL_VALUE,		sprintf(bufxy, "%d", zfx(fp->x)),
	    PANEL_VALUE_DISPLAY_LENGTH, 5,
	    0);
	*pi_y = panel_create_item(change_panel, PANEL_TEXT,
	    PANEL_VALUE_X,		ATTR_COL(27),
	    PANEL_VALUE_Y,		ATTR_ROW(row),
	    PANEL_VALUE,		sprintf(bufxy, "%d", zfy(fp->y)),
	    PANEL_VALUE_DISPLAY_LENGTH, 5,
	    0);
	}

static
get_f_pos(fp, pi_x, pi_y, zfx, zfy)
F_pos		*fp;
Panel_item	pi_x, pi_y;
int		(*zfx)(), (*zfy)();
{
	fp->x = zfx(atoi(panel_get_value(pi_x)));
	fp->y = zfy(atoi(panel_get_value(pi_y)));
	}

static
points_panel(p, closed, row, zfx, zfy)
struct f_point	*p;
int		row, closed;
int		(*zfx)(), (*zfy)();
{
	char buf[32];
	char bufxy[32];
	int  i;

	for (i = 0; p != NULL; i++) {
	    if (i >= MAX_POINTS)
		break;
	    px_panel[i] = panel_create_item(change_panel, PANEL_TEXT,
		PANEL_LABEL_X,		ATTR_COL(2),
		PANEL_LABEL_Y,		ATTR_ROW(row+i),
		PANEL_VALUE_X,		ATTR_COL(20),
		PANEL_VALUE_Y,		ATTR_ROW(row+i),
		PANEL_LABEL_STRING,	sprintf(buf, "p%d  x, y:", i+1),
		PANEL_VALUE,		sprintf(bufxy, "%d", zfx(p->x)),
		PANEL_VALUE_DISPLAY_LENGTH, 5,
		0);
	    py_panel[i] = panel_create_item(change_panel, PANEL_TEXT,
		PANEL_VALUE_X,		ATTR_COL(27),
		PANEL_VALUE_Y,		ATTR_ROW(row+i),
		PANEL_VALUE,		sprintf(bufxy, "%d", zfy(p->y)),
		PANEL_VALUE_DISPLAY_LENGTH, 5,
		0);
	    p = p->next;
	    if (closed  &&  (p == NULL  ||  p->next == NULL))
		break;
	    }
	}

static
get_points(p, closed, zfx, zfy)
struct f_point	*p;
int		(*zfx)(), (*zfy)();
{
	struct f_point	*q;
	int  i;

	for (q = p, i = 0;  q != NULL;  i++) {
	    if (i >= MAX_POINTS)
		break;
	    q->x = zfx(atoi(panel_get_value(px_panel[i])));
	    q->y = zfy(atoi(panel_get_value(py_panel[i])));
	    q = q->next;
	    if (closed) {
		if (q == NULL)
		    break;
		else if (q->next == NULL) {
		    q->x = p->x;
		    q->y = p->y;
		    break;
		    }
		}
	    }
	}

/* 
 * Add stuff for changing default values.
 */

#define put_default_vals() \
	generic_vals.thickness	= line_thickness; \
	generic_vals.color	= cur_color; \
	generic_vals.depth	= cur_depth; \
	generic_vals.style	= cur_line_style; \
	generic_vals.style_val	= cur_styleval; \
	generic_vals.pen	= cur_pen; \
	generic_vals.area_fill	= cur_area_fill

#define get_default_vals() \
	new_generic_values(NULL, NULL); \
	line_thickness	= generic_vals.thickness; \
	cur_color	= generic_vals.color; \
	cur_depth	= generic_vals.depth; \
	cur_line_style	= generic_vals.style; \
	cur_styleval	= generic_vals.style_val; \
	cur_pen		= generic_vals.pen; \
	cur_area_fill	= generic_vals.area_fill

new_default_style_values(ip, ie)
Panel_item	ip;
Event		*ie;
{
	double	f;
	char	buf[32];

	f = (int)panel_get_value(ip)/10.0;
	if (ip == dotgap_panel)
		panel_set(ip, PANEL_LABEL_STRING,
			  sprintf(buf, "dot gap = %3.1f", f), 0);
	else
		panel_set(ip, PANEL_LABEL_STRING,
			  sprintf(buf, "dash length = %3.1f", f), 0);
}

default_panel_items()
{
	char	buf[32];

	subtype_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(12),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(12),
	    PANEL_LABEL_STRING,		"Text alignment:",
	    PANEL_CHOICE_STRINGS,	"left justified", "center justified",
	    				   "right justified", 0,
	    PANEL_VALUE,		cur_texttype,
	    0);
	psfont_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(13),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(13),
	    PANEL_LABEL_STRING,		"PS Font:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		(cur_textflags != DEFAULT &&
					 (cur_textflags & PSFONT_TEXT)),
	    PANEL_NOTIFY_PROC,		new_font_names,
	    0);
	font_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(14),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(14),
	    PANEL_LABEL_STRING,		"Font:",
	    ATTR_LIST,			(cur_textflags != DEFAULT &&
					 (cur_textflags & PSFONT_TEXT)) ?
				       psfont_names : font_names,
	    PANEL_VALUE,		cur_textfont,
	    0);
	rigid_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(15),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(15),
	    PANEL_LABEL_STRING,		"Rigid:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		(cur_textflags == DEFAULT ||
					 (cur_textflags & RIGID_TEXT)),
	    0);
	special_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(16),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(16),
	    PANEL_LABEL_STRING,		"Special:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		(cur_textflags != DEFAULT &&
					 (cur_textflags & SPECIAL_TEXT)),
	    0);
	font_size_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(17),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(17),
	      PANEL_LABEL_STRING,	"Text size (pts):",
	      PANEL_VALUE,		sprintf(buf, "%d", cur_textsize),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      0);
	angle_panel = panel_create_item(change_panel, PANEL_TEXT,
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(18),
	      PANEL_VALUE_X,		ATTR_COL(20),
	      PANEL_VALUE_Y,		ATTR_ROW(18),
	      PANEL_LABEL_STRING,	"Text angle:",
	      PANEL_VALUE,		sprintf(buf, "%d", round(180/M_PI * cur_textangle)),
	      PANEL_VALUE_DISPLAY_LENGTH, 10,
	      0);
}

make_window_defaults()
{
	char	buf[32];

	win_setcursor(canvas_swfd, &wait_cursor);
	defaults++;
	put_default_vals();
	generic_window("Defaults", NULL, done_defaults, 1, 0);
	default_panel_items();
	dotgap_panel = panel_create_item(change_panel, PANEL_SLIDER, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(8),
	      PANEL_LABEL_STRING,	sprintf(buf, "dot gap = %3.1f", cur_dotgap),
	      PANEL_MIN_VALUE,		0,
	      PANEL_MAX_VALUE,		300,
	      PANEL_SLIDER_WIDTH,	300,
	      PANEL_VALUE,		(int)(10*cur_dotgap),
	      PANEL_SHOW_RANGE,		FALSE,
	      PANEL_LAYOUT,		PANEL_VERTICAL,
	      PANEL_NOTIFY_PROC,	new_default_style_values,
	      0);
	dashlength_panel = panel_create_item(change_panel, PANEL_SLIDER, 
	      PANEL_LABEL_X,		ATTR_COL(2),
	      PANEL_LABEL_Y,		ATTR_ROW(10),
	      PANEL_LABEL_STRING,	sprintf(buf, "dash length = %3.1f", cur_dashlength),
	      PANEL_MIN_VALUE,		0,
	      PANEL_MAX_VALUE,		300,
	      PANEL_SLIDER_WIDTH,	300,
	      PANEL_VALUE,		(int)(10*cur_dashlength),
	      PANEL_SHOW_RANGE,		FALSE,
	      PANEL_LAYOUT,		PANEL_VERTICAL,
	      PANEL_NOTIFY_PROC,	new_default_style_values,
	      0);
	property_selected();
	(void)window_fit(change_panel);
	(void)window_fit(change_frame);
	(void)window_set(change_frame, WIN_SHOW, TRUE, 0);
	win_setcursor(canvas_swfd, cur_cursor);
}

static
get_new_default_values()
{
	cur_texttype = (int)panel_get_value(subtype_panel);
	cur_textfont = (int)panel_get_value(font_panel);
	cur_textsize = atoi(panel_get_value(font_size_panel));
	cur_textangle = M_PI/180*atoi(panel_get_value(angle_panel));
	cur_dotgap = (int)panel_get_value(dotgap_panel)/10.0;
	cur_dashlength = (int)panel_get_value(dashlength_panel)/10.0;
	cur_textflags = (panel_get_value(rigid_panel) ? RIGID_TEXT : 0)
  		| (panel_get_value(special_panel) 	? SPECIAL_TEXT	: 0)
 		| (panel_get_value(psfont_panel)	? PSFONT_TEXT	: 0);
	get_default_vals();
}

done_defaults()
{
	switch(button_result) {
	case APPLY:
		get_new_default_values();
		break;
	case DONE:
		get_new_default_values();
		change_selected();
		defaults = 0;
		break;
	case CANCEL:
		change_selected();
		defaults = 0;
		break;
	}
}

#define C_THICKNESS		0
#define	C_COLOR			1
#define C_DEPTH			2
#define	C_STYLE			3
#define	C_PEN			4
#define C_AREA_FILL		5
#define C_SUBTYPE		6
#define C_PSFONT		7
#define C_FONT			8
#define	C_RIGID			9
#define C_SPECIAL		10
#define C_FONT_SIZE		11
#define C_ANGLE			12
#define C_HIDDEN		13

#define compound_item_selected(bit) \
	(value & (1 << (bit)))

#define	new_generic_compound_vals(x) \
	new_generic_values(NULL, NULL); \
	if (compound_item_selected(C_THICKNESS)) x->thickness = generic_vals.thickness; \
	if (compound_item_selected(C_COLOR)) x->color = generic_vals.color; \
	if (compound_item_selected(C_DEPTH)) x->depth = generic_vals.depth; \
	if (compound_item_selected(C_STYLE)) {x->style= generic_vals.style; \
	x->style_val = generic_vals.style_val;} \
	if (compound_item_selected(C_PEN)) x->pen = generic_vals.pen; \
	if (compound_item_selected(C_AREA_FILL)) x->area_fill = generic_vals.area_fill

compound_panel_items()
{
	char	buf[32];

	/*
	 * Additional items to go in the compound window beyond what's
	 * already in the generic and default windows.
	 */
	compound_item = panel_create_item(change_panel, PANEL_TOGGLE,
		PANEL_LAYOUT,		PANEL_VERTICAL,
		PANEL_LABEL_STRING,	"Items to change:",
		PANEL_LABEL_X,		ATTR_COL(50),
		PANEL_LABEL_Y,		ATTR_ROW(2),
		PANEL_CHOICE_STRINGS,	"Thickness", "Color", "Depth", "Style",
					"Pen", "Area fill", "Text alignment",
					"Text font family", "Text font",
					"Rigid text", "Special text",
					"Font size", "Text angle", "Hidden text", 0,
		0);

	hidden_text_panel = panel_create_item(change_panel, PANEL_CYCLE,
	    PANEL_LABEL_X,		ATTR_COL(2),
	    PANEL_LABEL_Y,		ATTR_ROW(19),
	    PANEL_VALUE_X,		ATTR_COL(20),
	    PANEL_VALUE_Y,		ATTR_ROW(19),
	    PANEL_LABEL_STRING,		"Hidden:",
	    PANEL_CHOICE_STRINGS,	"no", "yes", 0,
	    PANEL_VALUE,		(cur_textflags == DEFAULT ||
					 (cur_textflags & HIDDEN_TEXT)),
	    0);

}

make_window_compound(c)
F_compound	*c;
{
	win_setcursor(canvas_swfd, &wait_cursor);
	new_compound = copy_compound(c);
	new_compound->next = c;
	put_default_vals();
	generic_window("Compound", NULL, done_compound, 1, 0);
	default_panel_items();
	compound_panel_items();
	property_selected();
	(void) window_fit(change_panel);
	(void) window_fit(change_frame);
	(void) window_set(change_frame, WIN_SHOW, TRUE, 0);
	win_setcursor(canvas_swfd, cur_cursor);
}

static
get_new_compound_values(com)
F_compound	*com;
{
	F_line		*l;
	F_spline	*s;
	F_ellipse	*e;
	F_text		*t;
	F_arc		*a;
	F_compound	*c;
	unsigned int	value;
	
	/*
	 * We can only change something if the user has specified
	 * that a particular panel item is to be changed. Otherwise,
	 * we ignore the item.
	 */
	value = (unsigned int) panel_get_value(compound_item);
	 
	/*
	 * Scan through each set of objects in the compound and, if
	 * an applicable item has been selected, update that value
	 * in the object.
	 */
	for (l = com->lines; l != NULL; l = l->next) {
		new_generic_compound_vals(l);
	}
	for (s = com->splines; s != NULL; s = s->next) {
		new_generic_compound_vals(s);
	}
	for (e = com->ellipses; e != NULL; e = e->next) {
		new_generic_compound_vals(e);
	}
	for (t = com->texts; t != NULL; t = t->next) {
		if (compound_item_selected(C_SUBTYPE)) t->type =
			(int)panel_get_value(subtype_panel);
		if (compound_item_selected(C_FONT)) t->font =
			(int)panel_get_value(font_panel);
		if (compound_item_selected(C_FONT_SIZE)) t->size =
			atoi(panel_get_value(font_size_panel));
		if (compound_item_selected(C_ANGLE)) t->angle =
			M_PI/180*atoi(panel_get_value(angle_panel));
		if (compound_item_selected(C_RIGID)) {
			t->flags &= ~RIGID_TEXT;
			t->flags |= (panel_get_value(rigid_panel) ? RIGID_TEXT : 0);
		}
		if (compound_item_selected(C_SPECIAL)) {
			t->flags &= ~SPECIAL_TEXT;
			t->flags |= (panel_get_value(special_panel) ? SPECIAL_TEXT : 0);
		}
		if (compound_item_selected(C_PSFONT)) {
			t->flags &= ~PSFONT_TEXT;
			t->flags |= (panel_get_value(psfont_panel) ? PSFONT_TEXT : 0);
		}
		if (compound_item_selected(C_HIDDEN)) {
			t->flags &= ~HIDDEN_TEXT;
			t->flags |= (panel_get_value(hidden_text_panel) ? HIDDEN_TEXT : 0);
		}
	}
	for (a = com->arcs; a != NULL; a = a->next) {
		new_generic_compound_vals(a);
	}
	for (c = com->compounds; c != NULL; c = c->next) {
		get_new_compound_values(c);
	}
}

done_compound()
{
	F_compound	*old_compound;
	
	pw_batch_on(canvas_pixwin);
	erase_compound(new_compound);
	draw_compoundbox(new_compound, ERASE);
	old_compound = new_compound->next;
	switch(button_result) {
		case APPLY:
			get_new_compound_values(new_compound);
			/* 
			 * recompute the compound's bounding box
			 */
			compound_bound(new_compound, &new_compound->nwcorner.x,
				       &new_compound->nwcorner.y,
				       &new_compound->secorner.x,
				       &new_compound->secorner.y);
			draw_compound(new_compound);
			draw_compoundbox(new_compound, PAINT);
			break;
		case DONE:
			clean_up();
			get_new_compound_values(new_compound);
			set_action_object(F_CHANGE, O_COMPOUND);
			new_compound->next = NULL;
			delete_compound(&objects.compounds, old_compound);
			insert_compound(&objects.compounds, new_compound);
			set_latestcompound(old_compound);
			old_compound->next = new_compound;
			change_selected();
			/* 
			 * recompute the compound's bounding box
			 */
			compound_bound(new_compound, &new_compound->nwcorner.x,
				       &new_compound->nwcorner.y,
				       &new_compound->secorner.x,
				       &new_compound->secorner.y);
			draw_compound(new_compound);
			draw_compoundbox(new_compound, PAINT);
			show_pointmarker();
			set_modifiedflag();
			break;
		case CANCEL:
			clean_up();
			new_compound->next = NULL;
			free_compound(&new_compound);
			change_selected();
			draw_compound(old_compound);
			draw_compoundbox(old_compound, PAINT);
			show_pointmarker();
			break;
	}
	pw_batch_off(canvas_pixwin);
}

