/* copyright 1989 Phil Andrews, Pittsburgh Supercomputin Center */
/* all right reserved */
/* taken over from Chris Nuuja */
/****************************************************************************/
/*									    */
/*  GPLOT driver for the SUN workstation with SunCGI                        */
/*  Author: Christopher Nuuja, Pittsburgh Super Computing                   */
/*  Version 0.5 ( Monochrome devices handled )
/*  Version 0.2 ( Everything works, could use some cleaning up) 7/5/88	    */
/*  Version 0.1 ( Cell arrays, no edge attributes ) 7/1/88		    */
/*  Version 0.0 (No cell array routines yet) 6/18/88                        */
/*                                                                          */
/****************************************************************************/




#include <stdio.h>	/* to write diagnostic and error messages */
#include <cgidefs.h>	/* for scgi entry points */
#include "defs.h"	/* structures to interface to GPLOT */


/* The following macro is a shorthand for the debug command line option */
#define debugmode  Op[(int) debug].val.i 



/*
   Function return code conventions:  Unless otherwise stated,
   the following function return values have the following
   meanings:

	0	Terrible error, die immediately
	1	Successful function execution
	2	The function has made a survivable error
	4	The function has detected but not introduced a
		survivable error
	8	No error, but for some reason the device is currently
		incapable of fulfilling the request

*/




/*
			GLOBAL VARIABLE DECLARATION
	The following are global variables used by this driver to hold
	information about the current state of the device or GPLOT
*/


/*
   Pointers to provide storage for the driver's own copies of
   important gplot structure pointers.  These structures contain gplot internal
   values needed to interface with the driver.  These are defined in defs.h  
*/
static struct one_opt *Op;		/* Command line option array */
static struct mf_d_struct *State_c1;	/* Current CGM metafile desc. data */
static struct pic_d_struct *State_c2;	/* Current CGM picture descrip. data */
static struct control_struct *State_c3;	/* Current CGM control element data */
static struct attrib_struct *State_c5;	/* Current CGM attribute data */
static struct info_struct *Dev_info;	/* Device characteristics */
					/* communicated via this struct */
static int (**Ctrl_funcs)();


/*
   state_level is used as an image counter. State_level starts off -1
   (set by (float) scgi_setup), is incremented each time scgi_begin gets called
   ,and is decremented each time scgi_end gets called.  It may exceed zero if
   GPLOT is being used to composite multiple images on the screen;  in this
   case the driver will be called with an extra set of scgi_begin, scgi_bpage
   and scgi_epage, scgi_end calls bracketting the full set of calls associated
   with the composited images (including their scgi_begin's and scgi_end's).
   Thus scgi_bpage and scgi_epage have to check state_level to avoid doing
   repeat initializations.
*/
static Cint state_level;
 
/* 
   monochrome is used to determine whether or not to execute color related
   code.  It is set to null if the device is a monochome one.
   It is initialized during scgi_setup.
*/
static enum boolean monochrome;


/*
   Data concerning the physical display device, set by scgi_setup 
   xunits and yunuts are pixels per mm.  xext and yext are the dimensions of
   the screen in pixels
*/
static Cfloat xunits, yunits;
static Cint xext, yext;


/*
   Coordinate buffer, enlarged as needed by scgi_getcmem 
*/
static Ccoor *coorbuf;
static Cint coor_buf_sz=0;

/* 
   Cell array row buffers, statically allocated 
*/
#define ROWSIZE 1024
static Cint rowbuf[ROWSIZE];
static float redbuf[ROWSIZE],greenbuf[ROWSIZE],bluebuf[ROWSIZE];


/* 
   Image buffer, enlarged as needed by scgi_getimem
*/
static Cint *ibuf;
static Cint ibuf_sz=0;



/* 
   Variables to hold the minimum and maximum character sizes and the 
   default character height  NOTE d_c_height is not used
*/
static Cint c_h_min, c_h_max, d_c_height;
#define NUMFONTS 6

/*  
   A remapping of the font types and their respective indicies.  This is
   necessary because GPLOT often chooses the symbol font by default, and that
   font is for greek letters and symbols, not alphabetical characters.  This
   remapping makes the default font much more reasonable
*/
static Cint font_list[NUMFONTS] = {5,0,1,2,3,4};

/* 
   Total number of color indices available (ColorTable size). 
   Set up by scgi_setup
*/
static Cint clr_t_sz=0;

/*
   Direct color simulation variables, set by scgi_fake_dc and
   scgi_dc_colors.   The pointer clr_tbl points to a vector of
   floating point numbers.  Every three numbers represent the intensity of the
   red, green and blue elements of a color, thus every three consecutive
   floating point numbers represent one color.  This is not the physical
   color table for the Sun CGI system. It is designed to hold the direct color
   table representation. It is read into the physical color table in the
   procedure scgi_dc_colours.   Procedure scgi_ctab also uses an array of
   floating point numbers to represent  the color table it inserts,
   but that array is provided by GPLOT.  The flag dc_ready_flag is set by the
   function scgi_fake_dc.  dc_ready_flag is only reset by scgi_ctab.
   The driver believes it is in direct color or indexed color mode based on
   whether the color selection mode in State_c2 is or is not non-zero 
   at the time of the call to scgi_bpage.

   The pack_clr macro creates an integer between 0 and 255 out of the supplied
   red, green, and blue intensities ir,ig, and ib.  This integer is
   used as an index to the array best_clr, which will hold the index value of
   the physical color table that best matches the ir, ig, and ib intensitities.
   f_to_b is used to convert a floating point color-element intensity (read, 
   green, or blue), to a binary value <= 255.
*/
#define BEST_C_SZ 32768   /* = 32**3;  not a free parameter! */
#define f_to_b(f) ( (unsigned char) ((int)((f)*255.0) % 256 ))
#define b_to_f(b) ( (1./255.)*((float) (b)) )
#define mask_8_thru_4 248
#define pack_clr(ir,ig,ib) \
  	( ( ( (ir) & mask_8_thru_4 )<<7 ) \
	| ( ( (ig) & mask_8_thru_4 )<<2 ) \
	| ( ( (ib) & mask_8_thru_4 )>>3 ) )
#define pattern_clr(ir,ig,ib) \
	( 2*6*(ir / 64) + 2*(ig / 43) + (ib / 128) )
	
static Cint dc_init_flag=0, dc_ready_flag=0, dc_totclrs;
static Cint *best_clr;
static float *clr_tbl; 

/*
   The following variables are used for faking a gray scale for polygon 
   fill-colors on a monochrome devices.  SunCGI has a writable pattern table
   of size 50.  This is non-standard with cgm, but without it most metafile's
   are unreadable.  clr_pattern is an array of 3x3 pixel arrays.  Each 9
   digits represent a row-wise description of the pixels in that 3x3  pixel
   array.  These 3x3 arrays are replicated all over the interior of the
   polygon to be filled.  The formula y=.3*red + .59*green +.11*blue is used to
   determine intensity. 9*0.3=3, 9*0.59=5, 9*.11=1 (approximately), so there
   are 3 possible red pixels, 5 possible green pixels, and 1 possible blue
   pixel.  There are, therefore, 48 possible patterns.  pattern_clr()
   is used to convert red, green, and blue intensities into the matching
   pattern index for that color.
*/

#define PATTERN_TBL_SZ 48 	/* 6 green pixel possibilies * 4 red *2 blue */
#define PATTERN_SZ 9		/* 5 green + 3 red + 1 blue pixels = 9 */
#define PATT_WIDTH 3


static Cint clr_pattern[PATTERN_TBL_SZ][PATTERN_SZ] = {
	 1,1,1,1,1,1,1,1,1,    1,1,1,1,1,1,1,0,1,    0,1,1,1,1,1,1,1,1,
	 0,1,1,1,1,1,1,0,1,    0,1,1,1,1,1,1,1,0,    0,1,1,1,1,1,1,0,0, 
	 0,1,1,1,0,1,1,1,0,    0,1,1,1,0,1,1,0,0,    0,1,1,1,0,1,0,1,0,
	 0,1,1,1,0,1,0,0,0,    0,1,0,1,0,1,0,1,0,    0,1,0,1,0,1,0,0,0,  
	 1,0,1,1,1,1,1,1,1,    1,0,1,1,1,1,1,0,1,    0,0,1,1,1,1,1,1,1,  
	 0,0,1,1,1,1,1,0,1,    0,0,1,1,1,1,1,1,0,    0,0,1,1,1,1,1,0,0, 
	 0,0,1,1,0,1,1,1,0,    0,0,1,1,0,1,1,0,0,    0,0,1,1,0,1,0,1,0, 
	 0,0,1,1,0,1,0,0,0,    0,0,0,1,0,1,0,1,0,    0,0,0,1,0,1,0,0,0, 
	 1,0,1,1,1,0,1,1,1,    1,0,1,1,1,0,1,0,1,    0,0,1,1,1,0,1,1,1, 
	 0,0,1,1,1,0,1,0,1,    0,0,1,1,1,0,1,1,0,    0,0,1,1,1,0,1,0,0,  
	 0,0,1,1,0,0,1,1,0,    0,0,1,1,0,0,1,0,0,    0,0,1,1,0,0,0,1,0, 
	 0,0,1,1,0,0,0,0,0,    0,0,0,1,0,0,0,1,0,    0,0,0,1,0,0,0,0,0, 
	 1,0,1,0,1,0,1,1,1,    1,0,1,0,1,0,1,0,1,    0,0,1,0,1,0,1,1,1,  
	 0,0,1,0,1,0,1,0,1,    0,0,1,0,1,0,1,1,0,    0,0,1,0,1,0,1,0,0, 
	 0,0,1,0,0,0,1,1,0,    0,0,1,0,0,0,1,0,0,    0,0,1,0,0,0,0,1,0,
	 0,0,1,0,0,0,0,0,0,    0,0,0,0,0,0,0,1,0,    0,0,0,0,0,0,0,0,0 };


/*  
   If the top index and bottom index of the physical color table are set
   to the same value, SunCGI flips out and assigns white to the low index
   and black to the top index.  To prevent this,  the top index is maintained
   as a buffer color.  It is not written on unless the bottom index is about
   to be changed to an identical color.  These variables hold the current 
   color of the buffer color.  If the bottom index is about to be changed to 
   an identical color,  the buffer color is changed and these variables are
   updated.
*/
static unsigned char topRed=255, topGreen=255, topBlue=255;


/* 
   Variables needed by SCI.  Scgi_window and device initilaized by
   scgi_device_setup.  int_style and edg_vis initialized in scgi_bpage.
   Global values are needed for the interior style and edge visibility 
   attributes because the sun cgi function that sets them wants to  sets them 
   both at the same time, and gplot wants to set them individually.
*/

static Cint scgi_window; 	
static Cvwsurf scgi_device;
static Cintertype scgi_int_style;
static Cflag scgi_edg_vis;


/*
   timout to go to next images, or the end of metafile if no mouse input happens
*/
#define HOLD_FOR_INPUT (100 * 1000 * 1000)




/* 
			FUNCTION DECLARATIONS 
*/


 

/*
	This routine opens up the scgi window ( this driver will only open 
	at most one window at any one time, and it will be the size of the
	screen )
*/
scgi_begin(comment, file_name, prog_name)
char *comment, *file_name, *prog_name;
{

	if ( debugmode )
		{
		fprintf(stderr," scgi_begin: \n");
		fprintf(stderr,"    comment: <%s>\n",comment);
		fprintf(stderr,"    file name: <%s>\n",file_name);
		fprintf(stderr,"    program name: <%s>\n",prog_name);
		}
                                                  
	state_level++;
	if (state_level > 0) return(1);  /* already set up a window */

	/* else set up window with the file name as window title */

	return(1);
}

/* 
	This routine shuts down the SCGI window if GPLOT is not 
	currently in a composing state.
*/ 
scgi_end(pages_done)
int pages_done;
{
	if (debugmode)
		fprintf(stderr," scgi_end: pages done = %d\n",pages_done);
	state_level--;
	if (state_level > 0) return(1);  /* not yet done */

	close_vws(scgi_window);
	close_cgi();
	if (coor_buf_sz>0) { free(coorbuf); coor_buf_sz= 0; };
	if (clr_t_sz>0) { free(clr_tbl); };

	return(1);
}

/*                                   
	This routine starts a new page.  The offsets xoff and yoff are in 
	in pixels.  Rotation is in degrees.  SCGI offers no rotation features,
	so this parameter is simply ignored.  rg, gb, and bb are the intensities
	of the red, green, and blue color-elements of the backround color.
*/                                                              
scgi_bpage(pic_title,xoff, yoff, rotation, rb, gb, bb, pgnum, xsize, ysize)
char *pic_title;
float rotation, rb, gb, bb;
int xoff, yoff, pgnum;
int xsize, ysize;
{                   
	Cint rc,index;
	float bclist[3];
	char *malloc();

	if (debugmode) 
 		{
		fprintf(stderr,
			" scgi_bpage:   page number %d\n",pgnum);
		fprintf(stderr,"    x, y offset, rotation = %d %d %f\n",
			xoff,yoff,rotation);
		};
	

	if (state_level > 0)  return(1); 
		/* The page is already set up */
	

	/* Clear the workstation window and set backround color if not mono*/
		/* first change color table index 0 to that of the bckgrd */
		/*  This must be done for the monochrome device too, because */
		/*  although the background color is not changed, that */
		/*  color must still be added to the color_pattern table  */
	bclist[0] = rb;
	bclist[1] = gb;
	bclist[2] = bb;
	index = 0;
	scgi_ctab(index, 1, bclist);
	clear_view_surface(scgi_window, ON, index);

	/*
	Set proper color selection mode for this frame, skipping
	index 0 (just set) if it's indexed color 
	*/
	if (State_c2->c_s_mode) 	/* Direct color */
		{ 
		if ( !dc_ready_flag ) rc= scgi_dc_colors(); 
		else rc= 1;
		}
                                                     
	else		/* Indexed color - load color table */
	   	rc= scgi_ctab(1, State_c1->max_c_index, State_c5->ctab);
	if (!rc) return(rc);
                                                        
	/* Reset some attributes to their proper values */
	rc= scgi_t_align(State_c5->text_align.hor,State_c5->text_align.ver,
		State_c5->text_align.cont_hor,State_c5->text_align.cont_ver);
	if (!rc) return(rc);
	rc= scgi_c_height(State_c5->c_height);
	if (!rc) return(rc);
	rc= scgi_t_font(State_c5->t_f_index);
	if (!rc) return(rc);
	rc= scgi_mk_size(State_c5->mk_size.i,State_c5->mk_size.r);
	if (!rc) return(rc);
	rc= scgi_mk_type(State_c5->mk_type);
	if (!rc) return(rc);
	rc= scgi_e_vis(State_c5->edge_vis);
	if (!rc) return(rc);
	rc= scgi_fl_style(State_c5->int_style);
	if (!rc) return(rc);
	rc= scgi_l_type(State_c5->line_type);
	if (!rc) return(rc);
	rc= scgi_l_width(State_c5->line_width.i,State_c5->line_width.r);
	if (!rc) return(rc);
	rc= scgi_e_type(State_c5->edge_type);
	if (!rc) return(rc);
	rc= scgi_e_width(State_c5->edge_width.i,State_c5->edge_width.r);
	if (!rc) return(rc);
	rc= scgi_fl_colour(State_c5->fill_colour.red,State_c5->fill_colour.green,
		      	State_c5->fill_colour.blue,State_c5->fill_colour.ind);
	if (!rc) return(rc);

	if (monochrome) return(rc);

	/* Else, a colour device */

	rc= scgi_t_colour(State_c5->text_colour.red,State_c5->text_colour.green,
			State_c5->text_colour.blue,State_c5->text_colour.ind);
	if (!rc) return(rc);
	rc= scgi_mk_colour(State_c5->mk_colour.red,State_c5->mk_colour.green,
			State_c5->mk_colour.blue,State_c5->mk_colour.ind);
	if (!rc) return(rc);
	rc= scgi_l_colour(State_c5->line_colour.red,State_c5->line_colour.green,
			State_c5->line_colour.blue,State_c5->line_colour.ind);
	if (!rc) return(rc);
	rc= scgi_e_colour(State_c5->edge_colour.red,State_c5->edge_colour.green,
			State_c5->edge_colour.blue,State_c5->edge_colour.ind);
  
	return(rc);
}                                                                     

/* 
	This routine ends the page.  With SunCGI, we do not need to do anything
	here.
*/
			
scgi_epage(copies)
int copies;
{

	if (debugmode) fprintf(stderr," scgi_epage: copies = %d\n",copies);

	/* 
	state_level might be relevant here, but isn't needed for this
	device.  No need to call deactivate window since we only open at most
	one window at a time
	*/
	scgi_ready_prompt();
    	clear_view_surface(scgi_window, ON, 0);

	return(1);                                              
}
                                                       
/*                                                          
  	Set the text colour with an index for the color table. The color table 
	is filled with GPLOT's indexed color table, or the direct_color
	approximation table, depending on whether GPLOT is in direct-color-mode
	or not.  If GPLOT is in direct-color-mode, then we use the
	best_clr macro on the color-element values r, g, and b to get the best
	index to the color table, else we use the variable index as the index.
*/
scgi_t_colour(r, g, b, index)
float r, g, b;
int index;
{
	Cint rc= 1;

	if (debugmode) 
		fprintf(stderr," scgi_t_colour: r, g, b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (monochrome) return(1);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else 
			{
			fprintf(stderr, 
		     	   " Direct text color with indexed color set!\n");
			rc= 4;
			index= 1;
			}
		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
			fprintf(stderr, 
			   " Indexed text color with direct color set!\n");
			rc= 4;
			};
		};

	if ( index < clr_t_sz ) text_color(index);
	else rc= 8;

	return(rc);
}
                
/* 
	Write text.
*/
scgi_text(x, y, is_cont, buffer)
int x, y;
enum boolean is_cont;
char *buffer;
{
	Ccoor location;

	if (debugmode) 
		{
		fprintf(stderr," scgi_text: x= %d, y= %d\n",x,y);
		fprintf(stderr,"      text=<%s>\n",buffer);
		}
	location.x = x;
	location.y = y;
	text(&location,buffer);
	return(1);
}

/*
	Set the text alignment.
*/
scgi_t_align(hor, ver, cont_hor, cont_ver)
enum hor_align hor;
enum ver_align ver;
float cont_hor, cont_ver;
{
	Cint rc=1;		/* the return code */
	Chaligntype halign;
	Cvaligntype valign;

	if (debugmode) 
		fprintf(stderr," scgi_t_align: hor= %d, ver= %d\n",hor,ver);
	if ((int)hor>5)
		{
		fprintf(stderr,
		   " Unknown or unimplemented hor = %d\n", hor);
		return(8);
		}
	if ((int)ver>7)
		{
		fprintf(stderr,
		   " Unknown or unimplemented ver = %d\n", ver);
		return(8);
		}
	switch (hor) 
		{
		case normal_h : halign = NRMAL; break;
		case left_h   : halign = LFT; break;
		case center_h : halign = CNTER; break;
		case right_h  : halign = RGHT; break;
		case cont_h   : halign = CNT; break;
		default	      : fprintf(stderr,
		  "Unknown or unsupported alignment type; using NRMAL . \n");
	   	   		  halign = NRMAL;
				  rc = 2;
				  break;
		}
	switch (ver) 
		{
		case normal_v : valign = NORMAL; break;
		case top_v    : valign = TOP; break;
		case cap_v    : valign = CAP; break;
		case half_v   : valign = HALF; break;
		case base_v   : valign = BASE; break;
		case bottom_v : valign = BOTTOM; break;
		case cont_v   : valign = CONT; break;
		default	      : fprintf(stderr,
		  "Unknown or unsupported alignment type; using 'NORMAL'.\n");
				  valign = NORMAL;
				  rc = 2;
		}
	text_alignment(halign, valign, cont_hor, cont_ver);
/*	halign = hor_align_list[ (int) hor ];
	valign = valign_list[ (int) ver ];
*/

	return(rc);
}

/*
Plot a set of lines.
*/
scgi_pline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	Ccoorlist polycoors;	/* The list of points */
	int rc=1;


	if (debugmode) 
		fprintf(stderr," scgi_pline: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs <= 1 ) return(1);

	rc = make_coorlist(x1_ptr,y1_ptr,&polycoors,no_pairs);

	polyline(&polycoors);

	return(rc);
}

/* plot a set of lines between alternate points */
scgi_dpline(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	Ccoorlist polycoors;
	int rc=1;

	if (debugmode) 
		fprintf(stderr," scgi_dpline: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs <= 1 ) return(1);
	rc = make_coorlist(x1_ptr,y1_ptr,&polycoors,no_pairs);
	disjoint_polyline(&polycoors);

	return(rc);
}

/* 
	Set the marker colour.  Very similiar to scgi_text_colour.
*/
scgi_mk_colour(r, g, b, index)
float r, g, b;
int index;
{
	Cint rc= 1;

	if (debugmode) 
		fprintf(stderr," scgi_mk_colour: r, g, b, index= %f %f %f %d\n",
			r,g,b,index);

	if (monochrome) return(1);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
	    	{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else 
			{
			fprintf(stderr, 
			   " Direct marker color with indexed color set!\n");
			rc= 4;
		       	index= 1;
			}
      		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
			fprintf(stderr, 
			   " Indexed marker color with direct color set!\n");
			rc= 4;
			};
		};

	if ( index < clr_t_sz ) marker_color(index);
	else rc= 8;

	return(rc);
}

/*
	Set marker size.  This driver implementation uses sizes in absolute
	(number of pixels), so mk_a_size is the paramter that will be used.
*/
scgi_mk_size(mk_a_size,mk_s_size)
int mk_a_size;
float mk_s_size;
{
	if (debugmode) fprintf(stderr," scgi_mk_size: %d pixels /n",
				mk_a_size);

	marker_size( (float) mk_a_size );

	return(1);	
}

/*
	Set marker type.
*/
scgi_mk_type(marker)
int marker;
{                                              
	Cint rc=1;		/* the return code */
	Cmartype type;

	if (debugmode) fprintf(stderr,
			" scgi_mk_type: marker type set to %d\n",marker);

	switch(  marker )
		{
		case 1:	type= DOT; break;
		case 2:	type= PLUS; break;
		case 3:	type= ASTERISK ;break;
		case 4:	type= CIRCLE ;break;
		case 5:	type= X ;break;
		default	     :	fprintf(stderr,
		  "Unknown or unsupported marker type; using '*'.\n");
				type= ASTERISK;
				rc = 2;
		};

	marker_type(type);

	return(rc);
}

/* 
	Plot a series of markers.
*/
scgi_pmarker(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	Ccoorlist polycoors;
	int rc=1;


	if (debugmode) 
		fprintf(stderr," scgi_pmarker: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs < 1 ) return(1);

	rc = make_coorlist(x1_ptr,y1_ptr,&polycoors,no_pairs);
	polymarker(&polycoors);
	return(rc);
}

/* 
      	Set the text font. 
*/
scgi_t_font(index)
int index;
{
	Cint font;

	if (debugmode) fprintf(stderr," scgi_t_font:  font index= %d \n",
		index);

	if ( (index<=NUMFONTS) && (index>0) )	
		{ 
		font = font_list[index];
		text_font_index(font);
		}
	else
		fprintf(stderr," Font index %d out of range; ignored.\n",index);
	return(1);
}

/*        
  	Set character height, and set Dev_info->c_height and c_width to
	the values which result.
*/                
scgi_c_height(t_a_height)
int t_a_height;
{

	if (debugmode) fprintf(stderr,
		" scgi_c_height: character height set to %d pixels\n",
		t_a_height);

	/*
	   Note, character_height sets the height in VDC units, but this is
	   the same as pixel units since the VDC maps directly onto the screen
	*/

	if ( t_a_height < c_h_min ) Dev_info->c_height= c_h_min;
	else if ( t_a_height > c_h_max ) Dev_info->c_height= c_h_max;
	else Dev_info->c_height= t_a_height;
	/* Assume square characters */
	Dev_info->c_width= Dev_info->c_height;

	character_height(Dev_info->c_height);

	return(1);
}

/*
	Set filled area interior style.
*/
scgi_fl_style(style)
enum is_enum style;
{
	Cint rc=1;		/* the return code */

	if (debugmode) fprintf(stderr,
			" scgi_fl_style: interior style set to style %d\n",
			(int)style);

	switch( style )
		{
		case hollow  :	scgi_int_style = HOLLOW; break;
		case solid_i :	if (monochrome)
					{
					scgi_int_style = PATTERN;
					pattern_size(1,1); break; 
					/* size of one block in the */
					/* pattern = 1 pixel */
					}
				else
					{
					scgi_int_style = SOLIDI; break;
					
					}
		case pattern :	scgi_int_style = PATTERN;
				pattern_size(3,3);  break;
				/*  This way, a pattern fill looks different */
				/*  than a gray scale color-fill-fake pattern */
		case hatch   :	scgi_int_style = HATCH; break;
		case empty   :	scgi_int_style = HOLLOW; break;
		default	     :  fprintf(stderr,
		      "Unknown or unsupported interior style; using hollow.\n");
				scgi_int_style = HOLLOW;
		   		rc = 2;
		};

	interior_style(scgi_int_style, scgi_edg_vis);
	return(rc);
}

/* 
	Set polygon fill colour.
*/
scgi_fl_colour(r, g, b, index)
float r, g, b;                     
int index;
{
	Cint rc= 1, cindex;		/* the return code */

	if (debugmode) 
		fprintf(stderr," scgi_fl_colour: r,g,b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (monochrome) 
		{
		rc = scgi_mfl_colour(r,g,b,index);
		return(rc);
		}

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			cindex= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else 
			{
			fprintf(stderr, 
			   " Direct fill color with indexed color set!\n");
			rc= 4;
			cindex= 1;
			}
		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
			fprintf(stderr, 
			   " Indexed fill color with direct color set!\n");
			rc= 4;
			cindex= 1;
			}
		else
			cindex = index;
		};

	if ( cindex < clr_t_sz ) fill_color(cindex);
	else rc= 8;

	return(rc);
}

/* 
	Set polygon fill colour as a pattern for monochrome device.
*/
scgi_mfl_colour(r, g, b, index)
float r, g, b;                     
int index;
{
	Cint rc= 1, cindex;		/* the return code */

	if (debugmode) 
		fprintf(stderr," scgi_mfl_colour: r,g,b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			cindex= pattern_clr( 
				f_to_b(r), f_to_b(g), f_to_b(b) ); 
		else 
			{
			fprintf(stderr, 
			   " Direct fill color with indexed color set!\n");
			rc= 4;
			cindex= 0;
			}
		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
			fprintf(stderr, 
			   " Indexed fill color with direct color set!\n");
			rc= 4;
			cindex= 0;
			}
		else
			cindex = index;
		};

	if ( cindex < PATTERN_TBL_SZ ) pattern_index(cindex);
	else rc= 8;

	return(rc);
}
/* 
	Draw a polygon.
*/
scgi_pgon(no_pairs, x1_ptr, y1_ptr)
int no_pairs, *x1_ptr, *y1_ptr;
{
	Ccoorlist polycoors;
	int rc=1;

	if (debugmode) 
		fprintf(stderr," scgi_pgon: %d coordinate pairs.\n",no_pairs);

	if ( no_pairs <= 2 ) return(2);	/* must have at least three points */

	rc = make_coorlist(x1_ptr,y1_ptr,&polycoors,no_pairs);

	if (scgi_int_style == HOLLOW)
		{
		if (monochrome)
			{
			scgi_e_vis(ON);		/* Make sure edge is visible */
			polygon(&polycoors);
			scgi_e_vis(scgi_edg_vis);    /* no permanent change */
			}
		else 		/* a color device */
			{
			scgi_e_vis(ON);
			/* Make edge color same as fill color */
			/* But, only for this polygon         */
			scgi_e_colour(State_c5->fill_colour.red,
			   State_c5->fill_colour.green, 
			   State_c5->fill_colour.blue,
			   State_c5->fill_colour.ind);
			polygon(&polycoors);
			scgi_e_colour(State_c5->edge_colour.red,
			   State_c5->edge_colour.green, 
			   State_c5->edge_colour.blue,
			   State_c5->edge_colour.ind);
			scgi_e_vis(scgi_edg_vis);  /* set it back */
			};
		}
	else      /* Not a hollow fill style */
		polygon(&polycoors);

	return(rc);
}

/*                   
	Set the line type. 
*/              
scgi_l_type(l_type)
enum line_enum l_type;
{
	Cint rc=1;		/* the return code */
	Clintype ttype;

	if (debugmode) fprintf(stderr," scgi_l_type: type= %d.\n",l_type);
	switch (l_type) 
		{
		case solid_l  :	ttype = SOLID; break;
		case dash     :	ttype = DASHED; break;
		case dot_l    :	ttype = DOTTED; break;
		case dash_dot :	ttype = DASHED_DOTTED; break;
		case dash_d_d :	ttype = DASH_DOT_DOTTED; break;
		default       : fprintf(stderr, 
		  "Unknown or unsupported line type: %d; using solid lines.\n",
		   l_type);
				ttype = SOLID;
		   		rc = 2;
		}

	line_type(ttype);

	return(rc);
}

/*                   
	Set the edge type. 
*/              
scgi_e_type(e_type)
enum line_enum e_type;
{
	Cint rc=1;		/* the return code */
	Clintype ttype;

	if (debugmode) fprintf(stderr," scgi_e_type: type= %d.\n",e_type);
	switch (e_type) 
		{
		case solid_l  :	ttype = SOLID; break;
		case dash     :	ttype = DASHED; break;
		case dot_l    :	ttype = DOTTED; break;
		case dash_dot :	ttype = DASHED_DOTTED; break;
		case dash_d_d :	ttype = DASH_DOT_DOTTED; break;
		default       : fprintf(stderr, 
		  "Unknown or unsupported edge type: %d; using solid edges.\n",
		   e_type);
				ttype = SOLID;
		   		rc = 2;
		}

	perimeter_type(ttype);

	return(rc);
}

/* 
Set the line width. 
*/
scgi_l_width(l_a_width,l_s_width)
int l_a_width;
float l_s_width;
{
	if (debugmode) fprintf(stderr," scgi_l_width: width scaled to %d\n",
		l_a_width);

	line_width((Cfloat) l_a_width);

	return(1);
}

/* 
Set the edge width. 
*/
scgi_e_width(e_a_width,e_s_width)
int e_a_width;
float e_s_width;
{
	if (debugmode) fprintf(stderr," scgi_e_width: width scaled to %d\n",
		e_a_width);

	perimeter_width((Cfloat) e_a_width);

	return(1);
}

/*
	Set the line colour.
*/
scgi_l_colour(r, g, b, index)
float r, g, b;
int index;
{
	Cint rc= 1;                       
                                                                
	if (debugmode) 
		fprintf(stderr," scgi_l_colour: r, g, b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (monochrome) return(1);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else                                 
			{
			fprintf(stderr, 
			   " Direct line color with indexed color set!\n");
			rc= 4;
			index= 1;
			}
     		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
	       		fprintf(stderr, 
			   " Indexed line color with direct color set!\n");
			rc= 4;
	    		};
		};

	if ( index < clr_t_sz ) line_color(index);
	else rc= 8;

	return(rc);
}

/*
	Set the edge colour.
*/
scgi_e_colour(r, g, b, index)
float r, g, b;
int index;
{
	Cint rc= 1;                       
                                                                
	if (debugmode) 
		fprintf(stderr," scgi_e_colour: r, g, b, index= %f %f %f %d\n", 
			r,g,b,index);

	if (monochrome) return(1);

	if (State_c2->c_s_mode == d_c_mode) /* Direct color */
		{
		if (dc_ready_flag)
     			index= best_clr[
				pack_clr( f_to_b(r), f_to_b(g), f_to_b(b) )];
		else                                 
			{
			fprintf(stderr, 
			   " Direct edge color with indexed color set!\n");
			rc= 4;
			index= 1;
			}
     		}

	else /* Indexed color */
		{
		if (dc_ready_flag) 
			{
	       		fprintf(stderr, 
			   " Indexed edge color with direct color set!\n");
			rc= 4;
	    		};
		};

	if ( index < clr_t_sz ) perimeter_color(index);
	else rc= 8;

	return(rc);
}
/* 
      	Set the edge visibility flag. 
*/
scgi_e_vis(vis)
enum boolean vis;
{
	if (debugmode) fprintf(stderr," scgi_e_vis:  set to = %d \n",
		(int) vis);

	if ( (vis == on) || (vis == off) )	
		{
		scgi_edg_vis = (Cflag) vis;	/* change edge-vis global */
		interior_style(scgi_int_style, scgi_edg_vis);
		}
	else
		fprintf(stderr,"Edge visibility flag invalid; call ignored.\n");
	return(1);
}

/* 
	Draw a cell array.
*/
scgi_carray(p, q, r, nx, ny, prec, image_ptr, mode, no_bytes)
int p[2], q[2], r[2], nx, ny, prec, mode;
unsigned char *image_ptr;
long int no_bytes;
{
	Cint imagesize,rc,rdummy[2],zero= 0;
	Ccoor pcoor, qcoor, rcoor;
	Cint *colorind;

	if (debugmode) 
		{
		fprintf(stderr,
			" scgi_carray: nx= %d, ny= %d, prec= %d, mode= %d\n",
			nx,ny,prec,mode);
		fprintf(stderr,"      p= [%d,%d], q= [%d,%d], r= [%d,%d].\n",
			p[0],p[1],q[0],q[1],r[0],r[1]);
		};


	/* Check row buffer space availability */
	if ( nx > ROWSIZE )
		{
		fprintf(stderr,
		  " Cell array too wide for internal buffer; call ignored.\n");
		return(8);
		};

	/* Make sure buffer memory is available */
	imagesize= nx*ny;
	if ( scgi_getimem(imagesize) != 1 )
		{ 
		fprintf(stderr," Error allocating memory for image buffer.");
		return(2);
		};

	/* Set up coordinates */
	pcoor.x = p[0]; pcoor.y = p[1];
	qcoor.x = q[0]; qcoor.y = q[1];
	rcoor.x = r[0]; rcoor.y = r[1];


	/* Copy the image into the integer image buffer */

	if (monochrome)      /* Ibuf will be filled with zeros and ones */ 
		scgi_cmca(nx,ny,prec,mode,image_ptr);
	else if (State_c2->c_s_mode)		/* Direct color */    
		{
		if (dc_ready_flag)
			scgi_cdca(nx,ny,prec,mode,image_ptr);
		else 
			{
			fprintf(stderr, 
			  " Direct color cell array with indexed color set!\n");
			return(4);
			}
	    	}

 	else		/* Indexed color */
		{
	     	if (!dc_ready_flag)
			scgi_cica(nx,ny,prec,mode,image_ptr);
		else 
			{
			fprintf(stderr, 
			  " Indexed color cell array with direct color set!\n");
			return(4);
			}
	    	}
	colorind = ibuf;

	/* Draw the cell array */
	cell_array(&pcoor, &qcoor, &rcoor, nx, ny, colorind);

	return(1);
}


/*                
	This routine implements the copying of a cell array into the 
	(previously allocated) image memory buffer and transforming it into 
	monochrome.  Return 1 if successful.  The acronym is 
	'scgi_copy_monochrome_cell_array'.
*/

scgi_cmca(nx,ny,prec,mode,image_ptr)
int nx,ny,prec,mode;
unsigned char *image_ptr;
{
	Cint i, j, pixel, *cellptr;
	extern unsigned char *cla_m_row();

	if (debugmode) 
		fprintf(stderr,
			" scgi_cmca: nx= %d, ny= %d, prec= %d, mode= %d\n",
			nx,ny,prec,mode);

	cellptr = ibuf;
	for (j=0; j<ny; j++)
		{ 
		image_ptr= cla_m_row(image_ptr,nx,rowbuf,prec,mode);
		for (i=0; i<nx; i++) 
			{
			if (rowbuf[i] == 0)
				*cellptr++ = 1;
			else
				*cellptr++ = 0;
			}
		};

	return(1);
}


/*                                           
	This routine implements the copying of an indexed color cell
	array into the (previously allocated) image memory buffer.
	Return 1 if successful.  The acronym is 'scgi_copy_indexed_
	cell_array'.
*/

scgi_cica(nx,ny,prec,mode,image_ptr)
int nx,ny,prec,mode;
unsigned char *image_ptr;
{
	Cint i,j,*cellptr;
	extern unsigned char *cla_i_row();

	if (debugmode) 
		fprintf(stderr,
			" scgi_cica: nx= %d, ny= %d, prec= %d, mode= %d\n",
			nx,ny,prec,mode);

	cellptr = ibuf;
	for (j=0; j<ny; j++)
		{ 
  		image_ptr= cla_i_row(image_ptr, nx, rowbuf, prec, mode);
		for (i=0; i<nx; i++) 
			{
			*cellptr= (rowbuf[i]>=0 && rowbuf[i]<clr_t_sz ) ? 
					rowbuf[i] : 0;
			cellptr++;
			};
		};

	return(1);
}

/*                
	This routine implements the copying of a direct color cell
	array into the (previously allocated) image memory buffer.
	Return 1 if successful.  The acronym is 'scgi_copy_direct_
	cell_array'.
*/

scgi_cdca(nx,ny,prec,mode,image_ptr)
int nx,ny,prec,mode;
unsigned char *image_ptr;
{
	Cint i, j, pixel, *cellptr;
	extern unsigned char *cla_dc_row();

	if (debugmode) 
		fprintf(stderr,
			" scgi_cdca: nx= %d, ny= %d, prec= %d, mode= %d\n",
			nx,ny,prec,mode);

	cellptr = ibuf;
	for (j=0; j<ny; j++)
		{ 
		image_ptr= cla_dc_row(image_ptr,nx,redbuf,greenbuf,
			bluebuf,prec,mode);
		for (i=0; i<nx; i++) 
			{
     			pixel= best_clr[ pack_clr(
				f_to_b(redbuf[i]),
				f_to_b(greenbuf[i]),
				f_to_b(bluebuf[i])) ];
			*cellptr= ( pixel>=0 && pixel<clr_t_sz ) ? pixel : 0;
			cellptr= cellptr ++;
			};
		};

	return(1);
}

                     
/* 
	Attempt to change the actual SUN color_table entries beginning at
	beg_index.  This is used for installing GPLOTS's indexed color_table
	or the direct_color simulation table.  Return 1 if successful.
*/
scgi_ctab(beg_index, no_entries, pctab)
int beg_index; 	/* beginning index */ 
int no_entries; 	/* number of entries to add starting at beg_index */
float *pctab;	/* direct colour array, *(pctab + i*3) is the red */
		/*   entry for the i'th index, followed by g and b */
{
       	Cint index, rc=1, entry_max, max_index;
	unsigned char *cptrR, *cptrG, *cptrB;
	float *ctptr;
	Ccentry clist, top_clist;
	char *malloc();

	if (debugmode)
		fprintf(stderr,
		   " scgi_ctab: %d entries with first index= %d\n",
			no_entries,beg_index);

	if (monochrome)
		{
		rc = scgi_m_ctab(beg_index, no_entries, pctab);
		return(rc);
		}

	if (beg_index > clr_t_sz)
		{
		fprintf(stderr, 
		   "Beginning index greater than color table size.  Call ignored");
		return(2);
		}

	/* LOCK off top color entry index */
	max_index = clr_t_sz - 1;
	entry_max = no_entries + beg_index - 1;
	if ( entry_max > max_index ) 
		{
		no_entries = max_index - beg_index+1;
		fprintf(stderr,
		   "Too many colors for table size, only first %d colors added \n",
		   no_entries );
 		rc = 2;
		}


/*    		allocate memory for colorlist structure		*/

	if ( 
	    ((cptrR = (unsigned char *)
		malloc((no_entries+1)*sizeof(unsigned char) )) == 0 )
	     ||
	    ((cptrG = (unsigned char *)
		malloc((no_entries+1)*sizeof(unsigned char) )) == 0 )
	     ||
	    ((cptrB = (unsigned char *)
		malloc((no_entries+1)*sizeof(unsigned char) )) == 0 )
	   )
		{ 
		fprintf(stderr,
			" Unable to allocate color table memory.\n");
		return(2);
		};

	clist.ra = cptrR;
	clist.ga = cptrG;
	clist.ba = cptrB;

	/* ctptr is trashable,pctab start address is preserved */
	ctptr = pctab + (3 * beg_index); 

	for (index = 0; index < no_entries; index++)
		{
		clist.ra[index] = f_to_b( *ctptr );
		clist.ga[index] = f_to_b( *(ctptr+1) );
		clist.ba[index] = f_to_b( *(ctptr+2) );
		ctptr+=3;
		};
	clist.n = no_entries;
	/*
	   check that top and bottom colors of table are not the same, as
	   this will cause Suncgi to substitute black for the top and white
	   for the bottom
	*/
	if ( beg_index == 0)
		if ( 
		     (clist.ra[0] == topRed) &&
		     (clist.ga[0] == topGreen) &&
		     (clist.ba[0] == topBlue)
		   )
			{
			if (debugmode)
			   fprintf(stderr, "Changing buffer color \n");
			topRed = topRed + 1 % 256;
			top_clist.ra = &topRed;
			top_clist.ga = &topGreen;
			top_clist.ba = &topBlue;
			top_clist.n = 1;
			/* write new buffer color at top index of c-table */
			color_table(max_index+1,&top_clist);
			}


	color_table(beg_index,&clist);

	free(cptrR);		/* free color table memory */
	free(cptrG);
	free(cptrB);

	/* wiped out previous color table with and indexed one, reset flag */
	dc_ready_flag = 0;	

	return(rc);
}

/*
     The color table simulation for monochrome monitors.  Uses the pattern
     table instead
*/
scgi_m_ctab(beg_index, no_entries, pctab)
int beg_index; 	/* beginning index */ 
int no_entries; 	/* number of entries to add starting at beg_index */
float *pctab;	/* direct colour array, *(pctab + i*3) is the red */
		/*   entry for the i'th index, followed by g and b */
{
       	Cint index,patt_index, rc=1, entry_max, max_index;
	unsigned char red, green, blue;
	float *ctptr;

	if (debugmode)
		fprintf(stderr,
		   " scgi_m_ctab: %d entries with first index= %d\n",
			no_entries,beg_index);

	if (beg_index > PATTERN_TBL_SZ-1)
		{
		fprintf(stderr, 
		   "Beginning index greater than color table size.  Call ignored");
		return(2);
		}

	max_index = PATTERN_TBL_SZ;
	entry_max = no_entries + beg_index;
	if ( entry_max > max_index ) 
		{
		entry_max = PATTERN_TBL_SZ;
		no_entries = max_index - beg_index;
		fprintf(stderr, 
		   "Too many colors for table size, only first %d colors added\n", no_entries );
 		rc = 2;
		}


	/* ctptr is trashable,pctab start address is preserved */
	ctptr = pctab + (3 * beg_index); 

	for (index = beg_index; index < entry_max; index++)
		{
		red   = f_to_b( *ctptr );
		green = f_to_b( *(ctptr+1) );
		blue  = f_to_b( *(ctptr+2) );
		patt_index = pattern_clr(red, green, blue);
		pattern_table(index,PATT_WIDTH,PATT_WIDTH,
		   clr_pattern[patt_index]);
		ctptr+=3;
		};

	/* wiped out previous color table with and indexed one, reset flag */
	dc_ready_flag = 0;	

	return(rc);
}

/*
	This function converts GPLOTS's x and y buffers into a coordinate
	list that SCGI can use.
*/
make_coorlist(xlist,ylist,polycoors,numcoors)
Cint *xlist,*ylist,numcoors;	/* GPLOT's coordinate representation */
Ccoorlist *polycoors;		/* initially null, returned as a list of filled
				   Ccoor structures */
{
	Cint coornum, *xptr, *yptr;


	if (debugmode) fprintf(stderr,
	   " make_coorlist: a coorlist %d long.\n",numcoors);

	xptr = xlist; yptr = ylist;
	if ( scgi_getcmem(numcoors) != 1 )
		{ 
		fprintf(stderr," Error allocating memory for coord buffer.");
		return(2);
		};
	(*polycoors).ptlist = coorbuf;

	for (coornum = 0; coornum < numcoors; coornum++)
		{
		(*polycoors).ptlist[coornum].x = *xptr++;
		(*polycoors).ptlist[coornum].y = *yptr++;
		}
        (*polycoors).n = numcoors;
	return(1);
}
/*
	This function checks that sufficient memory is available for
	coordinate storage, allocating more as needed.  Returns 1 if
	successful.
*/	
scgi_getcmem(new_buf_sz)
int new_buf_sz;
{
	char *malloc();

	if (new_buf_sz > coor_buf_sz)
		{
		if (debugmode) fprintf(stderr,
		   " scgi_getcmem: increasing coordlist mem to %d.\n",
		   new_buf_sz);
		if (coor_buf_sz>0) { free(coorbuf); coor_buf_sz= 0; };
		if ( (coorbuf= (Ccoor *)malloc(new_buf_sz*sizeof(Ccoor))) == 0 )
			{ 
			fprintf(stderr,"Unable to allocate memory.\n");
		 	return(2);
			};
		coor_buf_sz= new_buf_sz;
		};
	return(1);
}

/*
	This function checks that sufficient memory is available for
	image storage, allocating more as needed.  Returns 1 if
	successful.
*/	
scgi_getimem(isize)
int isize;
{
	char *malloc();

	if (isize>ibuf_sz)
		{
 		if (debugmode) fprintf(stderr,
		   " scgi_getimem: increasing image mem to %d ints.\n",isize);
 		if (ibuf_sz>0) { free(ibuf); ibuf_sz= 0; };
		if ( (ibuf= (Cint *)malloc(isize*sizeof(int))) == 0 )
			{ 
			fprintf(stderr," Unable to allocate image memory.\n");
		 	return(2);
			};
		ibuf_sz= isize;
		};

	return(1);
}

            
/*                              
	This routine sets up a color table with which to 'fake' direct color
  	on an indexed color device.  Return 1 if setup was successful.
*/
scgi_fake_dc()  
{
	float r,g,b,rtbl,gtbl,btbl,rnxt,gnxt,bnxt;
	Cint ir,ig,ib,nr,ng,nb,index,itbl,irtbl,igtbl,ibtbl,
		roff,goff,ctoffset,egg;
	char *malloc();

	if (debugmode)
		{
		fprintf(stderr,
		" scgi_fake_dc: setting up direct color simulation.\n");
		if (monochrome)
			{ 
			fprintf( stderr, "Monochrome device, exiting. \n");
			dc_init_flag=1;
			return(2);
			}
		}
 
	if (dc_init_flag) return(1);

	if (clr_t_sz==0)
		{
		fprintf(stderr," SCGI driver not initialized.\n");
		return(2);
		};
	if (clr_t_sz<8)
		{
		fprintf(stderr,
			" Color table too small to fake direct color\n");
		return(2);
		};

	nb= scgi_cube_rt(clr_t_sz);
	ng= scgi_sqr_rt(clr_t_sz/nb);
	nr= clr_t_sz/(nb*ng);
	dc_totclrs= nr*ng*nb;
	ctoffset= clr_t_sz - dc_totclrs;

/* Allocate memory for color table and best-color correspondence table */ 

	if ( (clr_tbl= (float *)malloc(3*dc_totclrs*sizeof(float) )) 
		== 0 )
		{ 
		fprintf(stderr,
			" Unable to allocate color table memory.\n");
		return(2);
		};
	if ( (best_clr= (Cint *)malloc(BEST_C_SZ*sizeof(int))) == 0 )
		{ 
		fprintf(stderr,
			" Unable to allocate best-color list memory.\n");
		return(2);
		};

	/* Build the color table */
	itbl= 0;
	for (ir=0; ir<nr; ir++)
		for (ig=0; ig<ng; ig++)
			for (ib=0; ib<nb; ib++)
			   {
		 	   clr_tbl[3*itbl]= ((float) ir)/((float)(nr-1));
		 	   clr_tbl[3*itbl+1]= ((float) ig)/((float)(ng-1));
		 	   clr_tbl[3*itbl+2]= ((float) ib)/((float)(nb-1));
			   itbl++; };
	/* Build the nearest-color-index table */
	irtbl= 0;  roff= 0;
	rtbl= clr_tbl[0];
	rnxt= clr_tbl[3*ng*nb];
	for ( ir=7; ir<256; ir+=8 )
		{
		r= b_to_f(ir);
		if ( (r-rtbl>rnxt-r) && ( irtbl < nr-1 ) )
		      	{
			irtbl++;  roff += ng*nb;
			rtbl= rnxt;
			rnxt= clr_tbl[ 3*(irtbl+1)*ng*nb ];
			};
		igtbl= 0;  goff= 0;
		gtbl= clr_tbl[3*roff+1];
		gnxt= clr_tbl[3*(roff+nb)+1];
		for ( ig=7; ig<256; ig+=8 )
			{
			g= b_to_f(ig);
			if ( (g-gtbl>gnxt-g) && (igtbl < ng-1 ) )
				{
				igtbl++;  goff += nb;
				gtbl= gnxt;
				gnxt= clr_tbl[ 3*(roff+(igtbl+1)*nb) +1];
				};
			ibtbl= 0;
			btbl= clr_tbl[3*(roff+goff)+2];
			bnxt= clr_tbl[3*(roff+goff+1)+2];
			for ( ib=7; ib<256; ib+=8 )
				{                      
				b= b_to_f(ib);
				if ( (b-btbl>bnxt-b) && ( ibtbl < nb-1 ) )
					{
				     	ibtbl++;
					btbl= bnxt;
					bnxt= clr_tbl[3*(roff+goff+ibtbl+1)+2];
					};
				index= pack_clr( (unsigned char) ir,
				   (unsigned char) ig,(unsigned char) ib);
				best_clr[index]= ctoffset+roff+goff+ibtbl;
				}; 
			};
		};
	 dc_init_flag= 1;
	 return(1);
 }

/*                        
	This routine swaps in the direct color simulation color map
	if it is not currently in effect.  If necessary, the map and
	related data structures are generated by calling scgi_fake_dc.
	Returns 1 if successful.
*/
scgi_dc_colors()
{
	Cint rc=1;
	Cint beg_index, index;
	unsigned char *cptrR, *cptrG, *cptrB;
	float *ctptr;
	Ccentry clist;

	if (debugmode) fprintf(stderr,
		" scgi_dc_colors: swapping in direct color sim table.\n");
 
	if (monochrome)
		{ 
		rc = scgi_mdc_colors();
		return(rc);
		}
 
	if (dc_ready_flag) return(1);

	if ( 
	    ((cptrR = (unsigned char *)
		malloc(clr_t_sz*sizeof(unsigned char) )) == 0 )
	     ||
	    ((cptrG = (unsigned char *)
		malloc(clr_t_sz*sizeof(unsigned char) )) == 0 )
	     ||
	    ((cptrB = (unsigned char *)
		malloc(clr_t_sz*sizeof(unsigned char) )) == 0 )
	   )
		{ 
		fprintf(stderr,
			" Unable to allocate color table memory.\n");
		return(2);
		};

	clist.ra = cptrR;
	clist.ga = cptrG;
	clist.ba = cptrB;


	if ( (dc_init_flag) || ( (rc=scgi_fake_dc()) ==1 ) )
		{
	        /* ctptr is trashable,clr_tbl start address is preserved */
 		ctptr= clr_tbl;
		beg_index = clr_t_sz - dc_totclrs;
		for (index=0; index<dc_totclrs; index++)
			{
			clist.ra[index] = f_to_b(ctptr[0]);
			clist.ga[index] = f_to_b(ctptr[1]);
			clist.ba[index] = f_to_b(ctptr[2]);
			ctptr +=3;
			};
		clist.n =  dc_totclrs;
		color_table(beg_index,&clist);
		dc_ready_flag= 1;
		};

	return(rc);
}

scgi_mdc_colors()
{
       	Cint index,patt_index, rc=1, entry_max, max_index;
	unsigned char red, green, blue;
	float *ctptr;

	if (debugmode)
		fprintf(stderr,
		   " scgi_mdc_colors \n");


	max_index = PATTERN_TBL_SZ; 

	for (index = 0; index < max_index; index++)
		{
		pattern_table(index,PATT_WIDTH,PATT_WIDTH,clr_pattern[index]);
		};

	/* signal that we are ready to simulate direct color */
	dc_ready_flag = 1;	

	return(rc);
}


/* 
	This routine returns the greatest integer less than or equal to
	the cube root of the small integer i.
*/
scgi_cube_rt(i)
int i;
{
	Cint j;
	for (j=0; j<i; j++) if (j*j*j>i) return(j-1);
	return(i);
}

/* 
	This routine returns the greatest integer less than or equal to 
	the square root of the small integer i.
*/
scgi_sqr_rt(i)
int i;
{
	Cint j;
	for (j=0; j<i; j++) if (j*j>i) return(j-1);
	return(i);
}

/*
   This routine holds the display for timeout, or the first mouse input,
   then continues.  It is used to hold an image on the screen
*/

scgi_ready_prompt()
{
    Cawresult stat;
    Cflag defflag = OFF;
    Cint index;
    FILE *fp;


#if 1
    Cawresult valid;
    Ccoor point;
    Cdevoff devclass = IC_LOCATOR;
    Cint devnum = 2;		/* mouse */
    Ceqflow overflow;
    Cinrep ivalue;
    Cint replost;
    Cint time_stamp;
    Cint timeout = -1;		/* wait for input forever pg 91 */
    Cint tracktype = 1;		/* track type pg 89 */
    Cint trigger = 2;		/* trigger number; 2 is left mouse button pg
				 * 86 */
    Cmesstype message_link;
    Cqtype qstat;

    if (debugmode)
	fprintf(stderr, "scgi_ready_prompt \n");

    point.x = 1;
    point.y = 1;
    ivalue.xypt = &point;
    ivalue.string = "READY";

    initialize_lid(devclass, devnum, &ivalue);
    associate(trigger, devclass, devnum);
    track_on(devclass, devnum, tracktype,
	     (Ccoorpair *) 0, &ivalue);
    enable_events(devclass, devnum);
    await_event(timeout, &valid, &devclass, &devnum, &ivalue,
		&message_link, &replost, &time_stamp, &qstat, &overflow);
    disable_events(devclass, devnum);
    dissociate(trigger, devclass, devnum);
    release_input_device(devclass, devnum);
#endif
#if 0

    if ((fp = fopen("/dev/tty", "r")) == NULL)
	{
	fprintf(stderr, " Error opening device input \n");
	exit(8);

    if (fgetc(fp) == EOF)
	exit(1);

    fclose(fp);
#endif

}


/*
 	This routine initializes the SCGI system and this driver with GPLOT's
	relevant information
*/
void scgi_device_setup(dev_type)
char *dev_type;
{
	Cacttype soft = CLEAR, hard = CLEAR; /* hard not usedin SunCGI */
	Cacttype intern = CLEAR;	/* intern not used in SunCGI */
	Cexttype extent = VIEWSURFACE;
	Cint output, input;


	if (debugmode) fprintf(stderr," scgi_device_setup \n");

	/* Check device type (color or not) and set global flags */
	if (0 == strncmp(dev_type, "sunbw", 5))
		{
		monochrome=on;
		NORMAL_VWSURF(scgi_device, PIXWINDD);
		}
	else
		{
		monochrome=off;
	        NORMAL_VWSURF(scgi_device, CGPIXWINDD);
		/* allow more than eight colors */ 
		scgi_device.cmapsize = 256;
		strcpy(scgi_device.cmapname, "gplotmp"); 
		};


	/* inform scgi about the device */
	open_cgi();
	clear_control(soft, hard, intern, extent);
	open_vws(&scgi_window, &scgi_device);
}
/*                 
	This routine sets up the interface between the SCGI driver
	and GPLOT.  It appears at the end of the module so that it
	will have access to the symbolic names of functions defined
	so far.
*/                                   
void scgi_setup(pOp, pDev_info, pc1, pc2, pc3, pc5, pdelim, pmfdesc, ppdesc,
	 	pmfctrl, pgprim, pattr, pescfun, pextfun, pctrl,pargc, argv)
struct one_opt 		*pOp;	/* the command line options, in only */
struct info_struct *pDev_info;	/* device info to fill out, out only */
struct mf_d_struct 	*pc1;	/* the class 1 elements, in only */
struct pic_d_struct 	*pc2;	/* the class 2 elements, in only */
struct control_struct	*pc3;	/* the class 3 elements, in only */
struct attrib_struct	*pc5;	/* the class 5 elements, in only */
int (*pdelim[])(); 		/* delimiter functions, out only */
int (*pmfdesc[])();    		/* garbage, for now */
int (*ppdesc[])();     		/* garbage, for now */
int (*pmfctrl[])();    		/* metafile control functions */
int (*pgprim[])();     		/* graphical primitives, out only */
int (*pattr[])();      		/* the attribute functions, out only */
int (*pescfun[])();    		/* the escape functions, out only   */
int (*pextfun[])();    		/* the external functions, out only */
int (*pctrl[])();      		/* garbage, for now */
 
int *pargc;			/* count of command line args, */
char *argv[];			/* vector of command line args, */
				/* both pargc and argv could be altered here */
{
	Cint xbase, ybase;
	extern int cla_init();


	Ccoor top, bot;


	/* store the command line argument pointer */
	Op = pOp;

	if (debugmode) fprintf(stderr," scgi_setup \n");
                                                  
	/* store the CGM data structure and device info pointers*/
	State_c1 = pc1;
	State_c2 = pc2;
	State_c3 = pc3;
	State_c5 = pc5;
	Ctrl_funcs = pctrl;
	Dev_info = pDev_info;

	/* Fill out the function pointer arrays for GPLOT */

	Ctrl_funcs[(int) E_Pic] = 0;	/* disable the wait-between-pictures */
					/* control function, this driver has */
					/* its own */
	/* the delimiter functions */
	pdelim[(int) B_Mf] 	= scgi_begin;
	pdelim[(int) E_Mf]	= scgi_end;
	pdelim[(int) B_Pic_Body]= scgi_bpage;
	pdelim[(int) E_Pic]	= scgi_epage;

	/* the graphical primitives */
	pgprim[(int) PolyLine]	= scgi_pline;
	pgprim[(int) Dis_Poly]	= scgi_dpline;
	pgprim[(int) PolyMarker]= scgi_pmarker;
	pgprim[(int) Text]	= scgi_text;
	pgprim[(int) Polygon]	= scgi_pgon;
	pgprim[(int) Cell_Array]	= scgi_carray;

	/* the attributes */
	pattr[(int) LType]	= scgi_l_type;
	pattr[(int) LWidth]	= scgi_l_width;
	pattr[(int) LColour]	= scgi_l_colour;
	pattr[(int) EType]	= scgi_e_type;
	pattr[(int) EdWidth]	= scgi_e_width;
	pattr[(int) EdColour]	= scgi_e_colour;
	pattr[(int) EdVis]	= scgi_e_vis;
	pattr[(int) MColour]	= scgi_mk_colour;
	pattr[(int) MType]	= scgi_mk_type; 
	pattr[(int) TColour]	= scgi_t_colour;
	pattr[(int) MSize]	= scgi_mk_size;
	pattr[(int) CHeight]	= scgi_c_height;
	pattr[(int) TAlign]	= scgi_t_align;
	pattr[(int) FillColour]	= scgi_fl_colour;
	pattr[(int) IntStyle]	= scgi_fl_style;
	pattr[(int) ColTab]	= scgi_ctab;


	/* initialize the cell_array utility software */
	cla_init(pc1,pc2,pc5,pgprim,pattr); 


	/* 
	The following code initializes the SCGI system and informs GPLOT
	of SCGI's and the output device's capabilities.  Note that all
	coordinate values in SCGI refer to the VDC (virtual device coordinates).
	A one to one mapping between the VDC and the physical screen size is
	used so that all coordinates effectively refer to the screen.  SCGI
	requires all coordinates to be in integers, which is what GPLOT gives
	it, so no coordinate scaling is necessary.  Also, all SCGI primitives
	take scaled (num. pixels) or percent of VDC values for sizes.  We will
	only use scaled (pixel) values.
	*/

	/*
	   Initialize SCGI for this device and set up the color table size on
 	   this device to 256.  The first elment of argv is a flag telling 
	   whether this is a monochrome device or not.  scgi_device_setup
	   will set the global monochrome flag appropriatly.  The first element
	   of argv must be of type enum boolean.
	*/
	scgi_device_setup(pOp[(int) device].val.str);

	/* 
	Inquire about display size and pixel counts, and use them
	to set device size and resolution.  Note we assume the
	window will be square and that the distances in GPLOT is in
	inches ( = meters * 3.29*12 inches/meter).
	xext, yext are pixels lengths of screen.  xunits, yunits are pixels/mm.
	*/
	xbase = ybase = 0;
	/* Note, xunits and yunits return 0 by default, don't use them */
	inquire_physical_coordinate_system(scgi_window,&xbase,&ybase,
		&xext,&yext,&xunits, &yunits);
	Dev_info->pxl_in 	= yext/10;
	Dev_info->ypxl_in 	= yext/10;
	Dev_info->y_size 	= 10;
	Dev_info->x_size 	= Dev_info->y_size;

	/* 
	Fill out all the simple parts of the device info structure.
	Some data will be set later in the routine.
	*/
	Dev_info->x_offset	= 0.0;
	Dev_info->y_offset	= 0.0;
	Dev_info->capability	= h_center + v_center + string_text; 
	if (monochrome)
		Dev_info->capability += no_colour;
	strcpy(Dev_info->out_name, ".SCGI"); /* This won't be used */
	Dev_info->rec_size	= 80;

	/*
	   Set up SCGI's VDC (Virtual Device Coordinates ) to be the same as
	   the screen attirbutes.  This overrides the VDC facilitly,
	   all SCGI functions effectively refer to the physical screen
	*/
	top.x = xext;
	top.y = yext;
	bot.x = 0;
	bot.y = 0;
	vdc_extent(&bot, &top);

	/* Set up default line width, and edge width (assumed same) */
	Dev_info->d_l_width	= 0.0; 		
	Dev_info->d_e_width	= Dev_info->d_l_width;

	/* 
	   Set up the default marker size, which is 4 % of the screen height 
	   in pixels in SCGI's default setting 
	*/
	Dev_info->d_m_size	= (int) (.04*yext);  

	/* 
	   Set up global Character information.  We leave the _text flags in
	   GPLOT alone and have GPLOT draw all characters from its internal
	   font set.  String text can not be implemented at the driver level
	   because SCGI's text attribute setting functions do not apply to 
	   string quality text (p68 SCGI ref. manual).
	*/
	text_precision( CHARACTER );	/* make all 6 fonts usable */
	/* Character heights minimum and maximums in pixels */
	c_h_min= 1;
	c_h_max= yext;		/* length of screen width */
	/* Character height default is 1% of the default normal window */
	Dev_info->c_height= (int)(0.01 * Dev_info->y_size);
	d_c_height= Dev_info->c_height;		/* Save a local copy of this */
	/* Assume square characters to get character width */
	Dev_info->c_width	= Dev_info->c_height;
	/* Set up global color table information */
	if (!monochrome) 
		clr_t_sz = scgi_device.cmapsize - 1;  
		/*  Make space for the top buffer color  */
	line_width_specification_mode( ABSOLUTE );
	perimeter_width_specification_mode( ABSOLUTE );
	marker_size_specification_mode( ABSOLUTE );

	state_level= -1;  /* Just starting driver */
}

