/*** FILE "mapgif.cc" ********************************************** 130793 ***/

#include "add/mapgif.h"
#include "include/rgb.h"
#include "private/fatal.h"

/*** GIF Grammar **************************************************************/
/*
                  <>    grammar word
                  ::=   defines symbol
                  *     zero or more occurrences
                  +     one or more occurrences
                  |     alternate element
                  []    optional element

<GIF Data Stream> ::=     Header <Logical Screen> <Data>* Trailer

<Logical Screen> ::=      Logical Screen Descriptor [Global Color Table]

<Data> ::=                <Graphic Block>  |
                          <Special-Purpose Block>

<Graphic Block> ::=       [Graphic Control Extension] <Graphic-Rendering Block>

<Graphic-Rendering Block> ::=  <Table-Based Image>  |
                               Plain Text Extension

<Table-Based Image> ::=   Image Descriptor [Local Color Table] Image Data

<Special-Purpose Block> ::=    Application Extension  |
                               Comment Extension
*/
/*** MapGIFBase ***************************************************************/

MapGIFBase::MapGIFBase(const char* const filename):
  ImageFile(filename,       //open image-file...
            ios::in|        //...for reading...
            ios::binary|    //...a binary file, that is...
            ios::nocreate)  //...and if it doesn't exist, it shouldn't be
                            //created
{ if (!ImageFile)                          //if the file couldn't be opened...
    FATAL("Can't open image-file");        //...report it

  Comments.Insert('\0');                   //make sure the comments-string ends
                                           //with a '\0'
}

void MapGIFBase::GetImage(n1 infoOnly,
                          pix x1, pix y1, pix x2, pix y2,
                          MapInBase::Gravity g,
                          spix CustomX, spix CustomY)
{ _infoOnly=infoOnly;  //replicate data, -necessary because nested functions
  _x1=x1;              //ain't allowed
  _y1=y1;
  _x2=x2;
  _y2=y2;
  _g=g;
  _CustomX=CustomX;
  _CustomY=CustomY;

  colors=0;  //the number of colors found is zero, so far

  GIF_Data_Stream();  //since the GIF-grammar starts with the symbol
                      //'<GIF Data Stream>', the associated function is called
                      //to read the image

  ImageFile.close();  //make sure the file gets closed NOW; it's not a good idea
                      //to wait until destruction, where it otherwise would
                      //happen automatically, since the user is in control of,
                      //when this happens!
}

pix MapGIFBase::SizeX() const
{ return sizeX;  //return the x-dimension of the active area
}

pix MapGIFBase::SizeY() const
{ return sizeY;  //return the y-dimension of the active area
}

double MapGIFBase::GetAspectRatio() const
{ return aspectRatio;  //return the ratio associated with the last image read
}

n16 MapGIFBase::Colors() const
{ return colors;  //return the number of colors found in the image
}

const char* MapGIFBase::GetComments() const
{ return Comments.GetAll();  //return the comments-string
}

const char* MapGIFBase::Version() const
{ return version;  //return the version-number-string
}

/* Check, that the 'n8' type doesn't contain more than 8 bits; if it does, an
 * image-file can't be read correctly, and the file shouldn't be compileable:
 */
#if N8_WIDTH!=8
#error "mapgif.cc: Unable to process GIF files"
#endif

inline n8 MapGIFBase::Get()
{ n8 byte;
  if (!ImageFile.get(byte))  //read the next byte, if it doesn't exists, report
                             //it...
    FATAL("Unexpected end of file stream found - GIF image corrupted!");

  return byte;  //return the byte just read
}

inline n8 MapGIFBase::Peak()
{ n8 byte=Get();            //read the next byte...
  ImageFile.putback(byte);  //...and put it back on top of the file stream

  return byte;              //return the byte just read
}

void MapGIFBase::Skip(n16 bytes)
{ n16 i=0;        //counter
  while(i<bytes)  //while Get hasn't been called 'bytes' times...
  { Get();        //...call Get
    i++;
  }
}

void MapGIFBase::SkipSubBlocks()
{ n8 size;
  while((size=Get())!=0)  //get the next byte, if it's zero, it's a block
                          //terminator, otherwise, it's the size of a block,
                          //that will be skipped...
    Skip(size);           //...by calling Skip
}

void MapGIFBase::GIF_Data_Stream()
{ /* <GIF Data Stream> ::= Header <Logical Screen> <Data>* Trailer
   */

  /* Header - signature: */
  if (Get()!='G' ||  //if the signature isn't "GIF", report it...
      Get()!='I' ||
      Get()!='F')
    FATAL("Can't recognize file stream as a GIF data stream - bad signature!");

  /* Header - version number: */
  version[0]=Get();  //read the version number
  version[1]=Get();
  version[2]=Get();

  if (version[0]!='8' ||                       //if the version number isn't
      (version[1]!='7' && version[1]!='9') &&  //"87a" or "89a", report it...
      version[2]!='a')
    FATAL("Can't recognize version number of GIF data stream - bad version!");
  version[3]='\0';  //let the version number end with a '\0'

  /* <Logical Screen>: */
  Logical_Screen();  //let someone else handle this part

  /* <Data>*: */
  while(Peak()!=0x3b)  //if the Trailer isn't next...
    Data();            //...the next bytes are of the "Data" kind

  /* Trailer: */
  if (Get()!=0x3b)  //if the Trailer isn't found (it most certainly should be;
                    //the condition in the loop above has just failed!)...
    FATAL("Can't recognize end of GIF data stream - bad trailer!");
}

void MapGIFBase::Logical_Screen()
{ /* <Logical Screen> ::= Logical Screen Descriptor [Global Color Table]
   */

  /* Logical Screen Descriptor - Logical Screen Width: */
  sizeX=Get()|(Get()<<8);  //get width of the active area, consists of two
                           //bytes, the least significant byte comes first

  /* Logical Screen Descriptor - Logical Screen Height: */
  sizeY=Get()|(Get()<<8);  //get height of the active area, consists of two
                           //bytes, the least significant byte comes first

/* Note: The associativity of the operator '|' is left-to-right, so I'm always
 *       sure, which call to Get, that goes first; there are no ambiguities
 *       here!
 */

  /* Determine x-gravity: */
  switch(_g & (1+2))       //look at the 1th and the 2nd bit of the gravity
  { case MapInBase::LowX:
      x0_activeArea=_x1;  //the active area begins, when the window begins
      break;

    case MapInBase::CenterX:
      x0_activeArea=_x1+((_x2-_x1+1)-sizeX)/2;  //the active area is centered
                                                //in relation to the window
                                                //(it doesn't matter, which one
                                                //of the active area and the
                                                //window is biggest!)
      break;

    case MapInBase::HighX:
      x0_activeArea=_x2-(sizeX-1);  //the active ends, when the window ends
      break;

    case MapInBase::CustomX:
      x0_activeArea=_x1+_CustomX;  //_CustomX determines, where the active area
                                   //begins, relative to where the window begins
      break;
  }

  /* Determine y-gravity: */
  switch(_g & (4+8))       //look at the 3rd and the 4th bit of the gravity
  { case MapInBase::LowY:
      y0_activeArea=_y1;  //the active area begins, when the window begins
      break;

    case MapInBase::CenterY:
      y0_activeArea=_y1+((_y2-_y1+1)-sizeY)/2;  //the active area is centered
                                                //in relation to the window
                                                //(it doesn't matter, which one
                                                //of the active area and the
                                                //window is biggest!)
      break;

    case MapInBase::HighY:
      y0_activeArea=_y2-(sizeY-1);  //the active ends, when the window ends
      break;

    case MapInBase::CustomY:
      y0_activeArea=_y1+_CustomY;  //_CustomY determines, where the active area
                                   //begins, relative to where the window begins
      break;
  }

  /* Logical Screen Descriptor - <Packed Fields>: */
  n8 packedFields=Get();  //get the packed fields, one byte only

  /* Logical Screen Descriptor - Global Color Table Flag: */
  n1 existColorTable=packedFields & 128;  //set the flag indicating, if a global
                                          //color table exists

  /* Logical Screen Descriptor - Color Resolution: */
  //"This value should be set to indicate the richness of the original palette,
  // even if not every color from the whole palette is available on the source
  // machine", states the document for version 89a.
  //They wanna store data about the capabilities of the machine, that saved the
  //original image? And they say "...a protocol intended for the on-line
  //transmission and interchange of raster graphic data in a way that is
  //independent of the hardware used in their creation or display."
  //-so this Color Resolution is INDEPENDENT OF THE HARDWARE and ain't set in
  //the CREATION of the image, eh?
  //In no way I'll do anything but ignore their sick, self-contradictory CRAP!

  /* Logical Screen Descriptor - Sort Flag: */
  //I've chosen to ignore this; I think it's a CRIPPLED construction, which I
  //don't at all find usefull!

  /* Logical Screen Descriptor - Size of Global Color Table: */
  n16 Colors=2<<(packedFields & 7);  //calculate the number of colors

  /* Logical Screen Descriptor - Background Color Index: */
  //I've chosen to ignore this; I think it's a CRIPPLED construction, which I
  //don't at all find usefull!
  Get();  //skip Background Color Index

  /* Logical Screen Descriptor - Pixel Aspect Ratio: */
  { n8 ar=Get();                          //read the aspect ratio...
    if (ar==0)                            //...if it doesn't have a meaningfull
                                          //value...
      aspectRatio=1;                      //...the aspect ratio of this class is
                                          //set to 1
    else                                  //...otherwise...
      aspectRatio=(double(Get())+15)/64;  //...it's calculated using the formula
                                          //specified in the CompuServe
                                          //documents
  }

  /* [Global Color Table]: */
  if (existColorTable)       //if a global color table exists...
    ReadColorTable(Colors);  //...let someone have a close look at it
}

void MapGIFBase::Data()
{ /* <Data> ::= <Graphic Block>  |
   *            <Special-Purpose Block>
   */

  /* The following checking, that has to be done to know, what should be done
   * next, is too hard without some kind of drawing, showing what values the
   * next bytes may have:
   *
   * <Data> -*-> <Graphic Block> -*-> [Graphic Control Extension] (0x21,0xF9)
   *         |                    |
   *         |          <Graphic-Rendering Block> -> <Table-Based Image>
   *         |                    |                           |
   *         |                    |                   Image Descriptor (0x2C)
   *         |                    |
   *         |                    *-> Plain Text Extension (0x21,0x01)
   *         |
   *         *-> <Special-Purpose Block> -*-> Application Extension (0x21,0xFF)
   *                                      |
   *                                      *-> Comment Extension (0x21,0xFE)
   */

  switch(Get())            //get Extension Introducer for the next block...
  { case 0x21:             //...if it's an extension block, additional
                           //checking has to be performed...
     switch(Peak())        //...peak at the Label; can't decide without it...
     { case 0x01:          //...if the Label is a Plain Text Label...
       case 0xf9:          //...or a Graphic Control Label...
         Graphic_Block();  //...the right fan-out is known
         break;

       case 0xfe:                  //...if the Label is a Comment Label...
       case 0xff:                  //...or a Application Label...
         Special_Purpose_Block();  //...the right fan-out is known
         break;

       default:              //...can't recognize Label...
         FATAL("Unrecognized Label - GIF image corrupted!");
     }
     break;

    case 0x2c:          //...the next block is an Image Descriptor...
      Graphic_Block();  //...so the right fan-out is called
      break;

    default:            //...can't recognize Extension Introducer...
      FATAL("Unrecognized Extension Introducer - GIF image corrupted!");
  }
}

void MapGIFBase::Graphic_Block()
{ /* <Graphic Block> ::= [Graphic Control Extension] <Graphic-Rendering Block>
   */

  if (Peak()==0xf9)   //if the Label is a Graphic Control Label...
  { /* [Graphic Control Extension]: */
    Get();            //skip Label
    SkipSubBlocks();  //can't cope with all this interactive disposal shit,
                      //just scan across it
  }
  else              //...otherwise it can only be a Graphic-Rendering Label
                    //(other Labels can't show up here, not even if the image
                    //is corrupted; this is guaranteed by the checks performed
                    //in _Data_!)...
  { /* <Graphic-Rendering Block>: */
    Graphic_Rendering_Block();  //let someone else handle this case
  }
}

void MapGIFBase::Graphic_Rendering_Block()
{ /* <Graphic-Rendering Block> ::= <Table-Based Image>  |
   *                               Plain Text Extension
   */

  if (Peak()!=0x01)  //if the Label isn't a Plain Text Label...
  { /* <Table-Based Image>: */
    Table_Based_Image();  //let someone else handle this case
  }
  else              //...otherwise it can only be a Plain Text Label
                    //(other Labels can't show up here, not even if the image
                    //is corrupted; this is guaranteed by the checks performed
                    //in _Data_!)...
  { /* Plain Text Extension: */
    SkipSubBlocks();  //skip the extension
  }
}

void MapGIFBase::Table_Based_Image()
{ /* <Table-Based Image> ::= Image Descriptor [Local Color Table] Image Data
   */

  /* Image Descriptor - Image Left Position: */
  pix x0=Get()|(Get()<<8);  //get left position of image, consists of two bytes,
                            //the least significant byte comes first

  /* Image Descriptor - Image Top Position: */
  pix y0=Get()|(Get()<<8);  //get top position of image, consists of two bytes,
                            //the least significant byte comes first

  /* Image Descriptor - Image Width: */
  pix sizeX=Get()|(Get()<<8);  //get width of image, consists of two bytes, the
                               //least significant byte comes first

  /* Image Descriptor - Image Height: */
  pix sizeY=Get()|(Get()<<8);  //get height of image, consists of two bytes, the
                               //least significant byte comes first

/* Note: The associativity of the operator '|' is left-to-right, so I'm always
 *       sure, which call to Get, that goes first; there are no ambiguities
 *       here!
 */

  if (x0>SizeX() ||        //if the first corner is outside the active area...
      y0>SizeY() ||
      x0+sizeX>SizeX() ||  //...or the last corner is outside the active area...
      y0+sizeY>SizeY())
                           //...report it
    FATAL("Image doesn't fit into the active area - GIF image corrupted!");

  /* Image Descriptor - <Packed Fields>: */
  n8 packedFields=Get();  //get the packed fields, one byte only

  /* Image Descriptor - Local Color Table Flag: */
  n1 existColorTable=packedFields & 128;  //set the flag indicating, if a local
                                          //color table exists

  /* Image Descriptor - Interlace Flag: */
  //Even though I don't like all the interactive crap, that have been put into
  //the format, I think it's best to be able to decode interlaced images!
  interlaced=packedFields & 64;  //set the flag indicating, if the image is
                                 //interlaced

  /* Image Descriptor - Sort Flag: */
  //I've chosen to ignore this; I think it's a CRIPPLED construction, which I
  //don't at all find usefull!

  /* Image Descriptor - Reserved: */
  //the two reserved bits ain't toutched, of course

  /* Image Descriptor - Size of Local Color Table: */
  n16 Colors=2<<(packedFields & 7);  //calculate the number of colors

  /* [Local Color Table]: */
  if (existColorTable)       //if a local color table exists...
    ReadColorTable(Colors);  //...let someone have a close look at it

  /* Image Data: */
  if (_infoOnly)                      //if the image itself shouldn't be read...
  { Get();                            //...skip the code-size...
    SkipSubBlocks();                  //...and skip all trailing sub-blocks
  }                                   //if the image should be read...
  else
  { bitsLeft=0;                       //no bits are in a waiting position
    numBitsLeft=0;

    rootCodeSize=Get();               //get the (root) code-size

    CC=1<<rootCodeSize;               //determine the clear code
    EOI=CC+1;                         //determine the end-of-information code

    EOImissing=1;                     //the EOI code hasn't been seen yet

    interlacePass=1;                  //if the image should be interlaced, the
                                      //first pass has the number '1'

    Init();                           //initialize string table

    minX=x0_activeArea+x0;            //determine the first corner of the image
    minY=y0_activeArea+y0;
    maxX=x0_activeArea+x0+(sizeX-1);  //determine the last corner of the image
    maxY=y0_activeArea+y0+(sizeY-1);

    x=minX;                           //initialize the pixel-position
    y=minY;


    n8 size;                          //temporary, the size of a block

    while((size=Get())!=0)            //while the block-size isn't a Block
                                      //Terminator...
    { n8 i=0;                         //counter
      do
      { NextByte(Get());              //process the next byte...
        i++;
      }
      while(i<size);                  //...while there are still bytes left in
                                      //the block
    }
  }
}

void MapGIFBase::Special_Purpose_Block()
{ /* <Special-Purpose Block> ::= Application Extension  |
   *                             Comment Extension
   */

  if (Get()==0xfe)  //if the Label is a Comment Label...
  { /* Comment Extension: */
    Comments.ForgetLast();  //remove the '\0', that exists in the end of
                            //Comments

    /* Scan through all sub-blocks, while storing all characters in 'Comments':
     */
    n8 size;                      //temporary, the size of a block
    while((size=Get())!=0)        //while the block-size isn't a Block
                                  //Terminator...
    { n8 i=0;                     //counter
      do
      { Comments.Insert(Get());  //get the next comment-character
        i++;
      }
      while(i<size);             //...while there are still bytes left in the
                                 //block
    }

    Comments.Insert('\0');  //let the comments end with a '\0'
  }
  else              //...otherwise it can only be an Application Label
                    //(other Labels can't show up here, not even if the image
                    //is corrupted; this is guaranteed by the checks performed
                    //in _Data_!)...
  { /* Application Extension: */
   SkipSubBlocks();  //skip all sub-blocks, including the one beginning with
                     //Application Identifier
  }
}

void MapGIFBase::Init()
{ firstCode=1;               //initially, the first real code after
                             //initialization hasn't been seen

  codeSize=rootCodeSize+1;   //determine the number of bits in codes
  codeMask=(1<<codeSize)-1;  //set the code-mask to match the code-size

  nextFree=CC+2;             //new codes begin with the code CC+2

  n16 i=0;                   //counter
  while(i<CC)                //while not all root-codes have been initialized...
  { Table[i].firstByte=i;    //...set the first...
    Table[i].lastByte=i;     //...and the last byte
    i++;
  }
}

inline void MapGIFBase::NextByte(n8 byte)
{ bitsLeft=(byte<<numBitsLeft) | bitsLeft;  //the new byte is placed to the left
                                            //of the bits, that already are in a
                                            //waiting position
  numBitsLeft+=8;                           //it's 8 bits, that has just been
                                            //added to 'bitsLeft'

  while (numBitsLeft>=codeSize)             //while the next code can be built
                                            //from 'bitsLeft' alone...
  { n16 code=bitsLeft & codeMask;           //...the next code is masked off...

    bitsLeft=bitsLeft >> codeSize;          //...the code is removed from 
    numBitsLeft-=codeSize;                  //'bitsLeft'...

    NextCode(code);                         //...and the code is processed

/* Note: It very important, that the call to NextCode is the last thing within
 *       this loop, since NextCode may change 'codeMask' and 'codeSize'!
 */
  }
}

inline void MapGIFBase::NextCode(n16 code)
{ if (EOImissing)              //if the EOI code hasn't been seen...
  { if (code==CC)              //...if the code is a clear code...
      Init();                  //...the table is initialized
    else
    { if (code==EOI)           //...if the code is a end-of-information code...
        EOImissing=0;          //...mark, that it has been seen
      else                     //...otherwise, the code is a real code...
      { if (firstCode)         //...if it's the first code, it's special...
        { if (code>CC)         //...if the first code isn't a root-code,
                               //something is wrong
            FATAL("Bad code - GIF image corrupted!");
          firstCode=0;         //mark, that the first code has been seen
        }
        else                   //...otherwise...
        { if (nextFree>=4096)  //...if the table is about to overflow...
            FATAL("Bad code - GIF image corrupted!");

          if (code<nextFree)   //...if the code exists in the table...
            Table[nextFree].lastByte=Table[code].firstByte;
            //...the code, that's added on top of the table, is associated with
            //a string, that ends on the same byte, the string, that's
            //associated with the current code, begins with...
          else                   //...if the code doesn't exist in the table...
          { if (code!=nextFree)  //...if the new code does't lie right on top
                                 //of the table...
              FATAL("Bad code - GIF image corrupted!");

            Table[nextFree].lastByte=Table[oldCode].firstByte;
            //...the code, that's added on top of the table, is associated with
            //a string, that ends on the same byte, the string, that's
            //associated with the previous code, begins with...
          }

          Table[nextFree].firstByte=Table[oldCode].firstByte;
          //...and the first byte of the string is identical to the first byte
          //of the string, that's associated with the previous code...
          Table[nextFree].firstBytes=oldCode;
          //...the first part of the string, is the string associated with the
          //previous code...

          if (nextFree==codeMask &&    //...if all codes of the current
                                       //code-size is used up...
            nextFree!=4095)            //...and all 12 bits isn't used up...
          { codeSize+=1;               //...pump the code-size one up...
            codeMask=(codeMask<<1)|1;  //...and make the code-mask one bit wider
          }

/* Note: The check "nextFree!=4095" is necessary, because some images have the
 *       clear code coming just one code too late. It's not very clever.
 */
          nextFree++;  //the next code to be add to the table should be one
                       //bigger than the one, just added
        }
        Output(code);  //real codes, -including the first one-, is translated to
                       //a sequence of pixel output...
        oldCode=code;  //...and the code is remembered
      }
    }
  }
}

inline void MapGIFBase::Output(n16 code)
{ if (code>CC)                       //if the code has more than one pixel
                                     //associated with it...
    Output(Table[code].firstBytes);  //...output the first pixels first

  if (x>maxX)                        //if the right side of the image is
                                     //reached...
  { x=minX;                          //...move the position to the left...
    if (interlaced)                  //...if the image is interlaced...
    { switch(interlacePass)          //...find the current pass-number...
      { case 1:                      //...first pass...
          y+=8;                      //...move 8 lines down...
          if (y>maxY)                //...if the buttom is reached...
          { interlacePass++;         //...start on the next pass
            y=minY+4;
          }
          break;

        case 2:                      //...second pass...
          y+=8;                      //...move 8 lines down...
          if (y>maxY)                //...if the buttom is reached...
          { interlacePass++;         //...start on the next pass
            y=minY+2;
          }
          break;

        case 3:                      //...third pass...
          y+=4;                      //...move 4 lines down...
          if (y>maxY)                //...if the buttom is reached...
          { interlacePass++;         //...start on the next pass
            y=minY+1;
          }
          break;

        case 4:                      //...fourth pass...
          y+=2;                      //...move 2 lines down...
          if (y>maxY)                //...if the buttom is reached...
            FATAL("Too much output - GIF image corrupted!");
          break;
      }
    }                                
    else                             //...if the image is non-interlaced...
    { y++;                           //...move 1 line down...
      if (y>maxY)                    //...if the buttom is reached...
        FATAL("Too much output - GIF image corrupted!");
    }
  }

  if (x>=spix(_x1) && x<=spix(_x2) &&  //only if the position is inside the
      y>=spix(_y1) && y<=spix(_y2))    //window...
    Pixel(Table[code].lastByte,x,y);   //..the pixel is set

  x++;  //increase the x-coordinate
}

void MapGIFBase::ReadColorTable(n16 Colors)
{ colors+=Colors;            //the number of colors is increased

  if (_infoOnly)             //if the image itself shouldn't be read...
    Skip(3*Colors);          //...skip the color table
  else                       //...otherwise...
  { ColorTableInit(Colors);  //...initialize the color table...

    n16 i=0;                 //counter
    while(i<Colors)          //...while not all colors have been read
    { n16 red  =Get()<<8;    //...get the next three primaries
      n16 green=Get()<<8;    //(I use 16 bits per primary!)...
      n16 blue =Get()<<8;
      NextColorTableEntry(red,green,blue);  //...and process the next color
      i++;
    }
  }
}

/*** MapGIF *******************************************************************/

MapGIF::MapGIF(const char* const filename):
  MapGIFBase(filename)  //open the image-file
{ GetImage(1,0,0,0,0,MapInBase::LowX,0,0);  //let GetImage do the work, only the
                                            //first argument takes effect
}

MapGIF::MapGIF(const char* const filename, MapBase& M,
               MapInBase::Gravity g,
               spix CustomX, spix CustomY):
  MapGIFBase(filename)  //open the image-file
{ R=&M;                 //set the handle to the map

  GetImage(0,0,0,M.MaxX(),M.MaxY(),g,CustomX,CustomY);  //let GetImage do the
                                                        //work
}

MapGIF::MapGIF(const char* const filename, MapBase& M,
               pix x1, pix y1, pix x2, pix y2,
               MapInBase::Gravity g,
               spix CustomX, spix CustomY):
  MapGIFBase(filename)  //open the image-file
{ if (x1>x2 ||          //if the window isn't inside the map...
      y1>y2 ||
      x2>M.MaxX() ||
      y2>M.MaxY())
    FATAL("Bad area definition!");

  R=&M;  //set the handle to the map

  GetImage(0,x1,y1,x2,y2,g,CustomX,CustomY);  //let GetImage do the work
}

void MapGIF::ColorTableInit(n16 Colors)
{ index=0;  //initialize the index
}

void MapGIF::NextColorTableEntry(n16 red, n16 green, n16 blue)
{ BitColors[index]=BitIntensity(red,green,blue);  //calculate the bit-color

  index++;  //increase the index
}

void MapGIF::Pixel(n8 color, pix x, pix y)
{ R->BitColor(BitColors[color]);  //select the right bit-color...
  R->PixelNC(x,y);                //...and set the pixel
}

/*** MapPCGIF *****************************************************************/

MapPCGIF::MapPCGIF(const char* const filename):
  MapGIFBase(filename)  //open the image-file
{ GetImage(1,0,0,0,0,MapInBase::LowX,0,0);  //let GetImage do the work, only the
                                            //first argument takes effect
}

MapPCGIF::MapPCGIF(const char* const filename, MapPCBase& M,
                   MapInBase::Gravity g,
                   spix CustomX, spix CustomY,
                   n1 OverrideBW):
  MapGIFBase(filename)  //open the image-file
{ deallocateColors=1;        //deallocation should occur

  overrideBW=OverrideBW;     //set the overriding attributes
  BnotOverrided=OverrideBW;
  WnotOverrided=OverrideBW;

  R=&M;  //set the handle to the map

  GetImage(0,0,0,M.MaxX(),M.MaxY(),g,CustomX,CustomY);  //let GetImage do the
                                                        //work
}

MapPCGIF::MapPCGIF(const char* const filename, MapPCBase& M,
                   pix x1, pix y1, pix x2, pix y2,
                   MapInBase::Gravity g,
                   spix CustomX, spix CustomY,
                   n1 OverrideBW):
  MapGIFBase(filename)  //open the image-file
{ if (x1>x2 ||          //if the window isn't inside the map...
      y1>y2 ||
      x2>M.MaxX() ||
      y2>M.MaxY())
    FATAL("Bad area definition!");

  deallocateColors=1;        //deallocation should occur

  overrideBW=OverrideBW;     //set the overriding attributes
  BnotOverrided=OverrideBW;
  WnotOverrided=OverrideBW;

  R=&M;  //set the handle to the map

  GetImage(0,x1,y1,x2,y2,g,CustomX,CustomY);  //let GetImage do the work
}

MapPCGIF::~MapPCGIF()
{ DeallocColors();  //deallocate colors
}

void MapPCGIF::ForgetColors()
{ deallocateColors=0;  //change the deallocation flag, so deallocation doesn't
                       //take place
}

void MapPCGIF::DeallocColors()
{ if (deallocateColors)                   //if the deallocation flag is set...
  { const word n=AllocatedColors.Size();  //...get the number of colors...
    n16* ac=AllocatedColors.GetAll();     //...get the colors...

    word i=0;                  //counter
    while(i<n)                 //while not all colors have been deallocated...
    { R->DeallocColor(ac[i]);  //...deallocate the next color
      i++;
    }

    deallocateColors=0;  //clear the deallocation flag
  }
}

void MapPCGIF::ColorTableInit(n16 Colors)
{ index=0;                      //initialize the index

  if (Colors<=R->FreeColors())  //if there are colors enough...
  { Bfree=0;                    //...dont't toutch Black and White
    Wfree=0;
  }
  else                         //...otherwise...
  { Bfree=BnotOverrided;       //...override Black and White, if allowed
    Wfree=WnotOverrided;
  }
}

void MapPCGIF::NextColorTableEntry(n16 red, n16 green, n16 blue)
{ if (!overrideBW)                      //if Black and White not may be
                                        //overrided...
  { if (red==0 && green==0 && blue==0)  //...if the color is black...
    { Entries[index++]=0;               //...use the entry for Black...

      return;                           //...and leave
    }
    else
    { if (red==MAX_R && green==MAX_G && blue==MAX_B)  //...if the color is
                                                      //White...
      { Entries[index++]=R->GetWhite();               //...use the entry for
                                                      //White...

        return;                                       //...and leave
      }
    }
  }

  if (Bfree)                          //if Black is free for overriding...
  { R->ColorNC(0);                    //...use the entry for Black...
    R->PaletteEntry(red,green,blue);
    Entries[index++]=0;

    Bfree=0;                          //...mark it as used...
    BnotOverrided=0;

    return;                           //...and leave
  }

  n16 ac;
  if ((ac=R->AllocColor())!=0)          //...if a color can be allocated...
  { R->ColorNC(ac);                     //...use the allocated entry...
    R->PaletteEntry(red,green,blue);
    AllocatedColors.Insert(ac);
    Entries[index++]=ac;

    return;                             //...and leave
  }
  else                                  //...otherwise...
  { if (Wfree)                          //...if White is free for overriding...
    { R->ColorNC(R->GetWhite());        //...use the entry for White...
      R->PaletteEntry(red,green,blue);
      Entries[index++]=R->GetWhite();

      Wfree=0;                          //...mark it as used...
      WnotOverrided=0;

      return;                           //...and leave
    }
    else
      FATAL("The map hasn't got enough free colors!");
  }
}

void MapPCGIF::Pixel(n8 color, pix x, pix y)
{ R->ColorNC(Entries[color]);  //select the right palette entry...
  R->PixelNC(x,y);             //...and set the pixel
}

/*** MapSCGIF *****************************************************************/

MapSCGIF::MapSCGIF(const char* const filename):
  MapGIFBase(filename)  //open the image-file
{ GetImage(1,0,0,0,0,MapInBase::LowX,0,0);  //let GetImage do the work, only the
                                            //first argument takes effect
}

MapSCGIF::MapSCGIF(const char* const filename, MapSCBase& M,
                   MapInBase::Gravity g,
                   spix CustomX, spix CustomY):
  MapGIFBase(filename)  //open the image-file
{ R=&M;                 //set the handle to the map

  GetImage(0,0,0,M.MaxX(),M.MaxY(),g,CustomX,CustomY);  //let GetImage do the
                                                        //work
}

MapSCGIF::MapSCGIF(const char* const filename, MapSCBase& M,
                   pix x1, pix y1, pix x2, pix y2,
                   MapInBase::Gravity g,
                   spix CustomX, spix CustomY):
  MapGIFBase(filename)  //open the image-file
{ if (x1>x2 ||          //if the window isn't inside the map...
      y1>y2 ||
      x2>M.MaxX() ||
      y2>M.MaxY())
    FATAL("Bad area definition!");

  R=&M;  //set the handle to the map

  GetImage(0,x1,y1,x2,y2,g,CustomX,CustomY);  //let GetImage do the work
}

void MapSCGIF::ColorTableInit(n16 NumColors)
{ index=0;  //initialize the index
}

void MapSCGIF::NextColorTableEntry(n16 red, n16 green, n16 blue)
{ Red[index]=red;      //store the triplet
  Green[index]=green;
  Blue[index]=blue;
  index++;
}

void MapSCGIF::Pixel(n8 color, pix x, pix y)
{ R->Color(Red[color],Green[color],Blue[color]);  //select the right color...
  R->PixelNC(x,y);                                //...and set the pixel
}

/******** "mapgif.cc" *********************************************************/

