/*************************************************/
/*                                               */
/*  H E L P F I L E - D E C O M P I L E R  V1.0  */
/*                                               */
/*  Author: Volker Reichel                       */
/*     Bhlstrae 8                              */
/*     7507 Pfinztal 2                           */
/*                                               */
/*  Last chages: 31.01.1992                      */
/*************************************************/

#include <portab.h>
#include <stdio.h>
#include <tos.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "HELP_RC.H"

/*-------- VARIABLES: ---------------------------*/
/*-------- For management of name-tables --------*/
NAME_ENTRY *namelist = NULL; /* Names found      */
int name_cnt = 0;            /* Number of names  */
NAME_ENTRY **name_array;     /* As table         */
NAME_ENTRY *link_list = NULL;/* Link names found */
int  link_cnt = 0;           /* Number of them   */

/*--------- For managing the subindexes ---------*/
SUB_IDX_ENTRY subidx_scrs[INDEX_CNT]; 

/*--------- The search-word tables --------------*/ 
SRCHKEY_ENTRY *key_table = NULL;
SRCHKEY_ENTRY *c_key_table = NULL;

/*---------- The screen-table -------------------*/
long *screen_table;  /* File-offsets of screens  */
int screen_cnt = 0;  /* Number of screens        */
UBYTE *screen_done;  /* 'Done' marking           */

/*---------- The string-table -------------------*/
UBYTE *string_tab;          /* Coded strings     */

/*---------- File-streams -----------------------*/
FILE *hlpfile = NULL;     /* Input help-file     */
FILE *txtfile = NULL;     /* Text output-file    */
FILE *scrfile = NULL;     /* Screen output-file  */
FILE *logfile = NULL;     /* Log-file            */

char filename[80];      
char hlpname[80];      /* Name of help-file      */
char txtname[80];      /* Name of text-file      */
char scrname[80];      /* Name of screen-file    */
char logname[80];      /* Name of log-file       */

char *options;           /* Passed options       */
HLPHDR hlphdr;          /* Header of help-file   */

int glbref_cnt; /* Number of external references */
int warnings;   /* Number of warnings issued     */
int errors;     /* Number of errors              */

/*------------- Some flags for control ----------*/
UBYTE log_flag = FALSE;  
UBYTE txt_flag = FALSE;
UBYTE scr_flag = FALSE;

char msg[512];          /* Buffer for messages   */
char bold_on[80];
char bold_off[80];
char form_feed[80];

/*-------------- Some messages ------------------*/
#define no_ram_msg      Msgs[NO_RAM]
#define ill_opt_msg     Msgs[ILL_OPT]
#define log_opn_err     Msgs[LOG_OPN_ERR]
#define hlp_nf_msg      Msgs[HLP_NF]
#define no_hf_msg       Msgs[NO_HF]
#define hdr_err_msg     Msgs[HDR_SIZE_ERR]
#define rd_sens_msg     Msgs[RD_SENS_TAB]
#define rd_caps_msg     Msgs[RD_CAPS_TAB]
#define rd_scr_msg      Msgs[RD_SCR_TAB]
#define rd_scr_err      Msgs[RD_SCR_ERR]
#define rd_str_msg      Msgs[RD_STR_TAB]
#define rd_str_err      Msgs[RD_STR_ERR]
#define rd_idx_msg      Msgs[RD_IDX]
#define rd_idx_err      Msgs[RD_IDX_ERR]
#define set_attr_msg    Msgs[SET_ATTR]
#define link_msg        Msgs[SET_LINK]
#define decomp_msg      Msgs[DECOMP]
#define decomp_err      Msgs[DECOMP_ERR]
#define file_creat_err  Msgs[F_CREAT_ERR]
#define final_msg       Msgs[FINAL]
#define scr_cnt_msg     Msgs[SCR_CNT]
#define idx_warn_msg    Msgs[IDX_WARN]
#define wr_nt_msg       Msgs[WR_NAME_TAB]
#define wr_lk_msg       Msgs[WR_LINK_TAB]
#define lk_head_msg     Msgs[LINK_HEAD]
#define lk_cnt_msg      Msgs[LINK_CNT]
#define ill_code_msg    Msgs[ILL_CODE]
#define abort_msg       Msgs[ABORT]
#define glb_ref_msg     Msgs[GLB_REF]
#define nt_head_msg     Msgs[NAME_TAB_HEAD]
#define name_cnt_msg    Msgs[NAME_CNT]
#define opt_msg         Msgs[OPTION]
#define maketxt_msg     Msgs[MAKETEXT]
#define hlp_rc1         Msgs[HELP_RC1]
#define hlp_rc2         Msgs[HELP_RC2]
#define hlp_rc3         Msgs[HELP_RC3]

char *Msgs[] = {
   "\n*** Insufficient memory",
   "\nIllegal option character '%c'!\n",
   "\n\n*** Cannot open log-file %s! ***\n",
   "\n\n*** Help-file %s not found! ***\n",
   "\n\n*** The file %s is not a help-file! ***\n",
   "\n\n*** Size of HLPHDR-structure incorrect! ***\n",
   "\n\tReading sensitive search-word table...",
   "\n\tReading capsensitive search-word table...",
   "\n\tReading screen-table...",
   "\n\n*** Cannot read screen-table! ***\n",
   "\n\tReading string-table...",
   "\n\n*** Cannot read string-table! ***\n",
   "\n\tReading index-screen...",
   "\n\nCannot process index! ***\n",
   "\n\tSetting name-atributes...",
   "\n\tRevising \\link-cross-references...",
   "\n\tDecompiling screens...",
   "\n\nCannot decompile help-file!\n\n",
   "\n\nCannot create file %s!\n\n",
   "\n\n%d Errors. %d Warnings. %d Cross-references to other HELP-files.",
   "\t%d screens found.",
   "\n\n*** More than 27 entries in index! ***\n",
   "\n\tWriting name-table...",
   "\n\tWriting link-table...",
   "\n\n%s\n\t\t\tLink-table\n%s\n",
   "\n%d link cross-references found.",
   "\n\n*** Inadmissible screen-code 0x%X! ***\n",
   "\n******* Program will be broken off! *******",
   "\n\t*** WARNING: Global cross-reference <%s>",
   "\n\n%s\n\t\t\tName-table\n%s\n",
   "\n%d names found.",
   "Options (L=Log, S=SCR-file,T=TXT-file): ",
   "\n\tCreating text-file...",
   "\n\t\t\tHELPFILE DECOMPILER  Ver. 1.0\t"__DATE__,
   "\n\t\t\t=============================\n",
   "\n\t\t\t\t(c) Volker Reichel\n\n",
   "VR compiled: "__DATE__" "__TIME__
};


/*----- 'get_nibble()' gets its data from here ------*/
UBYTE *curr_coded_text;
UBYTE must_read = TRUE;       /* For get_nibble    */

/*-----------------------------------------------------------*/
/*-------------- Prototypes ---------------------------------*/
/*-----------------------------------------------------------*/
char *strsave( char *s);
void strfill(char *s, char c, int cnt);
int strint(char *s, char **lp);
int strin(char c, char *m);
void trans_bstr(char *s, UBYTE *bstr);
void read_info(void);
UBYTE get_nibble(void);
UBYTE get_byte(void);
long decode(int index, char *txtbuf);
void wr_header(void);
void wr_keytables(void);
char *get_keyword(SRCHKEY_ENTRY *keytable, int i);
int read_header(void);
int read_screen_table(void);
int read_string_table(void);
int read_Index(void);
int read_Link(void);
int read_key_table(SRCHKEY_ENTRY **ptable, int which);
int is_helpfile(void);
int get_name(char *pos, char *name);
UWORD screen_index(UWORD scr_code);
int is_dir_screen(long offset);
int rd_sidx_names(SUB_IDX_ENTRY subidx_code);
void ins_name(NAME_ENTRY **namelist,int *name_cnt, char *sname, 
              UWORD code, UBYTE attr, UWORD lnk_idx);
int find_name(NAME_ENTRY *namelist, char *sname, NAME_ENTRY **pelem);
void corr_attrs(NAME_ENTRY *namelist);
void setup_namearr(NAME_ENTRY *namelist);
void order_nametable(NAME_ENTRY *namelist);
int write_names(NAME_ENTRY *namelist);
void wr_nametable(void);
void wr_linktable(void);
void transform(char *source,long length,char *d);
int decompile(void);
int make_txtfile(void);
void init_rc(void);
void get_options(void);
void wr_options(void);
void wr_msg(char *s, UBYTE device);
void open_log(void);
int attr_cmp(NAME_ENTRY **elem1,NAME_ENTRY **elem2);

/*--------- Some general routines --------------*/
char *strsave( char *s)
{
  char *p;
  
  p = calloc(strlen(s)+1,1);
  if (p)
    strcpy(p,s);
  return( p );
}


void strfill(char *s, char c, int cnt)
{
  while (cnt-- > 0) 
    *s++ = c;
  *s = EOS;
}


/*--------------------------------------------------------*/
/*  strin:                                                */
/*--------------------------------------------------------*/
/*  Tests whether character 'c' is present in string 'm'. */
/*--------------------------------------------------------*/
int strin(char c, char *m)
{
  while (*m && *m != c)
    m++;
  return( *m && *m == c);
}

/*----------------------------------------------------*/
/*  strint:                                           */
/*----------------------------------------------------*/
/*  Converts the string beginning at 's' into an      */
/*  integer value.                                    */
/*  If the first character of 's' is a '$', then a    */
/*  hex number will be expected. Otherwise a decimal  */
/*  number.                                           */
/*  After the call '*lp' points to the first          */
/*  character that does not belong to a number.       */
/*----------------------------------------------------*/
int strint(char *s, char **lp)
{
  int value;
  int base = 10;
  
  value = 0;
  if (*s == '$') {
    base = 16;
    s++;
  }
  while (*s) {
    *lp = s;
    if (isdigit(*s)) {
      value *= base;
      value += *s - '0';
    }
    else
      if (base == 16 && isxdigit(*s)) {
        value *= 16;
        value += *s - 'A' + 10;
      }
      else
        break;
    s++;
  }  
  return( value );
}


void trans_bstr(char *s, UBYTE *bstr)
{
  static char *numbers = "0123456789ABCDEF";

  while (*bstr)
    if ((*bstr < 0x20) || (*bstr > 0x7F)) {
      *s++ = '\\';
      *s++ = numbers[*bstr >> 4];
      *s++ = numbers[*bstr++ & 0x0F];
    }
    else
      *s++ = (char) *bstr++;
  *s = EOS;
}

 
void init_rc(void)
{
  char *p;
  
  glbref_cnt = 0;
  screen_cnt = 0;
  warnings   = 0;
  errors     = 0;
  p = strchr(filename,'.');
  if (p)
    *p = EOS;             /* Delete extensions */
  sprintf(hlpname,"%s.HLP",filename);
  sprintf(logname,"%s.LOG",filename);
  sprintf(txtname,"%s.TXT",filename);
  sprintf(scrname,"%s.SCR",filename);
}


void get_options(void)
{    
  char *s;
  
  s = options;
  while (*s)
    switch (*s++) {
      case 'L':
      case 'l': log_flag = TRUE;
            break;
      case 'T':
      case 't': txt_flag = TRUE;
            break;
      case 'S':
      case 's': scr_flag = TRUE;
            break;
      case '-':
      case ' ':
      case '\t':
            break;
      default: 
            sprintf(msg,ill_opt_msg,*(s-1));
            wr_msg(msg,TO_ALL);
    } /* switch */
    log_flag = log_flag || !(txt_flag || scr_flag);
}


/*--------------------------------------------------------*/
/*  read_info:                                            */
/*--------------------------------------------------------*/
/*  Reads from the file HELP_RC.INF the control sequences */
/*  to be used to adapt the output.                       */
/*  'bold_on', 'bold_off', 'form_feed' are used by        */
/*  'make_txtfile'.                                       */
/*  File format:                                          */
/*    Comments are preceded by '*'. They begin in the     */
/*    first column and finish at the end of the line.     */
/*    First comes the character string to switch on bold  */
/*    printing, followed by that for switching it off.    */
/*    At the end one also has to specify the character    */
/*    string to be used when changing screens.            */
/*    The character string is to be input as pairs of     */
/*    hex numbers, separated by commas. It finishes at    */
/*    the end of the line.                                */
/*--------------------------------------------------------*/
void read_info(void)
{
  FILE *info_file;
  int cnt = 0;
  char *s, *sp, *lp;
  char line[180];
  
  info_file = fopen("HELP_RC.INF","r");
  if (info_file == NULL) {
    strcpy(bold_on,BOLD_ON);
    strcpy(bold_off,BOLD_OFF);
    strcpy(form_feed,FORM_FEED);
    return;
  }
  
  /*--------- File present ------------*/
  while (!feof(info_file) && (cnt < 3)) {
    fgets(line,80,info_file);
    if ( *line && !strin(*line,"\n \t*")) {
      switch (cnt) {
        case 0 :
          s = bold_on; break;
        case 1 : 
          s = bold_off; break;
        case 2 :
          s = form_feed; break;
      }
      cnt++;
      lp = sp = line;
      while (*sp && strin(*sp," \t,$")) {
        while (*sp && strin(*sp," \t,"))
          sp++;
        if (*sp) {
          *s++ = strint(sp,&lp);
          sp = lp;
        }
      } /* while */
      *s = '\0';
    } /* if */
  } /* while */
  fclose(info_file);
}



void wr_nametable(void)
{
  char bar[81];
  
  wr_msg(wr_nt_msg,TO_SCREEN);
  strfill(bar,'=',80);
  fprintf(logfile,nt_head_msg,bar,bar);
  write_names(namelist);
  fprintf(logfile,name_cnt_msg,name_cnt);
}
  
 
void wr_linktable(void)
{
  char bar[81];
  
  wr_msg(wr_lk_msg,TO_SCREEN);
  strfill(bar,'=',80);
  fprintf(logfile,lk_head_msg,bar,bar);
  write_names(link_list);
  fprintf(logfile,lk_cnt_msg,link_cnt);
}


void wr_options(void)
{
  sprintf(msg,"Options: %s\n\n",options);
  wr_msg(msg,TO_ALL);
}


void wr_msg(char *s, UBYTE device)
{
  if (device & TO_SCREEN) {
    printf("%s",s);
  }
  if (log_flag && (device & TO_LOG))
      fprintf(logfile,"%s",s);
}


void open_log(void)
{
  if (!logfile) {
    logfile = fopen(logname,"w");
    if (!logfile) {
      printf(log_opn_err,logname);
      errors++;
      log_flag = FALSE;
    } /* Log-file cannot be created */
    else {
      setvbuf(logfile,NULL,_IOFBF,32*1024L);
      fprintf(logfile,"%s%s%s",hlp_rc1,
                               hlp_rc2,
                               hlp_rc3);
    }
  } /* Logfile nicht geffnet */
}


/*----------------------------------------------*/
/*  get_nibble:                                 */
/*----------------------------------------------*/
/*  Reads the next half-byte from the input.    */
/*  'curr_coded_text' points to the next byte.  */
/*----------------------------------------------*/
UBYTE get_nibble(void)
{
  static UBYTE byte_read;
  UBYTE nibble;
  
  if (must_read) {
    byte_read = *curr_coded_text++;
    nibble = byte_read >> 4;
    must_read = FALSE;
  }
  else {
    nibble = byte_read & 0x0F;
    must_read = TRUE;
  }
  return( nibble );
}


/*----------------------------------------------*/
/*  get_byte:                                   */
/*----------------------------------------------*/
/*  Reads the next two nibbles from the input   */
/*  and returns them as a byte.                 */
/*----------------------------------------------*/
UBYTE get_byte(void)
{
  UBYTE byte;
  
  byte = get_nibble();
  byte <<= 4;
  byte += get_nibble();
  return( byte );
}


void wr_header(void)
{
  char char_string[50];
  
  trans_bstr(char_string,
             (UBYTE *) hlphdr.char_table);
  fprintf(logfile,"\nHeader of help-file %s\n\n",
                  hlpname);
  fprintf(logfile,"\tScreens\t\t\t\t\t: %4ld\n",
                  hlphdr.scr_tab_size>>2);
  fprintf(logfile,
         "\tStart string-table\t: %ld (0x%p)\n",
         hlphdr.str_offset,hlphdr.str_offset);
  fprintf(logfile,
              "\tLength\t\t\t\t\t: %ld (0x%p)\n",
              hlphdr.str_size,hlphdr.str_size);
  fprintf(logfile,
              "\tThe most frequent characters\t: %s\n",
              char_string);
  fprintf(logfile,
      "\tStart sensitive table\t: %ld (0x%p)\n",
      hlphdr.sens_offset,hlphdr.sens_offset);
  fprintf(logfile,
               "\tLength\t\t\t\t\t: %ld (0x%p)\n",
               hlphdr.sens_size,hlphdr.sens_size);
  fprintf(logfile,
         "\tNumber of sensitive words\t: %ld (0x%p)\n",
         hlphdr.sens_cnt,hlphdr.sens_cnt);
  fprintf(logfile,
       "\tStart capsens. table\t: %ld (0x%p)\n",
       hlphdr.caps_offset,hlphdr.caps_offset);
  fprintf(logfile,
       "\tLength\t\t\t\t\t: %ld (0x%p)\n",
       hlphdr.caps_size,hlphdr.caps_size);
  fprintf(logfile,
       "\tNumber of capsens words\t: %ld (0x%p)\n",
       hlphdr.caps_cnt,hlphdr.caps_cnt);
}


int read_key_table(SRCHKEY_ENTRY **ptable,
                  int which)
{
  long offset;
  long size;
  
  *ptable = NULL;
  if (which == SENS_TABLE) {
    offset = hlphdr.sens_offset;
    size   = hlphdr.sens_size;
    wr_msg(rd_sens_msg,TO_SCREEN);
  }
  
  if (which == CAP_TABLE) {
    offset = hlphdr.caps_offset;
    size   = hlphdr.caps_size;
    wr_msg(rd_caps_msg,TO_SCREEN);
  }
  
  if (size != 0) {
    fseek(hlpfile,offset,SEEK_SET);
    *ptable = (SRCHKEY_ENTRY *) malloc(size);
    if (*ptable != NULL)
      fread(*ptable,1,size,hlpfile);
  }
  return( *ptable != NULL );
}


/*---------------------------------------------*/
/*  read_coded:                                */
/*---------------------------------------------*/
/*  Reads in the 'Index' screen.               */
/*---------------------------------------------*/
int read_coded(int index, UBYTE *coded_text)
{
  long code_length, bytes_read;
  
  code_length = screen_table[index+1] 
                           - screen_table[index];
  fseek(hlpfile,screen_table[index],SEEK_SET);
  bytes_read = 
         fread(coded_text,1,code_length,hlpfile);
  return( bytes_read == code_length );
}

    
void wr_keytable(SRCHKEY_ENTRY *table, int cnt)
{
  int i;
  
  fprintf(logfile,
               "\tPosition   Code  Search-word\n");
  for(i=0;i<cnt;i++)
    fprintf(logfile,"\t   0x%p   %x  \"%s\"\n",
                    table[i].pos,
                    table[i].code,
                    get_keyword(table, i));
}



void wr_keytables(void)
{
  char bar[81];
  
  if (hlphdr.caps_cnt > 0) {
    strfill(bar,'c',80);
    fprintf(logfile,
         "\n\n%s\n\t\tCapsensitive table\n%s\n",
         bar,bar);
    wr_keytable(c_key_table,(int)hlphdr.caps_cnt);
  }
  
  if (hlphdr.sens_cnt > 0) {  
    strfill(bar,'s',80);
    fprintf(logfile,
            "\n\n%s\n\t\tSensitive table\n%s\n",
            bar,bar);
    wr_keytable(key_table,(int) hlphdr.sens_cnt);
  }
}


/*----------------------------------------------*/
/*  get_keyword:                                */
/*----------------------------------------------*/
/*  Gets the 'i'th keyword from the search-word */
/*  table. ('i' starts from 0)                  */
/*----------------------------------------------*/
char *get_keyword( SRCHKEY_ENTRY *keytable, int i)
{
  return((char *) (&keytable[i])+keytable[i].pos);
}


/*----------------------------------------------*/
/*  corr_attrs:                                 */
/*----------------------------------------------*/
/*  Corrects the assumption that all names are  */
/*  screen-names. For this the attribute that   */
/*  corresponds to its affiliation to the       */
/*  search-word tables is set.                  */
/*----------------------------------------------*/
void corr_attrs(NAME_ENTRY *namelist)
{  
  int i;
  char *search_name;
  NAME_ENTRY *elem;
  
  wr_msg(set_attr_msg,TO_SCREEN);
  /*----- First the sensitive names ------*/
  for (i=0; i < hlphdr.sens_cnt; i++) {
    search_name = get_keyword(key_table,i);
    if (find_name(namelist, search_name, &elem))
      elem->name_attr = SENSITIVE;
  }
  
  /*----- Now for the capsensitive names */
  for (i=0; i < hlphdr.caps_cnt; i++) {
    search_name = get_keyword(c_key_table,i);
    if (find_name(namelist, search_name, &elem))
      elem->name_attr = CAP_SENS;
  }
}


/*--------------------------------------------------*/
/*  decode:                                         */
/*--------------------------------------------------*/
/*  Decodes the screen given by ScreenTable[index]. */
/*  'plain_text' must point to a sufficiently       */
/*  large storage area.                             */ 
/*  The return is the length of the decoded screen. */
/*--------------------------------------------------*/
long decode(int index, char *plain_text)
{
  static UBYTE first_call = TRUE;
  static UBYTE *code_buffer = NULL;
  UBYTE nibble;
  UWORD idx;
  UWORD str_len;
  ULONG offset;
  char *p;
  long size = 0L;
  
  if (first_call) {
    code_buffer = malloc(MAXCODEDSIZE);
    first_call = FALSE;
  }
 
  /*------------- Read the screen -------*/
  if (!read_coded(index,code_buffer))
    return( 0L );
    
  curr_coded_text = code_buffer;
  must_read = TRUE;  /* No byte read yet       */
  
  /*------------ Now also decode it -----------*/
  while (TRUE) {
    nibble = get_nibble();
    if (nibble == CHAR_DIR) {
      *plain_text++ = (char) get_byte();
      size++;
    }
    else
    if (nibble < CHAR_DIR) { 
      *plain_text++ = hlphdr.char_table[nibble];
      size++;
    }
    else
    if (nibble == STR_TABLE) {
      *plain_text++ = (char) CR;
      *plain_text++ = (char) LF;
      size += 2;
    }
    else 
    if (nibble < STR_TABLE) {
      idx = get_byte() << 4;
      idx += get_nibble();
      str_len = (UWORD) 
             (((long *) string_tab)[idx+1] - 
              ((long *) string_tab)[idx]);
      offset = ((long *) string_tab)[idx];
      p = (char *) string_tab+offset;
      size += str_len;
      while (str_len-- > 0) {
        *plain_text++ = (char) (*p ^ 0xA3);
        p++;
      }
    }
    else {
      *plain_text = EOS;
      break;
    }
  }
  return(size);
}


/*----------------------------------------------*/
/*  screen_index:                               */
/*----------------------------------------------*/
/*  Calculates from a reference code the index  */
/*  to the screen-table.                        */ 
/*----------------------------------------------*/
UWORD screen_index(UWORD scr_code)
{
  UWORD index;
  
  if ((scr_code & 0x0004) > 0x0004) {
    sprintf(msg,ill_code_msg,scr_code);
    wr_msg(msg,TO_ALL);
    wr_msg(abort_msg,TO_ALL);
    errors++;
    exit(1);
  }
  index = (((scr_code & 0x7FF8) >> 1) - 4) >> 2;
  return( index );
}


/*----------------------------------------------*/
/*  get_name:                                   */
/*----------------------------------------------*/
/*  Returns in 'name' the string starting from  */
/*  'pos' and finishing at the next ESC_CHR.    */
/*  In addition the total length of the string  */
/*  is returned.                                */
/*----------------------------------------------*/
int get_name(char *pos, char *name)
{
  char *s;
      
  s = name;
  while (*pos != ESC_CHR)
    *s++ = *pos++;
  *s = '\0';
  return( (int) (s - name + 1) );
}


/*-----------------------------------------------*/
/*  find_name:                                   */
/*-----------------------------------------------*/
/*  Searches in the name-list 'namelist' for the */
/*  name 'sname' and if successfull returns a    */
/*  pointer '*pelem' to the found entry.         */
/*-----------------------------------------------*/
int find_name(NAME_ENTRY *namelist, char *sname,
              NAME_ENTRY **pelem)
{
  UBYTE found = FALSE;
  
  while ((namelist != NULL) && !found) {
    found = strcmp(namelist->name,sname) == 0;
    *pelem = namelist;
    namelist = namelist->next;
  }
  
  return( found );
}


/*----------------------------------------------*/
/*  find_code:                                  */
/*----------------------------------------------*/
/*  Searches in the name-table for the cross-   */
/*  reference 'code' code.                      */
/*----------------------------------------------*/
int find_code(UWORD search_code, 
              NAME_ENTRY **pelem)
{
  int i;
  
  for (i=0; i < name_cnt; i++)
    if (name_array[i]->scr_code == search_code) {
      *pelem = name_array[i];
      return( TRUE );
    }
  return( FALSE );
}


int write_names(NAME_ENTRY *namelist)
{
  static char *attr_str[ATTR_CNT] = 
           { "SCREEN_NAME ",
             "CAPSENSITIVE",
             " SENSITIVE  ",
             "   LINK     ",
           };
  int i = 0;
  
  fprintf(logfile,
"Name\t\t\t\tAttribute    Code     ScreenOffset\n");

  while (namelist != NULL) {
    fprintf(logfile,"<%-32.32s> %s 0x%X",
                    namelist->name,
                    attr_str[namelist->name_attr],
                    namelist->scr_code);
    if (namelist->name_attr == LINK)
      fprintf(logfile," from 0x%X",
                      namelist->link_index);
    else
      fprintf(logfile," = 0x%X",
              screen_index(namelist->scr_code));
    fprintf(logfile,"\n");
    namelist = namelist->next;
    i++;
  }
  return( i );
}


/*-----------------------------------------------*/
/* ins_name:                                     */
/*-----------------------------------------------*/
/*  Inserts the name 'sname' with its attributes */
/*  code 'attr' and 'lnk_idx' into the name-list */
/*  '*namelist'. During this the number of       */
/*  insertions will be counted in '*name_cnt'.   */
/*-----------------------------------------------*/
void ins_name(NAME_ENTRY **namelist,int *name_cnt,
              char *sname, UWORD code, UBYTE attr, 
              UWORD lnk_idx)
{
  NAME_ENTRY *new;
  
  new = malloc( sizeof(NAME_ENTRY) );
  if (!new) {
    sprintf(msg,"%s for names ***\n",no_ram_msg);
    wr_msg(msg,TO_ALL);
    wr_msg(abort_msg,TO_ALL);
    errors++;
    exit(1);
  }
  /* Insert at start of list  */
  new->next = *namelist; 
  new->name_attr = attr;
  new->scr_code = code;
  new->name = strsave(sname);
  if (attr == LINK)
    new->link_index = lnk_idx;
  *namelist = new;
  (*name_cnt)++;
}


/*-----------------------------------------------*/
/*  attr_cmp:                                    */
/*-----------------------------------------------*/
/* Compares two elements of the name-list for    */
/* their attribute-value.                        */
/*-----------------------------------------------*/
int attr_cmp(NAME_ENTRY **elem1,NAME_ENTRY **elem2)
{
  ULONG val1, val2;
  UWORD idx1, idx2;
  
  idx1 = screen_index((*elem1)->scr_code);
  idx2 = screen_index((*elem2)->scr_code);
    
  val1 = (idx1 << 4) + (*elem1)->name_attr;
  val2 = (idx2 << 4) + (*elem2)->name_attr;
  
  return((val1<val2) ? -1 : (val1>val2) ? 1 : 0);
}


/*-----------------------------------------*/
/*  setup_namearr:                         */
/*-----------------------------------------*/
/*  Creates a dynamic array and saves the  */
/*  names-list to it.                      */
/*-----------------------------------------*/
void setup_namearr(NAME_ENTRY *namelist)
{
  int  arr_idx;
  
  name_array = 
            malloc(name_cnt*sizeof(NAME_ENTRY *));
  if (!name_array) {
    sprintf(msg,"\n%s for name_array!\n",
            no_ram_msg);
    wr_msg(msg,TO_ALL);
    errors++;
    exit(1);
  }
  
  arr_idx = 0;
  while ((arr_idx < name_cnt) 
          && (namelist != NULL)) {
    name_array[arr_idx++] = namelist;
    namelist = namelist->next;
  }
}


/*-------------------------------------------------*/
/*  transform:                                     */
/*-------------------------------------------------*/
/*  Transforms the source so that it can be output */
/*-------------------------------------------------*/
void transform(char *source, long length, char *d)
{
  char *s, *limit;
  NAME_ENTRY *elem;
  char name[80];
  UWORD code;
  UBYTE global = FALSE;    /* Global reference */

  s = source; limit = source + length;
  while (s < limit)
    switch (*s) {
      case ESC_CHR :
        code = (* (UBYTE *)(s+1)) << 8;
        code += *(UBYTE *)(s+2);
        s += 3;         /* Point to name */
        s += get_name(s,name);
        if (code == 0xFFFF) {
          sprintf(msg,glb_ref_msg,name);
          wr_msg(msg,TO_ALL);
          warnings++;
          glbref_cnt++;
          global = TRUE;
        }
        if (find_name(link_list,name,&elem) 
            || global)
        {
          if (!global) 
            find_code(elem->scr_code,&elem);
          strcpy(d,"\\link(\""); d += 7;
          if (global) {
            strcpy(d,"%%GLOBAL%%"); d += 10; 
            global = FALSE;
          } 
          else {
            strcpy(d,elem->name); 
            d += strlen(elem->name);
          }
          strcpy(d,"\")"); d += 2;
          strcpy(d,name);  d += strlen(name);
          strcpy(d,"\\#"); d += 2;
        } 
        else {
          strcpy(d,"\\#"); d += 2;
          strcpy(d,name);  d += strlen(name);
          strcpy(d,"\\#"); d += 2;
        }
        break;

      case CR :
        *d++ = '\n';
        s += 2;          /* Jump over LF */
        break;

      case BACKSLASH:    /* Must be doubled up  */
        *d++ = *s++;
        *d++ = '\\';
        break;
      default :
        *d++ = *s++;
    } /* switch */
    
  *d = '\0';
}


/*----------------------------------------------*/
/*  decompile:                                  */
/*----------------------------------------------*/
/*  Re-creates a readable text from a HLP-file. */
/*----------------------------------------------*/
int decompile(void)
{
  int i=0;
  UWORD last_code;
  UBYTE new_screen = TRUE;
  char *result;
  char *textbuffer;
  long textlength;

  wr_msg(decomp_msg,TO_SCREEN);
  result = malloc(TXTBUFSIZE);
  if (!result) {
    sprintf(msg,"%s for result!",no_ram_msg);
    wr_msg(msg,TO_ALL);
    errors++;
    return( FALSE );
  }
      
  textbuffer = malloc(MAXCODEDSIZE);
  if (!textbuffer) {
    sprintf(msg,"%s for text-buffer!",no_ram_msg);
    wr_msg(msg,TO_ALL);
    errors++;
    return( FALSE);
  }
  
  /*----- Sort by attributes -----*/
  setup_namearr(namelist);
  qsort(name_array,name_cnt,sizeof(NAME_ENTRY *),
        attr_cmp);

  scrfile = fopen(scrname,"w");
  if (!scrfile) {
    sprintf(msg,file_creat_err,scrname);
    wr_msg(msg,TO_ALL);
    errors++;
    return(FALSE);
  }
  
  setvbuf(scrfile,NULL,_IOFBF,32*1024L);
  
  last_code = name_array[0]->scr_code;
  while (i < name_cnt) {
    while ((i < name_cnt) && 
          (name_array[i]->scr_code == last_code)) 
    {
      if (new_screen) {
        fprintf(scrfile,"\n\nscreen( ");
        new_screen = FALSE;
      }
      else
        fprintf(scrfile,",\n\t\t");

      switch (name_array[i]->name_attr) {
        case SCR_NAME :
          fprintf(scrfile,"\"%s\"",
                        name_array[i]->name);
          break;
        case SENSITIVE :
          fprintf(scrfile,"sensitive(\"%s\")",
                          name_array[i]->name);
          break;
        case CAP_SENS :
          fprintf(scrfile,"capsensitive(\"%s\")",
                          name_array[i]->name);
          break;
      } /* switch */
      i++;             /* Next entry */
    } /* while */
    fprintf(scrfile," )\n");
    textlength = decode(screen_index(last_code),
                        textbuffer);
    transform(textbuffer,textlength,result);
    fputs(result,scrfile);
    fputs("\n\\end",scrfile); 
    /*fprintf(scrfile,"%s\n\\end",result);*/

    if (i < name_cnt) 
      last_code = name_array[i]->scr_code;
    new_screen = TRUE;
  } /* while */
  fprintf(scrfile,"\n");
  fclose(scrfile);
  return( TRUE );
}


/*----------------------------------------------*/
/*  make_txtfile:                               */
/*----------------------------------------------*/
/*  Lists all screens in order.                 */
/*  'bold_on' or 'bold_off' determine how       */
/*  cross-references are emphasised.            */
/*  'form_feed' determines how screens are to   */
/*  be separated.                               */
/*----------------------------------------------*/
int make_txtfile(void)
{
  int index;
  char *textbuffer, *tp, *limit;
  long size;
  
  wr_msg(maketxt_msg,TO_SCREEN);
  txtfile = fopen(txtname,"w");
  if (!txtfile) {
    sprintf(msg,file_creat_err,txtname);
    wr_msg(msg,TO_ALL);
    errors++;
    return( FALSE );
  }
  
  setvbuf(txtfile,NULL,_IOFBF,32*1024L);
  
  textbuffer = malloc(TXTBUFSIZE);
  if (!textbuffer) {
    wr_msg(no_ram_msg,TO_ALL);
    errors++;
    return( FALSE );
  }
  
  for (index = 0; index < screen_cnt; index++) {
    if (!is_dir_screen(screen_table[index])) {
      size = decode(index,textbuffer);
      tp = textbuffer;
      limit = textbuffer + size;
      while (tp < limit) {
        switch (*tp) {
          case ESC_CHR:
            fputs(bold_on,txtfile);
            tp += 3;     /* Skip over cross-reference */
            while (*tp != ESC_CHR)
              fputc(*tp++,txtfile);
            fputs(bold_off,txtfile);
            tp++;  
            break;
          case CR:
            tp += 2;      /* Skip over LF */
            fputc('\n',txtfile);
            break;
          default:
            fputc(*tp++,txtfile);
        } /* switch */
      } /* while */
      fputs(form_feed,txtfile);
    } /* if */
  } /* for */
  
  free(textbuffer);
  fclose(txtfile);

  return( TRUE );
}


/*------------------------------------------------*/
/*  rd_sidx_names:                                */
/*------------------------------------------------*/
/* Enters all the names present in this sub-index */
/* screen into the name-table. The assumption     */
/* that they are all screen-names will be         */
/* corrected later.                               */
/*------------------------------------------------*/
int rd_sidx_names(SUB_IDX_ENTRY subidx_code)
{
  static char *plain_text = NULL;
  long size;
  char *pos;           
  UWORD scr_index;    /* Index in screen-table */
  UWORD screen_code;    /* that belongs to the name */
  char screen_name[80];    
  static UBYTE first_call = TRUE;
  
  if (first_call) {
    plain_text = malloc(MAXCODEDSIZE);
    first_call = FALSE;
  }
  if (!plain_text) 
    return( FALSE );
    
  scr_index = screen_index(subidx_code);
  if (!screen_done[scr_index]) {
    size = decode(scr_index,plain_text);

    /* Work through every entry of the IV */
    pos = plain_text;
    while (pos < plain_text + size) {
      if (*pos == ESC_CHR) {
        pos++;
        screen_code = * (UBYTE *) pos++ << 8;
        screen_code += * (UBYTE *) pos++;
        pos += get_name(pos,screen_name);
        ins_name(&namelist,&name_cnt,screen_name,
                 screen_code,SCR_NAME,0);
      }
      else
        pos++;
    } /* while */
    screen_done[scr_index] = TRUE;
  } /* if */
  return( TRUE );
}


/*----------------------------------------------*/
/*  is_dir_screen:                              */
/*----------------------------------------------*/
/* Determines whether the offset belongs to a   */
/* screen.                                      */
/*----------------------------------------------*/
int is_dir_screen( long offset )
{
  int i;
  
  for (i=0; i < INDEX_CNT; i++)
    if (screen_table[screen_index(subidx_scrs[i])]
        == offset)
      return( TRUE );
  return( FALSE );
}


/*----------------------------------------------*/
/*  read_Link:                                  */
/*----------------------------------------------*/
/* Searches all screens except the Copyright-,  */
/* Index- and all Subindex-screens for the      */
/* cross-references they contain. If during     */
/* this a cross-reference is found that is not  */
/* contained in the name-table, then it is      */
/* certain that this reference has been         */
/* generated via a '\link' instruction, since   */
/* '\link' instructions do not produce entries  */
/* in the search-word tables. The reference     */
/* will be included in the link-list.           */
/*----------------------------------------------*/
int read_Link(void)
{
  int i;
  long size;
  char *pos;
  char name[80];
  UWORD to_code;    
  NAME_ENTRY *elem; 
  static char *plain_text = NULL;
  static UBYTE first_call = TRUE;
  char *limit;
  
  wr_msg(link_msg,TO_SCREEN);
  if (first_call) {
    plain_text = malloc(TXTBUFSIZE);
    first_call = FALSE;
  }
  
  if (!plain_text)
    return( FALSE );

  /*--- Screen 0 Copyright screen 1 Index -----*/
  for(i=2;i<(hlphdr.scr_tab_size >> 2)-1;i++) {
    /*------ Is it a directory screen?---*/
    if (!is_dir_screen(screen_table[i])) { 
      /*------ Fetch page and decode it ---*/
      size = decode(i,plain_text);
      /*----- Work through every name -----*/
      /*----- entry of a screen -----------*/
      pos = plain_text; 
      limit = plain_text + size;
      while (pos < limit) {
        if (*pos == ESC_CHR) {
          pos++;
          to_code = * (UBYTE *) pos++ << 8;
          to_code += * (UBYTE *) pos++;
          pos += get_name(pos,name);
          if (!find_name(namelist,name,&elem) &&
              !find_name(link_list,name,&elem)) {
            ins_name(&link_list,&link_cnt,name,
                     to_code,LINK,i);
          }
        } /* if ESC_CHR */
        else
          pos++;
      } /* while */
    } /* if */
  } /* for */
  return( TRUE );
}


/*----------------------------------------------*/
/*  read_Index:                                 */
/*----------------------------------------------*/
/* An Index-screen will be read and the screens */
/* belonging to the letters A to Z and the      */
/* 'Miscellaneous' entry will be determined.    */
/* Following this all names present on the Sub- */
/* index screens will be read in.               */
/*----------------------------------------------*/
int read_Index(void)
{
  
  char *plain_idx_text;    /* Decoded Index */ 
  long size;                     /* Its length */
  char *limit;
  int  sub_idx = 0;     /* Entry being worked on */
  int  i;
  char *pos;
  char dummy[80];        
  UWORD screen_code;

  wr_msg(rd_idx_msg,TO_SCREEN);
  plain_idx_text = malloc(0x1000L);
  if (!plain_idx_text)
    return( FALSE );
    
  screen_done[INDEX_SCR] = TRUE;
  size = decode(INDEX_SCR,plain_idx_text);
  
  /* Work through every entry of the Index */
  pos = plain_idx_text;
  limit = plain_idx_text + size;
  while (pos < limit) {
    if (*pos == ESC_CHR) {
      pos++;
      screen_code = * (UBYTE *) pos++ << 8;
      screen_code += * (UBYTE *) pos++;
      if (sub_idx >= INDEX_CNT) {
        wr_msg(idx_warn_msg,TO_ALL);
        warnings++;
      }
      else 
        subidx_scrs[sub_idx] = screen_code;
      sub_idx++;
      pos += get_name(pos,dummy);
    }
    else
      pos++;
  }
  
  /* Now work through every Sub-index */
  for (i=0;i<INDEX_CNT;i++)
    if (!rd_sidx_names(subidx_scrs[i]))
      return( FALSE );
    
  free( plain_idx_text );
  return( TRUE );
}


/*----------------------------------------------*/
/*  is_helpfile:                                */
/*----------------------------------------------*/
/* Are we dealing with an HC2.0 file?           */
/*----------------------------------------------*/
int is_helpfile(void)
{
  char buffer[4];
  
  fseek(hlpfile,0x54L,SEEK_SET);
  fread(buffer,1,4L,hlpfile);
  return(
    !strncmp(buffer,HC_VERS,strlen(HC_VERS)));
}


/*------------------------------------------*/
/*  read_header:                            */
/*------------------------------------------*/
/*  Reads the description block from the    */
/*  help-file.                              */ 
/*------------------------------------------*/
int read_header(void)
{
  fseek(hlpfile,0x58L,SEEK_SET);
  return( 
      fread(&hlphdr,1,sizeof(HLPHDR),hlpfile)
      == HEADER_SIZE );
}
  

/*-------------------------------------------*/
/*  read_screen_table:                       */
/*-------------------------------------------*/
/* Reads the table with the screen offsets   */
/* from the help-file.                       */
/*-------------------------------------------*/  
int read_screen_table(void)
{
  long bytes_read;
  int  i;
  
  wr_msg(rd_scr_msg,TO_SCREEN);
  fseek(hlpfile,0x88L,SEEK_SET);
  screen_table = 
            (long *) malloc(hlphdr.scr_tab_size);
  if (!screen_table) {
    sprintf(msg,"%s for screen_table",no_ram_msg);
    wr_msg(msg,TO_ALL);
  }
  screen_cnt = (int) hlphdr.scr_tab_size >> 2;
  screen_done = (UBYTE *) malloc(screen_cnt);
  if (!screen_done) {
    sprintf(msg,"\n%s for screen done!\n",
            no_ram_msg);
    wr_msg(msg,TO_ALL);
  }
  
  
  for(i=0;i < screen_cnt; i++) {
    screen_done[i] = FALSE;
  }
    
  bytes_read = 
    fread(screen_table,1,hlphdr.scr_tab_size,
          hlpfile);
  return( (bytes_read == hlphdr.scr_tab_size) 
          && screen_table && screen_done);
}
  
  
/*---------------------------------------------*/
/*  read_string_table:                         */
/*---------------------------------------------*/
/*  Reads in the table with the code-strings.  */
/*---------------------------------------------*/  
int read_string_table(void)
{
  long bytes_read;
  
  wr_msg(rd_str_msg,TO_SCREEN);
  string_tab = (UBYTE *) malloc(hlphdr.str_size);
  if (!string_tab) {
    sprintf(msg,"%s for string_table",no_ram_msg);
    wr_msg(msg,TO_ALL);
  }
  fseek(hlpfile,hlphdr.str_offset,SEEK_SET);
  bytes_read = 
      fread(string_tab,1,hlphdr.str_size,hlpfile);
  return( (bytes_read == hlphdr.str_size) 
          && string_tab);
}


int main(int argc, char *argv[])
{
  char buf[40];
  
  printf("%s%s%s",hlp_rc1,hlp_rc2,hlp_rc3);
  
  read_info();
  if (argc > 2) {
    options = argv[1];
    strcpy(filename,argv[2]);
  }
  else {
    printf("Name of the HELP-file: ");
    gets(filename);
    printf(opt_msg);
    gets(buf);
    options = buf;
  }
  init_rc();
  get_options();
  if (log_flag) open_log();
  wr_options();
 
  hlpfile = fopen(hlpname,"rb");
  if (!hlpfile) {
    printf(hlp_nf_msg,hlpname);
    errors++;
    goto end;
  }
  setvbuf(hlpfile,NULL,_IOFBF,4*1024L);

  if (!is_helpfile()) {
    printf(no_hf_msg,hlpname);
    errors++;
    goto end;
  }
  
  if (!read_header()) {
    printf(hdr_err_msg);
    errors++;
    goto end;
  }
  if (log_flag) wr_header();
  
  read_key_table(&key_table,SENS_TABLE);
  read_key_table(&c_key_table,CAP_TABLE);
  if (log_flag) wr_keytables();
  
  if (!read_screen_table()) {
    wr_msg(rd_scr_err,TO_ALL);
    errors++;
    goto end;
  }
  sprintf(msg,scr_cnt_msg,screen_cnt);
  wr_msg(msg,TO_SCREEN);
  
  if (!read_string_table()) {
    wr_msg(rd_str_err,TO_ALL);
    errors++;
    goto end;
  }
 
  if (!read_Index()) {
    wr_msg(rd_idx_err,TO_ALL);
    errors++;
    goto end;
  }
  
  corr_attrs(namelist);
  if (log_flag) wr_nametable();

  read_Link();
  if (log_flag) wr_linktable();

  if (scr_flag) {  
    if (!decompile()) {
      wr_msg(decomp_err,TO_ALL);
      errors++;
      goto end;
    } /* if */
  } /* if */

  if (txt_flag) {
    if (!make_txtfile()) {
      sprintf(msg,file_creat_err,txtname);
      wr_msg(msg,TO_ALL);
      errors++;
      goto end;
    }
  }

end:  

  sprintf(msg,final_msg,errors,warnings,
                        glbref_cnt);
  wr_msg(msg,TO_ALL);
  fclose(hlpfile);
  if (log_flag) fclose(logfile);
  puts("\nFinished!");
  Cnecin();
  return( errors );
}
