#ifndef SPRITE_INCLUDED
#define SPRITE_INCLUDED

#pragma pack(1)

#ifdef __cplusplus
extern "C" {
#endif

#define MAX_SPRITES              64

#define MAX_SPRITE_CX            128
#define MAX_SPRITE_CY            128

typedef LONG SPRERROR;

#define SPR_ERR_NOERROR          0x00000000L
#define SPR_ERR_GENERROR         0xFFFFFFFFL
#define SPR_ERR_BADHANDLE        0x00000001L
#define SPR_ERR_BMPTOOLARGE      0x00000002L
#define SPR_ERR_RESOURCE         0x00000003L
#define SPR_ERR_NOMEMORY         0x00000004L
#define SPR_ERR_HASPLAYGROUND    0x00000005L
#define SPR_ERR_HASNOPLAYGROUND  0x00000006L
#define SPR_ERR_PLAYGROUNDFULL   0x00000007L
#define SPR_ERR_HASBACKGROUND    0x00000008L

typedef LHANDLE HSPRITE;
typedef HSPRITE *PHSPRITE;

typedef LHANDLE HPLAYGROUND;
typedef HPLAYGROUND *PHPLAYGROUND;

SPRERROR EXPENTRY SprCreatePlayground(HAB habAnchor,PHPLAYGROUND phpgPlay);
//-------------------------------------------------------------------------
// This function creates a playground to which sprites can be added.
//
// Input:  habAnchor - anchor block of the calling thread.
// Output:  phpgPlay - points to the variable with the HPLAYGROUND handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprDestroyPlayground(HPLAYGROUND hpgPlay);
//-------------------------------------------------------------------------
// This function destroys the playground including any sprites that are
// still members of it.  All resources consumed by the playground,
// including the back bitmap, are returned to the system.
//
// Input:  hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprAddSprite(HPLAYGROUND hpgPlay,HSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function labels a sprite as a "member" of the specified playground.
// Doing so allows the application to control the sprite's position,
// visibility, etc. on a drawing surface.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to add
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprRemoveSprite(HPLAYGROUND hpgPlay,HSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function removes the sprite from the membership list of the
// specified playground.  The sprite can then be added to another
// playground, or this one at a later time.
//
// Since there is a limited number of sprites that can be members of
// a playground, this function can be used to temporarily remove unused
// sprites from a playground so that others can be used.
//
// Input:  hpgPlay - handle to the playground
//         hsSprite - handle to the sprite to remove
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprSetPlaygroundBack(HPLAYGROUND hpgPlay,
                                       HBITMAP hbmNew,
                                       HBITMAP *phbmOld);
//-------------------------------------------------------------------------
// This function sets the background bitmap of the playground.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application unless the bitmap is "unset" from the playground (by
// calling this function again with a different handle).
//
// Input:  hpgPlay - handle to the playground
//         hbmNew - handle to the new bitmap to used as the background
// Output:  phbmOld - points to the handle to the old background bitmap.
//          This can be NULL, meaning that the application isn't interested
//          in receiving this value.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQueryPlaygroundBack(HPLAYGROUND hpgPlay,
                                         HBITMAP *phbmBack);
//-------------------------------------------------------------------------
// This function returns the handle of the background bitmap currently in
// use.
//
// Input:  hpgPlay - handle to the playground
// Output:  phbmBack - points to the handle to the background bitmap.
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprSetPlaygroundSize(HPLAYGROUND hpgPlay,PSIZEL pszlSize);
//-------------------------------------------------------------------------
// This function sets the playground size for playgrounds that do not have
// a bitmap set as the background.
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the size of the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQueryPlaygroundSize(HPLAYGROUND hpgPlay,PSIZEL pszlSize);
//-------------------------------------------------------------------------
// This function returns the size of the playground.  For playgrounds with
// bitmaps set as the background, the returned value is the size of the
// bitmap.  Otherwise, the returned value is that which was specified on
// the last call to SprSetPlaygroundSize().
//
// Input:  hpgPlay - handle to the playground
//         pszlSize - points to the variable to receive the size of the
//                    playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprSetBackColor(HPLAYGROUND hpgPlay,LONG lBackColor);
//-------------------------------------------------------------------------
// This function sets the new background color of the playground and is
// only valid if the playground doesn't have a bitmap.
//
// Input:  hpgPlay - handle to the playground
//         lBackColor - specifies the new background color
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQueryBackColor(HPLAYGROUND hpgPlay,PLONG plBackColor);
//-------------------------------------------------------------------------
// This function redraws the playground and all sprites belonging to the
// playground.
//
// Input:  hpsDraw - handle to the HPS to draw the playground in
//         hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprSetUpdateFlag(HPLAYGROUND hpgPlay,BOOL bUpdate);
//-------------------------------------------------------------------------
// This function sets the update flag for the playground.  If FALSE, no
// drawing actually takes place in any of the functions requiring an HPS,
// and the value of the HPS handle may be NULLHANDLE.  If TRUE, updating
// is reenabled, but you should still call SprDrawPlayground() to refresh
// the screen with the current contents.
//
// Input:  hpgPlay - handle to the playground
//         bUpdate - specifies the new update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQueryUpdateFlag(HPLAYGROUND hpgPlay,PBOOL pbUpdate);
//-------------------------------------------------------------------------
// This function returns the setting of the update flag.  See the notes
// for SprSetUpdateFlag() for more information about this setting.
//
// Input:  hpgPlay - handle to the playground
//         pbUpdate - points to the variable to receive the update flag
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprDrawPlayground(HPS hpsDraw,HPLAYGROUND hpgPlay);
//-------------------------------------------------------------------------
// This function redraws the playground and all sprites belonging to the
// playground.
//
// Input:  hpsDraw - handle to the HPS to draw the playground in
//         hpgPlay - handle to the playground
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprCreateSprite(HAB habAnchor,
                                  HBITMAP hbmBitmap,
                                  PHSPRITE phsSprite);
//-------------------------------------------------------------------------
// This function creates a sprite from the specified bitmap.  The sprite
// cannot be moved, shown, etc., however, until it is associated with a
// playground.
//
// The color black is used as the transparency color.  If you need to use
// black in the bitmap without it becoming transparent, use the next
// closest color.  <grin>
//
// New sprites are initialized as being at position (0,0) and hidden.
//
// Note that, once this function is called, the bitmap is managed by
// the sprite subsystem.  The bitmap should *NOT* be deleted by the
// application or else unpredictable results will occur.
//
// Input:  habAnchor - anchor block of the calling thread.
//         hbmBitmap - handle to the bitmap
// Output:  phsSprite - points to the sprite handle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprDestroySprite(HSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function destroys the sprite and returns all resources to the
// system.
//
// Input:  hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprSetSpritePosition(HPS hpsDraw,
                                       HSPRITE hsSprite,
                                       PPOINTL pptlNew);
//-------------------------------------------------------------------------
// This function changes the position of the sprite.  This function is
// optimized so that, if the rectangle bounding the sprite at the new
// position overlaps the old, only one "bit blit" to the specified HPS
// is done, eliminating flicker.
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in once it is
//                   moved
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQuerySpritePosition(HSPRITE hsSprite,PPOINTL pptlPos);
//-------------------------------------------------------------------------
// This function returns the current position of the sprite.  Note that
// a sprite has a current position even if it is hidden.
//
// Input:  hsSprite - handle to the sprite
// Output:  pptlPos - points to the current position
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQuerySpriteSize(HSPRITE hsSprite,PSIZEL pszlSize);
//-------------------------------------------------------------------------
// This function returns the current size of the sprite.
//
// Input:  hsSprite - handle to the sprite
// Output:  pszlSize - points to the current size
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQuerySpriteRect(HSPRITE hsSprite,PRECTL prclRect);
//-------------------------------------------------------------------------
// This function returns the bounding rectangle of the sprite at its
// current position.
//
// Input:  hsSprite - handle to the sprite
// Output:  prclRect - points to the current bounding rectangle
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprSetSpriteVisibility(HPS hpsDraw,
                                         HSPRITE hsSprite,
                                         BOOL bVisible);
//-------------------------------------------------------------------------
// This function shows or hides a sprite.
//
// Input:  hpsDraw - handle to the HPS to draw in once the sprite is
//                   shown or hidden
//         hsSprite - handle to the sprite
//         bVisible - new visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprQuerySpriteVisibility(HSPRITE hsSprite,PBOOL pbVisible);
//-------------------------------------------------------------------------
// This function returns the visibility state of the sprite
//
// Input:  hsSprite - handle to the sprite
// Output:  pbVisible - points to the visibility state
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

SPRERROR EXPENTRY SprDrawSprite(HPS hpsDraw,HSPRITE hsSprite);
//-------------------------------------------------------------------------
// This function draws a sprite
//
// Input:  hpsDraw - handle to the HPS to draw the sprite in
//         hsSprite - handle to the sprite
// Returns:  SPR_ERR_NOERROR if successful, SPR_ERR_* constant otherwise
//-------------------------------------------------------------------------

#ifdef __cplusplus
}
#endif

#pragma pack()

#endif                           // #ifndef SPRITE_INCLUDED
