/******************************************************************************
*******************************************************************************
**
** EXR0R3 - A ring 0/ring 3 IFS skeleton
** Copyright (C) 1993 by Andre Asselin
**
** R0STUBS.C - Ring 0 stubs
**
** History:
** 5/25/93 - created
** 11/26/93 - all stubs filled in
**
*******************************************************************************
******************************************************************************/

#include "r0inc.h"

#if defined(__cplusplus)
extern "C" {
#endif

/******************************************************************************
** Name of IFS
******************************************************************************/

char pascal FS_NAME[] = "EXR0R3";


/******************************************************************************
** Attributes of IFS
******************************************************************************/

#define SUP_REMOTE
//#define SUP_UNC
//#define SUP_LOCK
//#define SUP_LVL7
//#define SUP_FILEIO
//#define SUP_PAGEIO

unsigned long int pascal FS_ATTRIBUTE = 0
#if defined(SUP_REMOTE)
                        | FSA_REMOTE   /* If file system is remote          */
#endif
#if defined(SUP_UNC)
                        | FSA_UNC      /* If IFS supports UNC               */
#endif
#if defined(SUP_LOCK)
                        | FSA_LOCK     /* If IFS supports file locking      */
#endif
#if defined(SUP_LVL7)
                        | FSA_LVL7     /* If IFS supports QPathInfo level 7 */
#endif
                        ;


#if defined(__cplusplus)
}
#endif


/******************************************************************************
*******************************************************************************
**
** Primary entry points
**
*******************************************************************************
******************************************************************************/





/*-----------------------------------------------------------------------------
--
-- Volume management
--
-----------------------------------------------------------------------------*/


/******************************************************************************
**
** FS_ATTACH - Attach or detach a drive or device
**
** Parameters
** ----------
** unsigned short flag                  indicates attaching/detaching
**   values:
**     FSA_ATTACH               attach drive/device
**     FSA_DETACH               detach drive/device
**     FSA_ATTACH_INFO          return info on attached drive/device
** char far *pDev                       drive or device that is being attached/detached
** struct vpfsd far *pvpfsd             pointer to FSD dependant volume parameters
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** void far *pParm                      UNVERIFIED pointer to FSD dependant attachment info
** unsigned short far *pLen             length of area pointed to by pParm
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_ATTACH(unsigned short flag, 
                                               char far *pDev,
                                               struct vpfsd far *pvpfsd,
                                               struct cdfsd far *pcdfsd,
                                               void far *pParm,
                                               unsigned short far *pLen) {
   unsigned short rc;

   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if ((flag == FSA_ATTACH || flag == FSA_DETACH) && *pLen != 0) {
      rc = FSH_PROBEBUF(PB_OPREAD, pParm, *pLen);
   } else if (*pLen != 0) {
      rc = FSH_PROBEBUF(PB_OPWRITE, pParm, *pLen);
   }
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_ATTACH;
   CPData.OpData->attach.flag = flag;
   strncpy(CPData.OpData->attach.Dev, pDev, sizeof(CPData.OpData->attach.Dev));
   CPData.OpData->attach.vpfsd = *pvpfsd;
   CPData.OpData->attach.cdfsd = *pcdfsd;
   CPData.OpData->attach.Len = *pLen;
   if (flag != FSA_ATTACH_INFO)
      memcpy(CPData.Buf, pParm, *pLen);

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pvpfsd = CPData.OpData->attach.vpfsd;
   *pcdfsd = CPData.OpData->attach.cdfsd;
   *pLen = CPData.OpData->attach.Len;
   if (flag == FSA_ATTACH_INFO)
      memcpy(pParm, CPData.Buf, *pLen);

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_MOUNT - Examine a volume to determine if the IFS knows its format
**
** Parameters
** ----------
** unsigned short flag                  indicates function to perform
**   values:
**     MOUNT_MOUNT              mount or accept the volume
**     MOUNT_VOL_REMOVED        volume has been removed
**     MOUNT_RELEASE            release all resources associated with the volume
**     MOUNT_ACCEPT             accept the volume in preparation for formatting
** struct vpfsi far *pvpfsi             pointer to FSD independant volume parameters
** struct vpfsd far *pvpfsd             pointer to FSD dependant volume parameters
** unsigned short hVPB                  volume handle
** char far *pBoot                      pointer to sector 0 data
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_MOUNT(unsigned short flag, struct vpfsi far *pvpfsi,
                              struct vpfsd far *pvpfsd, unsigned short hVPB, char far *pBoot)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FSINFO - Get/Set file system information
**
** Parameters
** ----------
** unsigned short flag                  indicates function to perform
**   values:
**     INFO_RETREIVE                    retrieve information
**     INFO_SET                         set information
** unsigned short hVPB                  volume handle
** char far *pData                      UNVERIFIED pointer to data buffer
** unsigned short cbData                length of data buffer
** unsigned short level                 type of information to return
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FSINFO(unsigned short flag, unsigned short hVPB,
                               char far *pData, unsigned short cbData,
                               unsigned short level)
{
   unsigned short rc;
   struct vpfsi *pvpfsi;
   struct vpfsd *pvpfsd;

   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (flag == INFO_SET && cbData != 0) {
      rc = FSH_PROBEBUF(PB_OPREAD, pData, cbData);
   } else if (cbData != 0) {
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData);
   }
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FSINFO;
   CPData.OpData->fsinfo.flag = flag;
   FSH_GETVOLPARM(hVPB, &pvpfsi, &pvpfsd);
   CPData.OpData->flushbuf.vpfsd = *pvpfsd;
   memcpy(CPData.Buf, pData, cbData);
   CPData.OpData->fsinfo.cbData = cbData;
   CPData.OpData->fsinfo.level = level;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   if (flag == INFO_RETREIVE)
      memcpy(pData, CPData.Buf, cbData);

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FLUSHBUF - Flush buffers for a specified volume
**
** Parameters
** ----------
** unsigned short hVPB                  handle to volume to flush
** unsigned short flag                  indicates whether to discard or retain cache
**   values:
**     FLUSH_RETAIN     retain cached information
**     FLUSH_DISCARD    discard cached information
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FLUSHBUF(unsigned short hVPB, unsigned short flag)
{
   unsigned short rc;
   struct vpfsi *pvpfsi;
   struct vpfsd *pvpfsd;

   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FLUSHBUF;
   FSH_GETVOLPARM(hVPB, &pvpfsi, &pvpfsd);
   CPData.OpData->flushbuf.vpfsd = *pvpfsd;
   CPData.OpData->flushbuf.flag = flag;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}



/*-----------------------------------------------------------------------------
--
-- Directory management
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_CHDIR - Change current directory
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     CD_EXPLICIT      creating a new current directory
**     CD_VERIFY        verifying a current directory
**     CD_FREE          freeing an instance of a current directory
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pDir                       pointer to directory to change to
** unsigned short iCurDirEnd            offset to the end of the current directory in pDir
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_CHDIR(unsigned short flag, struct cdfsi far *pcdfsi,
                              struct cdfsd far *pcdfsd, char far *pDir,
                              unsigned short iCurDirEnd)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_CHDIR;
   CPData.OpData->chdir.flag = flag;
   if (flag != CD_FREE)
      CPData.OpData->chdir.cdfsi = *pcdfsi;
   CPData.OpData->chdir.cdfsd = *pcdfsd;
   if (flag == CD_EXPLICIT)
      strncpy(CPData.OpData->chdir.Dir, pDir, sizeof(CPData.OpData->chdir.Dir));
   CPData.OpData->chdir.iCurDirEnd = iCurDirEnd;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->chdir.cdfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_MKDIR - Make a new directory
**
** Parameters
** ----------
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pName                      pointer to directory name to create
** unsigned short iCurDirEnd            offset to the end of the current directory
**                                      in pName
** char *pEABuf                         UNVERIFIED pointer to EAs to attach
**                                      to new directory
** unsigned short flags                 0x40 = directory is non 8.3 filename
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_MKDIR(struct cdfsi far *pcdfsi, struct cdfsd far *pcdfsd,
                              char far *pName, unsigned short iCurDirEnd,
                              char far *pEABuf, unsigned short flags)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (pEABuf != NULL) {
      rc = FSH_PROBEBUF(PB_OPREAD, pEABuf, sizeof(EAOP));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pEABuf)->fpFEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pEABuf)->fpFEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pEABuf)->fpFEAList,
                        ((EAOP *)pEABuf)->fpFEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_MKDIR;
   CPData.OpData->mkdir.cdfsi = *pcdfsi;
   CPData.OpData->mkdir.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->mkdir.Name, pName, sizeof(CPData.OpData->mkdir.Name));
   CPData.OpData->mkdir.iCurDirEnd = iCurDirEnd;
   if (pEABuf != NULL)
      memcpy(CPData.Buf, ((EAOP *)pEABuf)->fpFEAList,
             ((EAOP *)pEABuf)->fpFEAList->cbList);
   CPData.OpData->mkdir.flags = flags;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->mkdir.cdfsd;
   if (pEABuf != NULL)
      ((EAOP *)pEABuf)->oError = CPData.OpData->mkdir.oError;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_RMDIR - Delete directory
**
** Parameters
** ----------
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pName                      pointer to directory name to delete
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_RMDIR(struct cdfsi far *pcdfsi, struct cdfsd far *pcdfsd,
                              char far *pName, unsigned short iCurDirEnd)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_RMDIR;
   CPData.OpData->rmdir.cdfsi = *pcdfsi;
   CPData.OpData->rmdir.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->rmdir.Name, pName, sizeof(CPData.OpData->rmdir.Name));
   CPData.OpData->rmdir.iCurDirEnd = iCurDirEnd;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->rmdir.cdfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}



/*-----------------------------------------------------------------------------
--
-- File management
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_CHGFILEPTR - Change current location in file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** long offset                          signed offset
** unsigned short type                  indicates seek type
**   values:
**     CFP_RELBEGIN     move pointer relative to begining of file
**     CFP_RELCUR       move pointer relative to current position in file
**     CFP_RELEND       move pointer relative to end of file
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_CHGFILEPTR(struct sffsi far *psffsi, struct sffsd far *psffsd,
                                   long offset, unsigned short type, unsigned short IOflag)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_CHGFILEPTR;
   CPData.OpData->chgfileptr.sffsi = *psffsi;
   CPData.OpData->chgfileptr.sffsd = *psffsd;
   CPData.OpData->chgfileptr.offset = offset;
   CPData.OpData->chgfileptr.type = type;
   CPData.OpData->chgfileptr.IOflag = IOflag;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->chgfileptr.sffsi;
   *psffsd = CPData.OpData->chgfileptr.sffsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_CLOSE - Close an open file
**
** Parameters
** ----------
** unsigned short type                  indicates close type
**   values:
**     FS_CL_ORDINARY   this is not the final close of the file
**     FS_CL_FORPROC    this is the final close of the file for the process
**     FS_CL_FORSYS     this is the final close of the file for the whole system
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_CLOSE(unsigned short type, unsigned short IOflag,
                              struct sffsi far *psffsi, struct sffsd far *psffsd)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_CLOSE;
   CPData.OpData->close.type = type;
   CPData.OpData->close.IOflag = IOflag;
   CPData.OpData->close.sffsi = *psffsi;
   CPData.OpData->close.sffsd = *psffsd;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->close.sffsi;
   *psffsd = CPData.OpData->close.sffsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_COMMIT - Commit a file to disk
**
** Parameters
** ----------
** unsigned short type                  indicates commit type
**   values:
**     FS_COMMIT_ONE    commit this one file
**     FS_COMMIT_ALL    commit all files
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_COMMIT(unsigned short type, unsigned short IOflag,
                               struct sffsi far *psffsi, struct sffsd far *psffsd)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_COMMIT;
   CPData.OpData->commit.type = type;
   CPData.OpData->commit.IOflag = IOflag;
   CPData.OpData->commit.sffsi = *psffsi;
   CPData.OpData->commit.sffsd = *psffsd;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->commit.sffsi;
   *psffsd = CPData.OpData->commit.sffsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_COPY - Copy a file
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     DCPY_EXISTING    if destination file exists, replace it
**     DCPY_APPEND      source file should be appended to destination file
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pSrc                       pointer to source filename
** unsigned short iSrcCurDirEnd         offset to the end of the current directory in pSrc
** char far *pDst                       pointer to destination filename
** unsigned short iDstCurDirEnd         offset to the end of the current directory in pDst
** unsigned short nameType              0x40 = destination is non 8.3 filename
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_COPY(unsigned short flag, struct cdfsi far *pcdfsi,
                             struct cdfsd far *pcdfsd, char far *pSrc,
                             unsigned short iSrcCurDirEnd, char far *pDst,
                             unsigned short iDstCurDirEnd, unsigned short nameType)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_COPY;
   CPData.OpData->copy.flag = flag;
   CPData.OpData->copy.cdfsi = *pcdfsi;
   CPData.OpData->copy.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->copy.Src, pSrc, sizeof(CPData.OpData->copy.Src));
   CPData.OpData->copy.iSrcCurDirEnd = iSrcCurDirEnd;
   strncpy(CPData.OpData->copy.Dst, pDst, sizeof(CPData.OpData->copy.Dst));
   CPData.OpData->copy.iDstCurDirEnd = iDstCurDirEnd;
   CPData.OpData->copy.nameType = nameType;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->copy.cdfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_DELETE - Delete a file
**
** Parameters
** ----------
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pFile                      pointer to filename to delete
** unsigned short iCurDirEnd            offset to the end of the current directory in pFile
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_DELETE(struct cdfsi far *pcdfsi, struct cdfsd far *pcdfsd,
                               char far *pFile, unsigned short iCurDirEnd)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_DELETE;
   CPData.OpData->fsdelete.cdfsi = *pcdfsi;
   CPData.OpData->fsdelete.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->fsdelete.File, pFile, sizeof(CPData.OpData->fsdelete.File));
   CPData.OpData->fsdelete.iCurDirEnd = iCurDirEnd;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->fsdelete.cdfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FILEATTRIBUTE - Get/Set DOS file attributes
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     FA_RETRIEVE      retrieve attribute
**     FA_SET           set attribute
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pName                      pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short far *pAttr            pointer to the attribute
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FILEATTRIBUTE(unsigned short flag, struct cdfsi far *pcdfsi,
                                      struct cdfsd far *pcdfsd, char far *pName,
                                      unsigned short iCurDirEnd, unsigned short far *pAttr)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FILEATTRIBUTE;
   CPData.OpData->fileattribute.flag = flag;
   CPData.OpData->fileattribute.cdfsi = *pcdfsi;
   CPData.OpData->fileattribute.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->fileattribute.Name, pName, sizeof(CPData.OpData->fileattribute.Name));
   CPData.OpData->fileattribute.iCurDirEnd = iCurDirEnd;
   CPData.OpData->fileattribute.Attr = *pAttr;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   if (flag == FA_RETRIEVE)
      *pAttr = CPData.OpData->fileattribute.Attr;
   *pcdfsd = CPData.OpData->fileattribute.cdfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FILEINFO - Get/Set file information
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     FI_RETRIEVE      retrieve information
**     FI_SET           set information
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** unsigned short level                 level of information to get/set
** char far *pData                      UNVERIFIED? pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FILEINFO(unsigned short flag, struct sffsi far *psffsi,
                                 struct sffsd far *psffsd, unsigned short level,
                                 char far *pData, unsigned short cbData,
                                 unsigned short IOflag)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (flag == FI_SET && cbData != 0) {
      rc = FSH_PROBEBUF(PB_OPREAD, pData, cbData);
   } else if (cbData != 0) {
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData);
   }
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }

   // Special EA check
   if (flag == FI_RETRIEVE && level == 3) {
      // Check to make sure the buffer's at least big enough for an EAOP
      if (cbData < sizeof(EAOP)) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check accessability of GEA

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpGEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, 
                        ((EAOP *)pData)->fpGEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Same thing for FEA

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPWRITE, ((EAOP *)pData)->fpFEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpFEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPWRITE, ((EAOP *)pData)->fpFEAList,
                        ((EAOP *)pData)->fpFEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }

   if (flag == FI_SET && level == 2) {
      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpFEAList, sizeof(ULONG));
      if (rc != NO_ERROR)
         return rc;

      // Check that it is <=64K
      if (((EAOP *)pData)->fpFEAList->cbList > 0xffff)
         return ERROR_INVALID_PARAMETER;

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpFEAList,
                        ((EAOP *)pData)->fpFEAList->cbList);
      if (rc != NO_ERROR)
         return rc;
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FILEINFO;
   CPData.OpData->fileinfo.flag = flag;
   CPData.OpData->fileinfo.sffsi = *psffsi;
   CPData.OpData->fileinfo.sffsd = *psffsd;
   CPData.OpData->fileinfo.level = level;
   // Copy either the buffer or the GEA list as appropriate
   if (flag == FI_RETRIEVE && level == 3) {
      memcpy(CPData.Buf, ((EAOP *)pData)->fpGEAList,
                        ((EAOP *)pData)->fpGEAList->cbList);
   } else if (flag == FI_SET && level == 2) {
      memcpy(CPData.Buf, ((EAOP *)pData)->fpFEAList,
                        ((EAOP *)pData)->fpFEAList->cbList);
   } else {
      memcpy(CPData.Buf, pData, cbData);
   }
   CPData.OpData->fileinfo.cbData = cbData;
   CPData.OpData->fileinfo.IOflag = IOflag;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   if (flag == FI_RETRIEVE) {
      // If level 3, copy results starting at the oError entry in the EAOP
      if (level != 3) {
         memcpy(pData, CPData.Buf, cbData);
      } else {
         memcpy(((EAOP *)pData)->fpFEAList, CPData.Buf,
                ((FEALIST *)&CPData.Buf)->cbList);
      }
   }
   if (flag == FI_RETRIEVE && level == 3 ||
       flag == FI_SET && level == 2)
      ((EAOP *)pData)->oError = CPData.OpData->fileinfo.oError;
   *psffsi = CPData.OpData->fileinfo.sffsi;
   *psffsd = CPData.OpData->fileinfo.sffsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FILEIO - Atomic I/O operations
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** char far *pCmdList                   UNVERIFIED pointer to information area
** unsigned short cbCmdList             size of area pointed to by pCmdList
** unsigned short far *poError          UNVERIFIED pointer to offset within
**                                      pCmdList of command that caused an error
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FILEIO(struct sffsi far *psffsi, struct sffsd far *psffsd,
                               char far *pCmdList, unsigned short cbCmdList,
                               unsigned short far *poError, unsigned short IOflag)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_MOVE - Move/rename a file
**
** Parameters
** ----------
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pSrc                       pointer to source filename
** unsigned short iSrcCurDirEnd         offset to the end of the current directory in pSrc
** char far *pDst                       pointer to destination filename
** unsigned short iDstCurDirEnd         offset to the end of the current directory in pDst
** unsigned short flags                 0x40 = destination is non 8.3 filename
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_MOVE(struct cdfsi far *pcdfsi, struct cdfsd far *pcdfsd,
                             char far *pSrc, unsigned short iSrcCurDirEnd,
                             char far *pDst, unsigned short iDstCurDirEnd,
                             unsigned short flags)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_MOVE;
   CPData.OpData->move.cdfsi = *pcdfsi;
   CPData.OpData->move.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->move.Src, pSrc, sizeof(CPData.OpData->move.Src));
   CPData.OpData->move.iSrcCurDirEnd = iSrcCurDirEnd;
   strncpy(CPData.OpData->move.Dst, pDst, sizeof(CPData.OpData->move.Dst));
   CPData.OpData->move.iDstCurDirEnd = iDstCurDirEnd;
   CPData.OpData->move.flags = flags;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->move.cdfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_NEWSIZE - Change size of file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** unsigned long len                    new length of file
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_NEWSIZE(struct sffsi far *psffsi, struct sffsd far *psffsd,
                                unsigned long len, unsigned short IOflag)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_NEWSIZE;
   CPData.OpData->newsize.sffsi = *psffsi;
   CPData.OpData->newsize.sffsd = *psffsd;
   CPData.OpData->newsize.len = len;
   CPData.OpData->newsize.IOflag = IOflag;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->newsize.sffsi;
   *psffsd = CPData.OpData->newsize.sffsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_OPENCREATE - Open or create a new file
**
** Parameters
** ----------
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pName                      pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** unsigned long ulOpenMode             sharing and access mode
** unsigned short usOpenFlag            action to take when file exists/doesn't exist
** unsigned short far *pusAction        returns the action that the IFS took
** unsigned short usAttr                OS/2 file attributes
** char far *pEABuf                     UNVERIFIED pointer to EAs to attach to new file
** unsigned short far *pfgenflag        flags returned by the IFS
**   values:
**     FOC_NEEDEAS      indicates there are critical EAs associated with the file
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_OPENCREATE(struct cdfsi far *pcdfsi, struct cdfsd far *pcdfsd,
                                   char far *pName, unsigned short iCurDirEnd,
                                   struct sffsi far *psffsi, struct sffsd far *psffsd,
                                   unsigned long ulOpenMode, unsigned short usOpenFlag,
                                   unsigned short far *pusAction, unsigned short usAttr,
                                   char far *pEABuf, unsigned short far *pfgenflag)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (pEABuf != NULL) {
      rc = FSH_PROBEBUF(PB_OPREAD, pEABuf, sizeof(EAOP));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pEABuf)->fpFEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pEABuf)->fpFEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pEABuf)->fpFEAList,
                        ((EAOP *)pEABuf)->fpFEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_OPENCREATE;
   CPData.OpData->opencreate.cdfsi = *pcdfsi;
   CPData.OpData->opencreate.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->opencreate.Name, pName, sizeof(CPData.OpData->opencreate.Name));
   CPData.OpData->opencreate.iCurDirEnd = iCurDirEnd;
   CPData.OpData->opencreate.sffsi = *psffsi;
   CPData.OpData->opencreate.sffsd = *psffsd;
   CPData.OpData->opencreate.ulOpenMode = ulOpenMode;
   CPData.OpData->opencreate.usOpenFlag = usOpenFlag;
   CPData.OpData->opencreate.usAttr = usAttr;
   if (pEABuf != NULL)
      memcpy(CPData.Buf, ((EAOP *)pEABuf)->fpFEAList,
             ((EAOP *)pEABuf)->fpFEAList->cbList);

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->opencreate.cdfsd;
   *psffsi = CPData.OpData->opencreate.sffsi;
   *psffsd = CPData.OpData->opencreate.sffsd;
   *pusAction = CPData.OpData->opencreate.usAction;
   *pfgenflag = CPData.OpData->opencreate.fgenflag;
   if (pEABuf != NULL)
      ((EAOP *)pEABuf)->oError = CPData.OpData->opencreate.oError;
 

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_PATHINFO - get/set file information by filename
**
** Parameters
** ----------
** unsigned short flag                  indicates flavor of call
**   values:
**     PI_RETRIEVE      retrieve information
**     PI_SET           set information
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pName                      pointer to filename
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short level                 level of information to get/set
** char far *pData                      UNVERIFIED pointer to information area
** unsigned short cbData                size of area pointed to by pData
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_PATHINFO(unsigned short flag, struct cdfsi far *pcdfsi,
                                 struct cdfsd far *pcdfsd, char far *pName,
                                 unsigned short iCurDirEnd, unsigned short level,
                                 char far *pData, unsigned short cbData)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (flag == PI_RETRIEVE && cbData != 0) {
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData);
   } else if (cbData != 0) {
      rc = FSH_PROBEBUF(PB_OPREAD, pData, cbData);
   }
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }

   // Special EA check
   if (flag == PI_RETRIEVE && level == 3) {
      // Check to make sure the buffer's at least big enough for an EAOP
      if (cbData < sizeof(EAOP)) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check accessability of GEA

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpGEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, 
                        ((EAOP *)pData)->fpGEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Same thing for FEA

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPWRITE, ((EAOP *)pData)->fpFEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpFEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPWRITE, ((EAOP *)pData)->fpFEAList,
                        ((EAOP *)pData)->fpFEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }

   if (flag == PI_SET && level == 2) {
      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpFEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpFEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpFEAList,
                        ((EAOP *)pData)->fpFEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_PATHINFO;
   CPData.OpData->pathinfo.flag = flag;
   CPData.OpData->pathinfo.cdfsi = *pcdfsi;
   CPData.OpData->pathinfo.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->pathinfo.Name, pName, sizeof(CPData.OpData->pathinfo.Name));
   CPData.OpData->pathinfo.iCurDirEnd = iCurDirEnd;
   CPData.OpData->pathinfo.level = level;
   // Copy either the buffer or the GEA list as appropriate
   if (flag == PI_RETRIEVE && level == 3) {
      memcpy(CPData.Buf, ((EAOP *)pData)->fpGEAList,
                        ((EAOP *)pData)->fpGEAList->cbList);
   } else if (flag == PI_SET && level == 2) {
      memcpy(CPData.Buf, ((EAOP *)pData)->fpFEAList,
                        ((EAOP *)pData)->fpFEAList->cbList);
   } else {
      memcpy(CPData.Buf, pData, cbData);
   }
   CPData.OpData->pathinfo.cbData = cbData;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   if (flag == PI_RETRIEVE) {
      // If level 3, copy results starting at the oError entry in the EAOP
      if (level != 3) {
         memcpy(pData, CPData.Buf, cbData);
      } else {
         memcpy(((EAOP *)pData)->fpFEAList, CPData.Buf,
                ((FEALIST *)&CPData.Buf)->cbList);
      }
   }
   if (flag == PI_RETRIEVE && level == 3 ||
       flag == PI_SET && level == 2)
      ((EAOP *)pData)->oError = CPData.OpData->pathinfo.oError;
   *pcdfsd = CPData.OpData->pathinfo.cdfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_READ - read data from a file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** char far *pData                      UNVERIFIED pointer to buffer
** unsigned short far *pLen             length of buffer
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_READ(struct sffsi far *psffsi, struct sffsd far *psffsd,
                             char far *pData, unsigned short far *pLen,
                             unsigned short IOflag)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (*pLen != 0) 
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, *pLen);
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_READ;
   CPData.OpData->read.sffsi = *psffsi;
   CPData.OpData->read.sffsd = *psffsd;
   CPData.OpData->read.Len = *pLen;
   CPData.OpData->read.IOflag = IOflag;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->read.sffsi;
   *psffsd = CPData.OpData->read.sffsd;
   *pLen = CPData.OpData->read.Len;
   memcpy(pData, CPData.Buf, *pLen);

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_WRITE - write data to a file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** char far *pData                      UNVERIFIED pointer to buffer
** unsigned short far *pLen             length of buffer
** unsigned short IOflag                bitfield of I/O suggestions
**   values:
**     IOFL_WRITETHRU   write all updated data before returning
**     IOFL_NOCACHE     don't cache any new data
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_WRITE(struct sffsi far *psffsi, struct sffsd far *psffsd,
                              char far *pData, unsigned short far *pLen,
                              unsigned short IOflag)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (*pLen != 0)
      rc = FSH_PROBEBUF(PB_OPREAD, pData, *pLen);
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_WRITE;
   CPData.OpData->write.sffsi = *psffsi;
   CPData.OpData->write.sffsd = *psffsd;
   memcpy(CPData.Buf, pData, *pLen);
   CPData.OpData->write.Len = *pLen;
   CPData.OpData->write.IOflag = IOflag;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->write.sffsi;
   *psffsd = CPData.OpData->write.sffsd;
   *pLen = CPData.OpData->write.Len;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}



/*-----------------------------------------------------------------------------
--
-- Directory management
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_FINDCLOSE - End a directory search
**
** Parameters
** ----------
** struct fsfsi far *pfsfsi             pointer to FSD independant search record
** struct fsfsd far *pfsfsd             pointer to FSD dependant search record
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FINDCLOSE(struct fsfsi far *pfsfsi, struct fsfsd far *pfsfsd)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FINDCLOSE;
   CPData.OpData->findclose.fsfsi = *pfsfsi;
   CPData.OpData->findclose.fsfsd = *pfsfsd;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pfsfsd = CPData.OpData->findclose.fsfsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FINDFIRST - Begin a new directory search
**
** Parameters
** ----------
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pName                      pointer to filename mask
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short attr                  attribute mask
** struct fsfsi far *pfsfsi             pointer to FSD independant search record
** struct fsfsd far *pfsfsd             pointer to FSD dependant search record
** char far *pData                      UNVERIFIED pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short far *pcMatch          maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned short flags                 indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FINDFIRST(struct cdfsi far *pcdfsi, struct cdfsd far *pcdfsd,
                                  char far *pName, unsigned short iCurDirEnd,
                                  unsigned short attr, struct fsfsi far *pfsfsi,
                                  struct fsfsd far *pfsfsd, char far *pData,
                                  unsigned short cbData, unsigned short far *pcMatch,
                                  unsigned short level, unsigned short flags)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (cbData != 0)
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData);
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }

   // Special EA check
   if (level == 3) {
      // Check to make sure the buffer's at least big enough for an EAOP
      if (cbData < sizeof(EAOP)) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpGEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, 
                        ((EAOP *)pData)->fpGEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FINDFIRST;
   CPData.OpData->findfirst.cdfsi = *pcdfsi;
   CPData.OpData->findfirst.cdfsd = *pcdfsd;
   strncpy(CPData.OpData->findfirst.Name, pName, sizeof(CPData.OpData->findfirst.Name));
   CPData.OpData->findfirst.iCurDirEnd = iCurDirEnd;
   CPData.OpData->findfirst.attr = attr;
   CPData.OpData->findfirst.fsfsi = *pfsfsi;
   CPData.OpData->findfirst.fsfsd = *pfsfsd;
   // Copy either the buffer or the GEA list as appropriate
   if (level != 3)
      memcpy(CPData.Buf, pData, cbData);
   else
      memcpy(CPData.Buf, ((EAOP *)pData)->fpGEAList,
                        ((EAOP *)pData)->fpGEAList->cbList);
   CPData.OpData->findfirst.cbData = cbData;
   CPData.OpData->findfirst.cMatch = *pcMatch;
   CPData.OpData->findfirst.level = level;
   CPData.OpData->findfirst.flags = flags;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pcdfsd = CPData.OpData->findfirst.cdfsd;
   *pfsfsd = CPData.OpData->findfirst.fsfsd;
   // If level 3, copy results starting at the oError entry in the EAOP
   if (level != 3) {
      memcpy(pData, CPData.Buf, cbData);
   } else {
      memcpy(pData + sizeof(EAOP), CPData.Buf, cbData - sizeof(EAOP));
      ((EAOP *)pData)->oError = CPData.OpData->findfirst.oError;
   }
   *pcMatch = CPData.OpData->findfirst.cMatch;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FINDFROMNAME - Restart directory search
**
** Parameters
** ----------
** struct fsfsi far *pfsfsi             pointer to FSD independant search record
** struct fsfsd far *pfsfsd             pointer to FSD dependant search record
** char far *pData                      UNVERIFIED pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short far *pcMatch          maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned long position               position in directory to restart search from
** char far *pName                      pointer to filename to restart search from
** unsigned short flags                 indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FINDFROMNAME(struct fsfsi far *pfsfsi, struct fsfsd far *pfsfsd,
                                     char far *pData, unsigned short cbData,
                                     unsigned short far *pcMatch, unsigned short level,
                                     unsigned long position, char far *pName,
                                     unsigned short flags)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (cbData != 0)
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData);
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }

   // Special EA check
   if (level == 3) {
      // Check to make sure the buffer's at least big enough for an EAOP
      if (cbData < sizeof(EAOP)) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpGEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, 
                        ((EAOP *)pData)->fpGEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FINDFROMNAME;
   CPData.OpData->findfromname.fsfsi = *pfsfsi;
   CPData.OpData->findfromname.fsfsd = *pfsfsd;
   if (level != 3)
      memcpy(CPData.Buf, pData, cbData);
   else
      memcpy(CPData.Buf, ((EAOP *)pData)->fpGEAList,
                        ((EAOP *)pData)->fpGEAList->cbList);
   CPData.OpData->findfromname.cbData = cbData;
   CPData.OpData->findfromname.cMatch = *pcMatch;
   CPData.OpData->findfromname.level = level;
   CPData.OpData->findfromname.position = position;
   strncpy(CPData.OpData->findfromname.Name, pName, sizeof(CPData.OpData->findfromname.Name));
   CPData.OpData->findfromname.flags = flags;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pfsfsd = CPData.OpData->findfromname.fsfsd;
   // If level 3, copy results starting at the oError entry in the EAOP
   if (level != 3) {
      memcpy(pData, CPData.Buf, cbData);
   } else {
      memcpy(pData + sizeof(EAOP), CPData.Buf, cbData - sizeof(EAOP));
      ((EAOP *)pData)->oError = CPData.OpData->findfirst.oError;
   }
   *pcMatch = CPData.OpData->findfromname.cMatch;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FINDNEXT - Continue directory search
**
** Parameters
** ----------
** struct fsfsi far *pfsfsi             pointer to FSD independant search record
** struct fsfsd far *pfsfsd             pointer to FSD dependant search record
** char far *pData                      UNVERIFIED pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short far *pcMatch          maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned short flags                 indicates whether to return position information
**   values:
**     FF_NOPOS         don't return any position information
**     FF_GETPOS        return position information in buffer
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FINDNEXT(struct fsfsi far *pfsfsi, struct fsfsd far *pfsfsd,
                                 char far *pData, unsigned short cbData,
                                 unsigned short far *pcMatch, unsigned short level,
                                 unsigned short flags)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters
   if (cbData != 0)
      rc = FSH_PROBEBUF(PB_OPWRITE, pData, cbData);
   if (rc != NO_ERROR) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return rc;
   }

   // Special EA check
   if (level == 3) {
      // Check to make sure the buffer's at least big enough for an EAOP
      if (cbData < sizeof(EAOP)) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that at least the first ULONG is accessible
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, sizeof(ULONG));
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }

      // Check that it is <=64K
      if (((EAOP *)pData)->fpGEAList->cbList > 0xffff) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return ERROR_INVALID_PARAMETER;
      }

      // Check that we can access the whole list
      rc = FSH_PROBEBUF(PB_OPREAD, ((EAOP *)pData)->fpGEAList, 
                        ((EAOP *)pData)->fpGEAList->cbList);
      if (rc != NO_ERROR) {
         FSH_SEMCLEAR(&CPData.BufLock);
         return rc;
      }
   }


   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FINDNEXT;
   CPData.OpData->findnext.fsfsi = *pfsfsi;
   CPData.OpData->findnext.fsfsd = *pfsfsd;
   if (level != 3)
      memcpy(CPData.Buf, pData, cbData);
   else
      memcpy(CPData.Buf, ((EAOP *)pData)->fpGEAList,
                        ((EAOP *)pData)->fpGEAList->cbList);
   CPData.OpData->findnext.cbData = cbData;
   CPData.OpData->findnext.cMatch = *pcMatch;
   CPData.OpData->findnext.level = level;
   CPData.OpData->findnext.flags = flags;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *pfsfsd = CPData.OpData->findnext.fsfsd;
   // If level 3, copy results starting at the oError entry in the EAOP
   if (level != 3) {
      memcpy(pData, CPData.Buf, cbData);
   } else {
      memcpy(pData + sizeof(EAOP), CPData.Buf, cbData - sizeof(EAOP));
      ((EAOP *)pData)->oError = CPData.OpData->findfirst.oError;
   }
   *pcMatch = CPData.OpData->findnext.cMatch;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FINDNOTIFYCLOSE - End a directory update request
**
** Parameters
** ----------
** unsigned short handle                handle of update request to close
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FINDNOTIFYCLOSE(unsigned short handle)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FINDNOTIFYFIRST - Begin a new directory update request
**
** Parameters
** ----------
** struct cdfsi far *pcdfsi             pointer to FSD independant current directory
** struct cdfsd far *pcdfsd             pointer to FSD dependant current directory
** char far *pName                      pointer to filename mask
** unsigned short iCurDirEnd            offset to the end of the current directory in pName
** unsigned short attr                  attribute mask
** unsigned short far *pHandle          pointer to place where FSD stores its handle
** char far *pData                      UNVERIFIED pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short far *pcMatch          maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned long timeout                timeout in milliseconds
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FINDNOTIFYFIRST(struct cdfsi far *pcdfsi, struct cdfsd far *pcdfsd,
                                        char far *pName, unsigned short iCurDirEnd,
                                        unsigned short attr, unsigned short far *pHandle,
                                        char far *pData, unsigned short cbData,
                                        unsigned short far *pcMatch, unsigned short level,
                                        unsigned long timeout)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_FINDNOTIFYNEXT - Continue directory update request
**
** Parameters
** ----------
** unsigned short handle                directory update handle
** char far *pData                      UNVERIFIED pointer to information area
** unsigned short cbData                size of area pointed to by pData
** unsigned short far *pcMatch          maximum number of entries to return*
**                                      number of entries actually returned
** unsigned short level                 level of information to return
** unsigned long timeout                timeout in milliseconds
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FINDNOTIFYNEXT(unsigned short handle, char far *pData,
                                       unsigned short cbData, unsigned short far *pcMatch,
                                       unsigned short level, unsigned long timeout)
{
   return ERROR_NOT_SUPPORTED;
}



/*-----------------------------------------------------------------------------
--
-- FSD Extended Interface
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_IOCTL - Perform an IOCTL on the IFS
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** unsigned short cat                   function category
** unsigned short func                  function to perform
** char far *pParm                      UNVERIFIED pointer to parameter area
** unsigned short lenParm               size of area pointed to by pParm
** unsigned short far *pParmLenInOut    length of parameters passed in pParm
** char far *pData                      UNVERIFIED pointer to information area
** unsigned short lenData               size of area pointed to by pData
** unsigned short far *pDataLenInOut    length of parameters passed in pData
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_IOCTL(struct sffsi far *psffsi, struct sffsd far *psffsd,
                              unsigned short cat, unsigned short func,
                              char far *pParm, unsigned short lenParm,
                              unsigned far *pParmLenInOut, char far *pData,
                              unsigned short lenData, unsigned far *pDataLenInOut)
{
   return ERROR_NOT_SUPPORTED;
}



/*-----------------------------------------------------------------------------
--
-- Miscellaneous Functions
--
-----------------------------------------------------------------------------*/



/******************************************************************************
**
** FS_NMPIPE - Perform a named pipe operation
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** unsigned short OpType                operation to perform
**   values:
**     NMP_GetPHandState
**     NMP_SetPHandState
**     NMP_PipeQInfo
**     NMP_PeekPipe
**     NMP_ConnectPipe
**     NMP_DisconnectPipe
**     NMP_TransactPipe
**     NMP_READRAW
**     NMP_WRITERAW
**     NMP_WAITPIPE
**     NMP_CALLPIPE
**     NMP_QNmPipeSemState
** union npoper far *pOpRec             data for operation
** char far *pData                      pointer to user data
** char far *pName                      pointer to remote named pipe
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_NMPIPE(struct sffsi far *psffsi, struct sffsd far *psffsd,
                               unsigned short OpType, union npoper far *pOpRec,
                               char far *pData, char far *pName)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_PROCESSNAME - Canonicalize a filename
**
** Parameters
** ----------
** char far *pNameBuf                   filename to canonicalize
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_PROCESSNAME(char far *pNameBuf)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_PROCESSNAME;
   strncpy(CPData.OpData->processname.NameBuf, pNameBuf, sizeof(CPData.OpData->processname.NameBuf));

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   strcpy(pNameBuf, CPData.OpData->processname.NameBuf);

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_SETSWAP - Inform IFS that it owns the swap file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_SETSWAP(struct sffsi far *psffsi,
                                        struct sffsd far *psffsd)
{
   return NO_ERROR;
}



/******************************************************************************
*******************************************************************************
**
** Locking support entry points
**
*******************************************************************************
******************************************************************************/



#if defined(SUP_FILEIO)
/******************************************************************************
**
** FS_CANCELLOCKREQUEST - Unlock a range in a file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** void far *pLockRange                 range to unlock
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_CANCELLOCKREQUEST(struct sffsi far *psffsi, struct sffsd far *psffsd,
                                          void far *pLockRange)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_CANCELLOCKREQUEST;
   CPData.OpData->cancellockrequest.sffsi = *psffsi;
   CPData.OpData->cancellockrequest.sffsd = *psffsd;
   CPData.OpData->cancellockrequest.LockRange = *pLockRange;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->cancellockrequest.sffsi;
   *psffsd = CPData.OpData->cancellockrequest.sffsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}


/******************************************************************************
**
** FS_FILELOCKS - Lock a range in a file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** void far *pUnLockRange               range to unlock
** void far *pLockRange                 range to lock
** unsigned long timeout                time in milliseconds to wait
** unsigned long flags                  flags
**   values:
**     0x01                     sharing of this file region is allowed
**     0x02                     atomic lock request
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_FILELOCKS(struct sffsi far *psffsi, struct sffsd far *psffsd,
                                  void far *pUnLockRange, void far *pLockRange,
                                  unsigned long timeout, unsigned long flags)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_FILELOCKS;
   CPData.OpData->filelocks.sffsi = *psffsi;
   CPData.OpData->filelocks.sffsd = *psffsd;
   CPData.OpData->filelocks.UnLockRange = *pUnLockRange;
   CPData.OpData->filelocks.LockRange = *pLockRange;
   CPData.OpData->filelocks.timeout = timeout;
   CPData.OpData->filelocks.flags = flags;

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results
   *psffsi = CPData.OpData->filelocks.sffsi;
   *psffsd = CPData.OpData->filelocks.sffsd;

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}

#endif


/******************************************************************************
*******************************************************************************
**
** UNC entry point
**
*******************************************************************************
******************************************************************************/


#if defined(SUP_UNC)

/******************************************************************************
**
** FS_VERIFYUNCNAME - Check if the IFS controls the server in question
**
** Parameters
** ----------
** unsigned short flag                  flags
**   values:
**     VUN_PASS1                pass 1 poll
**     VUN_PASS2                pass 2 poll
** char far *pName                      pointer to server in UNC format
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_VERIFYUNCNAME(unsigned short flag, char far *pName)
{
   unsigned short rc;
   
   rc = PreSetup();
   if (rc != NO_ERROR)
      return rc;


   // Verify parameters

   // Make sure the control program is still attached (this is how we
   // know our buffers are still valid)
   if (!CPAttached) {
      FSH_SEMCLEAR(&CPData.BufLock);
      return (ERROR_NOT_READY);
   }


   // Setup parameters -- from here to the PostSetup(), we MUST NOT block
   CPData.OpData->funccode = CPFC_VERIFYUNCNAME;
   CPData.OpData->verifyuncname.flag = flag;
   strncpy(CPData.OpData->verifyuncname.Name, pName, sizeof(CPData.OpData->verifyuncname.Name));

   rc = PostSetup();
   if (rc != NO_ERROR)
      return rc;


   // Copy results

   rc = CPData.OpData->rc;

   // We CANNOT access the buffers after this line
   FSH_SEMCLEAR(&CPData.BufLock);

   return rc;
}

#endif


/******************************************************************************
*******************************************************************************
**
** Swapper entry points
**
*******************************************************************************
******************************************************************************/

#if defined(SUP_PAGEIO)

/******************************************************************************
**
** FS_ALLOCATEPAGESPACE - Adjust size of the page file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** unsigned long ISize                  new size of page file
** unsigned long IWantContig            minimum continuity requirement
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_ALLOCATEPAGESPACE(struct sffsi far *psffsi,
                                          struct sffsd far *psffsd, 
                                          unsigned long lSize, 
                                          unsigned long lWantContig)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_DOPAGEIO - Perform I/O to the page file
**
** Parameters
** ----------
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** struct PageCmdHeader far *pPageCmdList operations to perform
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_DOPAGEIO(struct sffsi far *psffsi, struct sffsd far *psffsd,
                                 struct PageCmdHeader far *pPageCmdList)
{
   return ERROR_NOT_SUPPORTED;
}


/******************************************************************************
**
** FS_OPENPAGEFILE - Open the paging file
**
** Parameters
** ----------
** unsigned long far *pFlag             flags
**     PGIO_FIRSTOPEN           first open of page file
**     PGIO_PADDR               physical addresses required
**     PGIO_VADDR               16:16 virtual address required
** unsigned long far *pcMaxReq          pointer to maximum request list size
** char far *pName                      pointer to filename
** struct sffsi far *psffsi             pointer to FSD independant file instance
** struct sffsd far *psffsd             pointer to FSD dependant file instance
** unsigned long OpenMode               sharing and access mode
** unsigned short OpenFlag              action to take when file exists/doesn't exist
** unsigned short Attr                  OS/2 file attributes
** unsigned long Reserved               reserved
**
******************************************************************************/

#pragma argsused
short int far pascal _export _loadds FS_OPENPAGEFILE(unsigned long far *pFlags,
                                     unsigned long far *pcMaxReq,
                                     char far *pName, struct sffsi far *psffsi,
                                     struct sffsd far *psffsd, unsigned short OpenMode,
                                     unsigned short OpenFlag, unsigned short Attr,
                                     unsigned long Reserved)
{
   return ERROR_NOT_SUPPORTED;
}

#endif
