/* atm.h - general ATM declarations */
 
/* Written 1995,1996 by Werner Almesberger, EPFL LRC */
 

#ifndef _LINUX_ATM_H
#define _LINUX_ATM_H

/*
 * BEGIN_xx and END_xx markers are used for automatic generation of
 * documentation. Do not change them.
 */

#include <linux/atmsap.h>
#include <linux/atmioc.h>


/* general ATM constants */
#define ATM_CELL_SIZE		    53	/* ATM cell size incl. header */
#define ATM_CELL_PAYLOAD	    48	/* ATM payload size */
#define ATM_AAL0_SDU		    52	/* AAL0 SDU size */
#define ATM_MAX_AAL34_PDU	 65535	/* maximum AAL3/4 PDU payload */
#define ATM_AAL5_TRAILER	     8	/* AAL5 trailer size */
#define ATM_MAX_AAL5_PDU	 65535	/* maximum AAL5 PDU payload */
#define ATM_MAX_CDV		  9999	/* maximum (default) CDV */
#define ATM_NOT_RSV_VCI		    32	/* first non-reserved VCI value */

#define ATM_MAX_VPI		   255	/* maximum VPI at the UNI */
#define ATM_MAX_VPI_NNI		  4096	/* maximum VPI at the NNI */
#define ATM_MAX_VCI		 65535	/* maximum VCI */

/*
 * The following items should be added to sys/socket.h aka linux/socket.h
 */

/* address families */
#define AF_ATMPVC	6		/* ATM PVCs */
#define AF_ATMSVC	7		/* ATM SVCs */

/* "protcol" values for the socket system call */
#define ATM_AAL0	0		/* "raw" ATM cells */
#define ATM_AAL1	1		/* AAL1 (CBR) */
#define ATM_AAL2	2		/* AAL2 (VBR) */
#define ATM_AAL34	3		/* AAL3/4 (data) */
#define ATM_AAL5	5		/* AAL5 (data) */
#define ATM_SAAL	12		/* signaling AAL */

/*
 * UGLY - there should only be one protocol family (PF_ATM) for both
 * address families. A quick glance at some kernel internals seems to
 * suggest to me that doing it the "right" way might involve some
 * swimming against the stream ...
 */

/* protocol families */
#define PF_ATMPVC	AF_ATMPVC
#define PF_ATMSVC	AF_ATMSVC

/* layers for getsockopt/setsockopt */
#define SOL_ATM		2
#define SOL_AAL		3

/* ATM layer (values used for flags must be 2^n) */
#define SO_SETCLP	1		/* set CLP bit value - TODO */
#define SO_CIRANGE	3		/* connection identifier range;
					   socket must be bound or connected */
/* AAL layer */
#define SO_AALTYPE	0		/* AAL type, get only */

/* socket layer */
#define SO_BCTXOPT	16		/* not ATM specific - should go */
#define SO_BCRXOPT	17		/*   somewhere else */


/* for SO_BCTXOPT and SO_BCRXOPT */

struct atm_buffconst {
	unsigned long	buf_fac;	/* buffer alignment factor */
	unsigned long	buf_off;	/* buffer alignment offset */
	unsigned long	size_fac;	/* buffer size factor */
	unsigned long	size_off;	/* buffer size offset */
	unsigned long	min_size;	/* minimum size */
	unsigned long	max_size;	/* maximum size, 0 = unlimited */
};


/* ATM cell header (for AAL0) */

/* BEGIN_CH */
#define ATM_HDR_GFC_MASK	0xf0000000
#define ATM_HDR_GFC_SHIFT	28
#define ATM_HDR_VPI_MASK	0x0ff00000
#define ATM_HDR_VPI_SHIFT	20
#define ATM_HDR_VCI_MASK	0x000ffff0
#define ATM_HDR_VCI_SHIFT	4
#define ATM_HDR_PTI_MASK	0x0000000e
#define ATM_HDR_PTI_SHIFT	1
#define ATM_HDR_CLP		0x00000001
/* END_CH */


/* PTI codings */

/* BEGIN_PTI */
#define ATM_PTI_US0	0  /* user data cell, congestion not exp, SDU-type 0 */
#define ATM_PTI_US1	1  /* user data cell, congestion not exp, SDU-type 1 */
#define ATM_PTI_UCES0	2  /* user data cell, cong. experienced, SDU-type 0 */
#define ATM_PTI_UCES1	3  /* user data cell, cong. experienced, SDU-type 1 */
#define ATM_PTI_SEGF5	4  /* segment OAM F5 flow related cell */
#define ATM_PTI_E2EF5	5  /* end-to-end OAM F5 flow related cell */
#define ATM_PTI_RSV_RM	6  /* reserved for traffic control/resource mgmt */
#define ATM_PTI_RSV	7  /* reserved */
/* END_PTI */


/* for SO_CIRANGE */

#define ATM_CI_MAX	-1		/* use maximum range of VPI/VCI */

struct atm_cirange {
	char	vpi_bits;		/* 1..8, ATM_CI_MAX (-1) for maximum */
	char	vci_bits;		/* 1..16, ATM_CI_MAX (-1) for maximum */
};

/*
 * The following items should stay in linux/atm.h, which should be linked to
 * netatm/atm.h
 */

/* Traffic description */

#define ATM_NONE	0		/* no traffic */
#define ATM_UBR		1
#define ATM_CBR		2
#define ATM_VBR		3
#define ATM_ABR		4

#define ATM_MAX_PCR	-1		/* maximum available PCR */

struct atm_trafprm {
	unsigned char	class;		/* traffic class (ATM_UBR, ...) */
	int		max_pcr;	/* maximum PCR in cells per second */
	int		min_pcr;	/* minimum PCR in cells per second */
	int		max_cdv;	/* maximum CDV in microseconds */
	int		max_sdu;	/* maximum SDU in bytes */
};

/* PVC addressing */

#define ATM_ITF_ANY	-1		/* "magic" PVC address values */
#define ATM_VPI_ANY	-1
#define ATM_VCI_ANY	-1
#define ATM_VPI_UNSPEC	-2
#define ATM_VCI_UNSPEC	-2


struct sockaddr_atmpvc {
	unsigned short 	sap_family;	/* address family, AF_ATMPVC  */
	struct {			/* PVC address */
		short	itf;		/* ATM interface */
		short	vpi;		/* VPI (only 8 bits at UNI) */
		int	vci;		/* VCI (only 16 bits at UNI) */
	}		sap_addr;	/* PVC address */
	struct atm_trafprm sap_txtp;	/* TX traffic parameters */
	struct atm_trafprm sap_rxtp;	/* RX traffic parameters */
};

/* SVC addressing */

#define	ATM_ESA_LEN	20		/* ATM End System Address length */
#define ATM_E164_LEN	12		/* maximum E.164 number length */

#define ATM_MAX_BLLI	16		/* maximum number of BLLI elements */

#define ATM_AFI_DCC	0x39		/* DCC ATM Format */
#define ATM_AFI_ICD	0x47		/* ICD ATM Format */
#define ATM_AFI_E164	0x45		/* E.164 ATM Format */

struct sockaddr_atmsvc {
    unsigned short 	sas_family;	/* address family, AF_ATMSVC */
    struct {				/* SVC address */
        unsigned char	prv[ATM_ESA_LEN];/* private ATM address */
        char		pub[ATM_E164_LEN+1]; /* public address (E.164) */
    					/* unused addresses must be bzero'ed */
        struct atm_blli	*blli;		/* local SAP, low-layer information */
        struct atm_bhli	bhli;		/* local SAP, high-layer information */
    }			sas_addr;	/* SVC address */
    struct atm_trafprm	sas_txtp;	/* TX traffic parameters */
    struct atm_trafprm	sas_rxtp;	/* RX traffic parameters */
};


/*
 * Some stuff for linux/sockios.h
 */

struct atmif_sioc {
    int number;
    int length;
    void *arg;
};


#define SIOCSIFATMTCP	_IO('a',ATMIOC_ITF)	/* set ATMTCP mode */

/*
 * The following ioctls are obsolete and will be removed in a later release
 */

#define SIOCGIFATMADDR	_IOR('a',ATMIOC_ITF+1,struct atmif_sioc)
					/* get local ATM address */
#define SIOCSIFATMADDR	_IOR('a',ATMIOC_ITF+2,struct atmif_sioc)
					/* set local ATM address */

#ifdef __KERNEL__

#include <linux/net.h>	/* struct net_proto */


void atmpvc_proto_init(struct net_proto *pro);
void atmsvc_proto_init(struct net_proto *pro);

#endif /* __KERNEL__ */

#endif
