/* 
 *  atrun.c - run jobs queued by at; run with root privileges.
 *  Copyright (C) 1993  Thomas Koenig
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/* System Headers */

#include <sys/fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <dirent.h>
#include <errno.h>
#include <pwd.h>
#include <signal.h>
#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <syslog.h>

/* Local headers */

#include "gloadavg.h"

/* Macros */

#ifndef ATJOB_DIR
#define ATJOB_DIR "/usr/spool/atjobs/"
#endif

#ifndef ATSPOOL_DIR
#define ATSPOOL_DIR "/usr/spool/atspool/"
#endif

#ifndef LOADAVG_MX
#define LOADAVG_MX 1.5
#endif

/* File scope variables */

static char *namep;
static char rcsid[] = "$Id: atrun.c,v 1.9 1993/10/12 21:01:49 kernel Exp kernel $";

/* Local functions */
static void
perr(const char *a)
{
    syslog(LOG_ERR, "%s: %m", a);
    exit(EXIT_FAILURE);
}

static int
write_string(int fd, const char* a)
{
    return write(fd, a, strlen(a));
}

static void
run_file(const char *filename, uid_t uid, gid_t gid)
{
/* Run a file by by spawning off a process which redirects I/O,
 * spawns a subshell, then waits for it to complete and spawns another
 * process to send mail to the user.
 */
    pid_t pid;
    int fd;
    int queue;
    char mailbuf[9];
    char *mailname = NULL;
    FILE *stream;
    int send_mail = 0;
    struct stat buf;
    off_t size;

    pid = fork();
    if (pid == -1)
	perr("Cannot fork");
    
    else if (pid > 0)
	return;

    /* Let's see who we mail to.  Hopefully, we can read it from
     * the command file; if not, send it to the owner, or, failing that,
     * to root.
     */
    stream=fopen(filename, "r");
    if ((stream != NULL)
	&& (fscanf(stream, "#! /bin/sh\n# mail %8s %d",
		   mailbuf, &send_mail) == 2))
    {
	mailname = mailbuf;
	if (stream != NULL)
	    fclose(stream);
    }
    else
    {
	struct passwd *pentry;

	pentry = getpwuid(uid);
	if (pentry == NULL)
	    mailname = "root";
	else
	    mailname = pentry->pw_name;
    }
    if (chdir(ATSPOOL_DIR) < 0)
	perr("Cannot chdir to " ATSPOOL_DIR);
    
    /* Create a file to hold the output of the job we are
     * about to run.  Reassign stdin to /dev/null and stdout and stderr
     * to it, to prepare things to fork.
     */    
    if((fd=open(filename,
		O_WRONLY | O_CREAT | O_EXCL, S_IWUSR | S_IRUSR)) < 0)
	perr("Cannot create output file");

    write_string(fd, "Subject: Output from your job ");
    write_string(fd, filename);
    write_string(fd, "\n\n");
    fstat(fd, &buf);
    size = buf.st_size;

    if (chown(filename, uid, gid) < 0)
	perr("Cannot give away output file");

    close(STDIN_FILENO);
    close(STDOUT_FILENO);
    close(STDERR_FILENO);
 
    pid = fork();
    if (pid < 0)
	perr("Error in fork");

    else if (pid == 0)
    {
	char *nul = NULL;
	char **nenvp = &nul;

	/* Set up things for the child; we want no standard input,
	 * and standard output and error sent to our output file.
	 */
	if (open("/dev/null",O_RDONLY) != STDIN_FILENO)
	    perr("Error in I/O redirection");
    
	if (dup(fd) != STDOUT_FILENO)
	    perr("Error in I/O redirection");

	if (dup(fd) != STDERR_FILENO)
	    perr("Error in I/O redirection");

	close(fd);
	if (chdir(ATJOB_DIR) < 0)
	    perr("Cannot chdir to " ATJOB_DIR);

	queue = *filename;
	if (queue > 'b')
	    nice(queue - 'b');
	
	if (setgid(gid) < 0)
	    perr("Cannot change group");

	if (setuid(uid) < 0)
	    perr("Cannot set user id");

	if(execle("/bin/sh","sh",filename ,(char *) NULL, nenvp) != 0)
	    perr("Exec failed");
    }
    /* We're the parent.  Let's wait.
     */
    close(fd);
    waitpid(pid, (int *) NULL, 0);

    stat(filename, &buf);
    if ((buf.st_size != size) || send_mail)
    {
	/* Fork off a child for sending mail
	 */
	pid = fork();
	if (pid < 0)
	    perr("Fork failed");
	else if (pid == 0)
	{
	    if (open(filename, O_RDONLY) != STDIN_FILENO)
		perr("Cannot reopen output file");
	    
	    execl(MAIL_CMD, MAIL_CMD, mailname, (char *) NULL);
	}
	waitpid(pid, (int *) NULL, 0);
    }
    unlink(filename);
    exit(EXIT_SUCCESS);
}

/* Global functions */

int
main(int argc, char *argv[])
{
/* Browse through  ATJOB_DIR, checking all the jobfiles wether they should
 * be executed and or deleted. The queue is coded into the first byte of
 * the job filename, the date (in minutes since Eon) as a hex number in the
 * following eight bytes, followed by a dot and a serial number.  A file
 * which has not been executed yet is denoted by its execute - bit set.
 * For those files which are supposed to be executed, a child is forked
 * off which takes care of I/O redirection, then calls run_file() to
 * handle execution. Files which already have run are removed during
 * the next invocation.
 */
    DIR *spool;
    struct dirent *dirent;
    struct stat buf;
    int older;
    unsigned long ctm;
    char queue;

    openlog("atrun", LOG_PID, LOG_CRON);

    namep = argv[0];
    if (chdir(ATJOB_DIR) != 0)
	perr("Cannot change to " ATJOB_DIR);

    /* Main loop. Open spool directory for reading and look over all the
     * files in there. If the filename indicates that the job should be run
     * and the x bit is set, fork off a child which sets its user and group
     * id to that of the files and exec a /bin/sh which executes the shell
     * script. Unlink older files if they should no longer be run.  For
     * deletion, their r bit has to be turned on.
     */
    if ((spool = opendir(".")) == NULL)
	perr("Cannot read " ATJOB_DIR);

    while ((dirent = readdir(spool)) != NULL)
    {
	if (stat(dirent->d_name,&buf) != 0)
	    perr("Cannot stat in " ATJOB_DIR);

	/* We don't want directories
	 */
	if (!S_ISREG(buf.st_mode)) 
	    continue;

	if (sscanf(dirent->d_name,"%c%8lx",&queue,&ctm) != 2)
	    continue;

	if ((queue == 'b') && (gloadavg() > LOADAVG_MX))
	    continue;

	older = (time_t) ctm*60 <= time(NULL);
	
	/* The file is executable and old enough
	 */
	if (older && (S_IXUSR & buf.st_mode))
	{
	    /* Now we know we want to run the file, we can turn off
	     * the execute bit
	     */
	    if (chmod(dirent->d_name, S_IRUSR) != 0)
		perr("Cannot change file permissions");

	    run_file(dirent->d_name, buf.st_uid, buf.st_gid);
	}
	/*  Delete older files
	 */
	if (older && !(S_IXUSR & buf.st_mode) && (S_IRUSR & buf.st_mode))
	    unlink(dirent->d_name);
    }
    closelog();
    exit(EXIT_SUCCESS);
}
